/**
 * @defgroup IPCProvider IPCProvider
 * @ingroup Mediaplayer
 * @author Thomas Porsch
 *
  * Public interface for IPC provider used by all components which communicate with the PlayerEngine via DBus
 * @{
 */

#ifndef _IPCPROVIDER_H_
#define _IPCPROVIDER_H_

#include <map>
#include <vector>
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "IpcClient.h"
#include "TypeDefinitions.h"
#include "SMF.h"

class IPCListener
{
public:
    virtual bool belongs(const char* messageName, const char* params) = 0;
    virtual bool belongs(tU32 serial) = 0;

    virtual void RouteSignal(const char* name, const char* params) = 0;
    virtual void RouteMethodAnswer(tU32 serial, const char* params) = 0;
    virtual void RouteMethodRequest(tU32 serial, const char* name, const char* params) = 0;
    virtual void RouteError(tU32 serial, const char* error) = 0;
};

class IPCProvider : public ILocalSPM, public TFThread, public IpcClient
{

public:

//SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the IPCProvider componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    IPCProvider(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~IPCProvider();

    /**
     * This function is used by LocalSPM to create the IPCProvider.
     * Create the IPCProvider state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the IPCProvider initialization.
     * Init the IPCProvider state machine.
     * Initialize variables.
     * Register IPCProviderSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * This function is used by LocalSPM to start the IPCProvider.
     * From now on all other mediaplayer components are available.
     * Start the IPCProvider thread and the state machine.
     * Start IPC client thread
     * Register observer at PlayerEngine via DBUS -> RegisterObserver(service="de.bosch.mediaplayer",path="/MediaPlayer",interface="local.MediaPlayer").
     * PlayerEngine command has return value of success.
     * Comment: The register stuff should be done in first init transition of state machine
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the IPCProvider.
     * @attention: running in SPM thread context
     * Clear observer and method call map.
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * This function is used by LocalSPM to cleanup the IPCProvider.
     * @attention: running in SPM thread context
     * Deregister IPCProviderSM with dispatcher.
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();


//PlayerEngine interface part

    /**
     * Function registers an internal observer(DeviceDispatcher, USBControl or iPodControl).
     * Register observer not necessary. It's already done for MediaPlayer.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult RegisterObserver(SMF *component) const;

    /**
     * Function unregisters an internal observer(DeviceDispatcher, USBControl or iPodControl).
     * Reset pointer to observer in case of observer is currently active.
     * Do not unregister MediaPlayer observer! We still need it for other observer.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult UnregisterObserver(SMF *component);

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Save pointer to observer (DeviceDispatcher, USBControl or iPodControl).
     * Switch observer to MediaPlayer at PlayerEngine via DBUS -> SwitchObserver(service="de.bosch.mediaplayer",path="/MediaPlayer",interface="local.MediaPlayer").
     * PlayerEngine command has return value of success.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult SwitchObserver(SMF *component);

    /**
     * Function transfers assign_audio_output_device command to PlayerEngine.
     * Assign audio output device at PlayerEngine via DBUS -> AssignAudioOutputDevice(audioOutputDevice).
     * PlayerEngine command has return value of success.
     *
     * @param[in] component pointer to component which called function
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult AssignAudioOutputDevice(SMF *component, const tAudioOutputDevice audioOutputDevice);

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * Assign audio input device at PlayerEngine via DBUS -> AssignAudioInputDevice(audioInputDevice).
     * PlayerEngine command has return value of success.
     *
     * @param[in] component pointer to component which called function
     * @param[in] audioInputDevice device name (ALSA) of audio input
     * @return != 0: error, = 0: OK
     */
    tResult AssignAudioInputDevice(SMF *component, const tAudioInputDevice audioInputDevice);

    /**
     * Function transfers deallocate_out_device command to PlayerEngine.
     * Send DeAllocateOutDevice to PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult DeAllocateOutDevice(SMF *component);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * Send DeAllocateInDevice to PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult DeAllocateInDevice(SMF *component);

    /**
     * Function transfers new_play_slot command to PlayerEngine.
     * Send NewPlaySlot to PlayerEngine via DBUS -> NewPlaySlot(URL, position, playpointFormat=Absolute).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     *
     * @param[in] component pointer to component which called function
     * @param[in] URL file path and name including mount point
     * @param[in] position playtime
     * @param[in] playpointFormat format of PlaypointPosition parameter
     *            (percentage or absolute time value in milliseconds)
     * @return != 0: error, = 0: OK
     */
    tResult NewPlaySlot(SMF *component, const tURL URL, const tPlaytime position, const tPlaypointFormat playpointFormat);

    /**
     * Function transfers stop_slot command to PlayerEngine.
     * Send StopSlot to PlayerEngine via DBUS -> StopSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=STOPPEDSTATE) later.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult StopSlot(SMF *component);

    /**
     * Function transfers pause_slot command to PlayerEngine.
     * Send PauseSlot to PlayerEngine via DBUS -> PauseSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PAUSEDSTATE) later.
     *
     * @param[in] component pointer to component which called function
     * @param[in] URL file path and name including mount point
     * @param[in] position playtime
     * @param[in] playpointFormat format of PlaypointPosition parameter
     *            (percentage or absolute time value in milliseconds)
     * @return != 0: error, = 0: OK
     */
    tResult PauseSlot(SMF *component, const tURL URL, const tPlaytime position, const tPlaypointFormat playpointFormat);

    /**
     * Function transfers resume_play_slot command to PlayerEngine.
     * Send ResumePlaySlot to PlayerEngine via DBUS -> ResumePlaySlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult ResumePlaySlot(SMF *component);

    /**
     * Function transfers seek_to_slot command to PlayerEngine.
     * Send SeekToSlot to PlayerEngine via DBUS -> SeekToSlot(position, playpointFormat=Absolute).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later???
     *
     * @param[in] component pointer to component which called function
     * @param[in] position playtime
     * @param[in] playpointFormat format of PlaypointPosition parameter
     *            (percentage or absolute time value in milliseconds)
     * @return != 0: error, = 0: OK
     */
    tResult SeekToSlot(SMF *component, const tPlaytime position, const tPlaypointFormat playpointFormat);

    /**
     * Function transfers play_from_device_slot command to PlayerEngine.
     * Send PlayFromDeviceSlot to PlayerEngine via DBUS -> PlayFromDeviceSlot(URL, position, playpointFormat=Absolute).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     *
     * @param[in] component pointer to component which called function
     * @param[in] encoding encoding type = audio
     * @param[in] bitRate bit rate of the decoded audio
     * @param[in] sampleRate sample rate of the decoded audio
     * @param[in] channels number of channels of the decoded audio
     * @param[in] sampleWidth width of the sample in bits
     * @param[in] sampleDepth depth of the sample in bits
     * @param[in] sampleSignedness 0/1 in case of sample unsigned/signed
     * @param[in] sampleEndianness 0/1 in case of sample little/big endian
     * @return != 0: error, = 0: OK
     */
    tResult PlayFromDeviceSlot(SMF *component,
        const tEncoding encoding,
        const tBitRate bitRate,
        const me::samplerate_i sampleRate,
        const me::channels_i channels,
        const me::samplewidth_i sampleWidth,
        const me::sampledepth_i sampleDepth,
        const me::signedness_e sampleSignedness,
        const me::endianess_e sampleEndianness);

    /**
     * Function transfers start_ff_slot command to PlayerEngine.
     * Send StartFfSlot to PlayerEngine via DBUS -> StartFfSlot(rate).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=FASTFORWARDSTATE) later???
     *
     * @param[in] component pointer to component which called function
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfSlot(SMF *component, const tCueingRate rate);

    /**
     * Function transfers stop_ff_slot command to PlayerEngine.
     * Send StopFfSlot to PlayerEngine via DBUS -> StopFfSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later???
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult StopFfSlot(SMF *component);

    /**
     * Function transfers start_frev_slot command to PlayerEngine.
     * Send StartFrevSlot to PlayerEngine via DBUS -> StartFrevSlot(rate).
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=FASTREVERSESTATE) later???
     *
     * @param[in] component pointer to component which called function
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevSlot(SMF *component, const tCueingRate rate);

    /**
     * Function transfers stop_frev_slot command to PlayerEngine.
     * Send StopFrevSlot to PlayerEngine via DBUS -> StopFrevSlot().
     * PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later???
     *
     * @param[in] component pointer to component which called function
     * @return != 0: error, = 0: OK
     */
    tResult StopFrevSlot(SMF *component);

    /**
     * Register a listener who likes receive a specific dbus service updates
     * (e.g BTDaemon listener , PlayerEngine listener)
     *
     * @param[in] listener pointer to instance which implements IPCListener interface to received dbus updates from a service
     * @return none
     */
    void RegisterListener(IPCListener *listener);

    /**
     * Gets the interface name from the DBUS property Params
     *
     * @param[out] interfacename
     * @param[in] params
     * @return none
     */
    void GetInterfaceName(OUT tInterfaceName &interfacename, IN const char* params);

private:
    tBoolean SignalReceived(const char* name, const char* params);
    tBoolean MethodAnswered(dbus_uint32_t serial, const char* params);
    tBoolean MethodRequested(dbus_uint32_t serial, const char* name, const char* params);
    tBoolean Error(dbus_uint32_t serial, const char* errorName, const char* errorMessage);
    void AddFilters();
    void RemoveFilters();

    tBoolean m_Initialized;           /**< init complete: PlayerEngine is created and MediaPlayer is registered */
    tInteger m_RegisterSerial;           /**< dbus serial received from the RegisterObserver method call, required to track the method answer */
    tInteger m_SetVideoLayerSerial;      /**< dbus serial received from the SetVideoLayer method call, required to track the method answer */
    SMF *m_SuspendComponent;   /**< pointer to component of suspended method call */
    tAudioOutputDevice m_SuspendAudioOutputDevice;  /**< audio output device of suspended method call */
    volatile tBoolean mStopThread;
    Lock mEndOfThread;

    SMF *m_Observer;      /**< pointer to current observer */
    map<tInteger, SMF *> m_MethodCalls;  /**< map of DBus serial and component which call method */

    vector <IPCListener *> mListeners;    /**< list of all local components which implements IPCListener interface and listens for dbus responses*/

    Lock mLockObserver;
};

#endif /*_IPCPROVIDER_H_*/

/** @} */
