/*
 * MessageQueueTest.cpp
 *
 *  Created on: Jul 1, 2013
 *      Author: tritonsu
 */

#include "MessageQueueTest.h"
#include "MessageQueue.h"
#include "ElapsedTimer.h"
#include "Utils.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "TraceDefinitions.h"
#include "FunctionTracer.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#ifdef TARGET_BUILD
#include "trcGenProj/Header/MessageQueueTest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#endif
#endif

static MessageQueue *queue ;

void HappyDayTestMessageSend(MessageQueue *msgQ, char *message, int prio)
{
    void *Sendbuf;
    Sendbuf = msgQ->CreateMessageBuffer(strlen_r(message)+1);
    strcpy((char *)Sendbuf, message);
    msgQ->Push(Sendbuf, strlen_r(message)+1,prio);
}

void* MessageQueueTest::WriteThreadFunction(void *pQueue)
{
    //MessageQueue *pMessageQueue = (MessageQueue *)pQueue;
    (void)pQueue;
    sleep (2);
    if(queue != NULL)
    {
        HappyDayTestMessageSend(queue, (char *)"Message Arrived after 3 sec", 25);
    }

    pthread_exit(0);
    return NULL;
}

void MessageQueueTest::HappyDayTest()
{
    ENTRY_TEST
 
    MessageQueue msgQueue("MQ1");

    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage1_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage2_prio2", 2);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage3_prio1", 1);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage4_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage5_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage6_prio6", 6);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage7_prio7", 7);

    ETG_TRACE_USR3(("OK - Message queue size:%d",(int)msgQueue.GetCurMessagesCount()));

    CPPUNIT_ASSERT(msgQueue.GetCurMessagesCount() == 7);

    void *receive_buffer;
    size_t receive_size;

    while(!msgQueue.Pop(&receive_buffer,&receive_size))
    {
        ETG_TRACE_USR3(("OK - read queue -> message_size:%d,message:%s",receive_size,(char*)receive_buffer));
        msgQueue.DropMessage(receive_buffer);
    }
}

void MessageQueueTest::WrapAroundTest()
{
    ENTRY_TEST
 #define CHECK_CONTENTS_WA 1

    MessageQueue msgQueue("MQ2");

    int integralSize = msgQueue.GetIntegralSize();
    enum {messageSize = 128};
    int pushIndex = 0;

    for(int i=0; i<integralSize-1; i++) {

        /* allocate the send-buffer */
        unsigned char *Sendbuf = (unsigned char *)msgQueue.CreateMessageBuffer(messageSize);

#if CHECK_CONTENTS_WA
        /* fill it with data */
        srand(pushIndex);
        for(int j=sizeof(int); j<messageSize; j++) {
            Sendbuf[j] = (unsigned char)rand();
        }
        memcpy(Sendbuf, &pushIndex, sizeof(pushIndex)); // seed value for later check
        pushIndex++;
#endif // CHECK_CONTENTS

        /* send it */
        msgQueue.Push(Sendbuf,messageSize,10);
    }

    /*
     * now tail points to last element in integral queue size, head points to the first unread element
     */

    unsigned char *receive_buffer;
    size_t receive_size;
    for(int i=0; i<integralSize/2; i++) {

        msgQueue.Pop((void **)&receive_buffer,&receive_size);

#if CHECK_CONTENTS_WA
        /* check contents */
        int seed;
        memcpy(&seed, receive_buffer, sizeof(seed));
        srand(seed);
        for(int j=sizeof(int); j<messageSize; j++) {
            if (receive_buffer[j] != (unsigned char)rand()) {
                CPPUNIT_ASSERT(0);
            }
        }
#endif // CHECK_CONTENTS

        /* free the buffer */
        msgQueue.DropMessage(receive_buffer);
    }
    /*
     * now head points to the middle of the queue, tails is still at the end of the queue
     */

    /*
     * now, add ONE plus the read number of elements
     */
    for(int i=0; i<(1+(integralSize/2)); i++) {

        /* allocate the send-buffer */
        unsigned char *Sendbuf = (unsigned char *)msgQueue.CreateMessageBuffer(messageSize);

#if CHECK_CONTENTS_WA
        /* fill it with data */
        srand(pushIndex);
        for(int j=sizeof(int); j<messageSize; j++) {
            Sendbuf[j] = (unsigned char)rand();
        }
        memcpy(Sendbuf, &pushIndex, sizeof(pushIndex)); // seed value for later check
        pushIndex++;
#endif // CHECK_CONTENTS

        /* send it */
        msgQueue.Push(Sendbuf,messageSize,10);
    }
    /*
     * the pushes above have increased the size of the queue by integral size.
     * the head is still in the middle of the old queue space.
     * tail should be before head
     */

    int elementReadCounter = 0;
    while(!msgQueue.Pop((void **)&receive_buffer,&receive_size))
    {
        elementReadCounter++;

#if CHECK_CONTENTS_WA
        /* check contents */
        int seed;
        memcpy(&seed, receive_buffer, sizeof(seed));
        srand(seed);
        for(int j=sizeof(int); j<messageSize; j++) {
            if (receive_buffer[j] != (unsigned char)rand()) {
                CPPUNIT_ASSERT(0);
            }
        }
#endif // CHECK_CONTENTS

        /* free the buffer */
        msgQueue.DropMessage(receive_buffer);
    }

    ETG_TRACE_USR2(("WrapAroundTest: elementReadCounter=%d, integralSize=%d", elementReadCounter, integralSize));
    CPPUNIT_ASSERT(integralSize == elementReadCounter);

    //empty queue here.

    //verify extension part in reallocated queue buffer space

    int numberOfTestElements = 5 * integralSize;
    elementReadCounter = 0;
    for(int i=0; i < numberOfTestElements; i++) {

        /* allocate the send-buffer */
        unsigned char *Sendbuf = (unsigned char *)msgQueue.CreateMessageBuffer(messageSize);

#if CHECK_CONTENTS_WA
        /* fill it with data */
        srand(pushIndex);
        for(int j=sizeof(int); j<messageSize; j++) {
            Sendbuf[j] = (unsigned char)rand();
        }
        memcpy(Sendbuf, &pushIndex, sizeof(pushIndex)); // seed value for later check
        pushIndex++;
#endif // CHECK_CONTENTS

        /* send it */
        msgQueue.Push(Sendbuf,messageSize,10);
    }

    while(!msgQueue.Pop((void **)&receive_buffer,&receive_size))
    {
        elementReadCounter++;

#if CHECK_CONTENTS_WA
        /* check contents */
        int seed;
        memcpy(&seed, receive_buffer, sizeof(seed));
        srand(seed);
        for(int j=sizeof(int); j<messageSize; j++) {
            if (receive_buffer[j] != (unsigned char)rand()) {
                CPPUNIT_ASSERT(0);
            }
        }
#endif // CHECK_CONTENTS

        /* free the buffer */
        msgQueue.DropMessage(receive_buffer);
    }

    ETG_TRACE_USR2(("WrapAroundTest: elementReadCounter=%d, numberOfTestElements=%d", elementReadCounter, numberOfTestElements));
    CPPUNIT_ASSERT(numberOfTestElements == elementReadCounter);
}

void MessageQueueTest::SpeedTest()
{
    ENTRY_TEST
 
    MessageQueue msgQueue("MQ1");

    /* send speed */
    ElapsedTimer measTime;
    unsigned long durationUS;

    enum {noOfTestMessages = 100000};
    enum {messageSize = 128};
    enum {maxPrio = 70};

#define CHECK_CONTENTS 1

    measTime.Restart();
    for(int i=0; i<noOfTestMessages; i++) {

        /* allocate the send-buffer */
        unsigned char *Sendbuf = (unsigned char *)msgQueue.CreateMessageBuffer(messageSize);

#if CHECK_CONTENTS
        /* fill it with data */
        srand(i);
        for(int j=sizeof(int); j<messageSize; j++) {
            Sendbuf[j] = (unsigned char)rand();
        }
        memcpy(Sendbuf, &i, sizeof(i)); // seed value for later check
#endif // CHECK_CONTENTS

        /* send it */
        msgQueue.Push(Sendbuf,messageSize,(rand()*maxPrio)/RAND_MAX);
    }
    durationUS = measTime.ElapsedUS();
    CPPUNIT_ASSERT(msgQueue.GetCurMessagesCount() == noOfTestMessages);
    ETG_TRACE_USR2(("Queue write: in %f ms/message", ((float)durationUS / 1000.) / noOfTestMessages ));

    unsigned char *receive_buffer;
    size_t receive_size;

    /* restart time meas */
    measTime.Restart();

    /* receive one message */
    while(!msgQueue.Pop((void **)&receive_buffer,&receive_size))
    {

#if CHECK_CONTENTS
        /* check contents */
        int seed;
        memcpy(&seed, receive_buffer, sizeof(seed));
        srand(seed);
        for(int j=sizeof(int); j<messageSize; j++) {
            if (receive_buffer[j] != (unsigned char)rand()) {
                CPPUNIT_ASSERT(0);
            }
        }
#endif // CHECK_CONTENTS

        /* free the buffer */
        msgQueue.DropMessage(receive_buffer);
    }
    durationUS = measTime.ElapsedUS();
    ETG_TRACE_USR2(("Queue read: in %f ms/message", ((float)durationUS / 1000.) / noOfTestMessages ));
}

void MessageQueueTest::MessageWaitFixedTime()
{
    ENTRY_TEST
 
    queue = new MessageQueue("MQ2");
    pthread_t threadHandle;
    pthread_attr_t threadAttr;
    pthread_attr_init(&threadAttr);

    pthread_create(&threadHandle, &threadAttr, &MessageQueueTest::WriteThreadFunction, queue);

    sleep (1);

    void *receive_buffer = NULL;
    size_t receive_size = 0;

    receive_buffer = queue->WaitForMessage(&receive_size, 5000);

    if(receive_buffer && receive_size)
    {
        ETG_TRACE_USR3(("OK - message received after timed wait : %s",(char*)receive_buffer));
        queue->DropMessage(receive_buffer);
    }
    else
    {
        ETG_TRACE_USR3(("ERROR - no message could be received"));
    }
    delete queue;
    queue = NULL;
}

void MessageQueueTest::MessageWaitUnlimited()
{
    ENTRY_TEST
 
    queue = new MessageQueue("MQ4");
    pthread_t threadHandle;
    pthread_attr_t threadAttr;
    pthread_attr_init(&threadAttr);

    pthread_create(&threadHandle, &threadAttr, &MessageQueueTest::WriteThreadFunction, queue);

    void *receive_buffer = NULL;
    size_t receive_size = 0;

    receive_buffer = queue->WaitForMessage(&receive_size,0);

    if(receive_buffer && receive_size)
    {
        ETG_TRACE_USR3(("OK - message received after unlimited wait : %s",(char*)receive_buffer));
        queue->DropMessage(receive_buffer);
    }
    else
    {
        ETG_TRACE_USR3(("ERROR - no message could be received , an error occurred during wait"));
    }
    delete queue;
    queue = NULL;
}

void MessageQueueTest::MessageWaitTimeout()
{
    ENTRY_TEST
 
    queue = new MessageQueue("MQ3");
    pthread_t threadHandle;
    pthread_attr_t threadAttr;
    pthread_attr_init(&threadAttr);

    pthread_create(&threadHandle, &threadAttr, &MessageQueueTest::WriteThreadFunction, queue);

    sleep (1);

    void *receive_buffer = NULL;
    size_t receive_size = 0;

    receive_buffer = queue->WaitForMessage(&receive_size,2);

    if(receive_buffer && receive_size)
    {
        ETG_TRACE_USR3(("ERROR - message received after wait : %s",(char*)receive_buffer));
        queue->DropMessage(receive_buffer);
    }
    else
    {
        ETG_TRACE_USR3(("OK - no message could be received within timeout of 2sec"));
    }
    delete  queue;
    queue = NULL;
}

void MessageQueueTest::MessagePeekAndDelete()
{
    ENTRY_TEST
 
    MessageQueue msgQueue("MQ5");

    /* send first some messages */
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage1_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage2_prio2", 2);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage3_prio1", 1);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage4_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage5_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage6_prio6", 6);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage7_prio7", 7);

    /* now search a specific message by peeking into the queue */
    int iMessage = 0;
    int iMessageFound = -1;
    while(1) {

        /* peek one message */
        size_t msgSize;
        char *message = (char *)msgQueue.Peek(iMessage, &msgSize);
        if (message == NULL) break;

        /* is this the desired message? */
        if (!strcmp(message, "testmessage4_prio3")) {
            iMessageFound = iMessage;
            break;
        }

        /* next message number */
        iMessage++;
    }
    CPPUNIT_ASSERT(iMessageFound != -1);

    /* delete this message */
    msgQueue.Remove(iMessageFound);

    /* check the number of messages */
    CPPUNIT_ASSERT(msgQueue.GetCurMessagesCount() == 6);

    char *receive_buffer;
    size_t receive_size;

    /* read all remaining messages */
    iMessage = 0;
    while(!msgQueue.Pop((void **)&receive_buffer,&receive_size))
    {
        /* check the contents */
        switch(iMessage) {
        case 0:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage3_prio1") == 0);
            break;
        case 1:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage2_prio2") == 0);
            break;
        case 2:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage1_prio3") == 0);
            break;
        case 3:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage5_prio3") == 0);
            break;
        case 4:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage6_prio6") == 0);
            break;
        case 5:
            CPPUNIT_ASSERT(strcmp(receive_buffer, "testmessage7_prio7") == 0);
            break;
        }
        iMessage++;

        /* free the buffer */
        msgQueue.DropMessage(receive_buffer);
    }
}

void MessageQueueTest::MessageFlush()
{
    ENTRY_TEST
 
    MessageQueue msgQueue("MQ6");

    /* send first some messages */
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage1_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage2_prio2", 2);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage3_prio1", 1);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage4_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage5_prio3", 3);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage6_prio6", 6);
    HappyDayTestMessageSend(&msgQueue, (char *)"testmessage7_prio7", 7);

    /* delete all messages */
    msgQueue.Flush();

    /* check the number of messages */
    CPPUNIT_ASSERT(msgQueue.GetCurMessagesCount() == 0);
}


void MessageQueueTest::StressCreateDeleteQTest()
{
    ENTRY_TEST
    MessageQueue *pMsgQ = NULL;

    for(int i=0;i<100;i++)
    {
        pMsgQ = new MessageQueue("MsgQFullTextSearch");
        CPPUNIT_ASSERT(pMsgQ!=NULL);

        pMsgQ->Flush();

        delete pMsgQ;
        
    }
    

    ETG_TRACE_USR1(("          ***End   %s***", __PRETTY_FUNCTION__));
}

