/*
 * RemoteStateMachineManager.cpp
 *
 *  Created on: Mar 17, 2014
 *      Author: thm3hi
 */
/* ETG definitions */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FRAMEWORK
#ifdef TARGET_BUILD
#include "trcGenProj/Header/RemoteStateMachineManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_FRAMEWORK
#endif
#endif

#include "FunctionTracer.h"
#include "TraceDefinitions.h"

#include "BaseTypes.h"
#include <RemoteStateMachineManager.h>
#include <Dispatcher.h>

/*lint -save -e801 */

/* switches */
#define DBUS_SEND_ON 1

#define DBUS_INTERFACE "org.bosch.GenericMediaPlayer"
#define DBUS_OBJECT    "/MediaEngine"
#define DBUS_SIGNAL    "SMF_EventAndAnswer"

static DBusObjectPathVTable DbusVTable = {
        RemoteStateMachineManager::DbusUnregister,
        RemoteStateMachineManager::DbusSignalReceived,
        NULL,
        NULL,
        NULL,
        NULL
};

int RemoteStateMachineManager::Init()
{
    ENTRY_INTERNAL
    trace("RemoteStateMachineManager:Init");
    int res;
    DBusError err;

    /* init stop flag */
    mStopThread = 0;    //CID 17782 (#1 of 2): Uninitialized scalar field (UNINIT_CTOR)

    /* init some values */
    m_threadId = 0;

    /* switch dbus to multi threadded locking */
    dbus_threads_init_default();

    /* init the Dbus error values */
    dbus_error_init(&err);

    /* connect to DBus session bus (needed for send and receive */
    mConnection = dbus_bus_get(DBUS_BUS_SESSION, &err);
    trace("RemoteStateMachineManager:Init:dbus_bus_get:session:", DBUS_BUS_SESSION, " conn:", (int_t)mConnection);        
    if(!mConnection) {
        trace("RemoteStateMachineManager:dbus_bus_get:err:", err.message);
        return MP_ERR_DISP_OPEN_DBUS;
    }

    /* set flag not to exit process on disconnection */
    dbus_connection_set_exit_on_disconnect(mConnection, FALSE);

    /* add a filter to get only specific signals */
    char filter[128];
    snprintf(filter, sizeof(filter), "type='signal',interface='%s'", DBUS_INTERFACE);
    trace("RemoteStateMachineManager:Init:dbus_bus_add_match:filter:", filter);            
    dbus_bus_add_match(mConnection, filter ,NULL);

#if 1
    /* connect a callback to the filter */
    res = dbus_connection_try_register_object_path(mConnection, DBUS_OBJECT, &DbusVTable, NULL, &err);
    if (res==0 && !dbus_error_has_name(&err, DBUS_ERROR_OBJECT_PATH_IN_USE)) {
        trace("RemoteStateMachineManager:dbus_connection_try_register_object_path:err:", err.message);
    }
#else // test
    dbus_connection_add_filter(mConnection, RemoteStateMachineManager::DbusSignalReceived, NULL, NULL);
#endif
    return MP_NO_ERROR;
}

int RemoteStateMachineManager::StartDbusMainLoop()
{
    ENTRY
    trace("RemoteStateMachineManager:StartDbusMainLoop");    
    int res;

    /* init stop flag */
    mStopThread = 0;

    /* spawn receiver thread */
    pthread_attr_init(&m_attr);
    pthread_attr_setstacksize (&m_attr, 10000000);
    res = pthread_create(&m_threadId, &m_attr, &DbusDispatcherThread, this);
    if (res) {
        return MP_ERR_DISP_SPAWN_RECV_THREAD;
    }

    return MP_NO_ERROR;
}

int RemoteStateMachineManager::DeInit()
{
   ENTRY_INTERNAL
   trace("RemoteStateMachineManager:DeInit");       

    /* end receiver thread */
    mStopThread = 1;

    /* wait for end */
    if (m_threadId) {
        pthread_cancel(m_threadId);
        pthread_join(m_threadId, NULL);
        m_threadId = 0;
    }

    return MP_NO_ERROR;
}

int SetPriority(int prio)
{
    ENTRY_INTERNAL

    int policy;
    struct sched_param param;

    pthread_getschedparam(pthread_self(), &policy, &param);

    param.__sched_priority = prio;
    policy = SCHED_FIFO;

    pthread_setschedparam(pthread_self(), policy, &param);

    return 0;
}

void* RemoteStateMachineManager::DbusDispatcherThread(void *_threadCommunication)
{
   ENTRY_INTERNAL
   trace("RemoteStateMachineManager:DbusDispatcherThread");           

    RemoteStateMachineManager *_this = (RemoteStateMachineManager *)_threadCommunication;

    /* set the dbus thread to high priority */
    if((NULL != _this) && (NULL != _this->mConnection))
    {
        SetName("DbusDisaptcher");
        SetPriority(50);
        pthread_setcancelstate(PTHREAD_CANCEL_ENABLE ,NULL);
        pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS ,NULL);
        /* process Dbus messages */
        while(!_this->mStopThread) {

            // wait that the sending thread has the chance to send something
            usleep(20000);

            dbus_connection_read_write_dispatch(_this->mConnection, 20);
        }
    }
    return NULL;
}

int RemoteStateMachineManager::AssembleMessage(char *message, size_t messageSize, DBusMessageIter *args)
{
    ENTRY_INTERNAL

    int ret = -1;
    char *pMessage = message;
    size_t filled = 0;

    /* reset the message */
    *pMessage = 0;

    /* test if there is a next argument */
    if(!dbus_message_iter_has_next(args)) {

        /* this is a good case: there is no message at all, which can be the case for a answer message */
        return 0;
    }

    /* iterate through the arguments and form a message */
    while(1) {

        /* test if there is a next argument */
        if(!dbus_message_iter_has_next(args)) {
            break;
        }

        /* switch to next signal argument */
        dbus_message_iter_next(args);


        /* get first type */
        int argType;
        argType = dbus_message_iter_get_arg_type(args);
        if (argType != DBUS_TYPE_STRING) {
            break;
        }

        /* get one item */
        char *item = NULL;
        dbus_message_iter_get_basic(args, &item);

        /* check the size */
        int itemLen = strlen_r(item);
        if ((filled + itemLen + 2) >= messageSize) { // messageLen + itemLen + seperator + 0
            break;
        }

        /* add it to the message */
        strcpy(pMessage, item);
        filled += itemLen;
        pMessage += itemLen;

        /* switch to next signal argument */
        dbus_message_iter_next(args);

        /* get the seperator */
        argType = dbus_message_iter_get_arg_type(args);
        if (argType != DBUS_TYPE_INT32) {
            break;
        }
        int seperator;
        dbus_message_iter_get_basic(args, &seperator);

        /* append it to the string */
        *pMessage++ = seperator;
        filled++;

        /* was it a 0: end of message */
        if (seperator == 0) {
            ret = 0;
            break;
        }
    }

    return ret;
}

DBusHandlerResult RemoteStateMachineManager::DbusSignalReceived (DBusConnection *connection, DBusMessage *pDBusMessage, void *user_data)
{
    ENTRY_INTERNAL
   trace("RemoteStateMachineManager:DbusSignalReceived:con:", (int_t)connection, " msg:", string_t(pDBusMessage));           
    
    //debug ETG_TRACE_USR3(("[%d]: called", getpid()));

    (void)connection;
    (void)user_data;

    if ( pDBusMessage)
    {
        /* for debugging */
        /* was it a signal to the SMF function? */
        if (dbus_message_is_signal(pDBusMessage, DBUS_INTERFACE, DBUS_SIGNAL))
        {

            #define NU(a) ((a)==NULL ? "null" : (a))

            const char *interface_name = dbus_message_get_interface( pDBusMessage );;
            const char *member = dbus_message_get_member( pDBusMessage );;
            const char *destination = dbus_message_get_destination( pDBusMessage );
            const char *sender = dbus_message_get_sender( pDBusMessage ) ;

            ETG_TRACE_USR3(("[%d]: interface %40s, member : %30s , destination : %20s , sender :  %20s",
                    getpid(), NU(interface_name), NU(member), NU(destination), NU(sender)));

            DBusMessageIter args;
            int argType;
            if (!dbus_message_iter_init(pDBusMessage, &args)) {
                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
            }

            /* readout the senders pid and check if the message is from my process. If yes: discard it */
            argType = dbus_message_iter_get_arg_type(&args);
            if (argType == DBUS_TYPE_INT32) {
                int sendersPid;
                dbus_message_iter_get_basic(&args, &sendersPid);
                if (sendersPid == getpid()) {
                    //debug ETG_TRACE_USR3(("[%d]: not for me", getpid()));
                    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
                }
            } else {
                return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
            }

            tAllParameters message;
            tAllParameters answer;
            message[0] = 0;
            answer[0] = 0;

            /* get the message */
            int ret;
            ret = AssembleMessage(message, sizeof(message), &args);
            if (ret) {
                ETG_TRACE_ERR(("[%d]: on AssembleMessage(message)", getpid()));
                return DBUS_HANDLER_RESULT_HANDLED;
            }

            /* get the answer (if present) */
            ret = AssembleMessage(answer, sizeof(answer), &args);
            if (ret) {
                ETG_TRACE_ERR(("[%d]: on AssembleMessage(answer)", getpid()));
                return DBUS_HANDLER_RESULT_HANDLED;
            }

            /* send message to local state machines */
            Dispatcher &dispatcher = Dispatcher::GetInstance();

            tAllParameters UTF8UnFakedMessage;
#if TARGET_BUILD
            /* to some (unkown) reason the dbus communication disconnects if a valid UTF8 special "Umlaut"
             * is transferred, so we pack them into a faked UTF8 frame
             * */
            fromUtf8Fake(OUT UTF8UnFakedMessage, IN sizeof(UTF8UnFakedMessage), IN message);
#else
            strncpy_r(OUT UTF8UnFakedMessage, IN message, IN sizeof(UTF8UnFakedMessage));
#endif

            /* is a answer defined? */
            if (strlen_r(answer)) {

                tAllParameters UTF8UnFakedAnswer;
#if TARGET_BUILD
                /* to some (unkown) reason the dbus communication disconnects if a valid UTF8 special "Umlaut"
                 * is transferred, so we pack them into a faked UTF8 frame
                 * */
                fromUtf8Fake(OUT UTF8UnFakedAnswer, IN sizeof(UTF8UnFakedAnswer), IN answer);
#else
                strncpy_r(OUT UTF8UnFakedAnswer, IN answer, IN sizeof(UTF8UnFakedAnswer));
#endif

                ETG_TRACE_USR3(("[%d]: message: %s", getpid(), UTF8UnFakedMessage));
                ETG_TRACE_USR3(("[%d]: answer: %s", getpid(), UTF8UnFakedAnswer));
                dispatcher.SendMessageAnswerLocal(UTF8UnFakedMessage, UTF8UnFakedAnswer);
            } else {

                ETG_TRACE_USR3(("[%d]: message: %s", getpid(), UTF8UnFakedMessage));
                ETG_TRACE_USR3(("[%d]: answer: NULL", getpid()));
                dispatcher.SendMessageAnswerLocal(UTF8UnFakedMessage, NULL);
            }

            return DBUS_HANDLER_RESULT_HANDLED;
        }
    }

    /* message was not consumed by this callback/thread/receiver, try another one */
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

int RemoteStateMachineManager::AppendMessage(DBusMessageIter *args, const char *message)
{
    ENTRY_INTERNAL
   trace("RemoteStateMachineManager:AppendMessage");               

    /*
     * iterate through the message and put multiple strings into the args.
     * Every 0xff or 0xfe seperates the strings. the end of the mesage is marked with an int 0x0
     */
    int ret = MP_ERR_DISP_DBUS_NO_MEM_FOR_ARG;

    char *localMessage = strdup(message);
    if (!localMessage) {
        return ret;
    }

    unsigned char *lPtr = (unsigned char *)localMessage;
    int localMessageLen = strlen_r(localMessage);
    unsigned char *lPtrEnd = (unsigned char *)(localMessage + localMessageLen);

    /* loop over characters of message */
    char *localMessageItemStart = localMessage;
    for(; lPtr <= lPtrEnd; lPtr++) {

        /* found special character? */
        if (*lPtr == 0x0 || *lPtr == 0xfe || *lPtr == 0xff) {

            /* add the item */
            int seperator = *lPtr;

            /* end the string here */
            *lPtr = 0;

            /* add it to arguments */
            if (!dbus_message_iter_append_basic(args, DBUS_TYPE_STRING , &localMessageItemStart))
            {
                goto end;
            }

            /* add the seperator to the message */
            if (!dbus_message_iter_append_basic(args, DBUS_TYPE_INT32, &seperator))
            {
                goto end;
            }

            /* if it was an 0: string ends here */
            if (seperator == 0) break;

            /* set next start of item */
            localMessageItemStart = (char *)(lPtr+1);
        }
    }

    /* no error */
    ret = MP_NO_ERROR;

end:
    free(localMessage);
    return ret;
}


int RemoteStateMachineManager::SendMessage(const char *message, const char *answer)
{
    ENTRY_INTERNAL
   trace("RemoteStateMachineManager:SendMessage");                   
    DBusMessage     *pMessage;
    DBusMessageIter args;

#if 0 // for test: switch off dbus access at all
    return MP_NO_ERROR;
#else

    if(NULL == mConnection){
        ETG_TRACE_ERR(("null dbus_connection"));
        trace("RemoteStateMachineManager:SendMessage:err:mConnection:0");                           
        return MP_ERR_DISP_OPEN_DBUS;
    }

    /* create the signal */
    pMessage = dbus_message_new_signal (DBUS_OBJECT,
                                        DBUS_INTERFACE,
                                        DBUS_SIGNAL);
    if (NULL == pMessage){
        trace("RemoteStateMachineManager:SendMessage:no mem");                                  
        return MP_ERR_DISP_DBUS_NO_MEM;
    }

    /* start appending arguments to Dbus message */
    dbus_message_iter_init_append(pMessage, &args);

    /* append my pid to the message */
    int pid = getpid();
    if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_INT32, &pid))
    {
        return MP_ERR_DISP_DBUS_NO_MEM_FOR_ARG;
    }

    tAllParameters UTF8FakedMessage;
#if TARGET_BUILD
    /* to some (unkown) reason the dbus communication disconnects if a valid UTF8 special "Umlaut"
     * is transferred, so we pack them into a faked UTF8 frame
     * */
        toUtf8Fake(OUT UTF8FakedMessage, IN sizeof(UTF8FakedMessage), IN message);
#else
        strncpy_r(OUT UTF8FakedMessage, IN message, IN sizeof(UTF8FakedMessage));
#endif

    /* append SMF message to Dbus message */
    int ret;
    ret = AppendMessage(&args, UTF8FakedMessage);
    if (ret) {
        return MP_ERR_DISP_DBUS_NO_MEM_FOR_ARG;
    }

    /* if present, append also the desired answer to the signal */
    if (answer) {

        tAllParameters UTF8FakedAnswer;
#if TARGET_BUILD
        /* to some (unkown) reason the dbus communication disconnects if a valid UTF8 special "Umlaut"
         * is transferred, so we pack them into a faked UTF8 frame
         * */
        toUtf8Fake(OUT UTF8FakedAnswer, IN sizeof(UTF8FakedAnswer), IN answer);
#else
        strncpy_r(OUT UTF8FakedAnswer, IN answer, IN sizeof(UTF8FakedAnswer));
#endif

        ret = AppendMessage(&args, UTF8FakedAnswer);
        if (ret) {
            return MP_ERR_DISP_DBUS_NO_MEM_FOR_ARG;
        }
    }

#if DBUS_SEND_ON
    /* send the message via Dbus */
    if (!dbus_connection_send(mConnection, pMessage, NULL)) {
        trace("RemoteStateMachineManager:SendMessage:dbus_connection_send");                                         
        ETG_TRACE_ERR(("dbus_connection_send"));
        return MP_ERR_DISP_DBUS_SEND;
    }
#endif

    dbus_message_unref(pMessage);

    return MP_NO_ERROR;
#endif
}

void RemoteStateMachineManager::DbusUnregister (DBusConnection *connection, void *user_data)
{
    (void)connection;
    (void)user_data;
}
