/*
 * Lock.cpp
 *
 *  Created on: 28.01.2015
 *      Author: Thömel
 */

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "Lock.h"

#include "FunctionTracer.h"
#include "Utils.h"

Lock::Lock()
{
    /* init the lock */
    pthread_mutexattr_t mta;
    MP_FATAL_ASSERT(pthread_mutexattr_init(&mta) == 0);
    MP_FATAL_ASSERT(pthread_mutexattr_settype(&mta, PTHREAD_MUTEX_RECURSIVE) == 0);
    MP_FATAL_ASSERT(pthread_mutex_init(&mutex, &mta) == 0);

    /* set some members */
    mToValue = 120; // default time out value
    isLocked = 0;  // lock not locked
}

int Lock::lock()
{
    struct timespec abs_time;
    int err;

relock:
    clock_gettime(CLOCK_REALTIME , &abs_time);
    abs_time.tv_sec += mToValue; // try it

    while(1) {
        if (!mToValue) {
            err = pthread_mutex_lock(&mutex);
        } else {
            err = pthread_mutex_timedlock(&mutex, &abs_time);
        }

        /* error? */
        switch(err) {
        case EINVAL:
            /*
             * The mutex was created with the protocol attribute having the value PTHREAD_PRIO_PROTECT
             * and the calling thread's priority is higher than the mutex's current priority ceiling.
             */
            return -1;
        case EBUSY:
            /*
             * The mutex could not be acquired because it was already locked.
             * So this call to lock() was successful
             */
            return 0;
        case EDEADLK:
            /*
             * The current thread already owns the mutex.
             * So this call to lock() was successful (in case for recursive lock enabled)
             */
            return 0;
        case EAGAIN:
            /*
             * The mutex could not be acquired because the maximum number of recursive locks for
             * mutex has been exceeded.
             * retry after a short sleep it until this maximum number allows a lock.
             */
            usleep(16L*1000L);
            goto relock;
        case ETIMEDOUT:
            /*
             * The mutex could not be locked before the specified timeout expired.
             * End the loop here to prevent endless blocking - but it is severe!
             */
            break;
        case EPERM:
            /*
             * The current thread does not own the mutex.
             * Fatal error, cannot handle with this case - software design error.
             */
            return -1;
        case EINTR:
            /*
             * this should not happen but to be secure...
             * In former times a signal can cause a return of a lock call. But the lock was not acquired.
             * Retry to lock it
             */
            continue;
        }

        /*
         * unknown or no error:
         * break the lock loop and go to the timeout value check.
         */
        break;
    }

    /*
     * in the case of timeout or unexpected error: re-check the current system time against the calculated absolute timeout value.
     * Was the system clock adjusted?: If yes: re-start the lock loop with a new absolut timeout value.
     */
    if (err != 0) {

        /* if lock tried with timeout */
        if (mToValue) {

            /* check if the timeout is plausible */
            struct timespec absTimeAfterTimeout;
            clock_gettime(CLOCK_REALTIME , &absTimeAfterTimeout);

            /* check if the calculated timeout time matches with the current real time */
            if (abs(abs_time.tv_sec - absTimeAfterTimeout.tv_sec) > 1) {

                /* timeout was not plausible, maybe system clock was adjusted */
                goto relock;
            }
        }

        /* timeout was plausible: return error */
        return -1;
    } else {
        isLocked = 1;
    }
    return 0;
}

void Lock::unlock()
{
    pthread_mutex_unlock(&mutex);
    isLocked = 0;
}

Lock::~Lock()
{
    pthread_mutex_unlock(&mutex);
    pthread_mutex_destroy(&mutex);
    isLocked = 0;
}

int Lock::setNotReantrant()
{
    pthread_mutexattr_t mta;
    pthread_mutexattr_init(&mta);
    return pthread_mutex_init(&mutex, &mta);
}

void Lock::setTimeout(const unsigned int sec)
{
    mToValue = sec;
}
