/**
 * @defgroup Framework StateMachineFramework
 * @author Matthias Th�mel
 * 
 * This is the interface of the state machine framework
 * 
 * @{
 */
#ifndef SMF_H_
#define SMF_H_

#include <stdio.h>
#include <iostream>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <vector>
#include <pthread.h>
#include "MessageQueue.h"
#include "Timer.h"
#include "Utils.h"

using namespace std;

#include "me.hpp"
using namespace me;

#define MARSHAL_SEPARATOR ((unsigned const char)0xff)
#define DOUBLE_MARSHAL_SEPARATOR ((unsigned const char)0xfe)

/**
 * The state machine framework (SMF) needs to have a generated file with the state machine 
 * creation calls and needs to be inheritated by an user state machine where the 
 * generated code will overload the virtual method Create()
 *
 * @author Matthias Thoemel
 */
class SMF : public Timer
{
public:
    /**
     * State type tag for forward declaration
     */
    struct tState_tag { };

    /**
     * Transition type tag for forward declaration
     */
    struct tTransition_tag { };

    /**
     * general function callback type
     */
    typedef int(tFn)(void *data);

    /**
     * enum for the type of transition
     */
    typedef enum {
        isNormalTransition = 0, /**< normal, internal transition */
        isDefaultTransition,    /**< internal default transition (drawn with a star) */
        isEventTransition,      /**< an internal transition that is bound to an event */
        isHistoryTransition     /**< the history transition which leads to the latest active state (H) */
    } tTransitionType;

    /**
     * enum for the type conditional or not
     */
    typedef enum {
        hasNoCondition = 0,
        hasCondition
    } tConditional;

    /**
     * enum for an inner or outer transition type
     */
    typedef enum {
        isOuterTransition = 0,
        isInnerTransition
    } tTransitionPosition;

    /**
     * enum for the type of state
     */
    typedef enum {
        isNormalState = 0,
        isHistoryState,
        isCompoundState,
        isFinalState,
        isForkState
    } tStateType;

    /**
     * Event type for public events going into the state machine framework
     */
    typedef struct tEvent {
        tEvent()
        {
            name = NULL;
            noOfStates = 0;
            noOfTrans = 0;
        }
        ~tEvent()
        {
            state.clear();
            trans.clear();
        }
        const char *name;                   /**< name of the event */
        vector<tState_tag *> state;         /**< state to which this event is bound */
        unsigned int noOfStates;            /**< number of states to which this event is bound */
        vector<tTransition_tag *> trans;    /**< transition that will be internally fired */
        unsigned int noOfTrans;             /**< number of transitions this event is bound to */
    } tEvent;

    enum {STATE_MACHINE_FINISHED = -2};		// error code if the state machine ran to final state

    enum {AnswerTimeOutValue = 3000};        // Default timer value for EventAnswer pair

    /**
     * struct type for one transition
     */
    typedef struct tTransition {
        tTransition()
        {
            _this = NULL;
            name = NULL;
            trueName = NULL;
            falseName = NULL;
            type = isNormalTransition;
            cond = hasNoCondition;
            conditionalPosition = NULL;
            pos = isOuterTransition;
            event = NULL;
            guard = NULL;
            guardName = NULL;
            condition = NULL;
            conditionName = NULL;
            action = NULL;
            actionName = NULL;
            elseAction = NULL;
            elseActionName = NULL;
            source = NULL;
            target = NULL;
            elseTarget = NULL;
            arguments = NULL;
        };

        SMF *_this;
        const char *name;           /**< name of the transition */
        const char *trueName;       /**< name of TRUE transition */
        const char *falseName;      /**< name of FALSE transition */
        tTransitionType type;       /**< type of the transition*/
        tConditional cond;          /**< is this condition a conditional one? */
        tState_tag *conditionalPosition;/**< state where the conditional bubble is placed */
        tTransitionPosition pos;    /**< is this transition inside or outside of a state / compound? */
        tEvent *event;              /**< pointer to an event which is bound to this transition */
        tFn *guard;                 /**< user function for the guard (return 0: dont do transition) */
        const char *guardName;      /**< name of user guard function for SD printing */
        tFn *condition;             /**< user function for the condition (return 0: fals, return 1: true-transition will be taken) */
        const char *conditionName;  /**< name of user condition function for SD printing */
        tFn *action;                /**< user function for the (true-)action (after guard) for this transition */
        const char *actionName;     /**< name of user action function for SD printing */
        tFn *elseAction;            /**< user function for the false-action (after guard) for this transition */
        const char *elseActionName; /**< name of user else action function for SD printing */
        tState_tag *source;         /**< source state this transition comes from */
        tState_tag *target;         /**< (true-)target state this transition leads to */
        tState_tag *elseTarget;     /**< false-target state this transition may lead to */
        char *arguments;            /**< arguments string which can be used by user callbacks. will be provided by SendEvent() */
    } tTransition;

    /**
     * struct type for concurrent context
     */
    typedef struct tStateContext {
        tStateContext()
        {
            active = 0;
            currentState = NULL;
            compoundState = NULL;
            MQInternal = NULL;
            MQInternalName = NULL;
            LockStatus = 0;
            LockMessage[0] = 0;
            AnswerTimerID = 0;
            AnswerTimeoutValue = 0;
            SpecialAnswerTimeoutValue = 0;
            ReleaseEvent = NULL;
            ReleaseEventParameter[0] = 0;

        };

        ~tStateContext()
        {
            if (MQInternalName) delete[] MQInternalName;
        };

        int active;                     /**< context is active (=1) or not */

        /* current state */
        void *currentState;             /**< pointer to current state */
        void *compoundState;            /**< pointer to last used compound state */

        /* elements for internal message/answer queue (per context) */
        MessageQueue *MQInternal;      /**< Internal Queue */
        Lock MQInternalLock;           /**< lock for filling the internal queue */
        const char *MQInternalName;    /**< Internal Queue name */
        int LockStatus;                /**< que is locked for a Message/Answer pair */
        tGeneralString LockMessage;    /**< for debugging: holds the name of the lock message during message/answer lock */

        /* message/answer timeout handling */
        Timer AnswerTimer;             /**< timeout timer */
        timer_t AnswerTimerID;         /**< answer timer id */
        int AnswerTimeoutValue;        /**< timeout value, can be changed by user */
        int SpecialAnswerTimeoutValue; /**< timeout value, can be changed by user for every release event */

        /* for answer timeout and release waiting: */
        const tEvent *ReleaseEvent;            /**< event to release current state machine from waiting state */
        tAllParameters ReleaseEventParameter;  /**< event parameter to release current state machine from waiting state */

    } tStateContext;

    /**
     * struct type for one state
     */
    typedef struct tState_tag_tag {
        tState_tag_tag()
        {
            _this = NULL;
            name = NULL;
            stateID = 0;
            type = isNormalState;
            init = NULL;
            backupInit = NULL;
            hasHistory = 0;
            history = NULL;
            compound = NULL;
            noOfTransitions = 0;
            enter = NULL;
            enterName = NULL;
            exit = NULL;
            exitName = NULL;
            alreadyPrinted = 0;
            holdsContext = 0;
            context = NULL;
            hasContext = 0;
            entered = 0;
            delayQueue = NULL;
        };
        ~tState_tag_tag()
        {
            transitions.clear();

            /* was a delay queue created, delete it */
            if (delayQueue) {
                delete delayQueue;
            }
        }

        SMF *_this;
        const char *name;                   /**< name of the state */
        int stateID;
        tStateType type;                    /**< type of the state */
        tTransition *init;                  /**< pointer to init transition if compound state */
        tTransition *backupInit;            /**< backup pointer to init transition if compound state */
        int hasHistory;                     /**< marker if this state has a history state in it */
        struct tState_tag_tag *history;     /**< pointer to history transition (set by exiting this state) */
        struct tState_tag_tag *compound;    /**< pointer to possible compound state this state is in */
        vector<tTransition *> transitions;  /**< list of transition pointers which can leave this state */
        unsigned int noOfTransitions;       /**< number of transitions in list */
        tFn *enter;                         /**< pointer to user function for the entry action */
        const char *enterName;              /**< name of user entry action function for SD printing */
        tFn *exit;                          /**< pointer to user function for the exit action */
        const char *exitName;               /**< name of user exit action function for SD printing */
        int alreadyPrinted;                 /**< flag for tracing: state already printed in SD diagram */
        int holdsContext;                   /**< holds context (=1) or not */
        tStateContext *context;             /**< context for concurrency state */
        int hasContext;                     /**< flag if this state is a context holder */
        int entered;                        /**< was this state already entered (important for context switching transitions) */
        MessageQueue *delayQueue;           /**< delay queue for delayed messages (will be created on first DelayEvent call in this state) */
        Lock mDelayQueueLock;
    } tState;

    /**
     * struct type for to send response
     */
    typedef struct tSendResponse_
    {
        tSendResponse_()
        {
            error = 0;
            timerId = 0;
        }
        ~tSendResponse_(){}
        const tEvent *res_event;
        int error;
        Timer rtimer;
        timer_t timerId;
    }tSendResponse;
    /* needed for the generated code */
    void NotImplemented(const char *name);
    tState *CreateState(const char *name, int stateID, tFn *entry, const char *entryName, tFn *exit, const char *exitName);
    tState *CreateHistoryState(const char *name);
    tState *CreateCompoundState(const char *name, int stateID, tFn *entry, const char *entryName, tFn *exit, const char *exitName);
    tState *CreateFinalState(const char *name, int stateID, tFn *entry, const char *entryName);
    tTransition *CreateTransition(const char *name, tFn *guard, const char *guardName, tFn *action, const char *actionName, tState *target, int GCon = 1);
    tTransition *CreateDefaultTransition(const char *name, tFn *guard, const char *guardName, tFn *action, const char *actionName, tState *target);
    tTransition *CreateEventTransition(tEvent *event, const char *name, tFn *guard, const char *guardName, tFn *action, const char *actionName, tState *target);
    tTransition *CreateEventConditionalTransition(tEvent *event, const char *name, tState *conditionalPosition, tFn *guard, const char *guardName, tFn *condition, const char *conditionName,
                                                        tFn *actionTrue, const char *actionTrueName, tFn *actionFalse, const char *actionFalseName,
                                                        const char *trueName, tState *targetTrue, const char *falseName, tState *targetFalse);
    tTransition *CreateHistoryTransition(const char *name, tFn *guard, const char *guardName, tFn *action, const char *actionName, tState *target);
    int AddTransition(tState *state, tTransition *trans);
    int AddState(tState *CompoundState, tState *state);
    void SetRoot(tState *state);
    void SetConcurrent(tState *state);
    void SetInitTransition(tState *state, tTransition *trans);
    tEvent *CreateEvent(const char *name);
    int DelayEvent(void *trans);
    tState *CreateForkState(const char *name);
    int RemoveMessageByName(const char *messageName);
    int RemoveMessageByName(MessageQueue *queue, const char *messageName);

    /* needed for state machine verification */
    int ExpectReset(void);
    int ExpectSetString(const char *inputString);
    int Expect(const char *inputString);
    int GetExpectErrorCount(void);

    /* public user function prototypes */
    SMF();
    virtual ~SMF();

    /**
     * This function creates the state machine. It must be overloaded by user with
     * a generated function out of the ArgoUML tool xmi to C++ converter perl script.
     *
     */
    virtual void Create() { };

    /**
     * runs the receive thread
     *
     */
    void Run();

    /**
     * Inits a user state machine. Must be initial called by user
     *
     *
     * @return =0: ok
     */
    tResult Init(const void *userContext = NULL);

    /**
     * Destroys a state machine definition (not implemented yet)
     *
     */
    void Destroy(void);

    /**
     * Starts a small state mnachine to do a synchronized event / answer pair to a foriegn state machine
     *
     * @param[in] sendMessage message with parameters to send to the message/answer pair in target state machine
     * @param[in] parameter parameter string to send message
     * @param[in] userContext user context pointer, retrieval by #GetUserContext()
     * @return = 0: ok
     */
    tResult DoEventAnswer(const char *sendMessage, const char *parameter, const void *userContext = NULL,
            const int answerTimeoutMS =AnswerTimeOutValue,const char *answerTimeoutParams = NULL);

    /**
     * Weave m1::nif_t type serialisation into SMF.
     *
     * @param[in] sendMessage message with parameters to send to the message/answer pair in target state machine
     * @param[in] m1::msg_t supporting automatic serialisation.
     * @param[in] userContext user context pointer, retrieval by #GetUserContext()
     * @return = 0: ok
     */
    tResult DoEventAnswer(const char *sendMessage, m1::msg_t const& msg, const void *userContext = NULL,
            const int answerTimeoutMS =AnswerTimeOutValue,const char *answerTimeoutParams = NULL);
            
    /**
     * Sends an event by pointer of event message created by CreateEvent
     *
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendEvent(const tEvent *event, const char *parameters);

    /**
	 * PSARCC-4630
     * Sends an event with priority "ExternalResultingInternal" by pointer of event message created by CreateEvent
     * Lesser priority than Internal
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */	
    tResult SendEventToResultInternal(const tEvent *event, const char *parameters);

    /**
     * Sends an event with priority "urgent" by pointer of event message created by CreateEvent
     *
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendUrgentEvent(const tEvent *event, const char *parameters);

    /**
     * Flush the external queue and sends an event with priority "urgent"
     *
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendForceEvent(const tEvent *event, const char *parameters);

    /**
     * Sends an event which is defined by its name instead of its pointer with priority "external"
     *
     * @param[in] eventName name of the event which shall be forwarded to the state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendEventByName(const char *eventName, const char *parameters);

    /**
     * Sends an event which is defined by its name instead of its pointer with priority "internal"
     *
     * @param[in] eventName name of the event which shall be forwarded to the state machine
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendInternalEventByName(const char *eventName, const char *parameters);

    /**
     * Sends an event by pointer of event message created by CreateEvent
     *
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] priority priority of the event -> Priority_Internal ,Priority_AnwerTimeOut or Priority_External
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendEvent(const tEvent *event, const int priority, const char *parameters);

    /**
     * Sends an event by pointer of event message created by CreateEvent, and receive answer event
     * when action is finished.
     *
     * @param[in] event pointer to event which shall be forwarded to state machine
     * @param[in] answer name of the message thatwill be sent as an answer to this request
     * @param[in] parameters parameters to format the parameters to the event (NOT to the answer)
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendEventAnswer(const tEvent *event, const char *answer, const char *parameters);

    /**
     * Sends an event by name, and receive answer event
     * when action is finished.
     *
     * @param[in] eventName name which shall be forwarded to state machine
     * @param[in] eventAnswerName answer name which shall be answered after the first argument event completion
     * @param[in] parameters parameters string to format the appended integer values to a event
     *                   argument which is sent to the guard, action and condition user callback
     * @return = 0: ok
     *         = STATE_MACHINE_FINISHED: final state reached
     */
    tResult SendEventAnswerByName(const char *eventName, const char *eventAnswerName ,const char *parameters);

    /**
     * Sends answer event by pointer of event message created by CreateEvent
     *
     * @param[in] argument- which shall be passed with the answer to the receiving state machine
     * @return = 0: ok
     *         = 1: if the answer is not expected
     */
    tResult SendAnswer(const char *argument);

    /**
     * Sends error event instead of answer by pointer of event message created by CreateEvent
     *
     * @param[in] argument- which shall be passed with the answer to the receiving state machine
     * @return = 0: ok
     *         = 1: if the answer is not expected
     */
    tResult SendActionError(const char *argument);

    /**
     * StateMachin Main loop
     *
     * @return = 0: ok
     */
    tResult StateMachine_Main();

    /**
     * Set the name for StateMachine and the name will be used for mque defination
     *
     * @param[in] name the new name of this state machine
     * @return = 0: ok
     */
    tResult SetSMName(const char *name);

    /**
     * Function to get the State Machine name without SM counter
     *
     * @return : pointer to state machine name
     */
    const char* GetSMName(void);

    /**
     * Function to get the full qualified state machine name
     *
     * @return : pointer to state machine name
     */
    const char* GetSMNameFull(void);

    /**
     * To get current state ID
     *
     * @return - stateID
     */
    tResult GetState(void);

    /**
     * To get current state Name
     * @param[inout] stateName gets filled with current state name
     * @return nothing
     */
    void GetCurrentState(char *stateName, size_t size);
    const char *GetCurrentState();

    /**
     * method checks if the state machine is in a specific state
     *
     * @param[in] stateEnum state enumeration value casted to integer
     * @return =0: not in specific state
     * 		   =1: is in the specific state
     */
    tResult IsInState(tInteger stateEnum);

    /**
     * To set the time out value for the SendEventAnswer pair
     *
     * @param[in] TimeOut - Time out value in millisecond
     * @return = 0: ok
     */
    tResult SetAnswerTimeout(const int TimeOut);

    /**
     * Sets the event which will release the state machine from the current waiting state
     *
     * @param[in] event - Pointer to event definition
     * @param[in] parameter - Pointer to parameter (optional)
     * @param[in] specialTimeoutValue - set special timeout for current event only (optional)
     * @return = 0: ok
     */
    tResult RegisterReleaseEvent(const tEvent *event, const char * parameter = NULL, const int specialTimeoutValue = -1);

    /**
     * Sets the event which will release the state machine from the current waiting state
     *
     * @param[in] event - name of release event
     * @param[in] parameter - Pointer to parameter (optional)
     * @param[in] specialTimeoutValue - set special timeout for current event only (optional)
     * @return = 0: ok
     */
    tResult RegisterReleaseEvent(const char *event, const char * parameter = NULL, const int specialTimeoutValue = -1);
    /**
     * Sets the event which will send a update with no response of Answer event
     *
     * @param[in] event - Pointer to event definition
     * @param[in] parameter - Pointer to parameter (optional)
     * @param[in] specialTimeoutValue - set special timeout for current event only (optional)
     * @return = 0: ok
     */
    tResult RegisterNoResponseEvent(const tEvent *event, const int error,  const int specialTimeoutValue = -1);

    /**
     * Sets the event which will send a update with no response of Answer event
     *
     * @param[in] event - name of release event
     * @param[in] parameter - Pointer to parameter (optional)
     * @param[in] specialTimeoutValue - set special timeout for current event only (optional)
     * @return = 0: ok
     */
    tResult RegisterNoResponseEvent(const char *event,const int error = 0, const int specialTimeoutValue = -1);

    /**
     * Clears the release event at the framework which was registered by RegisterReleaseEvent()
     *
     * @return = 0: ok
     */
    tResult ClearReleaseEvent();

    /**
     * Releases the state machine from the current waiting state by firing the registered release
     * event (done by  RegisterReleaseEvent)
     *
     * @return = 0: ok
     */
    tResult ReleaseWaiting();

    /**
     * sets the count for a internal send message retry loop
     *
     * @param[in] retryCount - new retry count (0: disable retries)
     * @return = 0: ok
     */
    tResult SetRetryCount(int retryCount);

    /**
     * sets the waiting time for the internal send message retry loop
     *
     * @param[in] retryWaitTimeUS - new retry loop sleep time in microseconds
     * @return = 0: ok
     */
    tResult SetRetryWaitTime(unsigned long retryWaitTimeUS);

    /**
     * Resets a state machine to begin operation again
     *
     * @return = 0: ok
     */
    tResult Reset(const void *userContext = NULL);

    /**
     * The Error Handler is called on a non zero return of a user callback method. If this handler returns
     * zero a internal retry loop calls the user callback method again. If this error handler returns non zero,
     * the retry loop will be finished.
     *
     * The default implementation of the handler throws an assert and returns -1;
     *
     * @param[in] error - error number that was returned by user call back function
     * @param[in] trans - Pointer to the current transition
     * @param[in] fnName - name of the user callback function that failed
     * @return = 0: retry the user callback for the specific transition
     *         != 0: stop the retry loop
     */
    virtual tResult ErrorHandler(const tResult error, const tTransition *trans, const char *fnName);
    /**
     * Marshals values to transfer them via the SMF to target state machine
     *
     * @param[in] serialized - string buffer that will be written
     * @param[in] size - size of the string buffer
     * @param[in] format - description of the following parameters:
     * 					'i': integer
     * 					't': text
     * @param[in] ... - values that must be marshaled
     * @return = pointer to serialized
     */
    static char *Marshal(char *serialized, const size_t size, const char *format, ...);
    static char *Marshal(char *serialized, const size_t size, unsigned const char separator, const char *format, ...);
    static char *MarshalToUtf8(char *serialized, const size_t size, unsigned const char separator, const char *format, ...);

    /**
     * Un-Marshals values after transfer via the SMF to a target state machine
     *
     * @param[in] serialized - string buffer that contains the values after message passing
     * @param[in] format - description of the following parameters:
     * 					'i': integer
     * 					't': text
     * 					'a': write the complete contents of the serialized into one character buffer
     * @param[out] ... - values that will be written according to format description
     * @return = 0: ok
     */
    static tResult UnMarshal(const char *serialized, const char *format, ...);
    static tResult UnMarshal(const char *serialized, const unsigned char separator, const char *format, ...);
    static tResult UnMarshalFromUtf8(const char *serialized, const unsigned char separator, const char *format, ...);

    SMF *GetInstance()
    {
    	return this;
    }

    /**
     * Gets the target state name for the current transition the user function is in
     *
     * param[out] state name / pointer to target state of current transition.
     * @return != 0: error, = 0: OK
     */
    tResult GetTargetState(OUT tState &state); //Roadmap 13010

    /**
     * Gets the user context pointer set by Init() or Create()
     *
     * @return user context pointer
     */
    const void *GetUserContext() {return mUserContext;};

    /**
     * Set / Get the transient attribute
     */
    const bool IsTransient() {return mTransient;};
    //void SetTransient(const bool set) {mTransient = set;};

private:
	enum { printStringSize = 1024 };		// size of all string print buffers
	enum {									// message priority
            Priority_Urgent = 10,
			Priority_Internal = 20,
            Priority_External_Resulting_Internal = 25,  //Priority for NEW_LIST
			Priority_AnwerTimeOut = 30,
			Priority_Delayed = 39,
			Priority_External = 40
	};
	typedef enum {
		MessageAnswerUnlock = 0,
		MessageAnswerLock
	} mLock;

	/* internal function prototypes */
	static char *Marshal(char *serialized, const size_t size, const unsigned char separator, const char *format, va_list vl);
	static tResult UnMarshal(const char *serialized, const unsigned char separator, const char *format, va_list vl);
	int _Reset();
	void DebugPrint(const char *format, ...);
	tState *_CreateState(const char *name, int stateID, tFn *entry, const char *entryName, tFn *exit, const char *exitName);
	int DoTransition(tTransition *trans);
	int IsPartOf(tState *target, tState *compound);
	int CreateComponent();
	tEvent *CreateEventInternal(const char *name, int GCon = 1);
	int DecreateComponent();
	int PrintEnter(tState *state);
	int PrintExit(tState *state);
	int PrintGuard(tTransition *trans);
	int PrintAction(tTransition *trans);
	int PrintConditionAction(tTransition *trans);
	int SDPrintState(tState *state);
	int SDPrintEvent(tState *currentState, tEvent *event, char *arguments);
	int SDPrintNotConsumedEvent(tEvent *event);
	int SDPrintStateTransition(tState *currentState, tState *compoundState, tState *lastCurrentState, tTransition *trans);
	int SDPrintEnterAction(tState *currentState);
	int SDPrintExitAction(tState *currentState);
	int SDPrintAction(tTransition *trans, int falseAction = 0);
	int SDStartTransitionSection(void);
	int SDPrintCondition(tState *currentState, tTransition *trans);
	int SDPrintGuard(tState *currentState, tTransition *trans);
	void SDPrint(const char *buffer);
	int TransferDelayedEvents(tState *currentState);
    void* GetMQmessage(MessageQueue* mq, const char *nameOfQueue, int TimeOut);
    void* ReadMQmessage(MessageQueue* mq, const char *nameOfQueue);
    int CheckForInternalMessage();
    int CheckForExternalMessage();
    void EventAnswerTimeOut();
    static bool TimerCallBack(timer_t timerID , void* instance ,const void *userData);
    int NextEventProcess(char *eventName, char *arguments, tStateContext *processingContext = NULL);
    tStateContext *GetStateContext(tState *state);
    void EventNoResponseMsg(char *smName, int error);
	static bool ResponseTimerCallBack(timer_t timerID , void* instance ,const void *userData);
	tResult ClearResponseEvent();
    /* Garbage Collector */
    typedef enum {
        tEventGCType,
        tStateGCType,
        tTransitionGCType,
        tStateContextGCType
    } tClassType;
    void registerGC(void *ptr, tClassType type, int GCon);
    typedef struct {
        tClassType type;
        void *ptr;
    } tPointerGC;
    tPointerGC *pointersGC;
    size_t numberOfPointersGC;
    void freeGC();
    Lock lockGC;

    /* internal members values */
    vector<tEvent *> mKnownEvents;
    tState *mRootState;
    tTransition *mCurrentTrans;
    vector<tState *> mInitedStates;
	char *mSDActionName;           // for SD printing
	char *mSDTransactionName;      // for SD printing
	char mSDEnterAction[printStringSize]; // for SD printing
	tEvent *mNULL_EVENT;
    int mIsCreated;                 // flag if this state machine is already created
    MessageQueue *mMQExternal;				// External Queue
	const char *mMQExternalName;	// External Queue name
	const char *mSMName;			// StateMachine name without state machine counter
	const char *mSMFullName;		// StateMachine name with state machine counter
	char mSMNamePostfix[20];		// postfix to the state machine name (is the class name)
	vector<MessageQueue *> mAllDelayQueues; // vector which holds all created delay queues

	/* context handling */
	vector<tStateContext *> mContextDirectory; // array of all contexts used in state machine
	tStateContext *mCurrentContext; // needed for context specific calls from SM implementation to SMF (like SendAnswer), set by DoTransition

	/* for send message retry function: */
	int mRetryCount;
	unsigned long mUSSleepTime;

	/* user context */
	const void *mUserContext;

	/* for expect: */
	vector<const char *> mExpectedString;
	int mCurrentExpectString;
	int mExpectErrorCount;

	/* attibutes */
	bool mTransient; //true for RequestResponseSM, default false
	tSendResponse mResponseTimer;             /**< Response timeout timer */
};

#endif //SMF_H_

/** @} */
