/**
 * @addtogroup  Ipc
 *
 * @author      Dinesh D
 *
 * Provide IPC interface to communicate with the components which are in a
 * different processes (e.g. PlayerEngine which can be communicated via dbus IPC ).*
 *
 * @{
 */
#ifndef IPCCLIENT_H_
#define IPCCLIENT_H_

#include <iostream>
#include "FunctionTracer.h"
#include <string>
#include <dbus/dbus.h>
#include <map>
#include <list>
#include <sstream>

#include "IpcTypes.h"
#include "IpcBase.h"

using namespace std;
/**
 * IPC Interface class
 */
struct DbusParam
{
    string value;
    list<DbusParam> ParamList;
    int type;
};
typedef list<DbusParam> DbusParamList;

class IpcClient : public IpcBase
{

public:

    /**
     * Constructor
     *
     * @return void
     */
    IpcClient();

    /**
     * Destructor
     *
     * @return void
     */
    virtual ~IpcClient();

    /**
     * Sets the dbus path (dynamic) to be used for any new dbus message send based on its interface.
     *
     *@param[in] interface interface of the remote service 
     *@param[in] path dbus path of the remote service 
     */
    void OverridePath(const char* interface,const char* path);

    template <class ParamType1>
    int MethodCallNoOverride(const IpcMessageInfo &messageInfo,ParamType1 param1)
    {
        m_Mutex.lock();
        InitMethodCallNoOverride(messageInfo);
        AppendMessage(param1);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }

    template <class ParamType1,class ParamType2>
       int MethodCallNoOverride(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2)
       {
           m_Mutex.lock();
           InitMethodCallNoOverride(messageInfo);
           AppendMessage(param1);
           AppendMessage(param2);
           int serial = SendMethodCall();
           m_Mutex.unlock();
           return serial;
       }
    /**
     * Calls dbus method with no parameter
     *
     * @param[in] messageInfo type contains the method information like dbus path , interface and method name itself
     *
     */
    int MethodCall(const IpcMessageInfo &messageInfo)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }

    /**
     * Calls dbus method with single parameter
     *
     * @param[in] messageInfo   type contains the method information like dbus path , interface and method name itself
     * @param[in] param1        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     *
     */
    template <class ParamType1>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }

    /**
     * Calls dbus method with two parameters
     *
     * @param[in] messageInfo   type contains the method information like dbus path , interface and method name itself
     * @param[in] param1        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param2        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     *
     */
    template <class ParamType1,class ParamType2>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        AppendMessage(param2);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }

    /**
     * Calls dbus method with three parameters
     *
     * @param[in] messageInfo   type contains the method information like dbus path , interface and method name itself
     * @param[in] param1        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param2        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param3        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     *
     */
    template <class ParamType1,class ParamType2,class ParamType3>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2,ParamType3 param3)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        AppendMessage(param2);
        AppendMessage(param3);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }

    /**
     * Calls dbus method with eight parameters
     *
     * @param[in] messageInfo   type contains the method information like dbus path , interface and method name itself
     * @param[in] param1        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param2        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param3        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param4        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param5        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param6        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param7        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     * @param[in] param8        parameter to the method (can be of any basic dbus data type like int , bool , double , string)
     *
     */
    template <class ParamType1,class ParamType2,class ParamType3,class ParamType4,class ParamType5,class ParamType6,class ParamType7,class ParamType8>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2,ParamType3 param3,
        ParamType4 param4,ParamType5 param5,ParamType6 param6,ParamType7 param7,ParamType8 param8)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        AppendMessage(param2);
        AppendMessage(param3);
        AppendMessage(param4);
        AppendMessage(param5);
        AppendMessage(param6);
        AppendMessage(param7);
        AppendMessage(param8);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }


    template <class ParamType1,class ParamType2,class ParamType3,class ParamType4,class ParamType5>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2,ParamType3 param3,ParamType4 param4,ParamType5 param5)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        AppendMessage(param2);
        AppendMessage(param3);
        AppendMessage(param4);
        AppendMessage(param5);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
      }

    template <class ParamType1,class ParamType2,class ParamType3,class ParamType4>
    int MethodCall(const IpcMessageInfo &messageInfo,ParamType1 param1,ParamType2 param2,ParamType3 param3,ParamType4 param4)
    {
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(param1);
        AppendMessage(param2);
        AppendMessage(param3);
        AppendMessage(param4);
        int serial = SendMethodCall();
        m_Mutex.unlock();
        return serial;
    }
    void MethodCallWaitForReply(const IpcMessageInfo &messageInfo, DbusParamList &dbusParamList,bool isSystemBus,int reply_timeout,bool& isError,string& params,bool isByteArrayNeeded = false)
    {
        cout<<"MethodCallWaitForReply"<<endl;
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        for (DbusParamList::iterator ii = dbusParamList.begin(); ii != dbusParamList.end(); ++ii)
        {
            append_arg(&mMessageIter,(*ii).type,(*ii).value.c_str(),(*ii).ParamList);
        }
        cout<<"exit MethodCallWaitForReply"<<endl;
        SendMethodCallWaitForReply(isSystemBus,reply_timeout,isError,params,isByteArrayNeeded);
        m_Mutex.unlock();
    }
    void MethodCallWaitForReply(const IpcMessageInfo &messageInfo, map<string, unsigned int> paramlist ,bool isSystemBus,int reply_timeout,bool& isError,string& params,bool isByteArrayNeeded = false)
    {
        cout<<"MethodCallWaitForReply"<<endl;
        m_Mutex.lock();
        InitMethodCall(messageInfo);
        AppendMessage(paramlist);
        cout<<"exit MethodCallWaitForReply"<<endl;
        SendMethodCallWaitForReply(isSystemBus,reply_timeout,isError,params,isByteArrayNeeded);
        m_Mutex.unlock();
    }

    void MethodCallWaitForReplyNoOverride(const IpcMessageInfo &messageInfo, DbusParamList &dbusParamList,bool isSystemBus,int reply_timeout,bool& isError,string& params,bool isByteArrayNeeded = false)
    {
        cout<<"MethodCallWaitForReplyNoOverride"<<endl;
        m_Mutex.lock();
        InitMethodCallNoOverride(messageInfo);
        for (DbusParamList::iterator ii = dbusParamList.begin(); ii != dbusParamList.end(); ++ii)
        {
            append_arg(&mMessageIter,(*ii).type,(*ii).value.c_str(),(*ii).ParamList);
        }
        cout<<"exit MethodCallWaitForReplyNoOverride"<<endl;
        SendMethodCallWaitForReply(isSystemBus,reply_timeout,isError,params,isByteArrayNeeded);
        m_Mutex.unlock();
    }
    void SendMethodCallWaitForReply(bool isSystemBus,int reply_timeout,bool& isError,string& params,bool isByteArrayNeeded);

private:
    DBusMessage    *mMessage;
    DBusMessageIter mMessageIter;
    map<string, string> mPath;// pair of interface : path to be used while sending message to remote dbus service
    Lock mPathLock;

    int InitMethodCall( const IpcMessageInfo &messageInfo);
    int InitMethodCallNoOverride(const IpcMessageInfo &messageInfo);
    int AppendMessage(void *param) const;
    int AppendMessage(bool param);
    int AppendMessage(int param);
    int AppendMessage(unsigned int param);
    int AppendMessage(const char *param);            // character string
    int AppendMessage(const unsigned char  array[]); // array of byte 'ay'
    int AppendMessage(const dbus_uint32_t array[]);  // array of unsigned 32 bit int 'au'
    int AppendMessage(const unsigned char param);
    int AppendMessage(map<string,unsigned int>);
    int AppendMessage(string);
    int AppendMessage(list<string>&);
    int AppendMessage(signed short param);
    int SendMethodCall(void);
    int append_variant(DBusMessageIter *iter, char type, void *val);
    bool append_arg(DBusMessageIter *iter, int type, const char *value , DbusParamList ParamList);
    int dict_append_entry(DBusMessageIter *dict, const char *key, int type, void *val);

};

#endif /*IPCCLIENT_H_*/

/** @} */
