/*
 * FC_MediaPlayer_rootdaemon_client.cpp
 *
 *  This file implements IClientSharedlib.h (the client side of the root daemon)
 *  it should be compiled as shared library and copied to the sharedlibs loading area to be loadable from the root daemon server side
 *
 *  Created on: Jan 16, 2018
 *      Author: Ganesh Rengasamy
 */

#include <stdio.h> //fopen
#include <stdlib.h> //system()
#include <string.h>
#include <stddef.h>
#include <signal.h>
#include <sys/mount.h>
#include <syslog.h>
#include <errno.h>
#ifdef TARGET_BUILD
#include <libkmod.h>
#endif
#include <glob.h>
#include <sys/types.h>
#include <sys/stat.h> //stat()
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h> // for fork(), execv()
#include <sys/wait.h>  // for waitpid()

#include <IClientSharedlib.h>
#include <FC_MediaPlayer_rootdaemon_client.h>

#include "iPodControlCommon.h"
/*lint -save -e1773 */

#define STR_GADGET                              "gadget"
#define STR_HOST                                "host"

#define IAP2_OK                               0
#define IAP2_CTL_ERROR                       -1

#ifndef TRUE
# define TRUE                                  1
#endif
#define GMP_COMMANDS_DELIM "\377"
std::string validPathNames[] = {"dev_sd"};

///////////////////////////////HELPER FUNCTIONS///////////////////////////////

////////////////// String Helper functions - From Framework //////////////////
// str*_f added as framework calls str*_r
char *strncat_f(char *dest, const char *src, size_t sizeOfDest)
{
    /* sanity: empty source string */
    if (src == NULL || *src == 0) { // CID 13997 (#1 of 2): Dereference after null check (FORWARD_NULL)
        return dest;
    }

    int iCopy = 0;
    char *_src = (char *)src;
    char *ret = dest;
    sizeOfDest--; // les one for the trailing zero
    while(*_src && sizeOfDest) {
        if (iCopy == 0 && *dest == 0) iCopy = 1;    // end of dest string found
        if (iCopy == 1) *dest = *_src++;            // copy src to dest
        dest++;
        sizeOfDest--;
    }
    *dest = 0;
    return ret;
}

int strlen_f(const char *src)
{
    if(src == NULL) { //terminate src on NULL
        return 0;
    }
    return strlen(src);
}

char *strncpy_f(char *dest, const char *src, size_t sizeOfDest)
{
    if(src == NULL) { //terminate dest on NULL
        dest[0] = 0;
        return dest;
    }
    dest[sizeOfDest-1] = 0; //implicit null-character termination if source is longer than sizeOfDest
    return strncpy(dest, src, sizeOfDest-1);
}
/////////////////////////////////////////////////////////////////////////

//////////////////////////IAP related functions//////////////////////////
int iAP2Common_Write(const char* path, const char* subPath, const char* value, int checkBeforeWrite)
{
    int status = 0;
    char valuePath[256];
    int file = 0;

    int ret = snprintf(valuePath, 256, "%s/%s", path, subPath);
    if (ret >= 0 && ret < 256)
    {
        file = open(valuePath, checkBeforeWrite ? O_RDWR : O_WRONLY);
        if (file >= 0)
        {
            const size_t len = strlen_f(value);

            /* check before writing the same value */
            if (checkBeforeWrite == TRUE)
            {
                char buffer[len + 1]; /* to capture longer entries */
                ret = read(file, buffer, sizeof(buffer));
                if (ret == (int)len)
                {
                    if (0 == strncmp(buffer, value, len)) /* without trailing \0 */
                    {
                        /* no need to write */
                        status = TRUE;
                    }
                    // ETG_TRACE_USR3(("iAP2Common_Write():  read: %s", buffer));
                }
                else if (ret < 0)
                {
                    syslog(LOG_ERR, "iAP2Common_Write():  read: %d %s", errno, strerror(errno));
                }
            }

            /* write or skip if already the same value */
            if (status == 0)
            {
                ret = write(file, value, len);
                if (ret == (int)len)
                {
                    /* successful write */
                    status = 1;
                }
                else if (ret < 0)
                {
                    syslog(LOG_ERR, "iAP2Common_Write():  write: %d %s", errno, strerror(errno));
                }
            }

            close(file);
        }
        else
        {
            syslog(LOG_ERR, "iAP2Common_Write():  open: %d %s", errno, strerror(errno));
        }
    }

    return status;
}

int iap2Common_WriteValue(const char* path, const char* value)
{
    int status = 0;
    int file = 0;
    int ret = 0;

    file = open(path, O_WRONLY);
    if (file >= 0)
    {
        const size_t len = strlen_f(value) + 1;

        /* write or skip if already the same value */
        ret = write(file, value, len);
        if (ret == (int)len)
        {
            /* successful write */
            status = 1;
        }
        else if (ret < 0)
        {
            syslog(LOG_ERR, "WriteValue():  write: %d %s", errno, strerror(errno));
        }

        close(file);
    }
    else
    {
        syslog(LOG_ERR, "WriteValue():  open: %d %s", errno, strerror(errno));
    }

    return status;
}

#ifdef IPODCONTROL_IAP2_PF_CONFIGFS_NON_ROOT

int iap2InitGadgetConfiguration(iAP2_usbg_config_t* usb_gadget_configuration,
        const bool carPlayEnabled,
        const bool nativeTransportEnabled,
        const bool carLifeEnabled,
        const char * deviceID,
        const char * name,
        const char * modelIdentifier,
        const char * manufacturer,
        const char * serialNumber,
        const char * vendorID,
        const char * productID,
        const char * bcdDevice,
        const char * mountLocation,
        const char * udcDeviceName,
        const char * initEndPoint,
        const char * appName)
{
    int ret = IAP2_OK;
    syslog(LOG_INFO, "Enter iap2InitGadgetConfiguration ");
    char iAP2GadgetName[STRING_MAX]         = {0};
    strncpy_f(iAP2GadgetName, "iAP_Interface_", sizeof(iAP2GadgetName));
    strncat_f(iAP2GadgetName, deviceID, sizeof(iAP2GadgetName));
    usb_gadget_configuration->iAP2GadgetName                = (U8*)strndup((const char*)iAP2GadgetName, (STRING_MAX - 1) );

    char iAP2FFS_InstanceName[STRING_MAX]   = {0};
    strncpy_f(iAP2FFS_InstanceName, "ffs_", sizeof(iAP2FFS_InstanceName));
    strncat_f(iAP2FFS_InstanceName, deviceID, sizeof(iAP2FFS_InstanceName));
    usb_gadget_configuration->iAP2FFS_InstanceName = (U8*)strndup((const char*)iAP2FFS_InstanceName, STRING_MAX - 1);

    if(name) usb_gadget_configuration->iAP2AccessoryName = (U8*)strndup(name, (STRING_MAX-1));
    if(modelIdentifier) usb_gadget_configuration->iAP2AccessoryModelIdentifier = (U8*)strndup(modelIdentifier, (STRING_MAX-1));
    if(manufacturer) usb_gadget_configuration->iAP2AccessoryManufacturer = (U8*)strndup(manufacturer, (STRING_MAX-1));
    if(serialNumber) usb_gadget_configuration->iAP2AccessorySerialNumber = (U8*)strndup(serialNumber, (STRING_MAX-1));
    if(vendorID) usb_gadget_configuration->iAP2AccessoryVendorId = (U8*)strndup(vendorID, (STRING_MAX-1));
    if(productID) usb_gadget_configuration->iAP2AccessoryProductId = (U8*)strndup(productID, (STRING_MAX-1));
    if(bcdDevice) usb_gadget_configuration->iAP2AccessoryBcdDevice = (U8*)strndup(bcdDevice, (STRING_MAX-1));
    if(mountLocation) usb_gadget_configuration->iAP2ConfigFS_MountLocation = (U8*)strndup(mountLocation, (STRING_MAX-1));
    if(udcDeviceName) usb_gadget_configuration->iAP2UdcDeviceName = (U8*)strndup(udcDeviceName, (STRING_MAX-1));

    if(carLifeEnabled)  //Fix for NCG3D-116507
    {
        usb_gadget_configuration->CarPlayEnabled = false;
        usb_gadget_configuration->iAP2_UAC_NCM_gadgets_disable = true;
    }
    else if(nativeTransportEnabled)  //Fix for NCG3D-150785
    {
        usb_gadget_configuration->CarPlayEnabled = false;
        usb_gadget_configuration->iAP2_UAC_NCM_gadgets_disable = false;
    }
    else
    {
        syslog(LOG_INFO, "Device in Carplay mode, hence NCM will be configured");
    }

    if(carPlayEnabled) {
        //CARPLAY ON
        usb_gadget_configuration->CarPlayEnabled = TRUE;

        char iAP2NCM_InstanceName[STRING_MAX]   = {0};
        strncpy_f(iAP2NCM_InstanceName, "USB_", sizeof(iAP2NCM_InstanceName));
        strncat_f(iAP2NCM_InstanceName, deviceID, sizeof(iAP2NCM_InstanceName));
        usb_gadget_configuration->iAP2NCM_InstanceName = (U8*)strndup((const char*)iAP2NCM_InstanceName, (STRING_MAX - 1) );

    } else {
        //CARPLAY OFF
        char iAP2UAC2_InstanceName[STRING_MAX]   = {0};
        strncpy_f(iAP2UAC2_InstanceName, "uac2_", sizeof(iAP2UAC2_InstanceName));
        strncat_f(iAP2UAC2_InstanceName, deviceID, sizeof(iAP2UAC2_InstanceName));
        usb_gadget_configuration->iAP2UAC2_InstanceName = (U8*)strndup((const char*)iAP2UAC2_InstanceName, (STRING_MAX - 1) );

#if defined(TARGET_BUILD_GEN3) && !defined(GEN3X86)
        usb_gadget_configuration->iAP2_UAC2_Attrs = new (usbg_f_uac2_attrs);
        if(usb_gadget_configuration->iAP2_UAC2_Attrs) {
            memset(usb_gadget_configuration->iAP2_UAC2_Attrs, 0, sizeof(usbg_f_uac2_attrs));
            usb_gadget_configuration->iAP2_UAC2_Attrs->c_chmask     = 3;
            usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate_def  = 44100;
            usb_gadget_configuration->iAP2_UAC2_Attrs->c_ssize      = 2;
            usb_gadget_configuration->iAP2_UAC2_Attrs->delay_tout   = 80;
#ifdef VARIANT_S_FTR_ENABLE_CHY
            usb_gadget_configuration->iAP2_UAC2_Attrs->p_chmask     = 3;
#else
            usb_gadget_configuration->iAP2_UAC2_Attrs->p_chmask     = 0;
#endif
            usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate_def  = 44100;
            usb_gadget_configuration->iAP2_UAC2_Attrs->p_ssize      = 2;
            usb_gadget_configuration->iAP2_UAC2_Attrs->c_srate      = "44100,48000";
            usb_gadget_configuration->iAP2_UAC2_Attrs->p_srate      = "44100,48000";
        }
#endif
    }

    usb_gadget_configuration->iAP2FFSConfig.initEndPoint = (U8*)strndup(initEndPoint, (STRING_MAX - 1) );
    if(nativeTransportEnabled) {
        usb_gadget_configuration->iAP2FFSConfig.nativeTransport = TRUE;

        int numApps = 1; //restriction to single native transport app support.
        usb_gadget_configuration->iAP2FFSConfig.iOSAppNames = new U8*[numApps];
        usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier = new U8[numApps];
        if(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames
                && usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier) {

            usb_gadget_configuration->iAP2FFSConfig.nativeTransport = TRUE;
            usb_gadget_configuration->iAP2FFSConfig.iOSAppCnt = numApps;
            for(int i = 0; i < numApps; i++) {
                if(appName) usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[i] = (U8*)strndup((const char*)appName, (STRING_MAX - 1) );
                usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier[i] = i+1; //count from zero
            }
        }
    }

    return ret;
}

void iap2DeInitGadgetConfiguration(iAP2_usbg_config_t* usb_gadget_configuration)
{
    // those values are created with strdup and therefore need to be freed
    free(usb_gadget_configuration->iAP2GadgetName);
    free(usb_gadget_configuration->iAP2FFS_InstanceName);
    free(usb_gadget_configuration->iAP2AccessoryName);
    free(usb_gadget_configuration->iAP2AccessoryModelIdentifier);
    free(usb_gadget_configuration->iAP2AccessoryManufacturer);
    free(usb_gadget_configuration->iAP2AccessorySerialNumber);
    free(usb_gadget_configuration->iAP2AccessoryVendorId);
    free(usb_gadget_configuration->iAP2AccessoryProductId);
    free(usb_gadget_configuration->iAP2AccessoryBcdDevice);
    free(usb_gadget_configuration->iAP2ConfigFS_MountLocation);
    free(usb_gadget_configuration->iAP2UdcDeviceName);
    free(usb_gadget_configuration->iAP2UAC2_InstanceName);
#if defined(TARGET_BUILD_GEN3) && !defined(GEN3X86)
    // created with new
    delete usb_gadget_configuration->iAP2_UAC2_Attrs;
#endif
    // created with strndup
    free(usb_gadget_configuration->iAP2NCM_InstanceName);
    free(usb_gadget_configuration->iAP2FFSConfig.initEndPoint);
    // created with new[]
    delete [] usb_gadget_configuration->iAP2FFSConfig.iOSAppIdentifier;
    if(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames) {
        for(unsigned int i = 0; i < usb_gadget_configuration->iAP2FFSConfig.iOSAppCnt; i++) {
            // strdup needs to be freed - not deleted
            free(usb_gadget_configuration->iAP2FFSConfig.iOSAppNames[i]);
        }
        // created with new[]
        delete [] usb_gadget_configuration->iAP2FFSConfig.iOSAppNames;
    }
}

#endif
///////////////////////////////////////////////////////////////////////////////////

uid_t GetGidByName(const char *groupName)
{
    if(groupName) {
        struct group *grp = getgrnam(groupName); /* don't free, see getgrnam() for details */
        if(grp) return grp->gr_gid;
    }
    return -1;
}

int ExecuteSystemCommand(char *argv[])
{
    if (!argv)       /* just checking... */
        return(1);

    pid_t pid;
    int status = -1;
    char *const envp[]  = { (char*)"GST_REGISTRY_UPDATE=no", (char*)"GST_REGISTRY_FORK=no", (char*)"LD_LIBRARY_PATH=/opt/bosch/security/lib", NULL};

#if 0 //SIGCHLD platform bug
    struct sigaction sigAction;
    memset(&sigAction, 0, sizeof(sigAction));
    sigAction.sa_handler = SIG_DFL;
    sigemptyset(&sigAction.sa_mask);
    sigAction.sa_flags   = 0;
    sigaction(SIGCHLD, &sigAction, 0);
#endif

    sig_t intsave, quitsave;
    sigset_t mask, omask;
    int pstat;

    sigemptyset(&mask);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_BLOCK, &mask, &omask);
    switch (pid = vfork()) {
        case -1:        /* error */
            syslog(LOG_INFO, "vfork error: %d", pid);
            sigprocmask(SIG_SETMASK, &omask, NULL);
            return(-1);
        case 0:             /* child */
            syslog(LOG_INFO, "child execv");
            sigprocmask(SIG_SETMASK, &omask, NULL);
            execv(argv[0], argv);
            syslog(LOG_INFO, "execv failed %d", errno);
            _exit(127);
    }

    syslog(LOG_INFO, "parent wait for %d", pid);
    syslog(LOG_INFO, "parent wait for %d", pid);
    // http://man7.org/linux/man-pages/man3/bsd_signal.3.html
    // change bsd_signal to signal
    intsave = (sig_t)  signal(SIGINT, SIG_IGN);
    quitsave = (sig_t) signal(SIGQUIT, SIG_IGN);
    pid = waitpid(pid, (int *)&pstat, 0);

    syslog(LOG_INFO, "waitpid returned: pid %d", pid);
    syslog(LOG_INFO, "waitpid returned: pstat %d", pstat);
    syslog(LOG_INFO, "waitpid returned: WIFEXITED %d", WIFEXITED(pstat));
    syslog(LOG_INFO, "waitpid returned: WEXITSTATUS %d", WEXITSTATUS(pstat));

    if(pid != -1 && WIFEXITED(pstat)) {
        //0-256 valid range, LSB 8 bit exit code of child process
        char cstatus = WEXITSTATUS(pstat);
        //cast to signed integer return value
        status = (cstatus & 0x80) ? (0xffffff00 | cstatus) : cstatus;
    }

    sigprocmask(SIG_SETMASK, &omask, NULL);
    (void)signal(SIGINT, intsave);
    (void)signal(SIGQUIT, quitsave);

    syslog(LOG_INFO, "ExecuteSystemCommand: returned %d", status);
    return status;
}

//>--Roadmap CMG3G-10221 : 'Scene Recorder '
int setfilePermission(const string path, string mode)
{
    int ret = -1;
    mode_t mType = 0;
    syslog(LOG_INFO, "setfilePermission: path=%s ",path.c_str());
    syslog(LOG_INFO, "setfilePermission: mode=%s ",mode.c_str());
    if(!path.empty() && !mode.empty())
    {
        mType = S_IFREG; //regular filetype
        std::transform(mode.begin(), mode.end(), mode.begin(), ::tolower);
        if(mode.find("r") != string::npos)
        {
            mType+= S_IRUSR|S_IRGRP;
        }
        if(mode.find("w") != string::npos)
        {
            mType+= S_IWUSR|S_IWGRP;
        }
        if(mode.find("x") != string::npos)
        {
            mType+= S_IXUSR|S_IXGRP;
        }
        syslog(LOG_INFO, "chmod :- mtype=%d ",mType);
        if ((ret = chmod(path.c_str(), mType)) != 0)
        {
            syslog(LOG_ERR, "setfilePermission: chmod failed %d - %s ", errno, strerror(errno));
        }
        else
        {
            syslog(LOG_INFO, "setfilePermission: File permissions changed successfully");
        }
    }
    else
    {
        syslog(LOG_ERR, "setfilePermission:Invalid filepath or filemode");
    }
    return ret;
}

int setMountOperation(const string path, const string filetype, const string mode)
{
    int retStatus = -1;
    if(!path.empty() && !mode.empty() && !filetype.empty())
    {
        syslog(LOG_INFO, "GMP_MOUNT_OPERATION mount point %s Mode %s filetype %s", path.c_str(),mode.c_str(),filetype.c_str());
        if(mode == "rw ")
        {
            retStatus = mount(path.c_str(),path.c_str(),filetype.c_str(),MS_REMOUNT,NULL);
        }
        else
        {
            retStatus = mount(path.c_str(),path.c_str(),filetype.c_str(),MS_REMOUNT | MS_RDONLY,NULL);
        }
        if(0 == retStatus)
        {
            syslog(LOG_INFO, "GMP_MOUNT_OPERATION : mount operation successful");
        }
        else
        {
            syslog(LOG_ERR, "GMP_MOUNT_OPERATION : unable to mount - %s ",strerror(errno));
        }
    }
    else
    {
        syslog(LOG_ERR, "setMountOperation:Invalid filepath or filemode or filetype.");
    }
    return retStatus;
}
//<--Roadmap CMG3G-10221 : 'Scene Recorder '

int ExecuteUnmountOperation(const string args)
{
   int result = ERR_CANNOT_EXEC_CMD;
   if(!args.empty())
   {
       syslog(LOG_INFO, "ExecuteUnmountOperation: mountpoint: [%s]", args.c_str());
       if(0 == umount(args.c_str()))
       {
           syslog(LOG_INFO, "ExecuteUnmountOperation: unmount operation is successful");
           result = ERR_NONE;
       }
       else
       {
           syslog(LOG_ERR, "ExecuteUnmountOperation: unsuccessful operation [%s]", strerror(errno));
           result = ERR_CANNOT_EXEC_CMD;
       }
   }
   else
   {
       result = ERR_INVALID_CMD_ARGS;
   }
   return result;
}
int LoadIAPModules(const string args)
{
    int result = ERR_CANNOT_EXEC_CMD;
    tBoolean isValidModName=false;
    std::string modname;
    std::string params;
    std::string delimiter = GMP_COMMANDS_DELIM;
    int err = -999;
    if(!args.empty())
    {
        size_t pos = args.find(delimiter);
        if(pos == string::npos)
        {
            modname = args;
            params = "";
        }
        else
        {
            //For MODULE_NAME_G_FFS
            modname = args.substr(0, pos);
            params = args.substr(pos + delimiter.length());
        }

        syslog(LOG_INFO, "GMP_IAP_LOAD_MODULE '%s', params '%s'", modname.c_str(), params.c_str());
        //For modules other than MODULE_NAME_G_FFS ,params should be empty.
        if((modname.compare(MODULE_NAME_G_FFS)!=0 && params.empty()) || (modname.compare(MODULE_NAME_G_FFS)==0 && !params.empty())) // Valid number of Args.
        {
            std::string validModNames[] = {MODULE_NAME_CONFIGFS,MODULE_NAME_LIBCOMPOSITE,MODULE_NAME_USB_F_FS,MODULE_NAME_U_EITHER,MODULE_NAME_G_FFS};
            size_t size=sizeof(validModNames)/sizeof(validModNames[0]);
            for(tU8 index=0;index<size;++index)
            {
                if(modname.find(validModNames[index]) != string::npos)
                {
                    isValidModName = true;
                    break;
               }
            }
        }
    }
    if(isValidModName)
    {
#ifdef TARGET_BUILD
        struct kmod_module *mod = 0;
        struct kmod_ctx *ctx = 0;

        ctx = kmod_new(NULL, NULL);
        if (!ctx)
        {
            syslog(LOG_ERR, "kmod_new failed: %s", strerror(errno));
            err = -1;
        }
        else
        {
            err = kmod_module_new_from_name(ctx, modname.c_str(), &mod);
            if (err != 0)
            {
                syslog(LOG_ERR, "kmod_module_new_from_name(...,%s,..) failed: %s", modname.c_str(), strerror(errno));
            }
            else
            {
                err = kmod_module_insert_module(mod, KMOD_PROBE_APPLY_BLACKLIST, params.c_str());
                if (err != 0)
                {
                    syslog(LOG_ERR, "kmod_module_insert_module(%s) failed: %s", modname.c_str(), strerror(errno));
                    result = ERR_CANNOT_EXEC_CMD;
                }
                else
                {
                    syslog(LOG_INFO, "kmod_module_insert_module(%s) ok", modname.c_str());
                    result = ERR_NONE;
                }
                kmod_module_unref(mod);
            }
            kmod_unref(ctx);
        }
#endif
     }
     else
     {
        syslog(LOG_ERR, "GMP_IAP_LOAD_MODULE: invalid module name: %s", modname.c_str());
        result = ERR_INVALID_CMD_ARGS;
     }
     return result;
}

int UnloadIAPModules(const string args)
{
     int result = ERR_CANNOT_EXEC_CMD;
     int err = -999;
     tBoolean isValidModName=false;
     std::string modname;
     // Non-empty args
     if(!args.empty())
     {
         std::string validModNames[] = {MODULE_NAME_CONFIGFS,MODULE_NAME_LIBCOMPOSITE,MODULE_NAME_USB_F_FS,MODULE_NAME_U_EITHER,MODULE_NAME_G_FFS,MODULE_NAME_USB_F_NCM,MODULE_NAME_USB_F_UAC2};
         size_t size=sizeof(validModNames)/sizeof(validModNames[0]);
         for(tU8 index=0;index<size;++index)
         {
             if(args.compare(validModNames[index]) == 0)
             {
                 isValidModName = true;
                 break;
             }
         }
     }
     // Valid module name is found
     if(isValidModName)
     {
         /* get the arguments */
         modname = args;
         syslog(LOG_INFO, "GMP_IAP_UNLOAD_MODULE '%s''", modname.c_str());
 #ifdef TARGET_BUILD
         struct kmod_module *mod = 0;
         struct kmod_ctx *ctx = 0;

         ctx = kmod_new(NULL, NULL);
         if (!ctx)
         {
             syslog(LOG_ERR, "kmod_new failed: %s", strerror(errno));
             err = -1;
         }
         else
         {
             err = kmod_module_new_from_name(ctx, modname.c_str(), &mod);
             if (err != 0)
             {
                 syslog(LOG_ERR, "kmod_module_new_from_name(...,%s,..) failed: %s", modname.c_str(), strerror(errno));
             }
             else
             {
                 err = kmod_module_remove_module(mod, KMOD_PROBE_APPLY_BLACKLIST);
                 if (err != 0)
                 {
                     syslog(LOG_ERR, "kmod_module_remove_module(%s) failed: %s", modname.c_str(), strerror(errno));
                     result = ERR_CANNOT_EXEC_CMD;
                 }
                 else
                 {
                     syslog(LOG_INFO, "kmod_module_remove_module(%s) ok", modname.c_str());
                     result = ERR_NONE;
                 }
                 kmod_module_unref(mod);
             }
             kmod_unref(ctx);
         }
 #endif
      }
      else
      {
          syslog(LOG_ERR, "GMP_IAP_UNLOAD_MODULE: invalid module name: %s", modname.c_str());
          result = ERR_INVALID_CMD_ARGS;
      }
      return result;
}

int ExecuteIAP2CommonWrite(string args)
{
    int result = ERR_CANNOT_EXEC_CMD;
    std::string delimiter = GMP_COMMANDS_DELIM;
    std::string otgPath;
    std::string key;
    std::string value;
    tBoolean isValidKey=false;
    int err = -999;

    if(!args.empty())
    {
        size_t pos = args.find(delimiter);
        otgPath = args.substr(0, pos); //otgPath
        args.erase(0, pos + delimiter.length());
        pos = args.find(delimiter);
        key = args.substr(0,pos); //key
        args.erase(0, pos + delimiter.length());
        value = args.substr(0,args.find(delimiter)); //value

        std::string validKeyValues[] = {STR_FORCE_UDC,STR_DEFAULT_UDC,STR_VBUS,STR_VBUS_AUTO,STR_VBUS_AUTO_USB1,STR_ROLE,STR_BRIDGE,STR_CONTROL,STR_PORTPOWER};
        size_t size = sizeof(validKeyValues)/sizeof(validKeyValues[0]);

        //Validating the key value
        for(tU8 index=0;index<size;++index)
        {
            if(key.compare(validKeyValues[index]) == 0)
            {
                isValidKey = true;
                break;
            }
        }
    }
    // valid key is found
    if(isValidKey)
    {
        //switch OTG to Value
        if (TRUE != iAP2Common_Write(otgPath.c_str(), key.c_str(), value.c_str(), TRUE))
        {
            syslog(LOG_ERR, "GMP_IAP_COMMON_WRITE():  writing  %40s/%40s to %s failed", otgPath.c_str(), key.c_str(), value.c_str());
            err = IAP2_CTL_ERROR; //= -1
            result = ERR_CANNOT_EXEC_CMD;
        }
        else
        {
            syslog(LOG_INFO, "GMP_IAP_COMMON_WRITE(%40s/%40s, %s) ok", otgPath.c_str(), key.c_str(), value.c_str());
            err = IAP2_OK; //= 0
            result = ERR_NONE;
        }
    }
    else
    {
        syslog(LOG_ERR, "GMP_IAP_LOAD_MODULE: invalid key name: %s", key.c_str());
        result = ERR_INVALID_CMD_ARGS;
    }
    return result;
}

int ExecuteIAPMountOperation()
{
    int err = -1;
    int result = ERR_CANNOT_EXEC_CMD;
    err = mkdir(FUNCTION_FS_PATH, 0755);
    syslog(LOG_INFO, "mkdir(%s):%d", FUNCTION_FS_PATH, err);
 #if 1
    /* temp workaround - till group is released*/
     err = mount(FUNCTION_FS_NAME, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, 0, "mode=0777");
     syslog(LOG_INFO, "mount(%s,%s,%s,0,mode=0777)", FUNCTION_FS_NAME, FUNCTION_FS_PATH, FUNCTION_FS_TYPE);
 #else // solution when the carplay is switched to user mode
                 /* target */

     char cbuf[512];
     snprintf(cbuf, sizeof(cbuf), "mode=0770,gid=%d", GetGidByName("aid_gadgetffs"));
     mount(FUNCTION_FS_NAME, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, MS_NOEXEC, cbuf);
     syslog(LOG_INFO, "mount(%s,%s,%s,%d,%s)", FUNCTION_FS_NAME, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, MS_NOEXEC, cbuf);
 #endif
     if(0 == err)
     {
         result = ERR_NONE;
     }
     else
     {
         syslog(LOG_ERR, "mount(%s): %d, %s", FUNCTION_FS_PATH, errno, strerror(errno));
         result = ERR_CANNOT_EXEC_CMD;
     }
     return result;
}

int ExecuteIAPUnmountOperation()
{
    int result = ERR_CANNOT_EXEC_CMD;
    syslog(LOG_INFO, "umount(%s)", FUNCTION_FS_PATH);
    if(0 == umount(FUNCTION_FS_PATH))
    {
        result = ERR_NONE;
    }
    else if (EBUSY ==  errno || EAGAIN == errno)
    {
        result = ERR_CANNOT_EXEC_CMD;
    }
    syslog(LOG_ERR, "umount(%s): %d, %s", FUNCTION_FS_PATH, errno, strerror(errno));
    return result;
}

int ExecuteIAPMountConfigs(const string args)
{
   int result = ERR_CANNOT_EXEC_CMD;
   int err = -1;
   if(!args.empty()) //Expect NON-EMPTY arguments
   {
       syslog(LOG_INFO, "Enter GMP_IAP_MOUNT_CONFIGFS:");
       /* temp workaround - till group is released*/
       err = mount(CONFIGFS_NAME, CONFIGFS_PATH, CONFIGFS_TYPE, 0, "mode=0777");
       syslog(LOG_INFO, "mount(%s,%s,%s,0,mode=0777)", CONFIGFS_NAME, CONFIGFS_PATH, CONFIGFS_TYPE);
       err = chmod(CONFIGFS_PATH, 0777);
       syslog(LOG_INFO, "chmod(%s,0,mode=0777): %d", CONFIGFS_PATH, err);
 #ifndef IPODCONTROL_IAP2_PF_CONFIGFS_NON_ROOT
       err = chmod(CONFIGFS_USB_GADGET_PATH, 0777);
       syslog(LOG_INFO, "chmod(%s,0,mode=0777): %d", CONFIGFS_USB_GADGET_PATH, err);
       err = 0;
 #else
       syslog(LOG_INFO, "GMP_IAP_MOUNT_CONFIGFS:");
       //non root user mode
       /* get the arguments */
       bool carPlayEnabled = false;
       bool nativeTransportEnabled = false;
       bool carLifeEnabled = false;
       char * tmp = 0;
       char * deviceID = 0;
       char * name = 0;
       char * modelID = 0;
       char * manufacturer = 0;
       char * serialNumber = 0;
       char * vendorID = 0;
       char * productID = 0;
       char * bcdDevice = 0;
       char* carLife = 0;
       const char * mountLocation = CONFIGFS_PATH;
       char * udcDeviceName = 0;
       const char * initEndPoint = "/dev/ffs/ep0";
       char * appName = 0;

       char *iter_ptr;
       char * arguments = new char [args.length()+1];
       memset(arguments ,0 ,args.length()+1);
       strncpy(arguments,args.c_str(),args.length());
       syslog(LOG_INFO, "arguments: %s", arguments);

       iter_ptr = strndup(arguments,args.length());
       char *iter = iter_ptr;
       carPlayEnabled = !strcmp("true", strsep(&iter,GMP_COMMANDS_DELIM));
       nativeTransportEnabled = !strcmp("true", strsep(&iter,GMP_COMMANDS_DELIM));
       carLife = strsep(&iter,GMP_COMMANDS_DELIM);
       carLifeEnabled = !strcmp("true", carLife);
       //carLifeEnabled = !strcmp("true", strsep(&iter,GMP_COMMANDS_DELIM));
       deviceID = strsep(&iter,GMP_COMMANDS_DELIM);
       name = strsep(&iter,GMP_COMMANDS_DELIM);
       modelID = strsep(&iter,GMP_COMMANDS_DELIM);
       manufacturer = strsep(&iter,GMP_COMMANDS_DELIM);
       serialNumber = strsep(&iter,GMP_COMMANDS_DELIM);
       vendorID = strsep(&iter,GMP_COMMANDS_DELIM);
       productID = strsep(&iter,GMP_COMMANDS_DELIM);
       bcdDevice = strsep(&iter,GMP_COMMANDS_DELIM);
       udcDeviceName = strsep(&iter,GMP_COMMANDS_DELIM);
       appName = strsep(&iter,GMP_COMMANDS_DELIM);

       syslog(LOG_INFO, "carPlayEnabled: %d", carPlayEnabled);
       syslog(LOG_INFO, "nativeTransportEnabled: %d", nativeTransportEnabled);
       syslog(LOG_INFO, "carLife: %s", carLife);
       syslog(LOG_INFO, "carLifeEnabled: %d", carLifeEnabled);
       syslog(LOG_INFO, "deviceID: %s", deviceID);
       syslog(LOG_INFO, "name: %s", name);
       syslog(LOG_INFO, "modelID: %s", modelID);
       syslog(LOG_INFO, "manufacturer: %s", manufacturer);
       syslog(LOG_INFO, "serialNumber: %s", serialNumber);
       syslog(LOG_INFO, "vendorID: %s", vendorID);
       syslog(LOG_INFO, "productID: %s", productID);
       syslog(LOG_INFO, "bcdDevice: %s", bcdDevice);
       syslog(LOG_INFO, "udcDeviceName: %s", udcDeviceName);
       syslog(LOG_INFO, "appName: %s", appName);
       syslog(LOG_INFO, "mountLocation: %s", mountLocation);
       syslog(LOG_INFO, "initEndPoint: %s", initEndPoint);

       iAP2_usbg_config_t usb_gadget_configuration;
       memset(&usb_gadget_configuration, 0, sizeof(usb_gadget_configuration));

       err = iap2InitGadgetConfiguration(&usb_gadget_configuration, carPlayEnabled,nativeTransportEnabled,carLifeEnabled,
                         deviceID, name, modelID, manufacturer, serialNumber, vendorID, productID, bcdDevice, mountLocation, udcDeviceName, initEndPoint, appName);
       syslog(LOG_INFO, "iap2InitGadgetConfiguration(): %d", err);
       if(err == IAP2_OK)
       {
           err = iAP2InitializeGadget(&usb_gadget_configuration);
           syslog(LOG_INFO, "iAP2InitializeGadget(): %d", err);
           if(err == IAP2_OK)
           {
               err = mkdir(FUNCTION_FS_PATH, 0755);
               syslog(LOG_INFO, "mkdir(%s): %d", FUNCTION_FS_PATH, err);
               if((err == 0) || (err == EEXIST))
               {
                   err =  mount((char*)usb_gadget_configuration.iAP2FFS_InstanceName, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, 0, "mode=0777");
                   syslog(LOG_INFO, "mount(%s,%s,%s,0,mode=0777): %d", (char*)usb_gadget_configuration.iAP2FFS_InstanceName, FUNCTION_FS_PATH, FUNCTION_FS_TYPE, err);
                   if(err == IAP2_OK)
                   {
                       err = chmod(FUNCTION_FS_PATH, 0777);
                       syslog(LOG_INFO, "chmod(%s,0,mode=0777): %d", FUNCTION_FS_PATH, err);
                       if(err == IAP2_OK)
                       {
                           err = chmod(FUNCTION_FS_PATH_EP, 0777);
                           syslog(LOG_INFO, "chmod(%s,0,mode=0777): %d", FUNCTION_FS_PATH_EP, err);
                           if(err == IAP2_OK)
                           {
                               err = iAP2InitializeFFSGadget(&usb_gadget_configuration);
                               result = ERR_NONE;
                               syslog(LOG_INFO, "iAP2InitializeFFSGadget(%s): %d", (char*)initEndPoint, err);
                           }
                       }
                   }
               }
           }
       }
       //clear up
       free(iter_ptr);
       iap2DeInitGadgetConfiguration(&usb_gadget_configuration);
       delete[] arguments;
 #endif
    }
    else
    {
        result = ERR_INVALID_CMD_ARGS;
    }
    return result;
}

int ExecuteIAPUnmountConfigs()
{
    int result = ERR_CANNOT_EXEC_CMD;
    int err = -1;
 #ifdef IPODCONTROL_IAP2_PF_CONFIGFS_NON_ROOT
    err = iAP2DeInitializeGadget();
    syslog(LOG_INFO, "iAP2DeInitializeGadget(): %d", err);
 #endif
    if (0 == access(CONFIGFS_PATH, F_OK))
    {
        if(0 == umount(CONFIGFS_PATH))
        {
            result = ERR_NONE;
        }
        else if (EBUSY ==  errno || EAGAIN == errno)
        {
            result = ERR_CANNOT_EXEC_CMD;
        }
        syslog(LOG_ERR, "umount(%s): %d, %s", CONFIGFS_PATH, errno, strerror(errno));

        //err = rmdir(CONFIGFS_PATH);
        //syslog(LOG_INFO, "rmdir(%s): %i", CONFIGFS_PATH, err);
    }
    if (0 == access(FUNCTION_FS_PATH, F_OK))
    {
        if(0 == umount(FUNCTION_FS_PATH))
        {
            result = ERR_NONE; // Set default as no error
        }
        else if (EBUSY ==  errno || EAGAIN == errno)
        {
            result = ERR_CANNOT_EXEC_CMD;
        }
        syslog(LOG_ERR, "umount(%s): %d, %s", FUNCTION_FS_PATH, errno, strerror(errno));
        err = rmdir(FUNCTION_FS_PATH);
        syslog(LOG_INFO, "rmdir(%s): %i", FUNCTION_FS_PATH, err);
    }
    return result;
}

int ChangeFilePermissions(const string args)
{
    int result = ERR_CANNOT_EXEC_CMD;
    std::string delimiter = GMP_COMMANDS_DELIM;
    int err = -1;
    if(!args.empty()) //Expect NON-EMPTY arguments
    {
        size_t pos = args.find(delimiter);
        std::string pathToChange = args.substr(0, pos);
        std::string mode = args.substr(pos + delimiter.length());
        if((pathToChange.find(validPathNames[0]) != string::npos) && (!mode.empty()))
        {
            syslog(LOG_INFO, "GMP_FILE_CHMOD Path %s and Mode %s", pathToChange.c_str(), mode.c_str());
            err = setfilePermission(pathToChange.c_str(),mode.c_str());
            if(!err)
            {
                syslog(LOG_INFO, "GMP_FILE_CHMOD, setfilePermission() successful operation");
                result = ERR_NONE;
            }
            else
            {
                syslog(LOG_INFO, "GMP_FILE_CHMOD, setfilePermission() unsuccessful operation [path: %s mode: %s]",pathToChange.c_str(), mode.c_str());
                result = ERR_CANNOT_EXEC_CMD;
            }
        }
        else
        {
            syslog(LOG_INFO, "GMP_FILE_CHMOD unsuccessful operation [path: %s mode: %s]",pathToChange.c_str(), mode.c_str());
            result = ERR_INVALID_CMD_ARGS;
        }
    }
    else
    {
        result = ERR_INVALID_CMD_ARGS;
    }
    return result;
}

int ExecuteMountOperation(string args)
{
    int result = ERR_CANNOT_EXEC_CMD;
    std::string delimiter = GMP_COMMANDS_DELIM;
    int err = -1;
    if(!args.empty()) //Expect NON-EMPTY arguments
    {
        size_t pos = args.find(delimiter);
        std::string pathToChange = args.substr(0, pos); //pathToChange
        args.erase(0, pos + delimiter.length());
        pos = args.find(delimiter);
        std::string fileSystemType = args.substr(0,pos); //fileSystemType
        args.erase(0, pos + delimiter.length());
        std::string mode  = args.substr(0,args.find(delimiter)); //mode

        if((pathToChange.find(validPathNames[0]) != string::npos) && (fileSystemType.compare("unknown")!=0))
        {
            syslog(LOG_INFO, "GMP_MOUNT_OPERATION Path %s , Filetype %s and Mode %s", pathToChange.c_str(), fileSystemType.c_str(), mode.c_str());
            err = setMountOperation(pathToChange.c_str(),fileSystemType.c_str(),mode.c_str());
            if(!err)
            {
                result = ERR_NONE;
            }
            else
            {
                syslog(LOG_INFO, "GMP_MOUNT_OPERATION, setMountOperation() unsuccessful operation [path: %s mode: %s]",pathToChange.c_str(), mode.c_str());
                result = ERR_CANNOT_EXEC_CMD;
            }
        }
        else
        {
            syslog(LOG_INFO, "GMP_MOUNT_OPERATION unsuccessful operation [path: %s mode: %s]",pathToChange.c_str(), mode.c_str());
            result = ERR_INVALID_CMD_ARGS;
        }
    }
    else
    {
        result = ERR_INVALID_CMD_ARGS;
    }
    return result;
}

int ReloadUdevRules()
{
    int result = ERR_CANNOT_EXEC_CMD;
    char *execvArgs[3] = {(char *)"/bin/udevadm",(char *)"control --reload-rules",NULL};

    syslog(LOG_INFO, "ReloadUdevRules(), executing[%s %s]", execvArgs[0],execvArgs[1]);
    if(0 == ExecuteSystemCommand(execvArgs))
    {
        syslog(LOG_INFO, "ReloadUdevRules(): Successful operation");
        result = ERR_NONE;
    }
    else
    {
        syslog(LOG_INFO, "ReloadUdevRules(): unsuccessful operation");
        result = ERR_CANNOT_EXEC_CMD;
    }
    return result;
}

//---------------------------------------------------------------------------------------------------------
///////////////////////////////////////////END OF HELPER FUNCTIONS//////////////////////////////////////////
//---------------------------------------------------------------------------------------------------------


#ifdef __cplusplus
extern "C"
{
#endif

    const char * getClientName()
   {
      return "FC_Mediaplayer";//TODO put it in header file
   }

    const char * getClientGroupName()
   {
       return (char *)"aid_mediaplayer";
   }

   const char * getClientUserName()
   {
       return (char *)"aid_mediaplayer";
   }

    CmdData command(const int cmdNum, std::string args)
   {
      CmdData result;
      string command_str;
      int err = -999;
      string pat;
      std::string delimiter = GMP_COMMANDS_DELIM;
      switch(cmdNum)
      {
         /*cmdNum = 0*/
         case GMP_CMD_UNMOUNT: //Call in USBControl::Umount
         {
              result.errorNo = ExecuteUnmountOperation(args);
              break;
         }

         /*cmdNum = 1*/
         case GMP_IAP_LOAD_MODULE: //Call in iPodControlIAP::load_modules
         {
             result.errorNo = LoadIAPModules(args);
             break;
         }

          /*cmdNum = 2*/
         case GMP_IAP_UNLOAD_MODULE: //Call in iPodControlIAP::unload_modules_pre_umount // iPodControlIAP::unload_modules()
         {
             result.errorNo = UnloadIAPModules(args);
             break;
         }

         /*cmdNum = 3*/
         case GMP_IAP_COMMON_WRITE: //Call in iPodControlIAP::iap2CommonWrite
         {
             result.errorNo = ExecuteIAP2CommonWrite(args);
             break;
         }

         /*cmdNum = 4*/
         case GMP_IAP_MOUNT: //Call in iPodControlIAP::SwitchToHostMode
         {
             result.errorNo = ExecuteIAPMountOperation();
             break;
         }

         /*cmdNum = 5*/
         case GMP_IAP_UMOUNT: //Call in iPodControlIAP::SwitchToDeviceMode
         {
             result.errorNo = ExecuteIAPUnmountOperation();
             break;
         }

         /*cmdNum = 6*/
         case GMP_IAP_MOUNT_CONFIGFS:  // mount for ipod  //Call in iPodControlIAP::SwitchToHostMode
         {
             result.errorNo = ExecuteIAPMountConfigs(args);
             break;
         }

         /*cmdNum = 7*/
         case GMP_IAP_UMOUNT_CONFIGFS: // mount for ipod  //Call in iPodControlIAP::SwitchToDeviceMode
         {
             result.errorNo = ExecuteIAPUnmountConfigs();
             break;
         }

         /*cmdNum = 8*/
         case GMP_FILE_CHMOD: //Call in ListControl::SetFilePermissions
         {
             result.errorNo = ChangeFilePermissions(args);
             break;
         }

         /*cmdNum = 9*/
         case GMP_MOUNT_OPERATION: // Call in ListControl::ReMountFolder
         {
             result.errorNo = ExecuteMountOperation(args);
             break;
         }

         /*cmdNum = 10*/
         case GMP_CMD_MANUAL_UDEVRELOADRULES: //call in TraceCmdManager::TTFIScmdUDEVTrigger
         {
             result.errorNo = ReloadUdevRules();
             break;
         }

         default:
            syslog(LOG_INFO, "FC_Mediaplayer_rootdaemon_client:command(),default");
            result.errorNo = ERR_UNKNOWN_CMD;
      }
      syslog(LOG_INFO, "CmdData command() cmdNum: %d result: %d", cmdNum,result.errorNo);
      return result;
   }

   #ifdef __cplusplus
}
#endif


CmdData execRootCommand(const char * clientName, const int cmdNum, const char * args, const long timeoutSec)
{
   uid_t euid = geteuid();

   if(0 == euid)
   {
      (void)timeoutSec; // Is made void. Concerned developer has to extend if needed in future.
      //User is root: execute root operation directly
      syslog(LOG_INFO, "FC_MediaPlayer_rootdaemon_client:execRootCommand() calling command(), clientname[%s],cmdnum[%d], args[%s]",clientName, cmdNum, args);
      return command(cmdNum, std::string(args));
   }
   else
   {
      //use root daemon to perform root operation
      syslog(LOG_INFO, "FC_MediaPlayer_rootdaemon_client:execRootCommand() calling performRootOp(), cmdnum[%d]", cmdNum);
      syslog(LOG_INFO, "FC_MediaPlayer_rootdaemon_client:execRootCommand() calling performRootOp(), clientname[%s],cmdnum[%d], args[%s]",clientName, cmdNum, args);
      return RootDaemonHelper::performRootOp(clientName, cmdNum, std::string(args), -1/*indefinite timeout*/);
   }
}
