#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/iPodControlIndexer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#endif
#endif

#include <string.h>
#include "FunctionTracer.h"
#include "VarTrace.h"
#include "LocalSPM.h"
#include "iPodControlIndexer.h"

iPodControlIndexerContext::iPodControlIndexerContext()
{
	ENTRY_INTERNAL;
	Reset();
}

iPodControlIndexerContext::~iPodControlIndexerContext()
{
    ENTRY_INTERNAL;
    Reset();
}

void iPodControlIndexerContext::Reset()
{
    ENTRY;

    mStrategy = IPOD_INDEXER_STRATEGY_UNKNOWN;
    mIndexedItems = 0;
    mRemovedItems = 0;
    mTrackIndex = 0;
    mTrackCount = 0;
    mPlaylistIndex = 0;
    mPlaylistCount = 0;
    mVideoIndex = 0;
    mVideoCount = 0;
    mChunkSize = 1;
    mChunkPerf = 0.0;
    mRetrieveError = false;
    mMediaLibraryUpdateProgress = 0;
    miTunesRadioLibraryUpdateProgress = 0;
    mMediaLibraryItemsCompleted = false;

    mMediaLibraryUID[0] = 0;
    miTunesRadioLibraryUID[0] = 0;
    mNowPlayingLibraryUID[0] = 0;
    mMediaLibraryRevision[0] = 0;
    miTunesRadioLibraryRevision[0] = 0;
    mFingerprint[0] = 0;
    mPaused = false;
    mMediaLibraryReset = false;

    VARTRACE(mObjectList.size());
    for(unsigned int i = 0; i < mObjectList.size(); i++) {
        delete mObjectList[i].ptr;
        mObjectList[i].ptr = 0;
    }
    vector<tIndexObject>().swap(mObjectList); //used instead of clear to guarantee a memory reallocation

    VARTRACE(mPlaylistObjectList.size());
    vector<tPlaylistObject>().swap(mPlaylistObjectList); //used instead of clear to guarantee a memory reallocation

    VARTRACE(mPlaylistFileTransferObjectList.size());
    tIPODPlaylistFileTransfer::iterator it;
    for (it = mPlaylistFileTransferObjectList.begin(); it != mPlaylistFileTransferObjectList.end(); ++it) {
        delete [] (it->second.buffer);
    }
    tIPODPlaylistFileTransfer().swap(mPlaylistFileTransferObjectList); //used instead of clear to guarantee a memory reallocation
}

void iPodControlIndexerContext::TraceOut() const
{
	ENTRY_INTERNAL;
	tGeneralString totaltime;
	tGeneralString cachetime;
	mTotalTime.ElapsedPrint(totaltime);
	mCacheTime.ElapsedPrint(cachetime);

	int totalms = mTotalTime.ElapsedMS();
	double totalperf = totalms ? mIndexedItems*1000.0 / totalms : 0.0;

	tURL url = {0};
	mPath.GetURL(url, -1, -1, 0ULL, 0ULL, NULL);

	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR3(("*** IPOD INDEXER - STRATEGY %d", (int)mStrategy));
	ETG_TRACE_USR3(("*** FINGERPRINT    %s", mFingerprint));
	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR2(("*** TOTAL ITEMS    %d", mIndexedItems));
	ETG_TRACE_USR2(("*** TOTAL TIME     %s", totaltime));
	ETG_TRACE_USR2(("*** TOTAL PERF     %.2f/sec", totalperf));
	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR2(("*** TOTAL REMOVED  %d", mRemovedItems));
	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR3(("*** CACHE ITEMS    %d", mObjectList.size()));
	ETG_TRACE_USR3(("*** CACHE TIME     %s", cachetime));
	ETG_TRACE_USR2(("*** CHUNK PERF     %.2f/sec", mChunkPerf));
	ETG_TRACE_USR2(("*** CHUNK SIZE     %d", mChunkSize));
	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR3(("*** URL %s", url));
	ETG_TRACE_USR3(("*******************************"));
	ETG_TRACE_USR3(("*** TrackIndex     %d", mTrackIndex));
	ETG_TRACE_USR3(("*** TrackCount     %d", mTrackCount));
	ETG_TRACE_USR3(("*** PlaylistIndex  %d", mPlaylistIndex));
	ETG_TRACE_USR3(("*** PlaylistCount  %d", mPlaylistCount));
	ETG_TRACE_USR3(("*** VideoIndex     %d", mVideoIndex));
	ETG_TRACE_USR3(("*** VideoCount     %d", mVideoCount));
	ETG_TRACE_USR3(("*******************************"));
}

void iPodControlIndexerContext::PushObject(tMediaObjectPtr ptr, tMetadataStatus status)
{
	ENTRY_INTERNAL;
	if(status == MDS_SUCCESS) {
		mIndexedItems++;
	} else if(status == MDS_REMOVED) {
		mRemovedItems++;
	}

	tIndexObject obj;
	obj.ptr = ptr;
	obj.status = status;

	mObjectList.push_back(obj);
}

tBoolean iPodControlIndexerContext::PopObject(tMediaObjectPtr &ptr, tMetadataStatus &status)
{
	ENTRY_INTERNAL;

	VARTRACE(mObjectList.size());
    VARTRACE(mPlaylistObjectList.size());
    VARTRACE(mPlaylistFileTransferObjectList.size());

	if(mObjectList.empty()) {
		return false;
	}

	tIndexObject obj = mObjectList.front();
	mObjectList.erase(mObjectList.begin());
	ptr = obj.ptr;
	status = obj.status;

	return true;
}

void iPodControlIndexerContext::PushPlaylistObject(const tU64 parentUUID, const vector<tU64> uuidList)
{
    ENTRY;
    VARTRACE(uuidList.size());
    VARTRACE(parentUUID);
    vector<tPlaylistObject>::iterator it;
    vector<tPlaylistObject> cleanedUpList;
    bool isCleanUpNeeded = false;
	//NCG3D-55130: Remove previous updates for same playlist(parentUUID). Keep other playlists untouched.
    for(it = mPlaylistObjectList.begin(); it != mPlaylistObjectList.end(); ++it)
    {
    	if(it->parentUUID == parentUUID) // ignore all existing objects of Playlist(parentUUID).
    	{
			isCleanUpNeeded = true;
    		mIndexedItems = mIndexedItems - it->uuidList.size();
    		ETG_TRACE_USR3(("Removed previous Playlist items after receiving latest update"));
    	}
    	else
    	{
    		cleanedUpList.push_back(*it);  //take backup of other objects.
    	}
    }

    if(isCleanUpNeeded)
    {
    	VARTRACE(cleanedUpList.size());
    	mPlaylistObjectList.assign(cleanedUpList.begin(),cleanedUpList.end());
    	ETG_TRACE_USR3(("Assign mPlayListObjectList with cleanedUpList"));
    	VARTRACE(mPlaylistObjectList.size());
    }
    mIndexedItems += uuidList.size();

    tPlaylistObject obj;
    obj.parentUUID = parentUUID;
    obj.uuidList = uuidList;
    obj.listIndex = 0;

    if(uuidList.size() > 0) {
        mPlaylistObjectList.push_back(obj);
    }
}

tBoolean iPodControlIndexerContext::PopPlaylistObject(tMediaObjectPtr &ptr, const tDeviceID deviceID, const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    ptr = NULL;

    if(mPlaylistObjectList.empty()) {
        return false;
    }

    tPlaylistObject obj = mPlaylistObjectList.front();
    VARTRACE(obj.parentUUID);
    VARTRACE(obj.uuidList.size());
    VARTRACE(obj.listIndex);

    tU64 uuid = 0;

    if(obj.uuidList.size() > obj.listIndex) {
        uuid = obj.uuidList[obj.listIndex];
    }
    VARTRACE(uuid);

    //setup mediaObject
    if(uuid > 0) {
        //set the mediaobject
        tMediaObjectPtr mediaObjectPtr = new tMediaObject; //Attention: Has to be deleted by Indexer::CheckMetadata
        if (!mediaObjectPtr) {
            ETG_TRACE_ERR(("No memory"));
            return false;
        }
        //set general parameter
        InitMediaObject(OUT *mediaObjectPtr);
        strncpy_r(mediaObjectPtr->mountPoint, mountPoint, sizeof(mediaObjectPtr->mountPoint));
        mediaObjectPtr->deviceID = deviceID;

        snprintf(mediaObjectPtr->UUID, sizeof(mediaObjectPtr->UUID), IPODCONTROL_UUID_FORMAT, uuid);
        snprintf(mediaObjectPtr->parentUUID, sizeof(mediaObjectPtr->parentUUID), IPODCONTROL_UUID_FORMAT, obj.parentUUID);

        iPodControlMediaPath path(LTY_PLAYLIST_SONG);
        path.GetURL(mediaObjectPtr->fileName, obj.listIndex, -1, uuid, obj.parentUUID, "");
        strncpy_r(mediaObjectPtr->albumArtString, mediaObjectPtr->fileName, sizeof(mediaObjectPtr->albumArtString));

        mediaObjectPtr->trackNumber = (tTrackNumber)obj.listIndex;
        mediaObjectPtr->fileFormat = FFT_UNKNOWN;
        mediaObjectPtr->mediaType = MTY_UNKNOWN;
        mediaObjectPtr->fileType = FT_AUDIO;
        mediaObjectPtr->catType = CTY_PLAYLIST_ITEM;

        ptr = mediaObjectPtr;

    } else { //lint !e429 deleted by Indexer
        ETG_TRACE_ERR(("Invalid UUID"));
    }

    mPlaylistObjectList.front().listIndex++;

    if(mPlaylistObjectList.front().listIndex >= mPlaylistObjectList.front().uuidList.size()) {
        //free list element
        mPlaylistObjectList.erase(mPlaylistObjectList.begin());
    }

    return true;
}

void iPodControlIndexerContext::PushPlaylistFileTransferObject(const tU8 playlistFileTransferID, const tPlaylistFileTransferObj obj)
{
    ENTRY_INTERNAL;
    mPlaylistFileTransferObjectList[playlistFileTransferID] = obj;
}

tBoolean iPodControlIndexerContext::PopPlaylistFileTransferObject(const tU8 playlistFileTransferID, tPlaylistFileTransferObj &obj)
{
    ENTRY_INTERNAL;

    tIPODPlaylistFileTransfer::iterator it = mPlaylistFileTransferObjectList.find(playlistFileTransferID);
    if (it != mPlaylistFileTransferObjectList.end()) {
        obj = it->second;
        mPlaylistFileTransferObjectList.erase(it);
        return true;
    }
    return false;
}

void iPodControlIndexerContext::IncrementPath(const tResult result)
{
	ENTRY_INTERNAL;
	//set next path for indexing purpose
	//incrementing indices in the database hierarchy
	//On error or end the path is made invalid to prevent further usage
	//NOTE: After single increment step a new iPodControlIAP::SelectRecord call
	//      is necessary in order to update the map for mCount and mName
	//INDEXING ORDER:
	//1)TRACKS 		(LTY_GENRE_ARTIST_ALBUM_SONG)
	//2)VIDEO 		(LTY_VIDEO_EPISODE)
	//3)PLAYLISTS 	(LTY_PLAYLIST)

	int itemcount = mPath.GetCount(mPath.IsPlaylist() ? IPOD_CAT_PLAYLIST : IPOD_CAT_TRACK);
	VARTRACE(mTrackIndex);
	VARTRACE(itemcount);
	VARTRACE(result);
	bool end_reached = false;
	if(result != MP_NO_ERROR) {
		end_reached = true;
	} else if(mTrackIndex >= itemcount) {
		mTrackIndex = 0;
		if(!mPath.Increment()) {
			if(mPath.IsPlaylist()) {
				end_reached = true;
			} else if(mPath.IsVideo()) {
				if(mPlaylistCount > 0) {
					mPath = iPodControlMediaPath(LTY_PLAYLIST);
					mTrackIndex = 1; //skip default playlist
				} else {
					end_reached = true;
				}
			} else {
				if(mVideoCount > 0) {
					mPath = iPodControlMediaPath(LTY_VIDEO_EPISODE, 0 );
				} else if (mPlaylistCount > 0) {
					mPath = iPodControlMediaPath(LTY_PLAYLIST);
					mTrackIndex = 1; //skip default playlist
				} else {
					end_reached = true;
				}
			}
		}
	}

	if(end_reached) {
		//reached end of indexing paths
		ETG_TRACE_USR3(("*******************************"));
		mPath = iPodControlMediaPath(); //reset path, make it invalid for further usage
	}
}

void iPodControlIndexerContext::OnBeginCaching()
{
	ENTRY_INTERNAL;
	mCacheTime.Restart();
}

void iPodControlIndexerContext::OnEndCaching(const tSize size)
{
	ENTRY_INTERNAL;
	//calc current chunk performance
	int cachems = mCacheTime.ElapsedMS();
	mChunkPerf = cachems ? (size ? size : mObjectList.size())*1000.0 / cachems : 0.0;

	//set next chunk size
	mChunkSize = (int) (LocalSPM::GetDataProvider().iPodControlChunkTimeMS() * mChunkPerf / 1000.0);

	//limit new chunk value
	if(mChunkSize < 1) {
		mChunkSize = 1;
	} else if(mChunkSize > LocalSPM::GetDataProvider().iPodControlChunkMax()) {
		mChunkSize = LocalSPM::GetDataProvider().iPodControlChunkMax();
	}
}

