#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/iPodControlHandleMap.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#endif
#endif

#include "FunctionTracer.h"
#include "VarTrace.h"
#include <Utf8String.h>
#include "iPodControlHandleMap.h"
#include <unistd.h> //usleep()

iPodControlHandleMap::iPodControlHandleMap() {
    ENTRY_INTERNAL;
}

void iPodControlHandleMap::Clear() {
    ENTRY;
    Locker locker(&m_Mutex);
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i]) {
            delete mList[i];
        }
    }
    mList.clear();
}

void iPodControlHandleMap::AddHandle(const tDeviceInfo deviceInfo) {
    ENTRY;

    VARTRACE(deviceInfo.mountPoint);
    VARTRACE(deviceInfo.deviceID);
    VARTRACE(deviceInfo.connectionType);
    VARTRACE(deviceInfo.deviceName);

    Locker locker(&m_Mutex);

    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i]) {
            //check for existing deviceID
            if (mList[i]->m_DeviceInfo.deviceID == deviceInfo.deviceID) {
                ETG_TRACE_USR3(("Found entry at %u by deviceID %d, mountpoint '%s'", i, deviceInfo.deviceID, mList[i]->m_DeviceInfo.mountPoint));
                mList[i]->m_DeviceInfo = deviceInfo;
                return;
            } else if(!strcmp(mList[i]->m_DeviceInfo.mountPoint, deviceInfo.mountPoint)) {
                ETG_TRACE_USR3(("Found entry at %u by mountPoint '%40s', deviceID %d", i, deviceInfo.mountPoint, mList[i]->m_DeviceInfo.deviceID));
                mList[i]->m_DeviceInfo = deviceInfo;
                return;
            }
        } else {
            ETG_TRACE_ERR(("Null pointer or empty mount point in iPod handle list"));
        }
    }

    ETG_TRACE_USR4(("Append new handle, index %d", mList.size()));

    /* add the new list to the vector */
    iPodControlHandle *newHandle = new iPodControlHandle;
    if(newHandle) {
        newHandle->m_DeviceInfo = deviceInfo;
        mList.push_back(newHandle);
    }
} //lint !e429 deleted by Clear()

void iPodControlHandleMap::UpdateHandle(const tDeviceInfo deviceInfo) {
    ENTRY;

    VARTRACE(deviceInfo.mountPoint);
    VARTRACE(deviceInfo.deviceID);
    VARTRACE(deviceInfo.accessoryName);

    Locker locker(&m_Mutex);

    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i]) {
            bool found_entry = false;
            //check for existing deviceID
            if (mList[i]->m_DeviceInfo.deviceID == deviceInfo.deviceID) {
                ETG_TRACE_USR3(("Found entry at %u by deviceID %d, mountpoint '%s'", i, deviceInfo.deviceID, mList[i]->m_DeviceInfo.mountPoint));
                found_entry = true;
            } else if(!strcmp(mList[i]->m_DeviceInfo.mountPoint, deviceInfo.mountPoint)) {
                ETG_TRACE_USR3(("Found entry at %u by mountPoint '%40s', deviceID %d", i, deviceInfo.mountPoint, mList[i]->m_DeviceInfo.deviceID));
                found_entry = true;
            }
            if(found_entry) {
                //update some particular properties
                //accessoryName [GMMY16-24687]
                VARTRACE(mList[i]->m_DeviceInfo.accessoryName);
                strncpy_r(mList[i]->m_DeviceInfo.accessoryName, deviceInfo.accessoryName, sizeof(mList[i]->m_DeviceInfo.accessoryName));

                //connectionCount [GMMY17-12353]
                mList[i]->m_DeviceInfo.connectionCount = deviceInfo.connectionCount;
                return;
            }
        } else {
            ETG_TRACE_ERR(("Null pointer or empty mount point in iPod handle list"));
        }
    }
    ETG_TRACE_ERR(("UpdateHandle failed - no entry"));
}

vector<string> iPodControlHandleMap::GetAll()
{
    ENTRY_INTERNAL;
    vector<string> list;
    Locker locker(&m_Mutex);
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i]) {
            list.push_back(string(mList[i]->m_DeviceInfo.mountPoint));
        }
    }
    return list;
}

void iPodControlHandleMap::Reset(const tMountPoint mountPoint, const bool all) {
    ENTRY_INTERNAL;
    LockIAP2(mountPoint);
    LockIndexer(mountPoint);

    Locker locker(&m_Mutex);
    ResetPrivate(FindIndex(mountPoint), all);

    UnlockIndexer(mountPoint);
    UnlockIAP2(mountPoint);
}

void iPodControlHandleMap::ResetElapsedTime(
        const tMountPoint mountPoint, const tiPodElaspedTime elapsedtime) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    ResetElapsedTimePrivate(FindIndex(mountPoint), elapsedtime);
}

tBoolean iPodControlHandleMap::IsTimeElapsed( const tMountPoint mountPoint,
        const tiPodElaspedTime elapsedtime, const tPlaytime interval) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return IsTimeElapsedPrivate(FindIndex(mountPoint), elapsedtime, interval);
}

void iPodControlHandleMap::SuspendRemainingTime(const tMountPoint mountPoint,
        const tiPodElaspedTime elapsedtime, const unsigned int suspendlimit) {
    ENTRY_INTERNAL;
    m_Mutex.lock();
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        int delay = suspendlimit - mList[indx]->GetElapsedTimeMS(elapsedtime);
        int total_delay = 0;
        bool iPod_valid = mList[indx]->m_iPodID > 0 || mList[indx]->mp_IAP2Device;

        //delay processing of iap call (specified by elapsedtime) according to apple ATS requirements
        while (iPod_valid && delay > 0 && total_delay < IPODCONTROL_SUSPEND_MAX_MS) {

            m_Mutex.unlock();

            ETG_TRACE_USR3(("Suspending (type %d) remaining %d/%d ms", (int)elapsedtime, delay, suspendlimit));
            int sleepMS = (delay > IPODCONTROL_SLEEP_INTERVAL_MS) ? IPODCONTROL_SLEEP_INTERVAL_MS : (delay % IPODCONTROL_SLEEP_INTERVAL_MS);
            delay -= sleepMS;
            total_delay += sleepMS;
            usleep(sleepMS * 1000);

            m_Mutex.lock();

            //update from map
            indx = FindIndex(mountPoint);
            if (indx >= 0 && indx < (int) mList.size()) {
                iPod_valid = mList[indx]->m_iPodID > 0 || mList[indx]->mp_IAP2Device;
                delay = suspendlimit - mList[indx]->GetElapsedTimeMS(elapsedtime);
            } else {
                ETG_TRACE_ERR(("Index out of range"));
                iPod_valid = false;
            }
        }
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    m_Mutex.unlock();
}

int iPodControlHandleMap::CreateConfiOSApp(IPOD_IOS_APP * &pApps,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return CreateConfiOSAppPrivate(pApps, FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::AddAppInfo(const tMountPoint mountPoint,
        const tProtocolName protocol, const tBundleSeedID bundleID,
        const tAppName appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return AddAppInfoPrivate(FindIndex(mountPoint), protocol, bundleID, appName);
}

vector<tIPODAppInfo> iPodControlHandleMap::GetAppInfos(const tMountPoint mountPoint, const bool hostmode, const bool nativeTransport) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetAppInfosPrivate(FindIndex(mountPoint), hostmode, nativeTransport);
}

tBoolean iPodControlHandleMap::RemoveAppInfo(const tMountPoint mountPoint,
        const tAppName appName, tSessionID &sessionID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return RemoveAppInfoPrivate(FindIndex(mountPoint), appName, sessionID);
}

tBoolean iPodControlHandleMap::AddAppInfoFromSPI(const tMountPoint mountPoint,
        const tProtocolName protocol, const tBundleSeedID bundleID,
        const tAppName appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return AddAppInfoFromSPIPrivate(FindIndex(mountPoint), protocol, bundleID, appName);
}
vector<tIPODAppInfo> iPodControlHandleMap::GetAppInfosFromSPI(const tMountPoint mountPoint ) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetAppInfosFromSPIPrivate(FindIndex(mountPoint));
}
void iPodControlHandleMap::ClearAppInfofromSPI(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    ClearAppInfofromSPIPrivate(FindIndex(mountPoint));//, appName, sessionID);
}
void iPodControlHandleMap::TraceOptions(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    TraceOptionsPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetiPodID(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_iPodID;
    } else {
        ETG_TRACE_USR3(("iPodControlHandleMap::GetiPodID() Index out of range"));
    }
    return -1;
}

int iPodControlHandleMap::GetiPodID(const tDeviceID deviceID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    /* search for the element */
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i] && mList[i]->m_DeviceInfo.deviceID == deviceID) {
            return mList[i]->m_iPodID;
        }
    }
    ETG_TRACE_ERR(("iPodControlHandleMap::GetiPodID deviceID not found"));
    return -1;
}

void * iPodControlHandleMap::GetIAP2Device(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->mp_IAP2Device;
    } else {
        ETG_TRACE_USR3(("Index out of range"));
    }
    return NULL;
}

void * iPodControlHandleMap::GetIAP2InitParameter(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->mp_IAP2InitParameter;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return NULL;
}

void * iPodControlHandleMap::GetIAP2AlbumArtBuffer(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->mp_IAP2AlbumArtBuffer;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return NULL;
}

void * iPodControlHandleMap::GetQueueListBuffer(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->mp_QueueListBuffer;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return NULL;
}

Lock * iPodControlHandleMap::GetIAP2Mutex(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    Lock * pMutex = NULL;
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        pMutex = &mList[indx]->m_IAP2Mutex;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetIAP2Mutex() Index out of range"));
    }
    return pMutex;
}

void iPodControlHandleMap::LockIAP2(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Lock * pMutex = GetIAP2Mutex(mountPoint);
    if(pMutex) {
        pMutex->lock();
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::LockIAP2() failed"));
    }
}

void iPodControlHandleMap::UnlockIAP2(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Lock * pMutex = GetIAP2Mutex(mountPoint);
    if(pMutex) {
        pMutex->unlock();
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::UnlockIAP2() failed"));
    }
}

void iPodControlHandleMap::GetMountPoint(tMountPoint &mountPoint,
        const int iPodID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    mountPoint[0] = 0;
    int indx = FindIndex(iPodID);
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mountPoint, mList[indx]->m_DeviceInfo.mountPoint, sizeof(mountPoint));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetMountPoint() Index out of range"));
    }
}

void iPodControlHandleMap::GetMountPoint(tMountPoint &mountPoint,
        const void * iap2Device) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    mountPoint[0] = 0;
    int indx = FindIndex(iap2Device);
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mountPoint, mList[indx]->m_DeviceInfo.mountPoint, sizeof(mountPoint));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetMountPoint() Index out of range"));
    }
}

void iPodControlHandleMap::GetMountPointByPEHandle(tMountPoint &mountPoint,
        const tPEHandle handle) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    /* search for the element */
    mountPoint[0] = 0;
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i] && mList[i]->m_PEHandle == handle) {
            strncpy_r(mountPoint, mList[i]->m_DeviceInfo.mountPoint, sizeof(mountPoint));
            return;
        }
    }
    ETG_TRACE_ERR(("iPodControlHandleMap::iAPGetMountPointByPEHandle handle not found"));
}

void iPodControlHandleMap::GetMountPointFromDeviceID(tMountPoint &mountPoint,
        const tDeviceID deviceID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    mountPoint[0] = 0;
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i] && mList[i]->m_DeviceInfo.mountPoint[0]) {
            if (mList[i]->m_DeviceInfo.deviceID == deviceID) {
                strncpy_r(mountPoint, mList[i]->m_DeviceInfo.mountPoint, sizeof(mountPoint));
                return;
            }
        } else {
            ETG_TRACE_ERR(("Null pointer or empty mount point in iPod handle list"));
        }
    }
}

void iPodControlHandleMap::GetDeviceInfo(const tMountPoint mountPoint, tDeviceInfo &deviceInfo)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetDeviceInfoPrivate(FindIndex(mountPoint), deviceInfo);
}

bool iPodControlHandleMap::GetOption(const tMountPoint mountPoint,
        const tiPodOptions option) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetOptionPrivate(FindIndex(mountPoint), option);
}

iPodControlMediaPath iPodControlHandleMap::GetDBPath(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetDBPathPrivate(FindIndex(mountPoint));
}

iPodControlMediaPath iPodControlHandleMap::GetPBPath(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPBPathPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetPlayerState(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPlayerStatePrivate(FindIndex(mountPoint));
}

tPEPlaybackState iPodControlHandleMap::GetPEPlaybackState(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPEPlaybackStatePrivate(FindIndex(mountPoint));
}

tPEPlaybackState iPodControlHandleMap::GetLastPlaybackState(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetLastPlaybackStatePrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetNowPlayingTrackIndex(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetNowPlayingTrackIndexPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetNowPlayingTrackCount(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetNowPlayingTrackCountPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetNowPlayingChapterIndex(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetNowPlayingChapterIndexPrivate(FindIndex(mountPoint));
}

void iPodControlHandleMap::GetNowPlayingMediaObject(tMediaObject &mediaObject, const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetNowPlayingMediaObjectPrivate(mediaObject, FindIndex(mountPoint));
}

int iPodControlHandleMap::GetPollThreadIndex(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPollThreadIndexPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetPBTrackIndex(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPBTrackIndexPrivate(FindIndex(mountPoint));
}

tPlaytime iPodControlHandleMap::GetElapsedPlaytime(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetElapsedPlaytimePrivate(FindIndex(mountPoint));
}

tPlaytime iPodControlHandleMap::GetTotalPlaytime(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetTotalPlaytimePrivate(FindIndex(mountPoint));
}

void iPodControlHandleMap::GetDeviceName(tDeviceName &name,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetDeviceNamePrivate(name, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetLaunchApp(tAppName &name,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetLaunchAppPrivate(name, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetFocusApp(tAppName &name,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetFocusAppPrivate(name, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetOTGPath(tPath &path,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetOTGPathPrivate(path, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetPowerPortPath(tPath &path,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetPowerPortPathPrivate(path, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetUDCDevice(tPath &udc,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetUDCDevicePrivate(udc, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetHubVendorID(tPath &vendor,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetHubVendorIDPrivate(vendor, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetHubProductID(tPath &product,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetHubProductIDPrivate(product, FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::GetRoleSwitched(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetRoleSwitchedPrivate(FindIndex(mountPoint));
}

tConnectionState iPodControlHandleMap::GetConnectionState(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetConnectionStatePrivate(FindIndex(mountPoint));
}

tConnectionType iPodControlHandleMap::GetConnectionType(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetConnectionTypePrivate(FindIndex(mountPoint));
}

tDeviceID iPodControlHandleMap::GetDeviceID(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetDeviceIDPrivate(FindIndex(mountPoint));
}

me::samplerate_i iPodControlHandleMap::GetSampleRate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetSampleRatePrivate(FindIndex(mountPoint));
}

tBitRate iPodControlHandleMap::GetBitRate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetBitRatePrivate(FindIndex(mountPoint));
}

tCodecType iPodControlHandleMap::GetCodecType(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetCodecTypePrivate(FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::GetLocationInformationUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetLocationInformationUpdatePrivate(FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::GetVehicleStatusUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetVehicleStatusUpdatePrivate(FindIndex(mountPoint));
}

iPodControlIndexerContext * iPodControlHandleMap::GetIndexerContext(
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetIndexerContextPrivate(FindIndex(mountPoint));
}

Lock * iPodControlHandleMap::GetIndexerMutex(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    Lock * pMutex = NULL;
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        pMutex = &mList[indx]->m_IndexerMutex;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return pMutex;
}

void iPodControlHandleMap::LockIndexer(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Lock * pMutex = GetIndexerMutex(mountPoint);
    if(pMutex) {
        pMutex->lock();
    } else {
        ETG_TRACE_ERR(("LockIndexer() failed"));
    }
}

void iPodControlHandleMap::UnlockIndexer(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Lock * pMutex = GetIndexerMutex(mountPoint);
    if(pMutex) {
        pMutex->unlock();
    } else {
        ETG_TRACE_ERR(("UnlockIndexer() failed"));
    }
}

int iPodControlHandleMap::GetAlbumArtID(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetAlbumArtIDPrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetQueueListID(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetQueueListIDPrivate(FindIndex(mountPoint));
}

IPOD_OTG_TYPE iPodControlHandleMap::GetOTGType(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetOTGTypePrivate(FindIndex(mountPoint));
}

int iPodControlHandleMap::GetOTGPortNumber(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetOTGPortNumberPrivate(FindIndex(mountPoint));
}

void iPodControlHandleMap::GetAlbumArtString(tAlbumArt &albumArt,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetAlbumArtStringPrivate(albumArt, FindIndex(mountPoint));
}

tSessionID iPodControlHandleMap::GetSessionIDFromAppName(
        const tMountPoint mountPoint, const tAppName appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetSessionIDFromAppNamePrivate(FindIndex(mountPoint), appName);
}

unsigned int iPodControlHandleMap::GetProtocolIDFromAppName(
        const tMountPoint mountPoint, const tAppName appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetProtocolIDFromAppNamePrivate(FindIndex(mountPoint), appName);
}

void iPodControlHandleMap::GetActiveSessionList(vector<string> &list,
        const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetActiveSessionListPrivate(list, FindIndex(mountPoint));
}

void iPodControlHandleMap::GetAppNameBySessionID(const tMountPoint mountPoint,
        const tSessionID sessionId, tAppName &appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetAppNameBySessionIDPrivate(FindIndex(mountPoint), sessionId, appName);
}

void iPodControlHandleMap::GetAppNameByProtocolID(const tMountPoint mountPoint,
        const unsigned char protocolId, tAppName &appName, tSessionID &sessionId) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetAppNameByProtocolIDPrivate(FindIndex(mountPoint), protocolId, appName, sessionId);
}
void iPodControlHandleMap::GetProtocolNameByProtocolID(const tMountPoint mountPoint,
        const unsigned char protocolId, tProtocolName &protocolName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetProtocolNameByProtocolIDPrivate(FindIndex(mountPoint), protocolId, protocolName);
}


tBoolean iPodControlHandleMap::GetTrackFinished(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetTrackFinishedPrivate(FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::GetStreamingMode(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetStreamingModePrivate(FindIndex(mountPoint));
}

tPEHandle iPodControlHandleMap::GetPEHandle(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPEHandlePrivate(FindIndex(mountPoint));
}

tPlaybackMode iPodControlHandleMap::GetPlaybackMode(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPlaybackModePrivate(FindIndex(mountPoint));
}

tRepeatMode iPodControlHandleMap::GetRepeatMode(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetRepeatModePrivate(FindIndex(mountPoint));
}

bool iPodControlHandleMap::GetRepeatInitFlag(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetRepeatInitFlagPrivate(FindIndex(mountPoint));
}

tDiPOCallState iPodControlHandleMap::GetCallState(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetCallStatePrivate(FindIndex(mountPoint));
}

unsigned int iPodControlHandleMap::GetCallDuration(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetCallDurationPrivate(FindIndex(mountPoint));
}
tDiPOCallStateItem iPodControlHandleMap::GetCallStateByUUID(const tMountPoint mountPoint, const tGeneralString callUUID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);

    tDiPOCallState list = GetCallStatePrivate(FindIndex(mountPoint));
    for(unsigned int i = 0; i < list.size(); i++) {
        if(!strcmp(callUUID, list.at(i).callUUID)) {
            return list.at(i);

        }
    }

    //return new item
    tDiPOCallStateItem item;
    InitDiPOCallStateItem(item);
    strncpy_r(item.callUUID, callUUID, sizeof(item.callUUID));
    return item;
}

tDiPOCommunications iPodControlHandleMap::GetCommunications(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetCommunicationsPrivate(FindIndex(mountPoint));
}
void iPodControlHandleMap::GetDiPOLocationInfo(const tMountPoint mountPoint,tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetDiPOLocationInfoPrivate(FindIndex(mountPoint),startStopLocationInfo,locationInfoType);
}

void iPodControlHandleMap::GetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    GetDiPOGPRMCDataStatusValuesPrivate(FindIndex(mountPoint),GPRMCDataStatusValues);
}

tBTProfile iPodControlHandleMap::GetBTProfile(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetBTProfilePrivate(FindIndex(mountPoint));
}

IPOD_HID_REPORT iPodControlHandleMap::GetLastHIDReport(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetLastHIDReportPrivate(FindIndex(mountPoint));
}

tDiPOPower iPodControlHandleMap::GetPowerUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetPowerUpdatePrivate(FindIndex(mountPoint));
}

tDiPORouteGuidanceUpdate iPodControlHandleMap::PullRGUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return PullRGUpdatePrivate(FindIndex(mountPoint));
}

tDiPORouteGuidanceManeuverUpdate iPodControlHandleMap::PullRGManeuverUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return PullRGManeuverUpdatePrivate(FindIndex(mountPoint));
}

tDiPORouteGuidanceUpdate iPodControlHandleMap::GetRGUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetRGUpdatePrivate(FindIndex(mountPoint));
}

tDiPORouteGuidanceManeuverUpdate iPodControlHandleMap::GetRGManeuverUpdate(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetRGManeuverUpdatePrivate(FindIndex(mountPoint));
}

tDiPOCaps iPodControlHandleMap::GetLastWirelessCarplayStatus(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetLastWirelessCarplayStatusPrivate(FindIndex(mountPoint));
}

tBTLimitationActionState iPodControlHandleMap::GetBTLimitationActionState(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetBTLimitationActionStatePrivate(FindIndex(mountPoint));
}

tConnectionState iPodControlHandleMap::GetBTLimitationModeConnectionState(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetBTLimitationModeConnectionStatePrivate(FindIndex(mountPoint));
}

tBoolean iPodControlHandleMap::GetIsCPWActiveForUSBMountPoint(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetIsCPWActiveForUSBMountPointPrivate(FindIndex(mountPoint));
}

void iPodControlHandleMap::GetCPWMountPointbyUSBMountPoint(const tMountPoint mountPoint, tMountPoint& CPWMountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);

    /* search for the element */
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i]) {
            if (!strncmp(mList[i]->m_DeviceInfo.appleDeviceUSBSerialNumber,mountPoint,sizeof(mList[i]->m_DeviceInfo.appleDeviceUSBSerialNumber)) && (DCT_WIFI == mList[i]->m_DeviceInfo.connectionType)) {
                strncpy_r(CPWMountPoint, mList[i]->m_DeviceInfo.mountPoint, sizeof(CPWMountPoint));
                return;
            }
        } else {
            ETG_TRACE_ERR(("Null pointer in iPod handle list"));
        }
    }
    ETG_TRACE_ERR(("iPodControlHandleMap::GetCPWMountPointbyUSBMountPoint handle not found"));
}

void iPodControlHandleMap::GetUSBMountPointbyCPWMountPoint(const tMountPoint mountPoint, tMountPoint& USBMountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);

    // Get deviceUUID of device with Wifi mountpoint
    tUUID deviceUUID = {0};
    tInteger index = FindIndex(mountPoint);
    if ((index >= 0) && (index < (int) mList.size()))
    {
        GetDeviceUUIDPrivate(FindIndex(mountPoint), deviceUUID);
    }
    else
    {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetUSBMountPointbyCPWMountPoint(): Error:  mount point index is invalid"));
    }

    /* search for the element */
    for (unsigned int i = 0; i < mList.size(); i++)
    {
        if (mList[i])
        {
            if ( (0 == strncmp(mList[i]->m_DeviceInfo.deviceUUID, deviceUUID, sizeof(deviceUUID)) )
                    && (DCT_USB == mList[i]->m_DeviceInfo.connectionType))
            {
                strncpy_r(USBMountPoint, mList[i]->m_DeviceInfo.mountPoint, sizeof(USBMountPoint));
                break;
            }
        }
        else
        {
            ETG_TRACE_ERR(("Null pointer in iPod handle list"));
        }
    }

    if( 0 == USBMountPoint[0] )
    {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetUSBMountPointbyCPWMountPoint handle not found"));
    }
}

void iPodControlHandleMap::GetDeviceUUID(const tMountPoint mountPoint, tUUID& deviceUUID)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    tInteger index = FindIndex(mountPoint);
    if ((index >= 0) && (index < (int) mList.size()))
    {
        GetDeviceUUIDPrivate(FindIndex(mountPoint), deviceUUID);
    }
    else
    {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetDeviceUUID(): Error:  mount point index is invalid"));
    }
}

void iPodControlHandleMap::GetDeviceUUIDPrivate(const int indx, tUUID& deviceUUID)
{
    ENTRY_INTERNAL;
    strncpy_r(deviceUUID, mList[indx]->m_DeviceInfo.deviceUUID, sizeof(mList[indx]->m_DeviceInfo.deviceUUID));
}

tDiPODeviceTime iPodControlHandleMap::GetDeviceTimeUpdate(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetDeviceTimeUpdatePrivate(FindIndex(mountPoint));
}

bool iPodControlHandleMap::GetWaitingforRoleSwitchResponse(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    return GetWaitingforRoleSwitchResponsePrivate(FindIndex(mountPoint));
}

//Setter
void iPodControlHandleMap::SetDeviceUUID(const tMountPoint mountPoint,
        const tUUID deviceUUID)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size())
    {
        strncpy_r(mList[indx]->m_DeviceInfo.deviceUUID, deviceUUID, sizeof(mList[indx]->m_DeviceInfo.deviceUUID));
    }
    else
    {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetDeviceUUID() Index out of range"));
    }
}

void iPodControlHandleMap::SetiPodID(const tMountPoint mountPoint,
        const int iPodID) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_iPodID = iPodID;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetiPodID() Index out of range"));
    }
}

void iPodControlHandleMap::SetIAP2Device(const tMountPoint mountPoint, void * iap2Device) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->mp_IAP2Device = iap2Device;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetIAP2InitParameter(const tMountPoint mountPoint, void * iap2InitParameter) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->mp_IAP2InitParameter = iap2InitParameter;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetIAP2AlbumArtBuffer(const tMountPoint mountPoint, void * iap2AlbumArtBuffer) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->mp_IAP2AlbumArtBuffer = iap2AlbumArtBuffer;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetQueueListBuffer(const tMountPoint mountPoint, void * buffer) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    int indx = FindIndex(mountPoint);
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->mp_QueueListBuffer = buffer;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetOption(const tMountPoint mountPoint,
        const tiPodOptions option, const bool set) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetOptionPrivate(FindIndex(mountPoint), option, set);
}

void iPodControlHandleMap::SetDBPath(const tMountPoint mountPoint,
        const iPodControlMediaPath path) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetDBPathPrivate(FindIndex(mountPoint), path);
}

void iPodControlHandleMap::SetPBPath(const tMountPoint mountPoint,
        const iPodControlMediaPath path) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPBPathPrivate(FindIndex(mountPoint), path);
}

void iPodControlHandleMap::SetPlayerState(const tMountPoint mountPoint,
        const int state) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPlayerStatePrivate(FindIndex(mountPoint), state);
}

void iPodControlHandleMap::SetPEPlaybackState(const tMountPoint mountPoint,
        const tPEPlaybackState state) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPEPlaybackStatePrivate(FindIndex(mountPoint), state);
}

void iPodControlHandleMap::SetLastPlaybackState(const tMountPoint mountPoint,
        const tPEPlaybackState state) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetLastPlaybackStatePrivate(FindIndex(mountPoint), state);
}

void iPodControlHandleMap::SetNowPlayingTrackIndex(
        const tMountPoint mountPoint, const int track) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetNowPlayingTrackIndexPrivate(FindIndex(mountPoint), track);
}

void iPodControlHandleMap::SetNowPlayingTrackCount(
        const tMountPoint mountPoint, const int count) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetNowPlayingTrackCountPrivate(FindIndex(mountPoint), count);
}

void iPodControlHandleMap::SetNowPlayingChapterIndex(
        const tMountPoint mountPoint, const int track) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetNowPlayingChapterIndexPrivate(FindIndex(mountPoint), track);
}

void iPodControlHandleMap::SetNowPlayingTags(const tMountPoint mountPoint,
        const tMediaType mediatype, const tMetadata MetadataField1,
        const tMetadata MetadataField2, const tMetadata MetadataField3,
        const tMetadata MetadataField4, const tTitle title, const tUUID uuid) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetNowPlayingTagsPrivate(mediatype, MetadataField1, MetadataField2, MetadataField3, MetadataField4, title, uuid, FindIndex(mountPoint));
}

void iPodControlHandleMap::SetPollThreadIndex(const tMountPoint mountPoint,
        const int threadIndex) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPollThreadIndexPrivate(FindIndex(mountPoint), threadIndex);
}

void iPodControlHandleMap::SetPBTrackIndex(const tMountPoint mountPoint,
        const int track) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPBTrackIndexPrivate(FindIndex(mountPoint), track);
}

void iPodControlHandleMap::SetElapsedPlaytime(const tMountPoint mountPoint,
        const tPlaytime tme) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetElapsedPlaytimePrivate(FindIndex(mountPoint), tme);
}

void iPodControlHandleMap::SetTotalPlaytime(const tMountPoint mountPoint,
        const tPlaytime tme) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetTotalPlaytimePrivate(FindIndex(mountPoint), tme);
}

void iPodControlHandleMap::SetDeviceName(const tMountPoint mountPoint,
        const char* name) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetDeviceNamePrivate(FindIndex(mountPoint), name);
}

void iPodControlHandleMap::SetLaunchApp(const tMountPoint mountPoint,
        const char* name) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetLaunchAppPrivate(FindIndex(mountPoint), name);
}

void iPodControlHandleMap::SetFocusApp(const tMountPoint mountPoint,
        const char* name) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetFocusAppPrivate(FindIndex(mountPoint), name);
}

void iPodControlHandleMap::SetOTGPath(const tMountPoint mountPoint,
        const char* path) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetOTGPathPrivate(FindIndex(mountPoint), path);
}

void iPodControlHandleMap::SetPowerPortPath(const tMountPoint mountPoint,
        const char* path) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPowerPortPathPrivate(FindIndex(mountPoint), path);
}

void iPodControlHandleMap::SetUDCDevice(const tMountPoint mountPoint,
        const char* udc) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetUDCDevicePrivate(FindIndex(mountPoint), udc);
}

void iPodControlHandleMap::SetHubVendorID(const tMountPoint mountPoint,
        const char* vendor) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetHubVendorIDPrivate(FindIndex(mountPoint), vendor);
}

void iPodControlHandleMap::SetHubProductID(const tMountPoint mountPoint,
        const char* product) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetHubProductIDPrivate(FindIndex(mountPoint), product);
}

void iPodControlHandleMap::SetRoleSwitched(const tMountPoint mountPoint,
        const tBoolean state) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetRoleSwitchedPrivate(FindIndex(mountPoint), state);
}

void iPodControlHandleMap::SetConnectionState(const tMountPoint mountPoint,
        const tConnectionState state) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetConnectionStatePrivate(FindIndex(mountPoint), state);
}

void iPodControlHandleMap::SetDeviceID(const tMountPoint mountPoint,
        const tDeviceID id) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetDeviceIDPrivate(FindIndex(mountPoint), id);
}

void iPodControlHandleMap::SetSampleRate(const tMountPoint mountPoint,
        const me::samplerate_i sampleRate) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetSampleRatePrivate(FindIndex(mountPoint), sampleRate);
}

void iPodControlHandleMap::SetBitRate(const tMountPoint mountPoint,
        const tBitRate bitRate) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetBitRatePrivate(FindIndex(mountPoint), bitRate);
}

void iPodControlHandleMap::SetCodecType(const tMountPoint mountPoint,
        const tCodecType codecType) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetCodecTypePrivate(FindIndex(mountPoint), codecType);
}

void iPodControlHandleMap::SetLocationInformationUpdate(const tMountPoint mountPoint,
        const tBoolean locationInformationUpdate) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetLocationInformationUpdatePrivate(FindIndex(mountPoint), locationInformationUpdate);
}

void iPodControlHandleMap::SetVehicleStatusUpdate(const tMountPoint mountPoint,
        const tBoolean vehicleStatusUpdate) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetVehicleStatusUpdatePrivate(FindIndex(mountPoint), vehicleStatusUpdate);
}

void iPodControlHandleMap::SetAlbumArtID(const tMountPoint mountPoint,
        const int format) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetAlbumArtIDPrivate(FindIndex(mountPoint), format);
}

void iPodControlHandleMap::SetQueueListID(const tMountPoint mountPoint,
        const int format) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetQueueListIDPrivate(FindIndex(mountPoint), format);
}

void iPodControlHandleMap::SetOTGType(const tMountPoint mountPoint,
        const IPOD_OTG_TYPE type) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetOTGTypePrivate(FindIndex(mountPoint), type);
}

void iPodControlHandleMap::SetOTGPortNumber(const tMountPoint mountPoint,
        const int port) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetOTGPortNumberPrivate(FindIndex(mountPoint), port);
}

void iPodControlHandleMap::SetLastHIDReport(const tMountPoint mountPoint,
        const IPOD_HID_REPORT report) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetLastHIDReportPrivate(FindIndex(mountPoint), report);
}

void iPodControlHandleMap::SetAlbumArtString(const tMountPoint mountPoint,
        const tAlbumArt albumArt) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetAlbumArtStringPrivate(FindIndex(mountPoint), albumArt);
}

void iPodControlHandleMap::SetSessionIDByIndex(const tMountPoint mountPoint,
        const tSessionID sessionId, const unsigned char protocolIndex,
        tAppName &appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetSessionIDByIndexPrivate(FindIndex(mountPoint), sessionId, protocolIndex,
            appName);
}

void iPodControlHandleMap::ClearSessionID(const tMountPoint mountPoint,
        const tSessionID sessionId, tAppName &appName) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    ClearSessionIDPrivate(FindIndex(mountPoint), sessionId, appName);
}

void iPodControlHandleMap::SetTrackFinished(const tMountPoint mountPoint,
        const tBoolean set) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetTrackFinishedPrivate(FindIndex(mountPoint), set);
}

void iPodControlHandleMap::SetStreamingMode(const tMountPoint mountPoint,
        const tBoolean set) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetStreamingModePrivate(FindIndex(mountPoint), set);
}

void iPodControlHandleMap::SetPEHandle(const tMountPoint mountPoint,
        const tPEHandle handle) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPEHandlePrivate(FindIndex(mountPoint), handle);
}

void iPodControlHandleMap::SetPlaybackMode(const tMountPoint mountPoint,
        const tPlaybackMode mode) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetPlaybackModePrivate(FindIndex(mountPoint), mode);
}

void iPodControlHandleMap::SetRepeatMode(const tMountPoint mountPoint,
        const tRepeatMode mode) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetRepeatModePrivate(FindIndex(mountPoint), mode);
}

void iPodControlHandleMap::SetRepeatInitFlag(const tMountPoint mountPoint,
        const bool flag) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetRepeatInitFlagPrivate(FindIndex(mountPoint), flag);
}

void iPodControlHandleMap::SetBTProfile(const tMountPoint mountPoint,
        const tBTProfile btProfile) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetBTProfilePrivate(FindIndex(mountPoint), btProfile);
}

void iPodControlHandleMap::SetNowPlayingMediaObject(const tMediaObject mediaObject, const tMountPoint mountPoint) {
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetNowPlayingMediaObjectPrivate(mediaObject, FindIndex(mountPoint));
}

void iPodControlHandleMap::UpdateOrInsertCallState(const tMountPoint mountPoint, const tDiPOCallStateItem callStateItem)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    UpdateOrInsertCallStatePrivate(FindIndex(mountPoint), callStateItem);
}
void iPodControlHandleMap::UpdateCallDuration(const tMountPoint mountPoint, const U32 callDuration)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    UpdateCallDurationPrivate(FindIndex(mountPoint), callDuration);
}

void iPodControlHandleMap::RemoveDisconnectedCallState(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    RemoveDisconnectedCallStatePrivate(FindIndex(mountPoint));
}

void iPodControlHandleMap::SetCommunications(const tMountPoint mountPoint, const tDiPOCommunications comm)
{
    Locker locker(&m_Mutex);
    SetCommunicationsPrivate(FindIndex(mountPoint), comm);
}

void iPodControlHandleMap::SetDiPOLocationInfo(const tMountPoint mountPoint, const tBool startStopLocationInfo, const tDiPOLocationInfoType locationInfoType)
{
    Locker locker(&m_Mutex);
    SetDiPOLocationInfoPrivate(FindIndex(mountPoint),startStopLocationInfo,locationInfoType);
}

void iPodControlHandleMap::SetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint,const tDiPOGPRMCDataStatusValues GPRMCDataStatusValues)
{
    Locker locker(&m_Mutex);
    SetDiPOGPRMCDataStatusValuesPrivate(FindIndex(mountPoint),GPRMCDataStatusValues);
}

void iPodControlHandleMap::SetPowerUpdate(const tMountPoint mountPoint, const tDiPOPower power)
{
    Locker locker(&m_Mutex);
    SetPowerUpdatePrivate(FindIndex(mountPoint), power);
}

void iPodControlHandleMap::PushRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate)
{
    Locker locker(&m_Mutex);
    PushRGUpdatePrivate(FindIndex(mountPoint), RGUpdate);
}
void iPodControlHandleMap::PushRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate)
{
    Locker locker(&m_Mutex);
    PushRGManeuverUpdatePrivate(FindIndex(mountPoint), RGManeuverUpdate);
}

void iPodControlHandleMap::SetRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate)
{
    Locker locker(&m_Mutex);
    SetRGUpdatePrivate(FindIndex(mountPoint), RGUpdate);
}
void iPodControlHandleMap::SetRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate)
{
    Locker locker(&m_Mutex);
    SetRGManeuverUpdatePrivate(FindIndex(mountPoint), RGManeuverUpdate);
}

void iPodControlHandleMap::SetLastWirelessCarplayStatus(const tMountPoint mountPoint, tDiPOCaps diPOCaps)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetLastWirelessCarplayStatusPrivate(FindIndex(mountPoint), diPOCaps);
}

void iPodControlHandleMap::SetBTLimitationActionState(const tMountPoint mountPoint, tBTLimitationActionState btLimitation)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetBTLimitationActionStatePrivate(FindIndex(mountPoint), btLimitation);
}

void iPodControlHandleMap::SetBTLimitationModeConnectionState(const tMountPoint mountPoint, tConnectionState connectionState)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetBTLimitationModeConnectionStatePrivate(FindIndex(mountPoint), connectionState);
}

void iPodControlHandleMap::SetIsCPWActiveForUSBMountpoint(const tMountPoint mountPoint, const tBoolean isCPWActive)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetIsCPWActiveForUSBMountpointPrivate(FindIndex(mountPoint), isCPWActive);
}

void iPodControlHandleMap::SetAppleDeviceTransportIdentifiers(const tMountPoint mountpoint,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetAppleDeviceTransportIdentifiersPrivate(FindIndex(mountpoint),appleDeviceMACAddress,appleDeviceUSBSerialNumber);
}

void iPodControlHandleMap::SetDeviceTimeUpdate(const tMountPoint mountPoint, const tDiPODeviceTime deviceTime)
{
    Locker locker(&m_Mutex);
    SetDeviceTimeUpdatePrivate(FindIndex(mountPoint), deviceTime);
}

void iPodControlHandleMap::SetWaitingforRoleSwitchResponse(const tMountPoint mountPoint, bool isWaitingforRoleSwitchResponse)
{
    ENTRY_INTERNAL;
    Locker locker(&m_Mutex);
    SetWaitingforRoleSwitchResponsePrivate(FindIndex(mountPoint), isWaitingforRoleSwitchResponse);
}

//PRIVATE Helpers
int iPodControlHandleMap::FindIndex(const tMountPoint mountPoint) {
    ENTRY_INTERNAL;

    /* search for the element */
    for (unsigned int i = 0; i < mList.size(); i++) {
        if (mList[i] && mList[i]->m_DeviceInfo.mountPoint[0]) {
            if (!strcmp(mList[i]->m_DeviceInfo.mountPoint, mountPoint)) {
                return i;
            }
        } else {
            ETG_TRACE_ERR(("Null pointer or empty mount point at %d in iPod handle list", i));
        }
    }
    //error or not found
    return -1;
}

int iPodControlHandleMap::FindIndex(const int iPodID) {
    ENTRY_INTERNAL;

    if (iPodID > 0) {
        /* search for the element */
        for (unsigned int i = 0; i < mList.size(); i++) {
            if (mList[i]) {
                if (mList[i]->m_iPodID == iPodID) {
                    return i;
                }
            } else {
                ETG_TRACE_ERR(("Null pointer in iPod handle list"));
            }
        }
    } else {
        ETG_TRACE_ERR(("Invalid iPod ID"));
    }
    //error or not found
    return -1;
}

int iPodControlHandleMap::FindIndex(const void * iap2Device) {
    ENTRY_INTERNAL;

    if (iap2Device) {
        /* search for the element */
        for (unsigned int i = 0; i < mList.size(); i++) {
            if (mList[i]) {
                if (mList[i]->mp_IAP2Device == iap2Device) {
                    return i;
                }
            } else {
                ETG_TRACE_ERR(("Null pointer in iPod handle list"));
            }
        }
    } else {
        ETG_TRACE_ERR(("Invalid iap2Device"));
    }
    //error or not found
    return -1;
}
void iPodControlHandleMap::ResetPrivate(const int indx, const bool all) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->Reset(all);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::ResetPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::ResetElapsedTimePrivate(const int indx,
        const tiPodElaspedTime elapsedtime) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->ResetElapsedTime(elapsedtime);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::ResetElapsedTimePrivate() Index out of range"));
    }
}

tBoolean iPodControlHandleMap::IsTimeElapsedPrivate(const int indx,
        const tiPodElaspedTime elapsedtime, const tPlaytime elapsed) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetElapsedTimeMS(elapsedtime) > elapsed;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::IsTimeElapsedPrivate() Index out of range"));
    }
    return false;
}

int iPodControlHandleMap::CreateConfiOSAppPrivate(IPOD_IOS_APP * &pApps,
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->CreateConfiOSApp(pApps);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::CreateConfiOSAppPrivate() Index out of range"));
    }
    return 0;
}

tBoolean iPodControlHandleMap::AddAppInfoPrivate(const int indx,
        const tProtocolName protocol, const tBundleSeedID bundleID,
        const tAppName appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->AddAppInfo(protocol, bundleID, appName, AIO_DEVICE_MODE_ONLY);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}

vector<tIPODAppInfo> iPodControlHandleMap::GetAppInfosPrivate(const int indx, const bool hostmode, const bool nativeTransport) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetAppInfos(hostmode, nativeTransport);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    vector<tIPODAppInfo> emptylist;
    return emptylist;
}

tBoolean iPodControlHandleMap::RemoveAppInfoPrivate(const int indx,
        const tAppName appName, tSessionID &sessionID) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->RemoveAppInfo(appName, sessionID);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::RemoveAppInfoPrivate() Index out of range"));
    }
    return false;
}

tBoolean iPodControlHandleMap::AddAppInfoFromSPIPrivate(const int indx,
        const tProtocolName protocol, const tBundleSeedID bundleID,
        const tAppName appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->AddAppInfoFromSPI(protocol, bundleID, appName, AIO_EA_NATIVE_TRANSPORT);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}
vector<tIPODAppInfo> iPodControlHandleMap::GetAppInfosFromSPIPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetAppInfosFromSPI();
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    vector<tIPODAppInfo> emptylist;
    return emptylist;
}
void iPodControlHandleMap::ClearAppInfofromSPIPrivate(const int indx)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->ClearAppInfosFromSPI();//(appName, sessionID);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::RemoveAppInfoPrivate() Index out of range"));
    }
}
void iPodControlHandleMap::TraceOptionsPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->TraceOptions();
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::TraceOptionsPrivate() Index out of range"));
    }
}

bool iPodControlHandleMap::GetWaitingforRoleSwitchResponsePrivate(const int indx)
{
    ENTRY_INTERNAL;
    if(indx >= 0&& indx < (int)mList.size())
    {
        return mList[indx]->m_IsWaitingforRoleSwitchResponse;
    }
    else
    {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}

//private getters
void iPodControlHandleMap::GetDeviceInfoPrivate(const int indx, tDeviceInfo &deviceInfo) {
    ENTRY_INTERNAL;

    if (indx >= 0 && indx < (int) mList.size()) {
        deviceInfo = mList[indx]->m_DeviceInfo;
    } else {
        ETG_TRACE_USR3(("Index out of range"));
        InitDeviceInfo(deviceInfo);
    }
}

bool iPodControlHandleMap::GetOptionPrivate(const int indx,
        const tiPodOptions option) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetOption(option);
    } else {
        ETG_TRACE_USR3(("iPodControlHandleMap::GetOptionPrivate() Index out of range"));
    }
    return 0;
}

iPodControlMediaPath iPodControlHandleMap::GetDBPathPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_DBPath;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetDBPathPrivate() Index out of range"));
    }
    return iPodControlMediaPath();
}

iPodControlMediaPath iPodControlHandleMap::GetPBPathPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PBPath;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPBPathPrivate() Index out of range"));
    }
    return iPodControlMediaPath();
}

int iPodControlHandleMap::GetPlayerStatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PlayerState;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPlayerStatePrivate() Index out of range"));
    }
    return 0;
}

tPEPlaybackState iPodControlHandleMap::GetPEPlaybackStatePrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PlaybackState;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPEPlaybackStatePrivate() Index out of range"));
    }
    return PE_PBS_ERRORSTATE;
}

tPEPlaybackState iPodControlHandleMap::GetLastPlaybackStatePrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_LastPlaybackState;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return PE_PBS_ERRORSTATE;
}

int iPodControlHandleMap::GetNowPlayingTrackIndexPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingTrackIndex;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetNowPlayingTrackIndexPrivate() Index out of range"));
    }
    return -1;
}

int iPodControlHandleMap::GetNowPlayingTrackCountPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingTrackCount;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

int iPodControlHandleMap::GetNowPlayingChapterIndexPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingChapterIndex;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

void iPodControlHandleMap::GetNowPlayingMediaObjectPrivate(tMediaObject &mediaObject, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mediaObject = mList[indx]->m_NowPlayingMediaObject;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetNowPlayingMediaObjectPrivate() Index out of range"));
    }
}

int iPodControlHandleMap::GetPollThreadIndexPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PollThreadIndex;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

int iPodControlHandleMap::GetPBTrackIndexPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PBTrackIndex;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPBTrackIndexPrivate() Index out of range"));
    }
    return -1;
}

tPlaytime iPodControlHandleMap::GetElapsedPlaytimePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_ElapsedPlaytime;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return 0;
}

tPlaytime iPodControlHandleMap::GetTotalPlaytimePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingMediaObject.totalPlaytime;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetTotalPlaytimePrivate() Index out of range"));
    }
    return 0;
}

void iPodControlHandleMap::GetDeviceNamePrivate(tDeviceName &name,
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(name, mList[indx]->m_DeviceInfo.deviceName, sizeof(name));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetDeviceNamePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::GetLaunchAppPrivate(tAppName &name, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(name, mList[indx]->m_LaunchApp, sizeof(name));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetLaunchAppPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::GetFocusAppPrivate(tAppName &name, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(name, mList[indx]->m_FocusApp, sizeof(name));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetFocusAppPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::GetOTGPathPrivate(tPath &path, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(path, mList[indx]->m_OTGPath, sizeof(path));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetPowerPortPathPrivate(tPath &path, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(path, mList[indx]->m_PowerPortPath, sizeof(path));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetUDCDevicePrivate(tPath &udc, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(udc, mList[indx]->m_UDCDevice, sizeof(udc));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetHubVendorIDPrivate(tPath &vendor, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(vendor, mList[indx]->m_HubVendorID, sizeof(vendor));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetHubProductIDPrivate(tPath &product, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(product, mList[indx]->m_HubProductID, sizeof(product));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

tBoolean iPodControlHandleMap::GetRoleSwitchedPrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_RoleSwitched;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return FALSE;
}

tConnectionState iPodControlHandleMap::GetConnectionStatePrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_DeviceInfo.connectionState;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetConnectionStatePrivate() Index out of range"));
    }
    return CS_DISCONNECTED;
}

tConnectionType iPodControlHandleMap::GetConnectionTypePrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_DeviceInfo.connectionType;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return DCT_UNKNOWN;
}

tDeviceID iPodControlHandleMap::GetDeviceIDPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_DeviceInfo.deviceID;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetDeviceIDPrivate() Index out of range"));
    }
    return DEVICE_ID_NOT_SET;
}

me::samplerate_i iPodControlHandleMap::GetSampleRatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingMediaObject.sampleRate;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetSampleRatePrivate() Index out of range"));
    }
    return 0;
}

tBitRate iPodControlHandleMap::GetBitRatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_NowPlayingMediaObject.bitRate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tBitRate_init;
}

tCodecType iPodControlHandleMap::GetCodecTypePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_CodecType;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tCodecType_init;
}

tBoolean iPodControlHandleMap::GetLocationInformationUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_LocationInformationUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}

tBoolean iPodControlHandleMap::GetVehicleStatusUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_VehicleStatusUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}

iPodControlIndexerContext * iPodControlHandleMap::GetIndexerContextPrivate(
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->mp_Context;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return NULL;
}

int iPodControlHandleMap::GetAlbumArtIDPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_AlbumArtID;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

int iPodControlHandleMap::GetQueueListIDPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_QueueListID;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

IPOD_OTG_TYPE iPodControlHandleMap::GetOTGTypePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_OTGType;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return IPOD_OTG_NONE;
}

int iPodControlHandleMap::GetOTGPortNumberPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_OTGPortNumber;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return -1;
}

IPOD_HID_REPORT iPodControlHandleMap::GetLastHIDReportPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_LastHIDReport;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return IPOD_HID_REPORT_BUTTON_RELEASED;
}

void iPodControlHandleMap::GetAlbumArtStringPrivate(tAlbumArt &albumArt,
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(albumArt, mList[indx]->m_NowPlayingMediaObject.albumArtString, sizeof(albumArt));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetAlbumArtStringPrivate() Index out of range"));
    }
}

tSessionID iPodControlHandleMap::GetSessionIDFromAppNamePrivate(
        const int indx, const tAppName appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetSessionIDFromAppName(appName);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetSessionIDFromAppNamePrivate() Index out of range"));
    }
    return 0;
}

unsigned int iPodControlHandleMap::GetProtocolIDFromAppNamePrivate(
        const int indx, const tAppName appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->GetProtocolIDFromAppName(appName);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return 0;
}

void iPodControlHandleMap::GetActiveSessionListPrivate(vector<string> &list,
        const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->GetActiveSessionList(list);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetActiveSessionListPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::GetAppNameBySessionIDPrivate(const int indx,
        const tSessionID sessionId, tAppName &appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->GetAppNameBySessionID(sessionId, appName);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetAppNameBySessionIDPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::GetAppNameByProtocolIDPrivate(const int indx,
        const unsigned char protocolId, tAppName &appName,
        tSessionID &sessionId) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->GetAppNameByProtocolID(protocolId, appName, sessionId);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetProtocolNameByProtocolIDPrivate(const int indx,
        const unsigned char protocolId, tProtocolName &protocolName)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->GetProtocolNameByProtocolID(protocolId, protocolName);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

tBoolean iPodControlHandleMap::GetTrackFinishedPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_TrackFinished;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetTrackFinishedPrivate() Index out of range"));
    }
    return false;
}

tBoolean iPodControlHandleMap::GetStreamingModePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_StreamingMode;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetStreamingModePrivate() Index out of range"));
    }
    return false;
}

tPEHandle iPodControlHandleMap::GetPEHandlePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PEHandle;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPEHandlePrivate() Index out of range"));
    }
    return HANDLE_NONE;
}

tPlaybackMode iPodControlHandleMap::GetPlaybackModePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_PlaybackMode;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetPlaybackModePrivate() Index out of range"));
    }
    return tPlaybackMode_init;
}

tRepeatMode iPodControlHandleMap::GetRepeatModePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_RepeatMode;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetRepeatModePrivate() Index out of range"));
    }
    return tRepeatMode_init;
}

bool iPodControlHandleMap::GetRepeatInitFlagPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_RepeatInitFlag;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::GetRepeatInitFlagPrivate() Index out of range"));
    }
    return false;
}

tDiPOCallState iPodControlHandleMap::GetCallStatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_CallState;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPOCallState();
}

unsigned int iPodControlHandleMap::GetCallDurationPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_CallDuration;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return 0;
}

tDiPOCommunications iPodControlHandleMap::GetCommunicationsPrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_Communications;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPOCommunications();
}
void iPodControlHandleMap::GetDiPOLocationInfoPrivate(const int indx,tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        startStopLocationInfo = mList[indx]->m_LocationInformationUpdate;
        locationInfoType = mList[indx]->m_LocationInfoType;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::GetDiPOGPRMCDataStatusValuesPrivate(const int indx,tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        GPRMCDataStatusValues = mList[indx]->m_GPRMCDataStatusValues;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

tBTProfile iPodControlHandleMap::GetBTProfilePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_BTProfile;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    tBTProfile btProfile;
    InitBTProfile(btProfile);
    return btProfile;
}

tDiPOPower iPodControlHandleMap::GetPowerUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    VARTRACE(indx);
    VARTRACE(mList.size());
    if (indx >= 0 && indx < (int) mList.size()) {
        VARTRACE(mList[indx]->m_PowerUpdate);
        return mList[indx]->m_PowerUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPOPower();
}

tDiPORouteGuidanceUpdate iPodControlHandleMap::PullRGUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    tDiPORouteGuidanceUpdate RGUpdate;
    InitDiPORouteGuidanceUpdate(RGUpdate);
    if (indx >= 0 && indx < (int) mList.size()) {
        if(!mList[indx]->m_RGQueueUpdate.empty()){
            RGUpdate = mList[indx]->m_RGQueueUpdate.front();
            mList[indx]->m_RGQueueUpdate.pop();
            return RGUpdate;
        }
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPORouteGuidanceUpdate();
}

tDiPORouteGuidanceManeuverUpdate iPodControlHandleMap::PullRGManeuverUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate;
    InitDiPORouteGuidanceManeuverUpdate(RGManeuverUpdate);
    if (indx >= 0 && indx < (int) mList.size()) {
        if(!mList[indx]->m_RGManeuverQueueUpdate.empty()){
            RGManeuverUpdate = mList[indx]->m_RGManeuverQueueUpdate.front();
            mList[indx]->m_RGManeuverQueueUpdate.pop();
            return RGManeuverUpdate;
        }
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPORouteGuidanceManeuverUpdate();
}

tDiPORouteGuidanceUpdate iPodControlHandleMap::GetRGUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_RGUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPORouteGuidanceUpdate();
}

tDiPORouteGuidanceManeuverUpdate iPodControlHandleMap::GetRGManeuverUpdatePrivate(const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_RGManeuverUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPORouteGuidanceManeuverUpdate();
}

tDiPOCaps iPodControlHandleMap::GetLastWirelessCarplayStatusPrivate(const int indx)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        return mList[indx]->m_diPOCaps;
    } else {
       ETG_TRACE_ERR(("Index out of range"));
    }
    return DIPO_CAP_NONE;
}

tBTLimitationActionState iPodControlHandleMap::GetBTLimitationActionStatePrivate(const int indx)
{
    ENTRY_INTERNAL;
    if(indx >= 0&& indx < (int)mList.size()) {
        return mList[indx]->m_btLimitationActionState;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return BTLIMITATION_ACTION_STATE_INVALID;
}

tConnectionState iPodControlHandleMap::GetBTLimitationModeConnectionStatePrivate(const int indx)
{
    ENTRY_INTERNAL;
    if(indx >= 0&& indx < (int)mList.size()) {
        return mList[indx]->m_btLimitationConnectionState;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return CS_DISCONNECTED;
}

tBoolean iPodControlHandleMap::GetIsCPWActiveForUSBMountPointPrivate(const int indx)
{
    ENTRY_INTERNAL;
    if(indx >= 0&& indx < (int)mList.size()) {
        return mList[indx]->m_isCPWActive;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return false;
}

tDiPODeviceTime iPodControlHandleMap::GetDeviceTimeUpdatePrivate(const int indx)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size())
    {
        return mList[indx]->m_DeviceTimeUpdate;
    }
    else
    {
        ETG_TRACE_ERR(("Index out of range"));
    }
    return tDiPODeviceTime();
}

//private setters
void iPodControlHandleMap::SetOptionPrivate(const int indx,
        const tiPodOptions option, const bool set) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->SetOption(option, set);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetOptionPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetDBPathPrivate(const int indx,
        const iPodControlMediaPath path) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_DBPath = path;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetDBPathPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPBPathPrivate(const int indx,
        const iPodControlMediaPath path) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PBPath = path;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPBPathPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPlayerStatePrivate(const int indx,
        const int state) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PlayerState = state;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPlayerStatePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPEPlaybackStatePrivate(const int indx,
        const tPEPlaybackState state) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PlaybackState = state;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPEPlaybackStatePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetLastPlaybackStatePrivate(const int indx,
        const tPEPlaybackState state) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_LastPlaybackState = state;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetNowPlayingTrackIndexPrivate(const int indx,
        const int track) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingTrackIndex = track;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetNowPlayingTrackIndexPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetNowPlayingTrackCountPrivate(const int indx,
        const int count) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingTrackCount = count;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetNowPlayingChapterIndexPrivate(const int indx,
        const int track) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingChapterIndex = track;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetNowPlayingMediaObjectPrivate(const tMediaObject mediaObject, const int indx) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingMediaObject = mediaObject;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetNowPlayingMediaObjectPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetNowPlayingTagsPrivate(const tMediaType mediaType,
        const tMetadata MetadataField1, const tMetadata MetadataField2,
        const tMetadata MetadataField3, const tMetadata MetadataField4,
        const tTitle title, const tUUID uuid, const int indx) {
    ENTRY_INTERNAL;
    tMediaObject mediaObject;
    InitMediaObject(mediaObject);
    GetNowPlayingMediaObjectPrivate(mediaObject, indx);

    mediaObject.mediaType = mediaType;
    strncpy_r(mediaObject.MetadataField1, MetadataField1, sizeof(mediaObject.MetadataField1));
    strncpy_r(mediaObject.MetadataField2, MetadataField2, sizeof(mediaObject.MetadataField2));
    strncpy_r(mediaObject.MetadataField3, MetadataField3, sizeof(mediaObject.MetadataField3));
    strncpy_r(mediaObject.MetadataField4, MetadataField4, sizeof(mediaObject.MetadataField4));
    strncpy_r(mediaObject.title, title, sizeof(mediaObject.title));
    strncpy_r(mediaObject.UUID, uuid, sizeof(mediaObject.UUID));

    SetNowPlayingMediaObjectPrivate(mediaObject, indx);
}

void iPodControlHandleMap::SetPBTrackIndexPrivate(const int indx,
        const int track) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PBTrackIndex = track;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPBTrackIndexPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPollThreadIndexPrivate(const int indx,
        const int threadIndex) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PollThreadIndex = threadIndex;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetElapsedPlaytimePrivate(const int indx,
        const tPlaytime tme) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_ElapsedPlaytime = tme;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetTotalPlaytimePrivate(const int indx,
        const tPlaytime tme) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingMediaObject.totalPlaytime = tme;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetTotalPlaytimePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetDeviceNamePrivate(const int indx,
        const char* name) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_DeviceInfo.deviceName, name,
                sizeof(mList[indx]->m_DeviceInfo.deviceName));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetDeviceNamePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetLaunchAppPrivate(const int indx,
        const char* name) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_LaunchApp, name,
                sizeof(mList[indx]->m_LaunchApp));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetLaunchAppPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetFocusAppPrivate(const int indx,
        const char* name) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_FocusApp, name,
                sizeof(mList[indx]->m_FocusApp));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetFocusAppPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetOTGPathPrivate(const int indx,
        const char* path) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_OTGPath, path,
                sizeof(mList[indx]->m_OTGPath));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetPowerPortPathPrivate(const int indx,
        const char* path) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_PowerPortPath, path,
                sizeof(mList[indx]->m_PowerPortPath));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetUDCDevicePrivate(const int indx,
        const char* udc) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_UDCDevice, udc,
                sizeof(mList[indx]->m_UDCDevice));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetHubVendorIDPrivate(const int indx,
        const char* vendor) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_HubVendorID, vendor, sizeof(mList[indx]->m_HubVendorID));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetHubProductIDPrivate(const int indx,
        const char* product) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_HubProductID, product, sizeof(mList[indx]->m_HubProductID));
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetRoleSwitchedPrivate(const int indx,
        const tBoolean state) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RoleSwitched = state;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetConnectionStatePrivate(const int indx,
        const tConnectionState state) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_DeviceInfo.connectionState = state;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetConnectionStatePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetDeviceIDPrivate(const int indx,
        const tDeviceID id) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_DeviceInfo.deviceID = id;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetDeviceIDPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetSampleRatePrivate(const int indx,
        const me::samplerate_i sampleRate) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingMediaObject.sampleRate = sampleRate;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetSampleRatePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetBitRatePrivate(const int indx,
        const tBitRate bitRate) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_NowPlayingMediaObject.bitRate = bitRate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetCodecTypePrivate(const int indx,
        const tCodecType codecType) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_CodecType = codecType;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetLocationInformationUpdatePrivate(const int indx,
        const tBoolean locationInformationUpdate) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_LocationInformationUpdate = locationInformationUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetVehicleStatusUpdatePrivate(const int indx,
        const tBoolean vehicleStatusUpdate) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_VehicleStatusUpdate = vehicleStatusUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetAlbumArtIDPrivate(const int indx,
        const int format) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_AlbumArtID = format;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetQueueListIDPrivate(const int indx,
        const int format) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_QueueListID = format;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetOTGTypePrivate(const int indx,
        const IPOD_OTG_TYPE type) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_OTGType = type;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetOTGPortNumberPrivate(const int indx,
        const int port) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_OTGPortNumber = port;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetLastHIDReportPrivate(const int indx,
        const IPOD_HID_REPORT report) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_LastHIDReport = report;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetAlbumArtStringPrivate(const int indx,
        const tAlbumArt albumArt) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_NowPlayingMediaObject.albumArtString, albumArt,
                sizeof(mList[indx]->m_NowPlayingMediaObject.albumArtString));
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetAlbumArtStringPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetSessionIDByIndexPrivate(const int indx,
        const tSessionID sessionId, const unsigned char protocolIndex,
        tAppName &appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->SetSessionIDByIndex(sessionId, protocolIndex, appName);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetSessionIDByIndexPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::ClearSessionIDPrivate(const int indx,
        const tSessionID sessionId, tAppName &appName) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->ClearSessionID(sessionId, appName);
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::ClearSessionIDPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetTrackFinishedPrivate(const int indx,
        const tBoolean set) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_TrackFinished = set;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetTrackFinishedPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetStreamingModePrivate(const int indx,
        const tBoolean set) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_StreamingMode = set;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetStreamingModePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPEHandlePrivate(const int indx,
        const tPEHandle handle) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PEHandle = handle;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPEHandlePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetPlaybackModePrivate(const int indx,
        const tPlaybackMode mode) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PlaybackMode = mode;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetPlaybackModePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetRepeatModePrivate(const int indx,
        const tRepeatMode mode) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RepeatMode = mode;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetRepeatModePrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetRepeatInitFlagPrivate(const int indx,
        const bool flag) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RepeatInitFlag = flag;
    } else {
        ETG_TRACE_ERR(("iPodControlHandleMap::SetRepeatInitFlagPrivate() Index out of range"));
    }
}

void iPodControlHandleMap::SetBTProfilePrivate(const int indx,
        const tBTProfile btProfile) {
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_BTProfile = btProfile;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::UpdateOrInsertCallStatePrivate(const int indx, const tDiPOCallStateItem callStateItem)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        for(unsigned int i = 0; i < mList[indx]->m_CallState.size(); i++) {
            if(!strcmp(callStateItem.callUUID, mList[indx]->m_CallState.at(i).callUUID)) {
                mList[indx]->m_CallState.at(i) = callStateItem;
                return;
            }
        }
        //push new item
        mList[indx]->m_CallState.push_back(callStateItem);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::UpdateCallDurationPrivate(const int indx, const U32 callDuration)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_CallDuration = callDuration;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::RemoveDisconnectedCallStatePrivate(const int indx)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        for(tDiPOCallState::iterator it = mList[indx]->m_CallState.begin(); it != mList[indx]->m_CallState.end();) {
            if(it->status ==  DIPO_CALL_STATUS_DISCONNECTED) {
                it = mList[indx]->m_CallState.erase(it);
            } else {
                ++it;
            }
        }
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetCommunicationsPrivate(const int indx, const tDiPOCommunications comm)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_Communications = comm;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetDiPOLocationInfoPrivate(const int indx,const tBool startStopLocationInfo, const tDiPOLocationInfoType locationInfoType)
{
    ENTRY_INTERNAL;
        if (indx >= 0 && indx < (int) mList.size()) {
            mList[indx]->m_LocationInformationUpdate = startStopLocationInfo;
            mList[indx]->m_LocationInfoType = locationInfoType;
        } else {
            ETG_TRACE_ERR(("Index out of range"));
        }
}

void iPodControlHandleMap::SetDiPOGPRMCDataStatusValuesPrivate(const int indx,const tDiPOGPRMCDataStatusValues GPRMCDataStatusValues)
{
    ENTRY_INTERNAL;
        if (indx >= 0 && indx < (int) mList.size()) {
            mList[indx]->m_GPRMCDataStatusValues = GPRMCDataStatusValues;
        } else {
            ETG_TRACE_ERR(("Index out of range"));
        }
}

void iPodControlHandleMap::SetPowerUpdatePrivate(const int indx, const tDiPOPower power)
{
    ENTRY_INTERNAL;
    VARTRACE(indx);
    VARTRACE(power);
    VARTRACE(mList.size());
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_PowerUpdate = power;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::PushRGUpdatePrivate(const int indx, const tDiPORouteGuidanceUpdate RGUpdate)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RGQueueUpdate.push(RGUpdate);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::PushRGManeuverUpdatePrivate(const int indx, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RGManeuverQueueUpdate.push(RGManeuverUpdate);
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetRGUpdatePrivate(const int indx, const tDiPORouteGuidanceUpdate RGUpdate)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RGUpdate = RGUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetRGManeuverUpdatePrivate(const int indx, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_RGManeuverUpdate = RGManeuverUpdate;
    } else {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetLastWirelessCarplayStatusPrivate(const int indx,tDiPOCaps diPOCaps)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_diPOCaps = diPOCaps;
     } else {
         ETG_TRACE_ERR(("Index out of range"));
     }
}

void iPodControlHandleMap::SetBTLimitationActionStatePrivate(const int indx, tBTLimitationActionState btLimitationActionState)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_btLimitationActionState = btLimitationActionState;
     } else {
         ETG_TRACE_ERR(("Index out of range"));
     }
}


void iPodControlHandleMap::SetBTLimitationModeConnectionStatePrivate(const int indx, tConnectionState btLimitationModeConnectionState)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_btLimitationConnectionState = btLimitationModeConnectionState;
     } else {
         ETG_TRACE_ERR(("Index out of range"));
     }
}

void iPodControlHandleMap::SetIsCPWActiveForUSBMountpointPrivate(const int indx, const tBoolean isCPWActive)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        mList[indx]->m_isCPWActive = isCPWActive;
     } else {
         ETG_TRACE_ERR(("Index out of range"));
     }
}

void iPodControlHandleMap::SetAppleDeviceTransportIdentifiersPrivate(const int indx,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size()) {
        strncpy_r(mList[indx]->m_DeviceInfo.appleDeviceMACAddress, appleDeviceMACAddress, sizeof(mList[indx]->m_DeviceInfo.appleDeviceMACAddress));
        strncpy_r(mList[indx]->m_DeviceInfo.appleDeviceUSBSerialNumber, appleDeviceUSBSerialNumber, sizeof(mList[indx]->m_DeviceInfo.appleDeviceUSBSerialNumber));

     } else {
         ETG_TRACE_ERR(("Index out of range"));
     }
}

void iPodControlHandleMap::SetDeviceTimeUpdatePrivate(const int indx, const tDiPODeviceTime deviceTime)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size())
    {
        mList[indx]->m_DeviceTimeUpdate = deviceTime;
    }
    else
    {
        ETG_TRACE_ERR(("Index out of range"));
    }
}

void iPodControlHandleMap::SetWaitingforRoleSwitchResponsePrivate(const int indx, const bool isWaitingforRoleSwitchResponse)
{
    ENTRY_INTERNAL;
    if (indx >= 0 && indx < (int) mList.size())
    {
        mList[indx]->m_IsWaitingforRoleSwitchResponse = isWaitingforRoleSwitchResponse;
    }
    else
    {
         ETG_TRACE_ERR(("Index out of range"));
    }
}
