#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#ifdef TARGET_BUILD
#include "trcGenProj/Header/iPodControl.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_IPOD_CONTROL
#endif
#endif

#include "LocalSPM.h"
#include "Dispatcher.h"
#include "iPodControl.h"
#include "FunctionTracer.h"
#include "VarTrace.h"
#include "ElapsedTimer.h"
#include "BTDaemonProxy.h"
#include <glib.h>

#define FUNCTION_ID_APPCONTROL_CONNECT      1
#define FUNCTION_ID_APPCONTROL_COMMAND      2
#define FUNCTION_ID_APPCONTROL_CLOSE        3
#define FUNCTION_ID_STOP_PLAYBACK_ON_DETACH 4
#define FUNCTION_ID_IAP2_POLL               5
#define FUNCTION_ID_ROLE_SWITCH             6
#define FUNCTION_ID_PROCESS_USER_DATA       11
#define FUNCTION_ID_PLAYBACK_ACTION         12
#define FUNCTION_ID_CHANGE_SAMPLERATE       13

//SPM part

void iPodControl::Create()
{
    ENTRY;

    /* Create the state machine */
    iPodControlSM::Create();

    CreateDone(0);
}

tResult iPodControl::Init(tInitReason reason)
{
    ENTRY;
    VARTRACE(reason);


    /* Init the state machine */
    iPodControlSM::Init();
    SetAnswerTimeout(IPOD_CTRL_SM_ANSWER_TIMEOUT_MS); // Set answer timeout:: Ipod authentication up to 8secs

    /* Register state machine with dispatcher */
    Dispatcher::GetInstance().Register(IN this);
    if (!(LocalSPM::GetDataProvider().IsDeviceTypeSupported(DTY_BLUETOOTH)))
    {
        LocalSPM::GetInstance().GetIPCProvider().RegisterListener(&BTDaemonProxy::GetInstance());
    }

    //startup iAP library
    if (gIPODConnected) {
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP1()) {
            if (iAP1_Init() != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iPodControl::Init() - iAP1Init failed"));
            }
        }
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP2()) {
            if (iAP2_Init() != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iPodControl::Init() - iAP2Init failed"));
            }
        }
    }

    //----------------------------------------------------------------------------------------------------
    //At startup an intial sequence has to be called for the connected HUB in dependency of the configuration
    //in AIVI both types of HUB Mitsumi2PortES2 and Mitsumi2ES4 are expected to be enabled in configuraiton at the same time
    //but intial sequences are different - so software has to check which one is connected here
    //Note: iPodControlIAP2CarPlayViaMitsumi2PortES2HubEnabled is indeed calling iPodControlIAP2CarPlayViaMicrochipHubEnabled
    //but for clean coding we treat this sperately

    unsigned int uProductID,uVendorID; //tbd.: get this from central point
    char strBcdDevice[255], strProduct[255];

    if(    LocalSPM::GetDataProvider().iPodControlIAP2CarPlayViaMitsumi2PortES4HubEnabled()
        || LocalSPM::GetDataProvider().iPodControlIAP2CarPlayViaMitsumi2PortES2HubEnabled())
    {
        //fill out what is expected
           uProductID = LocalSPM::GetDataProvider().iPodControlIAP2CarMitsumi2PortES2And4Bridge_ProductID();
           uVendorID  = LocalSPM::GetDataProvider().iPodControlIAP2CarMitsumi2PortES2And4Bridge_VendorID();

           snprintf(strBcdDevice, sizeof(strBcdDevice),"%s",LocalSPM::GetDataProvider().iPodControlIAP2CarMitsumi2PortES2bcdDevice_A().c_str()); //128 is MitsumiPort2ES2 this not  MitsumiPort2ES2 (this is how it is defined with FO)
           snprintf(strProduct, sizeof(strProduct),(char*)"Bridge device");

           ETG_TRACE_USR2(("iPodControl::Init(): uProductID  : 0x%x",uProductID));
           ETG_TRACE_USR2(("iPodControl::Init(): uVendorID   : 0x%x",uVendorID));
           ETG_TRACE_USR2(("iPodControl::Init(): strBcdDevice: %s"  ,strBcdDevice));
           ETG_TRACE_USR2(("iPodControl::Init(): strProduct  : %s"  ,strProduct));
    }


    tBool bInitDone = FALSE;
    if(FALSE == bInitDone)
    {
        if(LocalSPM::GetDataProvider().iPodControlIAP2CarPlayViaMitsumi2PortES4HubEnabled())
        {

            if(TRUE == bCheckUdevForExpectedHub(uProductID, uVendorID, strProduct, strBcdDevice, FALSE /*bStrBcdDeviceEqualThis*/))
            {
                ETG_TRACE_FATAL(("iPodControl::Init() - InitCarPlayMicroChipMitsumiPort2ES4(): "))
                InitCarPlayMicroChipMitsumiPort2ES4();
                bInitDone = TRUE;
            }
        }
    }

    if(FALSE == bInitDone)
    {
        if(LocalSPM::GetDataProvider().iPodControlIAP2CarPlayViaMitsumi2PortES2HubEnabled())
        {
             if(TRUE == bCheckUdevForExpectedHub(uProductID, uVendorID, strProduct, strBcdDevice, TRUE /*bStrBcdDeviceEqualThis*/))
             {
                 ETG_TRACE_FATAL(("iPodControl::Init() - InitCarPlayMicroChipMitsumiPort2ES2(): "))
                 InitCarPlayMicroChipMitsumiPort2ES2();
                 bInitDone = TRUE;
             }
        }
    }

    if(FALSE == bInitDone)
    {
        if(LocalSPM::GetDataProvider().iPodControlIAP2CarPlayViaMicrochipHubEnabled())
        {
            ETG_TRACE_FATAL(("iPodControl::Init() - InitCarPlayMicroChipHub(): "))
            InitCarPlayMicroChipHub();
            bInitDone = TRUE;
        }
    }


    return InitDone(0);
}

tResult iPodControl::InitSM()
{
    ENTRY

    //init variables
    m_ActivePBMountPoint[0] = '\0';
    m_ActivePBSampleRate = samplerate_i_none;
    m_AudioOutputDevice[0] = '\0';
    m_RoleSwitchDeviceID = DEVICE_ID_NOT_SET;
    m_RemoteActivityDeviceID = DEVICE_ID_NOT_SET;

    m_FingerprintTimerID = 0;
    m_LastFingerprintDeviceID = DEVICE_ID_NOT_SET;
    m_IndexSession = 0;
    m_AudioInputDevice[0] = '\0'; // Fix for CAF-2023
    m_DeviceToInitMountPoint[0] = '\0';
    m_IsNowPlayingListAvail = FALSE;
    m_AlbumArtTimerID = 0;

    memset(m_mountPointPendingInitDevice, 0, sizeof(m_mountPointPendingInitDevice));
    m_protocolPendingInitDevice = IDP_NONE;
    m_bPendingInitDevice = false;

    InitAccessoryWifiCredentials();
    return MP_NO_ERROR;
}

tResult iPodControl::Run()
{
    ENTRY;

    /*Create and start threads */
    LocalSPM::GetThreadFactory().Do(IN this, 0, NULL); //iPodControl inclusive state machine

    return RunDone(0);
}

void iPodControl::Do(int functionID, void *ptr)
{
    ENTRY;

    //set the threads name
    LocalSPM::GetThreadFactory().SetName(iPodControlSM::GetSMNameFull());

    switch (functionID) {
        case 0:
        {
            while (iPodControlSM::STATE_MACHINE_FINISHED != iPodControlSM::StateMachine_Main()) { }
            break;
        }
        case FUNCTION_ID_APPCONTROL_CONNECT:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_APPCONTROL_CONNECT", __PRETTY_FUNCTION__));
            DoAppControlConnect((char *)ptr);
            break;
        }
        case FUNCTION_ID_APPCONTROL_COMMAND:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_APPCONTROL_COMMAND", __PRETTY_FUNCTION__));
            DoAppControlCommand((char *) ptr);
            break;
        }
        case FUNCTION_ID_APPCONTROL_CLOSE:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_APPCONTROL_CLOSE", __PRETTY_FUNCTION__));
            DoAppControlClose((char *) ptr);
            break;
        }
        case FUNCTION_ID_STOP_PLAYBACK_ON_DETACH:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_STOP_PLAYBACK_ON_DETACH", __PRETTY_FUNCTION__));
            DoStopPlaybackOnDetach((char *) ptr);
            break;
        }
        case FUNCTION_ID_IAP2_POLL:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_IAP2_POLL", __PRETTY_FUNCTION__));
            DoIAP2Poll((char*) ptr);
            break;
        }
        case FUNCTION_ID_ROLE_SWITCH:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_ROLE_SWITCH", __PRETTY_FUNCTION__));
            DoRoleSwitch((char *) ptr);
            break;
        }
        case FUNCTION_ID_PROCESS_USER_DATA:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_PROCESS_USER_DATA", __PRETTY_FUNCTION__));
            DoProcessUserData((char *) ptr);
            break;
        }
        case FUNCTION_ID_PLAYBACK_ACTION:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_PLAYBACK_ACTION", __PRETTY_FUNCTION__));
            DoPlaybackAction((char *) ptr);
            break;
        }
        case FUNCTION_ID_CHANGE_SAMPLERATE:
        {
            ETG_TRACE_USR3(("%s FUNCTION_ID_CHANGE_SAMPLERATE", __PRETTY_FUNCTION__));
            DoChangeSampleRate((char *) ptr);
            break;
        }
        default:
            ETG_TRACE_ERR(("Unhandled functionID %d", functionID));
            break;
    }
}

tResult iPodControl::Stop()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    //stop obsolete fingerprint timer
    StopFingerprintTimer();

    /* Send STOP message to own SM */
    ret = SendForceEvent(STOP_SM, (char *) NULL);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF"));
    }

    return ret;
}

tResult iPodControl::StopEventProcessed()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    if(false == LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

        /* unregister for further playback events */
        ret = LocalSPM::GetIPCProvider().UnregisterObserver(IN this);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iPodControl::StopEventProcessed() - UnregisterObserver() failed"));
        }
    }

    /* Send stop done to SPM in the transition to final state in state machine */
    return StopDone(0);
}

tResult iPodControl::Done()
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    if (gIPODConnected) {
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP1()) {
            iAP1_DeInit();
        }
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP2()) {
            iAP2_DeInit();
        }
    }

    /* Deregister state machine with dispatcher */
    Dispatcher::GetInstance().DeRegister(IN this);

    /* Send DONE message to own SM */
    ret = SendForceEvent(DONE, (char *) NULL);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF"));
    }

    return ret;
}

tResult iPodControl::DoneEventProcessed()
{
    ENTRY

    //Cleanup threads
    //Killing all threads without DB access
    LocalSPM::GetThreadFactory().CleanUpThreads(this);

    /* Send done done to SPM in the transition to final state in state machine */
    return DoneDone(0);
}

char *iPodControl::GetSMStateName(OUT tGeneralString stateName, size_t size)
{
    GetCurrentState((char *)stateName, size);
    return stateName;
}


int iPodControl::StartDoTask()
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    VARTRACE(m_bPendingInitDevice);
    if(true == m_bPendingInitDevice)
    {
        m_bPendingInitDevice = false;
        ret = SendInitDevice(IN m_mountPointPendingInitDevice, IN m_protocolPendingInitDevice);

        memset(m_mountPointPendingInitDevice, 0, sizeof(m_mountPointPendingInitDevice));
        m_protocolPendingInitDevice = IDP_NONE;
    }

    return ret;
}

//Playback control part

tResult iPodControl::StartAllocateAudioInput(const tDeviceID /*deviceID*/, const tMountPoint mountPoint)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    ETG_TRACE_USR3(("iPodControl::StartAllocateAudioInput()"));
    VARTRACE(mountPoint);

    //Store playing iPod device mountpoint
    strncpy_r(m_ActivePBMountPoint, mountPoint, sizeof(m_ActivePBMountPoint));

    /* iPod ALSA device to assign as audio input device */
    tDeviceInfo deviceInfo;
    iAPGetDeviceInfo(mountPoint, deviceInfo);

    //check for queue list
    const int queueListID = iAPGetQueueListID(mountPoint);
    VARTRACE(queueListID);

    /* Fix Provided for : NCG3D-52542, NCG3D-72271
     * If nowplayinglist is available from device, updating DB about NowPLayingListAvailable */
    if(m_IsNowPlayingListAvail){
        SendDBChangeNowPlayingListAvailable(mountPoint, queueListID, false);
    }

    if(deviceInfo.connectionType == DCT_BLUETOOTH) {

        RegisterReleaseEvent(START_AUDIO_PIPE_METHOD_RETURN);

        if(LocalSPM::GetDataProvider().UseEvolutionBtStack())
        {
            if(LocalSPM::GetDataProvider().RequestIAP2SPPConnection())
            {
                tMountPoint mountPoint  = {0};
                strncpy_r(mountPoint ,deviceInfo.accessoryName, sizeof deviceInfo.accessoryName);

                VARTRACE(mountPoint);
                ret = BTDaemonProxy::GetInstance().StartAudioPipe(IN mountPoint);
                if (ret != MP_NO_ERROR) {
                    ETG_TRACE_ERR(("BTDaemonProxy::StartAudioPipe() failed"));
                }
            }
            else
            {
                VARTRACE(deviceInfo.mountPoint);
                ret = BTDaemonProxy::GetInstance().StartAudioPipe(IN deviceInfo.mountPoint);
                if (ret != MP_NO_ERROR) {
                    ETG_TRACE_ERR(("BTDaemonProxy::StartAudioPipe() failed"));
                }
            }
        }
        else
        {
            VARTRACE(deviceInfo.serialNumber);
            tMountPoint deviceAddress  = {0};
            for (unsigned iter = 0; iter < 6; iter ++ )
            {
            unsigned char sum;
            sum  = 16*BTDaemonProxy::getHexInDecimal( (unsigned char)(deviceInfo.serialNumber[iter*2]) );
            sum += BTDaemonProxy::getHexInDecimal( (unsigned char)(deviceInfo.serialNumber[(iter*2)+1]) );
            deviceAddress[iter] = sum;
            }
            VARTRACE(deviceAddress);

            ret = BTDaemonProxy::GetInstance().StartAudioPipe(IN deviceAddress);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("BTDaemonProxy::StartAudioPipe() failed"));
            }
        }
        return MP_NO_ERROR;
    } else {
        if (deviceInfo.accessoryName[0] == 0) {
            ETG_TRACE_ERR(("GetAudioInputDevice() failed"));
        } else {

            //SUZUKI-16788
            if(iAP2_IsSet(mountPoint) && deviceInfo.connectionType != DCT_BLUETOOTH && !iAP2_IsHostMode(mountPoint)){
                ret = iAP2_StartUSBDeviceModeAudio(mountPoint);
                if (ret != MP_NO_ERROR) {
                    ETG_TRACE_ERR(("iAP2_StartUSBDeviceModeAudio failed"));
                }
            }

            if(false == LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

                /* Send AssignAudioInputDevice to PlayerEngine via DBUS */
                VARTRACE(deviceInfo.accessoryName);
                RegisterReleaseEvent(METHOD_RETURN);
                ret = LocalSPM::GetIPCProvider().AssignAudioInputDevice(IN this,
                        IN deviceInfo.accessoryName);
                if (ret != MP_NO_ERROR) {
                    ETG_TRACE_ERR(("AssignAudioInputDevice() failed"));
                }

            } else {

                ret = SendEvent(METHOD_RETURN, IN "0");
                if(MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("iPodControl::StartAllocateAudioInput() - Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }
            return MP_NO_ERROR;
        }
    }
    SendMethodResult(false); //go back to idle state

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartAudioPipeMethodReturn(tPipeName PipeName,tCodecType CodecType,tBitRate BitRate,tErrorMessage ErrorMessage)
{
    ENTRY;
    VARTRACE(PipeName);
    VARTRACE(CodecType);
    VARTRACE(BitRate);
    VARTRACE(ErrorMessage);
    tResult ret = MP_NO_ERROR;

    if(PipeName == NULL) {
        ret = 1;
        ETG_TRACE_ERR(("PipeName in NULL"));
        //release waiting state ""startAllocateAudioInput"
        SendEvent(METHOD_RETURN, IN "0");
    } else {
        const char *l_PipeName =(char *)PipeName;
        strncpy_r(OUT m_AudioInputDevice,IN l_PipeName,IN sizeof(m_AudioInputDevice)); //Fix for CAF-2023
        ETG_TRACE_USR3(("AudioInputDevice is %s",m_AudioInputDevice)); //Fix for CAF-2023

        if (m_ActivePBMountPoint[0] == 0) {
            ETG_TRACE_ERR(("m_ActivePBMountPoint[0] == 0"));
        } else {
            iAPSetCodecType(m_ActivePBMountPoint, CodecType);
            iAPSetBitRate(m_ActivePBMountPoint, BitRate);
        }

        if(false == LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

            RegisterReleaseEvent(METHOD_RETURN);
            ret = LocalSPM::GetIPCProvider().AssignAudioInputDevice(IN this,IN m_AudioInputDevice); // Fix for CAF-2023

            if (ret != MP_NO_ERROR) {
                 ETG_TRACE_ERR(("iPodControl::StartAllocateAudioInput() - AssignAudioInputDevice() failed"));
            }
        } else {

            ret = SendEvent(METHOD_RETURN, IN "0");
            if(MP_NO_ERROR != ret) {
                ETG_TRACE_ERR(("iPodControl::StartAudioPipeMethodReturn() - Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
        delete[] PipeName;
    }

    if(ErrorMessage != NULL) {
        delete[] ErrorMessage;
    }

    //SendMethodResult(ret); //go back to idle state

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartDeAllocateAudioInput()
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartDeAllocateAudioInput"));

    SendMethodResult(false); //go back to idle state

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartSwitchObserver(const tDeviceID deviceID, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartSwitchObserver"));
    VARTRACE(deviceID);
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    if(iAPGetConnectionType(mountPoint) == DCT_BLUETOOTH) {
        ret = BTDaemonProxy::GetInstance().SwitchObserver(DTY_IPOD);
        if(!ret) {
            ETG_TRACE_USR3(("BTDaemonProxy SwitchObserver Successful"));
        } else {
            ETG_TRACE_USR3(("BTDaemonProxy SwitchObserver failed"));
        }
    }

    if(LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

        /* Switch the observer to this state machine */
#if 1
        ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::SWITCH_OBSERVER", "", "iPodControlSM::METHOD_RETURN");
#else
        Dispatcher::GetInstance().SendMessage("MediaEngineSM::SWITCH_OBSERVER", "iPodControlSM");

        /* MediaEngine will not answer, so fake the answer */
        ret = SendEvent(METHOD_RETURN, IN "0");
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("iPodControl::StartSwitchObserver - Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
#endif
    } else {

        /* Send SwitchObserver to PlayerEngine via DBUS */
        RegisterReleaseEvent(METHOD_RETURN);
        ret = LocalSPM::GetIPCProvider().SwitchObserver(IN this);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("IPCProvider SwitchObserver() failed"));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartStreaming(const tDeviceID /*deviceID*/, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartStreaming()"));
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    //Clear remote activity flag (allowing PLAY command send to iPod) GMMY16-26007
    m_RemoteActivityDeviceID = DEVICE_ID_NOT_SET;

    if (m_ActivePBMountPoint[0] == 0) {
        ETG_TRACE_ERR(("iPodControl::StartStreaming() - m_ActivePBMountPoint[0] == 0"));
    } else {
        me::samplerate_i sampleRate = iAPGetSampleRate(mountPoint);
        tBitRate bitRate = iAPGetBitRate(mountPoint);
        VARTRACE(sampleRate);
        VARTRACE(bitRate);

        m_ActivePBSampleRate = sampleRate;

        if (!iAP_IsDeviceHandleAvail(mountPoint)) {
            ETG_TRACE_ERR(("iPodControl::StartStreaming() - iAP_IsDeviceHandleAvail() failed"));
        } else if(iAP2_IsHostMode(mountPoint)) {
            //CarPlay Mode
            SendPlaybackStatus(mountPoint, PE_PBS_LOADINGSTATE);//go back to idle state
            return MP_NO_ERROR;
        } else {

            //static streaming configuration parameter for iPod
            tEncoding encoding = "audio"; //Not valid for ipod
            //tBitRate bitRate = 128 << 10; //Not valid for ipod
            const me::channels_i channels = 2; //Number of channels of the decoded audio
            const me::samplewidth_i sampleWidth = 16; //Width of the sample in bits
            const me::sampledepth_i sampleDepth = 16; //Depth of the sample in bits
            me::signedness_e sampleSignedness = SIGNEDNESS_SIGNED; //0/1 in case of sample unsigned/signed.
            me::endianess_e sampleEndianness = ENDIANESS_BE; //0/1 in case of sample little/big endian

            if(iAPGetConnectionType(mountPoint) == DCT_BLUETOOTH) {
                sampleSignedness = SIGNEDNESS_SIGNED;                //1 as sample is signed.
                sampleEndianness = ENDIANESS_LE;        //0 as it is Little Endian
                tCodecType codecType = iAPGetCodecType(mountPoint);
                VARTRACE(codecType);

                switch(codecType)
                {
                    case CODEC_TYPE_SBC:
                    {
                        ETG_TRACE_USR4(("CODEC_TYPE_SBC "));
                        strncpy_r(OUT encoding,IN "sbc",IN sizeof(encoding));
                    }
                    break;
                    case CODEC_TYPE_MPEG12AUDIO:
                    {
                        ETG_TRACE_USR4(("CODEC_TYPE_MPEG12AUDIO "));
                        strncpy_r(OUT encoding,IN "mp3",IN sizeof(encoding));
                    }
                    break;
                    case CODEC_TYPE_MPEG24AAC:
                    {
                        ETG_TRACE_USR4(("CODEC_TYPE_MPEG24AAC "));
                        strncpy_r(OUT encoding,IN "aac",IN sizeof(encoding));
                    }
                    break;
                    default:
                    {
                        ETG_TRACE_ERR(("UNKNOWN CODEC"));
                    }
                    break;
                }
            }

            if(false == LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

                /* Send PlayFromDevice to PlayerEngine via DBUS */
                RegisterReleaseEvent(PLAYBACK_STATUS_RESPONSE);
                ret = LocalSPM::GetIPCProvider().PlayFromDeviceSlot(IN this,
                        IN encoding, IN bitRate, IN sampleRate, IN channels,
                            IN sampleWidth, IN sampleDepth, IN sampleSignedness,
                        IN sampleEndianness);

                    if(MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("iPodControl::StartStreaming() - PlayFromDeviceSlot() failed"));
                }
            } else {

                tDeviceInfo deviceInfo;
                iAPGetDeviceInfo(mountPoint, deviceInfo);

                tURL url;
                if(iAPGetConnectionType(mountPoint) == DCT_BLUETOOTH) {
                    snprintf(url, sizeof(tURL), "%s.%s.bt", m_AudioInputDevice, encoding); //Fix for CAF-2023
                } else {
                    snprintf(url, sizeof(tURL), "%s.pcm", deviceInfo.accessoryName);
                }
                ETG_TRACE_USR3(("iPodControl::StartStreaming:url:%s", url));

                tPEStateString args;
                SMF::Marshal((char *)args, sizeof(args)-1, DOUBLE_MARSHAL_SEPARATOR, "tiiltl", url, 1 /* speed */, -1 /*position*/, HANDLE_NONE, m_AudioOutputDevice, sampleRate);

                ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::PLAY", args, "iPodControlSM::START_STREAMING_ANSWER");

                if(MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("iPodControl:StartStreaming() - SendMessageAnswer() failed"));
                }
            }

            if(iAPGetConnectionType(mountPoint) == DCT_BLUETOOTH) {
                //no PE respose in case of BT streaming
                SendPlaybackStatus(mountPoint, PE_PBS_LOADINGSTATE);//go back to idle state
            }
            return MP_NO_ERROR;
        }
    }
    SendPlaybackStatus(mountPoint, PE_PBS_ERRORSTATE);//go back to idle state

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartStopStreaming()
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartStopStreaming()"));

    VARTRACE(m_ActivePBMountPoint);
    const tDeviceID deviceID = iAPGetDeviceID(m_ActivePBMountPoint);
    VARTRACE(deviceID);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterSTOP_STREAMING_ANSWER(OUT parameterString, IN size, IN deviceID);

    if (deviceID != DEVICE_ID_NOT_SET && iAP2_IsCarPlayMode(m_ActivePBMountPoint) && LocalSPM::GetDataProvider().iPodControlIAP2CarPlayModeEnabled()) {
        //CarPlay Mode
        RegisterReleaseEvent("STOP_STREAMING_ANSWER", parameterString, IPOD_CTRL_SM_ANSWER_TIMEOUT_MS);

        //CarPlay TAKE/BORROW HID Play extension
        const tDiPOStopReason diPOStopReason = (tDiPOStopReason)LocalSPM::GetDataProvider().DiPOStopReason();
        VARTRACE(diPOStopReason);
        LocalSPM::GetOutputWrapper().StopMediaPlayback(IN deviceID, IN diPOStopReason);
    } else {
        /* Send STOP_STREAMING_ANSWER message to release own waiting state */
        SendEvent(STOP_STREAMING_ANSWER, IN parameterString);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StopStreamingAnswer(const tDeviceID deviceID)
{
    ENTRY
    VARTRACE(deviceID);
    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    /* Send answer to waiting state machine */
    tPEPlaybackState status = PE_PBS_STOPPEDSTATE;
    if( LocalSPM::GetDataProvider().UseMediaEngine() ) {
        tPEHandle handle = iAPGetPEHandle(mountPoint);
        PlaybackStatusNew(IN handle, IN status, IN REASON_OK, IN ME_SPEED_NORMAL);
    }
    else {
        tMetadata metadata = { 0 };
        PlaybackStatus(IN status, IN metadata, IN metadata, IN metadata, IN metadata);
    }

    //release playing iPod device mountpoint
    iAPSetPEHandle(mountPoint, HANDLE_NONE);
    m_ActivePBMountPoint[0] = '\0';
    m_ActivePBSampleRate = samplerate_i_none;

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartPlay(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint, const tUUID uuid,
        const tPEHandle handle, const tPlaytime position, const tStreaming streaming) // finished: 80%
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);
    VARTRACE(uuid);
    VARTRACE(handle);
    VARTRACE(position);
    VARTRACE(streaming);
    VARTRACE(m_RemoteActivityDeviceID);

    const tBoolean playOnRemoteActivity = (deviceID != DEVICE_ID_NOT_SET) && (deviceID == m_RemoteActivityDeviceID);
    VARTRACE(playOnRemoteActivity);

    tPEPlaybackState CurrentIpodStatus = iAPGetCurrentPlaybackState(mountPoint);
    VARTRACE(CurrentIpodStatus);

    //Reset remote activity flag
    m_RemoteActivityDeviceID = DEVICE_ID_NOT_SET;

    //to move iPodControlSM to startPlay state
    SendEvent(METADATA_FOR_PLAYBACK, NULL);

    tBatchPlaybackList * pbatchlist = 0;

    //unmarshal batchlist pointer
    if(strstr(uuid, "batchlist="))
    {
        sscanf(uuid+10, "%p", &pbatchlist);
    }
    VARTRACE(pbatchlist!=NULL);

    //set internal streaming member
    iAPSetStreaming(mountPoint, streaming);

    //set the PE handle for further answers to PM
    iAPSetPEHandle(mountPoint, handle);

    //iPod may change sample rate while track selection
    //set PE_PBS_LOADINGSTATE to process the new sampling rate after selection is done.
    //See also: iPodControl::OnCBIAP1_NewiPodTrackInfo()
    iAPSetCurrentPlaybackState(mountPoint, PE_PBS_LOADINGSTATE);

    tBoolean sync_play_answer = false;
    if(!LocalSPM::GetDataProvider().iPodSiriActivated()){
        tResult ret = iAPPlaySelection(mountPoint, URL, position, pbatchlist, sync_play_answer, playOnRemoteActivity);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iPodControl::StartPlay iAPPlaySelection failed"));
            iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
        }
    }
    else {
        SendPlaybackStatusToPlayerManager(mountPoint,CurrentIpodStatus);
    }

    //free pbatchlist
    if(pbatchlist) {
        pbatchlist->clear();
        delete pbatchlist;
    }

    VARTRACE(sync_play_answer);
    if(sync_play_answer) {
        //answer sent by iPod event notification
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tObjectID ObjectID = OBJECT_ID_NONE;
        ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size, IN PE_PBS_ERRORSTATE, IN "", IN "", IN "", IN "",IN ObjectID);
        RegisterReleaseEvent(PLAYBACK_STATUS_RESPONSE, parameterString, IPOD_CTRL_SM_ANSWER_TIMEOUT_MS);
    } else {
        //send answer
        SendPlaybackStatus(mountPoint);
    }

    //Clear the iPodSiriActivated Dataprovider
    if(LocalSPM::GetDataProvider().iPodSiriActivated()){
        LocalSPM::GetDataProvider().iPodSiriActivated = 0;
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartStop(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartStop()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);

    tResult ret = MP_NO_ERROR;

    const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
    VARTRACE(isActive);

    const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
    VARTRACE(isStreaming);

    //reset internal streaming member
    iAPSetStreaming(mountPoint, false);

    //clear finished flag
    iAPSetTrackFinished(mountPoint, false);

    const tConnectionType connectionType = iAPGetConnectionType(mountPoint);
    VARTRACE(connectionType);

    const bool carPlayUserActionRestricted = LocalSPM::GetDataProvider().iPodControlIAP2CarPlayModeUserActionOnly() && iAP2_IsCarPlayMode(mountPoint);
    VARTRACE(carPlayUserActionRestricted);

    const tPEPlaybackState lastPlaybackStatus = iAPGetCurrentPlaybackState(mountPoint);
    VARTRACE(lastPlaybackStatus);

    if(isActive) { //check for active - NCG3D-12084
        //store last playback state, see config KeepLastPlaybackStateForStreamingDevicesEnabled
        iAPSetLastPlaybackState(mountPoint, lastPlaybackStatus);
    }

    if(carPlayUserActionRestricted) {
        //reset elapsed time for expected nowplaying update
        iAPResetElapsedTime(mountPoint, pbNotifications);

        if(!isStreaming) {
            //CarPlay TAKE/BORROW HID Play extension
            //MyMedia mode (non-streaming) set dipoStopReason to TAKE audio from iPhone
            LocalSPM::GetDataProvider().DiPOStopReason = (int) DIPO_STOP_REASON_DEVICE_SWITCHED;
        }
    }

    // Send pause to iPod device
    //check isActive in case of double STOP calls [GMMY17-11719]
    if(isActive && !carPlayUserActionRestricted && iAPGetConnectionState(mountPoint) == CS_CONNECTED && !LocalSPM::GetDataProvider().iPodSiriActivated()) {

        ret = iAPPlayPause(mountPoint);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iPodControl::StartStop() - pausing iPod failed"));
        }
        if(iAP2_IsSet(mountPoint) && connectionType != DCT_BLUETOOTH && !iAP2_IsHostMode(mountPoint)) {
            //SUZUKI-16788
            ret = iAP2_StopUSBDeviceModeAudio(mountPoint);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iAP2_StartUSBDeviceModeAudio failed"));
            }
        }
    }

    if(connectionType == DCT_BLUETOOTH) {
        ret = BTDaemonProxy::GetInstance().StopAudioPipe();
        if (ret != MP_NO_ERROR){
             ETG_TRACE_ERR(("StopAudioPipe failed"));
        }
        //PSARCC21-1827 :IPODContorlSM being the observer in BTDaemonProxy is Finished.Reset the Observer to Default BTControlSM
        ret = BTDaemonProxy::GetInstance().SwitchObserver(DTY_BLUETOOTH);

    }

    //ipod in paused state, PE in stopped state
    iAPSetCurrentPlaybackState(mountPoint, PE_PBS_STOPPEDSTATE);

    if (m_ActivePBMountPoint[0] == 0) {
        ETG_TRACE_ERR(("iPodControl::StartStop() - m_ActivePBMountPoint[0] == 0"));
    } else if(iAP2_IsHostMode(mountPoint)) {
        //CarPlay Mode
        //SendPlaybackStatus(mountPoint, PE_PBS_STOPPEDSTATE);
        //go back to idle state
        SendEvent(STOP_ANSWER, IN NULL); //GMMY16-24734
        return MP_NO_ERROR;
    } else {

        if(LocalSPM::GetDataProvider().UseMediaEngine()) { //pip2hi

            ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::STOP", NULL, "iPodControlSM::STOP_ANSWER");
            if(MP_NO_ERROR != ret) {
                ETG_TRACE_ERR(("iPodControl::StartStop() - PE STOP failed"));
            }

        } else {

            /* Send StopSlot to PlayerEngine via DBUS */
            //RegisterReleaseEvent(PLAYBACK_STATUS_RESPONSE);
            RegisterReleaseEvent(STOP_ANSWER); //GMMY16-24734
            ret = LocalSPM::GetIPCProvider().StopSlot(IN this);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iPodControl::StartStop() - stopping PE failed"));
            }
        }
        return MP_NO_ERROR;
    }
    //SendPlaybackStatus(mountPoint, PE_PBS_ERRORSTATE);
    //go back to idle state
    SendEvent(STOP_ANSWER, IN NULL); //GMMY16-24734

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartPause(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);

    tResult ret = MP_NO_ERROR;
    const tPEPlaybackState iPodPBStatus = iAPGetCurrentPlaybackState(mountPoint);
    if(iPodPBStatus == PE_PBS_FASTFORWARDSTATE || iPodPBStatus == PE_PBS_FASTREVERSESTATE) {
        //ATS issue: Send End FF/REV
        ret = iAPPlayFFREWStop(mountPoint);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iAPPlayFFREWStop failed"));
        }
    }

    ret = iAPPlayPause(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPPlayPause failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartResume(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartResume()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);

    tResult ret = iAPPlayResume(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartResume iAPPlayResume failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartFfwdStart(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint,
        const tCueingRate rate,const speedstate_e IsPlaybackSpeed)
{
    ENTRY;
    (void)IsPlaybackSpeed;
    ETG_TRACE_USR3(("iPodControl::StartFfwdStart()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);
    VARTRACE(rate);

    tResult ret = iAPPlayFFStart(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartFfwdStart iAPPlayFFStart failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartFfwdStop(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartFfwdStop()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);

    tResult ret = iAPPlayFFREWStop(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartFfwdStop iAPPlayFFREWStop failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartFrevStart(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint,
        const tCueingRate rate,const speedstate_e IsPlaybackSpeed)
{
    ENTRY;
    (void)IsPlaybackSpeed;

    ETG_TRACE_USR3(("iPodControl::StartFrevStart()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);
    VARTRACE(rate);



    tResult ret = iAPPlayREWStart(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartFfwdStart iAPPlayREWStart failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartFrevStop(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartFrevStop()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);

    tResult ret = iAPPlayFFREWStop(mountPoint);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartFrevStop iAPPlayFFREWStop failed"));
        iAPSetCurrentPlaybackState(mountPoint, PE_PBS_ERRORSTATE);
    }

    SendPlaybackStatus(mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartSeekTo(const tDeviceType deviceType,
        const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint,
        const tPlaytime position)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::StartSeekTo()"));
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);
    VARTRACE(position);

    tResult ret = iAPSeekTo(mountPoint, position);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartSeekTo failed"));
    }

    SendEvent(SEEK_TO_ANSWER, IN NULL);
    SendAnswer(IN NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::ActionStatus(const tReturnValue returnValue)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::ActionStatus() returnValue:%u", returnValue));

    tResult ret = MP_NO_ERROR;

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetDeviceDispatcher().ParameterMETHOD_RETURN(
            OUT parameterString, IN size, IN returnValue);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::ActionStatus() - Error while preparing parameter string"));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::BufferStatus(const tSuccess success)
{
    ENTRY
    VARTRACE(success);

        tResult ret = MP_NO_ERROR;

        /* Send answer to waiting state machine */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

#if 0
    /* map reason to success */
    tSuccess success = SC_NO;
    if (reason >= REASON_OK) {
        success = SC_YES;
    }
#endif

    ret = LocalSPM::GetPlayerManager().ParameterBUFFER_ANSWER(OUT parameterString, IN size, IN success);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::PlaybackStatus(const tPEPlaybackState status,  //Roadmap 13008 RemoteControl
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::PlaybackStatus()"));
    VARTRACE(status);
    VARTRACE(metadata1);
    VARTRACE(metadata2);
    VARTRACE(metadata3);
    VARTRACE(metadata4);


    /* get current now playing info from iPod */
    tMountPoint mountPoint = {0};
    strncpy_r(mountPoint, m_ActivePBMountPoint, sizeof(mountPoint));
    VARTRACE(mountPoint);

    tPEPlaybackState iPodStatus = iAPGetCurrentPlaybackState(mountPoint);
    VARTRACE(iPodStatus);

    //check for nowPlaying changed during StartPlay finished with flag==TRUE (SUZUKI-21753)
    iAPCheckUpdateNowPlaying(mountPoint);

    tMetadata meta1 = { 0 };
    tMetadata meta2 = { 0 };
    tMetadata meta3 = { 0 };
    tMetadata meta4 = { 0 };
    tTitle metaTitle = { 0 }; //empty title: PM does not update metadata.
    tMediaType mediaType = MTY_UNKNOWN;
    tUUID uid = { 0 };
    //Do not update tags on PLAY_ANSWER, it may cause delay and skipped playTime updates in PM
    iAPGetNowPlayingMetadata(meta1, meta2, meta3, meta4, metaTitle, mediaType, mountPoint,uid);

    VARTRACE(meta1);
    VARTRACE(meta2);
    VARTRACE(meta3);
    VARTRACE(meta4);
    VARTRACE(metaTitle);
    VARTRACE(mediaType);

    if(status != PE_PBS_ERRORSTATE) {
        //set current albumArt Status
        SendAlbumArtStatusToPlayerManager(mountPoint);
    }

    if(iPodStatus == PE_PBS_ERRORSTATE) {
        /* Send action error instead of answer to waiting state machine */
        HandleActionError(REASON_DEVICE_ERROR);
    } else {
        /* Send answer to waiting state machine */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        if(LocalSPM::GetDataProvider().UseMediaEngine()) {
            const tPEHandle handle = iAPGetPEHandle(mountPoint);
            const me::reason_e reason = REASON_OK;
            const me::speed_e speed = ME_SPEED_NORMAL;
            LocalSPM::GetDeviceDispatcher().ParameterPLAYBACK_STATUS(OUT parameterString, IN size, IN handle, IN iPodStatus, IN reason, IN speed);
            SendAnswer(IN parameterString);
        } else {
            LocalSPM::GetDeviceDispatcher().ParameterPLAY_ANSWER(OUT parameterString, IN size, IN iPodStatus, IN meta1, IN meta2, IN meta3, IN meta4, IN metaTitle, IN mediaType,IN(tConvertFlag)0,uid);
            SendAnswer(IN parameterString);
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::ForwardPlaybackStatus(const tPEPlaybackState status,
        const tMetadata metadata1,
        const tMetadata metadata2,
        const tMetadata metadata3,
        const tMetadata metadata4,
        const tObjectID ObjectID)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::ForwardPlaybackStatus()"));
    VARTRACE(status);
    VARTRACE(metadata1);
    VARTRACE(metadata2);
    VARTRACE(metadata3);
    VARTRACE(metadata4);
    tResult ret = MP_NO_ERROR;

    if(m_ActivePBMountPoint[0] == 0) {
        ETG_TRACE_USR3(("IPC event ignored for inactive iPod"));
        return MP_NO_ERROR;
    }

    ret = iAPCheckUpdateNowPlaying(m_ActivePBMountPoint);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::ForwardPlaybackStatus() - iAPCheckUpdateNowPlaying failed"));
    }
    SendAlbumArtStatusToPlayerManager(m_ActivePBMountPoint);

    /* get current now playing info from iPod handle */
    tPEPlaybackState iPodStatus = status;
    if (status == PE_PBS_LOADINGSTATE ||
        status == PE_PBS_PAUSEDSTATE ||
        status == PE_PBS_PLAYINGSTATE ||
        status == PE_PBS_ERRORSTATE) {
        //update from iPod handle
        iPodStatus = iAPGetCurrentPlaybackState(m_ActivePBMountPoint);
    }
    VARTRACE(iPodStatus);

    if( PE_PBS_ERRORSTATE == iPodStatus ) {
        /* Forward stream error */
        ForwardStreamError(REASON_DEVICE_ERROR);
    } else {
        SendPlaybackStatusToPlayerManager(m_ActivePBMountPoint, iPodStatus);
    }

    return MP_NO_ERROR;
}

tResult iPodControl::ForwardNowPlayingStatus(const tPEHandle handle, const tURL url)
{
    ENTRY;
    VARTRACE(handle);
    VARTRACE(url);
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAPGetMountPointByPEHandle(mountPoint, handle);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR( ("iPodControl::ForwardNowPlayingStatus() - iAPGetMountPointByPEHandle failed"));
    } else {
        VARTRACE(mountPoint);
        ret = iAPCheckUpdateNowPlaying(mountPoint);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("iPodControl::ForwardNowPlayingStatus() - iAPCheckUpdateNowPlaying failed"));
        }
        SendAlbumArtStatusToPlayerManager(mountPoint);
        SendNowPlayingStatusToPlayerManager(mountPoint);
    }
    return MP_NO_ERROR;
}

tResult iPodControl::HandleAnswerTimeout()
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));

    tResult ret = MP_NO_ERROR;

    /* Send releasing event to own waiting state machine */
    ret = ReleaseWaiting();
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("%s - Error while releasing waiting via SMF", __PRETTY_FUNCTION__));
    }

    if(SMF::GetState() == roleSwitch) {
        if(LocalSPM::GetDataProvider().DipoCommunicationError()) {
            tIpodCommError iPodCommError = IPOD_ROLESWITCH_ERROR;
            LocalSPM::GetOutputWrapper().UpdateDipoCommunicationError(m_RoleSwitchDeviceID,iPodCommError, m_ActivePBMountPoint);
        }
    }

    return ret;
}

tResult iPodControl::MessageNotConsumed()
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
#if 0 //Do not SendAnswer without reason, it may cause failure within communication to DeviceDispatcher
    /* Send answer to possible waiting state machine to release own SM in case of message answer pair*/
    tResult ret = SendAnswer(NULL);
    if(MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("%s - Error while sending answer via SMF", __PRETTY_FUNCTION__));
    }
#endif
    return MP_NO_ERROR;
}

tResult iPodControl::SetOutputDevice(const tAudioOutputDevice audioOutputDevice)
{
    ENTRY
    VARTRACE(audioOutputDevice);

    /* Store audio output device (alsa device name) internally */
    strncpy_r(OUT m_AudioOutputDevice, IN audioOutputDevice, IN sizeof(m_AudioOutputDevice));

    return MP_NO_ERROR;
}

tResult iPodControl::HandleActionError(const me::reason_e reason)
{
    ENTRY;

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send action error instead of answer to waiting state machine */
    ret = LocalSPM::GetDeviceDispatcher().ParameterACTION_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendActionError(IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending action error via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult iPodControl::ActionError(const me::reason_e reason)
{
    ENTRY;

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send action error to own state machine*/
    ret = ParameterACTION_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = SendEvent(ACTION_ERROR ,IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending action error via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult iPodControl::ForwardStreamError(const me::reason_e reason) const
{
    ENTRY;

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    /* Send STREAM_ERROR message to PlayerManger */
    char messageString[64];
    strncpy_r(OUT messageString, IN "PlayerManagerSM::STREAM_ERROR", IN sizeof(messageString));

    ret = LocalSPM::GetPlayerManager().ParameterSTREAM_ERROR(OUT parameterString, IN size, IN reason);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }

    return MP_NO_ERROR;
}

tResult iPodControl::PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, //Roadmap 13008
            const me::reason_e reason, const me::speed_e speed)
{
    ENTRY;
    VARTRACE(handle);
    VARTRACE(playbackState);
    VARTRACE(reason);
    VARTRACE(speed);

    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    if(REASON_OK > reason) {
        /* Send action error instead of answer to waiting state machine */
        HandleActionError(IN reason);
    } else {
        /* Send answer to waiting state machine */
        ret = LocalSPM::GetDeviceDispatcher().ParameterPLAY_ANSWER_NEW(OUT parameterString, IN size, IN handle, IN playbackState, IN reason, IN speed);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }

        ret = SendAnswer(IN parameterString);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("Error while sending answer via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

//Browsing VTIPOD part

tResult iPodControl::GetVTRecordFromCache(string &rValue, const tMountPoint mountPoint,
        const tListType listType, const tFilterTag filterTag1,
        const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
        const tRowID rowID, const tLimit limit, const tOffset offset)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(listType);
    VARTRACE(filterTag1);
    VARTRACE(filterTag2);
    VARTRACE(filterTag3);
    VARTRACE(filterTag4);
    VARTRACE(rowID);
    VARTRACE(limit);
    VARTRACE(offset);

    // **** called externally by VTIPOD ****
    // *** keep this function thread safe ***

    tResult ret = MP_NO_ERROR;
    iPodControlVTRecordKey key(listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID+offset);

    if(iAP2_IsSet(mountPoint)) {
        ret = iAPGetVTRecord(mountPoint, key, rValue);
        if(ret != MP_NO_ERROR) {
            //iAP2 protocol stack pushes all metadata to accessory
            //for VT enries there is no IAP2 call required but just read out internal iPodControliAP members from m_HandleMap
            ret = iAP2_RetrieveVTRecord(mountPoint, listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID, limit, offset);
            if(ret == MP_NO_ERROR) {
                iAPGetVTRecord(mountPoint, key, rValue);
            } else {
                //In order to prevent VTIPOD to call GET_VIRTUAL_METADATA by MesssageAnswer always return MP_NO_ERROR in case of iAP2
                //rValue := no entriy available (neither cache nor from device)
                ETG_TRACE_USR2(("iAPRetrieveVTRecord failed"));
                ret = MP_NO_ERROR;
            }
        }
    } else if(!iAP_IsDeviceHandleAvail(mountPoint)) {
        //device unavailable
        //In order to prevent VTIPOD to call GET_VIRTUAL_METADATA by MesssageAnswer return MP_NO_ERROR in case of connection loss
        //rValue := no entriy available
        ETG_TRACE_USR4(("Device not initialized"));
        ret = MP_NO_ERROR;
    } else { //else valid iAP1 connection

        /* Return what we have in metadata cache */
        ret = iAPGetVTRecord(mountPoint, key, rValue);
        if(ret != MP_NO_ERROR) {
            ETG_TRACE_USR3(("Not found in VT cache"));
            //returning MP_ERR_IPOD_METADATA caused VTIPOD to call GET_VIRTUAL_METADATA by MesssageAnswer
            //next function entry is IsNextVirtualMetadataAvailable()
        }
    }

    VARTRACE(ret);
    VARTRACE(rValue.c_str());
    return ret;
}

tResult iPodControl::IsNextVirtualMetadataAvailable(const tMountPoint mountPoint,
        const tListType listType, const tFilterTag filterTag1,
        const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
        const tRowID rowID, const tLimit limit, const tOffset offset)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(listType);
    VARTRACE(filterTag1);
    VARTRACE(filterTag2);
    VARTRACE(filterTag3);
    VARTRACE(filterTag4);
    VARTRACE(rowID);
    VARTRACE(limit);
    VARTRACE(offset);

    tResult ret = MP_NO_ERROR;
    iPodControlVTRecordKey key(listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID+offset);
    /* Return we have in metadata cache */
    string value;
    ret = iAPGetVTRecord(mountPoint, key, value);
    VARTRACE(value.c_str());
    VARTRACE(ret);
    return ret == MP_NO_ERROR;
}

tResult iPodControl::RetrieveVirtualMetadata(const tMountPoint mountPoint,
        const tListType listType, const tFilterTag filterTag1,
        const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
        const tRowID rowID, const tLimit limit, const tOffset offset)
{
    ENTRY;
    //retrieve metadata from iPod internal database
    if (MP_NO_ERROR != iAPRetrieveVTRecord(mountPoint, listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID, limit, offset)) {
        ETG_TRACE_USR2(("iAPRetrieveVTRecord failed"));
    }

    SendPutVirtualMetadata(mountPoint, listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID, limit, offset);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SendPutVirtualMetadata(const tMountPoint mountPoint,
        const tListType listType, const tFilterTag filterTag1,
        const tFilterTag filterTag2, const tFilterTag filterTag3, const tFilterTag filterTag4,
        const tRowID rowID, const tLimit limit, const tOffset offset)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    iPodControlVTRecordKey key(listType, filterTag1, filterTag2, filterTag3, filterTag4, rowID+offset);
    string value;
    ret = iAPGetVTRecord(mountPoint, key, value);
    //send answer to release waiting cursor SM in VTIPOD
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_USR2(("iAPGetVTRecord() failed"));
    }
    SendAnswer(IN value.c_str());

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::Umount(const tDeviceID deviceID, const tMountPoint mountPoint) //Roadmap 13003 SDCard
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(mountPoint);

    //NOT SUPPORTED BY IPOD - send an answer back to DeviceDispatcher
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    SMF::Marshal(parameterString, size - 1, "i", MP_ERR_UMOUNT_NOT_SUPPORTED);
    SendAnswer(IN parameterString);

    return MP_NO_ERROR;
}

//Indexing part

tResult iPodControl::IsInitRequired(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);

    VARTRACE(deviceInfo.mountPoint);
    VARTRACE(deviceInfo.deviceID);
    VARTRACE(deviceInfo.connectionType);
    tResult ret = MP_NO_ERROR;

    ret = iAPNewDevice(deviceInfo);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_USR3(("iPodControl::IsInitRequired() iAPNewDevice failed"));
    } else {
        tConnectionState connectionState = iAPGetConnectionState(deviceInfo.mountPoint);
        VARTRACE(connectionState);
        if (connectionState == CS_CONNECTED) {
            ETG_TRACE_USR3(("iPodControl::IsInitRequired() - iPod already initialized"));
            ret = MP_ERR_IPOD_INIT;
        }
    }

    if(ret == MP_NO_ERROR) {
        tBoolean bCarplayCapable = iAP2_IsCarPlayPossible(deviceInfo.mountPoint);
        VARTRACE(bCarplayCapable);
        //carplay device detection speed up, asking SPI if role switch is required to directly go for carplay
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP2() &&
           deviceInfo.connectionType != DCT_BLUETOOTH && deviceInfo.connectionType != DCT_WIFI /*CARPLAY_WIFI*/ &&
           deviceInfo.deviceType != DTY_IPOD && LocalSPM::GetDataProvider().iPodControlIAP2CheckRoleSwitchRequired() &&
           iAP2_IsHostModePossible(deviceInfo.mountPoint) &&
          (iAP2_IsNativeTransportAppConfigured(deviceInfo.mountPoint) || bCarplayCapable || LocalSPM::GetDataProvider().iPodControlSupportCarlife())) {

            if(LocalSPM::GetDataProvider().iPodControlIAP2CheckRoleSwitchRequiredTimeOutMS() == 0) {
                //no timeout wait for user choice
                iAPSetWaitingforRoleSwitchResponse(deviceInfo.mountPoint, true); //Store in HandleMap, fix for NCG3D-197143
                ret = LocalSPM::GetOutputWrapper().DIPORoleSwitchRequired(deviceInfo.deviceID,deviceInfo.mountPoint,bCarplayCapable); //USB serial number parameter added.
                if(ret != MP_NO_ERROR) {
                    //if SPI is not available: Init device mode
                    iAPSetWaitingforRoleSwitchResponse(deviceInfo.mountPoint, false);
                    SendInitInit(IN deviceInfoString);
                } else {
                    ETG_TRACE_USR3(("iPod initialization waiting for user interaction..."));
                }
            } else {
                /* Send ROLE_SWITCH_REQUIRED message to own SM */
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterROLE_SWITCH_REQUIRED(OUT parameterString, IN size, IN deviceInfo.deviceID);
                SendEvent(IN ROLE_SWITCH_REQUIRED, IN parameterString);
            }
        } else {
            SendInitInit(IN deviceInfoString);
        }
    } else {
        NoInitAnswer(IN deviceInfoString);
    }

    return MP_NO_ERROR;
}

tResult iPodControl::SendInitInit(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);
    ETG_TRACE_USR3(("iPodControl::SendInitInit() deviceID %d, mountpoint '%s'", deviceInfo.deviceID, deviceInfo.mountPoint));

    VARTRACE(deviceInfo.connectionType);
    tInitDeviceProtocol initProtocol = IDP_NONE;

    if(deviceInfo.connectionType == DCT_BLUETOOTH) {
        if(LocalSPM::GetDataProvider().iPodControlSupportIAP2() && (LocalSPM::GetDataProvider().iPodControlSupportIAP2BT() ||
                LocalSPM::GetDataProvider().IAPBTConnectionOnlyForSmartApp())) {
            initProtocol = IDP_IAP2BT;

            /*Get BTLimitation mode from BT Client Handler. */
            tBTLimitationActionState btLimitation;
            LocalSPM::GetOutputWrapper().GetBTLimitationModeFromBTClient(deviceInfo.serialNumber,btLimitation);
            VARTRACE(btLimitation);
            SetBTLimitationMode(deviceInfo.mountPoint,btLimitation);

        } else if(LocalSPM::GetDataProvider().iPodControlSupportIAP1() && LocalSPM::GetDataProvider().iPodControlSupportIAP1BT()) {
            initProtocol = IDP_IAP1BT;
        }
    } else if(deviceInfo.connectionType == DCT_WIFI) {
        initProtocol = IDP_IAP2_OVER_WIRELESS_CARPLAY;
    } else { //USB

        //skip iAP2 on known USB iAP2 devices (GMMY16-18313)
        const bool isKnownUSBiAP1Device = iAPIsKnownUSBiAP1Device(deviceInfo.mountPoint);
        VARTRACE(isKnownUSBiAP1Device);

        if(!isKnownUSBiAP1Device && LocalSPM::GetDataProvider().iPodControlSupportIAP2()) {
            initProtocol = IDP_IAP2;
        } else if(LocalSPM::GetDataProvider().iPodControlSupportIAP1()) {
            initProtocol = IDP_IAP1;
        }
    }
    VARTRACE(initProtocol);

    if(initProtocol != IDP_NONE) {
        SendInitDevice(deviceInfo.mountPoint, initProtocol);
    } else {
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        //error case - send indexer answer
        SendDeviceInitialized(deviceInfo.mountPoint);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::NoInitAnswer(const tDeviceInfoString deviceInfoString)
{
    ENTRY;
    tDeviceInfo deviceInfo;
    DataProvider::UnMarshalDeviceInfo(deviceInfoString, deviceInfo);
    VARTRACE(deviceInfo.mountPoint);
    VARTRACE(deviceInfo.deviceID);

    SendDeviceInitialized(deviceInfo.mountPoint);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

int iPodControl::IsInitDeviceConnection(const tMountPoint mountPoint, const tInitDeviceProtocol protocol) //Roadmap 13006 IAP2
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(protocol);
    tResult ret = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        ret = iAP1_InitDeviceConnection(mountPoint);
    } else if(IsProtocol_IAP2(protocol)) {

        tBoolean hostMode = IsProtocol_IAP2HostMode(protocol);
        tBoolean wiredCarPlay = IsProtocol_IAP2CarPlay(protocol);
        tBoolean nativeTransport = IsProtocol_IAP2NativeTransport(protocol);
        tBoolean wirelessCarplay =  IsProtocol_IAP2OverWirelessCarplay(protocol);
        tBoolean carLife =  IsProtocol_IAP2CarLife(protocol);
        if(wiredCarPlay)
        {
            SetIsWirelessCarplayActive(mountPoint);
        }

        ret = iAP2_InitDeviceConnection(mountPoint, hostMode, wiredCarPlay, nativeTransport,wirelessCarplay,carLife);

        if(ret == MP_NO_ERROR) {
            //setup IAP2 poll thread
            char *poll_mountPoint = new char[sizeof(tMountPoint)]; //deleted by Do function
            strncpy_r(poll_mountPoint, mountPoint, sizeof(tMountPoint));
            iAPSetPollThreadIndex(mountPoint, 1);
            int threadIndex = LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_IAP2_POLL, IN poll_mountPoint, this);
            if(threadIndex < 0) {
                ETG_TRACE_ERR(("POLL THREAD NOT STARTED"));
                ret = MP_ERR_IPOD_INIT;
                iAPSetPollThreadIndex(mountPoint, -1);
            }
        } //lint !e429 poll_mountPoint freed by DoIAP2Poll
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        ret = MP_ERR_IPOD_INIT;
    }

    if(ret == MP_NO_ERROR) {
        //setup answer timer
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        const tConnectionState connectionStateDefault = IsProtocol_IAP1(protocol) ? CS_HW_MALFUNCTION : CS_UNSUPPORTED; //iAP1 fallback!
        ParameterON_DEVICE_ATTACH(OUT parameterString, IN size,
                IN mountPoint, IN connectionStateDefault, IN protocol);
        const int timeoutval = IsProtocol_IAP2HostMode(protocol) ? 25000 : 20000; //GMMY16-12646
        RegisterReleaseEvent("ON_DEVICE_ATTACH", parameterString, timeoutval);

        //Store the MountPoint of the device that is under initialization
        strncpy_r(m_DeviceToInitMountPoint, mountPoint, sizeof(m_DeviceToInitMountPoint));
    } else {
      ETG_TRACE_ERR(("iPodControl::IsInitDeviceConnection failed %d", ret));
    }
    return (ret == MP_NO_ERROR);
}

tResult iPodControl::Disconnect(const tMountPoint mountPoint, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(protocol);
    tResult ret = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        ret = iAP1_DisconnectDeviceConnection(mountPoint, true);
    } else if(IsProtocol_IAP2(protocol)) {
        ret = iAP2_DisconnectDeviceConnection(mountPoint, true);
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        ret = MP_ERR_IPOD_INIT;
    }
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::iAP_DisconnectDeviceConnection failed"));
    }

    if(IsProtocol_BT(protocol) && !LocalSPM::GetDataProvider().IgnoreBTfallbackFromIAP()) {
        //IAP did not work, role back to DTY_BLUETOOTH
        if(LocalSPM::GetDataProvider().RequestIAP2SPPConnection())
        {

            tDeviceInfo deviceInfo;
            iAPGetDeviceInfo(mountPoint,deviceInfo);
            //BT mountpoint is replaced with RFCOMM ID for BTY_IPHONE, so update DB mount point with the copy of actual mount point (accessoryName)
            SendDBChangeDeviceTypeAndMountPoint(IN deviceInfo.deviceID,IN DTY_BLUETOOTH, IN deviceInfo.accessoryName);
        }
        else
        {
            SendDBChangeDeviceType(IN iAPGetDeviceID(mountPoint), IN DTY_BLUETOOTH);
        }

    } else {
        //send indexer answer
        SendDeviceInitialized(mountPoint);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::OnDeviceInitialized(const tMountPoint mountPoint,
        const tConnectionState connectionState, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(connectionState);
    VARTRACE(protocol);

    const tConnectionType connectionType = iAPGetConnectionType(mountPoint);
    VARTRACE(connectionType);
    tResult ret = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        ret = iAP1_OnDeviceInitialized(mountPoint, connectionState);
    } else if(IsProtocol_IAP2(protocol)) {
        ret = iAP2_OnDeviceInitialized(mountPoint, connectionState);
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        ret = MP_ERR_IPOD_INIT;
    }

    if(ret == MP_NO_ERROR && CS_CONNECTED == connectionState) {
        //Apply shuffle and repeat settings from device
        if(LocalSPM::GetDataProvider().iPodControlUpdateShuffleAndRepeatFromDevice()) {
            SendPlaybackModeToPlayerManager(mountPoint);
            SendRepeatModeToPlayerManager(mountPoint);
        }
    }

    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::iAPOnDeviceInitialized failed"));
    } else if(connectionType != DCT_BLUETOOTH) { /* NCG3D-126828 - Blocked setting of DIPO_CAP_NONE for iAP BT devices */
        //set DiPOCaps
        const tDiPOCaps diPOCaps = iAP2_GetDiPOCaps(mountPoint);
        VARTRACE(diPOCaps);
        SendDBChangeDiPOSettings(IN iAPGetDeviceID(mountPoint), IN diPOCaps, IN FALSE, IN "");
    }

    //USB fallback
    if(IDP_IAP2 == protocol || IsProtocol_IAP2HostMode(protocol)) {
        if(CS_UNSUPPORTED == connectionState) {
            if(LocalSPM::GetDataProvider().iPodControlSupportIAP1()) {
                //try IAP1 legacy
                SendInitDevice(mountPoint, IDP_IAP1);
                return MP_NO_ERROR;
            }
        }
    }

    //BT2 fallback
    if(IDP_IAP2BT == protocol) {
        if(CS_UNSUPPORTED == connectionState) {
            if(LocalSPM::GetDataProvider().iPodControlSupportIAP1() &&
               LocalSPM::GetDataProvider().iPodControlSupportIAP1BT()) {
                //try IAP1 BT legacy
                SendInitDevice(mountPoint, IDP_IAP1BT);
                return MP_NO_ERROR;
            }
        }
    }

    //BT1 Fallback
    if((IDP_IAP1BT == protocol ||  IDP_IAP2BT == protocol)&& CS_CONNECTED != connectionState && !LocalSPM::GetDataProvider().IgnoreBTfallbackFromIAP()) {
        //IAP did not work, role back to DTY_BLUETOOTH
        if(LocalSPM::GetDataProvider().RequestIAP2SPPConnection())
        {
            tDeviceInfo deviceInfo;
            iAPGetDeviceInfo(mountPoint,deviceInfo);
            //BT mountpoint is replaced with RFCOMM ID for BTY_IPHONE, so update DB mount point with the copy of actual mount point (accessoryName)
            SendDBChangeDeviceTypeAndMountPoint(IN deviceInfo.deviceID,IN DTY_BLUETOOTH, IN deviceInfo.accessoryName);
        }
        else
        {
            SendDBChangeDeviceType(IN iAPGetDeviceID(mountPoint), IN DTY_BLUETOOTH);
        }
        //before recevies connection state as Connected if device removed then disconnection needs to be handled for CPW.
    } else if(IDP_IAP2_OVER_WIRELESS_CARPLAY == protocol) {
        //when executes this conditon actual connection state may be changed to disconnected due to user performs immediate disconnection after connecting IAP2 WiFi
        tConnectionState currentConnectionState = iAPGetConnectionState(mountPoint);
        VARTRACE(currentConnectionState);
        if(CS_CONNECTED == currentConnectionState)
        {
            SendDBChangeConnectionState(IN iAPGetDeviceID(mountPoint),IN CS_ATTACHED);
        }
        else
        {
            SendDBChangeConnectionState(IN iAPGetDeviceID(mountPoint),IN currentConnectionState);
        }
    }
    else {
        SendDeviceInitialized(mountPoint);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::GetNumberOfFiles(const tDeviceID deviceID)
{
    //Device type is not supported for indexing, nevertheless indexer requires the total number of playable files from this device
    //Added for VAG iPod streaming

    ENTRY;
    VARTRACE(deviceID);

    tNumberOfFiles numberOfFiles = 1; //TODO: return total number of files here

    tResult ret = iAPGetNumberOfFiles(deviceID, numberOfFiles);
    if(ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPGetNumberOfFiles failed"));
    }
    /* Send ANSWER message to IndexerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIndexer().ParameterNUMBER_OF_FILES_ANSWER(OUT parameterString, IN size, IN deviceID, IN numberOfFiles);
    SendAnswer(IN parameterString);

    return MP_NO_ERROR;
}

tResult iPodControl::CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(lastFingerprint);

    tResult ret = MP_NO_ERROR;
    tFingerprint fingerprint = { 0 };
    tFingerprintStatus fingerprintStatus = FPS_NOT_AVAIL;
    tNumberOfFiles numberOfFiles = NUMBER_OF_FILES_NONE;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    StopFingerprintTimer(); //stop obsolete fingerprint timer

    if(iAP2_IsSet(mountPoint)){
        iAPIndexerGetFingerprint(mountPoint, fingerprint);
        if(fingerprint[0] != 0) {
            //fingerprint already available, in case of
            //REINDEXING
            ETG_TRACE_USR3(("iAP2_CalcFingerPrint() - fingerprint already available"));
            fingerprintStatus = FPS_OK_DELTA;
            //numberOfFiles = iAPIndexerGetTrackCount(mountPoint);
            ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerprintStatus, IN numberOfFiles, IN deviceID);
            SendEvent(PUT_FINGERPRINT, IN parameterString);
        } else {
            //start mediaLibrary updates on iPod
            ret = iAP2_CalcFingerPrint(mountPoint, lastFingerprint);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iAP2_CalcFingerPrint() failed"));
                fingerprintStatus = FPS_ERROR;
                ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerprintStatus, IN numberOfFiles, IN deviceID);
                SendEvent(PUT_FINGERPRINT, IN parameterString);
            } else {
                //Timeout for CBIAP2_MediaLibraryUpdate, indexer communication changed from messageAnswer to simple event
                StartFingerprintTimer(deviceID);
            }
        }
    } else {
        ret = iAP1_CalcFingerPrint(fingerprint, fingerprintStatus, numberOfFiles, mountPoint, lastFingerprint);
        if (ret != MP_NO_ERROR) {
            fingerprintStatus = FPS_ERROR;
            ETG_TRACE_ERR(("iPodControl::CalcFingerprint - iAPCalcFingerPrint() failed"));
        }
        if(ret == MP_NO_ERROR && LocalSPM::GetDataProvider().iPodControlIAP1UIDBasedIndexingSupport() && strcmp(fingerprint, lastFingerprint) != 0) {
            ret = ParameterFILE_COUNT_COMP(OUT parameterString,IN size,IN mountPoint,IN ++m_IndexSession);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret = SendEvent(FILE_COUNT_COMP,IN parameterString);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }else {
            ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerprintStatus, IN numberOfFiles, IN deviceID);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::OnFingerprint(const tFingerprint fingerprint,
        const tFingerprintStatus fingerprintStatus,
        const tNumberOfFiles numberOfFiles, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(fingerprint);
    VARTRACE(fingerprintStatus);
    VARTRACE(numberOfFiles);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;

    /* Send FINGERPRINT_AVAIL message to IndexerSM */
    char messageString[64];
    strncpy_r(messageString, "IndexerSM::FINGERPRINT_AVAIL",
            sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetIndexer().ParameterFINGERPRINT_AVAIL(
            OUT parameterString, IN size, IN fingerprint, IN fingerprintStatus,
            IN numberOfFiles, IN deviceID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::OnFingerprint - Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    //ret = SendAnswer(IN parameterString);
    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);

    //Don't return an error because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::IsNextMetadataAvailable(const tMountPoint mountPoint,
        const tReadPosition readPosition,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(mountPoint);
    VARTRACE(readPosition);

    tResult ret = MP_NO_ERROR;
    bool avail = false;
    /* Return we have in metadata cache */
    ret = iAPIndexerIsNextMetadataAvailable(mountPoint, avail);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPIndexerIsNextMetadataAvailable() failed"));
    }

    //check indexing resume
    iAPIndexerCheckResume(mountPoint);
    return avail;
}

tResult iPodControl::RetrieveMetadataFromDevice(const tMountPoint mountPoint,
        const tReadPosition readPosition,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(readPosition);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;
    if(iAP2_IsSet(mountPoint)){
        tFingerprint fingerprint;
        iAPIndexerGetFingerprint(mountPoint, fingerprint);
        VARTRACE(fingerprint);
        if(fingerprint[0] == 0 || iAPIsMediaLibraryUpdateCompleted(mountPoint)) {
            //indexing is finished, send MDS_FINISHED thru SendPutMetadata and iAPIndexerGetNextMediaObject
            //OR fingerprint is not available, CalcFingerprint need to be triggered first, see MDS_ERROR_RESTART handling [GMMY16-24058]
            SendPutMetadata(mountPoint, readPosition, deviceID);
        } else {
            //indexer cache ran empty or IAP2MediaLibraryUpdates delayed
            //wait for event METADATA_FOR_INDEX
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            ParameterMETADATA_FOR_INDEX(OUT parameterString, IN size, IN mountPoint, IN readPosition, IN deviceID);
            RegisterReleaseEvent("METADATA_FOR_INDEX", parameterString, 5000); //5secs timeout
        }
    } else {
        ret = iAPIndexerRetrieveMetadataFromDevice(IN mountPoint, IN readPosition, IN deviceID);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iAPIndexerRetrieveMetadataFromDevice() failed"));
        }
        SendPutMetadata(mountPoint, readPosition, deviceID);
    }
    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SendPutMetadata(const tMountPoint mountPoint,
        const tReadPosition readPosition,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(readPosition);
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;
    tMetadataStatus metadataStatus = MDS_DEVICE_ERROR;
    tMediaObjectPtr mediaObjectPtr = NULL;

    ret = iAPIndexerGetNextMediaObject(OUT metadataStatus, OUT mediaObjectPtr, IN mountPoint, IN readPosition);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPIndexerGetNextMediaObject() failed"));
    }

    /* Send PUT_METADATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterPUT_METADATA(OUT parameterString, IN size, IN metadataStatus, IN mediaObjectPtr);
    SendEvent(IN PUT_METADATA, IN parameterString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::ReceivedMetadata(const tMountPoint mountPoint,
        const tReadPosition readPosition,
        const tDeviceID deviceID)
{
    ENTRY;
    return SendPutMetadata(mountPoint, readPosition, deviceID);
}

tResult iPodControl::AnswerMetadata(const tMetadataStatus metadataStatus, const tMediaObjectPtr mediaObjectPtr)
{
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::AnswerMetadata()"));
    VARTRACE(metadataStatus);

    tResult ret = MP_NO_ERROR;

    /* Send METADATA_ANSWER message to IndexerSM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetIndexer().ParameterMETADATA_ANSWER(OUT parameterString,
            IN size, IN mediaObjectPtr, IN metadataStatus);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::AnswerMetadata() - Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    ret = SendAnswer(IN parameterString);

    //Don't return an error because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::RetrieveMetadataFromCache(tMetadata &/*metadata*/)
{
    /* Obsolete: DeviceControl is sending a media object instead of a metatdata string so an assembly is not necessary */
    ENTRY return MP_NO_ERROR;
}

tResult iPodControl::RemoveDeviceConnection(const tMountPoint mountPoint,
        const tDeviceID deviceID)
{
    ENTRY;
    //during HostMode, Device connection state is handle by IAP exclusively,
    //on detach callback REMOVE_DEVICE_CONNECTION_HOSTMODE is triggered.
    if(!iAP2_IsHostMode(mountPoint)) {

        const bool isWaitingforRoleSwitchResponse = iAPGetWaitingforRoleSwitchResponse(mountPoint);
        /* When device is disconnected before receiving role switch request response, device will be in device mode will update SPI on Communication error */
        if(isWaitingforRoleSwitchResponse && LocalSPM::GetDataProvider().DipoCommunicationError())
        {
            LocalSPM::GetOutputWrapper().UpdateDipoCommunicationError(deviceID,IPOD_COMM_ERROR, mountPoint);
            iAPSetWaitingforRoleSwitchResponse(mountPoint, false);
        }

        RemoveDeviceConnectionForced(mountPoint, deviceID);
    } else {
        ETG_TRACE_USR3(("RemoveDeviceConnection called on HOSTMODE"));
        if(LocalSPM::GetDataProvider().iPodControlIAP2CarPlayModeEnabled()) {
            //Update connection state - GMMY16-26410
            const tConnectionState connectionState = iAPGetConnectionState(mountPoint);
            VARTRACE(connectionState);
            if(connectionState == CS_CONNECTED) {
                //CarPlay device set to CS_ATTACHED, via Indexer set to CS_CONNECTED automatically
                //NOTE: Direct set to CS_CONNECTED is invalid/blocked in DB schema
                ETG_TRACE_USR3(("Set DB device ATTACHED"));
                SendDBChangeConnectionState(IN deviceID, IN CS_ATTACHED);
            }
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::RemoveDeviceConnectionForced(const tMountPoint mountPoint,
        const tDeviceID deviceID)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    if(iAP_IsDeviceHandleAvail(mountPoint)) {
        if(!iAP2_IsSet(mountPoint)) {
            ret = iAP1_DisconnectDeviceConnection(mountPoint, true);
        } else {
            if (m_FingerprintTimerID) {
                StopFingerprintTimer(deviceID); //stop obsolete fingerprint timer
                tFingerprint fingerprint = { 0 };
                tFingerprintStatus fingerprintStatus = FPS_ERROR;
                tNumberOfFiles numberOfFiles = 0;
                OnFingerprint(fingerprint, fingerprintStatus, numberOfFiles, deviceID);
            }
            ret = iAP2_DisconnectDeviceConnection(mountPoint, true);
        }
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iAPDisconnectDeviceConnection failed"));
        }
    } else {
        ETG_TRACE_USR3(("No connection found. Device may got already disconnected by IAP callback."));
    }

    //clear RemoteActivity flag on disconnection - GMMY17-11106
    if(deviceID == m_RemoteActivityDeviceID) {
        m_RemoteActivityDeviceID = DEVICE_ID_NOT_SET;
    }

    //clear DiPO parameters
    //reset DiPOCaps
    SendDBChangeDiPOSettings(IN deviceID, IN DIPO_CAP_NONE, IN FALSE, IN "");

    //NOTE: iPod has been disconnected while playing, audio stream needs to be closed (Playerengine::Stopslot)
    return MP_NO_ERROR;
}

tResult iPodControl::RemoveDeviceConnectionOnHold(const tMountPoint mountPoint,
        const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;
    RemoveDeviceConnectionForced(mountPoint, deviceID);

    //Set to CS_ON_HOLD
    SendDBChangeConnectionState(IN deviceID, IN CS_ON_HOLD);

    return ret;
}

//AlbumArt part

tResult iPodControl::GetAlbumArt(const tAlbumArt albumArtString)
{
    ENTRY;
    VARTRACE(albumArtString);

    tResult ret = MP_NO_ERROR;

    //retreive mount point from albumart string, prefixed
    tAlbumArtObjectPtr pAlbumArtObjectPtr = NULL;
    tMountPoint mountPoint = {0};
    tListType lty;
    int tag1;
    int tag2;
    int tag3;
    int tag4;
    int track;
    int chapter;
    tU64 uuid = 0;
    tU64 parentuuid = 0;
    tMetadata title = { 0 };
    int specialID = 0;
    lty = LTY_END_OF_EXTERNAL_LIST_TYPES;

    SMF::UnMarshalFromUtf8(albumArtString, IPODCONTROL_MARSHAL_SEPARATOR,
            IPODCONTROL_MARSHAL_FORMAT, mountPoint, &lty, &tag1, &tag2, &tag3, &tag4, &track,
            &chapter, &uuid, &parentuuid, title, &specialID);

    VARTRACE(mountPoint);
    VARTRACE(lty);
    VARTRACE(tag1);
    VARTRACE(tag2);
    VARTRACE(tag3);
    VARTRACE(tag4);
    VARTRACE(track);
    VARTRACE(chapter);
    VARTRACE(uuid);
    VARTRACE(parentuuid);
    VARTRACE(title);

    if (mountPoint[0] == 0) {
        ETG_TRACE_ERR(("Invalid mountpoint failed"));
        ret = MP_ERR_IPOD_NO_DEVICE;
        //calling HandleGetAlbumArtAnswer through own sate machine
        SendAlbumArtAnswer(pAlbumArtObjectPtr);
    } else {
        iAPSetAlbumArtString(mountPoint, albumArtString);
        if(iAP2_IsSet(mountPoint)){
            ret = iAP2_GetAlbumArt(mountPoint, title, uuid, pAlbumArtObjectPtr);
            //calling HandleGetAlbumArtAnswer through own sate machine
            SendAlbumArtAnswer(pAlbumArtObjectPtr);
        } else {
            ret = iAP1_GetAlbumArt(mountPoint, track, uuid);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_USR3(("iPodControl::GetAlbumArt() iAPGetAlbumArt failed"));
                //calling HandleGetAlbumArtAnswer through own sate machine
                SendAlbumArtAnswer(pAlbumArtObjectPtr);
            } else {
                //waiting for an async iAP callback for album art data
                RegisterReleaseEvent(GET_ALBUM_ART_ANSWER, NULL, 10000); //10secs timeout
            }
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::HandleGetAlbumArtAnswer(
        const tAlbumArtObjectPtr ptrToAlbumArtObject)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterGET_ALBUM_ART_ANSWER(parameterString, size, IN ptrToAlbumArtObject);
    ret = SendAnswer(IN parameterString);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::HandleGetAlbumArtAnswer() - Error while sending answer via SMF"));
        if (ptrToAlbumArtObject) {
            if (ptrToAlbumArtObject->imageData) {
                delete[] ptrToAlbumArtObject->imageData;
            }
            delete ptrToAlbumArtObject;
            ETG_TRACE_USR3(("AlbumArt buffer cleared"));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::AlbumArtAnswerNotConsumed(
        const tAlbumArtObjectPtr ptrToAlbumArtObject)
{
    ENTRY;
    if(ptrToAlbumArtObject) {
        if(ptrToAlbumArtObject->imageData) {
            delete [] ptrToAlbumArtObject->imageData;
        }
        delete ptrToAlbumArtObject;
        ETG_TRACE_USR3(("AlbumArt buffer cleared"));
    }
    return MP_NO_ERROR;
}

tResult iPodControl::SendTransferTags(const tMountPoint mountPoint,
            const vector<tTagTransfer> transferTags,
            tTagTransferStatus &transferStatus,
            vector<tUntransferredTag> &untransferredTags)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint);
    VARTRACE(transferTags);

    tResult res = MP_NO_ERROR;
    transferStatus = TAG_TRANSFER_SUCCESS;

    for (unsigned int i = 0; i < transferTags.size(); i++) {
        if (transferStatus == TAG_TRANSFER_SUCCESS) {
            //create xml file to be written to iPod
            tFilename tagFile = {0};
            res = CreateTagFile(tagFile, transferTags[i]);
            if (MP_NO_ERROR != res) {
                ETG_TRACE_ERR(("%s - CreateTagFile failed", __PRETTY_FUNCTION__));
                transferStatus = TAG_TRANSFER_COMM_ERROR;
            } else {
                TransferTagRR transferTagRR;
                char msgToSendString[64];
                strncpy_r(msgToSendString, "iPodControlSM::TRANSFER_TAG",
                        sizeof(msgToSendString));
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterTRANSFER_TAG(parameterString, size, mountPoint, tagFile);
                res = transferTagRR.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 10000);
                if (MP_NO_ERROR != res) {
                    ETG_TRACE_ERR(("%s - DoEventAnswer failed", __PRETTY_FUNCTION__));
                    transferStatus = TAG_TRANSFER_COMM_ERROR;
                } else {
                    transferStatus = transferTagRR.transferStatus;
                }
                //remove tmp file
                remove(tagFile);
            }
        }
        //check for errors
        if (transferStatus != TAG_TRANSFER_SUCCESS) {
            //error, fill untransferredTags
            tUntransferredTag untransferredTag;
            InitUntransferredTag(untransferredTag);
            untransferredTag.tagSongID = transferTags[i].tagSongID;
            strncpy_r(untransferredTag.tagTrackName, transferTags[i].tagTrackName, sizeof(untransferredTag.tagTrackName));
            strncpy_r(untransferredTag.tagArtistName, transferTags[i].tagArtistName, sizeof(untransferredTag.tagArtistName));
            untransferredTags.push_back(untransferredTag);
        }
    }

    VARTRACE(transferStatus);
    VARTRACE(untransferredTags);
    return res;
}

tResult iPodControl::TransferTag(const tMountPoint mountPoint, const tFilename tagfile)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint);
    VARTRACE(tagfile);

    tTagTransferStatus status = TAG_TRANSFER_COMM_ERROR;
    tResult ret = iAPTransferTag(status, mountPoint, tagfile);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::TransferTag() iAPTransferTag failed"));
    }
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterTRANSFER_TAG_ANSWER(OUT parameterString, IN size, IN status);
    SendEvent(TRANSFER_TAG_ANSWER, IN parameterString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::TransferTagAnswer(const tTagTransferStatus status)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(status);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterTRANSFER_TAG_ANSWER(OUT parameterString, IN size, IN status);
    SendAnswer(IN parameterString);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SendAppControlConnect(const tMountPoint mountPoint, const tProtocolName protocol,
        const tBundleSeedID bundleSeedID, const tAppName appName, const tAppLaunchOption launchOption)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(protocol);
    VARTRACE(bundleSeedID);
    VARTRACE(appName);
    VARTRACE(launchOption)
    tResult retVal = MP_NO_ERROR;

    //dispatch from service thread
    const size_t size = sizeof(tAllParameters);
    char *parameterString = new char[size]; //deleted by DoAppControlConnect
    ParameterAPPCONTROL_CONNECT(parameterString, size, mountPoint, protocol, bundleSeedID, appName, launchOption);

    LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_APPCONTROL_CONNECT, IN parameterString, this);

    VARTRACE(retVal)
    return retVal; //lint !e429 deleted by DoAppControlConnect
}

tResult iPodControl::DoAppControlConnect(char *parameterString)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(parameterString)
    tResult retVal = MP_NO_ERROR;

    iPodControlRR appControlRR;
    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::APPCONTROL_CONNECT",
            sizeof(msgToSendString));
    retVal = appControlRR.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);
    if (parameterString) {
        delete[] parameterString; //new by SendAppControlConnect
    }
    VARTRACE(retVal)
    return retVal;
}

tResult iPodControl::AddAppControlInfo(const tMountPoint mountPoint, const tProtocolName protocol,
        const tBundleSeedID bundleSeedID, const tAppName appName, const tAppLaunchOption launchOption)
{
    ENTRY;
    VARTRACE(mountPoint)
    VARTRACE(protocol)
    VARTRACE(bundleSeedID)
    VARTRACE(appName)
    VARTRACE(launchOption)

    const tBoolean deviceAvail = iAP_IsDeviceHandleAvail(mountPoint);
    VARTRACE(deviceAvail);

    bool added = false;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    if(deviceAvail && LocalSPM::GetDataProvider().iPodControlAppControlReauthEnabled()) { //SUZUKI-19198
        tResult retVal = iAPAddAppControlInfo(mountPoint, protocol, bundleSeedID, appName, added);
        if (retVal != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iAPAddAppControlInfo failed %d", retVal));
        }
    }

    VARTRACE(added);
    if (added) {
        iAPSetLaunchApp(mountPoint, appName);
        ParameterRECONNECT(OUT parameterString, IN size, IN mountPoint);
        SendEvent(RECONNECT, IN parameterString);
    } else {
        //Changed appName to bundleSeedID for launch app - Carlife
        //RTC-426240 Currently the understanding is bundleSeedID is coming as empty only for PSA where AppName should be passed to the LaunchApp message
        if('\0' == bundleSeedID[0]){
            ParameterLAUNCH_APP(OUT parameterString, IN size, IN mountPoint, IN appName, IN launchOption);
        } else {
        ParameterLAUNCH_APP(OUT parameterString, IN size, IN mountPoint, IN bundleSeedID, IN launchOption);
        }
        SendEvent(LAUNCH_APP, IN parameterString);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::RemoveAppControlInfo(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID)
{
    ENTRY;
    VARTRACE(mountPoint)
    VARTRACE(appName)
    VARTRACE(sessionID)

    const tBoolean deviceAvail = iAP_IsDeviceHandleAvail(mountPoint);
    VARTRACE(deviceAvail);

    bool removed = false;
    if(deviceAvail && LocalSPM::GetDataProvider().iPodControlAppControlReauthEnabled()) { //SUZUKI-19198
        tResult retVal = iAPRemoveAppControlInfo(mountPoint, appName, sessionID, removed);
        if (retVal != MP_NO_ERROR) {
                ETG_TRACE_ERR(("iAPRemoveAppControlInfo failed %d", retVal));
        }
    }
    VARTRACE(removed);
    if (removed || LocalSPM::GetDataProvider().iPodControlAppControlReconnectEnabled()) { //PSARCC-11861
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterRECONNECT(OUT parameterString, IN size, IN mountPoint);
        SendEvent(RECONNECT, IN parameterString);
    } else {
        //release the calling outer SM
        SendEvent(APPCONTROL_ANSWER, IN (char*) NULL);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::AppControlConnectReconnected(const tMountPoint mountPoint,
            const tConnectionState connectionState, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(connectionState);
    VARTRACE(protocol);

    tResult retVal = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        retVal = iAP1_OnDeviceInitialized(mountPoint, connectionState);
    } else if(IsProtocol_IAP2(protocol)) {
        retVal = iAP2_OnDeviceInitialized(mountPoint, connectionState);
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        retVal = MP_ERR_IPOD_INIT;
    }

    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPOnDeviceInitialized failed"));
    }

    tAppName appName = { 0 };
    iAPGetLaunchApp(mountPoint, appName);

    retVal = LaunchApp(mountPoint, appName, tAppLaunchOption_init);
    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s LaunchApp failed", __PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::AppControlCloseReconnected(const tMountPoint mountPoint,
        const tConnectionState connectionState, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(connectionState);
    VARTRACE(protocol);

    tResult retVal = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        retVal = iAP1_OnDeviceInitialized(mountPoint, connectionState);
    } else if(IsProtocol_IAP2(protocol)) {
        retVal = iAP2_OnDeviceInitialized(mountPoint, connectionState);
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        retVal = MP_ERR_IPOD_INIT;
    }

    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPOnDeviceInitialized failed"));
    }

    //release the calling outer SM
    retVal = AppControlAnswer();
    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s AppControlAnswer failed", __PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::AppControlAnswer()
{
    ENTRY;

    //release the calling outer SM
    SendAnswer(NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SendAppControlCommand(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer,
        const tCommandBufferPtr commandBuffer, const tUserContext userContext)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint);
    VARTRACE(appName);
    VARTRACE(sessionID);
    VARTRACE(sizeOfBuffer);
    iAPTraceData(commandBuffer, sizeOfBuffer);
    tResult retVal = MP_NO_ERROR;

    tCommandBufferPtr commandBufferCopy = new unsigned char[sizeOfBuffer]; //deleted by DoAppControlCommand
    if(commandBufferCopy){
        memcpy(commandBufferCopy, commandBuffer, sizeOfBuffer);
    }

    const size_t size = sizeof(tAllParameters);
    char *parameterString = new char[size]; //deleted by DoAppControlCommand
    SMF::Marshal(parameterString, size-1, IPODCONTROL_MARSHAL_FORMAT_APPCONTROL_COMMANDBUFFER tSourceAppID_format tRegisterID_format tCmdCounter_format tFunctionID_format tServiceID_format, mountPoint, appName, sessionID, sizeOfBuffer, commandBufferCopy, userContext.sourceAppID, userContext.registerID, userContext.cmdCounter, userContext.functionID, userContext.serviceID);

    LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_APPCONTROL_COMMAND, IN parameterString, this);

    VARTRACE(retVal)
    return retVal; //lint !e429 deleted by DoAppControlCommand
}

tResult iPodControl::DoAppControlCommand(char *parameterStr)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(parameterStr);
    tResult retVal = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    tAppName appName = {0};
    tSessionID sessionID = 0;
    tSize sizeOfBuffer = 0;
    tCommandBufferPtr commandBuffer = NULL;
    tUserContext userContext;

    //unpack and forward mountpoint
    SMF::UnMarshal(parameterStr, IPODCONTROL_MARSHAL_FORMAT_APPCONTROL_COMMANDBUFFER tSourceAppID_format tRegisterID_format tCmdCounter_format tFunctionID_format tServiceID_format, mountPoint, appName, &sessionID, &sizeOfBuffer, &commandBuffer, &userContext.sourceAppID, &userContext.registerID, &userContext.cmdCounter, &userContext.functionID, &userContext.serviceID);
    if (parameterStr) {
        delete[] parameterStr; //new by SendDiPORoleSwitchRequest
    }

    tAllParameters parameterString;
    const size_t size = sizeof(tAllParameters);
    ParameterAPPCONTROL_COMMAND(parameterString, size, mountPoint, appName, sessionID, sizeOfBuffer, commandBuffer);

    iPodControlRR appControlRR;
    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::APPCONTROL_COMMAND", sizeof(msgToSendString) - 1);
    retVal = appControlRR.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);

    if(commandBuffer) {
        delete[] commandBuffer; //new by SendAppControlCommand
    }

    LocalSPM::GetOutputWrapper().SendCommandAnswer(mountPoint, appName, sessionID, userContext);
    VARTRACE(retVal)
    return retVal;
}

tResult iPodControl::AppControlCommand(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer,
        const tCommandBufferPtr commandBuffer)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint)
    VARTRACE(appName)
    VARTRACE(sessionID)
    VARTRACE(sizeOfBuffer)

    tResult ret = iAPAppControlCommand(mountPoint, appName, sessionID,
            sizeOfBuffer, commandBuffer);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAPAppControlCommand failed", __PRETTY_FUNCTION__));
    }
    //release the calling outer SM
    SendEvent(APPCONTROL_ANSWER, IN (char*) NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SendAppControlClose(const tMountPoint mountPoint,
        const tAppName appName, const tSessionID sessionID)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(mountPoint)
    VARTRACE(appName)
    VARTRACE(sessionID)
    tResult retVal = MP_NO_ERROR;

    const size_t size = sizeof(tAllParameters);
    char *parameterString = new char[size]; //deleted by DoAppControlClose
    ParameterAPPCONTROL_CLOSE(parameterString, size, mountPoint, appName,
            sessionID);

    LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_APPCONTROL_CLOSE, IN parameterString, this);

    VARTRACE(retVal)
    return retVal; //lint !e429 deleted by DoAppControlClose
}

tResult iPodControl::DoAppControlClose(char *parameterString)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(parameterString)
    tResult retVal = MP_NO_ERROR;

    iPodControlRR appControlRR;
    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::APPCONTROL_CLOSE",
            sizeof(msgToSendString) - 1);

    retVal = appControlRR.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);
    if (parameterString) {
        delete[] parameterString; //new by SendAppControlClose
    }
    VARTRACE(retVal)
    return retVal;
}

tResult iPodControl::DoStopPlaybackOnDetach(char *mountPoint)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

   if (!mountPoint) {
        ETG_TRACE_ERR(("mountPoint null pointer"));
        ret = MP_ERR_IPOD_NO_DEVICE;
    } else {
        VARTRACE(mountPoint);
        VARTRACE(m_ActivePBMountPoint);
        if (!strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint))) {

            if(iAP2_IsHostMode(mountPoint)) {
                ETG_TRACE_USR3(("HostMode playback stopped"));
            } else {
                if (LocalSPM::GetDataProvider().UseMediaEngine()) {
                    //ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::STOP", NULL, "iPodControl::PLAYBACK_STATUS");
                    //call MediaEngine STOP from a dedicated SM
                    iPodControlRR iPodControl_rr;
                    char msgToSendString[64];
                    strncpy_r(msgToSendString, "MediaEngineSM::STOP", sizeof(msgToSendString));
                    ret = iPodControl_rr.DoEventAnswer(IN msgToSendString, IN NULL, NULL, 15000);
                    if (MP_NO_ERROR != ret) {
                        ETG_TRACE_ERR(("iPodControl::DoStopPlaybackOnDetach() - PE STOP failed"));
                    }
                } else {
                    ret = LocalSPM::GetIPCProvider().StopSlot(IN this);
                    if (ret != MP_NO_ERROR) {
                        ETG_TRACE_ERR(("Stopping PE failed"));
                    }
                }
            }
        }

        //return from AlbumArt waiting state
        SendAlbumArtAnswer(NULL);

        //notify app control clients about data session end
        iAPUpdateCloseAllActiveSessions(mountPoint);

        delete[] mountPoint; //new by OnCBIAP1_USBDetach
    }
    VARTRACE(ret);
    return ret;
}

void iPodControl::DoIAP2Poll(char* mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);

    if(mountPoint == NULL) {
        ETG_TRACE_ERR(("mountPoint == NULL"));
        return;
    }

    iAP2_PollFDs(mountPoint);

    delete[] mountPoint; //new by SM thread
}

tResult iPodControl::Reconnect(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint)

    //retrieve protocol from internal BEFORE DISCONNECT!
    tInitDeviceProtocol protocol = IDP_IAP1;
    const tConnectionType connectionType = iAPGetConnectionType(mountPoint);
    VARTRACE(connectionType);

    if(connectionType == DCT_BLUETOOTH) {
        protocol = iAP2_IsSet(mountPoint) ? IDP_IAP2BT : IDP_IAP1BT;
    } else if(iAP2_IsSet(mountPoint)) {
        if(!iAP2_IsHostMode(mountPoint)) {
            protocol = IDP_IAP2;
        } else if(iAP2_IsCarPlayMode(mountPoint) && iAP2_IsNativeTransportMode(mountPoint) ) {
            protocol = IDP_IAP2CARPLAY_NATIVE_TRANSPORT;
        } else if(iAP2_IsCarPlayMode(mountPoint)) {
            protocol = IDP_IAP2CARPLAY;
        } else if(iAP2_IsNativeTransportMode(mountPoint)){
            protocol = IDP_IAP2NATIVE_TRANSPORT;
        }
        VARTRACE(protocol);
    }

    tResult retVal = iAP2_IsHostMode(mountPoint) ? RemoveDeviceConnectionForced(mountPoint, iAPGetDeviceID(mountPoint)) : RemoveDeviceConnection(mountPoint, iAPGetDeviceID(mountPoint));
    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAPDisconnectDeviceConnection failed", __PRETTY_FUNCTION__));
    }

    //connect again using existing zargo function
    if(!IsInitDeviceConnection(mountPoint, protocol)) {
        retVal = MP_ERR_IPOD_INIT;
        ETG_TRACE_ERR(("Reconnect failed"));

        //send event directly
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterON_DEVICE_ATTACH(OUT parameterString, IN size, IN mountPoint, IN CS_HW_MALFUNCTION, protocol);
        SendEvent(ON_DEVICE_ATTACH, IN parameterString);
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::LaunchApp(const tMountPoint mountPoint, const tAppName appName, const tAppLaunchOption launchOption)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(appName);
    VARTRACE(launchOption);

    tResult retVal = iAPLaunchApp(mountPoint, appName, launchOption);
    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAPLaunchApp failed", __PRETTY_FUNCTION__));
    }

    //clear launch app
    iAPSetLaunchApp(mountPoint, "");

    //release the calling outer SM
    retVal = AppControlAnswer();
    if (retVal != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s AppControlAnswer failed", __PRETTY_FUNCTION__));
    }

    VARTRACE(retVal)

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

//iPod specific part

tResult iPodControl::CurrentSelectionChanged(tBoolean &changed, const tDeviceID deviceID)
{
    ENTRY;
    //Deliver flag if selection was changed via IAP since device connection on specific device
    return iAPCurrentSelectionChanged(changed, deviceID);
}

tResult iPodControl::GetCurrentPlaytime(tPlaytime &playtime, const tDeviceID deviceID)
{
    ENTRY;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(IN deviceID, OUT mountPoint);
    VARTRACE(mountPoint);

    //Deliver current playtime on specific device
    playtime = iAPGetElapsedPlaytime(IN mountPoint);
    VARTRACE(playtime);
    return MP_NO_ERROR;
}

tBoolean iPodControl::IsIAP2(const tMountPoint mountPoint)
{
    ENTRY_INTERNAL;
    return iAP2_IsSet(mountPoint);
}

tResult iPodControl::CompareMediaObjects(tBoolean &identical,
        const tMediaObject media1, const tPlaytime playtime1,
        const tMediaObject media2, const tPlaytime playtime2)
{
    ENTRY;
    tMountPoint hiddev = { 0 };
    tListType lty_1;
    tListType lty_2;
    int tag1_1;
    int tag2_1;
    int tag3_1;
    int tag4_1;
    int tag1_2;
    int tag2_2;
    int tag3_2;
    int tag4_2;
    int trackindex_1;
    int trackindex_2;
    int chapterindex_1;
    int chapterindex_2;
    tU64 uuid_1 = 0;
    tU64 uuid_2 = 0;
    tU64 parentuuid_1 = 0;
    tU64 parentuuid_2 = 0;
    tMetadata name_1 = { 0 };
    tMetadata name_2 = { 0 };
    int specialID_1 = 0;
    int specialID_2 = 0;

    SMF::UnMarshalFromUtf8(media1.fileName, IPODCONTROL_MARSHAL_SEPARATOR,
            IPODCONTROL_MARSHAL_FORMAT, hiddev, &lty_1, &tag1_1, &tag2_1, &tag3_1, &tag4_1, &trackindex_1, &chapterindex_1, &uuid_1, &parentuuid_1, name_1, &specialID_1);
    SMF::UnMarshalFromUtf8(media2.fileName, IPODCONTROL_MARSHAL_SEPARATOR,
            IPODCONTROL_MARSHAL_FORMAT, hiddev, &lty_2, &tag1_2, &tag2_2, &tag3_2, &tag4_2, &trackindex_2, &chapterindex_2, &uuid_2, &parentuuid_2, name_2, &specialID_2);

    VARTRACE(media1.fileName);
    VARTRACE(media2.fileName);

    identical = FALSE; //last mode issue GMMY16-19553
    if(uuid_1 > 0 && uuid_2 > 0) {
        //COMPARE BY UUIDS
        identical = uuid_1 == uuid_2;
    } else if(name_1[0] != 0 && name_2[0] != 0 ) {
        //COMPARE BY NAMES
    identical = !strcmp(name_1, name_2);
    }

    if(identical) {
        VARTRACE(playtime1);
        VARTRACE(playtime2);
        const tPlaytime variance = playtime1 > playtime2 ? playtime1 - playtime2 : playtime2 - playtime1;
        VARTRACE(variance);
        identical = variance < (tPlaytime)LocalSPM::GetDataProvider().iPodControlResumePositionVarianceMS();
    }
    VARTRACE(identical);
    return MP_NO_ERROR;
}

tResult iPodControl::GetVTTagByURL(tTagID &tag, tListType &lty, tMetadata &name, tUUID &uuid, const char* url)
{
    ENTRY;
    tMountPoint hiddev = { 0 };
    tTagID tag1 = 0;
    tTagID tag2 = 0;
    tTagID tag3 = 0;
    tTagID tag4 = 0;
    int chapterindex = 0;
    tU64 uuid64 = 0;
    tU64 parentuuid = 0;
    name[0] = 0;
    uuid[0] = 0;
    tag = 0;
    int specialID = 0;

    VARTRACE(url);
    SMF::UnMarshalFromUtf8(url, IPODCONTROL_MARSHAL_SEPARATOR,
            IPODCONTROL_MARSHAL_FORMAT, hiddev, &lty, &tag1, &tag2, &tag3, &tag4, &tag, &chapterindex, &uuid64, &parentuuid, name, &specialID);

    if(uuid64 != 0) {
        snprintf(uuid, sizeof(uuid), IPODCONTROL_UUID_FORMAT, uuid64);
    }
    VARTRACE(name);
    VARTRACE(uuid);

    if(lty == LTY_IPOD_QUEUE_LIST) {
        lty = LTY_CURRENT_SELECTION;
    }
    if(lty == LTY_ITUNES_RADIO_STATION) {
        lty = LTY_PLAYLIST;
    }
    VARTRACE(lty);
    //increment tags for VTIPOD compatibility
    //see also iPodControlIAP::iAPRetrieveVTRecord
    tag++;
    return MP_NO_ERROR;
}

tResult iPodControl::StartPlaybackAction(const tDeviceType deviceType, //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint,
            const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount) //finished 0%
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(URL);
    VARTRACE(mountPoint);
    VARTRACE(playbackAction);
    VARTRACE(skipcount);
    tResult ret = MP_NO_ERROR;

    //this allows further finished signals to PlayerManager
    iAPSetTrackFinished(mountPoint, false);

    switch (playbackAction) {
    case PBA_PLAY:
        ret = iAPPlayResume(mountPoint);
        break;
    case PBA_PAUSE:
    case PBA_STOP:
        ret = iAPPlayPause(mountPoint);
        break;
    case PBA_PREV:
        ret = iAPPlayPrev(mountPoint, skipcount);
        break;
    case PBA_NEXT:
        ret = iAPPlayNext(mountPoint, skipcount);
        break;
    case PBA_FFWD_START:
        ret = iAPPlayFFStart(mountPoint);
        break;
    case PBA_FREV_START:
        ret = iAPPlayREWStart(mountPoint);
        break;
    case PBA_FREV_STOP:
    case PBA_FFWD_STOP:
        ret = iAPPlayFFREWStop(mountPoint);
        break;
    default:
        ETG_TRACE_ERR(("iPodControl::StartPlaybackAction unsupported action"));
        ret = MP_ERR_IPOD_COMMAND;
        break;
    }

    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::StartPlay iAPPlaybackAction failed"));
    }

    const tPEHandle handle = iAPGetPEHandle(mountPoint);
    const tPEPlaybackState iPodPBStatus = iAPGetCurrentPlaybackState(mountPoint);
    const me::reason_e reason = REASON_OK;
    const me::speed_e speed = ME_SPEED_NORMAL;

    //send answer
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterPLAYBACK_ACTION_ANSWER(OUT parameterString, IN size, IN handle, IN iPodPBStatus, IN reason, IN speed);
    SendEvent(PLAYBACK_ACTION_ANSWER, IN parameterString);

    //JAC2-2425
    //send explicit playback state to PM for iAP1 lecacy
    if(!iAPGetOption(mountPoint, hasExtPlayStatusOption)) {
        SendPlaybackStatus(mountPoint);
    }
    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::SetPlaybackMode(const tDeviceType deviceType,  //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tMountPoint mountPoint, const tPlaybackMode playbackMode)
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(mountPoint);
    VARTRACE(playbackMode);
    tResult ret = iAPSetPlaybackMode(mountPoint, playbackMode);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::SetPlaybackMode iAPSetPlaybackMode failed"));
    }
    // The IAP will update the correkt status by calling a callback in our software.
    // the HMI of GM is toggling the switch by itself (which is not ok from my point of view)
    // if the following line would be on, the user sees: (coming from shuffle off):
    // - shuffle goes on (done by HMI)
    // - shuffle goes of (because iPOD is not as fast as should be) -> done by the line which is now commented out
    // - shuffle goes on (iPOD is ready now and has called callback at our software)
    // to prevent this toggling the line is commented out:

    // thoemel: SendPlaybackModeToPlayerManager(mountPoint);

    return MP_NO_ERROR;
}

tResult iPodControl::SetRepeatMode(const tDeviceType deviceType,    //Roadmap 13008 RemoteControl
            const tDeviceID deviceID, const tMountPoint mountPoint,
            const tRepeatMode repeatMode)
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(mountPoint);
    VARTRACE(repeatMode);
    tResult ret = iAPSetRepeatMode(mountPoint, repeatMode);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::SetRepeatMode iAPSetRepeatMode failed"));
    }
    //send update to upper level
    //SendRepeatModeToPlayerManager(mountPoint);
    return MP_NO_ERROR;
}

//**************************** callbacks **************************************
tResult iPodControl::OnCBIAP1_USBDetach(const S32 iPodID) {
    ENTRY;
    VARTRACE(iPodID);
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPGetMountPoint failed"));
    } else if(iAPGetConnectionState(mountPoint) == CS_CONNECTED && ILocalSPM::GetState() == COMPONENT_STATE_RUNNING) {
        const size_t size = sizeof(tMountPoint);
        char *mountPointNew = new char[size]; //deleted by DoStopPlaybackOnDetach
        strncpy_r(mountPointNew, mountPoint, size);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_STOP_PLAYBACK_ON_DETACH, IN (void *)mountPointNew, this);
    } //lint !e429 deleted by DoStopPlaybackOnDetach
    return ret;
}

tResult iPodControl::OnCBIAP1_USBAttach(const S32 success, const S32 iPodID) {
    ENTRY;
    VARTRACE(success);
    VARTRACE(iPodID);
    tResult ret = MP_NO_ERROR;

    if (success <= IPOD_OK) {
        /* Status response: Send answer to waiting state machine */
        tConnectionState state = success == IPOD_OK ? CS_CONNECTED : CS_DISCONNECTED;
        tMountPoint mountPoint = { 0 };
        ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iPodControl::OnCBIAP1_USBAttach(%d) - iAPGetMountPoint failed", iPodID));
        }

        const tConnectionType connectionType = iAPGetConnectionType(mountPoint);
        VARTRACE(connectionType);

        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tInitDeviceProtocol protocol = (connectionType == DCT_BLUETOOTH) ? IDP_IAP1BT : IDP_IAP1;
        ret = ParameterON_DEVICE_ATTACH(OUT parameterString, IN size, IN mountPoint, IN state, IN protocol);
        if (MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("iPodControl::OnCBIAP1_USBAttach() - Error while preparing parameter string"));
        }
        return SendEvent(ON_DEVICE_ATTACH, IN parameterString);

    } else {
        //success value 1 means IPOD_DETECT, an interim state during authentication
        //that has no effect to the overall iPodControl state machine
        ETG_TRACE_USR3(("iPodControl::OnCBIAP1_USBAttach(%d) IPOD_DETECTED", iPodID));
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_Notification(IPOD_NOTIFY_TYPE type,
        IPOD_NOTIFY_STATUS status, const S32 iPodID) {
    ENTRY; //callback event notification
    VARTRACE((int)type);
    VARTRACE(iPodID);

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_Notification(%d) - iAPGetMountPoint failed", iPodID));
    } else {

        const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
        const tBoolean isExpected = !iAPIsTimeElapsed(IN mountPoint, IN pbNotifications, IN LocalSPM::GetDataProvider().iPodControlPBNotificationTimeoutMS());
        VARTRACE(isStreaming);
        VARTRACE(isExpected);

        switch (type) {
        case IPOD_EVT_FLOW_CTRL:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE IPOD_EVT_FLOW_CTRL %u", status.waitMs));
            break;
        case IPOD_EVT_RADIO_TAG_STATUS:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_RADIO_TAG_STATUS %d", status.notifyStatus));
            break;
        case IPOD_EVT_CAMERA_STATUS:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_CAMERA_STATUS %d", status.notifyStatus));
            break;
        case IPOD_EVT_CHARGING:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_CHARGING %d", status.availableCurrent));
            break;
        case IPOD_EVT_DB_CHANGED:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_DB_CHANGED %d", status.notifyStatus));
            break;
        case IPOD_EVT_NOW_FOCUS_APP:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_NOW_FOCUS_APP '%s'", status.focusApp ? (char*)status.focusApp : "<null>"));
            iAPSetFocusApp(mountPoint, (char*)status.focusApp);
            if (!isExpected || isStreaming) {
                iAP1CheckRemoteActivity(mountPoint, isStreaming);
            }
            break;
        case IPOD_EVT_SESSION:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_SESSION %d", status.sessionId));
            break;
        case IPOD_EVT_CMD_COMPLETE:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_CMD_COMPLETE"));
            break;
        case IPOD_EVT_IPOD_OUT:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_IPOD_OUT %d", status.notifyStatus));
            break;
        case IPOD_EVT_BT_STATUS:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_BT_STATUS"));
            break;
        case IPOD_EVT_APP_DISPLAY_NAME:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_APP_DISPLAY_NAME '%s'", status.focusApp ? (char*)status.focusApp : "<null>"));
            break;
        case IPOD_EVT_ASSIST_TOUCH:
            ETG_TRACE_USR3(("CALLBACK IPOD_NOTIFY_TYPE: IPOD_EVT_ASSIST_TOUCH %d", status.notifyStatus));
            break;
        default:
            ETG_TRACE_ERR(("CALLBACK IPOD_NOTIFY_TYPE: UNKNOWN %d", (int)type));
            break;
        }
    }
    return ret;
}

#ifdef TARGET_BUILD_GEN3
tResult iPodControl::OnCBIAP1_RemoteEventNotification(IPOD_STATE_INFO_TYPE eventNum, IPOD_REMOTE_EVENT_NOTIFY_STATUS eventData, const U32 iPodID) {
    ENTRY; //callback event notification
    VARTRACE((int)eventNum);
    VARTRACE(iPodID);

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_RemoteEventNotification(%d) - iAPGetMountPoint failed", iPodID));
    } else {

        const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
        const tBoolean isExpected = !iAPIsTimeElapsed(IN mountPoint, IN pbNotifications, IN LocalSPM::GetDataProvider().iPodControlPBNotificationTimeoutMS());
        const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));

        VARTRACE(isStreaming);
        VARTRACE(isExpected);
        VARTRACE(isActive);

        tBoolean shuffleChange = false;
        tBoolean repeatChange = false;

        switch (eventNum) {
        case IPOD_STATE_INFO_TRACK_POSITION_MS:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_TRACK_POSITION_MS %u", eventData.trackPosMS));
            break;
        case IPOD_STATE_INFO_TRACK_INDEX:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_TRACK_INDEX %u", eventData.trackIndex));
            break;
        case IPOD_STATE_INFO_CHAPTER_INFO:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_CHAPTER_INFO count %u", eventData.chapterInfo.chapterCount));
            break;
        case IPOD_STATE_INFO_PLAY_STATUS:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_PLAY_STATUS %u", (unsigned int)eventData.playStatus));
            break;
        case IPOD_STATE_INFO_MUTE_VOLUME:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_MUTE_VOLUME uiVol %u", eventData.muteUiVol.uiVol));
            break;
        case IPOD_STATE_INFO_POWER_BATTERY:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_POWER_BATTERY batteryLevel %u", eventData.powerBattery.batteryLevel));
            break;
        case IPOD_STATE_INFO_EQUALIZER_STATE:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_EQUALIZER_STATE %u", eventData.eqState));
            break;
        case IPOD_STATE_INFO_SHUFFLE:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_SHUFFLE %d", (int)eventData.shuffle));
            iAPSetPlaybackModeFromIPOD(mountPoint, (int)eventData.shuffle);
            SendClearVTCache(mountPoint);
            shuffleChange = true;
            break;
        case IPOD_STATE_INFO_REPEAT:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_REPEAT %d", (int)eventData.repeat));
            iAPSetRepeatModeFromIPOD(mountPoint, (int)eventData.repeat);
            repeatChange = true;
            break;
        case IPOD_STATE_INFO_DATE_TIME:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_DATE_TIME hour %u", eventData.currDateTime.hour));
            break;
        case IPOD_STATE_INFO_ALARM:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_ALARM")); //depricated
            break;
        case IPOD_STATE_INFO_BACKLIGHT:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_BACKLIGHT %u", eventData.backlight));
            break;
        case IPOD_STATE_INFO_HOLD_SWITCH:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_HOLD_SWITCH %u", eventData.holdSwitch));
            break;
        case IPOD_STATE_INFO_SOUND_CHECK:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_SOUND_CHECK %u", eventData.soundCheck));
            break;
        case IPOD_STATE_INFO_AUDIOBOOK_SPEED:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_AUDIOBOOK_SPEED %u", (unsigned int)eventData.audiobook));
            break;
        case IPOD_STATE_INFO_TRACK_POSITION_S:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_TRACK_POSITION_S %u", eventData.trackPosSec));
            break;
        case IPOD_STATE_INFO_MUTE_EXTENDED_VOLUME:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_MUTE_EXTENDED_VOLUME absVol %u", eventData.muteUiAbsoluteVol.absVol));
            break;
        case IPOD_STATE_INFO_TRACK_CAPABILITIES:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_TRACK_CAPABILITIES"));
            break;
        case IPOD_STATE_INFO_NUM_OF_TRACKS_IN_PLAYLIST:
            ETG_TRACE_USR3(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY IPOD_STATE_INFO_NUM_OF_TRACKS_IN_PLAYLIST %u", eventData.playEngineContent));
            break;
        default:
            ETG_TRACE_ERR(("CALLBACK IPOD_REMOTE_EVENT_NOTIFY: UNKNOWN %d", (int)eventNum));
            break;
        }

        if(repeatChange || shuffleChange) {
            if (!isExpected || isStreaming) {
                if((repeatChange && LocalSPM::GetDataProvider().iPodRemoteEventEnterStreamingOnRepeatMode()) ||
                   (shuffleChange && LocalSPM::GetDataProvider().iPodRemoteEventEnterStreamingOnPlaybackMode())) {
                    iAP1CheckRemoteActivity(mountPoint, isStreaming);
                }
            }
            //GMMY16-19547
            if(repeatChange && (!isActive || isStreaming || LocalSPM::GetDataProvider().SetDeviceRepeatModeNonStreaming())) {
                SendRepeatModeToPlayerManager(mountPoint);
            }
            if(shuffleChange && (!isActive || isStreaming || LocalSPM::GetDataProvider().SetDevicePlaybackModeNonStreaming())) {
                SendPlaybackModeToPlayerManager(mountPoint);
            }
        }
    }
    return ret;
}
#endif

tResult iPodControl::OnCBIAP1_NotifyStatus(IPOD_CHANGED_PLAY_STATUS status, //Roadmap 13008 RemoteControl
        tU64 param, const S32 iPodID) {
    ENTRY; //callback play status change notification
    VARTRACE((int)status);
    VARTRACE((int)param);
    VARTRACE(iPodID);
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_NotifyStatus(%d) - iAPGetMountPoint failed", iPodID));
    } else {

        const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
        const tBoolean isFinished = iAPIsTrackFinished(mountPoint);
        const tBoolean isExpected = !iAPIsTimeElapsed(IN mountPoint, IN pbNotifications, IN LocalSPM::GetDataProvider().iPodControlPBNotificationTimeoutMS());
        const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
        const int IPODPlayerState = iAPGetIPODPlayerState(mountPoint);
        const tPEHandle handle = iAPGetPEHandle(mountPoint);
        const tPEPlaybackState iPodPBStatus = iAPGetCurrentPlaybackState(mountPoint);
        const tPlaytime lastElapsedPlaytime = iAPGetElapsedPlaytime(mountPoint);

        VARTRACE(isStreaming);
        VARTRACE(isFinished);
        VARTRACE(isExpected);
        VARTRACE(isActive);
        VARTRACE(iPodPBStatus);
        VARTRACE(IPODPlayerState);
        VARTRACE(lastElapsedPlaytime);
        VARTRACE(handle);

        tPEPlaybackState iPodPBStatusNew = PE_PBS_ERRORSTATE;

        if(isFinished && !isStreaming) {
            //ignore all interim notifications from iPod during transition to next track.
            ETG_TRACE_USR3(("Already in finished state, ignore notification for iPodID %s", mountPoint));
            return ret;
        }

        tU32 param32 = (tU32) param;

        bool playStatusChanged = false;
        switch (status) {
        case IPOD_STATUS_PLAYBACK_STOPPED: /*!< Status of playback stopped */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_PLAYBACK_STOPPED %d", (int)param));
            playStatusChanged = true;
            //cached VT record for all live list types got invalid
            SendClearVTCache(mountPoint);
            iAPSetIPODPlayerState(mountPoint, IPOD_PLAYER_STATE_STOPPED);
            iAPSetCurrentPlaybackState(mountPoint, PE_PBS_PAUSEDSTATE);

            if(isActive) {
                //sync play answer
                SendPlaybackStatus(mountPoint);

                if (!isExpected && !isStreaming) {
                    //reached end of list, send FINISHED_STATE to PM
                    SendFinishedStatusToPlayerManager(mountPoint);
                } else if(isStreaming) {
                    //Update HMI NowPlaying
                    iAPSetNowPlayingTrackIndex(mountPoint, -1);
                    //FIXME: SendPlaybackStatusNew()
                    //SendPlaybackStatus(mountPoint);
                    SendNowPlayingStatus(handle);
                    SendPlaytimeStatusToPlayerManager(mountPoint);
                }
            }
            break;
        case IPOD_STATUS_TRACK_CHANGED: /*!< Status of track index. Currently version of spec is Track index. */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_CHANGED %d", (int)param));

            //trigger update now playing metadata
            iAPSetNowPlayingTrackIndex(mountPoint, -1);

            //Clear during SA_OFF - GMMY17-13189
            //cached VT record for all live list types got invalid
            SendClearVTCache(mountPoint);

            //check for remote activity
            if (!isExpected) {
                if(isActive || IPODPlayerState == IPOD_PLAYER_STATE_PLAYING) { //GMMY16-20639
                    iAP1CheckRemoteActivity(mountPoint, isStreaming);
                }
            }
            if (isActive) {
                //sync play answer
                SendPlaybackStatus(mountPoint);

                if(isStreaming){
                    SendClearVTCache(mountPoint);
                }
#if 0   //Sending Finished state AND RemoteActivity conflicts. Hence, we have to rely on proper TrackEndDetection
                if (!isExpected && !isStreaming) {
                    //reached end of track, track end detection failed
                    SendFinishedStatusToPlayerManager(mountPoint);
                }
#endif
                //trigger a nowplaying update
                SendNowPlayingStatus(handle);
            }
            break;
        case IPOD_STATUS_FWD_SEEK_STOP: /*!< Status of Fastforward seek stop */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_FWD_SEEK_STOP %d", (int)param));
            playStatusChanged = true;
            if(!isExpected && !isStreaming && isActive && iPodPBStatus == PE_PBS_FASTFORWARDSTATE) {
                //reached end of track while FFWS, send FINISHED_STATE to PM
                SendFinishedStatusToPlayerManager(mountPoint);
            }
            break;
        case IPOD_STATUS_BWD_SEEK_STOP: /*!< Status of FastBackward seek stop */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_BWD_SEEK_STOP %d", (int)param));
            playStatusChanged = true;
            break;
        case IPOD_STATUS_TRACK_POSITION: /*!< Status of track position(ms)  */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_POSITION %d", (int)param));
            if (isActive) {
                iAPSetElapsedPlaytime(mountPoint, (tPlaytime) param);
                SendPlaytimeStatusToPlayerManager(mountPoint);

                if (!isStreaming) {
                    //TRACK END DETECTION
                    if (iAPGetNowPlayingTrackIndex(mountPoint) >= 0) {
                        TrackEndDetection(mountPoint, iPodPBStatus, lastElapsedPlaytime);
                    }else {
                        SendNowPlayingStatus(handle); //trigger nowPlaying update
                    }
                }
            }
            break;
        case IPOD_STATUS_CHAPTER_CHANGED: /*!< Status of chapter index.Currentry version of spec is Chapter index*/
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_CHAPTER_CHANGED %d", (int)param));
            //trigger update now playing metadata
            iAPSetNowPlayingTrackIndex(mountPoint, -1);

            //cached VT record for all live list types got invalid
            SendClearVTCache(mountPoint);

            if (isActive) {
                //sync play answer
                SendPlaybackStatus(mountPoint);
                SendNowPlayingStatus(handle);

                //check for remote activity
                if (!isStreaming) {
                    if(LocalSPM::GetDataProvider().iPodControlIAP1ChapterStreaming()) {
                        //SUZUKI-21829
                        //run embedded chapters of audiobook always in streaming mode to handle seekTo and PREV commands properly.
                        SendRemoteActivity(mountPoint);
                    }
                }
            }
            break;
        case IPOD_STATUS_PLAYBACK_EXTENDED: /*!< Status of playback(e.g. stopped,playing,paused etc) */

            switch (param32) {
            case IPODCONTROL_EXTENDED_STATUS_STOPPED:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_STOPPED"));
                iPodPBStatusNew = PE_PBS_PAUSEDSTATE;
                iAPSetIPODPlayerState(mountPoint, IPOD_PLAYER_STATE_STOPPED);
                playStatusChanged = true;
                break;
            case IPODCONTROL_EXTENDED_STATUS_FFW_START:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_FFW_START"));
                iPodPBStatusNew = PE_PBS_FASTFORWARDSTATE;
                playStatusChanged = true;
                break;
            case IPODCONTROL_EXTENDED_STATUS_REW_START:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_REW_START"));
                iPodPBStatusNew = PE_PBS_FASTREVERSESTATE;
                playStatusChanged = true;
                break;
            case IPODCONTROL_EXTENDED_STATUS_FFW_REW_STOPPED:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_FFW_REW_STOPPED"));
                iPodPBStatusNew = IPODPlayerState == IPOD_PLAYER_STATE_PLAYING ? PE_PBS_PLAYINGSTATE : PE_PBS_PAUSEDSTATE;
                playStatusChanged = true;
                break;
            case IPODCONTROL_EXTENDED_STATUS_PLAYING:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_PLAYING"));
                iPodPBStatusNew = PE_PBS_PLAYINGSTATE;
                iAPSetIPODPlayerState(mountPoint, IPOD_PLAYER_STATE_PLAYING);
                playStatusChanged = true;
                break;
            case IPODCONTROL_EXTENDED_STATUS_PAUSED:
                ETG_TRACE_USR3(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED IPODCONTROL_EXTENDED_STATUS_PAUSED"));
                iPodPBStatusNew = PE_PBS_PAUSEDSTATE;
                iAPSetIPODPlayerState(mountPoint, IPOD_PLAYER_STATE_PAUSED);
                playStatusChanged = true;
                break;
            default:
                ETG_TRACE_ERR(("CALLBACK IPOD_STATUS_PLAYBACK_EXTENDED UNKNOWN %u", param32));
                break;
            }
            if(playStatusChanged) {
                //check for remote activity
                if (!isActive && !isExpected && iPodPBStatusNew == PE_PBS_PLAYINGSTATE) {
                    iAP1CheckRemoteActivity(mountPoint, isStreaming);
                }
                if (isActive) {
                    iAPSetCurrentPlaybackState(mountPoint, iPodPBStatusNew);
                    //FIXME: SendPlaybackStatusNew()
                    SendPlaybackStatus(mountPoint);
                    //SendPlaybackStatusToPlayerManager(mountPoint, iPodPBStatusNew);
                }
            }
            break;
        case IPOD_STATUS_TRACK_TIME_OFFSET: /*!< Status of track time offset(s) */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_TIME_OFFSET %d", (int)param));
            break;
        case IPOD_STATUS_CHAPTER_MS_OFFSET: /*!< Status of chapter time offset(ms) */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_CHAPTER_MS_OFFSET %d", (int)param));
            break;
        case IPOD_STATUS_CHAPTER_SEC_OFFSET: /*!< Status of chapter time offset(s) */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_CHAPTER_SEC_OFFSET %d", (int)param));
            break;
        case IPOD_STATUS_TRACK_UID: /*!< Status of track unique identifier */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_UID %d", (int)param));
            break;
        case IPOD_STATUS_TRACK_PLAYBACK_MODE: /*!< Status of track playback mode */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_PLAYBACK_MODE %d", (int)param));
            playStatusChanged = true;
            break;
        case IPOD_STATUS_TRACK_LYRICS_READY: /*!< status of lyrics for the currently playing track are available for download */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_LYRICS_READY %d", (int)param));
            break;
        case IPOD_STATUS_TRACK_CAPABILITIES_CHANGED: /* Status of track capabilities changed */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_TRACK_CAPABILITIES_CHANGED %d", (int)param));
            break;
        case IPOD_STATUS_PLAYBACK_CONTENT_CHANGED: /* Status of number of tracks in new playlist changed */
            ETG_TRACE_USR3(("CALLBACK IPOD_CHANGED_PLAY_STATUS: IPOD_STATUS_PLAYBACK_CONTENT_CHANGED %d", (int)param));
            break;
        default:
            ETG_TRACE_USR2(("CALLBACK IPOD_CHANGED_PLAY_STATUS: UNKNOWN %d", (int)status));
            break;
        }

        if (playStatusChanged) {
            //reset ATS elapsed time for next play status iap call
            iAPResetElapsedTime(mountPoint, atsPlayStatus);
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_NotifyStateChange(IPOD_STATE_CHANGE state,
        const S32 iPodID) {
    ENTRY; //callback event notification
    ETG_TRACE_USR3(("iPodControl::OnCBIAP1_NotifyStateChange()"));
    VARTRACE((int)state);
    VARTRACE(iPodID);

    tResult ret = MP_NO_ERROR;

    switch (state) {
    case IPOD_STATE_NO_STATE_CHANGE: /*!< No state change.                                                     */
        ETG_TRACE_USR3(("CALLBACK IPOD_STATE_CHANGE: IPOD_STATE_NO_STATE_CHANGE"));
        ;
        break;
    case IPOD_STATE_GOING_TO_DEEP_SLEEP: /*!< Accessory power going to Deep Sleep state (no power).                */
        ETG_TRACE_USR3(("CALLBACK IPOD_STATE_CHANGE: IPOD_STATE_GOING_TO_DEEP_SLEEP"));
        ;
        break;
    case IPOD_STATE_GOING_TO_HIBERNATE: /*!< Accessory power going to Hibernate state (no power).                 */
        ETG_TRACE_USR3(("CALLBACK IPOD_STATE_CHANGE: IPOD_STATE_GOING_TO_HIBERNATE"));
        ;
        break;
    case IPOD_STATE_GOING_TO_LIGHT_SLEEP: /*!< Accessory power going to Light Sleep state (less than 5 mA current). */
        ETG_TRACE_USR3(("CALLBACK IPOD_STATE_CHANGE: IPOD_STATE_GOING_TO_LIGHT_SLEEP"));
        ;
        break;
    case IPOD_STATE_GOING_TO_POWER_ON: /*!< Accessory power going to the Power On state.                         */
        ETG_TRACE_USR3(("CALLBACK IPOD_STATE_CHANGE: IPOD_STATE_GOING_TO_POWER_ON"));
        ;
        break;
    default:
        ETG_TRACE_ERR(("CALLBACK IPOD_STATE_CHANGE: UNKNOWN %d", (int)state))
        ;
        break;
    }

    //NOTE: NotifyStateChange not processed
    return ret;
}

tResult iPodControl::OnCBIAP1_NewiPodTrackInfo(U32 uNewSampleRate, const S32 iPodID) {
    ENTRY; //callback new audio track info
    VARTRACE(uNewSampleRate);
    VARTRACE(iPodID);
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = { 0 };
    ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_NewiPodTrackInfo(%d) - iAPGetMountPoint failed", iPodID));
    } else {
        ret = OnSampleRateChange(mountPoint, (me::samplerate_i) uNewSampleRate);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("OnSampleRateChange() failed"));
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_OpenDataSession(unsigned char protocolIndex,
        unsigned short sessionId, const S32 iPodID) {
    ENTRY; //callback opened session to app info
    VARTRACE(iPodID);
    VARTRACE(protocolIndex);
    VARTRACE(sessionId);

    tMountPoint mountPoint = { 0 };
    tResult ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAP1_GetMountPoint() failed", __PRETTY_FUNCTION__));
    } else {
        tAppName appName = { 0 };
        tProtocolName protocolName = { 0 };
        iAPSetSessionIDByIndex(mountPoint, sessionId, protocolIndex, appName);
        iAPGetProtocolNameByProtocolID(mountPoint,protocolIndex, protocolName);
        //notify app control clients about data session opened
        ret = LocalSPM::GetOutputWrapper().AppControlConnect(mountPoint, appName, sessionId,protocolName);
    }
    VARTRACE(ret);
    return ret;
}

tResult iPodControl::OnCBIAP1_CloseDataSession(unsigned short sessionId,
        const S32 iPodID) {
    ENTRY; //callback closed session to app
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(iPodID);
    VARTRACE(sessionId);

    tMountPoint mountPoint = { 0 };
    tResult ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAP1_GetMountPoint() failed", __PRETTY_FUNCTION__));
    } else {
        tAppName appName = { 0 };
        iAPClearSessionID(mountPoint, sessionId, appName);

        //notify app control clients about data session closed
        ret = LocalSPM::GetOutputWrapper().AppControlClose(mountPoint, appName, sessionId);
    }
    VARTRACE(ret);
    return ret;
}

tResult iPodControl::OnCBIAP1_DataTransfer(unsigned short sessionId,
        unsigned char *data, unsigned short length, const S32 iPodID) {
    ENTRY; //callback app sent data
   VARTRACE(iPodID);
    VARTRACE(sessionId);
    VARTRACE(length);
    iAPTraceData(data, length);

    tMountPoint mountPoint = { 0 };
    tResult ret = iAP1_GetMountPoint(mountPoint, (int) iPodID);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("%s iAP1_GetMountPoint() failed", __PRETTY_FUNCTION__));
    } else {
        tAppName appName = { 0 };
        iAPGetAppNameBySessionID(mountPoint, sessionId, appName);

        //notify app control clients about data session closed
        ret = LocalSPM::GetOutputWrapper().AppControlCommand(mountPoint,
                appName, sessionId, length, data);
    }
    VARTRACE(ret);
    return ret;
}

tResult iPodControl::OnCBIAP1_RetrieveCategorizedDBRecord(U32 dbindex, U8* str IPODCONTROL_ID_END_PARAM) {
    ENTRY; //callback DB record
    tResult ret = MP_NO_ERROR;
#ifndef TARGET_BUILD_GEN3
    const U32 iPodID = 1; //only one single entry in Gen2 PF possible
#endif

    //determine the DB record map for the notified device
    tIPODDBRecords * pDBRecords = NULL;

    mMapMutex.lock();
    tIPODDBRecordPtrMap::iterator it = mDBRecordPtrMap.find(iPodID);
    if(it != mDBRecordPtrMap.end()) {
        pDBRecords = it->second;
    }
    mMapMutex.unlock();

    if (!pDBRecords) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_RetrieveCategorizedDBRecord() Null pointer pDBRecords"));
        ret = MP_ERR_IPOD_METADATA;
    } else {
        //validate metadata
        //U8* str should be const pointer
        tMetadata meta = { 0 };
        strncpy_r(meta, (const char*) str, sizeof(meta) - 1);
        LocalSPM::GetDataProvider().ImportAndCut(
                INOUT (FastUTF8::tString) meta, sizeof(meta));

        //push back to map
        pDBRecords->insert(make_pair((int)dbindex, string(meta)));
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_TrackInfo(U64 trackIndex, IPOD_TRACK_INFORMATION_TYPE infoType, IPOD_TRACK_INFORMATION_DATA *infoData IPODCONTROL_ID_END_PARAM) {
    ENTRY; //callback track info
    tResult ret = MP_NO_ERROR;

#ifndef TARGET_BUILD_GEN3
    const U32 iPodID = 1; //only one single entry in Gen2 PF possible
#endif

    //determine the DB track info map for the notified device
    tIPODTrackInfos * pTrackInfos = NULL;

    mMapMutex.lock();
    tIPODTrackInfoPtrMap::iterator it = mTrackInfoPtrMap.find(iPodID);
    if(it != mTrackInfoPtrMap.end()) {
        pTrackInfos = it->second;
    }
    mMapMutex.unlock();

    if (!infoData) {
        ETG_TRACE_ERR(("CALLBACK OnCBTrackInfo() NULL POINTER infoData [track=%d, infoType%d]", (int)trackIndex, (int)infoType ));
        ret = MP_ERR_IPOD_INVALID_PARAM;
    } else if (!pTrackInfos) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_TrackInfo() Null pointer pTrackInfos"));
        ret = MP_ERR_IPOD_METADATA;
    } else {
        tIPODTrackInfo sInfo;
        sInfo.type = infoType;
        sInfo.data = *infoData;

        switch (infoType) {
        case CAPABILITIES: /*!< Capabilities (media kind, skip when shuffle, has artwork,etc)) */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: CAPABILITIES 0x%x", (tU32)trackIndex, (int)infoData->caps.isAudiobook));
            break;
        case TRACK_NAME: /*!< Track name */
            sInfo.str = string((char*) infoData->trackName);
            sInfo.data.trackName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: TRACK_NAME %s", (tU32)trackIndex, (char*)infoData->trackName));
            break;
        case ARTIST_NAME: /*!< Artist name */
            sInfo.str = string((char*) infoData->artistName);
            sInfo.data.artistName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: ARTIST_NAME %s", (tU32)trackIndex, (char*)infoData->artistName));
            break;
        case ALBUM_NAME: /*!< Album name */
            sInfo.str = string((char*) infoData->albumName);
            sInfo.data.albumName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: ALBUM_NAME %s", (tU32)trackIndex, (char*)infoData->albumName));
            break;
        case GENRE_NAME: /*!< Genre name */
            sInfo.str = string((char*) infoData->genreName);
            sInfo.data.artistName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: GENRE_NAME %s", (tU32)trackIndex, (char*)infoData->genreName));
            break;
        case COMPOSER_NAME: /*!< Composer name*/
            sInfo.str = string((char*) infoData->composerName);
            sInfo.data.composerName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: COMPOSER_NAME %s", (tU32)trackIndex, (char*)infoData->composerName));
            break;
        case DURATION: /*!< Total track time duration */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: DURATION %d ms", (tU32)trackIndex, infoData->duration));
            break;
        case UID: /*!< Uniqui track identifier */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: UID 0x%08x%08x", (tU32)trackIndex, (int)((unsigned long long)infoData->trackUID>>32), (int)(infoData->trackUID&0xffffffff)));
            sInfo.data.trackUID = infoData->trackUID;
            break;
        case CHAPTER_COUNT: /*!< Chapter count */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: CHAPTER_COUNT %d", (tU32)trackIndex, (tU32)infoData->chapterCount));
            break;
        case CHAPTER_TIMES: /*!< Chapter times */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: CHAPTER_TIMES offset %d, chapterIndex %d, totalCount %d ", (tU32)trackIndex, infoData->chapterTimes.offset, infoData->chapterTimes.chapterIndex, infoData->chapterTimes.totalCount));
            break;
        case CHAPTER_NAMES: /*!< Chapter names */
            if(infoData->chapterNames.totalCount) {
                sInfo.str = string((char*) infoData->chapterNames.name);
            }
            sInfo.data.chapterNames.name = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: CHAPTER_NAMES chapterIndex %d, totalCount %d, name %s", (tU32)trackIndex, infoData->chapterNames.chapterIndex, infoData->chapterNames.totalCount, infoData->chapterNames.totalCount ? (char*)infoData->chapterNames.name : ""));
            break;
        case LYRIC_STRING: /*!< Lyrics of the song currently playing in the Playback Engine*/
            sInfo.str = string((char*) infoData->lyrics.lyric);
            sInfo.data.lyrics.lyric = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: LYRIC_STRING section %d, maxSection %d, lyric %s ", (tU32)trackIndex, infoData->lyrics.section, infoData->lyrics.maxSection, infoData->lyrics.lyric));
            break;
        case DESCRIPTION: /*!< Description */
            sInfo.str = string((char*) infoData->description);
            sInfo.data.description = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: DESCRIPTION %s", (tU32)trackIndex, infoData->description));
            break;
        case ALBUM_TRACK_INDEX: /*!< Number of albume index */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: ALBUM_TRACK_INDEX %d", (tU32)trackIndex, infoData->albumTrackIndex));
            break;
        case DISC_SET_ALBUM_INDEX: /*!< Number of disc set album */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: DISC_SET_ALBUM_INDEX %d", (tU32)trackIndex, infoData->discSetAlbumIndex)) ;
            break;
        case PLAY_COUNT: /*!< Count of track play(0 = track not played) */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: PLAY_COUNT %d", (tU32)trackIndex, infoData->playCount));
            break;
        case SKIP_COUNT: /*!< Count of track skip(0 = track not skipped) */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: SKIP_COUNT %d", (tU32)trackIndex, infoData->skipCount));
            break;
        case PODCAST_DATA: /*!< Date/time */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: PODCAST_DATA year %d month %d", (tU32)trackIndex, infoData->podcastReleaseDate.year, infoData->podcastReleaseDate.month));
            break;
        case LAST_PLAYED_DATA: /*!< Date/time. If data is all zeros: the track has never been played */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: LAST_PLAYED_DATA year %d month %d, day %d", (tU32)trackIndex, infoData->lastPlayedDate.year, infoData->lastPlayedDate.month, infoData->lastPlayedDate.day));
            break;
        case YEAR: /*!< Year is which track was released */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: YEAR %d", (tU32)trackIndex, infoData->yaer));
            break;
        case STAR_RATING: /*!< Star rating of track */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: STAR_RATING %d", (tU32)trackIndex, infoData->starRating));
            break;
        case SERIES_NAME: /*!< Series name */
            sInfo.str = string((char*) infoData->seriesName);
            sInfo.data.seriesName = 0;
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: SERIES_NAME %s", (tU32)trackIndex, infoData->seriesName));
            break;
        case SEASON_NUMBER: /*!< Season number */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: SEASON_NUMBER %d", (tU32)trackIndex, infoData->seasonNumber));
            break;
        case TRACK_VOLUME_ADJUST: /*!< Track volume attenuation/amplification adjestment */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: TRACK_VOLUME_ADJUST %d", (tU32)trackIndex, infoData->Volume));
            break;
        case EQ_PRESET: /*!< Track equalizer preset index */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: EQ_PRESET %d", (tU32)trackIndex, infoData->EQPreset));
            break;
        case SAMPLE_RATE: /*!< Track data sample rate*/
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: SAMPLE_RATE %d", (tU32)trackIndex, infoData->dataRate));
            break;
        case BOOKMARK_OFFSET: /*!< Bookmark offset from start of track in milliseconds */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: BOOKMARK_OFFSET %d", (tU32)trackIndex, infoData->bookmarkOffset));
            break;
        case START_AND_STOP_OFFSET: /*!< Start time */
            ETG_TRACE_USR3(("CALLBACK TRACKINFO %03u: START_AND_STOP_OFFSET startOffset %d, stopOffset %d", (tU32)trackIndex, infoData->timeOffset.startOffset, infoData->timeOffset.stopOffset));
            break;
        default:
            ETG_TRACE_ERR(("CALLBACK TRACKINFO %03u: UNKNOWN IPOD_TRACK_INFORMATION_TYPE %d", (tU32)trackIndex, (int)infoType));
            break;
        }

        //validate metadata
        tMetadata meta = { 0 };
        strncpy_r(meta, sInfo.str.c_str(), sizeof(meta) - 1);
        LocalSPM::GetDataProvider().ImportAndCut(
                INOUT (FastUTF8::tString) meta, sizeof(meta));
        sInfo.str = string(meta);

        //push back to map
        pTrackInfos->insert(make_pair((int) trackIndex, sInfo));
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_PlayingTrackInfo(IPOD_TRACK_INFO_TYPE infoType,
        const IPOD_TRACK_CAP_INFO_DATA* /*capData*/,
        const IPOD_TRACK_RELEASE_DATE_DATA* /*releaseData*/,
        const IPOD_TRACK_ARTWORK_COUNT_DATA *artworkCountData, U8* /*stringBuf*/ IPODCONTROL_ID_END_PARAM) {
    ENTRY; //callback track info
    tResult ret = MP_NO_ERROR;

#ifndef TARGET_BUILD_GEN3
    const U32 iPodID = 1; //only one single entry in Gen2 PF possible
#endif

    //determine the playing track info map for the notified device
    tIPODPlayingTrackInfo * pPlayingTrackInfo = NULL;

    mMapMutex.lock();
    tIPODPlayingTrackInfoPtrMap::iterator it = mPlayingTrackInfoPtrMap.find(iPodID);
    if(it != mPlayingTrackInfoPtrMap.end()) {
        pPlayingTrackInfo = it->second;
    }
    mMapMutex.unlock();

    if (!pPlayingTrackInfo) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_PlayingTrackInfo() Null pointer pPlayingTrackInfo"));
        ret = MP_ERR_IPOD_METADATA;
    } else {

        pPlayingTrackInfo->type = infoType;
        pPlayingTrackInfo->artworkCount = 0;

        switch (infoType) {
        case IPOD_TRACK_ARTWORK_COUNT: /*!< This is type for getting track artwork count */
            ETG_TRACE_USR3(("CALLBACK PLAYING_TRACKINFO: IPOD_TRACK_ARTWORK_COUNT"));

            if (!artworkCountData) {
                ETG_TRACE_ERR(("CALLBACK PLAYING_TRACKINFO: IPOD_TRACK_ARTWORK_COUNT: null pointer"));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            } else {
                int iartwork = -1;
                int formatid = pPlayingTrackInfo->artworkFormatID;
                if (formatid < 0) {
                    ETG_TRACE_ERR(("CALLBACK PLAYING_TRACKINFO: invalid format id"));
                } else {
                    do {
                        iartwork++;
                        if (artworkCountData[iartwork].count > 0 &&
                            artworkCountData[iartwork].id == formatid) {
                            pPlayingTrackInfo->artworkCount = artworkCountData[iartwork].count;
                        }
                        ETG_TRACE_USR3(("CALLBACK PLAYING_TRACKINFO: IPOD_TRACK_ARTWORK_COUNT #%d:", iartwork));
                        VARTRACE(artworkCountData[iartwork].count);
                        VARTRACE(artworkCountData[iartwork].id);
                    } while (!artworkCountData[iartwork].isLast);
                }
            }
            break;
        default:
            ETG_TRACE_ERR(("CALLBACK PLAYING_TRACKINFO: UNHANDLED IPOD_TRACK_INFO_TYPE %d", infoType));
            ret = MP_ERR_IPOD_INVALID_PARAM;
            break;
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP1_ArtworkData(IPOD_ALBUM_ARTWORK *artworkData, const S32 iPodID) {
    ENTRY; //callback artwork data
    //tResult ret = MP_NO_ERROR;
    VARTRACE(iPodID);
    tAlbumArtObjectPtr albumArtObjectPtr = 0;
    tMountPoint mountPoint = { 0 };

    if (!artworkData) {
        ETG_TRACE_ERR(("iPodControl::OnCBIAP1_ArtworkData() Null pointer artworkData"));
        //ret = MP_ERR_IPOD_METADATA;
    } else {
        ETG_TRACE_USR3(("iPodControl::OnCBIAP1_ArtworkData() got data"));
        VARTRACE(artworkData->telegramIndex);             /*!< Index number of artwork data */
        VARTRACE(artworkData->displayPixelFormatCode);    /*!< Format code of artwork */
        VARTRACE(artworkData->imageWidth);                /*!< Width of artwork */
        VARTRACE(artworkData->imageHeight);               /*!< Height of artwork */
        VARTRACE(artworkData->topLeftPointX);             /*!< X of topLeftPoint of artwork */
        VARTRACE(artworkData->topLeftPointY);             /*!< Y of topLeftPoint of artwork */
        VARTRACE(artworkData->bottomRightX);              /*!< X of bottom right of artwork */
        VARTRACE(artworkData->bottomRightY);              /*!< Y of bottom rigth of artwork */
        VARTRACE(artworkData->rowSize);                   /*!< Row size of artwork data */
        VARTRACE(artworkData->pixelDataLength);           /*!< length of artwork data */

        //validate data
        if (artworkData->pixelDataLength == 0 || !artworkData->pixelData) {
            ETG_TRACE_ERR(("iPodControl::OnCBIAP1_ArtworkData() Null pointer pixelData"));
        } else if (artworkData->displayPixelFormatCode != 2) {
            ETG_TRACE_ERR(("iPodControl::OnCBIAP1_ArtworkData() invalid displayPixelFormatCode"));
        } else if( artworkData->pixelDataLength != (U32)(artworkData->imageWidth * artworkData->imageWidth * 2)) {
            ETG_TRACE_ERR(("iPodControl::OnCBIAP1_ArtworkData() invalid color depth"));
        } else if(MP_NO_ERROR == iAP1_GetMountPoint(mountPoint, (int) iPodID)){

            ElapsedTimer elapsed;
            //create tAlbumArtObjectPtr
            albumArtObjectPtr = new tAlbumArtObject;    //Attention: Has to be deleted by DataProvider
            if(!albumArtObjectPtr) {
                ETG_TRACE_ERR(("Out of memory"));
            } else {
                iAPGetAlbumArtString(mountPoint, albumArtObjectPtr->albumArtString);

                //convert RGB565 to BMP file format
                albumArtObjectPtr->mimeType = MMT_BMP;
                albumArtObjectPtr->sizeX = artworkData->imageWidth;
                albumArtObjectPtr->sizeY = artworkData->imageHeight;

                int dataOff = 14 + 40 + 12;
                int imgsize = artworkData->pixelDataLength;

                int w = albumArtObjectPtr->sizeX;
                int h = albumArtObjectPtr->sizeY;
                int padding = (4-(w*2)%4)%4;
                int bmpsize = dataOff + imgsize + h*padding;

                VARTRACE(padding);
                VARTRACE(bmpsize);

                albumArtObjectPtr->imageSize = bmpsize;
                albumArtObjectPtr->imageData = new unsigned char[bmpsize];

                unsigned char bmpfileheader[14] = {'B', 'M', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
                unsigned char bmpinfoheader[40] = {40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 16, 0, 3, 0, 0, 0,  0, 0, 0, 0, 0,   0, 0, 0, 0, 0,   0, 0, 0, 0, 0,   0, 0, 0, 0, 0};
                unsigned char bmpcolortabel[12] = {0, 0xF8, 0, 0, 0xE0, 0x07, 0, 0, 0x1F, 0, 0, 0}; //rgb565 bitmask

                bmpfileheader[2] = (unsigned char) (bmpsize);
                bmpfileheader[3] = (unsigned char) (bmpsize >> 8);
                bmpfileheader[4] = (unsigned char) (bmpsize >> 16);
                bmpfileheader[5] = (unsigned char) (bmpsize >> 24);

                bmpfileheader[10] = (unsigned char) (dataOff);
                bmpfileheader[11] = (unsigned char) (dataOff >> 8);
                bmpfileheader[12] = (unsigned char) (dataOff >> 16);
                bmpfileheader[13] = (unsigned char) (dataOff >> 24);

                bmpinfoheader[4] = (unsigned char) (w);
                bmpinfoheader[5] = (unsigned char) (w >> 8);
                bmpinfoheader[6] = (unsigned char) (w >> 16);
                bmpinfoheader[7] = (unsigned char) (w >> 24);
                bmpinfoheader[8] = (unsigned char) (-h); //negative height: top-down image
                bmpinfoheader[9] = (unsigned char) (-h >> 8);
                bmpinfoheader[10] = (unsigned char) (-h >> 16);
                bmpinfoheader[11] = (unsigned char) (-h >> 24);

                bmpinfoheader[20] = (unsigned char) (imgsize);
                bmpinfoheader[21] = (unsigned char) (imgsize >> 8);
                bmpinfoheader[22] = (unsigned char) (imgsize >> 16);
                bmpinfoheader[23] = (unsigned char) (imgsize >> 24);

                int off = 0;
                memcpy(albumArtObjectPtr->imageData + off, bmpfileheader, 14); off +=14;
                memcpy(albumArtObjectPtr->imageData + off, bmpinfoheader, 40); off +=40;
                memcpy(albumArtObjectPtr->imageData + off, bmpcolortabel, 12); off +=12;

                if (padding) {
                    //32bit aligned lines
                    for (int i = 0; i < h; i++) {
                        memcpy(albumArtObjectPtr->imageData + off, artworkData->pixelData + (i * 2 * w), w * 2);
                        off += w * 2 + padding;
                    }
                } else {
                    memcpy(albumArtObjectPtr->imageData + off, artworkData->pixelData, artworkData->pixelDataLength);
                }

                tGeneralString elapsedStr = {0};
                elapsed.ElapsedPrint(elapsedStr);
                ETG_TRACE_USR3(("iPodControl::OnCBIAP1_ArtworkData() TIME %s", elapsedStr));
            }
        }
    }
    return SendAlbumArtAnswer(albumArtObjectPtr); //lint !e429 deleted by DataProvider
}

//IAP2 callbacks
#ifdef IPODCONTROL_IAP2_PF_AVAIL
tResult iPodControl::OnCBIAP2_DeviceState(iAP2Device_t* iap2Device, const iAP2DeviceState_t dState, void* /*context*/)
{
    ENTRY;
    VARTRACE(dState);
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);
    tConnectionState lastState = iAPGetConnectionState(mountPoint);
    VARTRACE(lastState);

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    iAP2DeviceErrorState_t deviceErrorState = iAP2_GetDeviceErrorState(iap2Device, NULL);
    VARTRACE(deviceErrorState);

    tConnectionState state = CS_ATTACHED;
    switch(dState)
    {
        case iAP2NotConnected :
            state = CS_DISCONNECTED;
            break;
        case iAP2TransportConnected :
            state = CS_ATTACHED;
            break;
        case iAP2LinkConnected:
            state = CS_ATTACHED;
            break;
        case iAP2AuthenticationPassed :
            state = CS_ATTACHED;
            break;
        case iAP2IdentificationPassed :
            state = CS_ATTACHED;
            break;
        case iAP2DeviceReady:
            state = CS_CONNECTED;
            break;
        case iAP2LinkiAP1DeviceDetected:
            state = CS_UNSUPPORTED;
            break;
        case iAP2ComError:
            if(deviceErrorState == iAP2NoError || deviceErrorState == iAP2IdentificationFailed) {
                state = CS_ATTACHED; //error ignored, ADIT does a re-identification automatically, see identification rejected CB
            } else {
                state = CS_DISCONNECTED;
                SendDBChangeDeviceState(IN deviceID, IN DS_NONE);
                if(iAP2_IsHostMode(mountPoint) && LocalSPM::GetDataProvider().DipoCommunicationError())
                {
                    LocalSPM::GetOutputWrapper().UpdateDipoCommunicationError(deviceID,IPOD_ROLESWITCH_ERROR, mountPoint);
                }
            }
            break;
        default:
            ETG_TRACE_ERR(("Invalid IAP2 device state %d", dState));
            break;
    }
    VARTRACE(state);

    iAPSetConnectionState(mountPoint, state); //reset connection state in order to detect "Removed device by user", see RemoveDeviceConnection()

    if(state == CS_UNSUPPORTED){
        // iAP1 device detected;
        SendDBChangeDeviceState(IN deviceID, IN DS_NONE);
    }

    bool sendAttached = false;
    bool rolebackHostMode = false;
    bool setDBDisconnected = false;
    tDeviceInfo deviceInfo;
    iAPGetDeviceInfo(mountPoint,deviceInfo);

    if(LocalSPM::GetDataProvider().iPodControlCarPlayWifiEnabled() && DCT_BLUETOOTH == deviceInfo.connectionType)
    {
        SetBTLimitationModeConnectionState(mountPoint,state);

        // Fix for NCG3D-162752: Not able to establish CPW after enabling the functionality from MD
        // 1) When SPP is connected, instead of triggering iAP BT session without wireless transport component,
        // initiate iAP BT session with wireless transport component.
        // 2) Ignore the BT Limitation mode updates and prevent Reconnection in MP on a BT Limitation mode change from NOT PREPARED to PREPARED
        // - to avoid iAP reconnection triggers from MP to ADIT.
        // TO DO: Clean up code wrt storing and retrieval of BT Limitation mode related parameters
        // Note: To be done after system testing of all Carplay related use cases and confirming that BT Limitation mode is not required.
        //if(CS_CONNECTED == state)
        //{
            //if IAP2 BT is connected then need to check whether IAP2 BT is required with WirelessTransport comp or not
            //based on BTLimitationMode
            //ReconnectIAPBTOnLimitationChange(mountPoint, deviceID);
        //}
        //End of fix
    }

    if(state == CS_ATTACHED || state == lastState) {
        //intermediate states during authentication
        //that has no effect to the overall iPodControl state machine
        return MP_NO_ERROR;
    } else if(state == CS_DISCONNECTED){
        //check previous connection state
        if(lastState == CS_CONNECTED) {
            //IAP2 Host role back
            if(iAP2_IsHostMode(mountPoint) || iAP2_IsWirelessCarPlayMode(mountPoint)) {
                rolebackHostMode = true;
                setDBDisconnected = true;
            } else if(ILocalSPM::GetState() == COMPONENT_STATE_RUNNING) {
                char *mountPointNew = new char[sizeof(tMountPoint)]; //deleted by DoStopPlaybackOnDetach
                strncpy_r(mountPointNew, mountPoint, sizeof(tMountPoint));
                LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_STOP_PLAYBACK_ON_DETACH, IN (void *)mountPointNew, this);
            }
        } else if(lastState == CS_ATTACHED) {//lint !e429 deleted by DoStopPlaybackOnDetach
            //connecting failure after reaching CS_ATTACHED state, e.g. on authentication

            //trigger RemoveDeviceConnection()/Disconnect()
            sendAttached = true;

            //IAP2 Host role back
            if(iAP2_IsHostMode(mountPoint)) {
                rolebackHostMode = true;
                setDBDisconnected = true;
            }
        }
        if((!LocalSPM::GetDataProvider().iPodControlCarPlayWifiEnabled() && deviceInfo.connectionType == DCT_BLUETOOTH) //Wifi check for fixing NCG3D-120729.
            || (deviceInfo.connectionType == DCT_WIFI)) //Wifi check for fixing Bug 595995.
        {
            setDBDisconnected = true;
        }
    } else if((deviceInfo.connectionType == DCT_BLUETOOTH)&& LocalSPM::GetDataProvider().IAPBTConnectionOnlyForSmartApp()){
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterON_DEVICE_ATTACH(OUT parameterString, IN size, IN mountPoint, IN CS_ATTACHED, IN IDP_IAP2BT);
        SendEvent(ON_DEVICE_ATTACH, IN parameterString);
    } else { //CS_CONNECTED || CS_UNSUPPORTED
        sendAttached = true;
    }

    if(rolebackHostMode) {
        //trigger RemoveDeviceConnection to clean up and clear dipoCaps flag
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ParameterREMOVE_DEVICE_CONNECTION_FORCED(OUT parameterString, IN size, IN mountPoint, IN deviceID);
        SendEvent(REMOVE_DEVICE_CONNECTION_FORCED, IN parameterString);
    }

    if(setDBDisconnected) {
        //update Mediaplayer device status
        ETG_TRACE_USR3(("HOST mode - set DB device removed"));
        SendDBChangeConnectionState(IN deviceID, IN CS_DISCONNECTED);
    }

    if(sendAttached) {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        const tConnectionType connectionType = iAPGetConnectionType(mountPoint);
        VARTRACE(connectionType);
        tInitDeviceProtocol protocol = IDP_IAP2;
        if(!iAP2_IsHostMode(mountPoint) && !iAP2_IsWirelessCarPlayMode(mountPoint)) {
            if(connectionType == DCT_BLUETOOTH) {
                protocol = IDP_IAP2BT;
            } else {
                protocol = IDP_IAP2;
            }
        } else if(iAP2_IsCarPlayMode(mountPoint) && iAP2_IsNativeTransportMode(mountPoint) ) {
            protocol = IDP_IAP2CARPLAY_NATIVE_TRANSPORT;
        } else if(iAP2_IsCarPlayMode(mountPoint)) {
            protocol = IDP_IAP2CARPLAY;
        } else if(iAP2_IsNativeTransportMode(mountPoint)){
            protocol = IDP_IAP2NATIVE_TRANSPORT;
        } else if(iAP2_IsWirelessCarPlayMode(mountPoint)){
            protocol = IDP_IAP2_OVER_WIRELESS_CARPLAY;
        }else if(iAP2_IsNativeTransportCarlifeMode(mountPoint)){
            protocol = IDP_IAP2CARLIFE_NATIVE_TRANSPORT;
        }
        VARTRACE(protocol);
        ParameterON_DEVICE_ATTACH(OUT parameterString, IN size, IN mountPoint, IN state, IN protocol);
        SendEvent(ON_DEVICE_ATTACH, IN parameterString);
    }

    return MP_NO_ERROR;
}

tResult iPodControl::OnCBIAP2_IdentificationAccepted(iAP2Device_t* iap2Device, iAP2IdentificationAcceptedParameter* /*idParameter*/, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return ret;
    }
    VARTRACE(mountPoint);

    iAPSetOption(mountPoint, hasIAP2Option, true);

    //ControlSessionV2
    const tDiPOCaps diPOCaps = iAP2_GetDiPOCaps(mountPoint);
    VARTRACE(diPOCaps);
    if(diPOCaps != DIPO_CAP_NONE) {
        const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);
        //set DiPOCaps
        SendDBChangeDiPOSettings(IN deviceID, IN diPOCaps, IN FALSE, IN "");
    }

    return MP_NO_ERROR;
}

tResult iPodControl::OnCBIAP2_DeviceInformationUpdate(iAP2Device_t* iap2Device, iAP2DeviceInformationUpdateParameter *deviceInformationUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!deviceInformationUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    VARTRACE(deviceInformationUpdateParameter->iAP2DeviceName_count);
    if(deviceInformationUpdateParameter->iAP2DeviceName_count &&
       deviceInformationUpdateParameter->iAP2DeviceName) {
        VARTRACE((char*)deviceInformationUpdateParameter->iAP2DeviceName[0]);
        tMountPoint mountPoint = {0};
        ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
        if (ret != MP_NO_ERROR) {
            ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
            return ret;
        }
        VARTRACE(mountPoint);
        iAPSetDeviceName(mountPoint, (const char *)deviceInformationUpdateParameter->iAP2DeviceName[0]);
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_DeviceLanguageUpdate(iAP2Device_t* iap2Device, iAP2DeviceLanguageUpdateParameter *deviceLanguageUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!deviceLanguageUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    VARTRACE(deviceLanguageUpdateParameter->iAP2DeviceName_count);
    if(deviceLanguageUpdateParameter->iAP2DeviceName_count &&
       deviceLanguageUpdateParameter->iAP2DeviceName) {
        VARTRACE((char*)deviceLanguageUpdateParameter->iAP2DeviceName[0]);
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_DeviceTimeUpdate(iAP2Device_t* iap2Device, iAP2DeviceTimeUpdateParameter *deviceTimeParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(iap2Device)
    {
        tMountPoint mountPoint = {0};
        ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
        VARTRACE(mountPoint);
        if (ret == MP_NO_ERROR)
        {
            if(deviceTimeParameter)
            {
                //update member
                tDiPODeviceTime deviceTime = iAP2GetDeviceTimeUpdate(mountPoint);

                if(deviceTimeParameter->iAP2DaylightSavingsOffsetMinutes_count &&
                        deviceTimeParameter->iAP2DaylightSavingsOffsetMinutes)
                {
                    VARTRACE(deviceTimeParameter->iAP2DaylightSavingsOffsetMinutes[0]);
                    deviceTime.s8DaylightSavingsOffsetMinutes = (tS8)deviceTimeParameter->iAP2DaylightSavingsOffsetMinutes[0];
                }

                if(deviceTimeParameter->iAP2SecondsSinceReferenceDate_count &&
                        deviceTimeParameter->iAP2SecondsSinceReferenceDate)
                {
                    VARTRACE(deviceTimeParameter->iAP2SecondsSinceReferenceDate[0]);
                    deviceTime.u64SecondsSinceReferenceDate = (tU64)deviceTimeParameter->iAP2SecondsSinceReferenceDate[0];
                }

                if(deviceTimeParameter->iAP2TimeZoneOffsetMinutes_count &&
                        deviceTimeParameter->iAP2TimeZoneOffsetMinutes)
                {
                    VARTRACE(deviceTimeParameter->iAP2TimeZoneOffsetMinutes[0]);
                    deviceTime.s16TimeZoneOffsetMinutes = (tS16)deviceTimeParameter->iAP2TimeZoneOffsetMinutes[0];
                }

                //store current state
                iAP2SetDeviceTimeUpdate(mountPoint, deviceTime);

                //send update trigger
                UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_DEVICETIME);
            }
            else
            {
                ETG_TRACE_ERR(("Invalid parameter"));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            }
        }
        else
        {
            ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
            ret = MP_ERR_IPOD_NO_DEVICE;
        }
    }
    else
    {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        ret = MP_ERR_IPOD_NO_DEVICE;
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_NowPlayingUpdate(iAP2Device_t* iap2Device, iAP2NowPlayingUpdateParameter* nowPlayingUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!nowPlayingUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = { 0 };
    ret = iAP2_GetMountPoint(mountPoint, (const void*) iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPGetMountPoint failed"));
    } else {
        VARTRACE(mountPoint);

        const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
        const tBoolean isFinished = iAPIsTrackFinished(mountPoint);
        const tBoolean isExpected = !iAPIsTimeElapsed(IN mountPoint, IN pbNotifications, IN LocalSPM::GetDataProvider().iPodControlPBNotificationTimeoutMS());
        const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
        const tPEHandle handle = iAPGetPEHandle(mountPoint);
        const int IPODPlayerState = iAPGetIPODPlayerState(mountPoint);
        const tPEPlaybackState iPodPBStatus = iAPGetCurrentPlaybackState(mountPoint);
        const tPlaytime lastElapsedPlaytime = iAPGetElapsedPlaytime(mountPoint);
        const tBoolean hostMode = iAP2_IsHostMode(mountPoint);
        const tBoolean wirelessCarPlay = iAP2_IsWirelessCarPlayMode(mountPoint);

        VARTRACE(isStreaming);
        VARTRACE(isFinished);
        VARTRACE(isExpected);
        VARTRACE(isActive);
        VARTRACE(iPodPBStatus);
        VARTRACE(IPODPlayerState);
        VARTRACE(lastElapsedPlaytime);
        VARTRACE(handle);
        VARTRACE(hostMode);

        tBoolean metadataChange = false;
        tBoolean playTimeChange = false;
        tBoolean playStatusChange = false;
        tBoolean appNameChange = false;
        tBoolean shuffleChange = false;
        tBoolean repeatChange = false;
        tBoolean albumArtChange = false;
        tBoolean queueListChange = false;
        tBoolean uuidChanged = false;

        tMediaObject mediaObject;
        InitMediaObject(mediaObject);
        iAPGetNowPlayingMediaObject(mediaObject, mountPoint);
        mediaObject.mediaType = MTY_UNKNOWN;
        tPlaytime elapsedPlaytime = iAPGetElapsedPlaytime(mountPoint);

        //check media item attribute updates
        iAP2MediaItem* mip = nowPlayingUpdateParameter->iAP2MediaItemAttributes;
        for(int i = 0; i < nowPlayingUpdateParameter->iAP2MediaItemAttributes_count && mip; i++) {

            if(mip[i].iAP2MediaItemPersistentIdentifier_count && mip[i].iAP2MediaItemPersistentIdentifier) { //iOS8
                VARTRACE(mip[i].iAP2MediaItemPersistentIdentifier[0]); //U64
                snprintf(mediaObject.UUID, sizeof(mediaObject.UUID), IPODCONTROL_UUID_FORMAT, mip[i].iAP2MediaItemPersistentIdentifier[0]);
                metadataChange = true;
                uuidChanged = true;
            }

            if(mip[i].iAP2MediaItemTitle_count && mip[i].iAP2MediaItemTitle) {
                VARTRACE(mip[i].iAP2MediaItemTitle[0]); //utf8
                strncpy_r(mediaObject.title, (const char *)mip[i].iAP2MediaItemTitle[0], sizeof(mediaObject.title));
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemPlaybackDurationInMilliseconds_count && mip[i].iAP2MediaItemPlaybackDurationInMilliseconds) {
                VARTRACE(mip[i].iAP2MediaItemPlaybackDurationInMilliseconds[0]);//U32
                mediaObject.totalPlaytime = (tPlaytime) mip[i].iAP2MediaItemPlaybackDurationInMilliseconds[0];
                iAPSetTotalPlaytime(mountPoint, mediaObject.totalPlaytime);
                //metadataChange = true; //handle TuneIn app spamming
                playTimeChange = true;
            }
            if(mip[i].iAP2MediaItemAlbumTitle_count && mip[i].iAP2MediaItemAlbumTitle) {
                VARTRACE(mip[i].iAP2MediaItemAlbumTitle[0]);//utf8
                strncpy_r(mediaObject.MetadataField4, (const char *)mip[i].iAP2MediaItemAlbumTitle[0], sizeof(mediaObject.MetadataField4));
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemAlbumTrackNumber_count && mip[i].iAP2MediaItemAlbumTrackNumber) {
                VARTRACE(mip[i].iAP2MediaItemAlbumTrackNumber[0]);//U16
                mediaObject.trackNumber = (tTrackNumber)mip[i].iAP2MediaItemAlbumTrackNumber[0];
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemAlbumTrackCount_count && mip[i].iAP2MediaItemAlbumTrackCount) {
                VARTRACE(mip[i].iAP2MediaItemAlbumTrackCount[0]);//U16
            }
            if(mip[i].iAP2MediaItemAlbumDiscNumber_count && mip[i].iAP2MediaItemAlbumDiscNumber) {
                VARTRACE(mip[i].iAP2MediaItemAlbumDiscNumber[0]);//U16
            }
            if(mip[i].iAP2MediaItemAlbumDiscCount_count && mip[i].iAP2MediaItemAlbumDiscCount) {
                VARTRACE(mip[i].iAP2MediaItemAlbumDiscCount[0]);//U16
            }
            //if(mip[i].iAP2MediaItemChapterCount_count && mip[i].iAP2MediaItemChapterCount) {
            //    VARTRACE(mip[i].iAP2MediaItemChapterCount[0]);//U16
            //}
            if(mip[i].iAP2MediaItemArtist_count && mip[i].iAP2MediaItemArtist) {
                VARTRACE(mip[i].iAP2MediaItemArtist[0]);//utf8
                strncpy_r(mediaObject.MetadataField2, (const char *)mip[i].iAP2MediaItemArtist[0], sizeof(mediaObject.MetadataField2));
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemGenre_count && mip[i].iAP2MediaItemGenre) {
                VARTRACE(mip[i].iAP2MediaItemGenre[0]);//utf8
                strncpy_r(mediaObject.MetadataField1, (const char *)mip[i].iAP2MediaItemGenre[0], sizeof(mediaObject.MetadataField1));
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemComposer_count && mip[i].iAP2MediaItemComposer) {
                VARTRACE(mip[i].iAP2MediaItemComposer[0]);//utf8
                strncpy_r(mediaObject.MetadataField3, (const char *)mip[i].iAP2MediaItemComposer[0], sizeof(mediaObject.MetadataField3));
                metadataChange = true;
            }
            if(mip[i].iAP2MediaItemIsLikeSupported_count && mip[i].iAP2MediaItemIsLikeSupported) {
                VARTRACE(mip[i].iAP2MediaItemIsLikeSupported[0]);//bool
            }
            if(mip[i].iAP2MediaItemIsBanSupported_count && mip[i].iAP2MediaItemIsBanSupported) {
                VARTRACE(mip[i].iAP2MediaItemIsBanSupported[0]);//bool
            }
            if(mip[i].iAP2MediaItemIsLiked_count && mip[i].iAP2MediaItemIsLiked) {
                VARTRACE(mip[i].iAP2MediaItemIsLiked[0]);//bool
            }
            if(mip[i].iAP2MediaItemIsBanned_count && mip[i].iAP2MediaItemIsBanned) {
                VARTRACE(mip[i].iAP2MediaItemIsBanned[0]);//bool
            }
            if(mip[i].iAP2MediaItemArtworkFileTransferIdentifier_count && mip[i].iAP2MediaItemArtworkFileTransferIdentifier) {
                VARTRACE(mip[i].iAP2MediaItemArtworkFileTransferIdentifier[0]);//U8
                iAPSetAlbumArtID(mountPoint, (int)mip[i].iAP2MediaItemArtworkFileTransferIdentifier[0]);
                albumArtChange = true;
                metadataChange = true;

                if(LocalSPM::GetDataProvider().iPodControlAlbumArtWaitTimer() && m_AlbumArtTimerID)
                {
                    StopAlbumArtTimer();
                }
            }
        }

        if(LocalSPM::GetDataProvider().iPodControlAlbumArtWaitTimer() && uuidChanged)
        {
            if(m_AlbumArtTimerID)
            {
                StopAlbumArtTimer();
            }
            //Fix for PSARCC30-4539 - Start Albumart Timer if the UUID changed and Albumartid is not changed.
            if(!albumArtChange)
            {
                StartAlbumArtTimer(mountPoint);
            }
        }

        //check playback attribute updates
        for(int i = 0; i < nowPlayingUpdateParameter->iAP2PlaybackAttributes_count && nowPlayingUpdateParameter->iAP2PlaybackAttributes; i++) {
            iAP2PlaybackAttributes* pap = nowPlayingUpdateParameter->iAP2PlaybackAttributes;

            if(pap[i].iAP2PlaybackAppName_count && pap[i].iAP2PlaybackAppName) {
                VARTRACE(pap[i].iAP2PlaybackAppName[0]); //utf8
                iAPSetFocusApp(mountPoint, (const char*)pap[i].iAP2PlaybackAppName[0]); //use with iOS7
                appNameChange = true;
            }
            if(pap[i].iAP2PlaybackAppBundleID_count && pap[i].iAP2PlaybackAppBundleID) {
                VARTRACE(pap[i].iAP2PlaybackAppBundleID[0]); //utf8
                iAPSetFocusApp(mountPoint, (const char*)pap[i].iAP2PlaybackAppBundleID[0]); //use with iOS8
                appNameChange = true;
            }
            if(pap[i].iAP2PlaybackStatus_count && pap[i].iAP2PlaybackStatus) {
                VARTRACE((int)pap[i].iAP2PlaybackStatus[0]); //enum
                iAPSetIPODPlayerState(mountPoint, (int)pap[i].iAP2PlaybackStatus[0] & 0xff);
                playStatusChange = true;
            }
            if(pap[i].iAP2PlaybackElapsedTimeInMilliseconds_count && pap[i].iAP2PlaybackElapsedTimeInMilliseconds) {
                VARTRACE(pap[i].iAP2PlaybackElapsedTimeInMilliseconds[0]); //U32
                elapsedPlaytime = (tPlaytime) pap[i].iAP2PlaybackElapsedTimeInMilliseconds[0];
                playTimeChange = true;
            }
            if(pap[i].iAP2PlaybackQueueListAvail_count && pap[i].iAP2PlaybackQueueListAvail) { //iOS8
                VARTRACE(pap[i].iAP2PlaybackQueueListAvail[0]); //U32
                if(!pap[i].iAP2PlaybackQueueListAvail[0]) {
                    iAPSetQueueListID(mountPoint, -1); //clear queue list
                    m_IsNowPlayingListAvail = false;
                }
                else{
                    m_IsNowPlayingListAvail = true;
                }
                queueListChange = true;
            }
            if(pap[i].iAP2PlaybackQueueListTransferID_count && pap[i].iAP2PlaybackQueueListTransferID) { //iOS8
                VARTRACE(pap[i].iAP2PlaybackQueueListTransferID[0]); //U32
                if(LocalSPM::GetDataProvider().iPodControlIAP2QueueListEnabled()) {
                    iAPSetQueueListID(mountPoint, (int)pap[i].iAP2PlaybackQueueListTransferID[0]);
                    queueListChange = true;
                }
            }
            if(pap[i].iAP2PlaybackQueueIndex_count && pap[i].iAP2PlaybackQueueIndex) {
                VARTRACE(pap[i].iAP2PlaybackQueueIndex[0]); //U32
                iAPSetNowPlayingTrackIndex(mountPoint, pap[i].iAP2PlaybackQueueIndex[0]);
                queueListChange = true;
            }
            if(pap[i].iAP2PlaybackQueueCount_count && pap[i].iAP2PlaybackQueueCount) {
                VARTRACE(pap[i].iAP2PlaybackQueueCount[0]); //U32
                iAPSetNowPlayingTrackCount(mountPoint, pap[i].iAP2PlaybackQueueCount[0]);
                queueListChange = true;
            }
            if(pap[i].iAP2PlaybackQueueChapterIndex_count && pap[i].iAP2PlaybackQueueChapterIndex) {
                VARTRACE(pap[i].iAP2PlaybackQueueChapterIndex[0]); //U32
                iAPSetNowPlayingChapterIndex(mountPoint, pap[i].iAP2PlaybackQueueChapterIndex[0]);
                queueListChange = true;
            }
            if(pap[i].iAP2PlaybackShuffleMode_count && pap[i].iAP2PlaybackShuffleMode) {
                VARTRACE((int)pap[i].iAP2PlaybackShuffleMode[0]); //enum
                iAPSetPlaybackModeFromIPOD(mountPoint, (int)pap[i].iAP2PlaybackShuffleMode[0] & 0xff);
                shuffleChange = true;
            }
            if(pap[i].iAP2PlaybackRepeatMode_count && pap[i].iAP2PlaybackRepeatMode) {
                VARTRACE((int)pap[i].iAP2PlaybackRepeatMode[0]); //enum
                iAPSetRepeatModeFromIPOD(mountPoint, (int)pap[i].iAP2PlaybackRepeatMode[0] & 0xff);
                repeatChange = true;
            }
            if(pap[i].iAP2PBMediaLibraryUniqueIdentifier_count && pap[i].iAP2PBMediaLibraryUniqueIdentifier) {
                VARTRACE(pap[i].iAP2PBMediaLibraryUniqueIdentifier[0]); //utf8
                iAPSetNowPlayingLibraryUID(mountPoint, (const char*)pap[i].iAP2PBMediaLibraryUniqueIdentifier[0]);
                //Fix for Pandora sourcing issue
                if(strlen_r((const char*)pap[i].iAP2PBMediaLibraryUniqueIdentifier[0]) > 0) {
                    //Workaround iOS8: iPhone sends new app name too late
                    tFingerprint mediaLibraryUID = {0};
                    iAPGetMediaLibraryUID(mountPoint, mediaLibraryUID);
                    VARTRACE(mediaLibraryUID);
                    tFingerprint iTunesRadioLibraryUID = {0};
                    iAPGetiTunesRadioLibraryUID(mountPoint, iTunesRadioLibraryUID);
                    VARTRACE(iTunesRadioLibraryUID);
                    if(!strcmp(mediaLibraryUID, (const char*)pap[i].iAP2PBMediaLibraryUniqueIdentifier[0])
                       || !strcmp(iTunesRadioLibraryUID, (const char*)pap[i].iAP2PBMediaLibraryUniqueIdentifier[0])) {
                        iAPSetFocusApp(mountPoint, IAP2_NOW_PLAYING_UPDATE_PLAYBACK_APP_BUNDLE_MUSIC);
                        appNameChange = true;
                    }
                }
            }
            if(pap[i].iAP2PBiTunesRadioAd_count && pap[i].iAP2PBiTunesRadioAd) {
                VARTRACE(pap[i].iAP2PBiTunesRadioAd[0]); //BOOL
            }
            if(pap[i].iAP2PBiTunesRadioStationName_count && pap[i].iAP2PBiTunesRadioStationName) {
                VARTRACE(pap[i].iAP2PBiTunesRadioStationName[0]); //utf8
            }
            if(pap[i].iAP2PBiTunesRadioStationMediaPlaylistPersistentID_count && pap[i].iAP2PBiTunesRadioStationMediaPlaylistPersistentID) {
                VARTRACE(pap[i].iAP2PBiTunesRadioStationMediaPlaylistPersistentID[0]); //U64
                if(pap[i].iAP2PBiTunesRadioStationMediaPlaylistPersistentID[0]) {
                    snprintf(mediaObject.UUID, sizeof(mediaObject.UUID), IPODCONTROL_UUID_FORMAT, pap[i].iAP2PBiTunesRadioStationMediaPlaylistPersistentID[0]);
                    metadataChange = true;
                }
            }
            if(pap[i].iAP2PlaybackSpeed_count && pap[i].iAP2PlaybackSpeed) { //iOS8
                VARTRACE(pap[i].iAP2PlaybackSpeed[0]); //U32
            }
            if(pap[i].iAP2SetElapsedTimeAvailable_count && pap[i].iAP2SetElapsedTimeAvailable) {
                VARTRACE(pap[i].iAP2SetElapsedTimeAvailable[0]); //utf8
            }
        }

        VARTRACE(metadataChange);
        VARTRACE(playStatusChange);
        VARTRACE(playTimeChange);
        VARTRACE(appNameChange);
        VARTRACE(shuffleChange);
        VARTRACE(repeatChange);
        VARTRACE(albumArtChange);
        VARTRACE(queueListChange);
        VARTRACE(m_IsNowPlayingListAvail);


        if( repeatChange &&  LocalSPM::GetDataProvider().SetDefaultRepeatModeToRPTListAppleDevice()
                && iAPGetRepeatInitFlag(mountPoint) && ( RPT_NONE == iAPGetRepeatMode(mountPoint)) )
        {
            tRepeatMode mode = RPT_LIST;
            SendSetRepeatMode(mountPoint,mode);
            iAPSetRepeatInitFlag(mountPoint,false);
            repeatChange = false;
        }

        //update PBS state to internal members first
        tBoolean playbackStatusSent = false;
        tPEPlaybackState iPodPBStatusNew = PE_PBS_ERRORSTATE;
        const int IPODPlayerStateNew = iAPGetIPODPlayerState(mountPoint);
        VARTRACE(IPODPlayerStateNew);

        if(playStatusChange) {
            //convert iPod playback state to PE PBS
            switch(IPODPlayerStateNew) {
            case IAP2_PLAYBACK_STATUS_STOPPED:
            case IAP2_PLAYBACK_STATUS_PAUSED:
                iPodPBStatusNew = PE_PBS_PAUSEDSTATE;
                break;
            case IAP2_PLAYBACK_STATUS_PLAYING:
                iPodPBStatusNew = PE_PBS_PLAYINGSTATE;
                break;
            case IAP2_PLAYBACK_STATUS_SEEK_FORWARD:
                iPodPBStatusNew = PE_PBS_FASTFORWARDSTATE;
                break;
            case IAP2_PLAYBACK_STATUS_SEEK_BACKWARD:
                iPodPBStatusNew = PE_PBS_FASTREVERSESTATE;
                break;
            default:
                ETG_TRACE_ERR(("INVALID iAP2PlaybackStatus"));
                break;
            }
            VARTRACE(iPodPBStatusNew);
            iAPSetCurrentPlaybackState(mountPoint, iPodPBStatusNew);
        }

        if(metadataChange) {
            //update now playing
            iAPSetNowPlayingMediaObject(mediaObject, mountPoint);
            //clear finished flag
            iAPSetTrackFinished(mountPoint, false);

            //sync play answer
            if(isActive) {
                SendPlaybackStatus(mountPoint);
                playbackStatusSent = true;
            }
        }
        if(playTimeChange) {
            //update elapsed playtime
            iAPSetElapsedPlaytime(mountPoint, elapsedPlaytime);
        }

        if (metadataChange || queueListChange || shuffleChange || albumArtChange) {
            //cached VT record for all live list types got invalid
            SendClearVTCache(mountPoint);

        }

        if(queueListChange){
            //clear isNowplayingListAvailable
            SendDBChangeNowPlayingListAvailable(mountPoint, iAPGetQueueListID(mountPoint), true);
        }

        //Note: iHeartRadio set empty title BEFORE pausing playbackatate
        // remote activity is causing trouble on audio sourceing back to mediaplayer
        // allow REMOTE ACTIVITY when (IPODPlayerStateNew == IAP2_PLAYBACK_STATUS_PLAYING) || isActive;
        tBoolean remoteActivitySuppressed = (IPODPlayerStateNew != IAP2_PLAYBACK_STATUS_PLAYING) && !isActive; //GMMY16-20639
        VARTRACE(remoteActivitySuppressed);
        tBoolean remoteActivitySent = remoteActivitySuppressed || (hostMode && (!LocalSPM::GetDataProvider().iPodControlIAP2CarPlayModeEnabled() || !isActive)); //iPod disabled for hostmode

        if(isFinished && !isStreaming) {
            //ignore all interim notifications from iPod during transition to next track.
            ETG_TRACE_USR3(("Already in finished state, ignore notification for iPodID %s", mountPoint));
        } else {
            if(metadataChange) {
                //check for remote activity
                if (!isExpected) {
                    iAP2CheckRemoteActivity(mountPoint, isStreaming, hostMode, remoteActivitySent);
                }
                if (isActive) {
                    if(albumArtChange){
                        //SendAlbumArtStatusToPlayerManager(mountPoint); //done by ForwardNowPlayingStatus
                    }
                    //trigger a nowplaying update
                    SendNowPlayingStatus(handle);
                }
            }

            if(playStatusChange) {
                //check for remote activity
                if (!isActive && !isExpected && iPodPBStatusNew == PE_PBS_PLAYINGSTATE) {
                    iAP2CheckRemoteActivity(mountPoint, isStreaming, hostMode, remoteActivitySent);
                }
                if (isActive && !playbackStatusSent) {
                    //TODO: 13010 - use SendPlaybackStatusNew()
                    SendPlaybackStatusToPlayerManager(mountPoint, iPodPBStatusNew);
                    playbackStatusSent = true;
                }
            }

            if(playTimeChange) {
                if (isActive) {
                    SendPlaytimeStatusToPlayerManager(mountPoint);
                    if (!isStreaming && !isExpected && !remoteActivitySent) {
                        //TRACK END DETECTION
                        TrackEndDetection(mountPoint, iPodPBStatus, lastElapsedPlaytime);
                    }
                }
            }

            if(appNameChange) {
                if (!isExpected || isStreaming) {
                    iAP2CheckRemoteActivity(mountPoint, isStreaming, hostMode, remoteActivitySent);
                }
            }
            if(repeatChange || shuffleChange) {
                if (!isExpected || isStreaming) {
                    if((repeatChange && LocalSPM::GetDataProvider().iPodRemoteEventEnterStreamingOnRepeatMode()) ||
                      (shuffleChange && LocalSPM::GetDataProvider().iPodRemoteEventEnterStreamingOnPlaybackMode())) {
                        iAP2CheckRemoteActivity(mountPoint, isStreaming, hostMode, remoteActivitySent);
                    }
                }
                //GMMY16-19547
                if(repeatChange && (!isActive || isStreaming || LocalSPM::GetDataProvider().SetDeviceRepeatModeNonStreaming())) {
                    SendRepeatModeToPlayerManager(mountPoint);
                    iAPSetRepeatInitFlag(mountPoint,false);
                }
                if(shuffleChange && (!isActive || isStreaming || LocalSPM::GetDataProvider().SetDevicePlaybackModeNonStreaming())) {
                    SendPlaybackModeToPlayerManager(mountPoint);
                }
            }
        }
        //update appControl properties
        if(hostMode || wirelessCarPlay) {
            const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
            if(metadataChange || appNameChange) {
                UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_NOWPLAYING);
            }
            if(playStatusChange) {
                UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_PLAYBACKSTATUS);
            }
            if(playTimeChange) {
                UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_PLAYTIME);
            }
            if(shuffleChange) {
                UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_SHUFFLEMODE);
            }
            if(repeatChange) {
                UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_REPEATMODE);
            }
        }
    }
    return ret;
}
tResult iPodControl::OnCBIAP2_USBDeviceModeAudioInformation(iAP2Device_t* iap2Device, iAP2USBDeviceModeAudioInformationParameter* usbDeviceModeAudioInformationParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!usbDeviceModeAudioInformationParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = { 0 };
    ret = iAP2_GetMountPoint(mountPoint, (const void*) iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAPGetMountPoint failed"));
    } else {
        VARTRACE(mountPoint);

        if(usbDeviceModeAudioInformationParameter->iAP2SampleRate_count) {
            VARTRACE(usbDeviceModeAudioInformationParameter->iAP2SampleRate[0]);
            //convert IAP2 enum to me::samplerate_i
            me::samplerate_i newSampleRate = 0;
            switch(usbDeviceModeAudioInformationParameter->iAP2SampleRate[0] & 0xff) {
            case IAP2_SAMPLERATE_8000HZ:
                newSampleRate = 8000;
                break;
            case IAP2_SAMPLERATE_11025HZ:
                newSampleRate = 11025;
                break;
            case IAP2_SAMPLERATE_12000HZ:
                newSampleRate = 12000;
                break;
            case IAP2_SAMPLERATE_16000HZ:
                newSampleRate = 16000;
                break;
            case IAP2_SAMPLERATE_22050HZ:
                newSampleRate = 22050;
                break;
            case IAP2_SAMPLERATE_24000HZ:
                newSampleRate = 24000;
                break;
            case IAP2_SAMPLERATE_32000HZ:
                newSampleRate = 32000;
                break;
            case IAP2_SAMPLERATE_44100HZ:
                newSampleRate = 44100;
                break;
            case IAP2_SAMPLERATE_48000HZ:
                newSampleRate = 48000;
                break;
            default:
                ETG_TRACE_ERR(("Invalid iAP2USBDeviceModeAudioSampleRate"));
                break;
            }
            ret = OnSampleRateChange(mountPoint, newSampleRate);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("OnSampleRateChange() failed"));
            }
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferSetup(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
    VARTRACE(isActive);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);

    //check for albumArt
    const int albumArtID = iAPGetAlbumArtID(mountPoint);
    VARTRACE(albumArtID);
    if(albumArtID > 0 && iAP2FileXferSession->iAP2FileTransferID == albumArtID) {

        tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetAlbumArtBuffer(mountPoint);
        //safety - clear obsolete buffer
        if(pBuf) {
            ETG_TRACE_USR3(("Found obsolete album art buffer for transferID %d - clearing", pBuf->transferID));
            delete [] pBuf->buffer;
            delete pBuf;
        }
        pBuf = 0;

        if(iAP2FileXferSession->iAP2FileXferRxLen == 0) {
            ETG_TRACE_USR3(("AlbumArt data not available"));
            //no data, no error
        } else if((int)iAP2FileXferSession->iAP2FileXferRxLen > LocalSPM::GetDataProvider().iPodControlIAP2AlbumArtMaxBytes()) {
            ETG_TRACE_ERR(("AlbumArt data size exceeds limit - cancel file transfer"));
            ret = MP_ERR_IPOD_INVALID_PARAM;
        } else {
            //create new buffer
            pBuf = new tFileXferBuf;
            if(pBuf) {
                pBuf->size = 0;
                pBuf->rxLen = 0;
                pBuf->transferID = iAP2FileXferSession->iAP2FileTransferID;
                pBuf->buffer = new U8[(unsigned int)iAP2FileXferSession->iAP2FileXferRxLen];

                if(pBuf->buffer) {
                    pBuf->size = iAP2FileXferSession->iAP2FileXferRxLen;
                } else {
                    ETG_TRACE_ERR(("No memory"));
                    ret = MP_ERR_IPOD_NO_MEM;
                }
            } else {
                ETG_TRACE_ERR(("No memory"));
                ret = MP_ERR_IPOD_NO_MEM;
            }
        }
        //store album art buffer
        iAPSetAlbumArtBuffer(mountPoint, pBuf);

        if(isActive) {
            //trigger now playing update (with set albumartString), see SendNowPlayingStatusToPlayerManager()
            SendAlbumArtStatusToPlayerManager(mountPoint);
        }
    } else { //lint !e429 pBuf freed at disconnect

        //check for queue list
        const int queueListID = iAPGetQueueListID(mountPoint);
        VARTRACE(queueListID);
        if(queueListID > 0 && iAP2FileXferSession->iAP2FileTransferID == queueListID) {
            tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetQueueListBuffer(mountPoint);
            //safety - clear obsolete buffer
            if(pBuf) {
                ETG_TRACE_USR3(("Found obsolete queue list buffer for transferID %d - clearing", pBuf->transferID));
                delete [] pBuf->buffer;
                delete pBuf;
            }
            pBuf = 0;

            if(iAP2FileXferSession->iAP2FileXferRxLen == 0) {
                ETG_TRACE_USR3(("QueueList data not available"));
                //no data, no error
            } else if((int)iAP2FileXferSession->iAP2FileXferRxLen > LocalSPM::GetDataProvider().iPodControlIAP2QueueListMaxBytes()) {
                ETG_TRACE_ERR(("QueueList data size exceeds limit - cancel file transfer"));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            } else {
                //create new buffer
                pBuf = new tFileXferBuf;
                if(pBuf) {
                    pBuf->size = 0;
                    pBuf->rxLen = 0;
                    pBuf->transferID = iAP2FileXferSession->iAP2FileTransferID;
                    pBuf->buffer = new U8[(unsigned int)iAP2FileXferSession->iAP2FileXferRxLen];

                    if(pBuf->buffer) {
                        pBuf->size = iAP2FileXferSession->iAP2FileXferRxLen;
                    } else {
                        ETG_TRACE_ERR(("No memory"));
                        ret = MP_ERR_IPOD_NO_MEM;
                    }
                } else {
                    ETG_TRACE_ERR(("No memory"));
                    ret = MP_ERR_IPOD_NO_MEM;
                }
            }
            //store album art buffer
            iAPSetQueueListBuffer(mountPoint, pBuf);

        } else { //lint !e429 pBuf freed at disconnect

            if(!iAPSetupPlaylistItems(mountPoint, iAP2FileXferSession)) { //lint !e429 deleted by iAP2_DisconnectDeviceConnection
                ETG_TRACE_USR3(("FileTransfer ignored - not matching data"));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            }
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferDataRcvd(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);

    //check for albumArt
    const int albumArtID = iAPGetAlbumArtID(mountPoint);
    VARTRACE(albumArtID);
    if(albumArtID > 0 && iAP2FileXferSession->iAP2FileTransferID == albumArtID) {
        if(iAP2FileXferSession->iAP2FileXferRxLen > 0 && iAP2FileXferSession->iAP2FileXferRxBuf) {
            tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetAlbumArtBuffer(mountPoint);
            if(pBuf && pBuf->buffer) {
                VARTRACE(pBuf->transferID);
                VARTRACE(pBuf->size);
                VARTRACE(pBuf->rxLen);

                if(pBuf->transferID == iAP2FileXferSession->iAP2FileTransferID) {
                    memcpy(pBuf->buffer + pBuf->rxLen, iAP2FileXferSession->iAP2FileXferRxBuf, (unsigned int)iAP2FileXferSession->iAP2FileXferRxLen);
                    pBuf->rxLen += iAP2FileXferSession->iAP2FileXferRxLen;
                } else {
                    ETG_TRACE_ERR(("No matching transfer ID - ignoring album art data"));
                }
            } else {
                ETG_TRACE_USR3(("No buffer - ignoring album art data"));
            }
        } else {
            ETG_TRACE_ERR(("invalid parameter"));
        }
    } else {
        //check for queue list
        const int queueListID = iAPGetQueueListID(mountPoint);
        VARTRACE(queueListID);
        if(queueListID > 0 && iAP2FileXferSession->iAP2FileTransferID == queueListID) {
            if(iAP2FileXferSession->iAP2FileXferRxLen > 0 && iAP2FileXferSession->iAP2FileXferRxBuf) {
                tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetQueueListBuffer(mountPoint);
                if(pBuf && pBuf->buffer) {
                    VARTRACE(pBuf->transferID);
                    VARTRACE(pBuf->size);
                    VARTRACE(pBuf->rxLen);

                    if(pBuf->transferID == iAP2FileXferSession->iAP2FileTransferID) {
                        memcpy(pBuf->buffer + pBuf->rxLen, iAP2FileXferSession->iAP2FileXferRxBuf, (unsigned int)iAP2FileXferSession->iAP2FileXferRxLen);
                        pBuf->rxLen += iAP2FileXferSession->iAP2FileXferRxLen;
                    } else {
                        ETG_TRACE_ERR(("No matching transfer ID - ignoring queue list data"));
                    }
                } else {
                    ETG_TRACE_USR3(("No buffer - ignoring queue list data"));
                }
            } else {
                ETG_TRACE_ERR(("invalid parameter"));
            }
        } else {
            if(!iAPDataRcvdPlaylistItems(mountPoint, iAP2FileXferSession)) {
                ETG_TRACE_USR3(("FileTransfer ignored - not matching data"));
            }
        }
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferSuccess(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
    VARTRACE(isActive);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);

    //check for albumArt
    const int albumArtID = iAPGetAlbumArtID(mountPoint);
    VARTRACE(albumArtID);
    if(albumArtID > 0 && iAP2FileXferSession->iAP2FileTransferID == albumArtID) {

        tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetAlbumArtBuffer(mountPoint);
        if(pBuf) {
            VARTRACE(pBuf->transferID);
            VARTRACE(pBuf->size);
            VARTRACE(pBuf->rxLen);

            if(pBuf->transferID == iAP2FileXferSession->iAP2FileTransferID) {
                if(pBuf->buffer && pBuf->size == pBuf->rxLen) {
                    if(isActive) {
                        //trigger now playing update (with set albumartString), see SendNowPlayingStatusToPlayerManager()
                        SendAlbumArtStatusToPlayerManager(mountPoint);
                    } else {
                        ETG_TRACE_USR3(("Received AlbumArt for inactive device"));
                        if(iAP2_IsHostMode(mountPoint) || iAP2_IsWirelessCarPlayMode(mountPoint)) { //PSARCC21-1781
                            UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_NOWPLAYING);
                        }
                    }
                } else {
                    ETG_TRACE_ERR(("Incomplete file transfer - cannot dispatch buffer album art data"));
                }
            } else {
                ETG_TRACE_ERR(("No matching transfer ID - cannot dispatch buffer album art data"));
            }
        } else {
            ETG_TRACE_USR3(("No buffer - ignoring album art data"));
        }
    } else {
        //check for queue list
        const int queueListID = iAPGetQueueListID(mountPoint);
        VARTRACE(queueListID);
        if(queueListID > 0 && iAP2FileXferSession->iAP2FileTransferID == queueListID) {

            tFileXferBuf * pBuf = (tFileXferBuf *)iAPGetQueueListBuffer(mountPoint);
            if(pBuf) {
                VARTRACE(pBuf->transferID);
                VARTRACE(pBuf->size);
                VARTRACE(pBuf->rxLen);

                if(pBuf->transferID == iAP2FileXferSession->iAP2FileTransferID) {
                    if(pBuf->buffer && pBuf->size == pBuf->rxLen) {
                        //trigger update for VTIPOD retrieval
                        SendClearVTCache(mountPoint);
                        if(isActive) {
                            //trigger nowplaying update
                            SendNowPlayingStatusToPlayerManager(mountPoint);
                            // New function for updating DB about NowPLayingListAvailable.
                            SendDBChangeNowPlayingListAvailable(mountPoint, queueListID, false);
                        } else {
                            ETG_TRACE_USR3(("Received QueueList for inactive device"));
                        }
                    } else {
                        ETG_TRACE_ERR(("Incomplete file transfer - cannot dispatch buffer queue list data"));
                    }
                } else {
                    ETG_TRACE_ERR(("No matching transfer ID - cannot dispatch buffer queue list data"));
                }
            } else {
                ETG_TRACE_USR3(("No buffer - ignoring queue list data"));
            }
        } else {
            if(iAPSuccessPlaylistItems(mountPoint, iAP2FileXferSession)) {
                //Wake up indexing state "loadMetadata"
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterMETADATA_FOR_INDEX(OUT parameterString, IN size, IN mountPoint, IN NULL, IN iAPGetDeviceID(mountPoint));
                SendEvent(IN METADATA_FOR_INDEX, IN parameterString);
            } else {
                ETG_TRACE_USR3(("FileTransfer ignored - not matching data"));
            }
        }
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferFailure(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);
    VARTRACE((int)(long)iAP2FileXferSession->iAP2FileXferRxBuf); // thoemel: the additional casting is due to 64 bit compiler

    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferCancel(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);
    VARTRACE((int)(long)iAP2FileXferSession->iAP2FileXferRxBuf); // thoemel: the additional casting is due to 64 bit compiler

    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferPause(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);
    VARTRACE((int)(long)iAP2FileXferSession->iAP2FileXferRxBuf); // thoemel: the additional casting is due to 64 bit compiler

    return ret;
}

tResult iPodControl::OnCBIAP2_FileTransferResume(const iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2FileXferSession) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2FileXferSession->iAP2FileTransferID);
    VARTRACE(iAP2FileXferSession->iAP2FileXferRxLen);
    VARTRACE((int)(long)iAP2FileXferSession->iAP2FileXferRxBuf); // thoemel: the additional casting is due to 64 bit compiler

    return ret;
}

tResult iPodControl::OnCBIAP2_MediaLibraryInfo(iAP2Device_t* iap2Device, iAP2MediaLibraryInformationParameter* mediaLibraryInfoParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!mediaLibraryInfoParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter_count);
    for(int i = 0; i < mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter_count; i++){
        if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter) {
            VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryName_count);
            if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryName_count) {
                VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryName[0]);
                VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaUniqueIdentifier_count);
                if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaUniqueIdentifier_count) {
                    VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaUniqueIdentifier[0]);
                    VARTRACE(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryType_count);
                    if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryType_count) {
                        VARTRACE((int)mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryType[0]);
                        if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryType[0] == IAP2_LIBRARY_TYPE_LOCAL_DEVICE) {
                            //set Media Library ID
                            iAPSetMediaLibraryUID(mountPoint, (const char*)mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaUniqueIdentifier[0]);
                        } else if(mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaLibraryType[0] == IAP2_LIBRARY_TYPE_ITUNES_RADIO) {
                            //set iTunes Radio Library ID
                            iAPSetiTunesRadioLibraryUID(mountPoint, (const char*)mediaLibraryInfoParameter->iAP2MediaLibraryInformationSubParameter[i].iAP2MediaUniqueIdentifier[0]);
                        }
                    }
                }
            }
        }
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_MediaLibraryUpdates(iAP2Device_t* iap2Device, iAP2MediaLibraryUpdateParameter* mediaLibraryUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!mediaLibraryUpdateParameter ||
       !mediaLibraryUpdateParameter->iAP2MediaLibraryUniqueIdentifier_count ||
       !mediaLibraryUpdateParameter->iAP2MediaLibraryUniqueIdentifier) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }
    VARTRACE(mediaLibraryUpdateParameter->iAP2MediaLibraryUniqueIdentifier[0]);

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    const bool indexing_paused = iAPIndexerIsPaused(mountPoint);
    VARTRACE(indexing_paused);

    tFingerprint mediaLibraryUID = {0};
    iAPGetMediaLibraryUID(mountPoint, mediaLibraryUID);
    VARTRACE(mediaLibraryUID);

    tFingerprint iTunesRadioLibraryUID = {0};
    iAPGetiTunesRadioLibraryUID(mountPoint, iTunesRadioLibraryUID);
    VARTRACE(iTunesRadioLibraryUID);

    bool isiTunesRadioUpdate = false;

    if(LocalSPM::GetDataProvider().iPodControlIAP2iTunesRadioEnabled() && strlen_r(iTunesRadioLibraryUID) > 0 && !strcmp(iTunesRadioLibraryUID, (const char*)mediaLibraryUpdateParameter->iAP2MediaLibraryUniqueIdentifier[0])) {
        isiTunesRadioUpdate = true;
    } else if(strlen_r(mediaLibraryUID) == 0 || strcmp(mediaLibraryUID, (const char*)mediaLibraryUpdateParameter->iAP2MediaLibraryUniqueIdentifier[0])) {
        ETG_TRACE_ERR(("Unknown media library"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }
    VARTRACE(isiTunesRadioUpdate);

    //Calculate FingerPrint depending on values below
    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    if(mediaLibraryUpdateParameter->iAP2MediaLibraryRevision_count) {
        VARTRACE(mediaLibraryUpdateParameter->iAP2MediaLibraryRevision[0]);
        if(isiTunesRadioUpdate) {
            iAPSetiTunesRadioLibraryRevision(mountPoint, (char *)mediaLibraryUpdateParameter->iAP2MediaLibraryRevision[0]);
        } else {
            iAPSetMediaLibraryRevision(mountPoint, (char *)mediaLibraryUpdateParameter->iAP2MediaLibraryRevision[0]);
        }
    }
    if(mediaLibraryUpdateParameter->iAP2MediaLibraryReset_count) {
        VARTRACE(mediaLibraryUpdateParameter->iAP2MediaLibraryReset_count); //type: none, no data
        if(!isiTunesRadioUpdate && LocalSPM::GetDataProvider().iPodControlIAP2IndexingIgnoreDeletedItemsAfterMediaLibraryReset()) {
            iAPSetMediaLibraryReset(mountPoint, true);
            ETG_TRACE_USR3(("Deleted MediaLibraryItems will be ignored"));
        }
    }
    if(mediaLibraryUpdateParameter->iAP2MediaLibraryIsHidingRemoteItems_count) {
        VARTRACE(mediaLibraryUpdateParameter->iAP2MediaLibraryIsHidingRemoteItems[0]);
    }

    //check progress

    const int lastMediaProgress = iAPGetMediaLibraryUpdateProgress(mountPoint);
    const int lastiTunesProgress = iAPGetiTunesRadioLibraryUpdateProgress(mountPoint);
    VARTRACE(lastMediaProgress);
    VARTRACE(lastiTunesProgress);

    int progress = 0;

    int lastprogress = isiTunesRadioUpdate ? iAPGetiTunesRadioLibraryUpdateProgress(mountPoint) : iAPGetMediaLibraryUpdateProgress(mountPoint);
    VARTRACE(lastprogress);

    if(mediaLibraryUpdateParameter->iAP2MediaLibraryUpdateProgress_count) {
        VARTRACE(mediaLibraryUpdateParameter->iAP2MediaLibraryUpdateProgress[0]);
        progress = mediaLibraryUpdateParameter->iAP2MediaLibraryUpdateProgress[0];
    }

    if(iAPGetMediaLibraryReset(mountPoint) && (progress == 100)) {
        if(!isiTunesRadioUpdate && LocalSPM::GetDataProvider().iPodControlIAP2IndexingIgnoreDeletedItemsAfterMediaLibraryReset()) {
            iAPSetMediaLibraryReset(mountPoint, false);
            ETG_TRACE_USR3(("Deleted MediaLibraryItems will not be ignored anymore"));
        }
    }

    //Do not send events if indexing is done
    if((lastprogress < 100 || LocalSPM::GetDataProvider().iPodControlIAP2DynamicReindexing()))
    {
        //stop obsolete fingerprint timer
        StopFingerprintTimer(deviceID);

        if(!iAP2_PushMediaLibraryUpdate(mountPoint, mediaLibraryUpdateParameter, isiTunesRadioUpdate) ) {
            ETG_TRACE_ERR(("iAP2_PushMediaLibraryUpdate failed"));
            ret = MP_ERR_IPOD_INVALID_PARAM;
        } else {

            tAllParameters parameterString;
            size_t size = sizeof(parameterString);

            //check for iTunes updates conficting media indexing
            if(!isiTunesRadioUpdate || lastMediaProgress > 0) {
                //set new fingerprint
                tFingerprint fingerprint = { 0 };

                tFingerprint mediaRevision = { 0 };
                iAPGetMediaLibraryRevision(mountPoint, mediaRevision);
                VARTRACE(mediaRevision);

                tFingerprint iTunesRevision = { 0 };
                iAPGetiTunesRadioLibraryRevision(mountPoint, iTunesRevision);
                VARTRACE(iTunesRevision);

                SMF::Marshal(fingerprint, sizeof(fingerprint) - 1, IPODCONTROL_MARSHAL_SEPARATOR, IPODCONTROL_MARSHAL_FORMAT_FINGERPRINT_IAP2, mediaLibraryUID, mediaRevision, iTunesRadioLibraryUID, iTunesRevision);
                iAPIndexerSetFingerprint(mountPoint, fingerprint);

                //force FPS_OK_DELTA in case of isiTunesRadioUpdate
                tFingerprintStatus fingerprintStatus = !isiTunesRadioUpdate && mediaLibraryUpdateParameter->iAP2MediaLibraryReset_count ? FPS_OK : FPS_OK_DELTA;
                VARTRACE(fingerprintStatus);

                //extrapolate total number of files
                tNumberOfFiles numberOfFiles = iAPIndexerGetTrackCount(mountPoint);
                if(numberOfFiles != NUMBER_OF_FILES_NONE) {
                    if(numberOfFiles == 0) {
                        numberOfFiles = NUMBER_OF_FILES_NONE;
                    } else if(progress > 0) {
                        numberOfFiles = 100 * numberOfFiles / (unsigned int)progress;
                        VARTRACE(numberOfFiles);
                    }
                }

                //NOTE: call fingerprint update first, update must reach the indexer before MDS_FINISHED
                //1. Update indexer fingerprint
                ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerprintStatus, IN numberOfFiles, IN deviceID);
                SendEvent(PUT_FINGERPRINT, IN parameterString);
            }

            const int cachecount = iAPIndexerGetCacheCount(mountPoint);
            VARTRACE(cachecount);

            //2. Wake up indexing state "loadMetadata"
            if(cachecount > 0) {
                ParameterMETADATA_FOR_INDEX(OUT parameterString, IN size, IN mountPoint, IN NULL, IN deviceID);
                SendEvent(IN METADATA_FOR_INDEX, IN parameterString);
            }

            //3.Check if indexing PAUSE is required in order to reduce memory consumption
            if((cachecount > LocalSPM::GetDataProvider().iPodControlIAP2IndexingCacheLimit()) && !indexing_paused) {
                //stop medialibraryupdates
                const tUserDataType userDataType = USRDATA_STOPMEDIALIBRARYUPDATES;
                tUserData userData = {0}; //no data, taken from config

                /* Send SIGNAL_USERDATA message to own SM */
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);

                ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
                SendEvent(SIGNAL_USERDATA, IN parameterString);
            }
        }
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_StartExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StartExternalAccessoryProtocolSessionParameter* startExternalAccessoryProtocolSessionParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!startExternalAccessoryProtocolSessionParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    unsigned char protocolID = 0;
    if(startExternalAccessoryProtocolSessionParameter->iAP2ExternalAccesoryProtocolIdentifier_count) {
        protocolID = startExternalAccessoryProtocolSessionParameter->iAP2ExternalAccesoryProtocolIdentifier[0];
        VARTRACE(protocolID);
    }
    unsigned short sessionID = 0;
    if(startExternalAccessoryProtocolSessionParameter->iAP2ExternalAccessoryProtocolSessionIdentifier_count) {
        sessionID = startExternalAccessoryProtocolSessionParameter->iAP2ExternalAccessoryProtocolSessionIdentifier[0];
        VARTRACE(sessionID);
    }

    if(protocolID > 0) {
        tAppName appName = { 0 };
        tProtocolName protocolName = { 0 };
        iAPSetSessionIDByIndex(mountPoint, sessionID, protocolID, appName);
        iAPGetProtocolNameByProtocolID(mountPoint,protocolID, protocolName);
        VARTRACE(appName);
        VARTRACE(sessionID);
        VARTRACE(protocolName);

        tDeviceID deviceId = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceId);

        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        const tUserDataType userDataType = USRDATA_APPCONTROL_CONNECT;
        tUserData userData = {0};
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tMountPoint_format tAppName_format tSessionID_format tProtocolName_format,mountPoint,appName,sessionID,protocolName);

        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
        SendEvent(SIGNAL_USERDATA, IN parameterString);
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_StopExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StopExternalAccessoryProtocolSessionParameter* stopExternalAccessoryProtocolSessionParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!stopExternalAccessoryProtocolSessionParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    unsigned short sessionID = 0;
    if(stopExternalAccessoryProtocolSessionParameter->iAP2ExternalAccessoryProtocolSessionIdentifier_count) {
        sessionID = stopExternalAccessoryProtocolSessionParameter->iAP2ExternalAccessoryProtocolSessionIdentifier[0];
        VARTRACE(sessionID);
    }

    tAppName appName = { 0 };
    iAPClearSessionID(mountPoint, sessionID, appName);
    VARTRACE(appName);

    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    const tUserDataType userDataType = USRDATA_APPCONTROL_CLOSE;
    tUserData userData = {0};
    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tMountPoint_format tAppName_format tSessionID_format,mountPoint,appName,sessionID);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tResult iPodControl::OnCBIAP2_StartEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* /*context*/)
{
   ENTRY;
   tResult ret = MP_NO_ERROR;

   if(!iap2Device) {
      ETG_TRACE_ERR(("No valid device pointer for IAP2"));
      return MP_ERR_IPOD_NO_DEVICE;
   }

   tMountPoint mountPoint = {0};
   ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
   if (ret != MP_NO_ERROR) {
      ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
      return MP_ERR_IPOD_NO_DEVICE;
   }
   VARTRACE(mountPoint);

   tDeviceID deviceId = iAPGetDeviceID(mountPoint);
   VARTRACE(deviceId);
   VARTRACE(iAP2iOSAppIdentifier);
   VARTRACE(sinkEndpoint);
   VARTRACE(sourceEndpoint);

   tAllParameters parameterString;
   size_t size = sizeof(parameterString);

   const tUserDataType userDataType = USRDATA_START_EA_NATIVE_TRANSPORT;
   tUserData userData = {0};
   SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR,"iii" ,iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint);

   ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
   SendEvent(SIGNAL_USERDATA, IN parameterString);

   return ret;
}

tResult iPodControl::OnCBIAP2_StopEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* /*context*/)
{
   ENTRY;
   tResult ret = MP_NO_ERROR;

   if(!iap2Device) {
      ETG_TRACE_ERR(("No valid device pointer for IAP2"));
      return MP_ERR_IPOD_NO_DEVICE;
   }

   tMountPoint mountPoint = {0};
   ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
   if (ret != MP_NO_ERROR) {
      ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
      return MP_ERR_IPOD_NO_DEVICE;
   }
   VARTRACE(mountPoint);

   tDeviceID deviceId = iAPGetDeviceID(mountPoint);
   VARTRACE(deviceId);
   VARTRACE(iAP2iOSAppIdentifier);
   VARTRACE(sinkEndpoint);
   VARTRACE(sourceEndpoint);

   tAllParameters parameterString;
   size_t size = sizeof(parameterString);

   const tUserDataType userDataType = USRDATA_STOP_EA_NATIVE_TRANSPORT;
   tUserData userData = {0};
   SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR,"iii" ,iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint);

   ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
   SendEvent(SIGNAL_USERDATA, IN parameterString);

   return ret;
}

tResult iPodControl::OnCBIAP2_iOSAppDataReceived(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8* iAP2iOSAppDataRxd, U16 iAP2iOSAppDataLength, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!iAP2iOSAppDataRxd || !iAP2iOSAppDataLength) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    VARTRACE(iAP2iOSAppIdentifier);
    VARTRACE(iAP2iOSAppDataLength);
    int temp_iAP2iOSAppDataLength = iAP2iOSAppDataLength;
    VARTRACE(temp_iAP2iOSAppDataLength);
    iAPTraceData(iAP2iOSAppDataRxd, iAP2iOSAppDataLength);

    tAppName appName = { 0 };
    tSessionID sessionID = 0;
    iAPGetAppNameByProtocolID(mountPoint, iAP2iOSAppIdentifier, appName, sessionID);
    VARTRACE(appName);
    VARTRACE(sessionID);

    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    const tUserDataType userDataType = USRDATA_APP_DATA_RECEIVED;
    tUserData userData = {0};
    U8* iAP2iOSAppDataRxdLocal = new U8[iAP2iOSAppDataLength];  //deleated by SignalUserData

    if(iAP2iOSAppDataRxdLocal)
    {
        memcpy(iAP2iOSAppDataRxdLocal,iAP2iOSAppDataRxd,iAP2iOSAppDataLength);
    }

    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "ttiip",mountPoint,appName,sessionID,temp_iAP2iOSAppDataLength, iAP2iOSAppDataRxdLocal);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tResult iPodControl::OnCBIAP2_StartLocationInformation(iAP2Device_t* iap2Device, iAP2StartLocationInformationParameter* startLocationInformationParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!startLocationInformationParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    //ret = iAP2SetLocationInformationUpdate(mountPoint, true);

    VARTRACE(startLocationInformationParameter->iAP2GlobalPositioningSystemFixData_count);             //GPGGA
    VARTRACE(startLocationInformationParameter->iAP2RecommendedMinimumSpecificGPSTransitData_count);   //GPRMC
    VARTRACE(startLocationInformationParameter->iAP2GPSSatellitesInView_count);                        //GPGSV
    VARTRACE(startLocationInformationParameter->iAP2VehicleHeadingData_count);                         //GPHDT
    VARTRACE(startLocationInformationParameter->iAP2VehicleSpeedData_count);                           //PASCD
    VARTRACE(startLocationInformationParameter->iAP2VehicleGyroData_count);                            //PAGCD
    VARTRACE(startLocationInformationParameter->iAP2VehicleAccelerometerData_count);                   //PAACD

    //trigger SPI update
    tDiPOLocationInfoType locationInfoType;
    locationInfoType.GPGGA = startLocationInformationParameter->iAP2GlobalPositioningSystemFixData_count;
    locationInfoType.GPRMC = startLocationInformationParameter->iAP2RecommendedMinimumSpecificGPSTransitData_count;
    locationInfoType.GPGSV = startLocationInformationParameter->iAP2GPSSatellitesInView_count;
    locationInfoType.GPHDT = startLocationInformationParameter->iAP2VehicleHeadingData_count;
    locationInfoType.PASCD = startLocationInformationParameter->iAP2VehicleSpeedData_count;
    locationInfoType.PAGCD = startLocationInformationParameter->iAP2VehicleGyroData_count;
    locationInfoType.PAACD = startLocationInformationParameter->iAP2VehicleAccelerometerData_count;

    iAPSetDiPOLocationInfo(mountPoint,TRUE, locationInfoType);
    UpdateAppControl(deviceId, APPCONTROL_UPDATE_DIPO_LOCATIONINFO);

    return ret;
}

tResult iPodControl::OnCBIAP2_StopLocationInformation(iAP2Device_t* iap2Device, iAP2StopLocationInformationParameter* stopLocationInformationParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!stopLocationInformationParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);
 //   ret = iAP2SetLocationInformationUpdate(mountPoint, false);

    //trigger SPI update
    tDiPOLocationInfoType locationInfoType;
    locationInfoType.GPGGA =  0;
    locationInfoType.GPRMC =  0;
    locationInfoType.GPGSV =  0;
    locationInfoType.GPHDT =  0;
    locationInfoType.PASCD =  0;
    locationInfoType.PAGCD =  0;
    locationInfoType.PAACD =  0;

    iAPSetDiPOLocationInfo(mountPoint,FALSE, locationInfoType);
    UpdateAppControl(deviceId, APPCONTROL_UPDATE_DIPO_LOCATIONINFO);

    return ret;
}

tResult iPodControl::OnCBIAP2_GPRMCDataStatusValuesNotification(iAP2Device_t* iap2Device, iAP2GPRMCDataStatusValuesNotificationParameter* GPRMCDataStatusValuesNotificationParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!GPRMCDataStatusValuesNotificationParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    VARTRACE(GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueA_count);
    VARTRACE(GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueV_count);
    VARTRACE(GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueX_count);

    //trigger SPI update
    tDiPOGPRMCDataStatusValues GPRMCDataStatusValues;
    GPRMCDataStatusValues.GPRMCDataStatusValueA = GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueA_count;
    GPRMCDataStatusValues.GPRMCDataStatusValueV = GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueV_count;
    GPRMCDataStatusValues.GPRMCDataStatusValueX = GPRMCDataStatusValuesNotificationParameter->iAP2GPRMCDataStatusValueX_count;

    iAPSetDiPOGPRMCDataStatusValues(mountPoint,GPRMCDataStatusValues);
    UpdateAppControl(deviceId, APPCONTROL_UPDATE_DIPO_GPRMCDATASTATUSVALUES_NOTIFICATION);

    return ret;
}

tResult iPodControl::OnCBIAP2_BluetoothConnectionUpdate(iAP2Device_t* iap2Device, iAP2BluetoothConnectionUpdateParameter* bluetoothConnectionUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!bluetoothConnectionUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    if(bluetoothConnectionUpdateParameter->iAP2BluetoothTransportComponentIdentifier) {
        VARTRACE(*bluetoothConnectionUpdateParameter->iAP2BluetoothTransportComponentIdentifier);
    }
    if(bluetoothConnectionUpdateParameter->iAP2BluetoothComponentProfiles) {
        const iAP2BluetoothComponentProfiles* p = bluetoothConnectionUpdateParameter->iAP2BluetoothComponentProfiles;
        VARTRACE(p->iAP2BluetoothHandsFreeProfile_count);
        VARTRACE(p->iAP2BluetoothPhoneBookAccessProfile_count);
        VARTRACE(p->iAP2BluetoothAudioVideoRemotecontrolProfile_count);
        VARTRACE(p->iAP2BluetoothAdvancedAudioDistributionProfile_count);
        VARTRACE(p->iAP2BluetoothHumanInterfaceDeviceProfile_count);
        VARTRACE(p->iAP2BluetoothiAP2LinkProfile_count);
        VARTRACE(p->iAP2BluetoothPersonalAreaNetworkAccessPointProfile_count);
        VARTRACE(p->iAP2BluetoothMessageAccessProfile_count);
        VARTRACE(p->iAP2BluetoothPersonalAreaNetworkClientProfile_count);

        //store current profile
        tBTProfile btProfile;
        btProfile.HandsFree =                       p->iAP2BluetoothHandsFreeProfile_count;
        btProfile.PhoneBookAccess =                 p->iAP2BluetoothPhoneBookAccessProfile_count;
        btProfile.AudioVideoRemoteControl =         p->iAP2BluetoothAudioVideoRemotecontrolProfile_count;
        btProfile.AdvancedAudioDistribution =       p->iAP2BluetoothAdvancedAudioDistributionProfile_count;
        btProfile.HumanInterfaceDevice =            p->iAP2BluetoothHumanInterfaceDeviceProfile_count;
        btProfile.iAP2Link =                        p->iAP2BluetoothiAP2LinkProfile_count;
        btProfile.PersonalAreaNetworkAccessPoint =  p->iAP2BluetoothPersonalAreaNetworkAccessPointProfile_count;
        btProfile.MessageAccess =                   p->iAP2BluetoothMessageAccessProfile_count;
        btProfile.PersonalAreaNetworkClient =       p->iAP2BluetoothPersonalAreaNetworkClientProfile_count;

        iAPSetBTProfile(mountPoint, btProfile);
        VARTRACE(btProfile);

        //send update trigger
        UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_BTPROFILES);

        //store BT iap link status
        iAPSetOption(mountPoint, hasBluetoothiAP2Link, p->iAP2BluetoothiAP2LinkProfile_count);

        vector<string> btmountPoints = iAPGetNumberOfiAP2Link();
        VARTRACE((int)(long)btmountPoints.size()); // thoemel: the additional casting is due to 64 bit compiler

        if(p->iAP2BluetoothiAP2LinkProfile_count && btmountPoints.size() >= 2) {
            //release double iAP2 connection

            const tConnectionType prefDCT = (tConnectionType) LocalSPM::GetDataProvider().iPodControlPreferredConnectionType();

            for(unsigned int i = 0; i < btmountPoints.size(); i++) {
                tDeviceInfo deviceInfo;
                iAPGetDeviceInfo(btmountPoints[i].c_str(), deviceInfo);

                if((deviceInfo.connectionType == DCT_BLUETOOTH && prefDCT == DCT_USB) ||
                   (deviceInfo.connectionType != DCT_BLUETOOTH && prefDCT != DCT_USB)) {

                    ETG_TRACE_USR3(("close redundant IAP2 connection (%s)", btmountPoints[i].c_str()));

                    tAllParameters parameterString;
                    size_t size = sizeof(parameterString);
                    ParameterREMOVE_DEVICE_CONNECTION_ONHOLD(OUT parameterString, IN size, IN btmountPoints[i].c_str(), IN deviceInfo.deviceID);
                    SendEvent(REMOVE_DEVICE_CONNECTION_ONHOLD, IN parameterString);
                }
            }
        }
    }
    return ret;
}

tResult iPodControl::OnCBIAP2_StartVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StartVehicleStatusUpdatesParameter* startVehicleStatusUpdatesParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!startVehicleStatusUpdatesParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    ret = iAP2SetVehicleStatusUpdate(mountPoint, true);

    VARTRACE(startVehicleStatusUpdatesParameter->iAP2Range_count);
    VARTRACE(startVehicleStatusUpdatesParameter->iAP2OutsideTemperature_count);
    VARTRACE(startVehicleStatusUpdatesParameter->iAP2RangeWarning_count);

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    //send initial vehicle status to iPod
    const tUserDataType userDataType = USRDATA_VEHICLESTATUS;
    tUserData userData = {0}; //no data, taken from config

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tResult iPodControl::OnCBIAP2_StopVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StopVehicleStatusUpdatesParameter* stopVehicleStatusUpdatesParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!stopVehicleStatusUpdatesParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);
    ret = iAP2SetVehicleStatusUpdate(mountPoint, false);

    return ret;
}

#ifdef IPODCONTROL_IAP2_PF_R22
tResult iPodControl::OnCBIAP2_CallStateUpdate(iAP2Device_t* iap2Device, iAP2CallStateUpdateParameter* csup, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!csup) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    tGeneralString callUUID = {0};
    if(csup->iAP2CallUUID_count && csup->iAP2CallUUID) {
        VARTRACE(csup->iAP2CallUUID[0]);
        strncpy_r(callUUID, (char *)csup->iAP2CallUUID[0], sizeof(callUUID));
    }

    //update current state
    tDiPOCallStateItem callStateItem = iAP2GetCallStateByUUID(mountPoint, callUUID);

    if(csup->iAP2RemoteID_count && csup->iAP2RemoteID) {
        VARTRACE(csup->iAP2RemoteID[0]);
        strncpy_r(callStateItem.remoteID, (char *)csup->iAP2RemoteID[0], sizeof(callStateItem.remoteID));
    }

    if(csup->iAP2DisplayName_count && csup->iAP2DisplayName) {
        VARTRACE(csup->iAP2DisplayName[0]);
        strncpy_r(callStateItem.displayName, (char *)csup->iAP2DisplayName[0], sizeof(callStateItem.displayName));
    }

    if(csup->iAP2Status_count && csup->iAP2Status) {
        VARTRACE(csup->iAP2Status[0]);
        //CallStateUpdateStatus
        switch(csup->iAP2Status[0]) {
            case IAP2_CALL_STATUS_DISCONNECTED:     /**< Call disconnected */
                callStateItem.status = DIPO_CALL_STATUS_DISCONNECTED;
                break;
            case IAP2_CALL_STATUS_SENDING:          /**< Call in progress */
                callStateItem.status = DIPO_CALL_STATUS_SENDING;
                break;
            case IAP2_CALL_STATUS_RINGING:          /**< Call waiting */
                callStateItem.status = DIPO_CALL_STATUS_RINGING;
                break;
            case IAP2_CALL_STATUS_CONNECTING:       /**< Call connecting */
                callStateItem.status = DIPO_CALL_STATUS_CONNECTING;
                break;
            case IAP2_CALL_STATUS_ACTIVE:           /**< Call in active */
                callStateItem.status = DIPO_CALL_STATUS_ACTIVE;
                break;
            case IAP2_CALL_STATUS_HELD:             /**< Call in hold */
                callStateItem.status = DIPO_CALL_STATUS_HELD;
                break;
            case IAP2_CALL_STATUS_DISCONNECTING:    /**< Call is disconnecting */
                callStateItem.status = DIPO_CALL_STATUS_DISCONNECTING;
                break;
            default:
                break;
        }
    }

    if(csup->iAP2Direction_count && csup->iAP2Direction) {
        VARTRACE(csup->iAP2Direction[0]);
        //CallStateUpdateDirection
        switch(csup->iAP2Direction[0]) {
            case IAP2_CALL_DIRECTION_UNKNOWN:
                callStateItem.direction = DIPO_CALL_DIRECTION_UNKNOWN;
                break;
            case IAP2_CALL_DIRECTION_INCOMING:
                callStateItem.direction = DIPO_CALL_DIRECTION_INCOMING;
                break;
            case IAP2_CALL_DIRECTION_OUTGOING:
                callStateItem.direction = DIPO_CALL_DIRECTION_OUTGOING;
                break;
            default:
                break;
        }
    }

    if(csup->iAP2AddressBookID_count && csup->iAP2AddressBookID) {
        VARTRACE(csup->iAP2AddressBookID[0]);
        strncpy_r(callStateItem.addressBookID, (char *)csup->iAP2AddressBookID[0], sizeof(callStateItem.addressBookID));
    }

    if(csup->iAP2Label_count && csup->iAP2Label) {
        VARTRACE(csup->iAP2Label[0]);
        strncpy_r(callStateItem.label, (char *)csup->iAP2Label[0], sizeof(callStateItem.label));
    }

    if(csup->iAP2Service_count && csup->iAP2Service) {
        VARTRACE(csup->iAP2Service[0]);
        //CallStateUpdateService
        switch(csup->iAP2Service[0]) {
            case IAP2_CALL_SERVICE_UNKNOWN:
                callStateItem.service = DIPO_CALL_SERVICE_UNKNOWN;
                break;
            case IAP2_CALL_SERVICE_TELEPHONY:
                callStateItem.service = DIPO_CALL_SERVICE_TELEPHONY;
                break;
            case IAP2_CALL_SERVICE_FACETIME_AUDIO:
                callStateItem.service = DIPO_CALL_SERVICE_FACETIMEAUDIO;
                break;
            case IAP2_CALL_SERVICE_FACETIME_VIDEO:
                callStateItem.service = DIPO_CALL_SERVICE_FACETIMEVIDEO;
                break;
            default:
                break;
        }
    }

    if(csup->iAP2IsConferenced_count && csup->iAP2IsConferenced) {
        VARTRACE(csup->iAP2IsConferenced[0]);
        callStateItem.isConferenced = csup->iAP2IsConferenced[0];
    }

    if(csup->iAP2ConferenceGroup_count && csup->iAP2ConferenceGroup) {
        VARTRACE(csup->iAP2ConferenceGroup[0]);
        callStateItem.conferenceGroup = csup->iAP2ConferenceGroup[0];
    }

    if(csup->iAP2DisconnectReason_count && csup->iAP2DisconnectReason) {
        VARTRACE(csup->iAP2DisconnectReason[0]);
        //CallStateUpdateDisconnectReason
        switch(csup->iAP2DisconnectReason[0]) {
            case IAP2_CALL_DISCONNECT_REASON_ENDED:
                callStateItem.disconnectReason = DIPO_CALL_DISCONNECT_REASON_ENDED;
                break;
            case IAP2_CALL_DISCONNECT_REASON_DECLINED:
                callStateItem.disconnectReason = DIPO_CALL_DISCONNECT_REASON_DECLINED;
                break;
            case IAP2_CALL_DISCONNECT_REASON_FAILED:
                callStateItem.disconnectReason = DIPO_CALL_DISCONNECT_REASON_FAILED;
                break;
            default:
                break;
        }
    }

    if(csup->iAP2StartTimestamp_count && csup->iAP2StartTimestamp)
    {
        VARTRACE(csup->iAP2StartTimestamp[0]);
        callStateItem.startTimestamp = csup->iAP2StartTimestamp[0];
    }

    //store current state
    iAP2UpdateOrInsertCallState(mountPoint, callStateItem);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_CALLSTATE);
    return ret;
}

tResult iPodControl::OnCBIAP2_CommunicationsUpdate(iAP2Device_t* iap2Device, iAP2CommunicationsUpdateParameter* cup, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!cup) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPOCommunications comm = iAPGetCommunications(mountPoint);

    if(cup->iAP2SignalStrength_count && cup->iAP2SignalStrength) {
        VARTRACE(cup->iAP2SignalStrength[0]);
        switch(cup->iAP2SignalStrength[0]) {
            case IAP2_SIGNAL_STRENGTH_0_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_0BARS;
                break;
            case IAP2_SIGNAL_STRENGTH_1_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_1BAR;
                break;
            case IAP2_SIGNAL_STRENGTH_2_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_2BARS;
                break;
            case IAP2_SIGNAL_STRENGTH_3_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_3BARS;
                break;
            case IAP2_SIGNAL_STRENGTH_4_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_4BARS;
                break;
            case IAP2_SIGNAL_STRENGTH_5_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_5BARS;
                break;
            default:
                break;
        }
    }

    if(cup->iAP2RegistrationStatus_count && cup->iAP2RegistrationStatus) {
        VARTRACE(cup->iAP2RegistrationStatus[0]);
        switch(cup->iAP2RegistrationStatus[0]) {
            case IAP2_REGISTRATION_STATUS_UNKNOWN:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_UNKNOWN;
                break;
            case IAP2_REGISTRATION_STATUS_NOT_REGISTERED:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_NOTREGISTERED;
                break;
            case IAP2_REGISTRATION_STATUS_SEARCHING:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_SEARCHING;
                break;
            case IAP2_REGISTRATION_STATUS_DENIED:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_DENIED;
                break;
            case IAP2_REGISTRATION_STATUS_HOME:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDHOME;
                break;
            case IAP2_REGISTRATION_STATUS_ROAMING:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDROAMING;
                break;
            case IAP2_REGISTRATION_STATUS_EMERGENCY_CALLS:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_EMERGENCYCALLONLY;
                break;
            default:
                break;
        }
    }

    if(cup->iAP2AirplaneModeStatus_count && cup->iAP2AirplaneModeStatus) {
        VARTRACE(cup->iAP2AirplaneModeStatus[0]);
        comm.airplaneModeStatus = (tBoolean)cup->iAP2AirplaneModeStatus[0];
    }

    if(cup->iAP2CarrierName_count && cup->iAP2CarrierName) {
        VARTRACE(cup->iAP2CarrierName[0]);
        strncpy_r(comm.carrierName, (char *)cup->iAP2CarrierName[0], sizeof(comm.carrierName));
    }

    if(cup->iAP2CellularSupported_count && cup->iAP2CellularSupported) {
        VARTRACE(cup->iAP2CellularSupported[0]);
        comm.cellularSupported = (tBoolean)cup->iAP2CellularSupported[0];
    }

    if(cup->iAP2TelephonyEnabled_count && cup->iAP2TelephonyEnabled) {
        VARTRACE(cup->iAP2TelephonyEnabled[0]);
        comm.telephonyEnabled = (tBoolean)cup->iAP2TelephonyEnabled[0];
    }

    if(cup->iAP2FaceTimeAudioEnabled_count && cup->iAP2FaceTimeAudioEnabled) {
        VARTRACE(cup->iAP2FaceTimeAudioEnabled[0]);
        comm.faceTimeAudioEnabled = (tBoolean)cup->iAP2FaceTimeAudioEnabled[0];
    }

    if(cup->iAP2FaceTimeVideoEnabled_count && cup->iAP2FaceTimeVideoEnabled) {
        VARTRACE(cup->iAP2FaceTimeVideoEnabled[0]);
        comm.faceTimeVideoEnabled = (tBoolean)cup->iAP2FaceTimeVideoEnabled[0];
    }

    if(cup->iAP2MuteStatus_count && cup->iAP2MuteStatus) {
        VARTRACE(cup->iAP2MuteStatus[0]);
        comm.muteStatus = (tBoolean)cup->iAP2MuteStatus[0];
    }

    if(cup->iAP2CurrentCallCount_count && cup->iAP2CurrentCallCount) {
        VARTRACE(cup->iAP2CurrentCallCount[0]);
        comm.currentCallCount = (tU8)cup->iAP2CurrentCallCount[0];
    }

    if(cup->iAP2NewVoicemailCount_count && cup->iAP2NewVoicemailCount) {
        VARTRACE(cup->iAP2NewVoicemailCount[0]);
        comm.newVoicemailCount = (tU8)cup->iAP2NewVoicemailCount[0];
    }

    if(cup->iAP2InitiateCallAvailable_count && cup->iAP2InitiateCallAvailable) {
        VARTRACE(cup->iAP2InitiateCallAvailable[0]);
        comm.initiateCallAvalable = (tBoolean)cup->iAP2InitiateCallAvailable[0];
    }

    if(cup->iAP2EndAndAcceptAvailable_count && cup->iAP2EndAndAcceptAvailable) {
        VARTRACE(cup->iAP2EndAndAcceptAvailable[0]);
        comm.endAndAcceptAvailable = (tBoolean)cup->iAP2EndAndAcceptAvailable[0];
    }

    if(cup->iAP2HoldAndAcceptAvailable_count && cup->iAP2HoldAndAcceptAvailable) {
        VARTRACE(cup->iAP2HoldAndAcceptAvailable[0]);
        comm.holdAndAcceptAvailable = (tBoolean)cup->iAP2HoldAndAcceptAvailable[0];
    }

    if(cup->iAP2SwapAvailable_count && cup->iAP2SwapAvailable) {
        VARTRACE(cup->iAP2SwapAvailable[0]);
        comm.swapAvailable = (tBoolean)cup->iAP2SwapAvailable[0];
    }

    if(cup->iAP2MergeAvailable_count && cup->iAP2MergeAvailable) {
        VARTRACE(cup->iAP2MergeAvailable[0]);
        comm.mergeAvailable = (tBoolean)cup->iAP2MergeAvailable[0];
    }

    if(cup->iAP2HoldAvailable_count && cup->iAP2HoldAvailable) {
        VARTRACE(cup->iAP2HoldAvailable[0]);
        comm.holdAvailable = (tBoolean)cup->iAP2HoldAvailable[0];
    }

    //store current state
    iAPSetCommunications(mountPoint, comm);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_COMMUNICATIONS);
    return ret;
}
#else
tResult iPodControl::OnCBIAP2_TelephonyCallStateInformation(iAP2Device_t* iap2Device, iAP2TelephonyCallStateInformationParameter* tcsip, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!tcsip) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    tGeneralString callUUID = {0};
    if(tcsip->iAP2UniqueCallID_count) {
        VARTRACE(tcsip->iAP2UniqueCallID[0]);
        strncpy_r(callUUID, (char *)tcsip->iAP2UniqueCallID[0], sizeof(callUUID));
    }

    //update current state
    tDiPOCallStateItem callStateItem = iAP2GetCallStateByUUID(mountPoint, callUUID);

    if(tcsip->iAP2CallStatePhoneNumber_count) {
        VARTRACE(tcsip->iAP2CallStatePhoneNumber[0]);
        //apply non empty information
        if(tcsip->iAP2CallStatePhoneNumber[0][0]) {
            strncpy_r(callStateItem.remoteID, (char *)tcsip->iAP2CallStatePhoneNumber[0], sizeof(callStateItem.remoteID));
        }
    }

    if(tcsip->iAP2CallStateCallerName_count) {
        VARTRACE(tcsip->iAP2CallStateCallerName[0]);
        //apply non empty information
        if(tcsip->iAP2CallStateCallerName[0][0]) {
            strncpy_r(callStateItem.displayName, (char *)tcsip->iAP2CallStateCallerName[0], sizeof(callStateItem.displayName));
        }
    }

    if(tcsip->iAP2CallStateStatus_count) {
        VARTRACE(tcsip->iAP2CallStateStatus[0]);
        //CallStateUpdateStatusLegcay
        switch(tcsip->iAP2CallStateStatus[0]) {
            case IAP2_TELEPHONY_CALL_DISCONNECTED:
                callStateItem.status = DIPO_CALL_STATUS_DISCONNECTED;
                break;
            case IAP2_TELEPHONY_CALL_ACTIVE:
                callStateItem.status = DIPO_CALL_STATUS_ACTIVE;
                break;
            case IAP2_TELEPHONY_CALL_WAITING:
                callStateItem.status = DIPO_CALL_STATUS_HELD;
                break;
            case IAP2_TELEPHONY_CALL_CONNECTING:
                callStateItem.status = DIPO_CALL_STATUS_CONNECTING;
                break;
            default:
                break;
        }
    }

    if(tcsip->iAP2CallStateDirection_count) {
        VARTRACE(tcsip->iAP2CallStateDirection[0]);
        //CallStateUpdateDirectionLegcay
        switch(tcsip->iAP2CallStateDirection[0]) {
            case IAP2_TELEPHONY_CALL_INCOMING:
                callStateItem.direction = DIPO_CALL_DIRECTION_INCOMING;
                break;
            case IAP2_TELEPHONY_CALL_OUTGOING:
                callStateItem.direction = DIPO_CALL_DIRECTION_OUTGOING;
                break;
            case IAP2_TELEPHONY_CALL_UNKNOWN:
                callStateItem.direction = DIPO_CALL_DIRECTION_UNKNOWN;
                break;
            default:
                break;
        }
    }

    if(tcsip->iAP2UniqueCallID_count) {
        VARTRACE(tcsip->iAP2UniqueCallID[0]);
    }
    if(tcsip->iAP2CallStatevCardFileTransferIdentifier_count) {
        VARTRACE(tcsip->iAP2CallStatevCardFileTransferIdentifier[0]);
    }

    //store current state
    iAP2UpdateOrInsertCallState(mountPoint, callStateItem);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_CALLSTATE);
    return ret;
}

tResult iPodControl::OnCBIAP2_TelephonyUpdate(iAP2Device_t* iap2Device, iAP2TelephonyUpdateParameter* tup, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!tup) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPOCommunications comm = iAPGetCommunications(mountPoint);

    if(tup->iAP2TelephonySignalStrength_count) {
        VARTRACE(tup->iAP2TelephonySignalStrength[0]);
        switch(tup->iAP2TelephonySignalStrength[0]) {
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_0_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_0BARS;
                break;
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_1_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_1BAR;
                break;
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_2_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_2BARS;
                break;
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_3_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_3BARS;
                break;
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_4_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_4BARS;
                break;
            case IAP2_TELEPHONY_SIGNAL_STRENGTH_5_BARS:
                comm.signalStrength = DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_5BARS;
                break;
            default:
                break;
        }
    }

    if(tup->iAP2TelephonyRegistrationStatus_count) {
        VARTRACE(tup->iAP2TelephonyRegistrationStatus[0]);
        switch(tup->iAP2TelephonyRegistrationStatus[0]) {
            case IAP2_TELEPHONY_REGISTRATION_UNKNOWN:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_UNKNOWN;
                break;
            case IAP2_TELEPHONY_REGISTRATION_NOT_REGISTERED:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_NOTREGISTERED;
                break;
            case IAP2_TELEPHONY_REGISTRATION_SEARCHING:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_SEARCHING;
                break;
            case IAP2_TELEPHONY_REGISTRATION_DENIED:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_DENIED;
                break;
            case IAP2_TELEPHONY_REGISTRATION_HOME:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDHOME;
                break;
            case IAP2_TELEPHONY_REGISTRATION_ROAMING:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDROAMING;
                break;
            case IAP2_TELEPHONY_REGISTRATION_EMERGENCY_CALLS:
                comm.registrationStatus = DIPO_COMMUNICATIONS_REGISTRATION_EMERGENCYCALLONLY;
                break;
            default:
                break;
        }
    }

    if(tup->iAP2TelephonyAirplaneModeStatus_count) {
        VARTRACE(tup->iAP2TelephonyAirplaneModeStatus[0]);
        comm.airplaneModeStatus = (tBoolean)tup->iAP2TelephonyAirplaneModeStatus[0];
    }

    if(tup->iAP2TelephonyTTYStatus_count) {
        VARTRACE(tup->iAP2TelephonyTTYStatus[0]);
    }

    if(tup->iAP2TelephonyMobileOperator_count) {
        VARTRACE(tup->iAP2TelephonyMobileOperator[0]);
        if(tup->iAP2TelephonyMobileOperator[0][0]) {
            strncpy_r(comm.carrierName, (char *)tup->iAP2TelephonyMobileOperator[0], sizeof(comm.carrierName));
        }
    }

    //store current state
    iAPSetCommunications(mountPoint, comm);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_COMMUNICATIONS);
    return ret;
}
#endif //IPODCONTROL_IAP2_PF_R22
#endif //IPODCONTROL_IAP2_PF_AVAIL

tResult iPodControl::OnCBIAP2_DeviceUUIDUpdate(iAP2Device_t *iap2Device, iAP2DeviceUUIDUpdateParameter* deviceUUIDUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if( NULL != iap2Device )
    {
        tMountPoint mountPoint = {0};
        ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
        if ( MP_NO_ERROR == ret )
        {
            VARTRACE(mountPoint);
            if( NULL != deviceUUIDUpdateParameter)
            {
                VARTRACE(deviceUUIDUpdateParameter->iAP2UUID_count);
                if( deviceUUIDUpdateParameter->iAP2UUID_count &&
                    deviceUUIDUpdateParameter->iAP2UUID )
                {
                    VARTRACE((char*)deviceUUIDUpdateParameter->iAP2UUID[0]);
                    tUUID deviceUUID;
                    strncpy_r(deviceUUID, (const char*)deviceUUIDUpdateParameter->iAP2UUID[0], sizeof(deviceUUID) - 1);

                    // Set device UUID in iPodControl handle map
                    iAPSetDeviceUUID(mountPoint, deviceUUID);

                    tDeviceID deviceID = iAPGetDeviceID(mountPoint);

                    // Set device UUID in DB
                    ret = SendDBSetDeviceUUID(deviceID, deviceUUID);

                    // Check for multiple iAP sessions in same device
                    CheckMultipleIAPSessionsForSameDevice(deviceID, deviceUUID);
                }
            }
            else
            {
                ETG_TRACE_ERR(("Invalid parameter"));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            }
        }
        else
        {
            ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        ret = MP_ERR_IPOD_NO_DEVICE;
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_WirelessCarPlayUpdate(iAP2Device_t *iap2Device, iAP2WirelessCarPlayUpdateParameter* wirelessCarPlayUpdateParameter, void* /*context*/)
{
    ENTRY;

    tResult ret = MP_NO_ERROR;

    if(NULL != iap2Device)
    {
        tMountPoint mountPoint = {0};
        ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
        if (MP_NO_ERROR == ret)
        {
            VARTRACE(mountPoint);
            if(NULL != wirelessCarPlayUpdateParameter)
            {
                VARTRACE(wirelessCarPlayUpdateParameter->iAP2Status_count);
                if(wirelessCarPlayUpdateParameter->iAP2Status_count)
                {
                    VARTRACE(wirelessCarPlayUpdateParameter->iAP2Status[0]);

                    tBool status = false;
                    status = (IAP2_WIRELESS_CARPLAY_STATUS_AVAILABLE == wirelessCarPlayUpdateParameter->iAP2Status[0]) ? true : false;

                    // Fix for RTC1-836469 Location Information is not exercised in iAP summary view
                    // If CPW session is already established (if diPOCaps is DIPO_CAP_CARPLAY_WIFI), then do not update diPOCaps back to DIPO_CAP_CARPLAY_WIFI_FEASIBLE
                    // Otherwise, subsequent location updates from SPI via onDiPOTransferDRData with deviceID as ZERO, will not be forwarded to Apple device.
                    tDiPOCaps diPOCaps = iAP2_GetDiPOCaps(mountPoint);
                    VARTRACE(diPOCaps);

                    // Check and update the value of DipoCaps
                    if(true == status)
                    {
                        if(DIPO_CAP_CARPLAY_WIFI != diPOCaps)
                        {
                            diPOCaps = DIPO_CAP_CARPLAY_WIFI_FEASIBLE;
                        }
                        else
                        {
                            ETG_TRACE_USR4((" Skipping update to DIPO_CAP_CARPLAY_WIFI_FEASIBLE from DIPO_CAP_CARPLAY_WIFI! "));
                        }
                    }
                    else
                    {
                        diPOCaps = DIPO_CAP_CARPLAY_WIFI_NOT_FEASIBLE;
                    }
                    VARTRACE(diPOCaps);

                    // Update DB and member variables if there is a change in value
                    if(diPOCaps != iAP2_GetDiPOCaps(mountPoint))
                    {
                        iAPSetLastWirelessCarplayStatus(mountPoint,diPOCaps);
                        tDeviceID deviceID = iAPGetDeviceID(mountPoint);
                        VARTRACE(deviceID);
                        // Set DiPOCaps in DB
                        SendDBChangeDiPOSettings(IN deviceID, IN diPOCaps, IN FALSE, IN "");
                    }
                }
            }
            else
            {
                ETG_TRACE_ERR((" Invalid parameter! "));
                ret = MP_ERR_IPOD_INVALID_PARAM;
            }
        }
        else
        {
            ETG_TRACE_ERR((" iAP2_GetMountPoint failed! "));
        }
    }
    else
    {
        ETG_TRACE_ERR((" No valid device pointer for IAP2! "));
        ret = MP_ERR_IPOD_NO_DEVICE;
    }

    return ret;
}

tResult iPodControl::OnCBIAP2_RequestAccessoryWiFiConfigurationInformation(iAP2Device_t* iap2Device, iAP2RequestAccessoryWiFiConfigurationInformationParameter */*requestAccessoryWiFiConfigurationInformationParameter*/, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return ret;
    }
    VARTRACE(mountPoint);

    //ToDo: CARPLAY WIFI: create a new INTERNAL EVENT to call new function from where the Wifi credentials can be updated to device.
    const tUserDataType userDataType = USRDATA_ACC_WIFI_CREDENTIALS;
    tUserData userData = {0};

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);

    ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
    } else {
        ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;
}

#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
tResult iPodControl::OnCBIAP2_DeviceTransportIdentifierNotification(iAP2Device_t* iap2Device,iAP2DeviceTransportIdentifierNotificationParameter *deviceTransportIdentifierNotificationParameter, void* context)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    tUSBSerial appleDeviceUSBSerialNumber = {0};
    tMACAddress appleDeviceMACAddress = {0};
    tMACAddress rawMACAddress = {0};
    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return ret;
    }
    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);
    if(!deviceTransportIdentifierNotificationParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }
    VARTRACE(deviceTransportIdentifierNotificationParameter->iAP2BluetoothTransportIdentifier_count);
    if(deviceTransportIdentifierNotificationParameter->iAP2BluetoothTransportIdentifier_count &&
            deviceTransportIdentifierNotificationParameter->iAP2BluetoothTransportIdentifier) {
        VARTRACE((char*)deviceTransportIdentifierNotificationParameter->iAP2BluetoothTransportIdentifier[0]);
        strncpy_r(rawMACAddress,(char*)deviceTransportIdentifierNotificationParameter->iAP2BluetoothTransportIdentifier[0], sizeof(rawMACAddress));
        VARTRACE(rawMACAddress);
        for(int i = 0, p = 0; i < strlen_r(rawMACAddress);i++)
        {
            if(rawMACAddress[i] != ':')
            {
                appleDeviceMACAddress[p] = rawMACAddress[i];
                p++;
            }

        }
        VARTRACE(appleDeviceMACAddress);
    }
    VARTRACE(deviceTransportIdentifierNotificationParameter->iAP2USBTransportIdentifier_count);
    if(deviceTransportIdentifierNotificationParameter->iAP2USBTransportIdentifier_count &&
            deviceTransportIdentifierNotificationParameter->iAP2USBTransportIdentifier) {
        VARTRACE((char*)deviceTransportIdentifierNotificationParameter->iAP2USBTransportIdentifier[0]);
        strncpy_r(appleDeviceUSBSerialNumber,(char*)deviceTransportIdentifierNotificationParameter->iAP2USBTransportIdentifier[0], sizeof(appleDeviceUSBSerialNumber));
    }
    SetAppleDeviceTransportIdentifiers(mountPoint,appleDeviceMACAddress,appleDeviceUSBSerialNumber);
    ret = SendDBsetAppleDeviceTransportIdentifiers(deviceId,appleDeviceMACAddress,appleDeviceUSBSerialNumber);
    return ret;
}
#endif
#endif

tResult iPodControl::OnCBIAP2_ListUpdate(iAP2Device_t* iap2Device, iAP2ListUpdateParameter* listUpdateParameter, void* context)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    U32 callDuration = 0;
    if(!listUpdateParameter || !listUpdateParameter->iAP2RecentsList){
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }
    ETG_TRACE_USR3(("listUpdateParameter->iAP2RecentsList->iAP2Duration_count:%d", listUpdateParameter->iAP2RecentsList->iAP2Duration_count));

    if(listUpdateParameter->iAP2RecentsList->iAP2Duration_count){

        ETG_TRACE_USR3(("listUpdateParameter->iAP2RecentsList->iAP2Duration:%d", listUpdateParameter->iAP2RecentsList->iAP2Duration[0]));
        callDuration = listUpdateParameter->iAP2RecentsList->iAP2Duration[0];

        iAP2UpdateCallDuration(mountPoint,callDuration);
        //send update trigger
        UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_CALLDURATION);
    }
    else
    {
        ETG_TRACE_USR3(("Call Duration not updated"));

    }
    VARTRACE(callDuration);

    return ret;
}

#ifdef IPODCONTROL_IAP2_PF_OOBBT_AVAIL
#ifdef TARGET_BUILD
tResult iPodControl::OnCBIAP2_OOBBTPairingLinkKeyInformation(iAP2Device_t* iap2Device, iAP2OOBBTPairingLinkKeyInformationParameter* oobBTPairingLinkKeyInformationParameter, void* context)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    tU64 linkLen = 0;
    tLinkKey linkKey;

    tU64 macLen = 0;
    tMACAddress appleDeviceMACAddress;


    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    tDeviceInfo deviceInfo;
    iAPGetDeviceInfo(mountPoint,deviceInfo);
    VARTRACE(deviceInfo.deviceID);

    if(oobBTPairingLinkKeyInformationParameter && oobBTPairingLinkKeyInformationParameter->iAP2LinkKey_count && oobBTPairingLinkKeyInformationParameter->iAP2LinkKey) {
        VARTRACE(oobBTPairingLinkKeyInformationParameter->iAP2LinkKey->iAP2BlobData);
        linkLen = oobBTPairingLinkKeyInformationParameter->iAP2LinkKey->iAP2BlobLength;
        string linkKeystr = string_to_hex(reinterpret_cast<const char*>(oobBTPairingLinkKeyInformationParameter->iAP2LinkKey->iAP2BlobData),linkLen);
        strcpy(linkKey,(const char*)linkKeystr.c_str());
        VARTRACE(linkKey);
    }

    if(oobBTPairingLinkKeyInformationParameter && oobBTPairingLinkKeyInformationParameter->iAP2AppleDeviceMACAddress_count && oobBTPairingLinkKeyInformationParameter->iAP2AppleDeviceMACAddress) {
        VARTRACE(oobBTPairingLinkKeyInformationParameter->iAP2AppleDeviceMACAddress->iAP2BlobData);
        macLen = oobBTPairingLinkKeyInformationParameter->iAP2AppleDeviceMACAddress->iAP2BlobLength;
        string macAddrstr = string_to_hex(reinterpret_cast<const char*>(oobBTPairingLinkKeyInformationParameter->iAP2AppleDeviceMACAddress->iAP2BlobData),macLen);
        strcpy(appleDeviceMACAddress,(const char*)macAddrstr.c_str());
        VARTRACE(appleDeviceMACAddress);
    }

    const tUserDataType userDataType = USRDATA_OOBBT_PAIRING_COMPLETION_INFORMATION;
    tUserData userData = {0};
    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "ittti",deviceInfo.deviceID,appleDeviceMACAddress,deviceInfo.deviceName,linkKey,OOB_APPLE);

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tResult iPodControl::OnCBIAP2_StartOOBBTPairing(iAP2Device_t* iap2Device, iAP2StartOOBBTPairingParameter* startOOBBTPairingParameter, void* context)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    VARTRACE(mountPoint);
    tDeviceID deviceId = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceId);

    const tUserDataType userDataType = USRDATA_OOBBT_PAIRING_ACCESSORY_INFORMATION;
    tUserData userData = {0};

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}
#endif
#endif  //IPODCONTROL_IAP2_PF_OOBBT_AVAIL

// private
tResult iPodControl::SendPlaybackStatus(const tMountPoint mountPoint,
        const tPEPlaybackState status) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    ETG_TRACE_USR3(("iPodControl::SendPlaybackStatus() mountPoint:%s", mountPoint));
    tResult ret = MP_NO_ERROR;

    VARTRACE(status);
    /* Status response: Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tMetadata meta = { 0 };

    tObjectID ObjectID = OBJECT_ID_NONE;
    ret = ParameterPLAYBACK_STATUS_RESPONSE(OUT parameterString, IN size,
            IN status, IN meta, IN meta, IN meta, IN meta,IN ObjectID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendPlaybackStatus() - Error while preparing parameter string"));
    }
    return SendEvent(PLAYBACK_STATUS_RESPONSE, IN parameterString);
}

tResult iPodControl::SendPlaybackStatusToPlayerManager(
        const tMountPoint mountPoint, const tPEPlaybackState status) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    ETG_TRACE_USR3(("iPodControl::SendPlaybackStatusToPlayerManager()"));
    VARTRACE(status);
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    tMetadata meta1 = { 0 };
    tMetadata meta2 = { 0 };
    tMetadata meta3 = { 0 };
    tMetadata meta4 = { 0 };
    tTitle metaTitle = { 0 };
    tUUID uid = { 0 };
    tMediaType mediaType = MTY_UNKNOWN;
    iAPGetNowPlayingMetadata(meta1, meta2, meta3, meta4, metaTitle, mediaType, mountPoint,uid);

    VARTRACE(meta1);
    VARTRACE(meta2);
    VARTRACE(meta3);
    VARTRACE(meta4);
    VARTRACE(metaTitle);
    VARTRACE(mediaType);
    VARTRACE(uid);

    /* Send PLAYBACK_STATUS_RESPONSE message to PlayerManger to update playback state */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::PLAYBACK_STATUS_RESPONSE",
            sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tObjectID ObjectID = OBJECT_ID_NONE;

    ret = LocalSPM::GetPlayerManager().ParameterPLAYBACK_STATUS_RESPONSE(
            OUT parameterString, IN size, IN status, IN meta1,
            IN meta2, IN meta3, IN meta4, IN metaTitle, IN mediaType,IN (tConvertFlag)0,uid,ObjectID);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendPlaybackStatusToPlayerManager() - Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    return Dispatcher::GetInstance().SendMessage( IN messageString,
            IN parameterString);
}

tResult iPodControl::SendNowPlayingStatus(const tPEHandle handle, const tURL url)
{
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    VARTRACE(handle);
    VARTRACE(url);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterNOW_PLAYING_STATUS(OUT parameterString, IN size,
            IN handle, IN url);
    return SendEvent(NOW_PLAYING_STATUS, IN parameterString);
}

tResult iPodControl::SendNowPlayingStatusToPlayerManager(
        const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    const tPEHandle handle = iAPGetPEHandle(mountPoint);

    tMetadata meta1 = { 0 };
    tMetadata meta2 = { 0 };
    tMetadata meta3 = { 0 };
    tMetadata meta4 = { 0 };
    tTitle metaTitle = { 0 };
    tMediaType mediaType = MTY_UNKNOWN;
    tUUID uid = { 0 };
    iAPGetNowPlayingMetadata(meta1, meta2, meta3, meta4, metaTitle, mediaType, mountPoint,uid);

    const tBitRate bitRate = 128 << 10; //Not valid for ipod
    me::samplerate_i sampleRate = iAPGetSampleRate(mountPoint);
    const tAudioChannelFormat audioChannelFormat = ACF_STEREO;

    VARTRACE(handle);
    VARTRACE(meta1);
    VARTRACE(meta2);
    VARTRACE(meta3);
    VARTRACE(meta4);
    VARTRACE(metaTitle);
    VARTRACE(mediaType);
    VARTRACE(bitRate);
    VARTRACE(sampleRate);
    VARTRACE(audioChannelFormat);
    VARTRACE(uid);

    /* Send PLAYBACK_STATUS_RESPONSE message to PlayerManger to update playback state */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::NOW_PLAYING_STATUS",
            sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterNOW_PLAYING_STATUS(
            OUT parameterString, IN size, IN handle, IN meta1,
            IN meta2, IN meta3, IN meta4, IN metaTitle, IN mediaType, IN bitRate, IN sampleRate, IN audioChannelFormat,IN (tConvertFlag)0,uid,NULL);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    return Dispatcher::GetInstance().SendMessage( IN messageString, IN parameterString);
}

tResult iPodControl::SendAlbumArtStatusToPlayerManager(const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    VARTRACE(mountPoint);

    tResult ret = MP_NO_ERROR;

    const tPEHandle handle = iAPGetPEHandle(mountPoint);
    tAvailable albumArtAvail = false;

    //Fix for Bug 536833 - ubdate the album art availability to playermanager as false while the Albumart timer is in progress.
    if(m_AlbumArtTimerID)
    {
        ETG_TRACE_USR3(("Albumart timer is running. So update the albumArt availibity as false"));
    }
    else
    {
        albumArtAvail = iAPIsAlbumArtAvailable(mountPoint);
    }

    VARTRACE(handle);
    VARTRACE(albumArtAvail);

    /* Send ALBUM_ART_STATUS message to PlayerManger to update album art */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::ALBUM_ART_STATUS",
            sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = LocalSPM::GetPlayerManager().ParameterALBUM_ART_STATUS(OUT parameterString, IN size, IN handle, IN albumArtAvail);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::SendPlaybackModeToPlayerManager(const tMountPoint mountPoint)       //Roadmap 13010, 13008
{
    ENTRY;
    VARTRACE(mountPoint);

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    tPlaybackMode playbackMode = iAPGetPlaybackMode(mountPoint);
    VARTRACE(playbackMode);

    /* Send PLAYBACK_MODE_STATUS message to PlayerManger */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::PLAYBACK_MODE_STATUS", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetPlayerManager().ParameterPLAYBACK_MODE_STATUS( OUT parameterString, IN size, IN deviceID, IN playbackMode);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::SendRepeatModeToPlayerManager(const tMountPoint mountPoint)       //Roadmap 13010, 13008
{
    ENTRY;
    VARTRACE(mountPoint);

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    tRepeatMode repeatMode = iAPGetRepeatMode(mountPoint);
    VARTRACE(repeatMode);

    /* Send REPEAT_MODE_STATUS message to PlayerManger */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::REPEAT_MODE_STATUS", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetPlayerManager().ParameterREPEAT_MODE_STATUS( OUT parameterString, IN size, IN deviceID, IN repeatMode);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::RemoteActivity(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    if(!iAPIsRCSupported(mountPoint) && !LocalSPM::GetDataProvider().IsPermanentStreamingActive(DTY_IPOD)) {
        if(!LocalSPM::GetDataProvider().iPodControlIAP1ChapterStreaming()) {
            ETG_TRACE_USR4(("Suppressing REMOTE_ACTIVITY, remote control not possible"));
            return ret;
        }
    }

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    const bool isCarPlayMode = iAP2_IsCarPlayMode(mountPoint);
    VARTRACE(isCarPlayMode);
    if(!isCarPlayMode) { //CarPlay remote activity issue GMMY17-10983
        m_RemoteActivityDeviceID = deviceID;
    }

    //cached VT record for all live list types got invalid
    iAPClearLiveVTRecords(mountPoint);

    /* Send REMOTE_ACTIVITY message to PlayerManger */
    char messageString[64];
    strncpy_r(messageString, "PlayerManagerSM::REMOTE_ACTIVITY", sizeof(messageString));
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetPlayerManager().ParameterREMOTE_ACTIVITY(
            OUT parameterString, IN size, IN deviceID);
    Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    return MP_NO_ERROR;
}

void iPodControl::iAP2CheckRemoteActivity(const tMountPoint mountPoint, const tBoolean isStreaming, const tBoolean hostMode, tBoolean &remoteActivitySent)
{
    ENTRY_INTERNAL;
    //VARTRACE(mountPoint);

    if(!remoteActivitySent)
    {
        if(LocalSPM::GetDataProvider().updateReadyToPlay()){
            SendUpdateReadyToPlay(mountPoint);      //Update ReadyToPlay Property to Connectivity
        }

        if(hostMode || !isStreaming){
            SendRemoteActivity(mountPoint);
        } else {
            OnFocusAppChanges(mountPoint);
        }
        remoteActivitySent = true; //prevent multipe events
    }
}

void iPodControl::iAP1CheckRemoteActivity(const tMountPoint mountPoint, const tBoolean isStreaming)
{
    ENTRY_INTERNAL;
    //VARTRACE(mountPoint);

    if(!isStreaming){
        SendRemoteActivity(mountPoint);
    } else {
        OnFocusAppChanges(mountPoint);
    }
}

tResult iPodControl::SendRemoteActivity(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterREMOTE_ACTIVITY(OUT parameterString, IN size, IN mountPoint);
    SendEvent(REMOTE_ACTIVITY, IN parameterString);

    return MP_NO_ERROR;
}

tResult iPodControl::SetRemoteActivity(const tDeviceID deviceID) //CarPlay Mode: SPI calls GET_AUDIO_DEVICE
{
    //from PlayerManager
    ENTRY;
    VARTRACE(deviceID);

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    const tBoolean isStreaming = iAPIsStreaming(IN mountPoint);
    VARTRACE(isStreaming);

    if(!isStreaming) {
        m_RemoteActivityDeviceID = deviceID; //CarPlay remote activity issue GMMY17-10983
        SendRemoteActivity(mountPoint);
    } else {
        OnFocusAppChanges(mountPoint);
    }
    return MP_NO_ERROR;
}

tResult iPodControl::StreamingModeOn(const tDeviceType deviceType, const tDeviceID deviceID, const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(deviceType);
    VARTRACE(deviceID);
    VARTRACE(mountPoint);

    const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
    const tBoolean isExpected = !iAPIsTimeElapsed(IN mountPoint, IN pbNotifications, IN LocalSPM::GetDataProvider().iPodControlPBNotificationTimeoutMS());

    VARTRACE(isActive);
    VARTRACE(isExpected);
    VARTRACE(m_RemoteActivityDeviceID);

    //switch off filter for CarPlay Browse feature
    //SUZUKI-18954
    //update PM with current iPod settings
    SendPlaybackModeToPlayerManager(mountPoint);
    SendRepeatModeToPlayerManager(mountPoint);

    if(m_RemoteActivityDeviceID == deviceID) { //Triggered by previous call REMOTE_ACTIVITY to PM
        //allocate Mediaplayer
        OnFocusAppChanges(mountPoint);
    }

    //keep flag to avoid double play commands to iPod - GMMY16-26007
    //m_RemoteActivityDeviceID = DEVICE_ID_NOT_SET;
    return MP_NO_ERROR;
}

tResult iPodControl::ClearVTCache(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);

    //cached VT record for all live list types got invalid
    iAPClearLiveVTRecords(mountPoint);

    return MP_NO_ERROR;
}

tResult iPodControl::SendClearVTCache(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterCLEAR_VT_CACHE(OUT parameterString, IN size, IN mountPoint);
    SendEvent(CLEAR_VT_CACHE, IN parameterString);

    return MP_NO_ERROR;
}

tResult iPodControl::OnFocusAppChanges(const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    tAppName appName = { 0 };
    iAPGetFocusApp(mountPoint, appName);
    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(appName);
    VARTRACE(deviceID);
    //return LocalSPM::GetOutputWrapper().OnFocusAppChanges(deviceID, appName);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    LocalSPM::GetCustomControl().ParameterON_FOCUSAPP_CHANGED(OUT parameterString, IN size, IN deviceID, IN appName);
    LocalSPM::GetCustomControl().SendEventByName("ON_FOCUSAPP_CHANGED", IN parameterString);
    return MP_NO_ERROR;
}

tResult iPodControl::SendFinishedStatusToPlayerManager(
        const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    if(!iAPIsTrackFinished(mountPoint)) {
        iAPSetTrackFinished(mountPoint, true); //prevent multiple NEXTs
        SendPlaybackStatusToPlayerManager(mountPoint, PE_PBS_FINISHEDSTATE);
        return MP_NO_ERROR;
    }
    return MP_NO_ERROR;
}

tResult iPodControl::SendPlaytimeStatusToPlayerManager(const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY;
    VARTRACE(mountPoint);

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    const tPEHandle handle = iAPGetPEHandle(mountPoint);
    VARTRACE(handle);

    tPETimeInfoStruct timeInfoStruct;
    InitPETimeInfoStruct(timeInfoStruct);
    timeInfoStruct.position.ms = iAPGetElapsedPlaytime(mountPoint);
    timeInfoStruct.duration.ms = iAPGetTotalPlaytime(mountPoint);

    VARTRACE(timeInfoStruct.position.ms);
    VARTRACE(timeInfoStruct.duration.ms);

    /* Marshal struct of timeInfo into a string */
    tPETimeInfo timeInfo;
    size_t size = sizeof(timeInfo);
    SMF::Marshal(OUT timeInfo,
            IN size-1,
            IN DOUBLE_MARSHAL_SEPARATOR,
            IN tPEBytes_format tPEPercentage_format tPEMilliseconds_format tPEBytes_format tPEMilliseconds_format,
            IN timeInfoStruct.position.bytes,
            IN timeInfoStruct.position.pct,
            IN timeInfoStruct.position.ms,
            IN timeInfoStruct.duration.bytes,
            IN timeInfoStruct.duration.ms);


    /* Send PLAYTIME_STATUS message to PlayerManger to update playtime */
    char messageString[64];
    strncpy_r(OUT messageString, IN "PlayerManagerSM::PLAYTIME_STATUS", IN sizeof(messageString));
    tAllParameters parameterString;
    size = sizeof(parameterString);

    tResult ret = LocalSPM::GetPlayerManager().ParameterPLAYTIME_STATUS(OUT parameterString, IN size, IN handle, IN deviceID, IN timeInfo);
    if (MP_NO_ERROR != ret){
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }

    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
    if (MP_NO_ERROR != ret){
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
    }
    return MP_NO_ERROR;
}

tResult iPodControl::SendInitDevice(const tMountPoint mountPoint, const tInitDeviceProtocol protocol) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    //VARTRACE(mountPoint);
    //VARTRACE(protocol);
    tResult ret = MP_NO_ERROR;

    /* Status response: Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterINIT_DEVICE(OUT parameterString, IN size, IN mountPoint, IN protocol);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendInitDevice() - Error while preparing parameter string"));
    }
    return SendEvent(INIT_DEVICE, IN parameterString);
}

tResult iPodControl::SendDeviceInitialized(const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    ETG_TRACE_USR3(("iPodControl::SendDeviceInitialized"));
    tResult ret = MP_NO_ERROR;

    /* Send response DEVICE_INITIALIZED message to Indexer */
    tDeviceInfo deviceInfo;
    iAPGetDeviceInfo(mountPoint, deviceInfo);
    VARTRACE(deviceInfo.mountPoint);
    VARTRACE(deviceInfo.deviceName);
    VARTRACE(deviceInfo.deviceID);
    VARTRACE(deviceInfo.connectionState);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ret = LocalSPM::GetIndexer().ParameterDEVICE_INITIALIZED(
            OUT parameterString, IN size, IN deviceInfo.deviceName, IN deviceInfo.deviceID,
            IN deviceInfo.connectionState);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendDeviceInitialized() - Error while preparing parameter string"));
        parameterString[0] = '\0';
    }
    return LocalSPM::GetIndexer().SendEventByName("DEVICE_INITIALIZED", IN parameterString);
}

tResult iPodControl::SendPauseOnTrackEnd(const tMountPoint mountPoint) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    ETG_TRACE_USR3(("iPodControl::SendPauseOnTrackEnd('%s')", mountPoint));
    tResult ret = MP_NO_ERROR;

    //Synchronous call to self state machine
    tDeviceType deviceType = DTY_IPOD;
    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    tURL URL = { 0 };

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterPAUSE( OUT parameterString, IN size, IN deviceType,
            IN deviceID, IN URL, IN mountPoint);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendPauseOnTrackEnd() - Error while preparing parameter string"));
    }
    return SendEventAnswerByName("PAUSE", "PlayerManagerSM::PLAYBACK_STATUS_RESPONSE", parameterString);
}

tResult iPodControl::SendAlbumArtAnswer(const tAlbumArtObjectPtr pAlbumArtObjectPtr) {
    //THIS METHOD IS THREAD-SAFE
    ENTRY_INTERNAL;
    ETG_TRACE_USR3(("iPodControl::SendAlbumArtAnswer()"));
    // thm4hi: not working on 64 bit: VARTRACE((int)pAlbumArtObjectPtr);
    tResult ret = MP_NO_ERROR;

    /* Status response: Send answer to waiting state machine */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterGET_ALBUM_ART_ANSWER(parameterString, size, IN pAlbumArtObjectPtr);
    ret = SendEvent(GET_ALBUM_ART_ANSWER, IN parameterString);
    if (MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("iPodControl::SendAlbumArtAnswer() - Error while sending internal event via SMF"));
        if (pAlbumArtObjectPtr) {
            if (pAlbumArtObjectPtr->imageData) {
                delete[] pAlbumArtObjectPtr->imageData;
            }
            delete pAlbumArtObjectPtr;
            ETG_TRACE_USR3(("AlbumArt buffer cleared"));
        }
    }
    return ret;
} //lint !e429 deleted by DataProvider

tResult iPodControl::SendMethodResult(const tReturnValue returnValue)
{
    ENTRY
    tResult ret = MP_NO_ERROR;

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterMETHOD_RETURN(OUT parameterString, IN size, IN returnValue);
    ret = SendEvent(METHOD_RETURN, IN parameterString);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }
    return ret;
}

tResult iPodControl::CreateTagFile(tFilename &tagFile, const tTagTransfer transferTagParam)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    tResult ret = MP_NO_ERROR;

    /* store data onto temp file */
    GenerateTempFileName(tagFile, "TAG");
    FILE *fpTag = fopen(tagFile, "wb");
    if (!fpTag) {
        ETG_TRACE_ERR(("iPodControl::CreateTagFile() fopen failed errno=%d '%s'", errno, strerror(errno)));
        ret =  MP_ERR_IPOD_FILE_ACCESS;
    } else {
        fprintf(fpTag, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
        fprintf(fpTag, "<!DOCTYPE plist PUBLIC \"-//Apple Computer//DTD PLIST 1.0//EN\"\n");
        fprintf(fpTag, "\"http://www.apple.com/DTDs/PropertyList-1.0.dtd\">\n");
        fprintf(fpTag, "<plist version=\"1.0\">\n");
        fprintf(fpTag, "<dict>\n");
        fprintf(fpTag, "    <key>MajorVersion</key>\n");
        fprintf(fpTag, "    <integer>1</integer>\n");
        fprintf(fpTag, "    <key>MinorVersion</key>\n");
        fprintf(fpTag, "    <integer>1</integer>\n");
        fprintf(fpTag, "    <key>ManufacturerID</key>\n");
        fprintf(fpTag, "    <integer>109155</integer>\n");
        fprintf(fpTag, "    <key>ManufacturerName</key>\n");
        fprintf(fpTag, "    <string>Robert Bosch CarMultimedia GmbH</string>\n");
        fprintf(fpTag, "    <key>DeviceName</key>\n");
        fprintf(fpTag, "    <string>NextGen</string>\n");
        fprintf(fpTag, "    <key>MarkedTracks</key>\n");
        fprintf(fpTag, "    <array>\n");
        fprintf(fpTag, "        <dict>\n");

        //ambiguous tag (mandatory)
        fprintf(fpTag, "            <key>AmbiguousTag</key>\n");
        fprintf(fpTag, "            <integer>%d</integer>\n", (int)transferTagParam.ambiguousTag);

        //button pressed tag (mandatory)
        fprintf(fpTag, "            <key>ButtonPressed</key>\n");
        fprintf(fpTag, "            <integer>%d</integer>\n", (int)transferTagParam.buttonPressed);

        //track name tag
        if (transferTagParam.tagTrackName[0] != 0) {
            fprintf(fpTag, "            <key>Name</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagTrackName);
        }
        //artist name tag
        if (transferTagParam.tagArtistName[0] != 0) {
            fprintf(fpTag, "            <key>Artist</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagArtistName);
        }
        //album name tag
        if (transferTagParam.tagAlbumName[0] != 0) {
            fprintf(fpTag, "            <key>Album</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagAlbumName);
        }
        //song id tag
        if (transferTagParam.tagSongID > 0) {
            fprintf(fpTag, "            <key>iTunesSongID</key>\n");
            fprintf(fpTag, "            <integer>%d</integer>\n", (int)transferTagParam.tagSongID);
        }
        //store id tag
        if (transferTagParam.tagStorefrontID > 0) {
            fprintf(fpTag, "            <key>iTunesStorefrontID</key>\n");
            fprintf(fpTag, "            <integer>%d</integer>\n", (int) transferTagParam.tagStorefrontID);
        }
        //channel number tag
        if (transferTagParam.tagChannelNumber[0] != 0) {
            fprintf(fpTag, "            <key>StationFrequency</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagChannelNumber);
        }
        //channel name tag
        if (transferTagParam.tagChannelName[0] != 0) {
            fprintf(fpTag, "            <key>StationCallLetters</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagChannelName);
        }
        //podcast feed tag
        if (transferTagParam.podcastFeedURL[0] != 0) {
            fprintf(fpTag, "            <key>PodcastFeedURL</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.podcastFeedURL);
        }
        //station url tag
        if (transferTagParam.stationURL[0] != 0) {
            fprintf(fpTag, "            <key>StationURL</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.stationURL);
        }
        //genre name tag
        if (transferTagParam.tagGenreName[0] != 0) {
            fprintf(fpTag, "            <key>Genre</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagGenreName);
        }
        //time stamp tag
        if (transferTagParam.tagTimeStamp[0] != 0) {
            fprintf(fpTag, "            <key>TimeStamp</key>\n");
            fprintf(fpTag, "            <date>%s</date>\n", transferTagParam.tagTimeStamp);
        }
        //program number tag
        //if (transferTagParam.tagProgramNumber >= 0) {
            fprintf(fpTag, "            <key>ProgramNumber</key>\n");
            fprintf(fpTag, "            <integer>%d</integer>\n", (int)transferTagParam.tagProgramNumber);
        //}
        //program number tag
        if (transferTagParam.tagAffiliateID[0] != 0) {
            fprintf(fpTag, "            <key>iTunesAffiliateID</key>\n");
            fprintf(fpTag, "            <string>%s</string>\n", transferTagParam.tagAffiliateID);
        }
        //unknown data tag
        if (transferTagParam.unknownDataSize > 0 && transferTagParam.unknownData[0] != 0) {
            //Example:
            //<data>MDcwNDEyMzQwMTAyMTEwNDA4MTJhYjU272g=</data>

            char* base64EncodedData = g_base64_encode (transferTagParam.unknownData,transferTagParam.unknownDataSize);
            fprintf(fpTag, "            <key>UnknownData</key>\n");
            if(base64EncodedData != NULL)
            {
                fprintf(fpTag, "            <data>%s</data>\n", base64EncodedData);
                g_free(base64EncodedData);
            }
            else
            {
                fprintf(fpTag, "            <data></data>\n");
            }
        }

        fprintf(fpTag, "        </dict>\n");
        fprintf(fpTag, "    </array>\n");
        fprintf(fpTag, "</dict>\n");
        fprintf(fpTag, "</plist>\n");

        fclose(fpTag);
    }
    return ret;
}

tResult iPodControl::OnSampleRateChange(const tMountPoint mountPoint, const me::samplerate_i newSampleRate)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(newSampleRate);
    tResult ret = MP_NO_ERROR;

    me::samplerate_i currentSampleRate = iAPGetSampleRate(mountPoint);
    if (newSampleRate != currentSampleRate) {
        iAPSetSampleRate(mountPoint, newSampleRate);
        const tBoolean isActive = !strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint));
        const tPEPlaybackState iPodStatus = iAPGetCurrentPlaybackState(mountPoint);
        if (!isActive || m_ActivePBSampleRate == samplerate_i_none /*|| iPodStatus == PE_PBS_STOPPEDSTATE || iPodStatus == PE_PBS_ERRORSTATE*/) {
            VARTRACE(isActive);
            VARTRACE(m_ActivePBSampleRate);
            VARTRACE(iPodStatus);
            ETG_TRACE_USR3(("No playing iDevice, ignore sample rate notification"));
        } else {
#if 0
            //Synchronous call to self state machine
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            ParameterSTART_STREAMING(OUT parameterString, IN size, IN mountPoint);
            ret = SendEventAnswerByName("START_STREAMING", "PlayerManagerSM::PLAYBACK_STATUS_RESPONSE", parameterString);
            if (ret != MP_NO_ERROR) {
                ETG_TRACE_ERR(("SendEventAnswerByName failed"));
            }
#else
            m_ActivePBSampleRate = newSampleRate;
            if(false == LocalSPM::GetDataProvider().UseMediaEngine() ) { //pip2hi

                //static streaming configuration parameter for iPod
                const tEncoding encoding = "audio"; //Not valid for ipod
                const tBitRate bitRate = 128 << 10; //Not valid for ipod
                const me::channels_i channels = 2; //Number of channels of the decoded audio
                const me::samplewidth_i sampleWidth = 16; //Width of the sample in bits
                const me::sampledepth_i sampleDepth = 16; //Depth of the sample in bits
                const me::signedness_e sampleSignedness = SIGNEDNESS_SIGNED; //0/1 in case of sample unsigned/signed.
                const me::endianess_e sampleEndianness = ENDIANESS_BE; //0/1 in case of sample little/big endian

                /* Send PlayFromDevice to PlayerEngine via DBUS */
                ret = LocalSPM::GetIPCProvider().PlayFromDeviceSlot(IN this,
                        IN encoding, IN bitRate, IN newSampleRate, IN channels,
                            IN sampleWidth, IN sampleDepth, IN sampleSignedness,
                        IN sampleEndianness);
                if (ret != MP_NO_ERROR) {
                    ETG_TRACE_ERR(("PlayFromDeviceSlot() failed"));
                }
                usleep(150); //WORKAROUND: give time to PE to restart pipeline with new sample rate
            } else {

                tDeviceInfo deviceInfo;
                iAPGetDeviceInfo(mountPoint, deviceInfo);

                tURL url;
                snprintf(url, sizeof(tURL), "%s.pcm", deviceInfo.accessoryName);

                const size_t size = sizeof(tAllParameters);
                char *parameterString = new char[size]; //deleted by DoChangeSampleRate
                SMF::Marshal(parameterString, size-1, DOUBLE_MARSHAL_SEPARATOR, "tiiltl", url, 1 /* speed */, -1 /*position*/, HANDLE_NONE, m_AudioOutputDevice, newSampleRate);

                //call MediaEngine PLAY from a dedicated SM
                //ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::PLAY", parameterString, "iPodControl::PLAYBACK_STATUS");

                LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_CHANGE_SAMPLERATE, IN parameterString, this);
            }
#endif
        }
    } else {
        ETG_TRACE_USR3(("Keep same sample rate %d", newSampleRate));
    }
    return ret;
}

tResult iPodControl::DoChangeSampleRate(char *parameterString)
{
    ENTRY;
    VARTRACE(parameterString)
    tResult ret = MP_NO_ERROR;

    if(LocalSPM::GetDataProvider().UseMediaEngine() ) {
        iPodControlRR iPodControl_rr;
        char msgToSendString[64];
        strncpy_r(msgToSendString, "MediaEngineSM::PLAY", sizeof(msgToSendString));

        ret = iPodControl_rr.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);
        VARTRACE(ret);
    }
    if (parameterString) {
        delete[] parameterString; //new by caller
    }
    return ret;
}

void iPodControl::StartFingerprintTimer(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    /* Set the timeout value */
    long timeoutValue = LocalSPM::GetDataProvider().iPodControlIAP2FingerprintTimeoutMS();

    //store last fingerprint parameters
    m_LastFingerprintDeviceID = deviceID;

    /* Start the timer, single shot */
    m_FingerprintTimer.StartTimer(OUT m_FingerprintTimerID, IN timeoutValue, 0L, &LocalSPM::GetIPODControl(), &FingerprintTimerCallBack, NULL);
}

void iPodControl::StopFingerprintTimer(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    if(deviceID != DEVICE_ID_NOT_SET && m_LastFingerprintDeviceID != deviceID) {
        //TODO: support multiple timer
        return;
    }

    /* Cancel the timer */
    if (m_FingerprintTimerID) {
        m_FingerprintTimer.CancelTimer(IN m_FingerprintTimerID);
        m_FingerprintTimerID = 0;
    }
}

bool iPodControl::FingerprintTimerCallBack(timer_t /*timerID*/ , void* instance, const void * /*userData*/)
{
    ENTRY;
    iPodControl *self = (iPodControl*)instance;
    if(self) {
        //Timeout for CBIAP2_MediaLibraryUpdate, no update received
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        self->ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN self->GetLastFingerprintDeviceID(), IN USRDATA_FINGERPRINTTIMEOUT, IN "");
        self->SendEvent(self->SIGNAL_USERDATA, IN parameterString);
    }
    return MP_NO_ERROR;
}

void iPodControl::TrackEndDetection(const tMountPoint mountPoint, const tPEPlaybackState iPodPBStatus, const tPlaytime lastElapsedPlaytime)
{
    ENTRY;
    const tPlaytime elapsedPlaytime = iAPGetElapsedPlaytime(mountPoint);
    const tPlaytime totalPlaytime = iAPGetTotalPlaytime(mountPoint);
    VARTRACE(elapsedPlaytime);
    VARTRACE(lastElapsedPlaytime);
    VARTRACE(totalPlaytime);

    //calculate remaining playtime
    const int lastStep = (int) elapsedPlaytime - (int) lastElapsedPlaytime;
    const int remTime = (int) totalPlaytime - (int) elapsedPlaytime;
    VARTRACE(lastStep);
    VARTRACE(remTime);

    if (totalPlaytime == 0 || totalPlaytime == PLAYTIME_NONE || remTime < 0) {
        ETG_TRACE_USR3(("INVALID PLAYTIME PARAMETERS"));
    } else if (remTime >= 0 && (remTime < LocalSPM::GetDataProvider().iPodControlTrackEndDetectionMS()
            ||(iPodPBStatus == PE_PBS_FASTFORWARDSTATE && lastStep > 0 && lastStep > remTime))) { //forcase of new playtime update during FFWD
        ETG_TRACE_USR3(("TRACK END DETECTION in %d ms", remTime));

        //SUZUKI-16783
        //if(!iAP2_IsSet(mountPoint) || iAPGetNowPlayingTrackCount(mountPoint) <= 1) {
            //send pause signal to own state machine, synch call
            SendPauseOnTrackEnd(mountPoint);
        //}
        SendFinishedStatusToPlayerManager(mountPoint);
    }
}

tResult iPodControl::SendDiPOActiveDevice(const tDeviceID deviceID, const tDiPOActive diPOActive, const tUserContext userContext)
{
    //DEPRECATED METHODE
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOActive);
    tResult ret = MP_NO_ERROR;
    tDiPOResponse diPOResponse = DIPO_ERROR_UNKNOWN;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    //update device table
    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID); //OK to call DBManagrer here, it is not the SM thread
    if(MP_NO_ERROR != ret) {
        ETG_TRACE_ERR(("LocalSPM::GetDBManager().GetDeviceInfo() failed"));
    } else {
        VARTRACE(deviceInfo.diPOCaps);
        VARTRACE(deviceInfo.diPOActive);
        VARTRACE(deviceInfo.diPOVersion);

        //  plausibility checks
        if(!IsAppleDevice(deviceInfo.deviceType)) {
           ETG_TRACE_ERR(("DiPOActive: Device is not an apple device"));
           diPOResponse = DIPO_ERROR_INVALIDHANLDE;
        } else if(!iAP_IsDeviceHandleAvail(mountPoint)) {
            ETG_TRACE_ERR(("DiPOActive: Device is not connected"));
            diPOResponse = DIPO_ERROR_INVALIDHANLDE;
        } else if(!iAP2_IsSet(mountPoint) && diPOActive) {
            ETG_TRACE_ERR(("DiPOActive: Device is not an IAP2 device"));
            diPOResponse = DIPO_ERROR_NOTSUPPORTED;
        } else if(!iAP2_IsHostMode(mountPoint) && diPOActive) {
            ETG_TRACE_ERR(("DiPOActive: Device does not run in host mode"));
            diPOResponse = DIPO_ERROR_NOTSUPPORTED;
        } else if(!IsDiPOCapability_CarPlay(deviceInfo.diPOCaps) && diPOActive) {
            ETG_TRACE_ERR(("DiPOActive: Device is not marked as CarPlay capable"));
            diPOResponse = DIPO_ERROR_NOTSUPPORTED;
        } else {
            if(deviceInfo.diPOActive != diPOActive) {
                deviceInfo.diPOActive = diPOActive;
                ret = LocalSPM::GetDBManager().SetDiPOParameters(IN deviceID, IN deviceInfo.diPOCaps, IN deviceInfo.diPOActive, IN deviceInfo.diPOVersion);
                if(MP_NO_ERROR != ret) {
                    ETG_TRACE_ERR(("LocalSPM::GetDBManager().SetDiPOParameters() failed"));
                } else {
                    ETG_TRACE_USR3(("DiPOActive: Device %s DiPO session", diPOActive ? "entered" : "closed"));
                }
            }
            diPOResponse = DIPO_OK;
        }
    }

    ret = LocalSPM::GetOutputWrapper().SendDiPOActiveDeviceAnswer(diPOResponse, userContext);
    return ret;
}

tResult iPodControl::SendDiPOStartIAP2OverCarplayWiFi(const tMountPoint mountPoint)
{
    ENTRY

    tResult ret = MP_NO_ERROR;

    const tUserDataType userDataType = USRDATA_START_IAP2_CARPLAY_WIRELESS;
    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    tUserData userData = {0};

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
    } else {
        ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    return ret;

}

tResult iPodControl::PrepareIAP2OverCarplayWiFi(const tMountPoint mountPoint)
{
    ENTRY
    VARTRACE(mountPoint);

    tResult ret = MP_NO_ERROR;
    tInitDeviceProtocol protocol = IDP_IAP2_OVER_WIRELESS_CARPLAY;
    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(m_ActivePBMountPoint);

    if(!strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint))) {
        //pause current playback
        StartStop(DTY_IPOD, deviceID, "", mountPoint);
        m_ActivePBMountPoint[0] = '\0'; //prevent following PE/ME actions [GMMY17-11719]
    }
//    RemoveDeviceConnection(mountPoint, deviceID); //Commented as iAP2-BT can co-exist with WiFI.
    SendInitDevice(mountPoint, protocol);
    return ret;

}

tResult iPodControl::SendDiPORoleSwitchRequest(const tDeviceID deviceID, const tDiPORoleStatus diPORoleStatus, const tUserContext userContext, const tAppInfo appInfo)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPORoleStatus);
    tResult ret = MP_NO_ERROR;
    tBoolean runRoleSwitch = false;
    tDiPOResponse diPOResponse = DIPO_ERROR_UNKNOWN;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);
    if(DIPO_CARLIFE_NATIVE_TRANSPORT_MODE == diPORoleStatus)
    {
        bool added = false;
        iAPAddAppControlInfoFromSPI(mountPoint,appInfo.ProtocolName,appInfo.BundleID,appInfo.AppName,added);
    }

    if(DIPO_CARPLAY_MODE == diPORoleStatus && !LocalSPM::GetDataProvider().iPodControlWiredCarPlayEnabled())
    {
        ETG_TRACE_ERR(("DiPORoleSwitch: CarPlay feature is not supported or blocked"));
        diPOResponse = DIPO_ERROR_NOTSUPPORTED;
    }
    else
    {
        //update device table
        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID); //OK to call DBManagrer here, it is not the SM thread
        if(MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("LocalSPM::GetDBManager().GetDeviceInfo() failed"));
        } else {
            VARTRACE(deviceInfo.deviceType);
            VARTRACE(deviceInfo.connectionType);

            const bool isHostMode = iAP2_IsHostMode(mountPoint);
            const bool isCarPlayMode = iAP2_IsCarPlayMode(mountPoint);
            const bool isNativeTransportMode = iAP2_IsNativeTransportMode(mountPoint);
            VARTRACE(isHostMode);
            VARTRACE(isCarPlayMode);
            VARTRACE(isNativeTransportMode);

            //  plausibility checks
            if(!IsAppleDevice(deviceInfo.deviceType)) {
               ETG_TRACE_ERR(("DiPORoleSwitch: Device is not an apple device"));
               diPOResponse = DIPO_ERROR_INVALIDHANLDE;
            } else if(deviceInfo.connectionType != DCT_USB) {
                ETG_TRACE_ERR(("DiPORoleSwitch: Device is not connected via USB"));
                diPOResponse = DIPO_ERROR_INVALIDHANLDE;
            } else if(!iAP_IsDeviceHandleAvail(mountPoint)) {
                ETG_TRACE_ERR(("DiPORoleSwitch: Device is not connected"));
                diPOResponse = DIPO_ERROR_INVALIDHANLDE;
            } else if(!iAP2_IsSet(mountPoint)) {
                ETG_TRACE_ERR(("DiPORoleSwitch: Device is not an IAP2 device"));
                diPOResponse = DIPO_ERROR_NOTSUPPORTED;
            } else if(!isHostMode && diPORoleStatus == DIPO_CLIENT_MODE) {
                ETG_TRACE_USR3(("DiPORoleSwitch: Device alread in device mode"));
                diPOResponse = DIPO_OK;
            } else if(diPORoleStatus == DIPO_CARPLAY_MODE && isHostMode && isCarPlayMode) {
                ETG_TRACE_USR3(("DiPORoleSwitch: Device is already in CarPlay mode"));
                diPOResponse = DIPO_OK;
            } else if(diPORoleStatus == DIPO_NATIVE_TRANSPORT_MODE && isHostMode && !isCarPlayMode && isNativeTransportMode) {
                ETG_TRACE_USR3(("DiPORoleSwitch: Device is already in NativeTransport mode"));
                diPOResponse = DIPO_OK;
            } else if(diPORoleStatus == DIPO_CARPLAY_NATIVE_TRANSPORT_MODE && isHostMode && isCarPlayMode && isNativeTransportMode) {
                ETG_TRACE_USR3(("DiPORoleSwitch: Device is already in CarPlay and NativeTransport mode"));
                diPOResponse = DIPO_OK;
            } else if(!isHostMode && !iAP2_IsHostModePossible(mountPoint)){
                ETG_TRACE_ERR(("DiPORoleSwitch: Host mode is not possible by HW/configuration"));
                diPOResponse = DIPO_ERROR_NOTSUPPORTED;
            } else if(diPORoleStatus == DIPO_CARPLAY_MODE && !iAP2_IsCarPlayPossible(mountPoint)){
                ETG_TRACE_ERR(("DiPORoleSwitch: CarPlay mode is not possible by device"));
                diPOResponse = DIPO_ERROR_NOTSUPPORTED;
            } else if(diPORoleStatus == DIPO_NATIVE_TRANSPORT_MODE && !iAP2_IsNativeTransportAppConfigured(mountPoint)){
                ETG_TRACE_ERR(("DiPORoleSwitch: Native transport mode is not possible by configuration"));
                diPOResponse = DIPO_ERROR_NOTSUPPORTED;
            } else if(diPORoleStatus == DIPO_CARPLAY_NATIVE_TRANSPORT_MODE && !(iAP2_IsCarPlayPossible(mountPoint) && iAP2_IsNativeTransportAppConfigured(mountPoint))){
                ETG_TRACE_ERR(("DiPORoleSwitch: CarPlay and Native transport mode is not possible by device/configuration"));
                diPOResponse = DIPO_ERROR_NOTSUPPORTED;
            } else {
                runRoleSwitch = true;
            }
        }
    }

    if(runRoleSwitch) {
        //dispatch from service thread
        const size_t size = sizeof(tAllParameters);
        char *parameterString = new char[size]; //deleted by DoRoleSwitch
        SMF::Marshal(parameterString, size-1, tMountPoint_format tDiPORoleStatus_format tSourceAppID_format tRegisterID_format tCmdCounter_format tFunctionID_format tServiceID_format , mountPoint, diPORoleStatus, userContext.sourceAppID, userContext.registerID, userContext.cmdCounter, userContext.functionID, userContext.serviceID);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_ROLE_SWITCH, IN parameterString, this);
    } else { //lint !e429 deleted by DoRoleSwitch
        ret = LocalSPM::GetOutputWrapper().SendDiPORoleSwitchRequestAnswer(diPOResponse, userContext);
    }
    return ret;
}

tResult iPodControl::DoRoleSwitch(char *parameterStr)
{
    ENTRY;
    ETG_TRACE_USR3(("%s", __PRETTY_FUNCTION__));
    VARTRACE(parameterStr)
    tResult ret = MP_NO_ERROR;
    tMountPoint mountPoint = {0};
    tDiPORoleStatus diPORoleStatus = tDiPORoleStatus_init;
    tUserContext userContext;

    //unpack and forward mountpoint
    SMF::UnMarshal(parameterStr, tMountPoint_format tDiPORoleStatus_format tSourceAppID_format tRegisterID_format tCmdCounter_format tFunctionID_format tServiceID_format , mountPoint, &diPORoleStatus, &userContext.sourceAppID, &userContext.registerID, &userContext.cmdCounter, &userContext.functionID, &userContext.serviceID);
    if (parameterStr) {
        delete[] parameterStr; //new by SendDiPORoleSwitchRequest
    }

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterROLE_SWITCH(OUT parameterString, IN size, IN mountPoint, IN diPORoleStatus);

    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::ROLE_SWITCH", sizeof(msgToSendString) - 1);
    iPodControlRR appControlRR;
    ret = appControlRR.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);

    VARTRACE(ret);
    VARTRACE(diPORoleStatus);

    const tBoolean deviceAvail = iAP_IsDeviceHandleAvail(mountPoint);
    const tBoolean hostMode = iAP2_IsHostMode(mountPoint);
    const tBoolean carPlayMode = iAP2_IsCarPlayMode(mountPoint);
    const tBoolean nativeTransport = iAP2_IsNativeTransportMode(mountPoint);

    VARTRACE(deviceAvail);
    VARTRACE(hostMode);
    VARTRACE(carPlayMode);
    VARTRACE(nativeTransport);

    //validate result status
    tDiPOResponse diPOResponse = DIPO_ERROR_UNKNOWN;
    if(deviceAvail &&
      ((diPORoleStatus == DIPO_CLIENT_MODE && !hostMode) ||
       (diPORoleStatus == DIPO_CARPLAY_MODE && carPlayMode && !nativeTransport) ||
       (diPORoleStatus == DIPO_NATIVE_TRANSPORT_MODE && !carPlayMode && nativeTransport) ||
       (diPORoleStatus == DIPO_CARPLAY_NATIVE_TRANSPORT_MODE && carPlayMode && nativeTransport)) ||
       (diPORoleStatus == DIPO_CARLIFE_NATIVE_TRANSPORT_MODE)) {
        diPOResponse = DIPO_OK;
        ETG_TRACE_USR3(("RoleSwitch Done"));
    } else {
        ETG_TRACE_ERR(("RoleSwitch ERROR"));
    }
    VARTRACE(diPOResponse);

    ret = LocalSPM::GetOutputWrapper().SendDiPORoleSwitchRequestAnswer(diPOResponse, userContext);
    return ret;
}

tResult iPodControl::PrepareRoleSwitch(const tMountPoint mountPoint, const tDiPORoleStatus diPORoleStatus)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(diPORoleStatus);

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    m_RoleSwitchDeviceID = deviceID; //store deviceID of the current role switching device, see RoleSwitchDeviceRemoved

    const bool isHostMode = iAP2_IsHostMode(mountPoint);
    const bool isNativeTransportAppConfigured = iAP2_IsNativeTransportAppConfigured(mountPoint);
    const bool isHostModePossible = iAP2_GetStoredHostModePossibility(mountPoint);
    const bool isCarPlayPossible = iAP2_IsCarPlayPossible(mountPoint);

    VARTRACE(isHostMode);
    VARTRACE(isHostModePossible);
    VARTRACE(isCarPlayPossible);
    VARTRACE(isNativeTransportAppConfigured);

    tInitDeviceProtocol protocol = IDP_IAP2;
    if(isHostModePossible) {
        switch(diPORoleStatus) {
            case DIPO_CARPLAY_MODE:
                if(isCarPlayPossible) {
                    protocol = IDP_IAP2CARPLAY;
                }
                break;
            case DIPO_NATIVE_TRANSPORT_MODE:
                if(isNativeTransportAppConfigured) {
                    protocol = IDP_IAP2NATIVE_TRANSPORT;
                }
                break;
            case DIPO_CARPLAY_NATIVE_TRANSPORT_MODE:
                if(isCarPlayPossible && isNativeTransportAppConfigured) {
                    protocol = IDP_IAP2CARPLAY_NATIVE_TRANSPORT;
                }
                break;
            case DIPO_CARLIFE_NATIVE_TRANSPORT_MODE:
                protocol = IDP_IAP2CARLIFE_NATIVE_TRANSPORT;
                break;
            default:
                break;
        }
    }
    VARTRACE(protocol);
    VARTRACE(m_ActivePBMountPoint);
    if(!strncmp(m_ActivePBMountPoint, mountPoint, strlen_r(mountPoint))) {
        //pause current playback
        StartStop(DTY_IPOD, deviceID, "", mountPoint);
        m_ActivePBMountPoint[0] = '\0'; //prevent following PE/ME actions [GMMY17-11719]
    }

    //remove current connection, including iAP2SwitchToDeviceMode
    if(isHostMode) {
        RemoveDeviceConnectionForced(mountPoint, deviceID);
        //SwitchToDeviceMode was called
        //wait until OTG comes back to continue with next init (device mode)
        tBoolean otg_available = iAP2HasOTGPort(mountPoint);
        VARTRACE(otg_available);
        //5 seconds retry max
        for(int i = 0; i < 5 && !otg_available; i++) {    /* reducing the delay to max 5 seconds for the ticket NCG3D-82012 */
            usleep(1000000);
            otg_available = iAP2HasOTGPort(mountPoint);
            VARTRACE(otg_available);
        }
    } else {
        RemoveDeviceConnection(mountPoint, deviceID);
    }

    //send init event even on previous failure to create a valid connection (default simple device mode)
    SendInitDevice(mountPoint, protocol);
    return MP_NO_ERROR;
}

tResult iPodControl::RoleSwitchAnswer()
{
    ENTRY;
    m_RoleSwitchDeviceID = DEVICE_ID_NOT_SET; //clear role switch deviceID

    //release the calling outer SM
    SendAnswer(IN NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::ModeDisconnect(const tMountPoint mountPoint, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(protocol);

    Disconnect(mountPoint, protocol);

    //release the calling outer SM
    SendEvent(ROLE_SWITCH_ANSWER, IN (char*) NULL);

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::RoleSwitchDeviceRemoved(const tMountPoint mountPoint, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(m_RoleSwitchDeviceID);
    tMountPoint USBSerialNumber = {0};
    iAPGetMountPointFromDeviceID(m_RoleSwitchDeviceID, USBSerialNumber);
    VARTRACE(USBSerialNumber);

    if(m_RoleSwitchDeviceID != deviceID) {
        //process event for non role switch device
        RemoveDeviceConnectionForced(mountPoint, deviceID);
    } //else ignore device removed event on host role switch

    return MP_NO_ERROR;
}

tResult iPodControl::RoleSwitchDeviceRemovedForced(const tMountPoint mountPoint, const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(m_RoleSwitchDeviceID);

    RemoveDeviceConnectionForced(mountPoint, deviceID);

    if(m_RoleSwitchDeviceID != deviceID && LocalSPM::GetDataProvider().iPodCommunicationError()) {
        tIpodCommError iPodCommError = IPOD_COMM_ERROR;
        LocalSPM::GetOutputWrapper().UpdateDipoCommunicationError(m_RoleSwitchDeviceID,iPodCommError, mountPoint);
    }

    return MP_NO_ERROR;
}

tResult iPodControl::OnModeInitialized(const tMountPoint mountPoint,
        const tConnectionState connectionState, const tInitDeviceProtocol protocol)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(connectionState);
    VARTRACE(protocol);

    tResult ret = MP_NO_ERROR;

    if(IsProtocol_IAP1(protocol)) {
        ret = iAP1_OnDeviceInitialized(mountPoint, connectionState);
    } else if(IsProtocol_IAP2(protocol)) {
        ret = iAP2_OnDeviceInitialized(mountPoint, connectionState);
    } else {
        //should never reach here
        ETG_TRACE_ERR(("NO IAP SUPPORTED AT ALL - PLEASE CHECK MEDIAPLAYER CONFIGURATION"));
        ret = MP_ERR_IPOD_INIT;
    }

    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iPodControl::OnModeInitialized failed"));
    }

    if(ret == MP_NO_ERROR && CS_CONNECTED == connectionState) {
        //Apply shuffle and repeat settings from device
        if(LocalSPM::GetDataProvider().iPodControlUpdateShuffleAndRepeatFromDevice()) {
            SendPlaybackModeToPlayerManager(mountPoint);
            SendRepeatModeToPlayerManager(mountPoint);
        }
    }

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    tDiPOCaps diPOCaps = iAP2_GetDiPOCaps(mountPoint);
    VARTRACE(diPOCaps);

    //update diPOCaps
    SendDBChangeDiPOSettings(IN deviceID, IN diPOCaps, IN FALSE, IN "");

    if(IDP_IAP2 == protocol) {
        if(CS_UNSUPPORTED == connectionState) {
            if(LocalSPM::GetDataProvider().iPodControlSupportIAP1()) {
                //try IAP1 legacy with another loop in state "roleSwitch"
                SendInitDevice(mountPoint, IDP_IAP1);
                return MP_NO_ERROR;
            }
        }
    }

    //Update connection state
    tDeviceInfo deviceInfo;
    iAPGetDeviceInfo(mountPoint, deviceInfo);
    VARTRACE(deviceInfo.deviceName);
    VARTRACE(deviceInfo.connectionState);

    if(iAP2_IsHostMode(mountPoint) && deviceInfo.connectionState == CS_CONNECTED) {
        if(LocalSPM::GetDataProvider().iPodControlIAP2CarPlayModeEnabled()) {
            //CarPlay device set to CS_ATTACHED, via Indexer set to CS_CONNECTED automatically
            //NOTE: Direct set to CS_CONNECTED is invalid/blocked in DB schema
            ETG_TRACE_USR3(("Set DB device ATTACHED"));
            SendDBChangeConnectionState(IN deviceID, IN CS_ATTACHED);
        } else {
            //CarPlay device stays CS_DISCONNECTED in DB
            //update device name (SUZUKI-21091)
            SendDBChangeDeviceName(IN deviceID, IN deviceInfo.deviceName);
        }
    } else {
        //update Mediaplayer device status
        //Media device, connection state is read from internal member
        SendDeviceInitialized(mountPoint);
    }

    //release the calling outer SM
    SendEvent(ROLE_SWITCH_ANSWER, IN (char*) NULL);
    return MP_NO_ERROR;
}

tResult iPodControl::SendDiPOTransferGPSData(const tDeviceID deviceID, const tDiPOGPGGAData diPOGPGGAData,const tDiPOGPRMCData diPOGPRMCData, const tDiPOGPGSVData diPOGPGSVData, const tDiPOGPHDTData diPOGPHDTData, const tUserContext userContext)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOGPGGAData);
    VARTRACE(diPOGPRMCData);
    VARTRACE(diPOGPGSVData);
    VARTRACE(diPOGPHDTData);
    tResult ret = MP_NO_ERROR;

    const tUserDataType userDataType = USRDATA_GPS;
    tUserData userData = {0};
    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tDiPOGPGGAData_format tDiPOGPRMCData_format tDiPOGPGSVData_format tDiPOGPHDTData_format, diPOGPGGAData, diPOGPRMCData, diPOGPGSVData, diPOGPHDTData);

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
    } else {
        ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    tDiPOResponse diPOResponse = (ret == MP_NO_ERROR) ? DIPO_OK : DIPO_ERROR_UNKNOWN;
    ret = LocalSPM::GetOutputWrapper().SendDiPOTransferGPSDataAnswer(diPOResponse, userContext);
    return ret;
}

tResult iPodControl::SendDiPOTransferDRData(const tDeviceID deviceID, const tDiPOPASCDData diPOPASCDData, const tDiPOPAGCDData diPOPAGCDData, const tDiPOPAACDData diPOPAACDData, const tUserContext userContext)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOPASCDData);
    VARTRACE(diPOPAGCDData);
    VARTRACE(diPOPAACDData);
    tResult ret = MP_NO_ERROR;

    const tUserDataType userDataType = USRDATA_DR;
    tUserData userData = {0};
    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tDiPOPASCDData_format tDiPOPAGCDData_format tDiPOPAACDData_format, diPOPASCDData, diPOPAGCDData, diPOPAACDData);

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
    } else {
        ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }

    tDiPOResponse diPOResponse = (ret == MP_NO_ERROR) ? DIPO_OK : DIPO_ERROR_UNKNOWN;
    ret = LocalSPM::GetOutputWrapper().SendDiPOTransferDRDataAnswer(diPOResponse, userContext);
    return ret;
}

tResult iPodControl::OutsideTemperatureChanged()
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    int updatecount = 0;
    //update all host mode connections
    vector<string> mountPoints = iAPGetMountPoints();
    for(unsigned int i = 0; i < mountPoints.size(); i++) {
        if(iAP2_IsSet(mountPoints[i].c_str()) && iAP2_IsHostMode(mountPoints[i].c_str())) {

            updatecount++;
            tDeviceID deviceID = iAPGetDeviceID(mountPoints[i].c_str());
            VARTRACE(deviceID);

            const tUserDataType userDataType = USRDATA_VEHICLESTATUS;
            tUserData userData = {0}; //no data, taken from config

            /* Send SIGNAL_USERDATA message to own SM */
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);

            ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
            SendEvent(SIGNAL_USERDATA, IN parameterString);
        }
    }

    VARTRACE(updatecount);
    return ret;
}

tResult iPodControl::SendSignalUserDataEvent(const tDeviceID deviceID, const tUserDataType& userDataType, const tUserData& userData)
{
    tResult res = MP_NO_ERROR;
    tAllParameters parameterString;
    const size_t size = sizeof(tAllParameters);
    memset(parameterString,0,size);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    res = SendEvent(SIGNAL_USERDATA, IN parameterString);
    if(res != MP_NO_ERROR)
    {
        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(res)));
    }
    return res;
}

tResult iPodControl::SendDiPOInitiateCall(const tDeviceID deviceID, const tDiPOInitiateCall initiateCall)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(initiateCall);

    tResult ret = MP_NO_ERROR;

    if(DEVICE_ID_NOT_SET == deviceID) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid device ID"));
    }
    else {
        const tUserDataType userDataType = USRDATA_INITIATECALL;
        tUserData userData = {0}; //no data
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "itit", initiateCall.type, initiateCall.destinationID, initiateCall.service,initiateCall.addressBookID);

        ret = SendSignalUserDataEvent(deviceID,userDataType,userData);
    }
    return ret;
}

tResult iPodControl::SendDiPOAcceptCall(const tDeviceID deviceID, const tDiPOAcceptCall acceptCall)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(acceptCall);

    tResult ret = MP_NO_ERROR;

    if(DEVICE_ID_NOT_SET == deviceID) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid device ID"));
    }
    else {
        const tUserDataType userDataType = USRDATA_ACCEPTCALL;
        tUserData userData = {0}; //no data
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "it", acceptCall.acceptAction, acceptCall.callUUID);

        ret = SendSignalUserDataEvent(deviceID,userDataType,userData);
    }
    return ret;
}

tResult iPodControl::SendDiPOEndCall(const tDeviceID deviceID, const tDiPOEndCall endCall)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(endCall);

    tResult ret = MP_NO_ERROR;

    if(DEVICE_ID_NOT_SET == deviceID) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid device ID"));
    }
    else {
        const tUserDataType userDataType = USRDATA_ENDCALL;
        tUserData userData = {0}; //no data
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "it", endCall.endAction, endCall.callUUID);
        ret = SendSignalUserDataEvent(deviceID,userDataType,userData);
    }
    return ret;
}

tResult iPodControl::SendSwapCalls(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    if(DEVICE_ID_NOT_SET == deviceID)
    {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid device ID"));
    }
    else
    {
        const tUserDataType userDataType = USRDATA_SWAPCALLS;
        tUserData userData = {0}; //no data
        const size_t size = sizeof(tAllParameters);
        char *parameterString = new char[size];
        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_PROCESS_USER_DATA, IN parameterString, this);
    }
    return ret;
}

tResult iPodControl::SendStartAudio(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    if(deviceID == DEVICE_ID_NOT_SET) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid moutPoint"));
    } else {
        const tUserDataType userDataType = USRDATA_STARTAUDIO;
        tUserData userData = {0}; //no data

        const size_t size = sizeof(tAllParameters);
        char *parameterString = new char[size]; //deleted by DoProcessUserData
        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_PROCESS_USER_DATA, IN parameterString, this);
    }
    return ret;
}

tResult iPodControl::SendStopAudio(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    if(deviceID == DEVICE_ID_NOT_SET) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid moutPoint"));
    } else {
        const tUserDataType userDataType = USRDATA_STOPAUDIO;
        tUserData userData = {0}; //no data

        const size_t size = sizeof(tAllParameters);
        char *parameterString = new char[size]; //deleted by DoProcessUserData
        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_PROCESS_USER_DATA, IN parameterString, this);
    }
    return ret;
}

tResult iPodControl::SendPlaybackAction(
        const tDeviceID deviceID,
        const tPlaybackAction playbackAction,
        const tNextPrevSkipCount nextPrevSkipCount)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(playbackAction);
    VARTRACE(nextPrevSkipCount);
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    if(mountPoint[0] == 0) {
        ret = MP_ERR_IPOD_NO_ACCESS;
        ETG_TRACE_ERR(("Invalid moutPoint"));
    } else {

        //dispatch from service thread
        const size_t size = sizeof(tAllParameters);
        char *parameterString = new char[size]; //deleted by DoPlaybackAction
        ParameterPLAYBACK_ACTION(OUT parameterString,
                        IN size,
                        IN DTY_IPOD,    //Not used in specific DeviceControl
                        IN deviceID,    //Not used in specific DeviceControl
                        IN "",          //Not used in specific DeviceControl
                        IN mountPoint,
                        IN playbackAction,
                        IN nextPrevSkipCount);

        LocalSPM::GetThreadFactory().Do(IN this, IN FUNCTION_ID_PLAYBACK_ACTION, IN parameterString, this);
    }
    VARTRACE(ret);
    return ret;
}

tResult iPodControl::DoPlaybackAction(char *parameterString)
{
    ENTRY;
    VARTRACE(parameterString)
    tResult ret = MP_NO_ERROR;

    iPodControlRR iPodControl_rr;
    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::PLAYBACK_ACTION", sizeof(msgToSendString));

    ret = iPodControl_rr.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);
    VARTRACE(ret);

    if (parameterString) {
        delete[] parameterString; //new by caller
    }
    return ret;
}

tResult iPodControl::GetBTProfileInfo(tBTProfileList &profileList)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    profileList.clear();

    vector<string> mountPoints = iAPGetMountPoints();
    for(unsigned int i = 0; i < mountPoints.size(); i++) {
        if(IsIAP2(mountPoints[i].c_str()) /*&& iAP_IsDeviceHandleAvail(mountPoints[i].c_str())*/) { //IsIAP2  implies a vaild connection
            tBTProfileInfo BTProfileInfo;
            InitBTProfileInfo(BTProfileInfo);
            BTProfileInfo.deviceID = iAPGetDeviceID(mountPoints[i].c_str());
            BTProfileInfo.BTProfile = iAPGetBTProfile(mountPoints[i].c_str());
            profileList.push_back(BTProfileInfo);
        }
    }

    VARTRACE(profileList.size());
    /*
    for(unsigned int i; i < profileList.size(); i++) {
        VARTRACE(profileList.at(i).deviceID);
        VARTRACE(profileList.at(i).BTProfile);
    }
    */
    return ret;
}

tResult iPodControl::GetDiPONowPlaying(tDeviceID &deviceID, tDiPONowPlaying &nowPlaying)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    InitDiPONowPlaying(nowPlaying);

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        tMediaObject mediaObject;
        InitMediaObject(mediaObject);
        iAPGetNowPlayingMediaObject(mediaObject, mountPoint);

        strncpy_r(OUT nowPlaying.title, IN mediaObject.title, IN sizeof(nowPlaying.title));
        strncpy_r(OUT nowPlaying.artist, IN mediaObject.MetadataField2, IN sizeof(nowPlaying.artist));
        strncpy_r(OUT nowPlaying.album, IN mediaObject.MetadataField4, IN sizeof(nowPlaying.album));
        strncpy_r(OUT nowPlaying.genre, IN mediaObject.MetadataField1, IN sizeof(nowPlaying.genre));
        strncpy_r(OUT nowPlaying.composer, IN mediaObject.MetadataField3, IN sizeof(nowPlaying.composer));

        iAPGetFocusApp(mountPoint, nowPlaying.appName);
        nowPlaying.albumTrackNumber = mediaObject.trackNumber;
        //nowPlaying.albumTrackCount = mediaObject.xx; //no member, see OnCBIAP2_NowPlayingUpdate
        //nowPlaying.albumDiscNumber = mediaObject.xx; //no member, see OnCBIAP2_NowPlayingUpdate
        //nowPlaying.albumDiscCount = mediaObject.xx; //no member yet, see OnCBIAP2_NowPlayingUpdate
        //nowPlaying.chapterCount = mediaObject.xx; //not yet supported by ADIT
        nowPlaying.queueIndex = iAPGetNowPlayingTrackIndex(mountPoint);
        nowPlaying.queueCount = iAPGetNowPlayingTrackCount(mountPoint);
        nowPlaying.queueChapterIndex = iAPGetNowPlayingChapterIndex(mountPoint);
        //nowPlaying.iTunesRadioAd = mediaObject.xx;//no member yet, see OnCBIAP2_NowPlayingUpdate
        //strncpy_r(OUT nowPlaying.iTunesRadioStationName, IN mediaObject.xx, IN sizeof(nowPlaying.iTunesRadioStationName)); //no member yet, see OnCBIAP2_NowPlayingUpdate

        //PSARCC21-1781
        //strncpy_r(OUT nowPlaying.albumArt, IN mediaObject.albumArtString, IN sizeof(nowPlaying.albumArt));
        // mediaObject.albumArtString is not set, need to generate albumartURL of current nowPlaying object
        if(m_AlbumArtTimerID){ //NCG3D-123595
            ETG_TRACE_USR3(("Albumart timer is running , So not updating the Albumart String"));
        }
        else if(iAPIsAlbumArtAvailable(mountPoint)) {
            iPodControlMediaPath albumArtPath;
            tU64 albumArtUUID = 0ULL;
            if(mediaObject.UUID[0] != 0) {
                albumArtUUID = (tU64)strtoull((char *)mediaObject.UUID, NULL, 16); //Hex
            }
            tURL albumArtURL = {0};
            albumArtPath.GetURL(OUT albumArtURL, nowPlaying.queueIndex, nowPlaying.queueChapterIndex, albumArtUUID, 0ULL, nowPlaying.title);
            strncpy_r(nowPlaying.albumArt, mountPoint, sizeof(nowPlaying.albumArt));
            strncat_r(nowPlaying.albumArt, albumArtURL, sizeof(nowPlaying.albumArt));
        }
        //VARTRACE(nowPlaying);
    }

    return ret;
}

tResult iPodControl::GetDiPOPlaybackStatus(tDeviceID &deviceID, tHMIPlaybackState &playbackState)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    playbackState = HMI_PBS_UNDEFINED;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        const int IPODPlayerState = iAPGetIPODPlayerState(mountPoint);

        //convert iPod playback state to HMI_PBS
        switch(IPODPlayerState) {
        case IAP2_PLAYBACK_STATUS_STOPPED:
            playbackState = HMI_PBS_STOPPED;
            break;
        case IAP2_PLAYBACK_STATUS_PAUSED:
            playbackState = HMI_PBS_PAUSED;
            break;
        case IAP2_PLAYBACK_STATUS_PLAYING:
            playbackState = HMI_PBS_PLAYING;
            break;
        case IAP2_PLAYBACK_STATUS_SEEK_FORWARD:
            playbackState = HMI_PBS_FFWD;
            break;
        case IAP2_PLAYBACK_STATUS_SEEK_BACKWARD:
            playbackState = HMI_PBS_FREV;
            break;
        default:
            ETG_TRACE_ERR(("INVALID iAP2PlaybackStatus"));
            break;
        }
        VARTRACE(playbackState);
    }
    return ret;
}

tResult iPodControl::GetDiPOPlaybackShuffleMode(tDeviceID &deviceID, tPlaybackMode &playbackMode)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    playbackMode = tPlaybackMode_init;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        playbackMode = iAPGetPlaybackMode(mountPoint);
        VARTRACE(playbackMode);
    }
    return ret;
}

tResult iPodControl::GetDiPOPlaybackRepeatMode(tDeviceID &deviceID, tRepeatMode &repeatMode)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    repeatMode = tRepeatMode_init;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        repeatMode = iAPGetRepeatMode(mountPoint);
        VARTRACE(repeatMode);
    }
    return ret;
}

tResult iPodControl::GetDiPOPlaytime(tDeviceID &deviceID, tPlaytime &elapsedPlaytime, tPlaytime &totalPlaytime)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    elapsedPlaytime = tPlaytime_init;
    totalPlaytime = tPlaytime_init;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        elapsedPlaytime = iAPGetElapsedPlaytime(mountPoint);
        VARTRACE(elapsedPlaytime);

        totalPlaytime = iAPGetTotalPlaytime(mountPoint);
        VARTRACE(totalPlaytime);
    }
    return ret;
}

tResult iPodControl::GetDiPOCallState(tDeviceID &deviceID, tDiPOCallState &callStateList)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    callStateList.clear();

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        callStateList = iAPGetCallState(mountPoint);
        iAP2RemoveDisconnectedCallState(mountPoint); //clean up list

        VARTRACE(callStateList.size());
        tDiPOCallState remainingCallStateList = iAPGetCallState(mountPoint);
        VARTRACE(remainingCallStateList.size());
        /*
        for(unsigned int i; i < callStateList.size(); i++) {
            VARTRACE(callStateList.at(i));
        }
        */
    }
    return ret;
}

tResult iPodControl::GetDiPOCallDuration(tDeviceID &deviceID, U32 &callDuration)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    callDuration = 0;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        callDuration = iAPGetCallDuration(mountPoint);
    }
    return ret;
}

tResult iPodControl::GetDiPOCommunications(tDeviceID &deviceID, tDiPOCommunications &communications)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    InitDiPOCommunications(communications);

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        communications = iAPGetCommunications(mountPoint);
        VARTRACE(communications);
    }
    return ret;
}
tResult iPodControl::GetDiPOLocationInfo(tDeviceID &deviceID, tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    startStopLocationInfo = false;
    InitDiPOLocationInfoType(locationInfoType);

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        //VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        iAPGetDiPOLocationInfo(mountPoint,startStopLocationInfo,locationInfoType);
    }
    return ret;
}

tResult iPodControl::GetDiPOGPRMCDataStatusValues(tDeviceID &deviceID, tMountPoint &USBSerialNumber, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    memset(USBSerialNumber,0,sizeof(USBSerialNumber));
    InitDiPOGPRMCDataStatusValues(GPRMCDataStatusValues);
    tDeviceInfo deviceInfo;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID);
        VARTRACE(deviceInfo.connectionType);
        if(deviceInfo.connectionType == DCT_WIFI){
            strncpy_r(USBSerialNumber, deviceInfo.appleDeviceUSBSerialNumber, sizeof(USBSerialNumber));
            VARTRACE(USBSerialNumber);
        }
        else if(deviceInfo.connectionType == DCT_USB){
            strncpy_r(USBSerialNumber, mountPoint, sizeof(USBSerialNumber));
            VARTRACE(USBSerialNumber);
        }

        iAPGetDiPOGPRMCDataStatusValues(mountPoint,GPRMCDataStatusValues);
    }
    return ret;
}


tResult iPodControl::SignalUserData(const tDeviceID deviceID, const tUserDataType userDataType, const tUserData userData)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(userDataType);

    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    if(userDataType == USRDATA_GPS || userDataType == USRDATA_DR) {
        tDiPOGPGGAData diPOGPGGAData = {0};
        tDiPOGPRMCData diPOGPRMCData = {0};
        tDiPOGPGSVData diPOGPGSVData = {0};
        tDiPOGPHDTData diPOGPHDTData = {0};
        tDiPOPASCDData diPOPASCDData = {0};
        tDiPOPAGCDData diPOPAGCDData = {0};
        tDiPOPAACDData diPOPAACDData = {0};

        if(userDataType == USRDATA_GPS) {
            SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tDiPOGPGGAData_format tDiPOGPRMCData_format tDiPOGPGSVData_format tDiPOGPHDTData_format, diPOGPGGAData, diPOGPRMCData, diPOGPGSVData, diPOGPHDTData);
            VARTRACE(diPOGPGGAData);
            VARTRACE(diPOGPRMCData);
            VARTRACE(diPOGPGSVData);
            VARTRACE(diPOGPHDTData);
        }

        if(userDataType == USRDATA_DR) {
            SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tDiPOPASCDData_format tDiPOPAGCDData_format tDiPOPAACDData_format, diPOPASCDData, diPOPAGCDData, diPOPAACDData);
            VARTRACE(diPOPASCDData);
            VARTRACE(diPOPAGCDData);
            VARTRACE(diPOPAACDData);
        }
        ETG_TRACE_USR3(("TRANSFERRING LOCATION DATA TO IPHONE..."));
        ret = iAP2_LocationInformation(mountPoint, diPOGPGGAData, diPOGPRMCData, diPOGPGSVData, diPOGPHDTData, diPOPASCDData, diPOPAGCDData, diPOPAACDData);
        ETG_TRACE_USR3(("TRANSFER LOCATION DATA TO IPHONE returned %d", ret));

    } else if(userDataType == USRDATA_VEHICLESTATUS) {
        ETG_TRACE_USR3(("VEHICLE STATUS UPDATE TO IPHONE..."));
        ret = iAP2_VehicleStatusUpdate(mountPoint, LocalSPM::GetDataProvider().OutsideTemperature());
        ETG_TRACE_USR3(("VEHICLE STATUS UPDATE returned %d", ret));

    } else if(userDataType == USRDATA_STARTAUDIO) {
        ETG_TRACE_USR3(("START USB DEVICE MODE AUDIO..."));
        ret = iAP2_StartUSBDeviceModeAudio(mountPoint);
        ETG_TRACE_USR3(("START USB DEVICE MODE AUDIO returned %d", ret));

    } else if(userDataType == USRDATA_STOPAUDIO) {
        ETG_TRACE_USR3(("STOP USB DEVICE MODE AUDIO..."));
        ret = iAP2_StopUSBDeviceModeAudio(mountPoint);
        ETG_TRACE_USR3(("STOP USB DEVICE MODE AUDIO returned %d", ret));
    } else if(userDataType == USRDATA_STOPMEDIALIBRARYUPDATES) {
        ret = iAP2_StopMediaLibraryUpdates(mountPoint);
    }else if(userDataType == USRDATA_APPCONTROL_CONNECT) {
        tMountPoint mountPoint = {0};
        tAppName appName = { 0 };
        unsigned short sessionID = 0;
        tProtocolName protocolName = { 0 };
        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tMountPoint_format tAppName_format tSessionID_format tProtocolName_format,mountPoint,appName,&sessionID,protocolName);

        ret = LocalSPM::GetOutputWrapper().AppControlConnect(mountPoint, appName, sessionID,protocolName);
    }else if (userDataType == USRDATA_APPCONTROL_CLOSE) {
        tMountPoint mountPoint = {0};
        tAppName appName = { 0 };
        unsigned short sessionID = 0;
        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tMountPoint_format tAppName_format tSessionID_format,mountPoint,appName,&sessionID);

        ret = LocalSPM::GetOutputWrapper().AppControlClose(mountPoint, appName, sessionID);
    }else if((userDataType == USRDATA_START_EA_NATIVE_TRANSPORT) || (userDataType == USRDATA_STOP_EA_NATIVE_TRANSPORT)){

       U8 iAP2iOSAppIdentifier;
       U8 sinkEndpoint;
       U8 sourceEndpoint;
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "iii" ,&iAP2iOSAppIdentifier, &sinkEndpoint, &sourceEndpoint);
       if(userDataType == USRDATA_START_EA_NATIVE_TRANSPORT)
       {
           ret = LocalSPM::GetOutputWrapper().NativeTransportStart(deviceID, iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint);
       }
       else
       {
           ret = LocalSPM::GetOutputWrapper().NativeTransportStop(deviceID, iAP2iOSAppIdentifier, sinkEndpoint, sourceEndpoint);
       }

   }else if(userDataType == USRDATA_APP_DATA_RECEIVED) {

       tMountPoint mountPoint = {0};
       tAppName appName = { 0 };
       int sessionID = 0;
       U8* iAP2iOSAppDataRxd = NULL;
       int iAP2iOSAppDataLength = 0;

       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR,"ttiip",mountPoint,appName,&sessionID,&iAP2iOSAppDataLength,&iAP2iOSAppDataRxd);

       VARTRACE(iAP2iOSAppDataLength);
       iAPTraceData(iAP2iOSAppDataRxd, iAP2iOSAppDataLength);

       ret = LocalSPM::GetOutputWrapper().AppControlCommand(mountPoint,
               appName, (unsigned short)sessionID,(U16) iAP2iOSAppDataLength, iAP2iOSAppDataRxd);

       if(iAP2iOSAppDataRxd)
       {
           delete[] iAP2iOSAppDataRxd;
       }

   } else if(userDataType == USRDATA_UPDATEAPPCONTROL) {
        tAppControlUpdateType updateType = tAppControlUpdateType_init;
        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tAppControlUpdateType_format, &updateType);
        switch(updateType) {
            case APPCONTROL_UPDATE_BTPROFILES:
                ret = LocalSPM::GetOutputWrapper().UpdateBTProfileInfo();
                break;
            case APPCONTROL_UPDATE_DIPO_NOWPLAYING:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPONowPlaying();
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOMetaData(); //deprecated
                break;
            case APPCONTROL_UPDATE_DIPO_PLAYTIME:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOPlaytime();
                break;
            case APPCONTROL_UPDATE_DIPO_PLAYBACKSTATUS:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOPlaybackStatus();
                break;
            case APPCONTROL_UPDATE_DIPO_SHUFFLEMODE:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOPlaybackShuffleMode();
                break;
            case APPCONTROL_UPDATE_DIPO_REPEATMODE:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOPlaybackRepeatMode();
                break;
            case APPCONTROL_UPDATE_DIPO_CALLSTATE:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOCallState();
                ret = LocalSPM::GetOutputWrapper().UpdatePhoneInfo(); //deprecated
                break;
            case APPCONTROL_UPDATE_DIPO_COMMUNICATIONS:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOCommunications();
                break;
            case APPCONTROL_UPDATE_DIPO_CALLDURATION:
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOCallDuration();
                break;
            case APPCONTROL_UPDATE_DIPO_LOCATIONINFO:
                ETG_TRACE_USR4(("SignalUserData: APPCONTROL_UPDATE_DIPO_LOCATIONINFO"));
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOLocationInfo();
                break;
            case APPCONTROL_UPDATE_DIPO_POWER:
                ETG_TRACE_USR4(("SignalUserData: APPCONTROL_UPDATE_DIPO_POWER"));
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOPower();
                break;
            case APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE:
                ETG_TRACE_USR4(("SignalUserData: APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE"));
                ret = LocalSPM::GetOutputWrapper().UpdateDiPORouteGuidance();
                break;
            case APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE_MANEUVER:
                ETG_TRACE_USR4(("SignalUserData: APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE_MANEUVER"));
                ret = LocalSPM::GetOutputWrapper().UpdateDiPORouteGuidanceManeuver();
                break;
            case APPCONTROL_UPDATE_DIPO_GPRMCDATASTATUSVALUES_NOTIFICATION:
                ETG_TRACE_USR4(("SignalUserData: APPCONTROL_UPDATE_DIPO_GPRMCDATASTATUSVALUES_NOTIFICATION"));
                ret = LocalSPM::GetOutputWrapper().UpdateDiPOGPRMCDataStatusValues();
                break;
                case APPCONTROL_UPDATE_DIPO_DEVICETIME:
                {
                    ret = LocalSPM::GetOutputWrapper().UpdateDiPODeviceTime();
                }
                break;
            default:
            ETG_TRACE_ERR(("Unhandled AppControlUpdateType"));
            break;
        }
    } else if(userDataType == USRDATA_FINGERPRINTTIMEOUT) {
        tFingerprint fingerprint;
        iAPIndexerGetFingerprint(mountPoint, fingerprint);
        ret = OnFingerprint(fingerprint, FPS_OK_SKIP_INDEXING, NUMBER_OF_FILES_NONE, deviceID);
    } else if(userDataType == USRDATA_TEST) {
        //TEST
        for(int i = 0; i < 100; i++) {
            RequestResponseSM rrSourceActivity;
            ret = rrSourceActivity.DoEventAnswer(IN "asdasdasd::asdasd", (char *)NULL, NULL, 200000);
        }
    } else if(userDataType == USRDATA_ROLE_SWITCH_REQUIRED_ANSWER) {
        tDiPOSwitchReqResponse diPOSwitchReqResponse = tDiPOSwitchReqResponse_init;
        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, tDiPOSwitchReqResponse_format, &diPOSwitchReqResponse);
        ret = RoleSwitchRequiredAnswer(deviceID, diPOSwitchReqResponse);
    } else if(userDataType == USRDATA_ACC_WIFI_CREDENTIALS) {

        tDeviceInfo deviceInfo;
        ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID);
        VARTRACE(deviceInfo.appleDeviceMACAddress);
        ret = LocalSPM::GetOutputWrapper().GetWiFiCredentials(deviceID,deviceInfo.appleDeviceMACAddress);

    }else if(userDataType == USRDATA_ACC_WIFI_CREDENTIALS_RESULT) {

        tWiFiAPCredentials wifiCredentials;
        m_AccessoryWiFiCredentialsMutex.lock();
        wifiCredentials = m_AccessoryWiFiCredentials;
        m_AccessoryWiFiCredentialsMutex.unlock();
        iAP2_AccessoryWiFiConfigurationInformation(mountPoint,wifiCredentials);
    } else if(userDataType == USRDATA_START_IAP2_CARPLAY_WIRELESS) {
        ret = PrepareIAP2OverCarplayWiFi(mountPoint);
    }else if(userDataType == USRDATA_OOBBT_PAIRING_ACCESSORY_INFORMATION) {
         ETG_TRACE_USR3(("OOB BT Pairing Accessory Information"));
         ret = iAP2_OOBTBTPairingAccessoryInformation(mountPoint);
         ETG_TRACE_USR3(("OOB BT Pairing Accessory Information returned %d", ret));
    } else if(userDataType == USRDATA_OOBBT_PAIRING_COMPLETION_INFORMATION) {

        tDeviceID deviceID;
        tMACAddress deviceAddress;
        tDeviceName deviceName;
        tLinkKey linkKey;
        tBTOobType oobType;

        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "ittti",&deviceID,deviceAddress,deviceName,linkKey,&oobType);
        ret = LocalSPM::GetOutputWrapper().AddOobPairedDevice(deviceID,deviceAddress,deviceName,linkKey,oobType);



   }else if(userDataType == USRDATA_INITIATECALL) {
       tDiPOInitiateCall initiateCall;
       memset(&initiateCall,0,sizeof(initiateCall));
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "itit",&(initiateCall.type),initiateCall.destinationID,&(initiateCall.service),initiateCall.addressBookID);
       ETG_TRACE_USR3(("Initiate Call"));
       ret = iAP2_InitiateCall(mountPoint,initiateCall);
       ETG_TRACE_USR3(("Initiate Call returned %d", ret));

   }else if(userDataType == USRDATA_ACCEPTCALL) {
       tDiPOAcceptCall acceptCall;
       memset(&acceptCall,0,sizeof(acceptCall));
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "it",&(acceptCall.acceptAction),acceptCall.callUUID);
       ETG_TRACE_USR3(("Accept Call"));
       ret = iAP2_AcceptCall(mountPoint,acceptCall);
       ETG_TRACE_USR3(("Accept Call returned %d", ret));

   }else if(userDataType == USRDATA_ENDCALL) {
       tDiPOEndCall endCall;
       memset(&endCall,0,sizeof(endCall));
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "it",&(endCall.endAction),endCall.callUUID);
       ETG_TRACE_USR3(("End Call"));
       ret = iAP2_EndCall(mountPoint,endCall);
       ETG_TRACE_USR3(("End Call returned %d", ret));

   }else if(userDataType == USRDATA_SWAPCALLS) {
       ETG_TRACE_USR3(("Swap Calls"));
       ret = iAP2_SwapCalls(mountPoint);
       ETG_TRACE_USR3(("Swap Calls returned %d", ret));
   }else if(userDataType == USRDATA_START_RG) {
       tUInt RouteGuidanceDisplayComponentID = RGDISPLAYCOMPONENTIDS_NONE;
       tBoolean SourceName = FALSE;
       tBoolean SourceSupportsRouteGuidance = FALSE;
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "iii" , &RouteGuidanceDisplayComponentID, &SourceName, &SourceSupportsRouteGuidance );
       VARTRACE(SourceName);
       VARTRACE(SourceSupportsRouteGuidance);
       VARTRACE(RouteGuidanceDisplayComponentID);
       ETG_TRACE_USR3(("TRANSFERRING COMPONENTID FOR START ROUTE GUIDANCE TO IPHONE..."));
       ret = iAP2_StartRouteGuidance(mountPoint, RouteGuidanceDisplayComponentID, SourceName, SourceSupportsRouteGuidance);
       ETG_TRACE_USR3(("TRANSFERRING COMPONENTID FOR START ROUTE GUIDANCE TO IPHONE returned %d", ret));
   }else if(userDataType == USRDATA_STOP_RG) {
       tUInt RouteGuidanceDisplayComponentID = RGDISPLAYCOMPONENTIDS_NONE;
       SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "i" , &RouteGuidanceDisplayComponentID );
       VARTRACE(RouteGuidanceDisplayComponentID);
       ETG_TRACE_USR3(("TRANSFERRING COMPONENTID FOR STOP ROUTE GUIDANCE TO IPHONE..."));
       ret = iAP2_StopRouteGuidance(mountPoint, RouteGuidanceDisplayComponentID);
       ETG_TRACE_USR3(("TRANSFERRING COMPONENTID FOR STOP ROUTE GUIDANCE TO IPHONE returned %d", ret));
   }else if(userDataType == USRDATA_LANGUAGEUPDATES) {
        ret = iAP2_IdentificationInformationUpdate();
        ETG_TRACE_USR3(("Identification Information Update returned %d", ret));
    }else if(userDataType == USRDATA_UPDATE_READY_TO_PLAY) {
        ret = OnUpdateReadyToPlay(mountPoint);
    }else if(userDataType == USRDATA_APPLE_HID_COMMAND) {
        tPlaybackHIDCommand playbackHIDCommand = HID_INVALID;
        tBTButtonEvent keyEvent = BUTTON_INVALID;
        tMountPoint mountPoint = {0};
        SMF::UnMarshal(userData, DOUBLE_MARSHAL_SEPARATOR, "iit" , &playbackHIDCommand, &keyEvent, mountPoint);
        ret = iAP2_PlaybackHIDCommand(IN playbackHIDCommand, IN keyEvent, IN mountPoint);
        ETG_TRACE_USR3(("iAP2_PlaybackHIDCommand returned %d", ret));
    } else {
       ETG_TRACE_ERR(("Unhandled UserDataType"));
    }

    return MP_NO_ERROR;
}

tResult iPodControl::ProcessUserData(const tDeviceID deviceID, const tUserDataType userDataType, const tUserData userData)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    ret = SignalUserData(deviceID, userDataType, userData);

    tReturnValue retval = (ret == MP_NO_ERROR);
    VARTRACE(retval);

    /* Send ANSWER message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterPROCESS_USERDATA_ANSWER(OUT parameterString, IN size, IN deviceID, IN retval);
    SendEvent(IN PROCESS_USERDATA_ANSWER, IN parameterString);

    return MP_NO_ERROR;
}

tResult iPodControl::ProcessUserDataAnswer(const tDeviceID deviceID, const tReturnValue retVal)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(retVal);

    //release the calling outer SM
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    SMF::Marshal(parameterString, size - 1, "i", retVal);

    SendAnswer(IN parameterString);

    return MP_NO_ERROR;
}

tResult iPodControl::DoProcessUserData(char *parameterString)
{
    ENTRY;
    VARTRACE(parameterString)
    tResult ret = MP_NO_ERROR;

    iPodControlRR iPodControl_rr;
    char msgToSendString[64];
    strncpy_r(msgToSendString, "iPodControlSM::PROCESS_USERDATA", sizeof(msgToSendString));

    ret = iPodControl_rr.DoEventAnswer(IN msgToSendString, IN parameterString, NULL, 15000);
    VARTRACE(ret)

    if (parameterString) {
        delete[] parameterString; //new by caller
    }
    VARTRACE(iPodControl_rr.result);

    return ret;
}

tResult iPodControl::FileCountComp(const tMountPoint mountPoint,const tIndexSession indexSession)
{

    ENTRY;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        tDeviceID deviceID=iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);
        tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
        VARTRACE(numberOfTrack);
        tNumberOfFiles numberOfVideo=iAPIndexerGetVideoCount(mountPoint);
        VARTRACE(numberOfVideo);
        tNumberOfFiles numberOfPlaylist=iAPIndexerGetPlaylistCount(mountPoint)-1; //skip default playlist
        VARTRACE(numberOfPlaylist);

        /* Send ALLOCATE message to DeviceDispatcherSM */
       char messageString[64];
        strncpy_r(messageString, "IndexerSM::FILE_COUNT_COMP_WITHDB",
                sizeof(messageString) - 1);
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        ret = LocalSPM::GetIndexer().ParameterFILE_COUNT_COMP_WITHDB(OUT parameterString,IN size,IN numberOfPlaylist, IN numberOfVideo,IN numberOfTrack,IN deviceID,IN indexSession);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }
        ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::FileCountCompAnswer(const tDeviceID deviceID, const tFingerprintStatus fingerPrintStatus,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(fingerPrintStatus);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        tMountPoint mountPoint;
        iAPGetMountPointFromDeviceID(deviceID,mountPoint);
        VARTRACE(mountPoint);
        if(FPS_OK_SKIP_INDEXING == fingerPrintStatus){
            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            tIndex iindex=0;
            tNumberOfFiles numberOfPlaylist=iAPIndexerGetPlaylistCount(mountPoint);
            tNumberOfFiles numberOfVideo=iAPIndexerGetVideoCount(mountPoint);
            tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
            VARTRACE(numberOfTrack);
            VARTRACE(numberOfVideo);
            VARTRACE(numberOfPlaylist);
            if(numberOfPlaylist)
            {
                iindex++; //skip default playlist
                int chunk = (int)(numberOfPlaylist - iindex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfPlaylist - iindex);
                ret=ParameterPLAYLIST_NAME_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN iindex,IN chunk,IN indexSession);
                if (MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret=SendEvent(PLAYLIST_NAME_COMP, IN parameterString);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }else if(numberOfVideo){
                //tAllParameters parameterString;
                //size_t size = sizeof(parameterString);
                //tIndex iindex=0;
                int chunk = (int)(numberOfVideo - iindex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfVideo - iindex);
                ret = ParameterVIDEO_UID_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN iindex ,IN chunk,IN indexSession);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(VIDEO_UID_COMP, IN parameterString);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }else if(numberOfTrack) {
                tIndex newIndex=0;
                int chunk = (int)(numberOfTrack - newIndex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfTrack - newIndex);
                ret = ParameterTRACK_UID_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN newIndex,IN chunk,IN indexSession);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(TRACK_UID_COMP, IN parameterString);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }

        }else{

            tAllParameters parameterString;
            size_t size = sizeof(parameterString);
            tFingerprint fingerprint;
            iAPIndexerGetFingerprint(mountPoint,fingerprint);
            VARTRACE(fingerprint);
            tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
            ret=ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
            if (MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret=SendEvent(PUT_FINGERPRINT, IN parameterString);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::PlaylistNameComp(const tMountPoint mountPoint,const tDeviceID deviceID,const tIndex plIndex,const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(plIndex);
    VARTRACE(chunk);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        char messageString[64];
        strncpy_r(messageString, "IndexerSM::PLAYLIST_NAME_COMP_WITHDB",
                sizeof(messageString) - 1);
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tNumberOfFiles numberOfPlaylist=iAPIndexerGetPlaylistCount(mountPoint);
        VARTRACE(numberOfPlaylist);
        if (numberOfPlaylist > 0 && plIndex < numberOfPlaylist) {
            tIPODDBRecords *playlistRecordsPtr = new tIPODDBRecords;
            if(NULL!=playlistRecordsPtr) {
                ret=iAP1_GetPlaylistNameList(mountPoint,plIndex,chunk,playlistRecordsPtr);
                if(MP_NO_ERROR == ret) {
                    vector<string> *playlistNameVectorPtr = new vector<string>; //deleted in Indexer
                    for(tUInt i=0;i<playlistRecordsPtr->size();i++){
                        tPlaylistName playlistName;
                        map<int, string>::iterator it = playlistRecordsPtr->find(plIndex+i);
                        if (it == playlistRecordsPtr->end()){
                            ETG_TRACE_ERR(("%s - NO ENTRY", __PRETTY_FUNCTION__));
                            break;
                        } else{
                            strncpy_r(playlistName,it->second.c_str(),sizeof(playlistName));
                            VARTRACE(playlistName)
                            if(playlistNameVectorPtr) {
                                playlistNameVectorPtr->push_back(it->second);
                            }
                        }
                    }
                    ret = LocalSPM::GetIndexer().ParameterPLAYLIST_NAME_COMP_WITHDB(OUT parameterString, IN size, IN deviceID, IN plIndex,IN chunk, IN numberOfPlaylist, IN playlistNameVectorPtr,IN indexSession);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                        parameterString[0] = '\0';
                    }
                    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                    }
                }else{ //lint !e429 deleted by Indexer
                    tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
                    tFingerprint fingerprint;
                    iAPIndexerGetFingerprint(mountPoint,fingerprint);
                    VARTRACE(fingerprint);
                    tFingerprintStatus fingerPrintStatus = FPS_OK;
                    ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                        parameterString[0] = '\0';
                    }
                    ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                    }
                    //return MP_NO_ERROR;
                }
                delete playlistRecordsPtr;
            }else{
                ETG_TRACE_ERR(("Error while allocating memory"));
            }
        }else {
            ETG_TRACE_ERR(("Error : There is no playlist it should not enter to this condition"));
        }
    }
    return MP_NO_ERROR;;
}

tResult iPodControl::PlaylistNameCompAnswer(const tDeviceID deviceID,const tFingerprintStatus fingerPrintStatus,const tIndex plIndex,const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(fingerPrintStatus);
    VARTRACE(plIndex);
    VARTRACE(chunk);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        tMountPoint mountPoint;
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        iAPGetMountPointFromDeviceID(deviceID,mountPoint);
        VARTRACE(mountPoint);
        tNumberOfFiles numberOfPlaylist=iAPIndexerGetPlaylistCount(mountPoint);
        VARTRACE(numberOfPlaylist);
        if( FPS_OK_SKIP_INDEXING != fingerPrintStatus)
        {
            tFingerprint fingerprint;
            iAPIndexerGetFingerprint(mountPoint,fingerprint);
            VARTRACE(fingerprint);
            tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
            ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
            return MP_NO_ERROR;;
        }
        if( /*FPS_OK_SKIP_INDEXING == fingerPrintStatus &&*/ (numberOfPlaylist == (plIndex+chunk)))
        {
            ETG_TRACE_USR3(("Playlist Comparison reached the end"));
            tIndex videoindex=0;
            tNumberOfFiles numberOfVideo=iAPIndexerGetVideoCount(mountPoint);
            tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
            VARTRACE(numberOfVideo);
            VARTRACE(numberOfTrack);
            if(numberOfVideo)
            {
                int videochunk = (int)(numberOfVideo - videoindex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfVideo - videoindex);
                ret = ParameterVIDEO_UID_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN videoindex,IN videochunk,IN indexSession);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(VIDEO_UID_COMP, IN parameterString);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
                return MP_NO_ERROR;
            }else if(numberOfTrack)
            {
                tIndex newIndex=0;
                int trackchunk = (int)(numberOfTrack - newIndex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfTrack - newIndex);
                ret = ParameterTRACK_UID_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN newIndex,IN trackchunk,IN indexSession);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(TRACK_UID_COMP, IN parameterString);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
                return MP_NO_ERROR;
            }
        }

        tIndex newIndex = plIndex + chunk;
        VARTRACE(newIndex)
        int newChunk = (int)(numberOfPlaylist - newIndex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfPlaylist - newIndex);
        VARTRACE(newChunk);
        ret = ParameterPLAYLIST_NAME_COMP(OUT parameterString, IN size, IN mountPoint,IN deviceID,IN newIndex,IN newChunk,IN indexSession);
        if(MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }
        ret = SendEvent(PLAYLIST_NAME_COMP, IN parameterString);
        if(MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return MP_NO_ERROR;;

}

tResult iPodControl::VideoUIDComp(const tMountPoint mountPoint,const tDeviceID deviceID,const tIndex vIndex,const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(vIndex);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        char messageString[64];
        strncpy_r(messageString, "IndexerSM::VIDEO_UID_COMP_WITHDB",
                sizeof(messageString) - 1);
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tNumberOfFiles numberOfVideo=iAPIndexerGetVideoCount(mountPoint);
        VARTRACE(numberOfVideo);
        if (numberOfVideo > 0 && vIndex < numberOfVideo) {
            tIPODTrackInfosPtr trackInfosPtr = new tIPODTrackInfos;
            if(NULL!=trackInfosPtr) {
                ret=iAP1_GetVideoUIDList(mountPoint,vIndex,chunk,trackInfosPtr);
                if(MP_NO_ERROR == ret) {
                    vector<string> *uidVectorPtr = new vector<string>;
                    for(tUInt i=0;i<trackInfosPtr->size();i++){
                        tUUID rUUID = {0};
                        pair<tIPODTrackInfos::const_iterator, tIPODTrackInfos::const_iterator> pr = trackInfosPtr->equal_range(vIndex + i);
                        for (tIPODTrackInfos::const_iterator it = pr.first; it != pr.second; ++it) {
                            if (it->second.type == UID) {
                                if(it->second.data.trackUID > 0) {
                                    snprintf(rUUID, sizeof(rUUID), IPODCONTROL_UUID_FORMAT, it->second.data.trackUID);
                                    VARTRACE(rUUID);
                                }
                            break;
                            }
                        }
                        string uuidStr(rUUID);
                        if(uidVectorPtr) {
                            uidVectorPtr->push_back(uuidStr);
                        }
                    }

                    ret = LocalSPM::GetIndexer().ParameterVIDEO_UID_COMP_WITHDB(OUT parameterString, IN size, IN deviceID, IN vIndex,IN chunk ,IN numberOfVideo,IN uidVectorPtr,IN indexSession);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                        parameterString[0] = '\0';
                    }
                    ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                    }
                }else{ //lint !e429 deleted by Indexer
                    tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
                    tFingerprint fingerprint;
                    iAPIndexerGetFingerprint(mountPoint,fingerprint);
                    VARTRACE(fingerprint);
                    tFingerprintStatus fingerPrintStatus = FPS_OK;
                    ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                        parameterString[0] = '\0';
                    }
                    ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
                    if(MP_NO_ERROR != ret)
                    {
                        ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                    }
                    //return MP_NO_ERROR;
                }
                delete trackInfosPtr;
            }else{
                ETG_TRACE_ERR(("Error while allocating memory"));
            }
        }else{
            ETG_TRACE_ERR(("Error : There is no Video it should not enter to this condition"));
        }
    }
    return MP_NO_ERROR;;
}

tResult iPodControl::VideoUIDCompAnswer(const tDeviceID deviceID,const tFingerprintStatus fingerPrintStatus,const tIndex vIndex,const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(fingerPrintStatus);
    VARTRACE(vIndex);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tMountPoint mountPoint;
        iAPGetMountPointFromDeviceID(deviceID,mountPoint);
        VARTRACE(mountPoint);
        tNumberOfFiles numberOfVideo=iAPIndexerGetVideoCount(mountPoint);
        VARTRACE(numberOfVideo);
        if( FPS_OK_SKIP_INDEXING != fingerPrintStatus)
        {

            tFingerprint fingerprint;
            iAPIndexerGetFingerprint(mountPoint,fingerprint);
            VARTRACE(fingerprint);
            tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
            ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
            if(MP_NO_ERROR != ret){
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
            return MP_NO_ERROR;
        }
        if( /*FPS_OK_SKIP_INDEXING == fingerPrintStatus &&*/ (numberOfVideo == (vIndex+chunk)))
        {

            tIndex trackindex=0;
            tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
            if(numberOfTrack){
                int newChunk = (int)(numberOfTrack - trackindex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfTrack - vIndex);
                ret = ParameterTRACK_UID_COMP(OUT parameterString, IN size, IN mountPoint, IN deviceID, IN trackindex, IN newChunk,IN indexSession);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(TRACK_UID_COMP, IN parameterString);
                if(MP_NO_ERROR != ret){
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
                return MP_NO_ERROR;
            }
        }
        tIndex newIndex = vIndex + chunk;
        VARTRACE(newIndex)
        int newChunk = (int)(numberOfVideo - newIndex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfVideo - newIndex);
        VARTRACE(newChunk);
        ret = ParameterVIDEO_UID_COMP(OUT parameterString, IN size, IN mountPoint,IN deviceID,IN newIndex,IN newChunk,IN indexSession);
        if(MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }
        ret = SendEvent(VIDEO_UID_COMP, IN parameterString);
        if(MP_NO_ERROR != ret){
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::TrackUIDComp(const tMountPoint mountPoint, const tDeviceID deviceID, const tIndex trackIndex, const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(mountPoint);
    VARTRACE(trackIndex);
    tResult ret = MP_NO_ERROR;
    char messageString[64];
    strncpy_r(messageString, "IndexerSM::TRACK_UID_COMP_WITHDB",
            sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
    if (numberOfTrack > 0 && trackIndex < numberOfTrack) {
        tIPODTrackInfosPtr trackInfosPtr = new tIPODTrackInfos;
        if(NULL!=trackInfosPtr) {
            ret=iAP1_GetTrackUIDList(mountPoint,trackIndex,chunk,trackInfosPtr);
            if(MP_NO_ERROR == ret) {
                vector<string> *uidVectorPtr = new vector<string>;
                for(tUInt i=0;i<trackInfosPtr->size();i++){
                    tUUID rUUID = {0};
                    pair<tIPODTrackInfos::const_iterator, tIPODTrackInfos::const_iterator> pr = trackInfosPtr->equal_range(trackIndex + i);
                    for (tIPODTrackInfos::const_iterator it = pr.first; it != pr.second; ++it) {
                        if (it->second.type == UID) {
                            if(it->second.data.trackUID > 0) {
                                snprintf(rUUID, sizeof(rUUID), IPODCONTROL_UUID_FORMAT, it->second.data.trackUID);
                                VARTRACE(rUUID);
                            }
                        break;
                        }
                    }
                    string uuidStr(rUUID);
                    if(uidVectorPtr) {
                        uidVectorPtr->push_back(uuidStr);
                    }
                }
                ret = LocalSPM::GetIndexer().ParameterTRACK_UID_COMP_WITHDB(OUT parameterString, IN size, IN deviceID, IN trackIndex,IN chunk, IN numberOfTrack, IN uidVectorPtr,IN indexSession);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
            }else{ //lint !e429 deleted by Indexer
                tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
                tFingerprint fingerprint;
                iAPIndexerGetFingerprint(mountPoint,fingerprint);
                VARTRACE(fingerprint);
                tFingerprintStatus fingerPrintStatus = FPS_OK;
                ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                    parameterString[0] = '\0';
                }
                ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
                if(MP_NO_ERROR != ret)
                {
                    ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
                }
                //return MP_NO_ERROR;
            }
            delete trackInfosPtr;
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::TrackUIDCompAnswer(const tDeviceID deviceID, const tFingerprintStatus fingerPrintStatus, const tIndex trackIndex, const tListSize chunk,const tIndexSession indexSession)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(fingerPrintStatus);
    VARTRACE(trackIndex);
    tResult ret = MP_NO_ERROR;
    if(m_IndexSession == indexSession)
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);
        tMountPoint mountPoint;
        iAPGetMountPointFromDeviceID(deviceID,mountPoint);
        VARTRACE(mountPoint);
        tNumberOfFiles numberOfTrack=iAPIndexerGetTrackCount(mountPoint);
        VARTRACE(numberOfTrack);
        if( FPS_OK_SKIP_INDEXING != fingerPrintStatus ||( /*FPS_OK_SKIP_INDEXING == fingerPrintStatus &&*/ (numberOfTrack == (trackIndex+chunk))))
        {

            tNumberOfFiles numberOfFiles=iAPIndexerGetTrackCount(mountPoint)+iAPIndexerGetVideoCount(mountPoint)+iAPIndexerGetPlaylistCount(mountPoint);
            tFingerprint fingerprint;
            iAPIndexerGetFingerprint(mountPoint,fingerprint);
            VARTRACE(fingerprint);
            ret = ParameterPUT_FINGERPRINT(OUT parameterString, IN size, IN fingerprint, IN fingerPrintStatus, IN numberOfFiles, IN deviceID);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
                parameterString[0] = '\0';
            }
            ret = SendEvent(PUT_FINGERPRINT, IN parameterString);
            if(MP_NO_ERROR != ret)
            {
                ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
            }
            return MP_NO_ERROR;
        }

        tIndex newIndex = trackIndex + chunk;
        VARTRACE(newIndex)
        int newChunk = (int)(numberOfTrack - newIndex) > iAPIndexerGetChunkSize(mountPoint) ? iAPIndexerGetChunkSize(mountPoint) : (int)(numberOfTrack - newIndex);
        VARTRACE(newChunk);
        ret = ParameterTRACK_UID_COMP(OUT parameterString, IN size, IN mountPoint,IN deviceID,IN newIndex,IN newChunk,IN indexSession);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameterString[0] = '\0';
        }
        ret = SendEvent(TRACK_UID_COMP, IN parameterString);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return MP_NO_ERROR;
}

tResult iPodControl::RoleSwitchRequired(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);

    tResult ret = MP_NO_ERROR;

    tDiPOSwitchReqResponse diPOSwitchReqResponse = DIPO_ROLE_SWITCH_NOT_REQUIRED; //default on error

    /* Send ROLE_SWITCH_REQUIRED_ANSWER message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterROLE_SWITCH_REQUIRED_ANSWER(OUT parameterString, IN size, IN deviceID, IN diPOSwitchReqResponse);

    tMountPoint USBSerialNumber = {0};
    iAPGetMountPointFromDeviceID(deviceID, USBSerialNumber);
    VARTRACE(USBSerialNumber);
    tBoolean bCarplayCapable = iAP2_IsCarPlayPossible(USBSerialNumber);
    VARTRACE(bCarplayCapable);
    ret = LocalSPM::GetOutputWrapper().DIPORoleSwitchRequired(deviceID,USBSerialNumber,bCarplayCapable);
    if(ret != MP_NO_ERROR) {
        SendEvent(IN ROLE_SWITCH_REQUIRED_ANSWER, IN parameterString);
    } else {
        RegisterReleaseEvent("ROLE_SWITCH_REQUIRED_ANSWER", parameterString, LocalSPM::GetDataProvider().iPodControlIAP2CheckRoleSwitchRequiredTimeOutMS());
    }
    return MP_NO_ERROR;
}

tResult iPodControl::RoleSwitchRequiredAnswer(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOSwitchReqResponse);

    //tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    VARTRACE(mountPoint);

    if(mountPoint[0] == 0) {
        ETG_TRACE_ERR(("Invalid device"));
    } else {
        iAPSetWaitingforRoleSwitchResponse(mountPoint, false);
        const tBoolean deviceAvail = iAP_IsDeviceHandleAvail(mountPoint);
        VARTRACE(deviceAvail);

        if(deviceAvail) {
            ETG_TRACE_USR4(("Device already initialized"));
        } else {

            tDiPORoleStatus diPORoleStatus = DIPO_CLIENT_MODE;
            switch(diPOSwitchReqResponse) {
                case DIPO_ROLE_SWITCH_NOT_REQUIRED:
                    diPORoleStatus = DIPO_CLIENT_MODE;
                    break;
                case DIPO_ROLE_SWITCH_REQUIRED_FOR_CARPLAY:
                    diPORoleStatus = DIPO_CARPLAY_MODE;
                    break;
                case DIPO_ROLE_SWITCH_REQUIRED_FOR_NATIVE_TRANSPORT:
                    diPORoleStatus = DIPO_NATIVE_TRANSPORT_MODE;
                    break;
                case DIPO_ROLE_SWITCH_REQUIRED_FOR_CARPLAY_AND_NATIVE_TRANSPORT:
                    diPORoleStatus = DIPO_CARPLAY_NATIVE_TRANSPORT_MODE;
                    break;
                case DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE_AND_NATIVE_TRANSPORT:
                    diPORoleStatus = DIPO_CARLIFE_NATIVE_TRANSPORT_MODE;
                    break;
                default:
                    diPORoleStatus = DIPO_CLIENT_MODE;
                    ETG_TRACE_ERR(("Invalid diPOSwitchReqResponse type"));
                    break;
            }
            VARTRACE(diPORoleStatus);
            if(diPORoleStatus != DIPO_CLIENT_MODE) {
                tAllParameters parameterString;
                size_t size = sizeof(parameterString);
                ParameterROLE_SWITCH(OUT parameterString, IN size, IN mountPoint, IN diPORoleStatus);
                //SendEvent(IN ROLE_SWITCH, IN parameterString);
                //using same RoleSwitch state as on DoRoleSwitch, so message answer (NULL) is needed
                SendEventAnswerByName("ROLE_SWITCH", "iPodControl::NULL_EVENT", IN parameterString);
            } else {
                //create deviceInfoString to match function parameter syntax of SendInitInit
                tDeviceInfo deviceInfo;
                iAPGetDeviceInfo(mountPoint, deviceInfo);
                tDeviceInfoString deviceInfoString;
                DataProvider::MarshalDeviceInfo(deviceInfoString, deviceInfo);
                SendInitInit(IN deviceInfoString);
            }
        }
    }

    return MP_NO_ERROR;
}

tResult iPodControl::OnDiPORoleSwitchRequiredResult(const tDeviceID deviceID, const tDiPOSwitchReqResponse diPOSwitchReqResponse,const tAppInfo appInfo)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOSwitchReqResponse);
    tMountPoint mountPoint = {0};
        iAPGetMountPointFromDeviceID(deviceID, mountPoint);
        VARTRACE(mountPoint);
    if(diPOSwitchReqResponse == DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE_AND_NATIVE_TRANSPORT)
    {
        bool added = false;
        iAPAddAppControlInfoFromSPI(mountPoint,appInfo.ProtocolName,appInfo.BundleID,appInfo.AppName,added);
    }

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    if(LocalSPM::GetDataProvider().iPodControlIAP2CheckRoleSwitchRequiredTimeOutMS() == 0) {
        //no timeout wait for user choice
        const tUserDataType userDataType = USRDATA_ROLE_SWITCH_REQUIRED_ANSWER;
        tUserData userData = {0};
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tDiPOSwitchReqResponse_format, diPOSwitchReqResponse);

        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
        SendEvent(SIGNAL_USERDATA, IN parameterString);
    } else {
        /* Send ROLE_SWITCH_REQUIRED_ANSWER message to own SM */
        ParameterROLE_SWITCH_REQUIRED_ANSWER(OUT parameterString, IN size, IN deviceID, IN diPOSwitchReqResponse);
        SendEvent(IN ROLE_SWITCH_REQUIRED_ANSWER, IN parameterString);
    }
    return MP_NO_ERROR;
}

tBoolean iPodControl::IsBatchPlayable(const tDeviceID deviceID)
{
    ENTRY;
    VARTRACE(deviceID);
    tBoolean ret = 0;
    if(LocalSPM::GetDataProvider().iPodControlIAP2BatchPlaying()) {

        tMountPoint mountPoint = {0};
        iAPGetMountPointFromDeviceID(deviceID, mountPoint);
        VARTRACE(mountPoint);
        ret = iAP2_IsSet(mountPoint);
    }
    VARTRACE(ret);
    return ret;
}

tResult iPodControl::SendDBChangeDeviceType(const tDeviceID deviceID, const tDeviceType newDeviceType)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(newDeviceType);

    /* Send DB_CHANGE_DEVICETYPE message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_DEVICETYPE", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_DEVICETYPE(OUT parameterString, IN size, IN deviceID, IN newDeviceType);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}
tResult iPodControl::SendDBChangeDeviceTypeAndMountPoint(const tDeviceID deviceID, const tDeviceType newDeviceType, const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(newDeviceType);

    /* Send DB_CHANGE_DEVICETYPE message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_DEVICETYPE_AND_MOUNTPOINT", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_DEVICETYPE_AND_MOUNTPOINT(OUT parameterString, IN size, IN deviceID, IN newDeviceType,IN mountPoint);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);

}


tResult iPodControl::SendDBChangeDeviceName(const tDeviceID deviceID, const tDeviceName newDeviceName)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(newDeviceName);

    /* Send DB_CHANGE_DEVICENAME message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_DEVICENAME", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_DEVICENAME(OUT parameterString, IN size, IN deviceID, IN newDeviceName);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::SendDBChangeDiPOSettings(const tDeviceID deviceID, const tDiPOCaps diPOCaps, const tDiPOActive diPOActive, const tDiPOVersion diPOVersion)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(diPOCaps);
    VARTRACE(diPOActive);
    VARTRACE(diPOVersion);

    /* Send DB_CHANGE_DIPOSETTINGS message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_DIPOSETTINGS", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_DIPOSETTINGS(OUT parameterString, IN size, IN deviceID, IN diPOCaps, IN diPOActive, IN diPOVersion);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::SendDBChangeConnectionState(const tDeviceID deviceID, const tConnectionState newConnectionState)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(newConnectionState);

    /* Send DB_CHANGE_CONNECTIONSTATE message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_CONNECTIONSTATE", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_CONNECTIONSTATE(OUT parameterString, IN size, IN deviceID, IN newConnectionState);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

tResult iPodControl::SendDBChangeDeviceState(const tDeviceID deviceID, const tDeviceState newDeviceState)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(newDeviceState);

    /* Send DB_CHANGE_CONNECTIONSTATE message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_DEVICESTATE", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_DEVICESTATE(OUT parameterString, IN size, IN deviceID, IN newDeviceState);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

//CARPLAY WIFI:
tResult iPodControl::SendDBSetDeviceUUID(const tDeviceID deviceID, const tUUID deviceUUID)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(deviceUUID);

    /* Send DB_SET_DEVICEUUID message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_SET_DEVICEUUID", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_SET_DEVICEUUID(OUT parameterString, IN size, IN deviceID, IN deviceUUID);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);

}

tResult iPodControl::SendDBSetAppleDeviceMACAddress(const tDeviceID deviceID, const tMACAddress appleDeviceMACAddress)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(appleDeviceMACAddress);

    /* Send DB_SET_APPLEDEVICEMACADDRESS message to iPodControlHelperSM */
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_SET_APPLEDEVICEMACADDRESS", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    LocalSPM::GetIPODControlHelper().ParameterDB_SET_APPLEDEVICEMACADDRESS(OUT parameterString, IN size, IN deviceID, IN appleDeviceMACAddress);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);

}
tResult iPodControl::SendDBsetAppleDeviceTransportIdentifiers(const tDeviceID deviceID,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber)
{
    ENTRY;
    VARTRACE(deviceID);
    VARTRACE(appleDeviceMACAddress);
    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_SET_APPLEDEVICETRANSPORTIDENTIFIERS", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    LocalSPM::GetIPODControlHelper().ParameterDB_SET_APPLEDEVICETRANSPORTIDENTIFIERS(OUT parameterString, IN size, IN deviceID, IN appleDeviceMACAddress, IN appleDeviceUSBSerialNumber);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}
void iPodControl::ReadAuthParameterFromRootFS()
{
    ENTRY;

    //read iPodAuth parameter from /etc/pfcfg/IPOD_AUTH.cfg if available
    FILE* authfp = fopen( "/etc/pfcfg/IPOD_AUTH.cfg", "r" );
    if(authfp != NULL)
    {
        while(1)
        {
          /* read one line */
          tGeneralString buffer;
          char *cerr = fgets(buffer, sizeof(tGeneralString), authfp);
          if (cerr != buffer) {
            break;
          }
          char *cEqualSign;

          //Go for next line as the current line is commented.
          if(buffer[0] == '#')
          {
            continue;
          }

          /* look for the 'IPOD_AUTH_DEV_NAME' and read the item */
          cEqualSign = strstr(buffer, "IPOD_AUTH_DEV_NAME");
          if (cEqualSign) {
            tGeneralString iPodControlAuthDevicename = {0};
            if(sscanf(cEqualSign, "%*s %s", &iPodControlAuthDevicename ) == 1)
            {
              VARTRACE(iPodControlAuthDevicename);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthDevicename = iPodControlAuthDevicename;
            }
            continue;
          }

          /* look for the 'IPOD_AUTH_IOCTL_REG' and read the item */
          cEqualSign = strstr(buffer, "IPOD_AUTH_IOCTL_REG");
          if (cEqualSign) {
            tGeneralString iPodControlRegAddr = {0};
            if(sscanf(cEqualSign, "%*s %s", &iPodControlRegAddr ) == 1)
            {
              VARTRACE(iPodControlRegAddr);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthIoctlRegAddr = iPodControlRegAddr;
            }
            continue;
          }

          //VARTRACE(buffer);
          /* look for the 'IPOD_AUTH_GPIO_RESET' and read the item */
          cEqualSign = strstr(buffer, "IPOD_AUTH_GPIO_RESET");
          if (cEqualSign) {
            tGeneralString iPodAuthGPIOReset = {0};
            if(sscanf(cEqualSign, "%*s %s", &iPodAuthGPIOReset ) == 1)
            {
              VARTRACE(iPodAuthGPIOReset);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthGPIOReset = iPodAuthGPIOReset;
            }
            continue;
          }

          cEqualSign = strstr(buffer, "IPOD_AUTH_GPIO_READY");
          if (cEqualSign) {
            tGeneralString iPodAuthGPIOReady = {0};
            if(sscanf(cEqualSign, "%*s %s", &iPodAuthGPIOReady ) == 1)
            {
              VARTRACE(iPodAuthGPIOReady);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthGPIOReady = iPodAuthGPIOReady;
            }
            continue;
          }

          cEqualSign = strstr(buffer, "IPOD_AUTH_DEV_COM_SHORT_WAIT");
          if (cEqualSign) {
            int iPodAuthShortWait = 0;
            if(sscanf(cEqualSign, "%*s %d", &iPodAuthShortWait ) == 1)
            {
              VARTRACE(iPodAuthShortWait);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthShortWait = iPodAuthShortWait;
            }
            continue;
          }

          cEqualSign = strstr(buffer, "IPOD_AUTH_DEV_COM_WAIT");
          if (cEqualSign) {
            int iPodAuthWait = 0;
            if(sscanf(cEqualSign, "%*s %d", &iPodAuthWait ) == 1)
            {
              VARTRACE(iPodAuthWait);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthWait = iPodAuthWait;
            }
            continue;
          }

          cEqualSign = strstr(buffer, "IPOD_AUTH_DEV_COM_LONG_WAIT");
          if (cEqualSign) {
            int iPodAuthLongWait = 0;
            if(sscanf(cEqualSign, "%*s %d", &iPodAuthLongWait ) == 1)
            {
              VARTRACE(iPodAuthLongWait);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthLongWait = iPodAuthLongWait;
            }
           // continue;
           break;
          }
#if 0
          cEqualSign = strstr(buffer, "IPOD_AUTH_CP_AUTODETECT");
          if (cEqualSign) {
            tGeneralString iPodAuthCpAutodetect = {0};
            if(sscanf(cEqualSign, "%*s %s", &iPodAuthCpAutodetect ) == 1)
            {
              VARTRACE(iPodAuthCpAutodetect);
              LocalSPM::GetDataProvider().iPodControlIAP2AuthAutoDetect = iPodAuthCpAutodetect;
            }
            break;
          }
#endif
        }
        fclose(authfp);
    }
}

void iPodControl::Test(int value)
{
    ENTRY;
    VARTRACE(value);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN value, IN USRDATA_TEST, IN "");
    SendEvent(SIGNAL_USERDATA, IN parameterString);
    sleep(1);
}

void iPodControl::UpdateAppControl(const tDeviceID deviceID, const tAppControlUpdateType updateType)
{
    ENTRY_INTERNAL;

    // Fix for NCG3D-196687: Skip updates for CP USB device ID when CPW is active for same device
    // By default Signal user data to App control for all device types
    bool bSignalUserData = true;

    // Power update will come for only USB mount point hence do not skip that
    if( APPCONTROL_UPDATE_DIPO_POWER != updateType )
    {
        tMountPoint mountPoint = {0};
        iAPGetMountPointFromDeviceID(deviceID, mountPoint);
        if( DCT_USB == iAPGetConnectionType(mountPoint) )
        {
            tBoolean isCPWActive = false;
            GetIsCPWActiveForUSBMountPoint(mountPoint, isCPWActive);
            VARTRACE(isCPWActive);

            // If CPW is active for the USB mount point, skip sending duplicate update for USB device ID
            if( true == isCPWActive )
            {
                bSignalUserData = false;
            }
        }
    }

    if( true == bSignalUserData )
    {
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        tUserData userData = {0};
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, tAppControlUpdateType_format, updateType);

        ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN USRDATA_UPDATEAPPCONTROL, IN userData);
        SendEvent(SIGNAL_USERDATA, IN parameterString);
    }

}

bool iPodControl::IsActiveAsCarPlay(const tDeviceName deviceName)
{
    ENTRY;
    bool ret = false;
    VARTRACE(deviceName);

    tMountPoint mountPoint = {0};
    if(deviceName[0] != 0 && iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        tDeviceInfo deviceInfo;
        iAPGetDeviceInfo(mountPoint, deviceInfo);
        VARTRACE(deviceInfo.deviceID);
        VARTRACE(deviceInfo.deviceName);
        ret = !strcmp(deviceName, deviceInfo.deviceName);
    }
    return ret;
}

tResult iPodControl::SendSetRepeatMode(const tMountPoint mountPoint, tRepeatMode repeatMode)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    tDeviceID deviceID = iAPGetDeviceID(mountPoint);

    ret = ParameterSET_REPEAT_MODE(OUT parameterString,
            IN size,
            IN DTY_IPOD,
            IN deviceID,
            IN mountPoint,
            IN repeatMode);
    if (MP_NO_ERROR != ret)
    {
        ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        parameterString[0] = '\0';
    }
    return SendEvent(SET_REPEAT_MODE ,IN parameterString);
}

tResult iPodControl::SendDBChangeNowPlayingListAvailable(const tMountPoint mountPoint, const int queueListID, tNowPlayingListAvailable clearNowPlayingList)
{
    ENTRY;
    VARTRACE(queueListID);
    VARTRACE(clearNowPlayingList);
    VARTRACE(mountPoint);

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    tNowPlayingListAvailable isNowPlayingListAvailable = false;

    char messageString[64];
    strncpy_r(messageString, "iPodControlHelperSM::DB_CHANGE_NOWPLAYINGLIST_AVAILABLE", sizeof(messageString) - 1);
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    if(clearNowPlayingList == true)
    {
        ETG_TRACE_USR3(("Forcefully clearing NowPlayingListAvailable in DB"));
    }
    else
    {
        bool isQueueListValid = false;
        const int nowplayingTrackCount = iAPGetNowPlayingTrackCount(mountPoint);
        VARTRACE(nowplayingTrackCount);
        tFileXferBuf * pQueueListBuf = (tFileXferBuf *)iAPGetQueueListBuffer(mountPoint);
        if(queueListID > 0 && pQueueListBuf) {
            VARTRACE(pQueueListBuf->rxLen);
            isQueueListValid = pQueueListBuf->rxLen == (8*nowplayingTrackCount);
        }
        if(isQueueListValid) {
            isNowPlayingListAvailable = true;
        }
    }

    VARTRACE(isNowPlayingListAvailable);

    LocalSPM::GetIPODControlHelper().ParameterDB_CHANGE_NOWPLAYINGLIST_AVAILABLE(OUT parameterString, IN size, IN deviceID, IN isNowPlayingListAvailable);
    return Dispatcher::GetInstance().SendMessage(IN messageString, IN parameterString);
}

void iPodControl::AccessoryWiFiCredentials(const tDeviceID deviceId,tWiFiAPCredentials wifiCredentials)
{
    ENTRY

    VARTRACE(wifiCredentials.channelNumber);
    VARTRACE(wifiCredentials.passPhrase);
    VARTRACE(wifiCredentials.ssid);
    VARTRACE(wifiCredentials.securityType);

    Locker locker(&m_AccessoryWiFiCredentialsMutex);
    m_AccessoryWiFiCredentials = wifiCredentials;


    const tUserDataType userDataType = USRDATA_ACC_WIFI_CREDENTIALS_RESULT;
    tUserData userData = {0}; //no data, taken from config

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceId , IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

}


void iPodControl::InitAccessoryWifiCredentials()
{
    ENTRY;
    const char *passPhrase = "12345678";
    const char *SSID = "My_VEHICLE_1234";

    m_AccessoryWiFiCredentials.channelNumber = 36;
    m_AccessoryWiFiCredentials.securityType  = WIFI_SECURITY_WPA_PERSONAL;
    strncpy_r(m_AccessoryWiFiCredentials.passPhrase,passPhrase,sizeof(m_AccessoryWiFiCredentials.passPhrase));
    strncpy_r(m_AccessoryWiFiCredentials.ssid,SSID,sizeof(m_AccessoryWiFiCredentials.ssid));

}

//CMG3G-14540. BTLimitationmode changes for CarPlay wireless.
void iPodControl::SendSetBTLimitationMode(tMACAddress btMacAddress, tBTLimitationActionState btLimitationActionState)
{
    ENTRY;
    //ToDO
    // Compare the BTLimitationMode value for the deviceID corresponding to btMacAddress stored in handleMap. If no data in map, consider this a first update
    // and trigger reconnection based on Wireless Transport component enabled / disabled.

    //1. Get Mount Point for btMacAddress, DTY_IPHONE & DCT_BLUETOOTH
    //2. Check btLimitationActionState value in handle map. If it is invalid, set the value & trigger

    if(BTLIMITATION_ACTION_STATE_INVALID == btLimitationActionState)
        return;

    VARTRACE(btMacAddress);
    VARTRACE(btLimitationActionState);

    bool foundMountPoint = false;
    tMountPoint mountPoint;
    tDeviceID deviceID = DEVICE_ID_NOT_SET;
    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfoList;
    memset(mountPoint,0,sizeof(mountPoint));
    LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(numberOfDevices,deviceInfoList,true);
    for(unsigned int i = 0; i<deviceInfoList.size(); i++)
    {
        if(!strcmp(deviceInfoList[i].appleDeviceMACAddress,btMacAddress)
                && (DTY_IPHONE == deviceInfoList[i].deviceType)
                && (DCT_BLUETOOTH == deviceInfoList[i].connectionType))
        {
            deviceID = deviceInfoList[i].deviceID;
            strncpy_r(mountPoint,deviceInfoList[i].mountPoint,sizeof(mountPoint));
            foundMountPoint = true;
            break;
        }
    }

    VARTRACE(mountPoint);
    VARTRACE(deviceID);
    VARTRACE(foundMountPoint);
    tBTLimitationActionState lastBTLimitationActionState;
    GetBTLimitationMode(mountPoint, lastBTLimitationActionState);
    //Get existing status of wireless transport component.
    if(foundMountPoint)
    {
        if(btLimitationActionState != lastBTLimitationActionState)
        {
            VARTRACE(btLimitationActionState);
            SetBTLimitationMode(mountPoint,btLimitationActionState);

            // Fix for NCG3D-162752: Not able to establish CPW after enabling the functionality from MD
            // 1) When SPP is connected, instead of triggering iAP BT session without wireless transport component,
            // initiate iAP BT session with wireless transport component.
            // 2) Ignore the BT Limitation mode updates and prevent Reconnection in MP on a BT Limitation mode change from NOT PREPARED to PREPARED
            // - to avoid iAP reconnection triggers from MP to ADIT.
            // TO DO: Clean up code wrt storing and retrieval of BT Limitation mode related parameters
            // Note: To be done after system testing of all Carplay related use cases and confirming that BT Limitation mode is not required.
            // ReconnectIAPBTOnLimitationChange(mountPoint, deviceID);
            // End of fix
        }
    }
}


tResult iPodControl::OnCBIAP2_PowerUpdate(iAP2Device_t* iap2Device, iAP2PowerUpdateParameter* cup, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!cup) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPOPower power = iAP2GetPowerUpdate(mountPoint);

    if(cup->iAP2BatteryChargeLevel_count && cup->iAP2BatteryChargeLevel) {
        VARTRACE(cup->iAP2BatteryChargeLevel[0]);
        power.BatteryChargeLevel=(tU16)cup->iAP2BatteryChargeLevel[0];
    }

    if(cup->iAP2AccessoryPowerMode_count && cup->iAP2AccessoryPowerMode) {
        VARTRACE(cup->iAP2AccessoryPowerMode[0]);
        switch(cup->iAP2AccessoryPowerMode[0]) {
            case IAP2_ACCESSORY_POWER_NO_POWER:
                power.AccessoryPowerMode = DIPO_ACCESSORY_MODE_RESERVED;
                break;
            case IAP2_ACCESSORY_POWER_LOW_POWER:
                power.AccessoryPowerMode = DIPO_ACCESSORY_MODE_LOW_POWER_MODE;
                break;
            case IAP2_ACCESSORY_POWER_HIGH_POWER:
                power.AccessoryPowerMode = DIPO_ACCESSORY_MODE_INTERMITTENT_HIGH_POWER_MODE;
                break;
            default:
                break;
        }
    }

    if(cup->iAP2MaximumCurrentDrawnFromAccessory_count && cup->iAP2MaximumCurrentDrawnFromAccessory) {
        VARTRACE(cup->iAP2MaximumCurrentDrawnFromAccessory[0]);
        power.MaximumCurrentDrawnFromAccessory=(tU16)cup->iAP2MaximumCurrentDrawnFromAccessory[0];
    }

    if(cup->iAP2DeviceBatteryWillChargeIfPowerIsPresent_count && cup->iAP2DeviceBatteryWillChargeIfPowerIsPresent) {
        VARTRACE(cup->iAP2DeviceBatteryWillChargeIfPowerIsPresent[0]);
        power.DeviceBatteryWillChargeIfPowerIsPresent = (tBoolean)cup->iAP2DeviceBatteryWillChargeIfPowerIsPresent[0];
    }

    if(cup->iAP2BatteryChargingState_count && cup->iAP2BatteryChargingState) {
        VARTRACE(cup->iAP2BatteryChargingState[0]);
        switch(cup->iAP2BatteryChargingState[0]) {
            case IAP2_BATTERY_CHARGING_STATE_DISABLED:
                power.BatteryChargingState = DIPO_BATTERY_CHARGE_STATE_DISABLED;
                break;
            case IAP2_BATTERY_CHARGING_STATE_CHARGING:
                power.BatteryChargingState = DIPO_BATTERY_CHARGE_STATE_CHARGING;
                break;
            case IAP2_BATTERY_CHARGING_STATE_CHARGED_NEW:
                power.BatteryChargingState = DIPO_BATTERY_CHARGE_STATE_CHARGED;
                break;
            default:
                break;
        }
    }

    if(cup->iAP2IsExternalChargerConnected_count && cup->iAP2IsExternalChargerConnected) {
        VARTRACE(cup->iAP2IsExternalChargerConnected[0]);
        power.IsExternalChargerConnected = (tBoolean)cup->iAP2IsExternalChargerConnected[0];
    }

    //store current state
    iAP2SetPowerUpdate(mountPoint, power);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_POWER);
    return ret;
}

tResult iPodControl::GetDiPOPower(tDeviceID &deviceID, tDiPOPower &power)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;
    InitDiPOPower(power);

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        power = iAP2GetPowerUpdate(mountPoint);
        VARTRACE(power);
    }
    return ret;
}
tBool iPodControl::bCheckUdevForExpectedHub(unsigned int uProductID, unsigned int uVendorID, const char *f_strProduct, const char *f_strBcdDevice, tBool bStrBcdDeviceNotEqualThis /*= FALSE*/)
{
    ENTRY
    //Buidl based on this information
    //-------------------------
    //MicroChipMitsumiPort2ES2
    //-------------------------

    /*
       udevadm monitor --property
       gives this info if you connect the hub

        DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1.3
        DEVTYPE=usb_device
        ID_BUS=usb
        ID_MODEL=Bridge_device
        ID_MODEL_ENC=Bridge\x20device
        ID_MODEL_ID=2530
        ID_REVISION=0128 <----------------------------------------BcdDevice
        ID_SERIAL=0424_Bridge_device
        ID_VENDOR=0424
        ID_VENDOR_ENC=0424
        ID_VENDOR_FROM_DATABASE=Standard Microsystems Corp.
        ID_VENDOR_ID=0424
        MAJOR=189
        MINOR=6
        PRODUCT=424/2530/128
        SEQNUM=1829
        SUBSYSTEM=usb
        TYPE=0/0/0
        USEC_INITIALIZED=692507521
    */

    //-------------------------
    //MicroChipMitsumiPort2ES4
    //-------------------------

    /*
        udevadm monitor --property
        gives this info if you connect the hub

        //gained with udevadm monitor --property
        DEVPATH=/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1.3
        DEVTYPE=usb_device
        ID_BUS=usb
        ID_MODEL=Bridge_device
        ID_MODEL_ENC=Bridge\x20device
        ID_MODEL_ID=2530
        ID_REVISION=0131<----------------------------------------BcdDevice
        ID_SERIAL=0424_Bridge_device
        ID_VENDOR=0424
        ID_VENDOR_ENC=0424
        ID_VENDOR_FROM_DATABASE=Standard Microsystems Corp.
        ID_VENDOR_ID=0424
        MAJOR=189
        MINOR=4
        PRODUCT=424/2530/131
        SEQNUM=1821
        SUBSYSTEM=usb
        TYPE=0/0/0
        USEC_INITIALIZED=594128896
    */
    /*
    to use c commands like udev_enumerate_add_match_subsystem and others
    this command is helpful to get hint for correct parameters

     udevadm info -a -p /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1.3

    Udevadm info starts with the device specified by the devpath and then
    walks up the chain of parent devices. It prints for every device
    found, all possible attributes in the udev rules key format.
    A rule to match, can be composed by the attributes of the device
    and the attributes from one single parent device.

        looking at device '/devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1.3':
        KERNEL=="1-1.3"
        SUBSYSTEM=="usb" <---------------------------------------subysystem usb
        DRIVER=="usb"
        ATTR{authorized}=="1"
        ATTR{avoid_reset_quirk}=="0"
        ATTR{bConfigurationValue}=="1"
        ATTR{bDeviceClass}=="00"
        ATTR{bDeviceProtocol}=="00"
        ATTR{bDeviceSubClass}=="00"
        ATTR{bMaxPacketSize0}=="64"
        ATTR{bMaxPower}=="0mA"
        ATTR{bNumConfigurations}=="1"
        ATTR{bNumInterfaces}==" 1"
        ATTR{bcdDevice}=="0131"<----------------------------------bcd device
        ATTR{bmAttributes}=="c0"
        ATTR{busnum}=="1"
        ATTR{configuration}==""
        ATTR{devnum}=="3"
        ATTR{devpath}=="1.3"
        ATTR{idProduct}=="2530" <----------------------------------product of hub
        ATTR{idVendor}=="0424" <----------------------------------vendor of hub
        ATTR{ltm_capable}=="no"
        ATTR{maxchild}=="0"
        ATTR{product}=="Bridge device"<----------------------------bridge device
        ATTR{quirks}=="0x0"
        ATTR{removable}=="fixed"
        ATTR{speed}=="480"
        ATTR{urbnum}=="10"
        ATTR{vbus_auto}=="enabled"
        ATTR{version}==" 2.00"

    */

    tBool bFound = FALSE;
    tBool bContinue = TRUE;

    //First WRONG PARAMETER check
    if(NULL == f_strProduct)
    {
        ETG_TRACE_USR2(("[ERROR]:iPodControl::bCheckUdevForExpectedHub() - f_strProduct is NULL"));
        return bFound;
    }
    if(NULL == f_strBcdDevice)
    {
        ETG_TRACE_USR2(("[ERROR]:iPodControl::bCheckUdevForExpectedHub() - f_strBcdDevice is NULL"));
        return bFound;
    }




    struct udev_enumerate* enumerate = NULL;
    struct udev* pUdev = NULL;


    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub() - bCheckForConnectedHUB "));
    int ret;

    pUdev = udev_new();
    if(!pUdev) {
        ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub(): (udev_new() returned NULL"));
        bContinue = FALSE;
    }

    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): - udev_new() ok (bContinue:0x%x)",bContinue));
    if(bContinue)
    {
        enumerate = udev_enumerate_new(pUdev);
        if (enumerate == NULL)
        {
            ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub():udev_enumerate_new() returned NULL"));
            bContinue = FALSE;
        }
    }

    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_new(pUdev) ok (bContinue:0x%x)",bContinue));
    //search subsystem 'usb'
    if(bContinue)
    {
        int ret = udev_enumerate_add_match_subsystem(enumerate, "usb");
        if(0 != ret)
        {
            ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub():udev_enumerate_add_match_subsystem failed: %d", ret));
            bContinue = FALSE;
        }
    }

    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_add_match_subsystem(enumerate, usb) ok (bContinue:0x%x)",bContinue));
    //search for  'idProduct'
    if(bContinue)
    {
        char l_idProduct[5];
        snprintf(l_idProduct, sizeof(l_idProduct),"%04x", uProductID);
        ret = udev_enumerate_add_match_sysattr(enumerate, (const char*)"idProduct", (const char*)l_idProduct);
        if(ret != 0)
        {
            ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub():udev_enumerate_add_match_sysattr failed: %d l_idProduct:%s", ret,l_idProduct));
            bContinue = FALSE;
        }
        ETG_TRACE_FATAL(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_add_match_sysattr: l_idProduct: %s ",l_idProduct));
    }


    //search for  'idVendor'
    if(bContinue)
    {
        char l_idVendor[5];
        snprintf(l_idVendor, sizeof(l_idVendor),"%04x", uVendorID);
        ret = udev_enumerate_add_match_sysattr(enumerate, (const char*)"idVendor", (const char*)l_idVendor);
        if(ret != 0)
        {
            ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub():udev_enumerate_add_match_sysattr failed: %d l_idVendor:%s", ret,l_idVendor));
            bContinue = FALSE;
        }
        ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_add_match_sysattr: l_idVendor: %s ",l_idVendor));
    }


    //search if device can be found with above search keys
    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_scan_devices(enumerate) now (bContinue:0x%x)",bContinue));
    if(bContinue)
    {
        ret = udev_enumerate_scan_devices(enumerate);
        if(ret != 0)
        {
            ETG_TRACE_ERR(("iPodControl::bCheckUdevForExpectedHub():udev_enumerate_scan_devices failed: %d", ret));
            bContinue = FALSE;
        }
    }

    ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub(): -  udev_enumerate_get_list_entry(enumerate) now (bContinue:0x%x)",bContinue));
    if(bContinue)
    {
        struct udev_list_entry *pDevices = udev_enumerate_get_list_entry(enumerate);
        struct udev_list_entry *pDevListEntry;
        udev_list_entry_foreach(pDevListEntry, pDevices)
        {
            const char *pPath = udev_list_entry_get_name(pDevListEntry);
            struct udev_device *pDevice = udev_device_new_from_syspath(pUdev, pPath);
            if(!pDevice)
            {
                ETG_TRACE_ERR(("udev_device_new_from_syspath() returned NULL"));
                bContinue = FALSE;
            }
            else
            {
                const char *pSysPath = udev_device_get_syspath (pDevice);

               // e.g. /devices/soc0/soc.0/2100000.aips-bus/2184000.usb/ci_hdrc.0/usb1/1-1/1-1.3
               ETG_TRACE_USR3(("-------------------------------------------------------------------------------------------------"));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_syspath: %s", pSysPath));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_sysattr_value[idVendor]:  %s", udev_device_get_sysattr_value (pDevice, "idVendor")));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_sysattr_value[idProduct]: %s", udev_device_get_sysattr_value (pDevice, "idProduct")));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_sysattr_value[product]:   %s", udev_device_get_sysattr_value (pDevice, "product")));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_sysattr_value[bcdDevice]: %s", udev_device_get_sysattr_value (pDevice, "bcdDevice")));
               ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub():found this: udev_device_get_sysattr_value[bcdDevice]: expected Equal or not: bStrBcdDeviceNotEqualThis=0x%x", bStrBcdDeviceNotEqualThis));
               ETG_TRACE_USR3(("-------------------------------------------------------------------------------------------------"));

               const char  *l_pStrBcdDevice = udev_device_get_sysattr_value (pDevice, "bcdDevice");
               const char  *l_pStrProduct   = udev_device_get_sysattr_value (pDevice, "product");

               //check for the difference (idVendor and idProduct already approved since yoused to find the )
               unsigned int uDifference = 0;

               //search if it is a bridge of this hub - (usable to trigger role switch for carplay)
               if(l_pStrProduct == NULL)
               {
                   ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrProduct NULL"));
                   uDifference++;
               }
               else
               {
                   if(0 != strcmp(f_strProduct,l_pStrProduct))
                   {
                       ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrProduct different"));
                       uDifference++;
                   }
                   else
                   {
                       ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrProduct equal"));
                   }
               }

               //check if expected BcdDevice
               if(l_pStrBcdDevice == NULL)
               {
                   ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrBcdDevice NULL"));
                   uDifference++;
               }
               else
               {

                   if(TRUE == bStrBcdDeviceNotEqualThis)
                   {
                       ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrBcdDevice expected equal %s",f_strBcdDevice));
                       if(0 != strcmp(f_strBcdDevice,l_pStrBcdDevice))
                       {
                           ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrBcdDevice related difference expected equal)"));
                           uDifference++;
                       }
                   }
                   else
                   {
                       ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrBcdDevice expected unequal  %s",f_strBcdDevice));
                       if(0 == strcmp(f_strBcdDevice,l_pStrBcdDevice))
                       {
                           ETG_TRACE_USR3(("iPodControl::bCheckUdevForExpectedHub(): l_pStrBcdDevice related difference (expected unequal)"));
                           uDifference++;
                       }
                   }

               }

               //-------------
               //result
               //-------------
               if(uDifference == 0)
               {
                   bFound = TRUE;
               }
               ETG_TRACE_USR2(("iPodControl::bCheckUdevForExpectedHub():uDifference: %d bFound=0x%x",uDifference,bFound));


               //here the check should be done - now

               udev_device_unref(pDevice);
            }
        }
    }

    if(enumerate)
    {
        udev_enumerate_unref(enumerate);
    }
    if(pUdev)
    {
        udev_unref(pUdev);
    }



    return bFound;
}


#ifdef IAP2_DEFAULT_ROUTEGUIDANCEDISPLAYCOMPONENT
tResult iPodControl::OnCBIAP2_RouteGuidanceUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceUpdateParameter* routeGuidanceUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!routeGuidanceUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPORouteGuidanceUpdate RGUpdate = iAP2GetRGUpdate(mountPoint);


    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0]); //ToDo: This needs to be changed to a vector/array/pointer to hold all items.
        RGUpdate.RouteGuidanceDisplayComponentID=(tU16)routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0];
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceState_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceState) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceState[0]);
        switch(routeGuidanceUpdateParameter->iAP2RouteGuidanceState[0]) {
            case IAP2_ROUTE_GUIDANCE_STATE_NOT_SET:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_SET:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_SET;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_ARRIVED:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_ARRIVED;
                break;
                case IAP2_ROUTE_GUIDANCE_STATE_LOADING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOADING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_LOCATING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOCATING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_REROUTING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_REROUTING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_PROCEED_TO_ROUTE:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_PROCEEDTOROUTE;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2ManeuverState_count && routeGuidanceUpdateParameter->iAP2ManeuverState) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2ManeuverState[0]);
        switch(routeGuidanceUpdateParameter->iAP2ManeuverState[0]) {
            case IAP2_MANEUVER_STATE_CONTINUE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_CONTINUE;
                break;
            case IAP2_MANEUVER_STATE_INITIAL:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_INITIAL;
                break;
            case IAP2_MANEUVER_STATE_PREPARE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_PREPARE;
                break;
            case IAP2_MANEUVER_STATE_EXECUTE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_EXECUTE;
                break;
            default:
                break;
        }
    }
    //for UTF-8 ManeuverDescription iAP2CurrentRoadName
    if(routeGuidanceUpdateParameter->iAP2CurrentRoadName_count && routeGuidanceUpdateParameter->iAP2CurrentRoadName) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2CurrentRoadName[0]);
        strncpy_r(RGUpdate.CurrentRoadName, (char *)routeGuidanceUpdateParameter->iAP2CurrentRoadName[0], sizeof(tGeneralString));
    }
    if(routeGuidanceUpdateParameter->iAP2DestinationName_count && routeGuidanceUpdateParameter->iAP2DestinationName) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DestinationName[0]);
        strncpy_r(RGUpdate.DestinationName, (char *)routeGuidanceUpdateParameter->iAP2DestinationName[0], sizeof(tGeneralString));
    }
    if(routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival_count && routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival[0]);
        RGUpdate.EstimatedTimeOfArrival=(tU64)routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival[0];
    }
    if(routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination_count && routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination[0]);
        RGUpdate.TimeRemainingToDestination=(tU64)routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination[0];
    }
    if(routeGuidanceUpdateParameter->iAP2DistanceRemaining_count && routeGuidanceUpdateParameter->iAP2DistanceRemaining) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemaining[0]);
        RGUpdate.DistanceRemaining=(tU32)routeGuidanceUpdateParameter->iAP2DistanceRemaining[0];
    }
    if(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr_count && routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr[0]);
        strncpy_r(RGUpdate.DistanceRemainingDisplayStr, (char *)routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits_count && routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits[0]);
        switch(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver[0]);
        RGUpdate.DistanceToNextManeuver=(tU32)routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver[0];
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr[0]);
        strncpy_r(RGUpdate.DistanceToNextManeuverDisplayStr, (char *)routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits[0]);
        switch(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobLength);
        tU16 maneuverListSize = (routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobLength)/2;     // blob is array of 1 byte data, we need to convert to 2 byte data.
        VARTRACE(maneuverListSize);
        vector<tU16> maneuverList;
        tU8 *ptrBlobData = routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobData;
            for(tU16 i = 0; i < maneuverListSize; ++i)
            {
                tU16 data = 0;
                data = ptrBlobData[i];
                data = data<<8;
                data |= ptrBlobData[i+1];
                maneuverList.push_back(data);
                VARTRACE(data);

            }
            RGUpdate.RouteGuidanceManeuverCurrentList = maneuverList;
        }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount[0]);
        RGUpdate.RouteGuidanceManeuverCount=(tU16)routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount[0];
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp) {
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp[0]);
        RGUpdate.RouteGuidanceVisibleInApp = (tBoolean)routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp[0];
    }

    if(routeGuidanceUpdateParameter->iAP2SourceName_count && routeGuidanceUpdateParameter->iAP2SourceName){
        VARTRACE(routeGuidanceUpdateParameter->iAP2SourceName[0]);
        strncpy_r(RGUpdate.SourceName, (char *)routeGuidanceUpdateParameter->iAP2SourceName[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance_count && routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance){
        VARTRACE(routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance[0]);
        RGUpdate.SourceSupportsRouteGuidance = (tBoolean)routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance[0];
    }

    //store current state
    iAP2SetRGUpdate(mountPoint, RGUpdate);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE);
    return ret;
}
#else
tResult iPodControl::OnCBIAP2_RouteGuidanceUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceUpdateParameter* routeGuidanceUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!routeGuidanceUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPORouteGuidanceUpdate RGUpdate;
    InitDiPORouteGuidanceUpdate(RGUpdate);

    tDiPORGDisplayComponentList RouteGuidanceDisplayComponentIDs;
    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID) {
        for(tU16 i = 0; i < routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count; ++i)
        {
            VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[i]);
            RGUpdate.RouteGuidanceDisplayComponentIDs.push_back(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[i]);
        }
        RGUpdate.RouteGuidanceDisplayComponentIDsAvailable = TRUE;
        //VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0]); //ToDo: This needs to be changed to a vector/array/pointer to hold all items.
        //RGUpdate.RouteGuidanceDisplayComponentID=(tU16)routeGuidanceUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0];
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceState_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceState) {
        RGUpdate.RouteGuidanceStateAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceState[0]);
        switch(routeGuidanceUpdateParameter->iAP2RouteGuidanceState[0]) {
            case IAP2_ROUTE_GUIDANCE_STATE_NOT_SET:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_SET:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_SET;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_ARRIVED:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_ARRIVED;
                break;
                case IAP2_ROUTE_GUIDANCE_STATE_LOADING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOADING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_LOCATING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOCATING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_REROUTING:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_REROUTING;
                break;
            case IAP2_ROUTE_GUIDANCE_STATE_PROCEED_TO_ROUTE:
                RGUpdate.RouteGuidanceState = DIPO_ROUTE_GUIDANCE_STATE_ROUTE_PROCEEDTOROUTE;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2ManeuverState_count && routeGuidanceUpdateParameter->iAP2ManeuverState) {
        RGUpdate.ManeuverStateAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2ManeuverState[0]);
        switch(routeGuidanceUpdateParameter->iAP2ManeuverState[0]) {
            case IAP2_MANEUVER_STATE_CONTINUE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_CONTINUE;
                break;
            case IAP2_MANEUVER_STATE_INITIAL:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_INITIAL;
                break;
            case IAP2_MANEUVER_STATE_PREPARE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_PREPARE;
                break;
            case IAP2_MANEUVER_STATE_EXECUTE:
                RGUpdate.ManeuverState = DIPO_MANEUVER_STATE_EXECUTE;
                break;
            default:
                break;
        }
    }
    //for UTF-8 ManeuverDescription iAP2CurrentRoadName
    if(routeGuidanceUpdateParameter->iAP2CurrentRoadName_count && routeGuidanceUpdateParameter->iAP2CurrentRoadName) {
        RGUpdate.CurrentRoadNameAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2CurrentRoadName[0]);
        strncpy_r(RGUpdate.CurrentRoadName, (char *)routeGuidanceUpdateParameter->iAP2CurrentRoadName[0], sizeof(tGeneralString));
    }
    if(routeGuidanceUpdateParameter->iAP2DestinationName_count && routeGuidanceUpdateParameter->iAP2DestinationName) {
        RGUpdate.DestinationNameAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DestinationName[0]);
        strncpy_r(RGUpdate.DestinationName, (char *)routeGuidanceUpdateParameter->iAP2DestinationName[0], sizeof(tGeneralString));
    }
    if(routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival_count && routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival) {
        RGUpdate.EstimatedTimeOfArrivalAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival[0]);
        RGUpdate.EstimatedTimeOfArrival=(tU64)routeGuidanceUpdateParameter->iAP2EstimatedTimeOfArrival[0];
    }
    if(routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination_count && routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination) {
        RGUpdate.TimeRemainingToDestinationAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination[0]);
        RGUpdate.TimeRemainingToDestination=(tU64)routeGuidanceUpdateParameter->iAP2TimeRemainingToDestination[0];
    }
    if(routeGuidanceUpdateParameter->iAP2DistanceRemaining_count && routeGuidanceUpdateParameter->iAP2DistanceRemaining) {
        RGUpdate.DistanceRemainingAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemaining[0]);
        RGUpdate.DistanceRemaining=(tU32)routeGuidanceUpdateParameter->iAP2DistanceRemaining[0];
    }
    if(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr_count && routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr) {
        RGUpdate.DistanceRemainingDisplayStrAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr[0]);
        strncpy_r(RGUpdate.DistanceRemainingDisplayStr, (char *)routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits_count && routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits) {
        RGUpdate.DistanceRemainingDisplayUnitsAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits[0]);
        switch(routeGuidanceUpdateParameter->iAP2DistanceRemainingDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGUpdate.DistanceRemainingDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver) {
        RGUpdate.DistanceToNextManeuverAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver[0]);
        RGUpdate.DistanceToNextManeuver=(tU32)routeGuidanceUpdateParameter->iAP2DistanceToNextManeuver[0];
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr) {
        RGUpdate.DistanceToNextManeuverDisplayStrAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr[0]);
        strncpy_r(RGUpdate.DistanceToNextManeuverDisplayStr, (char *)routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits_count && routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits) {
        RGUpdate.DistanceToNextManeuverDisplayUnitsAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits[0]);
        switch(routeGuidanceUpdateParameter->iAP2DistanceToNextManeuverDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGUpdate.DistanceToNextManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList) {
        RGUpdate.RouteGuidanceManeuverCurrentListAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobLength);
        tU16 maneuverListSize = (routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobLength)/2;     // blob is array of 1 byte data, we need to convert to 2 byte data.
        VARTRACE(maneuverListSize);
        vector<tU16> maneuverList;
        tU8 *ptrBlobData = routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCurrentList->iAP2BlobData;
            for(tU16 i = 0; i < maneuverListSize; i++)
            {
                tU16 data = 0;
                data = ptrBlobData[i];
                data = data<<8;
                data |= ptrBlobData[i+1];
                maneuverList.push_back(data);
                VARTRACE(data);

            }
            RGUpdate.RouteGuidanceManeuverCurrentList = maneuverList;
        }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount) {
        RGUpdate.RouteGuidanceManeuverCountAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount[0]);
        RGUpdate.RouteGuidanceManeuverCount=(tU16)routeGuidanceUpdateParameter->iAP2RouteGuidanceManeuverCount[0];
    }

    if(routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp_count && routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp) {
        RGUpdate.RouteGuidanceVisibleInAppAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp[0]);
        RGUpdate.RouteGuidanceVisibleInApp = (tBoolean)routeGuidanceUpdateParameter->iAP2RouteGuidanceVisiblelnApp[0];
    }

    if(routeGuidanceUpdateParameter->iAP2SourceName_count && routeGuidanceUpdateParameter->iAP2SourceName){
        RGUpdate.SourceNameAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2SourceName[0]);
        strncpy_r(RGUpdate.SourceName, (char *)routeGuidanceUpdateParameter->iAP2SourceName[0], sizeof(tGeneralString));
    }

    if(routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance_count && routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance){
        RGUpdate.SourceSupportsRouteGuidanceAvailable = TRUE;
        VARTRACE(routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance[0]);
        RGUpdate.SourceSupportsRouteGuidance = (tBoolean)routeGuidanceUpdateParameter->iAP2SourceSupportsRouteGuidance[0];
    }

    //store current state
    iAP2PushRGUpdate(mountPoint, RGUpdate);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE);
    return ret;
}
#endif

#ifdef IAP2_DEFAULT_ROUTEGUIDANCEDISPLAYCOMPONENT
tResult iPodControl::OnCBIAP2_RouteGuidanceManeuverUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceManeuverUpdateParameter* routeGuidanceManeuverUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!routeGuidanceManeuverUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate = iAP2GetRGManeuverUpdate(mountPoint); //changes need (gkm6kor)
    //ToDo: The below parameter needs to be re implemented as it can have multiple elements.
    if(routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count && routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0]);
        RGManeuverUpdate.RouteGuidanceDisplayComponentID=(tU16)routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID[0];
    }

     if(routeGuidanceManeuverUpdateParameter->iAP2Index_count && routeGuidanceManeuverUpdateParameter->iAP2Index) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2Index[0]);
        RGManeuverUpdate.Index=(tU16)routeGuidanceManeuverUpdateParameter->iAP2Index[0];
    }

    //for UTF-8 ManeuverDescription
    if(routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription_count && routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription[0]);
        strncpy_r(RGManeuverUpdate.ManeuverDescription, (char *)routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType_count && routeGuidanceManeuverUpdateParameter->iAP2ManeuverType) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType[0]);
        switch(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType[0]){

                    case IAP2_MANEUVER_TYPE_NO_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_NO_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_STRAIGHT_AHEAD:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_STRAIGHT_AHEAD;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_U_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_CONTINUE:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CONTINUE;
                        break;

                    case IAP2_MANEUVER_TYPE_ENTER_ROUNDABOUT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ENTER_ROUNDABOUT;
                        break;

                    case IAP2_MANEUVER_TYPE_EXIT_ROUNDABOUT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_EXIT_ROUNDABOUT;
                        break;

                    case IAP2_MANEUVER_TYPE_OFF_RAMP:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_OFF_RAMP;
                        break;

                    case IAP2_MANEUVER_TYPE_ON_RAMP:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ON_RAMP;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION;
                        break;

                    case IAP2_MANEUVER_TYPE_PROCEED_TO_BEGINNING:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_PROCEED_TO_THE_BEGINNING_OF_ROUTE;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVED_AT_DESTINATION;
                        break;

                    case IAP2_MANEUVER_TYPE_KEEP_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_KEEP_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_KEEP_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_KEEP_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_ENTER_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ENTER_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_EXIT_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_EXIT_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED_TO_ROUTE;
                        break;

                    case IAP2_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_A_UTURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_U_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_AT_THE_END_OF_ROAD_TURN_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_AT_THE_END_OF_ROAD_TURN_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_A_UTURN_WHEN_POSSIBLE:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN_WHEN_POSSIBLE;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTIONS:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTION;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_1:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_1;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_2:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_2;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_3:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_3;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_4:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_4;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_5:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_5;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_6:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_6;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_7:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_7;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_8:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_8;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_9:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_9;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_10:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_10;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_11:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_11;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_12:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_12;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_13:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_13;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_14:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_14;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_15:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_15;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_16:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_16;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_17:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_17;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_18:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_18;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_19:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_19;
                        break;

                    case IAP2_MANEUVER_TYPE_SHARP_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SHARP_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SHARP_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SHARP_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SLIGHT_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SLIGHT_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SLIGHT_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SLIGHT_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT;
                        break;

                    default:
                        break;
                }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName_count && routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName[0]);
        strncpy_r(RGManeuverUpdate.AfterManeuverRoadName, (char *)routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver[0]);
        RGManeuverUpdate.DistanceBetweenManeuver=(tU32)routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver[0];
    }
    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr[0]);
        strncpy_r(RGManeuverUpdate.DistanceBetweenManeuverDisplayStr, (char *)routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits[0]);
        switch(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide_count && routeGuidanceManeuverUpdateParameter->iAP2DrivingSide) {
            VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide[0]);
            switch(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide[0]) {
                case IAP2_DRIVING_SIDE_RIGHT:
                    RGManeuverUpdate.DrivingSide = DIPO_DRIVING_SIDE_RIGHT;
                    break;
                case IAP2_DRIVING_SIDE_LEFT:
                    RGManeuverUpdate.DrivingSide = DIPO_DRIVING_SIDE_LEFT;
                    break;
                default:
                    break;
            }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionType_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionType) {
            VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionType[0]);
            switch(routeGuidanceManeuverUpdateParameter->iAP2JunctionType[0]) {
                case IAP2_JUNCTION_TYPE_SINGLE_INTERSECTION_WITH_JUNCTION_ELEMENTS:
                    RGManeuverUpdate.JunctionType = DIPO_JUNCTION_TYPE_SINGLE_INTERSECTION;
                    break;
                case IAP2_JUNCTION_TYPE_ROUNDABOUT_WITH_JUNCTION_ELEMENTS:
                    RGManeuverUpdate.JunctionType = DIPO_JUNCTION_TYPE_ROUNDABOUT;
                    break;
                default:
                    break;
            }

    }
    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle) {
        //ToDo: This need re-implementation as the number of items can be more than 1.
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle[0]);
        RGManeuverUpdate.JunctionElementAngle=(tU16)routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle[0];
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle[0]);
        RGManeuverUpdate.JunctionElementExitAngle=(tU16)routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle[0];
    }

    //store current state
    iAP2SetRGManeuverUpdate(mountPoint, RGManeuverUpdate);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE_MANEUVER);
    return ret;
}
#else
tResult iPodControl::OnCBIAP2_RouteGuidanceManeuverUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceManeuverUpdateParameter* routeGuidanceManeuverUpdateParameter, void* /*context*/)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    if(!iap2Device) {
        ETG_TRACE_ERR(("No valid device pointer for IAP2"));
        return MP_ERR_IPOD_NO_DEVICE;
    }

    if(!routeGuidanceManeuverUpdateParameter) {
        ETG_TRACE_ERR(("Invalid parameter"));
        return MP_ERR_IPOD_INVALID_PARAM;
    }

    tMountPoint mountPoint = {0};
    ret = iAP2_GetMountPoint(mountPoint, (const void*)iap2Device);
    if (ret != MP_NO_ERROR) {
        ETG_TRACE_ERR(("iAP2_GetMountPoint failed"));
        return MP_ERR_IPOD_NO_DEVICE;
    }
    VARTRACE(mountPoint);

    //update member
    tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate;
    InitDiPORouteGuidanceManeuverUpdate(RGManeuverUpdate);

    tDiPORGDisplayComponentList RouteGuidanceDisplayComponentIDs;
    if(routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count && routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID) {
        for(tU16 i = 0; i < routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID_count; ++i)
        {
            VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID[i]);
            RGManeuverUpdate.RouteGuidanceDisplayComponentIDs.push_back(routeGuidanceManeuverUpdateParameter->iAP2RouteGuidanceDisplayComponentID[i]);
        }
        RGManeuverUpdate.RouteGuidanceDisplayComponentIDsAvailable = TRUE;
    }

     if(routeGuidanceManeuverUpdateParameter->iAP2Index_count && routeGuidanceManeuverUpdateParameter->iAP2Index) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2Index[0]);
        RGManeuverUpdate.IndexAvailable = TRUE;
        RGManeuverUpdate.Index=(tU16)routeGuidanceManeuverUpdateParameter->iAP2Index[0];
    }

    //for UTF-8 ManeuverDescription
    if(routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription_count && routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription[0]);
        RGManeuverUpdate.ManeuverDescriptionAvailable = TRUE;
        strncpy_r(RGManeuverUpdate.ManeuverDescription, (char *)routeGuidanceManeuverUpdateParameter->iAP2ManeuverDescription[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType_count && routeGuidanceManeuverUpdateParameter->iAP2ManeuverType) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType[0]);
        RGManeuverUpdate.ManeuverTypeAvailable = TRUE;
        switch(routeGuidanceManeuverUpdateParameter->iAP2ManeuverType[0]){

                    case IAP2_MANEUVER_TYPE_NO_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_NO_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_STRAIGHT_AHEAD:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_STRAIGHT_AHEAD;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_U_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_CONTINUE:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CONTINUE;
                        break;

                    case IAP2_MANEUVER_TYPE_ENTER_ROUNDABOUT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ENTER_ROUNDABOUT;
                        break;

                    case IAP2_MANEUVER_TYPE_EXIT_ROUNDABOUT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_EXIT_ROUNDABOUT;
                        break;

                    case IAP2_MANEUVER_TYPE_OFF_RAMP:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_OFF_RAMP;
                        break;

                    case IAP2_MANEUVER_TYPE_ON_RAMP:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ON_RAMP;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION;
                        break;

                    case IAP2_MANEUVER_TYPE_PROCEED_TO_BEGINNING:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_PROCEED_TO_THE_BEGINNING_OF_ROUTE;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVED_AT_DESTINATION;
                        break;

                    case IAP2_MANEUVER_TYPE_KEEP_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_KEEP_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_KEEP_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_KEEP_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_ENTER_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ENTER_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_EXIT_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_EXIT_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED_TO_ROUTE;
                        break;

                    case IAP2_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_A_UTURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_U_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_AT_THE_END_OF_ROAD_TURN_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_AT_THE_END_OF_ROAD_TURN_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT;
                        break;

                    case IAP2_MANEUVER_TYPE_MAKE_A_UTURN_WHEN_POSSIBLE:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_MAKE_U_TURN_WHEN_POSSIBLE;
                        break;

                    case IAP2_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTIONS:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTION;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_1:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_1;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_2:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_2;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_3:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_3;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_4:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_4;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_5:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_5;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_6:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_6;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_7:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_7;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_8:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_8;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_9:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_9;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_10:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_10;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_11:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_11;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_12:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_12;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_13:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_13;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_14:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_14;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_15:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_15;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_16:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_16;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_17:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_17;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_18:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_18;
                        break;

                    case IAP2_MANEUVER_TYPE_ROUNDABOUT_EXIT_19:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_19;
                        break;

                    case IAP2_MANEUVER_TYPE_SHARP_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SHARP_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SHARP_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SHARP_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SLIGHT_RIGHT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SLIGHT_RIGHT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_SLIGHT_LEFT_TURN:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_SLIGHT_LEFT_TURN;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT;
                        break;

                    case IAP2_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT:
                        RGManeuverUpdate.ManeuverType = DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT;
                        break;

                    default:
                        break;
                }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName_count && routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName[0]);
        RGManeuverUpdate.AfterManeuverRoadNameAvailable = TRUE;
        strncpy_r(RGManeuverUpdate.AfterManeuverRoadName, (char *)routeGuidanceManeuverUpdateParameter->iAP2AfterManeuverRoadName[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver[0]);
        RGManeuverUpdate.DistanceBetweenManeuverAvailable = TRUE;
        RGManeuverUpdate.DistanceBetweenManeuver=(tU32)routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuver[0];
    }
    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr[0]);
        RGManeuverUpdate.DistanceBetweenManeuverDisplayStrAvailable = TRUE;
        strncpy_r(RGManeuverUpdate.DistanceBetweenManeuverDisplayStr, (char *)routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayStr[0], sizeof(tGeneralString));
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits_count && routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits[0]);
        RGManeuverUpdate.DistanceBetweenManeuverDisplayUnitsAvailable = TRUE;
        switch(routeGuidanceManeuverUpdateParameter->iAP2DistanceBetweenManeuverDisplayUnits[0]) {
            case IAP2_DISTANCE_DISPLAY_UNITS_KM:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_KM;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_MILES:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_MILES;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_METERS:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_M;
                break;
                case IAP2_DISTANCE_DISPLAY_UNITS_YARDS:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_YARDS;
                break;
            case IAP2_DISTANCE_DISPLAY_UNITS_FT:
                RGManeuverUpdate.DistanceBetweenManeuverDisplayUnits = DIPO_DISTANCE_DISPLAY_UNIT_FT;
                break;
            default:
                break;
        }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide_count && routeGuidanceManeuverUpdateParameter->iAP2DrivingSide) {
            VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide[0]);
            RGManeuverUpdate.DrivingSideAvailable = TRUE;
            switch(routeGuidanceManeuverUpdateParameter->iAP2DrivingSide[0]) {
                case IAP2_DRIVING_SIDE_RIGHT:
                    RGManeuverUpdate.DrivingSide = DIPO_DRIVING_SIDE_RIGHT;
                    break;
                case IAP2_DRIVING_SIDE_LEFT:
                    RGManeuverUpdate.DrivingSide = DIPO_DRIVING_SIDE_LEFT;
                    break;
                default:
                    break;
            }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionType_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionType) {
            VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionType[0]);
            RGManeuverUpdate.JunctionTypeAvailable = TRUE;
            switch(routeGuidanceManeuverUpdateParameter->iAP2JunctionType[0]) {
                case IAP2_JUNCTION_TYPE_SINGLE_INTERSECTION_WITH_JUNCTION_ELEMENTS:
                    RGManeuverUpdate.JunctionType = DIPO_JUNCTION_TYPE_SINGLE_INTERSECTION;
                    break;
                case IAP2_JUNCTION_TYPE_ROUNDABOUT_WITH_JUNCTION_ELEMENTS:
                    RGManeuverUpdate.JunctionType = DIPO_JUNCTION_TYPE_ROUNDABOUT;
                    break;
                default:
                    break;
            }

    }
    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle[0]);
        RGManeuverUpdate.JunctionElementAngleAvailable = TRUE;
        for(int i=0; i<routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle_count; ++i){
            RGManeuverUpdate.JunctionElementAngle.push_back(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementAngle[i]);
        }
    }

    if(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle_count && routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle) {
        VARTRACE(routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle[0]);
        RGManeuverUpdate.JunctionElementExitAngleAvailable = TRUE;
        RGManeuverUpdate.JunctionElementExitAngle=(tU16)routeGuidanceManeuverUpdateParameter->iAP2JunctionElementExitAngle[0];
    }

    //store current state
    iAP2PushRGManeuverUpdate(mountPoint, RGManeuverUpdate);

    //send update trigger
    UpdateAppControl(iAPGetDeviceID(mountPoint), APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE_MANEUVER);
    return ret;
}
#endif

tResult iPodControl::GetRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        RGUpdate = iAP2GetRGUpdate(mountPoint);
        VARTRACE(RGUpdate);
    }
    return ret;
}

tResult iPodControl::GetRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        RGManeuverUpdate = iAP2GetRGManeuverUpdate(mountPoint);
        VARTRACE(RGManeuverUpdate);
    }
    return ret;
}

tResult iPodControl::PullRGUpdate(tDeviceID &deviceID, tDiPORouteGuidanceUpdate &RGUpdate)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        RGUpdate = iAP2PullRGUpdate(mountPoint);
        VARTRACE(RGUpdate);
    }
    return ret;
}

tResult iPodControl::PullRGManeuverUpdate(tDeviceID &deviceID, tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    deviceID = DEVICE_ID_NOT_SET;

    tMountPoint mountPoint = {0};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint)) {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        RGManeuverUpdate = iAP2PullRGManeuverUpdate(mountPoint);
        VARTRACE(RGManeuverUpdate);
    }
    return ret;
}

tResult iPodControl::SendDiPOStartRouteGuidance(const tDeviceID deviceID,tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs, tBoolean SourceName, tBoolean SourceSupportsRouteGuidance)
{
    ENTRY;

    tUInt RouteGuidanceDisplayComponentID = RGDISPLAYCOMPONENTIDS_NONE;
    VARTRACE(deviceID);
    VARTRACE(SourceSupportsRouteGuidance);
    VARTRACE(SourceName);
    tResult ret = MP_NO_ERROR;

    for(int i=0; i<RouteGuidanceDisplayComponentIDs.size(); ++i)
    {
        RouteGuidanceDisplayComponentID = RouteGuidanceDisplayComponentIDs[i];
        VARTRACE(RouteGuidanceDisplayComponentID);

        const tUserDataType userDataType = USRDATA_START_RG;
        tUserData userData = {0};
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "iii" ,  RouteGuidanceDisplayComponentID, SourceName, SourceSupportsRouteGuidance);

        /* Send SIGNAL_USERDATA message to own SM */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        } else {
            ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
            if( MP_NO_ERROR != ret ) {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
    }
    return ret;
}

tResult iPodControl::SendDiPOStopRouteGuidance(const tDeviceID deviceID,tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs)
{
    ENTRY;

    tUInt RouteGuidanceDisplayComponentID = RGDISPLAYCOMPONENTIDS_NONE;
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    for(int i=0; i<RouteGuidanceDisplayComponentIDs.size(); ++i)
    {
        RouteGuidanceDisplayComponentID = RouteGuidanceDisplayComponentIDs[i];
        VARTRACE(RouteGuidanceDisplayComponentID);

        const tUserDataType userDataType = USRDATA_STOP_RG;
        tUserData userData = {0};
        SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "i" , RouteGuidanceDisplayComponentID );

        /* Send SIGNAL_USERDATA message to own SM */
        tAllParameters parameterString;
        size_t size = sizeof(parameterString);

        ret = ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
        if( MP_NO_ERROR != ret ) {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
        } else {
            ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
            if( MP_NO_ERROR != ret ) {
                ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
            }
        }
    }
    return ret;
}

tBoolean iPodControl::IsInitializingDevice(const tMountPoint mountPoint,tConnectionState connectionState,tInitDeviceProtocol protocol)
{
    tBoolean retVal = false;

    if(!strncmp(m_DeviceToInitMountPoint,mountPoint,sizeof(m_DeviceToInitMountPoint)))
    {
        retVal = true;
    }
    return retVal;
}

tResult iPodControl::AddOobPairedDeviceResult(const tDeviceID deviceID,const tBoolean result)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    VARTRACE(deviceID);
    VARTRACE(result);

    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);

    ETG_TRACE_USR3(("OOB BT Pairing Completion Information"));
    ret = iAP2_OOBTBTPairingCompletionInformation(mountPoint,result);
    ETG_TRACE_USR3(("OOB BT Pairing Completion Information returned %d", ret));

    return ret;
}

tResult iPodControl::SetCurrentLanguage(const tLanguageType language)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    VARTRACE(m_ActivePBMountPoint);
    const tDeviceID deviceID = iAPGetDeviceID(m_ActivePBMountPoint);
    VARTRACE(deviceID);
    VARTRACE(language);

    tAllParameters parameterString;
    size_t size = sizeof(parameterString);
    switch(language)
    {
        case LNG_STANDARD_CHINESE:
        case LNG_TRADITIONAL_CHINESE:
            LocalSPM::GetDataProvider().iPodControlAccessoryInfoCurrentLanguage = "zh";
            break;
        default:
            LocalSPM::GetDataProvider().iPodControlAccessoryInfoCurrentLanguage = "en";
            break;
    }

    const tUserDataType userDataType = USRDATA_LANGUAGEUPDATES;
    tUserData userData = {0}; //no data, taken from config

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tBoolean iPodControl::StartAlbumArtTimer(const tMountPoint mountPoint)
{
    ENTRY;
    long timeoutValue = 700;

    /* Start the timer, single shot */
    tBoolean ret = m_AlbumArtTimer.StartTimer(OUT m_AlbumArtTimerID, IN timeoutValue, 0L, &LocalSPM::GetIPODControl(), &AlbumArtTimerCallBack, (void *)(char *)mountPoint);
    return ret;
}

void iPodControl::StopAlbumArtTimer()
{
    ENTRY;
    /* Cancel the timer */
    if (m_AlbumArtTimerID)
    {
        m_AlbumArtTimer.CancelTimer(IN m_AlbumArtTimerID);
        m_AlbumArtTimerID = 0;
    }

}

bool iPodControl::AlbumArtTimerCallBack(timer_t /*timerID */, void* instance ,const void * userData)
{
    ENTRY;
    iPodControl *self = (iPodControl*)instance;
    tMountPoint mountPoint = {0};
    strncpy_r(mountPoint, (const char*)userData, sizeof(tMountPoint));
    VARTRACE(mountPoint);
    if(self)
    {
        tDeviceID deviceID = self->iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);
        self->m_AlbumArtTimerID = 0;
        if(self->iAP2_IsHostMode(mountPoint) || self->iAP2_IsWirelessCarPlayMode(mountPoint)){
            ETG_TRACE_USR3(("Received AlbumArt for inactive device"));
            self->UpdateAppControl(deviceID, APPCONTROL_UPDATE_DIPO_NOWPLAYING);
        }
        else{
            self->SendAlbumArtStatusToPlayerManager(self->m_ActivePBMountPoint);
        }
    }
    return true;
}

//update clients (in this case connectivity) that the device has updated play and is ready for activation [for RTC bug : Bug 280485]
tResult iPodControl::OnUpdateReadyToPlay(const tMountPoint mountPoint) {
    ENTRY;

    tResult ret  = MP_NO_ERROR;
    char msgToSendString[64];
    tAllParameters parameters;
    size_t paramSize = sizeof(parameters);
    tPEPlaybackState playbackState = iAPGetCurrentPlaybackState(mountPoint);
    VARTRACE(playbackState);
    tBoolean BTDeviceConnected = false;

    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);

    tDeviceInfo deviceInfo;
    ret = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, deviceID);

    if(deviceInfo.connectionType == DCT_USB)
    {
        tNumberOfDevices numberOfDevices;
        vector<tDeviceInfo> deviceInfoList;

        ret = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(numberOfDevices,deviceInfoList,true);

        if (numberOfDevices > 0 )
        {
            for(tUInt j = 0; j < deviceInfoList.size(); j++)
            {
                VARTRACE(numberOfDevices);
                if( IsAppleDevice(deviceInfoList[j].deviceType)/*(DTY_IPHONE == deviceInfoList[j].deviceType || DTY_IPOD == deviceInfoList[j].deviceType)*/
                    && DCT_BLUETOOTH == deviceInfoList[j].connectionType
                    && !strncmp(deviceInfo.deviceName,deviceInfoList[j].deviceName, sizeof(deviceInfo.deviceName)))
                {
                    strncpy(deviceInfo.appleDeviceMACAddress,deviceInfoList[j].appleDeviceMACAddress,sizeof(deviceInfo.appleDeviceMACAddress));
                    BTDeviceConnected = true;
                }
            }
        }
    }

    VARTRACE(BTDeviceConnected);
    VARTRACE(deviceInfo.appleDeviceMACAddress);

    if(BTDeviceConnected)
    {
        strncpy_r(OUT msgToSendString, IN "CustomControlSM::UPDATE_READY_TO_PLAY", IN sizeof(msgToSendString));

        ret = LocalSPM::GetCustomControl().ParameterUPDATE_READY_TO_PLAY(OUT parameters, IN paramSize, IN deviceInfo.appleDeviceMACAddress , IN  playbackState);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while preparing parameter string (ErrorCode:%s)", errorString(ret)));
            parameters[0] = '\0';
        }

        ret = Dispatcher::GetInstance().SendMessage(IN msgToSendString, IN parameters);
        if (MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending message via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    return MP_NO_ERROR;
}

void iPodControl::SendUpdateReadyToPlay(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);
    const tDeviceID deviceID = iAPGetDeviceID(mountPoint);
    VARTRACE(deviceID);
    tResult ret = MP_NO_ERROR;

    const tUserDataType userDataType = USRDATA_UPDATE_READY_TO_PLAY;
    tUserData userData = {0}; //no data, taken from config

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    ret = SendEvent(SIGNAL_USERDATA, IN parameterString);
    if( MP_NO_ERROR != ret ) {
        ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
    }
}

void iPodControl::SetIsWirelessCarplayActive(const tMountPoint mountPoint)
{
    ENTRY;
    VARTRACE(mountPoint);
    tResult ret = MP_NO_ERROR;
    bool isCPWActive = false;
    tMountPoint CPWMountPoint = {0};
    ret = GetCPWMountPointbyUSBMountPoint(mountPoint,CPWMountPoint);
    if (MP_NO_ERROR == ret) {
        VARTRACE(CPWMountPoint);
        if(DIPO_CAP_CARPLAY_WIFI == iAP2_GetDiPOCaps(CPWMountPoint)){
            isCPWActive = true;
        }
    }
    VARTRACE(isCPWActive);
    SetIsCPWActiveForUSBMountpoint(mountPoint,isCPWActive);
}

void iPodControl::CheckMultipleIAPSessionsForSameDevice(const tDeviceID deviceID, const tUUID deviceUUID)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    tMountPoint mountPoint = {0};
    iAPGetMountPointFromDeviceID(deviceID, mountPoint);
    tDeviceInfo CurrentdeviceInfo;
    iAPGetDeviceInfo(mountPoint, CurrentdeviceInfo);

    tNumberOfDevices numberOfDevices;
    vector<tDeviceInfo> deviceInfoList;

    ret = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(numberOfDevices,deviceInfoList,true);
    if(MP_NO_ERROR != ret)
    {
       ETG_TRACE_ERR(("Error while retrieving MediaplayerDeviceConections (ErrorCode:%s)", errorString(ret)));
    }
    else
    {
        for(unsigned int i = 0; i<deviceInfoList.size(); i++)
        {
           // Fix for NCG3D-159085: The accessory supports simultaneous iAP sessions
           // Based on deviceUUID update, disconnect IAP2BT device in DB (if IAP via USB is active).
           // Also disconnect SPP for iAP UUID for the accessory name to be removed from Apple device in Settings->General->About

           // Fix for NCG3D-163114: BTA shows not connected or slow response
           // Modified the conditional statement that caused the BT device disconnection on IAP device UUID update.
           if((deviceID != deviceInfoList[i].deviceID) && (DTY_IPHONE == deviceInfoList[i].deviceType) && !strncmp(deviceUUID,deviceInfoList[i].deviceUUID,sizeof(tUUID)))
           {
               if((DCT_USB == CurrentdeviceInfo.connectionType) && (DCT_BLUETOOTH == deviceInfoList[i].connectionType) && (CS_DISCONNECTED != deviceInfoList[i].connectionState))
               {
                   ETG_TRACE_USR4((" Disconnect IAP2BT device in DB and Disconnect SPP "));
                   SendDBChangeConnectionState(IN deviceInfoList[i].deviceID, IN CS_DISCONNECTED);
                   ret = LocalSPM::GetOutputWrapper().SendDisconnectSPPService(IN deviceInfoList[i].appleDeviceMACAddress);
               }
               else if((DCT_BLUETOOTH == CurrentdeviceInfo.connectionType) && (DCT_USB == deviceInfoList[i].connectionType) && IsIAP2(deviceInfoList[i].mountPoint))
               {
                   ETG_TRACE_USR4((" Disconnect IAP2BT device in DB and Disconnect SPP "));
                   SendDBChangeConnectionState(IN deviceID, IN CS_DISCONNECTED);
                   ret = LocalSPM::GetOutputWrapper().SendDisconnectSPPService(IN CurrentdeviceInfo.appleDeviceMACAddress);
               }
               else
               {
                   ETG_TRACE_USR4(("No two IAP sessions(IAP and IAP2BT) found with UUID :%s",deviceUUID));
               }

               if(MP_NO_ERROR != ret)
               {
                   ETG_TRACE_ERR((" Posting of Disconnect SPP request via OutputWrapper failed! "));
               }
            }

           // For Wifi devices set CPW active for related USB mount point
           if( (DCT_WIFI == iAPGetConnectionType(mountPoint))
                 && ( 0 == strncmp(deviceUUID,deviceInfoList[i].deviceUUID,sizeof(tUUID)) )
                 && ( DCT_USB == deviceInfoList[i].connectionType) )
           {
               if( DIPO_CAP_CARPLAY_WIFI == iAP2_GetDiPOCaps(mountPoint) )
               {
                   ETG_TRACE_USR4((" Setting CPW Active for USB mount"));
                   SetIsCPWActiveForUSBMountpoint(deviceInfoList[i].mountPoint, true);
               }
           }
        }
    }
}

tResult iPodControl::SendAppleHIDCommand(const tPlaybackHIDCommand playbackHIDCommand,const tBTButtonEvent keyEvent)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;
    tDeviceID deviceID = iAPGetDeviceID(m_ActivePBMountPoint);

    VARTRACE(playbackHIDCommand);
    VARTRACE(keyEvent);

    const tUserDataType userDataType = USRDATA_APPLE_HID_COMMAND;
    tUserData userData = {0};
    SMF::Marshal(userData, sizeof(userData) - 1, DOUBLE_MARSHAL_SEPARATOR, "iit", playbackHIDCommand, keyEvent, m_ActivePBMountPoint);

    /* Send SIGNAL_USERDATA message to own SM */
    tAllParameters parameterString;
    size_t size = sizeof(parameterString);

    ParameterSIGNAL_USERDATA(OUT parameterString, IN size, IN deviceID, IN userDataType, IN userData);
    SendEvent(SIGNAL_USERDATA, IN parameterString);

    return ret;
}

tResult iPodControl::StoreInitDeviceEvent(const tMountPoint mountPoint, const tInitDeviceProtocol protocol)
{
    ENTRY;

    strncpy_r(m_mountPointPendingInitDevice, (const char*)mountPoint, sizeof(tMountPoint));
    VARTRACE(m_mountPointPendingInitDevice);

    m_protocolPendingInitDevice = protocol;
    m_bPendingInitDevice = true;

    return MP_NO_ERROR;
}

tResult iPodControl::GetDiPODeviceTime(tDeviceID &deviceID, tDiPODeviceTime &deviceTime)
{
    ENTRY;
    tResult ret = MP_NO_ERROR;

    tMountPoint mountPoint = {};
    if(iAPGetMountOfFirstHostModeDevice(mountPoint))
    {
        VARTRACE(mountPoint);

        deviceID = iAPGetDeviceID(mountPoint);
        VARTRACE(deviceID);

        deviceTime = iAP2GetDeviceTimeUpdate(mountPoint);
        VARTRACE(deviceTime);
    }
    return ret;
}

tResult iPodControl::StartAudioDevice()
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    VARTRACE(m_ActivePBMountPoint);

    if ((m_ActivePBMountPoint[0] == 0) || (!iAP_IsDeviceHandleAvail(m_ActivePBMountPoint)) || iAP2_IsHostMode(m_ActivePBMountPoint))
    {
        ETG_TRACE_ERR(("iPodControl::StartAudioDevice() - m_ActivePBMountPoint[0] == 0 or Invalid device handle"));
        ret = SendEvent(START_AUDIO_DEVICE_ANSWER, IN NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        VARTRACE(m_ActivePBSampleRate);
        tDeviceInfo deviceInfo;
        InitDeviceInfo(deviceInfo);
        tURL url = {};
        tPEStateString args = {};

        iAPGetDeviceInfo(m_ActivePBMountPoint, deviceInfo);
        snprintf(url, sizeof(tURL), "%s.pcm", deviceInfo.accessoryName);
        ETG_TRACE_USR3(("iPodControl::StartAudioDevice:url:%s", url));

        SMF::Marshal((char *)args, sizeof(args)-1, DOUBLE_MARSHAL_SEPARATOR, "tiiltl", url, 1 /* speed */, -1 /*position*/, HANDLE_NONE, m_AudioOutputDevice, m_ActivePBSampleRate);
        ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::PLAY", args, "iPodControlSM::START_AUDIO_DEVICE_ANSWER");
        if(MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("iPodControl:StartAudioDevice() - SendMessageAnswer() failed"));
        }
    }


    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StartAudioDeviceAnswer()
{
    ENTRY

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StopAudioDevice()
{
    ENTRY
    tResult ret = MP_NO_ERROR;
    VARTRACE(m_ActivePBMountPoint);

    if ((m_ActivePBMountPoint[0] == 0) || (!iAP_IsDeviceHandleAvail(m_ActivePBMountPoint)) || iAP2_IsHostMode(m_ActivePBMountPoint))
    {
        ETG_TRACE_ERR(("iPodControl::StopAudioDevice() - m_ActivePBMountPoint[0] == 0 or Invalid device handle"));
        ret = SendEvent(STOP_AUDIO_DEVICE_ANSWER, IN NULL);
        if(MP_NO_ERROR != ret)
        {
            ETG_TRACE_ERR(("Error while sending internal event via SMF (ErrorCode:%s)", errorString(ret)));
        }
    }
    else
    {
        ret = Dispatcher::GetInstance().SendMessageAnswer("MediaEngineSM::STOP", NULL, "iPodControlSM::STOP_AUDIO_DEVICE_ANSWER");
        if(MP_NO_ERROR != ret) {
            ETG_TRACE_ERR(("iPodControl:StartAudioDevice() - SendMessageAnswer() failed"));
        }
    }

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}

tResult iPodControl::StopAudioDeviceAnswer()
{
    ENTRY

    //Don't return an error in a transaction because we will not switch to new state even though we already did the action
    return MP_NO_ERROR;
}
