/**
 * @addtogroup iPodControl
 * @author Ulrich Deuper
 *
 * iPod indexer class
 * @{
 */

#ifndef IPODCONTROLINDEXER_H_
#define IPODCONTROLINDEXER_H_

//basic includes
#include "ElapsedTimer.h"
#include <vector>

//iPod includes
#include "iPodControlCommon.h"
#include "iPodControlMediaPath.h"

/** 
 * iPod control indexer class
 */

class iPodControlIndexerContext
{
public:
	iPodControlIndexerContext();				/**< constructor */
	~iPodControlIndexerContext();               /**< destructor */

	void Reset();

	void TraceOut() const;						/**< trace out members*/

	int GetTotalCacheCount() const {return mObjectList.size() + mPlaylistObjectList.size();};

	int GetStrategy() const {return mStrategy;};									/**< get indexing strategy*/
	void SetStrategy(const IPOD_INDEXER_STRATEGY strategy) {mStrategy = strategy;};	/**< set indexing strategy*/

	int GetTrackIndex() const {return mTrackIndex;};								/**< get indexing TrackIndex*/
	void SetTrackIndex(const int trackIndex) {mTrackIndex = trackIndex;};			/**< set indexing TrackIndex*/

	int GetTrackCount() const {return mTrackCount;};								/**< get indexing TrackCount*/
	void SetTrackCount(const int trackCount) {mTrackCount = trackCount;};			/**< set indexing TrackCount*/

	int GetPlaylistIndex() const {return mPlaylistIndex;};							/**< get indexing PlaylistIndex*/
	void SetPlaylistIndex(const int playlistIndex) {mPlaylistIndex = playlistIndex;};	/**< set indexing PlaylistIndex*/

	int GetPlaylistCount() const {return mPlaylistCount;};							/**< get indexing PlaylistCount*/
	void SetPlaylistCount(const int playlistCount) {mPlaylistCount = playlistCount;};	/**< set indexing PlaylistCount*/

	int GetVideoIndex() const {return mVideoIndex;};								/**< get indexing VideoIndex*/
	void SetVideoIndex(const int videoIndex) {mVideoIndex = videoIndex;};			/**< set indexing VideoIndex*/

	int GetVideoCount() const {return mVideoCount;};								/**< get indexing VideoCount*/
	void SetVideoCount(const int videoCount) {mVideoCount = videoCount;};			/**< set indexing VideoCount*/

	int GetMediaLibraryUpdateProgress() const
		{return mMediaLibraryUpdateProgress;};										/**< get indexing Progress*/
	void SetMediaLibraryUpdateProgress(const int progress)
		{mMediaLibraryUpdateProgress = progress;};									/**< set indexing Progress*/

	int GetiTunesRadioLibraryUpdateProgress() const
        {return miTunesRadioLibraryUpdateProgress;};                                /**< get indexing Progress*/
    void SetiTunesRadioLibraryUpdateProgress(const int progress)
        {miTunesRadioLibraryUpdateProgress = progress;};                            /**< set indexing Progress*/

	tBoolean GetMediaLibraryItemsCompleted() const
        {return mMediaLibraryItemsCompleted;};                                      /**< get indexing Progress*/
    void SetMediaLibraryItemsCompleted(const tBoolean set)
        {mMediaLibraryItemsCompleted = set;};                                       /**< set indexing Progress*/

	bool IsEmptyCache() const {return mObjectList.empty();};						/**< is empty object list */
	void PushObject(tMediaObjectPtr ptr, tMetadataStatus status);					/**< push object to list */
	tBoolean PopObject(tMediaObjectPtr &ptr, tMetadataStatus &status);  													/**< pop object from list */

	bool IsEmptyPlaylistCache() const {return mPlaylistObjectList.empty();};                        /**< is empty object list */
	void PushPlaylistObject(const tU64 parentUUID, const vector<tU64> uuidList);                    /**< push object to list */
	tBoolean PopPlaylistObject(tMediaObjectPtr &ptr, const tDeviceID deviceID, const tMountPoint mountPoint);                                               /**< pop object from list */

	void PushPlaylistFileTransferObject(const tU8 playlistFileTransferID, const tPlaylistFileTransferObj obj);
	tBoolean PopPlaylistFileTransferObject(const tU8 playlistFileTransferID, tPlaylistFileTransferObj &obj);
	int GetPlaylistFileTransferObjectListSize() { return mPlaylistFileTransferObjectList.size(); };

	iPodControlMediaPath & GetPath() {return mPath;};								/**< get reference to mPath*/

	void IncrementPath(const tResult result);										/**< set next valid path */

	int GetChunkSize() const { return mChunkSize; };								/**< get chunk size max */

	void OnBeginCaching();															/**< begin caching */
	void OnEndCaching(const tSize size=0);															/**< end caching */

	bool IsRetrievError() const {return mRetrieveError;};							/**< get retrieve error */
	void SetRetrievError(const bool set) {mRetrieveError = set;};					/**< set retrieve error */

	void GetMediaLibraryUID(tFingerprint &mediaLibraryUID) const
		{strncpy_r(mediaLibraryUID, mMediaLibraryUID, sizeof(mediaLibraryUID));};	/**< get mediaLibraryUID*/
	void SetMediaLibraryUID(const tFingerprint mediaLibraryUID)
		{strncpy_r(mMediaLibraryUID, mediaLibraryUID, sizeof(mMediaLibraryUID));};	/**< set mediaLibraryUID*/

    void GetiTunesRadioLibraryUID(tFingerprint &iTunesRadioLibraryUID) const
        {strncpy_r(iTunesRadioLibraryUID, miTunesRadioLibraryUID, sizeof(iTunesRadioLibraryUID));};      /**< get iTunesRadioUID*/
    void SetiTunesRadioLibraryUID(const tFingerprint iTunesRadioLibraryUID)
        {strncpy_r(miTunesRadioLibraryUID, iTunesRadioLibraryUID, sizeof(miTunesRadioLibraryUID));};     /**< set iTunesRadioUID*/

    void GetNowPlayingLibraryUID(tFingerprint &NowPlayingLibraryUID) const
        {strncpy_r(NowPlayingLibraryUID, mNowPlayingLibraryUID, sizeof(NowPlayingLibraryUID));};      /**< get NowPlayingUID*/
    void SetNowPlayingLibraryUID(const tFingerprint NowPlayingLibraryUID)
        {strncpy_r(mNowPlayingLibraryUID, NowPlayingLibraryUID, sizeof(mNowPlayingLibraryUID));};     /**< set NowPlayingUID*/

    void GetMediaLibraryRevision(tFingerprint &mediaLibraryRevision) const
        {strncpy_r(mediaLibraryRevision, mMediaLibraryRevision, sizeof(mediaLibraryRevision));};   /**< get mediaLibraryRevision*/
    void SetMediaLibraryRevision(const tFingerprint mediaLibraryRevision)
        {strncpy_r(mMediaLibraryRevision, mediaLibraryRevision, sizeof(mMediaLibraryRevision));};  /**< set mediaLibraryRevision*/

    void GetiTunesRadioLibraryRevision(tFingerprint &iTunesRadioLibraryRevision) const
        {strncpy_r(iTunesRadioLibraryRevision, miTunesRadioLibraryRevision, sizeof(iTunesRadioLibraryRevision));};      /**< get iTunesRadioRevision*/
    void SetiTunesRadioLibraryRevision(const tFingerprint iTunesRadioLibraryRevision)
        {strncpy_r(miTunesRadioLibraryRevision, iTunesRadioLibraryRevision, sizeof(miTunesRadioLibraryRevision));};     /**< set iTunesRadioRevision*/

	void GetFingerprint(tFingerprint &fingerprint) const
		{strncpy_r(fingerprint, mFingerprint, sizeof(fingerprint));};				/**< get fingerprint*/
	void SetFingerprint(const tFingerprint fingerprint)
		{strncpy_r(mFingerprint, fingerprint, sizeof(mFingerprint));};				/**< set fingerprint*/

    bool IsPaused() const {return mPaused;};                           /**< get stop */
    void SetPaused(const bool set) {mPaused = set;};                   /**< set stop */
	
    bool IsMediaLibraryReset() const {return mMediaLibraryReset;};                   /**< get MediaLibraryReset */
    void SetMediaLibraryReset(const bool set) {mMediaLibraryReset = set;};           /**< set MediaLibraryReset */

private:
	IPOD_INDEXER_STRATEGY mStrategy;		/**< strategy */
	iPodControlMediaPath mPath; 			/**< current media selection*/

	int mIndexedItems;						/**< total indexed items*/
	int mRemovedItems;						/**< total removed items*/

	int mTrackIndex;						/**< track index*/
	int mTrackCount;						/**< track count*/

	int mPlaylistIndex;						/**< playlist index*/
	int mPlaylistCount;						/**< playlist count*/

	int mVideoIndex;						/**< video index*/
	int mVideoCount;						/**< video count*/

	typedef struct {
		tMediaObjectPtr ptr;
		tMetadataStatus status;
	}tIndexObject;

    typedef struct {
        tU64 parentUUID;
        vector<tU64> uuidList;
        unsigned int listIndex;
    }tPlaylistObject;

	vector<tIndexObject> mObjectList;    	/**< media objects of indexing context */
	vector<tPlaylistObject> mPlaylistObjectList;       /**< media objects of indexing context */

	ElapsedTimer mTotalTime;				/**< total indexing time */
	ElapsedTimer mCacheTime;				/**< chunk indexing time */
	int mChunkSize;							/**< chunk indexing size */
	double mChunkPerf;						/**< chunk perf items/sec */

	bool mRetrieveError;				    /**< retrieve error */
	tFingerprint mMediaLibraryUID;			/**< iAP2MediaLibraryUniqueIdentifier */
	tFingerprint miTunesRadioLibraryUID;    /**<  iAP2iTunesRadioLibraryUniqueIdentifier */
	tFingerprint mNowPlayingLibraryUID;     /**<  current media library */
	tFingerprint mMediaLibraryRevision;     /**< iAP2MediaLibrary Revision */
	tFingerprint miTunesRadioLibraryRevision;     /**< iAP2MediaLibrary Revision */

	int mMediaLibraryUpdateProgress;		/**< iAP2MediaLibraryUpdateProgress */
	int miTunesRadioLibraryUpdateProgress;  /**< iAP2MediaLibraryUpdateProgress */

	tBoolean mMediaLibraryItemsCompleted;
	tFingerprint mFingerprint;				/**< fingerprint */

    tIPODPlaylistFileTransfer mPlaylistFileTransferObjectList;       /**< playlist objects of indexing context */
    bool mPaused;
	bool mMediaLibraryReset;
};

/**
 * iPod control indexer class
 */

class iPodControlVTRecordKey
{
public:
	/**< constructors */
	iPodControlVTRecordKey()
	: mListType(tListType_init), mFilterTag1(tFilterTag_init), mFilterTag2(
				tFilterTag_init), mFilterTag3(tFilterTag_init), mFilterTag4(tFilterTag_init), mIndex(0) {
	};

	iPodControlVTRecordKey(const tListType listType, const tFilterTag tag1,
			const tFilterTag tag2, const tFilterTag tag3, const tFilterTag tag4, const int index)
	: mListType(listType), mFilterTag1(tag1), mFilterTag2(tag2), mFilterTag3(
				tag3), mFilterTag4(tag4), mIndex(index) {
	};

	/**< operator */
	bool operator==(const iPodControlVTRecordKey &other) const {
		return (mListType == other.mListType &&
				mFilterTag1 == other.mFilterTag1 &&
				mFilterTag2 == other.mFilterTag2 &&
				mFilterTag3	== other.mFilterTag3 &&
				mFilterTag4	== other.mFilterTag4 &&
				mIndex == other.mIndex);
	};

	bool operator<(const iPodControlVTRecordKey &other) const {

		if (mListType < other.mListType) {
			return true;
		} else if (mListType == other.mListType) {
			if (mFilterTag1 < other.mFilterTag1) {
				return true;
			} else if (mFilterTag1 == other.mFilterTag1) {
				if (mFilterTag2 < other.mFilterTag2) {
					return true;
				} else if (mFilterTag2 == other.mFilterTag2) {
					if (mFilterTag3 < other.mFilterTag3) {
						return true;
					} else if (mFilterTag3 == other.mFilterTag3) {
						if (mFilterTag4 < other.mFilterTag4) {
							return true;
						} else if (mFilterTag4 == other.mFilterTag4) {
							if (mIndex < other.mIndex) {
								return true;
							}
						}
					}
				}
			}
		}

		return false;
	};

	tListType mListType;
private:
	tFilterTag mFilterTag1;
	tFilterTag mFilterTag2;
	tFilterTag mFilterTag3;
	tFilterTag mFilterTag4;
	int mIndex;
};

typedef map<iPodControlVTRecordKey, string> tIPODVTDeviceCache;
typedef map<string, tIPODVTDeviceCache> tIPODVTCache; //key = mountpoint

#endif /*IPODCONTROLINDEXER_H_*/

/** @} */
