/**
 * @addtogroup iPodControl
 * @author Ulrich Deuper
 *
 * iPod iAP access class wrapper
 * @{
 */

#ifndef IPODCONTROLIAP_H_
#define IPODCONTROLIAP_H_

//basic includes

#include "iPodControlCommon.h"
#include "iPodControlHandleMap.h"
#include <queue>


#include <libusb-1.0/libusb.h>



#define NO_RETRIAL (uint8_t)0


/**
 * iPod control handle class
 */
class iPodControlIAP {

public:
    iPodControlIAP(); /**< constructor */
    ~iPodControlIAP(); /**< destructor */

    //Init
    tResult iAP1_Init();     /**< initialize iAP1 library */
    tResult iAP2_Init();        /**< initialize iAP2 library */

    void iAP1_DeInit();         /**< disconnect iAP1 library */
    void iAP2_DeInit();         /**< disconnect iAP2 library */

    void iAP2_PollFDs(const tMountPoint mountPoint);     /**< poll IAP2 fds */

    tResult iAPNewDevice(const tDeviceInfo deviceInfo); /**< assign new iDevice, add to handle map */

    tResult iAP1_InitDeviceConnection(const tMountPoint mountPoint); /**< IAP1 initialize connection iDevice, trigger authentication procedure */
    tResult iAP2_InitDeviceConnection(const tMountPoint mountPoint, const tBoolean hostMode = false, const tBoolean carPlay = false, const tBoolean nativeTransport = false, const tBoolean wirelessCarplay = false,const tBoolean carLife = false); /**< IAP2 initialize connection iDevice, trigger authentication procedure */

    tResult iAP1_DisconnectDeviceConnection(const tMountPoint mountPoint, const bool resetAllParameters); /**< disconnect connection to IAP1 iDevice */
    tResult iAP2_DisconnectDeviceConnection(const tMountPoint mountPoint, const bool resetAllParameters, const bool callDisconnect = true); /**< disconnect connection to IAP2 iDevice */

    tResult iAP1_OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState); /**< prepared IAP1 iDevice after authentication procedure */
    tResult iAP2_OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState); /**< prepared IAP2 iDevice after authentication procedure */

    //Getter w/o IAP access
    tBoolean iAP_IsDeviceHandleAvail(const tMountPoint mountPoint);    /**< check handle availability */
    tBoolean iAP2_IsSet(const tMountPoint mountPoint);    /**< IAP2 mode */
    tBoolean iAP2_IsHostMode(const tMountPoint mountPoint); /** check for IAP2 Host Mode */
    tBoolean iAP2_IsCarPlayMode(const tMountPoint mountPoint); /** check for IAP2 Host+Dipo Mode */
    tBoolean iAP2_IsNativeTransportMode(const tMountPoint mountPoint); /** check for IAP2 NativeTransport Mode */
    tBoolean iAP2_IsWirelessCarPlayMode(const tMountPoint mountPoint); /** check for IAP2 WirelessCarPlay Mode */
    tBoolean iAP2_IsNativeTransportAppConfigured(const tMountPoint mountPoint);
    tBoolean iAP2_IsNativeTransportCarlifeMode(const tMountPoint mountPoint);
    tBoolean iAP2_IsHostModePossible(const tMountPoint mountPoint); /** check for IAP2 Host Mode */
    tBoolean iAP2_IsCarPlayPossible(const tMountPoint mountPoint);
    tDiPOCaps iAP2_GetDiPOCaps(const tMountPoint mountPoint);
    tBoolean iAP2_IsiTunesRadioPlaying(const tMountPoint mountPoint); /** check for IAP2 iTunes Radio Mode */
    tBoolean iAP2HasOTGPort(const tMountPoint mountPoint); /** check for OTG port avaiablility */

    tBoolean iAP2_GetStoredHostModePossibility(const tMountPoint mountPoint);
    tResult iAP1_GetMountPoint(tMountPoint &mountPoinit, const int iPodID); /**< get mountpoint */
    tResult iAP2_GetMountPoint(tMountPoint &mountPoinit, const void* iap2Device); /**< get mountpoint */

    vector<string> iAPGetMountPoints(); /**< get all mountpoints from map */
    tResult iAPGetMountPointByPEHandle(tMountPoint &mountPoint, const tPEHandle handle); /**< get mountpoint */
    tDeviceID iAPGetDeviceID(const tMountPoint mountPoint); /**< get device id */
    void iAPGetFocusApp(const tMountPoint mountPoint, tAppName &appName); /**< get focus app name */
    void iAPGetDeviceInfo(const tMountPoint mountPoint, tDeviceInfo &deviceInfo); /**< get device info */
    tConnectionState iAPGetConnectionState(const tMountPoint mountPoint);
    tConnectionType iAPGetConnectionType(const tMountPoint mountPoint);
    tPEPlaybackState iAPGetCurrentPlaybackState( const tMountPoint mountPoint); /**< get playing status play,pause, stopped */
    tResult iAPGetIPODPlayerState(const tMountPoint mountPoint); /**< get playing status play,pause, stopped */
    /**< get current playing metadata  */
    tPlaytime iAPGetElapsedPlaytime(const tMountPoint mountPoint); /**< get current elapsed play time */
    tPlaytime iAPGetTotalPlaytime(const tMountPoint mountPoint); /**< get current total play time */
    me::samplerate_i iAPGetSampleRate(const tMountPoint mountPoint); /**< get current sample rate */
    tBitRate iAPGetBitRate(const tMountPoint mountPoint); /**< get current bit rate */
    tCodecType iAPGetCodecType(const tMountPoint mountPoint); /**< get current codec type */
    int iAPGetNowPlayingTrackIndex(const tMountPoint mountPoint); /**< get current track index */
    int iAPGetPBTrackIndex(const tMountPoint mountPoint); /**< get current playing index */

    int iAPGetAlbumArtID(const tMountPoint mountPoint); /**< get albumart id */
    int iAPGetQueueListID(const tMountPoint mountPoint); /**< get queue list id */

    void * iAPGetAlbumArtBuffer(const tMountPoint mountPoint); /**< get albumart buffer */
    void * iAPGetQueueListBuffer(const tMountPoint mountPoint); /**< get QueueList buffer */

    void iAPGetNowPlayingMetadata(tMetadata &metadata1,
            tMetadata &metadata2,
            tMetadata &metadata3,
            tMetadata &metadata4,
            tTitle &metadataTitle,
            tMediaType &mediaType,
            const tMountPoint mountPoint,
            tUUID &uuid); /**< get iPod now playing metadata and mediatype */
    void iAPGetNowPlayingMediaObject(tMediaObject &mediaObject, const tMountPoint mountPoint);

    void iAPGetAlbumArtString(const tMountPoint mountPoint, tAlbumArt &albumArtString); /**< get current album art string */
    void iAPGetAppNameBySessionID(const tMountPoint mountPoint,
            const tSessionID sessionId, tAppName &appName); /**< get appname by session id*/
    void iAPGetAppNameByProtocolID(const tMountPoint mountPoint,
                const unsigned char protocolId, tAppName &appName, tSessionID &sessionId); /**< get appname by protocol id*/
    void iAPGetProtocolNameByProtocolID(const tMountPoint mountPoint,
            const unsigned char protocolId, tProtocolName &protocolName);/**< get ProtocolName by protocol id*/
    void iAPGetActiveSessionList(vector<string> &list, const tMountPoint mountPoint);
    void iAPGetLaunchApp(const tMountPoint mountPoint, tAppName &appName);         /**< get app to be launched after reeuthentication */
    tBoolean iAPIsTrackFinished(const tMountPoint mountPoint);    /**< get status track has been finished in non-streaming mode */

    int iAPGetNowPlayingTrackCount(const tMountPoint mountPoint); /**< get current track count */
    int iAPGetNowPlayingChapterIndex(const tMountPoint mountPoint); /**< get current chapter index */
    void iAPGetDiPOLocationInfo(const tMountPoint mountPoint,tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType);
    void iAPGetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint,tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues);
    tDiPOCaps iAPGetLastWirelessCarplayStatus(const tMountPoint mountPoint);
    tDiPORouteGuidanceUpdate iAP2GetRGUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceManeuverUpdate iAP2GetRGManeuverUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceUpdate iAP2PullRGUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceManeuverUpdate iAP2PullRGManeuverUpdate(const tMountPoint mountPoint);
    bool iAPGetWaitingforRoleSwitchResponse(const tMountPoint mountPoint);

    //Setter w/o IAP access
    void iAPSetConnectionState(const tMountPoint mountPoint, const tConnectionState connectionState); /**< set connectionState */
    void iAPSetCurrentPlaybackState(const tMountPoint mountPoint,
            const tPEPlaybackState status);/**< set playing status play,pause, stopped */
    void iAPSetLastPlaybackState(const tMountPoint mountPoint, const tPEPlaybackState status);/**< set last playing status play,pause, stopped */
    void iAPSetIPODPlayerState(const tMountPoint mountPoint, const int state); /**< set playing status play,pause, stopped */
    void iAPSetElapsedPlaytime(const tMountPoint mountPoint,
            const tPlaytime time); /**< set current elapsed track playtime */
    void iAPSetTotalPlaytime(const tMountPoint mountPoint,
            const tPlaytime total); /**< set current total track playtime */
    void iAPSetSampleRate(const tMountPoint mountPoint, const me::samplerate_i sampleRate); /**< set current sample rate */
    void iAPSetBitRate(const tMountPoint mountPoint, const tBitRate bitRate); /**< set current bit rate */
    void iAPSetCodecType(const tMountPoint mountPoint, const tCodecType codecType); /**< set current codec type */
    void iAPSetNowPlayingTrackIndex(const tMountPoint mountPoint,
            const int trackIndex); /**< set current track index */
    void iAPSetNowPlayingTrackCount(const tMountPoint mountPoint,
            const int trackIndex); /**< set current track index */
    void iAPSetNowPlayingChapterIndex(const tMountPoint mountPoint,
            const int trackIndex); /**< set current track index */
    void iAPSetPBTrackIndex(const tMountPoint mountPoint,
            const int trackIndex); /**< set playing track index */
    void iAPSetPollThreadIndex(const tMountPoint mountPoint,
                const int threadIndex); /**< set poll thread index index */

    void iAPSetAlbumArtID(const tMountPoint mountPoint, const int id); /**< set albumart id */
    void iAPSetQueueListID(const tMountPoint mountPoint, const int id); /**< set QueueList id */

    void iAPSetAlbumArtBuffer(const tMountPoint mountPoint, void * buffer); /**< set albumart buffer */
    void iAPSetQueueListBuffer(const tMountPoint mountPoint, void * buffer); /**< set queue list buffer */

    void iAPSetSessionIDByIndex(const tMountPoint mountPoint,
            const tSessionID sessionId, const unsigned char protocolIndex,
            tAppName &appName); /**< set session id by protocol index */
    void iAPClearSessionID(const tMountPoint mountPoint,
            const tSessionID sessionId, tAppName &appName); /**< clear session id */
    void iAPSetLaunchApp(const tMountPoint mountPoint,
            const tAppName appName); /**< set app to be launched after reeuthentication */
    void iAPSetTrackFinished(const tMountPoint mountPoint,
            const tBoolean set); /**< set status track has been finished in non-streaming mode */

    void iAPSetFocusApp(const tMountPoint mountPoint, const char* appName); /**< set the current focus app name */
    void iAPSetDeviceName(const tMountPoint mountPoint, const char* deviceName); /**< set the device name */
    void iAPSetAlbumArtString(const tMountPoint mountPoint, const tAlbumArt albumArtString);
    void iAPSetNowPlayingMediaObject(const tMediaObject mediaObject, const tMountPoint mountPoint);
    void iAPSetDiPOLocationInfo(const tMountPoint mountPoint, const tBool startStopLocationInfo, const tDiPOLocationInfoType locationInfoType);
    void iAPSetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint,const tDiPOGPRMCDataStatusValues GPRMCDataStatusValues);
    void iAP2PushRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate);
    void iAP2SetRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate);
    void iAP2SetRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void iAP2PushRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void iAPSetLastWirelessCarplayStatus(const tMountPoint mountPoint, tDiPOCaps diPOCaps);
    void iAPSetWaitingforRoleSwitchResponse(const tMountPoint mountPoint, bool isWaitingforRoleSwitchResponse);
    //Other methods w/o IAP access
    void iAPResetElapsedTime(const tMountPoint mountPoint,
            const tiPodElaspedTime elapsedtime); /**< reset elapsed time for delayed iap call */
    tBoolean iAPIsTimeElapsed( const tMountPoint mountPoint, const tiPodElaspedTime elapsedtime,
                    const tPlaytime interval);     /**< return if 'interval' time has elapsed */

    static void iAPTraceData(const unsigned char * data,
            const unsigned int size); /**< dump data buffer */

    //Other methods with IAP access
    tResult iAPCheckUpdateNowPlaying(const tMountPoint mountPoint); /**< checks for now playing update */
    tResult iAP2SetLocationInformationUpdate(const tMountPoint mountPoint, tBoolean enableUpdates);
    tResult iAP2SetVehicleStatusUpdate(const tMountPoint mountPoint, tBoolean enableUpdates);

    //Commands with IAP access
    tResult iAPPlaySelection(const tMountPoint mountPoint, const tURL URL, const tPlaytime position, const tBatchPlaybackList * pbatchlist, tBoolean &sync_play_answer, const tBoolean playOnRemoteActivity); /**< start playing selection */
    tResult iAPPlayNext(const tMountPoint mountPoint, const tNextPrevSkipCount skipcount); /**< playing action in streaming mode */
    tResult iAPPlayPrev(const tMountPoint mountPoint, const tNextPrevSkipCount skipcount); /**< playing action in streaming mode */
    tResult iAPPlayPause(const tMountPoint mountPoint); /**< toggle play/pause */
    tResult iAPPlayResume(const tMountPoint mountPoint); /**< toggle play/pause */
    tResult iAPPlayFFStart(const tMountPoint mountPoint); /**< fast forward start */
    tResult iAPPlayREWStart(const tMountPoint mountPoint); /**< fast rewind start */
    tResult iAPPlayFFREWStop(const tMountPoint mountPoint); /**< fast forward/rewind stop */

    tResult iAPSeekTo(const tMountPoint mountPoint, const tPlaytime position); /**< seekTo */

    tResult iAPGetVTRecord(const tMountPoint mountPoint, const iPodControlVTRecordKey key, string &rValue); /**< gets single iPod DB record for VTIPOD */

    tResult iAPRetrieveVTRecord(const tMountPoint mountpoint,
            const tListType listType, const tFilterTag tag1,
            const tFilterTag tag2, const tFilterTag tag3, const tFilterTag tag4, const int rowID,
            const tLimit limit, const tOffset offset);                         /**< retrieve single iPod DB record for VTIPOD */
    tResult iAP1_RetrieveVTRecord(const tMountPoint mountpoint,
            const tListType listType, const tFilterTag tag1,
            const tFilterTag tag2, const tFilterTag tag3, const tFilterTag tag4, const int rowID,
            const tLimit limit, const tOffset offset);                      /**< retrieve single iPod DB record for VTIPOD */
    tResult iAP2_RetrieveVTRecord(const tMountPoint mountpoint,
            const tListType listType, const tFilterTag tag1,
            const tFilterTag tag2, const tFilterTag tag3, const tFilterTag tag4, const int rowID,
            const tLimit limit, const tOffset offset);                      /**< retrieve single iPod DB record for VTIPOD */

    tResult iAPGetNumberOfFiles(const tDeviceID deviceID, tNumberOfFiles &numberOfFiles);
    tResult iAP1_CalcFingerPrint(tFingerprint &fingerprint,
            tFingerprintStatus &fingerprintStatus,
            tNumberOfFiles &numberOfFiles, const tMountPoint mountpoint, const tFingerprint lastFingerprint);     /**< calculate finger print and count the number of tracks */
    tResult iAP2_CalcFingerPrint(const tMountPoint mountpoint, const tFingerprint lastFingerprint);     /**< calculate finger print and count the number of tracks */

    tResult iAPIndexerIsNextMetadataAvailable(const tMountPoint mountPoint, bool &avail);                /**< check next mediaobject according to context */

    tResult iAPIndexerGetNextMediaObject(tMetadataStatus &metadataStatus,
            tMediaObjectPtr &nextMediaObject, const tMountPoint mountPoint,
            const tReadPosition readPosition);                                 /**< get next mediaobject from list according to context */

    tResult iAPIndexerRetrieveMetadataFromDevice(const tMountPoint mountPoint,
            const tReadPosition readPosition, const tDeviceID deviceID);     /**< indexer retrieves next mediaobject according to context */

    tResult iAP1_GetAlbumArt(const tMountPoint mountPoint, const int track, const tU64 uuid);                /**< retrieve album art image */
    tResult iAP2_GetAlbumArt(const tMountPoint mountPoint, const tMetadata title, const tU64 uuid, tAlbumArtObjectPtr &ptrToAlbumArtObject);    /**< retrieve album art image */
    tBoolean iAPIsAlbumArtAvailable(const tMountPoint mountPoint);

    tResult iAPTransferTag(tTagTransferStatus &status,
            const tMountPoint mountPoint, const tFilename tagfile); /**< transfer tag to iPod */

    tResult iAPAddAppControlInfo(const tMountPoint mountPoint,
            const tProtocolName protocol, const tBundleSeedID bundleSeedID,
            const tAppName appName, bool &added); /**< add app control info */
    tResult iAPRemoveAppControlInfo(const tMountPoint mountPoint,
            const tAppName appName, const tSessionID sessionID, bool &removed); /**< remove app control info */


    tResult iAPAddAppControlInfoFromSPI(const tMountPoint mountPoint,
            const tProtocolName protocol, const tBundleSeedID bundleSeedID,
            const tAppName appName, bool &added);
    tResult iAPClearAppInfofromSPI(const tMountPoint mountPoint);

    tResult iAPAppControlCommand(const tMountPoint mountPoint,
            const tAppName appName, const tSessionID sessionID,
            const tSize sizeOfBuffer, const tCommandBufferPtr commandBuffer); /**< app control command */
    tResult iAP2_LocationInformation(const tMountPoint mountPoint,
            const tDiPOGPGGAData diPOGPGGAData,
            const tDiPOGPRMCData diPOGPRMCData,
            const tDiPOGPRMCData diPOGPGSVData,
            const tDiPOGPRMCData diPOGPHDTData,
            const tDiPOGPRMCData diPOPASCDData,
            const tDiPOGPRMCData diPOPAGCDData,
            const tDiPOGPRMCData diPOPAACDData);
    tResult iAP2_VehicleStatusUpdate(
            const tMountPoint mountPoint,
            const tOutsideTemperature outsideTemperature);

    tResult iAPLaunchApp(const tMountPoint mountPoint, const tAppName appName, const tAppLaunchOption launchOption); /**< launch app */
    tResult iAPCurrentSelectionChanged(tBoolean &changed, const tDeviceID deviceID);    /**< get selection change status */
    tBoolean iAPIsStreaming(const tMountPoint mountPoint); /**< return streaming mode ON/OFF */
    tPEHandle iAPGetPEHandle(const tMountPoint mountPoint); /**< get PE handle */
    tBoolean iAPIsRCSupported(const tMountPoint mountPoint); /**< return remote control support ON/OFF */
    void iAPSetStreaming(const tMountPoint mountPoint, const tBoolean set); /**< set streaming mode ON/OFF */
    void iAPSetPEHandle(const tMountPoint mountPoint, const tPEHandle handle); /**< set PE handle */

    void iAPClearAllVTRecords(const tMountPoint mountPoint,const int limit = 0);        /**< clear all VT record for mountpoint*/
    void iAPClearLiveVTRecords(const tMountPoint mountPoint);                             /**< clear current selection VT record for mountpoint*/

    tResult iAPSetPlaybackMode(const tMountPoint mountPoint, const tPlaybackMode playbackMode);
    void iAPSetPlaybackModeFromIPOD(const tMountPoint mountPoint, const int playbackMode);
    tPlaybackMode iAPGetPlaybackMode(const tMountPoint mountPoint);

    tResult iAPSetRepeatMode(const tMountPoint mountPoint, const tRepeatMode repeatMode);
    void iAPSetRepeatModeFromIPOD(const tMountPoint mountPoint, const int repeatMode);
    tRepeatMode iAPGetRepeatMode(const tMountPoint mountPoint);
    void iAPSetRepeatInitFlag(const tMountPoint mountPoint, const bool repeatInitFlag);
    bool iAPGetRepeatInitFlag(const tMountPoint mountPoint);

    tDiPOCallState iAPGetCallState(const tMountPoint mountPoint);
    tDiPOCallStateItem iAP2GetCallStateByUUID(const tMountPoint mountPoint, tGeneralString const callUUID);
    void iAP2UpdateOrInsertCallState(const tMountPoint mountPoint, tDiPOCallStateItem const callStateItem);
    void iAP2RemoveDisconnectedCallState(const tMountPoint mountPoint);

    tDiPOCommunications iAPGetCommunications(const tMountPoint mountPoint);
    void iAPSetCommunications(const tMountPoint mountPoint, const tDiPOCommunications comm);
    tBTProfile iAPGetBTProfile(const tMountPoint mountPoint);
    void iAPSetBTProfile(const tMountPoint mountPoint, const tBTProfile btProfile);

    void iAPSetMediaLibraryUID(const tMountPoint mountPoint, const char * uid);
    void iAPGetMediaLibraryUID(const tMountPoint mountPoint, tFingerprint &mediaLibraryUID);
    void iAPSetiTunesRadioLibraryUID(const tMountPoint mountPoint, const char * uid);
    void iAPGetiTunesRadioLibraryUID(const tMountPoint mountPoint, tFingerprint &iTunesRadioUID);
    void iAPSetNowPlayingLibraryUID(const tMountPoint mountPoint, const char * uid);
    void iAPGetNowPlayingLibraryUID(const tMountPoint mountPoint, tFingerprint &NowPlayingUID);

    void iAPSetMediaLibraryRevision(const tMountPoint mountPoint, const char * Revision);
        void iAPGetMediaLibraryRevision(const tMountPoint mountPoint, tFingerprint &mediaLibraryRevision);
        void iAPSetiTunesRadioLibraryRevision(const tMountPoint mountPoint, const char * Revision);
        void iAPGetiTunesRadioLibraryRevision(const tMountPoint mountPoint, tFingerprint &iTunesRadioRevision);

    void iAPSetMediaLibraryReset(const tMountPoint mountPoint, const bool isReset);
    bool iAPGetMediaLibraryReset(const tMountPoint mountPoint);

    void iAPSetMediaLibraryUpdateProgress(const tMountPoint mountPoint, const int progress);
    int iAPGetMediaLibraryUpdateProgress(const tMountPoint mountPoint);
    void iAPSetiTunesRadioLibraryUpdateProgress(const tMountPoint mountPoint, const int progress);
        int iAPGetiTunesRadioLibraryUpdateProgress(const tMountPoint mountPoint);
    tBoolean iAPIsMediaLibraryUpdateCompleted(const tMountPoint mountPoint);

    int iAPIndexerGetTrackCount(const tMountPoint mountPoint);
    int iAPIndexerGetVideoCount(const tMountPoint mountPoint);
    int iAPIndexerGetPlaylistCount(const tMountPoint mountPoint);
    int iAPIndexerGetChunkSize(const tMountPoint mountPoint);

    void iAPIndexerSetFingerprint(const tMountPoint mountPoint, const char * fingerprint);
    void iAPIndexerGetFingerprint(const tMountPoint mountPoint, tFingerprint &fingerprint);
    int iAPIndexerGetCacheCount(const tMountPoint mountPoint);
    bool iAPIndexerIsPaused(const tMountPoint mountPoint);
    void iAPIndexerCheckResume(const tMountPoint mountPoint);
    void iAP2UpdateCallDuration(const tMountPoint mountPoint, const U32 callDuration);
    U32 iAPGetCallDuration(const tMountPoint mountPoint);

#ifdef IPODCONTROL_IAP2_PF_AVAIL
    tBoolean iAP2_PushMediaLibraryUpdate(const tMountPoint mountPoint, const iAP2MediaLibraryUpdateParameter *mediaLibraryUpdateParameter,const bool isiTunesRadioUpdate); /**< return mediaObject from iap2 media item */
    tBoolean iAPSetupPlaylistItems(const tMountPoint mountPoint, const iAP2FileTransferSession_t *iAP2FileXferSession);
    tBoolean iAPDataRcvdPlaylistItems(const tMountPoint mountPoint, const iAP2FileTransferSession_t *iAP2FileXferSession);
    tBoolean iAPSuccessPlaylistItems(const tMountPoint mountPoint, const iAP2FileTransferSession_t *iAP2FileXferSession);
#endif

    void iAPGetMountPointFromDeviceID(const tDeviceID deviceID, tMountPoint &mountPoint);
    bool iAPGetMountOfFirstHostModeDevice(tMountPoint &mountPoint);
    void iAPSetOption(const tMountPoint mountPoint, const tiPodOptions option, const bool set);
    bool iAPGetOption(const tMountPoint mountPoint, const tiPodOptions option);

    vector<string> iAPGetNumberOfiAP2Link();
    tResult iAP1_GetPlaylistNameList(const tMountPoint mountPoint,const tIndex index,const tListSize chunk,tIPODDBRecordsPtr iPODDBRecordsPtr);
    tResult iAP1_GetVideoUIDList(const tMountPoint mountPoint,const tIndex index,const tListSize chunk,tIPODTrackInfosPtr trackInfosPtr);
    tResult iAP1_GetTrackUIDList(const tMountPoint mountPoint,const tIndex index,const tListSize chunk,tIPODTrackInfosPtr trackInfosPtr);
    tResult iAP2_StartUSBDeviceModeAudio(const tMountPoint mountPoint);
    tResult iAP2_StopUSBDeviceModeAudio(const tMountPoint mountPoint);
    tResult iAP2_StopMediaLibraryUpdates(const tMountPoint mountPoint);
    iAP2DeviceErrorState_t iAP2_GetDeviceErrorState(iAP2Device_t* device, void* context);
    tResult iAP2_AccessoryWiFiConfigurationInformation(const tMountPoint mountPoint, tWiFiAPCredentials wifiCredentials);
    tResult iAP2_OOBTBTPairingAccessoryInformation(const tMountPoint mountPoint);
    tResult iAP2_OOBTBTPairingCompletionInformation(const tMountPoint mountPoint,const bool result);
    tResult iAP2_InitiateCall(const tMountPoint mountPoint, const tDiPOInitiateCall initiateCall);
    tResult iAP2_AcceptCall(const tMountPoint mountPoint, const tDiPOAcceptCall acceptCall);
    tResult iAP2_EndCall(const tMountPoint mountPoint, const tDiPOEndCall endCall);
    tResult iAP2_SwapCalls(const tMountPoint mountPoint);
    tResult iAP2_StartRouteGuidance(const tMountPoint mountPoint,tUInt RouteGuidanceDisplayComponentID, tBoolean SourceName, tBoolean SourceSupportsRouteGuidance);
    tResult iAP2_StopRouteGuidance(const tMountPoint mountPoint,tUInt RouteGuidanceDisplayComponentID);

    tResult iAPUpdateCloseAllActiveSessions(const tMountPoint mountPoint);
    tResult iAP2UpdateAllAppControlProperties(const tMountPoint mountPoint);
    bool iAPIsKnownUSBiAP1Device(const tMountPoint mountPoint);
    static tResult InitCarPlayMicroChipHub();
    static tResult TerminateCarPlayMicroChipHub();
    tDiPOPower iAP2GetPowerUpdate(const tMountPoint mountPoint);
    void iAP2SetPowerUpdate(const tMountPoint mountPoint, const tDiPOPower power);

        static tResult InitCarPlayMicroChipMitsumiPort2ES2();
        static tResult TerminateCarPlayMicroChipMitsumiPort2ES2();

        static tResult InitCarPlayMicroChipMitsumiPort2ES4();
        static tResult TerminateCarPlayMicroChipMitsumiPort2ES4();

    tResult GetBTLimitationMode(const tMountPoint mountPoint, tBTLimitationActionState& btLimitationActionSate);
    void SetBTLimitationMode(const tMountPoint mountPoint, tBTLimitationActionState btLimitationActionState);
    void SetBTLimitationModeConnectionState(const tMountPoint mountPoint, tConnectionState connectionState);
    void SetIsCPWActiveForUSBMountpoint(const tMountPoint mountPoint,const tBoolean isCPWActive);
    void SetAppleDeviceTransportIdentifiers(const tMountPoint mountpoint,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber);
    tResult GetBTLimitationModeConnectionState(const tMountPoint mountPoint, tConnectionState& connectionState);
    tResult GetIsCPWActiveForUSBMountPoint(const tMountPoint mountPoint, tBoolean& isCPWActive);
    tResult GetCPWMountPointbyUSBMountPoint(const tMountPoint mountPoint, tMountPoint& CPWMountPoint);
    tResult GetUSBMountPointbyCPWMountPoint(const tMountPoint mountPoint, tMountPoint& USBMountPoint);
    tResult iAP2_IdentificationInformationUpdate();
    tResult iAP2_PlaybackHIDCommand(const tPlaybackHIDCommand playbackHIDCommand, const tBTButtonEvent keyEvent, const tMountPoint mountPoint);
    void iAPSetDeviceUUID(const tMountPoint mountPoint, const tUUID deviceUUID);
    tDiPODeviceTime iAP2GetDeviceTimeUpdate(const tMountPoint mountPoint);
    void iAP2SetDeviceTimeUpdate(const tMountPoint mountPoint, const tDiPODeviceTime deviceTime);

private:
     static int send_libusb_control_transfer(libusb_device_handle *dev_handle,uint8_t request_type, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, unsigned char *data, uint16_t wLength, unsigned int timeout, uint8_t numOfRetrials=NO_RETRIAL);

    //Init
    void DisconnectAllOpenIAPConnections();
    void iAP1_IdentifyDevice(); /**< enter extended interface mode and read out options */
#ifdef IPODCONTROL_IAP2_PF_AVAIL
    int iAP2_PrepareFDsForPolling(iAP2GetPollFDs_t* getPollFDs, int* maxfd, fd_set* to_readfds, fd_set* to_writefds);
    iAP2Device_t * iAP2_InitDeviceStructure(const tMountPoint mountPoint, const tBoolean hostmode, const tBoolean carPlay, const tBoolean nativeTransport, const tBoolean wirelessCarplay,const tBoolean carLife = false);
#endif
    tResult iAP2_IdentifyDevice(); /**< enter extended interface mode and read out options */

    int SwitchToHostMode(const tMountPoint mountPoint);
        int SwitchToDeviceMode(const tMountPoint mountPoint, const tConnectionState lastConnectionState);
        tResult SwitchCarPlayMicroChipHub(const tMountPoint mountPoint, const int otgPort);
        tResult SwitchCarPlayMicroChipMitsumiPort2ES2Hub(const tMountPoint mountPoint, const int otgPort);
        tResult SwitchCarPlayMicroChipMitsumiPort2ES4Hub(const tMountPoint mountPoint, const int otgPort);

        S32 iap2CommonWrite(const char* otgGlob, const char* key, const char* value);
        S32 iap2CommonRead(const char* path, const char* subPath, char* value, tSize &len);

        tBoolean iAP2HasCarPlayCapability(const tMountPoint mountPoint);

        tBoolean iAP2HasTargetOTGPort(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);
        tBoolean iAP2HasMicrochipHub(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);
        tBoolean iAP2HasUnwiredHub(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);
        tBoolean iAP2HasMitsumi2PortES2Hub(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);
        tBoolean iAP2HasMitsumi2PortES4Hub(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);
        tBoolean iAP2HasMolexHub(IN const char* pSysPath, OUT int &otgPort, OUT tPath &otgPath, OUT tPath &powerPortPath, OUT tPath &udcDevice, OUT tPath &hubVendorID, OUT tPath &hubProductID);


    //Role Switch functions
        S32 load_modules(const bool carPlay);
        S32 unload_modules_pre_umount();
        S32 unload_modules();
        S32 iap2VendorRequestMonitor_Begin(iAP2USBVendorRequestMonitor_t* thisPtr, U16 vid, U16 pid, const char* serial);
        S32 iap2VendorRequestMonitor_WaitAndEnd(iAP2USBVendorRequestMonitor_t* thisPtr);
        void iap2VendorRequestMonitor_Release(iAP2USBVendorRequestMonitor_t* thisPtr);
        static S32 iap2findGlobPath(char* otgPath, const tSize len, const char * key, const char* otgGlob);
        static S32 _findDevice(iAP2USBVendorRequestMonitor_t* thisPtr, U16 vid, U16 pid, const char* serial);
        static S32 _compareSysPath(struct udev_device* device, const char* sysPath, BOOL* matches /* out */);
        static S32 _compareDeviceIds(iAP2USBVendorRequestMonitor_t* thisPtr, struct udev_device* device, const char* path, U16 vid, U16 pid, const char* serial, BOOL* found);
        static S32 _checkForDeviceRemoval(iAP2USBVendorRequestMonitor_t* thisPtr, BOOL* deviceFound);

    //Callbacks
    static void CBIAP1_USBDetach(const IPODCONTROL_ID_TYPE iPodID); /**< callback USB detach */
#ifndef TARGET_BUILD_GEN3
    static void    CBIAP1_USBAttach(const S32 success, const IPODCONTROL_ID_TYPE iPodID); /**< callback USB attach */
#else
    static void CBIAP1_USBAttach(const S32 success, IPOD_CONNECTION_TYPE connection, const IPODCONTROL_ID_TYPE iPodID); /**< callback USB attach */
#endif
    static void CBIAP1_Notification(IPOD_NOTIFY_TYPE type,
            IPOD_NOTIFY_STATUS status, const IPODCONTROL_ID_TYPE iPodID); /**< callback event notification */
#ifdef TARGET_BUILD_GEN3
    static void CBIAP1_RemoteEventNotification(IPOD_STATE_INFO_TYPE eventNum, IPOD_REMOTE_EVENT_NOTIFY_STATUS eventData, const U32 iPodID); /**< callback event notification */
#endif
    static void CBIAP1_NotifyStatus(IPOD_CHANGED_PLAY_STATUS status, U64 param,
            const IPODCONTROL_ID_TYPE iPodID); /**< callback play status change notification */
    static void CBIAP1_NotifyStateChange(IPOD_STATE_CHANGE stateChange,
            const IPODCONTROL_ID_TYPE iPodID); /**< callback power status change notification */
    static void CBIAP1_NewiPodTrackInfo(U32 uNewSampleRate, S32 sNewSoundCheckValue,
            S32 sNewTrackVolAdjustment, const IPODCONTROL_ID_TYPE iPodID); /**< callback new audio track info */
    static S32 CBIAP1_OpenDataSession(U8 protocolIndex, U16 sessionId,
            const IPODCONTROL_ID_TYPE iPodID); /**< callback opened session to app info */
    static void CBIAP1_CloseDataSession(U16 sessionId,
            const IPODCONTROL_ID_TYPE iPodID);/**< callback closed session to app */
    static S32 CBIAP1_DataTransfer(U16 sessionId, U8 *data, U16 length,
            const IPODCONTROL_ID_TYPE iPodID); /**< callback app sent data */
    static void CBIAP1_RetrieveCategorizedDBRecord(U32 index, U8* str IPODCONTROL_ID_END_PARAM); /**< callback DB Record */
    static void CBIAP1_TrackInfo(U64 index, IPOD_TRACK_INFORMATION_TYPE infoType,
            IPOD_TRACK_INFORMATION_DATA *infoData IPODCONTROL_ID_END_PARAM); /**< callback TrackInfo */
    static void CBIAP1_PlayingTrackInfo(IPOD_TRACK_INFO_TYPE infoType,
            const IPOD_TRACK_CAP_INFO_DATA *capData,
            const IPOD_TRACK_RELEASE_DATE_DATA *releaseData,
            const IPOD_TRACK_ARTWORK_COUNT_DATA *artworkCountData,
            U8* stringBuf IPODCONTROL_ID_END_PARAM); /**< callback IndexedPlayingTrackInfo */
    static void CBIAP1_ArtworkData(IPOD_ALBUM_ARTWORK *artworkData,
            const IPODCONTROL_ID_TYPE iPodID); /**< callback ArtworkData */

    //IAP2 callbacks
#ifdef IPODCONTROL_IAP2_PF_AVAIL
    static S32 CBIAP2_AuthenticationFailed(iAP2Device_t* iap2Device, iAP2AuthenticationFailedParameter* authParameter, void* context);
    static S32 CBIAP2_AuthenticationSucceeded(iAP2Device_t* iap2Device, iAP2AuthenticationSucceededParameter* authParameter, void* context);
    static S32 CBIAP2_IdentificationAccepted(iAP2Device_t* iap2Device, iAP2IdentificationAcceptedParameter* idParameter, void* context);
    static S32 CBIAP2_IdentificationRejected(iAP2Device_t* iap2Device, iAP2IdentificationRejectedParameter* idParameter, void* context);
    static S32 CBIAP2_PowerUpdate(iAP2Device_t* iap2Device, iAP2PowerUpdateParameter* powerupdateParameter, void* context);
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
    static S32 CBIAP2_RouteGuidanceUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceUpdateParameter* RouteGuidanceUpdateParameter, void* context);
    static S32 CBIAP2_RouteGuidanceManeuverUpdate(iAP2Device_t* iap2Device, iAP2RouteGuidanceManeuverUpdateParameter* RouteGuidanceManeuverUpdateParameter, void* context);
#endif
#endif
    static S32 CBIAP2_MediaLibraryInfo(iAP2Device_t* iap2Device, iAP2MediaLibraryInformationParameter* mediaLibraryInfoParameter, void* context);
    static S32 CBIAP2_MediaLibraryUpdates(iAP2Device_t* iap2Device, iAP2MediaLibraryUpdateParameter* mediaLibraryUpdateParameter, void* context);
    static S32 CBIAP2_NowPlayingUpdate(iAP2Device_t* iap2Device, iAP2NowPlayingUpdateParameter* nowPlayingUpdateParameter, void* context);
    static S32 CBIAP2_USBDeviceModeAudioInformation(iAP2Device_t* iap2Device, iAP2USBDeviceModeAudioInformationParameter* usbDeviceModeAudioInformationParameter, void* context);
    static S32 CBIAP2_StartExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StartExternalAccessoryProtocolSessionParameter* startExternalAccessoryProtocolSessionParameter, void* context);
    static S32 CBIAP2_StopExternalAccessoryProtocolSession(iAP2Device_t* iap2Device, iAP2StopExternalAccessoryProtocolSessionParameter* stopExternalAccessoryProtocolSessionParameter, void* context);
    static S32 CBIAP2_StartEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context);
    static S32 CBIAP2_StopEANativeTransport(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8 sinkEndpoint, U8 sourceEndpoint, void* context);
    static S32 CBIAP2_iOSAppDataReceived(iAP2Device_t* iap2Device, U8 iAP2iOSAppIdentifier, U8* iAP2iOSAppDataRxd, U16 iAP2iOSAppDataLength, void* context);
    static S32 CBIAP2_FileTransferSetup(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferDataRcvd(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferSuccess(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferFailure(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferCancel(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferPause(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_FileTransferResume(iAP2Device_t* iap2Device, iAP2FileTransferSession_t* iAP2FileXferSession, void* context);
    static S32 CBIAP2_DeviceState(iAP2Device_t* iap2Device, iAP2DeviceState_t dState, void* context);
    static S32 CBIAP2_DeviceInformationUpdate(iAP2Device_t* iap2Device, iAP2DeviceInformationUpdateParameter *deviceInformationUpdateParameter, void* context);
    static S32 CBIAP2_DeviceLanguageUpdate(iAP2Device_t* iap2Device, iAP2DeviceLanguageUpdateParameter *deviceLanguageUpdateParameter, void* context);
    static S32 CBIAP2_DeviceTimeUpdate(iAP2Device_t* iap2Device, iAP2DeviceTimeUpdateParameter *deviceTimeUpdateParameter, void* context);
    static S32 CBIAP2_DeviceHIDReport(iAP2Device_t* thisDevice, iAP2DeviceHIDReportParameter* thisParameter, void* context);
    static S32 CBIAP2_StartLocationInformation(iAP2Device_t* iap2Device, iAP2StartLocationInformationParameter* startLocationInformationParameter, void* context);
    static S32 CBIAP2_StopLocationInformation(iAP2Device_t* iap2Device, iAP2StopLocationInformationParameter* stopLocationInformationParameter, void* context);
    static S32 CBIAP2_GPRMCDataStatusValuesNotification(iAP2Device_t* iap2Device, iAP2GPRMCDataStatusValuesNotificationParameter* GPRMCDataStatusValuesNotificationParameter, void* context);
    static S32 CBIAP2_BluetoothConnectionUpdate(iAP2Device_t* iap2Device, iAP2BluetoothConnectionUpdateParameter* bluetoothConnectionUpdateParameter, void* context);
    static S32 CBIAP2_StartVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StartVehicleStatusUpdatesParameter* startVehicleStatusUpdatesParameter, void* context);
    static S32 CBIAP2_StopVehicleStatusUpdates(iAP2Device_t* iap2Device, iAP2StopVehicleStatusUpdatesParameter* stopVehicleStatusUpdatesParameter, void* context);
    static S32 CBIAP2_ListUpdate(iAP2Device_t* iap2Device, iAP2ListUpdateParameter* listUpdateParameter, void* context);
#ifdef IPODCONTROL_IAP2_PF_R22
    static S32 CBIAP2_CallStateUpdate(iAP2Device_t* iap2Device, iAP2CallStateUpdateParameter* callStateUpdateParameter, void* context);
    static S32 CBIAP2_CommunicationsUpdate(iAP2Device_t* iap2Device, iAP2CommunicationsUpdateParameter* communicationsUpdateParameter, void* context);
#else
    static S32 CBIAP2_TelephonyCallStateInformation(iAP2Device_t* iap2Device, iAP2TelephonyCallStateInformationParameter* telephonyCallStateInformationParameter, void* context);
    static S32 CBIAP2_TelephonyUpdate(iAP2Device_t* iap2Device, iAP2TelephonyUpdateParameter* telephonyUpdateParameter, void* context);
#endif //IPODCONTROL_IAP2_PF_R22
    //CARPLAY WIFI:
    static S32 CBIAP2_DeviceUUIDUpdate(iAP2Device_t* iap2Device, iAP2DeviceUUIDUpdateParameter* deviceUUIDUpdateParameter, void* context);
    static S32 CBIAP2_WirelessCarPlayUpdate(iAP2Device_t* iap2Device, iAP2WirelessCarPlayUpdateParameter* wirelessCarPlayUpdateParameter, void* context);
    static S32 CBIAP2_RequestAccessoryWiFiConfigurationInformation(iAP2Device_t* iap2Device, iAP2RequestAccessoryWiFiConfigurationInformationParameter *requestAccessoryWiFiConfigurationInformationParameter, void* context);
#ifdef IPODCONTROL_IAP2_PF_R26
#ifdef TARGET_BUILD
    static S32 CBIAP2_DeviceTransportIdentifierNotification(iAP2Device_t* iap2Device, iAP2DeviceTransportIdentifierNotificationParameter *deviceTransportIdentifierNotificationParameter, void* context);
#endif
#endif

#ifdef IPODCONTROL_IAP2_PF_OOBBT_AVAIL
#ifdef TARGET_BUILD
    static S32 CBIAP2_StartOOBBTPairing(iAP2Device_t* iap2Device, iAP2StartOOBBTPairingParameter* startOOBBTPairingParameter, void* context);
    static S32 CBIAP2_OOBBTPairingLinkKeyInformation(iAP2Device_t* iap2Device, iAP2OOBBTPairingLinkKeyInformationParameter* oobBTPairingLinkKeyInformationParameter, void* context);
#endif
#endif
#endif //IPODCONTROL_IAP2_PF_AVAIL

    //Multiple support
    tBoolean SelectDevice(const tMountPoint mountPoint); /**< select device by mountPoint*/

    //General
    tBoolean IsIAP2(const tMountPoint mountPoint); /**< check for IAP2 */
    tBoolean IsIAP2HostMode(const tMountPoint mountPoint); /**< check for IAP2 Host Mode */
    tBoolean IsIAP2CarPlayMode(const tMountPoint mountPoint); /**< check for IAP2 Host+Dipo Mode */
    tBoolean IsIAP2WirelessCarPlayMode(const tMountPoint mountPoint); /**<check for IAP2 over Carplay Wireless */
    tBoolean IsIAP2NativeTransportMode(const tMountPoint mountPoint); /**< check for IAP2 NativeTransport Mode */
    tBoolean IsIAP2NativeTransportCarlifeMode(const tMountPoint mountPoint);
    tBoolean IsIAP2HostModePossible(const tMountPoint mountPoint); /**< check if IAP2 Host Mode is possible*/
    tBoolean IsIAP2iTunesRadioPlaying(const tMountPoint mountPoint); /**< check if IAP2 itunes is playing*/
    tBoolean IsIAP2iTunesRadioLibrarySupported(const tMountPoint mountPoint);
    int GetNumberOfActiveHostModeConnections(); /**< return number of host mode connections */
    tBoolean CheckFocusMusicApp(); /**< check focus app and resume to iPod/music app */
    tBoolean IsFocusMusicApp(); /**< check focus app for iPod/music app */
    tBoolean SetConfiOSApp(const tMountPoint mountPoint); /**< configure iOS AppInfo support */

    //Playback
    tBoolean PlayControl(const IPOD_PLAY_CONTROL control); /**< playback control toggle play/pause, fast forward, etc */
    tBoolean iAP1_SeekTo(const tPlaytime position); /**< seek to */
    tBoolean PlayRecord(iPodControlMediaPath &path, int const index, const U64 uuid); /**< select and play record */
    tBoolean PlayChapter(int const chapter); /**< select and play record */
    tResult PlayMediaLibraryItems(const tU64 uuid, const tBatchPlaybackList * batchlist, const bool isiTunesRadio); /**< IAP2 play items by uuid */
    tResult PlayMediaLibraryCurrentSelection(const bool isiTunesRadio); /**< IAP2 play current item */
    tResult PlayMediaLibraryCollection(const tU64 collectionID, const iAP2MediaLibraryCollectionType collectionType, const int collectionStartIndexconst, const bool isiTunesRadio);
    tResult iAP2_SeekTo(const tPlaytime position); /**< seek to */
    tResult iAP2SetPlaybackQueueIndex(const int queueIndex); /**< set nowplaying queue index */

    tBoolean
    GetPlayingTitle(tMetadata &metadata, const int index); /**< get current playing metadata fields */
    tBoolean
    GetPlayingAlbum(tMetadata &metadata, const int index); /**< get current playing metadata fields */
    tBoolean
    GetPlayingArtist(tMetadata &metadata,const int index); /**< get current playing metadata fields */
    tBoolean
    GetPlayingGenre(tMetadata &metadata, const int index); /**< get current playing metadata fields */
    tBoolean
    GetPlayingChapter(tMetadata &metadata, const int index); /**< get current playing metadata fields */

    tBoolean GetPlayingTrackInfo(tIPODPlayingTrackInfo *ptrackInfo,
            const int index, const int chapter = 0); /**< get current playing track info */
    tBoolean GetTrackArtworkTimes(U32 trackIndex, U16 formatId,
            U16 artworkIndex, U16 artworkCount, U16* resultCount, U32* buffer); /**< get current artwork times */
    tBoolean GetTrackArtworkData(U32 trackIndex, U16 formatId, U32 timeOffset); /**< get current artwork data */

    //updater
    tBoolean UpdatePlayStatus();     /**< update iPod play status */
    tBool UpdateNowPlaying(const bool updatePlayStatus);                        /**< update iPod now playing index, metadata and mediatype */

    //Database
    tBoolean SelectRecord(iPodControlMediaPath &path);                             /**< select database record */
    tBoolean SelectDBRecord(const int cat, const int index);                     /**< select record */
    tBoolean SelectDBRecordCategory(iPodControlMediaPath &path, const int cat); /**< select database record category */
    tBoolean RetrieveName(const iPodControlMediaPath path, const int index,
            tMetadata &name);                                                     /**< retrieve name from DBRecord */
    tBoolean RetrieveCategorizedDBRecords(tIPODDBRecords *pRecords,
            const int cat, const int index, const int count);                     /**< retrieve DB record*/
    tBoolean GetTrackInfo(tIPODTrackInfos *pTrackInfos,
            const IPOD_TRACK_INFORMATION_BITFIELD bitfield, const int index,
            const int count, const bool fromDB);                                 /**< retrieve trackInfo*/
    tBoolean ResetDBSelection(iPodControlMediaPath &path);                         /**< reset database selection */
    int GetNumberCategorizedDBRecords(const int cat);                             /**< number of DB records */

    tResult IndexerStrategy1(iPodControlIndexerContext * pcontext);             /**< run indexer strategy 1 according to context */
    tResult IndexerStrategy2(iPodControlIndexerContext * pcontext);             /**< run indexer strategy 2 according to context */
    tResult IndexerStrategy2Tracks(iPodControlIndexerContext * pcontext);         /**< run indexer strategy 2 according to context */
    tResult IndexerStrategy2Videos(iPodControlIndexerContext * pcontext);         /**< run indexer strategy 2 according to context */
    tResult IndexerStrategy2Playlists(iPodControlIndexerContext * pcontext);    /**< run indexer strategy all according to context */

    tResult MediaObjectFromTrackInfo(iPodControlIndexerContext * pcontext,
            const tIPODTrackInfos trackInfos, const iPodControlMediaPath path,
            const int trackIndex);                                                 /**< return mediaObject for index from trackInfo map */

    tBoolean MediaObjectFromDBRecord(iPodControlIndexerContext * pcontext,
            const tIPODDBRecords dbrecords, const iPodControlMediaPath path,
            const int trackIndex);                                                 /**< return mediaObject for index from dbrecords map */

    //Display
    //Display remote
    //Storage
    //VTIPOD
    void PushVTRecord(const tMountPoint mountPoint, const iPodControlVTRecordKey key, const string value);     /**< push new vt record cache entry */

    tBoolean IsStreaming(const tMountPoint mountPoint); //return streaming mode ON/OFF
    tBoolean SendMediaPlaybackHIDReport(const IPOD_HID_REPORT report);
    tResult GetBTMAC(tDeviceAddress &deviceAddress);

    //CMG3G-8083 - iPodcommunication Error update to HMI
    void UpdateIpodCommunicationError(tResult res);

protected:
    Lock mMapMutex;    /**< lock access to protected maps */
    tIPODDBRecordPtrMap mDBRecordPtrMap;
    tIPODTrackInfoPtrMap mTrackInfoPtrMap;
    tIPODPlayingTrackInfoPtrMap mPlayingTrackInfoPtrMap;

private:
    bool IsModeliPodNano3G(const char * pstr) const {
        return (pstr && pstr[0] != 0 && 0 != strstr("MA978,PA978,PA979,MB245," //3G iPod nano: 4GB silver
                    "MA980,PA980,PA981,MB247," //3G iPod nano: 8GB silver
                    "MB249,PB249,PB250,MB251," //3G iPod nano: 8GB blue
                    "MB253,PB253,PB254,MB255," //3G iPod nano: 8GB green
                    "MB257,PB257,PB258,MB259," //3G iPod nano: 8GB red
                    "MB261,PB261,PB262,MB263," //3G iPod nano: 8GB black
                    "MB453,PB453,PB454,MB455,MB456", pstr));//3G iPod nano: 8GB pink
    };

    bool IsModeliPodClassic80GB(const char * pstr) const {
        return (pstr && pstr[0] != 0 && 0 != strstr("MB029,PB029,PB031," //iPod classic: 80GB silver
                    "MB147,PB147,PB148", pstr)); //iPod classic: 80GB black
    };

    int m_IAP1Init;             /**< IAP1 initialize state of iAP1 library */
    tMountPoint m_MountPoint;    /**< IAP1/IAP2 device mountPoint*/

    const unsigned long long m_NotificationMask;        /**< notification mask*/
    const unsigned int m_RemoteEventNotificationMask;   /**< notification mask*/

    iPodControlHandleMap m_HandleMap; /**< map class to manage multiple iPod handles */

    tIPODVTCache mVTCache;                    /**< VTRecord cache map */
    Lock mVTCacheMutex;                     /**< lock access to VTCache */

    map<string, int> mHostModeErrorCount;

    static const U8 sHIDReportDescriptor[];
    static const U8 sHIDReportDescriptor_NoRepeat[];

    static const iAP2USBDeviceModeAudioSampleRate USBDeviceSupportedAudioSampleRate[];

    static vector<string> sKnownUSBiAP1Devices;
    vector<string> mKnownUSBiAP1Devices;
};

#endif /*IPODCONTROLIAP_H_*/

/** @} */
