/**
 * @addtogroup iPodControl
 * @author Ulrich Deuper
 *
 * iPod handle map class for multiple device support
 * @{
 */

#ifndef IPODCONTROLHANDLEMAP_H_
#define IPODCONTROLHANDLEMAP_H_

#include <vector>
#include "iPodControlHandle.h"
#include "Lock.h"

using namespace std;

/**
 * iPod control handle map class
 */
class iPodControlHandleMap
{

public:
    iPodControlHandleMap(); /**< constructor */
    void Clear();             /**< Clear all map entries */
    void AddHandle(const tDeviceInfo deviceInfo);            /**< add a new device handle */
    void UpdateHandle(const tDeviceInfo deviceInfo);        /**< update device handle (alsa device, etc.) */

    vector<string> GetAll(); /**< return a list of all mountpoints from list */

    void Reset(const tMountPoint mountPoint, const bool all = true);                /**< reset device handle */
    void ResetElapsedTime(const tMountPoint mountPoint, const tiPodElaspedTime elapsedtime);    /**< reset elapsed time for delayed iap call */
    tBoolean IsTimeElapsed( const tMountPoint mountPoint, const tiPodElaspedTime elapsedtime,
                const tPlaytime elapsed);                                                /**< return if 'interval' time has elapsed */
    void SuspendRemainingTime(const tMountPoint mountPoint,
            const tiPodElaspedTime elapsedtime,
            const unsigned int suspendlimit);                                        /**< check elapsed time for delayed iap call */
    int CreateConfiOSApp(IPOD_IOS_APP * &pApps, const tMountPoint mountPoint);        /**< get iOS app configuration */
    tBoolean AddAppInfo(const tMountPoint mountPoint, const tProtocolName protocol,
            const tBundleSeedID bundleID, const tAppName appName);                     /**< add app info */
    vector<tIPODAppInfo> GetAppInfos(const tMountPoint mountPoint, const bool hostmode, const bool nativeTansport);                 /**< get app info list */
    tBoolean RemoveAppInfo(const tMountPoint mountPoint,
            const tAppName appName, tSessionID &sessionID);                         /**< remove app info */



    tBoolean AddAppInfoFromSPI(const tMountPoint mountPoint,
            const tProtocolName protocol, const tBundleSeedID bundleID,
            const tAppName appName);
    vector<tIPODAppInfo> GetAppInfosFromSPI(const tMountPoint mountPoint );
    void ClearAppInfofromSPI(const tMountPoint mountPoint) ;


    void TraceOptions(const tMountPoint mountPoint);                                  /**< trace out iPod options */

    //Getter
    int GetiPodID(const tMountPoint mountPoint);                                    /**< iPod id assigned by InitDeviceConnection, platform call */
    int GetiPodID(const tDeviceID deviceID);                                        /**< iPod deviceID assigned by AddHandle */

    void * GetIAP2Device(const tMountPoint mountPoint);                             /**< IAP2 device pointer */
    void * GetIAP2InitParameter(const tMountPoint mountPoint);                      /**< IAP2 parameter pointer */
    void * GetIAP2AlbumArtBuffer(const tMountPoint mountPoint);                     /**< IAP2 albumart buffer pointer */
    void * GetQueueListBuffer(const tMountPoint mountPoint);                        /**< IAP2 QueueList buffer pointer */
    Lock * GetIAP2Mutex(const tMountPoint mountPoint);                                /**< IAP2 mutex pointer */
    void LockIAP2(const tMountPoint mountPoint);                                     /**< IAP2 stack lock */
    void UnlockIAP2(const tMountPoint mountPoint);                                     /**< IAP2 stack unlock */

    void GetMountPoint(tMountPoint &mountPoint, const int iPodID);                  /**< iPod mount point assigned by AddHandle */
    void GetMountPoint(tMountPoint &mountPoint, const void * iap2Device);           /**< iPod mount point assigned by AddHandle */

    void GetMountPointByPEHandle(tMountPoint &mountPoint, const tPEHandle handle);  /**< iPod mount point assigned by AddHandle */
    void GetMountPointFromDeviceID(tMountPoint &mountPoint, const tDeviceID id);    /**< iPod mount point from id */

    void GetDeviceInfo(const tMountPoint mountPoint, tDeviceInfo &deviceInfo);      /**< iPod deviceinfo */

    bool GetOption(const tMountPoint mountPointconst, tiPodOptions option);          /**< iPod option according to enum tiPodOptions */
    iPodControlMediaPath GetDBPath(const tMountPoint mountPoint);                    /**< get current selected db path */
    iPodControlMediaPath GetPBPath(const tMountPoint mountPoint);                    /**< get current selected pb path */
    int GetPlayerState(const tMountPoint mountPoint);                                 /**< iPod internal playing state */
    tPEPlaybackState GetPEPlaybackState(const tMountPoint mountPoint);                /**< iPod playback state, mediaplayer */
    tPEPlaybackState GetLastPlaybackState(const tMountPoint mountPoint);            /**< iPod playback state, mediaplayer */

    int GetNowPlayingTrackIndex(const tMountPoint mountPoint);                        /**< iPod current track index */
    int GetNowPlayingTrackCount(const tMountPoint mountPoint);                        /**< iPod current track index */
    int GetNowPlayingChapterIndex(const tMountPoint mountPoint);                    /**< iPod current track index */

    void GetNowPlayingMediaObject(tMediaObject &mediaObject, const tMountPoint mountPoint);   /**< iPod media object */

    int GetPollThreadIndex(const tMountPoint mountPoint);                           /**< iPod requested poll thread index */
    int GetPBTrackIndex(const tMountPoint mountPoint);                              /**< iPod requested playing track index */
    tPlaytime GetElapsedPlaytime(const tMountPoint mountPoint);                        /**< iPod elapsed track playtime */
    tPlaytime GetTotalPlaytime(const tMountPoint mountPoint);                        /**< iPod total track playtime */
    void GetDeviceName(tDeviceName &name, const tMountPoint mountPoint);            /**< iPod device name */
    void GetLaunchApp(tAppName &name, const tMountPoint mountPoint);                /**< iPod launch app name */
    void GetFocusApp(tAppName &name, const tMountPoint mountPoint);                    /**< iPod focus app name */
    void GetOTGPath(tPath &path, const tMountPoint mountPoint);                     /**< iPod OTG path */
    void GetPowerPortPath(tPath &path, const tMountPoint mountPoint);               /**< iPod PowerPort path */
    void GetUDCDevice(tPath &udc, const tMountPoint mountPoint);                    /**< iPod UDC path */
    void GetHubVendorID(tPath &vendor, const tMountPoint mountPoint);               /**< iPod hub vendor ID */
    void GetHubProductID(tPath &product, const tMountPoint mountPoint);             /**< iPod hub product ID */
    tBoolean GetRoleSwitched(const tMountPoint mountPoint);                         /**< iPod role switched */

    tConnectionState GetConnectionState(const tMountPoint mountPoint);                /**< iPod connection state */
    tConnectionType GetConnectionType(const tMountPoint mountPoint);                /**< iPod connection type */
    tDeviceID GetDeviceID(const tMountPoint mountPoint);                            /**< iPod device ID */
    me::samplerate_i GetSampleRate(const tMountPoint mountPoint);                        /**< iPod sample rate */
    tBitRate GetBitRate(const tMountPoint mountPoint);                              /**< iPod bit rate */
    tCodecType GetCodecType(const tMountPoint mountPoint);                          /**< iPod codec type */
    tBoolean GetLocationInformationUpdate(const tMountPoint mountPoint);            /**< iPod LocationInformationUpdate */
    tBoolean GetVehicleStatusUpdate(const tMountPoint mountPoint);                  /**< iPod VehicleStatusUpdate */

    iPodControlIndexerContext * GetIndexerContext(const tMountPoint mountPoint);    /**< iPod indexer context*/
    Lock * GetIndexerMutex(const tMountPoint mountPoint);                            /**< Indexer mutex pointer */
    void LockIndexer(const tMountPoint mountPoint);                                 /**< Indexer stack lock */
    void UnlockIndexer(const tMountPoint mountPoint);                                 /**< Indexer stack unlock */

    int GetAlbumArtID(const tMountPoint mountPoint);                                   /**< iPod album art format id */
    int GetQueueListID(const tMountPoint mountPoint);                               /**< iPod QueueList id */
    IPOD_OTG_TYPE GetOTGType(const tMountPoint mountPoint);                         /**< OTG type */
    int GetOTGPortNumber(const tMountPoint mountPoint);                             /**< OTG Number */

    IPOD_HID_REPORT GetLastHIDReport(const tMountPoint mountPoint);
    void GetAlbumArtString(tAlbumArt &albumArt, const tMountPoint mountPoint);        /**< iPod current albumart string */
    tSessionID GetSessionIDFromAppName(const tMountPoint mountPoint, const tAppName appName);    /**< get session id from app name */
    unsigned int GetProtocolIDFromAppName(const tMountPoint mountPoint, const tAppName appName);   /**< get protocol id from app name */
    void GetActiveSessionList(vector<string> &list, const tMountPoint mountPoint);    /**< get active session list */
    void GetAppNameBySessionID(const tMountPoint mountPoint,
                            const tSessionID sessionId, tAppName &appName);         /**< get appname by session id*/
    void GetAppNameByProtocolID(const tMountPoint mountPoint,
            const unsigned char protocolId, tAppName &appName,
            tSessionID &sessionId);                                                 /**< get appname by protocol id*/
    void GetProtocolNameByProtocolID(const tMountPoint mountPoint,
            const unsigned char protocolId, tProtocolName &protocolName);           /**< get ProtocolName by protocol id*/
    tBoolean GetTrackFinished(const tMountPoint mountPoint);                        /**< get status track has been finished in non-streaming mode */
    tBoolean GetStreamingMode(const tMountPoint mountPoint);                        /**< get streaming mode on/off */
    tPEHandle GetPEHandle(const tMountPoint mountPoint);                            /**< get PEHandle */

    tPlaybackMode GetPlaybackMode(const tMountPoint mountPoint);                    /**< get playback mode */
    tRepeatMode GetRepeatMode(const tMountPoint mountPoint);                        /**< get repeat mode */
    bool GetRepeatInitFlag(const tMountPoint mountPoint);

    tDiPOCallState GetCallState(const tMountPoint mountPoint);                      /**< get CallState */
    tDiPOCallStateItem GetCallStateByUUID(const tMountPoint mountPoint, const tGeneralString callUUID);

    tDiPOCommunications GetCommunications(const tMountPoint mountPoint);            /**< get Communications */
    tBTProfile GetBTProfile(const tMountPoint mountPoint);                          /**< get BTProfile */
    unsigned int GetCallDuration(const tMountPoint mountPoint);
    void GetDiPOLocationInfo(const tMountPoint mountPoint,tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType);
    void GetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues);
    tDiPOPower GetPowerUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceUpdate GetRGUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceManeuverUpdate GetRGManeuverUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceUpdate PullRGUpdate(const tMountPoint mountPoint);
    tDiPORouteGuidanceManeuverUpdate PullRGManeuverUpdate(const tMountPoint mountPoint);
    tDiPOCaps GetLastWirelessCarplayStatus(const tMountPoint mountPoint);
    tBTLimitationActionState GetBTLimitationActionState(const tMountPoint mountPoint);
    tConnectionState GetBTLimitationModeConnectionState(const tMountPoint mountPoint);
    tBoolean GetIsCPWActiveForUSBMountPoint(const tMountPoint mountPoint);
    void GetCPWMountPointbyUSBMountPoint(const tMountPoint mountPoint, tMountPoint& CPWMountPoint);
    void GetUSBMountPointbyCPWMountPoint(const tMountPoint mountPoint, tMountPoint& USBMountPoint);
    void GetDeviceUUID(const tMountPoint mountPoint, tUUID& deviceUUID);
    tDiPODeviceTime GetDeviceTimeUpdate(const tMountPoint mountPoint);
    bool GetWaitingforRoleSwitchResponse(const tMountPoint mountPoint);

    //Setter
    void SetDeviceUUID(const tMountPoint mountPoint, const tUUID deviceUUID);
    void SetiPodID(const tMountPoint mountPoint, const int iPodID);                    /**< set current iPod ID */
    void SetIAP2Device(const tMountPoint mountPoint, void * iap2Device);                /**< set current IAP2 device handle */
    void SetIAP2InitParameter(const tMountPoint mountPoint, void * iap2InitParameter);    /**< set current IAP2 init parameter handle */
    void SetIAP2AlbumArtBuffer(const tMountPoint mountPoint, void * iap2AlbumArtBuffer);/**< set current IAP2 albumArt buffer handle */
    void SetQueueListBuffer(const tMountPoint mountPoint, void * buffer);               /**< set queue list */

    void SetOption(const tMountPoint mountPoint, const tiPodOptions option, const bool set);    /**< set current iPod option */
    void SetDBPath(const tMountPoint mountPoint, const iPodControlMediaPath path);    /**< set current selected db path */
    void SetPBPath(const tMountPoint mountPoint, const iPodControlMediaPath path);    /**< set current selected pb path */
    void SetPlayerState(const tMountPoint mountPoint, const int state);                /**< set current iPod player state */
    void SetPEPlaybackState(const tMountPoint mountPoint, const tPEPlaybackState state);/**< set current Mediaplayer playback state */
    void SetLastPlaybackState(const tMountPoint mountPoint, const tPEPlaybackState state);/**< set current Mediaplayer playback state */

    void SetNowPlayingTrackIndex(const tMountPoint mountPoint, const int track);        /**< set current iPod track index */
    void SetNowPlayingTrackCount(const tMountPoint mountPoint, const int count);        /**< set current iPod track index */
    void SetNowPlayingChapterIndex(const tMountPoint mountPoint, const int track);        /**< set current iPod track index */

    void SetNowPlayingTags(const tMountPoint mountPoint, const tMediaType mediaType,
            const tMetadata MetadataField1, const tMetadata MetadataField2,
            const tMetadata MetadataField3, const tMetadata MetadataField4,
            const tTitle title, const tUUID uuid);                                         /**< set nowplaying data */

    void SetPollThreadIndex(const tMountPoint mountPoint, const int threadIndex);       /**< set requested poll thread index */
    void SetPBTrackIndex(const tMountPoint mountPoint, const int track);                /**< set requested playing iPod track index */
    void SetElapsedPlaytime(const tMountPoint mountPoint, const tPlaytime time);        /**< set current elapsed track playtime */
    void SetTotalPlaytime(const tMountPoint mountPoint, const tPlaytime time);            /**< set current total track playtime */
    void SetDeviceName(const tMountPoint mountPoint, const char* name);                    /**< set iPod device name */
    void SetLaunchApp(const tMountPoint mountPoint, const char* name);                    /**< set iPod launch app name */
    void SetFocusApp(const tMountPoint mountPoint, const char* name);                    /**< set iPod focus app name */
    void SetOTGPath(const tMountPoint mountPoint, const char* path);                    /**< set iPod OTG path */
    void SetPowerPortPath(const tMountPoint mountPoint, const char* path);              /**< set iPod PowerPort path */
    void SetUDCDevice(const tMountPoint mountPoint, const char* udc);                   /**< set iPod UDC device name */
    void SetHubVendorID(const tMountPoint mountPoint, const char* vendor);              /**< set iPod hub vendor ID */
    void SetHubProductID(const tMountPoint mountPoint, const char* product);            /**< set iPod hub product ID */
    void SetRoleSwitched(const tMountPoint mountPoint, const tBoolean state);           /**< set iPod role switched */

    void SetConnectionState(const tMountPoint mountPoint, const tConnectionState state);/**< set iPod connection state */
    void SetDeviceID(const tMountPoint mountPoint, const tDeviceID id);                    /**< set iPod device id */
    void SetSampleRate(const tMountPoint mountPoint, const me::samplerate_i sampleRate);     /**< set iPod sample rate */
    void SetBitRate(const tMountPoint mountPoint, const tBitRate bitRate);              /**< set iPod bit rate */
    void SetCodecType(const tMountPoint mountPoint, const tCodecType codecType);        /**< set iPod codec type */
    void SetLocationInformationUpdate(const tMountPoint mountPoint, const tBoolean LocationInformationUpdate);        /**< set iPod LocationInformationUpdate enabled */
    void SetVehicleStatusUpdate(const tMountPoint mountPoint, const tBoolean vehicleStatusUpdate);        /**< set iPod VehicleStatusUpdate enabled */

    void SetAlbumArtID(const tMountPoint mountPoint, const int id);                        /**< set iPod album art format id */
    void SetQueueListID(const tMountPoint mountPoint, const int id);                    /**< set iPod queue list */
    void SetOTGType(const tMountPoint mountPoint, const IPOD_OTG_TYPE type);            /**< OTG type */
    void SetOTGPortNumber(const tMountPoint mountPoint, const int port);                /**< OTG port number */

    void SetLastHIDReport(const tMountPoint mountPoint, const IPOD_HID_REPORT report);
    void SetAlbumArtString(const tMountPoint mountPoint, const tAlbumArt albumArt);       /**< set current iPod albumArt string */
    void SetSessionIDByIndex(const tMountPoint mountPoint,
            const tSessionID sessionId, const unsigned char protocolIndex,
            tAppName &appName);                                                         /**< set session id by protocol index */
    void ClearSessionID(const tMountPoint mountPoint,
            const tSessionID sessionId, tAppName &appName);                             /**< clear session id */

    void SetTrackFinished(const tMountPoint mountPoint, const tBoolean set);            /**< set status track has been finished in non-streaming mode */
    void SetStreamingMode(const tMountPoint mountPoint, const tBoolean set);            /**< set streaming mode on/off */
    void SetPEHandle(const tMountPoint mountPoint, const tPEHandle handle);                /**< set PEHandle */
    void SetPlaybackMode(const tMountPoint mountPoint, const tPlaybackMode mode);        /**< set playback mode */
    void SetRepeatMode(const tMountPoint mountPoint, const tRepeatMode mode);            /**< set repeat mode */
    void SetRepeatInitFlag(const tMountPoint, const bool flag);

    void SetBTProfile(const tMountPoint mountPoint, const tBTProfile btProfile);        /**< set tBTProfile */

    void SetNowPlayingMediaObject(const tMediaObject mediaObject, const tMountPoint mountPoint);

    void UpdateOrInsertCallState(const tMountPoint mountPoint, const tDiPOCallStateItem callStateItem);
    void RemoveDisconnectedCallState(const tMountPoint mountPoint);
    void SetCommunications(const tMountPoint mountPoint, const tDiPOCommunications comm); /**< Set Communications */
    void UpdateCallDuration(const tMountPoint mountPoint, const U32 callDuration);
    void SetDiPOLocationInfo(const tMountPoint mountPoint, const tBool startStopLocationInfo, const tDiPOLocationInfoType locationInfoType); /**< Set DiPOLocationInfo */
    void SetDiPOGPRMCDataStatusValues(const tMountPoint mountPoint,const tDiPOGPRMCDataStatusValues GPRMCDataStatusValues);
    void SetPowerUpdate(const tMountPoint mountPoint, const tDiPOPower power);
    void SetRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate);
    void PushRGUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceUpdate RGUpdate);
    void SetRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void PushRGManeuverUpdate(const tMountPoint mountPoint, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void SetLastWirelessCarplayStatus(const tMountPoint mountPoint, tDiPOCaps diPOCaps);
    void SetBTLimitationActionState(const tMountPoint mountPoint, tBTLimitationActionState btLimitation);
    void SetBTLimitationModeConnectionState(const tMountPoint mountPoint, tConnectionState connectionState);
    void SetIsCPWActiveForUSBMountpoint(const tMountPoint mountPoint, const tBoolean isCPWActive);
    void SetAppleDeviceTransportIdentifiers(const tMountPoint mountpoint,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber);
    void SetDeviceTimeUpdate(const tMountPoint mountPoint, const tDiPODeviceTime deviceTime);
    void SetWaitingforRoleSwitchResponse(const tMountPoint mountpoint, const bool isWaitingforRoleSwitchResponse);

private:
    //helpers
    int FindIndex(const tMountPoint mountPoint);                                         /**< find existing device handle by path */
    int FindIndex(const int iPodID);                                                     /**< find existing device handle by id*/
    int FindIndex(const void * iap2Device);                                                /**< find existing device handle by iap2Device*/

    void ResetPrivate(const int index, const bool all);                                  /**< reset device handle */
    void ResetElapsedTimePrivate(const int index, const tiPodElaspedTime elapsedtime);    /**< reset elapsed time for delayed iap call */
    tBoolean IsTimeElapsedPrivate( const int index, const tiPodElaspedTime elapsedtime,
                const tPlaytime elapsed);                                                /**< return if 'interval' time has elapsed */

    int CreateConfiOSAppPrivate(IPOD_IOS_APP * &pApps, const int index);                /**< get iOS app configuration */
    tBoolean AddAppInfoPrivate(const int index, const tProtocolName protocol,
            const tBundleSeedID bundleID, const tAppName appName);                         /**< add app info */
    vector<tIPODAppInfo> GetAppInfosPrivate(const int index, const bool hostmode, const bool nativeTansport);                           /**< add app info */
    tBoolean RemoveAppInfoPrivate(const int index, const tAppName appName,
            tSessionID &sessionID);                                                     /**< remove app info */


    tBoolean AddAppInfoFromSPIPrivate(const int indx,
            const tProtocolName protocol, const tBundleSeedID bundleID,
            const tAppName appName);
    vector<tIPODAppInfo> GetAppInfosFromSPIPrivate(const int indx);
    void ClearAppInfofromSPIPrivate(const int indx);

    void TraceOptionsPrivate(const int index);                                           /**< trace out iPod options */

    //Getter
    void GetDeviceInfoPrivate(const int index, tDeviceInfo &deviceInfo);   /**< iPod deviceinfo */
    bool GetOptionPrivate(const int index, const tiPodOptions option);                  /**< iPod option according to enum tiPodOptions */
    iPodControlMediaPath GetDBPathPrivate(const int index);                                /**< get current selected db path */
    iPodControlMediaPath GetPBPathPrivate(const int index);                                /**< get current selected pb path */
    int GetPlayerStatePrivate(const int index);                              /**< iPod internal playing state */
    tPEPlaybackState GetPEPlaybackStatePrivate(const int index);                        /**< iPod playback state, mediaplayer */
    tPEPlaybackState GetLastPlaybackStatePrivate(const int index);                      /**< iPod playback state, mediaplayer */

    int GetNowPlayingTrackIndexPrivate(const int index);                                /**< iPod current track index */
    int GetNowPlayingTrackCountPrivate(const int index);                                /**< iPod current track index */
    int GetNowPlayingChapterIndexPrivate(const int index);                                /**< iPod current track index */

    void GetNowPlayingMediaObjectPrivate(tMediaObject &mediaObject, const int index);   /**< iPod media object */

    int GetPollThreadIndexPrivate(const int index);                                     /**< iPod requested thread index */
    int GetPBTrackIndexPrivate(const int index);                                        /**< iPod requested playing track index */
    tPlaytime GetElapsedPlaytimePrivate(const int index);                                /**< iPod elapsed track playtime */
    tPlaytime GetTotalPlaytimePrivate(const int index);                                    /**< iPod total track playtime */
    void GetDeviceNamePrivate(tDeviceName &name, const int index);                        /**< iPod device name */
    void GetLaunchAppPrivate(tAppName &name, const int index);                            /**< iPod launch app name */
    void GetFocusAppPrivate(tAppName &name, const int index);                            /**< iPod focus app name */
    void GetOTGPathPrivate(tPath &path, const int index);                               /**< iPod OTG path */
    void GetPowerPortPathPrivate(tPath &path, const int index);                         /**< iPod PowerPort path */
    void GetUDCDevicePrivate(tPath &udc, const int index);                              /**< iPod UDC device name */
    void GetHubVendorIDPrivate(tPath &vendor, const int index);                         /**< iPod hub vendor ID */
    void GetHubProductIDPrivate(tPath &product, const int index);                       /**< iPod hub product ID */
    tBoolean GetRoleSwitchedPrivate(const int index);                                   /**< iPod role switched */

    tConnectionState GetConnectionStatePrivate(const int index);                        /**< iPod connection state */
    tConnectionType GetConnectionTypePrivate(const int index);                          /**< iPod connection type */
    tDeviceID GetDeviceIDPrivate(const int index);                                        /**< iPod device ID */
    me::samplerate_i GetSampleRatePrivate(const int index);                                  /**< iPod sample rate */
    tBitRate GetBitRatePrivate(const int index);                                        /**< iPod bit rate */
    tCodecType GetCodecTypePrivate(const int index);                                    /**< iPod codec type */
    tBoolean GetLocationInformationUpdatePrivate(const int index);                      /**< iPod locationInformationUpdate */
    tBoolean GetVehicleStatusUpdatePrivate(const int index);                            /**< iPod VehicleStatusUpdate */

    iPodControlIndexerContext * GetIndexerContextPrivate(const int index);              /**< iPod indexer context */
    int GetAlbumArtIDPrivate(const int index);                                             /**< iPod album art format id */
    int GetQueueListIDPrivate(const int index);                                         /**< iPod queue list id */
    IPOD_OTG_TYPE GetOTGTypePrivate(const int index);                                   /**< iPod otg type */
    int GetOTGPortNumberPrivate(const int index);                                       /**< iPod otg port */

    IPOD_HID_REPORT GetLastHIDReportPrivate(const int index);
    void GetAlbumArtStringPrivate(tAlbumArt &albumArt, const int index);                  /**< iPod current albumart string */
    tSessionID GetSessionIDFromAppNamePrivate(const int index, const tAppName appName);    /**< get session id from app name */
    unsigned int GetProtocolIDFromAppNamePrivate(const int index, const tAppName appName); /**< get protocol id from app name */
    void GetActiveSessionListPrivate(vector<string> &list, const int index);            /**< get active session list */
    void GetAppNameBySessionIDPrivate(const int index,
            const tSessionID sessionId, tAppName &appName);                             /**< get appname by session id*/
    void GetAppNameByProtocolIDPrivate(const int index,
                const unsigned char protocolId, tAppName &appName,
                tSessionID &sessionId);                                                 /**< get appname by protocol id*/
    void GetProtocolNameByProtocolIDPrivate(const int indx,
            const unsigned char protocolId, tProtocolName &protocolName);               /**< get ProtocolName by protocol id*/
    tBoolean GetTrackFinishedPrivate(const int index);                                    /**< get status track has been finished in non-streaming mode */
    tBoolean GetStreamingModePrivate(const int index);                                    /**< get streaming mode on/off */
    tPEHandle GetPEHandlePrivate(const int index);                                        /**< get PEHandle */
    tPlaybackMode GetPlaybackModePrivate(const int index);                                /**< get playback mode */
    tRepeatMode GetRepeatModePrivate(const int index);                                    /**< get repeat mode */
    bool GetRepeatInitFlagPrivate(const int index);

    tDiPOCallState GetCallStatePrivate(const int index);                                /**< get CallState */
    tDiPOCommunications GetCommunicationsPrivate(const int index);                      /**< get Communications */
    tBTProfile GetBTProfilePrivate(const int index);                                    /**< get BTProfile */
    unsigned int GetCallDurationPrivate(const int indx);
    void GetDiPOLocationInfoPrivate(const int indx,tBool &startStopLocationInfo, tDiPOLocationInfoType &locationInfoType);
    void GetDiPOGPRMCDataStatusValuesPrivate(const int indx, tDiPOGPRMCDataStatusValues &GPRMCDataStatusValues);
    tDiPOPower GetPowerUpdatePrivate(const int indx);
    tDiPORouteGuidanceUpdate GetRGUpdatePrivate(const int indx);
    tDiPORouteGuidanceManeuverUpdate GetRGManeuverUpdatePrivate(const int indx);
    tDiPORouteGuidanceUpdate PullRGUpdatePrivate(const int indx);
    tDiPORouteGuidanceManeuverUpdate PullRGManeuverUpdatePrivate(const int indx);
    tDiPOCaps GetLastWirelessCarplayStatusPrivate(const int indx);
    tBTLimitationActionState GetBTLimitationActionStatePrivate(const int indx);
    tConnectionState GetBTLimitationModeConnectionStatePrivate(const int indx);
    tBoolean GetIsCPWActiveForUSBMountPointPrivate(const int indx);
    void GetDeviceUUIDPrivate(const int indx, tUUID& deviceUUID);
    tDiPODeviceTime GetDeviceTimeUpdatePrivate(const int indx);
    bool GetWaitingforRoleSwitchResponsePrivate(const int indx);

    //Setter
    void SetOptionPrivate(const int index, const tiPodOptions option, const bool set);    /**< set current iPod option */
    void SetDBPathPrivate(const int indext, const iPodControlMediaPath path);            /**< set current selected db path */
    void SetPBPathPrivate(const int indext, const iPodControlMediaPath path);            /**< set current selected pb path */
    void SetPlayerStatePrivate(const int index, const int state);                        /**< set current iPod player state */
    void SetPEPlaybackStatePrivate(const int index, const tPEPlaybackState state);        /**< set current Mediaplayer playback state */
    void SetLastPlaybackStatePrivate(const int index, const tPEPlaybackState state);    /**< set current Mediaplayer playback state */

    void SetNowPlayingTrackIndexPrivate(const int index, const int track);                /**< set current iPod track index */
    void SetNowPlayingTrackCountPrivate(const int index, const int track);                /**< set current iPod track index */
    void SetNowPlayingChapterIndexPrivate(const int index, const int track);            /**< set current iPod track index */

    void SetNowPlayingMediaObjectPrivate(const tMediaObject mediaObject, const int index);/**< set media object */
    void SetNowPlayingTagsPrivate(const tMediaType mediaType,
            const tMetadata MetadataField1, const tMetadata MetadataField2,
            const tMetadata MetadataField3, const tMetadata MetadataField4,
            const tTitle title, const tUUID uuid, const int index);                     /**< set nowplaying data */

    void SetPollThreadIndexPrivate(const int index, const int threadIndex);             /**< set requested poll thread index */
    void SetPBTrackIndexPrivate(const int index, const int track);                      /**< set requested playing iPod track index */
    void SetElapsedPlaytimePrivate(const int index, const tPlaytime time);                /**< set current elapsed track playtime */
    void SetTotalPlaytimePrivate(const int index, const tPlaytime time);                /**< set current total track playtime */
    void SetDeviceNamePrivate(const int index, const char* name);                        /**< set iPod device name */
    void SetLaunchAppPrivate(const int index, const char* name);                        /**< set iPod launch app name */
    void SetFocusAppPrivate(const int index, const char* name);                            /**< set iPod focus app name */
    void SetOTGPathPrivate(const int index, const char* path);                          /**< set iPod OTG path */
    void SetPowerPortPathPrivate(const int index, const char* path);                    /**< set iPod PowerPort path */
    void SetUDCDevicePrivate(const int index, const char* udc);                         /**< set iPod UDC device name */
    void SetHubVendorIDPrivate(const int index, const char* vendor);                    /**< set iPod hub vendor ID */
    void SetHubProductIDPrivate(const int index, const char* product);                  /**< set iPod hub product ID */
    void SetRoleSwitchedPrivate(const int index, const tBoolean state);                 /**< set iPod roloe switched */

    void SetConnectionStatePrivate(const int index, const tConnectionState state);        /**< set iPod connection state */
    void SetDeviceIDPrivate(const int index, const tDeviceID id);                        /**< set iPod device id */
    void SetSampleRatePrivate(const int index, const me::samplerate_i sampleRate);           /**< set iPod sampling rate */
    void SetBitRatePrivate(const int index, const tBitRate bitRate);                    /**< set iPod bit rate */
    void SetCodecTypePrivate(const int index, const tCodecType codecType);              /**< set iPod codec type */
    void SetLocationInformationUpdatePrivate(const int index, const tBoolean LocationInformationUpdate);    /**< set iPod LocationInformationUpdate */
    void SetVehicleStatusUpdatePrivate(const int index, const tBoolean vehicleStatusUpdate);                /**< set iPod VehicleStatusUpdate */

    void SetAlbumArtIDPrivate(const int index, const int format);                        /**< set iPod album art format id */
    void SetQueueListIDPrivate(const int index, const int format);                      /**< set iPod queue list id */
    void SetOTGTypePrivate(const int index, const IPOD_OTG_TYPE);                       /**< set iPod OTG type */
    void SetOTGPortNumberPrivate(const int index, const int port);                      /**< set iPod OTG port number */

    void SetLastHIDReportPrivate(const int index, const IPOD_HID_REPORT report);
    void SetAlbumArtStringPrivate(const int index, const tAlbumArt albumArt);           /**< set current iPod albumart string*/
    void SetSessionIDByIndexPrivate(const int index,
            const tSessionID sessionId, const unsigned char protocolIndex,
            tAppName &appName);                                                         /**< set session id by protocol index */
    void ClearSessionIDPrivate(const int index, const tSessionID sessionId,
            tAppName &appName);                                                         /**< clear session id */

    void SetTrackFinishedPrivate(const int index, const tBoolean set);                    /**< set status track has been finished in non-streaming mode */
    void SetStreamingModePrivate(const int index, const tBoolean set);                    /**< set streaming mode on/off */
    void SetPEHandlePrivate(const int index, const tPEHandle handle);                    /**< set PEHandle*/
    void SetPlaybackModePrivate(const int index, const tPlaybackMode mode);                /**< set playback mode*/
    void SetRepeatModePrivate(const int index, const tRepeatMode mode);                    /**< set repeat mode*/
    void SetRepeatInitFlagPrivate(const int index, const bool flag);                    /**< set repeat Init flag*/
    void SetBTProfilePrivate(const int index, const tBTProfile btProfile);              /**< set tBTProfile*/

    void UpdateOrInsertCallStatePrivate(const int indx, const tDiPOCallStateItem callStateItem);
    void RemoveDisconnectedCallStatePrivate(const int indx);
    void SetCommunicationsPrivate(const int indx, const tDiPOCommunications comm);
    void UpdateCallDurationPrivate(const int indx, const U32 callDuration);
    void SetDiPOLocationInfoPrivate(const int indx,const tBool startStopLocationInfo, const tDiPOLocationInfoType locationInfoType);
    void SetDiPOGPRMCDataStatusValuesPrivate(const int indx,const tDiPOGPRMCDataStatusValues GPRMCDataStatusValues);
    void SetPowerUpdatePrivate(const int indx, const tDiPOPower power);
    void SetRGUpdatePrivate(const int indx, const tDiPORouteGuidanceUpdate RGUpdate);
    void PushRGUpdatePrivate(const int indx, const tDiPORouteGuidanceUpdate RGUpdate);
    void SetRGManeuverUpdatePrivate(const int indx, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void PushRGManeuverUpdatePrivate(const int indx, const tDiPORouteGuidanceManeuverUpdate RGManeuverUpdate);
    void SetLastWirelessCarplayStatusPrivate(const int indx,tDiPOCaps diPOCaps);
    void SetBTLimitationActionStatePrivate(const int indx,tBTLimitationActionState btLimitationActionState);
    void SetBTLimitationModeConnectionStatePrivate(const int indx, tConnectionState btLimitationModeConnectionState);
    void SetIsCPWActiveForUSBMountpointPrivate(const int indx, const tBoolean isCPWActive);
    void SetAppleDeviceTransportIdentifiersPrivate(const int indx,const tMACAddress appleDeviceMACAddress,const tUSBSerial appleDeviceUSBSerialNumber);
    void SetDeviceTimeUpdatePrivate(const int indx, const tDiPODeviceTime deviceTime);
    void SetWaitingforRoleSwitchResponsePrivate(const int indx, bool isWaitingforRoleSwitchResponse);
    Lock m_Mutex;                      /**< lock access to member variables */
    vector<iPodControlHandle *> mList; /**< list of iPod handles */
};

#endif /*IPODCONTROLHANDLEMAP_H_*/

/** @} */
