/**
 * @author Sadaiyandi Ramadoss
 *
 */

#ifndef _DVDControl_H_
#define _DVDControl_H_

#include "DVDControlInterface.h"
#include "DVDControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "DVDBackendCMD.h"
#include "TouchEventHandler.h"
#include <vector>

class DVDControl : public DVDControlSM , public ILocalSPM, public TFThread , public TouchEventListener
{

public:

    //SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the DVDControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    DVDControl(const tComponentID componentID):ILocalSPM(componentID){};

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~DVDControl(){};

    /**
     * This function is used by LocalSPM to create the DVDControl.
     * @attention: running in SPM thread context
     * Create the DVDControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
     * This function is used by LocalSPM to trigger the DVDControl initialization.
     * @attention: running in SPM thread context
     * Init the DVDControl state machine.
     * Register DVDControlSM with dispatcher.
     * Inform LocalSPM that Init is ready -> InitDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the DVDControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the DVDControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the DVDControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the DVDControl.
     * @attention: running in SPM thread context
     * Deregister DVDControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(tGeneralString stateName, size_t size);

    /**
     * Returns answer if the component is a state machine
     *
     * @return true or false
     */
    tBoolean IsComponentSM() { return true; }; //component is a state machine

    /**
     * Copies the metadata of current track from gracenote
     *
     * @return pointer to current track gracenote metadata
     */
    tResult GetTrackTextInfoFromGN(tFilesPtr GNmetadata, unsigned char index);

    //Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * For USB no need to assign audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint=NULL);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * For USB no need to deallocate audio input device.
     * Send event to own SM immediately -> SendEvent(METHOD_RETURN, returnValue = true).
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/);

    /**
     * Function transfers ...
     * For USB no need to start streaming.
     * if old PlayerEngine
     * - Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_LOADINGSTATE, reason, speed).
     * - Release own waiting state -> SendEvent(START_STREAMING_ANSWER).
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming(const tDeviceID deviceID, const tMountPoint mountPoint=NULL); //Roadmap 13010: 100%

    /**
     * Function transfers ...
     * For USB no need to stop streaming
     * Release own waiting state -> SendEvent(STOP_STREAMING_ANSWER, deviceID=DEVICE_ID_NOT_SET).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopStreaming();

    /**
     * Function answers of STOP request to waiting state machine via SMF
     * if old PlayerEngine
     * - PlaybackStatus(playbackState=PE_PBS_STOPPEDSTATE, metadata, metadata, metadata, metadata) -> SendAnswer.
     * else //Roadmap 13010
     * - PlaybackStatusNew(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed) -> SendAnswer.
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult StopStreamingAnswer(tDeviceID deviceID);

    /**
     * Function transfers play command to PlayerEngine (1st part).
     * Update media object to play via TagInfo
     * Verify genre name via DataProvider::VerifyGenreName(&genreName) //Roadmap 13014
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object //Roadmap 13010
     * @param[in] position absolute playtime in milliseconds
     * @param[in] streaming streaming flag //Roadmap 13008
     * @return != 0: error, = 0: OK
     */
    tResult StartPlay(const tDeviceType deviceType, //Roadmap 13010: 100%, 13014, 13008: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint,
            const tUUID uuid,
            const tPEHandle handle,
            const tPlaytime position=0,
            const tStreaming streaming=false);

    /**
     * Function transfers play command to PlayerEngine (2nd part).
     * if DRM protected //Roadmap 13010
     * - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason = DRM_ERROR).
     * if old PlayerEngine
     * - Send NewPlaySlot to PlayerEngine via DBUS -> NewPlaySlot(URL, position, playpointFormat=Absolute).
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(PLAY_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, handle, m_OutputDevice, URL, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     *   - Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] handle handle (= objectID) to map metadata to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult StartPlayWithMetadata(const tPEHandle handle, const tURL mountPointURL);

    /**
     * Function transfers stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopSlot to PlayerEngine via DBUS -> StopSlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=STOPPEDSTATE) later.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=STOP)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStop(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL);

    /**
     * Function transfers pause command to PlayerEngine.
     * if old PlayerEngine
     * - Send PauseSlot to PlayerEngine via DBUS -> PauseSlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PAUSEDSTATE) later, but we don't use it.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(PAUSE_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PAUSE)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PAUSEDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartPause(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL);

    /**
     * Function transfers resume command to PlayerEngine.
     * if old PlayerEngine
     * - Send ResumePlaySlot to PlayerEngine via DBUS -> ResumePlaySlot().
     *   PlayerEngine response is sent by PlaybackStatusResponseSlot(CurrentPlaybackStatus=PLAYINGSTATE) later, but we don't use it.
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(RESUME_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartResume(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL);

    /**
     * Function transfers ffwd_start command to PlayerEngine.
     * if old PlayerEngine
     * - Send StartFfSlot to PlayerEngine via DBUS -> StartFfSlot(rate).
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FFWD_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=rate)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_FASTFORWARDSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStart(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL,
            const tCueingRate rate=10,
            const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers ffwd_stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopFfSlot to PlayerEngine via DBUS -> StopFfSlot().
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FFWD_STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFfwdStop(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL);

    /**
     * Function transfers frev_start command to PlayerEngine.
     * if old PlayerEngine
     * - Send StartFrevSlot to PlayerEngine via DBUS -> StartFrevSlot(rate).
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FREV_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=-rate)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_FASTREVERSESTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] rate rate of cueing (10= after every second jump 10s)
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStart(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL,
            const tCueingRate rate=10,
            const speedstate_e IsPlaybackSpeed = ME_SPEEDSTATE_OFF);

    /**
     * Function transfers frev_stop command to PlayerEngine.
     * if old PlayerEngine
     * - Send StopFrevSlot to PlayerEngine via DBUS -> StopFrevSlot().
     * - Fake PlayerEngine response: Send event to own SM immediately -> SendEvent(PLAYBACK_STATUS_RESPONSE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(FREV_STOP_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, speed=1)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer(handle, playbackState=PE_PBS_PLAYINGSTATE, reason, speed).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartFrevStop(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL);

    /**
     * Function transfers seek_to command to PlayerEngine.
     * if old PlayerEngine
     * - Send SeekToSlot to PlayerEngine via DBUS -> SeekToSlot(position, playpointFormat=Absolute).
     * - Fake PlayerEngine response: Send event to own SM immediately
     *    -> SendEvent(TICK_TIME_ELAPSED, elapsedPlaytime=position, totalPlaytime=PLAYTIME_NONE).
     * else //Roadmap 13010
     * - Release own waiting state -> SendEvent(SEEK_TO_ANSWER).
     * - Call PEClient::Action(&returnValue, peState with playState=PLAY, timeInfo=position)
     * - if (returnValue == true)
     *   - Send answer via SMF immediately -> SendAnswer().
     *   - Forward status to PlayerManager -> SendMessage(PLAYTIME_STATUS, handle, timeInfo).
     * - else
     *   - Forward action error to Device Dispatcher -> SendMessage(ACTION_ERROR, reason).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] position absolute playtime in milliseconds
     * @return != 0: error, = 0: OK
     */
    tResult StartSeekTo(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint=NULL,
            const tPlaytime position=0);

    /**
     * Function transfers start_buffering command to PlayerEngine.
     * Roadmap 13010 Gapless Play
     * Call PEClient::Action(&returnValue, peState with reason=BUFFER)
     * if (returnValue == true)
     * - Send answer via SMF immediately -> SendAnswer(success = SC_YES).
     * else
     * - Send answer via SMF immediately -> SendAnswer(success = SC_NO).
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @return != 0: error, = 0: OK
     */
    tResult StartBuffer(const tDeviceType deviceType, //Roadmap 13010: 100%
            const tDeviceID deviceID,
            const tURL URL,
            const tMountPoint mountPoint,
            const tPEHandle handle);

    /**
     * Function transfers discard_buffer command to PlayerEngine.
     * Roadmap 13017
     * Call PEClient::Action(&returnValue, peState with reason=FLUSH)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DiscardBuffer(); //Roadmap 13017: 100%

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Roadmap 13010
     * Answer via SMF -> SendAnswer(success).
     *
     * @param[in] success success of buffer request
     * @return != 0: error, = 0: OK
     */
    tResult BufferStatus(const tSuccess success); //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Answer via SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaybackStatusResponse update.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4, metadataTitle, mediaType).
     *
     * @param[in] status status of current media PlayerEngine for current song
     * @param[in] metadata1 coming from PlayerEngine with meta data content
     * @param[in] metadata2 coming from PlayerEngine with meta data content
     * @param[in] metadata3 coming from PlayerEngine with meta data content
     * @param[in] metadata4 coming from PlayerEngine with meta data content
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID = OBJECT_ID_NONE);

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(elapsedPlaytime, totalPlaytime).
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function sends playtime status to PlayerManager triggered by spontaneous Tick_TimeElapsed update.
     * Forward status to PlayerManager -> SendMessage(TICK_TIME_ELAPSED, elapsedPlaytime, totalPlaytime).
     * In case of playtime is at the beginning of the track (=0) call StartOfObject.
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime, const tPlaytime totalPlaytime);

    /**
     * Function only traces HMI layer sync view status triggered by spontaneous viewStatusDCReply update.
     *
     * @param[in] viewStatus HMI layer sync view status (e.g. ACTIVE, STOPPED, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardViewStatus(const tViewStatus viewStatus){return MP_NO_ERROR;};

    /**
     * Function sends ACTION_ERROR to waiting state machine.
     * Roadmap 13010
     *
     * @param[in] reason error type (PE_RS_ACTION_ERROR, PE_RS_DRM_ERROR, PE_RS_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason);//Roadmap 13010: 100%

    /**
     * Function sends ACTION_ERROR to its own state machine
     * Roadmap 13010
     *
     * @param[in] reason error type (PE_RS_ACTION_ERROR, PE_RS_DRM_ERROR, PE_RS_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ActionError(const me::reason_e reason);

    /**
     * Function sends STREAM_ERROR to DeviceDispatcher.
     * Roadmap 13010
     *
     * @param[in] reason error type (PE_RS_ACTION_ERROR, PE_RS_DRM_ERROR, PE_RS_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardStreamError(const me::reason_e reason) const; //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Roadmap 13010
     * Answer via SMF -> SendAnswer(handle, playbackState, reason, speed).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaybackStatus update.
     * Roadmap 13010
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS, handle, playbackState, reason, speed).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState playback status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous NowPlayingStatus update.
     * Roadmap 13010
     * Read meta data from TagInfo
     * Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult ForwardNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous NowPlayingStatus update.
     * Roadmap 13010
     * Read meta data from TagInfo
     * Forward status to PlayerManager -> SendMessage(NOW_PLAYING_STATUS, handle, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult ForwardNowPlayingStatusWithMetadata(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends answer to waiting state machine triggered by new playtime status.
     * Answer via SMF -> SendAnswer(handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult PlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous PlaytimeStatus update.
     * Roadmap 13010
     * Forward status to PlayerManager -> SendMessage(PLAYTIME_STATUS, handle, timeInfo).
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfo struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatusNew(const tPEHandle handle, const tPETimeInfo timeInfo); //Roadmap 13010: 100%

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Release waiting in SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();

    /**
     * Function stores audio output device name internally.
     * Roadmap 13010
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice); //Roadmap 13010: 100%

    /**
     * Function transfers playback mode to external device.
     * Roadmap 13008
     * For USB no need set the playback mode.
     * Restriction: Do it only in streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] playbackMode current playing sequence is executed in sequential or random order
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tDeviceType deviceType, //Roadmap 13008
            const tDeviceID deviceID,
            const tMountPoint mountPoint,
            const tPlaybackMode playbackMode);

    /**
     * Function transfers repeat mode to external device.
     * Roadmap 13008
     * For USB no need set the playback mode.
     * Restriction: Do it only in streaming mode
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] repeatMode repeat the current playing track or list or nothing
     * @return != 0: error, = 0: OK
     */
    tResult SetRepeatMode(const tDeviceType deviceType, //Roadmap 13008
            const tDeviceID deviceID,
            const tMountPoint mountPoint,
            const tRepeatMode repeatMode);

    /**
     * Function sends internal signal PLAYBACK_STATUS(handle, status, reason, speed).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState status of current media PlayerEngine for current song
     * @param[in] reason reason for status change
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaybackStatus(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed); //Roadmap 13010: 100%

    /**
     * Function sends internal signal NOW_PLAYING_STATUS(handle, URL).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] mountPointURL file path and name including mount point
     * @return != 0: error, = 0: OK
     */
    tResult SendNowPlayingStatus(const tPEHandle handle, const tURL mountPointURL); //Roadmap 13010: 100%

    /**
     * Function sends internal signal PLAYTIME_STATUS(handle, timeInfo).
     * Roadmap 13010
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] timeInfoStruct struct of time position and duration of a media object
     * @return != 0: error, = 0: OK
     */
    tResult SendPlaytimeStatus(const tPEHandle handle, const tPETimeInfoStruct timeInfoStruct); //Roadmap 13010: 100%

    /**
     * Function calls umount to the file system in order to detach it from the system.
     * Roadmap 13003 SDCard
     * In case of having multiple partitions on that device umount is applied to the pysical device path instead.
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     * DeviceControl returns errno back to calling SM via UMOUNT_ANSWER.
     *
     * @param[in] deviceID deviceID to be unmounted
     * @param[in] mountPoint mountPoint of device
     * @return != 0: error, = 0: OK
     */
    tResult Umount(const tDeviceID deviceID, const tMountPoint mountPoint); //Roadmap 13003: 100%

    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);


    //Indexing part

    /**
     * Function delivers answer to condition.
     * It decides if device needs an initialization when INIT_DEVICE_CONNECTION is signaled.
     * In case of USB no initialization is necessary.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsInitRequired(const tDeviceInfoString deviceInfoString);

    /**
     * Function sends internal signal INIT_DEVICE(mountPoint).
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendInitInit(const tDeviceInfoString deviceInfoString);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. USB or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, deviceID)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult NoInitAnswer(const tDeviceInfoString deviceInfoString);

    /**
     * Function checks if device is already initialized properly or malfunction has been detected.
     * @attention For USB no implementation (return value is always false to prevent the waiting state)
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    tResult IsInitDeviceConnection(const tMountPoint mountPoint,const tInitDeviceProtocol protocol);

    /**
     * Function is called if IsInitDeviceConnection delivers false.
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */

    tResult Disconnect(const tMountPoint mountPoint) const;

    /**
     * Function entered if authentication procedure succeeded.
     * @attention For USB no implementation
     *
     * @param[in] mountPoint mountpoint of device
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @return != 0: error, = 0: OK
     */
    tResult OnDeviceInitialized(const tMountPoint mountPoint, const tConnectionState connectionState,const tInitDeviceProtocol discType);


    tBoolean IsInitializingDevice(const tMountPoint mountPoint,tConnectionState connectionState,tInitDeviceProtocol protocol);


    /**
     * Function retrieves the total number of playable files
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult GetNumberOfFiles(const tDeviceID deviceID);

    /**
     * Function retrieves the fingerprint of the device content in order to check for modifications.
     * In case of USB the hash value of the folder structure is calculated recursively.
     * The number of media files for indexing will be counted.
     * Send answer to own waiting state machine after result is available.
     *    -> SendMessage(PUT_FINGERPRINT, fingerprint, fingerprintStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @param[in] lastFingerprint status of the last fingerprint read from db
     * @return != 0: error, = 0: OK
     */
    tResult CalcFingerprint(const tMountPoint mountPoint, const tDeviceID deviceID, const tFingerprint lastFingerprint); //Roadmap 13006 IAP2

    /**
     * Function sends answer to waiting state machine triggered by new fingerprint status (PUT_FINGERPRINT).
     * Answer via SMF -> SendAnswer(fingerprint, fingerprintStatus, numberOfMediaFiles).
     *
     * @param[in] fingerprint device identification
     * @param[in] fingerprintStatus status of the fingerprint answer [OK, NOT_AVAIL, ERROR, DELTA]
     * @param[in] numberOfFiles number of files to index
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult OnFingerprint(const tFingerprint fingerprint,
            const tFingerprintStatus fingerprintStatus,
            const tNumberOfFiles numberOfFiles,
            const tDeviceID deviceID);

    /**
     * Function delivers answer to condition.
     * Check cached list of metadata records already read from USB.
     * Return true if cache is not empty, otherwise false.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return true or false
     */
    tResult IsNextMetadataAvailable(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function retrieves a metadata record from device.
     * Based on the readPosition this function is able to continue extraction continuously within
     * one virtual list containing all music tracks, audiobooks, podcast and videos, etc.
     *
     * Get next media object from virtual file list -> GetNextObjectToIndex(&metadataStatus,deviceID,mountPoint,readPosition)
     * Send answer to own waiting state machine after collecting meta data is finished.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromDevice(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends internal signal PUT_METADATA(metadataStatus).
     * Send answer to own waiting state machine because meta data already available in cache.
     *    -> SendMessage(PUT_METADATA, metadataStatus).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult SendPutMetadata(const tMountPoint mountPoint, const tReadPosition readPosition, const tDeviceID deviceID);

    /**
     * Function sends answer to waiting state machine triggered by metadata available in cache (PUT_METADATA).
     * Get meta data from own cache -> RetrieveMetadataFromCache(&metadata).
     * Answer via SMF -> SendAnswer(metadata, metadataStatus).
     *
     * @param[in] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @param[in] mediaObjectPtr
     * @return != 0: error, = 0: OK
     */
    tResult AnswerMetadata(const tMetadataStatus metadataStatus, tMediaObjectPtr mediaObjectPtr);

    /**
     * Function retrieves the meta data of one media object from cache
     * @attention Obsolete: DVDControl does use a scan context with virtual file lists instead of metadata cache.
     *
     * @param[out] metadata meta data of one media object
     * @return != 0: error, = 0: OK
     */
    tResult RetrieveMetadataFromCache(tMetadata &metadata) const;

    /**
     * Function stores indexing last mode for removed device and resets scan context.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID);


    //AlbumArt part

    /**
     * Function triggers DoGetAlbumArt in a seperate working thread
     *
     * DoGetAlbumArt gets the album art from media object and stores it in a member variable.
     * Allocate memory for internal album art object.
     * Get album art from Taglib -> GetAlbumArt(&albumArtObject, albumArtString)
     * or from separate file -> fread(file) + GetAlbumArtFromSeparateFile(&albumArtObject, albumArtString).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     * @attention The caller must free this memory if not used anymore.
     *
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArt(const tAlbumArt albumArtString);

    /**
     * Function sends answer to waiting state machine triggered by answer from DoGetAlbumArt
     * working thread (GET_ALBUM_ART_ANSWER).
     * Answer via SMF -> SendAnswer(ptrToAlbumArtObject).
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    /**
     * Function clears allocated album art buffer
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @return != 0: error, = 0: OK
     */
    tResult AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    tResult ReadOneVideoMetaData(IN FILE *fpInput, IN char *tmpFileName, const char *format, ...);
    /**
     * Function gets the toc data and track info
     *
     * @param[in] mountpoint mountpoint of device
     * @param[in] path filepath
     * @param[in] index
     * @return != 0: error, = 0: OK
     */
    tResult CDDAGetFolderItem(const tMountPoint mountpoint, const tPath path, const tIndex index);

    /**
     * Function gets the toc data and track info
     *
     * @param[in] mountpoint mountpoint of device
     * @param[in] path filepath
     * @param[in] index
     * @return != 0: error, = 0: OK
     */
    tResult DVDGetFolderItem(const tMountPoint mountpoint, const tPath path, const tIndex index);

    /**
     * Function sends answer to waiting state machine triggered by answer from CDDAGetFolderItem
     */
    tResult CDDAGerFolderItemAnswer();

    tResult SendDeviceInitialized();

    tResult ejectDisc();

    tResult GetDiskError(OUT tBool& isError, OUT tU8& diskError);
    /**
     * Function get Disk error for Dvd (DiscError : FID 0x254)
     *
     * @param[OUT] error , disk error
     * @return != 0: error, = 0: OK
     */
    tResult GetDiskMechanicalInfo(OUT tU8& mechanicalInfo);
    /**
     * Function get mechanical Info for Dvd (mechanicalInfo : FID 0x255)
     *
     * @param[OUT] error , mechanical Info
     * @return != 0: error, = 0: OK
     */
    tResult GetDVDDriveInfo(OUT tU8& disctype,tU8& drivemode);
    /**
     * Function get dvd drive info for Dvd (DVDdriveInfo : FID 0x255)
     *
     * @param[OUT] disc type , drive mode
     * @return != 0: error, = 0: OK
     */


    /**
     * Function get COMMAND RESPONSE Info for Dvd (Skip info : FID 0x)
     *
     * @param[OUT] ,Key Command Response
     * @return != 0: error, = 0: OK
     */
    tResult GetKeyCommandResponse(OUT tU8& responseInfo);
    /**
     * Function get mechanical Info for Dvd (Skip info : FID 0x256)
     *
     * @param[OUT] ,skip
     * @return != 0: error, = 0: OK
     */
    tResult GetSkipInfo(OUT tBool& skip);
    /**
     * Function get tMenuPlaybackOngoing Info for Dvd (Skip info : FID 0x256)
     *
     * @param[OUT] ,menuOngoing
     * @return != 0: error, = 0: OK
     */
    tResult GetMenuPlaybackOngoing(OUT tBool& menuOngoing);
    /**
     * Function Get Dvd Direct Select Info for  (Skip info : FID 0x257)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetDirectSelectInfo(OUT tBool& dirceSelect);

    /**
     * Function get dvd dirct select Info for Dvd (Skip info : FID 0x258)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetAngleInfo(OUT tU8& angleTrack, tU8& totalAngle);
    /**
     * Function Get Dvd Subtitle Info for Dvd (Skip info : FID 0x259)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetSubtitleInfo(OUT tBool& state, OUT tU8& current, OUT tU8& total,OUT tU16& Lang);

    /**
     * Function Get Dvd Subtitle Info for Dvd (Skip info : FID 0x259)
     *
     * @param[OUT] ,dirceSelect
     * @return != 0: error, = 0: OK
     */
    tResult GetAudioChannelInfo(OUT tU8& totalChannels, OUT tBool& subWoofer, tU8& Assignment);
    /**
     * Function Get Set diplay mode setting (Skip info : FID 0x262)
     *
     * @param[OUT] , display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetDisplayMode(OUT tU8& displayMode);
    /**
     * Function Get Set diplay mode setting  (Skip info : FID 0x262)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult SetDisplayMode(IN tU8 displayMode);
    /**
     * Function Get Play Status Setting   (Skip info : FID 0x265)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetPlayStatusSetting(OUT tBool& playStatus);
    /**
     * Function Get Play Status Setting (Skip info : FID 0x265)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult SetPlayStatusSetting(IN tBool playStatus);
    /**
     * Function Get Dvd DRC Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,display mode
     * @return != 0: error, = 0: OK
     */
    tResult GetDRCSetting(OUT tU8& drc);
    /**
     * Function Get Dvd DRC Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetDRCSetting(IN tU8 drc);
    /**
     * Function Get Current AudioInfo Setting(Skip info : FID 0x266)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetCurrentAudioInfo(OUT tU8& soundformat, OUT tU8& VcdAudio, OUT tU8& DvdAudio,OUT tU8& totalOutput,OUT tU16& Lang);
    /**
     * Function SetAngleMarkSetting(Skip info : FID 0x264)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetAngleMarkSetting(IN tBool AngleMark);
    /**
     * Function GetAngleMarkSetting(Skip info : FID 0x264)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetAngleMarkSetting(OUT tBool& AngleMark);
    /**
     * Function Set Current Language Setting(Skip info : FID 0x263)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult SetLanguageSetting(IN tU16 Audio, IN tU16 SubTitel, IN tU16 Menu);
    /**
     * Function Get Current Language Setting(Skip info : FID 0x263)
     *
     * @param[IN] ,drc
     * @return != 0: error, = 0: OK
     */
    tResult GetLanguageSetting(OUT tU16& Audio, OUT tU16& SubTitel, OUT tU16& Menu);
    /**
     * This method  RequestDiscOperation used to ejct dvd drive
     * is used to Eject or Load DVD
     * @param[in] Operation drive Eject = 1 Load = 0
     * @return < 0: error, = 0: OK
     */
    tResult RequestDiscOperation(tU8 Operation);
    /**
     * This method  RequestVCDAudioChannel
     * is used to SelectionType and AudioMode
     * @param[in] SelectionType , AudioMode
     * @return < 0: error, = 0: OK
     */
    tResult RequestVCDAudioChannel(tU8 SelectionType, tU8 AudioMode);
    /**
     * This method  RequestDVDAudioChannel
     * is used to SelectionType and AudioMode
     * @param[in] SelectionType and AudioMode
     * @return < 0: error, = 0: OK
     */
    tResult RequestDVDAudioChannel(tU8 SelectionType, tU8 AudioMode);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] Action
     * @return < 0: error, = 0: OK
     */
    tResult SendNavigationKey(tU8 Action);
    /**
     * This method  SendTouchCommand used to set Touch Actions
     * @param[in] Action,Xcod,Ycod,maxXcod,maxYcod
     * @return < 0: error, = 0: OK
     */
    tResult SendTouchCommand(tU8 Action, tU16 Xcod, tU16 Ycod, tU16 maxXcod, tU16 maxYcod);
    /**
     * This method  RequestDirectSearch used direct search
     * @param[in] titel , chapter
     * @return < 0: error, = 0: OK
     */
    tResult RequestDirectSearch(tU32 titel, tU32 chapter);
    /**
     * This method  RequestAngleChange
     * @param[in] SelectionType , Angle
     * @return < 0: error, = 0: OK
     */
    tResult RequestAngleChange(tU8 SelectionType, tU8 Angle);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] MenuControl
     * @return < 0: error, = 0: OK
     */
    tResult SetMenuPlayBackControl(tU8 MenuControl);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] number
     * @return < 0: error, = 0: OK
     */
    tResult SelectDirectNumber(tU32 number);
    /**
     * This method  SendNavigationKey used to set Navi Actions
     * @param[in] number
     * @return < 0: error, = 0: OK
     */
    tResult RequestTitleSearch(tU32 number);
    /**
     * This method  RequestSubtitleChange
     * @param[in] state , selectionType , Subtitel
     * @return < 0: error, = 0: OK
     */
    tResult RequestSubtitleChange(tBool state, tU8  selectionType , tU8 Subtitel);
    /**
     * This method  RequestCMSkip
     * @param[in] SkipValue
     * @return < 0: error, = 0: OK
     */
    tResult RequestCMSkip(tU8 SkipValue);

    tResult DVDVideoSourceUpdate(tVideoSourceState status);

    tResult StartRippingByUser();
    tResult RequestDVDTemperature(tUserContext userContext);
    /**
     * Function is called when Media engine posts an answer for start Video streaming request
     * Video Management Related Implementation
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState status of requested action on video content
     * @param[in] reason , should be reason_ok if success
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return
     */

    tResult StartVideoStreamingAnswer(const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed);
    /**
     * Function is called when Media engine posts an answer for stop Video streaming request
     * Video Management Related Implementation
     *
     * @param[in] handle handle (= objectID) to map property update to an media object
     * @param[in] playbackState status of requested action on video content
     * @param[in] reason , should be reason_ok if success
     * @param[in] speed playback speed (1=normal play, 10=ffwd, -10=frev)
     * @return
     */
    tResult StopVideoStreamingAnswer (const tPEHandle handle, const tPEPlaybackState playbackState, const me::reason_e reason, const me::speed_e speed);

    bool checkRippingEnabled();
    void clearTOC();
    tCDTOCInfo* getCDTOC();

    //USB specific part
private:

    tResult SendAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);

    /*internal thread functions*/
    void DoGetAlbumArtThread(const tAlbumArt albumArtString);
    void DoCalcFingerprintThread(const char *inputString);
    void DoReadMetadataForPlaybackThread(const tURL completeFileName);

    /**
     * Function checks if device is still present in case of error code "file not found".
     * In case of error code of PlayerEngine (metadata2) is MEDIAPLAYER_ERROR_FILE_DOES_NOT_EXISTS
     * make read request on root directory to check if device is still present.
     * If not possible change error code to MEDIAPLAYER_ERROR_DEVICE_NOT_FOUND.
     *
     * @param[out] reason error reason
     * @param[in] errorCode error code to be checked
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult ValidateErrorCode(me::reason_e &reason, const tMetadata errorCode, const tMountPoint mountPoint) const;

    /**
     * Function gets next metadata from USB
     * In case of error code FILE_ERROR make read request on root directory to check if device is still present.
     * If not possible change error code to DEVICE_ERROR.
     * Collect meta data via TagInfo lib -> CollectData(mountPoint)
     *
     * @param[out] metadataStatus status of meta data collection [SUCCESS, FINISHED, FILE_ERROR, DEVICE_ERROR]
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] readPosition position of next media object to read
     * @return != 0: error, = 0: OK
     */
    tResult GetNextObjectToIndex(tMetadataStatus &metadataStatus, const tDeviceID deviceID, const tMountPoint mountPoint, const tReadPosition readPosition);

    /**
     * Function reads the metadata of an audio media object using the TagInfo lib
     * Read DRM protection into m_PlayMediaObject.notPlayable
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadAudioMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function reads the metadata of an video media object using the TagInfo lib
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadVideoMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function reads the metadata of an image media object using the TagInfo lib
     *
     * @param[in] mediaObject to update/fill an media object after extraction
     * @param[in] completeFileName file path and name including mount point
     * @return true or false
     */
    tReturnValue ReadImageMetadataFromFile(tMediaObject &mediaObject, const tURL completeFileName);

    /**
     * Function collects the metadata using the TagInfo lib
     * Do it only if collecting of metadata is configured in DataProvider -> CollectMetadata() //Roadmap 13012
     *
     * @param[in] mountPoint mount point of device
     * @return true or false
     */
    tReturnValue CollectData(const tMountPoint mountPoint); //Roadmap 13012: 100%

    /**
     * Function delivers answer if a video file is playable.
     * Check video metadata to meet requirements.
     * Return true if video file is playable, otherwise false.
     *
     * @param[in] codec used codec (e.g. CDC_MPEG4, CDC_DIVX, CDC_AVC, ...)
     * @param[in] profile used profile (e.g. Main, Simple, Advanced ...)
     * @param[in] level profile level
     * @param[in] width resolution width in pixel
     * @param[in] height resolution height in pixel
     * @param[in] frameRate frame rate in frames per second (fps)
     * @param[in] bitRate bit rate in bits per second (bps)
     * @return true or false
     */
    tReturnValue IsVideoPlayable(const tCodec codec,
            const tVideoProfileName profile,
            const tVideoProfileLevel level,
            const tSize width,
            const tSize height,
            const tFrameRate frameRate,
            const tBitRate bitRate);

    /**
     * Function removes the scan context for a device and releases temporary lists
     *
     * @param[in] deviceIndex index of device in scan context
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceFromContext(const tIndex deviceIndex);

    /**
     * Function gets the album art from separate file
     *
     * @param[out] albumArtObjectPtr pointer to an album art object
     * @param[in] albumArtString file name and path (URL) of image to display
     * @return != 0: error, = 0: OK
     */
    tResult GetAlbumArtFromSeparateFile(tAlbumArtObjectPtr &albumArtObjectPtr, const tAlbumArt albumArtString) const;

    void TransferMetaTags(void *info, tMediaObject &mediaObject);

    tResult SendInitDevice(const tMountPoint mountPoint, const tInitDeviceProtocol protocol);
    tResult StartPlaybackAction(const tDeviceType deviceType, const tDeviceID deviceID,
            const tURL URL, const tMountPoint mountPoint,const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount);
	tResult PlaybackAction(tDeviceType, tDeviceID, const char*, const char*, tPlaybackAction, tNextPrevSkipCount);
    tResult SignalUserData(const tDeviceID deviceID, const tUserDataType userDataType, const tUserData userData);
    tResult UpdatePlaybackStatus(tPEPlaybackState playbackState,int speed,tUInt currentTrackNo,tUInt currentGroupNo,tPlaytime elspsedTime,tPlaytime totalTime);
    void ForwardRepeatModeStatus(tRepeatMode repeatMode);
    void ForwardPlaybackModeStatus(tPlaybackMode playbackMode);

    /**
     * Function triggers the event in ripperSM to get the gracenote metadata for CDDA
     * @param[in] gracenote data request type for CDDA or Ripping
     * @return != 0: error, = 0: OK
     */
    tResult FetchedGNinfo(const tGNRequestType GNReqType);
    /**
     * Function gets the gracenote data from ripper control
     * @param[in] gracenote data available or not available
     * @return != 0: error, = 0: OK
     */
    tResult FinishedGNfetch(tMetadataAvailablefromGN GNdataAvailable);
    /**
     * Function sets the metadata available flag
     * @param[in] gracenote data available or not available
     * @return != 0: error, = 0: OK
     */
    tResult setGNMetadataAvailable(bool GNDataAvailable);
    /**
     * Function returns the value of the metadata available flag
     * @return != 0: error, = 0: OK
     */
    bool getGNMetadataAvailable();
    /**
     * Function triggers the ripping
     * R6392_UC1_LLD2.1
     * @param[out] audio format, if the format is DTS audio, the ripping will not triggered
     * @return != 0: error, = 0: OK
     */
    tResult StartRipping(tAudioFormat format);
    /**
     * Function stops the ripping
     * R6392_UC1_LLD2.1
     * @return != 0: error, = 0: OK
     */
    tResult StopRipping();
    /**
     * Function gives done ripping message to state machine
     * R6392_UC1_LLD2.1
     * @return != 0: error, = 0: OK
     */
    tResult DoneRipping(const tDoneRippingAnswerRequired isRippingAnswerRequired);
    tResult StartVideoStreaming();
    tResult StopVideoStreaming();
    tResult SendVideoProperty();


public:
    class DVDCallbacks : public DVDControlInterfaceCallback {
    public:
        DVDCallbacks();
        void update(me::state_t currentState);
        void resendNowPlaying(tBoolean resend);  //NCG3D-22900
        void signalEvent(tDVDEventID dvdEventId , tU8 value);
        void signalPlayBackState(tDVDPlayBackState& playBackInfo);
        void signalResumeInfo(ResumeBuffer& resumeBuffer);
        void signalEventAngleChange(tDVDAngleInfo& angleInfo);
        void signalEventSubtitleChange(tDVDSubtitleInfo& subtitel);
        void signalEventAudioInfoChange(tDVDAudioChannelInfo& audioChannel);
        void signalEventCurrentAudioChange(tDVDAudioOutputInfo& audioOutputInfo);
        void signalEventSetupInfoChange(tDVDSetupInfo& setupInfo);
        void signalEventTemperatureChange(tU16 temperature);

    private:
        int ForwardPlaybackStatus(me::state_t currentState);

        me::state_t mNotificationState;
        me::state_t mNotificationStateNew;
        int mResendPlaybackStatus;
        int mResendNowPlayingStatus;
        tGeneralString mObservingStateMachineName;
        bool m_bFirstTimeSetupInfo;

    };

    DVDCallbacks mCallbacks;

    private:

    int ForwardPlaybackStatus(me::state_t currentState);
    bool ConvertSpeedFormat(tPlaybackSpeed rate,tCueingRate& convertedRate);
    bool ConvertSpeedFormat(tCueingRate rate,tCueingRate& convertedRate);
    tResult FetchDVDVideoInterface(OUT tDeviceName& devicename);
    bool IsVideoDisc( tDiscType discType);
    Lock m_Mutex;                                       /**< lock access to member variables */
    class Lock m_ReadMetadataForPlaybackLock;

    tMediaObject m_PlayMediaObject;                     /**< last media object to play */
    tPlaytime m_StartPosition;                          /**< last position to play */

    DVDControlInterface *m_pDVDControlInterface;
    tFiles *mTocDataOneTrack;
    tDeviceInfo mDeviceInfo;

    std::map<unsigned int ,tRippedTrackInfo> mGNMetadata;
    tU16 mTrackNo;
    tU16 mGroupNo;
    tPEPlaybackState mPlaybackState ;
    tFiles mtocData;

    /*DVD property Update Types*/
    tDvdDiscError mDvdDiscError;
    tU8 mDiscMechanicalInfo;
    tBool mCMSkipAvailable;
    tBool mMenuPlayback;
    tBool mDirectSelectAvailable;
    tDVDAngleInfo mDVDAngleInfo;
    tDVDSubtitleInfo mDVDSubtitleInfo;
    tDVDAudioChannelInfo mDVDAudioChannelInfo;
    tDVDAudioOutputInfo mDVDAudioOutput;
    tOperationModeStatus m_dvdDriveMode;
    bool m_bAlreadyRipped;
    tU8 mKeyCommandResponse;
    tU16 m_resumeCount;
    bool m_bIsDVDControlActive;
    bool m_bEjectAfterDoneRipping;
    bool m_bVideoStreaming;
    bool m_bSendSourceSwitchNowPlayingUpdate;
    bool m_bResponseReceived;
    bool m_bIsDataDisc;
    tUserContext m_tempUserContext;
    bool m_bAutoReinsert;
    bool m_bdatadiscReinserted;
    bool m_bSourceStatePause;
    tURL m_Url;
    bool m_bPlayfromPause;
    int DvdVideoRegionCodeArray[7] = {0xFF, 0x01, 0x02 , 0x04 ,0x08 , 0x10 , 0x20 } ;
    tCDTOCInfo m_CDTOCInfo ;
    TouchEventHandler m_touchHandler;
};

#endif //_DVDControl_H_
