/**
 * @addtogroup DeviceControl
 * @author Reenuhaa Pechi S
 *
 * Public interface for BTControl
 * @{
 */

#ifndef BTCONTROL_H_
#define BTCONTROL_H_

#include "BTControlSM.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "RequestResponseSM.h"

/**
 * BT control class
 * All the playback related Requests for MediaPlayer's Source AVRCP 1.4 Bluetooth Device are handled.
 * AnyChange in Playback Related changes happened in AVRCP 1.4 Bluetooth Device will be captured and Sent to the PlayerManager.
 * PlayerEngine is Controlled for Playback of AVRCP 1.4 Bluetooth Devices.
 */

#define BT_COVERART_PATH "/tmp/mediaplayer"

class BTControl: public BTControlSM,
        public ILocalSPM,
        public TFThread
{

public:

    //SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the BTControl componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    BTControl(const tComponentID componentID) :
       ILocalSPM(componentID)
    {
    }
    ;

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~BTControl()
    {
    }
    ;

    /**
     * This function is used by LocalSPM to create the BTControl.
     * @attention: running in SPM thread context
     * Create the BTControl state machine (including create of message queue).
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create();

    /**
    */
    tResult Init(tInitReason reason);

    /**
     * Initialize own member variables
     *
     * @return != 0: error, = 0: OK
     */
    tResult InitSM();

    /**
     * This function is used by LocalSPM to start the BTControl.
     * @attention: running in SPM thread context
     * From now on all other mediaplayer components are available.
     * Start the BTControl thread and the state machine.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Run();

    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * This function is used by LocalSPM to stop the BTControl.
     * @attention: running in SPM thread context
     * Store last mode values.
     * Set state machine to final state -> SendMessage(STOP_SM).
     * LocalSPM will be informed after STOP_SM event is processed
     *
     * @return != 0: error, = 0: OK
     */
    tResult Stop();

    /**
     * Inform LocalSPM that Stop is ready -> StopDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopEventProcessed();

    /**
     * This function is used by LocalSPM to cleanup the BTControl.
     * @attention: running in SPM thread context
     * Deregister BTControlSM with dispatcher.
     * Set state machine to final state -> SendMessage(DONE).
     * LocalSPM will be informed after DONE event is processed.
     *
     * @return != 0: error, = 0: OK
     */
    tResult Done();

    /**
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return != 0: error, = 0: OK
     */
    tResult DoneEventProcessed();

    /**
     * Returns the current state the state machine is in (for debugging of shutdown problems)
     * @param[inout] stateName buffer for storing the current state name
     *
     * @return pointer to stateName
     */
    char *GetSMStateName(OUT tGeneralString stateName, IN size_t size);

    /**
    * Returns answer if the component is a state machine
    *
    * @return true or false
    */
    tBoolean IsComponentSM() {return true;}; //component is a state machine

    //Playback control part

    /**
     * Function transfers assign_audio_input_device command to PlayerEngine.
     * Send AssignAudioInputDevice to PlayerEngine via DBUS -> AssignAudioInputDevice(m_AudioInputDevice).
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartAllocateAudioInput(const tDeviceID deviceID, const tMountPoint mountPoint);

    /**
     * Function transfers deallocate_in_device command to PlayerEngine.
     * Send DeAllocateInDevice to PlayerEngine via DBUS.
     * PlayerEngine command has return value of success.
     *
     * @warning Not used yet!
     *
     * Comment: at the moment not used!
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartDeAllocateAudioInput();

    /**
     * Function transfers switch_observer command to PlayerEngine.
     * Switch observer at PlayerEngine via DBUS
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartSwitchObserver(const tDeviceID /*deviceID*/, const tMountPoint /*mountPoint*/);

    /**
     * Function transfers playFromDevice command to PlayerEngine.
     * Send playFromDevice to PlayerEngine via DBUS -> playFromDevice(m_AudioInputDevice).
     * PlayerEngine command has return value of success.
     *
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @return != 0: error, = 0: OK
     */
    tResult StartStreaming(const tDeviceID deviceID, const tMountPoint mountPoint);

    /**
     * Function transfers ...
     * For BT no need to stop streaming
     * Release own waiting state -> SendEvent(STOP_STREAMING_ANSWER, deviceID=DEVICE_ID_NOT_SET).
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartStopStreaming();

    /**
     * Function answers of STOP request to waiting state machine via SMF
     * if old PlayerEngine
     * - PlaybackStatus(playbackState=PE_PBS_STOPPEDSTATE, metadata, metadata, metadata, metadata) -> SendAnswer.
     * else //Roadmap 13010
     * - PlaybackStatusNew(handle, playbackState=PE_PBS_STOPPEDSTATE, reason, speed) -> SendAnswer.
     *
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult StopStreamingAnswer(tDeviceID deviceID);

    /**
     * Function sends answer to waiting state machine triggered by new playback status.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Add media type to message -> AddMediaType(mediaType).
     * Answer via SMF -> SendAnswer(status, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] status status of current media bt device for current song
     * @param[in] metadata1 coming from avrcp with status content
     * @param[in] metadata2 coming from avrcp with status content
     * @param[in] metadata3 coming from avrcp with status content
     * @param[in] metadata4 coming from avrcp with status content
     * @return != 0: error, = 0: OK
     */
    tResult PlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID);

    /**
     * Function sends playback status to PlayerManager triggered by spontaneous playback status update.
     * Validate error code -> ValidateErrorCode(metadata2).
     * Add media type to message -> AddMediaType(mediaType).
     * Forward status to PlayerManager -> SendMessage(PLAYBACK_STATUS_RESPONSE, status, metadata1, metadata2, metadata3, metadata4, mediaType).
     *
     * @param[in] status status of current media bt device for current song
     * @param[in] metadata1 coming from avrcp with status content
     * @param[in] metadata2 coming from avrcp with status content
     * @param[in] metadata3 coming from avrcp with status content
     * @param[in] metadata4 coming from avrcp with status content
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaybackStatus(const tPEPlaybackState status,
            const tMetadata metadata1,
            const tMetadata metadata2,
            const tMetadata metadata3,
            const tMetadata metadata4,
            const tObjectID ObjectID);

    /**
     * Function sends playtime status to PlayerManager triggered by spontaneous Tick_TimeElapsed update.
     * Forward status to PlayerManager -> SendMessage(TICK_TIME_ELAPSED, elapsedPlaytime, totalPlaytime).
     *
     * @param[in] elapsedPlaytime elapsed playtime of current media object
     * @param[in] totalPlaytime total playtime of current media object
     * @return != 0: error, = 0: OK
     */
    tResult ForwardPlaytimeStatus(const tPlaytime elapsedPlaytime,
            tPlaytime totalPlaytime);

    /**
     * Function sends answer to own waiting state machine triggered by SendMessageAnswer request timeout.
     * Release waiting in SMF -> ReleaseWaiting().
     *
     * @return != 0: error, = 0: OK
     */
    tResult HandleAnswerTimeout();

    /**
     * Function sends answer to possible waiting state machine to release own SM in case of message answer pair.
     * Answer to SMF -> SendAnswer().
     *
     * @return != 0: error, = 0: OK
     */
    tResult MessageNotConsumed();

   /**
     * Function stores audio output device name internally.
     *
     * @param[in] audioOutputDevice device name (ALSA) of audio output
     * @return != 0: error, = 0: OK
     */
    tResult SetOutputDevice(const tAudioOutputDevice audioOutputDevice);

   /**
     * Function transfers generic playback command, like NEXT, PREV, etc
     *
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] URL file path and name including mount point
     * @param[in] mountPoint mount point of device
     * @param[in] playbackAction action to be performed on the source (pause, next. prev , ffwd ,rwd etc)
     * @param[in] skipcount, number of tracks to be skipped by NEXT, PREV
     * @return != 0: error, = 0: OK
     */
    tResult StartPlaybackAction(const tDeviceType deviceType, const tDeviceID deviceID, //Roadmap 13008 RemoteControl
                const tURL URL, const tMountPoint mountPoint,const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount=0);

    /**
     * Function set the playback mode (shuffle on/off) on the device
     * @param[in] deviceType type of device on which the media object is stored
     * @param[in] deviceID device ID
     * @param[in] mountPoint mount point of device
     * @param[in] playbackMode, shuffle/normal mode
     * @return != 0: error, = 0: OK
     */
    tResult SetPlaybackMode(const tDeviceType deviceType,
                            const tDeviceID deviceID,
                            const tMountPoint mountPoint,
                            const tPlaybackMode playbackMode);



    tResult SetRepeatMode(const tDeviceType deviceType,
                          const tDeviceID deviceID,
                          const tMountPoint mountPoint,
                          const tRepeatMode repeatMode);

    tResult AvpSetRepeatModeMethodReturn(tBTMethodRequestStatus Status,tErrorMessage ErrorMessage);
    tResult AvpSetPlaybackModeMethodReturn(tBTMethodRequestStatus Status,tErrorMessage ErrorMessage);


    /**
     * Function sends answer to waiting state machine triggered by DBUS method return message.
     * Answer via SMF -> SendAnswer(returnValue).
     *
     * @param[in] returnValue success = true, otherwise false.
     * @return != 0: error, = 0: OK
     */
    tResult ActionStatus(const tReturnValue returnValue);

    /**
     * Function calls umount to the file system in order to detach it from the system.
     * Roadmap 13003 SDCard
     * NOT SUPPORTED BY BT
     *
     * By use of a local RRSM send UMOUNT to DeviceControl via RouteMessageAnswer, synchronous call.
     *
     * @param[in] deviceID deviceID to be unmounted
     * @param[in] mountPoint mountPoint of device
     * @return != 0: error, = 0: OK
     */
    tResult Umount(const tDeviceID deviceID, const tMountPoint mountPoint); //Roadmap 13003

    /* This Function does AvpGetFolderItem Method call on AVRCP 1.4 Device
    *
    * @param[in] Scope              Scope from which the Items need to be returned
    * @param[in] StartItem          Offset of the first returned item
    * @param[in] EndItem            Offset of the last returned item
    *
    * @return != 0: error, = 0: OK
    */
    tResult AvpGetFolderItem(tScope Scope,
                            tStartItem StartItem,
                            tEndItem EndItem);


    /**
     * This Function does AvpGetMediaPlayerItem Method call on AVRCP 1.4 Device
     *
     * @param[in] MaxNumber        Number of Players(in AVRCP 1.4 Device) whose Information needed.
     * @return != 0: error, = 0: OK
     */
    tResult AvpGetMediaPlayerItem(const tDeviceInfoString deviceInfoString,const tDeviceVersionUpdated IsDeviceVersionUpdated);
    tResult StartStop(tDeviceType deviceType,const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint);

    tResult StartPlay(const tDeviceType deviceType,const tDeviceID deviceID,
                      const tURL URL, const tMountPoint mountPoint,const tUUID uuid,
                      const tPEHandle handle, const tPlaytime position,const tStreaming streaming) ;
    tResult PlayReturnError();

    tResult IsInitRequired(const tDeviceInfoString deviceInfoString);
    tResult SendInitInit(const tDeviceInfoString deviceInfoString);
    tResult GetAlbumArt(const tAlbumArt albumArtString);
    tResult PlaybackStatusNew(const tPEHandle handle, const tPEPlaybackState playbackState, //Roadmap 13008
                const me::reason_e reason, const me::speed_e speed);
    tResult HandleGetAlbumArtAnswer(const tAlbumArtObjectPtr ptrToAlbumArtObject);
    tResult AlbumArtAnswerNotConsumed(const tAlbumArtObjectPtr ptrToAlbumArtObject);
    tResult GetAlbumArtFromSeparateFile(tAlbumArtObjectPtr &albumArtObjectPtr, const std::string albumArtpath);
    tResult AvpGetImageMethodReturn(tURL fullpath);

    /**
     * Function brings the DeviceControl to idle state from any other state.
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
    tResult RemoveDeviceConnection(const tMountPoint mountPoint, const tDeviceID deviceID);
    tResult StreamingModeOn(const tDeviceType DeviceType, const tDeviceID DeviceID, const tMountPoint MountPoint);
    void AvpPlayItemMethodReturn(tBTMethodRequestStatus Status);

    /**
     * Function sends ACTION_ERROR to waiting state machine.
     * Roadmap 13010
     *
     * @param[in] reason error type (REASON_ACTION_ERROR, REASON_DRM_ERROR, REASON_DEVICE_ERROR, ...)
     * @return != 0: error, = 0: OK
     */
    tResult HandleActionError(const me::reason_e reason);

    /**
         * Function sends NO_RESPONSE_MSG to waiting state machine.
         * AIVI-50524
         *
         * @param[in] response false for NOK true for OK
         * @return != 0: error, = 0: OK
    */
    tResult SendNoResponseMsg(const tResponseMsg response);
    void ForwardPlaybackModeStatus(tPlaybackMode playbackMode);
    void ForwardRepeatModeStatus(tRepeatMode repeatMode);

    /**
     * IsBatchPlayable
     * Function is used to check support for batch list playback support
     *
     * @param[in] deviceID
     * @return tBoolean
     */
    tBoolean IsBatchPlayable(const tDeviceID deviceID);

    /**
     * GetConnectedDeviceInfo
     * Function is used to get mConnectedDeviceInfo
     *
     * @return tDeviceInfo
     */
    tDeviceInfo GetConnectedDeviceInfo();

    void SetCurrentPlayer(string Playername);
    void GetPlayerId(tPlayerId &PlayerId);


    tBoolean SetupPlayerForBrowsing(string playerInfo);
    tResult GetPlayerAttributes();
    int UpdateDeviceVersion(tBTAvrcpVersion avrcpVersion);
    void DoCalculateItemCountInGivenPath(char *parameterStr);
    void DoGetAlbumArtThread (const tAlbumArt albumArtString);
    tPEPlaybackState GetCurrentPlaybackState();

    tBoolean getIsAvpChangePathFromStartPlay();

    void setIsAvpChangePathFromStartPlay(tBoolean isAvpChangePathFromStartPlay);

    tBoolean getIsPlayItemFromBrowseNeeded();

    void setIsPlayItemFromBrowseNeeded(tBoolean isPlayItemFromBrowseNeeded);

    tResult UpdateShuffleSupportToDB(int ShuffleSupport);

    tResult UpdateRepeatSupportToDB(int RepeatSupport);

    tResult UpdateVolumeToDevice(tVolume AudioVolume);

    tResult UpdateVolumeToAudio(tVolume DeviceVolume);

    tResult SendListChangedToHMI();

    tResult StartRetryPlayTimer();

    tResult StopRetryPlayTimer();

    tResult SetStateOfPlayinStartPlay(tStateOfPlayinStartPlay mStateOfPlayinStartPlay);

    tStateOfPlayinStartPlay GetStateOfPlayinStartPlay();

    void SetCoverArtConStatus(tCoverArtConStatus coverArtConStatus);

    tResult MoveToRootAndClearBTCache();

    tResult PlaybackAction(const tDeviceType deviceType, const tDeviceID deviceID, const tURL URL,const tMountPoint mountPoint, const tPlaybackAction playbackAction, const tNextPrevSkipCount skipcount=0);

    tResult StartSeekTo(const tDeviceType deviceType, const tDeviceID deviceID, const tURL URL, const tMountPoint mountPoint = NULL, const tPlaytime position = 0);

    void SetCoverArtInfo(const tBTCoverArtInfo coverartInfo);

    void ClearCoverArtInfo();

    void DeleteCurrentAlbumart();

    int TrackChanged();

   private:

    //ChangePath related Members
    tLevelCount mLevelUpCount;
    tLevelCount mLevelDownCount;
    tDirection  mDirectionRequested;
    tItemUIDArray mUIDArray;
    tItemUIDArray mUIDArray_Address;
    tBoolean mIsStoreInfoNeeded;
    tPlayerMode mPlayerModeCalled;
    tPlayerId mPlayerId;
    tCodecType mCodecType;
    tBitRate mBitRate;
    tPEPlaybackState mCurrentPlaybackState;
    tNumOfItems mCurBrowsedPthItemCnt;
    tPlaytime mCurTracktotalPlaytime;
    tPEHandle mHandle;
    string mPlayerName;

    //parameter to store the PlayerAttributes' value
    tEventsSupported mEventsSupported;
    //parameter Events Supported by the Player
    //tU8 mAttributesSupported;

    tBoolean mIsPlayItemFromBrowseNeeded;
    tBoolean mIsRequestListItemNeeded;
    tUUID mPlayItemUID;
    //Current BT Device Source's info
    tDeviceInfo mConnectedDeviceInfo; //once once bt device can be connected to the system

    tPlaybackMode mPlaybackModeRequested;
    tPlaybackMode mCurrentplaybackMode;
    tRepeatMode mRepeatModeRequested;
    tRepeatMode mCurrentRepeatMode;

    //tBTPlayerSettingAttributeMode mAttributeValueRequested;

    unsigned char *mCurrentBrowsePath;
    string mCurrentReqPath;
    string mCurrentReqPathForChange;//To save currentbrowsepath with division slash
    string mSelectedItemParentPath;
    string mSelectedItemName;

    //Member to avoid playing the url Passed during Source Switch to BT Media(i.e avoid Lastmode List)
    tBoolean mIsBTControlActive;

    //Member to check playback action FastReverse in progress.
    tBoolean mIsFastRevInProgress;

    //Member to avoid the trigger DEVICE_INITIALIZED while the same BT device version changed from 1.0 to 1.4
    tBoolean mIsDeviceVersionUpdated;

    tAudioOutputDevice m_AudioOutputDevice;
    tAudioInputDevice m_AudioInputDevice;
    tDeviceID m_RemoteActivityDeviceID;
    tBoolean mIsAvpChangePathFromStartPlay;

    tBoolean mIsSiriSessionActive;
    tResult SendRemoteActivity(const tMountPoint mountPoint);
    tResult RemoteActivity(const tMountPoint mountPoint);
    tResult SendPlaybackStatus(const tPEPlaybackState status);          /**< send current playback status to own state machine */
    tResult SendPlaytimeStatus(const tMountPoint mountPoint);           /**< send current playtime status to own state machine */
    tResult SendInitDevice(const tMountPoint mountPoint);               /**< send init device message to own state machine */
    tResult SendDeviceInitialized(const tMountPoint mountPoint);        /**< send response to indexer state machine */
    tResult SendPauseOnTrackEnd(const tMountPoint mountPoint);          /**< send pause signal to own state machine, synch call */

    /**
    *   This Function Retrieves the DeviceAddress from DeviceID .
    *   Append the Retrieved DeviceAddress to the generic Object Path eg."/com/alps/bt/dev_aa_bb_cc_dd_ee_ff/",
    *   where dev_aa_bb_cc_dd_ee_ff is the deviceAddress String.
    */
    tResult InitializeDevice(tDeviceID deviceId);

    /**
     * Function used to response to caller in order to handle device types
     * that do not need special initialization, e.g. USB or SD card
     * Get device properties from DBManager -> GetDeviceInfo(&deviceInfo, mountPoint)
     * Answer via SMF -> SendAnswer(deviceInfo.deviceName,deviceID,connectionState=CS_CONNECTED).
     *
     * @param[in] mountPoint mount point of device
     * @param[in] deviceID device ID
     * @return != 0: error, = 0: OK
     */
     tResult NoInitAnswer(const tDeviceInfoString deviceInfoString);

     /**
      * Function retrieves the total number of playable files
      *
      * @param[in] deviceID device ID
      * @return != 0: error, = 0: OK
      */
     tResult GetNumberOfFiles(const tDeviceID deviceID);

    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpGetMediaPlayerItem Method Call
     *
     *Ihis informs the Type,Sub_Type and Features Supported for each Player
     *
     * @param[in] Status            Status of the Request
     * @param[in] NetworkOrder      Endianess of the Data received
     * @param[in] Data              Data of the Items Received
     * @param[in] Size              Size of the Data
     * @param[in] Message           DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpGetMediaPlayerItemMethodReturn(tBTMethodRequestStatus status,
                                                    tNetworkOrder networkOrder,
                                                    tByteArray data,
                                                    tArrayLength size,
                                                    tErrorMessage message);

    /**
     * This Function does AvpSetPlayer Method call on AVRCP 1.4 Device
     *
     * @param[in] PlayerId      Id of the Palyer in AVRCP 1.4 Device
     * @param[in] PlayerMode    Mode in which Player is accessed. ie. Addresssed Mode or Browsed Mode
     * @return != 0: error, = 0: OK
     */
    tResult AvpSetPlayer(tPlayerId playerId,tPlayerMode playerMode);
    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpSetPlayer Method Call
     *
     *
     * @param[in] Status            Status of the Request
     * @param[in] NumOfItems        Number of Items in the current Browsed Path (this param is valid when  response is for setting a Player in Browsed Mode )
     * @param[in] DBAware           Database aware or Unaware Player
     * @param[in] Message           DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpSetPlayerMethodReturn(tBTMethodRequestStatus status,
                                    tNumOfItems numOfItems,
                                    tDBAware dBAware,
                                    tErrorMessage errorMessage);

    /**
     * This Function does StartAudioPipe Method call on AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult StartAudioPipe();
    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for StartAudioPipe Method Call
     * This give the Input Audio Pipe Name,Codec Type,BitRate for the AVRCP 1.4 Device
     *
     * @param[in] PipeName     Input AudioPipe Name
     * @param[in] CodecType    CodecType of the Data streamed into the Input AudioPipe
     * @param[in] BitRate      BitRate of the Data streamed into the Input AudioPipe
     * @param[in] Message      DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult StartAudioPipeMethodReturn(tPipeName pipeName,tCodecType codecType,tBitRate bitRate,tErrorMessage message);

    /**
     * This Function does StopAudioPipe Method call on AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult StopAudioPipe();

    /**
     * This Function does AvpChangePath Method call on AVRCP 1.4 Device
     *
     * @param[in] Direction       Direction of Navigation It is either Level_up or Level_down
     * @param[in] ItemUID         UID of the Folder (Value of this Param is really used if it is Level_down)
     * @return != 0: error, = 0: OK
     */
    tResult AvpChangePath(tDirection direction,tLevelCount levelUpCount,tLevelCount levelDownCount,tItemUIDArray itemUIDArray);

    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpChangePath Method Call
     *
     * This informs the Number of Items in the Newly changed path(when the ChangePath request is successful)
     * @param[in] Status            Status of the Request
     * @param[in] NumOfItems        Number of Items in the Newly changed pat
     * @param[in] Message           DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpChangePathMethodReturn(tBTMethodRequestStatus status,tNumOfItems numOfItems,tErrorMessage message,const tPath argPath);


    /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpGetFolderItem Method Call
     * This informs Status of the player control command sent to the Device
     *
     * @param[in] Status            Status of the Request
     * @param[in] Scope             Scope from which the Items are returned
     * @param[in] NetworkOrder      Endianess of the Data received
     * @param[in] Data              Data of the Items Received
     * @param[in] Size              Size of the Data
     * @param[in] Message           DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpGetFolderItemMethodReturn(tBTMethodRequestStatus status,
                                         tScope scope,
                                         tNetworkOrder networkOrder,
                                         tByteArray data,
                                         tArrayLength size,
                                         tErrorMessage message);
    /**
     * This Function does AvpCtrlCmd Method call on AVRCP 1.4 Device
     *
     * @param[in] Command     Player Control Command like Play,Pause,Next,Previous etc..
     * @param[in] Event       Button Action like Button Press/Button Release
     * @return != 0: error, = 0: OK
     */
    tResult AvpCtrlCmd(tBTControlCmd command,tBTButtonEvent event);
     /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpCtrlCmd Method Call
     * This informs Status of the player control command sent to the Device
     *
     * @param[in] Command           Player Control Command like Play,Pause,Next,Previous etc..
     * @param[in] Event             Button Action like Button Press/Button Release
     * @param[in] Status            Status of the Request
     * @param[in] Message           DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpCtrlCmdMethodReturn(tBTControlCmd command,
                                   tBTButtonEvent event,
                                   tBTMethodRequestStatus status,
                                   tErrorMessage errorMessage);

     /**
     * This Function does AvpGetCapabilities Method call on AVRCP 1.4 Device
     *
     * @param[in] CapabilityId    ID to Know the Events Supported by the Device
     * @return != 0: error, = 0: OK
     */
     //tResult  AvpGetCapabilities(tEventsSupported capabilityId);
     /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpGetCapabilities Method Call
     * This informs Attributes supported by player application(in the AVRCP 1.4 Device)
     *
     * @param[in] Status                Status of the Request
     * @param[in] CapabilityId          ID to Know the Events Supported by the Device
     * @param[in] EventsSupported       The value of Each bit in the param represents whether the Event is supported or Not
     * @param[in] Message               DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpGetCapabilitiesMethodReturn(tBTMethodRequestStatus status,
                                           tCapabilityId capabilityId,
                                           tEventsSupported eventsSupported,
                                           tErrorMessage message);
     /**
     * This Function does AvpConfigNotify Method call on AVRCP 1.4 Device
     *
     * @param[in] EventID    ID of the Events are represented by Each Bit in the param
     * @return != 0: error, = 0: OK
     */
    tResult AvpConfigNotify(tEventId eventID);

     /**
     * This Function does AvpListAttr Method call on AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult AvpListAttr();

     /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpListAttr Method Call
     * This informs Attributes supported by player application(in the AVRCP 1.4 Device)
     *
     * @param[in] Status                Status of the Request
     * @param[in] AttributesBit         Each bit in the param represents whether the Attribute is supported/not.
     *                                  Attributes include Equalizer,Repeat mode,Shuffle mode and Scan mode
     * @param[in] Message               DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpListAttrMethodReturn(tBTMethodRequestStatus status,tAttributesBit attributesBit,tErrorMessage message);

     /**
     * This Function does AvpListVal Method call on AVRCP 1.4 Device
     *
     * @param[in] AttributeId       ID  of the Attribute
     * @return != 0: error, = 0: OK
     */
    tResult AvpListVal(tAttributeId attributeId);

     /**
     * This Function gets Called on receiving the Response from the  AVRCP 1.4 Device
     * for AvpListVal Method Call
     * This informs player application(in the AVRCP 1.4 Device)setting Attribute's values and its current value
     *
     * @param[in] Status                Status of the Request
     * @param[in] SettingValuesArray    Array of the Values supported by the Attribute
     * @param[in] ArrayLength           Length of the SettingValuesArray
     * @param[in] CurrentValue          Current value of the Attribute
     * @param[in] Message               DBus Error Message(if Any)
     * @return != 0: error, = 0: OK
     */
    tResult AvpListValMethodReturn(tBTMethodRequestStatus status,tBTPlayerSettingAttributeMode attributeId,tByteArray settingValuesArray,tArrayLength arrayLength,tCurrentValue currentValue,tErrorMessage message);

     /**
     * This Function is used to set the  value for player application(in the AVRCP 1.4 Device)setting�s attribute
     *
     * @param[in] AttributeId       ID  of the Attribute
     * @param[in] AttributeValue    value of the Attribute
     * @return != 0: error, = 0: OK
     */
    //tResult AvpSetVal(tAttributeId AttributeId,tAttributeValue AttributeValue);

     tResult AvpSetValMethodReturn(tBTMethodRequestStatus status,tErrorMessage message);
     /**
     * This Function Registers for Signals indicated by the AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult RegisterAvpSignals();
    /**
    * This Function Registers for indication to Know the status of AvpConfigNotify Method Call
    *
    * @return != 0: error, = 0: OK
    */
    tResult RegisterAvpNotifySetCompleteInd();

    /**
    * This Function Registers for indication of player control status such as play, pause, etc. by the AVRCP 1.4 Device
    *
    * @return != 0: error, = 0: OK
    */
    tResult RegisterAvpCtrlCmdInd();

     /**
     * This Function Registers for indication of play status by the AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult RegisterAvpPlayStatusInd();

     /**
     * This Function Registers for the Notification of Events by the AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult RegisterAvpNotifyChgInd();

    /**
     * This Function gets Called on Capturing the AvpNotifySetComplete Signal
     * This Informs the event that has been registered for Notification
     *
     * @param[in] EventID  ID of the Events.
     * @return != 0: error, = 0: OK
     */
    tResult AvpNotifySetCompleteInd(tEventId eventID);

    /**
     * This Function gets Called on Capturing the AvpCtrlCmd Signal.
     * This Informs the Current player control status
     *
     * @param[in] Command  player control status in the AVRCP 1.4 Device ie. MEDIA_SUSPEND or MEDIA_RESUME
     * @return != 0: error, = 0: OK
     */
    tResult AvpCtrlCmdInd(tBTControlCmd command);

    /**
     * This Function gets Called on Capturing the AvpPlayStatus Signal.
     * This Informs the PlayStatus and Playtime Status of the NowPlaying Item in the AVRCP 1.4 Device
     *
     * @param[in] Status        Status of the Request
     * @param[in] SongLength    Total time length of the NowPlaying Item
     * @param[in] SongPosition  Elapsed time Length of the NowPlaying Item
     * @param[in] PlayStatus    Play Status like Playing/Paused etc..
     * @return != 0: error, = 0: OK
     */
    tResult AvpPlayStatusInd(tBTMethodRequestStatus status,tPlaytime songLength,tPlaytime songPosition,tBTPlayStatus playStatus);

    /**
     * This Function notifies the Events like Playback Status Changed,Track Changed etc..
     *
     * @param[in] EventId       ID of the Event
     * @param[in] NetworkOrder  Endianess of the Data received
     * @param[in] Data          Data (if Any)corresponding to the Event Notified
     * @param[in] Size          Size of the Data
     * @param[in] Message       DBus Error Message(if Any)
    * @return != 0: error, = 0: OK
     */
    tResult AvpNotifyChgInd(tEventId eventId,tNetworkOrder networkOrder,
                    tByteArray data,tArrayLength size,
                    tErrorMessage message);

    /**
     * This Function Unregisters for Signals that were registerd for Indication by AVRCP 1.4 Device
     *
     * @return != 0: error, = 0: OK
     */
    tResult UnRegisterAvpSignals();


    tResult MediaPlayerApplicationSettingsDone(tDeviceID deviceID,tConnectionState connectionState);

    tResult AvpPlayItem(tScope scope,tItemUID itemUID);
    int SetCurrentBrowsedPath(unsigned char *reqPath);
    int findFolderPathToMove(char *requestPath,string &PathToMove);
    int DoChangePath(tDirection direction,tLevelCount levelUpCount,tLevelCount levelDownCount,tItemUIDArray itemUIDArray);
    void DoPlayItemFromBrowse();
    void ClearFlags(tBool isInit);
    void ClearDeviceDetails();
    tResult MapBTControlCommandToPlaybackState(tPEPlaybackState &playbackState, const tBTControlCmd command);
    tResult SendMediaPlayerApplicationSettingsDone(tDeviceID deviceID,tConnectionState connectionState);
    tResult GetSupportedAttr();
    tResult GetSupportedAttrValue(const tBTPlayerSettingAttributeMode &attributeId);
    tResult ReqDevNotificationCapabilities();
    tResult DoDBChangeShuffleSupport(char *parameterStr);  /**< dispatched from service by thread factory */
    tResult DoDBChangeRepeatSupport(char *parameterStr);  /**< dispatched from service by thread factory */
    tResult DoDBChangeDeviceVersion(char *parameterStr);  /**< dispatched from service by thread factory */

    /**
     * The Function will update the NowPlayingList status to database.
     */
    tResult DoDBChangeNowPlayingListSupport(char *parameterStr);
    //int UpdateDeviceVersion(tBTAvrcpVersion avrcpVersion);
    tResult DoDeleteVTBluetoothCache(char *parameterStr);  /**< dispatched from service by thread factory */

    /**
     * This Function Resets all the Browse Informations of the currently browsed player(by invoking ClearBrowseInfoFlags())
     * and Request the stack to Set the Newly addressed player as Browsed Player
     *
     * @param[in] PlayerId          ID of Newly addressed Player
     * @return != 0: error, = 0: OK
     */
    tResult AvpChangeBrowsedPlayer(const tPlayerName playerId);
    tResult HandlePlayerChangeWhileGetFolderItemInProgress(const tPlayerName playerName);
    tResult HandlePlayerChangeWhileChangePathInProgress(const tPlayerName playerName);
    tResult AvpChangePathToRootMethodReturn(tBTMethodRequestStatus status,tNumOfItems numOfItems,tErrorMessage message);

    /**
     * This Function Resets all the Browse Informations of the currently browsed player
    */
    void ClearBrowseInfoFlags(tBool isInit);

    void ClearRemoteActivityFlags();

    tBoolean mIsRemoteActivityNeeded ;
    tBTActionOnPlayTime mActionOnPlayTime ;
    tPlaytime mTotalPlayTime;
    tPlaytime mElapsedPlayTime;
    void StorePlayTimeReceived(const tPlaytime &totalPlayTime,const tPlaytime &elapsedPlayTime,const tBTActionOnPlayTime ActionOnPlayTime = COMPARE_PLAY_TIME);
    tResult ValidatePlayTimeAndActivateBTDevice(const tPlaytime &elapsedPlayTime,const tPlaytime &totalPlayTime);
    tResult ActivateBTDevice();

    /* The following member functions and member variables are added for SIRI handling */
    tResult HandleVRSessionStatus(tVRSessionStatus vrstatus,tSiriStatus siristatus,tVRSupportStatus vrsupport);
    tBoolean StartSiriSessionTimer();
    void StopSiriSessionTimer();
    static bool SiriSessionTimerCallBack(timer_t timerID , void* instance ,const void *userData);
    void CancelTimers();
    tResult ClearSiriSessionFlags();
    Timer m_SiriSessionTimer;
    timer_t m_SiriSessionTimerID;
    tBoolean bRemoteActivityReqViaSiri;
    tMediaObject m_CurrentMediaObject;

    sem_t m_TaskSyncSem;
    tBoolean m_TaskSyncSemWait;
    Lock  m_TaskSyncSemMutex;
    Lock m_Mutex;
    Lock m_LockForStateOfPlayinStartPlay;
    std::vector<tTaskContext> mTaskVector;
    tResponseMsg m_response;

    //timer related variables amd functions for retrial of play.
    Timer mRetryPlayTimer;
    timer_t mRetryPlayTimerID;
    static bool RetryPlayTimerCallBack(timer_t timerID , void* instance, const void *userData);
    int mRetryPlayTimerCount;

    void SetTaskSyncSemWait(tBoolean Value);
    tBoolean GetTaskSyncSemWaitValue();
    int StartDoTask();
    tBoolean IsTaskAvailable();
    int StartTask();
    int EndTask();
    void AddTaskToVector(tTaskContext taskContext);
    void RemoveTaskFromVector(int TaskId);

    int ProcessContentChanged();
    int NowPlayingContentUpdated();
    tResult GetAndUpdateFolderItemsBulk(string curPath,tStartItem startItem,tEndItem endItem,tEndItem &lastItemretrived,tBoolean &isterminated,int &lastUpdatedBatch,tBoolean& cacheUpdated);
    tResult CheckFolderItem(tStartItem startItem);
    tResult ChangeFolderPath(const char * reqPath);

    void SetCurrentBrowsePath(string path);
    unsigned char* GetCurrentBrowsePath();
    void ClearCurrentBrowsePath();
    string GetCurrentReqPathForChange();
    void SetCurrentReqPathForChange(string CurrentReqPathForChange);
    string GetStringWithoutSlash(string name);
    int findFirstofDivisionSlash(string name);
    tListID NowPlayingListID;
    int AvpChangePathReturnError();
    int AvpGetFolderItemReturnError();
    tStateOfPlayinStartPlay mStateOfPlayinStartPlay;
    void SendPlaybackActionToBTControl(tPlaybackAction playbackAction);
    void StartNowplayingListUpdate();

    tBoolean mNowplayingListAvilable;

    //BT CoverArt functions
    tBTCoverArtInfo GetCoverArtInfo();
    tResult SendAlbumArtStatusToPlayerManager(const tBool albumartAvailable);
    void FillImageInfo(const std::string imagePath, tAlbumArtObjectPtr &albumArtObjectPtr);
    std::string GetObjectIDMatcher(const tObjectID &objectID);
    tBTCoverArtInfo mLatestCoverartInfo;
    std::string mCurrentAlbumartPath;
    Lock  m_CoverartInfoLock;
};

#endif /*BTControl_H_*/

/** @} */
