/**
 * @defgroup DataProvider DataProvider
 * @author Stefan Scherber, Matthias Thömel
 *
 * Public interface for data provider used by all components which must retrieve single data items like configurations
 * @{
 */

#ifndef DATAPROVIDER_H_
#define DATAPROVIDER_H_

#include <vector>
using namespace std;

#include "AlbumArtCache.h"
#include "DBManager.h"
#include "ILocalSPM.h"
#include "ThreadFactory.h"
#include "TypeDefinitions.h"
#include "FastUTF8.h"
#include "CddaMetaDataInfo.h"
class DataProvider : public ILocalSPM, public TFThread
{

public:

    //SPM part

    /**
     * Constructor of component.
     * This function is used by LocalSPM to store the DataProvider componentID in mComponentID.
     *
     * @param[in] componentID component ID assigned by SPM
     * @return void
     */
    DataProvider(const tComponentID componentID);

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~DataProvider();

    /**
     * This function is used by LocalSPM to cleanup the DataProvider.
     * Free and delete old album art object -> SetAlbumArtObjectToCache(ptrToAlbumArtObject=NULL)
     * Inform LocalSPM that Done is ready -> DoneDone(0)
     *
     * @return < 0: error, = 0: OK
     */
    tResult Done();


    /**
     * This function starts a user function in a thread from the thread factory's thread pool.
     * After leaving while loop used thread is stopped and released for other users
     *
     * @return void
     */
    void Do(int functionID, void *ptr);

    /**
     * Brief: returns statistic data as a string for the over all media player object statitcs
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    int Statistics(OUT tStatistics stat);

    //Data handling part

    /**
     * Sets a device as initialized
     *
     * Get visibility configuration for device which is not supported due to several reasons from DataProvider //Roadmap 13023
     *   -> UnsupportedDeviceTypeVisible()
     *   -> UnsupportedFileSystemVisible()
     *   -> UnsupportedPartitionVisible()
     * If device shall be visible
     * - Call DBManager::SetDeviceNameAndConnectionState( deviceID, deviceName, connectionState, deviceState)
     *
     * @param[in] deviceID ID of device
     * @param[in] deviceName name of device
     * @param[in] connectionState connection state of device
     * @param[in] deviceState state of device
     * @return < 0: error, = 0: OK
     */
    tResult DeviceInitialized(const tDeviceID deviceID, //Roadmap 13023: 100%
            const tDeviceName deviceName,
            const tConnectionState connectionState,
            const tDeviceState deviceState);

    /**
     * sets the current language
     *
     * calls: DBManager::SetLanguage(language)
     *
     * @param[in] language current language
     * @return < 0: error, = 0: OK
     */
    tResult SetLanguage(const tLanguageType language);

    /**
     * triggers the clearance of the complete database
     *
     * calls:
     * LocalSPM::Shutdown()
     * DBManager::ClearMediaPlayerData()
     * LocalSPM::Startup()
     *
     * @return < 0: error, = 0: OK
     */
    tResult ClearMediaPlayerData(void);

    /**
     * triggers the recreation of the complete database
     *
     * calls:
     * ClearMediaPlayerData()
     * DBManager::CreateTestDatabase()
     *
     * @param[in] createDevices switch if test devices should be created
     * @param[in] createSongs switch if test media object should be created
     * @return < 0: error, = 0: OK
     */
    tResult RecreateTestDatabase(const tBoolean createDevices, const tBoolean createSongs);

    /**
     * Retrieves the album art information
     *
     * Check if album art is already stored in cache for specific album art string
     *    -> GetAlbumArtObjectFromCache(ptrToAlbumArtObject, albumArtString)
     * If available (ptrToAlbumArtObject != NULL) send cached information immediately
     *    -> ForwardAlbumArtObject(ptrToAlbumArtObject, userContext)
     * Else retrieve album art object from device control -> GetAlbumArtObjectFromDevice(...)
     *
     * @param[in] albumArtString device ID + URL of image to read
     * @param[in] sizeX number of pixels in X direction
     * @param[in] sizeY number of pixels in Y direction
     * @param[in] mimeType type of picture (MMT_PNG, MMT_JPG, ...)
     * @param[in] userContext user context structure which will be forwarded to the chain in
     * case of an async update must be done after reading the album art.
     * @return < 0: error, = 0: OK
     */
    tResult GetMediaObjectAlbumArtInfo(const tAlbumArt albumArtString,
            const tSize sizeX,
            const tSize sizeY,
            const tMimeType mimeType,
            const tUserContext userContext);

    /**
     * Retrieves the album art image data
     *
     * Check if album art is already stored in cache for specific album art string
     *    -> GetAlbumArtObjectFromCache(ptrToAlbumArtObject, albumArtString)
     * If available (ptrToAlbumArtObject != NULL) send cached information immediately
     *    -> ForwardAlbumArtObject(ptrToAlbumArtObject, userContext)
     * Else retrieve album art object from device control -> GetAlbumArtObjectFromDevice(...)
     *
     * @param[in] albumArtString device ID + URL of image to read
     * @param[in] sizeX number of pixels in X direction
     * @param[in] sizeY number of pixels in Y direction
     * @param[in] mimeType type of picture (MMT_PNG, MMT_JPG, ...)
     * @param[in] userContext user context structure which will be forwarded to the chain in
     * case of an async update must be done after reading the album art.
     * @return < 0: error, = 0: OK
     */
    tResult GetMediaObjectAlbumArt(const tAlbumArt albumArtString,
            const tSize sizeX,
            const tSize sizeY,
            const tMimeType mimeType,
            const tUserContext userContext);

    static void CatchSEGVforDevil(int signal, siginfo_t *si, void *arg);

    /**
     * Converts the album art image to requested format and size
     *
     * internally uses devIL image library to convert the image
     *
     * @param[in] orgAlbumArtObject pointer to actual albumart object which need to be converted
     * @param[out] convertedAlbumArtObject pointer to new converted albumart object
     * @param[in] newSizeX requested sizeX
     * @param[in] newSizeY requested sizeY
     * @param[in] newMimeType requested new format
     *
     * @return < 0: error, = 0: OK
     */
    tResult ConvertAlbumArt(const tAlbumArtObjectPtr orgAlbumArtObject,
            tAlbumArtObjectPtr &convertedAlbumArtObject,
            const tSize newSizeX,
            const tSize newSizeY,
            const tMimeType newMimeType);

    /**
     * Get album art object from device control
     *
     * Spawn worker thread
     * Create RequestStateMachine
     * Store user context temporary
     * Get the device ID from the album art string -> DBManager::GetDeviceFromAlbumArtString(&deviceID, albumArtString)
     * RouteMessageAnswer(GET_ALBUM_ART) to specific DeviceControl (deviceType).
     *    -> deviceType "GET_ALBUM_ART albumArtString","REQUEST_SM GET_ALBUM_ART_ANSWER"
     * The thread finishes after album art object is delivered by device control with GET_ALBUM_ART_ANSWER event.
     *
     * @param[in]  albumArtString device ID + URL of image to read
     * @param[out] albumArtObjectPointer pointer to albumart object if retrieval is success
     * @return < 0: error, = 0: OK
     */
    tResult GetAlbumArtObjectFromDevice(const tAlbumArt albumArtString , tAlbumArtObjectPtr &albumArtObjectPointer);

    /**
     * Get videoThumbnail object using free Image
     *
     *
     * @param[in]  albumArtString device ID + URL of image to read
     * @param[out] albumArtObjectPointer pointer to albumart object if retrieval is success
     * @return < 0: error, = 0: OK
     */
    tResult GetThumbNailObject(OUT tAlbumArtObjectPtr &ThumbNailObjectPtr, IN const tThumbnailImage &Image,
            IN tSize Width, IN tSize Height, IN tMimeType MimeType,IN const char* filename);
    /**
     * Forwards album art object to OutputWrapper.
     *
     * Depending on functionID of user context and error occured or not one of the following methods is called.
     * if( functionID == GETMEDIAOBJECTALBUMARTINFO)
     *    if(imageSize > 0) no error
     *       OutputWrapper::SendGetMediaObjectAlbumArtInfoAnswer(mimeType, imageSize, userContext)
     *    else
     *       OutputWrapper::SendGetMediaObjectAlbumArtInfoError(errorCode, userContext)
     * if( functionID == GETMEDIAOBJECTALBUMART)
     *    if(imageSize > 0) no error
     *       OutputWrapper::SendGetMediaObjectAlbumArtAnswer(imageData, userContext)
     *    else
     *       OutputWrapper::SendGetMediaObjectAlbumArtError(errorCode, userContext)
     *
     * @param[in] ptrToAlbumArtObject pointer to allocated album art object
     * @param[in] userContext user context structure which was forwarded by GetAlbumArtObject
     * @return < 0: error, = 0: OK
     */
    tResult ForwardAlbumArtObject(const tAlbumArtObjectPtr ptrToAlbumArtObject, const tUserContext userContext);

    tResult ForwardAlbumArtInfo(const tAlbumArtObjectPtr ptrToAlbumArtObject, const tUserContext userContext);

    /**
     * set the activeSource Flag for a specific device ID
     *
     * It uses DBManager::ActiveMediaDeviceSet(...)
     *
     * @param[in] deviceID device ID for which the flag should be set
     * @param[in] activeSource flag value
     * @return < 0: error, = 0: OK
     */
    tResult ActiveMediaDeviceSet(const tDeviceID deviceID, const tBoolean activeSource);

    /**
     * Returns a list of device types that support autoplay
     *
     * @param[out] deviceTypes list of device types
     * @return < 0: error, = 0: OK
     */
    tResult GetAutoplayDeviceTypes(vector<tDeviceType> &deviceTypes);

    /**
     * Returns the indexing order priority of a device type
     *
     * @param[out] priority priority value (0: highest priority)
     * @param[in] deviceType device type to check
     * @return < 0: error, = 0: OK
     */
    tResult GetIndexingPriority(tPriority &priority, const tDeviceType deviceType);

    /**
     * Checks if a certain device type will be supported by the indexer
     *
     * @param[in] deviceType device type to check
     * @return true or false
     */
    tBoolean IsIndexingSupported(const tDeviceType deviceType);

    /**
     * Checks if a certain device type is indexable or not
     *
     * @param[in] deviceType device type to check
     * @return list of devices
     */
    tBoolean IsIndexableDevice(tDeviceType deviceType,tConnectionType connectionType);

    /**
     * Checks if a certain device type will be supported by the album art indexer
     * Roadmap 15009
     *
     * @param[in] deviceType device type to check
     * @return true or false
     */
    tBoolean IsAlbumArtIndexingSupported(const tDeviceType deviceType); //Roadmap 15009: 100%

    /**
     * This function checks AlbumArt support for the requested DeviceType
     *
     * @param[in] deviceType device type to check
     * @return true or false
     */
    tBoolean IsAlbumArtSupported(const tDeviceType deviceType);

    /**
     * Returns a list of device types that should run in permanent streaming mode
     *
     * @param[out] deviceTypes list of device types
     * @return < 0: error, = 0: OK
     */
    tResult GetPermanentStreamingDeviceTypes(vector<tDeviceType> &deviceTypes);

    /**
     * Returns vector of strings to find all supported files via extension
     *
     * @return tFileExtensions: all supported file extensions
     */
    tFileExtensions GetSupportedFileExtensions();

    /**
     * Returns vector of strings to find all supported files via file filters
     *
     * @return tFileExtensions: all supported file filters
     */
    tFileExtensions GetSupportedFileFilters();

    tResult GetSearchStringForValidPartition(tSearchString &searchString);
    tBool IsValidContentAvilable(const tMountPoint mountPoint);
    /**
     * Returns a fallback action for a failed action
     *
     * @param[out] outAction configured fallback action
     * @param[in] inAction failed action for which a fallback is needed
     * @return < 0: error, = 0: OK
     */
    tResult GetFallbackAction(tConfigTableAction &outAction, const tConfigTableAction inAction);

    /**
     * ???
     *
     * ???.
     *
     * @param[in] usedConfigTable used config table (CT_TEST, CT_DEVICE_ACTIVATION, CT_LIST_SELECTION)
     * @param[in] precondition precondition
     * @param[in] deviceID device ID of affected device
     * @param[out] action action
     * @return < 0: error, = 0: OK
     */
    tResult ProcessConfigTable(tConfigTableAction &action, const tConfigTable usedConfigTable, const tConfigTablePrecondition precondition, const tDeviceID deviceID);

    /**
     * Set test values for criteria
     *
     * @param[in] value1 test value of criteria 1
     * @param[in] value1 test value of criteria 2
     * @param[in] value1 test value of criteria 3
     * @return < 0: error, = 0: OK
     */
    tResult SetTestCriteria(const tInteger value1, const tInteger value2, const tInteger value3);

    /**
     * Reset test values for criteria to -1
     *
     * @return < 0: error, = 0: OK
     */
    tResult ResetTestCriteria();

    /**
     * Checks if we are in the startup phase
     *
     * @return true or false
     */
    tBoolean IsStartupPhase();

    /**
     * Checks if a certain device is the last played one
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsLastPlayedDevice(const tDeviceID deviceID);

    /**
     * Checks if any device is active
     *
     * @return true or false
     */
    tBoolean IsAnyDeviceActive();

    /**
     * Checks if any device is connected
     *
     * @param[in] deviceID device ID of excluded device
     * @return true or false
     */
    tBoolean IsAnyDeviceConnected(const tDeviceID deviceID);

    /**
     * Checks if autoplay is configured for the type of a certain device
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsDeviceAutoplay(const tDeviceID deviceID);

    /**
     * Checks if the type of a certain device is the same like the type of the last played device
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsSameDeviceTypeLikeBefore(const tDeviceID deviceID);

    /**
     * Checks if permanent streaming is configured for the type of a certain device
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsPermanentStreamingActive(const tDeviceID deviceID);
    tBoolean IsPermanentStreamingActive(const tDeviceType deviceType);

    /**
     * Checks if the last mode list is available for a certain device
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsLastModeListAvailable(const tDeviceID deviceID);

    /**
     * Checks if any DB content is available for a certain device
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsAnyDBContentAvailable(const tDeviceID deviceID);

    /**
     * Checks if the current selection on the iPod changed
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsCurrentIPodSelectionChanged(const tDeviceID deviceID);

    /**
     * Checks if formerconnectionstate for a device is hardware malfunction
     *
     * @param[in] deviceID device ID of affected device
     * @return true or false
     */
    tBoolean IsHwMalfuncInLastConnection(const tDeviceID deviceID);

    /**
     * imports a string into media player format. That includes:
     * - check if the string is GBK and convert it to UTF-8
     * - if string is longer than max length configured, cut it
     *
     * @param[inout] str in: original string, out: converted and cut string
     * @param[in] size size of the string
     *
     * @return nothing
     */
    void ImportAndCut(FastUTF8::tString str, const size_t size)
    {
        Convert2UTF8(str, size);
        CutMetadata(str, size);
    }

    void Convert2UTF8(FastUTF8::tString str, const size_t size);
    void CutMetadata(FastUTF8::tString str, const size_t size);

    /**
     * Returns the known application table for a given device type
     *
     * @param[in] deviceType the device type for which the table should be returned
     * @param[out] appTable contains all known applications for the given device type
     * @return != 0: error, = 0: OK
     */
    tResult GetApplicationTable(vector<tApplTableEntry> &appTable, const tDeviceType deviceType); //Roadmap 13008

    /**
     * Function checks if the genre contains the string "Genre_nnn" and tries to map the number nnn to a stored genre name.
     * the mapping table is in DataProvider.
     *
     * @param[in] genreID genre number as retrieved from the ID3V1 tag.
     * @param[inout] genre text contains the genre number at call and the translated name at exit.
     * @return != 0: error, = 0: OK
     */
    tResult VerifyGenreName(tUInt genreID ,tMetadata &genre); //Roadmap 13014

    /**
     * Checks if a certain device type is supported
     * Roadmap 13023
     *
     * @param[in] deviceType device type to check
     * @return true or false
     */
    tBoolean IsDeviceTypeSupported(const tDeviceType deviceType); //Roadmap 13023: 100%

    /**
     * Checks if a certain device type is supported
     * Roadmap 13023
     *
     * @param[in] fileSystemType file system type to check
     * @return true or false
     */
    tBoolean IsFileSystemSupported(const tFileSystemType fileSystemType); //Roadmap 13023: 100%

    /**
     * Checks if a certain device type is supported
     * Roadmap 13023
     *
     * @param[in] partitionNumber number of the partition on the device to check
     * @return true or false
     */
    tBoolean IsPartitionSupported(const tPartitionNumber partitionNumber); //Roadmap 13023: 100%

    /**
     * Returns default PEState for PEngine inclusive configuration values
     * Roadmap 13010
     *
     * @param[out] peState default state struct for PEngine
     * @return < 0: error, = 0: OK
     */
    tResult GetDefaultPEState(me::state_t &peState); //Roadmap 13010: 100%

    /**
     * Returns a list of internal devices
     * Roadmap 13005
     *
     * @param[out] deviceInfos list of internal devices
     * @return < 0: error, = 0: OK
     */
    tResult GetInternalDevices(vector<tDeviceInfo> &deviceInfos); //Roadmap 13005: 100%

    /**
     * Traces all GMP configuration item's name and its values
     *
     */
    void TraceMediaplayerConfigurations(void);

    static tResult MarshalDeviceInfo(tDeviceInfoString &deviceInfoString, const tDeviceInfo deviceInfo);
    static tResult UnMarshalDeviceInfo(const tDeviceInfoString deviceInfoString, tDeviceInfo &deviceInfo);

    /**
     * Returns available current for charging device depending on connected hub
     *
     * @param[out] current in mA
     * @return < 0: error, = 0: OK
     */
    int GetAvailableCurrentForDevice(const tPath hubVendorID, const tPath hubProductID, const int portNumber);

    /**
     * Checks if a certain combination of device type and object type is supported for last mode position
     * Roadmap 14017
     *
     * @param[in] deviceType device type to check
     * @param[in] objectType object type to check
     * @return true or false
     */
    tBoolean IsLastModePositionSupported(const tDeviceType deviceType, const tCategoryType objectType); //Roadmap 13017: 100%

    tBoolean IsiPodControlLocationInfoEnabled();

    /*15008_SCAN.
     *For the given deviceType,it will return TRUE if NowPlaying track scanning supported by that deviceType.
     *Else,it will return FALSE
     */
    tBoolean IsDeviceSupportTrackScanning(const tDeviceType deviceType);

    /* RoadMap 15004_CodesetConversionForNissan
     *
     * Checks is any of the metadata is a Latin1 string.
     * If found,the biflag representing that metadata in metadataconvertflag of mediaObject will be set.
     *
     * @param[inout] mediaObject whose metadata are to be checked
     * @return < 0: error, = 0: OK
     */
    tResult FindLatin1Metadata(tMediaObject &mediaObject,const void* _info);

    /* RoadMap 15004_CodesetConversionForNissan
     *
     * Checks is any of the Latin1 metadata is a  non-UTF8 string.
     * If found,the biflag representing that metadata in metadataconvertflag of mediaObject will be set.
     *
     * @param[inout] mediaObject whose metadata are to be checked
     * @return < 0: error, = 0: OK
     */
    tResult FindNonUTF8Latin1Metadata(tMediaObject &mediaObject);

    /* RoadMap 15004_CodesetConversionForNissan
     *
     * Checks the biflag representing that each metadata in metadataconvertflag of mediaObject
     * if any BitFlag is found set,then its corresponding metadata will be converted to a UTF-8 String
     *
     * @param[inout] mediaObject whose metadata are to be checked
     * @return < 0: error, = 0: OK
     */
    tResult ConvertNonUTF8Latin1Metadata2UTF8(tMediaObject &mediaObject,const void* _info);

    /* RoadMap 15004_CodesetConversionForNissan
     *
     * Wrapper function which invokes the functionalities  FindLatin1Metadata,FindNonUTF8Latin1Metadata and ConvertNonUTF8Latin1Metadata2UTF8
     *
     * @param[inout] mediaObject whose metadata are to be checked and converted
     * @return < 0: error, = 0: OK
     */
    tResult FindAndConvertLatin1MetadataEntries(tMediaObject &mediaobject,const void* _info);

    //To Dump the hex values of the mediaobject's metadata entries
    void HexDumpMediaObjectMetadataFields(tMediaObject &mediaObject,FILE* codesetConvLogFile);

    /**
     * Determine the read only flag of a file
     * Roadmap 16011
     *
     * @param[out] bReadOnlyFlag read only flag of the file
     * @param[in] fileMode file mode - permission triad for user, group and others in numeric octal notation
     * @param[in] fileUserID UID of the file
     * @param[in] fileGroupID GID of the file
     * @return true or false
     */
    tResult GetReadOnlyFlag(tBoolean &bReadOnlyFlag, const tFileMode fileMode, const tUserID fileUserID, const tGroupID fileGroupID);

    /*This function Adds the Device to the Vector mIgnoreLastModePauseDevices provided
     * the DeviceID is valid & doesn't exist in the vector
     *
     * @param[in] deviceID ID of the device to add into vector
     */
    void AddDeviceToIgnoreLastModePause(const tDeviceID deviceID);

    /*This function removes the Device from the Vector mIgnoreLastModePauseDevices provided
     * the DeviceID is valid
     *
     * @param[in] deviceID ID of the device to remove from vector
     */
    void RemoveDeviceFromLastModePauseList(const tDeviceID deviceID);

    /*This function checks the availability of given device ID in the mIgnoreLastModePauseDevices Vector
     *
     * @param[in] deviceID ID of the device to check its availability in vector
     * @return true or false
     */
    tBoolean IsIgnoreLastModePauseNeeded(const tDeviceID deviceID);

    void clearCache(void);

    tSPMSysPwrMode mSPMSystemState;

    /*Set the SPM Power Mode from the clienthandler of SPM*/
    void vUpdateSPMSysPwrMode(tSPMSysPwrMode SPMSysPwrMode);

    /*Get the SPM Power Mode*/
    tSPMSysPwrMode GetSPMSysPwrMode();

    /*Get Device file type string from enum type*/
    string GetDeviceFileSystemType(tFileSystemType etype);
    /*Get DB path from input path by user*/
    string ConvertDbPath(string Path);
    /*CDDA text common implementation for DVD and CDDA*/
    tResult parseCddaResponseBuffer(IN const tU8 *pu8RawATA,IN const tU16 textLength,IN const tU8 TrackNo);
    //get metadata for Album and Title
    tResult getCddaMetaDataForTitle(OUT string & Album , OUT MapCDDAMetadata & Title, IN tU8 TrackNo);
    //get metadata for Artist
    tResult getCddaMetaDataForArtist(OUT MapCDDAMetadata & Artist, IN tU8 TrackNo);

    //CMG3G-14540 : Wireless Carplay requirements for BTLimitationMode
    tBTLimitationModeDeviceList& getBTLimitationModeDeviceList();

    /**
     * Function fills codec restriction table from calibration
     *
     * @return != 0: error, = 0: OK
     */
    tResult FillCodecRestrictionTable();

    /**
     * Function delivers answer if a video file is playable.
     * Check video metadata to meet requirements.
     * Return true if video file is playable, otherwise false.
     *
     * @param[in] codec used codec (e.g. CDC_MPEG4, CDC_DIVX, CDC_AVC, ...)
     * @param[in] profile used profile (e.g. Main, Simple, Advanced ...)
     * @param[in] level profile level
     * @param[in] width resolution width in pixel
     * @param[in] height resolution height in pixel
     * @param[in] frameRate frame rate in frames per second (fps)
     * @param[in] bitRate bit rate in bits per second (bps)
     * @return true or false
     */
    tReturnValue IsVideoPlayable(const tCodec codec,
            const tVideoProfileName profile,
            const tVideoProfileLevel level,
            const tSize width,
            const tSize height,
            const tFrameRate frameRate,
            const tBitRate bitRate);

    /**
     * Function delivers answer if a audio file is playable.
     * Check audio metadata to meet requirements.
     * Return true if audio file is playable, otherwise false.
     *
     * @param[in] codec used codec (e.g. CDC_WMA, ...)
     * @param[in] version used version (e.g. Version 1, ...)
     * @return true or false
     */
    tReturnValue IsAudioPlayable(const tCodec codec, const tAudioVersionName version);

    /**
     * Need of CovertArt is limited to few list types.
     * Return true if the given list type needs coverart, otherwise false.
     *
     * @param[in] listtype of a list
     * @return true or false
     */
     tBoolean IsCoverArtInfoNeededInList(const tListType listType);

    /**
     * Need of Thumbnail is limited to few list types.
     * Return true if the given list type needs Thumbnail, otherwise false.
     *
     * @param[in] listtype of a list
     * @return true or false
     */
     tBoolean IsThumbnailInfoNeededInList(const tListType listType);

    /**
     * Checks whether device is connected or not
     * Return true if device connected, otherwise false.
     *
     * @param[in] mountPoint of the device
     * @return true or false
     */
    tReturnValue checkIfDeviceConnected(string mountPoint);

    /**
     * Checks whether database can be used instead of VTFile for list creation
     * Return true if possible, otherwise false.
     *
     * @param[in] deviceInfo info of the device for which list has to be created
     * @param[in] path path for which list has to be created
     * @param[in] listType type of list
     * @return true or false
     */
    tBoolean IsFileBrowsingByDb(const tDeviceInfo &deviceInfo, const char *path, const tListType &listType);

    /**
     * Set the CallControl configuration in DataPool
     * Return true if success, otherwise false.
     *
     * @param[in] config: Call controls config info for iAP2 identification.
     * @return true or false
     */
    tResult SetDiPOCallControlsConfiguration(const tDiPOCallControlsConfiguration config);

    /**
     * Get the CallControl configuration in DataPool
     * Return true if success, otherwise false.
     *
     * @param[out] config: Call controls config info for iAP2 identification.
     * @return true or false
     */
    tDiPOCallControlsConfiguration GetDiPOCallControlsConfiguration();

    /**
     * Set the SPM StandByMode Status
     *
     * @param[in] standbyModeStatus: Status of the SPM StandByMode.
     * @return void
     */
    void SetSPMStandByModeStatus(const bool standbyModeStatus);

    /**
     * Get the SPM StandByMode Status
     *
     * @return bool standbyModeStatus: Status of the SPM StandByMode.
     */
    bool GetSPMStandByModeStatus();

    /**
     * Set the Ipod Mute Status
     *
     * @param[in] IpodMuteStatus: Mute Status of Ipod source.
     * @return void
     */
    void SetIpodMuteStatus(const bool IpodMuteStatus);

    /**
     * Get the Ipod Mute Status
     *
     * @return bool IpodMuteStatus: Mute Status of the Ipod Source.
     */
    bool GetIpodMuteStatus();

private:
    tResult GetAlbumArtObject(tAlbumArtObjectPtr &aaObjectPtr,
            const tAlbumArt albumArtString, const tSize sizeX, const tSize sizeY,
            const tMimeType mimeType, const tUserContext userContext);
    tResult DoGetMediaObjectAlbumArt(char *parameterString);
    tResult DoGetMediaObjectAlbumArtInfo(char *parameterString);

    void setFileSystemtype();
    AlbumArtCache mAlbumArtCache;
    Lock mDevilLock;

    vector<tDeviceID> mIgnoreLastModePauseDevices; //NCG3D-25268

    tInteger mTestValue1;
    tInteger mTestValue2;
    tInteger mTestValue3;

    tSPMSysPwrMode mSPMSysPwrMode;
    std::map<tFileSystemType,string> mFileSystemType;

    bool mSPMStandByModeStatus;
    bool mIpodMuteStatus;
    Lock mLock;
    tBTLimitationModeDeviceList btLimitationModeDeviceList; //CMG3G-14540: Wireless Carplay requirements.

    /*CDDA text common Implementation */
    MapCDCode mTitelCDText; // CD title packet 0x80
    MapCDCode mArtistCDText;// CD Artist packet 0x81
    bool m_bArtistCDTextforAlbumPresent;
    string mAlbum; // CD text Album from packet 0x80
    int getnullPos(tU8 *ch, int length); // get null position of a character array
    tBool isValidPack(tU8 pty, int id, tU8 TrackNo); // check validity of a cd text packet
    // Common function for title and artist
    tResult getMetaData(OUT MapCDDAMetadata & metadata, IN MapCDCode &CDText, IN tU8 TrackNo,bool artist);

    vector<tCodecRestriction> m_CodecRestrictionTable;  /**< codec restriction table */

public:
    /**********************************************************************************************
     * The following definitions are for the on-the-fly definitions for single
     * and simple configuration values by all developers
     */

    /**
     * logs config value to the ETG trace by using name and value
     *
     * @param[in] dbName Name of the value to be used
     * @param[in] data   pointer to the slot where data is available
     *
     * @return < 0: error, = 0: OK
     */
    static void TraceConfigValue(const int data, const char *dbName);
    static void TraceConfigValue(const long data, const char *dbName);
    static void TraceConfigValue(const float data, const char *dbName);
    static void TraceConfigValue(string &data, const char *dbName);

    class ConfigItemBase
    {
    public:
        virtual void Trace()=0;
    };

    static vector <ConfigItemBase*> configurationItems; // pointer to list of configuration items

    // template for all simple config values in the data base
    template<typename T, typename dbName> class DatabaseItem : public ConfigItemBase
    {
    public:
        T mValue;
        const char *mDbName;
        int mLoaded;

        DatabaseItem()
        {
            dbName name;                // needed to get access to name and default value of item
            mLoaded = 0;                // value not yet loaded from db
            mDbName = name.get();        // set the name in db for this value
            mValue = name.defaultValue; // set the default value
            //TraceConfigValue(mValue, mDbName); // trace configuration value resolving its type

            configurationItems.push_back(this); // add item into master list of configurations
        }

        virtual T operator ()()
        {
            return mValue;
        }

        virtual T operator = (T Value)
        {
            mValue = Value;
            //TraceConfigValue(mValue, mDbName);    // trace configuration value resolving its type
            return mValue;
        }

        void Trace()
        {
            TraceConfigValue(mValue, mDbName); // trace configuration value resolving its type
        }
    };

    // macro to define every data base config item in one line
#define CREATE_DB_VALUE(name,type,def) \
        class name {\
        public: type defaultValue; \
        name() {defaultValue = def;} \
        const char *get() {return ""#name"";}\
}; \
DatabaseItem<type,name> name;

    /**
     * definition of all config values which are loaded out of the
     * database without any additional handling
     * (incl. default value if db has no apropriate value)
     */

    /**
     * This is only a demo creation of two different values:
     * if one wants to create additional values, the signature of this macro is:
     *
     * CREATE_DB_VALUE(dBValueName, realType, defaultValue)
     */
    CREATE_DB_VALUE(IntValue, int, 0); // only for demo // finished: 100%
    CREATE_DB_VALUE(FloatValue, float, 0); // only for demo // finished: 100%

    /**
     * definition of higher aggregated functions
     */
    float IntTimesFloat()   //((CustomControlSM &)LocalSPM::GetCustomControlRef()).ExpectSetString("on");
    {
        // accesses own data base values by local member value definition
        return IntValue() * FloatValue();
    }

    /**
     * example: returns the flag if customer wants autoplay
     *
     * this is the way how a data provider value my defined by the developer:
     *
     * <B>AutoPlayInPause</B> : name of database item, will be used as method name too
     *
     * <B>int</B>        : base type of database value for the item AutoPlayInPause
     *
     * <B>0</B>        : default value for item if no database value was given
     *
     * usage in user code:
   \code{.cpp}
     DataProvider dp;
     int autoPlay = dp.AutoPlayInPause()
   \endcode
     */

    CREATE_DB_VALUE(AutoPlayInLastModePause, int, 0 /**< default value */);
    CREATE_DB_VALUE(AutoPlayInPause, int, 1 /**< default value */);
    CREATE_DB_VALUE(PlayObjectFromActiveDevice, int, 0 /**< default value */);
    CREATE_DB_VALUE(PrevBufferTime, int, 5000 /**< default value */);
    CREATE_DB_VALUE(PrevBufferTimeAppleDevice, int, 2000 /**< default value */);
    CREATE_DB_VALUE(ContinuousCueing, int, 0 /**< default value */);
    CREATE_DB_VALUE(FFwdFRevSimulationRate, int, 10 /**< default value */); //cueing rate (0=off)
    CREATE_DB_VALUE(FFwdFRevTimeInterval, long, 1000); //time intervall in ms !Attention: Timer values <1000 will not work
    CREATE_DB_VALUE(StartupTimeMS, long, 30000);
    CREATE_DB_VALUE(SPMStartupTimeout, long, 20000); //20s for each step (->INIT, ->RUN) = 40s
    CREATE_DB_VALUE(SPMShutdownTimeout, long, 10000); //for each step (->STOP, ->DONE_SM, ->DONE)
    CREATE_DB_VALUE(SPMMaxShutdownDelay, long, 5000 /**< default value */);
    CREATE_DB_VALUE(MyMediaSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(MyMediaVisible, int, 1 /**< default value */);
    CREATE_DB_VALUE(DBMyMediaName, string, "MyMedia");
    //MaxNoOfDevicesInDB including internally created db device entries(MusicBox and MyMedia.db) + Indexable devices + Non-indexable devices
    //MaxNoOfDevicesInDB = 21, MaxNoOfIndexableDevicesInDB = 9, then "Nonindexable device count = 21 - 2(MusicBox and MyMedia.db) - 9 = 10"
    CREATE_DB_VALUE(MaxNoOfDevicesInDB, int, 21 /**< default value */); //including MusicBox and MyMedia.db
    CREATE_DB_VALUE(MaxNoOfIndexableDevicesInDB, int, 9 /**< default value */);  //musicbox is not included. Only including external indexable devices
    CREATE_DB_VALUE(GBKSupported, int, 0 /**< default value */);
    CREATE_DB_VALUE(EUC_KRSupported, int, 0 /**< default value */);
    CREATE_DB_VALUE(ARABICSupported, int, 0 /**< default value */);
    CREATE_DB_VALUE(FavoritesSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavGenreSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavArtistsSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavAlbumsSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavSongsSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavPlaylistsSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavPodcastsSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavAudiobooksSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavVideosSupported, int, 1 /**< default value */);
    CREATE_DB_VALUE(FavItemChangedUpdateLimit, int, 5 /**< default value */);
    //CREATE_DB_VALUE(DelCorruptMediaObjectFromDB, int, 0 /**< default value */); //Roadmap 13012 Obsolete: use FileErrorHandling instead
    CREATE_DB_VALUE(VideoLayerName, string, "0x14"); // valid for gen3armmake; not used for linuxx86make/gen3x86make
    CREATE_DB_VALUE(VideoLayerOrder, int, 5);
    CREATE_DB_VALUE(VideoLayerWidth, int, 800);
    CREATE_DB_VALUE(VideoLayerHeight, int, 480);
    CREATE_DB_VALUE(VideoIndexingOn, int, 1);
    CREATE_DB_VALUE(UsePictureManager, int, 0);
    CREATE_DB_VALUE(SlideShowTimeInterval, long, 3000); //time intervall in ms !Attention: Timer values <1000 will not work

    /* supported file types */
    CREATE_DB_VALUE(SupportedFileTypes, int, 39); // number of registered file types
    CREATE_DB_VALUE(SupportedFileTypeExtension_01, string, ".mp3");
    CREATE_DB_VALUE(SupportedFileTypeFormat_01, int, FFT_MP3);
    CREATE_DB_VALUE(SupportedFileTypeType_01, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_02, string, ".wma");
    CREATE_DB_VALUE(SupportedFileTypeFormat_02, int, FFT_WMA);
    CREATE_DB_VALUE(SupportedFileTypeType_02, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_03, string, ".aac");
    CREATE_DB_VALUE(SupportedFileTypeFormat_03, int, FFT_AAC);
    CREATE_DB_VALUE(SupportedFileTypeType_03, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_04, string, ".m4a");
    CREATE_DB_VALUE(SupportedFileTypeFormat_04, int, FFT_M4A);
    CREATE_DB_VALUE(SupportedFileTypeType_04, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_05, string, ".mp4");
    CREATE_DB_VALUE(SupportedFileTypeFormat_05, int, FFT_MP4);
    CREATE_DB_VALUE(SupportedFileTypeType_05, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_06, string, ".wav");
    CREATE_DB_VALUE(SupportedFileTypeFormat_06, int, FFT_WAV);
    CREATE_DB_VALUE(SupportedFileTypeType_06, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_07, string, ".ogg");
    CREATE_DB_VALUE(SupportedFileTypeFormat_07, int, FFT_OGG);
    CREATE_DB_VALUE(SupportedFileTypeType_07, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_08, string, ".3gp");
    CREATE_DB_VALUE(SupportedFileTypeFormat_08, int, FFT_3GP);
    CREATE_DB_VALUE(SupportedFileTypeType_08, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_09, string, ".3g2");
    CREATE_DB_VALUE(SupportedFileTypeFormat_09, int, FFT_3G2);
    CREATE_DB_VALUE(SupportedFileTypeType_09, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_10, string, ".aiff");
    CREATE_DB_VALUE(SupportedFileTypeFormat_10, int, FFT_AIFF);
    CREATE_DB_VALUE(SupportedFileTypeType_10, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_11, string, ".aif");
    CREATE_DB_VALUE(SupportedFileTypeFormat_11, int, FFT_AIFF);
    CREATE_DB_VALUE(SupportedFileTypeType_11, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_12, string, ".aa");
    CREATE_DB_VALUE(SupportedFileTypeFormat_12, int, FFT_AA);
    CREATE_DB_VALUE(SupportedFileTypeType_12, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_13, string, ".sbc");
    CREATE_DB_VALUE(SupportedFileTypeFormat_13, int, FFT_SBC);
    CREATE_DB_VALUE(SupportedFileTypeType_13, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_14, string, ".mpeg");
    CREATE_DB_VALUE(SupportedFileTypeFormat_14, int, FFT_MPEG);
    CREATE_DB_VALUE(SupportedFileTypeType_14, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_15, string, ".mpg");
    CREATE_DB_VALUE(SupportedFileTypeFormat_15, int, FFT_MPEG);
    CREATE_DB_VALUE(SupportedFileTypeType_15, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_16, string, ".avi");
    CREATE_DB_VALUE(SupportedFileTypeFormat_16, int, FFT_AVI);
    CREATE_DB_VALUE(SupportedFileTypeType_16, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_17, string, ".mov");
    CREATE_DB_VALUE(SupportedFileTypeFormat_17, int, FFT_MOV);
    CREATE_DB_VALUE(SupportedFileTypeType_17, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_18, string, ".divx");
    CREATE_DB_VALUE(SupportedFileTypeFormat_18, int, FFT_DIVX);
    CREATE_DB_VALUE(SupportedFileTypeType_18, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_19, string, ".wmv");
    CREATE_DB_VALUE(SupportedFileTypeFormat_19, int, FFT_WMV);
    CREATE_DB_VALUE(SupportedFileTypeType_19, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_20, string, ".m4v");
    CREATE_DB_VALUE(SupportedFileTypeFormat_20, int, FFT_M4V);
    CREATE_DB_VALUE(SupportedFileTypeType_20, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_21, string, ".flv");
    CREATE_DB_VALUE(SupportedFileTypeFormat_21, int, FFT_FLV);
    CREATE_DB_VALUE(SupportedFileTypeType_21, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_22, string, ".m3u");
    CREATE_DB_VALUE(SupportedFileTypeFormat_22, int, FFT_M3U);
    CREATE_DB_VALUE(SupportedFileTypeType_22, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_23, string, ".pls");
    CREATE_DB_VALUE(SupportedFileTypeFormat_23, int, FFT_PLS);
    CREATE_DB_VALUE(SupportedFileTypeType_23, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_24, string, ".xspf");
    CREATE_DB_VALUE(SupportedFileTypeFormat_24, int, FFT_XSPF);
    CREATE_DB_VALUE(SupportedFileTypeType_24, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_25, string, ".wpl");
    CREATE_DB_VALUE(SupportedFileTypeFormat_25, int, FFT_WPL);
    CREATE_DB_VALUE(SupportedFileTypeType_25, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_26, string, ".b4s");
    CREATE_DB_VALUE(SupportedFileTypeFormat_26, int, FFT_B4S);
    CREATE_DB_VALUE(SupportedFileTypeType_26, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_27, string, ".asx");
    CREATE_DB_VALUE(SupportedFileTypeFormat_27, int, FFT_ASX);
    CREATE_DB_VALUE(SupportedFileTypeType_27, int, FT_PLAYLIST);
    CREATE_DB_VALUE(SupportedFileTypeExtension_28, string, ".asf");
    CREATE_DB_VALUE(SupportedFileTypeFormat_28, int, FFT_WMA);
    CREATE_DB_VALUE(SupportedFileTypeType_28, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_29, string, ".m4b");
    CREATE_DB_VALUE(SupportedFileTypeFormat_29, int, FFT_M4B);
    CREATE_DB_VALUE(SupportedFileTypeType_29, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_30, string, ".m4p");
    CREATE_DB_VALUE(SupportedFileTypeFormat_30, int, FFT_M4P);
    CREATE_DB_VALUE(SupportedFileTypeType_30, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_31, string, ".m4r");
    CREATE_DB_VALUE(SupportedFileTypeFormat_31, int, FFT_M4R);
    CREATE_DB_VALUE(SupportedFileTypeType_31, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_32, string, ".flac");
    CREATE_DB_VALUE(SupportedFileTypeFormat_32, int, FFT_FLAC);
    CREATE_DB_VALUE(SupportedFileTypeType_32, int, FT_AUDIO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_33, string, ".dat");
    CREATE_DB_VALUE(SupportedFileTypeFormat_33, int, FFT_MPEG);
    CREATE_DB_VALUE(SupportedFileTypeType_33, int, FT_VIDEO);
    CREATE_DB_VALUE(SupportedFileTypeExtension_40, string, ".mkv");
    CREATE_DB_VALUE(SupportedFileTypeFormat_40, int, FFT_MKV);
    CREATE_DB_VALUE(SupportedFileTypeType_40, int, FT_VIDEO);

    /* image entries */
    CREATE_DB_VALUE(SupportedFileTypeExtension_34, string, ".gif");
    CREATE_DB_VALUE(SupportedFileTypeFormat_34, int, FFT_GIF);
    CREATE_DB_VALUE(SupportedFileTypeType_34, int, FT_IMAGE);
    CREATE_DB_VALUE(SupportedFileTypeExtension_35, string, ".bmp");
    CREATE_DB_VALUE(SupportedFileTypeFormat_35, int, FFT_BMP);
    CREATE_DB_VALUE(SupportedFileTypeType_35, int, FT_IMAGE);
    CREATE_DB_VALUE(SupportedFileTypeExtension_36, string, ".png");
    CREATE_DB_VALUE(SupportedFileTypeFormat_36, int, FFT_PNG);
    CREATE_DB_VALUE(SupportedFileTypeType_36, int, FT_IMAGE);
    CREATE_DB_VALUE(SupportedFileTypeExtension_37, string, ".tif");
    CREATE_DB_VALUE(SupportedFileTypeFormat_37, int, FFT_TIF);
    CREATE_DB_VALUE(SupportedFileTypeType_37, int, FT_IMAGE);
    CREATE_DB_VALUE(SupportedFileTypeExtension_38, string, ".jpg");
    CREATE_DB_VALUE(SupportedFileTypeFormat_38, int, FFT_JPEG);
    CREATE_DB_VALUE(SupportedFileTypeType_38, int, FT_IMAGE);
    CREATE_DB_VALUE(SupportedFileTypeExtension_39, string, ".jpeg");
    CREATE_DB_VALUE(SupportedFileTypeFormat_39, int, FFT_JPEG);
    CREATE_DB_VALUE(SupportedFileTypeType_39, int, FT_IMAGE);

    CREATE_DB_VALUE(SupportedFileTypeExtension_41, string, "");
    CREATE_DB_VALUE(SupportedFileTypeFormat_41, int, FFT_UNKNOWN);
    CREATE_DB_VALUE(SupportedFileTypeType_41, int, FT_UNKNOWN);

    /* default folder filter regular expressions*/
    CREATE_DB_VALUE(FileFilterCount, int, 5);
    CREATE_DB_VALUE(FileFilter_01, string, "^\\.$");
    CREATE_DB_VALUE(FileFilter_02, string, "^\\..$");
    CREATE_DB_VALUE(FileFilter_03, string, "recycle");
    CREATE_DB_VALUE(FileFilter_04, string, "System Volume Information");
    CREATE_DB_VALUE(FileFilter_05, string, "lost+found");
    CREATE_DB_VALUE(FileFilter_06, string, ".*\\.\\bmp3\\b$");
    CREATE_DB_VALUE(FileFilter_07, string, ".*\\.\\cmp3\\b$");


    CREATE_DB_VALUE(AutoplayDeviceTypes, int, 0);           /* number of registered device types for autoplay */
    CREATE_DB_VALUE(AutoplayDeviceType_1, int, DTY_USB);
    CREATE_DB_VALUE(AutoplayDeviceType_2, int, DTY_IPOD);
    CREATE_DB_VALUE(AutoplayDeviceType_3, int, DTY_SD);
    CREATE_DB_VALUE(AutoplayDeviceType_4, int, DTY_IPHONE);
    CREATE_DB_VALUE(AutoplayDeviceType_5, int, DTY_MTP);
    CREATE_DB_VALUE(AutoplayDeviceType_6, int, DTY_CDROM);
    CREATE_DB_VALUE(AutoplayDeviceType_7, int, DTY_BLUETOOTH);
    CREATE_DB_VALUE(AutoplayDeviceType_8, int, DTY_FLASH); /* attention: if you add here a line, take care about the method: DataProvider::GetAutoplayDeviceTypes */

    /* supported device types Roadmap 13023*/
    CREATE_DB_VALUE(SupportedDeviceTypes, int, 10);              /* number of registered supported device types */
    CREATE_DB_VALUE(SupportedDeviceType_01, int, DTY_USB);
    CREATE_DB_VALUE(SupportedDeviceType_02, int, DTY_SD);
    CREATE_DB_VALUE(SupportedDeviceType_03, int, DTY_IPOD);
    CREATE_DB_VALUE(SupportedDeviceType_04, int, DTY_IPHONE);
    CREATE_DB_VALUE(SupportedDeviceType_05, int, DTY_MTP);
    CREATE_DB_VALUE(SupportedDeviceType_06, int, DTY_CDROM);
    CREATE_DB_VALUE(SupportedDeviceType_07, int, DTY_BLUETOOTH);
    CREATE_DB_VALUE(SupportedDeviceType_08, int, DTY_FLASH);
    CREATE_DB_VALUE(SupportedDeviceType_09, int, DTY_DVD_DRIVE);/* attention: if you add here a line, take care about the method: DataProvider::IsDeviceTypeSupported */
    CREATE_DB_VALUE(SupportedDeviceType_10, int, DTY_MUSICBOX);

    /* supported file types for indexing */
    CREATE_DB_VALUE(IndexingSupportDeviceTypes,  int, 7);           /* number of registered device types which support indexing */
    CREATE_DB_VALUE(IndexingSupportDeviceType_1, int, DTY_IPHONE);     /* this device type has the highest priority */
    CREATE_DB_VALUE(IndexingSupportDeviceType_2, int, DTY_IPOD);      /* this device type has the 2nd highest priority */
    CREATE_DB_VALUE(IndexingSupportDeviceType_3, int, DTY_USB);    /* this device type has the 3rd highest priority */
    CREATE_DB_VALUE(IndexingSupportDeviceType_4, int, DTY_SD);
    CREATE_DB_VALUE(IndexingSupportDeviceType_5, int, DTY_MTP);
    CREATE_DB_VALUE(IndexingSupportDeviceType_6, int, DTY_CDROM);
    CREATE_DB_VALUE(IndexingSupportDeviceType_7, int, DTY_FLASH);
    CREATE_DB_VALUE(IndexingSupportDeviceType_8, int, DTY_BLUETOOTH);/* attention: if you add here a line, take care about the method: DataProvider::GetIndexingPriority*/

    /* AlbumArt Not supported DeviceTypes*/
    CREATE_DB_VALUE(AlbumArtUnSupportedDeviceTypes,  int, 3);
    CREATE_DB_VALUE(AlbumArtUnSupportedDeviceType_1, int, DTY_BLUETOOTH);
    CREATE_DB_VALUE(AlbumArtUnSupportedDeviceType_2, int, DTY_CDROM);
    CREATE_DB_VALUE(AlbumArtUnSupportedDeviceType_3, int, DTY_CDDA);

    /* supported file systems Roadmap 13023 */
    CREATE_DB_VALUE(SupportedFileSystems, int, 7);              /* number of registered file systems */
    CREATE_DB_VALUE(SupportedFileSystem_01, int, FSTY_FAT);
    CREATE_DB_VALUE(SupportedFileSystem_02, int, FSTY_NTFS);
    CREATE_DB_VALUE(SupportedFileSystem_03, int, FSTY_EXT4);
    CREATE_DB_VALUE(SupportedFileSystem_04, int, FSTY_HFS);
    CREATE_DB_VALUE(SupportedFileSystem_05, int, FSTY_EXFAT);
    CREATE_DB_VALUE(SupportedFileSystem_06, int, FSTY_ISO9660);
    CREATE_DB_VALUE(SupportedFileSystem_07, int, FSTY_UDF);
    CREATE_DB_VALUE(SupportedFileSystem_08, int, FSTY_UNKNOWN);

    /* supported device partition numbers Roadmap 13023 */
    CREATE_DB_VALUE(SupportedPartionNumbers, int, 5); /* number of registered partition numbers */
    CREATE_DB_VALUE(MultiPartitionSupported, int, 0) /*MultiPartitionSupport*/

    /* device types for permanent streaming Roadmap 13008 */
    CREATE_DB_VALUE(PermanentStreamingDeviceTypes, int, 1);           /* number of registered device types for permanent streaming */
    CREATE_DB_VALUE(PermanentStreamingDeviceType_1, int, DTY_BLUETOOTH);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_2, int, DTY_IPOD);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_3, int, DTY_IPHONE);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_4, int, DTY_DVD_DRIVE);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_5, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_6, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_7, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(PermanentStreamingDeviceType_8, int, DTY_UNKNOWN); /* attention: if you add here a line, take care about the method: DataProvider::GetPermanentStreamingDeviceTypes */

    /* application table Roadmap 13008 */
    CREATE_DB_VALUE(ApplicationNumbers, int, 8);   /* number of registered applications */
    //Pandora iAP1 and iAP2 iOS8
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_01, int, DTY_IPOD);
    CREATE_DB_VALUE(ApplicationNameNumber_01, string, "com.pandora");
    CREATE_DB_VALUE(ApplicationSourceNumber_01, int, 0xB5); //FBLOCK_INTERNETRADIO
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_02, int, DTY_IPHONE);
    CREATE_DB_VALUE(ApplicationNameNumber_02, string, "com.pandora");
    CREATE_DB_VALUE(ApplicationSourceNumber_02, int, 0xB5); //FBLOCK_INTERNETRADIO
    //Pandora iAP2 iOS7
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_03, int, DTY_IPOD);
    CREATE_DB_VALUE(ApplicationNameNumber_03, string, "Pandora");
    CREATE_DB_VALUE(ApplicationSourceNumber_03, int, 0xB5); //FBLOCK_INTERNETRADIO
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_04, int, DTY_IPHONE);
    CREATE_DB_VALUE(ApplicationNameNumber_04, string, "Pandora");
    CREATE_DB_VALUE(ApplicationSourceNumber_04, int, 0xB5); //FBLOCK_INTERNETRADIO
    //MusicApp iAP1 and iAP2 iOS8
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_05, int, DTY_IPOD);
    CREATE_DB_VALUE(ApplicationNameNumber_05, string, "com.apple.mobileipod");
    CREATE_DB_VALUE(ApplicationSourceNumber_05, int, 0xAB); //FBLOCK_MEDIAPLAYER
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_06, int, DTY_IPHONE);
    CREATE_DB_VALUE(ApplicationNameNumber_06, string, "com.apple.mobileipod");
    CREATE_DB_VALUE(ApplicationSourceNumber_06, int, 0xAB); //FBLOCK_MEDIAPLAYER
    //default source to mediaplayer
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_07, int, DTY_IPOD);
    CREATE_DB_VALUE(ApplicationNameNumber_07, string, "");
    CREATE_DB_VALUE(ApplicationSourceNumber_07, int, 0xAB); //FBLOCK_MEDIAPLAYER
    CREATE_DB_VALUE(ApplicationDeviceTypeNumber_08, int, DTY_IPHONE);
    CREATE_DB_VALUE(ApplicationNameNumber_08, string, "");
    CREATE_DB_VALUE(ApplicationSourceNumber_08, int, 0xAB); //FBLOCK_MEDIAPLAYER


    /* internal devices Roadmap 13005 */
    CREATE_DB_VALUE(InternalDevices, int, 0);                           /* number of registered internal devices */
    CREATE_DB_VALUE(InternalDeviceMountPoint_1, string, "/opt/bosch/test/data/GMP/test");
    CREATE_DB_VALUE(InternalDeviceName_1, string, "MyCarDevice_1");
    CREATE_DB_VALUE(InternalDeviceMountPoint_2, string, "/opt/bosch/test/data/GMP/stick_2");
    CREATE_DB_VALUE(InternalDeviceName_2, string, "MyCarDevice_2");     /* attention: if you add here a line, take care about the method: DataProvider::GetInternalDevices */

    CREATE_DB_VALUE(IndexingOrder, int, 0 /**< default value */);
    CREATE_DB_VALUE(IndexingDeviceTypePrio, int, 1 /**< default value */);
    CREATE_DB_VALUE(IndexingNotYetIndexedPrio, int, 1 /**< default value */);
    CREATE_DB_VALUE(IndexingFingerprintMaxMemory, int, 31249984 /**< default value */); //32GB
    CREATE_DB_VALUE(LimitNumberObjectsInDB, int, 100000 /**< default value */);
    CREATE_DB_VALUE(LimitNumberObjectsPerDevice, int, 100000 /**< default value */);
    CREATE_DB_VALUE(ScanMethod, int, 0 /**< default value */);
    CREATE_DB_VALUE(DBStoreMediaObjectMaxTimeMS, int, 2500);
    CREATE_DB_VALUE(DBStoreMediaObjectMaxCount, int, 300 /**< default value */);
    CREATE_DB_VALUE(DBDeleteMyMediaObjectMaxTimeMS, int, 1000 /**< default value */);
    CREATE_DB_VALUE(DBDeleteMyMediaObjectMaxCount, int, 4000 /**< default value */);
    CREATE_DB_VALUE(MaxMetadataStringLen, int, 90 /**< default value */);
    CREATE_DB_VALUE(DBUnknownText, string, "UNKNOWN"); /* attention: if string is not UNKNOWN the schema has to be attapted or a separate HMI view value has to be established */
    CREATE_DB_VALUE(NowPlayingUnknownText, string, "");
    CREATE_DB_VALUE(DBUpdateProgressInstructions, int, 100000 /**< default value */);
    CREATE_DB_VALUE(DBUpdateMyMediaPollingTimeUS, int, 150000 /**< default value */);
    CREATE_DB_VALUE(RepeatModeAdjustable, int, 1 /**< default value */);
    CREATE_DB_VALUE(DBDefaultRepeatMode, int, RPT_LIST /**< default value */);
    CREATE_DB_VALUE(DBDefaultVideoRepeatMode, int, RPT_NONE /**< default value */);

    /* Nowplaying FileList content of MSD,MTP Devices is tightly coupled wtih the current RepeatMode
     * So Nowplaying FileList content may be altered upon change in RepeatMode or user selects an item to play from FileBrowsing
     * Below configuration denotes user pereferred Nowplaying FileList content in RPT_ONE or RPT_OFF mode - NCG3D-132116 */
    CREATE_DB_VALUE(PreferredNewListContent, int, RPT_INVALID /**< default value */);

    CREATE_DB_VALUE(UseDefaultRepeatModeForWrapList, int, 0/**< default value */);
    CREATE_DB_VALUE(IgnoreRepeatModeOneByUserNext, int, 0/**< default value */);
    CREATE_DB_VALUE(CreateFollowingList, int, 0); //CRQ: CMG3G-8071
    CREATE_DB_VALUE(SetDeviceRepeatModeNonStreaming, int, 0 /**< default value */);
    CREATE_DB_VALUE(SetDevicePlaybackModeNonStreaming, int, 0 /**< default value */);
    CREATE_DB_VALUE(SaveLastModeInNotActive, int, 0/**< default value */);
    CREATE_DB_VALUE(DBDefaultLanguage, int, LNG_NA_ENGLISH /**< default value */);
    CREATE_DB_VALUE(DBDefaultMediaContext, int, MC_ALL /**< default value */);
    CREATE_DB_VALUE(DBFlushMediaObjectsAsThread, int, 1 /**< default value */);
    CREATE_DB_VALUE(DBRemoveExtensionFromFilename, int, 1 /**< default value */);
    CREATE_DB_VALUE(OnlyMediaObjectsInFileLists, int, 0); //CRQ: CMG3G-8348
    CREATE_DB_VALUE(PlayParentFolderInsteadOfGivenFolder, int, 0/**< default value */); //!!Workaround for scope 2.1 Renault.Replaced with clean soln in Scope 3.0 and higher!!//

    CREATE_DB_VALUE(DBVTFileMaxCacheSize, int, 50000 /**< default value */);
    CREATE_DB_VALUE(DBVTFileMaxDepthForScan, int, 11 /**< default value */);
    CREATE_DB_VALUE(DBVTFileMaxDepthForPlaylistRecurse, int, 0 /**< default value */);
    CREATE_DB_VALUE(DBVTFileShowAllPlaylistEntries, int, 1 /**< default value */);
    CREATE_DB_VALUE(InternalPlaylistStoragePath, string, "//tmp" /**< default value */); //Roadmap:13022: Internal playlist storage path



    CREATE_DB_VALUE(DeliverListSizeInPlayObjectToHMI, int, 0 /**< default value */);
    CREATE_DB_VALUE(DeliverPositionInNowPlayingToHMI, int, 1 /**< default value */);
    CREATE_DB_VALUE(DBUnifiedSchema, int, 1 /**< default value */);
    CREATE_DB_VALUE(CheckListTimerMS, long, 10000);
    CREATE_DB_VALUE(ShuffleSkippingStackLength, int, 0 /**< default value */); //Roadmap 13016
    CREATE_DB_VALUE(ShuffleReshuffleOnListWrap, int, 0 /**< default value */);

    CREATE_DB_VALUE(FullTextSearch, int, 0 /**< default value */);
    CREATE_DB_VALUE(FileErrorHandling, int, 0 /**< default value */); //Roadmap 13012
    CREATE_DB_VALUE(DBFileBrowsingByDB, int, 0 /**< default value */); //Roadmap 13012
    CREATE_DB_VALUE(DBFileListWhileIndexing, int, 0 /**< default value */);
    CREATE_DB_VALUE(CollectMetadata, int, 1 /**< default value */); //Roadmap 13012
    CREATE_DB_VALUE(ClearLastMode, int, 0 /**< default value */); //Roadmap 13023
    CREATE_DB_VALUE(ClearPlaybackMode, int, 1 /**< default value */);
    CREATE_DB_VALUE(ClearRepeatMode, int, 1 /**< default value */);
    CREATE_DB_VALUE(UnsupportedDeviceTypeVisible, int, 0 /**< default value */); //Roadmap 13023
    CREATE_DB_VALUE(UnsupportedFileSystemVisible, int, 0 /**< default value */); //Roadmap 13023
    CREATE_DB_VALUE(UnsupportedPartitionVisible, int, 0 /**< default value */); //Roadmap 13023
    CREATE_DB_VALUE(GaplessPlay, int, 0 /**< default value */); //Roadmap 13010
    CREATE_DB_VALUE(MicroMutes, int, 0 /**< default value */); //Roadmap 13018
    CREATE_DB_VALUE(FFVolumeReduction, int, 0 /**< default value */); //Roadmap 13018
    CREATE_DB_VALUE(FFVolumeReductionDB, int, -6 /**< default value */); //CRQ CMG3G-6576
    CREATE_DB_VALUE(CueingRate, int, 10 /**< default value */); // normal speed;

    /* Content Sharing options */
    CREATE_DB_VALUE(ContentSharingEnabled, int, 0 /**< default value */);      /**< General support for Content Sharing (CS) */

    /* iPodControl */
    CREATE_DB_VALUE(iPodControlSupportIAP1, int, 1 /**< default value */);        /**< General support for IAP1 protocol */
    CREATE_DB_VALUE(iPodControlSupportIAP2, int, 1 /**< default value */);        /**< General support for IAP2 protocol */

    /* iAP2ExternalAccessoryProtocolCarPlay */
    CREATE_DB_VALUE(iAP2ExternalAccessoryProtocolCarPlayEnabled, int, 0 /**< default Value */);      /**< Support for iAP2ExternalAccessoryProtocolCarPlay */

    /* iPodControl AlbumArt Wait Timer */
    CREATE_DB_VALUE(iPodControlAlbumArtWaitTimer, int, 0 /**< defaultValue */);      /**< AlbumArtWaitTimer for IAP2 */

    // IAP via BT
    CREATE_DB_VALUE(iPodControlSupportIAP1BT, int, 0 /**< default value */);      /**< General support for IAP1 protocol via Bluetooth */
    CREATE_DB_VALUE(iPodControlSupportIAP2BT, int, 0 /**< default value */);      /**< General support for IAP2 protocol via Bluetooth */
    CREATE_DB_VALUE(IAPBTConnectionOnlyForSmartApp, int, 0 /**< default value */);
    CREATE_DB_VALUE(iPodControlPreferredConnectionType, int, DCT_USB /**< default value */);      /**< DCT is preferred connection type */
    //CREATE_DB_VALUE(iPodControlBTMACAddressFile, string, "/var/opt/bosch/dynamic/connectivity/bt_module/bdaddr.txt" /**< default value */); /**< file to read BT MAC address */
    CREATE_DB_VALUE(iPodControlBTMACAddress, string, "" /**< default value */);   /**< BT MAC address */
    CREATE_DB_VALUE(iPodControlBTIAP2SPPUUID, string, "00000000decafadedecadeafdecacafe" /**< default value */);   /**< IAP2 SPP UUID */
    CREATE_DB_VALUE(RequestIAP2SPPConnection,int,0  /**< default value */);
    CREATE_DB_VALUE(IgnoreBTfallbackFromIAP,int,0  /**< default value */);

    // CarPlay (DiPO)
    CREATE_DB_VALUE(iPodControlSupportIAP2HostModeConnectionsMax, int, 1 /**< default value */);      /**< max count Hostmode support for IAP2 protocol */
    CREATE_DB_VALUE(iPodControlSupportIAP2iOSInTheCar, int, 1 /**< default value */);   /**< iOSInTheCar support for IAP2 protocol */
    CREATE_DB_VALUE(iPodControlIAP2HostModeErrorResetMS, int, 10000 /**< default value */);                   /**< hostmode error reset timeout */
    CREATE_DB_VALUE(iPodControlIAP2CheckRoleSwitchRequired, int, 0 /**< default value */);              /**< enable carplay device detection speed up, asking SPI if role switch is required to directly go for carplay */
    CREATE_DB_VALUE(iPodControlIAP2CheckRoleSwitchRequiredTimeOutMS, int, 5000 /**< default value */);    /**< timeout for checking SPI */
    CREATE_DB_VALUE(iPodControlIAP2CarPlayModeEnabled, int, 0 /**< default value */);   /* enable support for browsing & playback during CarPlay */
    CREATE_DB_VALUE(iPodControlIAP2CarPlayModeUserActionOnly, int, 1 /**< default value */);   /* send PLAY/PAUSE to iPhone in CarPlay mode only in case of an user action */
    CREATE_DB_VALUE(iPodControlIAP2CarPlayModeUserActionTimeoutMS, int, 3000 /**< default value */);   /**< time window in milliseconds for suppressing PLAY/PAUSE to iPhone in CarPlay mode */
    CREATE_DB_VALUE(GetAudioDeviceTimeout, long, 8000);  /**< !Attention: Timer values <1000 will not work*/
    CREATE_DB_VALUE(SourceActivityDelayEnabled, int, 0 /**< default value */);   /* enable delay SA_ON because of possible wrong source allocation due to a apple bug */
    CREATE_DB_VALUE(AppleBugWrongSourceTimeout, long, 3000);  /**< !Attention: Timer values <1000 will not work*/
    CREATE_DB_VALUE(DiPOStopReason, int, tDiPOStopReason_init);  /**< reason for stopping carplay playback, see topic "HID play" and Carplay TAKE/BORROW certification issue*/
    CREATE_DB_VALUE(DiPOStartReason, int, tDiPOStartReason_init);   //GMMY17-16339
    CREATE_DB_VALUE(iPodControlIAP2CarPlayViaUnwiredHubEnabled, int, 0 /**< default value */);   /* enable support for Unwired USB HUB for CarPlay*/
    CREATE_DB_VALUE(iPodControlIAP2CarPlayViaMicrochipHubEnabled, int, 0 /**< default value */);   /* enable support for Microchip USB HUB for CarPlay - be careful enabling this plus iPodControlIAP2CarPlayViaMitsumi2PortES2HubEnabled and iPodControlIAP2CarPlayViaMitsumi2PortES4HubEnabled*/
    CREATE_DB_VALUE(iPodControlIAP2CarPlayViaMitsumi2PortES2HubEnabled, int, 0 /**< default value */);   /* enable support for Mitsumi 2 Port ES2" Hub*/
    CREATE_DB_VALUE(iPodControlIAP2CarMitsumi2PortES2bcdDevice_A, string, "0128" /**< default value */);   /* value to rdifferentiate it from .e.g MicrochipHub with same productID and vendorID*/
    CREATE_DB_VALUE(iPodControlIAP2CarPlayViaMitsumi2PortES4HubEnabled, int, 0 /**< default value */);   /* enable support for Mitsumi 2 Port ES4" Hub*/
    CREATE_DB_VALUE(iPodControlIAP2CarMitsumi2PortES2And4Bridge_ProductID, int, 0x2530 /**< default value */);   /* value to rdifferentiate it from .e.g MicrochipHub with same productID and vendorID*/
    CREATE_DB_VALUE(iPodControlIAP2CarMitsumi2PortES2And4Bridge_VendorID,  int, 0x0424 /**< default value */);   /* value to rdifferentiate it from .e.g MicrochipHub with same productID and vendorID*/
    CREATE_DB_VALUE(iPodControlIAP2CarPlayViaMolexHubEnabled, int, 1 /**< default value */);   /* enable support for Molex USB HUB for CarPlay*/


    CREATE_DB_VALUE(iPodControlIAP2CarPlayIcon, string, "/var/opt/bosch/static/mediaplayer/ipod/CarPlay.png");   /**< CarPlay icon */
    CREATE_DB_VALUE(iPodControlIAP2CarPlayAlbumArtEnabled, int, 0);   /**< Apple Carplay logo as album art */
    CREATE_DB_VALUE(iPodControlIAP2ListUpdateEnabled, int, 0);

    CREATE_DB_VALUE(iPodControlSupportCarlife, int, 0 /**< default value */);   /**< Carlife app support */
    CREATE_DB_VALUE(iPodControlSupportMySpin, int, 0 /**< default value */);   /**< Myspin app support */

    //Location Info
    CREATE_DB_VALUE(iPodControlLocationInfoGPGGA, int, 1 /**< default value */);    /**< enable GPGGA info */
    CREATE_DB_VALUE(iPodControlLocationInfoGPRMC, int, 1 /**< default value */);    /**< enable GPRMC info */
    CREATE_DB_VALUE(iPodControlLocationInfoGPGSV, int, 0 /**< default value */);    /**< enable GPGSV info */
    CREATE_DB_VALUE(iPodControlLocationInfoGPHDT, int, 0 /**< default value */);    /**< enable GPHDT info */
    CREATE_DB_VALUE(iPodControlLocationInfoPASCD, int, 0 /**< default value */);    /**< enable PASCD info */
    CREATE_DB_VALUE(iPodControlLocationInfoPAGCD, int, 0 /**< default value */);    /**< enable PAGCD info */
    CREATE_DB_VALUE(iPodControlLocationInfoPAACD, int, 0 /**< default value */);    /**< enable PAACD info */

    //Vehicle Information
    CREATE_DB_VALUE(iPodControlVehicleStatusEnabled, int, 0 /**< default value */);    /**< enable general VehicleStatus info */
    CREATE_DB_VALUE(iPodControlVehicleStatusRange, int, 0 /**< default value */);    /**< enable Range info */
    CREATE_DB_VALUE(iPodControlVehicleStatusRangeWarning, int, 0 /**< default value */);    /**< enable RangeWarning info */
    CREATE_DB_VALUE(iPodControlVehicleStatusOutsideTemperature, int, 0 /**< default value */);    /**< enable OutsideTemperature info */

    CREATE_DB_VALUE(iPodControlGPRMCDataStatusValuesNotification, int, 0 /**< default value */);    /**< enable GPRMC DataStatus Values Notification */

    //iPod indexing
    CREATE_DB_VALUE(iPodControlIAP2DeltaIndexing, int, 1 /**< default value */);    /**< enable delta indexing */
    CREATE_DB_VALUE(iPodControlIAP2DeltaIndexingSyncCount, int, 10 /**< default value */);    /**< resync deta indexing IAP2, ignore last revision to get a fresh update from device*/
    CREATE_DB_VALUE(iPodControlIAP2DynamicReindexing, int, 1 /**< default value */);    /**< enable dynamic reindexing on spontaneous IAP2 MediaLibrary updates*/
    CREATE_DB_VALUE(iPodControlIAP2IndexingCacheLimit, int, 1000 /**< default value */);   /**< cache limit IAP2 MediaLibrary updates*/
    CREATE_DB_VALUE(iPodControlIAP2IndexingCacheFloor, int, 100 /**< default value */);    /**< cache floor IAP2 MediaLibrary updates*/
    CREATE_DB_VALUE(iPodControlIAP2IndexingIgnoreDeletedItemsAfterMediaLibraryReset, int, 1 /**< default value */);    /**< Deleted MediaLibraryItems will be ignored */
    CREATE_DB_VALUE(iPodControlUpdateShuffleAndRepeatFromDevice, int, 1);    /**< apply device setting shuffle and repeat from device */
    CREATE_DB_VALUE(iPodControlDisplayImage, string, "");   /**< upload display image to iPod screen */

    CREATE_DB_VALUE(iPodControlAccessoryInfoBTName, string, "vehicle");         /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoWIFIName, string, "vehicle");         /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoDisplayName, string, "");    /**< CarPlay display name NCG3D-9931 */

    CREATE_DB_VALUE(iPodControlAccessoryInfoModelName, string, "vehicle");        /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoModelNumber, string, "1.0");        /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSerialNumber, string, "00001");        /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoManufacturer, string, "BOSCH");        /**< iPod Accessory Info field */
    CREATE_DB_VALUE(iPodControlAccessoryInfoFWVersion, string, "1.0.0");        /**< iPod Accessory Info field firmware version */
    CREATE_DB_VALUE(iPodControlAccessoryInfoHWVersion, string, "1.0.0");        /**< iPod Accessory Info field hardware version */
    CREATE_DB_VALUE(iPodControlAccessoryInfoCurrentLanguage, string, "en");        /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguageCount, int, 1);    /**< iPod Accessory Info field language, see http://www.loc.gov/standards/iso639-2/php/English_list.php */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_1, string, "en");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_2, string, "ar");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_3, string, "bg");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_4, string, "zh");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_5, string, "hr");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_6, string, "cs");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_7, string, "da");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_8, string, "nl");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_9, string, "sq");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_10, string, "et");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_11, string, "fi");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_12, string, "fr");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_13, string, "de");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_14, string, "el");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_15, string, "he");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_16, string, "hi");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_17, string, "hu");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_18, string, "id");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_19, string, "it");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_20, string, "ja");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_21, string, "ko");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_22, string, "lv");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_23, string, "lt");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_24, string, "no");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_25, string, "fa");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_26, string, "pl");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_27, string, "pt");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_28, string, "ro");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_29, string, "ru");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_30, string, "sr");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_31, string, "sk");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_32, string, "sl");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_33, string, "es");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_34, string, "sv");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_35, string, "th");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_36, string, "tr");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_37, string, "uk");    /**< iPod Accessory Info field language */
    CREATE_DB_VALUE(iPodControlAccessoryInfoSupportedLanguage_38, string, "vi");    /**< iPod Accessory Info field language  */

    CREATE_DB_VALUE(iPodRemoteEventHandleInactiveDevice, int, 1 /**< default value */); //Roadmap 13008 Switch to streaming mode if a remote activity in a not active device is recognized
    CREATE_DB_VALUE(iPodRemoteEventEnterStreamingOnPlaybackMode, int, 1 /**< default value */); //Roadmap 13008 Trigger iPod remote activity if user changes playback (shuffle) mode on apple device
    CREATE_DB_VALUE(iPodRemoteEventEnterStreamingOnRepeatMode, int, 1 /**< default value */); //Roadmap 13008 Trigger iPod remote activity if user changes repeat mode on apple device
    CREATE_DB_VALUE(iPodControlChunkMax, int, 200 /**< default value */);        /**< max chunk value for iPod metadata list slice retrievals */
    CREATE_DB_VALUE(iPodControlVTChunkMax, int, 25 /**< default value */);        /**< max chunk value for VT iPod metadata list slice retrievals */
    CREATE_DB_VALUE(iPodControlChunkTimeMS, int, 1000 /**< default value */);    /**< time window in milliseconds for max delay on iPod metadata list slice retrieval */
    CREATE_DB_VALUE(iPodControlPBNotificationTimeoutMS, int, 5000 /**< default value */);    /**< time window in milliseconds for expected notifications from iPod after IAP call */
    CREATE_DB_VALUE(iPodControlTrackEndDetectionMS, int, 550 /**< default value */);        /**< interval for track end detection based on position updates */
    CREATE_DB_VALUE(iPodControlResumePositionVarianceMS, int, 2000 /**< default value */);        /**< resume position variance when continuing last track */
    CREATE_DB_VALUE(iPodControlPrevBufferTimeMS, int, 4000 /**< default value */);    /**< time interval that decides between PREV or SEEK to start of track within the first seconds */
    CREATE_DB_VALUE(iPodControlVTCacheLimit, int, 1000 /**< default value */);  /**< max number of cache entries per device */

    //values for suspending iOS communication, due to e.g. ATS requirements, HID commands
    CREATE_DB_VALUE(iPodControlATSDelayMS, int, 1100 /**< default value */);    /**< delay processing of iap call (specified by elapsedtime) according to apple ATS requirements */
    CREATE_DB_VALUE(iPodControlHIDSeekLeadMS, int, 1000 /**< default value */);    /**< time window in milliseconds for HID seek commands - ensures that iPod interprets a fast forward command and NOT just a NEXT */
    CREATE_DB_VALUE(iPodControlChapterSelectMS, int, 250 /**< default value */);   /**< time window in milliseconds for selection of audiobook chapter after calling SelecDBRecord */
    // Changed iPodControlIAP2InitCallbacksMS from 2000 to 5000 as a fix for issue NCG3D-11496
    CREATE_DB_VALUE(iPodControlIAP2InitCallbacksMS, int, 5000 /**< default value */);    /**< time window in milliseconds for IAP2 callbacks after initialization of device connection */

    CREATE_DB_VALUE(iPodControlIAP2AlbumArtMaxBytes, int, 0x100000 /**< default value */);    /**< max data size for JPEG albumArt buffer in bytes */
    CREATE_DB_VALUE(iPodControlIAP2QueueListMaxBytes, int, 0x100000 /**< default value */);  /**< max data size for queue list buffer in bytes */
    CREATE_DB_VALUE(iPodControlIAP2vCardMaxBytes, int, 0x100000 /**< default value */);  /**< max data size for vCard buffer in bytes */

    //iPod IAP2 accessory config values
    CREATE_DB_VALUE(iPodControlIAP2AuthDevicename, string, "/dev/i2c-2" /**< default value */);    /**< Authentication co-processor device path. */
    CREATE_DB_VALUE(iPodControlIAP2AuthIoctlRegAddr, string, "16" /**< default value */);        /**< Authentication co-processor I/O control register address. */
    CREATE_DB_VALUE(iPodControlIAP2AuthGPIOReset, string, "10" /**< default value */);            /**< GPIO Reset pin. */
    CREATE_DB_VALUE(iPodControlIAP2AuthGPIOReady, string, "-1" /**< default value */);            /**< GPIO Ready pin. */
    CREATE_DB_VALUE(iPodControlIAP2AuthShortWait, int, 1 /**< default value */);                /**< Time to wait after Authentication co-processor operation succeeds. */
    CREATE_DB_VALUE(iPodControlIAP2AuthWait, int, 10 /**< default value */);                    /**< Time to wait after Authentication co-processor operation fails. */
    CREATE_DB_VALUE(iPodControlIAP2AuthLongWait, int, 50 /**< default value */);                /**< Time to wait after Authentication co-processor operation fails very often. */
    CREATE_DB_VALUE(iPodControlIAP2UsbOtgGPIOPower, string, "111" /**< default value */);        /**< GPIO power pin */
    CREATE_DB_VALUE(iPodControlIAP2AccessoryVendorId, string, "4236" /**< default value */);    /**< decimal! Accessory Vendor ID 0x108C (Robert Bosch GmbH) */
    CREATE_DB_VALUE(iPodControlIAP2AccessoryProductId, string, "376" /**< default value */);    /**< decimal! Accessory Product ID 0x0178 Gen.3 Media Player */
    CREATE_DB_VALUE(iPodControlIAP2AccessoryBcdDevice, string, "1" /**< default value */);        /**< Accessory BCD ID */
    CREATE_DB_VALUE(iPodControlIAP2AccessoryInitEndPoint, string, "/dev/ffs/ep0" /**< default value */);      /**< Accessory init endpoint, gadget file system */
    CREATE_DB_VALUE(iPodControlIAP2FingerprintTimeoutMS, long, 10000);  /**< MediaLibrary Update: Timeout for fingerprint answer to indexer */

    CREATE_DB_VALUE(iPodControlIAP2GadgetAudioParams, string, "" /**< default value */);   /**< Extended option string for module load g_ffs with gadget audio */
    CREATE_DB_VALUE(iPodControlIAP2ConfigFSEnabled, int, 0);   /**< iAP2 hostmode based on ConfigFS driver (not GadgetFS) */
    CREATE_DB_VALUE(iPodControlIAP2ControlSessionVersion, int, 2);   /**< configure iAP2 control session V1 / V2 (default) */

    CREATE_DB_VALUE(iPodControlAppInfos, int, 0);           /**< list of iOS application info */
    CREATE_DB_VALUE(iPodControlAppInfoAppName_1,  string, "");
    CREATE_DB_VALUE(iPodControlAppInfoProtocol_1, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoBundleID_1, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoOption_1, int, 0);     /**< see enum tAppInfoOption */
    CREATE_DB_VALUE(iPodControlAppInfoAppName_2,  string, "");
    CREATE_DB_VALUE(iPodControlAppInfoProtocol_2, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoBundleID_2, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoOption_2, int, 0);     /**< see enum tAppInfoOption */
    CREATE_DB_VALUE(iPodControlAppInfoAppName_3,  string, "");
    CREATE_DB_VALUE(iPodControlAppInfoProtocol_3, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoBundleID_3, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoOption_3, int, 0);     /**< see enum tAppInfoOption */
    CREATE_DB_VALUE(iPodControlAppInfoAppName_4,  string, "");
    CREATE_DB_VALUE(iPodControlAppInfoProtocol_4, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoBundleID_4, string, "");
    CREATE_DB_VALUE(iPodControlAppInfoOption_4, int, 0);     /**< see enum tAppInfoOption */

    CREATE_DB_VALUE(iPodControlAppControlReauthEnabled, int, 0 );                /**< enabled re authentication on app control config changes */
    CREATE_DB_VALUE(iPodControlAppControlReconnectEnabled, int, 0 );             /**< enabled reconnect on app control close PSARCC-11861*/

    CREATE_DB_VALUE(iPodControlIAP1AppControlEnabled, int, 0 /**< default value */);    /**< enable iAP1 app control */
    CREATE_DB_VALUE(iPodControlIAP2AppControlEnabled, int, 0 /**< default value */);    /**< enable iAP2 app control */

    CREATE_DB_VALUE(iPodControlAlbumArtWidthMin, int, 120 /**< default value */);        /**< min width for albumArt resolution retrieved from iPod */
    CREATE_DB_VALUE(iPodControlAlbumArtWidthMax, int, 250 /**< default value */);        /**< max width for albumArt resolution retrieved from iPod */

    CREATE_DB_VALUE(iPodControlIAP1UIDBasedIndexingSupport, int, 1 /**< default value */);
    CREATE_DB_VALUE(iPodControlIAP1ChapterStreaming, int, 1 /**< default value */);     /**< play audiobook with embedded chapters always in streaming mode */
    CREATE_DB_VALUE(iPodControlIAP2RepeatDisabled, int, 0 /**< default value */);       /* disable repeat mode option for iAP2 devices */

    CREATE_DB_VALUE(iPodControlIAP2BatchPlaying, int, 1 /**< default value */);         /* enable support for batch list via PlaymediaLibraryItems */
    CREATE_DB_VALUE(iPodControlIAP2BatchLimit, int, 8000 /**< default value */);        /* limit for batch play list size via PlaymediaLibraryItems */

    CREATE_DB_VALUE(iPodControlIAP2iTunesRadioEnabled, int, 1 /**< default value */);   /* enable support for iTunes Radio playlists */
    CREATE_DB_VALUE(iPodControlIAP2QueueListEnabled, int, 1 /**< default value */);     /* enable support for IAP2 queue list transfer */
    CREATE_DB_VALUE(iPodControlIAP2QueueListResumeOnSelection, int, 1 /**< default value */);     /* send HID PLAY on queue list selection, may cause certification errors */

    CREATE_DB_VALUE(iPodControlIAP2BTTransportComponentEnabled, int, 1 /**< default value */);     /* enable support for IAP2 bluetooth transport component */
    CREATE_DB_VALUE(iPodControlIAP2BTConnectionUpdatesEnabled, int, 1 /**< default value */);     /* enable support for IAP2 bluetooth connection updates */
    CREATE_DB_VALUE(iPodControlIAP2CallStateUpdateEnabled, int, 1 /**< default value */);     /* enable support for IAP2 call state information */
    CREATE_DB_VALUE(iPodControlIAP2CommunicationsUpdateEnabled, int, 1 /**< default value */);     /* enable support for IAP2 communications information */
    CREATE_DB_VALUE(iPodControlIAP2RequestAppLaunchEnabled, int, 1 /**< default value */);     /* enable support for IAP2 request app launch */
    CREATE_DB_VALUE(iPodCommunicationError, int, 0);                                        /* For errors received from Apple(IAP1/IAP2) devices to Media player which needs to be updated to HMI */
    CREATE_DB_VALUE(DipoCommunicationError, int, 0);                                        /* For RoleSwitch Error update to SPI */
    CREATE_DB_VALUE(iPodSiriActivated, int, 0);
    CREATE_DB_VALUE(BTSourcePausedBySIRI, int, 0); /*Separate variable for BT device to indicate that source has been paused when BT device is activated */
    CREATE_DB_VALUE(iPodRepeatModeSequence_1, int, 1);        /* RPT_NONE, RPT_ONE, RPT_LIST */
    CREATE_DB_VALUE(iPodRepeatModeSequence_2, int, 0);        /* RPT_NONE, RPT_LIST, RPT_ONE */
    CREATE_DB_VALUE(iPodRouteGuidanceEnabled, int, 1);        /* Start & Stop Route Guidance update messages enabled */
    CREATE_DB_VALUE(updateReadyToPlay, int, 0);    /* Enable or disable ReadyToPlay Property update to Connectivity */
    CREATE_DB_VALUE(iPodControliAP2ProductPlanUUID, string, ""); /* Product Plan UUID string */

    CREATE_DB_VALUE(nextPossibleSourceEnabled, int, 0); /*NCG3D-86873:Next possible source information(For VR,SIRI activation) from audio will be available.*/
    CREATE_DB_VALUE(nextPossibleSourceEnabledTimeoutMs, int, 800); /*NCG3D-86873:Timeout for handling deallocation and sa_pause when nextPossibleSourceEnabled is enabled.*/

    CREATE_DB_VALUE(PowerHubEnabled, int, 0 /**< default value */);                     /* enable extra power for charging iPad (2100mA) */
    CREATE_DB_VALUE(PowerHubVendorID, string, "2996");                                  /**< UNWIRED HUB */
    CREATE_DB_VALUE(PowerHubProductID_A, string, "0112");                               /**< UNWIRED HUB 0x112 with AUX */
    CREATE_DB_VALUE(PowerHubProductID_A_Port_A, int, 2);                                /**< UNWIRED HUB 0x112 with AUX */
    CREATE_DB_VALUE(PowerHubProductID_A_Port_B, int, 3);                                /**< UNWIRED HUB 0x112 with AUX */
    CREATE_DB_VALUE(PowerHubProductID_B, string, "0113");                               /**< UNWIRED HUB 0x113 without AUX */
    CREATE_DB_VALUE(PowerHubProductID_B_Port_A, int, 3);                                /**< UNWIRED HUB 0x113 without AUX */
    CREATE_DB_VALUE(PowerHubProductID_B_Port_B, int, 4);                                /**< UNWIRED HUB 0x113 without AUX */

    CREATE_DB_VALUE(AvailableCurrentPowerHub, int, 2100);                               /**< Default value in mA. Must be one of the following, values - 0,1000,2100,2400 */
    CREATE_DB_VALUE(AvailableCurrentDefault, int, 1000);                                /**< Extra power, value in mA. Must be one of the following, values - 0,1000,2100,2400 */

    CREATE_DB_VALUE(CriticalMemoryUsageMB, int, 200);   /**< memory consumption limit, call ERRMEM trace and malloc_trim  */

    CREATE_DB_VALUE(KeepLastPlaybackStateForStreamingDevicesEnabled, int, 0 /**< default value */);     /* keep the last playback state for all streaming devices, iPod BT, when device is sourced again */

    /* codec restrictions Gen3 */
    CREATE_DB_VALUE(VideoHeightVariance, int, 20);                                      /**< additional control lines in video images. These lines should not be checked by codec restriction */
    CREATE_DB_VALUE(CodecRestrictions, int, 27); // number of codecs which have restrictions
    CREATE_DB_VALUE(CodecRestrictionCodec_01, int, CDC_MPEG4);
    CREATE_DB_VALUE(CodecRestrictionProfile_01, string, "SIMPLE");
    CREATE_DB_VALUE(CodecRestrictionSupported_01, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_01, float, 6);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_01, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_01, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_01, int, 40000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_01, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_02, int, CDC_MPEG4);
    CREATE_DB_VALUE(CodecRestrictionProfile_02, string, "ADVANCED"); //="ADVANCED SIMPLE"
    CREATE_DB_VALUE(CodecRestrictionSupported_02, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_02, float, 6);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_02, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_02, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_02, int, 40000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_02, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_03, int, CDC_DIVX);
    CREATE_DB_VALUE(CodecRestrictionProfile_03, string, "QMOBILE");
    CREATE_DB_VALUE(CodecRestrictionSupported_03, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_03, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_03, int, 176);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_03, int, 144);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_03, int, 460000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_03, int, 15);

    CREATE_DB_VALUE(CodecRestrictionCodec_04, int, CDC_DIVX);
    CREATE_DB_VALUE(CodecRestrictionProfile_04, string, "MOBILE");
    CREATE_DB_VALUE(CodecRestrictionSupported_04, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_04, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_04, int, 320);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_04, int, 240);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_04, int, 1240000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_04, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_05, int, CDC_DIVX);
    CREATE_DB_VALUE(CodecRestrictionProfile_05, string, "HOMETHEATRE");
    CREATE_DB_VALUE(CodecRestrictionSupported_05, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_05, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_05, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_05, int, 576);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_05, int, 6000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_05, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_06, int, CDC_XVID);
    CREATE_DB_VALUE(CodecRestrictionProfile_06, string, "MOBILE");
    CREATE_DB_VALUE(CodecRestrictionSupported_06, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_06, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_06, int, 352);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_06, int, 288);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_06, int, 1300000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_06, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_07, int, CDC_XVID);
    CREATE_DB_VALUE(CodecRestrictionProfile_07, string, "PORTABLE");
    CREATE_DB_VALUE(CodecRestrictionSupported_07, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_07, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_07, int, 640);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_07, int, 480);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_07, int, 4800000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_07, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_08, int, CDC_XVID);
    CREATE_DB_VALUE(CodecRestrictionProfile_08, string, "HOME");
    CREATE_DB_VALUE(CodecRestrictionSupported_08, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_08, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_08, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_08, int, 576);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_08, int, 4800000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_08, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_09, int, CDC_AVC);
    CREATE_DB_VALUE(CodecRestrictionProfile_09, string, "CONSTRAINED"); //="CONSTRAINED BASELINE"
    CREATE_DB_VALUE(CodecRestrictionSupported_09, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_09, float, 4.2);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_09, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_09, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_09, int, 50000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_09, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_10, int, CDC_AVC);
    CREATE_DB_VALUE(CodecRestrictionProfile_10, string, "BASELINE");
    CREATE_DB_VALUE(CodecRestrictionSupported_10, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_10, float, 4.2);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_10, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_10, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_10, int, 50000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_10, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_11, int, CDC_AVC); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_11, string, "MAIN");
    CREATE_DB_VALUE(CodecRestrictionSupported_11, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_11, float, 4.2);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_11, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_11, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_11, int, 50000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_11, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_12, int, CDC_AVC); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_12, string, "HIGH");
    CREATE_DB_VALUE(CodecRestrictionSupported_12, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_12, float, 4.2);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_12, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_12, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_12, int, 62500000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_12, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_13, int, CDC_WMV7);
    CREATE_DB_VALUE(CodecRestrictionProfile_13, string, "");
    CREATE_DB_VALUE(CodecRestrictionSupported_13, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_13, float, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_13, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_13, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_13, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_13, int, 0);

    CREATE_DB_VALUE(CodecRestrictionCodec_14, int, CDC_WMV8);
    CREATE_DB_VALUE(CodecRestrictionProfile_14, string, "");
    CREATE_DB_VALUE(CodecRestrictionSupported_14, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_14, float, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_14, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_14, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_14, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_14, int, 0);

    CREATE_DB_VALUE(CodecRestrictionCodec_15, int, CDC_WMV9);
    CREATE_DB_VALUE(CodecRestrictionProfile_15, string, "SIMPLE");
    CREATE_DB_VALUE(CodecRestrictionSupported_15, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_15, float, -1); //No restriction (only MEDIUM, but which number is it?)
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_15, int, 352);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_15, int, 288);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_15, int, 384000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_15, int, 15);

    CREATE_DB_VALUE(CodecRestrictionCodec_16, int, CDC_WMV9);
    CREATE_DB_VALUE(CodecRestrictionProfile_16, string, "MP"); //="MAIN"
    CREATE_DB_VALUE(CodecRestrictionSupported_16, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_16, float, -1); //No restriction (only MEDIUM, but which number is it?)
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_16, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_16, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_16, int, 20000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_16, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_17, int, CDC_WMV9); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_17, string, "ADVANCED");
    CREATE_DB_VALUE(CodecRestrictionSupported_17, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_17, float, 3);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_17, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_17, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_17, int, 45000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_17, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_18, int, CDC_VP6);
    CREATE_DB_VALUE(CodecRestrictionProfile_18, string, "SIMPLE");
    CREATE_DB_VALUE(CodecRestrictionSupported_18, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_18, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_18, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_18, int, 576);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_18, int, 768000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_18, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_19, int, CDC_VP6);
    CREATE_DB_VALUE(CodecRestrictionProfile_19, string, "ADVANCED");
    CREATE_DB_VALUE(CodecRestrictionSupported_19, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_19, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_19, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_19, int, 576);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_19, int, 768000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_19, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_20, int, CDC_SORENSON);
    CREATE_DB_VALUE(CodecRestrictionProfile_20, string, "");
    CREATE_DB_VALUE(CodecRestrictionSupported_20, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_20, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_20, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_20, int, 576);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_20, int, 6000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_20, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_21, int, CDC_H263);
    CREATE_DB_VALUE(CodecRestrictionProfile_21, string, "0"); //="0 BASELINE"
    CREATE_DB_VALUE(CodecRestrictionSupported_21, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_21, float, 70);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_21, int, 1408);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_21, int, 1152);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_21, int, 6000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_21, int, 25);

    CREATE_DB_VALUE(CodecRestrictionCodec_22, int, CDC_H263); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_22, string, "3");
    CREATE_DB_VALUE(CodecRestrictionSupported_22, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_22, float, 70);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_22, int, 1408);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_22, int, 1152);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_22, int, 20000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_22, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_23, int, CDC_VP8); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_23, string, "");
    CREATE_DB_VALUE(CodecRestrictionSupported_23, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_23, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_23, int, 1280);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_23, int, 720);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_23, int, 20000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_23, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_24, int, CDC_MPEG2); //new for Gen3
    CREATE_DB_VALUE(CodecRestrictionProfile_24, string, "");
    CREATE_DB_VALUE(CodecRestrictionSupported_24, int, 1);
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_24, float, -1); //No restriction
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_24, int, 1920);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_24, int, 1080);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_24, int, 50000000);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_24, int, 30);

    CREATE_DB_VALUE(CodecRestrictionCodec_25, int, CDC_WMA);
    CREATE_DB_VALUE(CodecRestrictionProfile_25, string, "VERSION 1");
    CREATE_DB_VALUE(CodecRestrictionSupported_25, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_25, float, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_25, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_25, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_25, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_25, int, 0);

    CREATE_DB_VALUE(CodecRestrictionCodec_26, int, CDC_WMA);
    CREATE_DB_VALUE(CodecRestrictionProfile_26, string, "VERSION 3");
    CREATE_DB_VALUE(CodecRestrictionSupported_26, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_26, float, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_26, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_26, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_26, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_26, int, 0);

    CREATE_DB_VALUE(CodecRestrictionCodec_27, int, CDC_WMA);
    CREATE_DB_VALUE(CodecRestrictionProfile_27, string, "VERSION 4");
    CREATE_DB_VALUE(CodecRestrictionSupported_27, int, 0); //Not supported
    CREATE_DB_VALUE(CodecRestrictionMaxLevel_27, float, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxWidth_27, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxHeight_27, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxBitRate_27, int, 0);
    CREATE_DB_VALUE(CodecRestrictionMaxFrameRate_27, int, 0);

    /* genre mapping Roadmap 13014 */
    /* source: VW requirement */
    CREATE_DB_VALUE(Genre_000, string, "Blues");
    CREATE_DB_VALUE(Genre_001, string, "Classic Rock");
    CREATE_DB_VALUE(Genre_002, string, "Country");
    CREATE_DB_VALUE(Genre_003, string, "Dance");
    CREATE_DB_VALUE(Genre_004, string, "Disco");
    CREATE_DB_VALUE(Genre_005, string, "Funk");
    CREATE_DB_VALUE(Genre_006, string, "Grunge");
    CREATE_DB_VALUE(Genre_007, string, "Hip-Hop");
    CREATE_DB_VALUE(Genre_008, string, "Jazz");
    CREATE_DB_VALUE(Genre_009, string, "Metal");
    CREATE_DB_VALUE(Genre_010, string, "New Age");
    CREATE_DB_VALUE(Genre_011, string, "Oldies");
    CREATE_DB_VALUE(Genre_012, string, "Other");
    CREATE_DB_VALUE(Genre_013, string, "Pop");
    CREATE_DB_VALUE(Genre_014, string, "R&B");
    CREATE_DB_VALUE(Genre_015, string, "Rap");
    CREATE_DB_VALUE(Genre_016, string, "Reggae");
    CREATE_DB_VALUE(Genre_017, string, "Rock");
    CREATE_DB_VALUE(Genre_018, string, "Techno");
    CREATE_DB_VALUE(Genre_019, string, "Industrial");
    CREATE_DB_VALUE(Genre_020, string, "Alternative");
    CREATE_DB_VALUE(Genre_021, string, "Ska");
    CREATE_DB_VALUE(Genre_022, string, "Death metal");
    CREATE_DB_VALUE(Genre_023, string, "Pranks");
    CREATE_DB_VALUE(Genre_024, string, "Soundtrack");
    CREATE_DB_VALUE(Genre_025, string, "Euro-Techno");
    CREATE_DB_VALUE(Genre_026, string, "Ambient");
    CREATE_DB_VALUE(Genre_027, string, "Trip-hop");
    CREATE_DB_VALUE(Genre_028, string, "Vocal");
    CREATE_DB_VALUE(Genre_029, string, "Jazz+Funk");
    CREATE_DB_VALUE(Genre_030, string, "Fusion");
    CREATE_DB_VALUE(Genre_031, string, "Trance");
    CREATE_DB_VALUE(Genre_032, string, "Classical");
    CREATE_DB_VALUE(Genre_033, string, "Instrumental");
    CREATE_DB_VALUE(Genre_034, string, "Acid");
    CREATE_DB_VALUE(Genre_035, string, "House");
    CREATE_DB_VALUE(Genre_036, string, "Game");
    CREATE_DB_VALUE(Genre_037, string, "Sound Clip");
    CREATE_DB_VALUE(Genre_038, string, "Gospel");
    CREATE_DB_VALUE(Genre_039, string, "Noise");
    CREATE_DB_VALUE(Genre_040, string, "Alt. Rock");
    CREATE_DB_VALUE(Genre_041, string, "Bass");
    CREATE_DB_VALUE(Genre_042, string, "Soul");
    CREATE_DB_VALUE(Genre_043, string, "Punk");
    CREATE_DB_VALUE(Genre_044, string, "Space");
    CREATE_DB_VALUE(Genre_045, string, "Meditative");
    CREATE_DB_VALUE(Genre_046, string, "Instrumental pop");
    CREATE_DB_VALUE(Genre_047, string, "Instrumental rock");
    CREATE_DB_VALUE(Genre_048, string, "Ethnic");
    CREATE_DB_VALUE(Genre_049, string, "Gothic");
    CREATE_DB_VALUE(Genre_050, string, "Darkwave");
    CREATE_DB_VALUE(Genre_051, string, "Techno-Industrial");
    CREATE_DB_VALUE(Genre_052, string, "Electronic");
    CREATE_DB_VALUE(Genre_053, string, "Pop-Folk");
    CREATE_DB_VALUE(Genre_054, string, "Eurodance");
    CREATE_DB_VALUE(Genre_055, string, "Dream");
    CREATE_DB_VALUE(Genre_056, string, "Southern Rock");
    CREATE_DB_VALUE(Genre_057, string, "Comedy");
    CREATE_DB_VALUE(Genre_058, string, "Cult");
    CREATE_DB_VALUE(Genre_059, string, "Gangsta");
    CREATE_DB_VALUE(Genre_060, string, "Top 40");
    CREATE_DB_VALUE(Genre_061, string, "Christian Rap");
    CREATE_DB_VALUE(Genre_062, string, "Pop/Funk");
    CREATE_DB_VALUE(Genre_063, string, "Jungle");
    CREATE_DB_VALUE(Genre_064, string, "Native American");
    CREATE_DB_VALUE(Genre_065, string, "Cabaret");
    CREATE_DB_VALUE(Genre_066, string, "New Wave");
    CREATE_DB_VALUE(Genre_067, string, "Psychedelic");
    CREATE_DB_VALUE(Genre_068, string, "Rave");
    CREATE_DB_VALUE(Genre_069, string, "Showtunes");
    CREATE_DB_VALUE(Genre_070, string, "Trailer");
    CREATE_DB_VALUE(Genre_071, string, "Lo-Fi");
    CREATE_DB_VALUE(Genre_072, string, "Tribal");
    CREATE_DB_VALUE(Genre_073, string, "Acid Punk");
    CREATE_DB_VALUE(Genre_074, string, "Acid Jazz");
    CREATE_DB_VALUE(Genre_075, string, "Polka");
    CREATE_DB_VALUE(Genre_076, string, "Retro");
    CREATE_DB_VALUE(Genre_077, string, "Musical");
    CREATE_DB_VALUE(Genre_078, string, "Rock & Roll");
    CREATE_DB_VALUE(Genre_079, string, "Hard Rock");
    CREATE_DB_VALUE(Genre_080, string, "Folk");
    CREATE_DB_VALUE(Genre_081, string, "Folk-Rock");
    CREATE_DB_VALUE(Genre_082, string, "National Folk");
    CREATE_DB_VALUE(Genre_083, string, "Swing");
    CREATE_DB_VALUE(Genre_084, string, "Fast Fusion");
    CREATE_DB_VALUE(Genre_085, string, "Bebob");
    CREATE_DB_VALUE(Genre_086, string, "Latin");
    CREATE_DB_VALUE(Genre_087, string, "Revival");
    CREATE_DB_VALUE(Genre_088, string, "Celtic");
    CREATE_DB_VALUE(Genre_089, string, "Bluegrass");
    CREATE_DB_VALUE(Genre_090, string, "Avantgarde");
    CREATE_DB_VALUE(Genre_091, string, "Gothic Rock");
    CREATE_DB_VALUE(Genre_092, string, "Progressive Rock");
    CREATE_DB_VALUE(Genre_093, string, "Psychedelic Rock");
    CREATE_DB_VALUE(Genre_094, string, "Symphonic Rock");
    CREATE_DB_VALUE(Genre_095, string, "Slow Rock");
    CREATE_DB_VALUE(Genre_096, string, "Big Band");
    CREATE_DB_VALUE(Genre_097, string, "Chorus");
    CREATE_DB_VALUE(Genre_098, string, "Easy Listening");
    CREATE_DB_VALUE(Genre_099, string, "Acoustic");
    CREATE_DB_VALUE(Genre_100, string, "Humour");
    CREATE_DB_VALUE(Genre_101, string, "Speech");
    CREATE_DB_VALUE(Genre_102, string, "Chanson");
    CREATE_DB_VALUE(Genre_103, string, "Opera");
    CREATE_DB_VALUE(Genre_104, string, "Chamber Music");
    CREATE_DB_VALUE(Genre_105, string, "Sonata");
    CREATE_DB_VALUE(Genre_106, string, "Symphony");
    CREATE_DB_VALUE(Genre_107, string, "Booty Bass");
    CREATE_DB_VALUE(Genre_108, string, "Primus");
    CREATE_DB_VALUE(Genre_109, string, "Porn Groove");
    CREATE_DB_VALUE(Genre_110, string, "Satire");
    CREATE_DB_VALUE(Genre_111, string, "Slow Jam");
    CREATE_DB_VALUE(Genre_112, string, "Club");
    CREATE_DB_VALUE(Genre_113, string, "Tango");
    CREATE_DB_VALUE(Genre_114, string, "Samba");
    CREATE_DB_VALUE(Genre_115, string, "Folklore");
    CREATE_DB_VALUE(Genre_116, string, "Ballad");
    CREATE_DB_VALUE(Genre_117, string, "Power Ballad");
    CREATE_DB_VALUE(Genre_118, string, "Rhythmic Soul");
    CREATE_DB_VALUE(Genre_119, string, "Freestyle");
    CREATE_DB_VALUE(Genre_120, string, "Duet");
    CREATE_DB_VALUE(Genre_121, string, "Punk Rock");
    CREATE_DB_VALUE(Genre_122, string, "Drum Solo");
    CREATE_DB_VALUE(Genre_123, string, "A cappella");
    CREATE_DB_VALUE(Genre_124, string, "Euro-House");
    CREATE_DB_VALUE(Genre_125, string, "Dance Hall");
    CREATE_DB_VALUE(Genre_126, string, "Goa");
    CREATE_DB_VALUE(Genre_127, string, "Drum & Bass");
    CREATE_DB_VALUE(Genre_128, string, "Club-House");
    CREATE_DB_VALUE(Genre_129, string, "Hardcore");
    CREATE_DB_VALUE(Genre_130, string, "Terror");
    CREATE_DB_VALUE(Genre_131, string, "Indie");
    CREATE_DB_VALUE(Genre_132, string, "BritPop");
    CREATE_DB_VALUE(Genre_133, string, "Negerpunk");
    CREATE_DB_VALUE(Genre_134, string, "Polsk Punk");
    CREATE_DB_VALUE(Genre_135, string, "Beat");
    CREATE_DB_VALUE(Genre_136, string, "Christian gangsta rap");
    CREATE_DB_VALUE(Genre_137, string, "Heavy Metal");
    CREATE_DB_VALUE(Genre_138, string, "Black Metal");
    CREATE_DB_VALUE(Genre_139, string, "Crossover");
    CREATE_DB_VALUE(Genre_140, string, "Contemporary Christian");
    CREATE_DB_VALUE(Genre_141, string, "Christian Rock");
    CREATE_DB_VALUE(Genre_142, string, "Merengue");
    CREATE_DB_VALUE(Genre_143, string, "Salsa");
    CREATE_DB_VALUE(Genre_144, string, "Thrash Metal");
    CREATE_DB_VALUE(Genre_145, string, "Anime");
    CREATE_DB_VALUE(Genre_146, string, "JPop");
    CREATE_DB_VALUE(Genre_147, string, "Synthpop");
    /* source: Internet http://de.wikipedia.org/wiki/Liste_der_ID3v1-Genres */
    CREATE_DB_VALUE(Genre_148, string, "Abstract");
    CREATE_DB_VALUE(Genre_149, string, "Art Rock");
    CREATE_DB_VALUE(Genre_150, string, "Baroque");
    CREATE_DB_VALUE(Genre_151, string, "Bhangra");
    CREATE_DB_VALUE(Genre_152, string, "Big Beat");
    CREATE_DB_VALUE(Genre_153, string, "Breakbeat");
    CREATE_DB_VALUE(Genre_154, string, "Chillout");
    CREATE_DB_VALUE(Genre_155, string, "Downtempo");
    CREATE_DB_VALUE(Genre_156, string, "Dub");
    CREATE_DB_VALUE(Genre_157, string, "EBM");
    CREATE_DB_VALUE(Genre_158, string, "Eclectic");
    CREATE_DB_VALUE(Genre_159, string, "Electro");
    CREATE_DB_VALUE(Genre_160, string, "Electroclash");
    CREATE_DB_VALUE(Genre_161, string, "Emo");
    CREATE_DB_VALUE(Genre_162, string, "Experimental");
    CREATE_DB_VALUE(Genre_163, string, "Garage");
    CREATE_DB_VALUE(Genre_164, string, "Global");
    CREATE_DB_VALUE(Genre_165, string, "IDM");
    CREATE_DB_VALUE(Genre_166, string, "Illbient");
    CREATE_DB_VALUE(Genre_167, string, "Industro-Goth");
    CREATE_DB_VALUE(Genre_168, string, "Jam Band");
    CREATE_DB_VALUE(Genre_169, string, "Krautrock");
    CREATE_DB_VALUE(Genre_170, string, "Leftfield");
    CREATE_DB_VALUE(Genre_171, string, "Lounge");
    CREATE_DB_VALUE(Genre_172, string, "Math Rock");
    CREATE_DB_VALUE(Genre_173, string, "New Romantic");
    CREATE_DB_VALUE(Genre_174, string, "Nu-Breakz");
    CREATE_DB_VALUE(Genre_175, string, "Post-Punk");
    CREATE_DB_VALUE(Genre_176, string, "Post-Rock");
    CREATE_DB_VALUE(Genre_177, string, "Psytrance");
    CREATE_DB_VALUE(Genre_178, string, "Shoegaze");
    CREATE_DB_VALUE(Genre_179, string, "Space Rock");
    CREATE_DB_VALUE(Genre_180, string, "Trop Rock");
    CREATE_DB_VALUE(Genre_181, string, "World Music");
    CREATE_DB_VALUE(Genre_182, string, "Neoclassical");
    CREATE_DB_VALUE(Genre_183, string, "Audiobook");
    CREATE_DB_VALUE(Genre_184, string, "Audio Theatre");
    CREATE_DB_VALUE(Genre_185, string, "Neue Deutsche Welle");
    CREATE_DB_VALUE(Genre_186, string, "Podcast");
    CREATE_DB_VALUE(Genre_187, string, "Indie Rock");
    CREATE_DB_VALUE(Genre_188, string, "G-Funk");
    CREATE_DB_VALUE(Genre_189, string, "Dubstep");
    CREATE_DB_VALUE(Genre_190, string, "Garage Rock");
    CREATE_DB_VALUE(Genre_191, string, "Psybient");

    CREATE_DB_VALUE(TestConfigTableColumns, int, 5); //precondition+numberOfCriteria+action
    CREATE_DB_VALUE(TestConfigTableRows, int, 6);
    //Attention: use max 8 criteria (static array)
    CREATE_DB_VALUE(TestConfigTableUsedCriteria, string, "1,2,3"); //1=CTC_TEST1, 2=CTC_TEST2, 3=CTC_TEST3
    //Attention: use max 20 config rows (static array)
    //Precondition 1
    CREATE_DB_VALUE(TestConfigTablePrecondition_01, int, CTP_TEST1);
    CREATE_DB_VALUE(TestConfigTableCriteria_01, string, "-1,1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_01, int, CTA_TEST1);
    CREATE_DB_VALUE(TestConfigTablePrecondition_02, int, CTP_TEST1);
    CREATE_DB_VALUE(TestConfigTableCriteria_02, string, "1,0,1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_02, int, CTA_TEST2);
    CREATE_DB_VALUE(TestConfigTablePrecondition_03, int, CTP_TEST1);
    CREATE_DB_VALUE(TestConfigTableCriteria_03, string, "0,0,1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_03, int, CTA_TEST3);
    //Precondition 2
    CREATE_DB_VALUE(TestConfigTablePrecondition_04, int, CTP_TEST2);
    CREATE_DB_VALUE(TestConfigTableCriteria_04, string, "-1,1,1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_04, int, CTA_TEST4);
    CREATE_DB_VALUE(TestConfigTablePrecondition_05, int, CTP_TEST2);
    CREATE_DB_VALUE(TestConfigTableCriteria_05, string, "1,0,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_05, int, CTA_TEST5);
    CREATE_DB_VALUE(TestConfigTablePrecondition_06, int, CTP_TEST2);
    CREATE_DB_VALUE(TestConfigTableCriteria_06, string, "-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(TestConfigTableAction_06, int, CTA_TEST6);

    CREATE_DB_VALUE(DeviceActivationTableColumns, int, 9); //precondition+numberOfCriteria+action
    CREATE_DB_VALUE(DeviceActivationTableRows, int, 9);
    //Attention: use max 8 criteria (static array)
    //4=CTC_STARTUP, 5=CTC_LAST_PLAYED_DEVICE, 6=CTC_ANY_DEVICE_ACTIVE, 7=CTC_ANY_DEVICE_CONNECTED, 8=CTC_AUTOPLAY_DEVICE, 9=CTC_SAME_DEVICE_TYPE_LIKE_BEFORE
    //14=CTC_HW_MALFUNC_IN_LAST_CONNECTION
    CREATE_DB_VALUE(DeviceActivationTableUsedCriteria, string, "4,5,6,7,8,9,14");
    //Attention: use max 20 config rows (static array)
    //Add device
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_01, int, CTP_ADD_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_01, string, "0,-1,0,1,1,1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_01, int, CTA_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_02, int, CTP_ADD_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_02, string, "0,-1,0,0,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_02, int, CTA_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_03, int, CTP_ADD_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_03, string, "0,-1,-1,-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_03, int, CTA_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_04, int, CTP_ADD_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_04, string, "-1,1,-1,-1,-1,-1,1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_04, int, CTA_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_05, int, CTP_ADD_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_05, string, "1,1,-1,-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_05, int, CTA_ACTIVATE_DEVICE_STOP_TIMER);
    //Activate device
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_06, int, CTP_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_06, string, "0,-1,-1,-1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_06, int, CTA_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_07, int, CTP_ACTIVATE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_07, string, "1,-1,-1,-1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_07, int, CTA_ACTIVATE_DEVICE_STOP_TIMER);
    //Remove device
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_08, int, CTP_REMOVE_DEVICE);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_08, string, "-1,-1,-1,-1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_08, int, CTA_DEACTIVATE_DEVICE);
    //Timer expired
    CREATE_DB_VALUE(DeviceActivationTablePrecondition_09, int, CTP_TIMER_EXPIRED);
    CREATE_DB_VALUE(DeviceActivationTableCriteria_09, string, "-1,-1,0,1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(DeviceActivationTableAction_09, int, CTA_ACTIVATE_FIRST_AVAILABLE_DEVICE);

    CREATE_DB_VALUE(ListSelectionTableColumns, int, 6); //precondition+numberOfCriteria+action
    CREATE_DB_VALUE(ListSelectionTableRows, int, 17);
    //Attention: use max 8 criteria (static array)
    //10=CTC_STREAMING_ACTIVE, 11=CTC_LAST_MODE_LIST_AVAILABLE, 12=CTC_ANY_DB_CONTENT_AVAILABLE, 13=CTC_IPOD_CURRENT_SELECTION_CHANGED
    CREATE_DB_VALUE(ListSelectionTableUsedCriteria, string, "10,11,12,13");
    //Attention: use max 20 config rows (static array)
    //USB
    CREATE_DB_VALUE(ListSelectionTablePrecondition_01, int, CTP_USB);
    CREATE_DB_VALUE(ListSelectionTableCriteria_01, string, "-1,0,0,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_01, int, CTA_PLAY_FOLDER_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_02, int, CTP_USB);
    CREATE_DB_VALUE(ListSelectionTableCriteria_02, string, "-1,0,1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_02, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_03, int, CTP_USB);
    CREATE_DB_VALUE(ListSelectionTableCriteria_03, string, "-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_03, int, CTA_PLAY_DB_LAST_MODE_LIST);
    //MTP
    CREATE_DB_VALUE(ListSelectionTablePrecondition_04, int, CTP_MTP);
    CREATE_DB_VALUE(ListSelectionTableCriteria_04, string, "-1,0,0,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_04, int, CTA_PLAY_FOLDER_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_05, int, CTP_MTP);
    CREATE_DB_VALUE(ListSelectionTableCriteria_05, string, "-1,0,1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_05, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_06, int, CTP_MTP);
    CREATE_DB_VALUE(ListSelectionTableCriteria_06, string, "-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_06, int, CTA_PLAY_DB_LAST_MODE_LIST);
    //iPod
    CREATE_DB_VALUE(ListSelectionTablePrecondition_07, int, CTP_IPOD);
    CREATE_DB_VALUE(ListSelectionTableCriteria_07, string, "0,0,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_07, int, CTA_PLAY_CURRENT_SELECTION_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_08, int, CTP_IPOD);
    CREATE_DB_VALUE(ListSelectionTableCriteria_08, string, "0,1,-1,1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_08, int, CTA_PLAY_DB_LAST_MODE_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_09, int, CTP_IPOD);
    CREATE_DB_VALUE(ListSelectionTableCriteria_09, string, "0,1,-1,0"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_09, int, CTA_PLAY_CURRENT_SELECTION_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_10, int, CTP_IPOD);
    CREATE_DB_VALUE(ListSelectionTableCriteria_10, string, "1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_10, int, CTA_PLAY_CURRENT_SELECTION_STREAMING_LIST);
    //MyMedia
    CREATE_DB_VALUE(ListSelectionTablePrecondition_11, int, CTP_MYMEDIA);
    CREATE_DB_VALUE(ListSelectionTableCriteria_11, string, "-1,0,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_11, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_12, int, CTP_MYMEDIA);
    CREATE_DB_VALUE(ListSelectionTableCriteria_12, string, "-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_12, int, CTA_PLAY_DB_LAST_MODE_LIST);
    //Bluetooth
    CREATE_DB_VALUE(ListSelectionTablePrecondition_13, int, CTP_BT);
    CREATE_DB_VALUE(ListSelectionTableCriteria_13, string, "0,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_13, int, CTA_PLAY_CURRENT_SELECTION_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_14, int, CTP_BT);
    CREATE_DB_VALUE(ListSelectionTableCriteria_14, string, "1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_14, int, CTA_PLAY_CURRENT_SELECTION_STREAMING_LIST);
    //CDDA
    CREATE_DB_VALUE(ListSelectionTablePrecondition_15, int, CTP_CDDA);
    CREATE_DB_VALUE(ListSelectionTableCriteria_15, string, "-1,0,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_15, int, CTA_PLAY_FOLDER_LIST);
    CREATE_DB_VALUE(ListSelectionTablePrecondition_16, int, CTP_CDDA);
    CREATE_DB_VALUE(ListSelectionTableCriteria_16, string, "-1,1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_16, int, CTA_PLAY_DB_LAST_MODE_LIST);
    //Content Sharing
    CREATE_DB_VALUE(ListSelectionTablePrecondition_17, int, CTP_CS);
    CREATE_DB_VALUE(ListSelectionTableCriteria_17, string, "-1,-1,-1,-1"); //0=false, 1=true, -1=don't care
    CREATE_DB_VALUE(ListSelectionTableAction_17, int, CTA_PLAY_CURRENT_SELECTION_STREAMING_LIST);

    CREATE_DB_VALUE(FallbackActions, int, 5);                               /* number of fallback actions */
    CREATE_DB_VALUE(FallbackInAction_01, int, CTA_PLAY_DB_LAST_MODE_LIST);
    CREATE_DB_VALUE(FallbackOutAction_01, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(FallbackInAction_02, int, CTA_PLAY_CURRENT_SELECTION_LIST);
    CREATE_DB_VALUE(FallbackOutAction_02, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(FallbackInAction_03, int, CTA_PLAY_DB_ALL_SONGS_LIST);
    CREATE_DB_VALUE(FallbackOutAction_03, int, CTA_PLAY_DB_ALL_VIDEOS_LIST);
    CREATE_DB_VALUE(FallbackInAction_04, int, CTA_PLAY_DB_ALL_VIDEOS_LIST);
    CREATE_DB_VALUE(FallbackOutAction_04, int, CTA_PLAY_FOLDER_LIST);
    CREATE_DB_VALUE(FallbackInAction_05, int, CTA_PLAY_CURRENT_SELECTION_STREAMING_LIST);
    CREATE_DB_VALUE(FallbackOutAction_05, int, CTA_PLAY_FOLDER_STREAMING_LIST);
    CREATE_DB_VALUE(FallbackInAction_06, int, CTA_NONE);
    CREATE_DB_VALUE(FallbackOutAction_06, int, CTA_NONE);
    CREATE_DB_VALUE(FallbackInAction_07, int, CTA_NONE);
    CREATE_DB_VALUE(FallbackOutAction_07, int, CTA_NONE);
    CREATE_DB_VALUE(FallbackInAction_08, int, CTA_NONE);
    CREATE_DB_VALUE(FallbackOutAction_08, int, CTA_NONE); /* attention: if you add here a line, take care about the method: DataProvider::GetFallbackAction */

    /*Roadmap 13010_CDROM (new PlayerEngine)*/
    /* Activate MediaEngine per default and deactivate in project dependent configurations */
    CREATE_DB_VALUE(UseMediaEngine, int, 1 /**< default value */);
    CREATE_DB_VALUE(SyncMediaEngineStartup, int, 1 /**< default value */);
    CREATE_DB_VALUE(ExclusiveAlsaDevice, int, 1 /**< default value */);
    CREATE_DB_VALUE(SupervisionThreadOn, int, 1 /**< default value */);
    CREATE_DB_VALUE(MediaEngineUseFork, int, 1 /**< default value */);

    // process priority & niceness boundaries and their mapped values in top;
    // chrt -r -p 99 pid   ->   top:  RT
    // chrt -r -p 98 pid   ->   top: -99
    // ...
    // chrt -r -p  1 pid   ->   top:  -2
    //
    // renice -20 -p pid   ->   top: -20
    // ...
    // renice  19 -p pid   ->   top:  19

    CREATE_DB_VALUE(MediaEngineProcessPriority, string, "30" /**< default value */);
    CREATE_DB_VALUE(MediaEngineProcessNiceness, string, "-4" /**< default value */);
    CREATE_DB_VALUE(UnmuteRampTypeForStartOfTrack, tRampType, SLOW_RAMP /**< default value */);   /**< default rampType for start of track */

    /* PlayerEngine configuration Roadmap 13010 */
    //CREATE_DB_VALUE(PEConfigRampIntermediateSteps, int, 2 /**< default value */);    /**< Max: 16 steps */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_01, int, 20 /**< default value */);   /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_01, int, 5000 /**< default value */);   /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_02, int, 50 /**< default value */);   /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_02, int, 5000 /**< default value */);   /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_03, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_03, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_04, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_04, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_05, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_05, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_06, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_06, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_07, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_07, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_08, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_08, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_09, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_09, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_10, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_10, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_11, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_11, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_12, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_12, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_13, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_13, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_14, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_14, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_15, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_15, int, -1 /**< default value */);      /**< duration in milliseconds */
    CREATE_DB_VALUE(PEConfigRampStepPercentage_16, int, -1 /**< default value */);  /**< percentage of speed */
    CREATE_DB_VALUE(PEConfigRampStepDuration_16, int, -1 /**< default value */);      /**< duration in milliseconds */

    CREATE_DB_VALUE(PEConfigDisplays, int, 1);    /**< Max: 8 steps */
    // display 1;
    CREATE_DB_VALUE(PEConfigDisplayWidth_01,   int,  800 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_01,  int,  480 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_01,   int,   20 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_01, int,   22 /**< default value */);   /**< video surface */
    // display 2;
    CREATE_DB_VALUE(PEConfigDisplayWidth_02,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_02,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_02,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_02, int,    0 /**< default value */);   /**< video surface */
    // display 3;
    CREATE_DB_VALUE(PEConfigDisplayWidth_03,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_03,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_03,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_03, int,    0 /**< default value */);   /**< video surface */
    // display 4;
    CREATE_DB_VALUE(PEConfigDisplayWidth_04,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_04,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_04,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_04, int,    0 /**< default value */);   /**< video surface */
    // display 5;
    CREATE_DB_VALUE(PEConfigDisplayWidth_05,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_05,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_05,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_05, int,    0 /**< default value */);   /**< video surface */
    // display 6;
    CREATE_DB_VALUE(PEConfigDisplayWidth_06,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_06,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_06,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_06, int,    0 /**< default value */);   /**< video surface */
    // display 7;
    CREATE_DB_VALUE(PEConfigDisplayWidth_07,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_07,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_07,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_07, int,    0 /**< default value */);   /**< video surface */
    // display 8;
    CREATE_DB_VALUE(PEConfigDisplayWidth_08,   int,    0 /**< default value */);   /**< display width */
    CREATE_DB_VALUE(PEConfigDisplayHeight_08,  int,    0 /**< default value */);   /**< display height */
    CREATE_DB_VALUE(PEConfigDisplayLayer_08,   int,    0 /**< default value */);   /**< video layer */
    CREATE_DB_VALUE(PEConfigDisplaySurface_08, int,    0 /**< default value */);   /**< video surface */

    CREATE_DB_VALUE(MediaEngineConfigureRetries, int, 2 /**< default value */);
    CREATE_DB_VALUE(MediaEngineConfigureTimeoutMs, int, 10000 /**< default value */);      /**< time in milliseconds MUST NOT BE LESS THAN 1000 */
    CREATE_DB_VALUE(MediaEngineTimeBetweenPingMs, int, 10000 /**< default value */);      /**< time in milliseconds */
    CREATE_DB_VALUE(MediaEnginePingTimeoutMs, int, 15000 /**< default value */);      /**< time in milliseconds */

    /*Roadmap:13026_DynamicSorting*/
    CREATE_DB_VALUE(DBSortUseUCAOn, int, 0 /**< default value */);
    CREATE_DB_VALUE(DBSortLanguage, string, "en_US" /**< default value */);     /*Ref:http://en.wikipedia.org/wiki/List_of_ISO_639-1_codes#Partial_ISO_639_table*/

    /*Wait time for the Playback when NEXT Command received (Roadmap:14001_MultipleNext)*/
    CREATE_DB_VALUE(MultipleNextWaitTime, long, 1000);  /**< default 0 means feature is disabled !Attention: Timer values <1000 will not work*/
    CREATE_DB_VALUE(MaximumAlbumArtSizeInMB, int, 4/*default value in MB*/); /** Maximum size of album art in MB (4MB=4194304 bytes)  */
    CREATE_DB_VALUE(AlbumArtFromFromSeparateFile, int, 0 /**< default value */);
    CREATE_DB_VALUE(ScaleUpAlbumArt, int, 1);

    /*Roadmap:15009_CoverArtFlow*/
    CREATE_DB_VALUE(AlbumArtThumbnail, int, 0); /**< Feature switch to collect and offer album art thumbnails */
    CREATE_DB_VALUE(ThumbnailWidth, int, 20);   /**< Width in pixel of converted album art thumbnail */
    CREATE_DB_VALUE(ThumbnailHeight, int, 20);  /**< Height in pixel of converted album art thumbnail */
    CREATE_DB_VALUE(ThumbnailMimeType, int, MMT_PNG); /**< Mime type of converted album art thumbnail */
    CREATE_DB_VALUE(CoverArtFlow, int, 0);      /**< Feature switch to collect and offer album arts for cover art flow */
    CREATE_DB_VALUE(CoverArtWidth, int, 280);   /**< Width in pixel of converted cover art */
    CREATE_DB_VALUE(CoverArtHeight, int, 280);  /**< Height in pixel of converted cover art */
    CREATE_DB_VALUE(CoverArtMimeType, int, MMT_PNG);  /**< Mime type of converted cover art */
    CREATE_DB_VALUE(CoverArtDirPath, string, "/var/opt/bosch/dynamic/media/coverart/"); /**< Path of directory of cover arts on the flash */
    CREATE_DB_VALUE(LimitCoverArtMemoryOnFlashKB, int, 314572800);          /**< Max memory on flash used for cover arts in KB (314572800 = 300GB)*/
    CREATE_DB_VALUE(LimitCovertArtInfoToListTypes, int, 0);
    CREATE_DB_VALUE(LimitCovertArtInfoToListType_01, int, LTY_END_OF_EXTERNAL_LIST_TYPES);

    CREATE_DB_VALUE(LimitThumbnailInfoToListTypes, int, 0);
    CREATE_DB_VALUE(LimitThumbnailInfoToListType_01, int, LTY_END_OF_EXTERNAL_LIST_TYPES);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceTypes, int, 3);            /**< Number of registered device types which support album art indexing */
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_01, int, DTY_USB);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_02, int, DTY_SD);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_03, int, DTY_FLASH);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_04, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_05, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_06, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_07, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(AlbumArtIndexingSupportDeviceType_08, int, DTY_UNKNOWN); /* attention: if you add here a line, take care about the method: DataProvider::IsLastModePositionSupported */

    /*Indexer fingerprint timeout*/
    CREATE_DB_VALUE(MonitorFingerPrintCalculation, bool, true) /** To decide whether fingertimer is required in indexer*/
    CREATE_DB_VALUE(IndexerFingerprintTimeoutMS, long, 60000); /** Timeout for fingerprint answer to indexer */
    CREATE_DB_VALUE(SearchForFirstPlayableFile, int, 1 /**< default value */);
    CREATE_DB_VALUE(FirstErrorTimeMS, long, 10000); /**< Time to prevent a second restart in case of a streaming error in ms */

    /*Switches for root daemon usage*/
    //Changing the default value to false for Feature branches
    CREATE_DB_VALUE(RootDaemonDirectCallGMPCommands, int, 0/*default*/); /** use the root daemon commands for GMP as local and direct calls */

    CREATE_DB_VALUE(EngineType, int, 0); /**< engine type, default: 0 = ET_UNDEFINED */
    CREATE_DB_VALUE(OutsideTemperature, int, INVALID_OUTSIDE_TEMPERATURE); /** outside temperature, will be set by Interface function */

    /*MTP reserved file size */
    CREATE_DB_VALUE(MaximumMTPFileSizeInMB, int, 8/*default value in MB*/); /** Maximum size of MTP file in MB  */
    CREATE_DB_VALUE(DBMTPMaxDepthForScan, int, 11 /**< default value */); /*max depth for MTP scan */
    CREATE_DB_VALUE(MaximumMTPVideoFileSizeInMB, int, 10/*default value in MB*/); /** Maximum size of MTP file in MB  */
    CREATE_DB_VALUE(MTPVideoSupport, int, 0/*default value*/); /** Configuration to enable video support in MTP devices */

    /* default MTP folder filter regular expressions*/
    CREATE_DB_VALUE(MTPFileFilterCount, int, 6);
    CREATE_DB_VALUE(MTPFileFilter_01, string, "^Android$");
    CREATE_DB_VALUE(MTPFileFilter_02, string, "^\\.");
    CREATE_DB_VALUE(MTPFileFilter_03, string, "^LOST.DIR$");
    CREATE_DB_VALUE(MTPFileFilter_04, string, "^DCIM$");
    CREATE_DB_VALUE(MTPFileFilter_05, string, "^Ringtones$");
    CREATE_DB_VALUE(MTPFileFilter_06, string, "^Notifications$");

    /*spare entries*/
    CREATE_DB_VALUE(MTPFileFilter_07, string, "^UNUSED$");
    CREATE_DB_VALUE(MTPFileFilter_08, string, "^UNUSED$");

    /*AVRCP < 1.4 support flag*/
    CREATE_DB_VALUE(AVRCPLowerProfileSupported, int, 0 /**< default value */);

    /*Activate BT if playback happens from a SIRI session flags */
    CREATE_DB_VALUE(AutoPlayBTOnRequestviaSiri, int, 0 /**< default value */);
    CREATE_DB_VALUE(SiriSessionTimeOutMS, int, 500 /**< default value */);    /**< timeout for Disabling the SIRI session flags */

    CREATE_DB_VALUE(RemoteActivitySupportForNonAppleDevice, int, 0 /**< default value */);

    /*Roadmap:14017_VideoLastModePosition*/
    CREATE_DB_VALUE(LastModePosDeviceTypes, int, 0);                /* number of registered device types which support last mode position */
    CREATE_DB_VALUE(LastModePosDeviceType_01, int, DTY_USB);
    CREATE_DB_VALUE(LastModePosDeviceType_02, int, DTY_SD);
    CREATE_DB_VALUE(LastModePosDeviceType_03, int, DTY_CDROM);
    CREATE_DB_VALUE(LastModePosDeviceType_04, int, DTY_FLASH);
    CREATE_DB_VALUE(LastModePosDeviceType_05, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(LastModePosDeviceType_06, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(LastModePosDeviceType_07, int, DTY_UNKNOWN);
    CREATE_DB_VALUE(LastModePosDeviceType_08, int, DTY_UNKNOWN);    /* attention: if you add here a line, take care about the method: DataProvider::IsLastModePositionSupported */

    /*Roadmap:14017_VideoLastModePosition*/
    CREATE_DB_VALUE(LastModePosCategoryTypes, int, 1);              /* number of registered category types which support last mode position */
    CREATE_DB_VALUE(LastModePosCategoryType_01, int, CTY_VIDEO);
    CREATE_DB_VALUE(LastModePosCategoryType_02, int, CTY_PODCAST);
    CREATE_DB_VALUE(LastModePosCategoryType_03, int, CTY_AUDIOBOOK);
    CREATE_DB_VALUE(LastModePosCategoryType_04, int, CTY_SONG);     /* attention: if you add here a line, take care about the method: DataProvider::IsLastModePositionSupported */

    /*Roadmap:14017_VideoLastModePosition*/
    CREATE_DB_VALUE(LastModePosAfterNext, int, 0);
    CREATE_DB_VALUE(LastModePosAfterPrev, int, 0);
    CREATE_DB_VALUE(LastModeDeltaTime, int, 5000);
    CREATE_DB_VALUE(LastModeDeltaTimeZero, int, 5000);

    /*Roadmap:15004_CodesetConversionForNissan*/
    CREATE_DB_VALUE(CodeSetConversionSupported, int, 0 /**< default value */); /*This feature will be enabled for A-IVI Project */
    CREATE_DB_VALUE(CodeSetConvLogFilePath,string, "/tmp/codesetConversionTest.txt" /**< default value */);/*For Metadata analysis (in HEX form)*/
    /* AutoPlay Roadmap topic:*/
    CREATE_DB_VALUE(SeparateMediaContent,int,0/**< default value */); /*This feature will be enabled for A-IVI Project */
    CREATE_DB_VALUE(CreateListForVideoSource,int,0/**< default value */);

    /*Roadmap:15008_SCAN --begin*/
    /*This feature will be enabled if Project needs it.Even then,Not All Devices could be scanned.
    Hence,MediaItem scanning possible If the DeviceType also supports*/
    CREATE_DB_VALUE(TrackScanningSupported, int, 0);
    CREATE_DB_VALUE(TrackScanTime, int, 15000); /*MediaItem scanning time */

    /*TRACK SCAN Supported DeviceTypes*/
    CREATE_DB_VALUE(TrackScanSupportedDeviceTypes,  int, 2);
    CREATE_DB_VALUE(TrackScanSupportedDeviceType_1, int, DTY_USB);
    CREATE_DB_VALUE(TrackScanSupportedDeviceType_2, int, DTY_SD);
    /*Roadmap:15008_SCAN --end*/

    CREATE_DB_VALUE(MaxImageSize, int, 10485760); //10 MB

    /* CAFI06-122 Update Device Portnumber in DeviceName */
    CREATE_DB_VALUE(AddPortNumberToDeviceName, int, 0);

    /*For CDDA Real Device (IVI) */
    CREATE_DB_VALUE(CDDAUseRealDevice, int, 0);

    CREATE_DB_VALUE(UseDVDControl, int, 0);
    CREATE_DB_VALUE(DVDRegionCode, int, 0);

    CREATE_DB_VALUE(ActivateBTDeviceOnRemoteActivity, int, 0); /* This feature will be enabled for GM projects */
    CREATE_DB_VALUE(UseArlSrcBtAudio, int, 0); /* This feature will be enabled for PSA and RNAIVI and SUZSLN */
    CREATE_DB_VALUE(UseArlSrcBtAudioForIAPBT, int, 1); /* This feature will be disabled for SUZSLN */

    /*Chek CoverArt in all available frame types */
    CREATE_DB_VALUE(FirstAvailableCoverArtSupport, int, 0); /* This feature will be enabled for AIVI project*/

    CREATE_DB_VALUE(ChangeFromRepeatAllToDefaultRepeatMode, int, 0); /* CMG3G-8071 : If User selects any Item to play via Browsing(other than currentlist) being in RPT_ALL mode,then RepeatMode should fall back to the Default RepeatMode*/
    CREATE_DB_VALUE(CheckActiveDeviceForAVActivation, int, 1) /* This feature will be disabled for GM */
    CREATE_DB_VALUE(BTAudioSourceAsStatic, int, 0); /* This will be enabled for RNAIVI. For RNAIVI BTAudio is a static source.*/

    CREATE_DB_VALUE(UseEvolutionBtStack, int, 0); /*Evolution stack can be dynamically enabled*/
    CREATE_DB_VALUE(BtNowPlayingListSupport, int, 0);
    CREATE_DB_VALUE(BTServiceFilePath, string, "/opt/bosch/connectivity/bluetoothservice/bluetoothservice_out.out"); /*Dynamic switching to Evolution Stack if fusion Stack is not enabled*/
    CREATE_DB_VALUE(IgnoreSpecialCharachters, int, 0); /* This will be enabled for RNAIVI. For Ignoring Special Charachters like 'A','An','The' from metadata.See Sorlib*/
    CREATE_DB_VALUE(InformVideoFileAsAudio, int, 0); /* This will inform Video files as audio for PSA. PSA has a requirement to play Video files with audio data and also shoe the same in song list*/

    //To read the Firmware version form Registry
    CREATE_DB_VALUE(FWVersionRegPath, string, "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS");/*The path to read Firmware version from the registry */
    CREATE_DB_VALUE(FWVersionRegKey, string, "BUILDVERSION_CUSTVERSTRING");/*The Registry key to read Firmware version from the registry */

    /* Check and set Apple device Repeat Mode to RPT_LIST if device setting is RPT_NONE */
    CREATE_DB_VALUE(SetDefaultRepeatModeToRPTListAppleDevice, int, 0);

    //CarPlay Wired (USB) support
    CREATE_DB_VALUE(iPodControlWiredCarPlayEnabled, int, 0);

    //CARPLAY WIFI: CarPlay Wifi support.
    CREATE_DB_VALUE(iPodControlCarPlayWifiEnabled, int, 0);

    //Config value for using iAP2 command SetNowPlayingInfo(index)
    CREATE_DB_VALUE(iPodControlIAP2UseSetNowPlayingInfo, int, 1);

    //Config value for using iAP2 command SwapCalls
    CREATE_DB_VALUE(iPodControlIAP2UseSwapCalls, int, 0);

    /* Roadmap:15027_Adjust Video Brightness */
    /* this will switch off/on the set video property feature */
    CREATE_DB_VALUE(SetVideoProperty, int, 0);

    /* NCG3D-66863 Adjustment Display settings for DVD Video playback */
    /* this will switch off/on the set video property feature */
    CREATE_DB_VALUE(SetDVDVideoProperty, int, 0);

    /* Min and Max values for each of video property type as suggested by ADIT plugin */
    /*Brightness range*/
    CREATE_DB_VALUE(MinBrightness, int, 0);
    CREATE_DB_VALUE(MaxBrightness, int, 200);

    /*Hue range*/
    CREATE_DB_VALUE(MinHue, int, -180);
    CREATE_DB_VALUE(MaxHue, int, 180);

    /*Saturation range*/
    CREATE_DB_VALUE(MinSaturation, int, 0);
    CREATE_DB_VALUE(MaxSaturation, int, 200);

    /*Contrast range*/
    CREATE_DB_VALUE(MinContrast, int, 0);
    CREATE_DB_VALUE(MaxContrast, int, 200);

    /*Brightness offset range*/
    CREATE_DB_VALUE(MinBrightnessOffset, int, -100);
    CREATE_DB_VALUE(MaxBrightnessOffset, int, 100);

    /*Hue offset range*/
    CREATE_DB_VALUE(MinHueOffset, int, -180);
    CREATE_DB_VALUE(MaxhueOffset, int, 180);

    /*Saturation offset range*/
    CREATE_DB_VALUE(MinSaturationOffset, int, 0);
    CREATE_DB_VALUE(MaxSaturationOffset, int, 200);

    /*Default video property levels */
    CREATE_DB_VALUE(DefaultVideoBrightness, int, 100);
    CREATE_DB_VALUE(DefaultVideoHue, int, 0);
    CREATE_DB_VALUE(DefaultVideoSaturation, int, 100);
    CREATE_DB_VALUE(DefaultVideoContrast, int, 100);
    CREATE_DB_VALUE(DefaultVideoBrightnessOffset, int, 0);
    CREATE_DB_VALUE(DefaultVideoHueOffset, int, 0);
    CREATE_DB_VALUE(DefaultVideoSaturationOffset, int, 0);
    //Config value for using playbackspeed values
    CREATE_DB_VALUE(MaximumPlaybackspeed, int, PBK_SPEED_32X );
    CREATE_DB_VALUE(MinimumPlaybackspeed, int, PBK_SPEED_0_2X);
    CREATE_DB_VALUE(DefaultPlaybackspeed, int, PBK_SPEED_1X);

    //>--Roadmap 16003 : 'CD Ripping With Gracenote'
    CREATE_DB_VALUE(CDRippingSupport, int, 0);
    CREATE_DB_VALUE(FormatOfRippedFile, string, "mp3");
    CREATE_DB_VALUE(RippedFilePath, string, "/var/opt/bosch/dynamic/media/CDRipping/");
    CREATE_DB_VALUE(CDRippingEncodingQuality,  int, 96);//default Encoding Quality is FM
    CREATE_DB_VALUE(PlaylistMulipleEntriesAllowed, int, 0 ); //Roadmap:13022: Internal playlist storage path
    //<--Roadmap 16003 : 'CD Ripping With Gracenote'

    //Add Device Syspath to DeviceInfo structure and send to clients in MediaplayerDeviceConnections-  CRQ  CMG3G-12660
    CREATE_DB_VALUE(YearMetadataSupported,int,1);
    CREATE_DB_VALUE(AddDeviceSysPath, int, 0);
    CREATE_DB_VALUE(KeyboardWithHashAtEnd, int, 0);
    CREATE_DB_VALUE(LimitRippingMemoryOnFlashKB,int, 7340032); // 7 GB
    CREATE_DB_VALUE(MinimumMemoryRequiredForRippingKB, int, 10240); // 10 MB
    CREATE_DB_VALUE(ReconnectMusicBoxTimeout, int, 10000); // 10 MB
    CREATE_DB_VALUE(UseResumeinfoFromDB,int,1);
    CREATE_DB_VALUE(PlayOnUserRequestWhenListFinished ,int,0);
    CREATE_DB_VALUE(FirstItemFromShuffleList ,int,0);//Suzuki CRQ 296694 and Bug 265009

    //>--Roadmap CMG3G-10221 : 'Scene Recorder '
    CREATE_DB_VALUE(EnableSRdevice,int,0);
    //<--Roadmap CMG3G-10221 : 'Scene Recorder ''

    //>--Roadmap 16015 : 'Internal DVD drive support'
    CREATE_DB_VALUE(V4lInterfaceName, string, "ipu1_csi0 capture");
    CREATE_DB_VALUE(V4lInputName, string, "ADV7182 YPbPr on AIN1/2/3");
    CREATE_DB_VALUE(SetFPGARegistryforDVD, int, 0);
    CREATE_DB_VALUE(V4lCaptureWidth, int, -1);
    CREATE_DB_VALUE(V4lCaptureHeight, int, -1);
    CREATE_DB_VALUE(V4lFPSNumerator, int, -1);
    CREATE_DB_VALUE(V4lFPSDenominator, int, -1);
    CREATE_DB_VALUE(V4lMotion, int, -1);
    CREATE_DB_VALUE(V4lPixFormat, int, 10);

    //<--Roadmap 16015 : 'Internal DVD drive support'

    //>>-- VideoThumbnail Config
    CREATE_DB_VALUE(VideoThumbnail, int, 0);
    CREATE_DB_VALUE(VideoThumbnailDuration, int, 87);/**< Percentage to get Seeking point for the video file> */
    CREATE_DB_VALUE(VideoThumbnailDirPath, string, "/var/opt/bosch/dynamic/media/videothumbnail/");
    CREATE_DB_VALUE(VideoThumbnailWidth, int, 280);   /**< Width in pixel of converted album art thumbnail */
    CREATE_DB_VALUE(VideoThumbnailHeight, int, 280);  /**< Height in pixel of converted album art thumbnail */
    CREATE_DB_VALUE(VideoThumbnailMimeType, int, MMT_JPG); /**< Mime type of converted album art thumbnail */
    //<<-- VideoThumbnail Config

    //>--Roadmap 17001 : 'Personalization'
    CREATE_DB_VALUE(PersonalizationSupported, int, 0 /**< default value */);
    //<--Roadmap 17001 : 'Personalization'

    //AbsoluteVolumeSupport for BT devices
    CREATE_DB_VALUE(AbsoluteVolumeSupportFeatureBT, int, 0 /**< default value */);
    CREATE_DB_VALUE(AbsoluteVolumeSupportForBTDevice, int, 0 /**< default value */);

    //>--Roadmap 16003 : 'CD Ripping With Gracenote'
    CREATE_DB_VALUE(YomiMetadataSupport, int, 0);
    //<--Roadmap 16003 : 'CD Ripping With Gracenote'

    /*Enable this only for sumilation configuration*/
    CREATE_DB_VALUE(UnitTestEnabled, int, 0);

    //Set value to enable highlighting the nowplaying song in the Current list of Bluetooth Folder browse.
    CREATE_DB_VALUE(EnableBTNowPlayingHighlight, int, 0);

    //set Niceness property on thread to reduce CPU load
    CREATE_DB_VALUE(ControlCPUThtreadLoad, int, 0/**< default value */);
    CREATE_DB_VALUE(IndexerThreadNiceValue, int, 19/**< default value */);

    //set Niceness property on thread to reduce CPU load for reading files for fingerprint Calculation
    CREATE_DB_VALUE(FingerprintThreadNiceValue, int, 19/**< default value */);

    // Album art extraction from BT device - AVRCP1.6 - for NowPlaying song
    CREATE_DB_VALUE(EnableBTNowPlayingCoverArt, int, 1);

    CREATE_DB_VALUE(iAP2BTDisconnectOniAP2USBConnection, int, 1);
    CREATE_DB_VALUE(SPMStandByMode, int, 0);

    /* After discussion with Christian ,on source from any other source to ET source like USB/BT/IPOD we use 20ms RAMP*/

    CREATE_DB_VALUE(RampvalueonSourceChange, int, 20);
};

/*Assign the unknown text configured in DataProvider if the metadata string is empty*/
void AssignUnknownText(char *p_ctext);

#endif /*DATAPROVIDER_H_*/

/** @} */
