/*lint -save -e1773 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#ifdef TARGET_BUILD
#include "trcGenProj/Header/FileSystemVTTest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_TEST
#endif
#endif

#include <stdlib.h>
#include <errno.h>
#include <memory.h>
#include <sqlite3.h>
#include <fcntl.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#include <vector>

using namespace std;

#include "FileSystemVTTest.h"
#include "TypeDefinitions.h"
#include <FunctionTracer.h>

static sqlite3 *mDBHandle;					// data base handle for all tests
static char mountPoint[1024];				// common mountpoint for all tests

void FileSystemVTTest::setUp()
{
	pthread_mutex_init(&mutex,NULL);
	pthread_cond_init(&condition,NULL);
	ticks.begin();
}

void FileSystemVTTest::tearDown()
{
	ticks.elapsed();
	pthread_mutex_destroy(&mutex);
	pthread_cond_destroy(&condition);
}

enum {
	SORT = 0,
	DONT_SORT = 1,
	ALL = 2,
	WO_UNKNOWN = 4
};

#define TEST_PRINT 0

/**********************************************************************
 * Test cases
 *********************************************************************/

int FileSystemVTTest::Query(char *deviceMountPoint, char *path, char *comparePath, int flags, vector<char *> expectedNames, vector<int> expectedTypes, int doAssert,int includeSubFolder)
{
    ENTRY_INTERNAL

    char query[1024];
	int res;
	int ncols;
	unsigned int rowCounter;
    sqlite3_stmt *stmt;	
	const char *tail;

	(void)comparePath;
	(void)includeSubFolder;
	/* setup the first query */
	if (flags & DONT_SORT) {
		if (flags & WO_UNKNOWN) {
			sprintf(query, "select mountpoint,path,filename,type,format,rowid from f where mountpoint match '%s' AND path match '%s' AND type < %d;", deviceMountPoint, path, FILE_TYPE_UNKNOWN);
		} else {
			sprintf(query, "select mountpoint,path,filename,type,format,rowid from f where mountpoint match '%s' AND path match '%s';", deviceMountPoint, path);
		}
	} else {
		if (flags & WO_UNKNOWN) {
			sprintf(query, "select mountpoint,path,filename,type,format,rowid from f where mountpoint match '%s' AND path match '%s' AND type < %d order by type, UPPER(filename);", deviceMountPoint, path, FILE_TYPE_UNKNOWN);
		} else {
			sprintf(query, "select mountpoint,path,filename,type,format,rowid from f where mountpoint match '%s' AND path match '%s' order by type, UPPER(filename);", deviceMountPoint, path);
		}
	}
#if 0
	printf("Query='%s'\n", query);
#else
    ETG_TRACE_USR2(("Query='%s'", query));
#endif
	
	/* search for files in mount point */
    res = sqlite3_prepare(mDBHandle, query, (int)strlen_r(query), &stmt, &tail);
    if (doAssert) CPPUNIT_ASSERT(res == SQLITE_OK);
    else if (res) return res;

    /* Iterate over results */
    ncols = sqlite3_column_count(stmt); 
    if (doAssert) CPPUNIT_ASSERT(ncols == 6);

    /* do the first step */
    res = sqlite3_step(stmt);
    if (doAssert) CPPUNIT_ASSERT(res == SQLITE_ROW);
    else if (res != SQLITE_ROW) return res;

    rowCounter = 0;

	/* loop over all results */
    while(res == SQLITE_ROW)
    {
#if 0// test print
		const char* stype = NULL;

		/* get the element type */
		int ntype = sqlite3_column_int(stmt,  3);
		switch (ntype)
		{
			case FILE_TYPE_DIRECTORY:
				stype = "DIR";
				break;
			case FILE_TYPE_AUDIOOBJECT:
				stype = "MEDIAOBJECT";
				break;
			case FILE_TYPE_PLAYLIST:
				stype = "PLAYLIST";
				break;
			case FILE_TYPE_IMAGE:
			    stype = "IMAGE";
			    break;
			default:
				stype = "Unexpected";
				break;
		}

		/* print all other columns */
#if 0
		printf( "mountpoint=%s type=%d/%s path=%s name=%s format=%d rowid=%d\n",
				 sqlite3_column_text(stmt, 0),
				 ntype,
				 stype,
				 sqlite3_column_text(stmt, 1),
				 sqlite3_column_text(stmt, 2),
				 sqlite3_column_int (stmt, 4),
				 sqlite3_column_int (stmt, 5));
#else
        ETG_TRACE_USR2(("mountpoint=%40s type=%d/%40s",
                (char *)sqlite3_column_text(stmt, 0),
                ntype,
                stype));
        ETG_TRACE_USR2(("path=%40s name=%40s format=%d rowid=%d",
                (char *)sqlite3_column_text(stmt, 1),
                (char *)sqlite3_column_text(stmt, 2),
                sqlite3_column_int (stmt, 4),
                sqlite3_column_int (stmt, 5)));
#endif
#endif
#if 1
		/* compare result with expectations */
		if (rowCounter < expectedNames.size()) {
		    if (doAssert) CPPUNIT_ASSERT(strcmp((char *)sqlite3_column_text(stmt,  0), deviceMountPoint) == 0); // check the mountpoint
#if 0
            printf("(char *)sqlite3_column_text(stmt, 1)=%s\n", (char *)sqlite3_column_text(stmt, 1));
#else
            ETG_TRACE_USR2(("(char *)sqlite3_column_text(stmt, 1)=%s", (char *)sqlite3_column_text(stmt, 1)));
#endif
		    //if (doAssert) CPPUNIT_ASSERT(strcmp((char *)sqlite3_column_text(stmt,  1), comparePath) == 0); // check the path
		    if (doAssert) CPPUNIT_ASSERT(strcmp((char *)sqlite3_column_text(stmt,  2), expectedNames[rowCounter]) == 0); // check the name
		    if (doAssert) CPPUNIT_ASSERT(sqlite3_column_int(stmt, 3) == expectedTypes[rowCounter]); // check the type
		} else {
		    if (doAssert) CPPUNIT_ASSERT(rowCounter < expectedNames.size()); // check number of expected items against number of read ones
		}
#endif
		/* next row */
        res = sqlite3_step(stmt);
        rowCounter++;
    }

    if (doAssert) CPPUNIT_ASSERT(rowCounter == expectedNames.size()); // check number of expected items against number of read ones

	/* finalize the sql statement */
    res = sqlite3_finalize(stmt);
    if (doAssert) CPPUNIT_ASSERT(res == SQLITE_OK);
    else if (res) return res;

    return 0;
}

void FileSystemVTTest::QueryCount(char *deviceMountPoint, char *path, int countDirectories,
		int countMediaObjects, int countPlaylists,int includeSubFolder)
{
    ENTRY_INTERNAL

    char query[1024];
	int res;
	int ncols;
    sqlite3_stmt *stmt;
	const char *tail;
	int testedDirectoires = 0;
	int testedMediaObjects = 0;
	int testedPlaylists = 0;
	int mediaObjectCount = 0;

	(void)includeSubFolder;

	sprintf(query, "select type,COUNT(rowid) from f where mountpoint match '%s' AND path match '%s' group by type order by type;", deviceMountPoint, path);

#if 0
    printf("query: '%s'\n", query);
#else
    ETG_TRACE_USR2(("query: '%s'", query));
#endif

	/* search for files in mount point */
    res = sqlite3_prepare(mDBHandle, query, (int)strlen_r(query), &stmt, &tail);
    CPPUNIT_ASSERT(res == SQLITE_OK);

    /* Iterate over results */
    ncols = sqlite3_column_count(stmt);
    CPPUNIT_ASSERT(ncols == 2);

    /* do the first step */
    res = sqlite3_step(stmt);
    CPPUNIT_ASSERT(res == SQLITE_ROW);

	/* loop over all results */
    while(res == SQLITE_ROW)
    {

#if 0
        printf("type=%d, count=%d\n", sqlite3_column_int(stmt, 0), sqlite3_column_int(stmt, 1));
#else
        ETG_TRACE_USR2(("type=%d, count=%d", sqlite3_column_int(stmt, 0), sqlite3_column_int(stmt, 1)));
#endif
#if 1
		/* compare after type id */
    	switch(sqlite3_column_int (stmt,  0)) {
    	case FILE_TYPE_DIRECTORY:
    		CPPUNIT_ASSERT(sqlite3_column_int (stmt, 1) == countDirectories); // check the count
    		testedDirectoires = 1;
    		break;
    	case FILE_TYPE_AUDIOOBJECT:
    	case FILE_TYPE_VIDEOOBJECT: // add values to one media object count and check it later
    		mediaObjectCount += sqlite3_column_int (stmt, 1);
    		break;
    	case FILE_TYPE_PLAYLIST:
    		CPPUNIT_ASSERT(sqlite3_column_int (stmt, 1) == countPlaylists); // check the count
    		testedPlaylists = 1;
    		break;
    	default:
    	    break;
    	};
#endif
		/* next row */
		res = sqlite3_step(stmt);
	}

    // if media object count, check it
    if (mediaObjectCount) {
		CPPUNIT_ASSERT(mediaObjectCount == countMediaObjects); // check the count
		testedMediaObjects = 1;
    }

    /* check all non tested */
    if (!testedDirectoires) CPPUNIT_ASSERT(countDirectories == 0);
    if (!testedMediaObjects) CPPUNIT_ASSERT(countMediaObjects == 0);
    if (!testedPlaylists) CPPUNIT_ASSERT(countPlaylists == 0);

	/* finalize the sql statement */
    res = sqlite3_finalize(stmt);
    CPPUNIT_ASSERT(res == SQLITE_OK);
}

void FileSystemVTTest::QueryFirstLevel()
{
     ENTRY_TEST

    char path[1024];
	vector<char *> expectedNames;
	vector<int> expectedTypes;

	/* set the path name */
	strcpy(path, "");

    /* set the expected file names */
    expectedNames.push_back((char *)"albumart"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
    expectedNames.push_back((char *)"music"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
    expectedNames.push_back((char *)"playlists"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
    expectedNames.push_back((char *)"video"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
    expectedNames.push_back((char *)"Garbage-Garbage-01-Supervixen.MP3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-02-Queer.MP3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-03-Only_Happy_When_It_Rains.MP3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 4/*countDirectories*/, 3/*countMediaObjects*/, 0/*countPlaylists*/);

    /* do the query */
    Query(mountPoint, path, path, SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QuerySecondLevel()
{
     ENTRY_TEST

	char path[1024];
	vector<char *> expectedNames;
	vector<int> expectedTypes;

	/* set the path name */
	strcpy(path, "music");

    /* set the expected file names */
    expectedNames.push_back((char *)"Alanis_Morissette"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
	expectedNames.push_back((char *)"Alan_Parsons_Project"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
	expectedNames.push_back((char *)"Apoptygma_Berzerk"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
	expectedNames.push_back((char *)"Garbage"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
	expectedNames.push_back((char *)"Grönemeyer_-_Anders"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
	expectedNames.push_back((char *)"Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET"); expectedTypes.push_back(FILE_TYPE_DIRECTORY);
    expectedNames.push_back((char *)"wavin_flag.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 6/*countDirectories*/, 1/*countMediaObjects*/, 0/*countPlaylists*/);

	/* do the query */
    Query(mountPoint, path, path, SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QuerySecondLevelVideo()
{
     ENTRY_TEST

	char path[1024];
	vector<char *> expectedNames;
	vector<int> expectedTypes;

	/* set the path name */
	strcpy(path, "video");

    /* set the expected file names */
    expectedNames.push_back((char *)"01_DivX5_medium_MediaClip_01.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"01_DivX6_medium_MediaClip_01.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"01_SIF_MediaClip_01.mp4"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"01_WMV_good_Q_MediaClip_01.wmv"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"01_XviD1.1.2_medium_MediaClip_01.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"128x96.mp4"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"3GP_MPEG4_MPEG4.3gp"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"AVI_DivX5_MP3.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"AVI_XviD_MP3.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"c.mpg"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"DivX5_AVI.avi"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"HD-1080i-25fps-HDMV-PAL-H.264.mpg"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"HD-1080i-25fps-HDMV-PAL-MPEG2.mpg"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"MP4_MPEG4_MPEG4.mp4"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"MPG_MPEG2_MPEG-Layer_2.mpg"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);
    expectedNames.push_back((char *)"QuickTime_H264_MP4-AAC.mov"); expectedTypes.push_back(FILE_TYPE_VIDEOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 16/*countMediaObjects*/, 0/*countPlaylists*/);

	/* do the query */
    Query(mountPoint, path, path, SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryThirdLevel()
{
    ENTRY_TEST

	char path[1024];
	vector<char *> expectedNames;
	vector<int> expectedTypes;

	/* set the path name */
	strcpy(path, "music/Garbage");

    /* set the expected file names */
	expectedNames.push_back((char *)"Garbage-Garbage-01-Supervixen.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-02-Queer.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-03-Only_Happy_When_It_Rains.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-04-As_Heaven_Is_Wide.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-05-Not_My_Idea.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-06-A_Stroke_of_Luck.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-07-Vow.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-08-Stupid_Girl.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-09-Dog_New_Tricks.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-10-My_Lover's_Box.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-11-Fix_Me_Now.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Garbage-Garbage-12-Milk.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 12/*countMediaObjects*/, 0/*countPlaylists*/);

	/* do the query */
    Query(mountPoint, path, path, SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryThirdLevel2()
{
    ENTRY_TEST

    char path[1024];
    vector<char *> expectedNames;
    vector<int> expectedTypes;

    /* set the path name */
    strcpy(path, "music/Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET");

    /* set the expected file names */
    expectedNames.push_back((char *)"00-mando_diao-bring_em_in-retail-2002-evighet.m3u"); expectedTypes.push_back(FILE_TYPE_PLAYLIST);
    expectedNames.push_back((char *)"01-sheepdog-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"02-sweet_ride-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"03-motown_blood-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"04-mr_moon-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"05-the_band-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"06-to_china_with_love-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"07-paralyzed-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"08-pusa-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"09-little_boy_jr-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"10-lady-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"11-bring_em_in-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"12-laurens_cathedral-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"00-mando_diao-bring_em_in-retail-2002-evighet.sfv"); expectedTypes.push_back(FILE_TYPE_UNKNOWN);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 12/*countMediaObjects*/, 1/*countPlaylists*/);

    /* do the query */
    Query(mountPoint, path, path, SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryThirdLevel2WOUnknown()
{
    ENTRY_TEST

    char path[1024];
    vector<char *> expectedNames;
    vector<int> expectedTypes;

    /* set the path name */
    strcpy(path, "music/Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET");

    /* set the expected file names */
    expectedNames.push_back((char *)"00-mando_diao-bring_em_in-retail-2002-evighet.m3u"); expectedTypes.push_back(FILE_TYPE_PLAYLIST);
    expectedNames.push_back((char *)"01-sheepdog-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"02-sweet_ride-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"03-motown_blood-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"04-mr_moon-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"05-the_band-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"06-to_china_with_love-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"07-paralyzed-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"08-pusa-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"09-little_boy_jr-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"10-lady-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"11-bring_em_in-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"12-laurens_cathedral-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 12/*countMediaObjects*/, 1/*countPlaylists*/);

    /* do the query */
    Query(mountPoint, path, path, SORT | WO_UNKNOWN, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistM3U()
{ 
    ENTRY_TEST

	char path[1024];
	char comparePath[1024];
	vector<char *> expectedNames;
	vector<int> expectedTypes;

	/* set the path name which is in fact a filename of a playlist */
	strcpy(path, "/music/Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET/00-mando_diao-bring_em_in-retail-2002-evighet.m3u");
	strcpy(comparePath, "/music/Mando_Diao-Bring_Em_In-RETAIL-2002-EVIGHET");

    /* set the expected file names */
	expectedNames.push_back((char *)"01-sheepdog-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"02-sweet_ride-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"03-motown_blood-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"04-mr_moon-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"05-the_band-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"06-to_china_with_love-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"07-paralyzed-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"08-pusa-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"09-little_boy_jr-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"10-lady-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"11-bring_em_in-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"12-laurens_cathedral-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 12/*countMediaObjects*/, 0/*countPlaylists*/);

	/* do the query */
    Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistConcatenated()
{
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

    /* set the path name which is in fact a filename of a playlist */
    strcpy(path, "/playlists/concatenated/ASX-05Tracks.asx");
    strcpy(comparePath, "/playlists/concatenated");

    /* set the expected file names */
	expectedNames.push_back((char *)"ASX-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"ASX-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

	expectedNames.push_back((char *)"WPL-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"WPL-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"WPL-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

	expectedNames.push_back((char *)"JoshWoodward-Breadcrumbs-01-Swansong.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"JoshWoodward-Breadcrumbs-02-2020.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"JoshWoodward-Breadcrumbs-03-BorderBlaster.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"JoshWoodward-Breadcrumbs-04-PrivateHurricane.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

	expectedNames.push_back((char *)"WPL-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"WPL-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"ASX-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"ASX-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

	expectedNames.push_back((char *)"Galipata-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Galipata-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
	expectedNames.push_back((char *)"Galipata-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

	expectedNames.push_back((char *)"ASX-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 17/*countMediaObjects*/, 0/*countPlaylists*/);

	/* do the query */
    Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistM3UExt()
{
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

     /*set the expected path for the each entries*/
     strcpy(comparePath, "/playlists/M3U");

     /* set the expected file names */
     expectedNames.push_back((char *)"M3U-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track06.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"M3U-Track07.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

     /* set the path name which is in fact a filename of a playlist */
     /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/
     strcpy(path, "/playlists/M3U_Plist_root.m3u");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 7/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 2 : playlist located in subfolder and entries located in upper level folder ***/
     strcpy(path, "/playlists/M3U/M3U_Subfolder/M3U_Plist_Subfolder.m3u");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 7/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 3 : playlist and entries located in the same folder****************************/
     strcpy(path, "/playlists/M3U/M3U_Plist_Samefolder.m3u");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 7/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistASX()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

     /*set the expected path for the each entries*/
     strcpy(comparePath, "/playlists/ASX");

     /* set the expected file names */
     expectedNames.push_back((char *)"ASX-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track06.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track07.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"ASX-Track08.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);


     /* set the path name which is in fact a filename of a playlist */
     /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/
     strcpy(path, "/playlists/ASX_USB_root.ASX");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 8/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 2 : playlist located in subfolder and entries located in upper level folder ***/
     strcpy(path, "/playlists/ASX/ASX_USB_Subfolder/ASX_USB_Subfolder.ASX");
     strcpy(comparePath, "/playlists/ASX/ASX_USB_Subfolder/..");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 8/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 3 : playlist and entries located in the same folder****************************/
     strcpy(path, "/playlists/ASX/ASX_USB_samefolder.ASX");
     strcpy(comparePath, "/playlists/ASX");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 8/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistPLS()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

     /*set the expected path for the each entries*/
     strcpy(comparePath, "/playlists/PLS");

     /* set the expected file names */
     expectedNames.push_back((char *)"PLS-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"PLS-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"PLS-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"PLS-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"PLS-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

     /* set the path name which is in fact a filename of a playlist */
     /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/
     strcpy(path, "/playlists/PLS_USB_root.pls");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 2 : playlist located in subfolder and entries located in upper level folder ***/
     strcpy(path, "/playlists/PLS/Playlistfile_subfolder/PLS_Plist_Subfolder.pls");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 3 : playlist and entries located in the same folder****************************/
     strcpy(path, "/playlists/PLS/PLS-Plist_Samefolder.pls");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistWPL()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

     /*set the expected path for the each entries*/
     strcpy(comparePath, "/playlists/WPL");

     /* set the expected file names */
     expectedNames.push_back((char *)"WPL-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"WPL-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"WPL-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"WPL-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"WPL-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"WPL-Track06.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

     /* set the path name which is in fact a filename of a playlist */
     /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/
     strcpy(path, "/playlists/WPL_Plist_root.wpl");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 6/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 2 : playlist located in subfolder and entries located in upper level folder ***/
     strcpy(path, "/playlists/WPL/WPL_Subfolder/WPL_Plist_Subfolder.wpl");
     strcpy(comparePath, "/playlists/WPL/WPL_Subfolder/..");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 6/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);

     /********************** Case 3 : playlist and entries located in the same folder****************************/
     strcpy(path, "/playlists/WPL/WPL_Plist_Samefolder.wpl");
     strcpy(comparePath, "/playlists/WPL");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 6/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistXSPF()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
     strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

     /*set the expected path for the each entries*/
     strcpy(comparePath, "/playlists/XSPF");

     /* set the expected file names */
     expectedNames.push_back((char *)"XSPF-Track01.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"XSPF-Track02.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"XSPF-Track03.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"XSPF-Track04.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
     expectedNames.push_back((char *)"XSPF-Track05.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

     /* set the path name which is in fact a filename of a playlist */
     /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/
     strcpy(path, "/playlists/XSPF_Plist_root.xspf");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT|WO_UNKNOWN, expectedNames, expectedTypes);

     /********************** Case 2 : playlist located in subfolder and entries located in upper level folder ***/
     strcpy(path, "/playlists/XSPF/XSPF_Subfolder/XSPF_Plist_Subfolder.xspf");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT|WO_UNKNOWN, expectedNames, expectedTypes);

     /********************** Case 3 : playlist and entries located in the same folder****************************/
     strcpy(path, "/playlists/XSPF/XSPF_Plist_Samefolder.xspf");
     /* do a count query */
     QueryCount(mountPoint, path, 0/*countDirectories*/, 5/*countMediaObjects*/, 0/*countPlaylists*/);
     /* do the query */
     Query(mountPoint, path, comparePath, DONT_SORT|WO_UNKNOWN, expectedNames, expectedTypes);
}

void FileSystemVTTest::QueryPlaylistB4S()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[1024];
    char comparePath[1024];
    char mountPoint[1024];

#ifndef TARGET_BUILD
    char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
    strcat(mountPoint, "/Customer/Simulation/CustomControl/stick_3");
    free(pwd);
#else
    strcpy(mountPoint, "/opt/bosch/test/data/GMP/stick_3");
#endif

    /*set the expected path for the each entries*/
    strcpy(comparePath, "/playlists/B4S");
    strcpy(path, "/playlists/B4S_plist_root.b4s");

    /* set the expected file names */
    expectedNames.push_back((char *)"B4S-Track06.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"B4S-Track07.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"B4S-Track08.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"B4S-Track09.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"B4S-Track10.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"B4S-Track11.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    /* set the path name which is in fact a filename of a playlist */
    /********************** Case 1 : playlist located in root and entries located in subfolder  ****************/

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 6/*countMediaObjects*/, 0/*countPlaylists*/);
    /* do the query */
    Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes);
}

void FileSystemVTTest::CreateVT()
{ 
    ENTRY_TEST

	int res;
	char *msg = NULL;

    /* Open an in-memory database */
    res = sqlite3_open(":memory:", &mDBHandle);
    CPPUNIT_ASSERT(res == SQLITE_OK);

    /* Enable loadable modules (DSOs) */
    res = sqlite3_enable_load_extension(mDBHandle, 1);
    CPPUNIT_ASSERT(res == SQLITE_OK);

    /* Load the fs virtual table. */
    res = sqlite3_load_extension(mDBHandle, "libVTFile.so", "sqlite3_extension_init", &msg);
    CPPUNIT_ASSERT(res == SQLITE_OK);
    if (msg) CPPUNIT_FAIL(msg);
	
	res = sqlite3_exec(mDBHandle, "create virtual table f using FileSystemVT", NULL, NULL, &msg);
    CPPUNIT_ASSERT(res == SQLITE_OK);
    if (msg) CPPUNIT_FAIL(msg);

	/* create the mount point name */
#ifndef TARGET_BUILD
	char *pwd = get_current_dir_name();
    strcpy(mountPoint, pwd);
	strcat(mountPoint, "/Customer/Simulation/CustomControl/test");
	free(pwd);
#else
	strcpy(mountPoint, "/opt/bosch/test/data/GMP/test");
#endif
}

void FileSystemVTTest::OpenDir_ReadDir_Performance()
{ 
    ENTRY_TEST

    DIR *dirp;
    //DIR *dirp2;
    struct dirent *entry;
    //struct stat statbuf;
    char baseDir[1024];
    char fileName[1024];
    unsigned int fileCount;
    //int dirCount;
    timeval startTime;
    timeval endTime;
    unsigned long durationUS;
    //int ret;

    /* create the mount point name */
#ifndef TARGET_BUILD
    strcpy(baseDir, get_current_dir_name());
    strcat(baseDir, "/Customer/Simulation/CustomControl/test/music/Garbage");
#else
    strcpy(baseDir, "/opt/bosch/test/data/GMP/test/music/Garbage");
#endif

    chdir(baseDir);

    dirp = opendir(".");
    //dirp = opendir(baseDir);
    CPPUNIT_ASSERT(dirp != NULL);

    fileCount = 0;
    //dirCount = 0;

    // meas the start time
    durationUS = 0;
    gettimeofday(&startTime, NULL);

    while(1) {

        // read one entry from directory
        entry = readdir(dirp);
        if (!entry) break;

        // do a stat on it
        strcpy(fileName, baseDir);
        strcat(fileName, "/");
        strcat(fileName, entry->d_name);

#if 0
        stat(entry->d_name, &statbuf);
        if (S_ISDIR(statbuf.st_mode)) {
            dirCount++;
        }
#endif

#if 0
        dirp2 = opendir(fileName);
        if (dirp2) {
            dirCount++;
            closedir(dirp2);
        }
#endif

#if 0
        ret = chdir(fileName);
        if (!ret) {
            dirCount++;
            chdir(baseDir);
        }
#endif

#if 0
        ret = open(fileName, O_DIRECTORY | O_NOATIME | O_NONBLOCK);
        if (ret != -1) {
            dirCount++;
            close(ret);
        }
#endif

        fileCount++;
    }

    // meas the end time
    gettimeofday(&endTime, NULL);

    if (!fileCount) fileCount = 1;

    // calc the duration and sum it up
    durationUS += (endTime.tv_usec - startTime.tv_usec) + ((endTime.tv_sec - startTime.tv_sec)*1000000L);
#if 0
    printf("perf: OpenDir_ReadDir_Performance: %lf us / fc=%d\n", (double)(durationUS)/fileCount, fileCount);
    //printf("perf: OpenDir_ReadDir_Performance: dirCount=%d\n", dirCount);
#else
    ETG_TRACE_USR2(("perf: OpenDir_ReadDir_Performance: %lf us / fc=%d", ((double)durationUS/fileCount), fileCount));
    //ETG_TRACE_USR2(("perf: OpenDir_ReadDir_Performance: dirCount=%d", dirCount));
#endif
}

void FileSystemVTTest::DeleteVT()
{ 
    ENTRY_TEST

    sqlite3_close(mDBHandle);
}

void FileSystemVTTest::OpenWrongDir()
{ 
    ENTRY_TEST

    char path[1024];
    char comparePath[1024];
    vector<char *> expectedNames;
    vector<int> expectedTypes;
    int res;

    /* set the path name which is in fact a filename of a playlist */
    strcpy(path, "music/does-not-exist");
    strcpy(comparePath, "");

    /* do the query for a not existing directory */
    res = Query(mountPoint, path, comparePath, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    if (res != SQLITE_DONE) {
        //fprintf(stderr, "ERROR: sqlite3_step: %s\n", sqlite3_errmsg(mDBHandle)); fflush(stderr);
    }
    CPPUNIT_ASSERT(res != SQLITE_DONE);

    /* clean up */
    remove("/tmp/my-dir-link");
    remove("/tmp/my-dir");

    /* create a dir, link it, remove target dir and try to do a query on it */
    res = mkdir("/tmp/my-dir", 0777);

    /* create the link */
    char command[1024];
    sprintf(command, "ln -s /tmp/my-dir /tmp/my-dir-link");
    res = system(command);
#if 0
    printf("system('ln -s /tmp/my-dir /tmp/my-dir-link') = %d\n", res);
#else
    ETG_TRACE_USR2(("system('ln -s /tmp/my-dir /tmp/my-dir-link') = %d", res));
#endif

    /* remove the target dir */
    remove("/tmp/my-dir");

    /* do the query for a not existing directory link */
    res = Query((char *)"/tmp", (char *)"my-dir-link", comparePath, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    if (res != SQLITE_ROW && res != SQLITE_DONE) {
        //fprintf(stderr, "ERROR: sqlite3_step: %s\n", sqlite3_errmsg(mDBHandle)); fflush(stderr);
    }
    CPPUNIT_ASSERT(res != SQLITE_DONE);

    /* remove the link */
    res = remove("/tmp/my-dir-link");
    CPPUNIT_ASSERT(res == 0);

    sleep(1);
}

void FileSystemVTTest::QueryIncludeSubFolder()
{ 
    ENTRY_TEST

    vector<char *> expectedNames;
    vector<int> expectedTypes;

    char path[256];
    strcpy(path, "recursive:/music/");

    expectedNames.push_back((char *)"00-mando_diao-bring_em_in-retail-2002-evighet.m3u"); expectedTypes.push_back(FILE_TYPE_PLAYLIST);

    expectedNames.push_back((char *)"01-sheepdog-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"02-sweet_ride-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"03-motown_blood-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"04-mr_moon-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"05-the_band-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"06-to_china_with_love-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"07-paralyzed-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"08-pusa-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"09-little_boy_jr-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"10-lady-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"11-bring_em_in-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"12-laurens_cathedral-evighet.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    expectedNames.push_back((char *)"Garbage-Garbage-01-Supervixen.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-02-Queer.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-03-Only_Happy_When_It_Rains.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-04-As_Heaven_Is_Wide.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-05-Not_My_Idea.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-06-A_Stroke_of_Luck.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-07-Vow.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-08-Stupid_Girl.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-09-Dog_New_Tricks.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-10-My_Lover's_Box.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-11-Fix_Me_Now.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Garbage-Garbage-12-Milk.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    expectedNames.push_back((char *)"Herbert_Grönemeyer_-_Bleibt_Alles_Anders.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Herbert_Grönemeyer_-_Fanatisch.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);
    expectedNames.push_back((char *)"Herbert_Grönemeyer_-_Nach_Mir.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    expectedNames.push_back((char *)"wavin_flag.mp3"); expectedTypes.push_back(FILE_TYPE_AUDIOOBJECT);

    expectedNames.push_back((char *)"bleibtallesanders.jpg"); expectedTypes.push_back(FILE_TYPE_IMAGE);

    /* do a count query */
    QueryCount(mountPoint, path, 0/*countDirectories*/, 28/*countMediaObjects*/, 1/*countPlaylists*/, 1/*includeSubFolder*/);

    /* do a content query */
    Query(mountPoint, path, path, SORT | WO_UNKNOWN, expectedNames, expectedTypes, 1/*doAssert*/, 1/*includeSubFolder*/);
}

#if 0 // not used
void FileSystemVTTest::VTCacheTest()
{
	// used only for validation during developement test
    ENTRY
    ENTRY_TEST

    extern void FSCacheReset(void);
    //extern bool FSCacheDelete(const char *path);
    extern int FSCacheLog(const char* logfile);
    extern int FSCacheGetSize(void);

    int size = FSCacheGetSize();
#if 0
    printf("FSCacheGetSize : %d\n",size);
#else
    ETG_TRACE_USR2(("FSCacheGetSize : %d",size));
#endif
    FSCacheReset();
#if 0
    printf("FSCacheGetSize after reset: %d\n",FSCacheGetSize());
#else
    ETG_TRACE_USR2(("FSCacheGetSize after reset: %d",FSCacheGetSize()));
#endif

    strcpy(mountPoint, "/usr");

    /* do the query */
    char path[1024];
    vector<char *> expectedNames;
    vector<int> expectedTypes;

    strcpy(path, "/include");
    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
#if 0
    printf("FSCacheGetSize after query: %d\n",FSCacheGetSize());
#else
    ETG_TRACE_USR2(("FSCacheGetSize after query: %d",FSCacheGetSize()));
#endif
    FSCacheLog("/tmp/query1");

    strcpy(path, "/lib");
    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    size = FSCacheGetSize();
#if 0
    printf("FSCacheGetSize after query: %d\n",size);
#else
    ETG_TRACE_USR2(("FSCacheGetSize after query: %d",size));
#endif
    FSCacheLog("/tmp/query2");

    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    CPPUNIT_ASSERT(size == FSCacheGetSize());

    strcpy(path, "/bin");
    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    size = FSCacheGetSize();
#if 0
    printf("FSCacheGetSize after query: %d\n",size);
#else
    ETG_TRACE_USR2(("FSCacheGetSize after query: %d",size));
#endif
    FSCacheLog("/tmp/query3");

    strcpy(path, "/share/doc");
    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    size = FSCacheGetSize();
#if 0
    printf("FSCacheGetSize after query: %d\n",size);
#else
    ETG_TRACE_USR2(("FSCacheGetSize after query: %d",size));
#endif
    FSCacheLog("/tmp/query4");

    strcpy(mountPoint, "/home/tritonsu");
    strcpy(path, "/test");
    Query(mountPoint, path, path, DONT_SORT, expectedNames, expectedTypes, 0/*doAssert*/);
    size = FSCacheGetSize();
#if 0
    printf("FSCacheGetSize after query: %d\n",size);
#else
    ETG_TRACE_USR2(("FSCacheGetSize after query: %d",size));
#endif
    FSCacheLog("/tmp/query5");
}
#endif // not used
