/*******************************************************************************
*
* FILE:          sortlib_helper.c
*
* SW-COMPONENT:  Sorting Lib
*
* PROJECT:       Bosch-GM-NextGen
*
* DESCRIPTION:   Sorting Lib for Sqlite
*
* AUTHOR:        Nishant Parashar
*
* COPYRIGHT:     (c) 2011 MontaVista Software LLC
*
*******************************************************************************/
#include <stdlib.h>
#include "sortlib_helper.h"

unsigned short* sortlib_calloc_ushort( unsigned long numElements )
{
  return (unsigned short*)calloc( numElements, sizeof(unsigned short) );
}

int sortlib_isdigit( unsigned short c )
{
  if ( SORTLIB_UTF_CODE_0 <= c && SORTLIB_UTF_CODE_9 >= c )
    return TRUE;
  else
    return FALSE;
}

int sortlib_compare_nums( unsigned num1, unsigned num2 )
{
  if( num1 > num2 )
  {
    return SORTLIB_GREATER ;
  }

  if ( num1 < num2 )
  {
    return SORTLIB_LESS;
  }

  return SORTLIB_EQUAL;

}

sortlib_dataArray sortlib_utf8_to_utf16( const char* utf8String, unsigned length )
{
  const char* inputUtfString = utf8String;
  unsigned char bomIsParsed = FALSE;
  unsigned short charToReplace = 0xFFFD;
  int moreNeeded = 0;
  int posOfError = -1;
  unsigned int mainuc = 0;
  unsigned int minimumuc = 0;
  unsigned lenghtOfResult = 0;

  int flag = 3 < length &&
      FALSE == bomIsParsed &&
      0xEF == (unsigned char)inputUtfString[0] &&
      0xBB == (unsigned char)inputUtfString[1] &&
      0xBF == (unsigned char)inputUtfString[2];
  
  if ( flag )
  {
    // BOM starting
    bomIsParsed = TRUE;
    length -= 3;
    inputUtfString += 3;    
  }

  unsigned pos = 0;
  unsigned char tempCh;
  int isInvalid = 0;
  lenghtOfResult = length + 1 + moreNeeded;
  unsigned short* ushortResult = sortlib_calloc_ushort(lenghtOfResult);
  unsigned short *workCh = ushortResult;
  
  for ( pos = 0; pos < length; ++pos )
  {
    tempCh = inputUtfString[pos];
    
    if ( moreNeeded )
    {
      if ( 0x80 == (tempCh & 0xC0) )
      {
        --moreNeeded;
        mainuc = ( mainuc << 6 ) | ( tempCh & 0x3F );        
        if ( !moreNeeded )
        {
          flag =  ( 0xFEFF == mainuc ) && ( FALSE == bomIsParsed ); 
          if (flag)
          {
            // skip to byte order mark
          }
          else if ( mainuc < 0x110000 && mainuc > 0xFFFF )
          {
            *workCh = ( mainuc >> 10 ) + 0xd7C0;
            ++workCh;
            *workCh = ( mainuc % 0x400 ) + 0xDC00;
            ++workCh;
          }
          else if ( (mainuc >= 0xFFFE) ||
                    (mainuc >= 0xD800 && mainuc <= 0xDFFF) || 
                    (mainuc < minimumuc))
          {
            // big error
            ++isInvalid;
            *workCh = charToReplace;
            ++workCh;
            
          }
          else
          {
            *workCh = mainuc;
            ++workCh;
          }
          
          bomIsParsed = TRUE;
          
        }
      }
      else
      {
        // we have an error
        bomIsParsed = TRUE;
        *workCh = charToReplace;
        ++workCh;
        moreNeeded = 0;
        ++isInvalid;        
        pos = posOfError;
      }
    }
    else
    {
      if ( 128 > tempCh )
      {
        bomIsParsed = TRUE;
        *workCh = tempCh;
        ++workCh;        
      }
      else if ( 0xc0 == ( tempCh & 0xe0 ) )
      {
        bomIsParsed = TRUE;
        posOfError = pos;
        minimumuc = 0x80;
        mainuc = tempCh & 0x1f;
        moreNeeded = 1;        
      }
      else if ( 0xE0 == (tempCh & 0xF0) )
      {
        posOfError = pos;
        minimumuc = 0x800;
        mainuc = tempCh & 0x0F;
        moreNeeded = 2;        
      }
      else if ( 0xF0 == (tempCh & 0xf8) )
      {
        bomIsParsed = TRUE;
        minimumuc = 0x10000;
        posOfError = pos;
        mainuc = tempCh & 0x07;
        moreNeeded = 3;        
      }
      else
      {
        // we have an error
        bomIsParsed = TRUE;
        ++isInvalid;
        *workCh = charToReplace;
        ++workCh;        
      }
    }
  } //lint -e850

  if ( 0 < moreNeeded )
  {
    // this UTF sequence is unterminated 
    for ( pos = posOfError; pos < length; ++pos)
    {
      ++isInvalid;
      *workCh = charToReplace;
      ++workCh;      
    }
  }

  sortlib_dataArray d;
  d.len = workCh - ushortResult;

  if ( d.len > lenghtOfResult )
  {
    d.len = lenghtOfResult;
  }

  d.data = sortlib_calloc_ushort(d.len+1);

  for ( pos = 0; pos < d.len; ++pos )
  {
    d.data[pos] = ushortResult[pos];
  }

  free(ushortResult);

  return d;
}
