/**************************************************************************//**
* \file      VTFileSystem.cpp
* \ingroup   MediaPlayer
*
*            Implementation of IPOD virtual table.
*			 The virtual table is named "IPODVT"
*            See .h file for description.
*
* \remark    Copyright: 2012, Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark    Scope:     MediaPlayer
* \remark    Authors:
* 			 - Matthias Thömel
*            - CM-AI/PJ-CD6 Xu Benny
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string>

#include "IPODCursor.h"

/*lint -save -e826 */

/* DDL defining the structure of the vtable */
static const char* ddl = "create table vtable (" \
		"mountpoint text, " \
		"LTY integer, " \
		"tag1 integer, " \
		"tag2 integer, " \
		"tag3 integer, " \
		"tag4 integer, " \
		"_limit integer,"
		"_offset integer,"
		"count integer, " \
		"ID integer, " \
		"name text, " \
		"url text, " \
		"uuid text, " \
		"active integer, " \
		"tracknumber integer " \
		") ";

static int VTIPODDestructor(sqlite3_vtab *pVtab) // finished: 100%
{
    tVTIPOD *p = (tVTIPOD*)pVtab; //lint !e826
	
    /* free the virtual table */
    sqlite3_free(p);
	
    return 0;
}

static int VTIPODCreate( sqlite3 *db, void *p_aux, int argc, const char * const* argv, sqlite3_vtab **pp_vt,   char **pzErr ) // finished: 100%
{
    (void)p_aux;
    (void)argc;
    (void)argv;
    (void)pzErr;

	sqlite3_vtab *vTabObject;
	int rc;

	/* allocate the virtual table object */
	vTabObject = (sqlite3_vtab *)sqlite3_malloc(sizeof(sqlite3_vtab));
	if (!vTabObject) return SQLITE_NOMEM;
	memset(vTabObject, 0, sizeof(sqlite3_vtab));

	/* declare the table to the sqlite */
	rc = sqlite3_declare_vtab(db, ddl);
    if(rc != SQLITE_OK)
    {
    	sqlite3_free(vTabObject);
        return SQLITE_ERROR;
    }

    /* set the resulting pointer to new vtab struct */
    *pp_vt = vTabObject;

    return SQLITE_OK;
}

static int VTIPODConnect( sqlite3 *db, void *p_aux, int argc, const char * const *argv, sqlite3_vtab **pp_vt, char **pzErr ) // finished: 100%
{
    return VTIPODCreate(db, p_aux, argc, argv, pp_vt, pzErr);
}

static int VTIPODDisconnect(sqlite3_vtab *pVtab) // finished: 100%
{
    return VTIPODDestructor(pVtab);
}

static int VTIPODDestroy(sqlite3_vtab *pVtab) // finished: 100%
{
    return VTIPODDestructor(pVtab);
}

static int VTIPODOpen(sqlite3_vtab *pVTab, sqlite3_vtab_cursor **pp_cursor) // finished: 100%
{
    tVTIPOD* p_vt         = (tVTIPOD*)pVTab; //lint !e826
    p_vt->base.zErrMsg = NULL;    
	
	tVTIPODCursor *cursor = (tVTIPODCursor*)sqlite3_malloc(sizeof(tVTIPODCursor));
	if (cursor == NULL)	{
		return SQLITE_NOMEM;
	}
	
	/* reset memory */
	memset(cursor, 0, sizeof(tVTIPODCursor));
		
	/* init the cursor */
	cursor->endOfDataset = 0;
	cursor->objectCount_rowID = 0;

	/* create the request/response state machine */
	cursor->rrIPODControl = new(RequestResponseIPODControl);
		
	/* return the pointer to the cursor memory */
    *pp_cursor = (sqlite3_vtab_cursor*)cursor;
	
    return SQLITE_OK;
}

static int VTIPODClose(sqlite3_vtab_cursor *cur) // finished: 100%
{
    tVTIPODCursor *cursor = (tVTIPODCursor*)cur; //lint !e826
	
    /* free the request/response stte machine */
    delete cursor->rrIPODControl;

    /* free cursor memory */
    sqlite3_free(cursor);

    return SQLITE_OK;
}

static int VTIPODEof(sqlite3_vtab_cursor *cur) // finished: 100%
{
    return ((tVTIPODCursor*)cur)->endOfDataset; //lint !e826
}

static int VTIPODNext(sqlite3_vtab_cursor *cur) // finished: 100%
{	
	return IPODCursorNext((tVTIPODCursor *)cur); //lint !e826
}

static int VTIPODColumn(sqlite3_vtab_cursor *cur, sqlite3_context *ctx, int i) // finished: 100%
{
	tVTIPODCursor *cursor = (tVTIPODCursor*)cur; //lint !e826
	//tIPODNode* d = cursor->currentNode;

	//printf("VTIPODColumn: %d/%d\n", cursor->objectCount_rowID, i);
			
	switch (i) {
	case 0: // mountpoint
		sqlite3_result_text(ctx, cursor->IPODMountpoint, strlen_r(cursor->IPODMountpoint), SQLITE_STATIC);
		break;
	case 1: // LTY
		sqlite3_result_int(ctx,  cursor->listType);
		break;
	case 2: // tag1
		sqlite3_result_int(ctx,  cursor->tag1.tag | IPOD_MARKER_BIT);
		break;
	case 3: // tag2
		sqlite3_result_int(ctx,  cursor->tag2.tag | IPOD_MARKER_BIT);
		break;
	case 4: // tag3
		sqlite3_result_int(ctx,  cursor->tag3.tag | IPOD_MARKER_BIT);
		break;
	case 5: // tag4
		sqlite3_result_int(ctx,  cursor->tag4.tag | IPOD_MARKER_BIT);
		break;
	case 6: // _limit
		sqlite3_result_int(ctx,  cursor->limit);
		break;
	case 7: // _offset
		sqlite3_result_int(ctx,  cursor->offset);
		break;
	case 8: // count
		sqlite3_result_int(ctx,  cursor->IPODCount);
		break;
	case 9: // ID
		sqlite3_result_int(ctx,  cursor->IPODID | IPOD_MARKER_BIT);
		break;
	case 10: // name
		sqlite3_result_text(ctx, cursor->IPODName, strlen_r(cursor->IPODName), SQLITE_STATIC);
		break;
	case 11: // url
		sqlite3_result_text(ctx, cursor->IPODURL, strlen_r(cursor->IPODURL), SQLITE_STATIC);
		break;
	case 12: // uuid
		sqlite3_result_text(ctx, cursor->IPODUUID, strlen_r(cursor->IPODUUID), SQLITE_STATIC);
		break;
	case 13: // Active
		sqlite3_result_int(ctx,  cursor->IPODActive);
		break;
	case 14: // TrackNumber
        sqlite3_result_int(ctx,  cursor->IPODTrackNumber);
        break;
	default:
		sqlite3_result_int(ctx, 0);
		break;
	}
	
    return SQLITE_OK;
}

static int VTIPODRowid(sqlite3_vtab_cursor *cur, sqlite_int64 *p_rowid) // finished: 100%
{
    tVTIPODCursor *cursor = (tVTIPODCursor*)cur; //lint !e826
	
    /* Just use the current row count as the rowid. */
    *p_rowid = cursor->objectCount_rowID | IPOD_MARKER_BIT;
	
    return SQLITE_OK;
}

static int VTIPODFilter( sqlite3_vtab_cursor *p_vtc, int idxNum, const char *idxStr, int argc, sqlite3_value **argv ) // finished: 80%
{
    (void)idxNum;
    (void)idxStr;

	tVTIPODCursor *cursor = (tVTIPODCursor*)p_vtc;

	/* set the filter criteria */
	if (argc == 8) {
		strncpy_r(cursor->IPODMountpoint, (const char *)sqlite3_value_text(argv[0]), sizeof(cursor->IPODMountpoint));
		cursor->listType = sqlite3_value_int(argv[1]);
		cursor->tag1.tag = sqlite3_value_int(argv[2]) & IPOD_MARKER_MASK;
		cursor->tag2.tag = sqlite3_value_int(argv[3]) & IPOD_MARKER_MASK;
		cursor->tag3.tag = sqlite3_value_int(argv[4]) & IPOD_MARKER_MASK;
		cursor->tag4.tag = sqlite3_value_int(argv[5]) & IPOD_MARKER_MASK;
		cursor->limit = sqlite3_value_int(argv[6]);
		cursor->offset = sqlite3_value_int(argv[7]);
	} else {
		return SQLITE_ERROR;
	}

	/* reset the control information */
	cursor->endOfDataset = 0;
	cursor->objectCount_rowID = 0;

	/* get the first entry */
	return IPODCursorNext(cursor);
}

/******************************/
static int VTIPODHasConstraint(sqlite3_index_info *p_info, int col, int opmask) // finished: 100%
{
    int i;
    for(i = 0; i < p_info->nConstraint; i++)
    {
        if(p_info->aConstraint[i].iColumn == col)
        {
            /* If specific operations are specified */
            if(opmask != 0)
            {
                /* Check that at least one is satisfied */
                if(p_info->aConstraint[i].op & opmask)
                {
                    return i;
                }
            }
            return i;
        }
    }    
    return -1;
}
static int VTIPODBestIndex(sqlite3_vtab *tab, sqlite3_index_info *pIdxInfo) // finished: 100%
{
    (void)tab;

    int i = 0;
	int ops = SQLITE_INDEX_CONSTRAINT_MATCH | SQLITE_INDEX_CONSTRAINT_EQ;

	/* loop over the elements: LTY, tag1, tag2, tag3, tag4, mountpoint, _limit, _offset */
	for (int col=0; col<8; col++) {

		/* if found in our table, mark them to be transferred to the filter function */
		if((i = VTIPODHasConstraint(pIdxInfo, col, ops)) > -1) {

			/* Then we want the value to be passed to xFilter() */
			pIdxInfo->aConstraintUsage[i].argvIndex = col+1;
		}
	}

    return SQLITE_OK;
}

static void VTIPODMatchFunction(sqlite3_context* ctx, int argc, sqlite3_value** argv) // finished: 100%
{
    (void)argc;
    (void)argv;

    sqlite3_result_int(ctx, 1);
}

static int VTIPODFindFunction( sqlite3_vtab *pVtab, // finished: 100%
                      int nArg,
                      const char *zName,
                      void (**pxFunc)(sqlite3_context*,int,sqlite3_value**),
                      void **ppArg )
{
    (void)pVtab;
    (void)nArg;
    (void)ppArg;

    if(strcmp(zName, "match") == 0)
    {
        *pxFunc = VTIPODMatchFunction;

        return 1;
    }
    return SQLITE_OK;
}

/////////////////////////////////////////////////////////////////////////////////////////////
static sqlite3_module VTIPODModule =
{
    0, /* iVersion */
    VTIPODCreate, /* xCreate - create a vtable */
    VTIPODConnect, /* xConnect - associate a vtable with a connection */
    VTIPODBestIndex, /* xBestIndex - best index */
    VTIPODDisconnect, /* xDisconnect - disassociate a vtable with a connection */
    VTIPODDestroy, /* xDestroy - destroy a vtable */
    VTIPODOpen, /* xOpen - open a cursor */
    VTIPODClose, /* xClose - close a cursor */
    VTIPODFilter, /* xFilter - configure scan constraints */
    VTIPODNext, /* xNext - advance a cursor */
    VTIPODEof, /* xEof - inidicate end of result set*/
    VTIPODColumn, /* xColumn - read data */
    VTIPODRowid, /* xRowid - read data */
    NULL, /* xUpdate - write data */
    NULL, /* xBegin - begin transaction */
    NULL, /* xSync - sync transaction */
    NULL, /* xCommit - commit transaction */
    NULL, /* xRollback - rollback transaction */
    VTIPODFindFunction, /*  xFindFunction - function overloading */
    NULL, /* *xRename */
    NULL, /* *xSavepoint */
    NULL, /* *xRelease */
    NULL  /* *xRollbackTo */
};

int VTIPOD_register(sqlite3 *db) // finished: 100%
{
    return sqlite3_create_module(db, "IPODVT", &VTIPODModule, NULL);
}

