/* -------------------------------------------------------------------------- */
/**
 *   @defgroup peengine peengine.cpp
 *   @ingroup  VTCDDA
 *   @author   Stephan Pieper, 2013
 *
 *   Virtual table CDDA to access /dev/cdaudio.
 */
/* -------------------------------------------------------------------------- */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_CDDA
#ifdef TARGET_BUILD
#include "trcGenProj/Header/VTCDDA.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_VT_CDDA
#endif
#endif

/* -------------------------------------------------------------------------- */

#include "FunctionTracer.h"

/* -------------------------------------------------------------------------- */

//#include <stdio.h>
//#include <stdlib.h>
//#include <memory.h>
//#include <string>

#include "VTCDDA.h"

#include "FastUTF8.h"
#include "Utils.h"
#include "TypeDefinitions.h"

/* -------------------------------------------------------------------------- */

/*lint -save -e1773 */

/* -------------------------------------------------------------------------- */
static char SMName[64];
void SetSMNameVTCDDA(char *name)
{
    strncpy_r(SMName,name,sizeof(SMName));
}

int CDDACursorNext(tVTCDDACursor *p_cur)
{
    /*send read cache entry event to cddacontrol sm */

    ENTRY_INTERNAL

    char msgToSendString[64];
    int res = SQLITE_OK;

    if((strlen_r(SMName) > 0) && (strncmp("DVDControlSM",SMName,sizeof(SMName) ) == 0))
    {
        strncpy_r(OUT msgToSendString, IN SMName, IN sizeof(msgToSendString));
        strncat_r(OUT msgToSendString, IN "::DVD_GET_FOLDER_ITEM", IN sizeof(msgToSendString));
    }
    else
    {
        strncpy_r(OUT msgToSendString, IN "CDDAControlSM", IN sizeof(msgToSendString));
        strncat_r(OUT msgToSendString, IN "::CDDA_GET_FOLDER_ITEM", IN sizeof(msgToSendString));
    }


    tAllParameters parameterString;

    if((NULL != p_cur) && (NULL != p_cur->rrCDDAControl))
    {
        p_cur->rrCDDAControl->isEOF = true;
        p_cur->rrCDDAControl->Marshal(parameterString, sizeof(parameterString) - 1,
                tMountPoint_format tPath_format tIndex_format, p_cur->mountPoint, p_cur->path, p_cur->rowIndex);

        /* do the request and wait for the answer */
        res = p_cur->rrCDDAControl->DoEventAnswer(msgToSendString, parameterString, NULL, CDDA_CTRL_SM_ANSWER_TIMEOUT_MS);

        if (res) { // request did not happen
            return SQLITE_ERROR ;
        }

        if(true == p_cur->rrCDDAControl->isEOF)
        {
            ETG_TRACE_USR3(("End of the query reached"));
            return SQLITE_OK;
        }
       ETG_TRACE_USR3(("Filename received is %s", p_cur->rrCDDAControl->mFileInfo.fileName));
       p_cur->rowIndex ++;
       return SQLITE_OK;
    }
    else
        return SQLITE_ERROR;
}

/* -------------------------------------------------------------------------- */
        
static int VTCDDADestructor(sqlite3_vtab *pVtab) // finished: 100%
{
    tVTCDDA *p = (tVTCDDA*)pVtab; //lint !e826

    if(NULL != p){
        /* free the virtual table */
        sqlite3_free(p);
    }

    return 0;
}

/* -------------------------------------------------------------------------- */

/* DDL defining the structure of the vtable */
static const char* ddl = "create table vtable (" \
        "mountpoint text, " \
        "path text, " \
        "filename text, " \
        "itemid integer, " \
        "type integer, " \
        "format integer, " \
        "metadata1 text, " \
        "metadata2 text, " \
        "metadata3 text, " \
        "metadata4 text, " \
        "playtime integer " \
        ") ";

/* -------------------------------------------------------------------------- */

static int VTCDDACreate( sqlite3 *db, void *p_aux, int argc, const char * const* argv , sqlite3_vtab **pp_vt,   char **pzErr ) // finished: 100%
{
    (void)argc;
    (void)argv;
    (void)p_aux;
    (void)pzErr;

    sqlite3_vtab *vTabObject;
    int rc;

    /* allocate the virtual table object */
    vTabObject = (sqlite3_vtab *)sqlite3_malloc(sizeof(sqlite3_vtab));
    if (!vTabObject) return SQLITE_NOMEM;
    memset(vTabObject, 0, sizeof(sqlite3_vtab));

    /* declare the table to the sqlite */
    rc = sqlite3_declare_vtab(db, ddl);
    if(rc != SQLITE_OK)
    {
        sqlite3_free(vTabObject);
        return SQLITE_ERROR;
    }

    /* set the resulting pointer to new vtab struct */
    *pp_vt = vTabObject;

    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAConnect( sqlite3 *db, void *p_aux, int argc, const char * const *argv, sqlite3_vtab **pp_vt, char **pzErr ) // finished: 100%
{
    return VTCDDACreate(db, p_aux, argc, argv, pp_vt, pzErr);
}

/* -------------------------------------------------------------------------- */

static int VTCDDADisconnect(sqlite3_vtab *pVtab) // finished: 100%
{
    return VTCDDADestructor(pVtab);
}

/* -------------------------------------------------------------------------- */

static int VTCDDADestroy(sqlite3_vtab *pVtab) // finished: 100%
{
    return VTCDDADestructor(pVtab);
}

/* -------------------------------------------------------------------------- */

static int VTCDDAOpen(sqlite3_vtab *pVTab, sqlite3_vtab_cursor **pp_cursor) // finished: 100%
{
    tVTCDDA* p_vt         = (tVTCDDA*)pVTab; //lint !e826
    p_vt->base.zErrMsg = NULL;
    tVTCDDACursor *cursor = (tVTCDDACursor*)sqlite3_malloc(sizeof(tVTCDDACursor));
    if (cursor == NULL) {
        return SQLITE_NOMEM;
    }

    /* reset memory */
    memset(cursor, 0, sizeof(tVTCDDACursor));
    cursor->rrCDDAControl = NULL;
    cursor->rrCDDAControl =(RequestResponseCDDAControl*) new RequestResponseCDDAControl;
    if(NULL == (cursor->rrCDDAControl))
    {
        return SQLITE_NOMEM;
    }
    /* return the pointer to the cursor memory */
    *pp_cursor = (sqlite3_vtab_cursor*)cursor;

    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAClose(sqlite3_vtab_cursor *cur) // finished: 100%
{
    tVTCDDACursor *cursor = (tVTCDDACursor*)cur; //lint !e826

    /* free cursor memory */
    if(cursor)
    {
        delete cursor->rrCDDAControl;
        sqlite3_free(cursor);
    }
    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAEof(sqlite3_vtab_cursor *cur) // finished: 100%
{
    if((NULL != ((tVTCDDACursor*)cur)) && (NULL != ((tVTCDDACursor*)cur)->rrCDDAControl)) //lint !e826
    {
        if(false == ((tVTCDDACursor*)cur)->rrCDDAControl->isEOF) //lint !e826
        {
            return 0 ; // end of result set
        }
    }
    return 1;
}

/* -------------------------------------------------------------------------- */

static int VTCDDANext(sqlite3_vtab_cursor *cur) // finished: 100%
{
    tVTCDDACursor *cursor = (tVTCDDACursor*)cur; //lint !e826
    if((NULL != cursor) && (NULL != cursor->rrCDDAControl))
    {
       return CDDACursorNext(cursor); //lint !e826
    }
    else
        return SQLITE_ERROR;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAColumn(sqlite3_vtab_cursor *cur, sqlite3_context *ctx, int i) // finished: 100%
{
    tVTCDDACursor *cursor = (tVTCDDACursor*)cur; //lint !e826
    if((NULL != cursor) && (NULL != cursor->rrCDDAControl))
    {
        switch (i) {
            case 0: // mountpoint
                sqlite3_result_text(ctx, cursor->mountPoint, strlen_r(cursor->mountPoint), SQLITE_STATIC);
                break;

            case 1: // path
                sqlite3_result_text(ctx, cursor->path, strlen_r(cursor->path), SQLITE_STATIC);
                break;

            case 2: // name
                sqlite3_result_text(ctx, cursor->rrCDDAControl->mFileInfo.fileName, strlen_r(cursor->rrCDDAControl->mFileInfo.fileName), SQLITE_STATIC);
                break;

            case 3: // itemid
                sqlite3_result_int(ctx,  cursor->rrCDDAControl->mFileInfo.trackNumber);
                break;

            case 4: // type
                sqlite3_result_int(ctx,  cursor->rrCDDAControl->mFileInfo.type);
                break;

            case 5: // format
                sqlite3_result_int(ctx,  cursor->rrCDDAControl->mFileInfo.fileFormat);
                break;

            case 6: // metadata1
                sqlite3_result_text(ctx, cursor->rrCDDAControl->mFileInfo.metaData1, strlen_r(cursor->rrCDDAControl->mFileInfo.metaData1), SQLITE_STATIC);
                break;

            case 7: // metadata2
                sqlite3_result_text(ctx, cursor->rrCDDAControl->mFileInfo.metaData2, strlen_r(cursor->rrCDDAControl->mFileInfo.metaData2), SQLITE_STATIC);
                break;

            case 8: // metadata3
                sqlite3_result_text(ctx, cursor->rrCDDAControl->mFileInfo.metaData3, strlen_r(cursor->rrCDDAControl->mFileInfo.metaData3), SQLITE_STATIC);
                break;

            case 9: // metadata4
                sqlite3_result_text(ctx, cursor->rrCDDAControl->mFileInfo.metaData4, strlen_r(cursor->rrCDDAControl->mFileInfo.metaData4), SQLITE_STATIC);
                break;

            case 10: // playtime
                sqlite3_result_int(ctx,  cursor->rrCDDAControl->mFileInfo.playTime);
                break;

            default:
                sqlite3_result_int(ctx, 0);
                break;
        }
        return SQLITE_OK;
    }
    else
        return SQLITE_ERROR;
}

/* -------------------------------------------------------------------------- */

static int VTCDDARowid(sqlite3_vtab_cursor *cur, sqlite_int64 *p_rowid) // finished: 100%
{
    tVTCDDACursor *cursor = (tVTCDDACursor*)cur; //lint !e826

    if(NULL != cursor){
        /* Just use the current row count as the rowid. */
        *p_rowid = cursor->rowIndex;
        return SQLITE_OK;
    }
    else
        return SQLITE_ERROR;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAFilter( sqlite3_vtab_cursor *p_vtc, int idxNum, const char *idxStr, int argc, sqlite3_value **argv ) // finished: 80%
{
    (void)idxNum;
    (void)idxStr;

     tVTCDDACursor *cursor = (tVTCDDACursor*)p_vtc; //lint !e826

    if((NULL != cursor) && (NULL != cursor->rrCDDAControl))
    {
        /* set the filter criteria */
        if (argc == 2)
        {
            strncpy_r(cursor->mountPoint, (const char *)sqlite3_value_text(argv[0]), sizeof(cursor->mountPoint));
            strncpy_r(cursor->path, (const char *)sqlite3_value_text(argv[1]), sizeof(cursor->path));
        }
        else
        {
            return SQLITE_ERROR;
        }

        /* reset the control information */
        cursor->rrCDDAControl->isEOF = false;
        cursor->rowIndex = 0;
        return CDDACursorNext(cursor);
    }
    else
        return SQLITE_ERROR;
}

/* -------------------------------------------------------------------------- */

static int VTCDDAHasConstraint(sqlite3_index_info *p_info, int col, int opmask) // finished: 100%
{
    int i;
    for(i = 0; i < p_info->nConstraint; i++)
    {
        if(p_info->aConstraint[i].iColumn == col)
        {
            /* If specific operations are specified */
            if(opmask != 0)
            {
                /* Check that at least one is satisfied */
                if(p_info->aConstraint[i].op & opmask)
                {
                    return i;
                }
            }
            return i;
        }
    }
    return -1;
}

/* -------------------------------------------------------------------------- */

static int VTCDDABestIndex(sqlite3_vtab *tab, sqlite3_index_info *pIdxInfo) // finished: 100%
{
    (void)tab;

    int i = 0;
    int ops = SQLITE_INDEX_CONSTRAINT_MATCH | SQLITE_INDEX_CONSTRAINT_EQ;

    /* loop over the elements: LTY, tag1, tag2, tag3, tag4, mountpoint, _limit, _offset */
    for (int col=0; col<6; col++) {

        /* if found in our table, mark them to be transferred to the filter function */
        if((i = VTCDDAHasConstraint(pIdxInfo, col, ops)) > -1) {

            /* Then we want the value to be passed to xFilter() */
            pIdxInfo->aConstraintUsage[i].argvIndex = col+1;
        }
    }

    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */

static void VTCDDAMatchFunction(sqlite3_context* ctx, int argc, sqlite3_value** argv) // finished: 100%
{
    (void)argc;
    (void)argv;

    sqlite3_result_int(ctx, 1);
}

/* -------------------------------------------------------------------------- */

static int VTCDDAFindFunction( sqlite3_vtab *pVtab, // finished: 100%
                      int nArg,
                      const char *zName,
                      void (**pxFunc)(sqlite3_context*,int,sqlite3_value**),
                      void **ppArg )
{
    (void)nArg;
    (void)pVtab;
    (void)ppArg;

    if(strcmp(zName, "match") == 0)
    {
        *pxFunc = VTCDDAMatchFunction;

        return 1;
    }
    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */

static sqlite3_module VTCDDAModule =
{
    0, /* iVersion */
    VTCDDACreate, /* xCreate - create a vtable */
    VTCDDAConnect, /* xConnect - associate a vtable with a connection */
    VTCDDABestIndex, /* xBestIndex - best index */
    VTCDDADisconnect, /* xDisconnect - disassociate a vtable with a connection */
    VTCDDADestroy, /* xDestroy - destroy a vtable */
    VTCDDAOpen, /* xOpen - open a cursor */
    VTCDDAClose, /* xClose - close a cursor */
    VTCDDAFilter, /* xFilter - configure scan constraints */
    VTCDDANext, /* xNext - advance a cursor */
    VTCDDAEof, /* xEof - inidicate end of result set*/
    VTCDDAColumn, /* xColumn - read data */
    VTCDDARowid, /* xRowid - read data */
    NULL, /* xUpdate - write data */
    NULL, /* xBegin - begin transaction */
    NULL, /* xSync - sync transaction */
    NULL, /* xCommit - commit transaction */
    NULL, /* xRollback - rollback transaction */
    VTCDDAFindFunction, /*  xFindFunction - function overloading */
    NULL, /* *xRename */
    NULL, /* *xSavepoint */
    NULL, /* *xRelease */
    NULL  /* *xRollbackTo */
};

/* -------------------------------------------------------------------------- */

int VTCDDA_register(sqlite3 *db) // finished: 100%
{
    return sqlite3_create_module(db, "CDDAVT", &VTCDDAModule, NULL);
}

/* -------------------------------------------------------------------------- */

extern "C" int sqlite3_extension_init( sqlite3 *db, char **pzErrMsg, const sqlite3_api_routines *pApi ) // finished: 100%
{
    (void)pzErrMsg;

    SQLITE_EXTENSION_INIT2(pApi);

    if(VTCDDA_register(db) != SQLITE_OK)
    {
        return SQLITE_ERROR;
    }
    return SQLITE_OK;
}

/* -------------------------------------------------------------------------- */
