
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/Query.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#endif
#endif

#include "Query.h"
#include "FunctionTracer.h"
#include "ThreadFactory.h"
#include "LocalSPM.h"
#include "TimeTrace.h"

#include <stdarg.h>
#include <cstring>
#include <sys/stat.h>

#define EXTENDED_TRACES 0
#define ENABLE_ENTRY_TRACES 0

static map<tListType, tSQLStatement> mSQLStatements;
static tBoolean mQueriesCreated = 0;

typedef struct {
    int inUse;
    sqlite3 *db;
    ThreadFactory::tThreadID threadID;
    sqlite3_stmt *statement;
    char query[MAX_QUERY_LENGTH];
} tQueryCache;

static vector<tQueryCache> mQueryCache;
static Lock mQueryCacheLock;

#define USE_QUERY_CACHE 1

tResult Query::DoLock()
{
    //ETG_TRACE_USR4(("Query::DoLock: 0x%x", this));
    return mLock.lock();
}

tResult Query::UnLock()
{
    //ETG_TRACE_USR4(("Query::UnLock: 0x%x", this));
    mLock.unlock();
    return MP_NO_ERROR;
}

void Query::CheckFatalError(sqlite3 *db, int sqlite3ResultCode)
{
    int createRecreateFile = 0;

    /* check if the database as returned a massive error */
    if (sqlite3ResultCode == SQLITE_CORRUPT) { // in this case a db must be recreated anyway
        createRecreateFile = 1;
    } else if (sqlite3ResultCode == SQLITE_ERROR) { // this is a general error
        const char *errMsg = sqlite3_errmsg(db);
        if (errMsg) {
            if (strstr(errMsg, "no such table")) { // do only a recreate if a table was missing
                createRecreateFile = 1;
            }
        }
    }

    /* should a recreate file be created due to an fatal db error? */
    if (createRecreateFile) {
        /* write a file onto the disk to issue a data base file delete on next restart of system */
        FILE *fp;
        fp = fopen(Database::mDatabaseRecreateFile, "r");
        /* if no recreate file exists create a new one */
        if (!fp) {
            fp = fopen(Database::mDatabaseRecreateFile, "w");
            ETG_TRACE_FATAL (("created: sqlite3ResultCode=%d, %s", sqlite3ResultCode, Database::mDatabaseRecreateFile));
            ETG_TRACE_ERRMEM(("created: sqlite3ResultCode=%d, %s", sqlite3ResultCode, Database::mDatabaseRecreateFile));
        }
        fclose(fp);
    }
}

tResult Query::Prepare(sqlite3_stmt **statement, sqlite3 *db, const char *query)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res = MP_NO_ERROR;
    ThreadFactory::tThreadID testThreadID;

    if (!db) {
        return MP_ERR_DB_SELECT_FAILED;
    }

    mQueryCacheLock.lock();

    testThreadID = LocalSPM::GetThreadFactory().GetThreadId();
    ETG_TRACE_USR4(("threadID=0x%x", (unsigned int)testThreadID));

    /* look for the query in the cache */
    tBoolean found = false;
    for(unsigned int i=0; i < mQueryCache.size(); i++) {
        if (mQueryCache[i].db == db && mQueryCache[i].inUse == 0 && mQueryCache[i].threadID == testThreadID) {
            if (!strcmp(mQueryCache[i].query, query)) {

                /* found! */
                *statement = mQueryCache[i].statement;
                mQueryCache[i].inUse = 1;

                ETG_TRACE_USR4(("reusing statement: 0x%x, 0x%x, 0x%x, %s",
                        mQueryCache[i].db, (unsigned int)mQueryCache[i].threadID, mQueryCache[i].statement, mQueryCache[i].query));
#if 0
                ETG_TRACE_USR4(("reusing statement: 0x%x, 0x%x, 0x%x, %200s",
                        mQueryCache[i].db, (unsigned int)mQueryCache[i].threadID, mQueryCache[i].statement, mQueryCache[i].query));

                const char *tmpQuery = mQueryCache[i].query;
                unsigned int lenQuery = strlen_r(mQueryCache[i].query);
                if(lenQuery > 200) ETG_TRACE_USR4(("reusing statement: %200s", tmpQuery+200));
                if(lenQuery > 400) ETG_TRACE_USR4(("reusing statement: %200s", tmpQuery+400));
                if(lenQuery > 600) ETG_TRACE_USR4(("reusing statement: %200s", tmpQuery+600));
                if(lenQuery > 800) ETG_TRACE_USR4(("reusing statement: %200s", tmpQuery+800));
#endif
                found = true;
                break;
            }
        }
    }

    if(false == found)
    {
        int queryLength = (int)strlen_r(query);
        if(queryLength > (MAX_QUERY_LENGTH - 1))
        {
            ETG_TRACE_USR1(("Query length = %d exceeds max limit",queryLength));
        }
        /* not found in cache: prepare it */
        res = sqlite3_prepare_v2(db, query, queryLength, statement, NULL);
        if (res != SQLITE_OK) {
            ETG_TRACE_ERR(("Prepare:sqlite3_prepare_v2: %s", sqlite3_errmsg(db)));
            ETG_TRACE_ERR(("Prepare:sqlite3_prepare_v2: %s", query));
            CheckFatalError(db, res);
            res = MP_ERR_DB_SELECT_FAILED;
        }
        else {
            /* create new cache entry */
            tQueryCache qc;
            qc.inUse = 1;
            qc.threadID = testThreadID;
            qc.db = db;
            qc.statement = *statement;
            strncpy_r(qc.query, query, sizeof(qc.query));
            mQueryCache.push_back(qc);

            ETG_TRACE_USR4(("created statement: 0x%x, 0x%x, 0x%x, %s", qc.db, (unsigned int)qc.threadID, *statement, qc.query));
#if 0
            ETG_TRACE_USR4(("created statement: 0x%x, 0x%x, 0x%x, %200s", qc.db, (unsigned int)qc.threadID, *statement, qc.query));

            const char *tmpQuery = query;
            unsigned int lenQuery = strlen_r(query);
            if(lenQuery > 200) ETG_TRACE_USR4(("created statement: %200s", tmpQuery+200));
            if(lenQuery > 400) ETG_TRACE_USR4(("created statement: %200s", tmpQuery+400));
            if(lenQuery > 600) ETG_TRACE_USR4(("created statement: %200s", tmpQuery+600));
            if(lenQuery > 800) ETG_TRACE_USR4(("created statement: %200s", tmpQuery+800));
#endif
            }
    }

    mQueryCacheLock.unlock();
    return res;
}

tResult Query::Finalize(sqlite3_stmt *statement) const
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif

    mQueryCacheLock.lock();

    for(unsigned int i=0; i < mQueryCache.size(); i++) {
        if (mQueryCache[i].statement == statement) {

            sqlite3_reset(statement);

            ETG_TRACE_USR4(("re-new statement: 0x%x, 0x%x, 0x%x, %s",
                    mQueryCache[i].db,
                    (unsigned int)mQueryCache[i].threadID,
                    mQueryCache[i].statement,
                    mQueryCache[i].query));

            mQueryCache[i].inUse = 0;
            break;
        }
    }

    mQueryCacheLock.unlock();

    return MP_NO_ERROR;
}

tResult Query::Finalize(sqlite3 *db)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif

    mQueryCacheLock.lock();

    for(unsigned int i=0; i<mQueryCache.size(); i++) {
        if (mQueryCache[i].db == db) {

            sqlite3_finalize(IN mQueryCache[i].statement);

            ETG_TRACE_USR4(("deleting statement: 0x%x", mQueryCache[i].statement));

            mQueryCache.erase(mQueryCache.begin()+i);
            i--; //lint -e850
        }
    }

    mQueryCacheLock.unlock();

    return MP_NO_ERROR;
}

int Query::GetQueryCacheCount()
{
    mQueryCacheLock.lock();
    int usedEntries = mQueryCache.size();
    mQueryCacheLock.unlock();
    return usedEntries;
}

int Query::QueryLengthForListType(tListType listType)
{
    int len = -1;
    if(mSQLStatements[listType].query)
    {
        len = strlen_r(mSQLStatements[listType].query);
    }
    return len;
}

Query::Query() // finished: 100%
{
    /* init some members */
    mStatement = NULL;
    mSelectString = NULL;
    mDB = LocalSPM::GetDBManager().GetDatabase();

    /* queries are static and common to all instances, so create them only once */
    if (!mQueriesCreated) {

        // include all query definitions
        #include "Queries.h"

        mQueriesCreated = 1;
    }

    /* configure the transaction begin/end lock */
    gTransactionLock.setNotReantrant();

    DoLock();
}

Query::~Query() // finished: 100%
{
    /* finalize the sql statement */
    if (mStatement) {

        mDB->UnRegister(this);

#if USE_QUERY_CACHE
        Finalize(IN mStatement);
#else
        // finalize the query
        sqlite3_finalize(mStatement);
#endif
        mStatement = NULL;
    }

    if (mSelectString) {
        free(mSelectString);
        mSelectString = NULL;
    }

    UnLock();
}

tResult Query::Select(const tListType listType, const tDeviceID deviceID, const char *format, ...) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;
    va_list arguments;

    va_start(arguments, format);

    res = Select(IN listType, IN deviceID, 0, INDEX_NO_LIMIT, IN format, IN arguments);

    va_end(arguments);

    return res;
}

tResult Query::Select(const tListType listType, const tDeviceID deviceID, const tIndex startIndex, const tIndex noOfRows, const char *format, ...) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;
    va_list arguments;

    va_start(arguments, format);

    res = Select(IN listType, IN deviceID, IN startIndex, IN noOfRows, IN format, IN arguments);

    va_end(arguments);

    return res;
}

tResult Query::Select(const tListType listType, const tDeviceID deviceID, const tIndex startIndex, const tIndex noOfRows, const char *format, va_list vl) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    /* prepare the sql statement */
    int noOfBindings = 0;
    int valueIndex;
    tResult res = 0;
    int deviceIdPhraseRemoved = 0;
    int deviceIdPhraseReplaced = 0;
    char *startPtr;

    /* if already a statement is active: finalize that */
    if (mStatement) {
        mDB->UnRegister(this);
#if USE_QUERY_CACHE
        Finalize(IN mStatement);
#else
        res = sqlite3_finalize(mStatement);
        if (res != SQLITE_OK) {
            ETG_TRACE_ERR(("Select:sqlite3_finalize: %s", sqlite3_errmsg(mDB->Handle())));
            return MP_ERR_DB_SELECT_FAILED;
        }
#endif
    }

    /* prepare the query */
    mListType = listType;
    if (!mSQLStatements[listType].query) {
        ETG_TRACE_ERR(("Unsupported ListType=%d", listType));
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }

    if (mSelectString) {
        free(mSelectString);
        mSelectString = NULL;
    }
    mSelectString = strdup(mSQLStatements[listType].query);

    if (NULL == mSelectString) return MP_ERR_DB_SELECT_FAILED;

    /* lowercase the complete select query */
    for (unsigned int i = 0; i<strlen_r(mSelectString); i++) mSelectString[i] = tolower(mSelectString[i]);

    /* modify query according to deviceID == MY_MEDIA */
    if (deviceID == MY_MEDIA) {

        /* find the [main].[mediaobjects] pattern and replace it with [myme].[mediaobjects] */
        startPtr = strstr(mSelectString, "[main].[mediaobjects]");
        if (startPtr) {
            memcpy(startPtr, "[myme]", strlen_r("[myme]"));

            /* is there a second [main] reference? */
            startPtr = strstr(startPtr, "[main].[mediaobjects]");
            if (startPtr) {
                memcpy(startPtr, "[myme]", strlen_r("[myme]"));
            }

            /* remove the (? or deviceid=?) pattern */
            startPtr = strstr(mSelectString, "(? or deviceid");
            if (startPtr) {

                /* wipe out the phrase */
                char *cptr;
                for(cptr=startPtr; *cptr != 0 && *cptr != ')'; cptr++) *cptr = ' ';
                if (*cptr) {
                    *cptr = ' '; // the closing bracket
                }

                /* replace it with a '1' as true */
                memcpy(startPtr, "1", 1);

                /* remember this change for later calculations of place holders */
                deviceIdPhraseRemoved = 1;
            }

            ETG_TRACE_USR3(("select modified: %s", mSelectString));
//            ETG_TRACE_USR3(("select modified: %200s", mSelectString));
//            size_t len = strlen_r(mSelectString);
//            while(200 < len)
//            {
//                ETG_TRACE_USR3(("%200s", mSelectString+200));
//                len = len-200;
//            }
        }
    } else {
        /* replace the (? or deviceid=?) || (? or [main].[mediaobjects].deviceid=?) pattern with deviceid=? */
        startPtr = strstr(mSelectString, "(? or deviceid");
        if (!startPtr) {
            startPtr = strstr(mSelectString, "(? or [main].[mediaobjects].deviceid");
        }
        if (startPtr) {

            /* wipe out the phrase partly */
            char *cptr;
            for(cptr=startPtr; (*cptr != 0) && (*cptr != 'd') && (*cptr != '['); cptr++) *cptr = ' '; // remove '(? or '
            cptr = strstr(cptr, ")");
            if (*cptr) {
                *cptr = ' '; // the closing bracket
            }

            deviceIdPhraseReplaced = 1;
        }

    }



#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN mSelectString);
    if (res) {
        ETG_TRACE_ERR(("Select:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("Select:Prepare: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    //ETG_TRACE_USR4(("Select:sqlite3_prepare_v2: %s", mSelectString));
    res = sqlite3_prepare_v2(mDB->Handle(), mSelectString, (int)strlen_r(mSelectString), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("Select:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("Select:sqlite3_prepare_v2: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#endif

    /* get the number of current bindings */
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* set every open binding to 1 (which enables all "? OR" terms) */
    for (int i=1; i<=noOfBindings - 2 /* the last 2 are limit and offset */; i++) {
        res = sqlite3_bind_int(mStatement, i, 1); /* enables all (? OR ...) expressions to true */
    }

    /* set limit and offset bindings */
    mBindingForLimit = noOfBindings-1; // position of limit binding (always the last-1 one)
    mLimit = noOfRows;
    res = sqlite3_bind_int(mStatement, mBindingForLimit, noOfRows);
    mBindingForOffset = noOfBindings; // position of offset binding (always the last one)
    mStartIndex = startIndex;
    res = sqlite3_bind_int(mStatement, mBindingForOffset, startIndex);

    if (deviceIdPhraseRemoved) {

        /* subtract the limit and offset binding */
        noOfBindings -= 2;

        // start with value index 1
        valueIndex = 1;

    } else {

        if (deviceIdPhraseReplaced) {

            /* the first place holder in the sql statement the deviceID now */
            res = sqlite3_bind_int(mStatement, 1, deviceID);

            /* subtract the deviceid=? and the limit and offset binding */
            noOfBindings -= 3;

            // start with value index 2 (1 is for deviceID)
            valueIndex = 2;

        } else { // code for selects without deviceID selection but with (? or id=?) selection

            /* only if specific id is given */
            if (deviceID != MY_MEDIA) {

                /* the first place holders in the sql statement are for the device id */
                res = sqlite3_bind_int(mStatement, 1, 0);
                res = sqlite3_bind_int(mStatement, 2, deviceID);
            }

            /* subtract the (? or deviceid=? and the limit and offset binding */
            noOfBindings -= 4;

            // start with value index 3 (1 and 2 are for deviceID)
            valueIndex = 3;
        }
    }

    /* do the real user provided bindings */
    const char *fmt = format;
    while(*fmt && noOfBindings > 0) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
            int integer;
            integer = va_arg(vl, int );
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            valueIndex++;
            break;
        case 'l': // long long integer
            long long int ll_integer;
            ll_integer = va_arg(vl, long long int );
            res = sqlite3_bind_int64(mStatement, valueIndex, ll_integer);
            noOfBindings--;
            valueIndex++;
            break;
        case 't': // c-string
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            valueIndex++;
            break;
        case 's': // skip input value
            va_arg(vl, int );
            break;
        case '-': // ignore column binding
        default:
            valueIndex++;
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("Select:sqlite3_bind_(*): error: %d", res));
            ETG_TRACE_ERR(("Select:sqlite3_bind_(*): %s", mSelectString));
        }

        // next char in format */
        fmt++;
    }

    /* set some values for this query */
    mPrevRow = startIndex-1;
    mRow = startIndex;
    mEOF = 0;

    return MP_NO_ERROR;
}

tResult Query::Explain(char *planResult, size_t size) // finished: 100%
{
    //tResult res;
    char queryString[2048];
    sqlite3_stmt *statement;
    int sqlError;
    int noOfBindings;

    /* create the explain query string */
    snprintf(queryString, sizeof(queryString)-1, "EXPLAIN QUERY PLAN %s", mSelectString);

    *planResult = 0;
    strncat_r(planResult, mSelectString, size);
    strncat_r(planResult, "\n", size);

    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(mDB->Handle(), queryString, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(): %s", sqlite3_errmsg(mDB->Handle())));
        return -1;
    }

    /* dummy bind all parameters */
    /* get the number of current bindings */
    noOfBindings = sqlite3_bind_parameter_count(statement);

    /* set every open binding to 1 (which enables all "? OR" terms) */
    for (int i=1; i<=noOfBindings; i++) {
        /*res =*/sqlite3_bind_int(statement, i, 0);
    }

    /* get the resulting lines */
    while(1) {

        /* do the explain */
        sqlError = sqlite3_step(statement);
        if (sqlError == SQLITE_DONE) break;
        if (sqlError != SQLITE_OK && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_step(count): %s", sqlite3_errmsg(mDB->Handle())));
            return -1;
        }

        /* get one text line */
        const unsigned char *cptr;
        cptr = sqlite3_column_text(statement,  3);
        if (cptr == NULL) break;
        strncat_r(planResult, (const char *)cptr, size);
        strncat_r(planResult, "\n", size);
    }

    /* finalize explain query */
    sqlite3_finalize(statement);

    return MP_NO_ERROR;
}

tResult Query::SetLimit(const tIndex limit) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* reset the query */
    res = End();
    if (res) {
        ETG_TRACE_ERR(("SetLimit: error in End: %d/%s", res, errorString(res)));
        return res;
    }

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* bind the new limit value */
    res = sqlite3_bind_int(mStatement, mBindingForLimit, limit);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("SetLimit:sqlite3_bind_int: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        return MP_ERR_DB_UNEXPECTED;
    }

    mLimit = limit;
    mPrevRow = mStartIndex-1;
    mRow = mStartIndex;
    mEOF = 0;

    return MP_NO_ERROR;
}

tResult Query::SetRow(tRowNumber row) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* reset the query */
    res = End();
    if (res) {
        ETG_TRACE_ERR(("SetRow: error in End: %d/%s", res, errorString(res)));
        return res;
    }

    if (!mStatement) {
        ETG_TRACE_ERR(("SetRow:no prepared statement"));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* bind the new offset value */
    res = sqlite3_bind_int(mStatement, mBindingForOffset, row);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("SetRow:sqlite3_bind_int: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* set the new values */
    mPrevRow = row-1;
    mRow = row;
    mStartIndex = row;
    mEOF = 0;

    return MP_NO_ERROR;
}

tResult Query::SetLimitAndRow(const tIndex limit, const tRowNumber row)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* reset the query */
    res = End();
    if (res) {
        ETG_TRACE_ERR(("SetLimitAndRow: error in End: %d/%s", res, errorString(res)));
        return res;
    }

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* bind the new offset value */
    res = sqlite3_bind_int(mStatement, mBindingForOffset, row);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("SetRow:sqlite3_bind_int: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* bind the new limit value */
    res = sqlite3_bind_int(mStatement, mBindingForLimit, limit);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("SetLimit:sqlite3_bind_int: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* set the new values */
    mPrevRow = row-1;
    mRow = row;
    mStartIndex = row;
    mLimit = limit;
    mEOF = 0;

    return MP_NO_ERROR;
}

tResult Query::End()
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;
#if EXTENDED_TRACES
    ETG_TRACE_USR2(("End: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    mDB->UnRegister(IN this);

    /* reset the query */
    res = sqlite3_reset(mStatement);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("End:sqlite3_reset: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* reset the current row to the first defined by start index */
    mPrevRow = mStartIndex-1;
    mRow = mStartIndex;
    mEOF = 0;

    return MP_NO_ERROR;
}

tResult Query::ResetRow() // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif

    /* force a new step to old row */
    mEOF = 0;

    /* set the new row value to minimum */
    return SetRow(IN 0);
}

tResult Query::GetRow(tRowNumber &row) const
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    row = mRow;
    return MP_NO_ERROR;
}

tResult Query::Save()
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    mSavedRow = mRow;
    mSavedLimit = mLimit;
    return MP_NO_ERROR;
}

tResult Query::Restore()
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    SetLimit(mSavedLimit);
    SetRow(mSavedRow);

    return MP_NO_ERROR;
}


tResult Query::Peek(const char *format, ...) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Peek: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    va_list vl;
    va_start(vl, format);
    res = Query::GetInternal(0, format, vl);
    va_end(vl);
    return res;
}

tResult Query::Get(const char *format, ...) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Get: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    va_list vl;
    va_start(vl, format);
    res = Query::GetInternal(1, format, vl);
    va_end(vl);
    return res;
}

tResult Query::Get(const int doStep, const char *format, ...) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Get(%d): 0x%x, %s", doStep, this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    va_list vl;
    va_start(vl, format);
    res = Query::GetInternal(doStep, format, vl);
    va_end(vl);
    return res;
}

#define PERF_MEAS 0

void Query::SqlTrace(void *, const char *traceString)
{
    ETG_TRACE_USR4(("SqlTrace: %s", traceString));
}

void Query::SqlProfile(void *, const char *_traceString, sqlite3_uint64 nanosecs)
{
    const char *traceString = _traceString;
    if (!traceString) traceString = "(null)";
    ETG_TRACE_USR4(("SqlProfile(%d ms): %s", (int)(nanosecs / 1000000L), traceString));
}

tResult Query::GetInternal(const int doStep, const char *format, va_list vl) // finished: 100%
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
#if PERF_MEAS
    TimeTrace          ticks("Query::GetInternal");
#endif

    /* end of list? */
    if (mEOF) {
        return MP_ERR_DB_END_OF_LIST;
    }

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

#if PERF_MEAS
    ticks.elapsed();
#endif

#if EXTENDED_TRACES
    ETG_TRACE_USR4(("mRow=%d, mPrevRow=%d, mStartIndex=%d, mLimit=%d", mRow, mPrevRow, mStartIndex, mLimit));
#endif

    /* step if needed */
    if (mRow != mPrevRow) {

        /* slow down the update my media thread to get cpu time for browsing */
        mDB->SlowDownUpdateMyMedia();

#if PERF_MEAS
        sqlite3_trace(mDB->Handle(), SqlTrace, NULL);
        sqlite3_profile(mDB->Handle(), SqlProfile, NULL);
#endif
        /* do a step the get the current row */
        mLastSqlError = sqlite3_step(mStatement);
#if PERF_MEAS
        sqlite3_trace(mDB->Handle(), NULL, NULL);
        sqlite3_profile(mDB->Handle(), NULL, NULL);
#endif

        if (mLastSqlError == SQLITE_DONE) {
            mEOF = 1;
            return MP_ERR_DB_END_OF_LIST;
        }
        if (mLastSqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("GetInternal:sqlite3_step: %s", sqlite3_errmsg(mDB->Handle())));
            ETG_TRACE_ERR(("%s", mSelectString));
            CheckFatalError(mDB->Handle(), mLastSqlError);
            return MP_ERR_DB_SELECT_FAILED;
        }
        mPrevRow = mRow;
    }
#if PERF_MEAS
    ticks.elapsed();
#endif

    /* get the data out of the current sqlite row */
    const char *fmt = format;
    int valueIndex = 0;
    while(*fmt) {

        /* read out the values after format spec */
        switch(*fmt) {
        case 'i': // integer
        {
            int *iPtr;
            iPtr = va_arg(vl, int *);
            *iPtr = sqlite3_column_int(mStatement, valueIndex);
            break;
        }
        case 'l': // long long int
        {
            long long int *lliPtr;
            lliPtr = va_arg(vl, long long int *);
            *lliPtr = sqlite3_column_int64(mStatement, valueIndex);
            break;
        }
        case 'b': // blob
        {
            tBlob *blobPtr;
            blobPtr = va_arg(vl, tBlob *);
            unsigned char *dbBlob;
            dbBlob = (unsigned char *)sqlite3_column_blob(mStatement, valueIndex);
            blobPtr->length = sqlite3_column_bytes(mStatement, valueIndex);
            blobPtr->dataPtr = (unsigned char *)malloc(blobPtr->length);
            memcpy(blobPtr->dataPtr, dbBlob, blobPtr->length);
            break;
        }
        case 't': // c-string (unsave because no size of target memory available)
        case 'T': // c-string (save version because as a second parameter the size of the target memory is provided)
        {
            char *cPtr;
            cPtr = va_arg(vl, char *);
            const unsigned char *dbValue;
            dbValue = sqlite3_column_text(mStatement, valueIndex);

            /* use the save variant? */
            if (*fmt == 'T') {
                size_t size = va_arg(vl, int); // get the new additional value for the size of the target memory
                if (dbValue) {
                    strncpy_r(cPtr, (const char *)dbValue, size);
                } else {
                    strncpy_r(cPtr, "", size);
                }
            } else { // unsave variant */
                if (dbValue) {
                    strcpy(cPtr, (const char *)dbValue);
                } else {
                    strcpy(cPtr, "");
                }
            }
            break;
        }
        case '-': // ignore column
        default:
            break;
        }

        // next char in format */
        fmt++;
        valueIndex++;
    }
#if PERF_MEAS
    ticks.elapsed();
#endif

    /* was this a get with step? */
    if (doStep) {

        /* step one step forward for the next Get */
        mRow++;
    }

    return MP_NO_ERROR;
}

tResult Query::Insert(const tListType listType)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    int res;

    /* prepare the sql statement */
    const char *query;

    /* if already a statement is active: finalize that */
    if (mStatement) {
        mDB->UnRegister(this);
#if USE_QUERY_CACHE
        Finalize(IN mStatement);
#else
        res = sqlite3_finalize(mStatement);
        if (res != SQLITE_OK) {
            ETG_TRACE_ERR(("Insert:sqlite3_finalize: %s", sqlite3_errmsg(mDB->Handle())));
            return MP_ERR_DB_INSERT_FAILED;
        }
#endif
    }

    /* prepare the query */
    mListType = listType;
    query = mSQLStatements[listType].query;
    if (!query) {
        ETG_TRACE_ERR(("Unsupported ListType=%d", listType));
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }
#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN query);
    if (res) {
        ETG_TRACE_ERR(("Insert:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("Insert:Prepare: %s", query));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    res = sqlite3_prepare_v2(mDB->Handle(), query, (int)strlen_r(query), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("Insert:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        return MP_ERR_DB_INSERT_FAILED;
    }
#endif
    return MP_NO_ERROR;
}

tResult Query::Update(const tListType listType)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    int res;

    /* prepare the sql statement */
    const char *query;

    /* if already a statement is active: finalize that */
    if (mStatement) {
        mDB->UnRegister(this);
#if USE_QUERY_CACHE
        Finalize(IN mStatement);
#else
        res = sqlite3_finalize(mStatement);
        if (res != SQLITE_OK) {
            ETG_TRACE_ERR(("Update:sqlite3_finalize: %s", sqlite3_errmsg(mDB->Handle())));
            return MP_ERR_DB_UPDATE_FAILED;
        }
#endif
    }

    /* prepare the query */
    mListType = listType;
    query = mSQLStatements[listType].query;
    if (!query) {
        ETG_TRACE_ERR(("Unsupported ListType=%d", listType));
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }
#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN query);
    if (res) {
        ETG_TRACE_ERR(("Update:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("Update:Prepare: %s", query));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    res = sqlite3_prepare_v2(mDB->Handle(), query, (int)strlen_r(query), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("Update:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        return MP_ERR_DB_UPDATE_FAILED;
    }
#endif
    return MP_NO_ERROR;
}

tResult Query::Delete(const tListType listType)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    int res;

    /* prepare the sql statement */
    const char *query;

    /* if already a statement is active: finalize that */
    if (mStatement) {
        mDB->UnRegister(this);
#if USE_QUERY_CACHE
        Finalize(IN mStatement);
#else
        res = sqlite3_finalize(mStatement);
        if (res != SQLITE_OK) {
            ETG_TRACE_ERR(("Delete:sqlite3_finalize: %s", sqlite3_errmsg(mDB->Handle())));
            return MP_ERR_DB_DELETE_FAILED;
        }
#endif
    }

    /* prepare the query */
    mListType = listType;
    query = mSQLStatements[listType].query;
    if (!query) {
        ETG_TRACE_ERR(("Unsupported ListType=%d", listType));
        return MP_ERR_DB_UNSUPPORTED_LIST_TYPE;
    }
#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN query);
    if (res) {
        ETG_TRACE_ERR(("Delete:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("Delete:Prepare: %s", query));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    res = sqlite3_prepare_v2(mDB->Handle(), query, (int)strlen_r(query), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("Delete:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        return MP_ERR_DB_DELETE_FAILED;
    }
#endif
    return MP_NO_ERROR;
}

tResult Query::Execute()
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /*
     * lock must be done by caller!
     */

    /* slow down the update my media thread to get cpu time for this action */
    mDB->SlowDownUpdateMyMedia();

    /* execute the insert */
    mLastSqlError = sqlite3_step(mStatement);

    res = End();
    if (res) {
        ETG_TRACE_ERR(("Execute: error in End: %d/%s", res, errorString(res)));
        return res;
    }

    if (mLastSqlError != SQLITE_DONE) {
        ETG_TRACE_ERR(("Execute:sqlite3_step: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("%s", mSQLStatements[mListType].query));
        CheckFatalError(mDB->Handle(), mLastSqlError);
        return MP_ERR_DB_EXECUTE_FAILED;
    }

    return MP_NO_ERROR;
}

tResult Query::Put(const char *format, ...)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Put: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* get the number of current bindings */
    int noOfBindings;
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* do the real user provided bindings */
    int valueIndex = 1;
    tResult res = 0;
    va_list vl;
    va_start(vl, format);
    const char *fmt = format;
    while(*fmt && noOfBindings) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
        {
            int integer;
            integer = va_arg(vl, int);
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            break;
        }
        case 'l': // long long integer
        {
            long long int ll_integer;
            ll_integer = va_arg(vl, long long int);
            res = sqlite3_bind_int64(mStatement, valueIndex, ll_integer);
            noOfBindings--;
            break;
        }
        case 'b':  // blob
        {
            tBlob *blobPtr;
            blobPtr = va_arg(vl, tBlob *);
            res = sqlite3_bind_blob(mStatement, valueIndex, blobPtr->dataPtr, blobPtr->length, SQLITE_TRANSIENT);
            noOfBindings--;
            break;
        }
        case 't': // c-string
        {
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            break;
        }
        case '-': // ignore column
        default:
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("Put: Error in bind: %d", res));
        }

        // next char in format */
        fmt++;
        valueIndex++;
    }
    va_end(vl);

    /* execute the insert */
    return Execute();
}

tResult Query::Change(const char *format, ...)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Change: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* get the number of current bindings */
    int noOfBindings;
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* set all bindings to 1 which enables the default values */
    for(int i=1; i<=noOfBindings; i++) {
        sqlite3_bind_int(mStatement, i, 1);
    }

    /* do the real user provided bindings */
    int valueIndex = 1;
    tResult res = 0;
    va_list vl;
    va_start(vl, format);
    const char *fmt = format;
    while(*fmt && noOfBindings) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
            int integer;
            integer = va_arg(vl, int );
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            break;
        case 'l': // long long int
            long long int ll_integer;
            ll_integer = va_arg(vl, long long int );
            res = sqlite3_bind_int64(mStatement, valueIndex, ll_integer);
            noOfBindings--;
            break;
        case 't': // c-string
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            break;
        case '-': // ignore column
        default:
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("Change: Error in bind: %d", res));
        }

        // next char in format */
        fmt++;
        valueIndex++;
    }
    va_end(vl);

    /* execute the change */
    return Execute();
}

tResult Query::Remove(const char *format, ...)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
#if EXTENDED_TRACES
    ETG_TRACE_USR2(("Remove: 0x%x, %s", this, mSQLStatements[mListType].query));
#endif
    mDB->Register(IN this);

    if (!mStatement) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* get the number of current bindings */
    int noOfBindings;
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* set all bindings to 1 which enables the default values */
    for(int i=1; i<=noOfBindings; i++) {
        sqlite3_bind_int(mStatement, i, 1);
    }

    /* do the real user provided bindings */
    int valueIndex = 1;
    tResult res = 0;
    va_list vl;
    va_start(vl, format);
    const char *fmt = format;
    while(*fmt && noOfBindings) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
            int integer;
            integer = va_arg(vl, int );
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            break;
        case 't': // c-string
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            break;
        case '-': // ignore column
        default:
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("Remove: Error in bind: %d", res));
        }

        // next char in format */
        fmt++;
        valueIndex++;
    }
    va_end(vl);

    /* execute the delete */
    return Execute();
}

tResult Query::BeginTransaction(const tListType listType)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;
    char *errmsg;

    /* lock the transaction to prevent parallel transactions */
    gTransactionLock.lock();

    /* set the list type in order to get propper query registry locking */
    mListType = listType;

    /* start the transaction directly */
    res = sqlite3_exec(
            mDB->Handle(),
            "BEGIN TRANSACTION;",
            NULL,
            NULL,
            &errmsg
          );

    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("BeginTransaction: %s", sqlite3_errmsg(mDB->Handle())));
        res = MP_ERR_DB_BEGIN_TRANSACTION_FAILED;
    }
    if(errmsg)
    {
        sqlite3_free(errmsg);
    }
    gTransactionLock.unlock();
    return res;
}

tResult Query::EndTransaction()
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    mLastSqlError = sqlite3_exec(
            mDB->Handle(),
            "COMMIT TRANSACTION;",
            NULL,
            NULL,
            NULL
          );

    /* unlock the transaction */
    gTransactionLock.unlock();

    if (mLastSqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("EndTransaction: %s", sqlite3_errmsg(mDB->Handle())));
        return MP_ERR_DB_END_TRANSACTION_FAILED;
    }

    return MP_NO_ERROR;
}

tResult Query::PositionSelect(const Query *parent, const tDeviceID deviceID, const char *format, ...)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* copy the parents sql command */
    if (mSelectString) {
        free(mSelectString);
        mSelectString = NULL;
    }
    mSelectString = strdup(parent->mSelectString);
    if (!mSelectString) {
        return MP_ERR_DB_UNEXPECTED;
    }

    /* lowercase the complete select query */
    for(unsigned int i=0; i<strlen_r(mSelectString); i++) mSelectString[i] = tolower(mSelectString[i]);

    /* modify the command to get only the id as result column */
    char *cWtritePtr = mSelectString;
    char *cReadPtr = mSelectString;

    enum {
        SELECT_SEARCH,
        ROWID_SEARCH,
        END_OF_COLUMNS_SEARCH,
        OFFSET_SEARCH,
        END
    } state = SELECT_SEARCH;
    int copyIt = 0;

    while(*cReadPtr) {

        switch(state) {
        case SELECT_SEARCH:
            if ((!memcmp(cReadPtr, "select rowid", 12)) ||
                (!memcmp(cReadPtr, "select [main].[mediaobjects].rowid", 34))) { // found
                state = ROWID_SEARCH;
                copyIt = 1; // start copying
            }
            break;
        case ROWID_SEARCH:
            if (!memcmp(cReadPtr, "rowid", 5)) {
                strcpy(cWtritePtr, "id ");
                cWtritePtr += 3;
                copyIt = 0;
                state = END_OF_COLUMNS_SEARCH;
            }
            break;
        case END_OF_COLUMNS_SEARCH:
            if (!memcmp(cReadPtr, "from [", 6)) {
                copyIt = 1;
                state = OFFSET_SEARCH;
            }
            break;
        case OFFSET_SEARCH:
            if (!memcmp(cReadPtr, "offset ?", 8)) {
                strcpy(cWtritePtr, "offset ?;");
                cWtritePtr += 9;
                copyIt = 0;
                state = END;
            }
            break;
        case END:
            break;
        }

        if (state == END) break;

        /* copy one character */
        if (copyIt) {
            *cWtritePtr++ = *cReadPtr;
        }

        /* next read position */
        cReadPtr++;
    }
    *cWtritePtr = 0;

#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN mSelectString);
    if (res) {
        ETG_TRACE_ERR(("PositionPrepare:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("PositionPrepare:Prepare: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    //ETG_TRACE_USR4(("Select:sqlite3_prepare_v2: %s", mSelectString));
    res = sqlite3_prepare_v2(mDB->Handle(), mSelectString, (int)strlen_r(mSelectString), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("PositionPrepare:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("PositionPrepare:sqlite3_prepare_v2: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#endif

    /* get the number of current bindings */
    int noOfBindings;
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* set every open binding to 1 (which enables all "? OR" terms) */
    for (int i=1; i<=noOfBindings - 2 /* the last 2 are limit and offset */; i++) {
        res = sqlite3_bind_int(mStatement, i, 1); /* enables all (? OR ...) expressions to true */
    }

    /* set limit and offset bindings */
    mBindingForLimit = noOfBindings-1; // position of limit binding (always the last-1 one)
    mLimit = -1;
    res = sqlite3_bind_int(mStatement, mBindingForLimit, mLimit);
    mBindingForOffset = noOfBindings; // position of offset binding (always the last one)
    mStartIndex = 0;
    res = sqlite3_bind_int(mStatement, mBindingForOffset, mStartIndex);

    /* do the real user provided bindings */
    va_list vl;
    va_start(vl, format);
    int valueIndex = 1; // start index in binding list

    /* if it is no MY_MEDIA, fill in the device ID */
    if (deviceID != MY_MEDIA) {
        res = sqlite3_bind_int(mStatement, 1, deviceID);
        valueIndex++;
    }

    const char *fmt = format;
    noOfBindings -= 2; /* subtract the limit / offset binding */
    while(*fmt && noOfBindings > 0) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
            int integer;
            integer = va_arg(vl, int );
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            valueIndex++;
            break;
        case 't': // c-string
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            valueIndex++;
            break;
        case 's': // skip input value
            va_arg(vl, int );
            break;
        case '-': // ignore column binding
        default:
            valueIndex++;
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): error: %d", res));
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): %s", sqlite3_errmsg(mDB->Handle())));
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): %s", mSelectString));
        }

        // next char in format */
        fmt++;
    }

    return MP_NO_ERROR;
}

tResult Query::PositionSelect(const tListType listType, const tDeviceID deviceID, const char *format, ...)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* copy the parents sql command */
    if (mSelectString) {
        free(mSelectString);
        mSelectString = NULL;
    }
    mSelectString = strdup(mSQLStatements[listType].query);
    if (NULL == mSelectString) return MP_ERR_DB_SELECT_FAILED;

    /* lowercase the complete select query */
    for(unsigned int i=0; i<strlen_r(mSelectString); i++) mSelectString[i] = tolower(mSelectString[i]);

#if USE_QUERY_CACHE
    res = Prepare(OUT &mStatement, IN mDB->Handle(), IN mSelectString);
    if (res) {
        ETG_TRACE_ERR(("PositionPrepare:Prepare: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("PositionPrepare:Prepare: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#else
    //ETG_TRACE_USR4(("Select:sqlite3_prepare_v2: %s", mSelectString));
    res = sqlite3_prepare_v2(mDB->Handle(), mSelectString, (int)strlen_r(mSelectString), &mStatement, NULL);
    if (res != SQLITE_OK) {
        ETG_TRACE_ERR(("PositionPrepare:sqlite3_prepare_v2: %s", sqlite3_errmsg(mDB->Handle())));
        ETG_TRACE_ERR(("PositionPrepare:sqlite3_prepare_v2: %s", mSelectString));
        return MP_ERR_DB_SELECT_FAILED;
    }
#endif

    /* get the number of current bindings */
    int noOfBindings;
    noOfBindings = sqlite3_bind_parameter_count(mStatement);

    /* set every open binding to 1 (which enables all "? OR" terms) */
    for (int i=1; i<=noOfBindings - 2 /* the last 2 are limit and offset */; i++) {
        res = sqlite3_bind_int(mStatement, i, 1); /* enables all (? OR ...) expressions to true */
    }

    /* set limit and offset bindings */
    mBindingForLimit = noOfBindings-1; // position of limit binding (always the last-1 one)
    mLimit = -1;
    res = sqlite3_bind_int(mStatement, mBindingForLimit, mLimit);
    mBindingForOffset = noOfBindings; // position of offset binding (always the last one)
    mStartIndex = 0;
    res = sqlite3_bind_int(mStatement, mBindingForOffset, mStartIndex);

    /* do the real user provided bindings */
    va_list vl;
    va_start(vl, format);
    int valueIndex = 1; // start index in binding list

    /* if it is no MY_MEDIA, fill in the device ID */
    if (deviceID != MY_MEDIA) {
        res = sqlite3_bind_int(mStatement, 1, deviceID);
        valueIndex++;
    }

    const char *fmt = format;
    noOfBindings -= 2; /* subtract the limit / offset binding */
    while(*fmt && noOfBindings > 0) {

        /* bind the values after format spec */
        switch(*fmt) {
        case 'i': // integer
            int integer;
            integer = va_arg(vl, int );
            res = sqlite3_bind_int(mStatement, valueIndex, integer);
            noOfBindings--;
            valueIndex++;
            break;
        case 't': // c-string
            char *cPtr;
            cPtr = va_arg(vl, char *);
            res = sqlite3_bind_text(mStatement, valueIndex, cPtr, strlen_r(cPtr), SQLITE_TRANSIENT);
            noOfBindings--;
            valueIndex++;
            break;
        case 's': // skip input value
            va_arg(vl, int );
            break;
        case '-': // ignore column binding
        default:
            valueIndex++;
            break;
        }

        if (res) {
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): error: %d", res));
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): %s", sqlite3_errmsg(mDB->Handle())));
            ETG_TRACE_ERR(("PositionPrepare:sqlite3_bind_(*): %s", mSelectString));
        }

        // next char in format */
        fmt++;
    }

    return MP_NO_ERROR;
}

tResult Query::PositionGet(tPosition &position, const tObjectID objectID)
{
#if ENABLE_ENTRY_TRACES
    ENTRY_INTERNAL;
#endif
    tResult res;

    /* save current list position */
    Save();

    /* reset the list position */
    res = SetRow(IN 0);
    if (res) return res;

    /* loop to search for position */
    position = 0;
    while(1) {
        tObjectID testObjectID;

        res = Get(IN 1, "i", OUT &testObjectID);
        if (res) break;
        if (objectID == testObjectID) break;

        position++;
    }

    Restore();

    return res;
}
