#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/Database.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#endif
#endif

#include <TimeTrace.h>
#include "FunctionTracer.h"
#include "VarTrace.h"

#include "VideoTagInfo.h"
#define USE_VIDEO_METADATA_READ_PROCESS 1

#include <stdarg.h>
#include <cstring>
#include <sys/stat.h>
#include <string>
#include <vector>
#include <errno.h>
//#include "/usr/include/asm/param.h" /* for the HZ value */
#include <asm/param.h> /* for the HZ value */

#define USE_TAGINFO

// needed for CreateTestDatabase: taglib
#ifndef USE_TAGINFO
#include <taglib/fileref.h>
#include <taglib/mpegfile.h> //mp3 file
#include <taglib/id3v2tag.h> //tag
#include <taglib/id3v2frame.h> //frame
#else
#include <taginfo.h>
using namespace std;
using namespace TagInfo;
#endif

#include "Database.h"
#include "TypeDefinitions.h"
#include "LocalSPM.h"
#include "DataProvider.h"
#include "sortlib/sortlib.h"
#include "UCASort/UCASort.h"
#include "DBTriggerManager.h"
#include "VTFileContext.h"
#include "VTBluetoothContext.h"
/*
 * define the collation for all ordered lists indizes (is used by the schema-2.0 include files
 */
#define NO_COLLATE " "
#define SORTLIB_COLLATE " COLLATE SORT_LIB "
#define NOCASE_COLLATE " COLLATE NOCASE "

/* this switches the collation for all queries underneath */
#define COLLATE SORTLIB_COLLATE

// Static constants initialization
const char* Database::mDatabaseFile = "/var/opt/bosch/dynamic/media/db/MyMedia.db";
const char* Database::mDatabaseRecreateFile = "/var/opt/bosch/dynamic/media/db/MyMedia.db.recreate";

#define QUICK_CHECK 1

#define MYMEDIA_IN_MEMORY 1

#define DEBUG_OPEN_DB_HANDLES 0

#if MYMEDIA_IN_MEMORY
const char* Database::mMyMediaDatabaseFile = ":memory:";
#else
const char* Database::mMyMediaDatabaseFile = "/var/opt/bosch/dynamic/media/db/MyMediaTest.db";
#endif

static tVTFileContext mVTFileContext;
static tVTBluetoothContext mVTBluetoothContext;
static int FileTypesFilled = 0;
static vector<sqlite3 *> mFreeDBConnections;
static Lock mFreeDBConnectionsLock;
static unsigned int mOpenedConnections = 0;
static int mShutDown = 0;
static int mStopUpdate = 0;
static timeval mProgressCallbackCallTime;
static FILE *testFile = NULL;
int mSlowDownUpdateMyMedia;

tResult Database::CheckSqlite3Version(sqlite3 *dbHandle) const
{
    ENTRY;
    const char *sqlCommand = "select sqlite_version();";
    sqlite3_stmt *statement;
    int sqlError;
    static int print = 1;

    sqlError = sqlite3_prepare_v2(dbHandle, sqlCommand, -1, &statement, NULL);

    /* error on sqlite3_prepare */
    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
        ETG_TRACE_ERR(("sqlite3_step: sql=%s", sqlCommand));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* process it */
    sqlError = sqlite3_step(statement);
    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
        ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
        ETG_TRACE_ERR(("sqlite3_step: sql=%s", sqlCommand));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* get the version string */
    const char *version = (const char *)sqlite3_column_text(statement, 0);
    if (!version) {
        ETG_TRACE_ERR(("sqlite3_column_text: %s", sqlite3_errmsg(dbHandle)));
        ETG_TRACE_ERR(("sqlite3_column_text: sql=%s", sqlCommand));
        return MP_ERR_DB_UNEXPECTED;
    }

    sqlite3_finalize(statement);

    if (print) {
        ETG_TRACE_USR1(("used sqlite version: %s", version));
        print = 0;
    }

    /* read out the version numbers */
    int major;
    int minor;
    int subminor;
    sscanf(version, "%d.%d.%d", &major, &minor, &subminor);
    if (major < 3 && minor < 7 && subminor < 15) {
        ETG_TRACE_FATAL(("Wrong sqlite3 version: %s", version));
        return MP_ERR_DB_UNEXPECTED;
    }

    return MP_NO_ERROR;
}

tResult Database::Refresh()
{
    ENTRY_INTERNAL

    Refresh(Handle());

    return MP_NO_ERROR;
}

tResult Database::Refresh(sqlite3 *dbHandle)
{
    ENTRY_INTERNAL

    /* re-register the busy callback with the current calculated instruction count */
    sqlite3_progress_handler(dbHandle, LocalSPM::GetDataProvider().DBUpdateProgressInstructions(), ProgressCallbackCaller, this);

    return MP_NO_ERROR;
}

tResult Database::Sync(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL
    int nPackets;
    int error;

    error = sqlite3_wal_checkpoint_v2(dbHandle, "Main", SQLITE_CHECKPOINT_RESTART, NULL, OUT &nPackets);
    ETG_TRACE_USR1(("sqlite3_wal_checkpoint_v2(Main): error=%d packets=%d", error, nPackets));

    return MP_NO_ERROR;
}

Database::Database() : mDatabase(0)
{
    ENTRY_INTERNAL
    tResult res;
    static int first = 1;

    /* very first inits */
    if (first) {
        first = 0;
        mFreeDBConnectionsLock.setNotReantrant();
        mShutDown = 0;
        memset(&mProgressCallbackCallTime, 0, sizeof(mProgressCallbackCallTime));

        /* accept URI's as filenames */
        sqlite3_config(SQLITE_CONFIG_URI, 1);

        /* if non-memory db: remove the mymedia db */
#if !MYMEDIA_IN_MEMORY
        remove(mMyMediaDatabaseFile);
#endif
    }

    mSlowDownUpdateMyMedia = 0;

    /* check if a previous run wants us to recreate the database because of a malformed error */
    FILE *fp;
    fp = fopen(Database::mDatabaseRecreateFile, "r");

    /* if no recreate file exists open the existing database */
    if (!fp) {

        ETG_TRACE_USR1(("OK: Open stored database - not existing: %s", Database::mDatabaseRecreateFile));
        mDatabase = OpenDatabase();

    /* a recreate file exists: do not open the database and force recreation */
    } else {

        ETG_TRACE_FATAL (("existing: %s", Database::mDatabaseRecreateFile));
        ETG_TRACE_ERRMEM(("existing: %s", Database::mDatabaseRecreateFile));

        /* close and remove the recreation trigger file */
        fclose(fp);
        remove(Database::mDatabaseRecreateFile);

        mDatabase = NULL;
    }

    if (!mDatabase) { // database open error or recreation needed

        /* create a new database */
        res = RecreateDatabase();
        if (res != MP_NO_ERROR) {
            ETG_TRACE_ERR(("Error on recreat of data base"));
        }

        /* try to open this */
        mDatabase = OpenDatabase();
        MP_FATAL_ASSERT(mDatabase);

    // database is open:
    } else {

        //25-01-2016 Removed integrity check from startup sequence due to performance issues
        //Instead this check is planned to be called at shutdown, additional time from SPM needs to be configured
        tBoolean withExplain = false;

        /* check if the database is up to date */
        res = CheckDatabaseIntegrity(IN mDatabase, IN withExplain);
        if (res != MP_NO_ERROR) { // if not ok:

            // for sure, close the database
            CloseDatabase(mDatabase);

            // recreate the database
            CloseAllHandles();

            res = RecreateDatabase();
            if (res != MP_NO_ERROR) {
                ETG_TRACE_ERR(("Error on recreat of data base"));
            }

            mDatabase = OpenDatabase();
            MP_FATAL_ASSERT(mDatabase);
        }
    }

    /* check the sqlite version */
    CheckSqlite3Version(mDatabase);

    /* attach trigger callbacks to this connection */
    DBTriggerManager::GetInstance().AttachDB(mDatabase);
}

void Database::Close()
{
    ENTRY_INTERNAL
    if (mDatabase) {

        if (testFile) fprintf(testFile, "%p\t%p\t%s\n", mDatabase, this, "Close");

        DBTriggerManager::GetInstance().DetachDB(mDatabase);

        CloseDatabase(mDatabase);

        mDatabase = NULL;
    }
}

Database::~Database()
{
    ENTRY_INTERNAL
    Close();
}

tResult Database::DeInit()
{
    ENTRY_INTERNAL

    /* set the shut down flag to break the busy callbacks and the running update thread */
    mShutDown = 1;

    /* remove all triggers */
    if (mDatabase) {

        DBTriggerManager::GetInstance().DeRegisterAllTriggers();

        // detach from db
        DBTriggerManager::GetInstance().DetachDB(mDatabase);
    }

    if (testFile) fprintf(testFile, "%p\t%p\t%s\n", mDatabase, this, "DeInit");

    // close this handle (goes back into cache)
    CloseDatabase(mDatabase);

    /* close all open handles in cache */
    CloseAllHandles();

    mDatabase = NULL;
    mShutDown = 0;

    return MP_NO_ERROR;
}

sqlite3* Database::Handle()
{
#if 0 // debug
    if (!testFile) {
        testFile = fopen("/tmp/testfile.txt", "w");
    }
#endif

    if (testFile) fprintf(testFile, "%p\t%p\n", mDatabase, this);
    if (testFile) fflush(testFile);

    return mDatabase;
}

tResult Database::CreateMyMediaDatabase(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL

    /* include the schema as header file */
    const char *sqlCreate =
    #include "schema-2.0-myMedia.h"
    ;
    const char *restOfCommand = sqlCreate;
    const char *endOfCommand = sqlCreate + strlen_r(sqlCreate);
    const char *lastCommand;
    sqlite3_stmt *statement;
    int sqlError;

#if MYMEDIA_IN_MEMORY
    char sql[256];
    snprintf(sql, sizeof(sql), "ATTACH DATABASE 'file:%s' as MyMe;", mMyMediaDatabaseFile);
    sqlError = sqlite3_exec(dbHandle, sql, NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec(%64s): %s", sql, sqlite3_errmsg(dbHandle)));
        return -1;
    }
#else
    /* only for test */
    sqlite3 *dbHandleMyMe;

    sqlError = sqlite3_open_v2(
            mMyMediaDatabaseFile,
            &dbHandleMyMe,
            SQLITE_OPEN_READWRITE |
            SQLITE_OPEN_CREATE,
            NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_open_v2: failed: %d, errno=%d/%s", sqlError, errno, strerror(errno)));
    }

    sqlite3_close(dbHandleMyMe);

    char sql[256];
    snprintf(sql, sizeof(sql), "ATTACH DATABASE 'file:%s' as MyMe;", mMyMediaDatabaseFile);
    sqlError = sqlite3_exec(dbHandle, sql, NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec(%64s): %s", sql, sqlite3_errmsg(dbHandle)));
        return -1;
    }
#endif

    /* process the commands in the included command list */
    while(restOfCommand != endOfCommand) {

        /* interpret one command */
        lastCommand = restOfCommand;
        sqlError = sqlite3_prepare_v2(dbHandle, restOfCommand, -1, &statement, &restOfCommand);

        /* error on sqlite3_prepare */
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_prepare_v2: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
                char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
                ETG_TRACE_ERR(("sqlite3_prepare_v2: sql=%s", singleCommand));
                free(singleCommand);
            }
            break;
        }

        /* process it */
        sqlError = sqlite3_step(statement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
                char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
                ETG_TRACE_ERR(("sqlite3_step: sql=%s", singleCommand));
                free(singleCommand);
            }
            break;
        }

        sqlite3_finalize(statement);
    }

    /* attach index helper file */
#include "index_helper.h"

    /* create a Indexhelper db file */
    FILE *fpDB = fopen("/tmp/index_helper.db", "wb");
    if (fpDB) {
        fwrite(db_IndexHelper_db, db_IndexHelper_db_len, 1, fpDB);
        fclose(fpDB);

        /* attach this db to the main db */
        snprintf(sql, sizeof(sql), "ATTACH DATABASE 'file:/tmp/index_helper.db' as IndexHelper;");
        sqlError = sqlite3_exec(dbHandle, sql, NULL, NULL, NULL);
    }

    return MP_NO_ERROR;
}

tResult Database::DeleteMyMediaDatabase(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL
    const char *deleteAll = "DELETE FROM [MyMe].[MediaObjects];";
    int sqlError;

    sqlError = sqlite3_exec(dbHandle, deleteAll, NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    return MP_NO_ERROR;
}

tResult Database::UpdateMain(sqlite3 *dbHandle, const tDeviceID deviceID) const
{
    ENTRY_INTERNAL
    const char *update = "DELETE FROM [Main].[MediaObjects] WHERE ID IN "
                         "(SELECT [Main].[MediaObjects].ID FROM [Main].[MediaObjects] LEFT JOIN MyMe.[MediaObjects] ON ([Main].[MediaObjects].ID = MyMe.[MediaObjects].ID) "
                         "WHERE ([Main].[MediaObjects].ID IS NULL OR MyMe.[MediaObjects].ID IS NULL) AND [Main].[MediaObjects].DeviceID=?);";
    sqlite3_stmt *statement;
    int sqlError;

    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, update, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* set the deviceID */
    sqlError = sqlite3_bind_int(statement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int(count): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* do the command */
    sqlError = sqlite3_step(statement);

    sqlite3_finalize(statement);

    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
        ETG_TRACE_ERR(("sqlite3_step: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    return MP_NO_ERROR;
}

tResult Database::DetermineFolderContent(sqlite3 *dbHandle, const tDeviceID deviceID) const
{
    ENTRY_INTERNAL
    int sqlError;
    tResult res = MP_NO_ERROR;

    sqlite3_stmt *selectFolderStatement;
    const char *selectFolder = "SELECT ID FROM [Main].[MediaObjects] "
                               "WHERE DeviceID = ? AND CategoryID = 17;";

    /* prepare the select statement */
    sqlError = sqlite3_prepare_v2(dbHandle, selectFolder, -1, &selectFolderStatement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(selectFolder): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    sqlite3_stmt *selectMediaTypeStatement;
    const char *selectMediaType = "SELECT MediaContent FROM [Main].[FolderHierarchy] JOIN [Main].[MediaObjects] ON ([Main].[FolderHierarchy].ChildID = [Main].[MediaObjects].FolderID) "
                                  "WHERE [FolderHierarchy].DeviceID = ? AND [FolderHierarchy].ParentID = ? GROUP BY MediaContent;";

    /* prepare the select statement */
    sqlError = sqlite3_prepare_v2(dbHandle, selectMediaType, -1, &selectMediaTypeStatement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(selectMediaType): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    sqlite3_stmt *updateFolderStatement;
    const char *updateFolder = "UPDATE OR IGNORE [Main].[MediaObjects] "
                               "SET MediaContent = ? WHERE ID = ?;";

    /* prepare the select statement */
    sqlError = sqlite3_prepare_v2(dbHandle, updateFolder, -1, &updateFolderStatement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(updateFolder): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* get the ID's of folders */
    /* set the deviceID */
    sqlError = sqlite3_bind_int(selectFolderStatement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int(selectFolder): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* chunk-loop */
    int folderID;
    while(1) {

        /* do the select command */
        sqlError = sqlite3_step(selectFolderStatement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_ROW) {
            break;
        }

        /* get the folderID */
        folderID = sqlite3_column_int(selectFolderStatement, 0);

        /* get the media types */
        /* set the deviceID */
        sqlError = sqlite3_bind_int(selectMediaTypeStatement, 1, deviceID);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int(selectMediaType): %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }

        /* set the folderID */
        sqlError = sqlite3_bind_int(selectMediaTypeStatement, 2, folderID);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int(selectMediaType): %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }

        /* chunk-loop */
        int mediaContent = 0;
        while(1) {

            /* do the select command */
            sqlError = sqlite3_step(selectMediaTypeStatement);
            if (sqlError != SQLITE_OK && sqlError != SQLITE_ROW) {
                break;
            }

            /* get the media types */
            mediaContent |= sqlite3_column_int(selectMediaTypeStatement, 0);
        }

        /* reset the select command */
        sqlError = sqlite3_reset(selectMediaTypeStatement);
        if (sqlError != SQLITE_OK) {
            break;
        }

        /* update the folder */
        /* set the media content */
        sqlError = sqlite3_bind_int(updateFolderStatement, 1, mediaContent);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int(updateFolder): %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }

        /* set the folderID */
        sqlError = sqlite3_bind_int(updateFolderStatement, 2, folderID);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int(updateFolder): %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }

        /* do the update command */
        sqlError = sqlite3_step(updateFolderStatement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
            ETG_TRACE_ERR(("sqlite3_step(updateFolder): %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* reset the update command */
        sqlError = sqlite3_reset(updateFolderStatement);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_reset(updateFolder): %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* shutdown requested? */
        if (mShutDown) {
            ETG_TRACE_USR2(("shutdown requested"));
            res = MP_ERR_DB_UPD_MYMEDIA_DB_SHUTDOWN;
            mShutDown++;
            break;
        }
    }

    /* finalize the statements */
    sqlite3_finalize(selectFolderStatement);
    sqlite3_finalize(selectMediaTypeStatement);
    sqlite3_finalize(updateFolderStatement);

    return res;
}

tResult Database::CleanupLiveTags(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL
    const char *update = "DELETE FROM [Main].LiveTags WHERE TagID IN "
                         "(SELECT DISTINCT [Main].LiveTags.TagID FROM [Main].LiveTags INNER JOIN [Main].[MediaObjects] ON (([Main].LiveTags.TagID = [Main].[MediaObjects].ID) OR "
                         "([Main].LiveTags.TagID = [Main].[MediaObjects].FilterTag1ID) OR ([Main].LiveTags.TagID = [Main].[MediaObjects].FilterTag2ID) OR "
                         "([Main].LiveTags.TagID = [Main].[MediaObjects].FilterTag3ID) OR ([Main].LiveTags.TagID = [Main].[MediaObjects].FilterTag4ID)));";
    sqlite3_stmt *statement;
    int sqlError;

    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, update, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* do the command */
    sqlError = sqlite3_step(statement);

    sqlite3_finalize(statement);

    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
        ETG_TRACE_ERR(("sqlite3_step: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    return MP_NO_ERROR;
}

tResult Database::SpeedupDatabase(sqlite3 *dbHandle, tDeviceID deviceID) const
{
    ENTRY;

#if 1 // single index method

    const char *indices[] = {
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag3ID_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag3ID_LowerFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_TrackNumber_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag4ID_TrackNumber_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag2ID_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag2ID_LowerFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_TrackNumber_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_LowerFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_LowerFilterTag2_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag3ID_InitialTitle_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag2ID_InitialTitle_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_InitialTitle_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag2ID_InitialFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_InitialFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_FilterTag1ID_InitialFilterTag2_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_InitialFilterTag3_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_InitialFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_InitialFilterTag2_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_InitialFilterTag1_Index;",
            (char *)"ANALYZE MediaObjects_InitialTitle_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_InitialTitle_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_UUID_CategoryID_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_Path_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_Path_Index;",
            (char *)"ANALYZE MediaObjects_Path_CategoryID_URL_Index;",
            (char *)"ANALYZE MediaObjects_CategoryID_URL_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_Path_CategoryID_URL_Index;",
            (char *)"ANALYZE MediaObjects_Title_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_Title_Index;",
            (char *)"ANALYZE MediaObjects_LowerFilterTag3_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_LowerFilterTag3_Index;",
            (char *)"ANALYZE MediaObjects_LowerFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_LowerFilterTag4_Index;",
            (char *)"ANALYZE MediaObjects_LowerFilterTag2_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_LowerFilterTag2_Index;",
            (char *)"ANALYZE MediaObjects_FilterTag1_Index;",
            (char *)"ANALYZE MediaObjects_LowerFilterTag1_Index;",
            (char *)"ANALYZE MediaObjects_DeviceID_CategoryID_LowerFilterTag1_Index;",
            (char *)"ANALYZE sqlite_autoindex_MediaObjects_2;",
            (char *)"ANALYZE sqlite_autoindex_MediaObjects_1;",
            (char *)"ANALYZE FolderHierarchy_ChildID_ParentID;",
            (char *)"ANALYZE FolderHierarchy_Name;",
            NULL
    };
    int sqlError;

    if (deviceID == MY_MEDIA) return MP_NO_ERROR;

    /* refresh the progress instruction count */
    //Refresh();

    for(unsigned int i=0; indices[i]; i++) {

        ETG_TRACE_USR2(("command: %s", indices[i]));

        sqlError = sqlite3_exec(dbHandle, indices[i], NULL, NULL, NULL);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }
        usleep(10000L);
    }

#else
    (void)deviceID;

    ElapsedTimer measTime; /* measure the time it takes to analyze the Mediaobjects table of the DB */

    int sqlError;
    sqlError = sqlite3_exec(dbHandle, "ANALYZE [Main].[Mediaobjects];", NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* end time sampling */
    unsigned long durationMS = measTime.ElapsedMS();
    if(2000 < durationMS) {
        ETG_TRACE_FATAL (("Analyze of GMP DB last too long: %lums", durationMS));
        ETG_TRACE_ERRMEM(("Analyze of GMP DB last too long: %lums", durationMS));

        int err = system("/opt/bosch/mediaplayer/bin/analyze.sh");
        if (err != 0) {
            ETG_TRACE_ERR(("return value: %i, errno: %i", err, errno));
        }
    }
#endif

    /* sync with all open db handles */
    Sync(dbHandle);

    return MP_NO_ERROR;
}

tResult Database::CopyDeviceToFromVirtualTable(sqlite3 *dbHandle, const tBoolean fromVirtual, const tBoolean toVirtual,
        const tDeviceID deviceID, tDeviceID newDeviceID/*=DEVICE_ID_NOT_SET*/) const
{
    ENTRY_INTERNAL
    sqlite3_stmt *statement;
    int sqlError;

    if (DEVICE_ID_NOT_SET == newDeviceID)
    {
        newDeviceID = deviceID;
    }

    char fromDevice[16];
    if (fromVirtual)
    {
        strncpy_r(OUT fromDevice, "VirtualDevices", IN sizeof(fromDevice));
    }
    else
    {
        strncpy_r(OUT fromDevice, "Devices", IN sizeof(fromDevice));
    }

    char toDevice[16];
    if (toVirtual)
    {
        strncpy_r(OUT toDevice, "VirtualDevices", IN sizeof(toDevice));
    }
    else
    {
        strncpy_r(OUT toDevice, "Devices", IN sizeof(toDevice));
    }

    char copyDevice[2048];
    snprintf (copyDevice, sizeof(copyDevice),
            "INSERT OR REPLACE INTO [MAIN].[%s]"
            "("
            "  [ID],"
            "  [UUID],"
            "  [SerialNumber],"
            "  [FirmwareVersion],"
            "  [FriendlyName],"
            "  [MountPoint],"
            "  [ConnectionCount],"
            "  [LastConnected],"
            "  [CheckSum],"
            "  [Active],"
            "  [RepeatMode],"
            "  [PlaybackMode],"
            "  [DeviceType],"
            "  [IndexingState],"
            "  [ConnectionState],"
            "  [IndexingProgress],"
            "  [FormerConnectionState],"
            "  [AudioInputDevice],"
            "  [DeviceState],"
            "  [FileSystemType],"
            "  [PartitionNumber],"
            "  [TotalSize],"
            "  [FreeSize],"
            "  [ProductID],"
            "  [NumberOfAudioFiles],"
            "  [NumberOfVideoFiles],"
            "  [NumberOfImageFiles],"
            "  [DiPOCapable],"
            "  [DiPOActive],"
            "  [DiPOVersion],"
            "  [ConnectionType]"
            ")"
            "SELECT"
            "  ?,"
            "  [UUID],"
            "  [SerialNumber],"
            "  [FirmwareVersion],"
            "  [FriendlyName],"
            "  [MountPoint],"
            "  [ConnectionCount],"
            "  [LastConnected],"
            "  [CheckSum],"
            "  [Active],"
            "  [RepeatMode],"
            "  [PlaybackMode],"
            "  [DeviceType],"
            "  [IndexingState],"
            "  [ConnectionState],"
            "  [IndexingProgress],"
            "  [FormerConnectionState],"
            "  [AudioInputDevice],"
            "  [DeviceState],"
            "  [FileSystemType],"
            "  [PartitionNumber],"
            "  [TotalSize],"
            "  [FreeSize],"
            "  [ProductID],"
            "  [NumberOfAudioFiles],"
            "  [NumberOfVideoFiles],"
            "  [NumberOfImageFiles],"
            "  [DiPOCapable],"
            "  [DiPOActive],"
            "  [DiPOVersion],"
            "  [ConnectionType]"
            "FROM"
            "  [MAIN].[%s]"
            "WHERE"
            "  [MAIN].[%s].[ID] = ?;"
            ,toDevice, fromDevice, fromDevice);

    /*
     * do the device copy:
     */
    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, (const char *)copyDevice, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* set the new deviceID */
    sqlError = sqlite3_bind_int(statement, 1, newDeviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* set the select deviceID */
    sqlError = sqlite3_bind_int(statement, 2, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* do the copy command */
    sqlError = sqlite3_step(statement);
    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
        ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    /* finalize query */
    sqlite3_finalize(statement);

    return MP_NO_ERROR;
}

tResult Database::UpdateMyMediaDatabase(int *stopFlag, sqlite3 *dbHandle, tDeviceID deviceID) const
{
    ENTRY_INTERNAL
    sqlite3_stmt *statement;
    int sqlError;
    tResult res = MP_NO_ERROR;
    const char *updateAll =
        "INSERT OR REPLACE INTO [MyMe].[MediaObjects]"
        "("
        "  [ID],"
        "  [DeviceID],"
        "  [CategoryID],"
        "  [FilterTag1ID],"
        "  [FilterTag2ID],"
        "  [FilterTag3ID],"
        "  [FilterTag4ID],"
        "  [UUID],"
        "  [ParentUUID],"
        "  [Path],"
        "  [URL],"
        "  [AlbumArt],"
        "  [TrackNumber],"
        "  [TotalPlaytime],"
        "  [NotPlayable],"
        "  [Title],"
        "  [FilterTag1],"
        "  [FilterTag2],"
        "  [FilterTag3],"
        "  [FilterTag4],"
        "  [MetadataConvertFlag],"
        "  [CompilationFlag],"
        "  [LowerFilterTag1],"
        "  [LowerFilterTag2],"
        "  [LowerFilterTag3],"
        "  [LowerFilterTag4],"
        "  [InitialTitle],"
        "  [InitialFilterTag1],"
        "  [InitialFilterTag2],"
        "  [InitialFilterTag3],"
        "  [InitialFilterTag4],"
        "  [FolderID],"
        "  [MediaContent],"
        "  [Size],"
        "  [DateTime],"
        "  [FileMode],"
        "  [UserID],"
        "  [GroupID]"
        ")"
        "SELECT"
        "  [ID],"
        "  [DeviceID],"
        "  [CategoryID],"
        "  [FilterTag1ID],"
        "  [FilterTag2ID],"
        "  [FilterTag3ID],"
        "  [FilterTag4ID],"
        "  [UUID],"
        "  [ParentUUID],"
        "  [Path],"
        "  [URL],"
        "  [AlbumArt],"
        "  [TrackNumber],"
        "  [TotalPlaytime],"
        "  [NotPlayable],"
        "  [Title],"
        "  [FilterTag1],"
        "  [FilterTag2],"
        "  [FilterTag3],"
        "  [FilterTag4],"
        "  [MetadataConvertFlag],"
        "  [CompilationFlag],"
        "  [LowerFilterTag1],"
        "  [LowerFilterTag2],"
        "  [LowerFilterTag3],"
        "  [LowerFilterTag4],"
        "  [InitialTitle],"
        "  [InitialFilterTag1],"
        "  [InitialFilterTag2],"
        "  [InitialFilterTag3],"
        "  [InitialFilterTag4],"
        "  [FolderID],"
        "  [MediaContent],"
        "  [Size],"
        "  [DateTime],"
        "  [FileMode],"
        "  [UserID],"
        "  [GroupID]"
        "FROM"
        "  [MAIN].[MediaObjects]"
        "WHERE"
        "  [MAIN].[MediaObjects].[DeviceID] = ? LIMIT ? OFFSET ?;";
    const char *countAll = "SELECT count() FROM [MAIN].[MediaObjects] WHERE [MAIN].[MediaObjects].[DeviceID] = ?;";
    int objectCount;

    /*
     get the number of objects to store
     */
    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, countAll, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(count): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* set the deviceID */
    sqlError = sqlite3_bind_int(statement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int(count): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* do the count */
    sqlError = sqlite3_step(statement);
    if (sqlError != SQLITE_OK && sqlError != SQLITE_ROW) {
        ETG_TRACE_ERR(("sqlite3_step(count): %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* get the count */
    objectCount = sqlite3_column_int(statement,  0);

    /* finalize count query */
    sqlite3_finalize(statement);

    /*
     * do the my media update:
     */
    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, updateAll, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* set the deviceID */
    sqlError = sqlite3_bind_int(statement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
        return -1;
    }

    /* chunk-loop */
    int offset = 0;
    int limit = 100; // start value for limit, will be reset after first inserts
    ElapsedTimer measTime; /* meas the time it takes to update the my media with one calculated chunk */
    unsigned long durationUS;
    while(1) {

        /* bind the limit and offset values */
        sqlError = sqlite3_bind_int(statement, 2, limit);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }
        sqlError = sqlite3_bind_int(statement, 3, offset);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
            return -1;
        }

        /* start time sampling */
        measTime.Restart();

        /* do the command */
        sqlError = sqlite3_step(statement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
            break;
        }

        /* reset the command */
        sqlError = sqlite3_reset(statement);
        if (sqlError != SQLITE_OK) {
            break;
        }

        /* end time sampling */
        durationUS = measTime.ElapsedUS();

        /* end of objects? */
        if ((offset + limit) > objectCount) {
            break;
        }

        /* set new offset */
        offset += limit;

        ETG_TRACE_USR2(("written %d objects in %d ms", limit, (int)(durationUS / 1000.)));

        /* recalc the cache size for next stores */
        int limitNew;
        limitNew = (int)((double)900 / ((durationUS / 1000.) / limit));

        /* set new limit */
        limit = (limit/2) + (limitNew/2); // a little integrating here...
        // limit = limitNew;

        ETG_TRACE_USR2(("new limit: %d", limit));

        /* enable others to run */
        sched_yield();

        /* does somebody wants to slow down this task? */
        if (mSlowDownUpdateMyMedia) {

            ETG_TRACE_USR2(("slow down"));

            /* sleep a second */
            int iLoop = 10;
            while(iLoop-- && !*stopFlag && !mShutDown) {
                usleep(100000L);
            }

            /* reset the limit */
            limit = 25;

            mSlowDownUpdateMyMedia = 0;
        }

        /* shutdown requested? */
        if (mShutDown) {
            ETG_TRACE_USR2(("shutdown requested"));
            res = MP_ERR_DB_UPD_MYMEDIA_DB_SHUTDOWN;
            mShutDown++;
            break;
        }

        /* stop requested */
        if (*stopFlag) {
            ETG_TRACE_USR2(("stop requested"));
            res = MP_ERR_DB_UPD_MYMEDIA_DB_STOPPED;
            break;
        }
    }

    sqlite3_finalize(statement);

    if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
        ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_UNEXPECTED;
    }

    return res;
}

tResult Database::RemoveDeviceFromMyMedia(int *stopFlag, sqlite3 *dbHandle, tDeviceID deviceID) const
{
    ENTRY_INTERNAL
    tResult res = MP_NO_ERROR;
    sqlite3_stmt *statement;
    sqlite3_stmt *minMaxStatement;
    int sqlError;
    const char *removeElements =
            "DELETE FROM [MyMe].[MediaObjects] WHERE [MyMe].[MediaObjects].[DeviceID] = ? AND [MediaObjects].[ID] >= ? AND [MediaObjects].[ID] <= ?;";
    const char *getMinMaxID = "SELECT MIN(ID),MAX(ID) FROM [MyMe].[MediaObjects] WHERE [MyMe].[MediaObjects].[DeviceID] = ?;";
     int maxID;
    int fromID;
    int noOfIDs = 200;
    ElapsedTimer measTime; /* meas the time it takes to update the my media with one calculated chunk */
    unsigned long durationUS;

    /*
     get the min and max id of area to delete
     */
    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, getMinMaxID, -1, &minMaxStatement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2(count): %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_QUERY_FAILED;
    }

    /* set the deviceID */
    sqlError = sqlite3_bind_int(minMaxStatement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int(count): %s", sqlite3_errmsg(dbHandle)));
        sqlite3_finalize(minMaxStatement);
        return MP_ERR_DB_QUERY_FAILED;
    }

    /*
     * do the my media delete for a device:
     */
    /* prepare the update statement */
    sqlError = sqlite3_prepare_v2(dbHandle, removeElements, -1, &statement, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_prepare_v2: %s", sqlite3_errmsg(dbHandle)));
        return MP_ERR_DB_QUERY_FAILED;
    }

    /* set the deviceID */
    sqlError = sqlite3_bind_int(statement, 1, deviceID);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_bind_int(count): %s", sqlite3_errmsg(dbHandle)));
        sqlite3_finalize(minMaxStatement);
        sqlite3_finalize(statement);
        return MP_ERR_DB_QUERY_FAILED;
    }

    /* chunk-loop */
    while(1) {

        /* get the current min and max values */
        sqlError = sqlite3_step(minMaxStatement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_step(count): %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* get the min and max */
        fromID = sqlite3_column_int(minMaxStatement,  0);
        maxID = sqlite3_column_int(minMaxStatement,  1);

        /* reset the query */
        sqlError = sqlite3_reset(minMaxStatement);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_reset(count): %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        ETG_TRACE_USR2(("ID's: min=%d, max=%d", fromID, maxID));

        /* no id's found? end of operation reached */
        if (maxID == 0) break;

        /* bind the limit and offset values */
        sqlError = sqlite3_bind_int(statement, 2, fromID);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
            break;
        }
        sqlError = sqlite3_bind_int(statement, 3, fromID + noOfIDs);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_bind_int: %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* start time sampling */
        measTime.Restart();

        /* do the command */
        sqlError = sqlite3_step(statement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE) {
            ETG_TRACE_ERR(("sqlite3_step: %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* reset the command */
        sqlError = sqlite3_reset(statement);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_reset: %s", sqlite3_errmsg(dbHandle)));
            break;
        }

        /* end time sampling */
        durationUS = measTime.ElapsedUS();

        ETG_TRACE_USR2(("deleted %d objects in %d ms", noOfIDs, (int)(durationUS / 1000.)));

        /* end of objects? */
        if ((fromID + noOfIDs) > maxID) {
            ETG_TRACE_USR2(("delete finished"));
            break;
        }

        /* recalc the cache size for next stores */
        int noOfIDsNew;
        noOfIDsNew = (int)((double)LocalSPM::GetDataProvider().DBDeleteMyMediaObjectMaxTimeMS() / ((durationUS / 1000.) / noOfIDs));

        /* set new limit */
        noOfIDs = (noOfIDs/2) + (noOfIDsNew/2); // a little integrating here...

        /* limit the number */
        if (noOfIDs > LocalSPM::GetDataProvider().DBDeleteMyMediaObjectMaxCount()) {
            noOfIDs = LocalSPM::GetDataProvider().DBDeleteMyMediaObjectMaxCount();
        }

        ETG_TRACE_USR2(("new limit: %d", noOfIDs));

        /* enable others to run */
        sched_yield();

        /* does somebody wants to slow down this task? */
        if (mSlowDownUpdateMyMedia) {

            ETG_TRACE_USR2(("slow down"));

            /* sleep a second */
            int iLoop = 10;
            while(iLoop-- && !*stopFlag && !mShutDown) {
                usleep(100000L);
            }

            /* reset the limit */
            noOfIDs = 100;

            mSlowDownUpdateMyMedia = 0;
        }

        /* shutdown requested? */
        if (mShutDown) {
            ETG_TRACE_USR2(("shutdown requested"));
            res = MP_ERR_DB_UPD_MYMEDIA_DB_SHUTDOWN;
            mShutDown++;
            break;
        }

#if 1 // stop is not supported for delete
        /* stop requested */
        if (*stopFlag) {
            res = MP_ERR_DB_UPD_MYMEDIA_DB_STOPPED;
            ETG_TRACE_USR2(("stop requested"));
            break;
        }
#endif
    }

    /* finalize the update statement */
    sqlite3_finalize(statement);

    /* finalize count query statement */
    sqlite3_finalize(minMaxStatement);

    return res;
}

void Database::SlowDownUpdateMyMedia(void) const
{
    mSlowDownUpdateMyMedia = 1;
}

tResult Database::ConfigureConnection(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL

    tResult result = MP_ERR_DB_UNEXPECTED;
    std::vector<std::string> connectionConfigurationScript;
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA FOREIGN_KEYS = OFF;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA JOURNAL_MODE = OFF;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA LOCKING_MODE = NORMAL;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA SYNCHRONOUS = OFF;")); // NORMAL
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA AUTOMATIC_INDEX = OFF;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA SECURE_DELETE = OFF;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA TEMP_STORE = MEMORY;"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA ENCODING = \"UTF-8\";"));
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA CACHE_SIZE = 4000;")); //default 2000
    connectionConfigurationScript.push_back(std::string(
            "PRAGMA PAGE_SIZE = 8192;")); //default 1024

    typedef std::vector<std::string>::const_iterator tStringVectorIterator;
    tStringVectorIterator connectionConfigurationScriptIterator =
            connectionConfigurationScript.begin();
    tInteger sqliteResult = SQLITE_ERROR;

    /* configure the db connection to be in serialized mode */
    sqliteResult = sqlite3_config(SQLITE_CONFIG_SERIALIZED);
    if (SQLITE_OK == sqliteResult) {
        ETG_TRACE_ERR(("sqlite3_config(SQLITE_CONFIG_SERIALIZED): error=%d", sqliteResult));
    }

    // Execute the database initialization script
    while (connectionConfigurationScript.end() !=
            connectionConfigurationScriptIterator)
    {
        char* sqlError = NULL;

        // Execute SQL query
        sqliteResult = sqlite3_exec(
                dbHandle,
                connectionConfigurationScriptIterator->c_str(),
                NULL, // No callback, as no data selected
                NULL, // No callback arguments
                &sqlError);

        // Check for success
        if (SQLITE_OK == sqliteResult)
        {
            result = MP_NO_ERROR;
            connectionConfigurationScriptIterator++;
        }
        else
        {
            // Failed to execute SQL query.
            ETG_TRACE_ERR(("Error: sqlite3_exec(%64s): %64s",
                    connectionConfigurationScriptIterator->c_str(),
                    sqlite3_errmsg(dbHandle)));

            // Error handling
            if (NULL == sqlError)
            {
                ETG_TRACE_FATAL(("Undefined SQLite3 error."));
            }
            else
            {
                ETG_TRACE_FATAL(("SQLite3 error: %s.", sqlError));
                sqlite3_free(sqlError);
            }

            result = MP_ERR_DB_DATABASE_CONNECTION_FAILURE;
            break;
        }
    }

    return result;
}

tResult Database::CloseDatabase(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL

    /* instead of closing the db, the used handle goes into a free list to be reused by next OpenDatabase() */
    mFreeDBConnectionsLock.lock();
    mFreeDBConnections.push_back(dbHandle);
    mFreeDBConnectionsLock.unlock();

    return MP_NO_ERROR;
}

void Database::CloseAllHandles() const
{
    ENTRY_INTERNAL
    tInteger sqliteResult = SQLITE_ERROR;
    tResult res;

    mFreeDBConnectionsLock.lock();

    // check the free cache count
    if (mFreeDBConnections.size() != mOpenedConnections) {
        ETG_TRACE_ERR(("connection cache leak: mOpenedConnections=%d, mFreeDBConnections.size()=%d", mOpenedConnections, mFreeDBConnections.size()));
    }

    for(unsigned int i=0; i<mFreeDBConnections.size(); i++) {

        /* detach the my media data base */
        const char *detach = "DETACH DATABASE MyMe;";

        sqliteResult = sqlite3_exec(mFreeDBConnections[i], detach, NULL, NULL, NULL);
        if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: failed"));
        }

        /* detach the index helper data base */
        const char *detach2 = "DETACH DATABASE IndexHelper;";

        sqliteResult = sqlite3_exec(mFreeDBConnections[i], detach2, NULL, NULL, NULL);
        if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: failed"));
        }

        /* finalize all cached queries */
        res = Query::Finalize(mFreeDBConnections[i]);
        if (res) {
            ETG_TRACE_ERR(("query.Finalize: failed"));
        }

        sqliteResult = sqlite3_close_v2(mFreeDBConnections[i]); // the v2 variant closes the file but leaves a zombie connection open.

        if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_close: failed: error=%d", sqliteResult));
        }

        ETG_TRACE_USR2(("closed handle: 0x%x", mFreeDBConnections[i]));
    }

    mFreeDBConnections.clear();
    mOpenedConnections = 0;

    mFreeDBConnectionsLock.unlock();
}

tResult Database::ResetDatabaseIntegrity(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL

    char* sqlError = NULL;
    tInteger sqliteResult = SQLITE_ERROR;
    std::string resetIntegrityQuery =
            "UPDATE"
            "  [MAIN].[SchemaVersion] "
            "SET"
            "  [Integrity] = 1"
            ";";

    // Execute SQL query for integrity reset.
    sqliteResult = sqlite3_exec(
            dbHandle,
            resetIntegrityQuery.c_str(),
            NULL, // No callback, as no data selected
            NULL, // No callback arguments
            &sqlError);

    // Check for success
    if (SQLITE_OK != sqliteResult)
    {
        // Failed to execute SQL query.
        ETG_TRACE_FATAL(("Failed to execute: %s.", resetIntegrityQuery.c_str()));

        // Error handling
        if (NULL == sqlError)
        {
            ETG_TRACE_FATAL(("Undefined SQLite3 error."));
        }
        else
        {
            ETG_TRACE_FATAL(("SQLite3 error: %s.", sqlError));
            sqlite3_free(sqlError);
        }

        return MP_ERR_DB_DATABASE_SCHEMA_FAILURE;
    }

    return MP_NO_ERROR;
}

tResult Database::CheckDatabaseIntegrity(sqlite3 *dbHandle, const tBoolean withExplain/*=true*/) const
{
    ENTRY;

    tResult result = MP_ERR_DB_UNEXPECTED;

    sqlite3_stmt* sqliteStatement = NULL;
    tInteger sqliteResult = SQLITE_ERROR;

    if (withExplain) {

        char queryString[128];
        sqlite3_stmt *statement;

        /* create the explain query string */
#if QUICK_CHECK
        strncpy_r(OUT queryString, IN "PRAGMA quick_check;", IN sizeof(queryString));
#else
        strncpy_r(OUT queryString, IN "PRAGMA integrity_check;", IN sizeof(queryString));
#endif

        /* prepare the update statement */
        sqliteResult = sqlite3_prepare_v2(dbHandle, queryString, -1, &statement, NULL);
        if (sqliteResult != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_prepare_v2(): %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));
            return MP_ERR_DB_INTEGRITY_FAILURE;
        }

        /* get the resulting lines */
        int integrityOk = 0;
        int lineNo = 1;
        TimeTrace ticks("IntegrityCheck");
        while(1) {

            /* do the explain */
            sqliteResult = sqlite3_step(statement);
            if (sqliteResult == SQLITE_DONE) break;
            if (sqliteResult != SQLITE_OK && sqliteResult != SQLITE_ROW) {
                ETG_TRACE_ERR(("sqlite3_step(): %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));
                break;
            }

            /* get one text line */
            const char *cptr;
            cptr = (const char *)sqlite3_column_text(statement,  0);
            if (cptr == NULL) break;

            /* check if it is OK */
            if (lineNo == 1) {
                if (!strcmp(cptr, "ok")) {
                    integrityOk = 1;
                    ETG_TRACE_USR2(("integrityCheck: %s", cptr));
                } else {
                    ETG_TRACE_ERR(("integrityError: %s", cptr));
                }
            } else {
                ETG_TRACE_ERR(("integrityError: %s", cptr));
            }

            lineNo++;
        }

        /* finalize pragma query */
        sqlite3_finalize(statement);

        ticks.elapsed();

        if (!integrityOk) {
            ETG_TRACE_ERR(("DB integrity damaged"));
            return MP_ERR_DB_INTEGRITY_FAILURE;
        }
    }

    /* Check DB version and stored integrity flag */
    char query[] =
            "SELECT"
            "  [Version],"
            "  [Integrity] "
            "FROM"
            "  [MAIN].[SchemaVersion];";

    // Prepare the database integrity check statement.
    sqliteResult = sqlite3_prepare_v2(dbHandle,
            query,
            sizeof(query),
            &sqliteStatement,
            NULL);

    // Check for success
    if (SQLITE_OK == sqliteResult)
    {
        // Execute the integrity check statement.
        sqliteResult = sqlite3_step(sqliteStatement);

        // Check for success
        // Retrieve the selected data if available.
        if (SQLITE_ROW == sqliteResult)
        {
            // Check whether the schema version is valid.
            if (mSchemaVersion == sqlite3_column_int(sqliteStatement, 0))
            {
                // Check the database integrity status.
                if (FALSE == sqlite3_column_int(sqliteStatement, 1))
                {
                    // The database integrity is invalid.
                    ETG_TRACE_ERR(("The database integrity is invalid."));
                    result = MP_ERR_DB_INTEGRITY_FAILURE;
                }
                else
                {
                    result = MP_NO_ERROR;
                }
            }
            else
            {
                ETG_TRACE_ERR(("Invalid database schema version."));
                result = MP_ERR_DB_INVALID_SCHEMA_VERSION;
            }
        }
        else
        {
            // Failed to execute the integrity check statement.
            ETG_TRACE_ERR(("Failed to execute the integrity check statement: %s", sqlite3_errmsg(dbHandle)));

            // Handle the error
            result = MP_ERR_DB_STEP_FAILURE;
        }

        // Destroy the integrity check statement.
        sqliteResult = sqlite3_finalize(sqliteStatement);

        // Check for success
        if (SQLITE_OK != sqliteResult)
        {
            // Failed to finalize the integrity check statement.
            ETG_TRACE_ERR(("Failed to finalize the integrity check statement."));
        }
    }
    else
    {
        // Failed to prepare the integrity check statement.
        ETG_TRACE_ERR(("Failed to prepare the integrity check statement: %d/%s", sqliteResult, sqlite3_errmsg(dbHandle)));

        // Handle the error
        result = MP_ERR_DB_PREPARE_FAILURE;
    }

    return result;
}

tResult Database::CreateDatabase(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL

    /* include the schema as header file for the standard case sensitive behavior
     * Standard schema with no conditions based on special characters
     * */

#define UNIFY_COLLATE NO_COLLATE
#define UNIFY_TAG1 "[NEW].[FilterTag1]"
#define UNIFY_TAG2 "[NEW].[FilterTag2]"
#define UNIFY_TAG3 "[NEW].[FilterTag3]"
#define UNIFY_TAG4 "[NEW].[FilterTag4]"

#define INITIAL_TITLE "substr(upper([NEW].[Title]), 1, 1)"
#define INITIAL_TAG1 "substr(upper([NEW].[FilterTag1]), 1, 1)"
#define INITIAL_TAG2 "substr(upper([NEW].[FilterTag2]), 1, 1)"
#define INITIAL_TAG3 "substr(upper([NEW].[FilterTag3]), 1, 1)"
#define INITIAL_TAG4 "substr(upper([NEW].[FilterTag4]), 1, 1)"
    const char *sqlCreate_Standard =
    #include "schema-2.0.h"
    ;

    /* include the schema as header file for the standard case sensitive behavior
     * Standard schema with conditions based on special characters(Ignore A,An,The from prefix)
     * */
#undef INITIAL_TITLE
#undef INITIAL_TAG1
#undef INITIAL_TAG2
#undef INITIAL_TAG3
#undef INITIAL_TAG4

#define INITIAL_TITLE "CASE WHEN (length([NEW].[Title]) >= 3 AND [NEW].[Title] LIKE 'A %' ) THEN (substr(upper([NEW].[Title]),3,1)) WHEN (length([NEW].[Title]) >= 4 AND [NEW].[Title] LIKE 'An %' ) THEN (substr(upper([NEW].[Title]),4,1)) WHEN (length([NEW].[Title]) >= 5 AND [NEW].[Title] LIKE 'The %' ) THEN (substr(upper([NEW].[Title]),5,1)) ELSE(substr(upper([NEW].[Title]),1,1)) END"
#define INITIAL_TAG1 "CASE WHEN (length([NEW].[FilterTag1]) >= 3 AND [NEW].[FilterTag1] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag1]),3,1)) WHEN (length([NEW].[FilterTag1]) >= 4 AND [NEW].[FilterTag1] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag1]),4,1)) WHEN (length([NEW].[FilterTag1]) >= 5 AND [NEW].[FilterTag1] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag1]),5,1)) ELSE(substr(upper([NEW].[FilterTag1]),1,1)) END"
#define INITIAL_TAG2 "CASE WHEN (length([NEW].[FilterTag2]) >= 3 AND [NEW].[FilterTag2] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag2]),3,1)) WHEN (length([NEW].[FilterTag2]) >= 4 AND [NEW].[FilterTag2] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag2]),4,1)) WHEN (length([NEW].[FilterTag2]) >= 5 AND [NEW].[FilterTag2] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag2]),5,1)) ELSE(substr(upper([NEW].[FilterTag2]),1,1)) END"
#define INITIAL_TAG3 "CASE WHEN (length([NEW].[FilterTag3]) >= 3 AND [NEW].[FilterTag3] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag3]),3,1)) WHEN (length([NEW].[FilterTag3]) >= 4 AND [NEW].[FilterTag3] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag3]),4,1)) WHEN (length([NEW].[FilterTag3]) >= 5 AND [NEW].[FilterTag3] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag3]),5,1)) ELSE(substr(upper([NEW].[FilterTag3]),1,1)) END"
#define INITIAL_TAG4 "CASE WHEN (length([NEW].[FilterTag4]) >= 3 AND [NEW].[FilterTag4] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag4]),3,1)) WHEN (length([NEW].[FilterTag4]) >= 4 AND [NEW].[FilterTag4] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag4]),4,1)) WHEN (length([NEW].[FilterTag4]) >= 5 AND [NEW].[FilterTag4] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag4]),5,1)) ELSE(substr(upper([NEW].[FilterTag4]),1,1)) END"
    const char *sqlCreate_IgnoreSp =
    #include "schema-2.0.h"
    ;

    /* include the same schema as header file for the unified behavior
     * Unified Behavior with no conditions based on special characters
     * */
#undef UNIFY_COLLATE
#undef UNIFY_TAG1
#undef UNIFY_TAG2
#undef UNIFY_TAG3
#undef UNIFY_TAG4
#undef INITIAL_TITLE
#undef INITIAL_TAG1
#undef INITIAL_TAG2
#undef INITIAL_TAG3
#undef INITIAL_TAG4

#define UNIFY_COLLATE NOCASE_COLLATE
#define UNIFY_TAG1 "LOWER([NEW].[FilterTag1])"
#define UNIFY_TAG2 "LOWER([NEW].[FilterTag2])"
#define UNIFY_TAG3 "LOWER([NEW].[FilterTag3])"
#define UNIFY_TAG4 "LOWER([NEW].[FilterTag4])"

#define INITIAL_TITLE "substr(upper([NEW].[Title]), 1, 1)"
#define INITIAL_TAG1 "substr(upper([NEW].[FilterTag1]), 1, 1)"
#define INITIAL_TAG2 "substr(upper([NEW].[FilterTag2]), 1, 1)"
#define INITIAL_TAG3 "substr(upper([NEW].[FilterTag3]), 1, 1)"
#define INITIAL_TAG4 "substr(upper([NEW].[FilterTag4]), 1, 1)"
    const char *sqlCreateUnified_Standard =
    #include "schema-2.0.h"
    ;

    /* include the same schema as header file for the unified behavior
     * Unified Behavior with conditions based on special characters(Ignore A,An,The from prefix)
     * */
#undef INITIAL_TITLE
#undef INITIAL_TAG1
#undef INITIAL_TAG2
#undef INITIAL_TAG3
#undef INITIAL_TAG4
#define INITIAL_TITLE "CASE WHEN (length([NEW].[Title]) >= 3 AND [NEW].[Title] LIKE 'A %' ) THEN (substr(upper([NEW].[Title]),3,1)) WHEN (length([NEW].[Title]) >= 4 AND [NEW].[Title] LIKE 'An %' ) THEN (substr(upper([NEW].[Title]),4,1)) WHEN (length([NEW].[Title]) >= 5 AND [NEW].[Title] LIKE 'The %' ) THEN (substr(upper([NEW].[Title]),5,1)) ELSE(substr(upper([NEW].[Title]),1,1)) END"
#define INITIAL_TAG1 "CASE WHEN (length([NEW].[FilterTag1]) >= 3 AND [NEW].[FilterTag1] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag1]),3,1)) WHEN (length([NEW].[FilterTag1]) >= 4 AND [NEW].[FilterTag1] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag1]),4,1)) WHEN (length([NEW].[FilterTag1]) >= 5 AND [NEW].[FilterTag1] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag1]),5,1)) ELSE(substr(upper([NEW].[FilterTag1]),1,1)) END"
#define INITIAL_TAG2 "CASE WHEN (length([NEW].[FilterTag2]) >= 3 AND [NEW].[FilterTag2] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag2]),3,1)) WHEN (length([NEW].[FilterTag2]) >= 4 AND [NEW].[FilterTag2] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag2]),4,1)) WHEN (length([NEW].[FilterTag2]) >= 5 AND [NEW].[FilterTag2] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag2]),5,1)) ELSE(substr(upper([NEW].[FilterTag2]),1,1)) END"
#define INITIAL_TAG3 "CASE WHEN (length([NEW].[FilterTag3]) >= 3 AND [NEW].[FilterTag3] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag3]),3,1)) WHEN (length([NEW].[FilterTag3]) >= 4 AND [NEW].[FilterTag3] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag3]),4,1)) WHEN (length([NEW].[FilterTag3]) >= 5 AND [NEW].[FilterTag3] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag3]),5,1)) ELSE(substr(upper([NEW].[FilterTag3]),1,1)) END"
#define INITIAL_TAG4 "CASE WHEN (length([NEW].[FilterTag4]) >= 3 AND [NEW].[FilterTag4] LIKE 'A %' ) THEN (substr(upper([NEW].[FilterTag4]),3,1)) WHEN (length([NEW].[FilterTag4]) >= 4 AND [NEW].[FilterTag4] LIKE 'An %' ) THEN (substr(upper([NEW].[FilterTag4]),4,1)) WHEN (length([NEW].[FilterTag4]) >= 5 AND [NEW].[FilterTag4] LIKE 'The %' ) THEN (substr(upper([NEW].[FilterTag4]),5,1)) ELSE(substr(upper([NEW].[FilterTag4]),1,1)) END"
    const char *sqlCreateUnified_IgnoreSp =
    #include "schema-2.0.h"
    ;

    const char *restOfCommand = sqlCreate_Standard;
    const char *lastCommand;
    const char *endOfCommand = sqlCreate_Standard + strlen_r(sqlCreate_Standard);
    sqlite3_stmt *statement;
    int sqlError = SQLITE_OK;

    /* switch to data base schema defined in data provider */
     if (LocalSPM::GetDataProvider().DBUnifiedSchema()) {
         if(LocalSPM::GetDataProvider().IgnoreSpecialCharachters())
         {
             restOfCommand = sqlCreateUnified_IgnoreSp;
             endOfCommand = sqlCreateUnified_IgnoreSp + strlen_r(sqlCreateUnified_IgnoreSp);
             ETG_TRACE_USR1(("Using UNIFIED Schema and Ignore Special Characters"));
         }
         else
         {
             restOfCommand = sqlCreateUnified_Standard;
             endOfCommand = sqlCreateUnified_Standard + strlen_r(sqlCreateUnified_Standard);
             ETG_TRACE_USR1(("Using UNIFIED Schema and NO Ignore Special Characters"));
         }
    } else {
        if(LocalSPM::GetDataProvider().IgnoreSpecialCharachters())
        {
            restOfCommand = sqlCreate_IgnoreSp;
            endOfCommand = sqlCreate_IgnoreSp + strlen_r(sqlCreate_IgnoreSp);
            ETG_TRACE_USR1(("Using STANDARD Schema and Ignore Special Characters"));
        }
        else
        {
            restOfCommand = sqlCreate_Standard;
            endOfCommand = sqlCreate_Standard + strlen_r(sqlCreate_Standard);
            ETG_TRACE_USR1(("Using STANDARD Schema and NO Ignore Special Characters"));
        }
    }

    /* start DB creation monitoring
     * write a file onto the disk to issue a data base file delete on next restart of system */
    FILE *fp;
    fp = fopen(Database::mDatabaseRecreateFile, "w");
    fclose(fp);
    ETG_TRACE_USR1(("created: %s", Database::mDatabaseRecreateFile));

    /* process the commands in the included command list */
    while(restOfCommand != endOfCommand) {

        /* interpret one command */
        lastCommand = restOfCommand;
        //printf("sqlite3_prepare: %60s\n", restOfCommand); fflush(stdout);
        sqlError = sqlite3_prepare_v2(dbHandle, restOfCommand, -1, &statement, &restOfCommand);

        /* error on sqlite3_prepare */
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_prepare_v2: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
                char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
                ETG_TRACE_ERR(("sqlite3_step: sql=%s", singleCommand));
                free(singleCommand);
            }
            break;
        }

        /* process it */
        sqlError = sqlite3_step(statement);
        if (sqlError != SQLITE_OK && sqlError != SQLITE_DONE && sqlError != SQLITE_ROW) {
            ETG_TRACE_ERR(("sqlite3_step: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
            const char *cSemicolon = strstr(lastCommand, ";");
            if (cSemicolon) {
                char *singleCommand = strndup(lastCommand, (size_t)(cSemicolon-lastCommand)+1);
                ETG_TRACE_ERR(("sqlite3_step: sql=%s", singleCommand));
                free(singleCommand);
            }
            break;
        }

        sqlite3_finalize(statement);
    }

    /* end DB creation monitoring
     * remove the recreation trigger file in case of success */
    if (sqlError == SQLITE_OK || sqlError == SQLITE_DONE || sqlError == SQLITE_ROW) {
        remove(Database::mDatabaseRecreateFile);
        ETG_TRACE_USR1(("removed: %s", Database::mDatabaseRecreateFile));
    }

    /* set the my media name from data provider */
    tGeneralString updateString;
    snprintf(updateString, sizeof(updateString), "update devices set FriendlyName = '%s' where ID=0;",
            LocalSPM::GetDataProvider().DBMyMediaName().c_str());
    sqlError = sqlite3_exec(IN dbHandle, updateString, NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
    }

    tMediaContext mediaContext;
    tRepeatMode repeatMode;
    LocalSPM::GetDBManager().GetMediaContext(OUT mediaContext);
    if(MC_VIDEO == mediaContext)
        repeatMode=(tRepeatMode)LocalSPM::GetDataProvider().DBDefaultVideoRepeatMode();
    else
        repeatMode=(tRepeatMode)LocalSPM::GetDataProvider().DBDefaultRepeatMode();
    /* set the default my media repeat mode */
    snprintf(updateString, sizeof(updateString), "update devices set RepeatMode = %d where ID=0;",
            repeatMode);
    sqlError = sqlite3_exec(IN dbHandle, updateString, NULL, NULL, NULL);
    if (sqlError != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_exec: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
    }

    /* set indexing state to NOT_SUPPORTED if my media is not visible */
    if (LocalSPM::GetDataProvider().MyMediaVisible() == 0) {
        snprintf(updateString, sizeof(updateString), "update devices set indexingState = 0 where ID=0;");
        sqlError = sqlite3_exec(IN dbHandle, updateString, NULL, NULL, NULL);
        if (sqlError != SQLITE_OK) {
            ETG_TRACE_ERR(("sqlite3_exec: %d/%s", sqlError, sqlite3_errmsg(dbHandle)));
        }
    }

    return MP_NO_ERROR;
}

tResult Database::RecreateDatabase() const
{
    ENTRY_INTERNAL
    tResult result = MP_ERR_DB_UNEXPECTED;
    tInteger sqliteResult = SQLITE_ERROR;
    sqlite3 *dbHandle;

    /* remove the data base file */
    remove(mDatabaseFile); // if not exist, dont care

    /* if non-memory db: remove the mymedia db */
#if !MYMEDIA_IN_MEMORY
    remove(mMyMediaDatabaseFile);
#endif

    /* remove cover art directory */
    if (LocalSPM::GetDataProvider().CoverArtFlow()) {

        tPath coverArtPath; //e.g. "/var/opt/bosch/dynamic/media/coverart"
        strncpy_r(OUT coverArtPath, IN LocalSPM::GetDataProvider().CoverArtDirPath().c_str(), IN sizeof(coverArtPath));
        if ((exists_directory(coverArtPath))
            &&
            (23 < strlen_r(coverArtPath))) { //do not delete directory recursively with a path length below "/var/opt/bosch/dynamic/"
            remove_directory(coverArtPath); // if not exist, dont care
        }
    }

    // open the data base
    // this method is only called on startup so nobody else will have access to the database
    sqliteResult = sqlite3_open_v2(
            mDatabaseFile,
            &dbHandle,
            SQLITE_OPEN_READWRITE |
            SQLITE_OPEN_CREATE,
            NULL);
    if (sqliteResult != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_open_v2: failed: %d, errno=%d/%s", sqliteResult, errno, strerror(errno)));

        // -> Handle the error
        MP_FATAL_ASSERT(sqliteResult == SQLITE_OK);
    }

    /* Enable loadable modules (DSOs) */
    result = AddModules(IN dbHandle);
    if (result != MP_NO_ERROR) {
        ETG_TRACE_ERR(("RecreateDatabase: error on adding modules"));
    }

    result = ConfigureConnection(IN dbHandle);
    if (result != MP_NO_ERROR) {
        ETG_TRACE_ERR(("RecreateDatabase: error on ConfigureConnection"));
    }

    /* add collation based on the configuration*/
    if(LocalSPM::GetDataProvider().DBSortUseUCAOn())
    {
        result = UCASort_RegisterSortFunction(dbHandle,LocalSPM::GetDataProvider().DBSortLanguage().c_str());
    }
    else
    {
        result = sortlib_register_sort_function(dbHandle);
    }


    if (result != MP_NO_ERROR) {
        ETG_TRACE_FATAL(("RecreateDatabase: error on sortlib_register_sort_function"));
    }

    // Create the database schema
    result = CreateDatabase(IN dbHandle);
    if (result != MP_NO_ERROR) {

        ETG_TRACE_FATAL(("CreateDatabase(..): failed"));
        MP_FATAL_ASSERT(sqliteResult == SQLITE_OK);
    }

    /* insert the db schema version number */
    result = ResetDatabaseIntegrity(IN dbHandle);
    if (result != MP_NO_ERROR) {
        ETG_TRACE_FATAL(("ResetDatabaseIntegrity(..): failed"));
    }

    // close the database
    sqlite3_close(dbHandle);

    return MP_NO_ERROR;
}

tResult Database::AddModules(sqlite3 *dbHandle) const
{
    ENTRY_INTERNAL
    tResult result = MP_NO_ERROR;

    /* Enable loadable modules (DSOs) */
    result = sqlite3_enable_load_extension(dbHandle, 1);
    if (result != SQLITE_OK) {
        ETG_TRACE_ERR(("AddModules: error on sqlite3_enable_load_extension"));
    }

    /* Load the virtual table for IPOD */
    result = sqlite3_load_extension(dbHandle, "libVTIPOD.so", "sqlite3_extension_init", NULL);
    if (result != SQLITE_OK) {
        ETG_TRACE_ERR(("AddModules: error on sqlite3_load_extension libVTIPOD.so"));
    }

    if (!FileTypesFilled)
    {
        for(int i=1; i<=LocalSPM::GetDataProvider().SupportedFileTypes(); i++) {
            tVTFileTypeEntry VTEntry;
            memset(&VTEntry, 0, sizeof(tVTFileTypeEntry));

            switch(i) {
            case 1:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_01().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_01();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_01();
                break;
            case 2:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_02().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_02();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_02();
                break;
            case 3:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_03().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_03();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_03();
                break;
            case 4:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_04().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_04();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_04();
                break;
            case 5:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_05().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_05();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_05();
                break;
            case 6:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_06().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_06();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_06();
                break;
            case 7:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_07().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_07();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_07();
                break;
            case 8:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_08().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_08();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_08();
                break;
            case 9:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_09().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_09();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_09();
                break;
            case 10:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_10().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_10();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_10();
                break;
            case 11:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_11().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_11();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_11();
                break;
            case 12:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_12().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_12();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_12();
                break;
            case 13:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_13().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_13();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_13();
                break;
            case 14:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_14().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_14();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_14();
                break;
            case 15:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_15().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_15();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_15();
                break;
            case 16:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_16().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_16();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_16();
                break;
            case 17:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_17().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_17();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_17();
                break;
            case 18:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_18().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_18();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_18();
                break;
            case 19:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_19().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_19();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_19();
                break;
            case 20:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_20().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_20();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_20();
                break;
            case 21:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_21().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_21();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_21();
                break;
            case 22:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_22().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_22();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_22();
                break;
            case 23:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_23().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_23();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_23();
                break;
            case 24:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_24().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_24();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_24();
                break;
            case 25:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_25().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_25();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_25();
                break;
            case 26:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_26().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_26();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_26();
                break;
            case 27:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_27().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_27();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_27();
                break;
            case 28:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_28().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_28();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_28();
                break;
            case 29:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_29().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_29();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_29();
                break;
            case 30:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_30().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_30();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_30();
                break;
            case 31:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_31().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_31();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_31();
                break;
            case 32:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_32().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_32();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_32();
                break;
            case 33:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_33().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_33();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_33();
                break;
            case 34:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_34().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_34();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_34();
                break;
            case 35:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_35().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_35();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_35();
                break;
            case 36:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_36().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_36();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_36();
                break;
            case 37:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_37().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_37();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_37();
                break;
            case 38:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_38().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_38();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_38();
                break;
            case 39:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_39().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_39();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_39();
                break;
            case 40:
                strncpy_r(OUT VTEntry.extension, IN LocalSPM::GetDataProvider().SupportedFileTypeExtension_40().c_str(), IN sizeof(VTEntry.extension));
                VTEntry.fileFormat = LocalSPM::GetDataProvider().SupportedFileTypeFormat_40();
                VTEntry.fileType = LocalSPM::GetDataProvider().SupportedFileTypeType_40();
                break;
            default:
                ETG_TRACE_FATAL(("[ERROR] Database::AddModules: Check mediaplayer configuration (SupportedFileTypes)"));
                break;
            }

            /* put it into the context */
            mVTFileContext.registeredFileTypes.push_back(VTEntry);
        }

        /*push file pattern which need to be skipped by VTFile*/
        for( int i=1; i<=LocalSPM::GetDataProvider().FileFilterCount(); i++)
        {
            switch(i)
            {
                case 1:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_01());
                    break;
                case 2:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_02());
                    break;
                case 3:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_03());
                    break;
                case 4:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_04());
                    break;
                case 5:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_05());
                    break;
                case 6:
                    mVTFileContext.registeredFileFilterPatterns.push_back(LocalSPM::GetDataProvider().FileFilter_06());
                    break;
                default:
                    ETG_TRACE_FATAL(("[ERROR] Database::AddModules: Check mediaplayer configuration (FileFilterCount)"));
                    break;
            }
        }

        FileTypesFilled = 1;

        /* set the path where internal playlist are storage */
        mVTFileContext.internalPlaylistPath.assign(LocalSPM::GetDataProvider().InternalPlaylistStoragePath());

        /* set the max depth to scan for mass storage devices */
        mVTFileContext.mMaxDepth = LocalSPM::GetDataProvider().DBVTFileMaxDepthForScan();

        /* set the max depth to recurse the concatenated playlist*/
        mVTFileContext.mSupportedPlaylistRecurseDepth = LocalSPM::GetDataProvider().DBVTFileMaxDepthForPlaylistRecurse();

        /* set the max limit for VTFile cache size */
        mVTFileContext.mMaxCacheSize = LocalSPM::GetDataProvider().DBVTFileMaxCacheSize();

        /* set flag to show all playlist entries even invalid and not accessible ones */
        mVTFileContext.mShowAllPlaylistEntries = LocalSPM::GetDataProvider().DBVTFileShowAllPlaylistEntries();
    }

    // printf("AddModules: mVTFileContext.registeredFileTypes.size()=%d\n", mVTFileContext.registeredFileTypes.size());

    // forward the context to the so vt file module
    VTFSEntrySetContext(&mVTFileContext);

    /* Load the fs virtual table. */
    char *errorMessage = NULL;
    result = sqlite3_load_extension(dbHandle, "libVTFile.so", "sqlite3_extension_init", &errorMessage);
    if (result != SQLITE_OK) {
        ETG_TRACE_ERR(("AddModules: error on sqlite3_load_extension libVTFile.so : %s",errorMessage));
        sqlite3_free(errorMessage);
    }

    extern void SetSMNameVTCDDA(char * smname);
    if(LocalSPM::GetDataProvider().UseDVDControl())
    {
        SetSMNameVTCDDA((char *)"DVDControlSM");
    }
    else
    {
        SetSMNameVTCDDA((char *)"CDDAControlSM");
    }

    /* Load the cdda virtual table. */
    errorMessage = NULL;
    result = sqlite3_load_extension(dbHandle, "libVTCDDA.so", "sqlite3_extension_init", &errorMessage);
    if (result != SQLITE_OK) {
        ETG_TRACE_ERR(("AddModules: error on sqlite3_load_extension libVTCDDA.so : %s",errorMessage));
        sqlite3_free(errorMessage);
    }

    if(LocalSPM::GetDataProvider().UseEvolutionBtStack())
    {
        mVTBluetoothContext.UseEvolutionBtStack=true;
    }else
    {
        mVTBluetoothContext.UseEvolutionBtStack=false;
    }
    VTBluetoothEntrySetContext(&mVTBluetoothContext);
    /* Load the virtual table for BT */
    errorMessage = NULL;
    result = sqlite3_load_extension(dbHandle, "libVTBluetooth.so", "sqlite3_extension_init", &errorMessage);
       if (result != SQLITE_OK) {
           ETG_TRACE_ERR(("AddModules: error on sqlite3_load_extension libVTBluetooth : %s",errorMessage));
           sqlite3_free(errorMessage);
       }

    /* Load the virtual table for MTP */
    errorMessage = NULL;
    result = sqlite3_load_extension(dbHandle, "libVTMTP.so", "sqlite3_extension_init", &errorMessage);
    if (result != SQLITE_OK)
    {
        ETG_TRACE_ERR(("AddModules: error on sqlite3_load_extension libVTMTP : %s",errorMessage));
        sqlite3_free(errorMessage);
    }

    return MP_NO_ERROR;
}

static Lock mLockRegister;
static vector<void *> mPtrRegister;


tResult Database::Register(const void *ptr) const
{
#if DEBUG_OPEN_DB_HANDLES
    ENTRY_INTERNAL
    mLockRegister.lock();
    mPtrRegister.push_back((void *)ptr);
    mLockRegister.unlock();
#else
    (void)ptr;
#endif
    return MP_NO_ERROR;
}

tResult Database::UnRegister(const void *ptr) const
{
#if DEBUG_OPEN_DB_HANDLES
    ENTRY_INTERNAL
    mLockRegister.lock();
    for(unsigned int i=0; i<mPtrRegister.size(); i++) {
        if (mPtrRegister[i] == ptr) {
            mPtrRegister.erase(mPtrRegister.begin()+i);
            i--;
        }
    }
    mLockRegister.unlock();
#else
    (void)ptr;
#endif
    return MP_NO_ERROR;
}

int Database::Shutdown(const int waitForAllThreadsFinished) const
{
    ENTRY;

    /* set the shutdown flag */
    mShutDown = 1;

#if 0
    /* wait for the shutdown start (max 3 seconds) (threads increasing this value) */
    for(unsigned int retry=0; retry < 6 && mShutDown == 1; retry++) {
        usleep(500);
    }
#endif
    /* now, at least one thread is running the shutdown now, but we do  not now if every thread is doing so. */

    if (waitForAllThreadsFinished) {

        //synchronize with ThreadFactory
        int running_threads = LocalSPM::GetThreadFactory().GetThreadsRunning();
        VARTRACE(running_threads);

        while(running_threads) {
            sleep(1);
            running_threads = LocalSPM::GetThreadFactory().GetThreadsRunning();
            VARTRACE(running_threads);
        }
    }

    return MP_NO_ERROR;
}

int Database::BusyCallback(void *_dbHandle, int times)
{
    ENTRY_INTERNAL
    long sleepTimeUS = times * 10000;
    sqlite3 *dbHandle = (sqlite3 *)_dbHandle;

    /* ask a global variable if this busy should end because of shutdown */
    if (mShutDown) {
        ETG_TRACE_USR2(("DB busy, but asked for shutdown"));
        mShutDown++;
        return 0;
    }

    /* limit the sleep time to max shutdown delay time */
    long maxShutdownDelayUS = LocalSPM::GetDataProvider().SPMMaxShutdownDelay() * 1000L /* us */;
    if (sleepTimeUS > maxShutdownDelayUS) sleepTimeUS = maxShutdownDelayUS;
    if (!(times%10)) { // print only every 10th time
        ETG_TRACE_USR2(("DB busy, waiting for %d us: %s", (int)sleepTimeUS, sqlite3_errmsg(dbHandle)));
#if DEBUG_OPEN_DB_HANDLES
        for(unsigned int i=0; i<mPtrRegister.size(); i++) {
                ETG_TRACE_USR2(("DB busy, open ptr[%d]: 0x%x", i, mPtrRegister[i]));
        }
#endif
    }

    /* sleep a while */
    usleep(sleepTimeUS);

    return 1; // = 1: continue trying
}

int Database::ProgressCallbackCaller(void *_ptr)
{
    ENTRY_INTERNAL
    Database *_this = (Database *)_ptr;
    return _this->ProgressCallback();
}

int Database::ProgressCallback() const
{
    ENTRY_INTERNAL
    /* ask a global variable if this progress should end because of shutdown */
    if (mShutDown) {
        ETG_TRACE_USR2(("DB in progress, but asked for shutdown"));
        mShutDown++;
        return 1; /* end this long runner */
    }

    /* ask a global variable if this progress should end because of user wish */
    if (mStopUpdate) {
        ETG_TRACE_USR2(("DB in progress, but asked for stop"));
        mStopUpdate = 0;
        return 1; /* end this long runner */
    }

    return 0; // = 0: continue trying
}

sqlite3 *Database::OpenDatabase()
{
    ENTRY;
    sqlite3 *dbHandle = NULL;
    tResult result = MP_NO_ERROR;
    tInteger sqliteResult = SQLITE_ERROR;

    /* is a free handle in the list? */
    mFreeDBConnectionsLock.lock();
    if (mFreeDBConnections.size()) {
        dbHandle = mFreeDBConnections.back();
        mFreeDBConnections.pop_back();
        mFreeDBConnectionsLock.unlock();
        Refresh(dbHandle);
        ETG_TRACE_USR2(("reusing handle: 0x%x", dbHandle));
        return dbHandle;
    }
    mFreeDBConnectionsLock.unlock();

    // open the data base
    sqliteResult = sqlite3_open_v2(
            mDatabaseFile,
            &dbHandle,
            SQLITE_OPEN_READWRITE
            ,
            NULL);
    if (sqliteResult != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_open_v2: failed: %d, errno=%d/%s", sqliteResult, errno, strerror(errno)));
        return NULL;
    }

    /* register busy handler */
    sqliteResult = sqlite3_busy_handler(dbHandle, BusyCallback, dbHandle);
    if (sqliteResult != SQLITE_OK) {
        ETG_TRACE_ERR(("sqlite3_busy_handler: failed: %d", sqliteResult));
        return NULL;
    }

    /* add sorting lib collation */
    if(LocalSPM::GetDataProvider().DBSortUseUCAOn())
    {
        result = UCASort_RegisterSortFunction(dbHandle,LocalSPM::GetDataProvider().DBSortLanguage().c_str());
    }
    else
    {
        result = sortlib_register_sort_function(dbHandle);
    }

    if (result != MP_NO_ERROR) {
        ETG_TRACE_FATAL(("OpenDatabase: error on sortlib_register_sort_function"));
        sqlite3_close(dbHandle);
        return NULL;
    }

    // Configure the connection
    result = ConfigureConnection(IN dbHandle);

    // Check for success
    if (result != MP_NO_ERROR)
    {
        ETG_TRACE_FATAL(("Failed to configure the database connection."));
        sqlite3_close(dbHandle);
        return NULL;
    }

    /* Enable loadable modules (DSOs) */
    result = AddModules(IN dbHandle);
    if (result != MP_NO_ERROR) {
        ETG_TRACE_FATAL(("OpenDatabase: error on adding modules"));
        sqlite3_close(dbHandle);
        return NULL;
    }

    Refresh(dbHandle);

    result = CreateMyMediaDatabase(dbHandle);
    if (result) return NULL;

    // count this connection
    mOpenedConnections++;

    ETG_TRACE_USR2(("created handle: 0x%x, count=%d", dbHandle, mOpenedConnections));

    return dbHandle;
}

tResult Database::ReadOneVideoMetaData(IN FILE *fpInput, IN char *tmpFileName, const char *format, ...)
{
    ENTRY

    tGeneralString buffer;

    /* read one line */
    char *cerr;
    cerr = fgets(buffer, sizeof(tGeneralString), fpInput);
    if (cerr != buffer) {
        fclose(fpInput);
        if(0 != remove(tmpFileName))
        {
            ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName));
        }
        return -1;
    }

    /* look for the '=' and read the item */
    char *cEqualSign = strstr(buffer, "=");
    if (!cEqualSign) {
        fclose(fpInput);

        if(0 != remove(tmpFileName))
        {
            ETG_TRACE_ERR(("Cannot delete temp file: %s", tmpFileName));
        }
        return -1;
    }
    cEqualSign++; // now pointing to value
    va_list vl;
    va_start(vl, format);
    vsscanf(cEqualSign, format, vl);
    va_end(vl);

    return MP_NO_ERROR;
}

tResult Database::ScanDirectory(const char *path, const tDeviceID deviceID)
{
    ENTRY_INTERNAL
    tResult res;
    tListID listID = LIST_ID_NONE;
    tListType listType = LTY_FILELIST_UNSORTED;
    tIndex row;
    tDeviceInfo deviceInfo;
    //tURL completeFileName;

    /* get the device info */
    res = LocalSPM::GetDBManager().GetDeviceInfo(OUT deviceInfo, IN deviceID);
    if (res) return res;

    /* create a list for the given path */
    res = LocalSPM::GetDBManager().CreateMediaPlayerFileList(INOUT listID, INOUT listType, IN path, IN deviceID);
    if (res) return res;

    /* loop over all elements of this list */
    for(row=0; 1; row++) {

        tMediaObject mediaObject;

        /* get one entry */
        res = LocalSPM::GetDBManager().GetMediaObjectDoStep(OUT mediaObject, IN listID, IN row);
        if (res == MP_ERR_DB_END_OF_LIST) break;
        if (res) return res;

        /* set the category type */
        switch(mediaObject.fileType) {
        case FT_FOLDER:
            res = ScanDirectory(mediaObject.fileName, deviceID);
            if (res) return res;
            continue;
        case FT_AUDIO:
            break;
        case FT_VIDEO:
            break;
        case FT_PLAYLIST:
            break;
        case FT_IMAGE:
            break;
        default: // unknown file type
            continue;
        }

        tURL completeFileName;
        strncpy_r(OUT completeFileName, IN deviceInfo.mountPoint, IN sizeof(completeFileName));
        strncat_r(OUT completeFileName, IN mediaObject.fileName, IN sizeof(completeFileName));

        mediaObject.metadataConvertFlag = 0;
        if (MTY_MUSIC_FILE == mediaObject.mediaType)
        {
            LocalSPM::GetUSBControl().ReadAudioMetadataFromFile(INOUT mediaObject, IN completeFileName);

            if(LocalSPM::GetDataProvider().IsAlbumArtSupported(mediaObject.deviceType))
            {
                /*overwrite the albumart string as filename (without mountpoint) for storing in DB, else it is with mountpoint*/
                strncpy_r(OUT mediaObject.albumArtString, IN mediaObject.fileName, IN sizeof(mediaObject.albumArtString));
            }
        }
        else if (MTY_VIDEO == mediaObject.mediaType)
        {
            LocalSPM::GetUSBControl().ReadVideoMetadataFromFile(INOUT mediaObject, IN completeFileName);

            if(LocalSPM::GetDataProvider().IsAlbumArtSupported(mediaObject.deviceType))
            {
                /*overwrite the albumart string as filename (without mountpoint) for storing in DB, else it is with mountpoint*/
                strncpy_r(OUT mediaObject.albumArtString, IN mediaObject.fileName, IN sizeof(mediaObject.albumArtString));
            }
        }
        else if (MTY_IMAGE == mediaObject.mediaType)
        {
            LocalSPM::GetUSBControl().ReadImageMetadataFromFile(INOUT mediaObject, IN completeFileName);
        }
        else if (MTY_PLAYLIST == mediaObject.mediaType)
        {
            /* Set the playlist name in tag1 to the title without extension*/
            FastUTF8::tString extension;
            FastUTF8::tString fileName = (FastUTF8::tString)strdup(mediaObject.title);
            FastUTF8::SplitExtension(OUT extension, INOUT fileName);
            strncpy_r(OUT mediaObject.MetadataField1, IN(char*)fileName, IN sizeof(mediaObject.MetadataField1));
            free(fileName);
        }
        else
        {
            ETG_TRACE_ERR(("Database::ScanDirectory -> Cannot extract meta data because of wrong media type"));
        }

        // set the rest
        //mediaObject.deviceID = deviceID;

        /* put the element into database */
        res = LocalSPM::GetDBManager().StoreMediaObject(IN mediaObject);
        if (res) return res;
    }

    LocalSPM::GetDBManager().ReleaseAccess(IN listID);

    /* release the list */
    res = LocalSPM::GetDBManager().ReleaseDBList(IN listID);
    if (res) return res;

    return MP_NO_ERROR;
}

tResult Database::CreateInternalDevices() const
{
    ENTRY
    tResult res;
    Query query;

    /* add internal flash as first devices */
    vector<tDeviceInfo> deviceInfos;
    tDeviceID deviceID;
    res = LocalSPM::GetDataProvider().GetInternalDevices(OUT deviceInfos);
    if (res != MP_NO_ERROR)
    {
        ETG_TRACE_ERR(("Error on get internal flash devices"));
    }

    for(tUInt i=0; i<deviceInfos.size(); i++)
    {
        /* first check if device is already in the database */
      //  res = query.Select(LTY_DEVICES, MY_MEDIA, "--it", IN 0, IN deviceInfos[i].serialNumber);
      //  if (res) return res;

        /* get the device info */
     //   res = query.Get("i", OUT &deviceID);
     //   query.End();

      //  if (res == MP_ERR_DB_END_OF_LIST) // device not known, seems to be a new one:
        {
            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfos[i]);
            if (res != MP_NO_ERROR)
            {
                ETG_TRACE_ERR(("Error on create internal flash device"));
            }
            else
            {
                deviceID = deviceInfos[i].deviceID;
            }
        }

        /* update MyMedia DB */
        res = LocalSPM::GetDBManager().UpdateMyMediaDatabase(IN deviceID);
        if (res != MP_NO_ERROR)
        {
            ETG_TRACE_ERR(("Error while updating MyMedia DB"));
        }
    }

    if(LocalSPM::GetDataProvider().ContentSharingEnabled()) {

        //add CS device to database
        tDeviceInfo deviceInfo;
        InitDeviceInfo(deviceInfo);

        //strncpy_r(deviceInfo.UUID, "ContentSharing", sizeof(deviceInfo.UUID));
        strncpy_r(deviceInfo.serialNumber, "ContentSharing", sizeof(deviceInfo.serialNumber));
        strncpy_r(deviceInfo.deviceName, "ContentSharing", sizeof(deviceInfo.deviceName));
        //strncpy_r(deviceInfo.mountPoint, "", sizeof(deviceInfo.mountPoint));
        //strncpy_r(deviceInfo.accessoryName, "", sizeof(deviceInfo.accessoryName));

        deviceInfo.connected = 1;
        deviceInfo.deviceType = DTY_CS;
        deviceInfo.connectionState = CS_ATTACHED;

        res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
        if (res != MP_NO_ERROR)
        {
            ETG_TRACE_ERR(("Error on creating CS device"));
        }
    }

    return MP_NO_ERROR;
}

tResult Database::CreateTestDatabase(const tBoolean createDevices, const tBoolean createSongs)
{
    ENTRY_INTERNAL
    tResult res;
    tDeviceID deviceID;

    if(createDevices)
    {
        tDeviceInfo deviceInfo;

        InitDeviceInfo(OUT deviceInfo);

        // add an usb device for the test data
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, "USB");
        if (res) {
            deviceInfo.activeSource = 1;
            deviceInfo.connected = 1;
            strncpy_r(OUT deviceInfo.deviceName, IN "USB", IN sizeof(deviceInfo.deviceName));
            deviceInfo.deviceType = DTY_USB;
            deviceInfo.connectionState = CS_CONNECTED;
            strncpy_r(OUT deviceInfo.serialNumber, IN "987654321", IN sizeof(deviceInfo.serialNumber));
            strncpy_r(OUT deviceInfo.deviceVersion, IN "1.0", IN sizeof(deviceInfo.deviceVersion));
            strncpy_r(OUT deviceInfo.UUID, IN "1", IN sizeof(deviceInfo.UUID));
            strncpy_r(OUT deviceInfo.accessoryName, IN "/dev/sda", IN sizeof(deviceInfo.accessoryName));
            deviceInfo.fileSystemType = FSTY_FAT;
            deviceInfo.partitionNumber = 1;
            deviceInfo.totalSize = 33554432; //32GB
            deviceInfo.freeSize = 20971520; //20GB
            deviceInfo.productID = 5767;
            deviceInfo.connectionType = DCT_USB;
#if 1 // for moduletests with data stored in view
#ifndef TARGET_BUILD // run in ubuntu
            strncpy_r(OUT deviceInfo.mountPoint, IN get_current_dir_name(), IN sizeof(deviceInfo.mountPoint));
            strncat_r(OUT deviceInfo.mountPoint, IN "/Customer/Simulation/CustomControl/test", IN sizeof(deviceInfo.mountPoint));
#else // run in target with data stored in view
            strncpy_r(OUT deviceInfo.mountPoint, IN "/opt/bosch/test/data/GMP/test", IN sizeof(deviceInfo.mountPoint));
#endif
#else // for a test with a stick on ubuntu:
            strncpy_r(OUT deviceInfo.mountPoint, IN "/media/KINGSTON", IN sizeof(deviceInfo.mountPoint));
#endif

            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
            if (res) return res;
        }

        // add an IPOD device for the test data
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_IPOD);
        if (res) {
            deviceInfo.activeSource = 0;
            deviceInfo.connected = 1;
            strncpy_r(OUT deviceInfo.deviceName, IN "IPOD", IN sizeof(deviceInfo.deviceName));
            deviceInfo.deviceType = DTY_IPOD;
            deviceInfo.connectionState = CS_CONNECTED;
            strncpy_r(OUT deviceInfo.serialNumber, IN "987654321-1", IN sizeof(deviceInfo.serialNumber));
            strncpy_r(OUT deviceInfo.deviceVersion, IN "1.0", IN sizeof(deviceInfo.deviceVersion));
            strncpy_r(OUT deviceInfo.UUID, IN "2", IN sizeof(deviceInfo.UUID));
            strncpy_r(OUT deviceInfo.accessoryName, IN "hw:3,0", IN sizeof(deviceInfo.accessoryName));
            deviceInfo.fileSystemType = FSTY_HFS;
            deviceInfo.partitionNumber = PARTITION_NUMBER_NONE;
            deviceInfo.totalSize = 0;
            deviceInfo.freeSize = 0;
            deviceInfo.productID = 4711;
            strncpy_r(OUT deviceInfo.appleDeviceMACAddress, IN "04f13ec34a28", IN sizeof(deviceInfo.appleDeviceMACAddress));
            deviceInfo.connectionType = DCT_USB;
            strncpy_r(OUT deviceInfo.mountPoint, IN "/dev/usb/hiddev0", IN sizeof(deviceInfo.mountPoint));

            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
            if (res) return res;
        }

        // add a second usb device for the test data for my media tests
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, "USB-2");
        if (res) {
            deviceInfo.activeSource = 0;
            deviceInfo.connected = 1;
            strncpy_r(OUT deviceInfo.deviceName, IN "USB-2", IN sizeof(deviceInfo.deviceName));
            deviceInfo.deviceType = DTY_USB;
            deviceInfo.connectionState = CS_CONNECTED;
            strncpy_r(OUT deviceInfo.serialNumber, IN "9874321", IN sizeof(deviceInfo.serialNumber));
            strncpy_r(OUT deviceInfo.deviceVersion, IN "1.1", IN sizeof(deviceInfo.deviceVersion));
            strncpy_r(OUT deviceInfo.UUID, IN "3", IN sizeof(deviceInfo.UUID)); // unique
            strncpy_r(OUT deviceInfo.accessoryName, IN "/dev/sdb", IN sizeof(deviceInfo.accessoryName));
            deviceInfo.fileSystemType = FSTY_FAT;
            deviceInfo.partitionNumber = 1;
            deviceInfo.totalSize = 4194304; //4GB
            deviceInfo.freeSize = 524288; //512MB
            deviceInfo.productID = 5652;
            deviceInfo.connectionType = DCT_USB;
#if 1 // for moduletests with data stored in view
#ifndef TARGET_BUILD
            strncpy_r(OUT deviceInfo.mountPoint, IN get_current_dir_name(), IN sizeof(deviceInfo.mountPoint));
            strncat_r(OUT deviceInfo.mountPoint, IN "/Customer/Simulation/CustomControl/stick_2", IN sizeof(deviceInfo.mountPoint));
#else
            strncpy_r(OUT deviceInfo.mountPoint, IN "/opt/bosch/test/data/GMP/stick_2", IN sizeof(deviceInfo.mountPoint));
#endif
#else // for a test with a stick on ubuntu:
            strncpy_r(OUT deviceInfo.mountPoint, IN "/media/KINGSTON", IN sizeof(deviceInfo.mountPoint));
#endif
            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
            if (res) return res;
        }

        // add a CDDA device for the test data for my media tests
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_CDROM, "CDDA");
        if (res) {
            deviceInfo.activeSource = 0;
            deviceInfo.connected = 1;
            strncpy_r(OUT deviceInfo.deviceName, IN "CDDA", IN sizeof(deviceInfo.deviceName));
            deviceInfo.deviceType = DTY_CDDA;
            deviceInfo.connectionState = CS_CONNECTED;
            strncpy_r(OUT deviceInfo.serialNumber, IN "cdda1234", IN sizeof(deviceInfo.serialNumber));
            strncpy_r(OUT deviceInfo.deviceVersion, IN "1.0.1", IN sizeof(deviceInfo.deviceVersion));
            strncpy_r(OUT deviceInfo.UUID, IN "4", IN sizeof(deviceInfo.UUID)); // unique
            strncpy_r(OUT deviceInfo.accessoryName, IN "/dev/cdda", IN sizeof(deviceInfo.accessoryName));
            deviceInfo.fileSystemType = FSTY_UNKNOWN;
            deviceInfo.partitionNumber = 1;
            deviceInfo.totalSize = 4194304; //4GB
            deviceInfo.freeSize = 524288; //512MB
            deviceInfo.productID = 767890;
            deviceInfo.connectionType = DCT_CDROM;
#if 1 // for moduletests with data stored in view
#ifndef TARGET_BUILD
            //internal cdrom strncpy_r(OUT deviceInfo.mountPoint, IN "/dev/sr0", IN sizeof(deviceInfo.mountPoint));
            strncpy_r(OUT deviceInfo.mountPoint, IN "/dev/sr2", IN sizeof(deviceInfo.mountPoint)); // external drive
#else
            strncpy_r(OUT deviceInfo.mountPoint, IN "/dev/sr0", IN sizeof(deviceInfo.mountPoint));
#endif
#else // for a test with a stick on ubuntu:
            strncpy_r(OUT deviceInfo.mountPoint, IN "/media/cdda", IN sizeof(deviceInfo.mountPoint));
#endif
            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
            if (res) return res;
        }

        // add a second usb-4 device for the test data for my media tests
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB, "USB-4");
        if (res) {
            deviceInfo.activeSource = 0;
            deviceInfo.connected = 1;
            strncpy_r(OUT deviceInfo.deviceName, IN "USB-4", IN sizeof(deviceInfo.deviceName));
            deviceInfo.deviceType = DTY_USB;
            deviceInfo.connectionState = CS_CONNECTED;
            strncpy_r(OUT deviceInfo.serialNumber, IN "9874321-4", IN sizeof(deviceInfo.serialNumber));
            strncpy_r(OUT deviceInfo.deviceVersion, IN "1.1", IN sizeof(deviceInfo.deviceVersion));
            strncpy_r(OUT deviceInfo.UUID, IN "4", IN sizeof(deviceInfo.UUID)); // unique
            strncpy_r(OUT deviceInfo.accessoryName, IN "/dev/sdb", IN sizeof(deviceInfo.accessoryName));
            deviceInfo.fileSystemType = FSTY_FAT;
            deviceInfo.partitionNumber = 1;
            deviceInfo.totalSize = 4194304; //4GB
            deviceInfo.freeSize = 524288; //512MB
            deviceInfo.productID = 5652;
            deviceInfo.connectionType = DCT_USB;
#if 1 // for moduletests with data stored in view
#ifndef TARGET_BUILD
            strncpy_r(OUT deviceInfo.mountPoint, IN get_current_dir_name(), IN sizeof(deviceInfo.mountPoint));
            strncat_r(OUT deviceInfo.mountPoint, IN "/Customer/Simulation/CustomControl/stick_4", IN sizeof(deviceInfo.mountPoint));
#else
            strncpy_r(OUT deviceInfo.mountPoint, IN "/opt/bosch/test/data/GMP/stick_4", IN sizeof(deviceInfo.mountPoint));
#endif
#else // for a test with a stick on ubuntu:
            strncpy_r(OUT deviceInfo.mountPoint, IN "/media/KINGSTON", IN sizeof(deviceInfo.mountPoint));
#endif
            res = LocalSPM::GetDBManager().AddDevice(INOUT deviceInfo);
            if (res) return res;
        }
    }

    if(createSongs)
    {
        /* scan USB device */
        res = LocalSPM::GetDBManager().GetDevice(OUT deviceID, DTY_USB);
        if (res) return res;

        /* start scanning the device */
        res = ScanDirForTest(deviceID);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;

        /* scan internal flash */
        tNumberOfDevices numberOfDevices;
        vector<tDeviceInfo> deviceInfos;

        /* get only connected device connections */
        res = LocalSPM::GetDBManager().GetMediaplayerDeviceConnections(OUT numberOfDevices, OUT deviceInfos);
        if (res && res != MP_ERR_DB_END_OF_LIST) return res;

        for(tUInt i=0; i<deviceInfos.size(); i++)
        {
            if(DTY_FLASH == deviceInfos[i].deviceType)
            {
                /* start scanning the device */
                res = ScanDirForTest(deviceInfos[i].deviceID);
                if (res && res != MP_ERR_DB_END_OF_LIST) return res;
            }
        }

        /* wait for end of all work */
        res = LocalSPM::GetDBManager().WaitForUpdateMyMediaFinished();
        if (res) return res;
    }

    return MP_NO_ERROR;
}

tResult Database::ScanDirForTest(const tDeviceID deviceID)
{
    ENTRY_INTERNAL
    tResult res;

    /* start scanning the device */
    res = ScanDirectory(IN "/", IN deviceID);
    if (res) return res;

    /* flush the media object cache */
    res = LocalSPM::GetDBManager().StoreMediaObjectEnd();
    if (res) return res;

    /* determine the media content (media types) of the folders inclusive there subfolders */
    res = DetermineFolderContent(IN mDatabase, IN deviceID);
    if (res) return res;

    /* set indexing state */
    tIndexingState indexingState = IDS_COMPLETE;
    tIndexingPercentComplete percent = 100;
    LocalSPM::GetDBManager().SetIndexingState(IN deviceID, IN indexingState, IN percent);

    /* set number of files */
    LocalSPM::GetDBManager().RecalculateNumberOfFiles(IN deviceID);

    /* set this device active */
    LocalSPM::GetDBManager().SetActiveDevice(IN deviceID);

    return MP_NO_ERROR;
}

tResult Database::GetFileFormat(tFileFormat &fileFormat, tFileType &fileType, const char *path) const
{
    ENTRY_INTERNAL

    /* init the return value */
    fileFormat = FFT_UNKNOWN;
    fileType = FT_UNKNOWN;

    FastUTF8::tString localPath;
    FastUTF8::tString extension;

    // double the path
    localPath = (FastUTF8::tString)strdup(path);
    FastUTF8::SplitExtension(OUT extension, INOUT localPath);

    /* extension found */
    if (extension)
    {
        // set a dot before the pure extension string
        extension--; // this must NOT be a FastUTF8 step because we want to put the . directly before the current char.
        *extension = '.';

        //ETG_TRACE_USR1(("GetFileFormat: mVTFileContext.registeredFileTypes.size(): %d", mVTFileContext.registeredFileTypes.size()));

        /* look for the file format and type */
        for(unsigned int i=0; i<mVTFileContext.registeredFileTypes.size(); i++) {

            if (FastUTF8::EndsWithNC((const FastUTF8::tString)extension, (const FastUTF8::tString)mVTFileContext.registeredFileTypes[i].extension)) {
                fileFormat = (tFileFormat)(mVTFileContext.registeredFileTypes[i].fileFormat);
                fileType = (tFileType)(mVTFileContext.registeredFileTypes[i].fileType);
                free(localPath);
                return MP_NO_ERROR;
            }
        }
    }

    /* file type not found */
    free(localPath);
    return MP_ERR_DB_FILE_TYPE_NOT_FOUND;
}

int Database::GetNumberOfVTCacheElements(int *cacheEntries, int *cacheHits, int *cacheMisses)
{
    if (cacheEntries) *cacheEntries = mVTFileContext.mNumberOfVTCacheElements;
    if (cacheHits)    *cacheHits = mVTFileContext.mCacheHits;
    if (cacheMisses)  *cacheMisses = mVTFileContext.mCacheMisses;

    return mVTFileContext.mNumberOfVTCacheElements;
}
tResult Database::ReleaseDatabaseMemory(sqlite3 *dbHandle)
{
    return sqlite3_db_release_memory(dbHandle);
}

tResult Database::DatabaseStatus(sqlite3 *dbHandle, int op)
{
    int Cur = 0, Hiwtr = 0;
    sqlite3_db_status(dbHandle, op, &Cur, &Hiwtr, 1);
    //ETG_TRACE_USR1(("DatabaseStatus: Operation = %d, Current Value = %d", op, Cur));
    return Cur;
}

tResult Database::AllocatedMemory(void)
{
    int MemoryUsed = 0;
    MemoryUsed = sqlite3_memory_used();
    //ETG_TRACE_USR1(("AllocatedMemory: Used Memory = %d", MemoryUsed));
    return MemoryUsed;
}

tResult Database::PrepareStatementCount(sqlite3 *dbHandle)
{
    // jump to beggining
    sqlite3_stmt* statement = NULL;
    int count = 0;

    statement = sqlite3_next_stmt(dbHandle, NULL);
    while (statement != NULL)
    {
        statement = sqlite3_next_stmt(dbHandle, statement);
        count++;
    }

    //ETG_TRACE_USR1(("PrepareStatementCount: open statement count = %d", count));
    return count;
}

