/*
 * DBTriggerManager.cpp
 *
 *  Created on: May 10, 2013
 *      Author: tritonsu
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/DBTriggerManager.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_DB_MANAGER
#endif
#endif


#include <string.h>
#include <stdlib.h>
#include "FunctionTracer.h"
#include "VarTrace.h"
#include "DBTriggerManager.h"
#include "LocalSPM.h"
#include "Dispatcher.h"


#define SQL_INSERT_TRIGGER "CREATE TRIGGER  IF NOT EXISTS %s AFTER INSERT ON %s %s BEGIN SELECT %s('%d','%d', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
#define SQL_DELETE_TRIGGER "CREATE TRIGGER  IF NOT EXISTS %s AFTER DELETE ON %s %s BEGIN SELECT %s('%d','%d', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
#define SQL_UPDATE_TRIGGER "CREATE TRIGGER  IF NOT EXISTS %s AFTER UPDATE %s %s ON %s %s BEGIN SELECT %s('%d','%d', '%s', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
#define TRIGGER_DROP      "DROP TRIGGER %s"
#define COMMA_DELIMITER ','
#define UNDERSCORE_DELIMITER '_'


#define RETURN_IF_NO_DB if(mDbHandle == NULL) return 0;

DBTriggerManager::DBTriggerManager(void)
{
    mDbHandle   = NULL;
    mTriggerID  = 1;
}


#define CB_FUNCTION_NAME "TriggerCallbackFn"
void TriggerCallbackFn(sqlite3_context *context, int num_values, sqlite3_value **values)
{
    ENTRY

    if((NULL != context) && (NULL != values) && (NULL != *values))
    {
        DBTriggerManager* instance = (DBTriggerManager*) sqlite3_user_data(context);
        if(NULL != instance)
        {
            instance->TriggerCallback(num_values,values);
        }
    }
}

tResult DBTriggerManager::AttachDB(sqlite3 * dbHandle)
{
    ENTRY

    mDbHandle   = dbHandle;

    /* create sql function which will be used for trigger callback */
    return sqlite3_create_function(mDbHandle, CB_FUNCTION_NAME, -1, SQLITE_UTF8, this,TriggerCallbackFn, NULL, NULL);
}

tResult DBTriggerManager::DetachDB(sqlite3 *dbHandle)
{
    ENTRY

    if (dbHandle == NULL)
        return -1;

    DeRegisterAllTriggers();

    mDbHandle = NULL;

    return sqlite3_create_function(dbHandle, CB_FUNCTION_NAME, -1, SQLITE_UTF8, NULL, NULL, NULL, NULL);
}

tBool DBTriggerManager::ExecuteQuey(const char* statement)
{
    ENTRY_INTERNAL

    char* sqlError;
    ETG_TRACE_USR3(("DBTriggerManager::Execute trigger statement: %s",statement));
    int sqlite3ResultCode = sqlite3_exec (mDbHandle, statement, 0, 0, &sqlError);
    VARTRACE(sqlite3ResultCode);
    if (SQLITE_OK != sqlite3ResultCode)
    {
        ETG_TRACE_ERR(("DBTrigger:Error while executing SQL statement(%200s) -> %s", statement, sqlError));
        sqlite3_free(sqlError);

        if (SQLITE_CORRUPT == sqlite3ResultCode) {
            /* write a file onto the disk to issue a data base file delete on next restart of system */
            FILE *fp;
            fp = fopen(Database::mDatabaseRecreateFile, "r");
            /* if no recreate file exists create a new one */
            if (!fp) {
                fp = fopen(Database::mDatabaseRecreateFile, "w");
                ETG_TRACE_FATAL (("created: sqlite3ResultCode=%d, %s", sqlite3ResultCode, Database::mDatabaseRecreateFile));
                ETG_TRACE_ERRMEM(("created: sqlite3ResultCode=%d, %s", sqlite3ResultCode, Database::mDatabaseRecreateFile));
            }
            fclose(fp);
        }

        return false;
    }
    return true;
}

tBool DBTriggerManager::TriggerCallback(int num_values, sqlite3_value **values)
{
    ENTRY

    mCallbackLock.lock();

    tTriggerMode type = (tTriggerMode)atoi((const char*)sqlite3_value_text(values[0]));
    tTriggerID triggerID = atoi((const char*)sqlite3_value_text(values[1]));

    //look for the trigger context matching the context and trigger the dispatcher message
    for(unsigned int iter = 0 ; iter < mActiveTriggers.size() ; iter++)
    {
        if(mActiveTriggers.at(iter)->triggerID == triggerID)
        {
            char message[128];
            strncpy_r(message,mActiveTriggers.at(iter)->smName,sizeof(message));
            strncat_r(message,"::",sizeof(message));
            strncat_r(message,mActiveTriggers.at(iter)->msgName,sizeof(message));

            char parameters[128];
            if(type == TRIGGER_ON_UPDATE)
            {
                LocalSPM::GetInstance().Marshal(parameters, sizeof(parameters)-1, "iit", sqlite3_value_int(values[4]), sqlite3_value_int(values[5]),sqlite3_value_text(values[3]));
            }
            else
            {
                LocalSPM::GetInstance().Marshal(parameters, sizeof(parameters)-1, "ii", sqlite3_value_int(values[3]), sqlite3_value_int(values[4]));
            }
            ETG_TRACE_USR3(("DBTriggerManager::TriggerCallback-> sm msg:%128s,parameters:%s",message,parameters));
            Dispatcher::GetInstance().SendMessage(message,parameters);
        }
    }

#ifndef TARGET_BUILD //debug
    switch(type)
    {
        case TRIGGER_ON_INSERT:
            ETG_TRACE_USR3(("###### TRIGGER_ON_INSERT :%d",num_values));
            ETG_TRACE_USR3(("triggerID:%d  , count:%d , inserted rowID:%d , tableName:%s",triggerID,sqlite3_value_int(values[3]),sqlite3_value_int(values[4]),sqlite3_value_text(values[2])));
            break;
        case TRIGGER_ON_DELETE:
            ETG_TRACE_USR3(("###### TRIGGER_ON_DELETE :%d",num_values));
            ETG_TRACE_USR3(("triggerID:%d  , count:%d , deleted rowID:%d , tableName:%s",triggerID,sqlite3_value_int(values[3]),sqlite3_value_int(values[4]),sqlite3_value_text(values[2])));
            break;
        case TRIGGER_ON_UPDATE:
            ETG_TRACE_USR3(("###### TRIGGER_ON_UPDATE : %d",num_values));
            ETG_TRACE_USR3(("triggerID:%d   , count:%d , updated rowID:%d , tableName:%128s,column name:%s",triggerID,sqlite3_value_int(values[4]),sqlite3_value_int(values[5]),sqlite3_value_text(values[2]),sqlite3_value_text(values[3])));
            break;
    }
#else
    (void)num_values;
#endif

    mCallbackLock.unlock();

    return true;
}

tTriggerID DBTriggerManager::CreateTrigger(tTriggerMode type, const char* tableName,
                                                         const char* callbackSMName,
                                                         const char* callbackMsgName,
                                                         const char* clause ,
                                                         const char* columnName,
                                                         const char* cbArg1,
                                                         const char* cbArg2)
{
    ENTRY;
    RETURN_IF_NO_DB;

    /*frame trigger name which is combination of table name , column name and clause also column name if required*/
    char triggerName[512];

    /*prepare sql statement to register for a trigger*/
    char statement[1024];
    mTriggerID++;
    switch(type)
    {
        case TRIGGER_ON_INSERT:
            sprintf (triggerName , "dbmanager%s_ON_INSERT_%d",tableName,mTriggerID);
//"CREATE TRIGGER  IF NOT EXISTS %s AFTER INSERT ON %s %s BEGIN SELECT %s('%d','%d', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
            sprintf (statement, SQL_INSERT_TRIGGER, triggerName, tableName,clause, CB_FUNCTION_NAME,
                    TRIGGER_ON_INSERT, mTriggerID,tableName, cbArg1, cbArg2, tableName);
            break;
        case TRIGGER_ON_DELETE:
            sprintf (triggerName , "dbmanager%s_ON_DELETE_%d",tableName,mTriggerID);
//"CREATE TRIGGER  IF NOT EXISTS %s AFTER DELETE ON %s %s BEGIN SELECT %s('%d','%d', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
            sprintf (statement, SQL_DELETE_TRIGGER, triggerName, tableName,clause, CB_FUNCTION_NAME,
                    TRIGGER_ON_DELETE, mTriggerID,tableName, cbArg1, cbArg2, tableName);
            break;
        case TRIGGER_ON_UPDATE:
        {
            char delimitedColumnName[512] = {0};

            /*RTC-341485:For multiple columns, replace comma with underscore in trigger name, to avoid syntax errors*/
            for(int i = 0; (i < strlen_r(columnName)) && (i < sizeof(delimitedColumnName) - 1); i++)
            {
                if(COMMA_DELIMITER == columnName[i])
                    delimitedColumnName[i] = UNDERSCORE_DELIMITER;
                else
                    delimitedColumnName[i] = columnName[i];
            }

            sprintf (triggerName , "dbmanager%s_ON_UPDATE_OF_%s_%d",tableName,delimitedColumnName,mTriggerID);
//"CREATE TRIGGER  IF NOT EXISTS %s AFTER UPDATE %s %s ON %s %s BEGIN SELECT %s('%d','%d', '%s', '%s', (%s),(%s),COUNT()/*to force for-each-statement*/) from %s; END"
            if(strlen_r(columnName) > 0)
                sprintf (statement, SQL_UPDATE_TRIGGER, triggerName, "OF" ,columnName , tableName ,clause ,CB_FUNCTION_NAME,
                        TRIGGER_ON_UPDATE, mTriggerID,tableName,columnName, cbArg1, cbArg2, tableName);
            else
                sprintf (statement, SQL_UPDATE_TRIGGER, triggerName, "" , "" ,tableName ,clause ,CB_FUNCTION_NAME,
                        TRIGGER_ON_UPDATE, mTriggerID,tableName,columnName, cbArg1, cbArg2, tableName);
            break;
        }
        default:
            ETG_TRACE_ERR(("DBTrigger::CreateTrigger -> Invalid trigger type received:%d",type));
            return 0;
    }

    /* execute the sql statement to register the trigger */
    if(ExecuteQuey(statement) == false)
        return 0;

    /*registration for the trigger is success full , hence store its context for future use by call back or unregister*/
    tTriggerContext *context = new tTriggerContext; //lint -e429 context will be freed in DeRegister or DeRegisterAllTriggers
    context->type = type;
    context->triggerID = mTriggerID;
    strncpy_r(context->smName,callbackSMName, sizeof(context->smName));
    strncpy_r(context->msgName,callbackMsgName, sizeof(context->msgName));
    strncpy_r(context->triggerName,triggerName, sizeof(context->triggerName));
    mActiveTriggers.push_back(context);

    return mTriggerID;
}

tTriggerID DBTriggerManager::RegisterOnInsert(const char* tableName, const char* callbackSMName , const char* callbackMsgName, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY;
    RETURN_IF_NO_DB;

    return CreateTrigger( TRIGGER_ON_INSERT, tableName, callbackSMName , callbackMsgName , clause, "", cbArg1, cbArg2);
}

tTriggerID DBTriggerManager::RegisterOnDelete(const char* tableName, const char* callbackSMName , const char* callbackMsgName, const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY;
    RETURN_IF_NO_DB;

    return CreateTrigger( TRIGGER_ON_DELETE, tableName, callbackSMName , callbackMsgName , clause, "", cbArg1, cbArg2);
}

tTriggerID DBTriggerManager::RegisteronUpdate(const char* tableName, const char* callbackSMName , const char* callbackMsgName ,const char* columnName , const char* clause, const char* cbArg1, const char* cbArg2)
{
    ENTRY;
    RETURN_IF_NO_DB;

    return CreateTrigger( TRIGGER_ON_UPDATE, tableName, callbackSMName , callbackMsgName , clause , columnName, cbArg1, cbArg2);
}

tBool DBTriggerManager::DeRegister(tTriggerID triggerID)
{
    ENTRY;
    RETURN_IF_NO_DB;
    //tResult ret;

    ETG_TRACE_USR3(("DBTriggerManager::DeRegister : %d",triggerID));

    unsigned int iter;
    for (iter = 0 ; iter < mActiveTriggers.size(); iter++)
    {
        /*look if triggerID is already registered one*/
        if(triggerID == mActiveTriggers.at(iter)->triggerID)
        {
            /*Drop the trigger matching for the requested triggerID*/
            char statement[512];
            sprintf(statement, TRIGGER_DROP, mActiveTriggers.at(iter)->triggerName);
            /*ret =*/ ExecuteQuey(statement);
            delete mActiveTriggers[iter];
            mActiveTriggers.erase (mActiveTriggers.begin()+iter);
            return true;
        }
    }

    ETG_TRACE_ERR(("DBTrigger::UnRegister -> No trigger was known with ID:%d",triggerID));
    return false;
}

tBool DBTriggerManager::DeRegisterAllTriggers(void)
{
    ENTRY;
    RETURN_IF_NO_DB;
    tResult ret = 0;

    unsigned int iter;
    for (iter = 0 ; iter < mActiveTriggers.size(); iter++)
    {
        /*Drop the trigger matching for the requested triggerID*/
        char statement[512];
        sprintf(statement, TRIGGER_DROP, mActiveTriggers.at(iter)->triggerName);
        ret = ExecuteQuey(statement);
        delete mActiveTriggers[iter];
    }
    mActiveTriggers.clear();
    return ret;
}


tResult DBTriggerManager::GetTriggers(vector<string> &triggerNames)
{
    ENTRY

    tResult res;
    Query query;

    /* do a select on the sqlite master table */
    res = query.Select(LTY_SQLITE_MASTER, MY_MEDIA, "it", IN 0, IN "trigger");
    if (res) return res;

    /* do a loop over all triggers in table */
    while(1) {
        char triggerName[64];
        /* get one name */
        res = query.Get("-T", OUT triggerName, IN sizeof(triggerName));
        if (res) break;

        ETG_TRACE_USR3(("DBManager::GetTriggers trigger:%s",triggerName));

        /* is it a db manager trigger? than store it */
        if (!strncmp("dbmanager", triggerName, strlen_r("dbmanager"))) {
            triggerNames.push_back(triggerName);
        }
    }
    query.End();

    return MP_NO_ERROR;
}

tResult DBTriggerManager::TriggerCleanupDB()
{
    ENTRY

    vector<string> triggerNames;
    GetTriggers(OUT triggerNames);

    /* delete all triggers from data base */
    for(unsigned int i=0; i<triggerNames.size(); i++) {
        char queryString[256];
        snprintf(queryString, sizeof(queryString)-1, "DROP TRIGGER IF EXISTS %s;", triggerNames[i].c_str());
        ExecuteQuey(queryString);
    }

    return MP_NO_ERROR;
}
