/**
 * @defgroup DBManager DBManager
 * @ingroup Mediaplayer
 * @author Matthias Dick
 *
 * Public interface for database manager to provide the database access.
 * @{
 */

#ifndef DBMANAGER_H_
#define DBMANAGER_H_

#include "ILocalSPM.h"
#include "TypeDefinitions.h"
#include "Lock.h"
#include <sqlite3.h>
#include "Database.h"
#include "Query.h"
#include "ThreadFactory.h"
#include "Timer.h"
#include "MyMediaUpdateManager.h"

#include <map>
#include <vector>
#include <string>
using namespace std;

#define DB_UNDEFINED_VALUE (unsigned int )0

class DBManager : public ILocalSPM, public TFThread
{

public:

    /**
     * Brief: Constructor of DBManager component which creates an instance.
     *        No further initialization of internal components and resources
     *        is performed in the constructor. See the explicit allocation
     *        and initialization functions Create() and Init(). Hence, the
     *        implementation is trivial.
     *
     * Synchronization: not re-entrant.
     *
     * Performance considerations: none.
     *
     * @return void
     */
    DBManager(tComponentID componentID):ILocalSPM(componentID){};

    /**
     * Destructor of component.
     *
     * @return void
     */
    virtual ~DBManager(){};

    // ***********************
    // * SPM related methods *
    // ***********************

    /**
     * Brief: performs an explicit allocation of the resources used by
     * the DBManager internally.
     *
     * Synchronization: not re-entrant.
     *
     * Performance considerations: none.
     *
     * @return void
     */
    void Create();

    /**
     * Brief: performs the initialization of the DBManager and its
     *        subcomponents, i.e.:
     *        - initialization of member variables;
     *        - establishment of a database connection;
     *        - integrity check of the database;
     *        - initialization of temporary and virtual tables;
     *        - create internal flash devices; //Roadmap 13005
     *
     * Synchronization: re-entrant.
     *
     * Performance considerations: can take up to 1 second if the database
     *                             recreation is required.
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    tResult Init(tInitReason reason); //Roadmap 13005: 100%

    /**
     * Brief: starts the component, implies that all other mediaplayer
     *        components are available.
     *        Since the DBManager does not require a state machine, the
     *        implementation of this function is trivial.
     *
     * Synchronization: re-entrant.
     *
     * Performance considerations: none.
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    tResult Run();

    /**
     * Brief: stops the component.
     *        Since the DBManager does not require a state machine, the
     *        implementation of this function is trivial.
     *
     * Synchronization: re-entrant.
     *
     * Performance considerations: none.
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    tResult Stop();

    /**
     * Brief: cleans up the resources used by the component, i.e:
     *        - release the database resources (e.g. prepared statements);
     *        - closing the database connection(s).
     *
     * Synchronization: re-entrant.
     *
     * Performance considerations: none.
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    tResult Done();

    /**
     * Brief: returns statistic data as a string for the over all media player object statitcs
     *
     * @return MEDIAPLAYER_SUCCESS on success and an error code otherwise.
     */
    int Statistics(OUT tStatistics stat);

    // ******************************
    // * End of SPM related methods *
    // ******************************

    /**
     * Function stores list attributes as last mode for a device
     * Reset all active device flags in Lastmode table
     * Add new entry into Lastmode table
     *
     * @param[in] listID ID of DB list
     * @param[in] objectID current active objectID
     * @param[in] position current active position in the list
     * @param[in] playtime current playtime
     * @return < 0: error, = 0: OK
     */
    virtual tResult StoreLastPlayed(const tListID listID, const tObjectID objectID, const tPosition position, const tPlaytime playtime);

    /**
     * Function deletes the last mode for a device from the LastMode table
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult RemoveLastPlayed(const tDeviceID deviceID); //Roadmap 13023: 100%

    /**
     * Function creates last mode list and delivers last mode attributes for a device
     * Get entry from Lastmode table for a specific device
     * Create list with retrieved last mode list attributes
     * Position to the last active object
     *
     * @param[in] deviceID specific deviceID
     * @param[out] listID ID of DB list
     * @param[out] objectID last active objectID
     * @param[out] playtime last playtime
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetLastPlayed(tListID &listID, tObjectID &objectID, tPlaytime &playtime, const tDeviceID deviceID);

    /**
     * Function stores the current playtime of a media object in the LastPlaytimes table
     * Roadmap 14017
     * if(playtime == 0)
     * - Remove last playtime for media object from DB -> RemoveLastPlaytime(objectID)
     *
     * @param[in] objectID ID of the media object
     * @param[in] playtime current play position of media object
     * @return < 0: error, = 0: OK
     */
    virtual tResult StoreLastPlaytime(const tObjectID objectID, const tPlaytime playtime); //Roadmap 14017: 100%

    /**
     * Function deletes the last playtime for a media object from the LastPlaytimes table
     * Roadmap 14017
     *
     * @param[in] objectID ID of the media object
     * @return < 0: error, = 0: OK
     */
    virtual tResult RemoveLastPlaytime(const tObjectID objectID); //Roadmap 14017: 100%

    /**
     * Function delivers the last playtime for a media object from the LastPlaytimes table
     * Roadmap 14017
     *
     * @param[in] objectID ID of the media object
     * @param[out] playtime last played position of the requested media object
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetLastPlaytime(tPlaytime &playtime, const tObjectID objectID); //Roadmap 14017: 100%

    /**
     * Function clears the last mode of a device (last list, last song, last position, repeat mode, playback mode)
     * Reset shuffle mode -> SetPlaybackMode(PBM_NORMAL)
     * Reset repeat mode -> SetRepeatMode(DBDefaultRepeatMode())
     * Clear not playable flag for all media objects of this device in DB -> ClearNotPlayableFlag(deviceID) //Roadmap 13012
     * Get clear last mode configuration from DataProvider -> ClearLastMode() //Roadmap 13023
     * if( clear last mode is configured )
     * - Clear last mode for this device
     *
     * @param[in] deviceID ID of the device
     * @return < 0: error, = 0: OK
     */
    virtual tResult ClearLastMode(const tDeviceID deviceID); //Roadmap 13012: 100%, 13023: 100%

    /**
     * Function resets all elements in the DB related to this device to playable again
     * Roadmap 13012
     * Recalculate number of playable files //Roadmap 13002
     *
     * @param[in] deviceID device ID of affected device
     * @return != 0: error, = 0: OK
     */
    virtual tResult ClearNotPlayableFlag(const tDeviceID deviceID); //Roadmap 13012: 100%, 13002: 100%

    virtual tResult StoreBTMACAddress(const tBTMACAddress address);
    virtual tResult GetBTMACAddress(tBTMACAddress &address);
    virtual tResult StoreScanContext(const tScanContext scanContext);
    virtual tResult RemoveScanContext(const tDeviceID deviceID);
    virtual tResult GetScanContext(tScanContext &scanContext, const tDeviceID deviceID);

    /**
     * Function stores cover art path and thumbnail blob for an album
     * Roadmap 15009
     *
     * @param[in] deviceID device ID of affected device
     * @param[in] albumID album ID
     * @param[in] coverArtPath path on flash to the related cover art
     * @param[in] thumbnailBlob blob struct of thumbnail image
     * @return < 0: error, = 0: OK
     */
    virtual tResult StoreAlbumArt(const tDeviceID deviceID, const tAlbumID albumID,const tTitle title,
            const tAlbumArt coverArtPath, const tImageBlob thumbnailBlob,const tDateTime lastModifiedTime); //Roadmap 15009: 100%

    /**
     * Function deletes all entries for a device from the AlbumArts table
     * Roadmap 15009
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult RemoveAlbumArts(const tDeviceID deviceID); //Roadmap 15009: 100%

    /**
     * Function delivers the cover art path from the AlbumArts table
     * Roadmap 15009
     *
     * @param[in] deviceID device ID of affected device
     * @param[in] albumID album ID
     * @param[out] coverArtPath path on flash to the related cover art
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetCoverArtPath(tAlbumArt &coverArtPath, const tDeviceID deviceID, const tAlbumID albumID); //Roadmap 15009: 100%

    /**
     * Function delivers the cover art path from the AlbumArts table
     * Roadmap 15009
     *
     * @param[in] deviceID device ID of affected device
     * @param[in] objectID Object's id
     * @param[out] coverArtPath path on flash to the related cover art
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetCoverArtPathByObjectId(tAlbumArt &coverArtPath, const tDeviceID deviceID, const tObjectID objectID);
    /**
     * Function delivers the thumbnail image from the AlbumArts table
     * Roadmap 15009
     *
     * @param[in] deviceID device ID of affected device
     * @param[in] albumID album ID
     * @param[out] thumbnailBlob blob struct of thumbnail image
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetThumbnailImage(tImageBlob &thumbnailBlob, const tDeviceID deviceID, const tAlbumID albumID); //Roadmap 15009: 100%

    virtual tResult GetSearchKeyboard(vector<tSearchKeyboard> &searchKeyboardList, const tListID listID);
    virtual tResult GetSearchKeyList(vector<tSearchKey> &searchKeys, const tListID listID);

    /**
     * Function maps device properties to device info for each device in the vector
     * Set deviceState to DS_INITIALIZING if connected, to DS_NONE if disconnected //Roadmap 13001
     * Map the disconnect reason in case of connectionState == CS_DISCONNECTED //Roadmap 15003
     *
     * @param[in] deviceInfos vector of device properties
     * @return != 0: error, = 0: OK
     */
    //virtual tResult DeviceChanged(const vector<tDeviceProperty>deviceProperties); //Roadmap 13001: 100%, 15003: 70%
    virtual tResult DeviceChanged(const vector<tDeviceInfo>deviceInfos); //Roadmap 13001: 100%, 15003: 70%
    virtual tResult GetListInfo(tListInfo &listInfo, const tListID listID, const tBoolean withListSize);
    virtual tResult GetPositionInList(tPosition &position, const tListID listID, const tObjectID objectID);
    virtual tResult GetPositionInParentFileList(tPosition &position, const tListID listID, const tPath nowPlayingFolder);
    virtual tResult SetLanguage(const tLanguageType language);
    virtual tResult GetLanguage(tLanguageType &language);
    virtual tResult SetMediaContext(const tMediaContext mediaContext);
    virtual tResult GetMediaContext(tMediaContext &mediaContext);
    tFileTypeSelection GetFTSByActiveMediaContext() const;
    tFileTypeSelection GetFTSByDefaultMediaContext() const;
    tFileTypeSelection GetFTSByMC(const tMediaContext mediaContext) const;

    virtual tResult GetObjectID(tObjectID &objectID, const tDeviceID deviceID, const tURL url);
    virtual tResult GetObjectType(tCategoryType &objectType, tObjectID &objectID);
    virtual tResult GetObjectDeviceID(tDeviceID &deviceID, const tObjectID objectID, const tCategoryType objectType);
    virtual tResult GetObjectDeviceID(vector<tDeviceID> &deviceIDs, const tObjectID objectID, const tCategoryType objectType);
    virtual tResult GetMediaObject(tMediaObject &mediaObject, tMediaObjectID &mediaObjectID);

    /* USBPortNumberInfo CAFI06-122 */
    virtual tResult UpdatePortNumberInfo(tDeviceID deviceID, tUSBPortNumber portNumber, const tConnectionState connectionstate);
    virtual tResult GetDevicePortNumberInfo(tDeviceID deviceID, tUSBPortNumber &retPortNumber);

    /* DeviceSysPathInfo CMG3G-12660 */
    virtual tResult UpdateSysPathInfoToMap(tDeviceID deviceID, tPath deviceSyspath, const tConnectionState connectionstate);
    virtual tResult GetSysPathInfoFromMap(tDeviceID deviceID, tPath &retDeviceSyspath);

    /**
     * Function returns a media object information
     *
     * special function: if user requests for example a genre "media object" this
     * function will return the info of first song found in this genre
     *
     *  if categoryType == 0
     *      DBManager::GetObjectType(&categoryType,mediaObjectId)
     *
     *  if song type:
     *     DBManager::GetMediaObject(&object,mediaObjectId)
     *     create media object out of it
     *
     *  if category type (eg: CTY_ALBUM)
     *     DBManager::GetAlbum(&albumTag,&albumName,mediaObjectId);
     *     DBManager::GetArtistOfAlbum(&artistTag,&artistName,albumTag);
     *     DBManager::GetGenreOfAlbum(&genreTag,&genreName,lbumTag);
     *     create media object
     *
     * @param[in] mediaObjectID id of the media object
     * @param[in] categoryType type id for a category (optional) to shorten the search
     * @param[out] mediaObject filled object with all media object data
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetMediaObject(tMediaObject &mediaObject, const tMediaObjectID mediaObjectID,
            const tCategoryType categoryType, const tDeviceID deviceID = MY_MEDIA,const tBoolean isRequestFromInterface = false);

    /**
     * Function updates media object in DB (notPlayable flag)
     * Roadmap 13012
     * Called by PlayerManager in stream error case
     *
     * @param[in] mediaObject media object to change the notPlayable flag
     * @return != 0: error, = 0: OK
     */
    virtual tResult UpdateMediaObject(const tMediaObject mediaObject); //Roadmap 13012: 100%

    virtual tResult SetCurrentRow(const tListID listID, const tRowNumber newRow);
    virtual tResult GetCurrentRow(tRowNumber &currentRow, const tListID listID);
    virtual tResult SetPosition(const tObjectID objectID, const tListID listID);
    virtual tResult SetPosition(tPosition &position, const tObjectID objectID, const tListID listID);
    virtual tResult SetPlaylistAttributes(const tListID listID, const tDeviceID deviceID, const tFilterTag1 filterTag1);
    virtual tResult RemoveMediaObject(const tMediaObjectID mediaObjectID);

    /**
     * Function removes a mediaObject defined by its UUID
     * Called by Indexer in partial or DELTA indexing mode (see Roadmap 13006 IAP2)
     *
     * @param[in] deviceID ID of the device
     * @param[in] UUID UUID of mediaObject to be removed from DB
     * @return != 0: error, = 0: OK
     */
    tResult RemoveMediaObjectByUUID(const tDeviceID deviceID, const tUUID UUID) const; //Roadmap 13006 IAP2: 100%

    virtual tResult RemoveMediaObjects(const tDeviceID deviceID);
    //virtual tResult RequestMediaPlayerFileListSlice(vector<tFiles> &mediaFileVector, const tDeviceID deviceID,
    //        const tPath path, const tIndex startIndex, const tIndex sliceSize);
    virtual tResult CreateList(tListID &dbListID, const tObjectID objectID,
            const tListType listType, const tPlayContinuation playContinuation, const tDeviceID deviceID);
    virtual tResult CreateList(tListID &listID, const tDeviceID deviceID, const tListType listType, const tPlayContinuation playContinuation,
            const tFilterTag1 filterTag1, const tFilterTag2 filterTag2, const tFilterTag3 filterTag3, const tFilterTag4 filterTag4,
            const tStreaming streaming = false, const tBoolean checkVTIPOD = true, const tBoolean lastPlayedList = false, const tPath lastPlayedPath = "",
            const char *pSearchString = NULL, tUserID userID=0); //additional param for Roadmap 17001 Personalization
    virtual tResult GetCurrentMediaObject(tMediaObject &mediaObject, const tListID listID);
    virtual tResult GetIndexingState(vector<tIndexingStateResult> &indexingState);
    virtual tResult GetIndexingState(tIndexingState &indexingState, const tDeviceID deviceID);
    virtual tResult SetIndexingState(const tDeviceID deviceID,
            const tIndexingState indexingState, const tIndexingPercentComplete percent);
    virtual tResult GetFingerprint(tFingerprint &fingerprint, const tDeviceID deviceID);
    virtual tResult SetFingerprint(IN const tDeviceID& deviceID, IN const tFingerprint fingerprint);
    virtual tResult SetFingerprint(const tDeviceID deviceID, const tFingerprint fingerprint, const tNumberOfFiles numberOfAudioFiles, const tNumberOfFiles numberOfVideoFiles); //Roadmap 13002: 100%

    /**
     * Function gets the number of files for a device
     * Roadmap 13002
     *
     * @param[in] deviceID ID of the device
     * @param[out] numberOfAudioFiles number of audio files in that device
     * @return != 0: error, = 0: OK
     */
    tResult GetNumberOfAudioFiles(tNumberOfFiles &numberOfAudioFiles, const tDeviceID deviceID) const; //Roadmap 13002: 100%

    /**
     * Function gets the number of files for a device
     * Roadmap 13002
     *
     * @param[in] deviceID ID of the device
     * @param[out] numberOfPlayableFiles number of playable files(audio + video) in that device
     * @return != 0: error, = 0: OK
     */
    tResult GetNumberOfPlayableFiles(tNumberOfFiles &numberOfPlayableFiles, const tDeviceID deviceID) const; //Roadmap 13002: 100%

    /**
     * Function sets the number of files for a device
     * Roadmap 13002
     *
     * @param[in] deviceID ID of the device
     * @param[in] numberOfAudioFiles number of playable audio files in that device
     * @return != 0: error, = 0: OK
     */
    tResult SetNumberOfAudioFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfAudioFiles); //Roadmap 13002: 100%


    /**
     * Function sets the number of Video files for a device
     * Roadmap 1612
     *
     * @param[in] deviceID ID of the device
     * @param[in] numberOfVideoFiles number of playable video files in that device
     * @return != 0: error, = 0: OK
     */
    tResult SetNumberOfVideoFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfVideoFiles);

    /**
     * Function sets the number of Image files for a device
     * Roadmap 1612
     *
     * @param[in] deviceID ID of the device
     * @param[in] numberOfImageFiles number of playable files in that device
     * @return != 0: error, = 0: OK
     */
    tResult SetNumberOfImageFiles(const tDeviceID deviceID, const tNumberOfFiles numberOfImageFiles);

    /**
     * Function recalculates number of playable files and set it for a device
     *
     * @param[in] deviceID ID of the device
     * @return != 0: error, = 0: OK
     */
    tResult RecalculateNumberOfFiles(const tDeviceID deviceID) const;

    virtual tResult GetRepeatMode(tRepeatMode &repeatMode, const tDeviceID deviceID);
    virtual tResult ResetAllRepeatModes(const tDeviceID deviceID);
    virtual tResult SetRepeatMode(const tDeviceID deviceID, const tRepeatMode repeatMode);
    virtual tResult GetPlaybackMode(tPlaybackMode &playbackMode, const tDeviceID deviceID,const tFileTypeSelection fileTypeSelection = FTS_AUDIO);
    virtual tResult ResetAllPlaybackModes(const tDeviceID deviceID);
    virtual tResult SetPlaybackMode(const tDeviceID deviceID, const tPlaybackMode playbackMode);

    /**
     * Function sets the connection state to disconnected for a device
     * The disconnectReason is DR_REMOVED //Roadmap 15003
     *
     * @param[in] deviceID ID of the device
     * @return != 0: error, = 0: OK
     */
    virtual tResult DeviceDisconnected(const tDeviceID deviceID); //Roadmap 15003: 100%

    /**
     * Function sets the connection state to connected for a device
     *
     * @param[in] deviceID ID of the device
     * @return != 0: error, = 0: OK
     */
    virtual tResult DeviceConnected(const tDeviceID deviceID);

    /**
     * Function sets the connection state to attached for a device
     *
     * @param[in] deviceID ID of the device
     * @return != 0: error, = 0: OK
     */
    virtual tResult DeviceAttached(const tDeviceID deviceID);

    virtual tResult SetDeviceNameAndConnectionState(const tDeviceID deviceID, const tDeviceName deviceName, //Roadmap 13023: 100%
            const tConnectionState connectionState, const tDeviceState deviceState);
    virtual tResult SetDeviceType(const tDeviceID deviceID, const tDeviceType deviceType);
    tResult SetDeviceTypeAndMountPoint(const tDeviceID deviceID, const tDeviceType deviceType, const tMountPoint mountPoint);
    virtual tResult SetDeviceName(const tDeviceID deviceID, const tDeviceName deviceName);
    virtual tResult SetDiPOParameters(const tDeviceID deviceID, const tDiPOCaps diPOCaps, //Roadmap 13034: 100%
            const tDiPOActive diPOActive, const tDiPOVersion diPOVersion);

    /**
     * Function gets the album art indexing complete flag for a device
     * Roadmap 15009
     *
     * @param[in] deviceID ID of the device
     * @param[out] complete album art indexing complete flag
     * @return != 0: error, = 0: OK
     */
    virtual tResult GetAlbumArtIndexingComplete(tAlbumArtIndexingComplete &complete, const tDeviceID deviceID); //Roadmap 15009: 100%

    /**
     * Function sets the album art indexing complete flag for a device
     * Roadmap 15009
     *
     * @param[in] deviceID ID of the device
     * @param[in] complete album art indexing complete flag
     * @return != 0: error, = 0: OK
     */
    virtual tResult SetAlbumArtIndexingComplete(const tDeviceID deviceID, const tAlbumArtIndexingComplete complete); //Roadmap 15009: 100%

    /**
     * Sets the active device ID
     *
     * @param[in] deviceID device ID to set active
     * @return < 0: error, = 0: OK
     */
    virtual tResult SetActiveDevice(const tDeviceID deviceID);
    virtual tResult ActiveMediaDeviceSet(const tDeviceID deviceID, const tBoolean active);

    /**
     * Returns the last active played device ID stored in DB
     * this is not the currently playing device. for that use: @ref GetActiveDevice
     *
     * @param[out] deviceID current last active device ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetLastActiveDevice(tDeviceID &deviceID);
    virtual tResult GetLastModeDevices(tNumberOfDevices &numberOfDevices, vector<tDeviceID> &deviceIDs); //Roadmap 13023: 100%

    //Roadmap 13012 (if feature switch DBFileBrowsingByDB is enabled use list type LTY_BROWSELIST otherwise LTY_FILELIST)
    virtual tResult CreateMediaPlayerFileList(tListID &listID, tListType &listType, const tPath path, const tDeviceID deviceID,
            const tStreaming streaming = false, const tFileTypeSelection fileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);  //Roadmap 13012: 100%
    virtual tResult CreateFileListOfMediaObjects(tListID &listID, tListType &listType, const tPath path, const tDeviceID deviceID,
            const tStreaming streaming = false, const tFileTypeSelection fileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST);
    virtual tResult AddDevice(tDeviceInfo &deviceInfo); //Roadmap 13001, 13023: 100%
    virtual tResult RemoveDevice(const tDeviceID deviceID, const tBoolean virtualDevice = false);

    /**
     * Returns the active device ID
     *
     * @param[out] deviceID current active device ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetActiveDevice(tDeviceID &deviceID);

    /**
     * returns the detailed device info
     *
     * @param[in] mountPoint this is the search criteria to find the device
     * @param[out] deviceInfo filled with detailed device info
     * @return < 0: error, = 0: OK
     */
    //tResult GetDeviceInfo(tDeviceInfo &deviceInfo, const tMountPoint mountPoint);

    /**
     * Returns the detailed device info
     * Roadmap 13023 (Set deviceInfo.connected to true in case of CS_CONNECTED or CS_UNSUPPORTED)
     * Roadmap 13002, 13023, 13001 (Fill additional attributes)
     *
     * @param[in] deviceID ID of the device
     * @param[in] virtualDevice device is stored in temporary virtual device table or in normal device table of DB
     * @param[out] deviceInfo filled with detailed device info
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetDeviceInfo(tDeviceInfo &deviceInfo, const tDeviceID deviceID, const tBoolean virtualDevice = false); //Roadmap 13023: 100%, 13002: 100%, 13001: 100%

    /**
     * Returns the connected devices
     * Roadmap 13023 (Set deviceInfo.connected to true in case of CS_CONNECTED or CS_UNSUPPORTED)
     * Roadmap 13002, 13023, 13001 (Fill additional attributes)
     *
     * @param[in] allConnectionStates = 0: only connected devices, = 1: all known devices
     * @param[in] virtualDevice device is stored in temporary virtual device table or in normal device table of DB
     * @param[in] connectionState device connection state [CS_CONNECTED or CS_HW_MALFUNCTION]
     * @param[out] numberOfDevices number of connected devices
     * @param[out] deviceInfo array with all device information
     *
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetMediaplayerDeviceConnections(tNumberOfDevices &numberOfDevices, vector<tDeviceInfo> &deviceInfo, //Roadmap 13023: 100%, 13002: 100%, 13001: 100%
            const tBoolean allConnectionStates = false, const tBoolean virtualDevice = false, const tConnectionState connectionState = CS_CONNECTED);

    virtual tResult GetConnectedDeviceCount(tNumberOfDevices &numberOfDevices);
    virtual tResult GetMediaObjectDoStep(tMediaObject &mediaObject, const tListID listID, const tIndex rowNumber, const tBool uuidOnly=0);
    virtual tResult GetMediaObject(tMediaObject &mediaObject, const tListID listID, const tIndex rowNumber);
    //Roadmap 13012 (Fill mediaObject and in case of CTY_SONG also NotPlayableReason)
    virtual tResult GetMediaObjectInternal(tMediaObject &mediaObject, const tBoolean doStep,//Roadmap 13012: 100%
            const tListID listID, const tIndex rowNumber, const tBool uuidOnly = 0);
    virtual tResult GetListSize(OUT tListSize &listSize, IN const tListID listID, IN const unsigned int uMinimalExpectedSize = DB_UNDEFINED_VALUE, IN const unsigned int uOffsetForSearch = DB_UNDEFINED_VALUE);
    virtual tResult GetFileListSize(tListSize &listSize, tNumberOfMediaObjects &numberOfMediaObjects, tNumberOfPlayLists &numberOfPlayLists,
            tNumberOfFolders &numberOfFolders, const tListID listID);

    /**
     * Recreates the data base and clears all related data
     * Clear media object cache
     * Release all DB lists
     * Recreate DB and reopen it
     * Create internal flash devices //Roadmap 13005
     *
     * @return < 0: error, = 0: OK
     */
    tResult ClearMediaPlayerData(void); //Roadmap 13005: 100%

    tResult StoreMediaObject(const tMediaObject mediaObject);
    //Roadmap 13012 (Fill mediaObject)
    tResult UpdateMediaObjectInternal(Query &queryMain, const tMediaObject mediaObject); //Roadmap 13012: 100%
    tResult UpdateDirectoriesInternal(Query &queryMain, const tMediaObject mediaObject);
    tResult StoreMediaObjectInternal(Query &queryMain, Query &queryMyMe, const tMediaObject mediaObject); //Roadmap 13012: 100%
    tResult StoreDirectoriesInternal(Query &queryMain, Query &queryMyMe, const tMediaObject mediaObject);
    tResult StoreMediaObjectEnd(const bool speedup = true);
    tResult GetNumberOfMediaObjectsInDB(tNumberOfMediaObjects &numberObjectsInDB) const;
    tResult GetNumberOfPlayableObjectsInDB(tNumberOfMediaObjects &numberOfAudioObjectsInDB, tNumberOfMediaObjects &numberOfVideoObjectsInDB, const tDeviceID deviceID) const; //Roadmap 130002
    tResult GetNumberOfRippedFile(tNumberOfMediaObjects &numberOfRippedFile, const tDeviceID deviceID, IN tTOC HashID, tCategoryType CategoryType) const;
    tResult GetNumberOfTotalRippedFile(tNumberOfMediaObjects &numberOfTotalRippedFile, const tDeviceID deviceID, tCategoryType CategoryType) const;

    tResult CreateTestDatabase(const tBoolean createDevices, const tBoolean createSongs);
    tResult ReleaseDBList(const tListID dbListID);
    tResult ReleaseAllDBLists();
    tResult IsVTIPODList(const tFilterTag1 filterTag1, const tFilterTag2 filterTag2, const tFilterTag3 filterTag3, const tFilterTag4 filterTag4,
            const tListType listType, const tDeviceInfo deviceInfo, const tBoolean lastPlayedList) const;
    tResult IsVTList(const tListID listID);
    tTriggerID OnInsertTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "NEW.rowid");
    tTriggerID OnDeleteTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");
    tTriggerID OnUpdateTrigger(const char* smName, const char* msgName, const char* tbl_name, const char* col_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");
    tResult OnInsertTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "NEW.rowid");
    tResult OnDeleteTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");
    tResult OnUpdateTrigger(tTriggerID &triggerID, const char* smName, const char* msgName, const char* tbl_name, const char* col_name, const char* clause = NULL, const char* cbArg1 = "COUNT()", const char* cbArg2 = "OLD.rowid");
    tResult UnregisterTrigger(const tTriggerID triggerID) const;
    tResult GetDevice(tDeviceID &deviceID, const tDeviceType deviceType, const tBoolean allDevices = false, const tBoolean virtualDevice = false, const tConnectionState connectionState = CS_CONNECTED);
    tResult GetDevice(tDeviceID &deviceID, const tMountPoint mountPoint, const tBoolean allDevices = false);
    tResult GetDevice(tDeviceID &deviceID, const tDeviceType deviceType, const tDeviceName deviceName, const tBoolean allDevices = false,
            const tConnectionState connectionState = CS_CONNECTED, const tConnectionType connectionType = DCT_UNKNOWN);
    tResult GetOldDevices(tNumberOfDevices &numberOfDevices, vector<tDeviceID> &deviceIDs);
    tResult GetDeviceFromAlbumArtString(tDeviceID &deviceID, const tAlbumArt albumArtString);
    tResult ScanDirForTest(const tDeviceID deviceID);
    tResult GenerateMountPoint(tDeviceInfo &deviceInfo) const;
    tResult StartListAccess(const tListID listID, const tIndex startIndex, const tIndex noOfRows);
    tResult EndListAccess(const tListID listID);
    tResult ReleaseAccess(const tListID listID);
    tResult UpdateMyMediaDatabase();

    /**
     * Function copies the current content of a device over to MyMedia DB
     * Called by Indexer in partial or DELTA indexing mode (see Roadmap 13006 IAP2)
     *
     * @param[in] deviceID ID of the device
     * @return != 0: error, = 0: OK
     */
    tResult UpdateMyMediaDatabase(const tDeviceID deviceID); //Roadmap 13006 IAP2: 100%

    tResult DeleteMyMedia();
    tResult WaitForUpdateMyMediaFinished(const tDeviceID deviceID);
    tResult WaitForUpdateMyMediaFinished();
    tResult WaitForSpeedupDBFinished();
    tResult SpeedupDB();
    tResult Sync();
    tResult RemoveDeviceFromMyMedia(const tDeviceID deviceID);
    //Roadmap 13002 (Calculate number of playable files and set it with SetNumberOfFiles(deviceID, numberOfFiles))
    //-> tph: maybe obsolete because already set with fingerprint
    tResult UpdateMain(const tDeviceID deviceID); //Roadmap 13002: 100%

    /**
     * Function resets the connection state of all connected devices
     * Roadmap 13024
     *
     * Loop over all connected devices
     * - Copy the old connection state to the FormerConnectionState column
     * - if (initReason == IR_UNDERVOLTAGE)
     *   - Set connection state to CS_UNDERVOLTAGE.
     *   else
     *   - Set connection state to CS_DISCONNECTED.
     * Start a timer to reset the FormerConnectionState column
     *
     * @param[in] initReason init reason (IR_BOOT or IR_UNDERVOLTAGE)
     * @return < 0: error, = 0: OK
    */
    tResult ResetDevices(const tInitReason initReason); //Roadmap 13024: 100%

    /**
     * Function resets the available state of all favorites
     *
     * Make all to non-available
    */
    tResult ResetAllFavorites();

    tResult ResetFormerConnection(void);

    /**
     * Function sets the connection state of all internal devcies to CS_ATTACHED.
     * This function must be called in Run function.
     *
     * @return < 0: error, = 0: OK
    */
    tResult SetInternalDevicesToAttached();

    Database *GetDatabase() {return mDB;};

    /**
     * Function moves a device from the device table to the virtual devices table
     * Roadmap 14004
     *
     * @param[in] deviceID ID of the device
     * @return < 0: error, = 0: OK
    */
    tResult MoveDeviceToFromVirtualTable(const tBoolean fromVirtual, const tBoolean toVirtual,
            const tDeviceID deviceID, tDeviceID newDeviceID = DEVICE_ID_NOT_SET); //Roadmap 14004

    /**
     * Function sets device ID explicitly
     * Roadmap 14004
     *
     * @param[in] oldDeviceID old ID of the device
     * @param[in] newDeviceID new ID of the device
     * @param[in] virtualDevice device is stored in temporary virtual device table or in normal device table of DB
     * @return < 0: error, = 0: OK
    */
    tResult SetDeviceID(const tDeviceID oldDeviceID, const tDeviceID newDeviceID, const tBoolean virtualDevice = false); //Roadmap 14004

    /**
     * Returns the answer to condition if the device is in the devices table and also in the virtual devices table.
     * Roadmap 14004
     *
     * @param[in] deviceName friendly name of device
     * @return true or false
    */
    tBoolean IsInBothDeviceTables(const tDeviceName deviceName); //Roadmap 14004

    /**
     * Function removes all entries in the virtual devices table
     * Roadmap 14004
     *
     * @return < 0: error, = 0: OK
    */
    tResult ResetVirtualDevices(void) const; //Roadmap 14004

    /**
     * Function removes all entries in the live tags table
     *
     * @return < 0: error, = 0: OK
    */
    tResult ResetLiveTags(void) const;

    /**
     * Function removes all entries in the live tags table related to given device ID
     *
     * @param[in] deviceID ID of the device
     * @return < 0: error, = 0: OK
    */
    //tResult RemoveLiveTags(const tDeviceID deviceID) const;

    /**
     * Function fills up missing tag IDs for the metadata
     * Update an existing live tags element (only CTY_SONG, CTY_VIDEO and CTY_PLAYLIST) //Roadmap 13012
     *
     * @param[in, out] mediaObject media object to update
     * @return < 0: error, = 0: OK
    */
    tResult FillupMetaDataIDs(tMediaObject &mediaObject); //Roadmap 13012: 100%

    /**
     * Function stores an element to the live tags table
     *
     * @param[in] tag filename or metadata string to tag
     * @param[in] catType type ID for a category (CTY_SONG, CTY_ARTIST, ...)
     * @param[in] deviceID ID of the device
     * @param[in] notPlayable not playable flag (playable or not or DRM protected) //Roadmap 13012
     * @return < 0: error, = 0: OK
    */
    tResult StoreLiveTagsElement(const tTag tag, const tCategoryType catType, const tDeviceID deviceID, const tNotPlayable notPlayable) const; //Roadmap 13012: 100%

    /**
     * Function updates the live tags table
     * Roadmap 13012
     *     *
     * @param[in] tagID tag ID (object ID) in live tag table
     * @param[in] notPlayable not playable flag (playable or not or DRM protected)
     * @return < 0: error, = 0: OK
    */
    tResult UpdateLiveTagsElement(const tTagID tagID, const tNotPlayable notPlayable) const; //Roadmap 13012: 100%

    /**
     * Function delivers an element from the live tags table
     *
     * @param[in] tagID tag ID (object ID) in live tag table
     * @param[out] tag filename or metadata string to tag
     * @param[out] catType type ID for a category (CTY_SONG, CTY_ARTIST, ...)
     * @param[out] deviceID ID of the device
     * @param[out] notPlayable not playable flag (playable or not or DRM protected) //Roadmap 13012
     * @return < 0: error, = 0: OK
    */
    tResult GetLiveTagsElement(tTag &tag, tCategoryType &catType, tDeviceID &deviceID, tNotPlayable &notPlayable, const tTagID tagID) const; //Roadmap 13012: 100%

    /**
     * Function delivers an element from the live tags table
     *
     * @param[in] tag filename or metadata string to tag
     * @param[in] catType type ID for a category (CTY_SONG, CTY_ARTIST, ...)
     * @param[out] tagID tag ID (object ID) in live tag table
     * @return < 0: error, = 0: OK
    */
    tResult GetLiveTagsElement(tTagID &tagID, const tCategoryType catType, const tTag tag) const;

    tResult GetURLOfMediaObject(tTag &url, const tTagID tagID, const tCategoryType catType) const;
    tResult RemoveOldestDevice(tDeviceID &freeDeviceID, bool indexableDevice = false);
    tResult FreeSpace();

    /**
     * Favorites information
     */
    tResult ActivateFavorite(const tFavoriteID favoriteID) const;
    tResult StoreFavorite(tFavoriteID &retFavID, const tObjectID objectID,
            const tCategoryType objectType, const tLabelText labelText, const tDescriptorText descriptorText) const;
    tResult DeleteFavorite(const tFavoriteID favoriteID) const;
    tResult ResetFavorites(void) const;
    tResult DeActivateAllFavorites(void) const;
    tResult UpdateFavoriteStatus(const tFavoriteID favoriteID, const tBool isAvailable, const tBool isActive) const;
    tResult GetFavoriteInfo(tFavoriteInfo &favoriteInfo, const tFavoriteID favID) const;
    tResult GetAllFavoriteInfo(vector<tFavoriteInfo> &favInfos) const;
    tResult GetObjectIdOfFavorite(tObjectID &objectId, tCategoryType &objectType, const tFavoriteID favoriteId) const;


    tResult DeleteVTFileCache(const tDeviceID deviceID);
    tResult DeleteVTBluetoothCache(const tMountPoint mountPoint);

    char *SelectFormatString(char *buffer, const tFilterTag1 tag1, const tFilterTag2 tag2, const tFilterTag3 tag3, const tFilterTag4 tag4, const tSearchString searchString /*added param Roadmap 16014 'full Text Search'*/) const;
    static tResult MaskOutFilterTagsByListType(tFilterTag1 &filterTag1, tFilterTag2 &filterTag2,
            tFilterTag3 &filterTag3,tFilterTag4 &filterTag4, const tListType listType);
    tResult GetFileFormat(tFileFormat &fileFormat, tFileType &fileType, const tFilename fileName) const;

    tCategoryType GetCTYByLTY(const tListType listType) const;
    tCategoryType GetCTYByMTY(const tMediaType mediaType) const;
    tMediaType GetMTYByCTY(const tCategoryType catType) const;
    tFileType GetFTByMTY(const tMediaType mediaType) const;
    tListType GetLTYByCTY(const tCategoryType catType) const;
    tResult GetVFTByFTS(tVTFileType &fileType1, tVTFileType &fileType2, tVTFileType &fileType3, tVTFileType &fileType4,
            const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const;
    tResult GetCTYByFTS(tCategoryType &catType1, tCategoryType &catType2, tCategoryType &catType3, tCategoryType &catType4,
            const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const;
    tResult GetMediaContentByFTS(tMediaContent &mediaContent,
            const tFileTypeSelection fileTypeSelection, const tBoolean onlyPlayableObjects) const;
    tResult GetFavoriteNameIPOD(tTitle &name, tObjectID &objectID, tCategoryType &objectType);
    tResult GetMetatdataIDs(tMediaObject &mediaObject);
    tResult GetObjectUUID(tDeviceID &deviceID, tUUID &UUID, const tObjectID objectID);
    tResult GetObjectIDByUUID(tObjectID &objectID, const tDeviceID deviceID, const tUUID UUID);
    tResult GetIPODMediaObjectByUUID(tMediaObject& mediaObject, const tDeviceID deviceID, const tUUID UUID);
    tResult GetNumberofImages(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID);
    tResult GetNumberofImagesByPath(tNumberOfMediaObjects &numberOfImages, const tDeviceID deviceID, const tPath path);

    /**
     * Function adds a artificial device for a mount point (acts like an usb stick)
     *
     * @param[in] mountpoint mountpoint (complete) to hold the mediaobjets for the new device
     * @return != 0: error, = 0: OK
     */
    tResult AddDevice(const tMountPoint mountPoint);

    /**
     * Function updates the Title of an InternalPlaylist located at path <playlistPath>
     * Roadmap 13022_PlaylistManagement
     *
     * @param[in] playlistPath path to the Internal playlist whose name has to be updated
     * @param[in] playlistName new Title name for the Internal playlist
     * @return != 0: error, = 0: OK
     */
    tResult UpdateInternalPlaylist(const tPath playlistPath, const tPlaylistName playlistName, tDeviceID deviceID = DEVICE_ID_NOT_SET); //Roadmap 13022

    /**
     * Check whether a particular playlist already exists or not
     *
     * @param[out] PlaylistNameCount count of playlist with same name
     * @param[in] deviceID device ID to set
     * @param[in] playlistName of playlist to check if exists
     * @return < 0: error, = 0: OK
     */
    tResult CheckifPlaylistExists(int &PlaylistNameCount,const tDeviceID deviceID, const tPlaylistName playlistName);
    /**
     * Sets the former connection state of a device
     *
     * @param[in] deviceID device ID to set
     * @param[in] formerConnectionState old connection state
     * @return < 0: error, = 0: OK
     */
    tResult setFormerConnectionState(const tDeviceID deviceID, const tConnectionState formerConnectionState);

   /**
     * Sets the connection state of a  device
     * Set the disconnect reason to DR_REMOVED in case of connectionState == CS_DISCONNECTED //Roadmap 15003
     *
     * @param[in] deviceID device ID to set
     * @param[in] connectionState new connection state
     * @return < 0: error, = 0: OK
     */
    tResult SetConnectionState(const tDeviceID deviceID, const tConnectionState connectionState,tDisconnectReason disconnectReason = DR_REMOVED) const;//Roadmap 15003: 100%

    /**
     * Sets the Device state of a  device
     * Set the disconnect reason to DR_REMOVED in case of deviceState == DS_NONE //NCG3D-52246
     *
     * @param[in] deviceID device ID to set
     * @param[in] deviceState new device state
     * @return < 0: error, = 0: OK
     */
    tResult SetDeviceState(const tDeviceID deviceID, const tDeviceState newDeviceState,tDisconnectReason disconnectReason = DR_REMOVED) const; //NCG3D-52246

    tResult MarkDeviceCommunicationError(const tDeviceID deviceID,const tConnectionCount connectionCount,const tDeviceType deviceType);

    /**
     * Function get the number of tracks(audio track and podcast and audiobooks)
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult GetNumberofTracks(tNumberOfMediaObjects &numberObjectsInDB, const tDeviceID deviceID);
    /**
     * Function get the number of videos
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult GetNumberOfVideos(tNumberOfMediaObjects &numberObjectsInDB,const tDeviceID deviceID);
    /**
     * Function get the number of playlist
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult GetNumberOfPlaylist(tNumberOfMediaObjects &numberObjectsInDB,const tDeviceID deviceID);
    /**
     * Function to create a temporary table to store the uid of audio and video tracks
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult CreateTrackTempTable();
    /**
     * Function to retrive the uid of audio tracks from media object table and
     * to store in the temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult FillTrackTempTable(const tDeviceID deviceID);
    /**
     * Function to retrive the uid of audio tracks from media object table and
     * to store in the temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult FillTrackTempTableWithVideoUID(const tDeviceID deviceID);
    /**
     * Function to delete the given uid from temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult RemoveTrackEntryFromTemp(const tPlaylistName playListName);
    /**
     * Function to get the number of entry available in temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult GetNumberofTrackTempEntry(tNumberOfMediaObjects &numberObjectsInDB);
    /**
     * Function to drop the temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult DropTrackTempTable();
    /**
     * Function to create a temporary table to store the playlist title
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult CreatePlaylistTempTable();
    /**
     * Function to retrive the playlist title from media object table and
     * to store in the temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult FillPlaylistTempTable(const tDeviceID deviceID);
    /**
     * Function to delete the given playlist name from temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult RemovePlaylistEntryFromTemp(const tPlaylistName playListName);
    /**
     * Function to get the number of playlist name entry available in temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult GetNumberofPlaylistTempEntry(tNumberOfMediaObjects &numberObjectsInDB);
    /**
     * Function to drop the temporary table
     * Roadmap 13033_IAP1_UID_BasedIndexing
     */
    tResult DropPlaylistTempTable();

    tResult UpdateShuffleSupport(const tDeviceID deviceID,const tShuffleSupport isShuffleSupported);
    tResult UpdateRepeatSupport(const tDeviceID deviceID,const tRepeatSupport isRepeatSupported);
    tResult UpdateDeviceVersion(const tDeviceID deviceID,const tDeviceVersion deviceVersion);
    tResult UpdateNowPlayingListAvailable(const tDeviceID, const tNowPlayingListAvailable isNowPlayingListAvailable);
    tResult UpdateDeviceUUID(const tDeviceID deviceID, const tUUID deviceUUID);
    tResult UpdateAppleDeviceMACAddress(const tDeviceID deviceID, const tMACAddress appleDeviceMacAddress);
    tResult UpdateAppleDeviceMACAddressAndUSBSerialNumber(IN const tDeviceID deviceID, IN const tMACAddress appleDeviceMACAddress, IN const tUSBSerial appleDeviceUSBSerialNumber);

    tResult MultiPartitionDeviceChanged(vector<tDeviceInfo> deviceInfos);
    tResult FindLargestValidPartition(vector<tDeviceInfo>& deviceInfos);
    tResult RemoveMultiPartitionDevice(tDeviceSerialNumber &serialNumber);
    tResult CheckValidPartition(tDeviceInfo deviceinfo);
    tResult UpdateDBListPath(const tDeviceID deviceID ,const tListID listID, const tPath lastPlayedPath);
    tResult CheckDatabaseIntegrity(const bool write_recreate_file);
    tResult UpdateDiscType(const tDeviceID deviceID,const tDiscType discType);

    /**
    * Check for mediacontent in data base and update if not done.
    */
    tResult UpdateFolderMediaContent(Query &queryMain, const tMediaObject mediaObject);
    tResult GetFolderMediaContent(Query &queryMain, tMediaContent &mediaContent,const tURL &url,IN const tDeviceID deviceID);
    /**
     * Function to retrieve the video properties from db
     * Roadmap 15027_AdjustVideoBrightness
     */
    tResult GetVideoBrightness(OUT tVideoBrightness &videoBrightness);
    tResult GetVideoHue(OUT tVideoHue &videoHue);
    tResult GetVideoSaturation(OUT tVideoSaturation &videoSaturation);
    tResult GetVideoContrast(OUT tVideoContrast &videoContrast);
    tResult GetVideoBrightnessOffset(OUT tVideoBrightnessOffset &videoBrightnessOffset);
    tResult GetVideoHueOffset(OUT tVideoHueOffset &videoHueOffset);
    tResult GetVideoSaturationOffset(OUT tVideoSaturationOffset &videoSaturationOffset);
    tResult GetVideoProperties(OUT me::vprops_t &mEVideoProperties);
    tResult UpdateVideoProperty (IN const tVideoProperty videoProperty,IN const tPropertyValue propertyValue);
      /* Added for Internal DVD  */
    tResult StoreDVDResumeInfo(const tDVDInfoBlob resumeBlob);
    tResult GetDVDResumeInfo(tDVDInfoBlob &resumeInfo);
    tResult StoreDVDSetupInfo(const tDVDInfoBlob setupInfo);
    tResult GetDVDSetupInfo(tDVDInfoBlob &setupInfo);
    tResult GetMetadataOfRippedFile(tRippedTrackInfo &trackInfo, const tDeviceID deviceID, IN tTOC HashID, IN tU32 trackIndex) const;
    tResult UpdateRippedFileCategory(tDeviceID deviceID, tRippedFilePath filepath, tCategoryType CategoryType);
    tResult UpdateAlbumArtStringForTocHash(tDeviceID deviceID, tAlbumArt albumArtString, const tUUID UUID);
    tResult GetAlbumArtStringForTocHash(const tUUID UUID, std::string &AlbumArtString);
    tResult SetMusicBoxAutoRipping(const tBoolean autoRipping);
    tResult GetMusicBoxAutoRipping(tBoolean &autoRipping);
    tResult SetMusicBoxEncodingQuality(const tU32 encodingQuality);
    tResult GetMusicBoxEncodingQuality(tU32 &encodingQuality);

    //>--Roadmap 16003 : 'CD Ripping With Gracenote'
    /**
     *
     * Roadmap 16003 : CD Ripping With Gracenote
     * Method to delete media objects
     *
     * @param[in] deviceID  id of the device
     * @param[in] mediObjectUrl the url to be deleted from mediaobjects table
     * @return != 0: error, = 0: OK
     */

    tResult DeleteMediaObject(IN const tDeviceID deviceID,IN const std::string mediObjectUrl);

    /**
     *
     * Roadmap 16003 : CD Ripping With Gracenote
     * Method to get the url of  media objects
     *
     * @param[out] url  of the mediaobject
     * @param[in] objectID  id of the device
     * @param[in] deviceID  id of the device
     *
     * @return != 0: error, = 0: OK
     */
    tResult GetURLOfMediaObject(OUT tURL &url,IN const tObjectID objectID, IN const tDeviceID deviceID) const;

    /**
     *
     * Roadmap 16003 : CD Ripping With Gracenote
     * Method to update the  metadata of  media object
     *
     * @param [in] metadata1 of the mediaobject
     * @param [in] metadata2 of the mediaobject
     * @param [in] metadata3 of the mediaobject
     * @param [in] metadata4 of the mediaobject
     * @param [in] year of the mediaobject
     * @param[in] url  of the mediaobject
     * @param[in] deviceID  id of the device
     * @param[in] mediaObjectUrl  url of the mediaobjects
     * @return != 0: error, = 0: OK
     */
    tResult UpdateMediaObject(const IN tEditMetaDataByUrl &editMetaDataByUrl);
    //<--Roadmap 16003 : 'CD Ripping With Gracenote'

    /*  Function to update the FormerConnectionState for the latest device of given deviceType
     *
     *  @param[in] deviceType
     *  @param[in] connectionState
     * @return < 0: error, = 0: OK
     */
    tResult UpdateFormerConnectionState(IN tDeviceType deviceType,IN tConnectionState formerconnectionState);
    tResult DisconnectDeviceOnCommunicationError(const tDeviceID deviceID);
    int DatabaseStatistics(void);
    tResult UpdateMusicBoxDevice(tDeviceID deviceID, tNumberOfFiles numberOfFiles, tIndexingState indexingState, tUInt RippingPercentage, tUInt TotalMemory, tUInt FreeMemory);

    /*  Get the tracknumbers which belong to a particular CD using UUID
     *
     *  @param[in] deviceId
     *  @param[in] UUID
     *  @param[OUT] VECTOR <TRACKNUMBER>
     * @return < 0: error, = 0: OK
     */
    tResult GetTrackNumbersByUUID(vector<tU16>& trackNumbers, const tDeviceID deviceID, const tUUID UUID);

    //>--Roadmap CMG3G-10221 : 'Scene Recorder '
    /*  Update file mode while changing file permission <<Road map topic CMG3G-10221>>
     *
     *  @param[in] deviceId
     *  @param[in] file mode
     *  @param[in] media object
     *  @param[in] media object url
     * @return < 0: error, = 0: OK
     */
    tResult UpdateMediaObject(IN const tDeviceID deviceID, IN const tFilePermission filePermission,const char* path);

    /*  Update file mode while changing file permission <<Road map topic CMG3G-10221>>
     *
     *  @param[in] deviceId
     *  @param[in] folder path
     *  @param[in] Category type
     *  @param[OUT] folder size and number for protected and total
     * @return < 0: error, = 0: OK
     */
    tResult GetFolderSizeInfo(IN const tDeviceID deviceID, IN const tPath Path,
            IN const tCategoryType type, OUT tFolderInfo& info);
    /*  Get Folder information for Delete operation
     *     <<Road map topic CMG3G-10221>>
     *
     *  @param[in] deviceId
     *  @param[in] folder path
     *  @param[OUT] Onject Id , catagory type and URL
     * @return < 0: error, = 0: OK
     */
    tResult GetDirInfoFromPath(IN const tDeviceID deviceID, IN const string dirPath, OUT vector<tDirInfo> &dirInfo);

     /*  Get File permission for media object
     *     <<Road map topic CMG3G-10221>>
     *  @param[in] deviceId
     *  @param[in] object ID`
     *  @param[OUT] File mode , User Id and Group ID
     * @return < 0: error, = 0: OK
     */
    tResult GetFileMode(IN const tDeviceID deviceID, IN const tObjectID obj, OUT tFileMode &FileMode, OUT tUserID &uId, OUT tGroupID &gId);
    /*  Get File size , date of creation and total playtime
         *     <<Road map topic CMG3G-10221>>
         *  @param[in] deviceId
         *  @param[in] object ID`
         *  @param[OUT] TotalPlaytime ,  Size and DateTime
         * @return < 0: error, = 0: OK
   */
   tResult GetFileLengthDateSize(IN const tDeviceID deviceID, IN const tObjectID obj, OUT tPlaytime &TotalPlaytime, OUT tDateTime &DateTime, OUT tFileSize &size);
    //<--Roadmap CMG3G-10221 : 'Scene Recorder '

    /*  Store Personalized favorite as per User
    *     <<Road map topic CMG3G-11800>>
    *  @param[in] object ID    *
    *  @param[in] objectType
    *  @param[in] labelText
    *  @param[in] descriptorText
    *  @param[in] userID
    *  @param[OUT] retFavID
    * @return < 0: error, = 0: OK
    */
    tResult StorePersonalizedFavorite(OUT tFavoriteID &retFavID,IN const tObjectID objectID,IN const tCategoryType objectType,IN const tLabelText labelText,IN const tDescriptorText descriptorText,IN const tUserID userID );
    //>--Roadmap CMG3G-11800  17001: 'Personalization'

    /*  Delete Personalized favorite as per User
    *     <<Road map topic CMG3G-11800>>
    *  @param[in] userID
    *  @param[in] objectID
    * @return < 0: error, = 0: OK
    */
    tResult DeletePersonalizedFavorite(IN const tObjectID objectID, IN const tUserID userID);
    //>--Roadmap CMG3G-11800  17001: 'Personalization'

    /*  Set the active user in the User table
    *     <<Road map topic CMG3G-11800>>
    *  @param[in] userID
    * @return < 0: error, = 0: OK
    */
    tResult SetCurrentUser(tUserID userID);

    /*  Get the active user from the users table
    *     <<Road map topic CMG3G-11800>>
    *  @param[out] userID
    * @return < 0: error, = 0: OK
    */
    tResult GetCurrentUser(tUserID &userID);

    /*  Delete Personalized favorite as per User
    *     <<Road map topic CMG3G-11800>>
    *  @param[in] userID
    *  @param[in] favID
    * @return < 0: error, = 0: OK
    */
    tIsFavorite IsUserFavoriteObject(const tObjectID objectID);

    /*  Delete all favorites for a User
     *     <<Road map topic CMG3G-11800>>
     *  @param[in] userID
     * @return < 0: error, = 0: OK
     */
     tResult DeleteAllUserFavorites(IN const tUserID userID);
     //>--Roadmap CMG3G-11800  17001: 'Personalization'

     /*  Delete all users and their favorites
      *     <<Road map topic CMG3G-11800>>
      * @return < 0: error, = 0: OK
      */
      tResult ResetPersonalization();

      /* Fills Yomimetadata and mediaobject tag in the YomiMetadata table
       * @param[in] MediaobjectTag, YomiTitle, YomiArtist, YomiAlbum
       * @return < 0: error, = 0: OK
       */
      tResult AddYomiMetadata(tObjectID MediaObjectTagYomiData, tYomiMetadata &yomiMetadata);
      /* Provides Yomimetadata  in the YomiMetadata table
       * @param[in] MediaobjectTag, YomiTitle, YomiArtist, YomiAlbum
       * @return < 0: error, = 0: OK
       */
      tResult GetYomiMetadata(IN tObjectID MediaObjectTagYomiData, OUT tYomiMetadata &yomiMetadata);

      tResult RemoveYomiMetadata(tObjectID MediaObjectTagYomiData);
      /* Updates edited Yomimetadata  in the YomiMetadata table
       * @param[in] yomiMetadata
       * @return < 0: error, = 0: OK
       */
      tResult UpdateYomiMetadata(tObjectID MediaObjectTagYomiData, tYomiMetadata &yomiMetadata);

      /* Fills TOC and uuid to the Music box ToC table for further mapping
       * @return < 0: error, = 0: OK
       */
      tResult AddMusicBoxToC(char* ToCString, tUUID uuid);
      /* Provides ToCString of the particular UUID
       * @return < 0: error, = 0: OK
       */
      tResult GetMusicBoxToC(tUUID uuid,std::string &ToCString);
      /* Deletes ToCString of the particular UUID
       * @return < 0: error, = 0: OK
       */
      tResult RemoveMusicBoxToC(tMediaObject &mediaObject);
      /* Provides uuid of the particular album from media object table
       * @return < 0: error, = 0: OK
       */
      tResult GetUUIDfromAlbumName(const tMetadata albumName,tUUID uuid);
      /* Updates gracenote metadata in mediaobjects table if Gracenote data is available
       * @return < 0: error, = 0: OK
       */
      tResult UpdateGNmetadatainMediaObject(const tRippedTrackInfo &rippedTrackData, IN tDeviceID deviceID, IN tUUID UUID, IN tTrackNumber trackNumber);

      // CRQ 300254 Support for Folder Up/Down command
      // Fetch and store IDs of all the folders in FolderHierarchy table for a particular deviceID, in the order it was indexed
      tResult GetFolderHierarchyForDeviceID(OUT vector<tObjectID>& folderHierarchy, IN const tDeviceID deviceID);
      // Find childID of received folderPath wrt the deviceID
      tResult GetChildIDForFolderPath(OUT tObjectID& childID, IN tDeviceID deviceID, IN tPath folderPath);
      // Find folderPath of received childID wrt the deviceID
      tResult GetFolderPathForChildID(OUT tPath& folderPath, IN tObjectID childID, IN tDeviceID deviceID);
      // Find yomiAlbum name for the given album ID
      tResult GetYomiAlbumNameForAlbumId(IN tObjectID albumID, tMetadata &yomiAlbum);
      tResult GetQuickFingerprint(tFingerprint &quickfingerprint, const tDeviceID deviceID);
      tResult SetQuickFingerprint(const tDeviceID deviceID, const tFingerprint fingerprint);
      tResult AddRouteGuidanceDisplayComponentInfo(const tDiPORGDisplayComponentInfo RGDisplayComponentInfo);
      tResult DeleteRouteGuidanceDisplayComponentInfo(const tU16 displayIdentifier);
      tResult UpdateRouteGuidanceDisplayComponentInfo(const tDiPORGDisplayComponentInfo RGDisplayComponentInfo);
      tResult GetRouteGuidanceDisplayComponentInfos(tDiPORGDisplayComponentList & RGDisplayComponentInfos);

    class DBList
    {
    public:
        DBList()
        {
            mListID = LIST_ID_NONE;
            mDeviceID = DEVICE_ID_NOT_SET;
            mQuery = NULL;
            mListType = LTY_END_OF_EXTERNAL_LIST_TYPES; /* lastmode */
            mIsVTIPODList = false;
            mPlayContinuation = PC_NO_REPEAT; /* lastmode */
            mFilterTag1.tag = 0; /* lastmode */
            mFilterTag2.tag = 0; /* lastmode */
            mFilterTag3.tag = 0; /* lastmode */
            mFilterTag4.tag  = 0; /* lastmode */
            mSearchString[0] ='\0'; //Roadmap 160014 //full text search
            mPath[0] = 0; /* lastmode */
            mStreaming = false; /* lastmode */
            mFileTypeSelection = FTS_AUDIO_VIDEO_PLAYLIST;
            InitMediaObject(mCurrentMediaObject);
            mCurrentRow = ROW_NUMBER_NONE;
            mRefCounter = 0;
            mUserID = 0;
        };

        tListID mListID;
        tDeviceID mDeviceID; /* lastmode */
        Query *mQuery;
        tListType mListType; /* lastmode */
        int mIsVTIPODList;  //switch VTIPOD handling
        tPlayContinuation mPlayContinuation; /* lastmode */
        tFilterTag1 mFilterTag1; /* lastmode */
        tFilterTag2 mFilterTag2; /* lastmode */
        tFilterTag3 mFilterTag3; /* lastmode */
        tFilterTag4 mFilterTag4; /* lastmode */
        tSearchString mSearchString; //Roadmap 160014 //full text search
        tPath mPath; /* lastmode */
        tStreaming mStreaming; /* lastmode */
        tFileTypeSelection mFileTypeSelection;
        tMediaObject mCurrentMediaObject;
        tRowNumber mCurrentRow;
        int mRefCounter;
        tUserID mUserID;

    };

private:

    map<tDeviceID,string> mPortNumberInfo;
    map<string,string> mValidPartitionInfo; //Zabeer
    map<tDeviceID,string> mDeviceSysPathInfo; //CMG3G-12660
    class UnlockOnExit
    {
    public:
        UnlockOnExit(DBList **currentDBList)
        {
            mCurrentDBList = currentDBList;
        }
        ~UnlockOnExit()
        {
            if (*mCurrentDBList) {
                if ((*mCurrentDBList)->mQuery) {
                    (*mCurrentDBList)->mQuery->UnLock();
                }
            }
        }
    private:
        DBList **mCurrentDBList;
    };

    class AutoDeRefDBListPtr
    {
    public:
        AutoDeRefDBListPtr(DBList *listPtr)
        {
            mListPtr = listPtr;
        }

        ~AutoDeRefDBListPtr()
        {
            if (mListPtr) {
                if (mListPtr->mRefCounter > 0) {
                    mListPtr->mRefCounter--;
                }
            }
        }

    private:
        DBList *mListPtr;
    };

    // the data base handling object
    Database *mDB;

    /* data base manager list management */
    DBList *CreateDBList();
    DBList *LookupDBList(const tListID dbListID);
    tResult ResetObjectCacheOfDBList(const tDeviceID deviceID);
    tResult FlushMediaObjectCache();
    tResult FlushMediaObjectCacheInternal(vector<tMediaObject> *objectCache);
    tResult WaitForFlushMediaObjectCacheEnd();

    unsigned int GetCountFlushBorder();
    void CalcCountFlushBorder(unsigned int cacheSize, unsigned long durationUS);

    vector<DBList *> mDBLists;
    Lock mDBListsLock;
    Lock mFlushBorderMutex;
    unsigned int mObjectCountFlushBorder;
    unsigned int mDBStoreMediaObjectMaxTime_ms;
    vector<tMediaObject> mMediaObjectCache;
    tLanguageType mActiveLanguage;
    tMediaContext mActiveMediaContext;
    Lock mFlushLock;
    class MyMediaUpdateManager mMMUpdt;

    void Do(int functionID, void *ptr);
    tResult UpdateObjectID(tMediaObject &mediaObject, const tListType listType) const;
    tResult StopSpeedupDB();
    static bool TimerCallBack(timer_t timerID, void* instance, const void *userData);

    static bool MemoryUsageTimerCallBack(timer_t timerID , void* instance ,const void *userData);
    void OnMemoryUsageTimeout();

    tResult GetVTTagForIPODPlaylist(tTagID &playlistTag) const;
    tResult GetVTTagForIPOD(tTagID &tagID) const;

    typedef class {
    public:
        Lock tableLock;
        vector<Query *> queries;
    } tRegistry;

    vector<tRegistry *> mRegistry;

    Timer mTimerResetDevices;
    timer_t mTimerIDResetDevices;
    Timer m_MemoryUsageTimer;
    timer_t m_MemoryUsageTimerID;
};

#endif /*DBMANAGER_H_*/
