/**************************************************************************//**
* \file      Utf8String.cpp
* \ingroup   MediaPlayer
*
*            Implementation of class Utf8String.
*            See .h file for description.
*
* \remark    Copyright: 2012, Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark    Scope:     MediaPlayer
* \remark    Authors:
*            - CM-AI/PJ-CD6 Xu Benny
******************************************************************************/

#include "Utf8String.h"
#include <memory.h>
using namespace std;

#include "TypeDefinitions.h"

/**************************************************************************************************/
/*
 * UTF8String iterator implementation
 */
UTF8String::Iterator::Iterator ()
{
	m_ulPos = 0xffffffff;
}
UTF8String::Iterator::Iterator (unsigned long ulPos)
{
	m_ulPos = ulPos;
}
UTF8String::Iterator::~Iterator ()
{
}
UTF8Char UTF8String::Iterator::getChar() const
{
	if (m_ulPos == 0xffffffff)
	{
		return ' ';
	}
	unsigned char* puBuff = (unsigned char*) m_ulPos;
			
	if ( puBuff[0] < 0xC0)
	{
		return  puBuff[0];
	}
	if ( puBuff[0] < 0xE0)
	{
		return (( puBuff[0] & 0x1F) << 6) |
		   ( puBuff[1] & 0x3F);
	}
	if ( puBuff[0] < 0xF0)
	{
		return (( puBuff[0] & 0x0F) << 12) |
		   (( puBuff[1] & 0x3F) << 6) |
		   ( puBuff[2] & 0x3F);
	}
	if ( puBuff[0] < 0xF8)
	{
		return (( puBuff[0] & 0x07) << 18) |
		   (( puBuff[1] & 0x3F) << 12) |
		   (( puBuff[2] & 0x3F) << 6) |
		   ( puBuff[3] & 0x3F);
	}
	MP_NORMAL_ASSERT(0);  /* unsupported unicode range */
	return '?';
}
		
int UTF8String::Iterator::getCharSize() const
{
	if (m_ulPos == 0xffffffff)
		return 0;
		
	unsigned char* puBuff = (unsigned char*) m_ulPos;
	if (*puBuff < 0xC0)
	{
	    MP_NORMAL_ASSERT(*puBuff < 0x80);  /* not the first byte of a UTF8 character */
		return 1;
	}
	if (*puBuff < 0xE0)		return 2;
	if (*puBuff < 0xF0)		return 3;
	if (*puBuff < 0xF8)		return 4;
	MP_NORMAL_ASSERT(0);  /* unsupported unicode range */
	return 1;	
}

unsigned long  UTF8String::Iterator::memory_addr() const
{
	return m_ulPos;
}

UTF8String::Iterator& UTF8String::Iterator::operator ++ ()
{
	if (m_ulPos == 0xffffffff)
		return (*this);
		
	unsigned char* puBuff = (unsigned char*) m_ulPos;
			
	MP_NORMAL_ASSERT(( puBuff[0] & 0xC0) != 0x80);  /* not a starting byte */
	
	int nSize = getCharSize();
	if (nSize > 0)
	{
		if (getChar() != '\0')
			m_ulPos += nSize;
	}
	else
		m_ulPos = 0xffffffff;
	return (*this);
}

UTF8String::Iterator  UTF8String::Iterator::operator ++ (int nOff)
{
    (void)nOff;
    Iterator& it = *this;
	++it;
	return it;
}

UTF8String::Iterator UTF8String::Iterator::operator -- (int nOff)
{
    (void)nOff;
    Iterator& it = *this;
	--it;
	return it;
}

UTF8String::Iterator& UTF8String::Iterator::operator -- ()
{
	if (m_ulPos == 0xffffffff)
		return (*this);

	unsigned char* puBuff = (unsigned char*) m_ulPos;
	int nOff = 0;
	do
	{
		/* skip all in-between utf8 bytes starting with 10xxxxxx */
		puBuff--;
		nOff += 1;
	}
	while ( ( puBuff[0] & 0xC0) == 0x80);
	MP_NORMAL_ASSERT( ( puBuff[0] & 0xC0) != 0x80 );   /* UTF8 string is corrupt */
		
	MP_NORMAL_ASSERT (nOff <= 4);
			
	m_ulPos -= nOff;
	return (*this);
}
		
UTF8String::Iterator& UTF8String::Iterator::operator += (int nOff)
{
	int nPos = 0;
	Iterator& it = *this;
	for (nPos = 0; nPos < nOff; nPos ++)
	{
		it++;
		if  (it.m_ulPos == 0xffffffff)
		{
			m_ulPos = 0xffffffff;
			break;
		}
	}
	return (*this);
}
bool UTF8String::Iterator::fine ()  const
{
	if (m_ulPos == 0xffffffff)
		return false;
	return true;
}

UTF8String::Iterator UTF8String::Iterator::operator + (int nOff) const
{
	if ((nOff < 0) || (m_ulPos == 0xffffffff) )
		return (*this);
		
	Iterator it = *this;
	
	int count = 0;
	for (count = 0; count < nOff; count ++)
	{
		it ++;
	}
	return it;
}

UTF8String::Iterator UTF8String::Iterator::operator - (int nOff) const
{
	if ((nOff < 0) || (m_ulPos == 0xffffffff) )
		return (*this);
	Iterator it = *this;
	
	int count = 0;
	for (count = 0; count < nOff; count ++)
	{
		it --;
	}
	return it;
}
		
bool UTF8String::Iterator::operator == ( const Iterator & ref) const
{
	return (m_ulPos == ref.m_ulPos);
}
bool UTF8String::Iterator::operator != ( const Iterator & ref) const
{
	return (m_ulPos != ref.m_ulPos);
}
bool UTF8String::Iterator::operator < ( const Iterator & ref) const
{
	if (m_ulPos == 0xffffffff)
		return false;				
	return (m_ulPos < ref.m_ulPos);
}

bool UTF8String::Iterator::operator > ( const Iterator & ref) const
{
	if (ref.m_ulPos == 0xffffffff)
		return false;
	return (m_ulPos > ref.m_ulPos);
}		
bool UTF8String::Iterator::operator <= ( const Iterator & ref) const
{
	if (m_ulPos == 0xffffffff)
		return false;
	return (m_ulPos <= ref.m_ulPos);
}

bool UTF8String::Iterator::operator >= ( const Iterator & ref) const
{
	if (ref.m_ulPos == 0xffffffff)
		return false;
	return (m_ulPos >= ref.m_ulPos);
}

UTF8String::Iterator& UTF8String::Iterator::operator= (const Iterator & ref)
{
    if( &ref == this || ref == *this) {}; // this is only needed for lint. There is no need to free any old memory in case of a self asignement.
	m_ulPos = ref.m_ulPos;
	return (*this);
} 

/**************************************************************************************************/
/******************************************************
 * Constructors
 ******************************************************/
UTF8String::UTF8String()
{
	m_nLen = 0;
	m_pBuffer = NULL;
	m_nBufSize = 0;	  
}

UTF8String::UTF8String(const UTF8String & u8str)
{
	m_nLen = 0;
	m_pBuffer = NULL;
	m_nBufSize = 0;	  
	
	init (u8str.m_pBuffer, u8str.m_nLen);
}

UTF8String::UTF8String(const char* pStr)
{
	m_nLen = 0;
	m_pBuffer = NULL;
	m_nBufSize = 0;	  
	
	init (pStr, strlen (pStr) );
}

/******************************************************
 * Destructors
 ******************************************************/
UTF8String::~UTF8String()
{
	free (m_pBuffer);
}
/******************************************************
 * Private member functions
 ******************************************************/
void UTF8String::init( const char* pBuffer, int nBufferSize)
{
	if (reserve (nBufferSize + 1) == false )
		return;
	
	memcpy (m_pBuffer, pBuffer, nBufferSize );
		
   	m_pBuffer[nBufferSize] = '\0';
   	m_nLen = nBufferSize; 
}

bool UTF8String::reserve(int nSize)
{
	if (nSize <= 0)
	{
		reset ();
		return false;
	}

	if (m_pBuffer  != NULL)
	{
		if (m_nBufSize >= nSize)
			return true;
		else
		{
			char* newBuffer = (char*)realloc(m_pBuffer, nSize);
			if (newBuffer == NULL)
				return false;
			
			m_pBuffer = newBuffer;
			m_nBufSize = nSize;		
		}
	}
	else
	{
		char* newBuffer = (char*) malloc(nSize);
		if (newBuffer == NULL)
			return false;			

		m_pBuffer = newBuffer;
		m_nBufSize = nSize;				
	}
	return true;
}
/**************************************************************************//**
 * Appends the zero terminated string at "pCStr" to the Utf8String. If
 * the string at "pubCStr" does not fit into the Utf8String's buffer, it
 * will be truncated.
 ******************************************************************************/	
void UTF8String::appendCStr( const char* pCStr)
{
	if (pCStr == NULL)
		return;
		
	int nlen = strlen (pCStr);
	const int nFreeBytes =  m_nBufSize -  m_nLen - 1;
	
	if (nFreeBytes < nlen)
	{
		if (reserve (nlen - nFreeBytes) == false)
			return;
	}
	
	while (*pCStr != '\0')
	{
		if (( m_nBufSize -  m_nLen) <= 1)
		{
			/* string buffer too small; truncate string */
			break;
		}
		m_pBuffer[ m_nLen++] = *(pCStr++);
	}
	m_pBuffer[ m_nLen] = '\0';	
}
/**************************************************************************//**
 * Appends the given "ulwChar" to the Utf8String in UTF8 format. If the
 * character does not fit into the Utf8String, it will be discarded.
 ******************************************************************************/	
void UTF8String::appendU8Char( UTF8Char u8Char)
{
	const int nFreeBytes =  m_nBufSize -  m_nLen - 1;

	int nCharSize = getCharSize(u8Char);
	if (nFreeBytes < nCharSize)
	{
		if (reserve (m_nLen + nCharSize + 1) == false)
			return;
	}

	if (u8Char < 0x80)
	{
		m_pBuffer[ m_nLen++] = (char) u8Char;
		m_pBuffer[ m_nLen] = '\0';
		return;
	}
	if (u8Char < 0x800)
	{
		m_pBuffer[ m_nLen++] = (char) (0xC0 | (u8Char >> 6));
		m_pBuffer[ m_nLen++] = (char) (0x80 | (u8Char & 0x3F));
		m_pBuffer[ m_nLen] = '\0';
		return;
	}
	if (u8Char < 0x10000)
	{
		m_pBuffer[ m_nLen++] = (char) (0xE0 | (u8Char >> 12));
		m_pBuffer[ m_nLen++] = (char) (0x80 | ((u8Char >> 6) & 0x3F));
		m_pBuffer[ m_nLen++] = (char) (0x80 | (u8Char & 0x3F));
		m_pBuffer[ m_nLen] = '\0';
		return;
	}
	if (u8Char < 0x200000)
	{
		m_pBuffer[ m_nLen++] = (char) (0xF0 | (u8Char >> 18));
		m_pBuffer[ m_nLen++] = (char) (0x80 | ((u8Char >> 12) & 0x3F));
		m_pBuffer[ m_nLen++] = (char) (0x80 | ((u8Char >> 6) & 0x3F));
		m_pBuffer[ m_nLen++] = (char) (0x80 | (u8Char & 0x3F));
		m_pBuffer[ m_nLen] = '\0';
		return;
	}
	MP_NORMAL_ASSERT(0);  /* invalid character */
	return;
}
/**************************************************************************//**
 * Returns the number of bytes this character code occupies in UTF8 format.
 ******************************************************************************/	
int UTF8String::getCharSize(UTF8Char u8Char) const
{
	  if (u8Char < 0x80)		return 1;
	  if (u8Char < 0x800)		return 2;
	  if (u8Char < 0x10000)		return 3;
	  MP_NORMAL_ASSERT(u8Char < 0x200000);
	  return 4;
}
unsigned long UTF8String::getCharAddress(int nU8CharId) const
{
	int nCount = 0;
	Iterator itChar = begin();

	while(itChar != end())
	{
		if (nCount == nU8CharId)
			return itChar.memory_addr();
		itChar ++;
		nCount++;
	}
	
	return itChar.memory_addr();	
}



int UTF8String::getByteIndex(int nU8CharId) const
{
	unsigned long memaddr = getCharAddress (nU8CharId);
	Iterator itChar = begin();
	return (memaddr - itChar.memory_addr() );
}

/**************************************************************************//**
 * Sets the string content to the zero terminated string at "pubCStr". If the
 * string at "pubCStr" does not fit into the Utf8String's buffer, it will be truncated.
 ******************************************************************************/	
void UTF8String::setCStr( const char* pCStr)
{
	if (pCStr == NULL)
	{
		clear();
		return;
	}

	int nlen = strlen_r(pCStr);
	if (reserve (nlen + 1) == false)
		return;
	
	clear();
	appendCStr (pCStr);
}
/**************************************************************************//**
 * Private helper function for comparing two zero-terminated strings "pCStr1" and "pCStr2".
 * \retval 0 if the strings match.
 * \retval -1 if 'pCStr1' < 'pCStr2'.
 * \retval +1 if 'pCStr1' > 'pCStr2'.
 ******************************************************************************/	
int UTF8String::compareCStrings(const char* pCStr1, const char* pCStr2) const
{
	if ( (pCStr1 == NULL) && (pCStr2 == NULL) )		return 0;
	if ( (pCStr1 == NULL) && (pCStr2 != NULL) )		return -1;
	if ( (pCStr1 != NULL) && (pCStr2 == NULL) )		return 1;
		
	while ((*pCStr1 == *pCStr2) && (*pCStr1 != '\0') &&	 (*pCStr2 != '\0'))
	{
		pCStr1++;
		pCStr2++;
	}

	if (*pCStr1 == *pCStr2)							return 0;
	if ((*pCStr1 == '\0') ||  (*pCStr1 < *pCStr2))	return -1;
	else											return 1;
}

void UTF8String::erase_subcontent(unsigned long start, unsigned long end)
{
	char* pubDest = (char*) start;
	const char* pubSrc = (const char*) end;
	const char* pubSrcLimit = (const char*) (m_pBuffer + m_nLen);
			
	if ( pubSrc == pubDest )
	{
		return;   /* nothing to be erased */
	}
	while ( pubSrc < pubSrcLimit )
	{
		*pubDest++ = *pubSrc++;
	}
	*pubDest = '\0';
	m_nLen =  pubDest -  m_pBuffer;
}
/******************************************************
 * UTF8String iterator functions
 ******************************************************/
UTF8String::Iterator UTF8String::begin( ) const
{
	return Iterator ( (unsigned long)m_pBuffer );
}

UTF8String::Iterator UTF8String::end( ) const
{
	return Iterator ( (unsigned long)m_pBuffer + m_nLen );
}
 

/******************************************************
 * UTF8String iterator
 ******************************************************/
 
void UTF8String::clear()
{
	if ( (m_nBufSize == 0) || (m_pBuffer == NULL) )
		return ;
		
	m_nLen = 0;
	m_pBuffer[0] = '\0';
}

bool UTF8String::empty() const
{
	if ( 	(m_pBuffer == NULL) 	||
			(m_nLen == 0)		||
			(m_nBufSize == 0) )
		return true;
	return false;
}

bool UTF8String::reset ()
{
	if (m_pBuffer != NULL)
	{
		free (m_pBuffer);
		m_pBuffer = NULL;
		m_nBufSize = 0;
	}
	m_nLen = 0;
	return true;
}
 
int UTF8String::size() const
{
	int nCount = 0;
	Iterator itChar = begin();

	while(itChar != end())
	{
		itChar ++;
		nCount++;
	}
	return nCount;
}

UTF8Char UTF8String::at (int nPos) const
{
	if (nPos < 0)
		return 0;
		
	int nCount = 0;
	Iterator itChar = begin();

	while(itChar != end())
	{
		itChar ++;
		nCount++;
		if (nCount == nPos)
			return itChar.getChar();
	}	
	return 0;
}
void UTF8String::limit (const Iterator& it)
{
	if ((it.fine () == false) || (it < begin()) || (it >= end()) )
		return;
	
	char* str = (char*) (it.memory_addr() );
	str[0] = '\0';
	m_nLen = it.memory_addr() - begin().memory_addr();
	return ;
}

bool UTF8String::erase(int nPos)
{
	if ( (nPos < 0) || (nPos >= size()) )
		return false;
	erase_subcontent (getCharAddress (nPos), getCharAddress (nPos + 1) );
	
	return true;
}

bool UTF8String::erase(int nStart, int nSize)
{ 
	if ( (nStart <= 0) || (nSize <= 0 ) )
		return false;

	int nU8Len = size();
	
	if ( nStart >= nU8Len )
		return false;
	
	erase_subcontent (getCharAddress (nStart), getCharAddress (nStart + nSize) );

	return true;
}
	
bool UTF8String::erase (const Iterator& it)
{	
	if (it.fine () == false)
		return false;
	
	if (it >= end())
		return false;	
		
	Iterator itNext = it ;
	itNext ++;
	
	erase_subcontent (it.memory_addr(), itNext.memory_addr() );
	return true;
}

bool UTF8String::erase (const Iterator& itStart, int nSize)
{
	if ( itStart.fine () == false)
		return false;
	
	if (itStart >= end())
		return false;
	if ((nSize <= 0) || (nSize > size()) )
		return false;
	
	Iterator it = itStart;	
	int nCount = 0;

	while(it != end())
	{
		it ++;
		nCount++;
		if (nCount == nSize)
		{
			erase_subcontent (itStart.memory_addr(), it.memory_addr() );	
			return true;
		}
	}
	if (it == end ())
	{
		erase_subcontent (it.memory_addr(), it.memory_addr() );
	}
	return false;
}

bool UTF8String::erase (const Iterator& itStart, const Iterator& itEnd)
{
	if ( (itStart.fine () == false) || (itEnd.fine() == false) )
		return false;
	
	if (itStart >= end())
		return false;
	if ( (itEnd < begin() ) || (itEnd > end ()) )
		return false;
		
	if(itEnd < itStart )
		return false;
		
	erase_subcontent (itStart.memory_addr(), itEnd.memory_addr() );	
	
	return true;
}
/**************************************************************************//**
 * Convert an UTF8 string to uppercase characters. Limitations: Works only for
 * - 0x0000..0x007F Basic Latin        (1 byte UTF8)
 * - 0x0080..0x00FF Latin 1 Supplement (2 byte UTF8)
 * - 0x0100..0x017F Latin Extended-A   (2 byte UTF8)
 * This should take care of the most common used characters.
 ******************************************************************************/		
void UTF8String::upper()
{
	const Iterator iterBegin = begin( );
	const Iterator iterEnd = end( );
	
	UTF8Char u8CharCode;
	UTF8Char u8CharCodeConverted;
	
	Iterator iter = iterBegin;
	
	while ( iter != iterEnd )
	{
		u8CharCode =  iter.getChar();
		u8CharCodeConverted = u8CharCode;
		
		/* Convert a..z into A..Z */
		if ( ( u8CharCode>=0x0061 ) && ( u8CharCode<=0x007A ) )
		{
			u8CharCodeConverted = u8CharCode - 0x0020;
		}
		
		/* convert Latin 1 supplement */
		else if ( ( u8CharCode>=0x00E0 ) && ( u8CharCode<=0x00FE ) )
		{
			if ( u8CharCode != 0x00F7 ) /* division sign not to be converted */
			{
				u8CharCodeConverted = u8CharCode - 0x0020;
			}
		}
		
		/* convert Latin Extended-A  Part A */
		else if ( ( ( u8CharCode>=0x0101 ) && ( u8CharCode<=0x0137 ) ) ||
		          ( ( u8CharCode>=0x014B ) && ( u8CharCode<=0x0163 ) ) ||
		          ( ( u8CharCode>=0x0167 ) && ( u8CharCode<=0x0177 ) ) )
		{
			if ( ( u8CharCode & 0x0001 ) == 0x0001 ) /* small character if code is odd */
			{
				u8CharCodeConverted = u8CharCode - 0x0001;
			}
		}
		
		/* convert Latin Extended-A  Part B */
		else if ( ( ( u8CharCode>=0x013A ) && ( u8CharCode<=0x0148 ) ) ||
		          ( ( u8CharCode>=0x017A ) && ( u8CharCode<=0x017E ) ) )
		{
			if ( ( u8CharCode & 0x0001 ) == 0x0000 ) /* small character if code is even */
			{
				u8CharCodeConverted = u8CharCode - 0x0001;
			}
		}
		else
		{
			/* no char found for up-conversion */
		}
		
		/* if CharCode was modified it needs to be exchanged in the source string */
		if ( u8CharCode!=u8CharCodeConverted )
		{
			/* size of utf8 coded character should stay the identical! */
			if ( getCharSize( u8CharCode )!= getCharSize( u8CharCodeConverted ) )
			{
				/* if the size stays not the same then we will not modify the sourcebuffer to avoid problems   this should n-e-v-e-r happen!  */
			    MP_NORMAL_ASSERT( 0 );
			}
			else
			{
				/* create UTF8 coding */
				UTF8String ustring;
				ustring.appendU8Char (u8CharCodeConverted);
			
				/* copy created code from workbuffer into main buffer */
				memcpy( (char*)iter.memory_addr(), &ustring.m_pBuffer[0], getCharSize( u8CharCodeConverted ) );
			}
		}
		iter ++;
	}
}

const char* UTF8String::c_str() const
{
	if ( (m_pBuffer == NULL) ||
		 (m_nBufSize == 0) ||
		 (m_nLen == 0) )
		return NULL;

	return  ( (char*)m_pBuffer );
}

UTF8String UTF8String::sub_str(const Iterator& itStart, const Iterator& itStop) const
{
	UTF8String str;
	
	if ((itStart.fine () == false) || (itStart < begin()) || (itStart >= end()) )
		return str;
	if ((itStop.fine () == false) || (itStop <= begin()) || (itStop > end()) )
		return str;
	if (itStart <= itStop)
		return str;
		
	str.init((const char*) (itStart.memory_addr()), itStop.memory_addr() - itStart.memory_addr() );
	return str;
}

/* 
 *
 */
UTF8String::Iterator UTF8String::find (UTF8Char u8Char) const
{
	return find (begin(), u8Char);
}
UTF8String::Iterator UTF8String::find (char cChar) const
{
	return find (begin(), cChar);
}

UTF8String::Iterator UTF8String::find (const Iterator& itStart, UTF8Char u8Char) const
{
	if ((itStart.fine () == false) || (itStart < begin()) || (itStart >= end()) )
		return end();
	
	Iterator it = itStart;	

	while(it != end())
	{
		if (it.getChar() == u8Char)
		{			
			return it;
		}
		it ++;
	}
	return end();
}

UTF8String::Iterator UTF8String::find (const Iterator& itStart, char cChar) const
{
	return find (itStart, (UTF8Char)cChar);
}

	
UTF8String::Iterator UTF8String::rfind (UTF8Char u8Char) const
{
	return rfind (end(), u8Char);
}

UTF8String::Iterator UTF8String::rfind (char cChar) const
{
	return rfind (end(), cChar);
}

UTF8String::Iterator UTF8String::rfind (const Iterator& itStart, UTF8Char u8Char) const
{
	if ((itStart.fine () == false) || (itStart < begin()) || (itStart > end()) )
		return end();
	
	Iterator it = itStart;	

	while(it >= begin())
	{
		if (it.getChar() == u8Char)
		{			
			return it;
		}

		if (it == begin()) break; // this extra if is needed because the -- operator accesses one byte before the buffer at the start of a string

		it --;
	}

	return end();
}

UTF8String::Iterator UTF8String::rfind (const Iterator& itStart, char cChar) const
{
	return rfind (itStart, (UTF8Char)cChar);
}
/******************************************************
 * operators overload
 ******************************************************/
UTF8String& UTF8String::operator += (const char* pStr)
{
	appendCStr (pStr);
	return *this;
}
UTF8String& UTF8String::operator += (UTF8Char  u8Char)
{
	appendU8Char (u8Char);
	return *this;
}
UTF8String& UTF8String::operator += (const UTF8String& uStr)
{
	appendCStr (uStr.c_str() );
	return *this;
}
	
UTF8String UTF8String::operator + (const char* pStr) const
{
	UTF8String temp = *this;
	temp.appendCStr (pStr);
	return temp;
}
UTF8String UTF8String::operator + (UTF8Char  u8Char) const
{
	UTF8String temp = *this;
	temp.appendU8Char (u8Char);
	return temp;
}

UTF8String UTF8String::operator + (const UTF8String& uStr) const
{
	UTF8String temp = *this;
	temp.appendCStr (uStr.c_str());
	return temp;
}

UTF8String& UTF8String::operator= (const UTF8String & ref)
{
    if( &ref == this || ref == *this) {}; // this is only needed for lint. There is no need to free any old memory in case of a self asignement.
	setCStr ( (const char*) ref.c_str());
	return (*this);
}

UTF8String& UTF8String::operator= (const char* pRef)
{
	setCStr ( (const char*) pRef);
	return (*this);
}

UTF8Char UTF8String::operator [](int nPos) const
{
	if (nPos > size())
		return 0;
	
	return at(nPos);
}

bool UTF8String::operator == ( const UTF8String & ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal == 0);
}

bool UTF8String::operator < ( const UTF8String & ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal == -1);
}

bool UTF8String::operator > ( const UTF8String & ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal == 1);
}

bool UTF8String::operator <= ( const UTF8String & ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal <= 0) ;
}

bool UTF8String::operator >= ( const UTF8String & ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal >= 0);
}

bool UTF8String::operator != (const UTF8String& ref) const
{
	int isequal = compareCStrings(m_pBuffer, ref.c_str());
	
	return (isequal != 0);
}

/** END OF FILE *************************************************************/
