/**************************************************************************//**
* \file      Utf8FileName.h
* \ingroup   MediaPlayer
*
*            Implementation of class UTF8FileName which handles UTF-8 format File/PATH name.
*            See .h file for description.
*
* \remark    Copyright: 2012, Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark    Scope:     MediaPlayer
* \remark    Authors:
*            - CM-AI/PJ-CD6 Xu Benny
******************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <cassert>
#include <string>
#include <sstream>
using namespace std;

#include <Utf8FileName.h>

UTF8FileName::UTF8FileName ()
{
}

UTF8FileName::UTF8FileName(const UTF8FileName& ref)
{
	m_fileName = ref.m_fileName;
	stripLeadingWhiteSpace();
	stripTrailingWhiteSpace();	
}

UTF8FileName::UTF8FileName( const char* pFileName )
{
	m_fileName = UTF8String (pFileName);
	stripLeadingWhiteSpace();
	stripTrailingWhiteSpace();	
}


UTF8FileName::UTF8FileName( const string& sFileName )
{
	m_fileName = UTF8String (sFileName.c_str());
}

UTF8FileName::~UTF8FileName()
{
}

void UTF8FileName::reset()
{
	m_fileName.reset();
}

bool UTF8FileName::isEmpty() const
{
	return m_fileName.empty ();
}


bool UTF8FileName::setFileName(const string& sFileName)
{
	m_fileName = sFileName.c_str();
	return true;
}

bool UTF8FileName::setFileName(const string& sPathName, const string& sFileName)
{
	if ( (sPathName.empty() == true) && (sFileName.empty() == true) )
		return false;
		
	stringstream ss;
	ss<<sPathName<<"/"<<sFileName;
	
	return setFileName (ss.str());
}

void UTF8FileName::removeEndFlag()
{
	char noteSep = '/';
	Utf8StringIter it = m_fileName.rfind(noteSep );

	if ( it == m_fileName.end( ) )
		return ;
		
	while ( it == ( m_fileName.end() - 1 ) )
	{
		m_fileName.erase(it);
		it = m_fileName.rfind(noteSep );
		if ( it == 0 )
			return;	
	}
	return;
}

UTF8FileName UTF8FileName::getParentDir() const
{	
	UTF8FileName emptyDir;
	UTF8FileName rootDir( "/");
	UTF8FileName curDir( "./");	
	if (isEmpty() == true)
		return emptyDir;
		
	if ( m_fileName ==  "/" )
		return  rootDir;
		
	UTF8FileName tmpFileName(m_fileName.c_str());
	char noteSep = '/';
	tmpFileName.removeEndFlag();

	UTF8String tmpStr = tmpFileName.m_fileName;
	
	Utf8StringIter it = tmpStr.rfind( noteSep );

	if ( it == tmpStr.end( ) )
		return curDir;
	/*
	 * Symbol '\' is not found, return current directory "."
	 */
	if ( it == 0 )
		return  rootDir;
	tmpStr.limit ( it );

	/*
	 * To check last character again
	 */
	it = tmpStr.rfind( noteSep );
	while ( it == ( tmpStr.end(  ) - 1 ) )
	{
		if ( it == 0 )
			return  rootDir;
		
		tmpStr.erase(it);
		it = tmpStr.rfind( noteSep );
	}
	return UTF8FileName(tmpStr.c_str());
}

string UTF8FileName::getPureName_cstr() const
{
	if (isEmpty() == true)
		return NULL;

	UTF8String tmpStr = m_fileName;
		
	if (tmpStr == UTF8String(( const char* )( "/" ) ))
		return "/";
	
	char noteSep = '/';
	Utf8StringIter it = tmpStr.rfind( noteSep );
	/*
	 * Symbol '\' is not found, return current directory "."
	 */
	if ( it == tmpStr.end( ) )
		return "./";
	
	while ( it == ( tmpStr.end() - 1 ) )
	{
		if ( it == 0 )
			return  "/";
		tmpStr.erase (it);
		
		it = tmpStr.rfind(noteSep );
	}
	it ++;
	
	string str((const char*) (it.memory_addr() ) );
	
	return  ( str );	
}

UTF8FileName UTF8FileName::getPureName() const
{
	UTF8FileName fileName;
	fileName.reset();
	
	string pureName = getPureName_cstr();		
	if (pureName.empty() == false)
		fileName.setFileName(string (pureName) );
	
	return fileName;
}

UTF8FileName& 	UTF8FileName::append ( const string & str )
{
	if (str.empty() == true)
		return *this;
		
	if (m_fileName.empty() == true)
	{
		setFileName(str);
		return *this;
	}
	
	stringstream ss;
	ss<<getString()<<str;
	
	string newValue = ss.str();
	
	setFileName(newValue);
	
	return *this;	
}
		
UTF8FileName& 	UTF8FileName::append ( char alphaChar )
{			
	stringstream ss;
	
	if (m_fileName.empty() == true)
		ss<<getString()<<alphaChar;
	else
		ss<<alphaChar;
	
	string newValue = ss.str();
	
	setFileName(newValue);
	
	return *this;	
}

UTF8FileName& 	UTF8FileName::appendPath ( const string & path )
{
	if (path.empty() == true)
		return *this;
		
	if (m_fileName.empty() == true)
	{
		setFileName(path);
		return *this;
	}
	
	stringstream ss;
	ss<<path<<"/"<<getString();
	
	string newValue = ss.str();
	
	setFileName(newValue);
	
	return *this;	
}

const char* UTF8FileName::getCString() const
{
	return (m_fileName.c_str());
}

string  UTF8FileName::getString() const
{
	string str( "" );
	const char* cstr = getCString();
	if (cstr != NULL)
		str = string(cstr);
	return str;
}

UTF8String  UTF8FileName::getUTF8String() const
{
	return m_fileName;
}

bool	UTF8FileName::isChildOf(const UTF8FileName& parentPath) const
{
	const string cur = this->getString();
	const string parent = parentPath.getString();
	
	if (cur.find(parent) == 0)
		return true;
	return false;
}

bool UTF8FileName::validateString(string & str)
{
	const char* repSymbol = "//";
	
	unsigned int ipos = str.find(repSymbol);
	if (ipos == string::npos)
		return false;
	
	string::iterator it = str.begin();
	it += ipos;
	
	str.erase( it);
	
	return true;
}

void UTF8FileName::validateFileName()
{
	if (m_fileName.empty() == true )
		return;
	
	stripLeadingWhiteSpace ();
	stripTrailingWhiteSpace();
	
	string str = string (m_fileName.c_str() );
	
	string::iterator it = str.begin();
	for (it=str.begin(); it<str.end(); it++)
	{
		if (*it == '\\')
			*it = '/';
	}	
	while ( validateString(str) == true){}
	
	m_fileName = str.c_str();
}

void UTF8FileName::stripLeadingWhiteSpace()
{
	const Utf8StringIter iterBegin = m_fileName.begin();
	const Utf8StringIter iterEnd = m_fileName.end();
	
	Utf8StringIter iter = iterBegin;
	
	while ( ( iter != iterEnd ) &&
	        ( iter.getChar(  ) == 0x20 ) )
	{
		iter ++;
	}
	m_fileName.erase( iterBegin, iter );
	
}
void UTF8FileName::stripTrailingWhiteSpace()
{
	const Utf8StringIter iterBegin = m_fileName.begin();
	Utf8StringIter iter = m_fileName.end();
	
	while ( iter != iterBegin )
	{
		iter --;
		if ( iter.getChar( ) != 0x20 )
		{
			break;
		}
		m_fileName.erase(iter );
	}
}

bool UTF8FileName::endsWith ( const string & str ) const
{
	string fn = string(m_fileName.c_str() );
	
	unsigned int nfn = fn.size();
	unsigned int nstr = str.size();
	
	if ( nfn < nstr )
		return false;
		
	unsigned int nidx = 	fn.rfind(str);
	if (nidx == string::npos)
		return false;
		
	if (nidx == (nfn - nstr) )
		return true;
	return false;
}

UTF8FileName& UTF8FileName::operator= (const UTF8FileName& ref)
{
    if( &ref == this || ref == *this) {}; // this is only needed for lint. There is no need to free any old memory in case of a self asignement.

	m_fileName = ref.getUTF8String();
	return (*this);
}

bool UTF8FileName::operator == (const UTF8FileName& ref) const
{
	return (m_fileName == ref.getUTF8String());
}

bool UTF8FileName::operator != (const UTF8FileName& ref) const
{
	return (m_fileName != ref.getUTF8String());		
}


/****EOF****/
