/*
 * FastUTF8.cpp
 *
 *  Created on: Dec 2, 2012
 *      Author: matthias
 */

#include <stdio.h>
#include "TypeDefinitions.h"

#include "FastUTF8.h"

int FastUTF8::CompareNC(tString c1, tString c2)
{
	return Compare(c1, c2, 1);
}

int FastUTF8::Compare(tString c1, tString c2)
{
	return Compare(c1, c2, 0);
}

int FastUTF8::Compare(tString c1, tString c2, int noCase)
{
	// standard character (MSBit is 0)
	// noCase is only affecting standard ASCII comparisons
	if (!(*c1 & 0x80)) {
		if (noCase) {
			return (tolower(*c1) - tolower(*c2));
		} else {
			return (*c1 - *c2);
		}
	}

	// UTF8: read out char len
	if ((*c1 & 0xe0) == 0xc0) { // 2 characters
		if (*c1 - *c2) return (*c1 - *c2); 			// compare first character, return if not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		return 0;									// both characters are equal
	}

	if ((*c1 & 0xf0) == 0xe0) { // 3 characters
		if (*c1 - *c2) return (*c1 - *c2); 			// compare first character, return if not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		return 0;									// both characters are equal
	}

	if ((*c1 & 0xf8) == 0xf0) { // 4 characters
		if (*c1 - *c2) return (*c1 - *c2); 			// compare first character, return if not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		c1++; c2++; 								// next characters
		if (!*c2 || *c1 - *c2) return (*c1 - *c2); 	// character not 0 and characters not equal
		return 0;									// both characters are equal
	}

	return MP_ERR_UTF8_UNKNOWN_CHARACTER_FORMAT;
}

int FastUTF8::isValid(tString nextString)
{
#if 0
    /* standard ASCII */
    if (!(*nextString & 0x80)) return 1;
#endif

    /* UTF8 */
    if ((*nextString & 0xe0) == 0xc0) { // two bytes
        if ((*(nextString+1) & 0xc0) == 0x80) return 1;
        return 0; // not valid
    }

    if ((*nextString & 0xf0) == 0xe0) { // three bytes
        if ((*(nextString+1) & 0xc0) == 0x80 &&
            (*(nextString+2) & 0xc0) == 0x80) return 1;
        return 0;
    }

    if ((*nextString & 0xf8) == 0xf0) { // four bytes
        if ((*(nextString+1) & 0xc0) == 0x80 &&
            (*(nextString+2) & 0xc0) == 0x80 &&
            (*(nextString+3) & 0xc0) == 0x80) return 1;
        return 0;
    }

    return 1;
}

FastUTF8::tString FastUTF8::Next(tString nextString)
{
	// standard character (MSBit is 0)
	if (!(*nextString & 0x80)) return nextString+1;

	// UTF8: read out char len
	if ((*nextString & 0xe0) == 0xc0) return nextString+2;
	if ((*nextString & 0xf0) == 0xe0) return nextString+3;
	if ((*nextString & 0xf8) == 0xf0) return nextString+4;

	// wrong formatted string
	return nextString+1;
}

FastUTF8::tString FastUTF8::Prev(tString prevString)
{
	while(1) {
	    prevString--; // step one char back
		if (!*prevString) return prevString; // if null, return
		if (!(*prevString & 0x80)) return prevString; // char was normal char
	}

	return NULL;
}

const FastUTF8::tString FastUTF8::EndsWith(const tString fileName, const tString extension)
{
	tString c;
	tString c1;
	tString c2;

	// loop over string
	for(c = (tString)fileName; *c; c = Next(c)) {

		// check extension
		for(c1 = c, c2 = (tString)extension; *c1 && *c2; c1 = Next(c1), c2 = Next(c2)) {
			if (Compare(c1, c2) != 0) break;
		}

		// ending found
		if (*c1 == 0 && *c2 == 0) {
			return c;
		}
	}

	return NULL;
}

const FastUTF8::tString FastUTF8::StartsWith(const tString fileName, const tString prefix)
{
    tString c1;
    tString c2;

    // loop over string
    for(c1 = fileName, c2 = (tString)prefix; *c1 && *c2; c1 = Next(c1), c2 = Next(c2))
    {
        if (Compare(c1, c2) != 0) break;
    }

    // ending found
    if (*c2 == 0) {
        return c1;
    }

    return NULL;
}


const FastUTF8::tString FastUTF8::EndsWithNC(const tString fileName, const tString extension)
{
	tString c;
	tString c1;
	tString c2;

	// loop over string
	for(c = (tString)fileName; *c; c = Next(c)) {

		// check extension
		for(c1 = c, c2 = (tString)extension; *c1 && *c2; c1 = Next(c1), c2 = Next(c2)) {
			if (CompareNC(c1, c2) != 0) break;
		}

		// ending found
		if (*c1 == 0 && *c2 == 0) {
			return c;
		}
	}

	return NULL;
}

const FastUTF8::tString FastUTF8::Split(tString &fileName, const tString fullName)
{
	return Split(fileName, fullName, (const FastUTF8::tString)"/"); //lint !e1773
}

int FastUTF8::Count(const tString fullName, const tString separator)
{
	tString c;
	int count = 0;

	// loop over fullName to find last slash
	for(c = (tString)fullName; *c; c = Next(c)) {
		if (!Compare(c, separator)) {
			count++;
		}
	}

	return count;
}

const FastUTF8::tString FastUTF8::Split(tString &fileName, const tString fullName, const tString separator)
{
	tString c;
	tString slash = NULL;

	// loop over fullName to find last separator
	for(c = (tString)fullName; *c; c = Next(c)) {
		if (!Compare(c, separator)) {
			slash = c;
		}
	}

	// no separator found
	if (slash == NULL) {
		//fileName = fullName;
		//*fullName = 0;
		fileName = NULL;

	// separator found
	} else {
		*slash = 0;
		slash++;

		// after last separator is something
		if(*slash) {
		    fileName = slash;
		} else {
		    fileName = NULL;
		}
	}

	return fullName;
}

const FastUTF8::tString FastUTF8::SplitExtension(tString &extension, const tString fullName)
{
    tString c;
    tString dot = NULL;

    // loop over fullName to find last dot (.)
    for(c = (tString)fullName; *c; c = Next(c)) {
        if (!Compare(c, (const FastUTF8::tString)".")) {  //lint !e1773
            dot = c;
        }
    }

    // no dot found
    if (dot == NULL) {
        //extension = fullName;
        //*fullName = 0;
        extension = NULL;

    // dot found
    } else {
        *dot = 0;
        dot++;
        extension = dot;
    }

    return fullName;
}
