
#ifndef _TYPEDEFINITIONS_H_
#define _TYPEDEFINITIONS_H_

#include <vector>
#include <string>
#include <cstring>
#include "TraceDefinitions.h"
#include "Framework_Errorcodes.h"
#include "BaseTypes.h"
#include "me.hpp"
#include "Utils.h"

using namespace std;

/*
 * error code enumeration
 */
typedef enum {

    MP_ERROR = -1,
    /* ListControl error codes */
    MP_ERR_LC_UNDEFINED_LIST_ID = 1000,
    MP_ERR_LC_NO_CTY_MATCH_TO_LTY,
    MP_ERR_LC_NO_CTY_IMPLEMENTATION,
    MP_ERR_LC_NOT_IMPLEMENTED_LIST_TYPE,
    MP_ERR_LC_NO_FURTHER_PARENT,
    MP_ERR_LC_GENERAL_ERROR,
    MP_ERR_LC_UNDEFINED_INDEX,
    MP_ERR_LC_INVALID_POSITION,
    MP_ERR_LC_UNDEFINED_OBJECTID,
    MP_ERR_LC_LIST_IS_NOT_PLAYABLE,
    MP_ERR_LC_NEXT_AT_END_OF_LIST,
    MP_ERR_LC_PREV_AT_START_OF_LIST,
    MP_ERR_LC_POSITION_NOT_PLAYABLE,
    MP_ERR_LC_EMPTY_LIST,
    MP_ERR_LC_INVALID_PATH,
    MP_ERR_LC_PLAYLIST_NOT_FOUND,
    MP_ERR_LC_PLAYLIST_FILE_CREATE,
    MP_ERR_LC_PLAYLIST_FILE_OPERATATION,
    MP_ERR_LC_PLAYLIST_DISK_FULL,
    MP_ERR_LC_PLAYLIST_FILE_PERMISSION,
    MP_ERR_LC_PLAYLIST_DATABASE_ERROR,
    MP_ERR_LC_PLAYLIST_MULTIPLE_ENTRIS_FOUND,


    /* CustomControl error codes */
    MP_ERR_CC_UNABLE_TO_CREATE_LIST = 2000,
    MP_ERR_CC_DEVICE_RECOGNITION_TIMEOUT,
    MP_ERR_CC_DEVICE_NOT_CONNECTED,
    MP_ERR_CC_INVALID_PARAM,
    MP_ERR_CC_UNSUPPORTED_DEVICE_TYPE,
    MP_ERR_CC_DEVICE_NOT_ACTIVE,
    MP_ERR_CC_NO_OBJECT_AVAILABLE,
    MP_ERR_CC_INVALID_PAYLOAD,
    MP_ERR_CC_MEMORY_ALLOC_FAILED,


    /* PlayerManager error codes */
    MP_ERR_PM_INVALID_PARAM = 3000,
    MP_ERR_PM_NO_LIST_ID,
    MP_ERR_PM_NO_MEDIA_OBJECT,
    MP_ERR_PM_NO_TOTAL_PLAYTIME,
    MP_ERR_PM_WRONG_STATE,

    /* IPOD error codes */
    MP_ERR_IPOD_INVALID_PARAM = 4000,
    MP_ERR_IPOD_NO_DEVICE,
    MP_ERR_IPOD_NO_ACCESS,
    MP_ERR_IPOD_INIT,
    MP_ERR_IPOD_AUTHENTICATION,
    MP_ERR_IPOD_PLAY,
    MP_ERR_IPOD_SELECT,
    MP_ERR_IPOD_SEEK,
    MP_ERR_IPOD_METADATA,
    MP_ERR_IPOD_TIMER,
    MP_ERR_IPOD_NO_ALBUMART,
    MP_ERR_IPOD_TIMEOUT,
    MP_ERR_IPOD_FILE_ACCESS,
    MP_ERR_IPOD_NO_MEM,
    MP_ERR_IPOD_NO_FREE_SPACE,
    MP_ERR_IPOD_COMMAND,

    /* DB Manager error codes */
    MP_ERR_DB_UNEXPECTED = 5000,
    MP_ERR_DB_ALREADY_INITIALIZED,
    MP_ERR_DB_INITIALIZATION_FAILURE,
    MP_ERR_DB_DATABASE_RECREATION_FAILURE,
    MP_ERR_DB_ALREADY_CONNECTED,
    MP_ERR_DB_DATABASE_FILE_FAILURE,
    MP_ERR_DB_DATABASE_CONNECTION_FAILURE,
    MP_ERR_DB_DATABASE_SCHEMA_FAILURE,
    MP_ERR_DB_NOT_CONNECTED,
    MP_ERR_DB_UNFINALIZED_STATEMENTS,
    MP_ERR_DB_DATABASE_CLOSE_FAILURE,
    MP_ERR_DB_PREPARE_FAILURE,
    MP_ERR_DB_STEP_FAILURE,
    MP_ERR_DB_INTEGRITY_FAILURE,
    MP_ERR_DB_INVALID_SCHEMA_VERSION,
    MP_ERR_DB_FINALIZE_FAILURE,
    MP_ERR_DB_LIST_NOT_FOUND,
    MP_ERR_DB_CREATE_LIST_CONTEXT,
    MP_ERR_DB_QUERY_FAILED,
    MP_ERR_DB_UNSUPPORTED_LIST_TYPE,
    MP_ERR_DB_UNSUPPORTED_CAT_TYPE,
    MP_ERR_DB_SELECT_FAILED,
    MP_ERR_DB_END_OF_LIST,
    MP_ERR_DB_INSERT_FAILED,
    MP_ERR_DB_DELETE_FAILED,
    MP_ERR_DB_UPDATE_FAILED,
    MP_ERR_DB_EXECUTE_FAILED,
    MP_ERR_DB_OPEN_FAILED,
    MP_ERR_DB_NOT_OPEN,
    MP_ERR_DB_LIST_IS_NO_PLAYLIST,
    MP_ERR_DB_RESET_IN_SUPSEND,
    MP_ERR_DB_BEGIN_TRANSACTION_FAILED,
    MP_ERR_DB_END_TRANSACTION_FAILED,
    MP_ERR_DB_REGISTER_FAILED,
    MP_ERR_DB_UNREGISTER_FAILED,
    MP_ERR_DB_SEARCHKEYS_NOT_SUPPORTED,
    MP_ERR_DB_FILE_TYPE_NOT_FOUND,
    MP_ERR_DB_UPD_MYMEDIA_DB_STOPPED,
    MP_ERR_DB_UPD_MYMEDIA_DB_SHUTDOWN,
    MP_ERR_DB_UPD_MYMEDIA_DB_RUNNING,
    MP_ERR_DB_UPD_MYMEDIA_DB_UNKNOWN,
    MP_ERR_DB_INVALID_HANDLE,
    MP_ERR_DB_INVALID_FTS,

    /* utf8 */
    MP_ERR_UTF8_UNKNOWN_CHARACTER_FORMAT = 6000,

    /* data provider */
    MP_ERR_DP_NO_ALBUM_ART_AVAIL = 7000,
    MP_ERR_DP_FORMAT_NOT_SUPPORTED,
    MP_ERR_DP_NO_DEVICE_FOUND,
    MP_ERR_DP_NO_ALBUM_ART_POINTER,

    /* general error codes */
    MP_ERR_DBUS = 8000,
    MP_ERR_UMOUNT_BUSY,
    MP_ERR_UMOUNT_NOT_SUPPORTED,
    MP_ERR_UMOUNT_ERROR,
    MP_ERR_ACTION_ERROR,
    MP_ERR_NO_PLAYER,
    MP_ERR_NO_PROFILE,

    /* MTP error codes */
    MP_ERR_MTP_GENERAL_ERROR     = 9000,
    MP_ERR_MTP_DECODEMOUNTPOINT,
    MP_ERR_MTP_UNKNOWN,
    MP_ERR_MTP_UNKNOWNSUFFIX,
    MP_ERR_MTP_WRONGINPARAM,
    MP_ERR_MTP_WRONGTYPE,
    MP_ERR_MTP_NODEVICE,
    MP_ERR_MTP_NODEVMATCH,
    MP_ERR_MTP_DECODEMOUNT,
    MP_ERR_MTP_ENCODEMOUNT,
    MP_ERR_MTP_FILENOSPACE,
    MP_ERR_MTP_FILEDELETE,
    MP_ERR_MTP_FILEEXISTS,
    MP_ERR_MTP_FILERENAME,
    MP_ERR_MTP_DIRNOTEXIST,
    MP_ERR_MTP_LIBMTP_TRACKNULL,
    MP_ERR_MTP_LIBMTP_GETFILE,
    MP_ERR_MTP_LIBMTP_GETDEVLIST,

    /* MediaEngine error codes */
    MP_ERR_MEDIA_ENGINE_GENERAL_ERROR     = 10000,
    MP_ERR_MEDIA_ENGINE_STARTUP_ERROR,

    /* MediaEngine error codes */
    MP_ERR_CDDA_OPEN_SG     = 10000,
    MP_ERR_CDDA_OPEN_SR,
    MP_ERR_CDDA_IOCTRL,
    MP_ERR_CDDA_CDTEXT,
    MP_ERR_CDDA_CD_NOT_MOUNTED,
    MP_ERR_CDDA_WRONG_URL,
    MP_ERR_CDDA_WRONG_TRACK_NO,
    MP_ERR_DVD_COMMAND_EXECUTION,
    MP_DVD_SKIP_NOT_POSSIBLE,
    MP_ERR_DVD_PLAYBACK,
    MP_DVD_INTERFACE_ERROR,
    /* BT Stack Error */
    MP_ERR_FILE_NOT_EXIST,

    /*Root Daemon Error */
    MP_ERR_RD_FILE_PERMISSION,

    /*File Operations Error*/
    MP_ERR_DEVICE_NOT_EXIST,
    MP_ERR_FILE_OPERATION_INPROGESS,
    MP_ERR_NO_SPACE,

    /*XML Parse Error*/
    MP_XML_PARSE_ERR,
    /*Gracenote error*/
    MP_ERR_GN_ALBUMART_FAILURE,
    MP_ERR_RIPPERSM_BUSY,

    MP_ERR_BT_BROWSE_FAILED,
    // DataPool IO Error codes.
    MP_ERR_DATAPOOL_READ_FAILURE,
    MP_ERR_DATAPOOL_WRITE_FAILURE,
    MP_ERR_BT_ALBUMART_FAILURE,

    // USB Fingerprint Error code
    MP_ABORTED_CALCFINGERPRINT,

    MP_LAST_ERROR_CODE
} tError;

const char *errorString(int res);
void InitErrorStrings(void);

/*
 * global test variables
 */
extern int gIPODConnected;
extern int gMTPConnected;


/********************
 * Global Definitions
 ********************/
#ifdef TARGET_BUILD
#define USB_ALSA_DEVICENAME "AdevModuleTest" //only used for module test on the target
#else
#define USB_ALSA_DEVICENAME "hw:0,0" //only used for module test on ubuntu
#endif

#define USB_CTRL_SM_ANSWER_TIMEOUT_MS 5500L
#define USB_CTRL_SM_METADATA_ANSWER_TIMEOUT_MS 8000L
#define IPOD_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define MTP_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define BT_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define CDDA_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define CS_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define DVD_CTRL_SM_ANSWER_TIMEOUT_MS 6800L
#define RIPPER_CTRL_SM_ANSWER_TIMEOUT_MS 100000L
#define PLAYBACK_RESPONSE_TIMEOUT 3000

#define DEV_DISP_SM_ANSWER_TIMEOUT_MS (IPOD_CTRL_SM_ANSWER_TIMEOUT_MS + 100) //6900

#define INDEXER_SM_ANSWER_TIMEOUT_MS 10000L
#define ALBUMART_INDEXER_SM_ANSWER_TIMEOUT_MS 2000L
#define PLAYER_MANAGER_SM_ANSWER_TIMEOUT_MS (DEV_DISP_SM_ANSWER_TIMEOUT_MS + 100) //7000
#define PICTURE_MANAGER_SM_ANSWER_TIMEOUT_MS 2000L

#define IPOD_CTRL_SM_PLAYTIME_SYNC_MS 999L    // playtime in HMI and device to be in sync

const char IPC_DBUS_PARAM_DELIMITER_CHAR='\n';
const int IPC_DBUS_PARAM_DELIMITER_INT='\n';
const char IPC_DBUS_PARAM_COMMA_DELIMITER_CHAR=',';

typedef unsigned long long tReason;
#define tReason_format "l"
#define tReason_init 0

/*******************
 * MOST Enumerations
 *******************/

/**
 * Type for the playback mode
 */
typedef enum {
    PBM_NORMAL = 0UL,   /**< Current playing sequence is executed in sequential order */
    PBM_RANDOM = 1UL,   /**< Current playing sequence is exectued in random order */
    PBM_INVALID = 2UL
} tPlaybackMode;
#define tPlaybackMode_format "i"
#define tPlaybackMode_init PBM_NORMAL

/**
 * Type for the repeat mode
 */
typedef enum {
    RPT_NONE = 0UL,                 /**< Continue sequentially through the currently playing list of media objects. Stop at the end of the list */
    RPT_ONE = 1UL,                  /**< Repeat the current playing track */
    RPT_LIST = 2UL,                 /**< Upon completion of the currently playing list of media objects, repeat playback of the entire list */
    RPT_LIST_WITH_SUBLISTS = 3UL,   /**< Upon completion of the currently playing list of media objects including all sub lists,
                                         repeat playback of the entire list and sublists*/

    RPT_ALL_LISTS = 4UL,             /* Upon completion of the currently playing list and each of its sublists
                                        repeat playback of the list and sublists

                                        Note:
                                        1. For IndexedList:RPT_ALL_LISTS behaviour is same as RPT_LIST_WITH_SUBLISTS.
                                        2. For FileList:
                                        A Sublist is a list of mediaObjects of currently playing folder's subfolder.Thus each of its subfolder's lists will be played.*/

    RPT_ALL = 5UL,                   /**< Upon completion of the currently playing list of media objects, repeat playback of all media objects on the device */

    RPT_INVALID = 6UL
} tRepeatMode;
#define tRepeatMode_format "i"
#define tRepeatMode_init RPT_NONE

/**
 * Type declaration of an enum containing the indexing state
 */
typedef enum
{
    IDS_NOT_SUPPORTED = 0UL,            /**< The device does not support indexing (and can only be browsed via folder structure) */
    IDS_CURRENTLY_NOT_SUPPORTED = 1UL,  /**< The device does currently not support indexing due to fingerprint calculation error */
    IDS_NOT_STARTED = 2UL,              /**< The device supports indexing but the indexing operation has not started */
    IDS_PARTIAL = 3UL,                  /**< The device has been partially indexed */
    IDS_COMPLETE_FULL_DB = 4UL,         /**< The device has only been partly indexed due to no more space in DB */
    IDS_COMPLETE = 5UL                  /**< The device has been fully indexed */
} tIndexingState;
#define tIndexingState_format "i"
#define tIndexingState_init IDS_NOT_SUPPORTED

/**
 * Type declaration of an enum containing the list type
 */
typedef enum
{
    LTY_GENRE = 0UL,                        /**< List of all genres */
    LTY_GENRE_SEARCH                    = 1UL,  /**< List of all genres - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ARTIST                          = 2UL,  /**< List of all artists */
    LTY_ARTIST_SEARCH                   = 3UL,  /**< List of all artists - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ALBUM                           = 4UL,  /**< List of all albums */
    LTY_ALBUM_SEARCH                    = 5UL,  /**< List of all albums - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_SONG                            = 6UL,  /**< List of all songs */
    LTY_SONG_SEARCH                     = 7UL,  /**< List of all songs - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_GENRE_ARTIST                    = 8UL,  /**< List of artists filtered by genre */
    LTY_GENRE_ARTIST_SEARCH             = 9UL,  /**< List of artists filtered by genre - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_GENRE_ARTIST_ALBUM              = 10UL, /**< List of albums filtered by genre and artist */
    LTY_GENRE_ARTIST_ALBUM_SEARCH       = 11UL, /**< List of albums filtered by genre and artist - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_GENRE_ARTIST_ALBUM_SONG         = 12UL, /**< List of songs filtered by genre, artist, and album */
    LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH  = 13UL, /**< List of songs filtered by genre, artist, and album - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_GENRE_ARTIST_SONG               = 14UL, /**< List of songs filtered by genre and artist */
    LTY_GENRE_ARTIST_SONG_SEARCH        = 15UL, /**< List of songs filtered by genre and artist - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_GENRE_ALBUM                     = 16UL, /**< List of albums filtered by genre */
    LTY_GENRE_ALBUM_SEARCH              = 17UL, /**< List of albums filtered by genre - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_GENRE_ALBUM_SONG                = 18UL, /**< List of songs filtered by genre and album */
    LTY_GENRE_ALBUM_SONG_SEARCH         = 19UL, /**< List of songs filtered by genre and album - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_GENRE_SONG                      = 20UL, /**< List of songs filtered by genre */
    LTY_GENRE_SONG_SEARCH               = 21UL, /**< List of songs filtered by genre - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ARTIST_ALBUM                    = 22UL, /**< List of albums filtered by artist */
    LTY_ARTIST_ALBUM_SEARCH             = 23UL, /**< List of albums filtered by artist - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ARTIST_ALBUM_SONG               = 24UL, /**< List of songs filtered by artist and album */
    LTY_ARTIST_ALBUM_SONG_SEARCH        = 25UL, /**< List of songs filtered by artist and album - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ARTIST_SONG                     = 26UL, /**< List of songs filtered by artist */
    LTY_ARTIST_SONG_SEARCH              = 27UL, /**< List of songs filtered by artist - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_ALBUM_SONG                      = 28UL, /**< List of songs filtered by album */
    LTY_ALBUM_SONG_SEARCH               = 29UL, /**< List of songs filtered by album - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_PODCAST                         = 30UL, /**< List of all podcasts */
    LTY_PODCAST_SEARCH                  = 31UL, /**< List of all podcasts  - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_PODCAST_EPISODE                 = 32UL, /**< List of episodes filtered by podcast */
    LTY_PODCAST_EPISODE_SEARCH          = 33UL, /**< List of episodes filtered by podcast - matching a search string */  //->Roadmap 160014 'full Text Search'
    LTY_AUDIOBOOK                       = 34UL, /**< List of all audiobook titles */
    LTY_AUDIOBOOK_SEARCH                = 35UL, /**< List of all audiobook titles - matching a search string  */ //->Roadmap 160014 'full Text Search'
    LTY_BOOKTITLE_CHAPTER               = 36UL, /**< List of chapters filtered by book */
    LTY_BOOKTITLE_CHAPTER_SEARCH        = 37UL, /**< List of chapters filtered by book  - matching a search string *///->Roadmap 160014 'full Text Search'
    LTY_AUTHOR                          = 38UL, /**< List of all audiobook authors */
    LTY_AUTHOR_SEARCH                   = 39UL, /**< List of all audiobook authors - matching a search string *///->Roadmap 160014 'full Text Search'

    LTY_AUTHOR_BOOKTITLE                = 40UL, /**< List of audiobook titles filtered by author */
    LTY_AUTHOR_BOOKTITLE_SEARCH         = 41UL, /**< List of audiobook titles filtered by author */ //->Roadmap 160014 'full Text Search'
    LTY_AUTHOR_BOOKTITLE_CHAPTER        = 42UL, /**< List of audiobook chapters filtered by title and author */
    LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH = 43UL, /**< List of audiobook chapters filtered by title and author *///->Roadmap 160014 'full Text Search'
    LTY_COMPOSER                        = 44UL, /**< List of all composers */
    LTY_COMPOSER_SEARCH                 = 45UL, /**< List of composers - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_COMPOSER_ALBUM                  = 46UL, /**< List of albums filtered by composer */
    LTY_COMPOSER_ALBUM_SEARCH           = 47UL, /**< List of albums filtered ba composer  - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_COMPOSER_ALBUM_SONG             = 48UL, /**< List of songs filtered by album and composer */
    LTY_COMPOSER_ALBUM_SONG_SEARCH      = 49UL, /**< List of songs filtered by album and composer - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_COMPOSER_SONG                   = 50UL, /**< List of songs filtered by composer */
    LTY_COMPOSER_SONG_SEARCH            = 51UL, /**< List of songs filtered by composer - matching a search string*///->Roadmap 160014 'full Text Search'*/
    LTY_VIDEO                           = 52UL, /**< List of all videos */
    LTY_VIDEO_SEARCH                    = 53UL, /**< List of all videos - matching a search string*///->Roadmap 160014 'full Text Search'*/
    LTY_VIDEO_EPISODE                   = 54UL, /**< List of video episodes filtered by video */
    LTY_VIDEO_EPISODE_SEARCH            = 55UL, /**< List of video episodes filtered by video  - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_PLAYLIST                        = 56UL, /**< List of all playlists */
    LTY_PLAYLIST_SEARCH                 = 57UL, /**< List of all playlists - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_PLAYLIST_SONG                   = 58UL, /**< List of songs filtered by playlist */
    LTY_PLAYLIST_SONG_SEARCH            = 59UL, /**< List of songs filtered by playlist - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_CURRENT_SELECTION               = 60UL, /**< List of current selection in the ipod */
    //LTY_CURRENT_SELECTION_SEARCH        = 61UL, /**< List of current selection in the ipod - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_IMAGE                           = 62UL, /**< List of all images */
    LTY_IMAGE_SEARCH                    = 63UL, /**< List of all images - matching a search string*///->Roadmap 160014 'full Text Search'  */
    LTY_PLAYLIST_INTERNAL               = 64UL, /**< List of Internal playlist created by the user */
    LTY_PLAYLIST_INTERNAL_SEARCH        = 65UL, /**< List of Internal playlist created by the user - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_CD                              = 66UL, /**< List of CD content */
    //LTY_CD_SEARCH                       = 67UL, /**< List of CD content - matching a search string*///->Roadmap 160014 'full Text Search'   */
    LTY_EPISODE_OF_VIDEO                = 68UL, /**< List of all video episodes */
    LTY_EPISODE_OF_VIDEO_SEARCH         = 69UL, /**< List of all video episodes - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_IMAGE_FOLDER                    = 70UL, /**< List of all folders having atleast one image in it */
    //LTY_IMAGE_FOLDER_SEARCH             = 71UL, /**< List of all folders having atleast one image in it - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_IMAGE_FOLDER_ITEM               = 72UL, /**< List of images filtered by image folder */
    //LTY_IMAGE_FOLDER_ITEM_SEARCH      = 73UL, /**< List of images filtered by image folder - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_COMPILATION                     = 74UL, /**< List of albums filtered by compilations */
    LTY_COMPILATION_SEARCH              = 75UL, /**< List of albums filtered by compilations - matching a search string*///->Roadmap 160014 'full Text Search'
    LTY_COMPILATION_SONG                = 76UL, /**< List of songs filtered by album and compilations */
    LTY_COMPILATION_SONG_SEARCH         = 77UL, /**< List of songs filtered by album and compilations  - matching a search string*///->Roadmap 160014 'full Text Search'

    LTY_YEAR                            = 78UL,
    LTY_YEAR_ALBUM                      = 79UL,
    LTY_YEAR_ALBUM_SEARCH               = 80UL,
    LTY_YEAR_ALBUM_SONG                 = 81UL,
    LTY_YEAR_ALBUM_SONG_SEARCH          = 82UL,
    LTY_USER_FAVORITES_SONG             = 83UL,
    LTY_USER_FAVORITES_SONG_SEARCH      = 84UL,
    LTY_END_OF_EXTERNAL_LIST_TYPES, // this list type marks the end of the public list types
    //==============================================================================================================

    /* list types used by database manager */
    LTY_FILELIST,                           /**< List of files and folders in a folder */
    LTY_FILELIST_SORTED_FOR_INDEXING,
    LTY_FILELIST_UNSORTED,                  /**< Unsorted list of files and folders in a folder */
    LTY_FILELIST_PLAYLIST,                  /**< Unsorted list of files used for a playlist */
    LTY_FILELIST_COUNT,                     /**< Count of files and folders in a folder */
    LTY_FILELIST_MEDIAOBJECTS,              /**< List of mediaobjects in a folder*/
    LTY_FILELIST_MEDIAOBJECTS_COUNT,        /**< Count of mediaobjects in a folder*/
    LTY_FILELIST_MEDIAOBJECTS_WITH_SUBFOLDERS,   /**< List of mediaobjects in a folder and all subfolders*/
    LTY_FILELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS,   /**< Count of mediaobjects in a folder and all subfolders*/
    LTY_BROWSELIST,                         /**< List of files and folders in a folder based on DB */
    LTY_BROWSELIST_COUNT,                   /**< Count of files and folders in a folder based on DB */
    LTY_BROWSELIST_MEDIAOBJECTS,            /**< List of mediaobjects in a folder based on DB*/
    LTY_BROWSELIST_MEDIAOBJECTS_COUNT,      /**< Count of mediaobjects in a folder based on DB*/
    LTY_BROWSELIST_MEDIAOBJECTS_WITH_SUBFOLDERS,   /**< List of mediaobjects in a folder and all subfolders based on DB*/
    LTY_BROWSELIST_MEDIAOBJECTS_ALL,        /**< List of mediaobjects in the root folder and all subfolders based on DB*/
    LTY_BROWSELIST_MEDIAOBJECTS_COUNT_WITH_SUBFOLDERS,   /**< Count of mediaobjects in a folder and all subfolders based on DB*/
    LTY_BROWSELIST_MEDIAOBJECTS_COUNT_ALL,  /**< Count of mediaobjects in the root folder and all subfolders based on DB*/
    LTY_MTP_FILELIST,                       /**< list of files and folder during MTP device browse via libmtp APIs */
    LTY_MTP_FILELIST_COUNT,                 /**< Count of files and folder during MTP device browse via libmtp APIs*/
    LTY_BLUETOOTH_FILELIST,                 /**< list of files and folder during BT Device folder browse via AVRCP 1.4 */
    LTY_BLUETOOTH_FILELIST_COUNT,           /**< Count of files and folder during BT Device folder browse via AVRCP 1.4  */
    LTY_DEVICES,                            /**< List of known devices */
    LTY_DEVICES_COUNT,                      /**< Count of known devices */
    LTY_DEVICES_INSERT,                     /**< insert of devices */
    LTY_DEVICES_UPDATE,                     /**< update of column in devices */
    LTY_DEVICES_UPDATE_COMMUNICATION_ERROR,
    LTY_DEVICES_UPDATE_DEVICE_DISCONNECTED_IN_COMMUNICATION_ERROR,
    LTY_DEVICES_UPDATE_PLAYBACKMODE,        /**< update of video and audio playback modes*/
    LTY_DEVICES_DELETE,                     /**< delete a row in devices */
    LTY_DEVICES_OLDEST,
    LTY_LATEST_DISC_EJECTED,
    LTY_DEVICES_LATEST_VALID_PARTITION,      /**<List of partitions of a given device*/
    LTY_VIRTUAL_DEVICES,                    /**< List of known virtual devices */
    LTY_VIRTUAL_DEVICES_DELETE,             /**< delete a row in virtual devices */
    LTY_IPOD_ACCESS,                        /**< List type to access the IPOD */
    LTY_IPOD_ACCESS_RANDOM,                 /**< List type to access the IPOD */
    LTY_IPOD_ACCESS_COUNT,                  /**< List type to access the IPOD */
    LTY_OBJECTS,
    LTY_MEDIAOBJECTS_INSERT,                /**< Insert of a media object */
    LTY_MEDIAOBJECTS,                       /**< One element of media objects selected by (deviceID) and/or URL and/or objectID */
    LTY_MEDIAOBJECTS_FILTERTAG4ID_OBJECTID_FILTER, /**< FilterTag4Id of a media object selected by objectID */
    LTY_MEDIAOBJECTS_OBJECTID_FILTER,       /**< One element of media objects selected by objectID or filtertag1-4 */
    LTY_MEDIAOBJECTS_DEVICEID_OBJECTID_FILTER,   /**< deviceID of media object element selected by (objectID) and/or filtertag1-4 */
    LTY_MEDIAOBJECTS_URL,                   /**< URL of media object element selected by (deviceID) and/or filtertag1ID */
    LTY_MEDIAOBJECTS_DEVICEID_UUID_BY_OBJECTID, /**< DeviceID and UUID of media object element selected by objectID */
    LTY_MEDIAOBJECTS_BY_UUID,               /**< One element of media objects selected by (deviceID) and/or UUID */
    LTY_IPOD_MEDIAOBJECTS_BY_UUID,          /**< One element of media objects selected by (deviceID) and/or UUID for Ipod/Iphone*/
    LTY_SONGS_FILTERTAGIDS_BY_UUID,         /**< Filtertags of song element selected by (deviceID) and/or UUID */
    LTY_SONGS_OBJECTID_BY_UUID,             /**< ObjectID of song element selected by (deviceID) and/or UUID */
    LTY_MEDIAOBJECTS_OBJECTID_BY_CATEGORY_FILTER, /**< ObjectID of media object element selected by (deviceID) and categoryID and filtertag1-4/title */
    LTY_MEDIAOBJECTS_OBJECTID_CATEGORYID_BY_TITLE, /**< ObjectID and category of media object element selected by (deviceID) and title */
    LTY_FOLDERHIERARCHY_PARENTID_BY_CHILD_NAME, /**< ParentID and depth of folder hierarchy element selected by (deviceID) and name of folder */
    LTY_MEDIAOBJECTS_COUNT,                 /**< Count of all media objects */
    LTY_MEDIAOBJECTS_PLAYABLE_FILTER_COUNT, /**< Count of all playable media objects (like CTY_SONG, CTY_VIDEO, CTY_PODCAST, CTY_AUDIOBOOK) selected by (deviceID) and/or (notPlayable) */
    LTY_MEDIAOBJECTS_DELETE,                /**< Delete one or more elements of media objects selected by deviceID and URL */
    LTY_SONGS_DELETE,                       /**< Delete one or more elements of media objects selected by (objectID) and/or (UUID) and/or (deviceID) */
    LTY_MEDIAOBJECTS_UPDATE,                /**< Update one or more elements of media objects selected by (deviceID) and/or URL and/or objectID */
    LTY_MEDIAOBJECTS_UPDATE_MEDIACONTENT,   /**< Update the media content if data base is not available when partial indexing done */
    LTY_MEDIAOBJECTS_GET_MEDIACONTENT,      /**< Read media content from the media object */
    LTY_MEDIAOBJECTS_MYME,
    LTY_MEDIAOBJECTS_MYME_COUNT,            /**< Count of all media objects in [MyMe]*/
    LTY_MEDIAOBJECTS_MYME_PLAYABLE_FILTER_COUNT, /**< Count of all playable media objects (like CTY_SONG, CTY_VIDEO, CTY_PODCAST, CTY_AUDIOBOOK) in [MyMe] selected by (deviceID) and/or (notPlayable)*/
    LTY_MEDIAOBJECTS_MYME_COPY,             /**< Copy of a media object from [Main] in [MyMe] selected by objectID */
    LTY_MEDIAOBJECTS_MYME_DEV_URL_COPY,     /**< Copy of a media object from [Main] in [MyMe] selected by deviceID and URL */
    LTY_MEDIAOBJECTS_MYME_DELETE,           /**< Delete one or more elements of media objects in [MyMe] selected by (objectID) and/or (UUID) and/or (deviceID)*/
    LTY_FAVORITES,
    LTY_FAVORITES_INSERT,
    LTY_FAVORITES_DELETE,
    LTY_FAVORITES_UPDATE,
    LTY_ITUNES_RADIO_STATION,
    LTY_IPOD_QUEUE_LIST,
    LTY_ALBUM_POSITION,                     /**< List of all albums in a device ordered by the album name(to get position for an album for coverflow) */
    LTY_ALBUM_UNKNOWN_ALBUMART,             /**< List of all albums without album art entry and also the album art entries whose cover arts are modified*/
    LTY_ALBUMART,
    LTY_ALBUMART_INSERT,
    LTY_ALBUMART_DELETE,
    LTY_ALBUMART_COVERARTPATH_OBJECTID_FILTER, /**< To get coverart path from Albumarts table*/
    LTY_USERS,                              /**< To get the users from the User table (17001_Personalization) */
    LTY_USERS_INSERT,                       /**< To insert the users to the User table (17001_Personalization) */
    LTY_USERS_DELETE,                       /**< To delete the users from the User table (17001_Personalization) */
    LTY_USERS_UPDATE,                       /**< To update the users table (17001_Personalization) */
    LTY_USER_FAVORITES,                     /**< To get the favorite information from favorites table for the user (17001_Personalization) */
    LTY_USER_FAVORITES_INSERT,              /**< To insert into favorites table to store a favorite for the user (17001_Personalization) */
    LTY_USER_FAVORITES_DELETE,              /**< To delete from favorites table for the user (17001_Personalization) */
    LTY_USER_FAVORITES_UPDATE,              /**< To update the favorites table for the user (17001_Personalization) */
    LTY_INDEXABLEDEVICES_COUNT,             /**< To count the number of indexable devices present in db */
    LTY_INDEXABLEDEVICES_OLDEST,            /**< To list indexable devices present in db on the basis of its timestamp */
    LTY_NONINDEXABLEDEVICES_OLDEST,         /**< To list non-indexable devices present in db on the basis of its timestamp */
    LTY_PLAYLISTNAME_COUNT,                 /**< To get count of playlist in same name based on platlistname and deviceID>*/

    //---Counts------------------------------------------------------------------------------------------------------------------
    LTY_GENRE_COUNT,                        /**< Count of all genres */
    LTY_GENRE_SEARCH_COUNT,                   /**< Count of all genres - which match search string*///->Roadmap 160014 'full text search'
    LTY_ARTIST_COUNT,                       /**< Count of all artists */
    LTY_ARTIST_SEARCH_COUNT,                  /**< Count of all artists - which match search string*///->Roadmap 160014 'full text search'
    LTY_ALBUM_COUNT,                        /**< Count of all albums */
    LTY_ALBUM_SEARCH_COUNT,                   /**< Count of all albums - which match search string*///->Roadmap 160014 'full text search'
    LTY_SONG_COUNT,                         /**< Count of all songs */
    LTY_SONG_SEARCH_COUNT,                    /**< Count of all songs - which match search string*///->Roadmap 160014 'full text search'
    LTY_GENRE_ARTIST_COUNT,                 /**< Count of artists filtered by genre */
    LTY_GENRE_ARTIST_SEARCH_COUNT,            /**< Count of artists filtered by genre - which match search string*///->Roadmap 160014 'full text search'

    LTY_GENRE_ARTIST_ALBUM_COUNT,           /**< Count of albums filtered by genre and artist */
    LTY_GENRE_ARTIST_ALBUM_SEARCH_COUNT,      /**< Count of albums filtered by genre and artist - which match search string*///->Roadmap 160014 'full text search'
    LTY_GENRE_ARTIST_ALBUM_SONG_COUNT,      /**< Count of songs filtered by genre, artist, and album */
    LTY_GENRE_ARTIST_ALBUM_SONG_SEARCH_COUNT, /**< Count of songs filtered by genre, artist and album - which match search string*///->Roadmap 160014 'full text search'
    LTY_GENRE_ARTIST_SONG_COUNT,            /**< Count of songs filtered by genre and artist */
    LTY_GENRE_ARTIST_SONG_SEARCH_COUNT,       /**< Count of songs filted by genre and artist - which match search string*///->Roadmap 160014 'full text search'
    LTY_GENRE_ALBUM_COUNT,                  /**< Count of albums filtered by genre */
    LTY_GENRE_ALBUM_SEARCH_COUNT,             /**< Count of albums filtered by genre - which match search string*///->Roadmap 160014 'full text search'
    LTY_GENRE_ALBUM_SONG_COUNT,             /**< Count of songs filtered by genre and album */
    LTY_GENRE_ALBUM_SONG_SEARCH_COUNT,        /**< Count of songs filtered by genre and album - which match search string*///->Roadmap 160014 'full text search'

    LTY_GENRE_SONG_COUNT,                   /**< Count of songs filtered by genre */
    LTY_GENRE_SONG_SEARCH_COUNT,              /**< Count of songs filtered by genre - which match search string*///->Roadmap 160014 'full text search'
    LTY_ARTIST_ALBUM_COUNT,                 /**< Count of albums filtered by artist */
    LTY_ARTIST_ALBUM_SEARCH_COUNT,            /**< Count of albums filtered by artists - which match search string*///->Roadmap 160014 'full text search'
    LTY_ARTIST_ALBUM_SONG_COUNT,            /**< Count of songs filtered by artist and album */
    LTY_ARTIST_ALBUM_SONG_SEARCH_COUNT,       /**< Count of songs filtered by artist and album - which match search string*///->Roadmap 160014 'full text search'
    LTY_ARTIST_SONG_COUNT,                  /**< Count of songs filtered by artist */
    LTY_ARTIST_SONG_SEARCH_COUNT,             /**< Count of songs filtered by artists - which match search string*///->Roadmap 160014 'full text search'
    LTY_ALBUM_SONG_COUNT,                   /**< Count of songs filtered by album */
    LTY_ALBUM_SONG_SEARCH_COUNT,              /**< Count of songs filtered by album - which match search string*///->Roadmap 160014 'full text search'

    LTY_PODCAST_COUNT,                      /**< Count of all podcasts */
    LTY_PODCAST_SEARCH_COUNT,                 /**< Count of all podcasts - which match search string*/ //->Roadmap 160014 'full text search'
    LTY_PODCAST_EPISODE_COUNT,              /**< Count of episodes filtered by podcast */
    LTY_PODCAST_EPISODE_SEARCH_COUNT,         /**< Count of episodes filtered by podcast - which match search string*/ //->Roadmap 160014 'full text search'
    LTY_AUDIOBOOK_COUNT,                    /**< Count of all audiobook titles */
    LTY_AUDIOBOOK_SEARCH_COUNT,               /**< Count of all audiobook titles  - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_BOOKTITLE_CHAPTER_COUNT,            /**< Count of chapters filtered by book */
    LTY_BOOKTITLE_CHAPTER_SEARCH_COUNT,       /**< Count of chapters filtered by book - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_AUTHOR_COUNT,                       /**< Count of all audiobook authors */
    LTY_AUTHOR_SEARCH_COUNT,                  /**< Count of all audiobook authors - which match search string*/ //->Roadmap 160014 'full text search'*/

    LTY_AUTHOR_BOOKTITLE_COUNT,             /**< Count of audiobook titles filtered by author */
    LTY_AUTHOR_BOOKTITLE_SEARCH_COUNT,        /**< Count of audiobook titles filtered by author - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_AUTHOR_BOOKTITLE_CHAPTER_COUNT,     /**< Count of audiobook chapters filtered by title and author */
    LTY_AUTHOR_BOOKTITLE_CHAPTER_SEARCH_COUNT,/**< Count of audiobook chapters filtered by title and author  - which match search string*///->Roadmap 160014 'full text search'*/
    LTY_COMPOSER_COUNT,                     /**< Count of all composers */
    LTY_COMPOSER_SEARCH_COUNT,                /**< Count of composer - which match search string*///->Roadmap 160014 'full text search'
    LTY_COMPOSER_ALBUM_COUNT,               /**< Count of albums filtered by composer */
    LTY_COMPOSER_ALBUM_SEARCH_COUNT,          /**< Count of albums filtered by composer - which match search string*///->Roadmap 160014 'full text search'
    LTY_COMPOSER_ALBUM_SONG_COUNT,          /**< Count of songs filtered by album and composer */
    LTY_COMPOSER_ALBUM_SONG_SEARCH_COUNT,     /**< Count of songs filtered by composer and album - which match search string*///->Roadmap 160014 'full text search'

    LTY_COMPOSER_SONG_COUNT,                /**< Count of songs filtered by composer */
    LTY_COMPOSER_SONG_SEARCH_COUNT,           /**< Count of songs filtered by composer - which match search string*/ //->Roadmap 160014 'full text search'
    LTY_VIDEO_COUNT,                        /**< Count of all videos */
    LTY_VIDEO_SEARCH_COUNT,                   /**< Count of all videos  - which match search string*///->Roadmap 160014 'full text search'
    LTY_VIDEO_EPISODE_COUNT,                /**< Count of video episodes filtered by video */
    LTY_VIDEO_EPISODE_SEARCH_COUNT,           /**< Count of video episodes filtered by video - which match search string*/ //->Roadmap 160014 'full text search'
    LTY_PLAYLIST_COUNT,                     /**< Count of all playlists */
    LTY_PLAYLIST_SEARCH_COUNT,                /**< Count of all playlists  - which match search string*///->Roadmap 160014 'full text search'*/
    LTY_PLAYLIST_SONG_COUNT,                /**< Count of songs filtered by playlist */
    LTY_PLAYLIST_SONG_SEARCH_COUNT,           /**< Count of songs filtered by playlist  - which match search string*/ //->Roadmap 160014 'full text search'

    LTY_CURRENT_SELECTION_COUNT,            /**< Count of current selection in the ipod */
    //LTY_CURRENT_SELECTION_COUNT,            /**< Count of current selection in the ipod  - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_IMAGE_COUNT,                        /**< Count of all images */
    LTY_IMAGE_SEARCH_COUNT,                   /**< Count of all images  - which match search string*/ //->Roadmap 160014 'full text search'
    LTY_PLAYLIST_INTERNAL_COUNT,            /**< Count of all internal playlists */
    LTY_PLAYLIST_INTERNAL_SEARCH_COUNT,       /**< Count of all internal playlists - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_CD_COUNT,                           /**< Count of CD content */
    //LTY_CD_SEARCH_COUNT,                    /**< Count of CD content - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_EPISODE_OF_VIDEO_COUNT,             /**< Count of all video episodes */
    LTY_EPISODE_OF_VIDEO_SEARCH_COUNT,        /**< Count of all video episodes - which match search string*/ //->Roadmap 160014 'full text search'*/

    LTY_IMAGE_FOLDER_COUNT,
    //LTY_IMAGE_FOLDER_SEARCH_COUNT,
    LTY_IMAGE_FOLDER_ITEM_COUNT,
    //LTY_IMAGE_FOLDER_ITEM_SEARCH_COUNT,
    LTY_COMPILATION_COUNT,                  /**< Count of all albums filtered by compilations */
    LTY_COMPILATION_SEARCH_COUNT,           /**< Count of all albums filtered by compilations - which match search string*/ //->Roadmap 160014 'full text search'*/
    LTY_COMPILATION_SONG_COUNT,             /**< Count of all songs filtered by album and compilations */
    LTY_COMPILATION_SONG_SEARCH_COUNT,      /**< Count of all songs filtered by album and compilations - which match search string*/ //->Roadmap 160014 'full text search'*/*/
    LTY_ALBUM_UNKNOWN_ALBUMART_COUNT,       /**< Count of all albums without album art entry */
    LTY_YEAR_COUNT,
    LTY_YEAR_ALBUM_COUNT,
    LTY_YEAR_ALBUM_SONG_COUNT,
    LTY_YEAR_ALBUM_SEARCH_COUNT,
    LTY_YEAR_ALBUM_SONG_SEARCH_COUNT,
    LTY_USER_FAVORITES_SONG_COUNT,
    LTY_USER_FAVORITES_SONG_SEARCH_COUNT,
    //--Keys-----------------------------------------------------------------------------------------------------------------
    LTY_ALBUM_SEARCH_KEY,                   /**< lists pair of album's first letter and count of rows starting with this letter  */
    LTY_ARTIST_SEARCH_KEY,                  /**< lists pair of artist's first letter and count of rows starting with this letter  */
    LTY_GENRE_SEARCH_KEY,                   /**< lists pair of genre's first letter and count of rows starting with this letter  */
    LTY_SONG_SEARCH_KEY,                    /**< lists pair of song's first letter and count of rows starting with this letter  */
    LTY_COMPOSER_SEARCH_KEY,                /**< lists pair of composer's first letter and count of rows starting with this letter  */
    LTY_GENRE_ARTIST_ALBUM_SEARCH_KEY,
    LTY_GENRE_ARTIST_SONG_SEARCH_KEY,
    LTY_GENRE_ARTIST_SEARCH_KEY,
    LTY_GENRE_ALBUM_SEARCH_KEY,
    LTY_YEAR_ALBUM_SEARCH_KEY,
    LTY_GENRE_SONG_SEARCH_KEY,
    LTY_ARTIST_ALBUM_SEARCH_KEY,
    LTY_ARTIST_SONG_SEARCH_KEY,
    LTY_COMPOSER_ALBUM_SEARCH_KEY,
    LTY_COMPOSER_SONG_SEARCH_KEY,
    LTY_PLAYLIST_SEARCH_KEY,
    LTY_VIDEO_SEARCH_KEY,
    LTY_VIDEO_EPISODE_SEARCH_KEY,
    LTY_PODCAST_SEARCH_KEY,
    LTY_PODCAST_EPISODE_SEARCH_KEY,
    LTY_AUDIOBOOK_SEARCH_KEY,

    //-----------------------------------------------------------------------------------------------------------------------
    LTY_LASTMODE_INSERT,
    LTY_LASTMODE_DELETE,
    LTY_LASTMODE_UPDATE,
    LTY_LASTMODE,
    LTY_LAST_PLAYTIME_INSERT,
    LTY_LAST_PLAYTIME_DELETE,
    LTY_LAST_PLAYTIME,
    LTY_SCAN_CONTEXT_INSERT,
    LTY_SCAN_CONTEXT_DELETE,
    LTY_SCAN_CONTEXT,

    LTY_LIVETAGS_INSERT,
    LTY_LIVETAGS_DELETE,
    LTY_LIVETAGS_SELECT,
    LTY_LIVETAGS_UPDATE,
    LTY_SQLITE_MASTER,    // to remove all triggers from database on startup
    LTY_SONG_GET_UUID,
    LTY_SYSTEM_INSERT,
    LTY_SYSTEM,
    LTY_TRACK_TEMP_CREATE,
    LTY_TRACK_TEMP_INSERT,
    LTY_TRACK_TEMP_DELETE,
    LTY_TRACK_TEMP_DELETE_ALL,
    LTY_TRACK_TEMP_SELECT_ALL,
    LTY_TRACK_TEMP_DROP,
    LTY_TRACK_TEMP_COUNT,
    LTY_TRACK_COUNT,
    LTY_VIDEO_TEMP_INSERT,
    LTY_VIDEO_TRACK_COUNT,
    LTY_PLAYLIST_TITLE_SELECT_ALL,
    LTY_PLAYLIST_TEMP_CREATE,
    LTY_PLAYLIST_TEMP_INSERT,
    LTY_PLAYLIST_TEMP_COUNT,
    LTY_PLAYLIST_TEMP_DELETE,
    LTY_PLAYLIST_TEMP_DROP,
    LTY_PLAYLIST_TEMP_SELECT_ALL,
    LTY_VIDEOPROPERTY_BRIGHTNESS,
    LTY_VIDEOPROPERTY_HUE,
    LTY_VIDEOPROPERTY_SATURATION,
    LTY_VIDEOPROPERTY_CONTRAST,
    LTY_VIDEOPROPERTY_BRIGHTNESSOFFSET,
    LTY_VIDEOPROPERTY_HUEOFFSET,
    LTY_VIDEOPROPERTY_SATURATIONOFFSET,
    LTY_VIDEOPROPERTY,
    LTY_VIDEOPROPERTY_UPDATE,
    LTY_DVDINFO_RESUME_INSERT,
    LTY_DVDINFO_RESUME,
    LTY_DVDINFO_SETUP_INSERT,
    LTY_DVDINFO_SETUP,
    LTY_MEDIAOBJECTS_RIPPED_FILE,
    LTY_MEDIAOBJECTS_RIPPED_FILE_METADATA,
    LTY_MEDIAOBJECTS_UPDATE_RIPPED_FILE,
    LTY_MEDIAOBJECTS_UPDATE_ALBUMARTSTRING,
    LTY_MEDIAOBJECTS_ALBUMART_FROM_UUID,
    LTY_MEDIAOBJECTS_TOTAL_RIPPED_FILE,
    LTY_MEDIAOBJECTS_UPDATE_MUSICBOX_DEVICE,
    LTY_MUSICBOX_PROP_UPDATE_ENC_QUALITY,
    LTY_MUSICBOX_PROP_UPDATE_AUTO_RIPPING,
    LTY_MUSICBOX_PROP_GET_ENC_QUALITY,
    LTY_MUSICBOX_PROP_GET_AUTO_RIPPING,
    LTY_END_OF_LTY,
    LTY_END_OF_LTY_COUNT,
    LTY_COMPILATION_SEARCH_KEY,
    LTY_TRACK_COUNT_SIZE,
    LTY_TRACK_COUNT_SIZE_FILEMODE,
    LTY_FILE_PERMISSION,
    LTY_DATETIME_SIZE_LENGTH,
    LTY_DIRECTORY_INFO_FROM_FOLDERPATH,
    LTY_DVDVIDEOPROPERTY_BRIGHTNESS,
    LTY_DVDVIDEOPROPERTY_HUE,
    LTY_DVDVIDEOPROPERTY_SATURATION,
    LTY_DVDVIDEOPROPERTY_CONTRAST,
    LTY_DVDVIDEOPROPERTY_BRIGHTNESSOFFSET,
    LTY_DVDVIDEOPROPERTY_HUEOFFSET,
    LTY_DVDVIDEOPROPERTY_SATURATIONOFFSET,
    LTY_DVDVIDEOPROPERTY,
    LTY_DVDVIDEOPROPERTY_UPDATE,
    LTY_MTPVIDEOPROPERTY_BRIGHTNESS,
    LTY_MTPVIDEOPROPERTY_HUE,
    LTY_MTPVIDEOPROPERTY_SATURATION,
    LTY_MTPVIDEOPROPERTY_CONTRAST,
    LTY_MTPVIDEOPROPERTY_BRIGHTNESSOFFSET,
    LTY_MTPVIDEOPROPERTY_HUEOFFSET,
    LTY_MTPVIDEOPROPERTY_SATURATIONOFFSET,
    LTY_MTPVIDEOPROPERTY,
    LTY_MTPVIDEOPROPERTY_UPDATE,
    LTY_YOMIMETADATA_INSERT,
    LTY_YOMIMETADATA_DELETE,
    LTY_YOMIMETADATA,
    LTY_MUSICBOXTOC_INSERT,
    LTY_MUSICBOXTOC_DELETE,
    LTY_MUSICBOXTOC,
    LTY_SONGS_ALBUMNAME_BY_UUID,
    LTY_MEDIAOBJECTS_UPDATE_GNDATA,
    LTY_FOLDERHIERARCHY_CHILDID_BY_DEVICEID,                   /**< List of all folders in a device in the order it was indexed */
    LTY_FOLDERHIERARCHY_CHILDID_BY_FOLDERPATH,    /**< Find childID of received folderPath wrt the deviceID >*/
    LTY_FOLDERHIERARCHY_FOLDERPATH_BY_CHILDID,     /**< Find folderPath of received childID wrt the deviceID >*/
    LTY_YOMIMETADATA_UPDATE,                       /**< Update yomimetadata in Yomi metadata table >*/
    LTY_MEDIAOBJECT_BY_ALBUMID,
    LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS,
    LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_INSERT,
    LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_UPDATE,
    LTY_ROUTEGUIDANCEDISPLAYCOMPONENTS_DELETE,
    LTY_DEVICEID_BY_SERIALNUMBER
#ifdef USE_PROTOTYPE_DIRECT_ADD_LTY_ALLSONGS
    ,LTY_ALLSONGS
#endif
} tListType;
#define tListType_format "i"
#define tListType_init LTY_END_OF_EXTERNAL_LIST_TYPES
tBoolean IsFileList(const tListType listType);
tBoolean IsPlayableList(const tListType listType);
tBoolean IsUserFavoriteList(const tListType listType);
/**
 * Type declaration of an enum describing the type of file
 */
typedef enum {
    FT_FOLDER = 0UL,    /**< The list item is a folder */
    FT_PLAYLIST = 1UL,  /**< The list item is a playlist file */
    FT_AUDIO = 2UL,     /**< The list item is an audio file */
    FT_VIDEO = 3UL,     /**< The list item is a video file */
    FT_IMAGE = 4UL,     /**< The list item is an image file */
    FT_UNKNOWN = 999UL
} tFileType;

/**
 * Type for the file type from the virtual table file system
 */
typedef enum
{
    FILE_TYPE_DIRECTORY = FT_FOLDER,
    FILE_TYPE_PLAYLIST = FT_PLAYLIST,
    FILE_TYPE_AUDIOOBJECT = FT_AUDIO,
    FILE_TYPE_VIDEOOBJECT = FT_VIDEO,
    FILE_TYPE_IMAGE = FT_IMAGE,
    FILE_TYPE_UNKNOWN,
    FILE_TYPE_FAILED
} tVTFileType;

/**
 * Type declaration of an enum describing the type of file for  selection
 */
typedef enum {
    FTS_AUDIO_VIDEO_PLAYLIST = 0UL, /**< The list shall contain folders and audio, video, playlist files */
    FTS_AUDIO = 1UL,                /**< The list shall contain folders and audio files */
    FTS_VIDEO = 2UL,                /**< The list shall contain folders and video files */
    FTS_PLAYLIST = 3UL,             /**< The list shall contain folders and playlist files */
    FTS_IMAGE = 4UL,                /**< The list shall contain folders and image files */
    FTS_AUDIO_PLAYLIST = 5UL,       /**< The list shall contain folders and audio, playlist files */
    FTS_ALL = 6UL                   /**< The list shall contain folders and all known files */

    //tbd.: Roadmap 160014 'Full text search' extensions
    //FTS_AUDIO_VIDEO_PLAYLIST_SEARCH,  /**< The list shall contain folders and audio, video, playlist files - matching a search string*/
    //FTS_AUDIO_SEARCH,             /**< The list shall contain folders and audio files - matching a search string*/
    //FTS_VIDEO_SEARCH,             /**< The list shall contain folders and video files - matching a search string*/
    //FTS_PLAYLIST_SEARCH,              /**< The list shall contain folders and playlist files - matching a search string*/
    //FTS_IMAGE_SEARCH,                /**< The list shall contain folders and image files - matching a search string*/
    //FTS_AUDIO_PLAYLIST_SEARCH,       /**< The list shall contain folders and audio, playlist files - matching a search string*/
    //FTS_ALL_SEARCH                   /**< The list shall contain folders and all known files - matching a search string*/
} tFileTypeSelection;

/**
 * Type declaration of an enum describing the media context
 */
typedef enum {
    MC_ALL = 0UL,   /**< File lists have no media content restriction */
    MC_AUDIO = 1UL, /**< File lists are restricted/filtered to audio content */
    MC_VIDEO = 2UL  /**< File lists are restricted/filtered to video content */
} tMediaContext;

#define tMediaContext_format "i"
#ifdef VARIANT_S_FTR_ENABLE_MP_AUTOPLAY
#define tMediaContext_init MC_AUDIO
#else
#define tMediaContext_init MC_ALL
#endif

/**
 * Type declaration of an enum describing the format of file
 */
typedef enum {
    FFT_MP3 = 0UL,      /**< MP3 audio file format */
    FFT_WMA = 1UL,      /**< Windows Media Audio (WMA) file format */
    FFT_AAC = 2UL,      /**< Advanced Audio Coding (AAC) audio file format */
    FFT_AAC_PLUS = 3UL, /**< High-Efficiency Advanced Audio Coding audio file format */
    FFT_M4A = 4UL,      /**< M4A container format for audio */
    FFT_MP4 = 5UL,      /**< MP4 Container format for audio and video */
    FFT_WAV = 6UL,      /**< Waveform (WAV) audio file format */
    FFT_OGG = 7UL,      /**< Ogg Vorbis audio file format */
    FFT_3GP = 8UL,      /**< 3GP and 3GPP container format for audio and video */
    FFT_3G2 = 9UL,      /**< 3G2 and 3GPP2 container format for audio and video */
    FFT_AIFF = 10UL,    /**< Audio Interchange File Format (AIFF) */
    FFT_AA = 11UL,      /**< Audible.com Audio Books */
    FFT_SBC = 12UL,     /**< Bluetooth Sub-Band Coding audio file format */
    FFT_MPEG = 13UL,    /**< MPEG video file format */
    FFT_AVI = 14UL,     /**< Audio Video Interleave (AVI) file format */
    FFT_MOV = 15UL,     /**< QuickTime multimedia format */
    FFT_DIVX = 16UL,    /**< DIVX file format */
    FFT_WMV = 17UL,     /**< Windows Media Video (WMV) file format */
    FFT_M4V = 18UL,     /**< iTunes Video file format */
    FFT_FLV = 19UL,     /**< Flash video format */
    FFT_M3U = 20UL,     /**< M3U playlist format */
    FFT_PLS = 21UL,     /**< PLS playlist format */
    FFT_XSPF = 22UL,    /**< XSPF playlist format */
    FFT_WPL = 23UL,     /**< WPL playlist format */
    FFT_B4S = 24UL,     /**< B4S playlist format */
    FFT_ASX = 25UL,     /**< ASX playlist format */
    FFT_JPEG = 26UL,    /**< JPEG image format */
    FFT_GIF = 27UL,     /**< GIF image format */
    FFT_BMP = 28UL,     /**< BMP image format */
    FFT_PNG = 29UL,     /**< PNG image format */
    FFT_TIF = 30UL,     /**< TIF image format */
    FFT_FLAC = 31UL,    /**< Free Lossless Audio Codec (FLAC) audio file format */
    FFT_M4B = 32UL,     /**< iTunes Audiobook file format */
    FFT_M4P = 33UL,     /**< DRM protected iTunes file format */
    FFT_M4R = 34UL,     /**< iTunes Ringtone file format */
    FFT_MKV = 35UL,     /**< MKV Video file format */
    FFT_UNKNOWN = 999UL
} tFileFormat;

/**
 * Type declaration of an enum describing the category type
 */
typedef enum {
    CTY_NONE = 0UL,     /**< Media object is not a category */
    CTY_GENRE = 1UL,    /**< Media object represents a genre */
    CTY_ARTIST = 2UL,   /**< Media object represents an artist */
    CTY_ALBUM = 3UL,    /**< Media object represents an album */
    CTY_SONG = 4UL,     /**< Media object represents a song */
    CTY_COMPOSER = 5UL, /**< Media object represents a composer */
    CTY_AUTHOR = 6UL,   /**< Media object represents an author */
    CTY_TITLE = 7UL,    /**< Media object represents an audiobook title */
    CTY_CHAPTER = 8UL,  /**< Media object represents a chapter  */
    CTY_NAME = 9UL,     /**< Media object represents a podcast or video name */
    CTY_EPISODE = 10UL, /**< Media object represents a podcast or video episode */
    CTY_PLAYLIST = 11UL,/**< Media object represents a playlist of other media objects */
    CTY_PODCAST = 12UL, /**< Media object represents a podcast object */
    CTY_VIDEO = 13UL,   /**< Media object represents a video object */
    CTY_PLAYLIST_ITEM = 14UL, /**< Media object represents a playlist entry object */
    CTY_FILE = 15UL,    /**< Media object represents a file */
    CTY_VTIPOD = 16UL,  /**< Media object represents an VTiPod URL */
    CTY_DIR = 17UL,      /**< Media object represents a directory (Roadmap 13012) */
    CTY_PLAYLIST_INTERNAL = 18UL,/**< Media object represents an internal playlist of other media objects */
    CTY_IMAGE = 19UL,    /**< Media object represents an image */
    CTY_AUDIOBOOK = 20UL, /**< Media object represents an audiobook object */
    CTY_RIPPING = 21UL , /**< Media object represents an ripped file */
    CTY_YEAR = 22UL,
    CTY_FAVORITES = 23UL  /**<Media Object represents a Favorite song of the user */
} tCategoryType;

/**
 * Type declaration of an enum containing playback action type
 */
typedef enum
{
    PBA_PLAY = 0UL,             /**< MediaPlayer starts/resumes playback of the current media file */
    PBA_PAUSE = 1UL,            /**< MediaPlayer pauses playback of the current media file */
    PBA_STOP = 2UL,             /**< MediaPlayer stops playback of the current media file */
    PBA_PREV = 3UL,             /**< MediaPlayer skips to the previous media file in the current playing sequence */
    PBA_NEXT = 4UL,             /**< MediaPlayer skips to the next media file in the current playing sequence */
    PBA_FREV_START = 5UL,       /**< MediaPlayer begins the fast reverse operation */
    PBA_FREV_STOP = 6UL,        /**< MediaPlayer stops the fast reverse operation */
    PBA_FFWD_START = 7UL,       /**< MediaPlayer begins the fast forward operation */
    PBA_FFWD_STOP = 8UL,        /**< MediaPlayer stops the fast forward operation */
    PBA_FOLDER_UP = 9UL,        /**< MediaPlayer skips to the next folder in the flat folder structure of the connected media source and starts playback of the first media file in it */
    PBA_FOLDER_DOWN = 10UL      /**< MediaPlayer skips to the previous folder in the flat folder structure of the connected media source and starts playback of the first media file in it */
} tPlaybackAction;
#define tPlaybackAction_format "i"
#define tPlaybackAction_init PBA_PLAY
///**
// * Type declaration of an enum containing playbackspeed status
// */
//typedef enum
//{
//    PLAYBACK_SPEED_OFF = 0UL,    // for FFWD/FREV via PlaybackAction interface
//    PLAYBACK_SPEED_ON        ,   // For PlaybackSpeed Interface
//    PLAYBACK_SPEED_INVALID
//}tPlaybackspeedState;
//#define tPlaybackspeedState_format "i"
//#define tPlaybackspeedState_init PLAYBACK_SPEED_OFF
/**
 * Type declaration of an enum containing playback speed value
 */
typedef enum
{

    PBK_SPEED_0_2X = 0UL,
    PBK_SPEED_0_5X = 1UL,
    PBK_SPEED_1X = 2UL,
    PBK_SPEED_2X = 3UL,
    PBK_SPEED_4X = 4UL,
    PBK_SPEED_8X = 5UL,
    PBK_SPEED_16X = 6UL,
    PBK_SPEED_32X = 7UL,
    PBK_SPEED_INCREASE = 8UL,
    PBK_SPEED_DECREASE = 9UL,
    PBK_SPEED_INVALID = 99UL
}tPlaybackSpeed;
#define tPlaybackSpeed_format "i"
#define tPlaybackSpeed_init PBK_SPEED_1X

/**
 * Type declaration of an enum containing playback direction
 */
typedef enum
{
    PBK_FORWARD = 0UL,
    PBK_BACKWARD = 1UL
}tPEPlaybackDirection;
#define tPEPlaybackDirection_format "i"
#define tPEPlaybackDirection_init PBK_FORWARD
/**
 * Type declaration of an enum containing external (HMI) play back state
 */
typedef enum
{
    HMI_PBS_PLAYING = 0UL,  /**< MediaPlayer is currently playing a media file */
    HMI_PBS_PAUSED = 1UL,   /**< MediaPlayer has paused playback of a media file */
    HMI_PBS_STOPPED = 2UL,  /**< MediaPlayer has stopped playback */
    HMI_PBS_FFWD = 3UL,     /**< MediaPlayer is currently fast forwarding a media file */
    HMI_PBS_FREV = 4UL,     /**< MediaPlayer is currently fast rewinding a media file */
    HMI_PBS_UNDEFINED = 999
} tHMIPlaybackState;

/**
 * Type declaration of an enum describing the NowPlaying states
 */
typedef enum
{
    NP_NEW_TRACK = 0UL,         /**< The MediaPlayer has transitioned to a different track */
    NP_SAME_TRACK = 1UL,        /**< The MediaPlayer is in same track */
    NP_TRACK_START = 2UL,       /**< The MediaPlayer has reached the start of the track */
    NP_TRACK_COMPLETE = 3UL,    /**< The MediaPlayer has reached the end of the current track */
    NP_LIST_START = 4UL,        /**< The MediaPlayer has reached the start of the current playlist */
    NP_LIST_COMPLETE = 5UL,     /**< The MediaPlayer has reached the end of the current playlist */
    NP_NO_DEVICE = 6UL,         /**< The MediaPlayer could not find a supported device from which to play media */
    NP_INVALID = 7UL            /* No currently playing track in Mediaplayer */
} tNowPlayingState;

/**
 * Type declaration of an enum describing the playpoint format (percentage or absolute)
 */
typedef enum
{
    PPF_PERCENTAGE = 0UL,   /**< PlaypointPosition is expressed as a percentage of the TotalPlaytime */
    PPF_ABSOLUTE = 1UL      /**< PlaypointPosition is expressed as an absolute time value in milliseconds */
} tPlaypointFormat;

/**
 * Type declaration of an enum describing the reason for a list change
 */
typedef enum
{
    LCH_CONTENT_CHANGED = 0UL,  /**< The complete content of the list has changed */
    LCH_ITEMS_ADDED = 1UL,      /**< Items have been added to the list */
    LCH_ITEMS_REMOVED = 2UL,    /**< Items have been removed from the list */
    LCH_ITEMS_CHANGED = 3UL     /**< Items have been changed in the list */
} tListChangeType;
#define tListChangeType_format "i"
#define tListChangeType_init LCH_CONTENT_CHANGED

/**
 * Type declaration of an enum describing the source activity option
 */
typedef enum
{
    SA_OFF = 0UL,   /**< Off */
    SA_ON = 1UL,    /**< On */
    SA_PAUSE = 2UL  /**< Pause */
} tSourceActivity;

/**
 * Type declaration of an enum describing the media type
 */
typedef enum {
    MTY_UNKNOWN = 0x00,             /**< Type of media object is not know */
    MTY_MUSIC_FILE = 0x01,          /**< Media type is a music file */
    MTY_PODCAST = 0x02,             /**< Media type is a podcast */
    MTY_AUDIOBOOK = 0x04,           /**< Media type is an audiobook */
    MTY_VIDEO = 0x08,               /**< Media type is a video */
    MTY_PLAYLIST = 0x10,            /**< Media type is a playlist */
    MTY_PLAYLIST_INTERNAL = 0x20,   /**< Media type is an internal playlist */
    MTY_AUDIO_STREAM = 0x40,        /**< Media type is streaming audio */
    MTY_IMAGE = 0x80                /**< Media type is an image */
} tMediaType;
#define tMediaType_format "i"
#define tMediaType_init MTY_UNKNOWN

/**
 * Type declaration of a media content (bitfield of media types)
 */
typedef tUInt tMediaContent;

/**
 * Type declaration of an enum containing the device type
 */
typedef enum {
    DTY_UNKNOWN = 0UL,      /**< Unknown device type */
    DTY_USB = 1UL,          /**< USB device */
    DTY_IPOD = 2UL,         /**< iPod */
    DTY_SD = 3UL,           /**< SD card */
    DTY_BLUETOOTH = 4UL,    /**< Bluetooth device. Not supported in GM implementations */
    DTY_IPHONE = 5UL,       /**< iPhone */
    DTY_MTP = 6UL,          /**< MTP device. Not supported in GM implementations */
    DTY_MSZUNE = 7UL,       /**< MSZune device. Not supported in GM implementations */
    DTY_CDROM = 8UL,        /**< CDROM. Not supported in GM implementations (Roadmap 13010) */
    DTY_MYMEDIA = 10UL,     /**< MyMedia (all connected devices) */
    DTY_FLASH = 11UL,       /**< Internal flash (Roadmap 13005) */
    DTY_CDDA = 12UL,        /**< CD DA. Not supported in GM implementations (Roadmap 15002) */
    DTY_CS = 13UL,          /**< Content Sharing (Roadmap 15016) */
    DTY_DVD_DRIVE =14UL,
    DTY_MUSICBOX = 15UL,    /**< MusicBox (Roadmap 16003) */
    DTY_UNSUPPORTED     = 99UL, /**< Unsupported device type like a USB mouse (Roadmap 13023) */
    DTY_NOT_USED_BY_GMP = 100UL /**< device is not added to DataBase by DBManager nor notified to HMI (NCG3D-13379) */
} tDeviceType;
#define tDeviceType_format "i"
#define tDeviceType_init DTY_UNKNOWN
tBoolean IsMassStorageDevice(const tDeviceType deviceType);
tBoolean IsAppleDevice(const tDeviceType deviceType);
tBoolean IsSRStorageDevice(const tDeviceType deviceType);

/**
 * Type declaration of an enum containing the connection type
 */
typedef enum {
    DCT_UNKNOWN = 0UL,      /**< Unknown connection type */
    DCT_USB = 1UL,          /**< USB connection */
    DCT_BLUETOOTH = 2UL,    /**< Bluetooth connection */
    DCT_CDROM = 3UL,        /**< CDROM & CDDA*/
    DCT_FLASH = 4UL,        /**< FLASH */
    DCT_WIFI = 5UL          /**< Carplay Wifi*/
} tConnectionType;
#define tConnectionType_format "i"
#define tConnectionType_init DCT_UNKNOWN

/**
 * Type declaration of an enum containing the tag transfer status
 */
typedef enum {
    TAG_TRANSFER_SUCCESS = 0UL,     /**< All tags have been transferred successfully.
                                    In this case, the ensuing UntransferredTagStream is empty. */
    TAG_TRANSFER_COMM_ERROR = 1UL,  /**< There was a communication error with the device and one or
                                    more of the tags could not be transferred. In this case, the
                                    Untransferred Tag Stream contains the tag information of the
                                    tags that could not be transferred. */
    TAG_TRANSFER_DEVICE_FULL = 2UL, /**< The device is full and cannot accept any more tags.
                                    In this case, one or more of the tags could not be transferred,
                                    and the Untransferred Tag Stream contains the tag information
                                    of the tag(s) that could not be transferred. */
    TAG_TRANSFER_NOT_SUPPORTED = 3UL    /**< The Tag Transfer feature is not supported for the current connected device.*/
} tTagTransferStatus;
#define tTagTransferStatus_format "i"
#define tTagTransferStatus_init TAG_TRANSFER_SUCCESS

/**
 * Type declaration of an enum for logical AV channel
 */
typedef enum
{
    LC_NONE = 0UL,                      /**<   */
    LC_MAIN_AUDIO = 1UL,                /**<   */
    LC_LVM = 2UL,                       /**<   */
    LC_ASYNCH_MSG = 3UL,                /**<   */
    LC_INCOMING_RING_TONE = 4UL,        /**<   */
    LC_PHONE = 5UL,                     /**<   */
    LC_SYNCH_MSG = 6UL,                 /**<   */
    LC_SPEECH_REC = 7UL,                /**<   */
    LC_EMER_PHONE = 8UL,                /**<   */
    LC_SHORT_MIX_ALERT = 9UL,           /**<   */
    LC_MIX_ALERT_MSG = 10UL,            /**<   */
    LC_ALERT_TONE = 11UL,               /**<   */
    LC_ADVISOR_PHONE = 12UL,            /**<   */
    LC_AUDIO_CUE = 13UL,                /**<   */
    LC_TRAFFIC = 14UL,                  /**<   */
    LC_EMER_MSG = 15UL,                 /**< This channel is used for PTY=31 traffic */
    LC_SOFT_ALERT_TONE = 16UL,          /**<   */
    LC_REAR_AUDIO = 17UL,               /**<   */
    LC_REAR_ASYNCH_MSG = 18UL,          /**<   */
    LC_REAR_INCOMING_RING_TONE = 19UL,  /**<   */
    LC_REAR_PHONE = 20UL,               /**<   */
    LC_REAR_SYNCH_MSG = 21UL,           /**<   */
    LC_REAR_SPEECH_REC = 22UL,          /**<   */
    LC_REAR_SHORT_MIX_ALERT = 23UL,     /**<   */
    LC_REAR_MIX_ALERT_MSG = 24UL,       /**<   */
    LC_REAR_ALERT_TONE = 25UL,          /**<   */
    LC_INTERNET_APP_AUDIO = 26UL,       /**<   */
    LC_REAR_SOFT_ALERT_TONE = 27UL,     /**<   */
    LC_REAR_INTERNET_APP_AUDIO = 28UL,  /**<   */
    LC_HEADPHONE_1 = 33UL,              /**<   */
    LC_HEADPHONE_2 = 34UL,              /**<   */
    LC_HEADPHONE_3 = 35UL,              /**<   */
    LC_HEADPHONE_4 = 36UL,              /**<   */
    LC_RSE_VIDEO_1 = 49UL,              /**<   */
    LC_RSE_VIDEO_2 = 50UL,              /**<   */
    LC_RSE_VIDEO_3 = 51UL,              /**<   */
    LC_RSE_VIDEO_4 = 52UL,              /**<   */
    LC_CLUSTER_VIDEO = 53UL,            /**<   */
    LC_CENTER_STACK_VIDEO = 54UL,       /**<   */
    LC_BT_AUDIO = 55UL       /**<   */
} tLogicalAVChannel;

/**
 * Type for the possible languages
 */
typedef enum {
    LNG_NA_ENGLISH = 0UL,               /**< North American English */
    LNG_GERMAN = 1UL,
    LNG_ITALIAN = 2UL,
    LNG_SWEDISH = 3UL,
    LNG_FRENCH = 4UL,                   /**< European French */
    LNG_SPANISH = 5UL,                  /**< European Spanish */
    LNG_DUTCH = 6UL,
    LNG_PORTUGUESE = 7UL,
    LNG_NORWEGIAN = 8UL,
    LNG_FINNISH = 9UL,
    LNG_DANISH = 10UL,
    LNG_GREEK = 11UL,
    LNG_JAPANESE = 12UL,
    LNG_ARABIC = 13UL,
    LNG_STANDARD_CHINESE = 14UL,
    LNG_POLISH = 15UL,
    LNG_TURKISH = 16UL,
    LNG_KOREAN = 17UL,
    LNG_TRADITIONAL_CHINESE = 18UL,
    LNG_UK_ENGLISH = 19UL,              /**< British English */
    LNG_HUNGARIAN = 20UL,
    LNG_CZECH = 21UL,
    LNG_SLOVAK = 22UL,
    LNG_RUSSIAN = 23UL,
    LNG_BRAZILIAN_PORTUGUESE = 24UL,
    LNG_THAI = 25UL,
    LNG_BULGARIAN = 26UL,
    LNG_ROMANIAN = 27UL,
    LNG_SLOVENIAN = 28UL,
    LNG_CROATIAN = 29UL,
    LNG_UKRAINIAN = 30UL,
    LNG_NA_FRENCH = 31UL,               /**< North American French */
    LNG_NA_SPANISH = 32UL,              /**< North American Spanish */
    LNG_CANTONESE = 33UL,
    LNG_HEBREW = 34UL,                  /**CMG3G-8461 VerticalKeyboard Extension */
    LNG_DEVANAGARI = 35UL               /**CMG3G-8461 VerticalKeyboard Extension */
} tLanguageType;

/**
 * Type declaration of an enum containing USB device connection status from VD_DeviceManager
 */
typedef enum {
    DM_CS_UNDEFINED = 0UL,              /**< If client connects but server has not yet values available from
                                        underlying driver layer this status could be communicated first. */
    DM_CS_WARNING = 1UL,                /**< VD_DeviceManager is taking expiration of a timer into account
                                        before announcing a state to connected clients. This state is thought
                                        to be communicated if timer has not been expired but current
                                        available states from underlying layer looks as if USB is not
                                        available anymore. Client should be aware to receive another message
                                        of DeviceConnectStatus informing about details after expiration of timer. */
    DM_CS_CONNECTED = 2UL,              /**< If user plugs USB device this status will be send. */
    DM_CS_REMOVED_BY_USR = 3UL,         /**< If user removes/unplugs USB device this status will be send. */
    DM_CS_UNAVAIL_BAT_LOWVOLT = 4UL,    /**< In case of low voltage USB device is not available anymore
                                        hence this status is used. */
    DM_CS_UNAVAIL_HW_MALFUNC     = 5UL,     /**< In case of voltage overrun, over current, short to ground
                                        at an USB port this status is send to clients. */
    DM_CS_UNAVAIL_CDROM_OVERTEMP = 6UL,      /**< if CDDrive has overtemperature this is notified by VD_DeviceManager
                                                                                      which is informed by FC_HeatControl. //Roadmap 13035_Overtemperature */
    DM_CS_UNAVAIL_HW_MALFUNC_PERMANENT=7UL  /**< In case of voltage overrun, over current, short to ground
                                        at an USB port for a long time this status is send to clients. */
} tDMConnectStatus;


/**
 * Type declaration of enum containing SLOT StATUS of CDDA/CDROM
 * if CD inserted into optical disc there is a gap in time if it is usable for playing
 * For this reason VD_DeviceManager notifies the slot status independently from
 * the deviceprpoperty list. In order to give the HMI notification of user action
 * 'disc inserted' immediately
 * (Roadmap: 15002_CDAudio_DetectionAndDiagnosis)
 */
typedef enum{
    DM_OPTDISC_INITIALISING                    =  0UL, //Optical disc and corresponding drivers are initialising
    DM_OPTDISC_INITIALISED                     =  1UL, //Optical disc and corresponding drivers are ready
    DM_OPTDISC_LASTMODE_EMTPY                  =  2UL, //Device has been switched off without disc in slot
    DM_OPTDISC_LASTMODE_INSERTED_CDDA          =  3UL, //Device has been switched off with CDDA in slot
    DM_OPTDISC_LASTMODE_INSERTED_CDROM         =  4UL, //Device has been switched off with CDROM in slot
    DM_OPTDISC_LASTMODE_INSERTED_CDERROR       =  5UL, //Device has been switched off with CDERROR
    DM_OPTDISC_INSERTING                       =  6UL, //Disc is currently inserting into slot.
    DM_OPTDISC_INSERTED_CDAUDIO                =  7UL, //Disc is fully inserted and recognized as CDDA (CDAUDIO)
    DM_OPTDISC_INSERTED_CDROM                  =  8UL, //Disc is fully inserted and recognized as CDROM (Data CD)
    DM_OPTDISC_INSERTED_CDERROR                =  9UL, //Disc is fully inserted with error
    DM_OPTDISC_INSERTED_AUTOMATIC_CDAUDIO      = 10UL, //After disc has been ejected it has been automatic inserted and recognized as CDDA (CDAUDIO)
    DM_OPTDISC_INSERTED_AUTOMATIC_CDROM        = 11UL, //After disc has been ejected it has been automatic inserted and recognized as CDROM (data CD)
    DM_OPTDISC_INSERTED_AUTOMATIC_CDERROR      = 12UL, //After disc has been ejected it has been automatic inserted but error occured
    DM_OPTDISC_EJECTING                        = 13UL, //Disc is currently ejecting
    DM_OPTDISC_EJECTED_READY_TO_REMOVE         = 14UL, //Disc is fully ejected and ready to be removed by the user (in configured it'll be automatically inserted if it is not removed within configured timeout)
    DM_OPTDISC_EJECTED_EMPTY                   = 15UL, //Ejected disc has been removed by user - slot is empty
    DM_OPTDISC_UNDEFINED_STATE                 = 0xFF  //state is undefined yet (initialised value)
}tDMOpticalDiscSlotState;

/*
 * HMI sends eject or insert to Mediaplayer - parameter is used to differentiate this
 *  (Roadmap: 15002_CDAudio_DetectionAndDiagnosis)
 */
typedef enum{
    DM_OPTDISC_CMD_EJECT                       = 0UL, //used to eject optical disc
    DM_OPTDISC_CMD_INSERT                      = 1UL  //used to force insertion of optical disc
}tDMOpticalDiscEjectCmd;

/*
 * return value for eject command
 *  (Roadmap: 15002_CDAudio_DetectionAndDiagnosis)
 */

typedef enum
{
    ACCEPTED,
    NOTACCEPTED,
    DEVICENOTREADY
}tDMOpticalDiscEjectMethodResult;


/*****************
 * MP Enumerations
 *****************/

/**
 * Type for the engine type
 */
typedef enum {
    ET_UNDEFINED = 0UL,
    ET_GASOLINE,
    ET_CNG,
    ET_DIESEL,
    ET_ELECTRIC
} tEngineType;

/**
 * Type for outside temperature
 */
typedef int tOutsideTemperature;
#define INVALID_OUTSIDE_TEMPERATURE -10000

/**
 * Type for the MIME type of an album art picture
 */
typedef enum
{
    MMT_PNG = 0UL,
    MMT_BMP,
    MMT_GIF,
    MMT_JPG
} tMimeType;

/**
 * Type declaration of an enum containing HMILayerSyncView status of PlayerEngine
 */
typedef enum
{
    VS_INVALID = 0UL,
    VS_ACTIVE,
    VS_STOPPED,
    VS_ABOUT_TO_CHANGE
} tViewStatus;

/**
 * Type declaration of an enum describing the next action after reaching the end of a list
 */
typedef enum {
    PC_NO_REPEAT = 0UL,           /**< do not repeat */
    PC_REPEAT_CURRENT_LEVEL, /**< repeat only the current level, no step up */
    PC_STEPUP_ONE_LEVEL
} tPlayContinuation;

/**
 * Type declaration of an enum listing the component identifier for all internal components
 */
typedef enum
{
    COMPONENT_ID_INDEXER = 0UL,
    COMPONENT_ID_USB_CONTROL = 1UL,
    COMPONENT_ID_IPOD_CONTROL = 2UL,
    COMPONENT_ID_MTP_CONTROL = 3UL,
    COMPONENT_ID_LIST_CONTROL = 4UL,
    COMPONENT_ID_CUSTOM_CONTROL = 5UL,
    COMPONENT_ID_PLAYER_MANAGER = 6UL,
    COMPONENT_ID_DEVICE_DISPATCHER = 7UL,

    COMPONENT_ID_DISPATCHER = 8UL,
    COMPONENT_ID_DB_MANAGER = 9UL,
    COMPONENT_ID_DATA_PROVIDER = 10UL,
    COMPONENT_ID_IPC_PROVIDER = 11UL,
    COMPONENT_ID_CONFIGURATION = 12UL,
    COMPONENT_ID_OUTPUTWRAPPER = 13UL,
    COMPONENT_ID_CUSTOMCONTROL = 14L,
    COMPONENT_ID_THREAD_FACTORY = 15UL,
    COMPONENT_ID_BT_CONTROL = 16UL,

    COMPONENT_ID_PICTURE_MANAGER = 17UL,
    COMPONENT_ID_MEDIA_ENGINE = 18UL,
    COMPONENT_ID_ROOT_DAEMON = 19UL,
    COMPONENT_ID_CDDA_CONTROL = 20UL,
    COMPONENT_ID_CS_CONTROL = 21UL,
    COMPONENT_ID_IPOD_CONTROL_HELPER = 22UL,
    COMPONENT_ID_ALBUMART_INDEXER = 23UL,
    COMPONENT_ID_DVD_CONTROL = 24UL,
    COMPONENT_ID_CD_RIPPER = 25UL,
#ifdef USE_PROTOTYPE_DIRECT_ADD_ID_SOURCEMANAGER
    COMPONENT_ID_SOURCEMANAGER = 26UL,
#endif
    COMPONENT_ID_UNKNOWN
} tComponentID;
#define tComponentID_format "i"
#define tComponentID_init COMPONENT_ID_UNKNOWN

/**
 * Local SPM component init reason definitions
 */
typedef enum
{
    IR_BOOT,
    IR_UNDERVOLTAGE
} tInitReason;

/**
 * Local SPM component state definitions
 */
typedef enum
{
    COMPONENT_STATE_CREATE = 0UL,
    COMPONENT_STATE_INIT   = 1UL ,
    COMPONENT_STATE_RUNNING = 2UL,
    COMPONENT_STATE_STOP = 3UL,
    COMPONENT_STATE_DONE = 4UL,
    COMPONENT_STATE_UNKNOWN
} tComponentState;

/**
 * Local SPM component state definitions
 */
typedef enum
{
    COMPONENT_GROUP_ALL = 0UL,
    COMPONENT_GROUP_SM   = 1UL ,
    COMPONENT_GROUP_NON_SM = 2UL,
} tComponentGroup;

/**
 * Local SPM state definitions
 */
typedef enum
{
    SPM_STATE_OFF = 0UL,
    SPM_STATE_NORMAL   = 1UL ,
    SPM_STATE_UNDERVOLTAGE = 2UL
} tSPMState;

/**
 * Type declaration of an enum for an element position in a list
 */
typedef enum {
    PIL_IN_THE_MIDDLE = 0UL,
    PIL_FIRST_ELEMENT,
    PIL_LAST_ELEMENT
} tPositionInList;

/**
 * Type declaration of an enum containing the playback state of PlayerEngine
 */
typedef enum
{
    PE_PBS_LOADINGSTATE = 0UL,
    PE_PBS_STOPPEDSTATE,
    PE_PBS_PLAYINGSTATE,
    PE_PBS_BUFFERINGSTATE,
    PE_PBS_PAUSEDSTATE,
    PE_PBS_ERRORSTATE,
    PE_PBS_FASTFORWARDSTATE,
    PE_PBS_FASTREVERSESTATE,
    PE_PBS_ABOUTTOFINISHSTATE,
    PE_PBS_FULLYBUFFEREDSTATE,
    PE_PBS_FINISHEDSTATE,
    PE_PBS_SEEKTOSTATE = 999,
    PE_PBS_LOADINGSTATEFORREMOTEACTIVITY,
    PE_PBS_INVALIDSTATEFORREMOTEACTIVITY,
    PE_PBS_UNKNOWN
} tPEPlaybackState;
#define tPEPlaybackState_format "i"
#define tPEPlaybackState_init PE_PBS_STOPPEDSTATE

/**
 * Type declaration of an enum describing the error codes of the PlayerEngine
 */
typedef enum
{
    PE_EC_FILE_NOT_FOUND = 0UL,
    PE_EC_DEVICE_NOT_PRESENT
} tPEErrorCode;

/**
 * Type declaration of an enum describing the fingerprint states
 */
typedef enum
{
    FPS_OK = 0UL,
    FPS_ERROR,
    FPS_NOT_AVAIL,
    FPS_OK_DELTA,   //Roadmap 13006 IAP2
    FPS_OK_SKIP_INDEXING
} tFingerprintStatus;
#define tFingerprintStatus_format "i"
#define tFingerprintStatus_init FPS_NOT_AVAIL

/**
 * Type declaration of an enum describing the metadata collecting states
 */
typedef enum
{
    MDS_SUCCESS = 0UL,
    MDS_FINISHED,
    MDS_FINISHED_RECURSIVE_EMERGENCY_BREAK,
    MDS_FILE_SKIP,
    MDS_FILE_ERROR,
    MDS_DEVICE_ERROR,
    MDS_REMOVED,    //Roadmap 13006 IAP2
    MDS_FLUSH,
    MDS_ERROR_RESTART
} tMetadataStatus;
#define tMetadataStatus_format "i"
#define tMetadataStatus_init MDS_SUCCESS

/**
 * Type declaration of an enum describing the reason for a ListIsEmpty notification
 */
typedef enum
{
    NOR_NEW_LIST = 0UL,
    NOR_END_OF_LIST,
    NOR_START_OF_LIST,
    NOR_LIST_IS_NOT_PLAYABLE,
    NOR_NO_OBJECT,
    NOR_DEVICE_REMOVED,
    NOR_VIDEO_SOURCE
} tNoObjectReason;
#define tNoObjectReason_format "i"
#define tNoObjectReason_init NOR_NO_OBJECT

/**
 * Type declaration for device connection status
 */
typedef enum
{
    CS_CONNECTED = 0UL,
    CS_ATTACHED,
    CS_DISCONNECTED,
    CS_HW_MALFUNCTION,
    CS_UNDERVOLTAGE,    //Roadmap 13024
    CS_UNSUPPORTED,     //Roadmap 13023
    CS_OVERTEMP,         //Roadmap 13025_CDDrive OverTemperature
    CS_ON_HOLD,         //Roadmap 13031_BT_DeviceSourceAllocation
    CS_AUTO_REINSERTED,
    CS_UNDEFINED        //Multi Partition Support
} tConnectionState;
#define tConnectionState_format "i"
#define tConnectionState_init CS_DISCONNECTED

/**
 * Type declaration of an enum containing the reason for the device announcment
 */
typedef enum
{
    CR_NEW_MEDIA,   /** media was attached during runtime */
    CR_SAME_MEDIA,  /** media was already connected in the last run or was in an error state */
    CR_ERROR        /** media is not notified as Disconnected by DeviceManager but couldn't play the device + mountpoint not accessible */
} tConnectionReason;

/**
 * Type declaration of an enum containing the indexing mode
 */
typedef enum
{
    IM_FULL_SCAN = 0UL,     /**< A full indexing scan is necessary for this device */
    IM_CONTINUE_SCAN        /**< The indexing scan will be continue for this device */
} tIndexingMode;

/**
 * Type declaration of an enum containing the indexing order
 */
typedef enum
{
    IO_AS_CONNECTED = 0UL,     /**< Order as connected (first come first serve) */
    IO_AS_CONNECTED_REVERSE    /**< Reverse order as connected (last come first serve) */
} tIndexingOrder;

/**
 * Type declaration of an enum containing the scan method
 */
typedef enum
{
    MS_DEEP = 0UL,  /**< Deep scan (recursive scan) */
    MS_FLAT         /**< Flat scan (scan whole directory first and then subdirectories) */
} tScanMethod;

/**
 * Type declaration of an enum containing the state of trigger switching
 */
typedef enum
{
    TS_OFF = 0UL,   /**< Deregister a DB trigger */
    TS_ON           /**< Register a DB trigger */
} tTriggerState;

/**
 * Type declaration of an enum containing all DB trigger types used in media player
 */
typedef enum
{
    TT_ALL = 0UL,                    /**< All component relevant triggers */
    TT_DB_DEVICE_INSERTED,           /**< New entry in device table (Devices) */
    TT_DB_DEVICE_CHANGED,            /**< One of the attributes of the device table has changed */
    TT_DB_DEVICE_ATTACHED,           /**< Attribute Devices.ConnectionState changed to ATTACHED */
    TT_DB_DEVICE_CONNECTED,          /**< Attribute Devices.ConnectionState changed to CONNECTED */
    TT_DB_DEVICE_REMOVED,            /**< Attribute Devices.ConnectionState changed to DISCONNECTED */
    TT_DB_DEVICE_ACTIVATED,          /**< Attribute Devices.Active changed to true */
    TT_DB_DEVICE_INDEXING_STATE,     /**< Attribute Devices.IndexingState changed */
    TT_DB_DEVICE_IDS_COMPLETE,       /**< Attribute Devices.IndexingState changed to COMPLETE */
    TT_DB_DEVICE_INDEXING_PROGRESS,  /**< Attribute Devices.IndexingProgress changed */
    TT_DB_DEVICE_REPEAT_MODE,        /**< Attribute Devices.RepeatMode changed */
    TT_DB_DEVICE_NUMBER_OF_FILES,    /**< Attribute Devices.NumberOfAudioFiles changed */
    TT_DB_FAVORITE_INSERTED,         /**< New entry in favorite table (Favorites) */
    TT_DB_FAVORITE_CHANGED,          /**< One of the attributes of the favorite table has changed */
    TT_DB_FAVORITE_DELETED,          /**< An entry in favorite table has deleted */
    TT_DB_DEVICE_SAME,                /**< Same device was inserted */
    TT_DB_DEVICE_NEW,                 /**< New device was inserted */
    TT_DB_DEVICE_VERSION_UPDATED
} tTriggerType;

/**
 * Type declaration of an enum describing the iPod database hierarchy
 */
typedef enum
{
    IPOD_DB_HIERARCHY_AUDIO = 1UL,    /**< audio selection */
    IPOD_DB_HIERARCHY_VIDEO           /**< video selection */
} tIPODDBHierarchy;

/**
 * Type declaration of an enum containing video and audio codecs with restrictions
 */
typedef enum
{
    CDC_OTHER = 0UL,
    CDC_MPEG2 = 1UL,
    CDC_MPEG4 = 2UL,
    CDC_DIVX = 3UL,
    CDC_XVID = 4UL,
    CDC_AVC = 5UL,
    CDC_WMV7 = 6UL,
    CDC_WMV8 = 7UL,
    CDC_WMV9 = 8UL,
    CDC_VP6 = 9UL,
    CDC_VP8 = 10UL,
    CDC_SORENSON = 11UL,
    CDC_H263 = 12UL,
    CDC_WMA = 13UL
} tCodec;

/**
 * Type declaration of an enum containing the streaming mode
 * Roadmap 13008
 */
typedef enum
{
    SM_OFF = 0UL,   /**< streaming mode is off */
    SM_ON           /**< streaming mode is on */
} tStreamingMode;

/**
 * Type declaration of an enum containing the play state of PEngine
 * Roadmap 13010
 */
//typedef enum : tS64 //lint !e1505
//// new c++0x standard
//{
//    PE_PS_NOP = 0,
//    PE_PS_STOP,
//    PE_PS_PAUSE,
//    PE_PS_PLAY,
//    PE_PS_SHUT,
//    PE_PS_SET
//} tPEPlaystate;
//#define tPEPlaystate_format "l"
//#define tPEPlaystate_init PE_PS_STOP

///**
// * Type declaration of an enum containing the reason for a PEngine interaction
// * Roadmap 13010
// */
//typedef enum : tS64 //lint !e1505 new c++0x standard
//{
//    PE_RS_NONE               =  1024, // default;
//    PE_RS_ACK                =   512, // acknowledge;
//    PE_RS_CODECS             =   256, // scan url for codecs;
//    PE_RS_IDLE               =   128, // out: low  cpu load
//    PE_RS_BUSY               =    64, // out: high "   "
//    PE_RS_EMPTY              =    32, // out: audible playback of buffered files ended (finished)
//    PE_RS_END_OF_STREAM      =    16, // out: end of stream (playback of current file ended; send next_handle);
//    PE_RS_NEW_TRACK          =     8, // out: request next url for buffering
//    PE_RS_END_OF_FILE        =     4, // out: end of file (internal use)
//    PE_RS_FLUSH              =     2, // in:  flush buffered files when current file is finished;
//    PE_RS_BUFFER             =     1, // in:  buffer given url
//    PE_RS_OK                 =     0, // okay
//    PE_RS_ERROR              =    -1, // unspecified failure
//    PE_RS_ACTION_ERROR       =    -2, // statechange failure (e.g. stop -> pause)
//    PE_RS_URL_ERROR          =    -4, // url not available (e.g. file does not exist)
//    PE_RS_FORMAT_ERROR       =    -8, // incorrect file format (not playable)
//    PE_RS_DRM_ERROR          =   -16, // drm protected file (not released)
//    PE_RS_DEVICE_ERROR       =   -32, // device unavailable (write error)
//    PE_RS_READ_ERROR         =   -64, // read error (e.g. damaged block)
//    PE_RS_FATAL_READ_ERROR   =  -128, // gst streaming error (former STREAM_ERROR)
//    PE_RS_UNDERVOLTAGE_ERROR =  -256, // device in undervoltage (read error)
//    PE_RS_ARGUMENT_ERROR     =  -512, // wrong argument;
//    PE_RS_CODEC_ERROR        = -1024, // codec failure;
//    PE_RS_PIPELINE_ERROR     = -2048, // pipeline failure;
//    PE_RS_FORK_ERROR         = -4096, // server process fork failure;
//    PE_RS_RESTART_ERROR      = -8192  // mediaengine restart after kill or reset;
//} tPEReason;
//#define tPEReason_format "l"
//#define tPEReason_init PE_RS_OK

/**
 * Type declaration of an enum containing the reason for a PEngine interaction
 * Roadmap 13010
 */
typedef enum : tS64 //lint !e1505 new c++0x standard
{
    PE_BTF_NONE = -1,
    PE_BTF_SBC  =  0,
    PE_BTF_MP3  =  1,
    PE_BTF_AAC  =  2
} tPEBTFormat;
#define tPEBTFormat_format "l"
#define tPEBTFormat_init PE_BTF_NONE

/**
 * Type declaration for a device name
 */
typedef char tStateName[32];
#define tStateName_format "t"

/**
 * Type declaration of an enum containing the activity status of the PEngine
 * Roadmap 13010
 */
typedef enum : tS64 //lint -e1505 new c++0x standard
{
    PE_ACS_IDLE = 0,        /* idle */
    PE_ACS_BUSY = 1         /* busy */
} tPEActivityStatus;
#define tPEActivityStatus_format "l"
#define tPEActivityStatus_init PE_ACS_IDLE

/**
 * Type declaration of an enum describing the file error handling
 * Roadmap 13010
 */
typedef enum {
    FEH_NO_ACTION = 0UL,        /**< no action */
    FEH_MARK_OBJECT,            /**< mark media object in DB as not playable */
    FEH_MARK_OBJECT_PERMANENT,  /**< mark media object in DB as not playable without clearance by disconnect */
    FEH_DELETE_OBJECT           /**< delete media object from DB */
} tFileErrorHandling;

/**
 * Type declaration of an enum describing the file error handling
 * Roadmap 13010
 */
typedef enum {
    SC_NO = 0UL,    /**< no success */
    SC_YES          /**< success */
} tSuccess;
#define tSuccess_format "i"
#define tSuccess_init SC_NO

/**
 * Type declaration of an enum describing the file error handling
 * Roadmap 13023
 */
typedef enum {
    FSTY_UNKNOWN = 0UL, /**< Unknown file system type */
    FSTY_FAT,
    FSTY_NTFS,
    FSTY_EXT4,
    FSTY_HFS,
    FSTY_EXFAT,
    FSTY_ISO9660,
    FSTY_UDF,
    FSTY_DISC_NONE,
    FSTY_DISC_JUDGING,
    FSTY_DISC_AUDIO_CD,
    FSTY_DISC_VCD_VER_1,
    FSTY_DISC_VCD_VER_2_PBC_OFF,
    FSTY_DISC_SUPER_VCD_PBC_OFF,
    FSTY_DISC_VCD_VER_2_PBC_ON,
    FSTY_DISC_SUPER_VCD_PBC_ON,
    FSTY_DISC_DVD_VIDEO,
    FSTY_DISC_DVD_AUDIO,
    FSTY_DISC_DVD_VR,
    FSTY_DISC_DATA
} tFileSystemType;

/**
 * Type declaration for device status
 * Roadmap 13023
 */
typedef enum
{
    DS_NONE = 0UL,              /**< No device state */
    DS_INITIALIZING,            /**< Device initializing is ongoing */
    DS_OK,                      /**< Device is supported */
    DS_UNSUPPORTED,             /**< Device type is not supported */
    DS_UNSUPPORTED_FILESYSTEM,  /**< File system on device is not supported */
    DS_UNSUPPORTED_PARTITION,   /**< Partition on device is not supported */
    DS_COMMUNICATION_ERROR      /**< Communication error to the device */
} tDeviceState;
#define tDeviceState_format "i"
#define tDeviceState_init DS_NONE

/**
 * Type declaration of an enum describing the audio channel format
 * Roadmap 13001
 */
typedef enum
{
    ACF_UNKNOWN = 0UL,   /**< Unknown format */
    ACF_MONO,            /**< Mono */
    ACF_STEREO           /**< Stereo */
} tAudioChannelFormat;
#define tAudioChannelFormat_format "i"
#define tAudioChannelFormat_init ACF_UNKNOWN

/**
 * Type declaration of an enum describing the source allocation state
 * Roadmap 13024
 */
typedef enum
{
    ALS_DEALLOCATED = 0UL,  /**< Deallocated */
    ALS_ALLOCATED = 1UL,    /**< Allocated but SourceActivity is OFF */
    ALS_ACTIVE = 2UL        /**< Allocated and SourceActivity is ON */
} tAllocateState;

/**
 * Type declaration of an enum describing if a file is playable
 * Roadmap 13012
 */
typedef enum
{
    FNP_PLAYABLE = 0UL,          /**< File is playable */
    FNP_DRM_PROTECTED = 1UL,     /**< File is DRM protected */
    FNP_FORMAT_ERROR = 2UL,      /**< File is not playable (format error) */
    FNP_READ_ERROR = 3UL,        /**< File is not playable (read error) */
    FNP_NOT_PLAYABLE = 4UL,      /**< File is not playable (general error) */
    FNP_PLAY_RESTRICTION  = 5UL  /**< File is restricted to play */
} tNotPlayable;

/**
 * Type declaration of an enum describing the config table
 */
typedef enum
{
    CT_TEST = 0UL,              /**< Test config table */
    CT_DEVICE_ACTIVATION = 1UL, /**< Device activation config table */
    CT_LIST_SELECTION = 2UL     /**< List selection config table */
} tConfigTable;

/**
 * Type declaration of an enum describing the config table precondition (entry trigger)
 */
typedef enum
{
    CTP_NONE = 0UL,             /**< No precondition */
    CTP_TEST1 = 1UL,            /**< Test precondition 1 */
    CTP_TEST2 = 2UL,            /**< Test precondition 2 */
    CTP_ADD_DEVICE = 3UL,       /**< Add device */
    CTP_ACTIVATE_DEVICE = 4UL,  /**< Activate device */
    CTP_REMOVE_DEVICE = 5UL,    /**< Remove device */
    CTP_TIMER_EXPIRED = 6UL,    /**< Timer expired */
    CTP_USB = 7UL,              /**< Device type USB */
    CTP_MTP = 8UL,              /**< Device type MTP */
    CTP_IPOD = 9UL,             /**< Device type iPod */
    CTP_MYMEDIA = 10UL,         /**< Device type MyMedia */
    CTP_BT = 11UL,              /**< Device type Bluetooth */
    CTP_CDDA = 12UL,            /**< Device type CDDA */
    CTP_CS = 13UL               /**< Device type Content Sharing */
} tConfigTablePrecondition;

/**
 * Type declaration of an enum describing the config table criteria
 */
typedef enum
{
    CTC_NONE = 0UL,                             /**< No criteria */
    CTC_TEST1 = 1UL,                            /**< Test criteria 1 */
    CTC_TEST2 = 2UL,                            /**< Test criteria 2 */
    CTC_TEST3 = 3UL,                            /**< Test criteria 3 */
    CTC_STARTUP = 4UL,                          /**< Startup phase */
    CTC_LAST_PLAYED_DEVICE = 5UL,               /**< Last played device */
    CTC_ANY_DEVICE_ACTIVE = 6UL,                /**< Any device active */
    CTC_ANY_DEVICE_CONNECTED = 7UL,             /**< Any device connected */
    CTC_AUTOPLAY_DEVICE = 8UL,                  /**< Autoplay device */
    CTC_SAME_DEVICE_TYPE_LIKE_BEFORE = 9UL,     /**< Same device type like before */
    CTC_STREAMING_ACTIVE = 10UL,                /**< Streaming mode active for device type */
    CTC_LAST_MODE_LIST_AVAILABLE = 11UL,        /**< Last mode list available for device */
    CTC_ANY_DB_CONTENT_AVAILABLE = 12UL,        /**< Any DB content available for device */
    CTC_IPOD_CURRENT_SELECTION_CHANGED = 13UL,  /**< Current selection on iPod is changed from MP */
    CTC_HW_MALFUNC_IN_LAST_CONNECTION = 14L
} tConfigTableCriteria;

/**
 * Type declaration of an enum describing the config table action (exit result)
 */
typedef enum
{
    CTA_NONE = 0UL,                                 /**< No action */
    CTA_TEST1 = 1UL,                                /**< Test action 1 */
    CTA_TEST2 = 2UL,                                /**< Test action 2 */
    CTA_TEST3 = 3UL,                                /**< Test action 3 */
    CTA_TEST4 = 4UL,                                /**< Test action 4 */
    CTA_TEST5 = 5UL,                                /**< Test action 5 */
    CTA_TEST6 = 6UL,                                /**< Test action 6 */
    CTA_ACTIVATE_DEVICE_STOP_TIMER = 7UL,           /**< Activate device and stop startup timer */
    CTA_ACTIVATE_DEVICE = 8UL,                      /**< Activate device */
    CTA_DEACTIVATE_DEVICE = 9UL,                    /**< Deactivate device */
    CTA_ACTIVATE_FIRST_AVAILABLE_DEVICE = 10UL,     /**< Activate first available device of DB */
    CTA_PLAY_DB_LAST_MODE_LIST = 11UL,              /**< Create and play DB last mode list */
    CTA_PLAY_DB_ALL_SONGS_LIST = 12UL,              /**< Create and play DB all songs list */
    CTA_PLAY_DB_ALL_VIDEOS_LIST = 13UL,             /**< Create and play DB all video files list */
    CTA_PLAY_FOLDER_LIST = 14UL,                    /**< Create and play list of first folder with content or all songs list in case of iPod */
    CTA_PLAY_FOLDER_STREAMING_LIST = 15UL,          /**< Create and play list of first folder with content
                                                         or all songs list in case of iPod in streaming mode */
    CTA_PLAY_CURRENT_SELECTION_LIST = 16UL,         /**< Create and play current selection list */
    CTA_PLAY_CURRENT_SELECTION_STREAMING_LIST = 17UL /**< Create and play current selection list in streaming mode */
} tConfigTableAction;

/**
 * Type declaration of an enum containing the init device protocol type, e.g. iPod IAP1, IAP2
 * Roadmap 13006
 */
typedef enum
{
    IDP_NONE = 0UL,
    IDP_IAP1,
    IDP_IAP1BT,

    IDP_IAP2,
    IDP_IAP2BT,
    IDP_IAP2CARPLAY,            /**< host mode */
    IDP_IAP2NATIVE_TRANSPORT,    /**< host mode */
    IDP_IAP2CARPLAY_NATIVE_TRANSPORT,    /**< host mode */
    IDP_DVD_BACKEND,
    IDP_IAP2_OVER_WIRELESS_CARPLAY,
    IDP_IAP2CARLIFE_NATIVE_TRANSPORT
} tInitDeviceProtocol;
#define tInitDeviceProtocol_format "i"
#define tInitDeviceProtocol_init IDP_NONE
tBoolean IsProtocol_IAP1(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2HostMode(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2CarPlay(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2NativeTransport(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_BT(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2OverWirelessCarplay(const tInitDeviceProtocol protocol);
tBoolean IsProtocol_IAP2CarLife(const tInitDeviceProtocol protocol);

/**
 * Type declaration of an enum containing the app launch option (e.g. user alert popup)
 * Roadmap mySpin
 */
typedef enum
{
    ALO_LAUNCH_WITH_USER_ALERT = 0UL, /**< Launch with user alert (default) */
    ALO_LAUNCH_WITHOUT_USER_ALERT     /**< Launch without any alert, but device must be unlocked and on Home Screen or Music/Video app */
} tAppLaunchOption;
#define tAppLaunchOption_format "i"
#define tAppLaunchOption_init ALO_LAUNCH_WITH_USER_ALERT

/**
 * Type declaration of an enum containing the app info option (e.g. native transport)
 * Roadmap 15014_MySpinSupport_without_CarPlay
 */
typedef enum
{
    AIO_DEVICE_MODE_ONLY = 0UL,     /**< app used for device mode only (default) */
    AIO_HOSTMODE_ONLY,              /**< app used for host mode only */
    AIO_HOST_AND_DEVICE_MODE,       /**< app used for host and device mode */
    AIO_EA_NATIVE_TRANSPORT         /**< app required native transport support (host mode only) */
} tAppInfoOption;
#define tAppInfoOption_format "i"
#define tAppInfoOption_init AIO_DEVICE_MODE_ONLY

/**
 * Type declaration of an enum describing the current mute state
 * Roadmap 13018
 */
typedef enum
{
    MUS_DEMUTED = 0UL,       /**< Demuted */
    MUS_MUTED = 1UL,         /**< Muted */
    MUS_DEMUTE_RUNNING = 2UL /**< A demute request is running */
} tMuteState;
#define tMuteState_format "i"
#define tMuteState_init MUS_DEMUTED

/**
 * Type declaration of an enum describing the current mute state
 * Roadmap 13018
 */
typedef enum
{
    VOS_NORMAL = 0UL,       /**< Normal volume 0db */
    VOS_REDUCED = 1UL       /**< Reduced volume -12dB */
} tVolumeState;

/**
 * Type declaration of an enum describing the user data type
 * Roadmap 13034_SPIInterface
 */
typedef enum
{
    USRDATA_UNKNOWN = 0UL,
    USRDATA_GPS = 1UL,          /**< GPS NMEA sentences */
    USRDATA_DR = 2UL,               /**< dead reckoning sensor data */
    USRDATA_VEHICLESTATUS = 4UL,      /**< iPod vehicle status update */
    USRDATA_STARTAUDIO = 5UL,      /**< start iap2 deviceMode audio */
    USRDATA_STOPAUDIO = 6UL,      /**< stop iap2 deviceMode  */
    USRDATA_STOPMEDIALIBRARYUPDATES = 7UL,      /**< stop iap2 media library updates  */
    USRDATA_UPDATEAPPCONTROL = 8UL,   /**< call outputwrapper, update appControl property  */
    USRDATA_FINGERPRINTTIMEOUT = 9UL,  /**< Timeout for CBIAP2_MediaLibraryUpdate, no update received  */
    USRDATA_TEST = 10UL,            /**< test function only */
    USRDATA_ROLE_SWITCH_REQUIRED_ANSWER = 11UL, /**< user selection on RoleSwitchRequiredTimeOutMS == 0 */
    USRDATA_APPCONTROL_CONNECT = 12UL,
    USRDATA_APPCONTROL_CLOSE = 13UL,
    USRDATA_START_EA_NATIVE_TRANSPORT = 14UL,
    USRDATA_STOP_EA_NATIVE_TRANSPORT = 15UL,
    USRDATA_APP_DATA_RECEIVED = 16UL,
    USRDATA_PLAYBACK_STATUS_UPDATES = 17UL,
    USRDATA_PLAYBACK_MODE_UPDATES = 18UL,
    USRDATA_ACC_WIFI_CREDENTIALS = 19UL,
    USRDATA_START_IAP2_CARPLAY_WIRELESS = 20UL,
    USRDATA_OOBBT_PAIRING_ACCESSORY_INFORMATION = 21UL,
    USRDATA_OOBBT_PAIRING_COMPLETION_INFORMATION = 22UL,
    USRDATA_DISKERROR_UPDATES = 30UL,
    USRDATA_MECHANICAL_STATUS_UPDATES = 31UL,
    USRDATA_DVD_SKIP_UPDATES = 32UL,
    USRDATA_DVD_DIRECTSELECT_UPDATES = 33UL,
    USRDATA_DVD_ANGLEINFO_UPDATES = 34UL,
    USRDATA_DVD_SUBTITLEINFO_UPDATES = 35UL,
    USRDATA_DVD_AUDIOCHANNEL_UPDATES = 36UL,
    USRDATA_DVD_DISPLAYMODE_UPDATES = 37UL,
    USRDATA_DVD_PLAYSTATUSSETTING_UPDATES = 38UL,
    USRDATA_DVD_DRC_SETTINGS_UPDATES = 39UL,
    USRDATA_DVD_AUDIO_OUTPUT_UPDATES = 40UL,
    USRDATA_DVD_ANGLE_MARK_UPDATES = 41UL,
    USRDATA_DVD_PLAYER_LANGUAGE_UPDATES = 42UL,
    USRDATA_DVD_DISC_DETERMINED = 43UL,
    USRDATA_DVD_KEY_RESPONSE_UPDATES = 44UL,
    USRDATA_DVD_RESUME_INFO_UPDATE = 45UL,
    USRDATA_SWAPCALLS = 46UL,
    USRDATA_START_RG = 47UL,
    USRDATA_STOP_RG = 48UL,
    USRDATA_DVD_VIDEO_SOURCE_STATUS = 49UL ,
    USRDATA_DVD_EJECT = 50UL,
    USRDATA_ACC_WIFI_CREDENTIALS_RESULT = 51UL,
    USRDATA_DVD_TEMPERATURE_REQUEST = 52UL,
    USRDATA_DVD_TEMPERATURE_RESPONSE = 53UL,
    USRDATA_DVD_MENUPLAYBACK_MODE = 54UL,
    USRDATA_DRIVEMODE_UPDATES = 55UL,
    USRDATA_LANGUAGEUPDATES = 56UL,
    USRDATA_DVD_TRACK_CHANGE = 57UL,
    USRDATA_UPDATE_READY_TO_PLAY = 58UL,
    USRDATA_INITIATECALL = 59UL,
    USRDATA_ACCEPTCALL = 60UL,
    USRDATA_ENDCALL = 61UL,
    USRDATA_APPLE_HID_COMMAND = 62UL
} tUserDataType;
#define tUserDataType_format "i"
#define tUserDataType_init USRDATA_UNKNOWN

typedef enum
{
    APPCONTROL_UPDATE_NONE = 0UL,  // default
    APPCONTROL_UPDATE_BTPROFILES,
    APPCONTROL_UPDATE_DIPO_NOWPLAYING,
    APPCONTROL_UPDATE_DIPO_PLAYTIME,
    APPCONTROL_UPDATE_DIPO_PLAYBACKSTATUS,
    APPCONTROL_UPDATE_DIPO_SHUFFLEMODE,
    APPCONTROL_UPDATE_DIPO_REPEATMODE,
    APPCONTROL_UPDATE_DIPO_CALLSTATE,
    APPCONTROL_UPDATE_DIPO_COMMUNICATIONS,
    APPCONTROL_UPDATE_DIPO_CALLDURATION,
    APPCONTROL_UPDATE_DIPO_LOCATIONINFO,
    APPCONTROL_UPDATE_DIPO_POWER,
    APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE,
    APPCONTROL_UPDATE_DIPO_ROUTE_GUIDANCE_MANEUVER,
    APPCONTROL_UPDATE_DIPO_GPRMCDATASTATUSVALUES_NOTIFICATION,
    APPCONTROL_UPDATE_DIPO_DEVICETIME
} tAppControlUpdateType;
#define tAppControlUpdateType_format "i"
#define tAppControlUpdateType_init APPCONTROL_UPDATE_NONE

/**
 * Type declaration of an enum describing the reason of a device disconnection
 * Roadmap 15003
 */
typedef enum
{
    DR_REMOVED = 0UL,                   /** < device is removed/unplugged from the target (permanently unavailable) */
    DR_TEMPERATURE = 1UL,               /** < device is in an over temperature situation (temporary unavailable) */
    DR_HW_MALFUNC = 2UL,                /** < device is in hardware malfunction situation like low voltage or over current (temporary unavailable) */
    DR_INTERNALDISCONNECT = 3UL,        /** < device is in an internal disconnect situation (temporary unavailable) */
    DR_HW_MALFUNC_PERMANENT = 4UL,      /** <device is in hardware malfunction permanent situation like low voltage or over current (permanently unavailable) */
    DR_AVP_DISCONNECTED = 5UL,          /** <device is in AVP disconnected state (unavailable until AVP reconnection) */
    DR_CONNECTED_OVER_USB = 6UL,        /** <BT paired iPhone is connected over USB (unavailable until USB connection exist) */
    DR_NOT_APPLICABLE = 7UL,            /** <BT device disconnect reason not applicable */
    DR_DEVICE_REQUEST = 8UL,            /** <BT device disconnect reason for Device request */
    DR_MODULE_REQUEST = 9UL,            /** <BT device disconnect reason for Module request */
    DR_AUTOMATIC = 10UL,                /** <BT device disconnect reason  for automatic*/
    DR_OUT_OF_RANGE = 11UL,             /** <BT device disconnect reason for BT Device out of range*/
    DR_CONNECT_TIMEOUT = 12UL,          /** <BT device disconnect reason for device or profile connection timeout */
    DR_CONNECT_REJECTED = 13UL,         /** <BT device disconnect reason for device or profile connection rejected */
    DR_PROTOCOL_NOT_SUPPORTED = 14UL,   /** <BT device disconnect reason for protocol not supported */
    DR_CONNECT_FAILED = 15UL,           /** <BT device disconnect reason for connection failed */
    DR_REMOVED_BY_MP = 16UL,             /**<BT paired phone is in internal disconnect situation */
    DR_SPP_DISCONNECTED = 17UL           /** <BT device disconnect reason SPP profile Disconnected */
} tDisconnectReason;

/**
 * Type declaration of an enum describing the search result for quicksearch
 * Roadmap 15006_QuickSearchForNissan
 */
typedef enum
{
    FOUND  = 0UL,
    FOLLOWING,
    ENDOFLIST,
    NOTSUPPORTED
} tSearchResult;

/**
 * Search File type for File list Quicksearch operation
 */
typedef enum
{
    FL_NONE = 0L,
    FL_FOLDER,
    FL_PLAYLIST,
    FL_TRACK
} tSearchFileType;

/**
 * CDInfo- Device State for RNAIVI
 */
typedef enum
{
    DEVICE_NOT_READY = 0L,
    DEVICE_READY
} tenOpticalDeviceState;

#define DVMGR_MAX_DRIVEVERSIONLENGTH 40
/**
 * Type declaration - context CD drive Version
 */
typedef char tOpticalDriveVersion[DVMGR_MAX_DRIVEVERSIONLENGTH];

/**
 * CDInfo- for RNAIVI
 */
typedef struct
{
    tenOpticalDeviceState  OpticalDeviceState;
    tOpticalDriveVersion   OpticalDriveVersion;
}trOpticalDiskCDInfo;

/***********************
 * Type definitions
 ***********************/

/**
 * Type declaration of a device ID
 */
typedef tUInt tDeviceID;
#define tDeviceID_format "i"
#define DEVICE_ID_NOT_SET ((tDeviceID)-1)
#define tDeviceID_init DEVICE_ID_NOT_SET
#define MY_MEDIA (0)

/**
 * Type declaration an Error code set into OutputWrapper on AlbumArt error
 */
typedef tUInt tErrorCode;

/**
 * Type declaration for a device name
 */
typedef char tDeviceName[128];
#define tDeviceName_format "t"

/**
 * Type declaration an user friendly device name
 */
typedef char tAccessoryName[128];
#define tAccessoryName_format "t"

/**
 * Type declaration for original serial ID
 */
typedef char tAccessoryName2[128];
#define tAccessoryName2_format "t"
/**
 * Typedef for a version string of a device
 */
typedef char tDeviceVersion[128];
#define tDeviceVersion_format "t"
/**
 * Typedef for a serial number of a device
 */
typedef char tDeviceSerialNumber[128];
#define tDeviceSerialNumber_format "t"

/**
 * Type declaration of a BT MAC address
 */
typedef char tBTMACAddress[128];

/**
 * Type declaration for BT Vehicle Friendly Name
 */
typedef char tBTVehicleFriendlyName[128];

/**
 * Typedef for a serial number of a device
 */
typedef char tTOC[128];
#define tTOC_format "t"
#define tTOC_init {0}

/**
 * Typedef for an USB port number of a device
 */
typedef char tUSBPortNumber[128];
#define PORTNUM_DEFAULT (char*)"-1"

/**
 * Type declaration of a directory path on root to which an additional filesystem is mounted
 */
typedef char tMountPoint[1024];
#define tMountPoint_format "t"

/**
 * Type declaration for a flag if an object is stored as a favorite for the current user or not
 */
typedef tBoolean tIsFavorite;

/**
 * Type declaration for a Interface name(DBUS)
 */
typedef char tInterfaceName[128];

/**
 * Type declaration of a directory path on root to which an additional filesystem is mounted
 */
typedef char tRippedFilePath[1024];
#define tRippedFilePath_format "t"

/**
 * Type declaration for a flag if a device is connected or not
 */
typedef tBoolean tDeviceConnected;
#define tDeviceConnected_format "i"

/**
 * Type declaration for a flag if a device version updated from 1.0 to 1.4 or 1.0 to 1.3
 */
typedef tBoolean tDeviceVersionUpdated;
#define tDeviceVersionUpdated_format "i"
#define tDeviceVersionUpdated_init 0
/**
 * Type declaration for a flag if a device is active source or not
 */
typedef tBoolean tDeviceActiveSource;
#define tDeviceActiveSource_format "i"
#define tDeviceActiveSource_init 0

/**
 * Type declaration of the count how often the device was ever connected to the target
 */
typedef tUInt tConnectionCount;
#define tConnectionCount_format "i"

/**
 * Type declaration of the number of connected devices
 */
typedef tUInt tDeviceCount;
#define tDeviceCount_format "i"
#define tDeviceCount_init 0

/**
 * Type declaration for the number of devices
 */
typedef tUInt tNumberOfDevices;

/**
 * Type declaration for the number of media objects
 */
typedef tUInt tNumberOfMediaObjects;
#define tNumberOfMediaObjects_format "i"
#define tNumberOfMediaObjects_init 0

/**
 * Type declaration for the number of playlists
 */
typedef tUInt tNumberOfPlayLists;

/**
 * Type declaration for the number of folders
 */
typedef tUInt tNumberOfFolders;

/**
 * Type declaration for the number of files
 */
typedef tUInt tNumberOfFiles;
#define tNumberOfFiles_format "i"
#define tNumberOfFiles_init 0
#define NUMBER_OF_FILES_NONE ((tNumberOfFiles)-1)

/**
 * Type declaration for the file path
 */
typedef char tPath[1024];
#define tPath_format "t"
#define tPath_init ""

/**
 * Type declaration for the media object (database) id
 */
typedef tUInt tMediaObjectID;

/**
 * Type declaration for track number
 */
typedef tUInt tTrackNumber;

/**
 * Type declaration for a flag if an object is playing or not
 */
typedef tBoolean tIsPlaying;

/**
 * Type declaration - context device notification
 */
typedef tUInt tVendorID;

/**
 * Type declaration - context device notification
 */
typedef tUInt tProductID;
#define tProductID_format "i"

/**
 * Type declaration - context device notification
 */
typedef tU64 tStorageSize;

/**
 * Type declaration - context device notification
 */
typedef char tProtocolUsb[128];

/**
 * Type declaration - context device notification
 */
typedef char tDeviceSpeed[128];

/**
 * Type declaration - context device notification
 */
typedef char tBaseClassCodeUsb[128];

/**
 * Type declaration - context device notification
 */
typedef char tSubClassCodeUsb[128];

/**
 * Type declaration - context device notification
 */
typedef char tManufacturer[256];

/**
 * Type declaration for a Response message
 */
typedef tS32 tResponseMsg;
#define tResponseMsg_format "i"

#define tResponseMsg_init 0

/**
 * Type definition for AppControl interface
 */
typedef char tProtocolName [128];
#define tProtocolName_format "t"

typedef char tBundleSeedID [128];
#define tBundleSeedID_format "t"

typedef char tAppName [128];
#define tAppName_format "t"

typedef tUInt tSessionID;
#define tSessionID_format "i"
#define tSessionID_init 0

typedef unsigned char *tCommandBufferPtr;
#define tCommandBufferPtr_format "p"
#define tCommandBufferPtr_init NULL

/**
 * Type declaration for Volume
 */
typedef tUInt tVolume;
#define tVolume_format "i"
#define tVolume_init 0

typedef struct
{
    tAppName AppName;
    tProtocolName ProtocolName;
    tBundleSeedID BundleID;
    tAppInfoOption option;
}tAppInfo;
void InitAppInfo(tAppInfo &appInfo);

/**
 * Type definitions for DiPO
 */

typedef tBoolean tDiPOActive;
#define tDiPOActive_format "i"
#define tDiPOActive_init false

typedef char tDiPOVersion[128];
#define tDiPOVersion_format "t"

typedef char tDiPOGPGGAData[1024];
#define tDiPOGPGGAData_format "t"

typedef char tDiPOGPRMCData[1024];
#define tDiPOGPRMCData_format "t"

typedef char tDiPOGPGSVData[1024];
#define tDiPOGPGSVData_format "t"

typedef char tDiPOGPHDTData[1024];
#define tDiPOGPHDTData_format "t"

typedef char tDiPOPASCDData[1024];
#define tDiPOPASCDData_format "t"

typedef char tDiPOPAGCDData[1024];
#define tDiPOPAGCDData_format "t"

typedef char tDiPOPAACDData[1024];
#define tDiPOPAACDData_format "t"

typedef enum
{
    DIPO_OK = 0UL,                  // OK
    DIPO_ERROR_UNKNOWN = 1UL,       // Unknown error
    DIPO_ERROR_INVALIDHANLDE = 2UL, // Invalid handle
    DIPO_ERROR_NOTSUPPORTED = 3UL   // Device does not support DIPO
} tDiPOResponse;
#define tDiPOResponse_format "i"
#define tDiPOResponse_init DIPO_ERROR_UNKNOWN

typedef enum
{
    DIPO_CLIENT_MODE = 0UL,             // DeviceMode
    DIPO_CARPLAY_MODE,                  // HostMode
    DIPO_NATIVE_TRANSPORT_MODE,         // HostMode
    DIPO_CARPLAY_NATIVE_TRANSPORT_MODE,  // HostMode
    DIPO_CARLIFE_NATIVE_TRANSPORT_MODE
} tDiPORoleStatus;
#define tDiPORoleStatus_format "i"
#define tDiPORoleStatus_init DIPO_CLIENT_MODE

typedef enum
{
    DIPO_ROLE_SWITCH_NOT_REQUIRED = 0UL,  // not required
    DIPO_ROLE_SWITCH_REQUIRED_FOR_CARPLAY = 1UL,     // required for CarPlay
    DIPO_ROLE_SWITCH_REQUIRED_FOR_NATIVE_TRANSPORT = 2UL,     // required for native transport
    DIPO_ROLE_SWITCH_REQUIRED_FOR_CARPLAY_AND_NATIVE_TRANSPORT = 3UL,     // required for CarPlay and native transport
    DIPO_ROLE_SWITCH_REQUIRED_FOR_CARLIFE_AND_NATIVE_TRANSPORT = 4UL    // required for CarLife and native transport
} tDiPOSwitchReqResponse;
#define tDiPOSwitchReqResponse_format "i"
#define tDiPOSwitchReqResponse_init DIPO_ROLE_SWITCH_NOT_REQUIRED

typedef enum
{
    DIPO_CAP_NONE = 0UL,  // default capability, devicemode
    DIPO_CAP_CARPLAY,                   //hostmode
    DIPO_CAP_NATIVE_TRANSPORT,          //hostmode
    DIPO_CAP_CARPLAY_NATIVE_TRANSPORT,  //hostmode
    DIPO_CAP_HOSTMODE_FEASIBLE,         //devicemode (for native transport)
    DIPO_CAP_CARPLAY_FEASIBLE,           //devicemode (includes hostmode possible)
    DIPO_CAP_CARPLAY_WIFI_FEASIBLE,     //CARPLAY WIFI: devicemode (includes CarPlay Wifi possible)
    DIPO_CAP_CARPLAY_WIFI_NOT_FEASIBLE,
    DIPO_CAP_CARPLAY_WIFI,               //CARPLAY WIFI: wireless mode ( CarPlay over Wireless )
    DIPO_CAP_IAP2BT,                        //IAP2BT : IAP2 connection only for Smart app.
    DIPO_CAP_CARLIFE
} tDiPOCaps;
#define tDiPOCaps_format "i"
#define tDiPOCaps_init DIPO_CAP_NONE
tBoolean IsDiPOCapability_HostMode(const tDiPOCaps caps);
tBoolean IsDiPOCapability_CarPlay(const tDiPOCaps caps);
tBoolean IsDiPOCapability_NativeTransport(const tDiPOCaps caps);

typedef enum
{
    DIPO_STOP_REASON_UNKNOWN = 0UL,  // default
    DIPO_STOP_REASON_DEVICE_SWITCHED,
    DIPO_STOP_REASON_SOURCE_SWITCHED,
    DIPO_STOP_REASON_SOURCE_SUSPENDED
} tDiPOStopReason;
#define tDiPOStopReason_format "i"
#define tDiPOStopReason_init DIPO_STOP_REASON_UNKNOWN

typedef enum
{
    DIPO_START_REASON_UNKNOWN = 0UL,  // default
    DIPO_START_REASON_REMOTE_ACTIVITY
} tDiPOStartReason;
#define tDiPOStartReason_format "i"
#define tDiPOStartReason_init DIPO_START_REASON_UNKNOWN

typedef struct
{
    unsigned int GPGGA : 1;
    unsigned int GPRMC : 1;
    unsigned int GPGSV : 1;
    unsigned int GPHDT : 1;
    unsigned int PASCD : 1;
    unsigned int PAGCD : 1;
    unsigned int PAACD : 1;
} tDiPOLocationInfoType;
void InitDiPOLocationInfoType(tDiPOLocationInfoType &DiPOLocationInfoType);

typedef struct
{
    unsigned int GPRMCDataStatusValueA : 1;
    unsigned int GPRMCDataStatusValueV : 1;
    unsigned int GPRMCDataStatusValueX : 1;
} tDiPOGPRMCDataStatusValues;
void InitDiPOGPRMCDataStatusValues(tDiPOGPRMCDataStatusValues &diPOGPRMCDataStatusValues);

typedef struct
{
    unsigned int HandsFree : 1;
    unsigned int PhoneBookAccess : 1;
    unsigned int AudioVideoRemoteControl : 1;
    unsigned int AdvancedAudioDistribution : 1;
    unsigned int HumanInterfaceDevice : 1;
    unsigned int iAP2Link : 1;
    unsigned int PersonalAreaNetworkAccessPoint : 1;
    unsigned int MessageAccess : 1;
    unsigned int PersonalAreaNetworkClient : 1;
} tBTProfile;
void InitBTProfile(tBTProfile &BTProfile);

typedef struct
{
    tDeviceID deviceID;
    tBTProfile BTProfile;
} tBTProfileInfo;
void InitBTProfileInfo(tBTProfileInfo &BTProfileInfo);

typedef vector<tBTProfileInfo> tBTProfileList;

typedef struct
{
    tGeneralString title;
    tGeneralString artist;
    tGeneralString album;
    tGeneralString genre;
    tGeneralString composer;
    tGeneralString albumArt;
    tAppName appName;
    tU16 albumTrackNumber;
    tU16 albumTrackCount;
    tU16 albumDiscNumber;
    tU16 albumDiscCount;
    tU16 chapterCount;
    tU32 queueIndex;
    tU32 queueCount;
    tU32 queueChapterIndex;
    tBoolean iTunesRadioAd;
    tGeneralString iTunesRadioStationName;
} tDiPONowPlaying;
void InitDiPONowPlaying(tDiPONowPlaying &DiPONowPlaying);

typedef enum
{
    DIPO_CALL_STATUS_DISCONNECTED = 0UL,  // default
    DIPO_CALL_STATUS_SENDING,
    DIPO_CALL_STATUS_RINGING,
    DIPO_CALL_STATUS_CONNECTING,
    DIPO_CALL_STATUS_ACTIVE,
    DIPO_CALL_STATUS_HELD,
    DIPO_CALL_STATUS_DISCONNECTING
} tDiPOCallStatus;
#define tDiPOCallStatus_format "i"
#define tDiPOCallStatus_init DIPO_CALL_STATUS_DISCONNECTED

typedef enum
{
    DIPO_CALL_DIRECTION_UNKNOWN = 0UL,  // default
    DIPO_CALL_DIRECTION_INCOMING,
    DIPO_CALL_DIRECTION_OUTGOING
} tDiPOCallDirection;
#define tDiPOCallDirection_format "i"
#define tDiPOCallDirection_init DIPO_CALL_DIRECTION_UNKNOWN

typedef enum
{
    DIPO_CALL_SERVICE_UNKNOWN = 0UL,  // default
    DIPO_CALL_SERVICE_TELEPHONY,
    DIPO_CALL_SERVICE_FACETIMEAUDIO,
    DIPO_CALL_SERVICE_FACETIMEVIDEO
} tDiPOCallService;
#define tDiPOCallService_format "i"
#define tDiPOCallService_init DIPO_CALL_SERVICE_UNKNOWN

typedef enum
{
    DIPO_CALL_DISCONNECT_REASON_ENDED = 0UL,  // default
    DIPO_CALL_DISCONNECT_REASON_DECLINED,
    DIPO_CALL_DISCONNECT_REASON_FAILED
} tDiPOCallDisconnectReason;
#define tDiPOCallDisconnectReason_format "i"
#define tDiPOCallDisconnectReason_init DIPO_CALL_DISCONNECT_REASON_ENDED

/**
 * Type declaration for Dipo Call Start timestamp
*/
typedef unsigned long long int tDiPOCallStartTimestamp;
#define tDiPOCallStartTimestamp_format "l"
#define tDiPOCallStartTimestamp_init 0

typedef struct
{
    tGeneralString remoteID;
    tGeneralString displayName;
    tDiPOCallStatus status;
    tDiPOCallDirection direction;
    tGeneralString callUUID;
    tGeneralString addressBookID;
    tGeneralString label;
    tDiPOCallService service;
    tBoolean isConferenced;
    tU8 conferenceGroup;
    tDiPOCallDisconnectReason disconnectReason;
    tDiPOCallStartTimestamp startTimestamp;
} tDiPOCallStateItem;
void InitDiPOCallStateItem(tDiPOCallStateItem &callStateItem);

typedef vector<tDiPOCallStateItem> tDiPOCallState;

typedef enum
{
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_0BARS = 0UL,  // default
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_1BAR,
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_2BARS,
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_3BARS,
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_4BARS,
    DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_5BARS
} tDiPOCommunicationsSignalStrength;
#define tDiPOCommunicationsSignalStrength_format "i"
#define tDiPOCommunicationsSignalStrength_init DIPO_COMMUNICATIONS_SIGNAL_STRENGTH_0BARS

typedef enum
{
    DIPO_COMMUNICATIONS_REGISTRATION_UNKNOWN = 0UL,  // default
    DIPO_COMMUNICATIONS_REGISTRATION_NOTREGISTERED,
    DIPO_COMMUNICATIONS_REGISTRATION_SEARCHING,
    DIPO_COMMUNICATIONS_REGISTRATION_DENIED,
    DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDHOME,
    DIPO_COMMUNICATIONS_REGISTRATION_REGISTEREDROAMING,
    DIPO_COMMUNICATIONS_REGISTRATION_EMERGENCYCALLONLY
} tDiPOCommunicationsRegistrationStatus;
#define tDiPOCommunicationsRegistrationStatus_format "i"
#define tDiPOCommunicationsRegistrationStatus_init DIPO_COMMUNICATIONS_REGISTRATION_UNKNOWN


typedef struct
{
    tDiPOCommunicationsSignalStrength signalStrength;
    tDiPOCommunicationsRegistrationStatus registrationStatus;
    tBoolean airplaneModeStatus;
    tGeneralString carrierName;
    tBoolean cellularSupported;
    tBoolean telephonyEnabled;
    tBoolean faceTimeAudioEnabled;
    tBoolean faceTimeVideoEnabled;
    tBoolean muteStatus;
    tU8 currentCallCount;
    tU8 newVoicemailCount;
    tBoolean initiateCallAvalable;
    tBoolean endAndAcceptAvailable;
    tBoolean holdAndAcceptAvailable;
    tBoolean swapAvailable;
    tBoolean mergeAvailable;
    tBoolean holdAvailable;
} tDiPOCommunications;
void InitDiPOCommunications(tDiPOCommunications &communications);

//CARPLAY WIFI:
typedef enum
{
    WIFI_SECURITY_NONE = 0,
    WIFI_SECURITY_WEP,
    WIFI_SECURITY_WPA_PERSONAL,
    WIFI_SECURITY_WPA2_PERSONAL
}tWiFiSecurityType;

typedef struct
{
    tGeneralString ssid;
    tGeneralString passPhrase;
    tU16 channelNumber;
    tWiFiSecurityType securityType;
}tWiFiAPCredentials;
void InittWiFiAPCredentials(tWiFiAPCredentials &wifiAPCredentials);

/**
 * Type definitions for CARPLAY WIFI BTLIMITATION MODE
 */
typedef enum
{
    BTTECHNOLOGY_IAP = 1,
    BTTECHNOLOGY_CARPLAY,
    BTTECHNOLOGY_INVALID = -1
}tBTTechnology;

typedef enum
{
    BTCOMMUNICATION_CHANNEL_BLUETOOTH = 4,
    BTCOMMUNICATION_CHANNEL_INVALID = -1
}tBTCommunicationChannel;

typedef enum
{
    BTLIMITATION_ACTION_STATE_IDLE = 0,
    BTLIMITATION_ACTION_STATE_PREPARED = 2,
    BTLIMITATION_ACTION_STATE_INVALID = -1
}tBTLimitationActionState;
#define tBTLimitationActionState_init BTLIMITATION_ACTION_STATE_INVALID;

typedef struct
{
    tBTMACAddress deviceBTMacAddress;
    tBTTechnology btTechnology;
    tBTCommunicationChannel btCommunicationChannel;
    tBTLimitationActionState btLimitationActionState;
}tBTLimitationMode;

typedef std::vector<tBTLimitationMode> tBTLimitationModeDeviceList;

/**
 * Type definitions for TagTransfer
 */
typedef unsigned char tAmbiguousTag;
typedef unsigned char tButtonPressed;
typedef char tTagTrackName[128];
typedef char tTagArtistName[128];
typedef char tTagAlbumName[128];
typedef char tTagGenreName[128];
typedef tUInt tTagSongID;
typedef tUInt tTagStorefrontID;
typedef char tTagChannelNumber[128];
typedef char tTagChannelName[128];
typedef char tTagTimeStamp[128];
typedef char tPodcastFeedURL[128];
typedef char tStationURL[128];
typedef unsigned char tTagProgramNumber;
typedef char tTagAffiliateID[128];
typedef unsigned char tUnknownDataSize;
typedef unsigned char tUnknownData[64];
typedef char tStationURLFrequency[128];

typedef struct
{
    tAmbiguousTag ambiguousTag;
    tButtonPressed buttonPressed;
    tTagTrackName tagTrackName;
    tTagArtistName tagArtistName;
    tTagAlbumName tagAlbumName;
    tTagGenreName tagGenreName;
    tTagSongID tagSongID;
    tTagStorefrontID tagStorefrontID;
    tTagChannelNumber tagChannelNumber;
    tTagChannelName tagChannelName;
    tTagTimeStamp tagTimeStamp;
    tPodcastFeedURL podcastFeedURL;
    tStationURL stationURL;
    tTagProgramNumber tagProgramNumber;
    tTagAffiliateID tagAffiliateID;
    tUnknownDataSize unknownDataSize;
    tUnknownData unknownData;
    tStationURLFrequency stationURLFrequency;
} tTagTransfer;
void InitTagTransfer(tTagTransfer &tagTransfer);

typedef struct
{
    tTagSongID tagSongID;
    tTagArtistName tagArtistName;
    tTagTrackName tagTrackName;
} tUntransferredTag;
void InitUntransferredTag(tUntransferredTag &untransferredTag);

/**
 * Type declaration of favorite ID
 */
typedef tUInt tFavoriteID;
#define tFavoriteID_format "i"
#define tFavoriteID_init 0

/**
 * Type declaration of the number of favorites
 */
typedef tUInt tFavCount;
#define tFavCount_format "i"
#define tFavCount_init 0

/**
 * Type declaration for the length of a list
 */
typedef tUInt tListSize;
#define tListSize_format "i"
#define tListSize_init 0


/**
 * Type declaration of a device control name
 */
typedef char tDeviceControlName[128];

/**
 * Type declaration of a message
 */
typedef char tMsg[1024];

/**
 * Type declaration of a database object ID
 */
typedef tUInt tObjectID;
#define tObjectID_format "i"
#define OBJECT_ID_NONE ((tObjectID)-1)
#define tObjectID_init OBJECT_ID_NONE

/**
 * Type for a row number in a database list
 */
typedef tUInt tRowNumber;
#define ROW_NUMBER_NONE ((tRowNumber)-1)

/**
 * Type for a trigger id for a database trigger
 */
typedef tUInt tTriggerID;

/**
 * Type for a filter tag object id
 */
typedef tUInt tFilterTag;
#define tFilterTag_format "i"
#define FILTER_TAG_UNUSED ((tFilterTag)-1)
#define tFilterTag_init 0

/**
 * Type for the name of a file
 */
typedef char tFilename[1024];
#define tFilename_format "t"
int GenerateTempFileName(tFilename &tmpFile, const char * prefix, const char * path = "/tmp");
/**
 * Type declaration of a list ID
 */
typedef tUInt tListID;
#define tListID_format "i"
//#define LIST_ID_NONE ((tListID)-1)
#define LIST_ID_NONE ((tListID)0)
#define tListID_init LIST_ID_NONE

/**
 * Type declaration of a song ID
 */
typedef tUInt tSongID;
#define tSongID_format "i"
#define tSongID_init 0

/**
 * Type declaration for metadata of one media object
 */
typedef char tMetadata[256];
#define tMetadata_format "t"
#define tMetadata_init {0}

/**
 * Type declaration for a song title
 */
typedef tMetadata tTitle;
#define tTitle_format "t"

/**
 * Type declaration for return value
 */
typedef tBoolean tReturnValue;
#define tReturnValue_format "i"
#define tReturnValue_init 0

/**
 * Type declaration of a skip count for next and prev playback action
 */
typedef tUInt tNextPrevSkipCount;
#define tNextPrevSkipCount_format "i"
#define tNextPrevSkipCount_init 0

/**
 * Type declaration of a playtime in milliseconds
 */
typedef tUInt tPlaytime;
#define tPlaytime_format "i"
#define tPlaytime_init 0
#define PLAYTIME_NONE ((tPlaytime)-1)
#define PLAYTIME_END_OF_OBJECT (tPlaytime)(-2)

/**
 * Type declaration of a element position in a list
 */
typedef tUInt tPosition;
#define tPosition_format "i"
#define tPosition_init 0
#define POSITION_NOT_SET (tPosition)(-1)

/**
 * Type declaration of a name of a ALSA output device
 */
typedef char tAudioOutputDevice[128];
#define tAudioOutputDevice_format "t"

/**
 * Type declaration of a name of a ALSA input device
 */
typedef char tAudioInputDevice[128];

/**
 * Type declaration of string for phone info
 */
typedef char tPhoneString[1024];


/**
 * Type declaration of the slideshow time in seconds
 */
typedef tUInt tSlideshowTime;
#define tSlideshowTime_format "i"
#define tSlideshowTime_init 0

/**
 * Type declaration of an index
 */
typedef tUInt tIndex;
#define tIndex_format "i"
#define INDEX_LAST_ACTIVE_OBJECT ((tIndex)-1)
#define INDEX_NO_LIMIT ((tIndex)-1)
#define INDEX_NONE ((tIndex)-2)
#define tIndex_init 0

/**
 * Typedef for the name of a file to play with mount point
 */
typedef char tURL[1024];
#define tURL_format "t"

/**
 * Typedef for the name of a PlayerName
 */
typedef char tPlayerName[1024];
#define tPlayerName_format "t"

/**
 * Type declaration for position (URL) of next media object to read
 */
typedef tURL tReadPosition;
#define tReadPosition_format "t"

/**
 * Type declaration of a cueing rate
 */
typedef tS64 tCueingRate;
#define tCueingRate_format "l"
#define tCueingRate_init 1

/**
 * Type declaration for fingerprint value
 */
typedef char tFingerprint[1024];
#define tFingerprint_format "t"
#define FINGERPRINT_INVALID "invalid"
#define FINGERPRINT_SHORTCUT "shortcut -> usedMemory: "

/**
 * Typedef for album art string "deviceID:MediaObjectPath" (device ID + URL of image to read)
 */
typedef char tAlbumArt[1024];
#define tAlbumArt_format "t"

/**
 * Typedef for album art handle of a media object in a BT device
 */
typedef char tAlbumArtHandle[1024];

/**
 * Typedef for album art indexing complete flag
 */
typedef tBoolean tAlbumArtIndexingComplete;

/**
 * Type declaration for the number pixels in x or y direction
 */
#ifndef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
typedef tUInt tSize;
#else
typedef size_t tSize;
#endif
#define tSize_format "i"
#define tSize_init 0

/**
 * Type declaration for a size of DVD Resume info in bytes
 */
typedef tUInt tDVDInfoSize;

/**
 * Type declaration of DVD Resume data
 */
typedef unsigned char *tDVDInfoData;
/**
 * Type declaration for a size of an image in bytes
 */
typedef tUInt tImageSize;

/**
 * Type declaration of image data
 */
typedef unsigned char *tImageData;

/**
 * Type declaration for percent completion of the indexing operation
 */
typedef tUInt tIndexingPercentComplete;

/**
 * Type declaration of a source number
 */
typedef tUInt tSourceNr;

/**
 * Type declaration of a source
 */
typedef tUInt tSource;

/**
 * Type declaration for a favorite label text
 */
typedef char tLabelText[128];

/**
 * Type declaration for a favorite descriptor text
 */
typedef char tDescriptorText[128];

/**
 * Type declaration for the favorite/album art availability attribute
 */
typedef tBoolean tAvailable;
#define tAvailable_format "i"
#define tAvailable_init 1

/**
 * Type declaration for the favorite active state attribute
 */
typedef tBoolean tActive;

/**
 * Type declaration for the list window start position (offset inside of a list)
 */
typedef tUInt tListWindowStartPosition;

/**
 * Type declaration of a state machine name
 */
typedef tGeneralString tStateMachineName;
#define tStateMachineName_format "t"

typedef tGeneralString tLinkKey;
#define tLinkKey_format "t"


/**
 * Type declaration of a state message name
 */
typedef tGeneralString tMessageName;
#define tMessageName_format "t"

/**
 * Type declaration of an album id
 */
typedef tUInt tAlbumID;
#define tAlbumID_format "i"
#define tAlbumID_init 0

/**
 * Type declaration of an album name
 */
typedef tGeneralString tAlbumName;
#define tAlbumName_format "t"

/**
 * Type declaration of an artist id
 */
typedef tUInt tArtistID;
#define tArtistID_format "i"
#define tArtistID_init 0

/**
 * Type declaration of an artist name
 */
typedef tGeneralString tArtistName;
#define tArtistName_format "t"

/**
 * Type declaration of an author id
 */
typedef tUInt tAuthorID;
#define tAuthorID_format "i"
#define tAuthorID_init 0

/**
 * Type declaration of an author name
 */
typedef tGeneralString tAuthorName;
#define tAuthorName_format "t"

/**
 * Type declaration of a book title id
 */
typedef tUInt tBookTitleID;
#define tBookTitleID_format "i"
#define tBookTitleID_init 0

/**
 * Type declaration of a book title name
 */
typedef tGeneralString tBookTitleName;
#define tBookTitleName_format "t"

/**
 * Type declaration of a chapter id
 */
typedef tUInt tChapterID;
#define tChapterID_format "i"
#define tChapterID_init 0

/**
 * Type declaration of a chapter name
 */
typedef tGeneralString tChapterName;
#define tChapterName_format "t"

/**
 * Type declaration of a composer id
 */
typedef tUInt tComposerID;
#define tComposerID_format "i"
#define tComposerID_init 0

/**
 * Type declaration of a composer name
 */
typedef tGeneralString tComposerName;
#define tComposerName_format "t"

/**
 * Type declaration of an episode id
 */
typedef tUInt tEpisodeID;
#define tEpisodeID_format "i"
#define tEpisodeID_init 0

/**
 * Type declaration of an episode name
 */
typedef tGeneralString tEpisodeName;
#define tEpisodeName_format "t"

/**
 * Type declaration of a genre id
 */
typedef tUInt tGenreID;
#define tGenreID_format "i"
#define tGenreID_init 0

/**
 * Type declaration of a genre name
 */
typedef tGeneralString tGenreName;
#define tGenreName_format "t"

/**
 * Type declaration of a podcast id
 */
typedef tUInt tPodcastID;
#define tPodcastID_format "i"
#define tPodcastID_init 0

/**
 * Type declaration of a podcast name
 */
typedef tGeneralString tPodcastName;
#define tPodcastName_format "t"

/**
 * Type declaration of a playlist id
 */
typedef tUInt tPlaylistID;
#define tPlaylistID_format "i"
#define tPlaylistID_init 0

/**
 * Type declaration of a playlistname
 */
typedef tGeneralString tPlaylistName;
#define tPlaylistName_format "t"

/**
 * Type declaration of a video id
 */
typedef tUInt tVideoID;
#define tVideoID_format "i"
#define tVideoID_init 0

/**
 * Type declaration of a video name
 */
typedef tGeneralString tVideoName;
#define tVideoName_format "t"

/**
 * Type declaration of a tag string
 */
typedef tGeneralString tTag;

/**
 * Type declaration of a tag ID
 */
typedef tUInt tTagID;

/**
 * Type declaration of a file extension
 */
typedef char tExtension[16];

/**
 * Type declaration of an encoding type
 */
typedef char tEncoding[64];

/**
 * Type declaration of a bit rate used for decoded audio/video
 */
typedef tUInt tBitRate;
#define tBitRate_format "i"
#define tBitRate_init 0

/**
 * Type declaration of a video profile name
 */
typedef char tVideoProfileName[64];

/**
 * Type declaration of a video profile level
 */
typedef float tVideoProfileLevel;

/**
 * Type declaration of a frame rate used for decoded video
 */
typedef tUInt tFrameRate;

/**
 * Type declaration of a audio version name
 */
typedef char tAudioVersionName[64];

/**
 * Type declaration of an UUID
 */
typedef tGeneralString tUUID;
#define tUUID_format "t"
#define tUUID_init {0}

/**
 * Type declaration of an Apple Device MAC Address
 */
typedef tGeneralString tMACAddress;
#define tMACAddress_format "t"
#define tMACAddress_init {0}

/**
 * Type declaration of Apple device USBSerial number.
 */
typedef tGeneralString tUSBSerial;
#define tUSBSerial_format "t"
#define tUSBSerial_init {0}

/**
 * Type declaration of a thread id (handle)
 */
typedef tUInt tThreadHandle;

/**
 * Type declaration of a priority value (0: highest priority)
 */
typedef tUInt tPriority;
#define PRIORITY_NONE ((tPriority)-1)

/**
 * Type declaration of a search string
 */
typedef char tSearchString[1024];

/**
 * Type declaration of fileExtensions
 */
typedef vector<std::string> tFileExtensions;

/**
 * Type declaration of row id used by VTIPOD
 */
typedef tUInt tRowID;
#define tRowID_format "i"
#define tRowID_init 0

/**
 * Type declaration of limit used by VTIPOD
 */
typedef tUInt tLimit;
#define tLimit_format "i"
#define tLimit_init -1

/**
 * Type declaration of offset used by VTIPOD
 */
typedef tUInt tOffset;
#define tOffset_format "i"
#define tOffset_init 0

/**
 * Type declaration of tVirtualCount used by VTIPOD
 */
typedef tUInt tVirtualCount;
#define tVirtualCount_format "i"
#define tVirtualCount_init 0

/**
 * Type declaration of tVirtualID used by VTIPOD
 */
typedef tUInt tVirtualID;
#define tVirtualID_format "i"
#define tVirtualID_init 0

/**
 * Type declaration of tVirtualName used by VTIPOD
 */
typedef tMetadata tVirtualName;
#define tVirtualName_format "t"

/**
 * Type declaration of tVirtualURL used by VTIPOD
 */
typedef tURL tVirtualURL;
#define tVirtualURL_format "t"

/**
 * Type declaration a layer name
 */
typedef char tLayerName[128];

/**
 * Type declaration of a media object handle (=objectID)
 * Roadmap 13010
 */
typedef tS64 tPEHandle;
#define HANDLE_NONE ((tPEHandle)-1)
#define tPEHandle_format "l"
#define tPEHandle_init HANDLE_NONE

/**
 * Type declaration of a device partition number
 * Roadmap 13023
 */
typedef tUInt tPartitionNumber;
#define PARTITION_NUMBER_NONE ((tPartitionNumber)-1)

typedef tUInt tNumberOfPartitions;

/**
 * Type declaration for a memory size in kbytes
 * Roadmap 13001
 */
typedef tUInt tMemorySize;

/**
 * Type declaration for a flag if a list is a streaming list
 * Roadmap 13008
 */
typedef tBoolean tStreaming;
#define tStreaming_format "i"
#define tStreaming_init 0

/**
 * Type declaration for a flag if system is in undervoltage situation
 * Roadmap 13024
 */
typedef tBoolean tUndervoltage;

/**
 * Type declaration for a flag if codec is VBR
 * Roadmap 14011
 */
typedef tBoolean tVBR;

/**
 * Type declaration of the user data
 * Roadmap 13034_SPIInterface
 */
typedef tAllParameters tUserData;
#define tUserData_format "t"

/**
 * Type declaration for HMI request flag
 * Roadmap 14001
 */
typedef tBoolean tHMIRequest;
#define tHMIRequest_format "i"
#define tHMIRequest_init 0

/* statistics string for all components */
typedef char tStatistics[4096];

/* batch playback list */
typedef vector<tU64> tBatchPlaybackList;

typedef tBoolean tNowPlayingListAvailable;
#define tNowPlayingListAvailable_format "i"
#define tNowPlayingListAvailable_init 0

typedef tBoolean tShuffleSupport;
#define tShuffleSupport_format "i"
#define tShuffleSupport_init 0

typedef tBoolean tRepeatSupport;
#define tRepeatSupport_format "i"
#define tRepeatSupport_init 0

typedef tUInt tConvertFlag;
#define tConvertFlag_format "i"
#define tConvertFlag_init 0

typedef tUInt tCompilation;
#define tCompilation_format "i"
#define tCompilation_init 0

/**
 * Type declaration for file mode - permission triad for user, group and others
 * Stored in numeric octal notation e.g. 0777 = -rwxrwxrwx
 * Roadmap 16011
 */
typedef tUInt tFileMode;

/**
 * Type declaration for user ID of file (UID)
 * Roadmap 16011
 */
typedef tUInt tUserID;

/**
 * Type declaration for group ID of file (GID)
 * Roadmap 16011
 */
typedef tUInt tGroupID;

/**
 * Type declaration for file size in Byte
 * Roadmap 16011
 */
typedef tU64 tFileSize;

/**
 * Type declaration of last file modification date and time in Y:m:d H:M:S
 * Roadmap 16011
 */
typedef char tDateTime[32];

/**
 * Type declaration for full text search
 * Roadmap 16014
 */

typedef tUInt tSearchIncrement;
#define tSearchIncrement_init    50

/**
 * Type declaration for full text search
 * Roadmap 16014
 */
typedef tUInt tDelaySec;
#define tDelaySec_init    0
#define tDelaySec_noDelay 0
/**
 * Type declaration of Maneuver list
 */
//typedef unsigned char *tManeuverList;

/**
 * Type declaration for a size of ManeuverList in bytes
 */
//typedef unsigned short tManeuverListSize;
/***********************
 * Structures and unions
 ***********************/
/**
 * Type declaration for the ManeuverList blob
 */
/*typedef struct
{
    tManeuverListSize ManeuverListSize;
    tManeuverList ManeuverList;
} tManeuverListBlob;
 */
typedef vector<tU16> tManeuverList;
typedef vector<tU16> tRouteGuidanceDisplayComponentIDList;
/**
 * Type declaration for the image blob
 */
typedef struct
{
    tImageSize imageSize;
    tImageData imageData;
} tImageBlob;
/**
 * Type declaration for the image blob
 */
typedef struct
{
    tDVDInfoSize dataSize;
    tDVDInfoData data;
} tDVDInfoBlob;
/**
 * Type declaration for the album art object
 */
typedef struct
{
    tAlbumArt albumArtString;
    tMimeType mimeType;
    tSize sizeX;
    tSize sizeY;
    tImageSize imageSize;
    tImageData imageData;
} tAlbumArtObject;

/*Type definition for Yomi MetaData Info */

typedef struct
{
  tMetadata YomiTitle;
  tMetadata YomiArtist;
  tMetadata YomiAlbum;
} tYomiMetadata;


/**
 * Type declaration for a pointer to an album art object
 */
typedef tAlbumArtObject *tAlbumArtObjectPtr;
#define tAlbumArtObjectPtr_format "p"
#define tAlbumArtObjectPtr_init 0

/**
 * Type declaration for contents of tag position 1
 */
typedef union
{
    tUInt tag;
    tUInt genre;
    tUInt podcast;
    tUInt author;
    tUInt video;
    tUInt playlist;
    tUInt yearID;
} tFilterTag1;

typedef enum {
    DISC_UNDECIDED = 0x00,
    DISC_NO_ACCESS = 0x00,
    DISC_NONE = 0x10,
    DISC_ERROR = 0x20,
    DISC_JUDGING = 0xF0,
    DISC_AUDIO_CD = 0x01,
    DISC_AUDIO_CD_WITH_TEXT = 0x11,
    DISC_VCD_VER_1 = 0x02,
    DISC_VCD_VER_2_PBC_OFF = 0x12,
    DISC_SUPER_VCD_PBC_OFF = 0x22,
    DISC_VCD_VER_2_PBC_ON = 0x03,
    DISC_SUPER_VCD_PBC_ON = 0x13,
    DISC_DVD_VIDEO = 0x04,
    DISC_DVD_AUDIO = 0x05,
    DISC_DVD_VR = 0x06,
    DISC_DATA = 0x09
}tDiscType;
#define tDiscType_format "i"
#define tDiscType_init DISC_NONE

/**
 * iPod marker bit used to indicate VTIPOD access
 */
#ifndef IPOD_MARKER_BIT
#define IPOD_MARKER_BIT ((tUInt)1 << ((sizeof(tFilterTag1)*8)-1)) // sets the top most bit in the ipod id field
#define IPOD_MARKER_MASK (~IPOD_MARKER_BIT) // sets the mask to hide the ipod marker bit
#endif

/**
 * Type declaration for contents of tag position 2
 */
typedef union
{
    tUInt tag;
    tUInt artist;
    tUInt audiobook;
} tFilterTag2;

/**
 * Type declaration for contents of tag position 3
 */
typedef union
{
    tUInt tag;
    tUInt composer;
} tFilterTag3;

/**
 * Type declaration for contents of tag position 4
 */
typedef union
{
    tUInt tag;
    tUInt album;
} tFilterTag4;


/**
 * \enum tenDevice_UnsupportedReason -
 *  USB Compliance - UnsupportedReason for device connected
 */
typedef enum
{
    OK                            = 0x00,
    NotMountedDevice              = 0x01,
    DisabledInConfiguration       = 0x02,
    NotSupportedByDVM             = 0x03,
    HubButNotCustomerHub          = 0x04
}tDeviceUnsupportedReason;

//Indicates the cover art connection status of the BTDevice
typedef enum
{
    NOTCONNECTED = 0,
    CONNECTED,
    UNKNOWN
}tCoverArtConStatus;

/**
 * Type for the device information from MOST catalog
 */
typedef struct {
    tDeviceID deviceID;
    tUUID UUID;
    tDeviceSerialNumber serialNumber;
    tDeviceVersion deviceVersion;       /**< FirmwareVersion */
    tDeviceName deviceName;             /**< FriendlyName */
    tMountPoint mountPoint;
    tUSBPortNumber portNumber;       /**< PortNumber where device is connected */
    tConnectionCount connectionCount;
    tDeviceActiveSource activeSource;
    tDeviceType deviceType;
    tDiscType discType;
    tConnectionType connectionType;
    tIndexingState indexedState;
    tConnectionState connectionState;
    tConnectionState formerConnectionState;
    tAccessoryName accessoryName;       /**< Additional name information e.g. alsa device name (AudioInputDevice) */
    tAccessoryName accessoryName2;      /**<Support for multi partition devices*/
    tDeviceState deviceState;           //Roadmap 13023
    tFileSystemType fileSystemType;     //Roadmap 13023
    tPartitionNumber partitionNumber;   //Roadmap 13023
    tMemorySize totalSize;              //Roadmap 13001
    tMemorySize freeSize;               //Roadmap 13001
    tNumberOfFiles numberOfAudioFiles;  //Roadmap 13002
    tNumberOfFiles numberOfVideoFiles;  //Roadmap 1612
    tNumberOfFiles numberOfImageFiles;  //Roadmap 1612
    tProductID productID;               //Roadmap 13006
    tDiPOCaps diPOCaps;                 //Roadmap 13034
    tDiPOActive diPOActive;             //Roadmap 13034
    tDiPOVersion diPOVersion;           //Roadmap 13034
    tPath lastPlayedFile;               //Roadmap 14005
    tDeviceConnected connected;
    tDisconnectReason disconnectReason; //Roadmap 15003
    tShuffleSupport isShuffleSupported;
    tRepeatSupport isRepeatSupported;
    tNowPlayingListAvailable isNowPlayingListAvailable;
    tUUID deviceUUID;                   //CARPLAY WIFI
    tMACAddress appleDeviceMACAddress;
    tPath deviceSyspath;                //Device Syspath -  CRQ  CMG3G-12660
    tUSBSerial appleDeviceUSBSerialNumber;
    tBTLimitationActionState btLimitationActionState;  //CMG3G-14540: BTLimitation Mode for Wireless CarPlay supported Apple BT device.
    tCoverArtConStatus coverArtConStatus; /* Task 317554 Album art extraction from AVRCP 1.6 supported BT device*/
} tDeviceInfo;
void InitDeviceInfo(tDeviceInfo &deviceInfo);

/**
 * marshaled device info
 */
typedef tAllParameters tDeviceInfoString;
#define tDeviceInfoString_format "t"

/**
 * Type for the device information coming from the vd device manager
 */
typedef struct {
    tDeviceSerialNumber serialNumber;
    tUSBPortNumber      USBPortNumber;
    tDeviceType         deviceType;
    tConnectionType     connectionType;
    tDeviceVersion      deviceVersion;
    tDeviceName         deviceName;         /**< user friendly name */
    tAccessoryName      accessoryName;      /**< Additional name information e.g. alsa device name */
    tAccessoryName2     accessoryName2;     /**< Added to support multipartition USB stick feature. This store the original serial ID of the device.*/
    tMountPoint         mountPoint;
    tDMConnectStatus    connectStatus;
    tStorageSize        sizeInByte;           //available through code merge NissanLCN2Kai, GMMY15, kept for VAG-MIBE diag
    tVendorID           vendorID;             //available through code merge NissanLCN2Kai, GMMY15, kept for VAG-MIBE diag
    tManufacturer       manufacturer;         //available through code merge NissanLCN2Kai, GMMY15, kept for VAG-MIBE diag
    tProductID          productID;            //available through code merge NissanLCN2Kai, GMMY15, kept for VAG-MIBE diag
    tStorageSize        totalSizekB;          //Roadmap 13001 ExtUserConfig
    tStorageSize        usedSizekB;           //Roadmap 13001 ExtUserConfig
    tStorageSize        freeSizekB;           //Roadmap 13001 ExtUserConfig
    tPartitionNumber    partitionNumber;      //Roadmap 13001 ExtUserConfig /**< supported for msc devices like SD-card & usb-stick*/
    tFileSystemType     fileSystemType;       //Roadmap 13001 ExtUserConfig
    tDeviceSpeed        deviceSpeed;          //Roadmap 13001 ExtUserConfig -diag
    tProtocolUsb        protocolUsb;          //Roadmap 13001 ExtUserConfig- diag  /** < usb protocol like usb1.0, usb2.0*/
    tBaseClassCodeUsb   baseClassCodeUsb;     //Roadmap 13001 ExtUserConfig -diag
    tSubClassCodeUsb    subBaseClassCodeUsb;  //Roadmap 13001 ExtUserConfig -diag
    tDeviceUnsupportedReason deviceUnsupportedReason; //USB Compliance - unsupported device
    tPath               deviceSyspath; //Device Syspath -  CRQ  CMG3G-12660

} tDeviceProperty;
void InitDeviceProperty(tDeviceProperty &devProp);

/**
 * Type for the information about a media object (@@@: see PlayItemFromList in MOST catalog)
 */
typedef struct
{
    unsigned char ObjectValid : 1;
    unsigned char ObjectInvalid : 1;
    unsigned char ListInvalid : 1;
    unsigned char PositionInvalid : 1;
    unsigned char GeneralError : 1;
} tMediaObjectInfo;
void InitMediaObjectInfo(tMediaObjectInfo &mediaObjectInfo);

/**
 * Type declaration of a file information
 */
typedef struct {
    tBoolean readOnlyFlag;      /**<Roadmap CMG3G-10221 Scene Recorder>*/
    tFileType type;             /**< FT_FOLDER, FT_PLAYLIST, FT_AUDIO, FT_VIDEO, FT_IMAGE */
    tTrackNumber trackNumber;   /**< sequence number in folder */
    tObjectID objectID;         /**< real object id from database */
    tURL fileName;              /**< URL of file */
    tFileFormat fileFormat;     /**< file format like FFT_MP3, FFT_AVI, FFT_GIF */
    tIsPlaying isPlaying;       /**< is currently playing? */
    tNotPlayable notPlayable;   /**< FNP_PLAYABLE, FNP_DRM_PROTECTED, FNP_FORMAT_ERROR, ... (Roadmap 13012) */
    int isEOF;                  /**< marker for list end (all other elements have no meaning) */
    tMetadata metaData1;        /**< album name */
    tMetadata metaData2;        /**< artist name */
    tMetadata metaData3;        /**< title */
    tMetadata metaData4;        /**< spare */
    tPlaytime playTime;         /**< playtime in milliseconds */
    tFileSize fileSize;         /**< fileSize file size */
    tDateTime dateTime;         /**< dateTime created date and time*/
    tAlbumArt coverArtString;   /**< path to a cover art image on the flash*/
} tFiles;

typedef tFiles *tFilesPtr;
#define tFilesPtr_format "p"
#define tFilesPtr_init 0

void InitFiles(tFiles &file);

/**
 * Type declaration for a letter in the vertical keyboard
 */
typedef struct {
    char letter[10];
    tBoolean available;
    tUInt startIndex;
    tUInt endIndex;
} tSearchKeyboard;

/**
 * Type declaration for a search letter in list along with its occurance
 */
typedef struct
{
    char letter[10];
    tUInt count;
}tSearchKey;

/**
 * Type declaration of a list
 */
typedef struct
{
    tListID listID;
    tListID parent;
    tListID next;
    tListType listType;
    tStreaming streaming; //Roadmap 13008
    tListSize listSize;
    tDeviceID deviceID; /* for which device was this list created */
    tPlayContinuation playContinuation;
    tFilterTag1 filterTag1;
    tFilterTag2 filterTag2;
    tFilterTag3 filterTag3;
    tFilterTag4 filterTag4;
    tPath path;
    tFileTypeSelection fileTypeSelection;
    int isVTIPODList;
} tListInfo;
void InitListInfo(tListInfo &listInfo);

/**
 * Type declaration of a media object
 */
typedef struct
{
    tAlbumArt albumArtString;
    tCategoryType catType;              /**< category type (e.g. CTY_SONG or CTY_ARTIST) */
    tDeviceID deviceID;                 /**< id of device this object is stored */
    tDeviceType deviceType;             /**< type of device for this object*/
    tDeviceVersion deviceVersion;
    tFileFormat fileFormat;
    tURL fileName;                      /**< complete name file combined from DB: mountPoint and pure file name*/
    tIsPlaying isPlaying;               /**< is currently playing? */
    tMediaType mediaType;               /**< type of medium: video, song, podcast, ... */
    tMetadata MetadataField1;
    tMetadata MetadataField2;
    tMetadata MetadataField3;
    tMetadata MetadataField4;
    tUInt MetadataTag1;
    tUInt MetadataTag2;
    tUInt MetadataTag3;
    tUInt MetadataTag4;
    tTitle title;
    tObjectID objectID;                 /**< real object id from database */
    tPlaytime totalPlaytime;            /**< total play time of this media object in milliseconds */
    tTrackNumber trackNumber;

    //tPlaytime elapsedPlaytime;          /**< Obsolete: please don not use anymore */
    tMountPoint mountPoint;             /**< name of mount point of iPod or USB device */
    tFileType fileType;                 /**< type of file */
    tUUID UUID;                         /**< unique ID for a playable object on iPod */
    tUUID parentUUID;                   /**< unique ID for parant object, see playlist support iPod */
    tIndex active;                      /**< position (index) of active object in VTiPod list (only set in LTY_CURRENT_SELECTION) */
    tBitRate bitRate;                   /**< bit rate (Roadmap 13001) */
    tVBR vbr;                           /**< VBR flag (Roadmap 14011) */
    me::samplerate_i sampleRate;        /**< sample rate (Roadmap 13001) */
    tAudioChannelFormat audioChannelFormat; /**< format of audio channel (MONO or STEREO or ...) (Roadmap 13001) */
    tPath path;                         /**< path to a file object (Roadmap 13012) */
    tNotPlayable notPlayable;           /**< FNP_PLAYABLE, FNP_DRM_PROTECTED, FNP_FORMAT_ERROR, ... (Roadmap 13012) */
    tConvertFlag metadataConvertFlag;
    tCompilation compilationFlag;
    tFileSize fileSize;                 /**< file size in Bytes (Roadmap 16011) */
    tDateTime dateTime;                 /**< last file modification date and time in Y:m:d H:M:S (Roadmap 16011) */
    tFileMode fileMode;                 /**< file mode - permission triad for user, group and others (Roadmap 16011) */
    tUserID userID;                     /**< UID of the file - root=0 (Roadmap 16011) */
    tGroupID groupID;                   /**< GID of the file - root=0 (Roadmap 16011) */
    tAlbumArt coverArtString;           /**< path to a cover art image on the flash (Roadmap 15009) */
    tImageBlob thumbnailBlob;           /**< size and data of an album art thumbnail (Roadmap 15009) */
    tUInt year;
    tUInt yearID;
    tIsFavorite isFavorite;             /**< if an object is stored as a favorite for the current user or not */
    tMetadata YomiTitle;                /**<  Roadmap 16003 - Yomi metadata. The yomi title will be only filled if it belongs to music box device*/
    tMetadata YomiArtist;               /**<  Roadmap 16003 - Yomi metadata. The yomi artist will be only filled if it belongs to music box device*/
    tMetadata YomiAlbum;                /**<  Roadmap 16003 Yomi metadata. The yomi album will be only filled if it belongs to music box device*/
} tMediaObject;
void InitMediaObject(tMediaObject &mediaObject);

/**
 * Type declaration for a pointer to a media object
 */
typedef tMediaObject *tMediaObjectPtr;
#define tMediaObjectPtr_format "p"
#define tMediaObjectPtr_init 0

/**
 * Type declaration of a Image object
 */
#if USE_IMAGE_OBJECT
typedef struct
{
    tFileType type;             /**< FT_FOLDER, FT_IMAGE*/
    tTitle folderTitle;
    tNumberOfMediaObjects numberofImages;
#if IMAGE_ATTRIBUTES_IN_METADATA
    tMetadata imageSize;          /**< Represents Image size in bytes */
    tMetadata dateTime;          /**< Represents Date taken and Time */
#else
    tFileSize fileSize;
    tDateTime dateTime;
#endif
    tMetadata dimension;          /**< Represents Dimension */
    tMetadata resolution;          /**< Represents Resolution */
    tFileFormat fileFormat;            /**< format of file (e.g FFT_PNG or FFT_TIF) */
    tObjectID objectID;                /**< real object id from database */
    tTitle imageTitle;                      /**< Name of the image file */
    tPath path;                        /**< path to a file object */
    tURL imageFile;                     /**< path with filename */
    tDeviceID deviceID;                /**< id of device this object is stored */
    tMountPoint mountPoint;            /**< name of mount point of iPod or USB device */
} tImageObject;
void InitImageObject(tImageObject &ImageObject);
#endif

/**
 * Information about the currently played media object and the current
 * active list ID in the player manager
 */
typedef struct
{
    tListID listID;             /**< unique id of the list stored in the database */
    tNowPlayingState state;     /**< state of the active playlist upon transitions between items in the list */
    tPosition position;         /**< position (or index) of the item within the specified list starting at 0 for the first element */
    tObjectID objectID;         /**< real object id from database */
    tMediaObject object;        /**< media object */
} tNowPlaying;

/**
 * Information about the currently shown image media object
 */
typedef struct
{
    tListID listID;             /**< unique id of the list stored in the database */
    tNowPlayingState state;     /**< state of the active playlist upon transitions between items in the list */
    tPosition position;         /**< position (or index) of the item within the specified list starting at 0 for the first element */
    tMediaObject object;        /**< media object */
    tFilename nextFile;         /**< probably next shown image file */
} tNowShowing;
void InitNowShowing(tNowShowing &nowShowing);

/**
 * Type declaration for an indexing context
 */
typedef struct {
    tDeviceID deviceID;                 /**< device id of device to index */
    tDeviceType deviceType;             /**< device type of device to index */
    tMountPoint mountPoint;             /**< mount point of device and start point for first item to index */
    tIndexingMode indexingMode;         /**< full indexing scan or scan has be continue */
    tFingerprint fingerprint;           /**< calculated fingerprint */
    tNumberOfFiles totalNumberOfFiles;  /**< total number of files to index */
    tNumberOfFiles indexedFileCount;    /**< number of files already indexed */
    tNumberOfFiles skippedFileCount;    /**< number of skipped files */
    tNumberOfFiles errorSkipCount;      /**< number of skipped files due to an error */
    tNumberOfFiles writtenFileCount;    /**< number of written files */
    tNumberOfFiles writtenFileCountForVideo; /**< number of written video files */
    tNumberOfFiles writtenFileCountForImage; /** < number of written image files */
    tReadPosition readPosition;         /**< position (URL) of next media object to read */
    tIndexingPercentComplete percent;   /**< percentage of indexing complete */
    tBoolean flushFirstPlayableObject;  /**< is first playable object already flushed to DB */
    tBoolean flushFirstPlayableVideoObject;  /**< is first playable video object already flushed to DB */
    tBoolean flushFirstPlayableImageObject;  /**< is first playable image object already flushed to DB */
} tIndexingContext;
void InitIndexingContext(tIndexingContext &indexingContext);

/**
 * Type declaration for an indexing information for a given device
 */
typedef struct
{
    tDeviceID deviceID;                 /**< device id of device on which the media objects are stored */
    tIndexingState state;               /**< state of indexing operation */
    tIndexingPercentComplete percent;   /**< percentage completion of indexing operation */
} tIndexingStateResult;
void InitIndexingStateResult(tIndexingStateResult &indexingStateResult);

/**
 * Type declaration for an album art indexing context
 * Roadmap 15009_CoverArtFlow
 */
typedef struct {
    tDeviceID deviceID;                     /**< device ID of device to index */
    tListID listID;                         /**< list ID of album art indexing list */
    tIndex startIndex;                      /**< start index of first element to index */
    tIndex sliceSize;                       /**< number of elements to index */
    tBoolean interimList;                   /**< interim list due to focus change */
    vector<tMediaObject> mediaObjectVector; /**< vector of media objects to index */
    tIndex objectIndex;                     /**< current index in media object vector*/
} tAlbumArtIndexingContext;
void InitAlbumArtIndexingContext(tAlbumArtIndexingContext &indexingContext);

/**
 * Type declaration for a directory context (used by tScanContext)
 */
typedef struct {
    tURL path;      /**< directory path */
    tListID listID; /**< list ID of this directory */
    tRowNumber row; /**< scan position of next object to scan in the list of this directory */
} tDirectoryContext;
void InitDirectoryContext(tDirectoryContext &directoryContext);

typedef enum
{
    AVAILABILITY_UNKNOWN = 0,
    AVAILABLE,
    NOT_AVAILABLE,
    IGNORE_AVAILABILITY_CHECK
}tScanStatus;

/**
 * Type declaration for a scan context of a device
 */
typedef struct {
    tDeviceID deviceID;                     /**< device ID of device to scan */
    tFingerprint fingerprint;               /**< calculated fingerprint */
    tFingerprint lastFingerprint;               /**< last stored fingerprint */
    tFingerprintStatus fingerprintStatus;   /**< fingerprint status*/
    tNumberOfFiles numberOfFiles;           /**< total number of media files on device */
    tURL firstPath;                         /**> directory path of first element (define starting point) */
    tRowNumber firstElementRow;             /**> position in the list of first element (define starting point) */
    size_t deep;                             /**< current deep level of scanning */
    vector<tDirectoryContext> dirContext;   /**< vector of directory context for each deep level */
    //vector<tURL> subdir;                  /**< vector of sub directories left to scan (only needed for flat scan) */
    tScanStatus firstAudioFileScanStatus;   //Status of subjecting first audio File for indexing
    tURL firstScannedAudioFileName;         //Name of first audio File subjected to indexing
} tScanContext;
void InitScanContext(tScanContext &scanContext);

/**
 * Type declaration for a user context
 */
typedef tUInt tSourceAppID;
#define tSourceAppID_format "i"
#define tSourceAppID_init 0

typedef tUInt tRegisterID;
#define tRegisterID_format "i"
#define tRegisterID_init 0

typedef tUInt tCmdCounter;
#define tCmdCounter_format "i"
#define tCmdCounter_init 0

typedef tUInt tFunctionID;
#define tFunctionID_format "i"
#define tFunctionID_init 0

typedef tUInt tServiceID;
#define tServiceID_format "i"
#define tServiceID_init 0

typedef struct
{
    tSourceAppID sourceAppID;
    tRegisterID registerID;
    tCmdCounter cmdCounter;
    tFunctionID functionID;
    tServiceID serviceID;
} tUserContext;
void InitUserContext(tUserContext &UserContext);

/**
 * Type declaration for a play context
 */
typedef struct
{
    tDeviceType deviceType;
    tDeviceID deviceID;
    tURL URL;
    tUUID uuid;
    tMountPoint mountPoint;
    tPEHandle handle;       //Roadmap 13010
    tPlaytime position;
    tStreaming streaming;    //Roadmap 13008
} tPlayContext;

/**
 * Type declaration of a favorite object
 */
typedef struct
{
    tFavoriteID favoriteId;         /**< unique id of the favorite stored in the database */
    tCategoryType objectType;       /**< type of the database object (e.g. genre, artist, album, song, etc.) */
    tObjectID objectID;             /**< global object id from database (i.e. ROWID) */
    tLabelText labelText;           /**< label text of the favorite stored in the database */
    tDescriptorText descriptorText; /**< descriptor text of the favorite stored in the database */
    tActive active;                 /**< favorite is currently playing */
    tAvailable available;           /**< favorite is available for play back with its device connected */
} tFavoriteInfo;
void InitFavoriteInfo(tFavoriteInfo &favoriteInfo);

/**
 * Type for the information about a database object (e.g. artist, album, genre, etc.)
 */
typedef struct
{
    tGeneralString objectName;
    tCategoryType objectType;
    tGeneralString objectPath;
    tDeviceID deviceID;
} tObjectInfo;

/**
 * Type for one entry in the file type entry data base supported by the media player
 */
typedef struct
{
    tExtension extension;
    tFileFormat fileFormat;
    tFileType fileType;
} tFileTypeEntry;
void InitFileTypeEntry(tFileTypeEntry &fileTypeEntry);

/**
 * Type for one entry in the codec restriction table
 */
typedef struct
{
    tCodec codec;
    tVideoProfileName profile;
    tBoolean supported;
    tVideoProfileLevel maxLevel;
    tSize maxWidth;
    tSize maxHeight;
    tBitRate maxBitRate;
    tFrameRate maxFrameRate;
} tCodecRestriction;

/**
 * Type for config values from generic media player to player engine @@@
 */
typedef char tConfigString[1024];
#define tConfigString_format "t"

/**
 * Type for one entry in the application table
 */
typedef struct {
    tAppName appName;
    tSource source;
} tApplTableEntry;

/**
 * Type declaration for bytes
 */
typedef tS64 tPEBytes;
#define BYTES_NONE ((tPEBytes)-1)
#define tPEBytes_format "l"
#define tPEBytes_init BYTES_NONE

/**
 * Type declaration for percentage
 */
typedef tS64 tPEPercentage;
#define PERCENTAGE_NONE ((tPEPercentage)-1)
#define tPEPercentage_format "l"
#define tPEPercentage_init PERCENTAGE_NONE

/**
 * Type declaration for milliseconds
 */
typedef tS64 tPEMilliseconds;
#define MILLISECONDS_NONE ((tPEMilliseconds)-1)
#define tPEMilliseconds_format "l"
#define tPEMilliseconds_init MILLISECONDS_NONE

/**
 * Type declaration for the time position in a media object in different units
 * Roadmap 13010
 */
typedef struct {
    tPEBytes bytes;
    tPEPercentage pct;
    tPEMilliseconds ms;
} tPEPosition;

/**
 * Type declaration for the duration of a media object in different units
 * Roadmap 13010
 */
typedef struct {
    tPEBytes bytes;
    tPEMilliseconds ms;
} tPEDuration;

/**
 * Type declaration for a time information object of a media object
 * Roadmap 13010
 */
typedef struct {
    tPEPosition position;   /* elapsed playtime */
    tPEDuration duration;   /* total playtime */
} tPETimeInfoStruct;
void InitPETimeInfoStruct(tPETimeInfoStruct &info);

typedef char tPETimeInfo[128];
#define tPETimeInfo_format "t"

/**
 * Type declaration for video property value
 */
typedef tS16 tPropertyValue;
#define tVideoProperty_init VP_UNDEFINED
#define tPropertyValue_init 0
/**
 * Type declaration for video properties type
 */
typedef tU32 tVideoBrightness;
typedef tS32 tVideoHue;
typedef tU32 tVideoSaturation;
typedef tU32 tVideoContrast;
typedef tS32 tVideoBrightnessOffset;
typedef tS32 tVideoHueOffset;
typedef tU32 tVideoSaturationOffset;

/**
 * Type for the video properties - roadmap 15027
 */
typedef enum {
    VP_BRIGHTNESS = 0UL,
    VP_HUE = 1UL,
    VP_SATURATION = 2UL,
    VP_CONTRAST = 3UL,
    VP_BRIGHTNESS_OFFSET = 4UL,
    VP_HUE_OFFSET = 5UL,
    VP_SATURATION_OFFSET = 6UL,
    VP_ALL = 98UL,
    VP_UNDEFINED = 99UL
} tVideoProperty;

#define tVideoProperty_format "i"
#define tPropertyValue_format "i"

/**
 * Holds all the video properties with reference to database - roadmap 15027
 */
//typedef struct
//{
//    tVideoBrightness brightness;
//    tVideoHue hue;
//    tVideoSaturation saturation;
//    tVideoContrast contrast;
//    tVideoBrightnessOffset brightnessoffset;
//    tVideoSaturationOffset saturationoffset;
//    tVideoHueOffset hueoffset;
//} tMEVideoProperties;

//void InitVideoProperties(tMEVideoProperties &meVideoProperties);

/**
 * this type is for the player engine state machine. it holds the marshalled representation of tPEState
 */
typedef char tPEStateString[2048];
#define tPEStateString_format "t"

/*
    Type declarations for Roadmap 13009 AVRCP
    Start
 */
/**
 * Type declaration of an enum describing the Playback Control Commands
 * that can be Sent to the AVRCP 1.4 BT Device
 *
 */
typedef enum
{
    BT_CMD_UNUSED = 0,
    BT_CMD_PLAY,
    BT_CMD_STOP,
    BT_CMD_PAUSE,
    BT_CMD_NEXTTRACK,
    BT_CMD_PREVTRACK,
    BT_CMD_FORWARD,
    BT_CMD_REWIND,
    BT_CMD_RESERVED1,
    BT_CMD_RESERVED2,
    BT_CMD_MEDIASUSPEND,
    BT_CMD_MEDIARESUME
} tBTControlCmd;
#define tBTControlCmd_format "i"
#define tBTControlCmd_init BT_CMD_UNUSED

/**
 * Type declaration of an enum describing the status of the MethodRequest
 * Sent to the AVRCP 1.4 BT Device
 *
 */
typedef enum ALPS_METHOD_REQUEST_STATUS_TYPE_E
{
    REQ_OK = 0,
    REQ_NOT_OK,
    REQ_NOT_SUPPORT,
    REQ_ACTION_TIMEOUT,
    REQ_NOT_SENT_TO_DEVICE
}tBTMethodRequestStatus;
#define tBTMethodRequestStatus_format "i"
#define tBTMethodRequestStatus_init REQ_NOT_OK

/**
 * Type declaration of an enum describing the Button actions for a Playback Control Command
 * Sent to the AVRCP 1.4 Device
 *
 */
typedef enum
{
    BUTTON_PRESS = 0,
    BUTTON_RELEASE,
    BUTTON_CLICK,
    BUTTON_INVALID = 99
}tBTButtonEvent;

#define tBTButtonEvent_format "i"
#define tBTButtonEvent_init BUTTON_PRESS

/**
 * Type declaration of an enum describing the current playbackStatus
 * of the Device which is sent as a method return for
 * ControlCommand method Request sent to the AVRCP 1.4 Device
 *
 */
typedef enum
{
    BT_PS_STOPPED = 0,
    BT_PS_PLAYING,
    BT_PS_PAUSED,
    BT_PS_FWDSEEK,
    BT_PS_REVSEEK,
    BT_PS_ERROR  = 255
} tBTPlayStatus;
#define tBTPlayStatus_format "i"
#define tBTPlayStatus_init BT_PS_STOPPED

/**
 * Type declaration of an enum describing the Current PlaybackStatus of AVRCP 1.4 Device
 * It is Updated via a Control Command Indication Signal by the device.
 *
 */
typedef enum BT_PLAYER_CONTROL_STATUS_E
{
    BT_MEDIA_SUSPEND = 0x0A,
    BT_MEDIA_RESUME
}tBTPlayerControlStatus;

/**
 * Typedef for the scope in which the AVRCP 1.4 Browse Command sent to the BT device
 */
typedef enum
{
    SCOPE_MP_VFS = 1,
    NOW_PLAYING = 3,
    SCOPE_INVALID = 4
}tScope;
#define tScope_format "i"
#define tScope_init SCOPE_MP_VFS

#define ALPS_EVENTS_SUPPORTED  0x01

/**
 * Typedef for the Events(ie Notifications) supported by the AVRCP 1.4 device
 */
typedef tUInt tEventsSupported;
#define tEventsSupported_format "i"
#define tEventsSupported_init 0


/**
 * Typedef for the number of Metadata Attributes of each MediaElement
 */
typedef tUInt tNumAttributes;

/**
 * Typedef for the Database Aware/Unaware feature supported by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tUInt tDBAware;
#define tDBAware_format "i"
#define tDBAware_init 0

/**
 * Typedef for the Capabilities(ie Events Supported) of the AVRCP 1.4 Device's MediaPlayer
 */
typedef tUInt tCapabilityId;
#define tCapabilityId_format "i"
#define tCapabilityId_init 0

/**
 * Typedef for the Attributes(Equaliser/Repeat/Shuffle/Scan) suppoted by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tUInt tAttributesBit;
#define tAttributesBit_format "i"
#define tAttributesBit_init 0

/**
 * Typedef for the Ids of the Attributes(Equaliser/Repeat/Shuffle/Scan) suppoted by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tU8 tAttributeId;

/**
 * Typedef for the current value of the Attributes(Equaliser/Repeat/Shuffle/Scan) suppoted by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tUInt tCurrentValue;
#define tCurrentValue_format "i"
#define tCurrentValue_init 0

/**
 * Typedef for the value set to the Attribute(Equaliser/Repeat/Shuffle/Scan) suppoted by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tU8 tAttributeValue;

/**
 * Typedef for CodecType supported by the AVRCP 1.4 Device
 */
typedef enum
{
    CODEC_TYPE_SBC = 0,
    CODEC_TYPE_MPEG12AUDIO,
    CODEC_TYPE_MPEG24AAC,
    CODEC_TYPE_INVALID
}tCodecType;
#define tCodecType_format "i"
#define tCodecType_init CODEC_TYPE_INVALID

/**
 * Typedef for Number of Players  suppoted by the AVRCP 1.4 Device
 */
typedef tU8 tNumOfPlayers;

/**
 * Typedef for UID of the MediaElement or the FolderItem present in the
 * Virtual FileSystem  of the AVRCP 1.4 Device's MediaPlayer
 */
typedef tU8* tItemUID;
#define tItemUID_format "p"
#define tItemUID_init NULL

/**
 * Typedef for UIDs of the FolderItem present in the
 * Virtual FileSystem  of the AVRCP 1.4 Device's MediaPlayer
 */
typedef tU8* tItemUIDArray;
#define tItemUIDArray_format "p"
#define tItemUIDArray_init NULL

/**
 * Typedef for ByteArray sent as the response for
 * BrowseCommand requests and MetadataAttributes of a mediaElement request
 */
typedef unsigned char *tByteArray;
#define tByteArray_format "p"
#define tByteArray_init NULL

/**
 * Typedef for list of Attributes sent as the response for
 * BrowseCommand requests and MetadataAttributes of a mediaElement request
 */
typedef tUInt* tAttributeArray;
#define tAttributeArray_format "p"
#define tAttributeArray_init NULL

/**
 * Typedef for NetworkOrder(ie Endianess) of the byteArray
 * sent as the response for AVRCP 1.4 Method Requests
 *
 */
typedef tBoolean tNetworkOrder;
#define tNetworkOrder_format "i"
#define tNetworkOrder_init 0

/**
 *Typedef for the Starting item that need to be returned for GetFolderItem request
 *
 */
typedef tUInt tStartItem;
#define tStartItem_format "i"
#define tStartItem_init 0
/**
 *Typedef for the Enditem that need to be returned for GetFolderItem request
 *
 */
typedef tUInt tEndItem;
#define tEndItem_format "i"
#define tEndItem_init 0

/**
 * Typedef for the Number of Level Changes(Levelup and Down)
 * need to be Done on the Virtual FileSystem of the AVRCP 1.4 Device's MediaPlayer
 *
 */
typedef tUInt tLevelCount;
#define tLevelCount_format "i"
#define tLevelCount_init 0

/**
 * Typedef for Lenght of the ByteArray sent as the response for
 * BrowseCommand requests and MetadataAttributes of a mediaElement request
 */
typedef tUInt tArrayLength;
#define tArrayLength_format "i"
#define tArrayLength_init 0


/**
 * Typedef for ErrorMessage of each D-Bus AVRCP Method Requests.
 *
 */
typedef char *tErrorMessage;
#define tErrorMessage_format "p"
#define tErrorMessage_init NULL

/**
 * Typedef for the Id of the MediaPlayer found in the AVRCP 1.4 Device
 *
 */
typedef tUInt tPlayerId;
#define tPlayerId_format "i"
#define tPlayerId_init 0
#define INVALID_PLAYER_ID ((tPlayerId)-1)


/**
 * Typedef for the Ids of the  Events sents by AVRCP 1.4 Device's MediaPlayer
 *
 */
typedef tUInt tEventId;
#define tEventId_format "i"
#define tEventId_init 0

/**
 * Typedef for the mode in which AVRCP 1.4 Device's MediaPlayer is used
 *
 */
typedef enum
{
    ADDRESSED_PLAYER = 1,
    BROWSED_PLAYER,
    INVALID_PLAYER_MODE
}tPlayerMode;
#define tPlayerMode_format "i"
#define tPlayerMode_init ADDRESSED_PLAYER


/**
 * Typedef for the Audio Input PipeName of AVRCP 1.4 Device
 *
 */
typedef unsigned char *tPipeName;
#define tPipeName_format "p"
#define tPipeName_init NULL

/**
 * Typedef for the Bluetooth Device address
 */
typedef char tDeviceAddress[6];

/**
 * Error numbers received on Select call from platform
 *
 */
#define SELECT_EINTR            4      /* Interrupted system call */
#define SELECT_EBADF            9      /* Bad file number */

/**
 * Type declaration of an enum describing the Notifications that will be sent by the AVRCP 1.4 Device
 */
typedef enum tenAlpsAvpNotifyEvents
{
    ALPS_EVENT_PLAYBACK_STATUS_CHANGED = 1,
    ALPS_EVENT_TRACK_CHANGED,
    ALPS_EVENT_TRACK_REACHED_END,
    ALPS_EVENT_TRACK_REACHED_START,
    ALPS_EVENT_PLAYBACK_POS_CHANGED,
    ALPS_EVENT_BATT_STATUS_CHANGED,
    ALPS_EVENT_SYSTEM_STATUS_CHANGED,
    ALPS_EVENT_PLAYER_APPLICATION_SETTING_CHANGED,
    ALPS_EVENT_NOW_PLAYING_CONTENT_CHANGED,
    ALPS_EVENT_AVAILABLE_PLAYERS_CHANGED,
    ALPS_EVENT_ADDRESSED_PLAYER_CHANGED,
    ALPS_EVENT_UIDS_CHANGED,
    /* To adapt Evo changes*/
    ALPS_EVENT_REPEAT_CHANGED,
    ALPS_EVENT_SHUFFLE_CHANGED
}tBTAvpNotifyEvent;

/**
 * Type declaration of an enum describing the id of the attributes supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum tenAlpsPlayerSettingsAttributes
{
    ALPS_ATTR_EQUALIZER = 1,
    ALPS_ATTR_REPEAT,
    ALPS_ATTR_SHUFFEL,
    ALPS_ATTR_SCAN,
    ALPS_ATTR_ERROR
}tBTPlayerSettingAttributeMode;
#define tBTPlayerSettingAttributeMode_format  "i"
#define tBTPlayerSettingAttributeMode_init ALPS_ATTR_ERROR

/**
 * Type declaration of the enum for the BitFlags describing the attributes supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum tenAlpsPlayerSettingsBitFlags
{
    BITFLAG_EQUALIZER = 0x01,
    BITFLAG_REPEAT    = 0x02,
    BITFLAG_SHUFFEL   = 0x04,
    BITFLAG_SCAN      = 0x08
}tBTPlayerSettingsBitFlag;

/**
 * Type declaration of the enum describing the values for the EqualizerMode supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum tenAlpsEqualizerMode
{
    ALPS_EQUALIZER_MODE_OFF = 1,
    ALPS_EQUALIZER_MODE_ON
}tBTEqualizerMode;

/**
 * Type declaration of the enum describing the values for the RepeatMode supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum tenAlpsRepeatMode
{
    ALPS_REPEAT_OFF = 1,
    ALPS_REPEAT_SINGLE,
    ALPS_REPEAT_ALL,
    ALPS_REPEAT_GROUP,
    ALPS_REPEAT_INVALID
}tBTRepeatMode;

/**
 * Type declaration of the enum describing the values for the ShuffleMode supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum tenAlpsShuffleMode
{
    ALPS_SHUFFLE_OFF = 1,
    ALPS_SHUFFLE_ALL,
    ALPS_SHUFFLE_GROUP,
    ALPS_SHUFFLE_INVALID
}tBTShuffleMode;

/**
 * Type declaration of the enum describing the values for the ScanMode supported by AVRCP 1.4 Device's MediaPlayer
 */
typedef enum temAlpsScanMode
{
    ALPS_SCAN_OFF = 1,
    ALPS_SCAN_ALL,
    ALPS_SCAN_GROUP
}tBTScanMode;

/**
 * Structure used to Parse the ByteArray of GetMediaPlayerItem response and Retrieve each MediaPlayerItem Information
 */
struct MediaPlayer_Item_info      //11 Bytes
{
    char    Item_Type;         //1
    short   Item_Length ;      //2
    short   Player_ID;         //2
    char    Major_Player_Type; //1
    int     Player_Sub_Type;   //4
    char    PlayBackStatus;    //1
}__attribute__((packed));
typedef struct MediaPlayer_Item_info tMediaPlayerItemInfo;

/**
 * Structure used to Parse the ByteArray of GetFolderItem response and Retrieve each FolderItem Information
 */
struct FolderItem_Item_info   //16 Bytes
{

    unsigned short   Item_Length ;  //2
    unsigned char    Item_UID[8];   //8
    unsigned char    Item_Sub_Type; //1
    unsigned char   IsPlayable;     //1
    unsigned short   Char_Set_ID;   //2
    unsigned short   Item_Name_Length; //2
}__attribute__((packed));
typedef struct FolderItem_Item_info tFolderItemInfo;

/**
 * Structure used to Parse the ByteArray of GetFolderItem response and Retrieve each MediaElement Information
 */
struct MediaElement_Item_info  //15 Bytes
{
    unsigned short   Item_Length ;  //2
    unsigned char    Item_UID[8];   //8
    unsigned char    Item_Sub_Type; //1
    unsigned short   Char_Set_ID;   //2
    unsigned short   Item_Name_Length; //2
}__attribute__((packed));
typedef struct MediaElement_Item_info tMediaElementItemInfo;

/**
 * Typedef for the Number of Items in the Current Browsed Path
 * of the Virtual FileSystem managed by the AVRCP 1.4 Device's MediaPlayer
 */
typedef tUInt tNumOfItems;
#define tNumOfItems_format "i"
#define tNumOfItems_init 0

typedef enum
{
    FOLDER_UP = 0,
    FOLDER_DOWN,
    LEVEL_CHANGE_NOT_NEEDED,
    LEVEL_CHANGE_INVALID
}tDirection;
#define tDirection_format "i"
#define tDirection_init FOLDER_UP

#define SIZE_OF_ITEM_UID 8
#define SIZE_OF_ATTRIBUTE_ARRAY 7

typedef enum
{
    AUDIO_FILE = 0,
    VIDEO_FILE,
    FOLDER,
    MEDIA_ELEMENT,
    INVALID
}tBTItemType;

typedef enum
{
    BT_ATTRIBUTE_TITLE = 1,
    BT_ATTRIBUTE_ARTIST,
    BT_ATTRIBUTE_ALBUM,
    BT_ATTRIBUTE_TRACK_NUM,
    BT_ATTRIBUTE_GENRE = 6,
    BT_ATTRIBUTE_MEDIA_LENGTH
}tBTAttribute;

typedef void *tVoidPtr;
#define tVoidPtr_format "p"
#define tVoidPtr_init 0

typedef tUInt tIndexSession;
#define tIndexSession_format "i"
#define tIndexSession_init 0

typedef enum tenMetadataConversionBitFlag
{
    BITFLAG_METADATAFIELD1 = 0x01,
    BITFLAG_METADATAFIELD2 = 0x02,
    BITFLAG_METADATAFIELD3 = 0x04,
    BITFLAG_METADATAFIELD4 = 0x08,
    BITFLAG_TITLE          = 0x10
} tMetadataConversionBitFlag;

typedef enum
{
    AVRCP_NO_METADATA = 0UL,
    AVRCP_METADATA = 1UL,
    AVRCP_BROWSING = 2UL,
    AVRCP_MYMEDIA = 3UL
} tBTAvrcpVersion;

typedef tBoolean tScanMode;
#define tScanMode_format "i"
#define tScanMode_init 0

/**
 * Structure to find/remove device by Serial Number
 * SerialNumber - UID of partition
 * AccessoryName2 - UID of device
 * removeOthers = true --> Remove all devices with accessoryName2 = AccessoryName2 other than the device with serialNumber = SerialNumber
 * countPartition = true --> Count all devices with accessoryName2 = AccessoryName2
 * if removeOthers = false and countPartition = false then find the device with serialNumber = SerialNumber
 * removeOthers and countPartition cannot be simultaneously true
 */
struct FindDeviceBySerialNumber
{
    tDeviceSerialNumber mSerialNumber;
    tAccessoryName2 mAccessoryName2;
    tBool mRemoveOthers;
    tBool mRemoveAll;
    FindDeviceBySerialNumber(const tDeviceSerialNumber SerialNumber,
            const tAccessoryName2 AccessoryName2,
            tBool removeOthers = false,
            tBool RemoveAll = false) : mRemoveOthers(removeOthers),mRemoveAll(RemoveAll)
    {

        strncpy_r(mSerialNumber,SerialNumber,sizeof(tDeviceSerialNumber)); //CID 17332
        strncpy_r(mAccessoryName2,AccessoryName2,sizeof(tAccessoryName2)); //CID 17332

    }
    bool operator()(const tDeviceInfo& deviceInfo) const
    {
      if(mRemoveAll) //RemoveAll
      {
          return strcmp(deviceInfo.accessoryName2,mAccessoryName2)==0;
      }
      else if(mRemoveOthers) //Remove other than
      {
          return (!strcmp(deviceInfo.serialNumber,mSerialNumber)==0) && (strcmp(deviceInfo.accessoryName2,mAccessoryName2)==0);
      }
      else //Find
      {
          return strcmp(deviceInfo.serialNumber,mSerialNumber)==0;
      }
    }
};
typedef struct FindDeviceBySerialNumber tFindDeviceBySerialNumber;

#define PATH_DELIMITER "/"
#define NOW_PLAYING_PATH "/NowPlaying"
typedef enum {
    STORE_VALID_PLAY_TIME = 0,
    COMPARE_PLAY_TIME,
    REMOTE_ACTIVITY_REQUEST_POSTED,
    NO_ACTION_NEEDED
}tBTActionOnPlayTime;
/**
 * Typedef for the Siri availability status
 */
typedef enum
{
    SIRI_NOT_AVAILABLE = 0,
    SIRI_AVAILABLE_ENABLED,
    SIRI_AVAILABLE_DISABLED,
    SIRI_AVAILABILITY_UNKNOWN
}tSiriStatus;
#define tSiriStatus_format "i"
#define tSiriStatus_init SIRI_AVAILABILITY_UNKNOWN

/**
 * Type declaration for iPod Communication Errors
 */
typedef enum
{
    IPOD_NO_ERROR = 0,
    IPOD_COMM_ERROR,
    IPOD_ROLESWITCH_ERROR
}tIpodCommError;

/**
 * Type declaration for a flag if VoiceRecognitionSession is active or not
 */
typedef tBoolean tVRSessionStatus;
#define tVRSessionStatus_format "i"
#define tVRSessionStatus_init 0

typedef tBoolean tVRSupportStatus;
#define tVRSupportStatus_format "i"
#define tVRSupportStatus_init 0

#define INVALID_PATH -1

#define CODESET_CONVERSION_TRACES_NEEDED  0

typedef struct {
    string objectPath;
    tBoolean isBrowsable;
    tBoolean isSearchable;
    tBoolean isDBAware;
    tBoolean isSetBrowsedPlayerSuccess;
} tBTPlayerInfo;

typedef struct {
    tBoolean AttributeSupported;
    int value;
} tBTAttributeData;

typedef struct
{
    tBTAttributeData Shuffle;
    tBTAttributeData Repeat;
}tPlayerAttributes;

void InitPlayerAttributes(tPlayerAttributes &BTSupportAttributes);
void InitBTPlayerInfo(tBTPlayerInfo &BTPlayerInfo);

/*For Ignoring Articles 'A','An,'The' from metadata- RNAIVI*/
#define SPECIALWORDSCOUNT   3

const string SpecialWordsTable[SPECIALWORDSCOUNT] =
{
        "A ",
        "An ",
        "The "
};

typedef struct
{
    int taskId;
    char *parameters;
}tTaskContext;

typedef unsigned char tUID[8];
typedef struct
{
    string name;
    tUID   uid;
    tBTItemType type;
    tBoolean isPlayable;
    string objectPath;
    string player;
}tBrowseElement;

typedef unsigned char tEndPoint;
typedef tU8   tEndPointDirection;
typedef tU8*  tCMDDataBlockPtr;
typedef tU8   tCMDDataBlockLength;
typedef tU16  tDataTransferLength;
typedef unsigned char tOpInfoData[0x3f];

typedef struct
{
    tU8*  cmd;
    tU32 dataLength;
    tResult  result;
    bool keyResponseRequired;
}tCMDDataStruct;


typedef enum
{
    DIPO_BATTERY_CHARGE_STATE_DISABLED = 0UL,  // default
    DIPO_BATTERY_CHARGE_STATE_CHARGING,
    DIPO_BATTERY_CHARGE_STATE_CHARGED
} tDiPOBatteryChargingState;
#define tDiPOBatteryChargingState_format "i"
#define tDiPOBatteryChargingState_init DIPO_BATTERY_CHARGE_STATE_DISABLED

typedef enum
{
    DIPO_ACCESSORY_MODE_RESERVED = 0UL,  // default
    DIPO_ACCESSORY_MODE_LOW_POWER_MODE,
    DIPO_ACCESSORY_MODE_INTERMITTENT_HIGH_POWER_MODE
} tDiPOAccessoryPowerMode;
#define tDiPOAccessoryPowerMode_format "i"
#define tDiPOAccessoryPowerMode_init DIPO_ACCESSORY_MODE_RESERVED

typedef struct
{
    tDiPOBatteryChargingState BatteryChargingState;
    tDiPOAccessoryPowerMode AccessoryPowerMode;
    tU16 BatteryChargeLevel;
    tU16 MaximumCurrentDrawnFromAccessory;
    tBoolean DeviceBatteryWillChargeIfPowerIsPresent;
    tBoolean IsExternalChargerConnected;
} tDiPOPower;
void InitDiPOPower(tDiPOPower &power);

#define CD_MAX_TRACK_NUMBER         99
//>>-----Roadmap 16003

/**
 * struct declaration for packing the user data
 */
typedef struct
{
    tListType listType;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;
    tDeviceID deviceID;
}tDeleteMediaByTagsParam;

/**
 * struct declaration for packing the user data
 */
typedef struct
{
    vector<tObjectID> objectIDs;
    tDeviceID deviceID;
}tDeleteMediaByObjectIDParam;

/**
 * struct declaration for packing the user data
 */
typedef struct
{
    vector<string> objectUrl;
    tDeviceID deviceID;
}tDeleteMediaByUrlParam;

/**
 * struct declaration for packing the user data
 */
typedef struct
{
    tObjectID objectID;
    tBool readOnly;
    tURL url;
}tFilePermission;
typedef struct
{
    tDeviceID deviceID;
    vector<tFilePermission> objectIDs;
}tSetFilePermission;

/**
 * enum declaration for FileOperation(EditMetadata and DeleteMediaContent Errors
 */
typedef enum
{
    FILE_OPERATION_OK,
    FILEPERM_ISSUE,
    DATABASE_ERROR,
    DEVICE_NOT_EXISITS,
    LISTTYPE_NOT_FOUND,
    FILE_OPERATION_ERROR,
    DEVICE_ERROR,
    OBJECT_NOT_FOUND,
    TAG_EDIT_ERROR
}tFileOperationErrorCodes;

/**
 * enum declaration for DeleteMediaContent Types
 */
typedef enum
{
    DELETEMEDIACONTENTBYFILERTAGS,
    DELETEMEDIACONTENTBYMEDIAOBJECTTAGS,
    DELETEMEDIACONTENTBYURLS,
    SETFILEPERMISSIONBYMEDIAOBJECTTAGS
}tDeleteMediaContentType;

/**
 * struct declaration for packing the data for EditMetaData
 */
typedef struct
{
    tObjectID MediaObjectID;
    tMetadata GenreField;
    tMetadata AlbumField;
    tMetadata ArtistField;
    tMetadata TitleField;
    tMetadata ComposerField;
    tU16      YearField;
    tDeviceID deviceID;
    tYomiMetadata yomiMetaData;
}tEditMetaDataByID;

/**
 * struct declaration for packing the data for EditMetaData
 */
typedef struct
{
    tURL MediaObjectUrl;
    tMetadata GenreField;
    tMetadata AlbumField;
    tMetadata ArtistField;
    tMetadata TitleField;
    tMetadata ComposerField;
    tU16      YearField;
    tDeviceID deviceID;
    tYomiMetadata yomiMetaData;
}tEditMetaDataByUrl;



/**
 * enum declaration for EditMetaData Types
 */
typedef enum
{
    EDITMETADATABYMEDIAOBJECTTAG,
    EDITMETADATABYURL

}tEditMetadatatType;

/**
 * struct declaration for packing the data for PlayList Operation Add entries into playlist
 */
typedef struct
{
    tPath PlayListPath;
    vector<string> filenamePaths;
    vector<tPosition> ListOfPositions;
    tDeviceID deviceID;
}tPlayListAddOperation;
//For deleting all the entries
#define DELETE_ALL POSITION_NOT_SET

/**
 * struct declaration for packing the data for PlayList Operation Delete entries into playlist
 */
typedef struct
{
    tPath PlayListPath;
    vector<tPosition> ListOfPositions;
    tDeviceID deviceID;
}tPlayListDeleteOperation;

typedef enum
{
    Encoding_Quality_Low  = 32,//Low
    Encoding_Quality_FM   = 96,//FM
    Encoding_Quality_CD   = 128,//CD
    Encoding_Quality_high = 320//High

}tEncodingQuality;


typedef struct
{
    tListType listType;
    tFilterTag1 tag1;
    tFilterTag2 tag2;
    tFilterTag3 tag3;
    tFilterTag4 tag4;
    tPath playlistPath;
    tDeviceID deviceID;
}tPlayListAddOperationByTags;


//<<-----Roadmap 16003
/**
 * Type for the dvd properties
 */
typedef enum {
    DIRECTSELECTAVAILABLE= 0,
    DISCERROR,
    CMSKIPAVAILABLE,
    DISCMECHANICALINFO,
    DVDDRIVEINFO,
    DISPLAYMODE,
    DRCSETTING,
    DVDANGLEINFO,
    DVDAUDIOCHANNELINFO,
    DVDCURRENTAUDIOINFO,
    DVDPLAYERLANGSETTING,
    DVDSUBTITLEINFO,
    ANGLEMARKSETTING,
    PLAYSTATUSSETTING,
    KEYCOMMANDRESPONSE,
    DVDMENUPLAYBACKONGOING
}tDvdProperty;


#define CD_TRACK_ARRAY_SIZE (CD_MAX_TRACK_NUMBER+1)
typedef struct
{
  unsigned int u32StartZLBA;
  unsigned char u8AdrCtrl;
  tMetadata title;
  tMetadata performer;
  tPlaytime playTime;
} tCDTrackTOC;

typedef struct
{
    unsigned char u8Valid;
    unsigned char u8MinTrack;
    unsigned char u8MaxTrack;
    unsigned int u32LastZLBA;
    tMetadata albumName;
    tMountPoint m_MountPoint;
    tCDTrackTOC arTrack[CD_TRACK_ARRAY_SIZE]; /*+1: satisfy l i n t*/

} tCDTOCInfo;

/**
 * Type declaration for a pointer to a media object
 */
typedef tCDTOCInfo *tCDTOCInfoPtr;
#define tCDTOCInfoPtr_format "p"
#define tCDTOCInfoPtr_init 0


typedef enum
{
   RIPPING_NOT_SUPPORTED = 0UL,  // default
   RIPPING_NOT_STARTED,
   RIPPING_PARTIAL,
   RIPPING_COMPLETED
} tRippingState;
#define tRippingState_format "i"
#define tRippingState_init RIPPING_NOT_SUPPORTED


typedef enum
{
   RIPPING_NO_ERROR = 0UL,  // default
   RIPPING_TRACK_READ_ERROR,
   RIPPING_COPY_PROTECTED_DISC_ERROR,
   RIPPING_IO_ERROR,
   RIPPING_DISC_FULL_ERROR,
   RIPPING_PERMISSION_MISS_ERROR
} tRippingError;
typedef enum
{
    SPM_SYSPWRMODE_UNDEFINED = 0,
    SPM_SYSPWRMODE_OFF,
    SPM_SYSPWRMODE_ACCESSORY,
    SPM_SYSPWRMODE_RUN,
    SPM_SYSPWRMODE_CRANK
}tSPMSysPwrMode;


#define MUSICBOX_DEVICE_ID  1
/*Folder info for
 * total file size , protected file size
 * total file number , protected file number*/
typedef struct
{
    tU32 totalNumber;
    tU64 totalSize;
    tU32 protectedNumber;
    tU64 protectedSize;
} tFolderInfo;

typedef enum
{
   DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET = 0UL,  // default
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_SET,
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_ARRIVED,
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOADING,
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_LOCATING,
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_REROUTING,
   DIPO_ROUTE_GUIDANCE_STATE_ROUTE_PROCEEDTOROUTE
} tDiPORouteGuidanceState;
#define tDiPORouteGuidanceState_format "i"
#define tDiPORouteGuidanceState_init DIPO_ROUTE_GUIDANCE_STATE_NO_ROUTE_SET

typedef enum
{
   DIPO_MANEUVER_STATE_CONTINUE = 0UL,  // default
   DIPO_MANEUVER_STATE_INITIAL,
   DIPO_MANEUVER_STATE_PREPARE,
   DIPO_MANEUVER_STATE_EXECUTE
} tDiPOManeuverState;
#define tDiPOManeuverState_format "i"
#define tDiPOManeuverState_init DIPO_MANEUVER_STATE_CONTINUE

typedef enum
{
   DIPO_DISTANCE_DISPLAY_UNIT_KM = 0UL,  // default
   DIPO_DISTANCE_DISPLAY_UNIT_MILES,
   DIPO_DISTANCE_DISPLAY_UNIT_M,
   DIPO_DISTANCE_DISPLAY_UNIT_YARDS,
   DIPO_DISTANCE_DISPLAY_UNIT_FT
} tDiPoDistanceDisplayUnits;
#define tDiPoDistanceDisplayUnits_format "i"
#define tDiPoDistanceDisplayUnits_init DIPO_DISTANCE_DISPLAY_UNIT_KM

#ifdef IAP2_DEFAULT_ROUTEGUIDANCEDISPLAYCOMPONENT
typedef struct
{
    tU16 RouteGuidanceDisplayComponentID;
    tDiPORouteGuidanceState RouteGuidanceState;
    tDiPOManeuverState ManeuverState;
    tGeneralString CurrentRoadName;
    tGeneralString DestinationName;
    tU64 EstimatedTimeOfArrival;
    tU64 TimeRemainingToDestination;
    tU32 DistanceRemaining;
    tGeneralString DistanceRemainingDisplayStr;
    tDiPoDistanceDisplayUnits DistanceRemainingDisplayUnits;
    tU32 DistanceToNextManeuver;
    tGeneralString DistanceToNextManeuverDisplayStr;
    tDiPoDistanceDisplayUnits DistanceToNextManeuverDisplayUnits;
    tManeuverList RouteGuidanceManeuverCurrentList;
    tU16 RouteGuidanceManeuverCount;
    tBoolean RouteGuidanceVisibleInApp;
    tGeneralString SourceName;
    tBoolean SourceSupportsRouteGuidance;
} tDiPORouteGuidanceUpdate;
#else
typedef struct
{
    tBoolean RouteGuidanceDisplayComponentIDsAvailable;
    tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs;
    tBoolean RouteGuidanceStateAvailable;
    tDiPORouteGuidanceState RouteGuidanceState;
    tBoolean ManeuverStateAvailable;
    tDiPOManeuverState ManeuverState;
    tBoolean CurrentRoadNameAvailable;
    tGeneralString CurrentRoadName;
    tBoolean DestinationNameAvailable;
    tGeneralString DestinationName;
    tBoolean EstimatedTimeOfArrivalAvailable;
    tU64 EstimatedTimeOfArrival;
    tBoolean TimeRemainingToDestinationAvailable;
    tU64 TimeRemainingToDestination;
    tBoolean DistanceRemainingAvailable;
    tU32 DistanceRemaining;
    tBoolean DistanceRemainingDisplayStrAvailable;
    tGeneralString DistanceRemainingDisplayStr;
    tBoolean DistanceRemainingDisplayUnitsAvailable;
    tDiPoDistanceDisplayUnits DistanceRemainingDisplayUnits;
    tBoolean DistanceToNextManeuverAvailable;
    tU32 DistanceToNextManeuver;
    tBoolean DistanceToNextManeuverDisplayStrAvailable;
    tGeneralString DistanceToNextManeuverDisplayStr;
    tBoolean DistanceToNextManeuverDisplayUnitsAvailable;
    tDiPoDistanceDisplayUnits DistanceToNextManeuverDisplayUnits;
    tBoolean RouteGuidanceManeuverCurrentListAvailable;
    tManeuverList RouteGuidanceManeuverCurrentList;
    tBoolean RouteGuidanceManeuverCountAvailable;
    tU16 RouteGuidanceManeuverCount;
    tBoolean RouteGuidanceVisibleInAppAvailable;
    tBoolean RouteGuidanceVisibleInApp;
    tBoolean SourceNameAvailable;
    tGeneralString SourceName;
    tBoolean SourceSupportsRouteGuidanceAvailable;
    tBoolean SourceSupportsRouteGuidance;
} tDiPORouteGuidanceUpdate;
#endif
void InitDiPORouteGuidanceUpdate(tDiPORouteGuidanceUpdate &RGUpdate);
#define RGDISPLAYCOMPONENTIDS_NONE 0

typedef enum
{
   DIPO_MANEUVER_TYPE_NO_TURN = 0UL,  // default
   DIPO_MANEUVER_TYPE_LEFT_TURN,
   DIPO_MANEUVER_TYPE_RIGHT_TURN,
   DIPO_MANEUVER_TYPE_STRAIGHT_AHEAD,
   DIPO_MANEUVER_TYPE_MAKE_U_TURN,
   DIPO_MANEUVER_TYPE_CONTINUE,
   DIPO_MANEUVER_TYPE_ENTER_ROUNDABOUT,
   DIPO_MANEUVER_TYPE_EXIT_ROUNDABOUT,
   DIPO_MANEUVER_TYPE_OFF_RAMP,
   DIPO_MANEUVER_TYPE_ON_RAMP,
   DIPO_MANEUVER_TYPE_ARRIVE_END_OF_NAVIGATION,
   DIPO_MANEUVER_TYPE_PROCEED_TO_THE_BEGINNING_OF_ROUTE,
   DIPO_MANEUVER_TYPE_ARRIVED_AT_DESTINATION,
   DIPO_MANEUVER_TYPE_KEEP_LEFT,
   DIPO_MANEUVER_TYPE_KEEP_RIGHT,
   DIPO_MANEUVER_TYPE_ENTER_FERRY,
   DIPO_MANEUVER_TYPE_EXIT_FERRY,
   DIPO_MANEUVER_TYPE_CHANGE_TO_DIFFERENT_FERRY,
   DIPO_MANEUVER_TYPE_MAKE_U_TURN_AND_PROCEED_TO_ROUTE,
   DIPO_MANEUVER_TYPE_USE_ROUNDABOUT_TO_MAKE_U_TURN,
   DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_LEFT,
   DIPO_MANEUVER_TYPE_END_OF_ROAD_TURN_RIGHT,
   DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_LEFT,
   DIPO_MANEUVER_TYPE_HIGHWAY_OFF_RAMP_RIGHT,
   DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_LEFT,
   DIPO_MANEUVER_TYPE_ARRIVE_AT_DESTINATION_RIGHT,
   DIPO_MANEUVER_TYPE_MAKE_U_TURN_WHEN_POSSIBLE,
   DIPO_MANEUVER_TYPE_ARRIVE_END_OF_DIRECTION,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_1,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_2,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_3,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_4,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_5,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_6,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_7,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_8,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_9,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_10,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_11,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_12,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_13,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_14,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_15,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_16,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_17,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_18,
   DIPO_MANEUVER_TYPE_ROUNDABOUT_EXIT_19,
   DIPO_MANEUVER_TYPE_SHARP_LEFT_TURN,
   DIPO_MANEUVER_TYPE_SHARP_RIGHT_TURN,
   DIPO_MANEUVER_TYPE_SLIGHT_LEFT_TURN,
   DIPO_MANEUVER_TYPE_SLIGHT_RIGHT_TURN,
   DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY,
   DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_LEFT,
   DIPO_MANEUVER_TYPE_CHANGE_HIGHWAY_RIGHT
} tDiPoManeuverType;
#define tDiPoManeuverType_format "i"
#define tDiPoManeuverType_init DIPO_MANEUVER_TYPE_NO_TURN

typedef enum
{
   DIPO_DRIVING_SIDE_RIGHT = 0UL,  // default
   DIPO_DRIVING_SIDE_LEFT
} tDiPODrivingSide;
#define tDiPODrivingSide_format "i"
#define tDiPODrivingSide_init DIPO_DRIVING_SIDE_RIGHT

typedef enum
{
   DIPO_JUNCTION_TYPE_SINGLE_INTERSECTION = 0UL,  // default
   DIPO_JUNCTION_TYPE_ROUNDABOUT
} tDiPoJunctionType;
#define tDiPoJunctionType_format "i"
#define tDiPoJunctionType_init DIPO_JUNCTION_TYPE_SINGLE_INTERSECTION

#ifdef IAP2_DEFAULT_ROUTEGUIDANCEDISPLAYCOMPONENT
typedef struct
{
    tU16 RouteGuidanceDisplayComponentID;
    tU16 Index;
    tGeneralString ManeuverDescription;
    tDiPoManeuverType ManeuverType;
    tGeneralString AfterManeuverRoadName;
    tU32 DistanceBetweenManeuver;
    tGeneralString DistanceBetweenManeuverDisplayStr;
    tDiPoDistanceDisplayUnits DistanceBetweenManeuverDisplayUnits;
    tDiPODrivingSide DrivingSide;
    tDiPoJunctionType JunctionType;
    tU16 JunctionElementAngle;
    tU16 JunctionElementExitAngle;
} tDiPORouteGuidanceManeuverUpdate;
#else
typedef struct
{
    tBoolean RouteGuidanceDisplayComponentIDsAvailable;
    tRouteGuidanceDisplayComponentIDList RouteGuidanceDisplayComponentIDs;
    tBoolean IndexAvailable;
    tU16 Index;
    tBoolean ManeuverDescriptionAvailable;
    tGeneralString ManeuverDescription;
    tBoolean ManeuverTypeAvailable;
    tDiPoManeuverType ManeuverType;
    tBoolean AfterManeuverRoadNameAvailable;
    tGeneralString AfterManeuverRoadName;
    tBoolean DistanceBetweenManeuverAvailable;
    tU32 DistanceBetweenManeuver;
    tBoolean DistanceBetweenManeuverDisplayStrAvailable;
    tGeneralString DistanceBetweenManeuverDisplayStr;
    tBoolean DistanceBetweenManeuverDisplayUnitsAvailable;
    tDiPoDistanceDisplayUnits DistanceBetweenManeuverDisplayUnits;
    tBoolean DrivingSideAvailable;
    tDiPODrivingSide DrivingSide;
    tBoolean JunctionTypeAvailable;
    tDiPoJunctionType JunctionType;
    tBoolean JunctionElementAngleAvailable;
    vector<tU16> JunctionElementAngle;
    tBoolean JunctionElementExitAngleAvailable;
    tU16 JunctionElementExitAngle;
} tDiPORouteGuidanceManeuverUpdate;
#endif
void InitDiPORouteGuidanceManeuverUpdate(tDiPORouteGuidanceManeuverUpdate &RGManeuverUpdate);

typedef struct
{
    tU16 Identifier;
    tGeneralString Name;
    tU16 CurrentRoadNameLength;
    tU16 DestinationNameLength;
    tU16 AfterManeuverRoadNameLength;
    tU16 ManeuverDescriptionLength;
    tU16 GuidanceManeuverCapacity;
} tDiPORGDisplayComponentInfo;
void InitDiPOSetRGDisplayInfo(tDiPORGDisplayComponentInfo &RGDisplayInfo);
typedef vector<tDiPORGDisplayComponentInfo> tDiPORGDisplayComponentList;

typedef struct
{
    tU8 isInitiateCallAvailable:1;
    tU8 isAcceptCallAvailable:1;
    tU8 isEndCallAvailable:1;
    tU8 isSwapCallsAvailable:1;
    tU8 isMergeCallsAvailable:1;
    tU8 isHoldStatusUpdateAvailable:1;
    tU8 isMuteStatusUpdateAvailable:1;
    tU8 isSendDTMFAvailable:1;
} tCallControlBits;

typedef union
{
    tU8 value;
    tCallControlBits bits;
}tDiPOCallControlsConfiguration;

typedef enum
{
    DIPO_INITIATECALLTYPE_DESTINATION = 0,
    DIPO_INITIATECALLTYPE_VOICEMAIL = 1,
    DIPO_INITIATECALLTYPE_REDIAL = 2
} tDiPOInitiateCallType;

typedef enum
{
    DIPO_INITIATECALLSERVICE_TELEPHONY = 1,
    DIPO_INITIATECALLSERVICE_FACE_TIME_AUDIO = 2,
    DIPO_INITIATECALLSERVICE_FACE_TIME_VIDEO = 3
} tDiPOInitiateCallSevice;

struct tDiPOInitiateCall
{
    tDiPOInitiateCallType type;
    tDiPOInitiateCallSevice service;
    tGeneralString destinationID;
    tGeneralString addressBookID;

    tDiPOInitiateCall()
    {
        type = DIPO_INITIATECALLTYPE_DESTINATION;
        service = DIPO_INITIATECALLSERVICE_TELEPHONY;
        memset(destinationID,0,sizeof(tGeneralString));
        memset(addressBookID,0,sizeof(tGeneralString));
    }
};

typedef enum
{
    DIPO_ACCEPTACTION_ACCEPT_OR_HOLD_AND_ACCEPT = 0,
    DIPO_ACCEPTACTION_END_AND_ACCEPT = 1
} tDiPOAcceptAction;

struct tDiPOAcceptCall
{
    tDiPOAcceptAction acceptAction;
    tGeneralString callUUID;

    tDiPOAcceptCall()
    {
        acceptAction = DIPO_ACCEPTACTION_ACCEPT_OR_HOLD_AND_ACCEPT;
        memset(callUUID,0,sizeof(tGeneralString));
    }
};

typedef enum
{
    DIPO_ENDACTION_END_OR_DECLINE = 0,
    DIPO_ENDACTION_END_ALL = 1
} tDiPOEndAction;

struct tDiPOEndCall
{
    tDiPOEndAction endAction;
    tGeneralString callUUID;

    tDiPOEndCall()
    {
        endAction = DIPO_ENDACTION_END_OR_DECLINE;
        memset(callUUID,0,sizeof(tGeneralString));
    }
} ;

typedef struct
{
    tS8  s8DaylightSavingsOffsetMinutes;
    tU64 u64SecondsSinceReferenceDate;
    tS16 s16TimeZoneOffsetMinutes;
} tDiPODeviceTime;
void InitDiPODeviceTime(tDiPODeviceTime &deviceTime);

/* Get object id , category type and URL from selected folder*/
typedef enum GRACENOTE_METHOD_REQUEST_STATUS_TYPE_E
{
    GRACENOTE_REQ_OK = 0,
    GRACENOTE_REQ_NOT_OK,
    GRACENOTE_REQ_NOT_ACTION,
    GRACENOTE_REQ_NOT_SUPPORT,
    GRACENOTE_REQ_ACTION_TIMEOUT
}tGracenoteMethodRequestStatus;
#define tGracenoteMethodRequestStatus_format "i"
#define tGracenoteMethodRequestStatus_init REQ_NOT_OK

typedef enum GRACENOTE_LOOKUP_MODE
{
 GRACENOTE_ONLINE = 1,
 GRACENOTE_OFFLINE = 0
}tGracenoteLookupMode;
#define tGracenoteLookupMode_format "i"
#define tGracenoteLookupMode_init GRACENOTE_ONLINE

typedef enum GRACENOTE_ALBUM_COVER_FORMAT
{
    FORMAT_NONE = 0,
    FORMAT_PNG ,
    FORMAT_JPEG
}tGracenoteCoverArtFormat;

#define tGracenoteCoverArtFormat_format "i"
#define tGracenoteCoverArtFormat_init REQ_NOT_OK

typedef char tGracenoteLicensePath[1024];
#define tGracenoteLicensePath_format "t"


typedef char tAlbumArtPath[1024];
#define tAlbumArtPath_format "t"
typedef unsigned int tStartAddressArray[100];
typedef char tGracenoteVersion[10];
#define tGracenoteLicensePath_format "t"

/* Get object id , category type and URL from selected folder*/
typedef struct
{
    tObjectID Id;
    tCategoryType Category;
    string URL;
}tDirInfo;


typedef enum GRACENOTE_INIT_STATUS
{
    GRACENOTE_INIT_NOT_DONE = 0,
    GRACENOTE_INIT_OK ,
    GRACENOTE_INIT_ERROR
}tGracenoteInitStatus;
#define tGracenoteInitStatus_format "i"
#define tGracenoteInitStatus_init GRACENOTE_INIT_NOT_DONE

typedef enum
{
    RIPPING,
    CDDA,
    CDDB_ONDEMAND
}tGNRequestType;
#define tGNRequestType_format "i"
#define tGNRequestType_init RIPPING
typedef bool tMetadataAvailablefromGN;
#define tMetadataAvailablefromGN_format "i"
#define tMetadataAvailablefromGN_init 0

typedef enum
{
    CD_AUDIO_LINEAR_PCM = 0x00 ,
    CD_AUDIO_CD_DTS = 0x10 ,
    CD_AUDIO_UNKNOWN = 0xFF,
    CD_AUDIO_INVALID = 0xF0
}tAudioFormat;
#define tAudioFormat_format "i"
#define tAudioFormat_init CD_AUDIO_INVALID

typedef tU32 tAlbumCount;
#define tAlbumCount_format "i"
#define tAlbumCount_init 0

typedef char* tAlbumInfo;
#define tAlbumInfo_format "p"
#define tAlbumInfo_init NULL

typedef char* tMetadataInfo;
#define tMetadataInfo_format "p"
#define tMetadataInfo_init NULL

typedef char* tGNAlbumArtFilename;
#define tGNAlbumArtFilename_format "p"
#define tGNAlbumArtFilename_init NULL

typedef tU32 tGNAlbumArtFilesize;
#define tGNAlbumArtFilesize_format "i"
#define tGNAlbumArtFilesize_init 0


typedef tU32 tAlbumIndex;
#define tAlbumIndex_format "i"
#define tAlbumIndex_init 0

typedef struct
{
    tMetadata MetadataField1;
    tMetadata MetadataField2;
    tMetadata MetadataField3;
    tMetadata MetadataField4;
    tMetadata YomiTitle;  //for storing yomi metadata title
    tMetadata YomiArtist; //for storing yomi metadata Artist
    tMetadata YomiAlbum; //for storing yomi metadata Album
    tUInt Year;
    tTitle title;
    tUInt CompilationFlag;
}tRippedTrackInfo;

typedef enum GRACENOTE_ALBUM_ART_STATUS
{
    GRACENOTE_ALBUM_ART_FAILURE = 0,
    GRACENOTE_ALBUM_ART_SUCCESS
}tAlbumArtDownload;
#define tAlbumArtDownload_format "i"
#define tAlbumArtDownload_init GRACENOTE_ALBUM_ART_FAILURE

typedef enum GRACENOTE_MEDIA_ERROR_CODES
{
    GRACENOTE_MEDIA_ERROR_FAILURE = 0,
    GRACENOTE_MEDIA_ERROR_SUCCESS
}tGracenoteMediaErrorCodes;
#define tGracenoteMediaErrorCodes_format "i"
#define tGracenoteMediaErrorCodes_init GRACENOTE_MEDIA_ERROR_FAILURE

/*
 * To receive property updates from Video Manager component
 * Currently used for DVD Video Source
 *  0=unknown, 1=inactive, 2=activating, 3=active, 4=blocked, 5=restricted (audio only)
 *  0=NONE, 1=RVCA, 2=RVCD, 3=DTV, 4=SPCX, 5=DVD, 6=USB
 */
typedef enum
{
    VIDEO_SOURCE_NONE  = 0,
    VIDEO_SOURCE_RVCA ,
    VIDEO_SOURCE_RVCD ,
    VIDEO_SOURCE_DTV ,
    VIDEO_SOURCE_SPCX ,
    VIDEO_SOURCE_DVD ,
    VIDEO_SOURCE_USB
}tVideoSource ;


typedef enum
{
    VIDEO_SOURCE_STATUS_UNKNOWN = 0,
    VIDEO_SOURCE_STATUS_INACTIVE ,
    VIDEO_SOURCE_STATUS_ACTIVATING ,
    VIDEO_SOURCE_STATUS_ACTIVE,
    VIDEO_SOURCE_STATUS_BLOCKED,
    VIDEO_SOURCE_STATUS_RESTRICTED_AUDIO_ONLY,
    VIDEO_SOURCE_STATUS_ERROR = 6
}tVideoSourceState;

/**
 * Type declaration for the image blob
 */
typedef struct
{
    tImageBlob imageBlob;
    tUInt sizeX;
    tUInt sizeY;
    tAlbumArt string;
} tThumbnailImage;

typedef enum
{
    OOB_GENERIC = 0L,
    OOB_APPLE
}tBTOobType;
/*Send response message property*/
typedef struct
{
    tResponseMsg response;
    tDeviceType deviceType;
}tResponseInfo;
/**
 * Function unifies different used codec strings to an internal used enumeration.
 *
 * @param[out] codecEnum internal used unify codec enum
 * @param[in] codecName codec string
 * @return != 0: error, = 0: OK
 */
void CodecStandardisation(tCodec &codecEnum, const tGeneralString codecName);

typedef tBoolean tDoneRippingAnswerRequired;
#define tDoneRippingAnswerRequired_format "i"
#define tDoneRippingAnswerRequired_init 0

typedef struct {
    int volumeSentToDevice;
    int currentVolumeReceivedFromAudio;
    bool waitingForVolumeUpdate;
    int currentVolumeinDevice;
} tAbsoluteVolumeSupportValues;
typedef enum
{
    PLAY_NOT_SENT = 0L,
    PLAY_SENT,
    PLAY_REPLY_RECEIVED
} tStateOfPlayinStartPlay;
typedef enum
{
    GNMETADATA_EXTRACTION_SUCCESS,
    GNMETADATA_NOMATCH_FOUND
}tGNmetadataExtractionResult;
typedef enum
{
  SLOW_RAMP = 0L,
  FAST_RAMP
}tRampType;
typedef enum
{
    HID_PLAY = 0,
    HID_PAUSE,
    HID_SCAN_NEXT_TRACK,
    HID_SCAN_PREV_TRACK,
    HID_RANDOM,
    HID_REPEAT,
    HID_VOICE,
    HID_MUTE,
    HID_INVALID = 99
}tPlaybackHIDCommand;
#define tPlaybackHIDCommand_format "i"
#define tPlaybackHIDCommand_init HID_PLAY

typedef struct
{
    tObjectID objectID=OBJECT_ID_NONE;
    std::string coverArthandle="";
    std::string mediaItemObjectPath="";
}tBTCoverArtInfo;

void InitBTCoverArtInfo(tBTCoverArtInfo &BTCoverArtInfo);

#endif //_TYPEDEFINITIONS_H_
