/*!
*******************************************************************************
* \file              ipodauth_tcliPodAuthService.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment - Virtual Device iPod Authentication
*  - DESCRIPTION:    iPod Authentication service handler.
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 21.04.2011 \version 1.3 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Updation of macro name IPODAUTH_S32GETSIGNDATA -> IPODAUTH_GETSIGNDATA due
* to return value change.
*
* \date 14.02.2011 \version 1.2 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Fixed the issue with getting accessory certificate
* - Change of concept in retrieving the basic Apple CP register values.
* - Implemented feature to write the Firmware verion to registry.
*
* \date 13.01.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Migrated codebase to use Infotainment Helper Library (IHL).
* - Added function to send error messages to clients.
*
* \date 18.10.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_IPODAUTHFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_IPODAUTHFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_IPODAUTHFI_SERVICEINFO
#include <midw_fi_if.h>

#define IHL_S_IMPORT_INTERFACE_FI_HELPERS
#include <ihl_if.h>
using namespace ihl::fi_helpers;

#include "ipodauth_tcliPodAuthService.h"
#include "ipodauth_tclRegistry.h"
#include "ipodauth_Trace.h"
#include "ipodauth_Wrapper.h"
#include <vector>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_IPODAUTH_SERVICE
#include "trcGenProj/Header/ipodauth_tcliPodAuthService.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

#define IPODAUTH_SERVICE_MAJOR_VER  MIDW_IPODAUTHFI_C_U16_SERVICE_MAJORVERSION
#define IPODAUTH_SERVICE_MINOR_VER  MIDW_IPODAUTHFI_C_U16_SERVICE_MINORVERSION

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/*!
 * \typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgiPodAuthenticationSelftestMethodStart> ipodauth_tMsgSelfTestMS
 * \brief Self test Method start eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgiPodAuthenticationSelftestMethodStart> ipodauth_tMsgSelfTestMS;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgiPodAuthenticationSelftestMethodResult> ipodauth_tMsgSelfTestMR
 * \brief Self test Method result eXtended Fi type with auto destructor feature.
 */

typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgiPodAuthenticationSelftestMethodResult> ipodauth_tMsgSelfTestMR;

/*!
 * \typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgSetSignatureMethodStart> ipodauth_tMsgSetSignMS
 * \brief Set iPod Signature Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgSetSignatureMethodStart> ipodauth_tMsgSetSignMS;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgSetSignatureMethodResult> ipodauth_tMsgSetSignMR
 * \brief Set iPod Signature Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgSetSignatureMethodResult> ipodauth_tMsgSetSignMR;

/*!
 * \typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgAuthenticationInitMethodStart> ipodauth_tMsgInitMS
 * \brief Authentication init Method start eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgAuthenticationInitMethodStart> ipodauth_tMsgInitMS;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgAuthenticationInitMethodResult> ipodauth_tMsgInitMR
 * \brief Authentication init Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgAuthenticationInitMethodResult> ipodauth_tMsgInitMR;

/*!
 * \typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgGenSignatureMethodStart> ipodauth_tMsgGenSignMS
 * \brief Generate Signature Method start eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgGenSignatureMethodStart> ipodauth_tMsgGenSignMS;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgGenSignatureMethodResult> ipodauth_tMsgGenSignMR
 * \brief Generate Signature Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgGenSignatureMethodResult> ipodauth_tMsgGenSignMR;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgGetChallengeMethodResult> ipodauth_tMsgGetChallengeMR
 * \brief Get Challenge data Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgGetChallengeMethodResult> ipodauth_tMsgGetChallengeMR;

/*!
 * \typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgiPodCertificateMethodStart> ipodauth_tMsgiPodCertMS
 * \brief iPod Certificate data Method start eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclXFiObjHandler<midw_ipodauthfi_tclMsgiPodCertificateMethodStart> ipodauth_tMsgiPodCertMS;

/*!
 * \typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgiPodCertificateMethodResult> ipodauth_tMsgiPodCertMR
 * \brief iPod Certificate data Method result eXtended Fi type with auto destructor feature.
 */
typedef ihl_tclFiObjHandler<midw_ipodauthfi_tclMsgiPodCertificateMethodResult> ipodauth_tMsgiPodCertMR;

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| Utilizing the Framework for message map abstraction.
|----------------------------------------------------------------------------*/

BEGIN_MSG_MAP(ipodauth_tcliPodAuthService, ahl_tclBaseWork)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_AUTHENTICATIONINIT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsAuthInit)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_GENSIGNATURE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsGenSign)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_SETSIGNATURE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsSetSign)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_GETCHALLENGE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsGetChallenge)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_IPODCERTIFICATE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsiPodCertif)
   ON_MESSAGE_SVCDATA(MIDW_IPODAUTHFI_C_U16_IPODAUTHENTICATIONSELFTEST, AMT_C_U8_CCAMSG_OPCODE_METHODSTART, vOnMsiPodSelfTest)
END_MSG_MAP()

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  ipodauth_tcliPodAuthService::ipodauth_tcliPodAuthService(ahl_tclBaseO ...
******************************************************************************/

/*explicit*/
ipodauth_tcliPodAuthService::ipodauth_tcliPodAuthService(ahl_tclBaseOneThreadApp *const cpoApp)
:ahl_tclBaseOneThreadService(cpoApp, CCA_C_U16_SRV_IPODAUTH
                             , IPODAUTH_SERVICE_MAJOR_VER, IPODAUTH_SERVICE_MINOR_VER)
                             , m_u8FirmwareMajVer(0), m_u8AuthProMajVer(0)
                             , m_u8AuthProMinVer(0), m_u32DeviceId(0)
                             , m_u16AccCertifLen(0)
                             , m_oAccCertificate(IPODAUTH_ACC_CERTIF_MAX_LEN)
{
   //TODO: Used new platform call instead to just (un)load pfcfg/IPOD_CFG file
   int initRet = iPodInitConnection();
   ETG_TRACE_USR1(("iAP library loaded: %s", initRet==IPOD_OK ? "OK" : "ERROR"));
   ETG_TRACE_USR1(("Initializing the Apple Co processor"));

   ETG_TRACE_USR1(("Reading the Apple Co processor registers"));
   tU8 u8MinVer   =  0;

   tS32 s32Result = IPODAUTH_S32GETFIRMWAREVERSION(&m_u8FirmwareMajVer, &u8MinVer);

   if (IPOD_OK == s32Result)
   {
      ETG_TRACE_USR1(("Firmware version: %d", m_u8FirmwareMajVer));

      // Make a registry entry for the firmware version.
      ipodauth_tclRegistry oRegistry(IPODAUTH_CP_REG_PATH);

      if (FALSE == oRegistry.bAdd(IPODAUTH_CP_FIRMWARE_VER, m_u8FirmwareMajVer))
      {
         ETG_TRACE_ERR(("Failed to add Firmware major version to registry"));
      }
      else if (FALSE == oRegistry.bAdd(IPODAUTH_CP_FIRMWARE_MIN_VER, u8MinVer))
      {
         ETG_TRACE_ERR(("Failed to add Firmware minor version to registry"));
      }
      else
      {
         ETG_TRACE_USR1(("Firmware version added to registry"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Failed to retrieve Firmware version with error: %d", s32Result));
   }

   s32Result   =  IPODAUTH_S32GETPROTOCOLVERSION(&m_u8AuthProMajVer, &m_u8AuthProMinVer);

   if (IPOD_OK == s32Result)
   {
      ETG_TRACE_USR1(("Auth Protocol Major version: %d, Minor version: %d"
         , m_u8AuthProMajVer, m_u8AuthProMinVer));
   }
   else
   {
      ETG_TRACE_ERR(("Failed to retrieve AuthProtocol version with error: %d", s32Result));
   }

   s32Result   =  IPODAUTH_S32GETDEVICEID((U32*)&m_u32DeviceId);

   if (IPOD_OK == s32Result)
   {
      ETG_TRACE_USR1(("Device id: %d", m_u32DeviceId));
   }
   else
   {
      ETG_TRACE_ERR(("Failed to retrieve Device id."));
   }

   IPODAUTH_GETCERTIFICATE(&m_u16AccCertifLen, &m_oAccCertificate[0]);


}  // ipodauth_tcliPodAuthService::ipodauth_tcliPodAuthService(ahl_tclBaseOneThreadApp ..

/******************************************************************************
** FUNCTION:  virtual ipodauth_tcliPodAuthService::~ipodauth_tcliPodAuthService()
******************************************************************************/

/*virtual*/
ipodauth_tcliPodAuthService::~ipodauth_tcliPodAuthService()
{
   ETG_TRACE_USR1(("~ipodauth_tcliPodAuthService() entered."));


   //TODO: Used new platform call instead to just (un)load pfcfg/IPOD_CFG file
   iPodDisconnect();
   ETG_TRACE_USR1(("iAP library unloaded"));

}  // ipodauth_tcliPodAuthService::~ipodauth_tcliPodAuthService()

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnLoopbackService(amt_tclS...
******************************************************************************/

/*virtual*/
tVoid ipodauth_tcliPodAuthService::vOnLoopback
(
   tU16 /*u16ServiceId*/
   , amt_tclServiceData *poMessage
)
{
   ETG_TRACE_USR1(("vOnLoopback() entered."));

   if (FALSE == ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage))
   {
      // No default handler found, check for get/upreg/relupreg messages.
      switch (poMessage->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         {
            ail_tenCommunicationError enResult = eUpdateRequestingClient(poMessage);

            if (AIL_EN_N_NO_ERROR != enResult)
            {
               ETG_TRACE_ERR(("GET with Function id:%d resulted in error"
                  , ETG_ENUM(IPODAUTH_FID, poMessage->u16GetFunctionID())));
            }
         }
         break;

         default:
         {
            // Trace: Nothing else to do since it is a loopback message.
            // Message will be deleted by the framework.
            // No error message to be sent back if loop back could not find a mapping
            // function - just perform the trace.
            ETG_TRACE_ERR(("vOnLoopbackService() could not find a mapping function."));
            ETG_TRACE_ERR(("Opcode: %d, FID: %d", ETG_ENUM(OP_CODES
               , poMessage->u8GetOpCode()), ETG_ENUM(IPODAUTH_FID
               , poMessage->u16GetFunctionID())));
         }
         break;
      }
   }
}  // tVoid ipodauth_tcliPodAuthService::vOnLoopback(tU16 /*u16ServiceId*/..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsAuthInit(amt_tclSer..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsAuthInit(amt_tclServiceData *poServData)
{
   ipodauth_tMsgInitMS oAuthInitMs(*poServData);

   tS32 s32Result    =  IPOD_OK;

   // Try to check if init or deinit has to be done
   if (FALSE != oAuthInitMs.bInit)
   {
      ETG_TRACE_USR4(("Processing init of communication with Apple CP.."));

      // Initialize the communication with CP.
      s32Result   =  IPODAUTH_S32INIT();
   }
   else
   {
      ETG_TRACE_USR4(("Processing deinit of communication with Apple CP.."));

      // Deinit the communication with CP.
      s32Result   =  IPODAUTH_S32DEINIT();
   }

   if (IPOD_OK == s32Result)
   {
      ipodauth_tMsgInitMR oAuthInitMr;

      // Send the method result.
      vSendMessage(*poServData, oAuthInitMr);
   }
   else
   {
      // Error message has to be sent.
      vSendError(*poServData);

      ETG_TRACE_ERR(("Init routine failed with error: %d", s32Result));
   }
}  // tVoid ipodauth_tcliPodAuthService::vOnMsAuthInit(amt_tclServiceData *p..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsSetSign(amt_tclSer..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsSetSign(amt_tclServiceData *poServData)
{
   ETG_TRACE_USR4(("Processing Set signature .."));

   ipodauth_tMsgSetSignMS oSetSignatureMs(*poServData);

   // Allocate the raw buffer for copying the signature from Apple CP.
   ihl_tclRawData oSignature(IPODAUTH_CP_SIGN_MAX_LEN);
   // Check if the buffer allocation was successful.
   NORMAL_M_ASSERT(FALSE == oSignature.bIsEmpty());

   tU16 u16SignLen   =  (tU16)oSetSignatureMs.Signature.ContainerDataList.size();
   ETG_TRACE_USR4(("Signature size: %d", u16SignLen));

   NORMAL_M_ASSERT(IPODAUTH_CP_SIGN_MAX_LEN >= u16SignLen);

   // Unmarshall the method start.
   for (tU16 u16Loop = 0; u16Loop < u16SignLen; ++u16Loop)
   {
      oSignature[u16Loop] = oSetSignatureMs.Signature.ContainerDataList[u16Loop];
      ETG_TRACE_USR4(("Signature[%d]: %d", u16Loop, oSignature[u16Loop]));
   }

   tS32 s32Result    =  IPODAUTH_S32SETSIGNATURE(u16SignLen, &oSignature[0]);

   if (IPOD_OK == s32Result)
   {
      ipodauth_tMsgSetSignMR oSetSignatureMr;

      // Send the method result.
      vSendMessage(*poServData, oSetSignatureMr);
   }
   else
   {
      // Error message has to be sent.
      vSendError(*poServData);
      ETG_TRACE_ERR(("Set signature failed with error: %d", s32Result));
   }
}  // tVoid ipodauth_tcliPodAuthService::vOnMsSetSign(amt_tclServiceData *poSe..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsGenSign(amt_tclSer..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsGenSign(amt_tclServiceData *poServData)
{
   ETG_TRACE_USR4(("Triggering signature generation process.."));

   // Unmarshall data into method start object.
   ipodauth_tMsgGenSignMS oGenSignatureMs(*poServData);

   // Allocate the raw buffer for copying the signature from Apple CP.
   ihl_tclRawData oSignature(IPODAUTH_CP_SIGN_MAX_LEN);
   // Check if the buffer allocation was successful.
   NORMAL_M_ASSERT(FALSE == oSignature.bIsEmpty());

   tU32 u32ChallengeSize   =  (tU32)oGenSignatureMs.iPodChallenge.ContainerDataList.size();

   NORMAL_M_ASSERT(IPODAUTH_CP_CHALLENGE_MAX_LEN >= u32ChallengeSize);

   ihl_tclRawData oiPodChallenge(u32ChallengeSize);

   for (tU32 u32Loop = 0; u32Loop < u32ChallengeSize; ++u32Loop)
   {
      oiPodChallenge[u32Loop] =  oGenSignatureMs.iPodChallenge.ContainerDataList[u32Loop];
   }

   tU16 oiPodChallengeLen = 0;

   tU16 u16SignDataLen     =  0;

   tS32 s32Result =  IPODAUTH_S32GETSIGNDATA(&oiPodChallenge[0], oiPodChallengeLen, &u16SignDataLen, &oSignature[0]);

   if (IPOD_OK == s32Result)
   {
      ETG_TRACE_USR4(("Signature length is : %d", u16SignDataLen));

      if ((0 != u16SignDataLen) && (IPODAUTH_CP_SIGN_MAX_LEN >= u16SignDataLen))
      {
         ipodauth_tMsgGenSignMR oGenSignatureMr;

         // Fill in the signature in the result
         for (tU16 u16Loop = 0; u16Loop < u16SignDataLen; ++u16Loop)
         {
            oGenSignatureMr.Signature.ContainerDataList.push_back(oSignature[u16Loop]);
         }

         // Send the method result.
         vSendMessage(*poServData, oGenSignatureMr);
      }
      else
      {
         NORMAL_M_ASSERT(0 != u16SignDataLen);
         NORMAL_M_ASSERT(IPODAUTH_CP_SIGN_MAX_LEN >= u16SignDataLen);

         // Error message has to be sent.
         vSendError(*poServData);
         ETG_TRACE_ERR(("Signature length is invalid"));
      }
   }
   else
   {
      // Error message has to be sent.
      vSendError(*poServData);
      ETG_TRACE_ERR(("Get signature failed with error"));
   }

}  // tVoid ipodauth_tcliPodAuthService::vOnMsGenSign(amt_tclServiceData *poS..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsGetChallenge(amt_tclSer..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsGetChallenge(amt_tclServiceData *poServData)
{
   ETG_TRACE_USR4(("Processing request to get Challenge data.."));

   // Allocate the raw buffer for copying the challenge data from Apple CP.
   ihl_tclRawData oChallengeData(IPODAUTH_CP_CHALLENGE_MAX_LEN);
   // Check if the buffer allocation was successful.
   NORMAL_M_ASSERT(FALSE == oChallengeData.bIsEmpty());

   tU16 u32ChallengeSize     =  0;

   // Get the challenge data.
   tS32 s32Result =  IPODAUTH_S32GETCHALLENGEDATA(&u32ChallengeSize, &oChallengeData[0]);

   if (IPOD_OK == s32Result)
   {
      ETG_TRACE_USR4(("Challenge data size returned is : %d", u32ChallengeSize));

      ipodauth_tMsgGetChallengeMR oGetChallengeMr;

      // Assert for the challenge data size.
      NORMAL_M_ASSERT(IPODAUTH_CP_CHALLENGE_MAX_LEN >= u32ChallengeSize);

      for (tU16 u16Loop = 0; u16Loop < u32ChallengeSize; ++u16Loop)
      {
         oGetChallengeMr.CPChallenge.ContainerDataList.push_back(oChallengeData[u16Loop]);
      }

      // Send the method result.
      vSendMessage(*poServData, oGetChallengeMr);
   }
   else
   {
      // Error message has to be sent.
      vSendError(*poServData);
      ETG_TRACE_ERR(("Getting Challenge data from CP failed with error: %d"
         , s32Result));
   }
}  // tVoid ipodauth_tcliPodAuthService::vOnMsGetChallenge(amt_tclServiceData ..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsiPodCertif(amt_tclServi..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsiPodCertif(amt_tclServiceData *poServData)
{
   ETG_TRACE_USR4(("Processing set iPod Certificate.."));

   // Unmarshall data into method start object.
   ipodauth_tMsgiPodCertMS oiPodCertMs(*poServData);

   // Allocate the raw buffer for copying the iPod Certificate from iPod.
   ihl_tclRawData oiPodCertif(IPODAUTH_CP_IPOD_CERTIF_MAX_LEN);
   // Check if the buffer allocation was successful.
   NORMAL_M_ASSERT(FALSE == oiPodCertif.bIsEmpty());

   tU16 u16iPodCertLen  =  (tU16)oiPodCertMs.Certificate.ContainerDataList.size();
   // Check if the length is more than expected?
   NORMAL_M_ASSERT(IPODAUTH_CP_IPOD_CERTIF_MAX_LEN >= u16iPodCertLen);

   // Copy the iPod certificate to the buffer.
   for (tU16 u16Loop = 0; u16Loop < u16iPodCertLen; ++u16Loop)
   {
      oiPodCertif[u16Loop] =  oiPodCertMs.Certificate.ContainerDataList[u16Loop];
   }

   // Set the iPod Certificate.
   tS32 s32Result =  IPODAUTH_S32SETCERTIFICATE(u16iPodCertLen, &oiPodCertif[0]);

   if (IPOD_OK == s32Result)
   {
      ipodauth_tMsgiPodCertMR oiPodCertMr;

      // Send the method result.
      vSendMessage(*poServData, oiPodCertMr);
   }
   else
   {
      // Error message has to be sent.
      vSendError(*poServData);
      ETG_TRACE_ERR(("Setting the iPod certificate to CP failed with error: %d"
         , s32Result));
   }

}  // tVoid ipodauth_tcliPodAuthService::vOnMsiPodCertif(amt_tclServiceData *..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vOnMsiPodSelfTest(amt_tclSer..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vOnMsiPodSelfTest(amt_tclServiceData *poServData)
{
   ETG_TRACE_USR4(("iPod Self test triggered..."));
   ipodauth_tMsgSelfTestMS oiPodSelfTestMs(*poServData);

   tU8 u8Certificate =  0;
   tU8 u8PriKey      =  0;
   tU8 u8RamCheck    =  0; // Depricated w.r.t new Apple Co-processor
   tU8 u8CheckSum    =  0; // Depricated w.r.t new Apple Co-processor

   tS32 s32Result = IPODAUTH_S32SELFTEST(&u8Certificate, &u8PriKey
      , &u8RamCheck, &u8CheckSum);

   ipodauth_tMsgSelfTestMR oiPodSelfTestMr;

   if (IPOD_OK == s32Result)
   {
      oiPodSelfTestMr.enResult.enType  =
         midw_fi_tcl_e16_iPodAuthSelfTestResult::FI_EN_IPODAUTHPASSED;
   }
   else
   {
      oiPodSelfTestMr.enResult.enType  =
         midw_fi_tcl_e16_iPodAuthSelfTestResult::FI_EN_IPODCERTNKEYNOTFOUND;
   }

   ETG_TRACE_USR4(("iPod Self test status: %d. Cert: %d, PriKey: %d"
      , ETG_ENUM(IPODAUTH_SELFTESTRESULT, oiPodSelfTestMr.enResult.enType)
      , ETG_ENUM(IPODAUTH_PRESENT, u8Certificate)
      , ETG_ENUM(IPODAUTH_PRESENT, u8PriKey)));

   // Send the method result.
   vSendMessage(*poServData, oiPodSelfTestMr);

}  // tVoid ipodauth_tcliPodAuthService::vOnMsiPodSelfTest(amt_tclServiceData *poServData)

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vSendError( const amt_tclSe..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vSendError
(
   const amt_tclServiceData &rfcoServiceData
   , tCU16 cu16Error
)
{
   // Create an error message.
   amt_tclServiceDataError oError(rfcoServiceData, cu16Error);
   // Post the message to message Queue.
   vPostMessage(oError);
}

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vSendMessage( const amt_tclSe..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vSendMessage
(
   const amt_tclServiceData  &rfcoServiceData
   , const fi_tclTypeBase &rfcoTypeBase
   , tU8 u8OpCode
)
{
   tU16 u16ServMajVer   =  1;
   tU16 u16ServMinVer   =  0;
   tU16 u16ServPatchVer =  0;

   // Get the Fi Versions.
   (tVoid)bGetServiceVersion(u16GetServiceID(), u16ServMajVer, u16ServMinVer
      , u16ServPatchVer);

   // Create the Visitor Message
   fi_tclVisitorMessage oOutVisitorMsg(rfcoTypeBase, u16ServMajVer);

   // Set the CCA message information
   oOutVisitorMsg.vInitServiceData
   (
      u16GetAppID()                                // Source app-ID
      , rfcoServiceData.u16GetSourceAppID()        // Dest. app-ID
      , AMT_C_U8_CCAMSG_STREAMTYPE_NODATA          // Stream type
      , 0                                          // Stream counter
      , rfcoServiceData.u16GetRegisterID()         // Registry ID
      , rfcoServiceData.u16GetCmdCounter()         // Command counter
      , u16GetServiceID()                          // Service-ID
      , rfcoServiceData.u16GetFunctionID()         // Function-ID
      , u8OpCode                                   // OpCode
   );

   // Post the message to message Queue.
   vPostMessage(oOutVisitorMsg);

   ETG_TRACE_USR4(("Message sent. Opcode: %d, Version: %d, ServiceId: %d, \
                   RegId: %d, FID: %x, SrcAppId: %d, TargAppId: %d, CmdCntr: %d"
                   , ETG_ENUM(OP_CODES, oOutVisitorMsg.u8GetOpCode())
                   , oOutVisitorMsg.u16GetVersion(), oOutVisitorMsg.u16GetServiceID()
                   , oOutVisitorMsg.u16GetRegisterID()
                   , oOutVisitorMsg.u16GetFunctionID()
                   , ETG_ENUM(ail_u16AppId, oOutVisitorMsg.u16GetSourceAppID())
                   , ETG_ENUM(ail_u16AppId, oOutVisitorMsg.u16GetTargetAppID())
                   , oOutVisitorMsg.u16GetCmdCounter()
                   ));

}  // tVoid ipodauth_tcliPodAuthService::vSendMessage(const amt_tclServiceData &rf..

/******************************************************************************
** FUNCTION:  tVoid ipodauth_tcliPodAuthService::vPostMessage( amt_tclServiceData..
******************************************************************************/

tVoid ipodauth_tcliPodAuthService::vPostMessage(amt_tclServiceData &rfoServiceData)
{
   if (FALSE != rfoServiceData.bIsValid())
   {
      // Post the message to message queue
      ail_tenCommunicationError enCommError =
         _poMainAppl->enPostMessage(&rfoServiceData, TRUE);

      if (AIL_EN_N_NO_ERROR != enCommError)
      {
         ETG_TRACE_ERR(("Communication Error : %d, could not post message"
            , ETG_ENUM(AIL_ERROR, enCommError)));
      }  // if (AIL_EN_N_NO_ERROR != enCommError)

   }  //if( FALSE != rfoServiceData.bIsValid())
   else
   {
      ETG_TRACE_ERR(("Invalid message!!!"));
   }  //End of if-else; if( FALSE != poServiceData->bIsValid())
}  //tVoid ipodauth_tcliPodAuthService::vPostMessage(amt_tclServiceData *poS....

/*******************************************************************************
** FUNCTION:    tBool ipodauth_tcliPodAuthService::bStatusMessageFactory(tU..
*******************************************************************************/

tBool ipodauth_tcliPodAuthService::bStatusMessageFactory
(
   tU16 u16FunctionId
   , amt_tclServiceData& roOutMsg
   , amt_tclServiceData* poInMsg
)
{
   tBool bRetVal  =  TRUE;

   ETG_TRACE_USR4(("Status message factory invoked for FID: %d", u16FunctionId));

   // Precondition for every case - Apple CP initialization has to be done.
   // Explicitly read the data explicitly from the Apple CP - especially to avoid
   // issues due to non availability of data.
   switch (u16FunctionId)
   {
      case MIDW_IPODAUTHFI_C_U16_FIRMWAREVERSION:
      {
         tU8 u8MinVer   =  0;
         tS32 s32Result = IPODAUTH_S32GETFIRMWAREVERSION(&m_u8FirmwareMajVer, &u8MinVer);

         // Check current status & previous status.
         // In case, it was read previously & the read failed just for this read
         // give the old value.
         if ((IPOD_OK == s32Result) || (0 != m_u8FirmwareMajVer))
         {
            midw_ipodauthfi_tclMsgFirmwareVersionStatus oMsgFirmVerStat;
            oMsgFirmVerStat.u8MajorVersion   =  m_u8FirmwareMajVer;

            ETG_TRACE_USR4(("Firmware version, MajVer: %d, MinVer: %d"
               , oMsgFirmVerStat.u8MajorVersion, oMsgFirmVerStat.u8MinorVersion));

            fi_tclVisitorMessage oMsg(oMsgFirmVerStat, IPODAUTH_SERVICE_MAJOR_VER);

            bRetVal = oMsg.bHandOver(&roOutMsg);
         }
         else
         {
            // Create an error message.
            amt_tclServiceDataError oError(*poInMsg, CCA_C_U16_ERROR_INTERNAL_FAILURE);
            bRetVal = oError.bHandOver(&roOutMsg);

            ETG_TRACE_ERR(("Failed to retrieve Firmware version with error: %d", s32Result));
         }

      }  // case MIDW_IPODAUTHFI_C_U16_FIRMWAREVERSION:
      break;

      case MIDW_IPODAUTHFI_C_U16_AUTHENTICATIONPROTOCOLVERSION:
      {

         tS32 s32Result   =  IPODAUTH_S32GETPROTOCOLVERSION(&m_u8AuthProMajVer
                                 , &m_u8AuthProMinVer);

         // Check current status & previous status.
         // In case, it was read previously & the read failed just for this read
         // give the old value.
         if ((IPOD_OK == s32Result) || (0 != m_u8AuthProMajVer))
         {
            ETG_TRACE_USR1(("Auth Protocol Major version: %d, Minor version: %d"
               , m_u8AuthProMajVer, m_u8AuthProMinVer));

            midw_ipodauthfi_tclMsgAuthenticationProtocolVersionStatus oMsgProVerStat;
            oMsgProVerStat.u8MajorVersion    =  m_u8AuthProMajVer;
            oMsgProVerStat.u8MinorVersion    =  m_u8AuthProMinVer;

            fi_tclVisitorMessage oMsg(oMsgProVerStat, IPODAUTH_SERVICE_MAJOR_VER);

            bRetVal = oMsg.bHandOver(&roOutMsg);
         }
         else
         {
            // Create an error message.
            amt_tclServiceDataError oError(*poInMsg, CCA_C_U16_ERROR_INTERNAL_FAILURE);
            bRetVal = oError.bHandOver(&roOutMsg);

            ETG_TRACE_ERR(("Failed to retrieve AuthProtocol version with error: %d", s32Result));
         }

      }  // case MIDW_IPODAUTHFI_C_U16_AUTHENTICATIONPROTOCOLVERSION:
      break;

      case MIDW_IPODAUTHFI_C_U16_DEVICEID:
      {
         tS32 s32Result   =  IPODAUTH_S32GETDEVICEID((U32*)&m_u32DeviceId);

         // Check current status & previous status.
         // In case, it was read previously & the read failed just for this read
         // give the old value.
         if ((IPOD_OK == s32Result) || (0 != m_u32DeviceId))
         {
            midw_ipodauthfi_tclMsgDeviceIDStatus oMsgDevIdStat;
            oMsgDevIdStat.u32DeviceID  =  m_u32DeviceId;

            ETG_TRACE_USR4(("Device id : %d", oMsgDevIdStat.u32DeviceID));

            fi_tclVisitorMessage oMsg(oMsgDevIdStat, IPODAUTH_SERVICE_MAJOR_VER);

            bRetVal = oMsg.bHandOver(&roOutMsg);
         }
         else
         {
            // Create an error message.
            amt_tclServiceDataError oError(*poInMsg, CCA_C_U16_ERROR_INTERNAL_FAILURE);
            bRetVal = oError.bHandOver(&roOutMsg);

            ETG_TRACE_ERR(("Failed to retrieve Device id."));
         }

      }  // case MIDW_IPODAUTHFI_C_U16_DEVICEID:
      break;

      case MIDW_IPODAUTHFI_C_U16_ACCESSORYCERTIFICATE:
      {
         // Whatever is the condition read the certificate always from Apple CP.
         IPODAUTH_GETCERTIFICATE(&m_u16AccCertifLen, &m_oAccCertificate[0]);

         midw_ipodauthfi_tclMsgAccessoryCertificateStatus oMsgAccCertifStat;

         for(tU16 u16Loop = 0; u16Loop < m_u16AccCertifLen; ++u16Loop)
         {
            oMsgAccCertifStat.Certificate.ContainerDataList.push_back(m_oAccCertificate[u16Loop]);
         }  // for(tU32 u32Loop = 0; u32Loop < u32CertSize; ++u32Loop)

         ETG_TRACE_USR4(("Accessory certificate size: %d", m_u16AccCertifLen));

         fi_tclVisitorMessage oMsg(oMsgAccCertifStat, IPODAUTH_SERVICE_MAJOR_VER);

         bRetVal = oMsg.bHandOver(&roOutMsg);
      }  // case MIDW_IPODAUTHFI_C_U16_ACCESSORYCERTIFICATE:
      break;

      default:
      {
         bRetVal  =  FALSE;
      }  // default:
      break;

   }  // switch (u16FunctionId)

   // Avoid Lint - Parameter needs to be neglected without LINT objection.
   // This parameter may be required for future.
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(poInMsg);

   return bRetVal;
}  // tBool ipodauth_tcliPodAuthService::bStatusMessageFactory(tU16 u16Functi..

////////////////////////////////////////////////////////////////////////////////

// <EOF>
