/*-----------------------------------------------------------------------------*
 * ReaderEOL.cpp                                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : G3G                                                   *
 * COPYRIGHT   : (c) 2012 - 2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file ReaderEOL.cpp
 *
 * \brief Implementation to open, read and close the EOL file.
 *
 * \version 05.03.2014, Koechling Christian (CM-AI/ECD1), initial version
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2020
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#include "osal_if.h"

#include "FunctionTracer.h"
#include "VarTrace.h"
#include "ReaderEOL.h"




/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_SERVICE_MEDIAPLAYER
#ifdef TARGET_BUILD
#include "trcGenProj/Header/ReaderEOL.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_GEN_MEDIAPLAYER_SERVICE_MEDIAPLAYER
#endif
#endif

/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------*
 * Constructor                                                                 *
 *-----------------------------------------------------------------------------*/
ReaderEOL::ReaderEOL():m_ioEolDesc(OSAL_ERROR)
{
    ENTRY
    ETG_TRACE_USR4(("Begin:ReaderEOL"));

   tU32  u32LastErrorCode;
   // Open the EOL Device.
   ETG_TRACE_USR4(("---> ReaderEOL entered"));
   m_ioEolDesc =  OSAL_IOOpen(OSAL_C_STRING_DEVICE_DIAG_EOL, OSAL_EN_READWRITE);

   if (OSAL_ERROR == m_ioEolDesc)
   {
      u32LastErrorCode =  OSAL_u32ErrorCode();
      ETG_TRACE_ERR(("[ERROR] ReaderEOL::ReaderEOL: EOL device open failed with error %u", u32LastErrorCode));
   }

    ETG_TRACE_USR4(("End  :ReaderEOL"));
}// ReaderEOL::ReaderEOL()


/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
ReaderEOL::~ReaderEOL()
{
    ENTRY
    ETG_TRACE_USR4(("Begin:~ReaderEOL"));
   if ((OSAL_ERROR != m_ioEolDesc) && (OSAL_ERROR == OSAL_s32IOClose(m_ioEolDesc)))
   {
      ETG_TRACE_ERR(("[ERROR] ReaderEOL : EOL device close failed -  error %u", OSAL_u32ErrorCode()));
   }

   m_ioEolDesc =  OSAL_ERROR;

   ETG_TRACE_USR4(("End  :~ReaderEOL"));
} // ReaderEOL::~ReaderEOL()  


/*-----------------------------------------------------------------------------*
 * tBool bRead(tCU8 cu8Table, tCU16 cu16Offset, tU8& rfu8Value)                *
 *-----------------------------------------------------------------------------*/
tBool ReaderEOL::bReadEOL(tCU8 cu8Table, tCU16 cu16Offset, tU8 &rfu8Value)
{
    ENTRY
    ETG_TRACE_USR4(("Begin:bReadEOL"));

   tBool bRetVal =  FALSE;
   tU32  u32LastErrorCode;

   if (OSAL_ERROR != m_ioEolDesc)
   {
      OSAL_trDiagEOLEntry oDiagEolEntry;
      oDiagEolEntry.u8Table         =  cu8Table;
      oDiagEolEntry.u16Offset       =  cu16Offset;
      oDiagEolEntry.u16EntryLength  =  sizeof(tU8);
      oDiagEolEntry.pu8EntryData    =  &rfu8Value;

      if (OSAL_ERROR != OSAL_s32IORead(m_ioEolDesc, (tS8*)&oDiagEolEntry ,(tS32)sizeof(oDiagEolEntry)))
      {
         bRetVal  =  TRUE;
      }
      else 
      {
         u32LastErrorCode =  OSAL_u32ErrorCode();
         ETG_TRACE_ERR(("[ERROR] ReaderEOL::bRead : offset: %d,Error:%d" , cu16Offset, u32LastErrorCode));
      }
   }  // if (OSAL_ERROR != m_ioEolDesc)

   ETG_TRACE_USR4(("End  :bReadEOL"));

   return bRetVal;
} // tBool ReaderEOL::bRead(tCU16 cu16Offset, tU8& rfu8Value)



////////////////////////////////////////////////////////////////////////////////
// <EOF>

