
/**
 * @addgroup OutputWrapper
 * @author Stefan Scherber
 *
 * Public interface for CCAOutputWrapper
 * @{
 */

#include "MediaPlayerInterface.h"
#include "OutputWrapper.h"
#include "Lock.h"


class fc_mediaplayer_tclApp;
/**
 * This class is used by the media player components to send messages to external components.
 */
class CCAOutputWrapper: public OutputWrapper
{

public:

    /**
     * The constructor sets all local member variables to default values
     *
     */
    CCAOutputWrapper(tComponentID componentId, fc_mediaplayer_tclApp* poMainAppl);


    /**
     * This function calls SendAVActivation of CCA client handler AVManager
     *
     * @param[in] sourceNr
     * @param[in] logicalAVChannel
     * @return < 0: error, = 0: OK
     */
    virtual tResult RequestAVActivation(const tSourceNr sourceNr, const tLogicalAVChannel logicalAVChannel, const tSource source=0, const tDeviceID deviceID = DEVICE_ID_NOT_SET);

    /**
     * This function is called by the mediaplayer core when playtime property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_PLAYTIME. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaytime();

    /**
     * This function is called by the mediaplayer core when nowplaying property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_NOWPLAYING. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateNowPlaying();

    /**
     * This function is called when the active device was changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_ACTIVEMEDIADEVICE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateActiveMediaDevice();

    /**
     * This function is called when any device changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_MEDIAPLAYERDEVICECONNECTIONS. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateMediaPlayerDeviceConnections();

    /**
     * This function is called when a device connection state has changed.
     * Informs the audio managment about new device
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDeviceConnection(const tDeviceID deviceID, const tConnectionReason connectionReason);

    /**
     * This function is called by the mediaplayer core when playbackstate property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_PLAYBACKSTATE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaybackState();

    /**
     * This function is called when the indexing state (or percentage) of a device changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_INDEXINGSTATE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateIndexingState();

    /**
     * This function is called when the playback mode (shuffle) of a device changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_PLAYBACKMODE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePlaybackMode();

    /**
     * This function is called when the repeat mode of a device changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_REPEATMODE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateRepeatMode();


    /**
     * DEPRECATED
     * This function is called by the mediaplayer core when phone information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPHONEINFO. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdatePhoneInfo();

    /**
     * This function is called by the mediaplayer core when phone information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_CONNECTEDBLUETOOTHPROFILES. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateBTProfileInfo();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPONOWPLAYING. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPONowPlaying();

     /**
      * DEPRECATED
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOMETADATA. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOMetaData();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPLAYTIME. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOPlaytime();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSTATUS. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOPlaybackStatus();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSHUFFLEMODE. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOPlaybackShuffleMode();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKREPEATMODE. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOPlaybackRepeatMode();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOCALLSTATE. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */

     virtual tResult UpdateDiPOCallDuration();

     virtual tResult UpdateDiPOCallState();

     /**
      * This function is called by the mediaplayer core when phone information has changed .
      * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
      * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONS. The framework then notifies all registered clients of the change.
      * @return < 0: error, = 0: OK
      */
     virtual tResult UpdateDiPOCommunications();

    /**
     * This function is called by the mediaplayer core when slideshow property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_SLIDESHOWSTATE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateSlideshowState();

    /**
     * This function is called by the mediaplayer core when now showing property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_NOWSHOWING. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateNowShowing();

    /**
     * This function is called by the mediaplayer core when slide show property changed.
     * The function called the function eUpdateClients of the class MediaPlayerService
     * with the parameter MPLAY_MPLAYFI_C_U16_SLIDESHOWTIME. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateSlideshowTime();

    /**
     * This function is called by the mediaplayer core when SPI component should start or stop with sending location information .
     * The function stores value internally and calls the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOLOCATIONINFO. The framework then notifies all registerd clients of the change.
     * @param[in] startStopLocationInfo start or stop with updates
     * @param[in] locationInfoType supported location info types
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOLocationInfo();

    /**
     * This function is called by the mediaplayer core to update SPI componenet on the GPRMC data status values notified by Apple device.
     * The function calls eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOGPRMCDATASTATUSVALUES. The framework then notifies all registerd clients of the change.
     * @param[in] DeviceID
     * @param[in] GPRMCDataStatusValues(A,V,X)
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPOGPRMCDataStatusValues();


    /*
     * This function is used to notify optical disc slot status towards HMI. I.e. VD_DeviceManager receives it from VD_MediaManager and forwards it though output wrapper to HMI
     *
     * @param[in] enOpticalDiscEjectCmd specify if command is eject or insert
     * @return != 0: error, = 0: OK
     */
     virtual tResult UpdateOpticalDiscSlotState(); //Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * This function is called by the mediaplayer core to ask SPI component if role switch is required.
     * The answer will be received asynchronously
     * @param[in] deviceID DeviceID
     * @param[in] USBSerialNumber USB serial number of Apple Device.
     * @return < 0: error, = 0: OK
     */
    virtual tResult DIPORoleSwitchRequired(const tDeviceID deviceID,const tMountPoint USBSerialNumber,const tBool DeviceCarplayCapability);
    virtual tResult GetBTLimitationModeFromBTClient(const tDeviceSerialNumber serialNumber, tBTLimitationActionState& btLimitation);

    /**
     * This function is called when favorite table changes
     * The function first stores the paramters in local member variables and then
     * calls the function eUpdateClients of the class FC_MediaPlayer_service_MediaPlayer.cpp
     * with the parameter MOST_MPLAYFI_C_U16_FAVORITECHANGES. The framework then notifies all registerd clients of the change.
     * The function eUpdateClienst calls the function GetFavoriteChanged to get the values of the property
     * @param[in] listChangeType Type of the list change
     * @param[in] listSize size of list
     * @param[in] favID changed fav ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendFavoritesChanged(tListChangeType listChangeType, tListSize listSize, tFavoriteID favID);


    /**
     * This function is called by framework to get the changed values of the favorite table
     * @param[out] listChangeType Type of the list change
     * @param[out] listSize size of list
     * @param[out] favID changed fav ID
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetFavoriteChanged(tListChangeType &listChangeType, tListSize &listSize, tFavoriteID &favID);

    /**
     * This function is called when a list changes
     * The function must first call a lock function to avoid parallel access. Then it stores the paramters in local member variables and
     * calls the function eUpdateClients of the class FC_MediaPlayer_service_MediaPlayer.cpp
     * with the parameter MOST_MPLAYFI_C_U16_MEDIAPLAYERLISTCHANGE. The function eUpdateClienst calls the function GetMediaPlayerListChnaged to get the values of the property
     * At last the framework then notifies all registerd clients of the change.
     * When the function eUpdateClients returns the unlock function must be called.
     * @param[in] listID ID of the changed list
     * @param[in] listChangeType type of change
     * @param[in] listSize size of list
     * @param[in] changedItems changed list items
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendMediaPlayerListChanged(tListID listID, tListChangeType listChangeType, tListSize listSize, vector<tPosition> changedItems);
    //virtual tResult SendMediaPlayerListChanged(tListID listID, tListSize listSize);

    /**
     * This function is called by framework to get the list chanegs
     * @param[out] listID ID of the changed list
     * @param[out] listSize size of list
     * @param[out] listChangeType type of change
     * @param[out] listSize size of list
     * @param[out] changedItems changed items
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetMediaPlayerListChanged(tListID &listID, tListChangeType &listChangeType, tListSize &listSize, vector<tPosition> &changedItems);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArtInfo was calculated
     * The function calls SendGetMediaObjectAlbumArtInfoAnswer function of MediaPlayerService class
     * @param[in] mimeType MIME image subtype
     * @param[in] size image size in bytes
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtInfoAnswer(tMimeType mimeType, tImageSize size, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArtInfo cannot provided
     * The function calls the SendGetMediaObjectAlbumArtInfoAnswer function of MediaPlayerService class
     * with parameter mimeType = NULL and size = 0
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtInfoError(tErrorCode errorCode, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt is calculated
     * The function calls SendGetMediaObjectAlbumArtAnswer function of MediaPlayerService class.
     * @param[in] imageData Byte stream of the image
     * @param[in] imageSize size of byte stream
     * @param[in] userContext context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtAnswer(tImageData imageData, tImageSize imageSize, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of GetMediaObjectAlbumArt cannot provided
     * The function calls the SendGetMediaObjectAlbumArtAnswer function of MediaPlayerService class
     * with parameter imageData = NULL
     * @param[in] errorCode
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendGetMediaObjectAlbumArtError(tErrorCode errorCode, tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of Command is calculated
     * @param[in] mountPoint a string containing the iPhone device HID name.
     * @param[in] appName a string containing App name of the IAP application.
     * @param[in] sessionID The session ID of the connected IAP application.
     * @param[in] userContext  client information
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendCommandAnswer(const tMountPoint mountPoint, const tAppName appName, const tSessionID sessionID, const tUserContext userContext);


    /**
     * This function is called by MediaPlayer when the result of DiPOActiveDevice is received
     * Sends answer message to client
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOActiveDeviceAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of DiPORoleSwitchRequest is received
     * Sends answer message to client
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPORoleSwitchRequestAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext);

    /**
     * This function is called by MediaPlayer to get WiFi credentials from SPI
     * Sends answer message to client
     * @param[in] deviceId Device ID of the device whose SPP connection is to be terminated
     * @param[in] appleDeviceMACAddress MAC address of Apple device
     * @return < 0: error, = 0: OK
     */
    virtual tResult GetWiFiCredentials(const tDeviceID deviceId, const tMACAddress appleDeviceMACAddress);

    /**
     * This function is called by MediaPlayer to send Disconnect BT SPP connection request to BT
     * Sends answer message to client
     * @param[in] deviceAddress Device address of the device whose SPP connection is to be terminated
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDisconnectSPPService(const tMACAddress deviceAddress);

    /**
     * This function is called by MediaPlayer when the result of DiPORoleSwitchRequest is received
     * Sends answer message to client
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOTransferGPSDataAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext);

    /**
     * This function is called by MediaPlayer when the result of DiPOTransferDRData is received
     * Sends answer message to client
     * @param[in] diPOResponse Result of operation
     * @param[in] userContext Context to assign the answer to the request
     * @return < 0: error, = 0: OK
     */
    virtual tResult SendDiPOTransferDRDataAnswer(const tDiPOResponse diPOResponse, const tUserContext userContext);

    /**
     * This function is used to send from Mediaplayer Interface to cllientHandler_DeviceManager command eject
     *
     * @param[in] enOpticalDiscEjectCmd specify if command is eject or insert
     * @return != 0: error, = 0: OK
     */
    virtual tResult SendEjectOpticalDisc(IN tDMOpticalDiscEjectCmd enOpticalDiscEjectCmd); //Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /**
     * This function is used to send asynchronous response for eject command to output wrapper.
     *
     * @param[in] entOpticalDiscEjectMethodResult if command was successfull or not see definition of tDMOpticalDiscEjectMethodResult
     * @return != 0: error, = 0: OK
     */
    virtual tResult SendOpticalDiscEjectMethodResult(IN tDMOpticalDiscEjectMethodResult entOpticalDiscEjectMethodResult);//Roadmap 15002_CDAudio_DetectionAndDiagnosis

    /* Functions for IPOD authentication */

    /**
    * AuthenticationInit is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[in] bInit if TRUE initialize authentication; if true deinitalize authentication
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AuthenticationInit(tBool bInit);

    /**
    * AccessoryCertificate is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[out] certLength  length of certificate
    * @param[out] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AccessoryCertificate(tU16 *certLength, tU8 *certData);

    /**
    * IPodCertificate is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[in] certLen  length of certificate
    * @param[in] certData certificate
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 IPodCertificate(tU16 certLen, tU8 *certData);

    /**
    * SetSignature is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[in] sigDataLen  length of signature
    * @param[in] sigData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 SetSignature(tU16 sigDataLen, tU8 *sigData);

    /**
    * GenSignature is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[in] ResponseBuffer  challange data from the IPOD
    * @param[out] SignatureDataLength length of the signature
    * @param[out] SignatureData signature
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 GenSignature(const tU8  *ResponseBuffer, tU16 *SignatureDataLength, tU8 *SignatureData);

    /**
    * GetChallenge is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[out] ChallengeDataLength  challange data length
    * @param[out] ChallengeData challange
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 GetChallenge(tU16 *ChallengeDataLength, tU8 *ChallengeData);

    /**
    * deviceID is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[out] DeviceID device ID
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 DeviceID (tU32 *DeviceID);

    /**
    * FirmwareVersion is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[out] majorVer  Major version of firmware
    * @param[out] minorVer  Minor version of firmware
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 FirmwareVersion (tU8 *majorVer, tU8 *minorVer);

    /**
    * AuthenticationProtocolVersion is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @param[out] majorVer  Major version of protocol
    * @param[out] minorVer  Minor version of protocol
    * @return IPOD_OK or IPOD_ERROR
    */
    virtual tS32 AuthenticationProtocolVersion (tU8 *majorVer, tU8 *minorVer);

    /**
    * IPodAuthenticationSelftest is called by fc_mediaplayer_tclClientHandler_IPodAuthentication object
    * @return IPOD_OK: passed or IPOD_ERROR: failed
    */
    virtual tS32 IPodAuthenticationSelftest (tU8 *certificate, tU8 *private_key, tU8 *ram_check, tU8 *checksum);

    /**
    * The function stores the arguments in member variables and calls eUpdateClients(MIDW_APPCONTROLFI_C_U16_SESSIONOPENED) of ServiceAppControl class
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlConnect(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID,const tProtocolName protocolName);

    /**
    * The function stores the arguments in member variables and calls eUpdateClients(MIDW_APPCONTROLFI_C_U16_DATARESPONSE) of ServiceAppControl class
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @param[in] sizeOfBuffer size of answer buffer
    * @param[in] answer message from IPOD
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlCommand(const tMountPoint deviceID,  const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer, const tCommandBufferPtr commandBuffer);

    /**
    * The function stores the arguments in member variables and calls eUpdateClients(MIDW_APPCONTROLFI_C_U16_SESSIONCLOSED) of ServiceAppControl class
    * @param[in] deviceID a string containing the iPhone device HID name.
    * @param[in] appName a string containing App name of the IAP application.
    * @param[in] sessionID The session ID of the connected IAP application.
    * @return < 0: error, = 0: OK
    */
    virtual tResult AppControlClose(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID);

    /**
     * The function stores the arguments in member variables and calls eUpdateClients(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTART) of ServiceAppControl class
     * @param[in] iAP2iOSAppIdentifier Identifier of connected iOS app
     * @param[in] sinkEndpoint USB endpoint identifier of communication sink.
     * @param[in] sourceEndpoint USB endpoint identifier of communication source.
     * @return < 0: error, = 0: OK
     */
    virtual tResult NativeTransportStart(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint );

   /**
    * The function stores the arguments in member variables and calls eUpdateClients(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTOP) of ServiceAppControl class
    * @param[in] iAP2iOSAppIdentifier Identifier of connected iOS app
    * @param[in] sinkEndpoint USB endpoint identifier of communication sink.
    * @param[in] sourceEndpoint USB endpoint identifier of communication source.
    * @return < 0: error, = 0: OK
    */
   virtual tResult NativeTransportStop(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint );

    /**
    * function to get app control connect  member variables
    */
    virtual tResult GetAppControlConnect(tMountPoint &deviceID, tAppName &appName, tSessionID &sessionID,tProtocolName &protocolName);

    /**
    * function to get app control command member variables
    */
    virtual tResult GetAppControlCommand(tMountPoint &deviceID,  tAppName &appName, tSessionID &sessionID, tSize &sizeOfBuffer,  tCommandBufferPtr &commandBuffer);

    /**
    * function to get app control close member variables
    */
    virtual tResult GetAppControlClose(tMountPoint &deviceID, tAppName &appName, tSessionID &sessionID);

    /**
     * function to get native transport start member variables
     */
    virtual tResult GetNativeTransportStart(tDeviceID &deviceId, tU8 &iAP2iOSAppIdentifier, tU8 &sinkEndpoint, tU8 &sourceEndpoint );

    /**
     * function to get native transport stop member variables
     */
    virtual tResult GetNativeTransportStop(tDeviceID &deviceId, tU8 &iAP2iOSAppIdentifier, tU8 &sinkEndpoint, tU8 &sourceEndpoint );

    /**
    * Request all connected devices from DeviceManager through Clienthandler DeviceManager
    */
    virtual tResult RequestConnectedDevices();

    /**
     * This function requests a mute state at the AVManager of the system.
     *
     * @param[in] muteState requested mute state (DEMUTED or MUTED)
     * @return != 0: error, = 0: OK
     */
    virtual tResult RequestMuteState(const tMuteState muteState, const tRampType rampType = SLOW_RAMP); //Roadmap 13018

    /**
     * This function requests a volume change at the AVManager of the system.
     *
     * @param[in] volumeState requested volume change to NORMAL or REDUCED
     * @return != 0: error, = 0: OK
     */
    virtual tResult RequestVolumeChange(const tVolumeState volumeState); //Roadmap 13018

    /**
     * This function is called when the scan mode of a device changed.
     * This function will call the function eUpdateClients of CCA framework
     * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_SCANMODE. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateScanMode(); //Roadmap 15008_SCAN

    /**
     * UpdateCDInfo will update the clients regarding changes in CD Drive Version and CD Device State
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateCDInfo();//CD info


    /** //Roadmap 16014 'full text search'
    *  This function calls method return first of fi method 'CreateMediaListWithSearchString' also referred as fc_mediaplayer_tclService_MediaPlayer::onCreateMediaPlayerListSearchString'
    *  it is first return
    *  @param [IN] listID is the ID of the created earch string list
    *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
    *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
    *
    *  @return < 0: error, = 0: OK
    */
    virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnFirst(IN tListID listIDParent, IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);

    /** //Roadmap 16014 'full text search'
    *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. And first answer i.e. 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
    *  has been send already. This method is called once or more. I.e. it is the second, third ... answer
    *  @param [IN] listID is the ID of the created earch string list
    *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
    *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
    *
    *  @return < 0: error, = 0: OK
    */
    virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);

    /** //Roadmap 16014 'full text search'
    *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. It is called after once 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
    * and then after once or more times 'Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle' has been called.
    * It is the last method return expected to be called after once 'CreateMediaListWithSearchString'  has been called.
    *
    *  @param [IN] listID is the ID of the created earch string list
    *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
    *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
    *
    *  @return < 0: error, = 0: OK
    */

    virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnLast(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);


    /** //Roadmap 16014 'full text search'
       *  This function calls method return after fi method 'CreateMediaListWithSearchString' has been called. It is called after once 'Send_onCreateMediaPlayerListSearchString_MethodReturnFirst'
       * and then after once or more times 'Send_onCreateMediaPlayerListSearchString_MethodReturnMiddle' has been called.
       * It is the last method return expected to be called after once 'CreateMediaListWithSearchString'  has been called IF SIZE OF THIS LIST HAS NOT BEEN FOUND due to abortion of mechanism.
       *
       *  @param [IN] listID is the ID of the created earch string list
       *  @param [IN] list size of this list. For the first answer this will be expected to be 0 or 1
       *  @param [IN] tDeviceID to keep answer in the same way like for other lists the device ID of the device used for the list is send here
       *
       *  @return < 0: error, = 0: OK
       */

    virtual tResult Send_onCreateMediaPlayerListSearchString_MethodReturnAbort(IN tListID listIDParent,IN tListID listID, IN tListSize listSize, IN tDeviceID deviceID);


    /**
    * This function is called when there is an error sent by apple device for accessory commands.
    * This function will call the function eUpdateClients of CCA framework.
    * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_IPODCOMMUNICATIONERROR, the framework then notifies all registerd clients of the change.
    * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateIpodCommunicationError(const tDeviceID deviceID, tIpodCommError iPodCommError); //CMG3G-8083

    /**
    * function to get iPod Communication Error  member variables
    */
    virtual tResult GetIpodCommunicationError(tDeviceID &deviceID, tIpodCommError &iPodCommError);

        /**
    * This function is called when there is an error sent by apple device for accessory commands during carplay.
    * This function will call the function eUpdateClients of CCA framework.
    * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_DIPOCOMMUNICATIONERROR, the framework then notifies all registerd clients of the change.
    * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDipoCommunicationError(const tDeviceID deviceID, tIpodCommError iPodCommError, const tMountPoint USBSerialNumber);

    /**
    * function to get iPod Communication Error  member variables
    */
    virtual tResult GetDipoCommunicationError(tDeviceID &deviceID, tIpodCommError &iPodCommError, tMountPoint &USBSerialNumber);

     /**
     * This function is called when the video property of video object is changed.
     * This function will call the function eUpdateClients of CCA framework
     * with the parameter MPLAY_MEDIAPLAYERFI_C_U16_VIDEOBRIGHTNESS/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOHUE/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOSATURATION/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOCONTRAST/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOBRIGHTNESSOFFSET/MPLAY_MEDIAPLAYERFI_C_U16_VIDEOHUEOFFSET depend upon the video property changed. The framework then notifies all registerd clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateVideoProperty(const tVideoProperty videoProperty); //Roadmap 15027_Adjust Video Brightness

    /**
    * This function is called by the mediaplayer core when PowerUpdate information has changed .
    * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
    * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOPOWER. The framework then notifies all registered clients of the change.
    * @return < 0: error, = 0: OK
    */
    virtual tResult UpdateDiPOPower();

    /**
    * This function is called by the mediaplayer core when ripping information has changed .
    * @return < 0: error, = 0: OK
    */
    virtual tResult UpdateRippingStatus();
    virtual tResult UpdateAutoRipping();
    /**
     * This function is called by the mediaplayer core when there is a ripping error .
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateRippingErrors();
    /**
     * This function is called by the mediaplayer core when gracenote lookup mode has changed .
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateGracenoteLookupMode();
    /**
    * This function is called by the mediaplayer core when ripping encoding quality has changed .
    * @return < 0: error, = 0: OK
    */
    virtual tResult UpdateRippingEncodingQuality();

    /**
     * This function is called by the mediaplayer core when there is gracenote metadata for the particular album .
     * @return < 0: error, = 0: OK
     */

    virtual tResult UpdateGNdataforAlbumOnUserDemand();
   /**
     * This function is called by the mediaplayer core when DVD controle property update happend
     * with the parameter . The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */

    virtual tResult UpdateDVDProperty(const tDvdProperty dvdProperty);

    /**
      * This function is called to get the Gracenote version.
      * @return < 0: error, = 0: OK
      */
    virtual tResult UpdateGracenoteVersion();

    virtual tResult UpdateGraceNoteResolveAmbiguity();

 //-->Roadmap 16003 : CD Ripping With Gracenote

    /**
    * This function is called when the delete media content operation is completed.
    * This function calls method return after fi method 'DeleteMediaContentByMediaObjectUrls' or 'DeleteMediaContentByMediaObjectTags' or  'DeleteMediaContentByFilterTags' is called
    * @param [IN] deleteMediaContentType is type of 'deletemediacontent' interface used
    * @param [IN] fileOperationErrorCodes is type of 'deletemediacontent' error code
    * @param [IN] numofFilesDeleted is number of files got deleted
    * @return < 0: error, = 0: OK
    */
    virtual tResult Send_OnDeleteMediaContentMethodResult(IN tDeleteMediaContentType deleteMediaContentType, IN tFileOperationErrorCodes fileOperationErrorCodes, IN tU16 numofFilesDeleted );


    /**
    * virtual function of base classes changed to write params to message queue. Test is listening to that queue to check correct paramenter
    */
    virtual tResult Send_OnEditMetadataMethodResult(IN tEditMetadatatType editMetadatatType, IN tFileOperationErrorCodes fileOperationErrorCodes  );


    /**
    * This function is called when the add entries into playlist  operation is completed.
    * This function calls method return after fi method 'AddFilenameToInternalPlaylist'  is called
    * @param [IN] playListOperationErrorCodes is type of playlist operation result code
    * @return < 0: error, = 0: OK
    */
    virtual tResult Send_OnAddFileNamesToPlaylistMethodResult(IN tResult playListOperationErrorCodes );

    /**
     * This function is called when the delete entries into playlist  operation is completed.
     * This function calls method return after fi method 'DeleteEntriesFromPlaylist'  is called
     * @param [IN] playListOperationErrorCodes is type of playlist operation result code
     * @return < 0: error, = 0: OK
     */
     virtual tResult Send_OnDeleteEntriesFromPlaylistMethodResult(IN tResult playListOperationErrorCodes );

    //<--Roadmap 16003 : 'CD Ripping With Gracenote'
    /**
     * AIVI-50524 Update streaming info to HMI
     * Can be overwritten by the derived class.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateStreamingInfo();
    /**
     * This function is called by the mediaplayer core when ROUTEGUIDANCE information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
    virtual tResult UpdateDiPORouteGuidance();

    /**
     * This function is called by the mediaplayer core when ROUTEGUIDANCEMANEUVER information has changed .
     * The function called the function eUpdateClients of the class FC_MediaPlayer_service_AppControl.cpp
     * with the parameter MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER. The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateDiPORouteGuidanceManeuver();

    //<--Roadmap CMG3G-10221 : 'Scene Recorder '
    /**
     * This function is called when the edit media metadata operation is completed.
     * This function calls method return after fi method 'Send_SetfilePermissionByMediaobjectTagsMethodResult'
     * @param [IN] fileOperationErrorCodes is type of set file permission result code
     * @param [IN] numofFiles is stype of set file permission result code
     * @return < 0: error, = 0: OK
     */
     tResult Send_SetfilePermissionByMediaobjectTagsMethodResult(IN tResult filePermissionError, IN tU16 numofFiles);
     //<--Roadmap CMG3G-10221 : 'Scene Recorder '


     /**
     * This function is called to update Videomanager for DVD Source activation status
     */
     virtual tResult UpdateDVDVideoSourceStatus(const bool activeStatus);

     /**
     * This function is called to acknowledge Videomanager for DVD Source request
     */
     virtual tResult vSendVideoStateAcknowledgement(const tVideoSourceState currentStatus);

     virtual tResult AddOobPairedDevice(const tDeviceID deviceID,const tMACAddress deviceAddress,const tDeviceName deviceName, const tLinkKey linkKey,const tBTOobType oobType);
     /**
     * This function is called to update DVD Temperature as a Method Result
     */
     tResult SendRequestDVDTemperatureAnswer(tU16 temperature , tUserContext userContext);
     /**
     This function is to update the playbackspeed value on HMI
     *
     **/
     virtual tResult UpdatePlaybackSpeed();

     /**
      * This function call Updates the volume  to the Audio Component
      *
      * @param[in] Volume
      */
     virtual void UpdateVolumeToAudio(int Volume);

     /**
      * This function call Updates that the Browse structure is cleared.
      *
      * @param[in] Device id
      */
     virtual void UpdateFolderBrowseStructureCleared();

     /**
      * This function call gets the Current Volume, from Audio component
      *
      * @param[out] Value of volume from Audio component (indicates the current value of the volume for ENTERTAINMENT in the head unit)
      */
     virtual void GetCurrentVolumeFromAudio();

     /**
       * Updating that the device has updated that it is in playing and Mediaplayer is ready to Play due to remote acivity
     */
     virtual void UpdateReadyToPlay();

     /**
     * This function is called by the MediaPlayer core when DeviceTime information has changed .
     * The function calls eUpdateClients with parameter
     * MPLAY_APPCONTROLFI_C_U16_DIPODEVICETIMEUPDATE.
     * The framework then notifies all registered clients of the change.
     * @return < 0: error, = 0: OK
     */
     virtual tResult UpdateDiPODeviceTime();

     /**
      * This function call Get method of DeviceManagerNotifyConnection property.
      */
     virtual void RequestDeviceManagerNotifyConnection();

private:

    /**
    * For service class access
    */
    fc_mediaplayer_tclApp *mpApp;

    /**
    * Lock to avoid parallel access
    */
    Lock mSendMediaPlayerListChangedLock;
    Lock mSendFavoritesChangedLock;
    Lock mSendAppControlConnectLock;
    Lock mSendAppControlCommandLock;
    Lock mSendAppControlCloseLock;
    Lock mSendNativeTransportStartLock;
    Lock mSendNativeTransportStopLock;

    /**
    * Var lock
    */
    Lock mAccessLock;

    tMountPoint mDeviceIDConnect;
    tAppName mAppNameConnect;
    tSessionID mSessionIDConnect;
    tProtocolName mProtocolNameConnect;

    tMountPoint mDeviceIDCommand;
    tAppName mAppNameCommand;
    tSessionID mSessionIDCommand;
    tSize mSizeOfBufferCommand;
    tCommandBufferPtr mCommandBufferCommand;

    tMountPoint mDeviceIDClose;
    tAppName mAppNameClose;
    tSessionID mSessionIDClose;

    tDeviceID mDeviceIdStart;
    tU8 mIAP2iOSAppIdentifierStart;
    tU8 mSinkEndpointStart;
    tU8 mSourceEndpointStart;

    tDeviceID mDeviceIdStop;
    tU8 mIAP2iOSAppIdentifierStop;
    tU8 mSinkEndpointStop;
    tU8 mSourceEndpointStop;

    tListID mListID;
    tListChangeType mListChangeType;
    tListSize mListSize;
    vector<tPosition> mChangedItems;

    tListChangeType mFavListChangeType;
    tFavoriteID mFavID;
    tListSize mFavListSize;


    tBool mStartStopLocationInfo;
    tDiPOLocationInfoType mLocationInfoType;

    tDeviceID mDeviceIdComm;
    tIpodCommError mIpodCommError;
    tMountPoint mUSBSerialNumberComm;

    /**
    * Internal function to set list changed member variables
    */
    tResult SetMediaPlayerListChanged(tListID listID, tListChangeType listChangeType, tListSize listSize, vector<tPosition> changedItems);

    /**
    * Internal function to set fav changed member variables
    */
    tResult SetFavoriteChanged(tListChangeType listChangeType, tListSize listSize, tFavoriteID favID);

    /**
    * Internal function to set app control connect  member variables
    */
    tResult SetAppControlConnect(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID,const tProtocolName protocolName);

    /**
    * Internal function to set app control command member variables
    */
    tResult SetAppControlCommand(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID, const tSize sizeOfBuffer, const tCommandBufferPtr commandBuffer);

    /**
    * Internal function to set app control close member variables
    */
    tResult SetAppControlClose(const tMountPoint deviceID, const tAppName appName, const tSessionID sessionID);

    /**
     * Internal function to set native transport start member variables
     */
    tResult SetNativeTransportStart(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint );

    /**
     * Internal function to set native transport stop member variables
     */
    tResult SetNativeTransportStop(const tDeviceID deviceId, const tU8 iAP2iOSAppIdentifier, const tU8 sinkEndpoint, const tU8 sourceEndpoint );

    /**
    * Internal function to clear app control connect  member variables
    */
    tResult ClearAppControlConnect();

    /**
    * Internal function to claer app control command member variables
    */
    tResult ClearAppControlCommand();

    /**
    * Internal function to clear app control close member variables
    */
    tResult ClearAppControlClose();

    /**
    * Internal function to clear native transport start member variables
    */
    tResult ClearNativeTransportStart();

    /**
    * Internal function to clear native transport stop member variables
    */
    tResult ClearNativeTransportStop();

    /**
    * Internal function to update HMI for iPod Communication Errors
    */
    tResult SetIpodCommunicationError(const tDeviceID deviceId, const tIpodCommError iPodCommError);

    /**
    * Internal function to update SPI for iPod Communication Errors
    */
    tResult SetDipoCommunicationError(const tDeviceID deviceID, const tIpodCommError iPodCommError, const tMountPoint USBSerialNumber);

    void GetSource(unsigned int &source ,const tDeviceType deviceType,const tConnectionType connectionType);

};

/** @} */
