/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SubStateDebounce.cpp
  * @brief        is (if configured via SPM_FEATURE_ENABLE_SUBSTATE_DEBOUNCE) used to debounce SubStates
  *               This class can be used (when called from generic SubStateHandler) to configure when a SubState
  *               is set. This prevents that SubStates change too often and cause high system load when the "real"
  *               substate changes frequently.
  * @copyright    (C) 2016 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SubStateDebounce.h"

// spm class definitions w/o interface
#include "spm_security.h"

// interfaces class definitions
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_IWupOnOffEvents.h"

#include "spm_IFactory.h"

// spm helper

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_PRJ
#include "trcGenProj/Header/spm_SubStateDebounce.cpp.trc.h"
#endif

#include "spm_trace.h"

// //#define SPM_TRACE_FILE_ID   SPM_FILE_SUBSTATEVERIFY

// #define SPM_CPLD_VER_INTR_SUPPORT 5  //-> V0.5

spm_tclSubStateDebounce*spm_tclSubStateDebounce::_pMyStaticRef = 0;

/*!
  * \fn
  *  \brief
  *    Constructor.
  *    here the project specific list of substates and the parameter for debouncing are defined
  *    The list of substates to be debounces is created
  *    The message queue is create to allow context changes
  *
  *  \param
  *    factory: reference to factory class - caller of this function
  *  \version
  *    1.0   - Initial
  ******
  */
spm_tclSubStateDebounce::spm_tclSubStateDebounce( const ISpmFactory& factory ) : ISpmSubStateDebounce( factory ){
   _pMyStaticRef            = this;

   _poclSubStateHandler     = NULL;
   _poclOnOffEventHandler   = NULL;

   _bTestMode               = FALSE;
   _hGpioQueueHandle        = OSAL_C_INVALID_HANDLE;
   _u32ThreadWaitingDelay   = SPM_HWGPIO_DELAY_IDLE;
   _bPollingActive          = FALSE;
   _u32LastTrigger          = 0;

   TGpioCfg tIgnConfig = { SPM_U32_IGNITION, TRUE, 0, FALSE, FALSE, 4000, 12, 0, 0, FALSE, 4000, TRUE };

   _tGpioCfg[eGpioIgnition] = tIgnConfig;

   // create my msgbox
   if ( OSAL_ERROR == OSAL_s32MessageQueueCreate(
           SPM_MSGBOX_GPIO_NAME,
           SPM_MSGBOX_GPIO_MAX_COUNT,
           sizeof( tU32 ),
           OSAL_EN_READWRITE,
           &_hGpioQueueHandle
           )
        ){
      _hGpioQueueHandle = OSAL_C_INVALID_HANDLE;
   }
   return;
}

/*!
  * \fn
  *  \brief
  *    Destructor.
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
spm_tclSubStateDebounce::~spm_tclSubStateDebounce( ){

   if ( _hGpioQueueHandle != OSAL_C_INVALID_HANDLE ){
      OSAL_s32MessageQueueClose( _hGpioQueueHandle );
      OSAL_s32MessageQueueDelete( SPM_MSGBOX_GPIO_NAME );
      _hGpioQueueHandle = OSAL_C_INVALID_HANDLE;
   }

   _poclSubStateHandler   = NULL;
   _poclOnOffEventHandler = NULL;

   _pMyStaticRef          = NULL;
   return;
}

/*!
  * \fn
  *  \brief
  *    Get references to all required LCM classes.
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,   ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOnOffEventHandler, ISpmWupOnOffEvents );
}

/*!
  * \fn
  *  \brief
  *    start polling SubStates
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vStartCommunication( ){

   _u32ThreadWaitingDelay = SPM_HWGPIO_DELAY_IDLE;
   _bPollingActive        = FALSE;

} // vStartCommunication

/*!
  * \fn
  *  \brief
  *    Nothing to do
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vTraceSignals( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Nothing to do
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vCheckSignals( ){
   return;
}

/*!
  * \fn
  *  \brief
  *    Specific implementation of BaseClass: Start the polling
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vOnStart( ){
   SPM_NULL_POINTER_CHECK( _poclOnOffEventHandler );
   for ( tU32 i = 0; i < eGpioMax; i++ ){
      ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::vOnStart(): check! %d", i ) );
      if ( _tGpioCfg[i].bEnabled && _tGpioCfg[i].bState && _poclOnOffEventHandler->bIsOnOffStateActive( _tGpioCfg[i].u32TriggerType ) ){
         ETG_TRACE_USR1( ( "spm_tclSubStateDebounce::vOnStart(): Update() --> %d", i ) );
         bUpdateSubState( _tGpioCfg[i].u32TriggerType, TRUE );
      }
   }
}

/*!
  * \fn
  *  \brief
  *    Nothing is done.
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vOnTerminate( ){
}

/*!
  * \fn
  *  \brief
  *    Get the time where no change must be detected to make the substate valid.
  *
  *  \param u32GpioId - ID of the SubState in the GpioCfg Array
  *
  *  \return absolute time of next trigger for this SubState
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tU32 spm_tclSubStateDebounce::u32GetNextTriggerGpio( tU32 u32GpioId ) const {
   tU32 u32Ret = SPM_HWGPIO_TO_INVALID;

   if ( u32GpioId < eGpioMax ){
      u32Ret = 0;
      if ( _tGpioCfg[u32GpioId].bBlocked ){
         u32Ret = _tGpioCfg[u32GpioId].u32StartCountingTime + _tGpioCfg[u32GpioId].u32BlockTime;
         // ETG_TRACE_USR4(("spm_tclSubStateDebounce::u32GetNextTriggerGpio(): Gpio %u: state -> update at: %d", ETG_ENUM(SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId), u32Ret));
      }
   }
   return( u32Ret );
} // u32GetNextTriggerGpio

/*!
  * \fn
  *  \brief
  *    Check if substate update must be send and send it if required
  *    This function does the substate update if bUpdateSubState returned false
  *    called cyclically from main function
  *
  *  \param u32GpioId - ID of the SubState in the GpioCfg Array
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vCheckCount( tU32 u32GpioId ){

   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   if ( u32GpioId < eGpioMax ){
      tU32 u32CurTime = OSAL_ClockGetElapsedTime( );

      ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::vCheckCount(): Gpio %u: state: %d/%d, %d changes in last %d sec!", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), _tGpioCfg[u32GpioId].bHwState, _tGpioCfg[u32GpioId].bDebouncedState, _tGpioCfg[u32GpioId].u32ChangeCount, u32CurTime - _tGpioCfg[u32GpioId].u32StartCountingTime ) );

      if ( _tGpioCfg[u32GpioId].bBlocked ){
         // we are already locked --> only trigger is needed for continous blocking
         if ( u32CurTime > ( _tGpioCfg[u32GpioId].u32StartCountingTime + _tGpioCfg[u32GpioId].u32BlockTime ) ){
            _tGpioCfg[u32GpioId].bBlocked             = FALSE;
            ETG_TRACE_USR1( ( "spm_tclSubStateDebounce::vCheckCount(): Gpio %u: Block mode disabled now!!!!", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ) ) );

            _poclSubStateHandler->vSetSubStateType( _tGpioCfg[u32GpioId].u32TriggerType, FALSE, FALSE );

            _tGpioCfg[u32GpioId].u32StartCountingTime = u32CurTime;
            _tGpioCfg[u32GpioId].u32ChangeCount       = 0;


         } else if ( 0 < _tGpioCfg[u32GpioId].u32ChangeCount ){
            _tGpioCfg[u32GpioId].bBlocked             = TRUE;
            ETG_TRACE_USR1( ( "spm_tclSubStateDebounce::vCheckCount(): Gpio %u: We are already in blocking mode and receive %d changes in last %d sec --> Block!!!!", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), _tGpioCfg[u32GpioId].u32ChangeCount, u32CurTime - _tGpioCfg[u32GpioId].u32StartCountingTime ) );
            _tGpioCfg[u32GpioId].u32StartCountingTime = u32CurTime;
            _tGpioCfg[u32GpioId].u32ChangeCount       = 0;
         }
      }

      if ( ( u32CurTime > ( _tGpioCfg[u32GpioId].u32StartCountingTime + _tGpioCfg[u32GpioId].u32ChangeIntervall ) )
           || ( _tGpioCfg[u32GpioId].u32MaxCount <= _tGpioCfg[u32GpioId].u32ChangeCount )
           ){
         tBool bPreconditionBlock = TRUE; //for events true in anycase, for state GPIOs check if current level is inactive
         if ( ( _tGpioCfg[u32GpioId].bState == TRUE ) && ( _tGpioCfg[u32GpioId].bDebouncedState == FALSE )
              ){
            bPreconditionBlock = FALSE;
         }
         if ( bPreconditionBlock && ( _tGpioCfg[u32GpioId].u32MaxCount <= _tGpioCfg[u32GpioId].u32ChangeCount ) ){
            _tGpioCfg[u32GpioId].bBlocked             = TRUE;
            ETG_TRACE_USR1( ( "spm_tclSubStateDebounce::vCheckCount(): Gpio %u: %d changes in last %d sec --> Block changes!!!!", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), _tGpioCfg[u32GpioId].u32ChangeCount, u32CurTime - _tGpioCfg[u32GpioId].u32StartCountingTime ) );
            _tGpioCfg[u32GpioId].bDebouncedState      = FALSE;
            _poclSubStateHandler->vSetSubStateType( _tGpioCfg[u32GpioId].u32TriggerType, FALSE, FALSE );


            _tGpioCfg[u32GpioId].u32StartCountingTime = u32CurTime;
            _tGpioCfg[u32GpioId].u32ChangeCount       = 0;
         }
      }
   }
} // vCheckCount

/*!
  * \fn
  *  \brief
  *    Check if the Substate status needs to be updated
  *    called cyclically from main function
  *
  *  \param u32GpioId - ID of the SubState in the GpioCfg Array
  *         bForce - always send update
  *
  *  \return SPM_HWGPIO_IDLE | SPM_HWGPIO_SEND | SPM_HWGPIO_WAITING
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tU8 spm_tclSubStateDebounce::u8UpdateGpio( tU32  u32GpioId,
                                           tBool bForce ){
   // function returns
   tU8 u8Ret = SPM_HWGPIO_IDLE;

   SPM_NULL_POINTER_CHECK_VAL( _poclSubStateHandler );

   if ( u32GpioId < eGpioMax ){
      tU32 u32CurTime     = OSAL_ClockGetElapsedTime( );
      tU32 u32TriggerTime = u32GetNextTriggerGpio( u32GpioId );

      if ( ( ( _tGpioCfg[u32GpioId].bDebouncedState != _tGpioCfg[u32GpioId].bHwState )
             && ( u32CurTime > u32TriggerTime )
             )
           || bForce ){
         tU32 u32SubState = _tGpioCfg[u32GpioId].u32TriggerType;
         _tGpioCfg[u32GpioId].bDebouncedState = _tGpioCfg[u32GpioId].bHwState;
         if ( _tGpioCfg[u32GpioId].bHwState ){
            if ( !_poclSubStateHandler->bIsTriggerSet( u32SubState ) ){
               _poclSubStateHandler->vSetSubStateType( u32SubState, TRUE, FALSE );
               ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::u8UpdateGpio(): Trigger substate for Gpio %u: state: %d, change detected at: %d", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), ETG_ENUM( SPM_BOOL_STATE, _tGpioCfg[u32GpioId].bHwState ), _tGpioCfg[u32GpioId].u32LastChange ) );
            }
            //here add clearing for events
            if ( !_tGpioCfg[u32GpioId].bState ){
               _tGpioCfg[u32GpioId].bHwState        = FALSE;
               _tGpioCfg[u32GpioId].bDebouncedState = FALSE;
               _poclSubStateHandler->vSetSubStateType( u32SubState, FALSE, FALSE );
            }

         } else {
            if ( _poclSubStateHandler->bIsTriggerSet( u32SubState ) ){
               _poclSubStateHandler->vSetSubStateType( u32SubState, FALSE, FALSE );
               ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::u8UpdateGpio(): Trigger substate for Gpio %u: state: %d, change detected at: %d", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), ETG_ENUM( SPM_BOOL_STATE, _tGpioCfg[u32GpioId].bHwState ), _tGpioCfg[u32GpioId].u32LastChange ) );
            }
         }
         u8Ret = SPM_HWGPIO_SEND;
      }

      if ( _tGpioCfg[u32GpioId].bDebouncedState != _tGpioCfg[u32GpioId].bHwState ){  //lint !e661 Warning 661
         u8Ret = SPM_HWGPIO_WAITING;
      }
   }

   ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::u8UpdateGpio(): Gpio %u: u8Ret: %d", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), u8Ret ) );

   return( u8Ret );
} // u8UpdateGpio

/*!
  * \fn
  *  \brief
  *    main function of the thread
  *
  *  \param
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::main( ){

   tU32 u32Prio = 0;
   tU32 u32Msg  = 0;

   if ( _bPollingActive ){
      _u32ThreadWaitingDelay = SPM_HWGPIO_DELAY_POLLING;
      // wait for a message
      // writes the received OSALMsgHandle in the message object
   }

   tS32 s32ReadBytes = OSAL_s32MessageQueueWait
                       (
      _hGpioQueueHandle,
      (tU8*)&u32Msg, /*pointer of the MsgHandle field*/
      sizeof( tU32 ),
      &u32Prio,
      _u32ThreadWaitingDelay
                       );

   if ( s32ReadBytes <= 0 ){
      SPM_NULL_POINTER_CHECK( _poclOnOffEventHandler );
      for ( tU32 i = 0; i < eGpioMax; i++ ){
         if ( _tGpioCfg[i].bEnabled && _tGpioCfg[i].bState ){
            tBool bActiveState = _poclOnOffEventHandler->bIsOnOffStateActive( _tGpioCfg[i].u32TriggerType );
            ETG_TRACE_USR1( ( "spm_tclSubStateDebounce::main(): Check: bDebouncedState '%u', bActiveState '%u'", ETG_ENUM( SPM_BOOL_STATE, _tGpioCfg[i].bDebouncedState ), ETG_ENUM( SPM_BOOL_STATE, bActiveState ) ) );
            if ( _tGpioCfg[i].bDebouncedState != bActiveState ){
               bUpdateSubState( _tGpioCfg[i].u32TriggerType, bActiveState );
            }
         }
      }
   }


   // ETG_TRACE_USR4(("spm_tclSubStateDebounce::main(): MessageQueueWait returned: %d", s32ReadBytes));
   tBool bBlockMode = FALSE;
   _u32ThreadWaitingDelay = SPM_HWGPIO_DELAY_IDLE;

   ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::main(): Check: GpioId '%u'", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32Msg ) ) );

   for ( tU8 i = 0; i < eGpioMax; i++ ){

      if ( _tGpioCfg[i].bEnabled ){
         vCheckCount( i );
         if ( SPM_HWGPIO_WAITING == u8UpdateGpio( i, _bPollingActive ) ){
            tU32 u32Tmp = u32GetNextTriggerGpio( i ) - OSAL_ClockGetElapsedTime( );
            if ( u32Tmp < _u32ThreadWaitingDelay ){
               _u32ThreadWaitingDelay = u32Tmp;
            }
         }
         if ( !bBlockMode ){
            bBlockMode = _tGpioCfg[i].bBlocked;
         }
      }
   }

   if ( bBlockMode ){
      _u32ThreadWaitingDelay = SPM_HWGPIO_DELAY_BLOCK;
   }
} // main

/*!
  * \fn
  *  \brief
  *    Check if the Substate status is handled here or not
  *
  *  \param u32Substate - ID of the SubState in Substatehandler
  *         bActive - current state of Substate
  *
  *  \return TRUE - handled here - do NOT update in Substatehandler
  *          FALSE - NOT handled here - do update in Substatehandler*
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tBool spm_tclSubStateDebounce::bUpdateSubState( tU32  u32Substate,
                                                tBool bActive ){

   tBool bRet      = FALSE;
   tU32  u32GpioId = u32GetGpioId( u32Substate );

   if ( u32GpioId < eGpioMax ){

      if ( _tGpioCfg[u32GpioId].bEnabled ){

         tBool bStateChange = TRUE;
         if ( _tGpioCfg[u32GpioId].bState ){
            if ( _tGpioCfg[u32GpioId].bHwState != bActive ){
               _tGpioCfg[u32GpioId].bHwState = bActive;
            } else {
               //no change detected --> don't trigger update
               bStateChange = FALSE;
            }
         } else if ( bActive ){
            _tGpioCfg[u32GpioId].bHwState = bActive;
         }

         if ( bStateChange ){
            ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::bUpdateSubState(): post substate for Gpio %u: ActiveState: %d", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), ETG_ENUM( SPM_BOOL_STATE, bActive ) ) );

            tU32 u32CurTime = OSAL_ClockGetElapsedTime( );

            if ( ( u32CurTime - _tGpioCfg[u32GpioId].u32LastChange ) > _tGpioCfg[u32GpioId].u32ChangeIntervall ){
               _tGpioCfg[u32GpioId].u32StartCountingTime = u32CurTime;
               _tGpioCfg[u32GpioId].u32ChangeCount       = 0;
            }

            _tGpioCfg[u32GpioId].u32LastChange = u32CurTime;
            _tGpioCfg[u32GpioId].u32ChangeCount++;

            // post message to leave callback context
            if ( FALSE == bPostMessage( u32GpioId ) ){
               SPM_RESET_VIA_ASSERT;
            }
         }
         bRet = TRUE;
      }
   }

            ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::bUpdateSubState(): Substate %u, GpioId %u, bActive: %u, bRet: %u", ETG_ENUM( SPM_SUBSTATETYPE, u32Substate ), ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ), ETG_ENUM( SPM_BOOL_STATE, bActive ), ETG_ENUM( SPM_BOOL_STATE, bRet ) ) );

   return( bRet );
} // bUpdateSubState

/*!
  * \fn
  *  \brief
  *    get ID of substate in GpioConfig
  *
  *  \param u32Substate - ID of the SubState in Substatehandler
  *
  *  \return ID
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tU32 spm_tclSubStateDebounce::u32GetGpioId( tU32 u32SubState ){
   for ( tU32 i = 0; i < eGpioMax; i++ ){
      if ( _tGpioCfg[i].u32TriggerType == u32SubState ){
            ETG_TRACE_USR4( ( "spm_tclSubStateDebounce::u32GetGpioId(): Substate %u, GpioId %u", ETG_ENUM( SPM_SUBSTATETYPE, u32SubState ), ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, i ) ) );
         return( i );
      }
   }

   return( eGpioMax );
}

/*!
  * \fn
  *  \brief
  *    Post a message to change context
  *
  *  \param u32Msg - u32GpioId
  *
  *  \return TRUE - message send
  *          FALSE - message could not be send
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tBool spm_tclSubStateDebounce::bPostMessage( tU32 u32Msg ){
   tBool bSuccess = FALSE;
   tS32  s32Ret   = OSAL_s32MessageQueuePost( _hGpioQueueHandle,
                                              (tCU8*)&u32Msg,
                                              sizeof( tU32 ),
                                              OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );

   if ( s32Ret == OSAL_OK ){
      bSuccess = TRUE;
   }
   return( bSuccess );
}

/*!
  * \fn
  *  \brief
  *    disable debouncing for a single SubState
  *
  *  \param u32Substate - SubstateID from Substatehandler
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vDisableSubstateDebounce( tU32 u32Substate ){
   tU32 u32GpioId = u32GetGpioId( u32Substate );

   if ( u32GpioId < eGpioMax ){
      ETG_TRACE_FATAL( ( "spm_tclSubStateDebounce::vDisableSubstateDebounce(): GpioId '%u'", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ) ) );
      _tGpioCfg[u32GpioId].bEnabled = FALSE;
   }
}

/*!
  * \fn
  *  \brief
  *    enable debouncing for a single SubState
  *
  *  \param u32Substate - SubstateID from Substatehandler
  *
  *  \version
  *    1.0   - Initial
  ******
  */
tVoid spm_tclSubStateDebounce::vEnableSubstateDebounce( tU32 u32Substate ){
   tU32 u32GpioId = u32GetGpioId( u32Substate );

   if ( u32GpioId < eGpioMax ){
      ETG_TRACE_FATAL( ( "spm_tclSubStateDebounce::vEnableSubstateDebounce(): GpioId '%u'", ETG_ENUM( SPM_SUBSTATEDEBOUNCE_GPIOS, u32GpioId ) ) );
      _tGpioCfg[u32GpioId].bEnabled = TRUE;
   }
}

// EOF

