/*!
 * \file spm_SubStateHandlerConfig.cpp
 *  \brief
 *    Configuration of SubStates
 *
 *  \note
 *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
 *  \see
 *  \version
 * Date      | Author            | Modification
 * 07.09.11  | TMS Fischer       | initial version
 ******
 */

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_StartupSystemConfig.h"

// interfaces class definitions
#include "spm_IOsLinux.h"
#include "spm_LxProcInfo.h"
#include "spm_IStartupInvestigationServer.h"

#include "spm_IRegistry.h"
#include "spm_IFactory.h"

// spm helper
#include "spm_StartupCommon.h"
#include "spm_Registry_pathes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_StartupSystemConfig.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
#define SPM_U8_CONFIG_OEM_TYPE_DIAMLER      0x10

#define SPM_U8_CONFIG_VEHICLE_TYPE_TRUCK    0x00
#define SPM_U8_CONFIG_VEHICLE_TYPE_COACH    0x01

#define SPM_U8_CONFIG_SYSTEM_TYPE_GENERIC   0x03
#define SPM_U8_CONFIG_SYSTEM_TYPE_EVOBUS    0x04

#define SPM_U8_CONFIG_DEST_REGION_US	    0x00
#define SPM_U8_CONFIG_DEST_REGION_CAN       0x01
#define SPM_U8_CONFIG_DEST_REGION_MEX       0x02
#define SPM_U8_CONFIG_DEST_REGION_UK        0x03 
#define SPM_U8_CONFIG_DEST_REGION_EUR       0x06


#define SPM_STR_STARTUP_REG_PATH_CODING  "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/CONFIG/REGISTRY/"

spm_tclStartupSystemConfig::spm_tclStartupSystemConfig(  const ISpmFactory& factory,
        const std::string& strConfigPath,
        const std::string& strInstance,
        tBool              bReadFirstRegistry )
: spm_tclStartupSystem( factory, strConfigPath, strInstance, bReadFirstRegistry ) {

}

spm_tclStartupSystemConfig::~spm_tclStartupSystemConfig( ){
}

tBool spm_tclStartupSystemConfig::bAlternativeProcConfig(tChar *strProcessName, tBool bValidationOnly){
	tBool           bRet         = TRUE;
    // get region
    reg_tclRegKey   oRegKey;
	
	ISpmRegistry* poclRegistry = dynamic_cast < ISpmRegistry* >( _crfFactory.getSpmObjHandler("ISpmRegistry") );
    SPM_NULL_POINTER_CHECK_VAL(_poclRegistry);
	
	//variant handling (used for SDS registry --> extract OEM Type first)
	if (!bValidationOnly) {
        tU8  u8OemType = 0;
        if ( DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "OEMType", &u8OemType, 1) ){

            tChar acPathBuffer[256] = { 0 };
            tChar acReg[256]        = { 0 };
            OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "OEMTYPE/%03d", u8OemType);

            ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Check for alternative registry (OEMTYPE depending). Try to open '%s'.", acPathBuffer ) );

            if (oRegKey.bOpen(acPathBuffer) ){
                ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                if (!oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                    // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
					ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): acReg: '%s'.", acReg ) );
                } else {
                    if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                        ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
                        return( bRet );
                    }
                    ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
                }
            }
        } else {
            ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): !!!!!!!!!!!! Failed to read KDS region entry --> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!" ) );
        }
    }
	
	
	
    //variant handling (extracting the Vehicle Type --> Truck/Coach)
    //Vehicle Type = 2 bytes
    tU8 u8VehicleType[2] = {0};
    tU16 u16VehicleType = SPM_U8_CONFIG_VEHICLE_TYPE_COACH; //setting default as Coach
    if ( DP_S32_NO_ERR ==  DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2)){

        if (!bValidationOnly) {
            tBool bFoundReg = FALSE;
			
	    ETG_TRACE_USR4(("spm_tclStartupSystem::bAlternativeProcConfig(): Vehicle type First byte : %x, Second byte : %x", u8VehicleType[0], u8VehicleType[1]));

	    //Motorola-HighLow-order needs to be reversed
	    u16VehicleType = static_cast<tU16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
	    ETG_TRACE_COMP(("spm_tclStartupSystem::bAlternativeProcConfig(): Vehicle type : %d", u16VehicleType));

            tChar acPathBuffer[256] = { 0 };
            tChar acReg[256]        = { 0 };
            OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "VEHICLETYPESELECTOR/%03d", u16VehicleType);

            ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Check for alternative registry (VEHICLETYPESELECTOR depending). Try to open '%s'.", acPathBuffer ) );

            if (oRegKey.bOpen(acPathBuffer) ){
                if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                    ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                    bFoundReg = TRUE;
                }
            } else if (oRegKey.bOpen(SPM_STR_STARTUP_REG_PATH_CODING "VEHICLETYPESELECTOR/DEFAULT") ){
                if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                    ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Path is not valid for Key. Use DEFAULT entry for '%s'.", strProcessName ) );
                    bFoundReg = TRUE;
                }
            }


            if (bFoundReg){
                if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                    ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
                    return( bRet );
                }
                if (  ( 0 == OSAL_s32StringCompare(strProcessName, "STARTCFG") )) {
                    // store in startup investigation
                    ISpmStartupInvestigationServer *_poclStartupInvest = dynamic_cast < ISpmStartupInvestigationServer* >( ISpmFactory::_poFactoryRef->getSpmObjHandler("ISpmStartupInvestigationServer") );

                    if (_poclStartupInvest){
                        SPM_STARTUPINVEST_INIT_STARTUPITEM
                        SPM_STARTUPINVEST_FORCE_ENTRY("STARTUP", ( (std::string)"Startup registry: '" + acReg + "' loaded." ) )
                    }
                }

                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
            }
        }
    } else {

        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): !!!!!!!!!!!! Failed to read 'VehicleInformation/VehicleType' --> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!" ) );
        //u8VariantSelection = SPM_U8_CONFIG_VEHICLE_TYPE_COACH;
    }
	
	
	//system type selector handling (extract Variant Handling first) --> Generic/Evobus
    tU8 u8SystemTypeSelector = SPM_U8_CONFIG_SYSTEM_TYPE_EVOBUS;
	if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "SystemTypeSelector", &u8SystemTypeSelector, 1)) {

		tBool bFoundReg = FALSE;

		tChar acPathBuffer[256] = { 0 };
		tChar acReg[256] = { 0 };
		OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof(acPathBuffer), SPM_STR_STARTUP_REG_PATH_CODING "SYSTEMTYPESELECTOR/%03d", u8SystemTypeSelector);

		ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): Check for alternative registry (SYSTEMTYPESELECTOR depending). Try to open '%s'.", acPathBuffer));

		if (oRegKey.bOpen(acPathBuffer)) {
			ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): Path is valid. Check for KEY: '%s'.", strProcessName));
			if (!oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof(acReg))) {
				// ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
			}
			else {
				if (!poclRegistry->bImportDataByFile(acReg, FALSE)) {
					ETG_TRACE_FATAL(("Could not read registry of process '%s'", acReg));
					return(bRet);
				}
				if ((0 == OSAL_s32StringCompare(strProcessName, "STARTCFG"))) {
					// store in startup investigation
					ISpmStartupInvestigationServer *_poclStartupInvest = dynamic_cast <ISpmStartupInvestigationServer*>(ISpmFactory::_poFactoryRef->getSpmObjHandler("ISpmStartupInvestigationServer"));

					if (_poclStartupInvest) {
						SPM_STARTUPINVEST_INIT_STARTUPITEM
							SPM_STARTUPINVEST_FORCE_ENTRY("STARTUP", ((std::string)"Startup registry: '" + acReg + "' loaded."))
					}
				}

				ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg));
			}
		}
	}
	else {
		ETG_TRACE_FATAL(("spm_tclStartupSystem::bAlternativeProcConfig(): !!!!!!!!!!!! Failed to read 'AIVIVariantCoding/SystemTypeSelector' --> Check target configuration (KDS 'AIVIVariantCoding') !!!!!!!!!!!!"));
	}
	
	
	// Destination region handling
    tBool           bRegionValid = FALSE;
    tU8             u8RegionCode = SPM_U8_CONFIG_DEST_REGION_EUR;
    if ( DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8RegionCode, 1) ){
        bRegionValid = TRUE;
    }

	if (!bValidationOnly) {
		//variant handling (used for navigation registry)
		if (bRegionValid) {

			tChar acPathBuffer[256] = { 0 };
			tChar acReg[256] = { 0 };
			OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof(acPathBuffer), SPM_STR_STARTUP_REG_PATH_CODING "REGION/%03d", u8RegionCode);

			ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): Check for alternative registry (REGION depending). Try to open '%s'.", acPathBuffer));

			if (oRegKey.bOpen(acPathBuffer)) {
				ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): Path is valid. Check for KEY: '%s'.", strProcessName));
				if (!oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof(acReg))) {
					// ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
				}
				else {
					if (!poclRegistry->bImportDataByFile(acReg, FALSE)) {
						ETG_TRACE_FATAL(("Could not read registry of process '%s'", acReg));
						return(bRet);
					}
					ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg));
				}
			}
		}
		else {
			ETG_TRACE_FATAL(("spm_tclStartupSystem::bAlternativeProcConfig(): !!!!!!!!!!!! Failed to read KDS region entry 'VehicleInformation/DestinationRegion1'--> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!"));
		}
	}

	
	return( bRet );
	
}  // bAlternativeProcConfig


tU32 spm_tclStartupSystemConfig::u32CalculateStartupRegistry(tU32 u32StartupConfig){
   /*!
     * \fn
     *  \brief
     *    Project configuration for support of CPU cores
     *
     *  \param
     *  \return  tU32
     *
     ******
     */

    // update startup configuration depending of active COREs
    tU32 u32NumberOfCores = 1;
    {
        spm_tclLxCpuInfo oCpuInfo;
        u32NumberOfCores = oCpuInfo.u32GetNumberOfCpus();
    }

    tU32 u32Offset = 0; //for single core
    if (u32NumberOfCores == 2) {
        u32Offset = SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
    } else if (u32NumberOfCores == 4) {
        u32Offset = 2*SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
    }

    tU32 u32CfgNb = u32StartupConfig % SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
    u32CfgNb += u32Offset;
    u32CfgNb = u32CfgNb % SPM_STARTUP_CONFIGURATION_MAX_NB;

    tChar cCfgRegs[SPM_STARTUP_CONFIGURATION_MAX_NB][200] = SPM_STARTUP_CONFIGURATION;
    ETG_TRACE_USR1( ( "spm_tclStartupSystemConfig::u32CalculateStartupRegistry(): Number of COREs: %d, Config: %d --> '%s'!", u32NumberOfCores, u32CfgNb, cCfgRegs[u32CfgNb] ) );

    //check if file is in FS
    OSAL_tIODescriptor fd = OSAL_IOOpen(cCfgRegs[u32CfgNb], OSAL_EN_READONLY);
    if(fd == OSAL_ERROR){
        ETG_TRACE_USR1( ( "spm_tclStartupSystemConfig::u32CalculateStartupRegistry(): File not found: '%s'. Set config w/o core dependencies.", cCfgRegs[u32CfgNb] ) );
        u32CfgNb = u32CfgNb % SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
        ETG_TRACE_USR1( ( "spm_tclStartupSystemConfig::u32CalculateStartupRegistry(): set startup file: '%s'", cCfgRegs[u32CfgNb] ) );
    } else {
        OSAL_s32IOClose(fd);
    }

    return u32CfgNb;
}

tVoid spm_tclStartupSystemConfig::vLoadSelectorRegistries( tChar *strProcessName) {
	
	// get region
    reg_tclRegKey   oRegKey;

    ISpmRegistry* poclRegistry = dynamic_cast < ISpmRegistry* >( _crfFactory.getSpmObjHandler("ISpmRegistry") );
    SPM_NULL_POINTER_CHECK(_poclRegistry);
	
	//variant handling (used for SDS registry --> extract OEM Type first)
    tU8  u8OemType = 0;
    if ( DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "OEMType", &u8OemType, 1) ){

        tChar acPathBuffer[256] = { 0 };
        tChar acReg[256]        = { 0 };
        OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "OEMTYPE/%03d", u8OemType);

        ETG_TRACE_USR1( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Check for alternative registry (OEMTYPE depending). Try to open '%s'.", acPathBuffer ) );

        tBool bFoundReg = FALSE;
        if (oRegKey.bOpen(acPathBuffer) ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        } else if (oRegKey.bOpen(SPM_STR_STARTUP_REG_PATH_CODING "OEMTYPE/DEFAULT") ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is not valid for Key. Use DEFAULT entry for '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        }
        if (bFoundReg){
            if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
            } else {
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
            }
        }
    } else {
        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): !!!!!!!!!!!! Failed to read KDS region entry --> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!" ) );
    }
	
	//variant handling (used for SDS registry --> extract vechile type -> Truck/Coach)
    //Vehicle Type = 2 bytes
    tU8 u8VehicleType[2] = {0};
    tU16 u16VehicleType = SPM_U8_CONFIG_VEHICLE_TYPE_COACH; //default as Coach
    if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2)){
      	ETG_TRACE_USR4(("spm_tclStartupSystem::vLoadSelectorRegistries(), Vehicle type First byte : %x, Second byte : %x", u8VehicleType[0], u8VehicleType[1]));

      	//Motorola-HighLow-order needs to be reversed
      	u16VehicleType = static_cast<tU16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
      	ETG_TRACE_COMP(("spm_tclStartupSystem::vLoadSelectorRegistries(), Vehicle type : %d", u16VehicleType));
 
        tBool bFoundReg = FALSE;

        tChar acPathBuffer[256] = { 0 };
        tChar acReg[256]        = { 0 };
        OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "VEHICLETYPESELECTOR/%03d", u16VehicleType);

        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Check for alternative registry (VEHICLETYPESELECTOR depending). Try to open '%s'.", acPathBuffer ) );

        if (oRegKey.bOpen(acPathBuffer) ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                bFoundReg = TRUE;

            }
        } else if (oRegKey.bOpen(SPM_STR_STARTUP_REG_PATH_CODING "VEHICLETYPESELECTOR/DEFAULT") ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is not valid for Key. Use DEFAULT entry for '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        }
        if (bFoundReg){
            if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
            } else {
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
            }
        }
    } else {

        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): !!!!!!!!!!!! Failed to read 'VehicleInformation/VehicleType' --> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!" ) );
        //u8VariantSelection = SPM_U8_CONFIG_OEM_TYPE_DIAMLER;
    }

    //system type selector handling (extract Variant Handling first) --> Generic/Evobus//
    tU8 u8SystemTypeSelector = SPM_U8_CONFIG_SYSTEM_TYPE_EVOBUS;

    if ( DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "SystemTypeSelector", &u8SystemTypeSelector, 1) ){
 		
	tChar acPathBuffer[256] = { 0 };
        tChar acReg[256]        = { 0 };
        OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "SYSTEMTYPESELECTOR/%03d", u8SystemTypeSelector);

        ETG_TRACE_USR1( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Check for alternative registry (SYSTEMTYPESELECTOR depending). Try to open '%s'.", acPathBuffer ) );

        tBool bFoundReg = FALSE;
        if (oRegKey.bOpen(acPathBuffer) ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        } else if (oRegKey.bOpen(SPM_STR_STARTUP_REG_PATH_CODING "SYSTEMTYPESELECTOR/DEFAULT") ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is not valid for Key. Use DEFAULT entry for '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        }
        if (bFoundReg){
            if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
            } else {
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
            }
        }
    } else {

        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): !!!!!!!!!!!! Failed to read 'AIVIVariantCoding/SystemTypeSelector' --> Check target configuration (KDS 'AIVIVariantCoding') !!!!!!!!!!!!" ) );
    }


	// Destination region handling
    tBool           bRegionValid = FALSE;
    tU8             u8RegionCode = 0;
    if ( DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8RegionCode, 1) ){
        bRegionValid = TRUE;
    }

    //variant handling (used for navigation registry)
    if (bRegionValid){

        tChar acPathBuffer[256] = { 0 };
        tChar acReg[256]        = { 0 };
        OSALUTIL_s32SaveNPrintFormat(acPathBuffer, sizeof( acPathBuffer ), SPM_STR_STARTUP_REG_PATH_CODING "REGION/%03d", u8RegionCode);

        ETG_TRACE_USR1( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Check for alternative registry (REGION depending). Try to open '%s'.", acPathBuffer ) );

        tBool bFoundReg = FALSE;
        if (oRegKey.bOpen(acPathBuffer) ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is valid. Check for KEY: '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        } else if (oRegKey.bOpen(SPM_STR_STARTUP_REG_PATH_CODING "REGION/DEFAULT") ){
            if (oRegKey.bQueryString(strProcessName, (tChar*)acReg, sizeof( acReg ) ) ){
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): Path is not valid for Key. Use DEFAULT entry for '%s'.", strProcessName ) );
                bFoundReg = TRUE;
            }
        }
        if (bFoundReg){
            if (!poclRegistry->bImportDataByFile(acReg, FALSE) ){
                ETG_TRACE_FATAL( ( "Could not read registry of process '%s'", acReg ) );
            } else {
                ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): registry is loaded [%d], process '%20s' -> '%s'", OSAL_ClockGetElapsedTime(), strProcessName, acReg) );
            }
        }

    } else {
        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::vLoadSelectorRegistries(): !!!!!!!!!!!! Failed to read KDS region entry 'VehicleInformation/DestinationRegion1'--> Check target configuration (KDS 'VehicleInformation') !!!!!!!!!!!!" ) );
    }
}

// EOF

