/*!
  * \file spm_service_csm.cpp
  *  \brief
  *   Project specific CAN adaption.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  *
  ******
  */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define CSM_C_PROJECT_USERSPACE_EVOBUS
#define MIDW_COMMON_S_IMPORT_INTERFACE_CSM
#include "midw_common_if.h"

// #define CSM_S_IMPORT_INTERFACE_GENERIC //lint -e750 PQM_authorized_multi_55 -> only used by paramount project
/*#define CSM_C_PROJECT_SMART_HU
#define CSM_C_VERSION                                     0x88010001  */     
//#include "csm_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define PNM_SIGNAL_CONSIDERED 0x00
#define PNM_SIGNAL_IGNORED    0x01

#define PNM_GRP10_SIGNAL_OFF 0
#define PNM_GRP10_SIGNAL_ON  1

#define SPM_U8_CONFIG_SYSTEM_TYPE_EVOBUS    0x04

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_service_csm.h"

// spm class definitions w/o interface
#include "spm_security.h"

// interfaces class definitions
#include "spm_ISystemStateManager.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISuperVisionClient.h"
#include "spm_IOsalProxy.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IStatistics.h"
#include "spm_IWakeupHandler.h"
#include "spm_SubStateHandlerConfig.h"
#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_PRJ
#include "trcGenProj/Header/spm_service_csm.cpp.trc.h"
#endif

#ifndef CSM_C_PROC_MESSAGE_QUEUE_NAME
#define CSM_C_PROC_MESSAGE_QUEUE_NAME         "PROC_CSM_MSG_QUEUE"
#endif
#include "spm_trace.h"


spm_tclCsmIf*spm_tclCsmIf::_pMyStaticRef    = 0;

spm_tclCsmIf::spm_tclCsmIf( const ISpmFactory& factory ) : ISpmCsmService( factory )
{
	ETG_TRACE_USR4(("SPM: !!!!!! spm_tclCsmIf !!!!!!"));
	_pMyStaticRef     	= this;
	_poMyCSMInterface 	= NULL;
	_poclWorkerServer	= NULL;   
	_poclSubStateHandler	= NULL;
	_poclSystemPowerManager	= NULL;
	_poclSystemStateManager	= NULL;
	_poLcmdbusclientprj	= NULL;
   
   return;
}

spm_tclCsmIf::~spm_tclCsmIf( )
{

	SPM_NULL_POINTER_CHECK( _poclWorkerServer );
	_poclWorkerServer->vRemoveClient((ISpmWorkerClient *)this );
	_poclSubStateHandler    = NULL;
	_poclSystemPowerManager = NULL;
	_poclWorkerServer       = NULL;   
	_poclSystemStateManager = NULL;
	_poLcmdbusclientprj	= NULL;

   delete _poMyCSMInterface;

   return;
}

tVoid spm_tclCsmIf::vGetReferences( )
{
	ETG_TRACE_USR4(("spm_tclCsmIf::vGetReferences :entered"));
   // get all needed references now -> all SPM objects are now available
	_poclSubStateHandler    = dynamic_cast < ISpmSubStateClient* >( _crfFactory.getSpmObjHandler( "ISpmSubStateClient" ) );
	_poclSystemPowerManager = dynamic_cast < ISpmSystemPowerManager* >( _crfFactory.getSpmObjHandler( "ISpmSystemPowerManager" ) );
	_poclWorkerServer       = dynamic_cast < ISpmWorkerServer* >( _crfFactory.getSpmObjHandler("ISpmWorkerServer") );
	_poclSystemStateManager = dynamic_cast < ISpmSystemStateManager* >(_crfFactory.getSpmObjHandler("ISpmSystemStateManager"));
	_poLcmdbusclientprj = dynamic_cast <spm_tclSpmDBusServiceHandler*>( _crfFactory.getClientHandler("spm_tclSpmDBusServiceHandler") );

   
	SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
	SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
	SPM_NULL_POINTER_CHECK(_poclWorkerServer);
	_poclWorkerServer->vAddClient((ISpmWorkerClient *) this );
	SPM_NULL_POINTER_CHECK(_poclSystemStateManager);
	SPM_NULL_POINTER_CHECK(_poLcmdbusclientprj);


}
tVoid spm_tclCsmIf::vStartCommunication(tVoid)
{
	ETG_TRACE_USR4(("spm_tclCsmIf::vStartCommunication(): entered"));	
    // wait for CAN interface, but wait max. 300ms --> if failed, try to connect in late phase
    if (!bConnectCsmIf(300)) {
        ETG_TRACE_FATAL(("spm_tclCsmIf::vStartCommunication(): Failed to connect to CAN."));
    }

}

tVoid spm_tclCsmIf::vStartLateCommunication( tVoid )
{
/*!
  * \fn
  *  \brief
  *    Synchronize the startup. Until now the FC_SPM is fully running.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

	ETG_TRACE_USR4(("spm_tclCsmIf::vStartLateCommunication(): entered"));

   // wait for CAN interface, but wait max. 300ms --> if failed, try to connect in late phase
   if (!bConnectCsmIf(300)) {
	   ETG_TRACE_FATAL(("spm_tclCsmIf::vStartLateCommunication(): Failed to connect to CAN."));
   }
   
} // vStartCommunication

tBool spm_tclCsmIf::bConnectCsmIf(tU32 u32MaxWaitingTime) 
{
	/*!
	* \fn
	*  \brief
	*    Instanciate CsmIf and set callback functions
	*
	*  \param
	*  \note
	*  \version
	*    1.0   - Initial
	******
	*/

	ETG_TRACE_USR4(("spm_tclCsmIf::bConnectCsmIf(): entered"));
	tU32 u32CsmHandle;
	tBool bRet = TRUE;

	if (_poMyCSMInterface == NULL) 
	{
		ETG_TRACE_USR4(("spm_tclCsmIf::bConnectCsmIf(): _poMyCSMInterface ==NULL"));
		tS32 s32CanActive = CSMInterface::CSM_s32WaitForCAN(u32MaxWaitingTime);
		if (s32CanActive < 0) 
		{
			ETG_TRACE_FATAL(("spm_tclCsmIf::bConnectCsmIf() CAN currently not active."));
			bRet = FALSE;
		} 
		else
		{
			ETG_TRACE_FATAL(("spm_tclCsmIf::bConnectCsmIf() Waiting for CAN: %d ms.", s32CanActive));

			_poMyCSMInterface = new CSMInterface();
			SPM_NULL_POINTER_CHECK_VAL(_poMyCSMInterface);
			
			if((getKDSPNMGRP10Status() == PNM_SIGNAL_CONSIDERED)&&(getKDSSystemType()== SPM_U8_CONFIG_SYSTEM_TYPE_EVOBUS))//Checking whether PNM_GRP10 signal is considered in KDS, PNM_GRP10 should ON only for Evobus variant
			{
				if (_poMyCSMInterface->CSM_lSignalCallbackInit((tVoid*)&u32CsmHandle, CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2 , vSignalInd) == CSM_C_NO_ERROR) 
				{
					vHandlePNMGrp10Stat();
				}
				else
				{
					ETG_TRACE_FATAL(("spm_tclCsmIf::bConnectCsmIf(): CSM_lSignalCallbackInit failed for 'CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2'"));
				}
			}
		}
	}
	else 
	{
		ETG_TRACE_USR4(("spm_tclCsmIf::bConnectCsmIf(): CheckCsmSignals"));
		//reload signals
		vCheckCsmSignals();
	}
	return bRet;
}
/*Reading  PNM_GRP10 signal status*/
tU8 spm_tclCsmIf::getKDSPNMGRP10Status()
{
 	tU8 u8PNMGRP10Status = PNM_SIGNAL_IGNORED;	//If PNM Group read fails then CAN Signal Ignored is considered. 
   	if (DP_S32_NO_ERR == DP_s32GetConfigItem("dev_NVM_Configuration", "PNMGroup10Switch", (tU8*) &u8PNMGRP10Status, 1))	//Configuring for PNMGroup10Switch to know the customer KDS value of the parameter
   	{
      		ETG_TRACE_USR4(("spm_tclCsmIf::getKDSPNMGRP10Status u8PNMGRP10Status(%d)", u8PNMGRP10Status));
   	}
    	else
   	{
     		ETG_TRACE_FATAL(("Error!!! spm_tclCsmIf::getKDSPNMGRP10Status u8PNMGRP10Status(%d)", u8PNMGRP10Status));
  	}
   return u8PNMGRP10Status;
}

/*Reading  SystemType - Evobus/generic */
tU8 spm_tclCsmIf::getKDSSystemType()
{
	tU8 u8SystemTypeSelector = 0;
	if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "SystemTypeSelector", &u8SystemTypeSelector, 1))
	{
		ETG_TRACE_USR4(("spm_tclCsmIf::getKDSSystemType u8SystemTypeSelector(%d)", u8SystemTypeSelector));
	}
	else
	{
		ETG_TRACE_FATAL(("Error!!! spm_tclCsmIf::getKDSSystemType u8SystemTypeSelector(%d)", u8SystemTypeSelector));
	}
	return u8SystemTypeSelector;
}

tVoid spm_tclCsmIf::vHandleMessage( tU32 u32Message, tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    Handles the Message from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm:
  ******
  */
   (tVoid)u32Parm;
   ETG_TRACE_USR4( ( "spm_tclCsmIf::vHandleMessage(): New mssage received: 0x%08x (param: 0x%08x)", u32Message, u32Parm ) );

   if ( SPM_U32_WORKER_CSM_SIG_IND == u32Message ){
       vSignalIndication( u32Parm );
   }
} // vHandleMessage

tBool spm_tclCsmIf::bHandleSynchrounousCall( tU32   u32Message, tVoid *args ){
/*!
  * \fn
  *  \brief
  *   spm_tclWorkerServer invokes this method for notifying messages.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] args: not used.
  ******
  */
   (tVoid)args;

   return( FALSE );
} // bHandleSynchrounousCall

/* ------------------------------------------------------------------------- */
tVoid spm_tclCsmIf::vSignalInd( tVoid *pHandle,
                                tU32   u32SignalId,
                                tU32   u32SignalStatus ){
/*!
  * \fn
  *  \brief
  *    callback function to inform about new signals from CSM.
  *
  *  \param
  *
  *  \return
  *
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

	ETG_TRACE_USR4(("spm_tclCsmIf::vSignalInd(): entered"));

	(tVoid)pHandle;
	(tVoid)u32SignalStatus;

	/* Check which signal is changed ! */
	ETG_TRACE_USR4(("spm_tclCsmIf::vSignalInd(): Signal Indication received: ID 0x%08x, Status: 0x%08x", u32SignalId, u32SignalStatus));
	//post message to handle Signal synchronous

	SPM_NULL_POINTER_CHECK(_pMyStaticRef->_poclSystemStateManager);
	SPM_NULL_POINTER_CHECK(_pMyStaticRef->_poclWorkerServer);

	_pMyStaticRef->vSignalIndication(u32SignalId);

}    // vSignalInd


tVoid spm_tclCsmIf::vSignalIndication(tU32 u32SignalId)
{
	/*!
	* \fn
	*  \brief
	*    callback function to inform about new signals from CSM.
	*
	*  \param
	*
	*  \return
	*
	*  \note
	*  \version
	*    1.0   - Initial
	******
	*/
	/* Check which signal is changed ! */
	ETG_TRACE_USR4(("spm_tclCsmIf::vSignalIndication(): Signal Indication received: ID 0x%08x", u32SignalId));

	switch (u32SignalId) 
	{
		case CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2 :
			_pMyStaticRef->vHandlePNMGrp10Stat();
			break;		
		
		default:
			ETG_TRACE_USR4(("Error: Signal Indication received: ID 0x%08x", u32SignalId));
			break;
	} // switch
}    // vSignalInd

bool spm_tclCsmIf::CheckOpcode() {
	/*!
	* \fn
	*  \brief
	*    Function to read the Opcode
	*
	*  \param
	*
	*  \return
			True: If opcode is 0x80
			False: signal read failed or opcode is not equal to 0x80
	*
	*  \note
	*  \version
	*    1.0   - Initial
	******
	*/

	ETG_TRACE_USR4(("spm_tclCsmIf::CheckOpcode(): entered"));

	tU8  u8Data_ReadBuffer;
	tU32 u32StatusRead = CSM_C_SIGNAL_DATA_INVALID;
	tU32 u32CsmHandle;
	bool bRet=FALSE;
	
	//Read opcode
	if (CSM_C_NO_ERROR <= _poMyCSMInterface->CSM_lSignalRead((tVoid*)&u32CsmHandle, CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2,
		&u8Data_ReadBuffer,1, &u32StatusRead)) 
	{
		//if data buffer contains valid data
		if (((u32StatusRead & CSM_C_SIGNAL_DATA_INVALID) == 0) && ((u32StatusRead & CSM_C_SIGNAL_TIMEOUT) == 0)) 
		{
				
			bRet = TRUE;
			ETG_TRACE_USR4(("ERROR:spm_tclCsmIf::CheckOpcode: VALID data"));

		}
		else 
		{
			ETG_TRACE_FATAL(("ERROR:spm_tclCsmIf::CheckOpcode: INVALID data"));
		}		
	}
	else 
	{
		ETG_TRACE_FATAL(("ERROR:spm_tclCsmIf::CheckOpcode: read failed"));
	} 
	return bRet;
	

}

tVoid spm_tclCsmIf::vHandlePNMGrp10Stat() {
	/*!
	* \fn
	*  \brief
	*    Function to read the CAN status of PNM_Grp10_Stat
	*
	*  \param
	*
	*  \return
	*
	*  \note
	*  \version
	*    1.0   - Initial
	******
	*/

	ETG_TRACE_USR4(("spm_tclCsmIf::vHandlePNMGrp10Stat(): entered"));
	
	tU8  u8Data_ReadBuffer;
	tU32 u32StatusRead = CSM_C_SIGNAL_DATA_INVALID;
	tU32 u32CsmHandle;

	SPM_NULL_POINTER_CHECK(_poMyCSMInterface);
	SPM_NULL_POINTER_CHECK(_poclSubStateHandler);


	ETG_TRACE_USR4(("spm_tclCsmIf::vHandlePNMGrp10Stat(): Signal CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2  received"));
	if (CheckOpcode ())
	{
		if (CSM_C_NO_ERROR <= _poMyCSMInterface->CSM_lSignalRead((tVoid*)&u32CsmHandle, CSM_C_SIG_RX_PNM_Grp10_Stat_SCA_C2,&u8Data_ReadBuffer,1, &u32StatusRead)) 
		{
			/*if data buffer contains valid data: */
			if (((u32StatusRead & CSM_C_SIGNAL_DATA_INVALID) == 0) && ((u32StatusRead & CSM_C_SIGNAL_TIMEOUT) == 0)) 
			{
				ETG_TRACE_USR4(("spm_tclCsmIf::vHandlePNMGrp10Stat: PNM_Grp10: 0x%08x", u8Data_ReadBuffer));
				_poLcmdbusclientprj->vPNMGRP10SignalstateUpdate(u8Data_ReadBuffer);
				if (u8Data_ReadBuffer == PNM_GRP10_SIGNAL_OFF)		//PNM_Grp10 Signal OFF (Battery low)
				{
					_poclSubStateHandler->vSetSubStateType(SPM_U32_SHUTDOWN_BATTERY_SAVING, TRUE);
					ETG_TRACE_USR4(("spm_tclCsmIf::vHandlePNMGrp10Stat: OFF"));					
				}
				else if(u8Data_ReadBuffer == PNM_GRP10_SIGNAL_ON) 	//PNM_Grp10 Signal ON (Battery normal)
				{
					_poclSubStateHandler->vSetSubStateType(SPM_U32_SHUTDOWN_BATTERY_SAVING, FALSE);
					ETG_TRACE_USR4(("spm_tclCsmIf::vHandlePNMGrp10Stat: ON"));					
				}
				else	
				{
					ETG_TRACE_USR4(("ERROR: spm_tclCsmIf::vHandlePNMGrp10Stat: Unused"));
				}
			} 
			else 
			{
				//TODO: Shutdown to be initiated based on the SYSFL-8946
				ETG_TRACE_FATAL(("ERROR:spm_tclCsmIf::vHandlePNMGrp10Stat: INVALID data or timed out"));
			}		
		}
		else
		{
			ETG_TRACE_FATAL(("ERROR:spm_tclCsmIf::vHandlePNMGrp10Stat: read failed"));
		}
	}
	else 
	{
		ETG_TRACE_FATAL(("ERROR:spm_tclCsmIf::vHandlePNMGrp10Stat: INVALID opcode"));
	}
}


/* ------------------------------------------------------------------------- */
tVoid spm_tclCsmIf::vCsmCommunicationRequest( tU8 u8ConnectState ){
/*!
  * \fn
  *  \brief
  *
  *  \param
  *
  *  \return
  *
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   (tVoid)u8ConnectState;
   ETG_TRACE_USR4( ( "spm_tclCsmIf::vCsmCommunicationRequest(): '''NO CSM INTERFACE IMPLEMENTED on generic SPM''!!!!!!!!" ) );
   return;
}

/* ------------------------------------------------------------------------- */
tVoid spm_tclCsmIf::vSendCsmBroadcastSignal( tU8 *pabData,
                                             tU8  u8Len,
                                             tU32 u32CsmSignal,
                                             tU8  u8TxType ){
/*!
  * \fn
  *  \brief
  *
  *  \param
  *
  *  \return
  *
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

   tU32 u32CsmHandle;
   tS32 s32Ret;

   SPM_NULL_POINTER_CHECK( _poMyCSMInterface );

   s32Ret = _poMyCSMInterface->CSM_lSignalWrite( (tVoid*)&u32CsmHandle, u32CsmSignal, pabData, u8Len, u8TxType );
   if ( CSM_C_NO_ERROR > s32Ret ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   return;
}


/* ------------------------------------------------------------------------- */
tU32 spm_tclCsmIf::u32ReadCsmBroadcastSignal( tU8 *pabData,
                                              tU8  u8Len,
                                              tU32 u32CsmSignal ){
/*!
  * \fn
  *  \brief
  *
  *  \param
  *
  *  \return
  *
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

   tU32 u32StatusRead = CSM_C_SIGNAL_DATA_INVALID;
   tU32 u32CsmHandle;
   tS32 s32Ret;

   SPM_NULL_POINTER_CHECK_VAL( _poMyCSMInterface );

   s32Ret = _poMyCSMInterface->CSM_lSignalRead( (tVoid*)&u32CsmHandle, u32CsmSignal,
                                                &pabData[0],
                                                u8Len,
                                                &u32StatusRead );

	/*
	bBuffer not read (i.e. bBuffer contains invalid data)
	In this case no default data for timeout has been defined.
	Default value definition is project specific - must be discussed
	with CAN developer
	*/
 if ( CSM_C_NO_ERROR > s32Ret )
   {
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   } 

   return( u32StatusRead );
} // u32ReadCsmBroadcastSignal

/* ------------------------------------------------------------------------- */
tVoid spm_tclCsmIf::vTraceCsmSignals( ) const

/*!
  * \fn
  *  \brief
  *    Trace out special CSM signals
  *
  *  \param
  *
  *  \return
  *
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
{
   return;
}

tVoid spm_tclCsmIf::vCheckCsmSignals( ) const {
	
	ETG_TRACE_USR4(("spm_tclCsmIf::vCheckCsmSignals :entered"));
	tVoid vHandlePNMGrp10Stat();
   return;
}



