/**
* @swcomponent Life Cycle Management
* @{
* @file        spm_VdDiagLogCcaServiceHandler.cpp
* @brief       Base class implementation of VdDialog Service.
*              Handling of diagnosis reports.
* @copyright   (C) 2018 Robert Bosch GmbH.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

//SPM  configuration
#include "spm_Config.h"

//my class header
#include "spm_VdDiagLogCcaServiceHandler.h"

//spm class definitions

//interfaces class definitions
#include "spm_ICcaServiceServer.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
    #include "trcGenProj/Header/spm_VdDiagLogCcaServiceHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
| local #define (scope: module-local)
|-----------------------------------------------------------------------*/
//#define SPM_TRACE_FILE_ID   SPM_FILE_VDDIAGLOGSERVICEHANDLER


#define DIAGLOG_FI_MAJOR_VERSION 1
#define DIAGLOG_FI_MINOR_VERSION 0

/*-------------------------------------------------------------------------- */
/* MSG MAP                                                                   */
/*-------------------------------------------------------------------------- */
BEGIN_MSG_MAP(spm_tclVdDiagLogServiceHandler, ICcaDiaglogProxy)
ON_MESSAGE(DIAGLOG_MAINFI_C_U16_SAVETESTRESULT,     vHandleSaveTestResult)
ON_MESSAGE(DIAGLOG_MAINFI_C_U16_SENDNEXTTESTRESULT, vHandleSendNextTestResult)
END_MSG_MAP()

spm_tclVdDiagLogServiceHandler::spm_tclVdDiagLogServiceHandler(const ISpmFactory& factory) : ICcaDiaglogProxy(factory) {
    ETG_TRACE_USR1(("VdDiagLogServiceHandler: Constructor: %dms", OSAL_ClockGetElapsedTime()));
}

spm_tclVdDiagLogServiceHandler::~spm_tclVdDiagLogServiceHandler() {
}

tVoid spm_tclVdDiagLogServiceHandler::vRegisterFids()
{
    vUpRegFid(DIAGLOG_MAINFI_C_U16_SENDNEXTTESTRESULT);

    ETG_TRACE_USR1(("VdDiagLogServiceHandler: service is available: %dms", OSAL_ClockGetElapsedTime()));

    // Because SPM is the first component to come up at the time the ITC data
    // is gathered the VD DiagLog service is not available.
    // So all gathered data is sent to VD DiagLog as a result of the service
    // being available in one bunch.
    // 
    std::set<tU16> allowed;
    std::map<tU16, diaglog_fi_tcl_e8_TestResult::tenType>::iterator it;

    for (it = _tc.begin(); it != _tc.end(); ++it) allowed.insert(it->first);

    vSendAllTestResults(allowed);
}


tVoid spm_tclVdDiagLogServiceHandler::vAddItc(tU16 itc, tU8  u8Result)
{
   diaglog_fi_tcl_e8_TestResult::tenType enTestResult = (diaglog_fi_tcl_e8_TestResult::tenType)u8Result;

   ETG_TRACE_USR1(("vAddItc Itc: %04x, Testresult: %d", itc, ETG_ENUM(SPM_TEST_RESULT, enTestResult)));
   _tc[itc]                                              = enTestResult;
}

tVoid spm_tclVdDiagLogServiceHandler::vSetItcError(tU16 u16ItcType,
                                    tU8  u8Result){
    // remember troublecode in map
    diaglog_fi_tcl_e8_TestResult::tenType enTestResult = (diaglog_fi_tcl_e8_TestResult::tenType)u8Result;
    bpstl::map<tU16, diaglog_fi_tcl_e8_TestResult::tenType>::iterator it = _tc.find(u16ItcType);

    ETG_TRACE_USR1(("vSetItcError: Service=%d, Found: %d, Itc: %04x, Testresult: %d", bIsServiceAvailable(), it != _tc.end(), u16ItcType, ETG_ENUM(SPM_TEST_RESULT, enTestResult)));

    if (it != _tc.end())
    {
        if (bIsServiceAvailable() && enTestResult != it->second)
        {
            // result has changed, send an update to VD DiagLog
            //send trigger if change detected
            it->second = enTestResult;

            diaglog_mainfi_tclMsgSaveTestResultMethodStart ms;

            diaglog_fi_tcl_TestResult item;

            item.TroubleCode   = u16ItcType;
            item.Result.enType = enTestResult;
            ms.TestResultList.TestResultList.push_back(item);

            fi_tclVisitorMessage oMethodStart(ms);

            oMethodStart.vInitServiceData(CCA_C_U16_APP_SPM,
                                          CCA_C_U16_APP_DIAGLOG,
                                          AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                          0,
                                          _u16RegID,
                                          0,
                                          CCA_C_U16_SRV_DIAGLOG,
                                          DIAGLOG_MAINFI_C_U16_SAVETESTRESULT,
                                          AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

            ETG_TRACE_USR4(("vSetItcError Itc: %04x, Testresult: %d", u16ItcType, ETG_ENUM(SPM_TEST_RESULT, enTestResult)));

            _poclCcaMsgHandler->bPostMessage( &oMethodStart);
        }
        else
        {
            it->second = enTestResult;
        }
    }
}


// handler functions -------------------------------------------
// check the response of savetestresult from FC DiagLog
tVoid spm_tclVdDiagLogServiceHandler::vHandleSaveTestResult(amt_tclServiceData *poMessage)
{
    tU32   u32OpCode      = poMessage->u8GetOpCode();

    switch (u32OpCode)
    {
    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
        {
            //ETG_TRACE_USR4(("DiagLog result"));
            break;
        }
    case AMT_C_U8_CCAMSG_OPCODE_ABORTRESULT:
    case AMT_C_U8_CCAMSG_OPCODE_METHODABORT:
    case AMT_C_U8_CCAMSG_OPCODE_ERROR:
    default:
        {
            ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
        }
        break;
    }
}


tVoid spm_tclVdDiagLogServiceHandler::vHandleSendNextTestResult(amt_tclServiceData *poMessage)
{
    tU32   u32OpCode      = poMessage->u8GetOpCode();

    if (u32OpCode == AMT_C_U8_CCAMSG_OPCODE_STATUS)
    {
        fi_tclVisitorMessage oMsg(poMessage); // create FI message from original message
        diaglog_mainfi_tclMsgSendNextTestResultStatus oStatus;

        if (oMsg.s32GetData(oStatus) == OSAL_ERROR)
        {
            return;
        }

        if (!oStatus.Status)
        {
            // this is only a confirmation message from the last upreg, can be ignored
            return;
        }

        // Evaluate a list if available. If list is empty then send all test results to
        // VD DiagLog otherwise only those which are part of the list.
        // 
        bpstl::set<tU16> allowed;
        bpstl::map<tU16, diaglog_fi_tcl_e8_TestResult::tenType>::iterator it;

        // list is empty so send all our test results
        for (it = _tc.begin(); it != _tc.end(); ++it)
        {
            ETG_TRACE_USR4(("vHandleSendNextTestResult: Code=0x%04x", it->first));
            allowed.insert(it->first);
        }
        vSendAllTestResults(allowed);
    }
}


tVoid spm_tclVdDiagLogServiceHandler::vSendAllTestResults(const bpstl::set<tU16>& incl)
{
    if (bIsServiceAvailable())
    {
        diaglog_mainfi_tclMsgSaveTestResultMethodStart ms;

        bpstl::map<tU16, diaglog_fi_tcl_e8_TestResult::tenType>::iterator it;

        for (it = _tc.begin(); it != _tc.end(); ++it)
        {
            // only include all allowed DTCs to the result list
            if (incl.find(it->first) != incl.end())
            {
                diaglog_fi_tcl_TestResult item;
                item.TroubleCode   = it->first;
                item.Result.enType = it->second;
                ms.TestResultList.TestResultList.push_back(item);
                ETG_TRACE_USR4(("VdDiagLogServiceHandler::ItcType = %04x, TestResult = %d", it->first, ETG_ENUM(SPM_TEST_RESULT,it->second)));
            }
        }

        fi_tclVisitorMessage oMethodStart(ms);

        oMethodStart.vInitServiceData(CCA_C_U16_APP_SPM,
                                      CCA_C_U16_APP_DIAGLOG,
                                      AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                      0,
                                      _u16RegID,
                                      0,
                                      CCA_C_U16_SRV_DIAGLOG,
                                      DIAGLOG_MAINFI_C_U16_SAVETESTRESULT,
                                      AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

        ETG_TRACE_USR4(("vSendAllTestResults to VdDiaglog: %d", incl.size()));

        _poclCcaMsgHandler->bPostMessage( &oMethodStart);
    }
}

