/**
  * \file spm_rootdaemon_client.cpp
  *  \brief
  *    This file contains the interface for the root daemon to execute those
  *    LCM/SPM related actions which require root permission to be performed
  *    properly.
  *
  *  \b PROJECT      : Generic
  *  \b SW-COMPONENT : LCM
  *  \b COPYRIGHT    : (c) 2015 Robert Bosch GmbH, Hildesheim
  */

/*
  ****************************************************************************

   INCLUDES

  ****************************************************************************
  */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <string>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_ERRMEM
   #include <linux/errmem.h>
#endif
#include "IClientSharedlib.h"
#include "ShellCmd.h"

#include "spm_rootdaemon.h"

extern "C"
{
/*
  ****************************************************************************

   TYPE DEFINITIONS

  ****************************************************************************
  */

typedef int (*tpfnRootCommand)(std::string args);

/*
  ****************************************************************************

   FUNCTION DECLARATIONS

  ****************************************************************************
  */

int SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetRunningThreads(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetThreadInfo(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById(std::string args);

int SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd(std::string args);

int SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess(std::string args);

int SPM_ROOTDAEMON_CLIENT_nSendSignalToAllProcesses(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetProcessHighWaterMarks(std::string args);

int SPM_ROOTDAEMON_CLIENT_nUnmountPartitions(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses(std::string args);

int SPM_ROOTDAEMON_CLIENT_nGetOOMInfo(std::string args);

int SPM_ROOTDAEMON_CLIENT_nKillProcess(std::string args);

int SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit(std::string args);

int SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit(std::string args);

int SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb(std::string args);

/*
  ****************************************************************************

   GLOBAL CONSTANTS

  ****************************************************************************
  */

static const tpfnRootCommand g_carpfnRootCommands[] =
{
   SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker,
   SPM_ROOTDAEMON_CLIENT_nGetRunningThreads,
   SPM_ROOTDAEMON_CLIENT_nGetThreadInfo,
   SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName,
   SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById,
   SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd,
   SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess,
   SPM_ROOTDAEMON_CLIENT_nSendSignalToAllProcesses,
   SPM_ROOTDAEMON_CLIENT_nGetProcessHighWaterMarks,
   SPM_ROOTDAEMON_CLIENT_nUnmountPartitions,
   SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses,
   SPM_ROOTDAEMON_CLIENT_nGetOOMInfo,
   SPM_ROOTDAEMON_CLIENT_nKillProcess,
   SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit,
   SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit,
   SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb
};

/*
  ****************************************************************************

   LOCAL FUNCTIONS

  ****************************************************************************
  */

int SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker(std::string args){
/**
  *  \brief Determine the top CPU processes by use of the 'top' shell command
  *         and write the results into the file '/tmp/_spm_top.log'.
  *
  *  \param [in] args : String with "processes" if only processes shall be
  *                     tracked but no tasks.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/_top.py");

   if (!(args == "processes")){
      cmd.addArgs(std::string("-t"));                      // Also track tasks beside processes
   }
   cmd.addArgs(std::string("-n 20 -o /tmp/_spm_top.log")); // Track processes only

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() >> 8); // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nTriggerTimeTracker

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetRunningThreads(std::string args){
/**
  *  \brief Determine the threads/task which are currently in 'running' state
  *         by use of the 'ps' shell command and write the results into the
  *         file '/tmp/_spm_task_info.log'.
  *
  *  \return Result of the shell command execution.
  */
   (void)args; // Parameter not used.

   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/_get_task_list.py");
   cmd.addArgs("-r -o /tmp/_spm_task_info.log");

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetRunningThreads() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() ) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetRunningThreads() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetRunningThreads

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetThreadInfo(std::string args){
/**
  *  \brief Get thread/task related information by use of the 'ps' shell
  *         command and write the results into the file
  *         '/tmp/_spm_task_info.log'.
  *
  *  \param [in] args : String "%s" with the name of the task/thread.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/_get_task_list.py -t");
   if (isTaskName(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nGetThreadInfo() => %s is not a valid task/process name",
             args.c_str());
      return -1;
   }
   cmd.addArgs("-o /tmp/_spm_task_info.log");

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetThreadInfo() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() ) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetThreadInfo() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetThreadInfo

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName(std::string args){
/**
  *  \brief Get process related information by use of the 'ps' shell command
  *         and write the results into the file '/tmp/_spm_task_info.log'.
  *
  *  \param [in] args : String "%s" with the name of the process.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/_get_task_list.py -p");
   if (isTaskName(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName() => %s is not a valid task/process name",
             args.c_str());
      return -1;
   }
   cmd.addArgs("-o /tmp/_spm_task_info.log");

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() ) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetProcessInfoByName

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById(std::string args){
/**
  *  \brief Get process related information by use of the 'ps' shell command
  *         and write the results into the file '/tmp/_spm_task_info.log'.
  *
  *  \param [in] args : String "%u" with the ID of the process.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/_get_task_list.py -i");
   if (isPid(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_nGetProrcessInfoById() => %s is not a valid pid",
             args.c_str());
      return -1;
   }
   cmd.addArgs("-o /tmp/_spm_task_info.log");

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetProcessInfoById

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd(std::string args){
/**
  *  \brief Send a signal to a process which is controlled by Systemd by use
  *         of the 'systemctl' shell command.
  *
  *  \param [in] args : String "%s %u" with signal name and service name.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;
   std::string signal_name;
   std::string service_name;

   ShellCmd cmd("systemctl");

   cmd.addArgs("kill -s");

   splitStringAtSpace(args, signal_name, service_name);

   if (isServiceName(service_name)) {
      cmd.addArgs(signal_name);
      cmd.addArgs(service_name);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd() => invalid service name: %s",
             service_name.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nSendSignalToProcessViaSystemd

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess(std::string args){
/**
  *  \brief Send a signal to a process by use of the 'kill' shell command.
  *
  *  \param [in] args : String "%u %u" with signal name and process ID.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;
   std::string signal_name;
   std::string pid;

   ShellCmd cmd("kill");

   splitStringAtSpace(args, signal_name, pid);

   if (isPid(pid)) {
      cmd.addArgs("-" + signal_name);
      cmd.addArgs(pid);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess() => %s is not a valid pid",
             pid.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() ) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nSendSignalToAllProcesses(std::string args){
/**
  *  \brief Send a signal to all process by use of the 'killall' shell command.
  *
  *  \param [in] args : String "%u %s" with signal ID and process name.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;
   std::string signal;
   std::string process_name;

   ShellCmd cmd("killall");

   splitStringAtSpace(args, signal, process_name);

   if (isSignal(signal)) {
      cmd.addArgs("-" + signal);
      cmd.addArgs(process_name);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nSendSignalToAllProcesses() => invalid signal %s",
             signal.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec() ) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSendSignalToProcess() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nSendSignalToAllProcesses

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetProcessHighWaterMarks(std::string args){
/**
  *  \brief Determine which processes have exceeded their high-water-mark
  *         memory consumption levels by accessing their 'status' files from
  *         the /proc virtual filesystem via a shell command and write the
  *         results into the file '/tmp/_spm_hwm_info.log'.
  *
  *  \param [in] args : String "%s" with the name and path of the XML
  *                     configuration file which contains a list of the
  *                     processes with their high-water-mark levels which
  *                     should be checked against exceedance.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/check_hwm.py -o /tmp/_spm_hwm_info.log -i");
   if (isPath(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nGetProcessHighWaterMarks() => invalid path to XML file %s",
             args.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nLogProcessHighWaterMarks() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nLogProcessHighWaterMarks() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetProcessHighWaterMarks

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nUnmountPartitions(std::string args){
/**
  *  \brief Remount a partition as read-only by use of the 'mount' shell
  *         command.
  *
  *  \param [in] args : String "%s %s" with the folder name to be remounted
  *                     as read-only and an optional file name as location
  *                     to store the results of this operation.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;
   std::string folder;
   std::string logfile;

   ShellCmd cmd("python");

   cmd.addArgs("/var/opt/bosch/static/spm/unmount.py");
   cmd.addArgs("-e");

   splitStringAtSpace(args, folder, logfile);

   if (isPath(folder)) {
      cmd.addArgs(folder);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nUnmountPartitions() => path to folder '%s' is invalid",
             folder.c_str());
      return -1;
   }

   if (!logfile.empty()) {
      if (isPath(logfile)) {
         cmd.addArgs(logfile);
      } else {
         syslog(LOG_ERR,
                "SPM_ROOTDAEMON_CLIENT_nUnmountPartitions() => invalid log file");
         return -1;
      }
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nUnmountPartitions() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nUnmountPartitions() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nUnmountPartitions

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses(std::string args){
/**
  *  \brief Get a list of the current processes of the system via the 'ps'
  *         shell command and store the result in the error memory.
  *
  *  \return Result of the shell command execution.
  */
   (void)args; // Parameter not used.

   int         nResult;

   ShellCmd cmd("ps");

   cmd.addArgs("-AlF");

   cmd.setOutputFile(std::string("/dev/errmem"));

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetCurrentProcesses

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nGetOOMInfo(std::string args){
/**
  *  \brief Get the out-of-memory info via the sysrq-trigger and store the
  *         result in the error-memory. To get all intended out-of-memory
  *         information into the error-memory, the log-level must be
  *         temporarily increased to level 8.
  *
  *  \return 0 on success, otherwise the error value.
  */
   (void)args; // Parameter not used.

   int  nResult                 = 0;
   int  nFdErrmem               = - 1;
   int  nFdSysRq                = - 1;
   int  nFdSysRqTrigger         = - 1;
   int  nCurrentKernelMsgLevel  = - 1;
   int  nElevatedKernelMsgLevel = 8;
   char chSysrq                 = '1';
   char chSysrqTriggerOOMInfo   = 'a';
   char chSysrqTriggerLogLevel4 = '4';
   char chSysrqTriggerLogLevel8 = '8';

   nFdErrmem = open("/dev/errmem", O_WRONLY);
   if ( nFdErrmem == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   nFdSysRq = open("/proc/sys/kernel/sysrq", O_WRONLY);
   if ( nFdSysRq == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   nFdSysRqTrigger = open("/proc/sysrq-trigger", O_WRONLY);
   if ( nFdSysRqTrigger == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

#ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_ERRMEM
   if ( ioctl(nFdErrmem,
              IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL,
              &nCurrentKernelMsgLevel) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   if ( ioctl(nFdErrmem,
              IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL,
              &nElevatedKernelMsgLevel) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }
#endif

   if ( write(nFdSysRq, &chSysrq, sizeof( char ) ) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   if ( write(nFdSysRqTrigger, &chSysrqTriggerLogLevel8, sizeof( char ) ) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   if ( write(nFdSysRqTrigger, &chSysrqTriggerOOMInfo, sizeof( char ) ) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

   // Wait 200 ms to gather OOM data and let it be stored in
   // error memory before switching back to log level 4.
   usleep(200000);

   if ( write(nFdSysRqTrigger, &chSysrqTriggerLogLevel4, sizeof( char ) ) == - 1 ){
      nResult = errno;
      goto error_out; //lint !e801 Use of goto is deprecated -- intentionally used to prevent deep structures
   }

error_out:
#ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_ERRMEM
   if ( ( nFdErrmem != - 1 ) && ( nCurrentKernelMsgLevel != - 1 ) ){
      (void) ioctl(nFdErrmem,
            IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL,
            &nCurrentKernelMsgLevel);
      if ( close(nFdErrmem) == -1){
          syslog(LOG_ERR,
                 "SPM_ROOTDAEMON_CLIENT_nGetOOMInfo() : Failed to close /dev/errmem. Error: %s",
                 strerror(errno));
      }
   }
#endif

   if ( nFdSysRq != - 1 ){
      if ( close(nFdSysRq) == -1){
          syslog(LOG_ERR,
                 "SPM_ROOTDAEMON_CLIENT_nGetOOMInfo() : Failed to close /proc/sys/kernel/sysrq. Error: %s",
                 strerror(errno));
      }
   }

   if ( nFdSysRqTrigger != - 1 ){
      if ( close(nFdSysRqTrigger) == -1){
          syslog(LOG_ERR,
                 "SPM_ROOTDAEMON_CLIENT_nGetOOMInfo() : Failed to close /proc/sysrq-trigger. Error: %s",
                 strerror(errno));
      }
   }

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nGetOOMInfo

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nKillProcess(std::string args){
/**
  *  \brief Kill a process via the 'killall' shell command.
  *
  *  \param [in] args : String "%s" with to be killed processes name and path.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd = ShellCmd("killall");

   cmd.addArgs(args);

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nKillProcess() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nKillProcess() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nKillProcess

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit(std::string args){
/**
  *  \brief Start a Systemd unit via the 'systemctl' shell command in a
  *         non-blocking and asynchronous mode.
  *
  *  \param [in] args : String "%s" with the Systemd unit to be started.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("systemctl");

   cmd.addArgs("--no-block start");

   if (isServiceName(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit() => invalid service name: %s",
             args.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nSystemCtlStartUnit

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit(std::string args){
/**
  *  \brief Stop a Systemd unit via the 'systemctl' shell command in a
  *         non-blocking and asynchronous mode
  *
  *  \param [in] args : String "%s" with the Systemd unit to be stopped.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("systemctl");

   cmd.addArgs("--no-block stop");

   if (isServiceName(args)) {
      cmd.addArgs(args);
   } else {
      syslog(LOG_ERR,
             "SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit() => invalid service name: %s",
             args.c_str());
      return -1;
   }

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nSystemCtlStopUnit

/******************************************************************************/

int SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb(std::string args){
/**
  *  \brief Start a process via the debugger.
  *
  *  \param [in] args : String "%s" with the name of the process.
  *
  *  \return Result of the shell command execution.
  */
   int         nResult;

   ShellCmd cmd("gdbserver");

   cmd.setBackgroundTask(true);

   cmd.addArgs(":2345");
   cmd.addArgs(args);

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb() => call ShellCmd::exec(%s)",
          cmd.getFullCommandLine().c_str());

   nResult = (cmd.exec()) >> 8; // >> 8 because the exit status of the executed program is stored in the upper 8 bits of a 16 bit value

   syslog(LOG_INFO,
          "SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb() => call returns with %d",
          nResult);

   return( nResult );
}  // SPM_ROOTDAEMON_CLIENT_nStartProcessViaGdb

/******************************************************************************/

CmdData SPM_ROOTDAEMON_CLIENT_rPerformRootOp(const int   cmdNum,
                                             std::string args){
/**
  *  \brief Common root daemon interface to execute the passed command.
  *
  *  \return Structure 'CmdData' with one of the predefined return values in
  *          field 'errorNo' and a free definable message-result-string in
  *          field 'message'.
  */
   CmdData rCmdData = { 0,0,0,0,0 };
   int     nResult  = 0;

   rCmdData.errorNo = ERR_NONE;

   if ( ( cmdNum >= 0 ) && ( cmdNum < NUMBER_OF_LCM_ROOT_COMMANDS ) ){
      // Sanitize args before using them
      stripNonASCII(args);
      nResult = g_carpfnRootCommands[cmdNum](args);
      sprintf(rCmdData.message, "%d", nResult);
   } else {
      rCmdData.errorNo = ERR_UNKNOWN_CMD;
   }

   return( rCmdData );
}  // SPM_ROOTDAEMON_CLIENT_rPerformRootOp

/*
  ****************************************************************************

   PUBLIC FUNCTIONS

  ****************************************************************************
  */

const char*getClientName(){
/**
  *  \brief Common root daemon interface function which returns a string with
  *         the name of this client.
  *
  *  \return String with the name of this client.
  */
   return( "lcm" );
}

/******************************************************************************/

const char*getClientGroupName(){
/**
  *  \brief Common root daemon interface function which returns a string with
  *         the related Linux file system access control GROUP.
  *
  *  \return String with the related Linux file system access control GROUP.
  */
   return( "eco_lcm" );
}

/******************************************************************************/

const char*getClientUserName(){
/**
  *  \brief Common root daemon interface function which returns a string with
  *         the related Linux file system access control USER.
  *
  *  \return String with the related Linux file system access control USER.
  */
   return( "eco_lcm" );
}

/******************************************************************************/

CmdData command(const int   cmdNum,
                std::string args){
/**
  *  \brief Common root daemon interface to execute a passed command.
  *
  *  \return Structure 'CmdData' with one of the predefined return values in
  *          field 'errorNo' and a free definable message-result-string in
  *          field 'message'.
  */
   return ( SPM_ROOTDAEMON_CLIENT_rPerformRootOp(cmdNum, args) );
}

/******************************************************************************/

CmdData SPM_ROOTDAEMON_CLIENT_rPerformRootOpAsRoot(const int   cmdNum,
                                                   std::string args){
/**
  *  \brief Local wrapper function to directly execute root operations without
  *         the help of the root-daemon while the LCM/SPM is itself being
  *         executed as root.
  *
  *  \return Structure 'CmdData' with one of the predefined return values in
  *          field 'errorNo' and a free definable message-result-string in
  *          field 'message'.
  */
   return ( SPM_ROOTDAEMON_CLIENT_rPerformRootOp(cmdNum, args) );
}

/******************************************************************************/

} // extern "C"

