/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     LifeCycleConsumerProxy.h
  * \brief    Implementation of the GENIVI LifeCycleConsumer interface
  *           the server is implemented by the application and the name
  *           of the bus and the object is passed via RegisterShutdownClient call
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  *
  * \par Copyright:
  * (c) 2013-2013 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
#ifndef __LIFECYCLECONSUMERPROXY_H
#define __LIFECYCLECONSUMERPROXY_H

#include <iostream>

#include "asf/core/Logger.h"

#include "org/genivi/NodeStateManager/LifeCycleConsumerProxy.h"

#include "lcm_AppMain.h"
#include "ILifeCycleConsumerProxy.h"

// include type definitions of Genivi
#include "org/genivi/NodeStateManager/LcmBasicTypesConst.h"
//  boost::shared_ptr is used here cause by ASF interface.

using namespace ::org::genivi::NodeStateManager::LcmBasicTypes;

#define SPM_U32_LCM_PROXY_SHUTDOWNTYPE_NOT      _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Not    /**< Client not registered for any shutdown           */
#define SPM_U32_LCM_PROXY_SHUTDOWNTYPE_NORMAL   _Nsm_Shutdown_Type_e__Nsm_Shutdown_Type_Normal /**< Client registered for normal shutdown            */
#define SPM_U32_LCM_PROXY_SHUTDOWNTYPE_FAST     _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Fast   /**< Client registered for fast shutdown              */
#define SPM_U32_LCM_PROXY_SHUTDOWNTYPE_RUNUP    _Nsm_Shutdown_Type_e__NSM_Shutdown_Type_Runup  /**< The shutdown type "run up" can not be used for
                                                                                                    registration. Clients which are registered and
                                                                                                    have been shut down, will automatically be
                                                                                                    informed about the "run up", when the shut down
                                                                                                    is canceled.                                    */

#define SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG       0x01000000U
#define SPM_U32_LCM_GET_PROXY_SHUTDOWN_TYPE(state)      ( ~SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG & state )

#define SPM_U32_LCM_SHUTDOWNTYPE_NOT      ( SPM_U32_LCM_PROXY_SHUTDOWNTYPE_NOT | SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG )
#define SPM_U32_LCM_SHUTDOWNTYPE_NORMAL   ( SPM_U32_LCM_PROXY_SHUTDOWNTYPE_NORMAL | SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG )
#define SPM_U32_LCM_SHUTDOWNTYPE_FAST     ( SPM_U32_LCM_PROXY_SHUTDOWNTYPE_FAST | SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG )
#define SPM_U32_LCM_SHUTDOWNTYPE_RUNUP    ( SPM_U32_LCM_PROXY_SHUTDOWNTYPE_RUNUP | SPM_U32_LCM_SHUTDOWNTYPE_APP_FLAG )

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

class AsfLcmConsumerProxy
{
protected:
boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy > _proxy;
std::string                                                                                      _busName;
std::string                                                                                      _objectName;
uint32                                                                                           _u32AppId;
uint32                                                                                           _u32CurrentState;
uint32                                                                                           _u32RequestedState;

public:
AsfLcmConsumerProxy(boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy > proxy,
                    const std::string                                                                              & busName,
                    const std::string                                                                              & objectName,
                    uint32                                                                                           u32AppId)
   : _proxy(proxy),
   _busName(busName),
   _objectName(objectName),
   _u32AppId(u32AppId),
   _u32CurrentState(SPM_U32_LCM_SHUTDOWNTYPE_RUNUP),
   _u32RequestedState(SPM_U32_LCM_SHUTDOWNTYPE_RUNUP)
{}
virtual ~AsfLcmConsumerProxy(){}
AsfLcmConsumerProxy& operator =(const AsfLcmConsumerProxy& rhs){
   if(this != &rhs){
      _proxy             = rhs._proxy;
      _busName           = rhs._busName;
      _u32AppId          = rhs._u32AppId;
      _objectName        = rhs._objectName;
      _u32CurrentState   = rhs._u32CurrentState;
      _u32RequestedState = rhs._u32RequestedState;
   }
   return( * this );
}

virtual const std::string getBusName(){ return( _busName ); }
virtual const std::string getObjectName(){ return( _objectName ); }
virtual uint32 u32GetAppId(){ return( _u32AppId ); }
virtual uint32 u32GetCurrentState(){ return( _u32CurrentState ); }
virtual uint32 u32GetRequestedState(){ return( _u32RequestedState ); }
virtual void vSetCurrentState(uint32 u32State){ _u32CurrentState = u32State; }
virtual void vSetRequestedState(uint32 u32State){ _u32RequestedState = u32State; }

virtual boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy > getProxy(){ return( _proxy ); }
};

typedef std::map < const boost::shared_ptr < ::asf::core::Proxy >, boost::shared_ptr < AsfLcmConsumerProxy > > AsfLcmConsumerProxyMap;

class LifeCycleConsumerProxy : public ILifeCycleConsumerProxy
   , private ::asf::core::ServiceAvailableIF
   , private ::org::genivi::NodeStateManager::LifeCycleConsumer::LifecycleRequestCallbackIF
{
private:
AsfLcmConsumerProxyMap _lcmConsumerProxyMap;

DECLARE_CLASS_LOGGER();

public:
LifeCycleConsumerProxy(lcm_tclAppMain *poMainAppl);
virtual ~LifeCycleConsumerProxy();

// ISpmBase methods
virtual tVoid vHandleMessage(lcm_tclBaseIf::TMsg */*pMsg*/){}

virtual tVoid vGetReferences(){}
virtual tVoid vStartCommunication();

virtual tVoid vTraceInfo(){}

virtual tVoid vHandleTraceMessage(const tUChar */*puchData*/){}
virtual const tChar*getName() const { return( "LifeCycleConsumerProxy" ); }

virtual tVoid AddLifeCycleConsumerProxy(const std::string strBusName,
                                        const std::string strObjectPath);

virtual tVoid SendAppStateChangeRequest(const tU32 u32AppId,
                                        const tU32 u32PowerData);

virtual tBool bHandleLifeCycleRequestResponses(const tU32 u32AppId, const ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e eNsmErrorStatus);

// ######################################
// ASF-Core implementations
// ######################################
// ServiceAvailableIF implementation
virtual void onAvailable(const boost::shared_ptr < ::asf::core::Proxy >& proxy,
                         const ::asf::core::ServiceStateChange         & stateChange);

virtual void onUnavailable(const boost::shared_ptr < ::asf::core::Proxy >& proxy,
                           const ::asf::core::ServiceStateChange         & stateChange);

// ######################################
// LifecycleConsumerProxy implementations
// ######################################
// LifecycleRequestMethodCallbackIF
virtual void onLifecycleRequestError(const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy >& proxy,
                                     const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifecycleRequestError > & error);

virtual void onLifecycleRequestResponse(const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy >  & proxy,
                                        const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifecycleRequestResponse >& response);

};

}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

#endif // ifndef __LIFECYCLECONSUMERPROXY_H

