/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
    *\file     DbusProxy.h
    *\brief    Implementation of the LcmCcaServer component
    *\brief    which implements handcrafted CcaServerHookStub
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  *
  * \par Copyright:
  * (c) 2013-2013 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
#ifndef __LCMDBUSPROXY_H
#define __LCMDBUSPROXY_H

#include <iostream>

#include "asf/core/Logger.h"
#include "asf/core/Timer.h"
#include "asf/core.h"

#include "org/freedesktop/systemd1/UnitProxy.h"
#include "org/freedesktop/systemd1/ServiceProxy.h"
#include "DbusManagerProxy.h"
#ifdef VARIANT_S_FTR_ENABLE_DBUS_PROPERTY
#include "bosch/org/freedesktop/DBus/PropertiesProxy.h"
#endif
#include "LcmConsumerStub.h"
#include "org/genivi/NodeStateManager/LifeCycleConsumerProxy.h"
#include "org/genivi/NodeStateManager/LifecycleControlStub.h"

#include "asf/stream/MemoryStream.h"
#include <set>

#include "lcm_AppMain.h"
#include "IDbusProxy.h"

#define SYSTEMD_IMPORT_INTERFACE_GENERIC
#include "org/bosch/cm/lcm/systemdinterface/systemd_if.h"

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

/*!
   \brief contains all info needed of a single job that is done by systemd to start / stop units
  */
class SpmJobElement
{
public:
typedef enum
{
   eJobQueued  = 0x00,
   eJobStarted = 0x01,
   eJobDone    = 0x02,
   eJobError   = 0xFF
} EJobState;

SpmJobElement() : _strJobName(""), _strServiceName(""), _eJobState(eJobQueued){}
SpmJobElement(const std::string& strServiceName,
              const std::string& strJobName) : _strJobName(strJobName), _strServiceName(strServiceName), _eJobState(eJobQueued){}
~SpmJobElement(){}

tVoid vSetJobStarted(){ _eJobState = eJobStarted; }
tVoid vSetJobDone(){ _eJobState = eJobDone; }
tVoid vSetJobError(){ _eJobState = eJobError; }
tVoid vSetServiceName(const std::string& strServiceName){ _strServiceName = strServiceName; }
const std::string getServiceName(){ return( _strServiceName ); }
const std::string getJobName(){ return( _strJobName ); }

private:
const std::string _strJobName;
std::string       _strServiceName;
EJobState         _eJobState;
};

/*!
   \brief is the list of all jobs triggered by LCM
  */
typedef std::list < SpmJobElement > TSpmJobList;

/*!
   \brief handles the list and getters its setters
  */
class SpmJobList
{
public:
SpmJobList(){ _lJobList.clear(); }
~SpmJobList(){ _lJobList.clear(); }

std::string  strGetJobNameByServiceName(const std::string& strServiceName);

std::string  strGetServiceNameByJobName(const std::string& strJobName);

tVoid vAddJob(const std::string& strServiceName,
              const std::string& strJobName);

tVoid vSetJobStarted(const std::string& strJobName);

tVoid vSetJobDone(const std::string& strJobName);

tVoid vSetJobError(const std::string& strJobName);

tBool bSetServiceName(const std::string& strServiceName,
                      const std::string& strJobName);

private:
TSpmJobList _lJobList;
};

class DbusProxy : public IDbusProxy
   , public SystemdIF
{
private:
SpmJobList         _JobList;

DECLARE_CLASS_LOGGER();

systemd_interface *_posystemd_interface;

private:
//avoid assignment
DbusProxy & operator = (const DbusProxy& proxy){ (tVoid)proxy; return *this;}
// avoid copy
DbusProxy(const DbusProxy& proxy):IDbusProxy(proxy){}

public:
DbusProxy(lcm_tclAppMain *poMainAppl);
virtual ~DbusProxy();

// ISpmBase methods
virtual tVoid vHandleMessage(lcm_tclBaseIf::TMsg */*pMsg*/){}

virtual tVoid vGetReferences(){}
virtual tVoid vStartCommunication();

virtual tVoid vTraceInfo(){}

virtual tVoid vHandleTraceMessage(const tUChar */*puchData*/){}
virtual const tChar*getName() const { return( "DbusProxy" ); }

virtual void vTraceAvailability(const ::asf::core::ServiceStateChange& stateChange);

virtual void vPostLcmRequest(tU32 u32AppId,
                             tU16 u16PowerType,
                             tU32 u32PowerData1,
                             tU32 u32PowerData2);


virtual std::string     unitNameEscape(const std::string UnitName);

virtual std::string ExtractApplicationName(const std::string strObjectName);

virtual std::string  strGetJobNameByServiceName(const std::string& strServiceName){ return( _JobList.strGetJobNameByServiceName(strServiceName) ); }
virtual std::string  strGetServiceNameByJobName(const std::string& strJobName){ return( _JobList.strGetServiceNameByJobName(strJobName) ); }
virtual tVoid vAddJob(const std::string& strServiceName,
                      const std::string& strJobName){ _JobList.vAddJob(strServiceName, strJobName); }
virtual tVoid vSetJobStarted(const std::string& strJobName){ _JobList.vSetJobStarted(strJobName); }
virtual tVoid vSetJobDone(const std::string& strJobName){ _JobList.vSetJobDone(strJobName); }
virtual tVoid vSetJobError(const std::string& strJobName){ _JobList.vSetJobError(strJobName); }
virtual tBool bSetServiceName(const std::string& strServiceName,
                              const std::string& strJobName){ return( _JobList.bSetServiceName(strServiceName, strJobName) ); }

// Systemd interface implementation
virtual bool OnAppWatchdog();

};

}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

#endif // ifndef __LCMDBUSPROXY_H

