/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_WorkerServer.h
  * @brief        This is header file contains functions for Worker Server which distributes
  *                  messages within the FC LCM to support loose coupling.
  * @copyright    (C) 2011 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <set>
#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_IWorkerClient.h"
#include "spm_baseClass.h"

class ISpmFactory;
class ISpmSupervisionManager;
class ISpmSyncHandler;

/*!
  *  \class spm_tclWorkerServer
  *  \brief This class server distributes messages within the FC SPM to support loose coupling.
          The worker server also distributes timer events to synchronous startup phases like early, startup, storage,
          cyclic.

          The spm_tclFactory object creates an instance of this class. It is used by class spm_tclProcessSupervision to post messages to workers.
  */
class spm_tclWorkerServer : public ISpmWorkerServer
{
public:
spm_tclWorkerServer( const ISpmFactory& factory );
virtual ~spm_tclWorkerServer( );

virtual tVoid vSetNumberOfStartupProcesses( tUInt nNumber ){ _nStartupNumber = nNumber; }
virtual tUInt nGetStartupTime( ) const { return( _nStartupTime ); }
virtual tVoid vSetStartupTime( );

virtual tVoid vHandleMsgQueue( );

protected:
// completion of spm_tclActive
virtual tVoid main( ); // main thread entry function

static tVoid vOnQueueCallback( tVoid *pArg );

public:
// Implementation of interface ISpmWorkerServer
virtual tVoid vAddClient( ISpmWorkerClient *pClient );

virtual tVoid vRemoveClient( ISpmWorkerClient *pClient );

virtual tBool bPostMessage( const std::string& strClientIfName,
                            tU32               u32Message,
                            tU32               u32Parm = 0 );

virtual tBool bPostMessage( const TSpmWorkerMessage& msg );

virtual tBool bPostMessageToWorker( tU32 u32Message,
                                    tU32 u32Parm = 0 );

virtual tVoid vSynchronousCall( tU32   u32Message,
                                tVoid *args );

virtual tVoid vChangePriority( tU32 u32Priority );

virtual tVoid vRestorePriority( );

virtual tVoid vStart( const std::string& strName,
                      tU32               u32Prio,
                      tU32               u32Stack );

virtual tVoid vTerminate( );

tVoid vSetShutdownSupervision( tU32 u32Timeout ){ _u32ShutdownSupervision = u32Timeout; }

protected:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar */*puchData*/ ){} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual const tChar*getName( ) const { return( "spm_tclWorkerServer" ); }

// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm );

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );

static tVoid vSupervision( tVoid *pArg );

virtual tVoid vOnMessageQueueError( ){}

virtual tVoid vOnTerminate( );

tBool bIsClientAvailable( ISpmWorkerClient *pClient ) const;

virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

tU32                           _u32MaxMsgCount;

private:
std::set < ISpmWorkerClient* > _clients;

OSAL_tTimerHandle              _hSupervisionTimer;
OSAL_tMQueueHandle             _hSpmWorkerQueue;
tU32                           _u32ShutdownSupervision;

OSAL_tSemHandle                _hSemId;

tUInt                          _nStartupNumber;
tUInt                          _nCurrentStartupEnded;
tUInt                          _nStartupTime;
ISpmSupervisionManager        *_poclSupervisionManager;
ISpmSyncHandler               *_poclSyncHandler;
tU32                           _u32ErrorRetryCnt;
};

