/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_trace.h
  * @brief        This is header file contains functions for handling LCM trace related functionalities.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <set>
#define TRACE_S_IMPORT_INTERFACE_TYPES
#include "trace_if.h"

#include "spm_Config.h"
#include "spm_traceFileIds.h"
#include "spm_traceCommands.h"
#include "spm_ITrace.h"

#define U16_M_ERRMEM_SPM_ERROR(err_code)  ((tU16)( TR_COMP_SPM + ( err_code & 0x3FF ) ))

#define SPM_TRACE_CLASS_SPM                  ( TR_COMP_SPM + 0x0011 )
#define SPM_TRACE_CLASS_SPM_LAM              ( TR_COMP_SPM + 0x0012 )
#define SPM_TRACE_CLASS_SPM_SSM              ( TR_COMP_SPM + 0x0013 )
#define SPM_TRACE_CLASS_SPM_CSUH             ( TR_COMP_SPM + 0x0014 )
#define SPM_TRACE_CLASS_SPM_CSH              ( TR_COMP_SPM + 0x0015 )
#define SPM_TRACE_CLASS_SPM_SSH              ( TR_COMP_SPM + 0x0016 )
#define SPM_TRACE_CLASS_SPM_CSM              ( TR_COMP_SPM + 0x0017 )
#define SPM_TRACE_CLASS_SPM_REG              ( TR_COMP_SPM + 0x0018 )
#define SPM_TRACE_CLASS_SPM_SEC              ( TR_COMP_SPM + 0x0019 )

#define SPM_TRACE_CLASS_SPM_SRV              ( TR_COMP_SPM + 0x001C )
#define SPM_TRACE_CLASS_SPM_PRX              ( TR_COMP_SPM + 0x001D )
#define SPM_TRACE_CLASS_SPM_STARTUP          ( TR_COMP_SPM + 0x0000 )

#define SPM_TRACE_CLASS_SPM_PRJ              ( TR_COMP_SPM + 0x0021 )
#define SPM_TRACE_CLASS_SPM_LAM_PRJ          ( TR_COMP_SPM + 0x0022 )
#define SPM_TRACE_CLASS_SPM_SSM_PRJ          ( TR_COMP_SPM + 0x0023 )
#define SPM_TRACE_CLASS_SPM_CSUH_PRJ         ( TR_COMP_SPM + 0x0024 )
#define SPM_TRACE_CLASS_SPM_CSH_PRJ          ( TR_COMP_SPM + 0x0025 )
#define SPM_TRACE_CLASS_SPM_SSH_PRJ          ( TR_COMP_SPM + 0x0026 )
#define SPM_TRACE_CLASS_SPM_CSM_PRJ          ( TR_COMP_SPM + 0x0027 )
#define SPM_TRACE_CLASS_SPM_REG_PRJ          ( TR_COMP_SPM + 0x0028 )
#define SPM_TRACE_CLASS_SPM_SEC_PRJ          ( TR_COMP_SPM + 0x0029 )

// makros to easy interaction with etg_vTraceBinary
/*!
   calculate the parameter u32CompAndLevelId for etg_vTraceBinary from TraceClass and TraceLevel
*/
#define SPM_CALC_ETG_TRACE_CLASS(SPM_TRACE_CLASS, TRACE_LEVEL)    ( ( ( 0xFFFFu & (etg_tU16)( SPM_TRACE_CLASS ) ) << 16 ) | (etg_tU16)( TRACE_LEVEL ) )

/*!
   redefine format values for ETG (check CMG3G-11112) - replace by ETG internal values if provided
*/
#define SPM_ETG_FORMAT_VALUE_EN_DONE    0
#define SPM_ETG_FORMAT_VALUE_EN_T8      1
#define SPM_ETG_FORMAT_VALUE_EN_T16     2
#define SPM_ETG_FORMAT_VALUE_EN_T32     3
#define SPM_ETG_FORMAT_VALUE_EN_T64     4
#define SPM_ETG_FORMAT_VALUE_EN_STRING  5
#define SPM_ETG_FORMAT_VALUE_EN_LIST    6

/*!
   redefine format value of bit numbers to specify the format of a parameter for ETG (check CMG3G-11112) - replace by ETG internal values if provided
*/
#define SPM_ETG_NUM_FORMAT_BITS         3

#define SPM_ETG_CREATE_FORMAT_STRING                                    tU32 u32Format = SPM_ETG_FORMAT_VALUE_EN_T16 << ( 0 * SPM_ETG_NUM_FORMAT_BITS);
#define SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING(FormatValue, index)     u32Format |= ( FormatValue << ( index * SPM_ETG_NUM_FORMAT_BITS) );

// spm classes have a range from TR_COMP_SPM + 0x0000 up to TR_COMP_SPM + 0x001f

// special traces which can not be realized with help of ETG. Therefore they are coded
// the old way.
#define I_SPM_01                    0x1101
#define I_SPM_02                    0x1102
#define I_SPM_03                    0x1103
#define I_SPM_04                    0x1104

#define I_SPM_ERROR                 0x1110  // defines for the TRACE_SPM_... commands
#define I_SPM_INFO                  0x1111  // to allow for quick debug output
#define I_SPM_INFO_STRING           0x1112
#define I_SPM_ERROR_FILE_ID         0x1113
#define I_SPM_WARNING_FILE_ID       0x1114
#define I_SPM_ERROR_FILE_ID_STR     0x1115
#define I_SPM_INFO_STRING_EXT       0x1116

#define I_SPM_EMTRACE_ENTRY         0x11ff  // error memory entries

#define I_SPM_SSH_01                0x1601  // substate handler output
#define I_SPM_SSH_02                0x1602
#define I_SPM_SSH_03                0x1603

#define I_SPM_SSH_04                0x1604
#define I_SPM_SSH_05                0x1605
#define I_SPM_SSH_06                0x1606
#define I_SPM_SSH_07                0x1607
#define I_SPM_SSH_08                0x1608

#define I_SPM_SSH_CONFIG_00         0x1610
#define I_SPM_SSH_CONFIG_01         0x1611

#define I_SPM_BUILDVERSION          0x3000 // do not change this number, it's part of the build tooling

// ******************************************************************************
// spm trace command which consider the suppression of early traces during
// startup.
// ******************************************************************************

// ******************************************************************************
// spm trace commands to help debug new code. If they should stay in code for
// longer then they should be replaced by ETG commands.
// ******************************************************************************

#ifdef SPM_FATAL_ASSERT_DISABLED
   #define SPM_FATAL_ASSERT(tFalse)  NORMAL_M_ASSERT(tFalse)
#else
   #define SPM_FATAL_ASSERT(tFalse)  FATAL_M_ASSERT(tFalse)
#endif

#define TRACE_SPM_FAILURE   SPM_FATAL_ASSERT(FALSE)

#define SPM_RESET_VIA_ASSERT SPM_FATAL_ASSERT(FALSE)

#define TRACE_SPM_FAILURE_THREAD_INFO   \
   {                                     \
      if (NULL != _poclSystemPowerManager){ _poclSystemPowerManager->vSpmTraceThreadInfo(); } \
      OSAL_s32ThreadWait(SPM_ERRMEM_WRITE_DELAY_MS); \
      SPM_FATAL_ASSERT(FALSE);                  \
   }

#define TRACE_SPM_ERROR_MY_THREAD(pSpm)   \
   {                                     \
      NORMAL_M_ASSERT(FALSE); \
      pSpm->vSpmTraceThreadInfo(); \
   }

#define TRACE_SPM_FAILURE_MY_THREAD(pSpm)   \
   {                                     \
      NORMAL_M_ASSERT(FALSE); \
      pSpm->vSpmTraceThreadInfo(); \
      OSAL_s32ThreadWait(SPM_ERRMEM_WRITE_DELAY_MS); \
      SPM_FATAL_ASSERT(FALSE);                  \
   }

extern "C"
{
extern tBool etg_bIsTraceActive(tU16 u16CompID,
                                tU16 u16LevelId);

}

/*!
  *  \class spm_tclTrace
  *  \brief
  *    This class handles trace related functionalities. It adds and removes client for tracing.
  *    This class uses class spm_tclWorkerServerConfig to add and remove clients.
  *    The spm_tclFactory object creates an instance of this class. Class spm_tclSubStateHandler uses this class
  *    to enable early traces and check whether traces are active or not.
  ******
  */
class spm_tclTrace : public ISpmTrace
{
public:
spm_tclTrace(const ISpmFactory& factory,
             tBool                 bEarlyFunctionalityAllowed);
virtual ~spm_tclTrace();

virtual tVoid vGetReferences();

virtual tVoid vStartCommunication();

// implement trace client interface
virtual tVoid vHandleTraceMessage(const tUChar *puchData);

// routine to replace the generated trcGen_bIsTraceActive routine
// from ETG tooling.
static inline tBool spm_bIsTraceActive(tU16 u16CompID,
                                       tU16 u16LevelId){
   if (_poclTraceRef && _poclTraceRef->_bEarlyTraceEnabled){
      return( etg_bIsTraceActive(u16CompID, u16LevelId) );
   }
   return( FALSE );
}

// implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage(tU32 u32Message,
                             tU32 u32Parm = 0);

virtual tBool bHandleSynchrounousCall(tU32   u32Message,
                                      tVoid *args);

virtual const tChar*getName() const { return( "spm_tclTrace" ); }

virtual tVoid vAddTraceClient(ISpmBase *oTraceClient);

virtual tVoid vRemoveTraceClient(ISpmBase *oTraceClient);

protected:
virtual tVoid vRegisterTraceCallback();

virtual tVoid vUnregisterTraceCallback();

private:
static spm_tclTrace   *_poclTraceRef;
ISpmWorkerServer      *_poclWorkerServer;
tBool                  _bEarlyTraceEnabled;
tLcmUString            _ustrData;
OSAL_tIODescriptor     _hDeviceTrace;
std::set < ISpmBase* > _traceClientsSet;

static tVoid vTraceHandling(const tUChar *puchData);

};


#undef trcGen_bIsTraceActive
#define trcGen_bIsTraceActive spm_tclTrace::spm_bIsTraceActive

