/*!
  * \file spm_trace.cpp
  *  \brief
  *         contains trace relevant controls
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 20.04.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"
#include "spm_WorkerServerConfig.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_trace.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

spm_tclTrace*spm_tclTrace::_poclTraceRef = NULL;

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// //#define SPM_TRACE_FILE_ID   SPM_FILE_TRACE

spm_tclTrace::spm_tclTrace( const ISpmFactory& factory,
                            tBool              bEarlyFunctionalityAllowed )
   : ISpmTrace( factory )
   , _poclWorkerServer( NULL )
   , _bEarlyTraceEnabled( FALSE ){

   _poclTraceRef = this;
   _hDeviceTrace = (OSAL_tIODescriptor)OSAL_ERROR;

   if ( bEarlyFunctionalityAllowed ){
      // if no delay wanted activate always the early trace.
      _bEarlyTraceEnabled = TRUE;
      ETG_TRACE_USR4( ( "Early trace is now enabled: %dms", OSAL_ClockGetElapsedTime( ) ) );
   }
}

spm_tclTrace::~spm_tclTrace( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   _poclWorkerServer->vRemoveClient( this );
   _poclWorkerServer = NULL;

   spm_tclTrace::vUnregisterTraceCallback( );
   _poclTraceRef     = NULL;
}

tVoid spm_tclTrace::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    get all needed (dependency) references to other SPM objects from the factory.
  *    The dependency here is WorkerServer.
  *
  *  \param none
  *  \details
  *    LCM factory invokes this method to initialize the dependency of the spm_tclTrace.
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
}

tVoid spm_tclTrace::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    Starts communication with other LCM components.
  *
  *  \param none
  *  \details
  *    LCM factory invokes this method after invoking the method - vGetReferences.
  *    In this method, vRegisterTraceCallback method is called to register a callback function at the TTFIS trace device.
  *    and adds itself to client list of worker server. Worker server is used to reduce the latency of inter-thread communication in LCM.
  ******
  */
   vRegisterTraceCallback( );

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tBool spm_tclTrace::bHandleSynchrounousCall( tU32 u32Message,
                                             tVoid* /*args*/ ){
/*!
  * \fn
  *  \brief
  *    message handler for messages received from WorkerServer
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] args: not used.
  *  \details
  *    spm_tclWorkerServer invokes this method for notifying messages.
  ******
  */
   if ( ( u32Message == SPM_U32_WORKER_BROADCAST_EARLY_DELAY )
        && !_bEarlyTraceEnabled ){
      _bEarlyTraceEnabled = TRUE;
      ETG_TRACE_USR4( ( "Early trace is now enabled: %dms", OSAL_ClockGetElapsedTime( ) ) );
   }
   return( FALSE );
}

tVoid spm_tclTrace::vAddTraceClient( ISpmBase *oTraceClient ){
/*!
  * \fn
  *  \brief
  *    method to add the trace client
  *  \param [in] oTraceClient: trace client which is to be added.
  *  \return none
  *
  *  \details
  *    This method is called during construction of LCM from factory. Adds a new trace client to the set of LCM trace clients.
  *    Only if added to this set the method "vHandleTraceMessage" of respective client (object) is called.
  ******
  */
   _traceClientsSet.insert( oTraceClient );
}

tVoid spm_tclTrace::vRemoveTraceClient( ISpmBase *oTraceClient ){
/*!
  * \fn
  *  \brief
  *    method to remove the trace client
  *  \param [in] oTraceClient: trace client which is to be removed.
  *  \return none
  *
  *  \details
  *    This method will remove the trace client which is already added to the set of LCM trace clients.
  ******
  */
   std::set < ISpmBase* >::iterator it = _traceClientsSet.find( oTraceClient );

   if ( it != _traceClientsSet.end( ) ){
      _traceClientsSet.erase( it );
   }
} // vRemoveTraceClient

tVoid spm_tclTrace::vRegisterTraceCallback( ){
/*!
  * \fn
  *  \brief
  *    This method registers a callback function at the TTFIS trace device.
  *  \param [in] none
  *  \return none
  *
  *  \details
  *    This method will activates the TTFIS trace command channel for LCM.
  *    After this function call only LCM trace commands will work. This method is called from vStartCommunication() method.
  *    This must not be called before all vGetReferences functions are processed. Otherwise trace commands could use uninitialized references.
  ******
  */
   _hDeviceTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );
   if ( _hDeviceTrace != OSAL_ERROR ){
      // register with the trace driver
      OSAL_trIOCtrlLaunchChannel oTraceChannel;

      oTraceChannel.enTraceChannel = TR_TTFIS_SYSMANAGER;
      oTraceChannel.pCallback      = (OSAL_tpfCallback)vTraceHandling; // need to cast here because prototype does not match

      if ( OSAL_s32IOControl( _hDeviceTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, ( intptr_t )&oTraceChannel ) == OSAL_ERROR ){
            ETG_TRACE_FATAL( ( "spm_tclTrace::vRegisterTraceCallback() => OSAL_s32IOControl(OSAL_C_S32_IOCTRL_CALLBACK_REG) failed" ) );
         if ( OSAL_s32IOClose( _hDeviceTrace ) == OSAL_OK ){
            _hDeviceTrace = OSAL_ERROR;
         } else {
            ETG_TRACE_FATAL( ( "spm_tclTrace::vRegisterTraceCallback() => OSAL_s32IOClose failed" ) );
         }
      }
   } else {
            ETG_TRACE_FATAL( ( "spm_tclTrace::vRegisterTraceCallback() => OSAL_IOOpen failed" ) );
   }
} // vRegisterTraceCallback

tVoid spm_tclTrace::vUnregisterTraceCallback( ){
/*!
  * \fn
  *  \brief
  *    This method un-registers the callback function from the TTFIS trace device.
  *  \param [in] none
  *  \return none
  *
  *  \details
  *    This method will deactivates the TTFIS trace command channel for LCM. After this function call LCM trace commands will not work anymore.
  *    This method is called from the destructor of spm_tclTrace class.
  ******
  */
   if ( _hDeviceTrace != OSAL_ERROR ){
      OSAL_trIOCtrlLaunchChannel oTraceChannel;

      // de-register from osal trace driver
      oTraceChannel.enTraceChannel = TR_TTFIS_SYSMANAGER;
      oTraceChannel.pCallback      = (OSAL_tpfCallback)vTraceHandling;

      if ( OSAL_s32IOControl( _hDeviceTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, ( intptr_t )&oTraceChannel ) == OSAL_OK ){
         if ( OSAL_s32IOClose( _hDeviceTrace ) == OSAL_OK ){
            _hDeviceTrace = OSAL_ERROR;
         } else {
            ETG_TRACE_FATAL( ( "spm_tclTrace::vUnregisterTraceCallback() => OSAL_s32IOClose failed" ) );
         }
      } else {
            ETG_TRACE_FATAL( ( "spm_tclTrace::vUnregisterTraceCallback() => OSAL_s32IOControl(OSAL_C_S32_IOCTRL_CALLBACK_UNREG) failed" ) );
      }
   }
} // vUnregisterTraceCallback

tVoid spm_tclTrace::vTraceHandling( const tUChar *puchData ){
/*!
  * \fn
  *  \brief
  *    This method is the callback method from the TTFIS trace device for LCM trace commands.
  *  \param [in] none
  *  \return none
  *
  *  \details
  *    This method will be called as callback function whenever a LCM trace command is executed.
  *    A worker message will be sent to "ISpmTrace" indicating that a ttfis command has come which should be served by the trace client.
  ******
  */
   if ( _poclTraceRef && _poclTraceRef->_poclWorkerServer ){
      ETG_TRACE_USR4( ( "spm_tclTrace::vTraceHandling(): Handle trace command via worker thread." ) );
      // In the below line puchData should not be assigned to _ustrData using = or assign(puchData) because
      // puchData is not NULL terminating string it is a buffer having length 256 sent by trace device.
      _poclTraceRef->_ustrData.assign( puchData, 256 );
      _poclTraceRef->_poclWorkerServer->bPostMessage( "ISpmTrace", SPM_U32_WORKER_TRA_COMMAND );
   }
} // vTraceHandling

tVoid spm_tclTrace::vHandleMessage( tU32 u32Message,
                                    tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    This method handles the Messages from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm:
  *  \details
  *    This method handles the worker message "SPM_U32_WORKER_TRA_COMMAND".
  *    The method "vHandleTraceMessage" of all trace clients (which are added by vAddTraceClient) will be called to serve the ttfis command.
  *  \note
  *    The method "vHandleTraceMessage" of all trace clients is be called because more than one trace client can contribute to one ttfis command.
  ******
  */
   (tVoid)u32Parm;

   ETG_TRACE_USR1( ( "spm_tclTrace::vHandleMessage(): Message received: %d.", u32Message ) );
   if ( u32Message == SPM_U32_WORKER_TRA_COMMAND ){
      std::set < ISpmBase* >::iterator it;

      for ( it = _traceClientsSet.begin( ); it != _traceClientsSet.end( ); ++it ){
         // simple chain of command, do not break if first client has served
         // the command. Then several objects are able to contribute to one
         // ttfis command.
         ( * it )->vHandleTraceMessage( (const tUChar*)_ustrData.c_str( ) );
      }
      ETG_TRACE_FATAL( ( "spm_tclTrace::vHandleMessage(): SpmTrace command processed!" ) );
   }
} // vHandleMessage

