/*!
  * \file spm_SuperVisionManager.cpp
  *  \brief
  *    All aspects of supervision functionality is implemented here (Like watchdog handling, evaluation of lifesigns).
  *
  *  \note
  *  PROJECT: NextGen
  *  SW-COMPONENT: FC SPM
  *  COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 14.07.11  | TMS Fischer       | initial version
  * 27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SuperVisionManager.h"

// interfaces class definitions
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IFactory.h"
#include "spm_ISuperVisionClient.h"
#include "spm_ILocalAppManager.h"
#include "spm_IApplicationDatabase.h"
// spm helper


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_SuperVisionManager.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_SUPERVISIONMANAGER

#define SPM_WRITE_DIRECTLY  TRUE

spm_tclSupervisionManager::spm_tclSupervisionManager( const ISpmFactory& factory,
                                                      tU32               u32WaitInterval )
   : ISpmSupervisionManager( factory ){
   _poclOsalProxy                           = NULL;
   _poclWorkerServer                        = NULL;
   _poclSystemPowerManager                  = NULL;
   _poclSubStateHandler                     = NULL;
   _poclLocalAppManager                     = NULL;

   _bEnableTriggerHwWdt                     = TRUE;
   _bTimeTrackerRunning                     = FALSE;
   _bWdtGetpEnabled                         = FALSE;

   _hHeartBeatTimer                         = 0;
   _u32PreWarnTime                          = 0;

   _u32CCAHeartBeatCount                    = 0;

   _u32WaitInterval                         = u32WaitInterval;
   _u32MakeTrackerRunning                   = 0;

   _u32TimeStampCCAHeartBeatTimerRetriggerd = 0;
   _u32TimeStampCCAHeartBeatTimerChecked    = 0;
   _u32WdtCCAHeartBeatTime                  = 0;

   _u32LastSupervisionCall                  = 0;
}

spm_tclSupervisionManager::~spm_tclSupervisionManager( ){
   // and now give platform the chance to shutdown
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   ETG_TRACE_USR4( ( "... and now give platform the chance to shutdown -> WDT 40s" ) );
   if ( _bEnableTriggerHwWdt ){
      _poclOsalProxy->bStartWatchdog( SPM_ARION_WDG_TRIGGER_INTERVAL_SHUTDOWN );
   }

   if ( _poclWorkerServer != NULL ){
      _poclWorkerServer->vRemoveClient( this );
   }
   _poclSystemPowerManager = NULL;
   _poclLocalAppManager    = NULL;
   _poclOsalProxy          = NULL;
   _poclWorkerServer       = NULL;
   _poclSubStateHandler    = NULL;
}

tVoid spm_tclSupervisionManager::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *         Resolve inter-dependencies on other components by getting references to them from spm_tclFactory.
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclLocalAppManager,    ISpmLocalAppManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );

   #ifndef LCM_UNIT_TESTS
      dp_tclSpmDpFeatureDisableHwWatchdog oHwWdg;
      tU8                                 u8WdtRestartCount = (tU8)oHwWdg.tGetData( ); // KSM2HI, todo, 27.02.2017 : The below logic doesn't make sense to me because
                                                                                       // dp_tclSpmDpFeatureDisableHwWatchdog doesn't sound like a counter but is assigned to one.
                                                                                       // And this counter is nowhere changed (increased or decreased). The datapool item
                                                                                       // dp_tclSpmDpFeatureDisableHwWatchdog is used nowhere else in the LCM so it has no effect on
                                                                                       // nothing. Why?
      if ( u8WdtRestartCount != 0 ){
         _bEnableTriggerHwWdt = FALSE;

         ETG_TRACE_FATAL( ( "*********** HW WATCHDOG IS DISABLED for '%d' MORE STARTUPS!!! **************", u8WdtRestartCount ) );

         {
            std::string u8Dummy( 4, 0 );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)u8Dummy.c_str( ), (tU16)u8Dummy.length( ), FALSE );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)u8Dummy.c_str( ), (tU16)u8Dummy.length( ), FALSE );
            {
               std::string szStr;
               szStr = "*********** HW WATCHDOG IS DISABLED for '" + std::to_string( u8WdtRestartCount ) + "' MORE STARTUPS!!! **************";
               _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ) );

            }
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)u8Dummy.c_str( ), (tU16)u8Dummy.length( ), FALSE );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)u8Dummy.c_str( ), (tU16)u8Dummy.length( ), FALSE );
         }
      }
   #endif // ifndef LCM_UNIT_TESTS

} // vGetReferences

tVoid spm_tclSupervisionManager::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *         adds itself to client list of worker server.
  *         Worker server is used to reduce the latency of inter-thread communication in Spm.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tBool spm_tclSupervisionManager::bHandleSynchrounousCall( tU32   u32Message,
                                                          tVoid *args ){
/*!
  * \fn
  *  \brief
  *         message handler for messages received from WorkerServer
  *
  *  \param[out]
  *         return false if message is not intended for you,enabling other clients
  *         in the client list of WorkerServer to receive the message
  *         return true if message is intended for you.
  ******
  */
   (tVoid)u32Message;
   (tVoid)args;

   return( FALSE );
}

tVoid spm_tclSupervisionManager::vAddSupervisionClient( ISpmSupervisionClient *client ){
/*!
  * \fn
  *  \brief
  *   Adds a given client handler to List of Client Handlers.
  *
  *  \param[in] client: pointer to any object derived from ISpmSupervisionClient.
  ******
  */
   tClientHdlList.insert( client );
}

tVoid spm_tclSupervisionManager::vRemoveSupervisionClient( ISpmSupervisionClient *client ){
   /*!
     * \fn
     *  \brief
     *   Removes a given client handler from List of clients
     *
     *  \param[in] client: pointer to any object derived from ISpmSupervisionClient.
     ******
     */
   if ( tClientHdlList.find( client ) != tClientHdlList.end( ) ){
      tClientHdlList.erase( tClientHdlList.find( client ) );
   }
}

tVoid spm_tclSupervisionManager::vOnStart( ){
/*!
  * \fn
  *  \brief
  *         called just before the main entry function of thread is called by spm_tclActive.
  *         Do your last minute initializations before the thread is acivated.
  *         In this case, the hardware watchdog is started.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   #ifndef SPM_FEATURE_SD_NOTIFY_BY_PROJECT_CONFIG
      if ( _bEnableTriggerHwWdt ){
         _poclOsalProxy->bStartWatchdog( SPM_ARION_WDG_TRIGGER_INTERVAL );
      }
   #endif
}

tVoid spm_tclSupervisionManager::vOnTerminate( ){
/*!
  * \fn
  *  \brief
  *         called just before the thread is to be terminated via spm_tclActive
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   _poclWorkerServer->vRemoveClient( this );
   // and now set trigger interval to 90 sec to survive last application shutdown -> next trigger when destructor is called
   // set prescale value
   #ifdef SPM_FEATURE_ENABLE_PROCESSOR_WDT
      ETG_TRACE_USR4( ( "... and now set trigger interval to 90 sec to survive last application shutdown" ) );
      if ( _bEnableTriggerHwWdt ){
         _poclOsalProxy->bStartWatchdog( SPM_ARION_WDG_TRIGGER_INTERVAL_APPLICATION_SHUTDOWN );
      }
   #endif
}

tVoid spm_tclSupervisionManager::main( ){
/*!
  * \fn
  *  \brief
  *        entry point of SupervisionManager thread.triggers hardware watchdog
  *        through osalproxy and checks CCA heartbeat
  *
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   //calculate wait delay
   tU32 u32CurrentTime  = OSAL_ClockGetElapsedTime( );


   tU32 u32WaitInterval = _u32WaitInterval;

   if ( ( ( u32CurrentTime - _u32LastSupervisionCall ) - _u32WaitInterval ) < _u32WaitInterval ){
      u32WaitInterval = _u32WaitInterval - ( ( u32CurrentTime - _u32LastSupervisionCall ) - _u32WaitInterval );
   }
   ETG_TRACE_USR4( ( "spm_tclSupervisionManager::main(): delay: %dms (%dms)", u32WaitInterval, u32CurrentTime - _u32LastSupervisionCall ) );

   _u32LastSupervisionCall = u32CurrentTime;


   if ( OSAL_OK != OSAL_s32ThreadWait( u32WaitInterval ) ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   if ( bIsTerminated( ) ){
      return;
   }
   {
      std::set < ISpmSupervisionClient* >::iterator iter;
      for ( iter = tClientHdlList.begin( ); iter != tClientHdlList.end( ); iter++ ){
         // ETG_TRACE_USR4(("spm_tclSupervisionManager::main(): '%s'", (*iter)->getName() ));
         ( * iter )->vCheckSupervisionState( );
      }
   }

   #ifdef SPM_FEATURE_ENABLE_PROCESSOR_WDT
      if ( _bEnableTriggerHwWdt ){
         _poclOsalProxy->bTriggerWatchdog( );
      }
   #endif

   vCheckCCAHeartBeat( );

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->bPostMessage( "ISpmSystemPowerManager", SPM_U32_WORKER_SPM_CHECK_EMTRACE_VERS );

   _poclWorkerServer->bPostMessage( "ISpmLocalAppManager",    SPM_U32_WORKER_LAM_APP_CHECK_REQ );
} // main

tVoid spm_tclSupervisionManager::vWriteStringToErrmem( std::string pszString ){
/*!
  * \fn
  *  \brief
  *         Writing a string to error memory
  *  \param[in]
  *         string to be put into error memory
  ******
  */
   if ( _poclSystemPowerManager && ( !pszString.empty( ) ) ){
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)pszString.c_str( ), (tU16)pszString.length( ), SPM_WRITE_DIRECTLY );

   } else {
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
}

tVoid spm_tclSupervisionManager::vCheckCCAHeartBeat( ){
/*!
  * \fn
  *  \brief
  *         Responsible for checking CCA heartbeat.keeps triggering CCA Heartbeat and  handles situation when cca heartbeat goes missing.
  *         CCA heartbeat is reset by CCA MessageHandler.
  ******
  */
   std::string szStr;

   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );

   if ( bIsCCAHeartBeatWarnTime( ) ){

      szStr = "SPM: SW Watchdog Supervision warning: CCA-HeartBeat missing for " + std::to_string( _u32CCAHeartBeatCount ) + "s!!!!";
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
      TRACE_SPM_ERROR_MY_THREAD( _poclSystemPowerManager );
   }
   vNewCCAHeartBeatInterval( );
   _u32TimeStampCCAHeartBeatTimerChecked = OSAL_ClockGetElapsedTime( );
} // vCheckCCAHeartBeat

tVoid spm_tclSupervisionManager::vDisableWdtRetriggering( ){
/*!
  * \fn
  *  \brief
  *        Checks the reset watchdog flag and accordingly stops retriggering of hw wdt
  ******
  */
   ETG_TRACE_USR1( ( "Disabling WDT retriggering." ) );
   vEnableRetriggerHwWdt( FALSE ); // Disable retriggering of WDT here.
}

tVoid spm_tclSupervisionManager::vTraceInfo( ){
   std::set < ISpmSupervisionClient* >::iterator iter;

   for ( iter = tClientHdlList.begin( ); iter != tClientHdlList.end( ); iter++ ){
      ( * iter )->vTraceInfo( );
   }
}

tVoid spm_tclSupervisionManager::vStartUpFinished( ){
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   if ( _bEnableTriggerHwWdt ){
      _poclOsalProxy->bStartWatchdog( SPM_ARION_WDG_TRIGGER_INTERVAL );
   }
}

// EOF

