/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_ProcessSupervision.h
  * @brief        This is header file contains functions for ProcessSupervision.
  * @copyright    (C) 2013 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include "spm_IProcessSupervision.h"
#include <csignal>
#include <vector>

class ISpmWorkerServer;
class ISpmProcessDatabase;
class ISpmSystemPowerManager;
typedef std::vector < pid_t > TPidVector;

/*!
  *  \class spm_tclProcessSupervision
  *  \brief This class implements the process supervision logic on processes started by SPM and additional set of processes.
   The additional processes are those processes not started by SPM, but which are to be supervised by FC SPM.

   spm_tclProcessDatabase stores the information about the processes which are to be supervised by FC SPM. If any process in this list
   is missing, it updates the error memory with needed process information.

   The spm_tclFactory object creates an instance of this class. The spm_tclOsLinux uses this class to dump the list of (spm) started linux processes.

  */
class spm_tclProcessSupervision : public ISpmProcessSupervision
{
public:
spm_tclProcessSupervision( const ISpmFactory& factory );
virtual ~spm_tclProcessSupervision( );

virtual tVoid vMainMonitor( );

// Overrides from spm_tclActive
virtual tVoid vOnStart( ){}
virtual tVoid vOnTerminate( ){}
virtual tVoid main( );

virtual tVoid vHandleTraceMessage( const tUChar *puchData );

// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );

virtual tBool bSupervisionStop( tVoid );
virtual tVoid vSupervisionStart( tVoid );
virtual tBool bSupervisionChange( const std::string& strProcessName,
                                  tBool              bSupervisionActive );

virtual tBool bSupervisionChangeAll( tBool bSupervisionActive );

virtual tVoid vDumpStartedProcessList( tBool bToErrmem,
                                       tBool bToTrace );

virtual tBool bGetProcessHandleByName( const std::string& strProcessName,
                                       tU32              *pu32handle );

virtual tBool bRemoveProcessFromSupervisionList( const std::string& strProcessName );

virtual tVoid vRememberAdditionalProzessToSupervise( const std::string& strProcessName,
                                                     tU32               u32SupervisionMode );

virtual tVoid vDumpOOMInfo( ) const;

protected:
virtual const tChar*getName( ) const { return( "spm_tclProcessSupervision" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

virtual tVoid vStartLateCommunication( );

/* variables: */
static ISpmWorkerServer *_poclWorkerServer;
ISpmProcessDatabase     *_poProcessDatabase;
ISpmSystemPowerManager  *_poclSystemPowerManager;
OSAL_tSemHandle          _hProcessSupervisionSem;
OSAL_tSemHandle          _hPrcSupervisionStartSem;

// PRMIMPL(PRM Implementation for SIGCHLD Interface)
OSAL_tIODescriptor       hDevice;
tU32                     u32Data;

tU32                     _bProcSupervisionWriteToErrmem;
// OSAL_tThreadID _hThreadProcessSupervision;

/* methods: */
tVoid vCheckForExternalProcessesToSupervise( );

tBool bAddAdditionalProzessToSupervise( const std::string& strProcessName );


tVoid vActivateAdditionalProcessSupervision( tVoid );
tBool bCheckIfProcessAlive( pid_t tPid ) const;

tInt iGetProcessIdList( const std::string& strPidName,
                        TPidVector       & procPidList ) const;

tBool bDoRAMtest( tS32 lNumberOfBytes,
                  tS32 lStride ) const;

tVoid vSetSupervisionForAllProcesses( tBool bEnableSupervision );

tBool bChangeSupervisionForProcess( const std::string& strProcessName,
                                    tBool              bSupervisionActive );

private:
static spm_tclProcessSupervision *_pMyStaticRef;
tBool                             _bStopProcessSupervision;
tBool                             _bProcSuperVisionStopped;

public:
#ifdef SIGNALHANDLER_SIGCHLD
   static tVoid vSigHandler( tS32       signum,
                             siginfo_t *info,
                             tVoid     *ptr );

#endif
// static tVoid vProcessSupervisionThread(tVoid* pvArg);

// PRMIMPL
static tVoid vNotifyHandler( const tU32 *pu32Param );

};

/* Process info defines : */
#define READ_BUF_SIZE   1024
#define ROOT_PATH_PROC  "/proc"
#define OUTSIDE_ROOT_PATH_PROC  ".."
#define PATH_PROC_STATUS "/proc/%s/status"
#define PATH_PROC_NAME "%*s %s"
#define SPM_PROCSV_THREAD                       "SpmPrcSv"

