/*!
  * \file spm_SupervisionEnableSupervisor.cpp
  *  \brief
  *    Implementation of the check if Supervision in LCM is enabled or not
  *    it checks the following parameter
  *    - WD_OFF pin of the debug board
  *    - /opt/bosch/reset_disable.txt
  *    - TTFis command: SPMRESETSUPERVISIONDISABLE
  *   This is the central class to decide wether supervision is active or not.
  *  \note
  *  PROJECT: NextGen
  *  SW-COMPONENT: FC SPM
  *  COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim
  *  \see
  *  \version
  * 1.0 | 22.02.2016 | Michael Rossner CM/ESO3 | initial Version
  ******
  */
#include <boost/format.hpp>
#include "spm_Config.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SupervisionEnableSupervisor.cpp.trc.h"
#endif

#include "spm_GlobDefs.h"
#include "spm_IOsalProxy.h"
#include "spm_IOsLinux.h"
#include "spm_IWorkerClient.h"
#include "spm_ISuperVisionManager.h"

#include "spm_IFactory.h"
#include "spm_SupervisionEnableSupervisor.h"
#include "spm_ILateServiceHandler.h"
#include <string>
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

const std::string                                                      cStrLcmSupervisionDisableFile = "/opt/bosch/disable_reset.txt";

spm_tclSupervisionEnableSupervisor*spm_tclSupervisionEnableSupervisor::_pMyStaticRef                 = NULL;

spm_tclSupervisionEnableSupervisor::spm_tclSupervisionEnableSupervisor( const ISpmFactory& factory ) : ISpmSupervisionEnableSupervisor( factory )
   , _poclSupervisionManager( NULL )
   , _bGlobalSupervisionStatus( TRUE )
   , _bTTFisDisableRequest_Active( FALSE )
   , _bDisableResetFileRequest_Active( FALSE )
   , _bWdOffDebugSwitchRequest_Active( FALSE )
   , _bExternalDisableRequest_Active( FALSE ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in] factory: spm factory object.
  *  \todo
  *    Integration of SIGNAL HANDLER(SIGCHLD) for Process Supervision.\n
  *    Testing Process Supervision on Additional Set of Processes.
  *  \version
  ******
  */
   _pMyStaticRef = this;
}

spm_tclSupervisionEnableSupervisor::~spm_tclSupervisionEnableSupervisor( ){
   /*!
     * \fn
     *  \brief
     *    Destructor
     *
     *  \param
     ******
     */
   _pMyStaticRef                    = NULL;

   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vRemoveSupervisionClient( this );

   _poclSupervisionManager          = NULL;
}

tVoid spm_tclSupervisionEnableSupervisor::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclSupervisionEnableSupervisor.
  *  \param
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
}

tVoid spm_tclSupervisionEnableSupervisor::vStartCommunication( ){
   /*!
     * \fn
     *  \brief
     *   adds this class as supervision client and does the first initial check
     *  \param
     ******
     */
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
   poclOsalProxy->bConfigureSccResetHandling( !_bGlobalSupervisionStatus, 0 );

   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vAddSupervisionClient( this );

   dp_tclSpmDpFeatureDisableSupervision oDisableSupervision;
   tU8                                  NumberOfCyclesWithoutSupervision = oDisableSupervision.tGetData( );
   if ( NumberOfCyclesWithoutSupervision > 0 ){
      ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vStartCommunication GlobalSupervisionState disabled for %u more rounds", NumberOfCyclesWithoutSupervision ) );
      _bTTFisDisableRequest_Active = TRUE;
      NumberOfCyclesWithoutSupervision--;
      oDisableSupervision << NumberOfCyclesWithoutSupervision;
   }
} // spm_tclSupervisionEnableSupervisor::vStartCommunication

tVoid spm_tclSupervisionEnableSupervisor::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  *   This method adds the instance(this) of spm_tclSupervisionEnableSupervisor as the client of spm_tclWorkerServer.
  *   and does the first check
  *  \param
  ******
  */
   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poLateSrvHandler, spm_ISpmLateServiceHandler);
   poLateSrvHandler->vSupervisionStateChange( _bGlobalSupervisionStatus );
} // vStartLateCommunication

tVoid spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests( ){
   /*!
     * \fn
     *  \brief
     *   this is the central function of this module it checks the inputs if currently any disable request
     *   is active and send the update about final supppervision state to lcmlate if any changes
     *  \param
     *  \todo replace magic number 0 by DEV_WUP_C_U8_BITMASK_SUPERVISION_METHOD_NONE
     ******
     */
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsLinux,   ISpmOsLinux );
   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poLateSrvHandler, spm_ISpmLateServiceHandler);
   tBool bSuperVisionStatusUpdate = TRUE;
   tBool bRet            = poclOsalProxy->bGetGpio( OSAL_EN_DEBUG_WD_OFF, &_bWdOffDebugSwitchRequest_Active );
   if ( FALSE == bRet ){
      ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests() bGetGpio false" ) );
      return;
   }
   if ( poclOsLinux->bIsFileInFs( cStrLcmSupervisionDisableFile ) ){
      _bDisableResetFileRequest_Active = TRUE;
   } else{
      _bDisableResetFileRequest_Active = FALSE;
   }
   if (( SPM_ACTIVE_STATE_WD_OFF == _bWdOffDebugSwitchRequest_Active)
      || (TRUE == _bDisableResetFileRequest_Active)
      || (TRUE == _bTTFisDisableRequest_Active)
      || (TRUE == _bExternalDisableRequest_Active)){
      bSuperVisionStatusUpdate = FALSE;
   }
   if ( _bGlobalSupervisionStatus != bSuperVisionStatusUpdate){
      _bGlobalSupervisionStatus = bSuperVisionStatusUpdate;
      ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests() _bGlobalSupervisionStatus changed, current state: %d\n",
                           ETG_ENUM( SPM_STATE_COMMON, _bGlobalSupervisionStatus ) ) );
      if(_bWdOffDebugSwitchRequest_Active){
         ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests() _bGlobalSupervisionStatus disabled due to WD_OFF is Active" ) );
      }
      if(_bDisableResetFileRequest_Active){
         ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests() _bGlobalSupervisionStatus disabled due to disable_reset.txt existed" ) );
      }
      poclOsalProxy->bConfigureSccResetHandling( !_bGlobalSupervisionStatus, 0 );
      poLateSrvHandler->vSupervisionStateChange( _bGlobalSupervisionStatus );
   }

} // spm_tclSupervisionEnableSupervisor::vCheckSupervisionDisableRequests


tVoid spm_tclSupervisionEnableSupervisor::vCheckSupervisionState( ){
/*!
  * \fn
  *  \brief
  *    check all possible blockers of supervision
  ******
  */
   vCheckSupervisionDisableRequests( );
} // main

// ex. case SPM_SLV_CMD_PROCESS_SUPERVISION_STOP:
tVoid spm_tclSupervisionEnableSupervisor::vSupervisionStop( ){
/*!
  * \fn
  *  \brief
  *      Stops supervision. It sets a bit that is evaluated next time the Supervisor thread is activated
  ******
  */
   if ( FALSE == _bExternalDisableRequest_Active ){
      ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vSupervisionStop() Supervision disabled due to external request by other LCM module" ) );
   }
   _bExternalDisableRequest_Active = TRUE;
   vCheckSupervisionState( );
} // bSupervisionStop

tVoid spm_tclSupervisionEnableSupervisor::vSupervisionStart( ){
/*!
  * \fn
  *  \brief
  *      Stops supervision. It clears a bit that is evaluated next time the Supervisor thread is activated
  ******
  */
   if ( TRUE == _bExternalDisableRequest_Active ){
      ETG_TRACE_ERRMEM( ( "spm_tclSupervisionEnableSupervisor::vSupervisionStart() Supervision enabled due to external request by other LCM module" ) );
   }
   _bExternalDisableRequest_Active = FALSE;
   vCheckSupervisionState( );
}

tVoid spm_tclSupervisionEnableSupervisor::vTriggerSupervisionState( ){
   /*!
     * \fn
     *  \brief
     *      currently empty
     ******
     */
}

tVoid spm_tclSupervisionEnableSupervisor::vTraceInfo( ){
   /*!
     * \fn
     *  \brief
     *        The traces out the Supervision enabled/disabled state.
     *  \note
     *  \version
     ******
     */
      ETG_TRACE_FATAL( ( "----- SUPERVISOR: SUPERVISION_ENABLE -----" ) );

   if ( TRUE == _bWdOffDebugSwitchRequest_Active ){
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision disabled due to WD_OFF" ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision enabled due to WD_OFF" ) );
   }

   if ( TRUE == _bDisableResetFileRequest_Active ){
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision disabled due to file %s", cStrLcmSupervisionDisableFile.c_str( ) ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision enabled due to file %s", cStrLcmSupervisionDisableFile.c_str( ) ) );
   }

   if ( TRUE == _bExternalDisableRequest_Active ){
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision disabled due to request by some other LCM module" ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision enabled due to request by some other LCM module" ) );
   }

   if ( TRUE == _bTTFisDisableRequest_Active ){
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision disabled due to TTFisCommand!" ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() Supervision enabled due to TTFisCommand!" ) );
   }

   if ( FALSE == _bGlobalSupervisionStatus ){
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() GlobalSupervisionState disabled" ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclSupervisionEnableSupervisor::vTraceInfo() GlobalSupervisionState enabled" ) );
   }
} // vTraceInfo

