/**
  * @swcomponent   Life Cycle Management
  * @{
  * @file          spm_RamLoadSupervisor.cpp
  * @PROJECT:      CMD project
  * @brief         implementation of spm_tclRamLoadSupervisor class
  *
  * ----------------------------------------------------------------------------
  * @copyright     (C) 2018 Robert Bosch GmbH.
  *                The reproduction, distribution and utilization of this file as well as the
  *                communication of its contents to others without express authorization is prohibited.
  *                Offenders will be held liable for the payment of damages.
  *                All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_RamLoadSupervisor.h"

// interfaces class definitions
#include "spm_IWorkerClient.h"
#include "spm_IStartupSupervisor.h"
#include "spm_ISuperVisionManager.h"
#include "spm_ISystemLoadStatistics.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_RamLoadSupervisor.cpp.trc.h"
#endif

#include "spm_trace.h"

/*!
  * The size of buffer read from /proc/meminfo.
  */
#define SPM_LX_RAM_INFO_TEMP_FILE_BUFFER_SIZE  128

/*!
  * Path of /proc/meminfo which reports statistics about memory usage on the system.
  */
#define SPM_LX_RAM_INFO_FILE_PATH              "/dev/root/proc/meminfo"

spm_tclRamLoadSupervisor::spm_tclRamLoadSupervisor( const ISpmFactory& factory )
   : ISpmSupervisionClient( factory ),
   _oRamInfo( NULL ),
   _poclSupervisionManager( NULL ),
   _poclRamUsageStatistic( NULL ),
   _u32TriggerCount( 0 ),
   _u32RamUsageCheckPeriodS( SPM_RAM_USAGE_CHECK_PERIOD_SECONDS ){
   _oRamInfo = new spm_tclLxRamInfo( );
   //Indicate the state of RAM Load Supervision.
   //RamUsageCheckPeriodS can be configurable by spm_Config header (0 mean disable)
   if ( _u32RamUsageCheckPeriodS != 0 ){
      _bSupervisionEnabled = TRUE;
   } else {
      _bSupervisionEnabled = FALSE;
   }
   ETG_TRACE_USR1( ( "RAM usage supervision %d", ETG_ENUM( SPM_ONOFF_STATE, _bSupervisionEnabled ) ) );
   ETG_TRACE_USR1( ( "   - supervision period = %u s", _u32RamUsageCheckPeriodS ) );
}// spm_tclRamLoadSupervisor

spm_tclRamLoadSupervisor::~spm_tclRamLoadSupervisor( ){
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vRemoveSupervisionClient( this );
   _poclSupervisionManager = NULL;
   _poclRamUsageStatistic  = NULL;
   if ( _oRamInfo ){
      delete _oRamInfo;
      _oRamInfo = NULL;
   }
}// spm_tclRamLoadSupervisor

tVoid spm_tclRamLoadSupervisor::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *        Get the references to all used LCM objects.
  ******
  */
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSupervisionManager, spm_tclSupervisionManager, ISpmSupervisionManager );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclRamUsageStatistic,  spm_tclRamUsageStatistics, ISpmSystemLoadStatistics );
} // vGetReferences

tVoid spm_tclRamLoadSupervisor::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *        The Supervision Concept follows a sort of Subject-Observer Pattern.
  *        All observers(Supervision Clients) are required to register themselves at the Subject(SupervisionManager).
  *        The interface exposed for registration is vAddSupervisionClient(ISpmsupervisonclient*).
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vAddSupervisionClient( this );
}// spm_tclRamLoadSupervisor

tVoid spm_tclRamLoadSupervisor::vCheckSupervisionState( ){
/*!
  * \fn
  *  \brief
  *        This method is periodically called by the supervision-manager in per second and checks if the RAM usage (in percent) of system.
  *        The reading and getting meminfo will do every SPM_RAM_USAGE_CHECK_PERIOD_SECONDS second.
  *        This also updates the RAM usage statistic data.
  ******
  */
   if ( ( _u32RamUsageCheckPeriodS != 0 ) && ( ++_u32TriggerCount % _u32RamUsageCheckPeriodS == 0 ) ){
      if ( _oRamInfo != NULL ){
         if ( _oRamInfo->bReadStatusFromProcFileSystem( ) == TRUE ){
            tU32 u32RamUsage = _oRamInfo->u32GetRamUsage( );
            _poclRamUsageStatistic->bUpdateLoading( u32RamUsage );
            ETG_TRACE_USR3( ( "RAM usage supervision => period = %u s, RAM usage = %u percent", _u32RamUsageCheckPeriodS, u32RamUsage ) );
         } else {
            ETG_TRACE_ERRMEM( ( "spm_tclRamLoadSupervisor::vCheckSupervisionState fail to read /proc/meminfo" ) );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclRamLoadSupervisor::vCheckSupervisionState _oRamInfo is NULL" ) );
      }
   }
}// vCheckSupervisionState

tVoid spm_tclRamLoadSupervisor::vTraceInfo( ){
/*!
  * \fn
  *  \brief
  *        Trace the current overall RAM usage.
  *
  ******
  */
   ETG_TRACE_FATAL( ( "----- SUPERVISOR: RAM USAGE -----" ) );

   ETG_TRACE_FATAL( ( "Overall RAM usage: %u", _oRamInfo->u32GetRamUsage( ) ) );
}// vTraceInfo

spm_tclLxRamInfo::spm_tclLxRamInfo( )
   : ISpmLxRamInfo( ),
   u32MemTotal( 0 ),
   u32MemFree( 0 ){

}// spm_tclLxRamInfo

spm_tclLxRamInfo::~spm_tclLxRamInfo( ){}

tU32 spm_tclLxRamInfo::u32GetRamUsage( tVoid ){
/*!
  * \fn
  *  \brief
  *        Calculate the RAM usage in percent.
  *
  ******
  */
   if ( ( u32MemTotal > u32MemFree ) && ( u32MemTotal != 0 ) ){
      return( ( ( u32MemTotal - u32MemFree ) * 100 ) / u32MemTotal );
   }
   return( 0 );
}// ~spm_tclLxRamInfo

tBool spm_tclLxRamInfo::bReadStatusFromProcFileSystem( tVoid ){
/*!
  * \fn
  *  \brief
  *        Read from /proc/meminfo and get the total memory size, free mem.
  *
  ******
  */
   OSAL_tIODescriptor hIODescriptor = OSAL_ERROR;
   std::string        strPath;
   tS32               s32FileSize   = 0;
   tChar              szFileBuffer[SPM_LX_RAM_INFO_TEMP_FILE_BUFFER_SIZE];

   strPath       = SPM_LX_RAM_INFO_FILE_PATH;
   hIODescriptor = OSAL_IOOpen( strPath.c_str( ), OSAL_EN_READONLY );
   if ( hIODescriptor != OSAL_ERROR ){
      s32FileSize                   = OSAL_s32IORead( hIODescriptor, (tPS8)szFileBuffer, SPM_LX_RAM_INFO_TEMP_FILE_BUFFER_SIZE );
      (tVoid)OSAL_s32IOClose( hIODescriptor );
      if ( s32FileSize == OSAL_ERROR ){
         ETG_TRACE_FATAL( ( "spm_tclLxRamInfo::bReadStatusFromProcFileSystem => OSAL_s32IORead() for file '%32s' failed with s32FileSize = %d", strPath.c_str( ), s32FileSize ) );
         return( FALSE );
      }
      szFileBuffer[s32FileSize - 1] = '\0';
      if ( 2 == sscanf( szFileBuffer,
                        "%*s%u%*s%*s%u%*s",
                        &u32MemTotal,
                        &u32MemFree
                        ) ){
         return( TRUE );
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclLxRamInfo::bReadStatusFromProcFileSystem can not get meminfo" ) );
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclLxRamInfo::bReadStatusFromProcFileSystem() => OSAL_IOOpen() for '%32s' failed", strPath.c_str( ) ) );
   }

   return( FALSE );
}// bReadStatusFromProcFileSystem

