/*!
  * \file spm_CpuLoadSupervisor.cpp
  *  \brief
  *    Periodically determine the currently running processes and their subordinate threads via the /proc file system.
  *    Collect thread related data like CPU-usage and priority and check if the CPU-usage of a thread has crossed a configurable threshold for a configurable duration.
  *    Only those threads are considered for the CPU-usage verification which have a priority higher than a configurable priority-threshold.
  *    If a suspicious thread was detected then write some thread related information into the error memory and trigger a callstack generation.
  *    If configured also perform a reset of the application processor as final action.
  *
  *    The name of this file 'spm_CpuLoadSupervisor.cpp' is missleading as it implements a supervision of the CPU 'usage' and not the CPU 'load'.
  *    A CPU load also exists in the Linux terminology but the CPU load has a completely different meaning than the CPU usage.
  *    The name of the file and the name of the class are kept and the term 'load' is not changed to 'usage' to avoid unnecessary effort.
  *
  *  \b PROJECT: NextGen
  *  \b SW-COMPONENT: FC SPM
  *  \b COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
  *  \version
  *    Date      | Author              | Modification
  *    09.03.16  | CM-AI/ESO3   Kalms  | Overall rework due to new requirements.
  *    23.11.13  | CM-AI/PJVW32 Kollai | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_CpuLoadSupervisor.h"

// interfaces class definitions
#include "spm_ISuperVisionManager.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IStartupSupervisor.h"
#include "spm_IOsalProxy.h"
#include "spm_ISystemLoadStatistics.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_CpuLoadSupervisor.cpp.trc.h"
#endif

#include "spm_trace.h"

spm_tclCpuLoadSupervisor::spm_tclCpuLoadSupervisor( const ISpmFactory& factory )
   : ISpmSupervisionClient( factory ),
   _u32CpuUsageCheckPeriodS( 10 ),
   _bDoApplicationProcessorReset( 0 ),
   _u32ThreadCpuUsagePerCoreThreshold( 95 ),
   _msCpuUsageRealtimeTimeout( 10000 ),
   _msCpuUsageNiceTimeout( 30000 ),
   _lNicePriority( 20 ),
   _lRealtimePriority( 0 ),
   _u32NumberOfCpus( 0 ),
   _siNumberOfProcs( 0 ),
   _siNumberOfThreads( 0 ),
   _u32NumberOfCalculatedProcs( 0 ),
   _u32NumberOfCalculatedThreads( 0 ),
   _u32Duration( 0 ),
   _u32TriggerCount( 0 ),
   _u32CpuUsageSupervisionThread( 0 ),
   _ulPeriodOverallTicksSupervisionThread( 0 ),
   _poclSupervisionManager( NULL ),
   _poclSystemPowerManager( NULL ),
   _poclStartupSupervisor( NULL ),
   _poclOsalProxy( NULL ),
   _poclCpuUsageStatistic( NULL ){
   _bTriggeringEnabled                = FALSE;

   _u32NumberOfCpus                   = _oCpuInfo.u32GetNumberOfCpus( );

   // Load configurable items from datapool.
   dp_tclSpmDpInternDataCpuUsageCheckPeriodS           oCpuUsageCheckPeriodS;
   _u32CpuUsageCheckPeriodS           = oCpuUsageCheckPeriodS.tGetData( );

   dp_tclSpmDpInternDataDoApplicationProcessorReset    oDoApplicationProcessorReset;
   _bDoApplicationProcessorReset      = oDoApplicationProcessorReset.tGetData( );

   dp_tclSpmDpInternDataThreadCpuUsagePerCoreThreshold oThreadCpuUsagePerCoreThreshold;
   _u32ThreadCpuUsagePerCoreThreshold = oThreadCpuUsagePerCoreThreshold.tGetData( );

   if ( _u32ThreadCpuUsagePerCoreThreshold >= 100 ){
      ETG_TRACE_FATAL( ( "Thread related CPU usage threshold is configured with %u which is not allowed (max = 99). Set value to 95", _u32ThreadCpuUsagePerCoreThreshold ) );
      _u32ThreadCpuUsagePerCoreThreshold = 95;
   }

   dp_tclSpmDpInternDataCpuUsageRealtimeTimeoutMs oCpuUsageRealtimeTimeoutMs;
   _msCpuUsageRealtimeTimeout = oCpuUsageRealtimeTimeoutMs.tGetData( );

   dp_tclSpmDpInternDataCpuUsageNiceTimeoutMs     oCpuUsageNiceTimeoutMs;
   _msCpuUsageNiceTimeout     = oCpuUsageNiceTimeoutMs.tGetData( );

   dp_tclSpmDpInternDataRealtimePriority          oRealtimePriority;
   _lRealtimePriority         = oRealtimePriority.tGetData( );

   dp_tclSpmDpInternDataNicePriority              oNicePriority;
   _lNicePriority             = oNicePriority.tGetData( );
   //Indicate the state of CPU Load Supervision.
   //CpuUsageCheckPeriodS can be configurable by registry key CPU_USAGE_CHECK_PERIOD_S (0 mean disable)
   if ( _u32CpuUsageCheckPeriodS != 0 ){
      _bSupervisionEnabled = TRUE;
   } else {
      _bSupervisionEnabled = FALSE;
   }
   ETG_TRACE_USR1( ( "CPU usage supervision %d", ETG_ENUM( SPM_ONOFF_STATE, _bSupervisionEnabled ) ) );
   ETG_TRACE_USR1( ( "CPU usage supervision => CPU usage supervision started for %u CPU(s)", _u32NumberOfCpus ) );
   ETG_TRACE_USR1( ( "   - supervision period = %u s", _u32CpuUsageCheckPeriodS ) );
   ETG_TRACE_USR1( ( "   - thread related CPU usage threshold = %u (in relation to ONE core)", _u32ThreadCpuUsagePerCoreThreshold ) );
   ETG_TRACE_USR1( ( "   - realtime supervision / priority = %d, timeout = %u ms", (tUInt)_lRealtimePriority, _msCpuUsageRealtimeTimeout ) );
   ETG_TRACE_USR1( ( "   - nice supervision / priority = %d, timeout = %u ms", (tUInt)_lNicePriority, _msCpuUsageNiceTimeout ) );
   ETG_TRACE_USR1( ( "   - perform application processor reset at high CPU usage detection = %u", _bDoApplicationProcessorReset ) );
} // spm_tclCpuLoadSupervisor

spm_tclCpuLoadSupervisor::~spm_tclCpuLoadSupervisor( ){
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vRemoveSupervisionClient( this );

   _poclSupervisionManager = NULL;
   _poclSystemPowerManager = NULL;
   _poclStartupSupervisor  = NULL;
   _poclOsalProxy          = NULL;
   _poclCpuUsageStatistic  = NULL;
} // ~spm_tclCpuLoadSupervisor

tVoid spm_tclCpuLoadSupervisor::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *        Get the references to all used LCM objects.
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclStartupSupervisor,  ISpmStartupSupervisor );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclCpuUsageStatistic, spm_tclCpuUsageStatistics, ISpmSystemLoadStatistics );
} // vGetReferences

tVoid spm_tclCpuLoadSupervisor::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *        The Supervision Concept follows a sort of Subject-Observer Pattern.
  *        All observers(Supervision Clients) are required to register themselves at the Subject(SupervisionManager).
  *        The interface exposed for registration is vAddSupervisionClient(ISpmsupervisonclient*).
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vAddSupervisionClient( this );
} // vStartCommunication

tVoid spm_tclCpuLoadSupervisor::vCheckSupervisionState( ){
/*!
  * \fn
  *  \brief
  *        This method is periodically called by the supervision-manager and checks if the CPU usage of
  *        processes/threads with a certain priority is above a configurable threshold.
  *
  *        This also updates the cpu usage statistic data.
  *
  *        In case a process/thread is detected to be operating beyong its limits, state information about this process/thread
  *        as well as callstacks are collected and written to the error memory. If configured also an application processor
  *        reset is performed as final action.
  ******
  */
   OSAL_tMSecond msStarttime                     = 0;
   tULong        ulAllCpuCoresPeriodOverallTicks = 0;

   SPM_NULL_POINTER_CHECK( _poclStartupSupervisor );

   if ( ( _poclStartupSupervisor->u8GetStartupState( ) >= SPM_U8_STARTUP_STATE_SYSTEM_UP )
        && ( _u32CpuUsageCheckPeriodS != 0 )
        && ( ++_u32TriggerCount % _u32CpuUsageCheckPeriodS == 0 )
        && ( _u32NumberOfCpus > 0 ) ){
      msStarttime                     = OSAL_ClockGetElapsedTime( );
      ulAllCpuCoresPeriodOverallTicks = _oCpuInfo.ulUpdateFromProcFileSystem( );
      vCheckCPUUsageOfAllThreads( ulAllCpuCoresPeriodOverallTicks );
      tU32 u32CpuUsage = _oCpuInfo.u32GetCpuUsage( "cpu" );
      _poclCpuUsageStatistic->bUpdateLoading( u32CpuUsage );
      ETG_TRACE_USR3( ( "CPU usage supervision => period = %u s, duration = %u ms, caused CPU usage = %u with %u of %u period ticks per core, processes = %u/%u, tasks = %u/%u", _u32CpuUsageCheckPeriodS, ( OSAL_ClockGetElapsedTime( ) - msStarttime ), ( _u32CpuUsageSupervisionThread * _u32NumberOfCpus ), (tUInt)_ulPeriodOverallTicksSupervisionThread, (tUInt)( ulAllCpuCoresPeriodOverallTicks / _u32NumberOfCpus ), _u32NumberOfCalculatedProcs, (tUInt)_siNumberOfProcs, _u32NumberOfCalculatedThreads, (tUInt)_siNumberOfThreads ) );
   }
   return;
} // vCheckSupervisionState

/*!
  * \fn
  *  \brief
  *        This method is used to clear all so far collected supervision results and start the supervision anew.
  *
  *        For the CPU usage supervision this method is intentionally empty because there is no use case to clear the supervision results.
  ******
  */
tVoid spm_tclCpuLoadSupervisor::vTriggerSupervisionState( ){
   // Intentionally empty.
} // vTriggerSupervisionState

tVoid spm_tclCpuLoadSupervisor::vTraceInfo( ){
/*!
  * \fn
  *  \brief
  *        Trace the current overall CPU usage and the CPU usge of each individual CPU core.
  *        Also trace all threads and processes with their current CPU usage.
  ******
  */
   ETG_TRACE_FATAL( ( "----- SUPERVISOR: CPU USAGE -----" ) );

   if ( _u32NumberOfCpus > 0 ){
      ETG_TRACE_FATAL( ( "Overall CPU usage: %d", _oCpuInfo.u32GetCpuUsage( "cpu" ) ) );

      for ( tU8 i = 0; i < _u32NumberOfCpus; i++ ){
         std::string strCpuName = "cpux";
         strCpuName[3] = (tChar)( 0x30 + i );
         ETG_TRACE_FATAL( ( "%4s usage: %d", strCpuName.c_str( ), _oCpuInfo.u32GetCpuUsage( strCpuName ) ) );
      }

      ETG_TRACE_FATAL( ( "Processes = %u", (tUInt)_siNumberOfProcs ) );

      std::map < tInt, spm_tclLxProcessStatusInfo >::iterator itP;
      for ( itP = _mapProcessCpuUsageMap.begin( ); itP != _mapProcessCpuUsageMap.end( ); ++itP ){
         ETG_TRACE_FATAL( ( "CPU usage on one core = %03u / calculated = %u / high usage pending = %u => caused by process %d '%32s'",
                            ( itP->second.u32GetCpuUsage( ) * _u32NumberOfCpus ),
                            itP->second.bIsCalculated( ),
                            itP->second.bIsHighCpuUsagePending( ),
                            itP->first,
                            itP->second.strGetProcName( ).c_str( ) ) );
      }

      ETG_TRACE_FATAL( ( "Threads = %u", (tUInt)_siNumberOfThreads ) );

      std::map < tInt, spm_tclLxThreadStatusInfo >::iterator itT;
      for ( itT = _mapThreadCpuUsageMap.begin( ); itT != _mapThreadCpuUsageMap.end( ); ++itT ){
         ETG_TRACE_FATAL( ( "CPU usage on one core = %03u / calculated = %u / high usage detected = %u => caused by thread %d '%32s' with parent process %d '%32s'",
                            ( itT->second.u32GetCpuUsage( ) * _u32NumberOfCpus ),
                            itT->second.bIsCalculated( ),
                            itT->second.bIsHighCpuUsageDetected( ),
                            itT->first, itT->second.strGetThreadName( ).c_str( ),
                            itT->second.nGetParentProcID( ),
                            itT->second.strGetParentProcName( ).c_str( ) ) );
      }
   } else {
      ETG_TRACE_FATAL( ( "No CPUs detected" ) );
   }
} // vTraceInfo

tVoid spm_tclCpuLoadSupervisor::vCheckCPUUsageOfAllThreads( tULong ulAllCpuCoresPeriodOverallTicks ){
/*!
  * \fn
  *  \brief
  *        Determine the currently running processes and their subordinate threads via the /proc file system.
  *        Collect thread related data like CPU-usage and priority and maintain (add/update) this
  *        data in a list to be able to compare the previous values against the latest ones. Check if the
  *        CPU-usage of a thread has crossed a configurable threshold for a configurable duration. Only those
  *        threads are considered for the CPU-usage verification which have a priority higher than a configurable
  *        priority-threshold. If such a suspicious thread was detected then write some thread related
  *        information into the error memory and trigger a callstack generation. If configured also perform
  *        a reset of the application processor as final action.
  ******
  */
   OSAL_trIOCtrlDir    *pProcDir;
   OSAL_trIOCtrlDirent *pProcDirEntry;
   OSAL_trIOCtrlDir    *pThreadDir;
   OSAL_trIOCtrlDirent *pThreadDirEntry;

   std::set < tInt >    nThreadIDSet;
   std::set < tInt >    nProcessIDSet;
   std::set < tInt >    nProcessIDHighUsagePendingSet;
   std::vector < tInt > nToBeRemovedThreadsVector;
   std::vector < tInt > nToBeRemovedProcessesVector;
   tInt                 nProcID               = 0;
   tInt                 nThreadID             = 0;
   std::string          strProcName;
   OSAL_tMSecond        msHighCpuUsageTimeout = 0;
   tLong                lHighCpuUsagePriority = 0;
   spm_tclLxStatusInfo  oLxStatusInfo;

   // Determine the current processes and threads via the /proc file system and update the thread list.
   if ( ( pProcDir = OSALUTIL_prOpenDir( "/dev/root/proc" ) ) == OSAL_NULL ){
      ETG_TRACE_FATAL( ( "spm_tclCpuLoadSupervisor::vCheckCPUUsageOfAllThreads(): Failed to open directory '/dev/root/proc'. OSAL error code = 0x%04X.", OSAL_u32ErrorCode( ) ) );
      return;
   }
   while ( ( pProcDirEntry = OSALUTIL_prReadDir( pProcDir ) ) != OSAL_NULL ){
      nProcID = atoi( (tChar*)pProcDirEntry->s8Name );
      if ( nProcID > 0 ){
         std::string strTaskPath = "/dev/root/proc/" + std::to_string( nProcID ) + "/task";
         if ( ( pThreadDir = OSALUTIL_prOpenDir( strTaskPath.c_str( ) ) ) == OSAL_NULL ){
            ETG_TRACE_USR1( ( "spm_tclCpuLoadSupervisor::vCheckCPUUsageOfAllThreads(): Failed to open directory '/dev/root/proc/%d/task'. OSAL error code = 0x%04X.", nProcID, OSAL_u32ErrorCode( ) ) );
            break;
         }
         nProcessIDSet.insert( nProcID );
         if ( _mapProcessCpuUsageMap.find( nProcID ) == _mapProcessCpuUsageMap.end( ) ){
            spm_tclLxProcessStatusInfo tNew( nProcID );
            _mapProcessCpuUsageMap[nProcID] = tNew;
         }
         strProcName = _mapProcessCpuUsageMap[nProcID].strGetProcName( );
         while ( ( pThreadDirEntry = OSALUTIL_prReadDir( pThreadDir ) ) != OSAL_NULL ){
            nThreadID = atoi( (tChar*)pThreadDirEntry->s8Name );
            if ( nThreadID > 0 ){
               nThreadIDSet.insert( nThreadID );
               if ( _mapThreadCpuUsageMap.find( nThreadID ) == _mapThreadCpuUsageMap.end( ) ){
                  spm_tclLxThreadStatusInfo tNew( nThreadID, nProcID, strProcName );
                  _mapThreadCpuUsageMap[nThreadID] = tNew;
               }
            }
         }
         OSALUTIL_s32CloseDir( pThreadDir );
      }
   }
   OSALUTIL_s32CloseDir( pProcDir );

   // Run through the process list. Calculate the CPU usage of each process. If the CPU usage exceeds the threshold, then set the
   // high-CPU-usage pending flag and insert the respective process ID into a SET container for later evaluation by the thread processing.
   _u32NumberOfCalculatedProcs = 0;
   if ( !_mapProcessCpuUsageMap.empty( ) ){
      std::map < tInt, spm_tclLxProcessStatusInfo >::iterator it1;
      for ( it1 = _mapProcessCpuUsageMap.begin( ); it1 != _mapProcessCpuUsageMap.end( ); ++it1 ){
         if ( nProcessIDSet.find( it1->first ) == nProcessIDSet.end( ) ){
            nToBeRemovedProcessesVector.push_back( it1->first );
         } else {
            it1->second.vCalculateCpuUsage( ulAllCpuCoresPeriodOverallTicks );
            _u32NumberOfCalculatedProcs++;
            if ( ( it1->second.u32GetCpuUsage( ) * _u32NumberOfCpus ) >= _u32ThreadCpuUsagePerCoreThreshold ){
               nProcessIDHighUsagePendingSet.insert( it1->first );
               it1->second.vSetHighCpuUsagePending( TRUE );
            } else {
               it1->second.vSetHighCpuUsagePending( FALSE );
            }
         }
      }

      // Remove no more existing processes from the process list.
      std::vector < tInt >::iterator it2;
      for ( it2 = nToBeRemovedProcessesVector.begin( ); it2 != nToBeRemovedProcessesVector.end( ); ++it2 ){
         _mapProcessCpuUsageMap.erase( * it2 );
      }
   }

   // Run through the thread list. Calculate the CPU-usage of the processes threads and check against the thresholds of CPU-usage, priority and pending duration.
   // Only consider those threads for CPU usage calculation if they belong to a process for which the CPU usage threshold is already exceeded.
   // Also stop to consider further threads of the same process if the accumulated CPU usage of the already considered threads have reached such
   // a level that no thread within this process can be left to exceed the CPU usage threshold.
   _u32NumberOfCalculatedThreads = 0;
   if ( !_mapThreadCpuUsageMap.empty( ) ){
      std::map < tInt, tULong >                              _mapThreadAccumulatedOverallPeriodTicksMap;
      std::map < tInt, spm_tclLxThreadStatusInfo >::iterator it1;
      for ( it1 = _mapThreadCpuUsageMap.begin( ); it1 != _mapThreadCpuUsageMap.end( ); ++it1 ){
         if ( nThreadIDSet.find( it1->first ) == nThreadIDSet.end( ) ){
            nToBeRemovedThreadsVector.push_back( it1->first );
         } else if ( it1->second.bIsHighCpuUsageDetected( ) == FALSE ){

            if ( it1->second.bIsSupervisionThread( ) == TRUE ){
               it1->second.vCalculateCpuUsage( ulAllCpuCoresPeriodOverallTicks );
               _u32NumberOfCalculatedThreads++;
               _u32CpuUsageSupervisionThread          = it1->second.u32GetCpuUsage( );
               _ulPeriodOverallTicksSupervisionThread = it1->second.ulGetPeriodOverallTicks( );
            } else {
               tInt nParentProcID = it1->second.nGetParentProcID( );
               if ( nProcessIDHighUsagePendingSet.find( nParentProcID ) == nProcessIDHighUsagePendingSet.end( ) ){
                  it1->second.vResetCalculationData( );
                  continue;
               } else {
                  if ( _mapThreadAccumulatedOverallPeriodTicksMap.find( nParentProcID ) != _mapThreadAccumulatedOverallPeriodTicksMap.end( ) ){
                     if ( _mapThreadAccumulatedOverallPeriodTicksMap[nParentProcID] > (tU32)( _mapProcessCpuUsageMap[nParentProcID].ulGetPeriodOverallTicks( ) / ( 100 / ( 100 - _u32ThreadCpuUsagePerCoreThreshold ) ) ) ){
                        it1->second.vResetCalculationData( );
                        continue;
                     }
                     it1->second.vCalculateCpuUsage( ulAllCpuCoresPeriodOverallTicks );
                     _u32NumberOfCalculatedThreads++;
                     _mapThreadAccumulatedOverallPeriodTicksMap[nParentProcID] += it1->second.ulGetPeriodOverallTicks( );
                  } else {
                     it1->second.vCalculateCpuUsage( ulAllCpuCoresPeriodOverallTicks );
                     _u32NumberOfCalculatedThreads++;
                     _mapThreadAccumulatedOverallPeriodTicksMap[nParentProcID] = it1->second.ulGetPeriodOverallTicks( );
                  }
               }
            }

            if ( ( ( it1->second.u32GetCpuUsage( ) * _u32NumberOfCpus ) >= _u32ThreadCpuUsagePerCoreThreshold ) && ( it1->second.lGetRawPriority( ) <= _lNicePriority ) ){
               it1->second.vUpdateHighUsageDuration( );

               if ( it1->second.lGetRawPriority( ) <= _lRealtimePriority ){
                  if ( it1->second.msGetHighUsageDuration( ) >= _msCpuUsageRealtimeTimeout ){
                     msHighCpuUsageTimeout = _msCpuUsageRealtimeTimeout;
                     lHighCpuUsagePriority = _lRealtimePriority;
                     it1->second.vSetHighCpuUsageDetected( TRUE );
                  }
               } else {
                  if ( it1->second.msGetHighUsageDuration( ) >= _msCpuUsageNiceTimeout ){
                     msHighCpuUsageTimeout = _msCpuUsageNiceTimeout;
                     lHighCpuUsagePriority = _lNicePriority;
                     it1->second.vSetHighCpuUsageDetected( TRUE );
                  }
               }

               if ( it1->second.bIsHighCpuUsageDetected( ) == TRUE ){
                  tU8         u8Dummy = 0;
                  std::string strError;
                  // A suspicious thread was detected so write some thread related information to trace and error memory.
                  // Finally trigger a callstack generation for all threads of the parent process and (if configured) also perform a reset of the application processor.
                  SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
                  std::string strThread                         = std::to_string( it1->first );
                  std::string strParentProcID                   = std::to_string( it1->second.nGetParentProcID( ) );
                  std::string strRawPriority                    = std::to_string( it1->second.lGetRawPriority( ) );
                  std::string strCpuUsage                       = std::to_string( it1->second.u32GetCpuUsage( ) * _u32NumberOfCpus );
                  std::string strNumberOfCpus                   = std::to_string( _u32NumberOfCpus );
                  std::string strHighCpuUsageTimeout            = std::to_string( msHighCpuUsageTimeout );
                  std::string strHighCpuUsagePriority           = std::to_string( lHighCpuUsagePriority );
                  std::string strThreadCpuUsagePerCoreThreshold = std::to_string( _u32ThreadCpuUsagePerCoreThreshold );
                  strError = "Thread " + strThread + " '" + it1->second.strGetThreadName( ) + "' with parent process " + strParentProcID + " '" + it1->second.strGetParentProcName( ) + "' and a raw-priority of " + strRawPriority + " has a CPU usage of " + strCpuUsage + " on one of " + strNumberOfCpus + " CPU cores for longer than " + strHighCpuUsageTimeout + " ms";
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
                  ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
                  strError = "Threads with a raw-priority lower or equal to " + strHighCpuUsagePriority + " should not cause a CPU usage higher or equal to " + strThreadCpuUsagePerCoreThreshold + " for longer than " + strHighCpuUsageTimeout + " ms";
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy,                sizeof( u8Dummy ),        FALSE );
                  ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
                  SPM_NULL_POINTER_CHECK( _poclOsalProxy );
                  _poclOsalProxy->bDumpProcessInfo( (tU32)it1->second.nGetParentProcID( ), TRUE, SPM_BL_OSALPROXY_IF_ENABLED );
                  if ( _bDoApplicationProcessorReset == TRUE ){
                     _poclOsalProxy->bResetProcessor( DEV_WUP_C_U8_APPLICATION_PROCESSOR, DEV_WUP_C_U8_RESET_MODE_LOGGED, DEV_WUP_C_U16_RESET_REASON_UNSPECIFIED );
                  }
               }
            } else {
               it1->second.vResetHighUsageDuration( );
            }
         }
      }

      // Remove no more existing threads from the thread list.
      std::vector < tInt >::iterator it2;
      for ( it2 = nToBeRemovedThreadsVector.begin( ); it2 != nToBeRemovedThreadsVector.end( ); ++it2 ){
         _mapThreadCpuUsageMap.erase( * it2 );
      }
   }

   _siNumberOfProcs   = _mapProcessCpuUsageMap.size( );
   _siNumberOfThreads = _mapThreadCpuUsageMap.size( );
} // vCheckCPUUsageOfAllThreads

