/*!
  * \file spm_ThermalManagementServiceHandler.cpp
  *  \brief
  *    Generates from a string input and a delimiter set a stream of tokens.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 15.03.11  | TMS Fischer       |
  * 18.02.09  | Thuc Nguyen       | replace spm_tclStringTokenizer by C++11 Regex
  ******
  */

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "spm_StringTokenizer.h"
#include <utility>
#include <algorithm>
#include <set>

/*!
 * \fn
 *  \brief
 *    Constructor
 *
 *  \param[in] inputString:     string to be examined for splitting
 *  \param[in] delimiters:      delimiters to split the input string str
 *  \param[in] isDelimiterKept: FALSE - delimiters are discarded while splitting
 *                              TRUE - delimiters are kept as tokens while splitting
 *  \return
 */
spm_tclStringTokenizer::spm_tclStringTokenizer( const std::string& inputString,
                                                const std::string& delimiters,
                                                tBool              isDelimiterKept )
    : _inputString( inputString ),
    _delimiters( delimiters ),
    _isDelimiterKept( isDelimiterKept ){
    std::string regexString = strBuildRegexString( strHandleSpecialDelimiters( ) );
    _regexObject = std::move( std::regex( regexString ) );
    vRestart( );
}

std::string spm_tclStringTokenizer::strBuildRegexString( const std::string& delimiters ) const {
    std::string resultString;

    if( _isDelimiterKept ){
        // ([delimiters]|[^delimiters]+)
        resultString += "([" + delimiters + "]|[^" + delimiters + "]+)";
    } else {
        // [^delimiters]
        resultString += "[^" + delimiters + "]+";
    }

    return resultString;
}

std::string spm_tclStringTokenizer::strHandleSpecialDelimiters( ) const {
    // Step#1: remove duplicated delimiters by mechanism of
    // std::set (containers that store unique elements following a specific order)
    std::set<StrValType> uniqueDelimiters(_delimiters.begin(), _delimiters.end());
    std::string resultString(uniqueDelimiters.begin(), uniqueDelimiters.end());

    // Step#2: characters '-', '^', '[', ']', '\' have special meaning in 
    // the regular expression string => must escape them with escape character "\"
    // or place them in a position where they do not take on their special meaning

    // + Step#2.1: Using the std::regex_replace to move special characters(hyphen character '-')
    //             to the end of delimiters if having
    //      Construct the pattern to match array of delimiter characters which
    //             may contain hyphen '-' with conditions:
    //      - Contain zero or more characters are not the hyphen '-'    => group $1: (([^-]*))
    //      - and followed by the hyphen '-'                            => group $2: ([-])
    //      - and followed by zero or more characters                   => group $3: (.*)
    //      If delimiters match the pattern, then swap group $2: ([-]) which
    //             contain hyphen "-" with group $3: (.*)
    resultString = std::regex_replace(resultString, std::regex("([^-]*)([-])(.*)"), "$1$3$2");

    // + Step#2.2: Using the std::regex_replace to escape special characters '^', '[', ']', '\'
    //             by prepending the backslash into them
    //      Construct the pattern to match any delimiter character of:
    //      - Character '^'     =>  '\\^'
    //      - or character '['  =>  '\\['
    //      - or character ']'  =>  '\\]'
    //      - or character '\'  =>  '\\\\'
    //      If any of delimiters match the pattern, then escape them all
    //             with backslash => '\\$0'
    resultString = std::regex_replace(resultString, std::regex("[\\^\\[\\]\\\\]"), "\\$0");

    return resultString;
}

/*!
 * \fn
 *  \brief
 *    Reset to get tokens from the begin of the input string
 *
 *  \param
 *  \return void
 */
tVoid spm_tclStringTokenizer::vRestart( ){
    _stringIndex    = 0;
    _isDelimiterChar = false;
    _currentRegexIterator = RegexStrIt(_inputString.begin(), _inputString.end(), _regexObject);
}

/*!
 * \fn
 *  \brief
 *    Get the token which is splitted from the input string by delimiters
 *
 *  \param
 *  \return token as a string
 */
std::string spm_tclStringTokenizer::oNextToken( ){
    std::string tokenString;

    if( bHasMoreTokens() ){
        tokenString = _currentRegexIterator->str();
        _isDelimiterChar = ( tokenString.size() == 1 ) && ( _delimiters.find(tokenString) != std::string::npos );
        _stringIndex = (*_currentRegexIterator).position() + tokenString.size();
        ++_currentRegexIterator;
    } else {
        // reach the end of the input string
        _isDelimiterChar = false;
        _stringIndex = _inputString.size();
    }

    return tokenString;
}
