/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_IStartupSystem.h
  * @brief        This is header file contains interface functions for StartupSystem (start-up configuration).
  *               also interfaces regarding registry read or update and process stop or start.
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <set>
#include <list>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISwBlock.h"
#include "spm_GlobDefs.h"
#include "spm_baseClass.h"
#include "spm_IWorkerClient.h"
#include "spm_ProcConfig.h"
#include "spm_IStartupSystemGroupServer.h"
class ISpmApplicationDatabase;
class ISpmCcaSupplierServer;
class ISpmSwBlockServer;

#define SPM_STARTUP_VALUE_SYSTEM_RECOVERY_START_TYPE  "SYSTEMRECOVERY"
#define SPM_STARTUP_VALUE_SUPERVISION_START_TYPE      "SUPERVISION"

// //////////////////////////////////////////////////////////////////////////////
// structures and classes

// //////////////////////////////////////////////////////////////////////////////
// interface for a client class to work with spm_tclActive (inside a thread)

/*!
  *  \class ISpmStartupSystem
  *  \brief This abstract class provides interfaces regarding registry read or update and process stop or start.
   This class also delivers pure virtual functions to read and evaluate registries and startup configuration .
   It also supplies interfaces to add startup system and update registry to valid variant.
  */
class ISpmStartupSystem : public spm_tclActive
   , public ISpmSwBlockClient
   , public ISpmWorkerClient
{
protected:
    typedef enum {
        eFinal        = 0,
        eEqual        = 1,
        eNotEqual     = 2,
        eLess         = 3,
        eLessEqual    = 4,
        eBigger       = 5,
        eBiggerEqual  = 6,
        eAnd          = 7,
        eOr           = 8,
        eFailed       = 9
    }EOperation;

    typedef struct {
        tBool      bResult;
        EOperation eNextOperation;
        std::string strRemaining;
    }TOperationResult;

public:
ISpmStartupSystem( const ISpmFactory& factory ) : ISpmWorkerClient( factory ){ }
virtual ~ISpmStartupSystem( ){ }

virtual tVoid vCheckForProcessToStart( )                                             = 0;

virtual TOperationResult eCheckNextOperation(std::string strCommad)                  = 0;

virtual tVoid vSystemStart( )                                                        = 0;

virtual tBool bStopProcess( const std::string& strProcessName )                      = 0;

virtual tBool bStartProcess( const std::string& strProcessName )                     = 0;

virtual tVoid vRemoveSwBlock( const std::string& strSwBlockName,
                              tBool              bStopProcWithLastBlock = TRUE )     = 0;

virtual tBool bForceProcessStop( const std::string& strProcessName )                 = 0;

virtual tVoid vTraceProcessConfiguration( )                                          = 0;

virtual std::string strGetProcLocationByBlockName( const std::string& strBlockName ) = 0;

virtual std::string strGetProcNameByBlockName( const std::string& strBlockName )     = 0;

virtual tU32 u32GetSupervisionMode( tU32 ProcId )                                    = 0;

virtual std::string  strGetProcName( tU32 u32ProcId )                                = 0;

virtual void vOnSyncPointReached( const std::string& strSyncPointName )              = 0;

virtual void vOnCheckStartup( )                                                      = 0;

virtual const tChar*getInterfaceName( ) const { return( "ISpmStartupSystem" ); }

virtual tVoid vSetGroupEvaluator( ISpmStartupSystemGroupServer *grp )                = 0;

virtual tBool bAlternativeProcConfig( tChar *strProcessName, tBool bValidationOnly = FALSE ) = 0;
virtual tBool bAlternativeProcConfig( const TProcConfiguration& oProcConf, tChar *strProcessName, tBool bValidationOnly = FALSE ) = 0;
virtual tVoid vLoadSelectorRegistries( tChar *strProcessName)                        = 0;

virtual tU32  u32CalculateStartupRegistry(tU32 u32StartupConfig)                     = 0;

};

