/*!
  * \file spm_WakeupHandler.cpp
  *  \brief
  *    Checks wakeup reason at startup .
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author             | Modification
  * 21.01.14  | CM-AI/EPB2 Kollai    | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------
#include "spm_Config.h"

#include "spm_WakeupHandler.h"

#include "spm_IOsalProxy.h"
#include "spm_IStatistics.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ISupervisionEnableSupervisor.h"
#include "spm_IOsLinux.h"
#include "spm_rootdaemon.h"

#include "spm_IFactory.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_WakeupHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
#define SPM_WAKEUPHANDLER_RESETCOUNTER_INIT 255

spm_tclWakeupHandler::spm_tclWakeupHandler( const ISpmFactory& factory )
   : ISpmWakeupHandler( factory )
   , _u8StartType( SPM_U8_STARTTYPE_COLDSTART )
   , _u8ResetReason( spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_WARMSTART )
   , _u8ResetReasonScc( spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_WARMSTART )
   , _u8ResetCounter( SPM_WAKEUPHANDLER_RESETCOUNTER_INIT )
   , _u32WakeupReason( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON )
   , _u32OrgWakeupReason( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON )
   , _bIsReset( FALSE )
   , _poclOsalProxy( NULL )
   , _poRstStat( NULL )
   , _poclSystemPowerManager( NULL ){
}

spm_tclWakeupHandler::~spm_tclWakeupHandler( ){
   _poclOsalProxy          = NULL;
   _poRstStat              = NULL;
   _poclSystemPowerManager = NULL;
}

tVoid spm_tclWakeupHandler::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poRstStat, spm_tclResetStatistics, ISpmStatistics );

   // get wakeup reason
   _poclOsalProxy->bGetWakeupReason( &_u32WakeupReason );
   _u32OrgWakeupReason = _u32WakeupReason;

   dp_tclSpmDpInternDataShutdownConfirmed oShutdownConfirmed;

#ifdef SPM_ENABLE_DM_VERITY_CHECK
   tU8 u8OperationalState  = _poclOsalProxy->u8GetOperationalState( );
   tU8 u8DmVerityState     = _poclOsalProxy->u8GetDmVerityState( );

   ETG_TRACE_FATAL( ( "spm_tclWakeupHandler::vGetReferences(): DM Verity: OperationalState: %d, last DmVerity check state: %d.",  u8OperationalState, u8DmVerityState) );

   if ((DEV_WUP_C_U8_DM_VERITY_STATE_PASSED == u8DmVerityState) && (u8OperationalState != DEV_WUP_C_U8_OPERATIONAL_STATE_DMVERITY_CHECK) ){
       //switch off DM Verity for this start
       ETG_TRACE_FATAL( ( "spm_tclWakeupHandler::vGetReferences(): DM Verity is switched OFF --> Complete check will be done after next reboot (in OPERATIONAL state 'STATE_DMVERITY_CHECK')." ) );
       //system("echo 0 > /sys/block/dm-0/dm/onoff");

       SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", SYSTEMCTL_START_UNIT, "rbcm-switchoffdmv.service" );

   }

   if ( (u8OperationalState == DEV_WUP_C_U8_OPERATIONAL_STATE_DMVERITY_CHECK) || (DEV_WUP_C_U8_DM_VERITY_STATE_FAILED == u8DmVerityState)) {
       ETG_TRACE_FATAL( ( "spm_tclWakeupHandler::vGetReferences(): Keep DM Verity active and lock interface!" ) );
       SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", SYSTEMCTL_START_UNIT, "rbcm-lockdmv_if.service" );
   }

#endif


   tBool                                  bConfirmedFlag = TRUE;
   if ( DP_U8_ELEM_STATUS_INVALID == oShutdownConfirmed.u8GetData( bConfirmedFlag ) ){
      ETG_TRACE_USR1( ( "spm_tclWakeupHandler::vGetReferences(): unable to read 'ShutdownConfirmed' from PRAM." ) );
      bConfirmedFlag = TRUE;
   }

   _poclOsalProxy->bGetStartType( &_u8StartType );
   _u8ResetReason    = _poclOsalProxy->u8GetResetReason( );
   _u8ResetReasonScc = _poclOsalProxy->u8GetResetReasonScc( );

   if ( ( _u8StartType == SPM_U8_STARTTYPE_RESTART_SCC )
        && ( _u8ResetReasonScc == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_ESCC_USER )
        ){
      _u8ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_USER;

   }

   ETG_TRACE_FATAL( ( "spm_tclWakeupHandler::vGetReferences() _u8StartType=%d _u8ResetReason=%d _u8ResetReasonScc=%d bConfirmedFlag=%d",
                      ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                      ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                      ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                      ETG_ENUM( SPM_BOOL_STATE,        bConfirmedFlag )
                      ) );

   if ( !bConfirmedFlag ){
      _bIsReset = TRUE;
   }
   // check if last shutdown was reset
   if ( ( _u8ResetReason != spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET ) /* iMx was powered ON */
        && ( _u8ResetReason != spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_OFF )   /* iMx was fully turned OFF before it was reset */
        ){
      _bIsReset        = TRUE;
      _u32WakeupReason = ( tU32 ) spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART;
   }

   if ( ( _u8ResetReason == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET ) /* Coldstart by connect to battery*/
        || ( _u8ResetReason == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_LOSS )  /* PowerLossWarning was active at reset */
        || ( _u8ResetReason == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_OFF )   /* e.g. Abort-of-shutdown scenario*/
        ){
      if ( _bIsReset ){
                             ETG_TRACE_ERR( ( "SPM: *********************************************************************************************" ) );
                             ETG_TRACE_ERR( ( "SPM: PowerOffCheck: Detected start with valid startup reason, but without previous valid shutdown" ) );
                             ETG_TRACE_ERR( ( "SPM: _u32WakeupReason=%u ResetReason=%u _u8ResetReasonScc=%d StartType=%u bConfirmedFlag=%d RstCnt=%d",
                             ETG_ENUM( SPM_WAKEUP_REASON,     _u32WakeupReason ),
                             ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                             ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                             ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                             ETG_ENUM( SPM_BOOL_STATE,        bConfirmedFlag ),
                          _poclOsalProxy->u8GetResetCounter( )
                          ) );
                             ETG_TRACE_ERRMEM( ( "SPM: *********************************************************************************************" ) );
                             ETG_TRACE_ERRMEM( ( "SPM: PowerOffCheck: Detected start with valid startup reason, but without previous valid shutdown" ) );
                             ETG_TRACE_ERRMEM( ( "SPM: _u32WakeupReason=%u ResetReason=%u _u8ResetReasonScc=%d StartType=%u bConfirmedFlag=%d RstCnt=%d",
                             ETG_ENUM( SPM_WAKEUP_REASON,     (tUInt)_u32WakeupReason ),
                             ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                             ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                             ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                             ETG_ENUM( SPM_BOOL_STATE,        bConfirmedFlag ),
                             _poclOsalProxy->u8GetResetCounter( )
                             ) );
         if ( _u8ResetReason == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE ){
                           ETG_TRACE_USR1( ( "SPM: Possible restart by late incomming wake-up reason" ) );
         }
                           ETG_TRACE_USR1( ( "SPM: ****************************************************** **************************************" ) );
      } else {
                           ETG_TRACE_USR1( ( "SPM: **********************************************************" ) );
                           ETG_TRACE_USR1( ( "SPM: PowerOffCheck: Detected start with previous valid shutdown" ) );
                           ETG_TRACE_USR1( ( "SPM: _u32WakeupReason=%u ResetReason=%u StartMode=%u _u8ResetReasonScc=%d,  bConfirmedFlag=%d RstCnt=%d",
                           ETG_ENUM( SPM_WAKEUP_REASON,  _u32WakeupReason ),
                           ETG_ENUM( SPM_LCMRESETREASON, _u8ResetReason ),
                           ETG_ENUM( SPM_STARTTYPE,      _u8StartType ),
                           _u8ResetReasonScc,
                           ETG_ENUM( SPM_BOOL_STATE,     bConfirmedFlag ),
                           _poclOsalProxy->u8GetResetCounter( )
                           ) );
         if ( _u8ResetReason == spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE ){
                        ETG_TRACE_USR1( ( "SPM: Possible restart by late incomming wake-up reason" ) );
         }
                        ETG_TRACE_USR1( ( "SPM: **********************************************************" ) );
      }
   } else {
                        ETG_TRACE_USR1( ( "SPM: ****************************************************************" ) );
                        ETG_TRACE_USR1( ( "SPM: PowerOffCheck: Detected (re)start." ) );
                        ETG_TRACE_USR1( ( "SPM: _u32WakeupReason=%u ResetReason=%u StartMode=%u _u8ResetReasonScc=%d,  bConfirmedFlag=%d RstCnt=%d",
                        ETG_ENUM( SPM_WAKEUP_REASON,  _u32WakeupReason ),
                        ETG_ENUM( SPM_LCMRESETREASON, _u8ResetReason ),
                        ETG_ENUM( SPM_STARTTYPE,      _u8StartType ),
                        _u8ResetReasonScc,
                        ETG_ENUM( SPM_BOOL_STATE,     bConfirmedFlag ),
                        _poclOsalProxy->u8GetResetCounter( )
                        ) );
                        ETG_TRACE_USR1( ( "SPM: ****************************************************************" ) );
   }

   #ifdef SPM_FEATURE_ERRMEM_INFO
      ETG_TRACE_ERRMEM( ( "SPM: WakeupReason=%u ResetReason=%u ResetReasonScc=%d StartType=%u bConfirmedFlag=%d RstCnt=%d",
                          ETG_ENUM( SPM_WAKEUP_REASON,     _u32WakeupReason ),
                          ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                          ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                          ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                          ETG_ENUM( SPM_BOOL_STATE,        bConfirmedFlag ),
                          _poclOsalProxy->u8GetResetCounter( )
                          ) );
   #endif

} // vGetReferences

tVoid spm_tclWakeupHandler::vStartCommunication( ){
}

tVoid spm_tclWakeupHandler::vStartLateCommunication( ){
   SPM_NULL_POINTER_CHECK( _poRstStat );

   dp_tclSpmDpInternDataConnectToBattery oConnectToBatteryState;
   dp_tclSpmDpInternDataResetSetBySpm    oResetBySpm;

   ETG_TRACE_USR1( ( "spm_tclWakeupHandler::vStartLateCommunication(): _bIsReset=%u, _u8StartType=%d, _u8ResetReason=%d, _u8ResetReasonScc=%d, oResetBySpm.tGetData()=%d",
                     ETG_ENUM( SPM_BOOL_STATE,        _bIsReset ),
                     ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                     ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                     ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                     oResetBySpm.tGetData( )
                     ) );

   if ( ( !_bIsReset ) && ( _u8StartType == SPM_U8_STARTTYPE_WARMSTART ) ){
      oResetBySpm << SPM_U32_SHUTDOWN_INVALID;
      oConnectToBatteryState.vSetData( FALSE );
      // normal startup
   } else if ( _u8StartType == SPM_U8_STARTTYPE_COLDSTART ){
      // connect to battry detected --> remember persitent
      if ( _u8ResetReason != spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET ){
         ETG_TRACE_FATAL( ( "spm_tclWakeupHandler::vStartLateCommunication(): Detected mismatch of Startmode: %u and  Reset reason: %u",
                            ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                            ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ) ) );
      }
      dp_tclSpmDpInternDataConnectToBatteryCount oConnBatCount;

      tU16                                       u16BatCount = oConnBatCount.tGetData( );
      u16BatCount++;
      oConnectToBatteryState.vSetData( TRUE );
      oConnBatCount.vSetData( u16BatCount );

      #ifdef SPM_FEATURE_ENABLE_CODE_REQUEST_BY_PWR_ON
         // check for Ubat drop
         dp_tclSpmDpUBatDropDetected oUBatDropDetect( TRUE );
      #endif

      _poRstStat->vSetNewData( spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_POWER_ON_RESET );

      oResetBySpm << SPM_U32_SHUTDOWN_INVALID;


   } else { // _u8StartType == SPM_U8_STARTTYPE_RESTART_AP or _u8StartType == SPM_U8_STARTTYPE_UNKNOWN
      oConnectToBatteryState.vSetData( FALSE );
      tU8  u8ResetSetBySpm = oResetBySpm.tGetData( );
      oResetBySpm << SPM_U32_SHUTDOWN_INVALID;

      ETG_TRACE_USR4( ( "spm_tclWakeupHandler::vStartLateCommunication(): !!! ResetSetBySpm %u !!!", ETG_ENUM( SPM_RESET_REASON, u8ResetSetBySpm ) ) );

      tU32 u32ResetReason  = _u8ResetReason;

      if ( u8ResetSetBySpm != SPM_U32_SHUTDOWN_INVALID ){

         switch ( u8ResetSetBySpm ){
            case SPM_U32_SHUTDOWN_RESET:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_SOFTWARE;
               break;

            case SPM_U32_SHUTDOWN_OVERTEMPERATURE:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_OVERTEMPERATURE;
               break;

            case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_DOWNLOAD;
               break;

            case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_ERROR:
            case SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_TERMINATED:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_DOWNLOAD_ERROR;
               break;

            case SPM_U32_SHUTDOWN_AFTER_LONG_PRESS:
            case SPM_U32_SHUTDOWN_USER_RESET:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_USER;
               break;

            case SPM_U32_SHUTDOWN_ECU_RESET:
            case SPM_U32_SHUTDOWN_ALTSW_TESTMANAGER:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_ECU;
               break;

            case SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL:
            case SPM_U32_SHUTDOWN_NOTIFY_PROBLEM:
            case SPM_U32_SHUTDOWN_APP_END_WANTED:
            case SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_APP_ERROR;
               break;

            case SPM_U32_SHUTDOWN_PROCESS_SUPERVISION:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_APP_ERROR;
               break;

            case SPM_U32_SHUTDOWN_WATCHDOG_MID_SUPERVISION:
            case SPM_U32_SHUTDOWN_WATCHDOG_HIGH_SUPERVISION:
            case SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION:
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_LCM_INTERN;
               break;

            default:
               ETG_TRACE_ERRMEM( ( "spm_tclWakeupHandler::vStartLateCommunication(): _bIsReset=%u, _u8StartType=%d, _u8ResetReason=%d, _u8ResetReasonScc=%d, u8ResetSetBySpm=%d",
                                   ETG_ENUM( SPM_BOOL_STATE,        _bIsReset ),
                                   ETG_ENUM( SPM_STARTTYPE,         _u8StartType ),
                                   ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReason ),
                                   ETG_ENUM( CFC_FI_LCMRESETREASON, _u8ResetReasonScc ),
                                   ETG_ENUM( SPM_RESET_REASON,      u8ResetSetBySpm )
                                   ) );
               u32ResetReason = spm_fi_tcl_e8_LcmResetReason::FI_EN_SPM_EHMI_UNKNOWN;
               break;
         } // switch
      }
      oResetBySpm << SPM_U32_SHUTDOWN_INVALID;
      _poRstStat->vSetNewData( u32ResetReason );
   }

   if ( ( _u8StartType != SPM_U8_STARTTYPE_WARMSTART ) && ( _u8StartType != SPM_U8_STARTTYPE_RESTART_AP ) ){
      _poRstStat->vSetNewData( (tU32)_u8ResetReasonScc );
   }
   // remember Wakeup reason
   SPM_GET_CLASS_REFERENCE_NEW_VAR( poWakeupStat, spm_tclWakeupStatistics, ISpmStatistics );
   poWakeupStat->vSetNewData( (tU32)_u32WakeupReason );
} // vStartCommunication

tVoid spm_tclWakeupHandler::vSetResetCounter( tU8 u8ResetCounter ){
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   _poclOsalProxy->bSetResetCounter( u8ResetCounter );
   _u8ResetCounter = u8ResetCounter;
}

tVoid spm_tclWakeupHandler::vSetWakeupConfiguration( tU32 u32ConfigNb ){
   (tVoid)u32ConfigNb;
}

/*! \fn spm_tclWakeupHandler::vCheckResetCounter()
  *  \brief Checks the reset counter and resets the reset counter if CyclicResetHandling should be disabled
  *  \param none
  *  \exception none
  *  \return none
  */
tVoid spm_tclWakeupHandler::vCheckResetCounter( ){
   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );

   // handling of reset counter
   _u8ResetCounter   = _poclOsalProxy->u8GetResetCounter( );
   tBool bSupervisionState;
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionEnableSupervisor, ISpmSupervisionEnableSupervisor );

   bSupervisionState = poclSupervisionEnableSupervisor->bIsSupervisionStopped( );
   if ( TRUE == bSupervisionState ){
      vSetResetCounter( 0 );
      const std::string strErrorBuffer = "Cyclic Reset Handling disabled due to Supervision settings (WD_OFF/disable_reset.txt/TTFis-SpmResetSupervisionDisable)";
      ETG_TRACE_FATAL( ( "%s", strErrorBuffer.c_str( ) ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrorBuffer.c_str( ), (tU16)strErrorBuffer.length( ) );
   }

   dp_tclSpmDpConfigDisableCyclicResetHandling oDisableCyclicReset;
   tBool                                       bDisableCyclicReset = oDisableCyclicReset.tGetData( );
   if ( bDisableCyclicReset == TRUE ){
      vSetResetCounter( 0 );
      std::string strErrorBuffer = "Cyclic Reset Handling disabled due to Datapool setting";
      ETG_TRACE_FATAL( ( "%s", strErrorBuffer.c_str( ) ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strErrorBuffer.c_str( ), (tU16)strErrorBuffer.length( ) );
   }

   ETG_TRACE_USR1( ( "ResetCounterhandling: ResetCounter: %d, WD_OFF state: %d, datapool DisableCyclicResetHandling state: %d",
                     _u8ResetCounter,
                     ETG_ENUM( SPM_BOOL_STATE, bSupervisionState ),
                     ETG_ENUM( SPM_BOOL_STATE, bDisableCyclicReset )
                     ) );
} // vCheckResetCounter


/*! \fn spm_tclWakeupHandler::u8GetResetCounter()
  *  \brief return value of reset counter
  *  \param none
  *  \exception none
  *  \return tU8
  */
tU8 spm_tclWakeupHandler::u8GetResetCounter( ){
   if ( SPM_WAKEUPHANDLER_RESETCOUNTER_INIT == _u8ResetCounter ){
      vCheckResetCounter( );
   }
   return( _u8ResetCounter );
} // u8GetResetCounter

