/************************************************************************
  * FILE:         spm_TimerHandler.cpp
  * PROJECT:      NextGen
  * SW-COMPONENT: fc_spm
  * ----------------------------------------------------------------------
  *
  * DESCRIPTION:  Helper class for Timer Handling.
  *
  * ----------------------------------------------------------------------
  * COPYRIGHT:    (c) 2018 Robert Bosch GmbH, Hildesheim
  * HISTORY:
  * Date      | Author               | Modification
  * 25.04.18  | CM-CI1/ERN2-E Kollai | initial version
  *
  *************************************************************************/

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"

#include "spm_TimerHandler.h"

#include "spm_IFactory.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_TimerHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------

spm_tclTimerHandler::spm_tclTimerHandler( const ISpmFactory& factory )
   : ISpmTimerHandler( factory )
, _hTimerHdl(OSAL_C_INVALID_HANDLE)
, _poclWorkerServer( NULL )

{
    _mapTimer.clear();

    if ( OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback) vTimerCallback, (tPVoid)this, &_hTimerHdl ) ) {
        _hTimerHdl = OSAL_C_INVALID_HANDLE;
    }
}

spm_tclTimerHandler::~spm_tclTimerHandler( ){

    if ( OSAL_OK == OSAL_s32TimerDelete( _hTimerHdl ) ) {
        _hTimerHdl = OSAL_C_INVALID_HANDLE;
    }

    SPM_NULL_POINTER_CHECK( _poclWorkerServer );
    _poclWorkerServer->vRemoveClient( this );
    _poclWorkerServer = NULL;
}

tVoid spm_tclTimerHandler::vGetReferences( ){
    SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
    _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclTimerHandler::vStartCommunication( ){
    OSAL_s32TimerSetTime( _hTimerHdl, 5000, 0 );
}

tVoid spm_tclTimerHandler::vHandleMessage( tU32 u32Message, tU32 u32Parm ){
   (tVoid)u32Parm;
   if ( u32Message == SPM_U32_WORKER_TMR_TIMER_EXPIRED) {
       vCheckTimerTable();
   }
}

OSAL_tpfCallback spm_tclTimerHandler::vTimerCallback(tVoid* pvArg) {
    spm_tclTimerHandler* poMyRef = (spm_tclTimerHandler*) pvArg;
    if (poMyRef) {
        poMyRef->_poclWorkerServer->bPostMessage( "ISpmTimerHandler", SPM_U32_WORKER_TMR_TIMER_EXPIRED, 0 );
    }
    return 0;
}


tVoid spm_tclTimerHandler::vStartTimer(std::string strClientIfName, std::string strTimeName, tU32 u32Intervall, tU32 u32CycleTime, tU32 u32Message) {

    ETG_TRACE_USR1( ( "spm_tclTimerHandler::vStartTimer(): New timer started '%30s' by '%30s' with interval %dms (cycle: %dms), msg to send: %d!",
            strTimeName.c_str(), strClientIfName.c_str(), u32Intervall, u32CycleTime, u32Message) );

    std::map < std::string, TTimerInfo >::iterator it = _mapTimer.find(strTimeName);

    if (it != _mapTimer.end()) {
        //timer will be restarted or deleted (if u32Intervall == 0)
        ETG_TRACE_USR1( ( "spm_tclTimerHandler::vStartTimer(): Timer already exists!" ) );
        if (u32Intervall == 0) {
            ETG_TRACE_USR1( ( "spm_tclTimerHandler::vStartTimer(): delete timer!" ) );
            _mapTimer.erase(strTimeName);
        } else {
            _mapTimer[strTimeName].strClientIfName = strClientIfName;
            _mapTimer[strTimeName].u32OsalExpireTime = OSAL_ClockGetElapsedTime() + u32Intervall;
            _mapTimer[strTimeName].u32CycleTime = u32CycleTime;
            _mapTimer[strTimeName].u32MessageToSend = u32Message;

            ETG_TRACE_USR1( ( "spm_tclTimerHandler::vStartTimer(): restart timer with with osal time --> %d!", _mapTimer[strTimeName].u32OsalExpireTime ) );
        }
    } else {
        TTimerInfo tInfo;
        tInfo.strClientIfName = strClientIfName;
        tInfo.u32OsalExpireTime = OSAL_ClockGetElapsedTime() + u32Intervall;
        tInfo.u32CycleTime = u32CycleTime;
        tInfo.u32MessageToSend = u32Message;

        _mapTimer[strTimeName] = tInfo;
    }
    vCheckTimerTable();
}

tVoid spm_tclTimerHandler::vCheckTimerTable() {

    tU32 u32RemainingTime, u32IntervalTime;
    if (OSAL_s32TimerGetTime(_hTimerHdl, &u32RemainingTime, &u32IntervalTime) == OSAL_OK) {
        if (u32RemainingTime == 0) {
            OSAL_s32TimerSetTime( _hTimerHdl, 5000, 0 );
        }
    }

    tU32 u32CurrentTime     = OSAL_ClockGetElapsedTime();
    tU32 u32EarliestTimeout = 0xFFFFFFFF;

    std::map < std::string, TTimerInfo >::iterator it;

    for (it = _mapTimer.begin(); it != _mapTimer.end(); ++it) {
        // check if timer has expired
        if (it->second.u32OsalExpireTime == 0) {
            //nothing to do --> timer stopped
        } else if (it->second.u32OsalExpireTime <= u32CurrentTime) {
            ETG_TRACE_USR1( ( "spm_tclTimerHandler::vCheckTimerTable(): timer %30s expired --> send message: %d", it->first.c_str(), it->second.u32MessageToSend) );
            //timer expired --> send message
            SPM_NULL_POINTER_CHECK( _poclWorkerServer );
            _poclWorkerServer->bPostMessage( it->second.strClientIfName, it->second.u32MessageToSend, 0 );
            if (it->second.u32CycleTime != 0) {
                ETG_TRACE_USR1( ( "spm_tclTimerHandler::vCheckTimerTable(): Timer is a cyclic timer --> restart") );
                it->second.u32OsalExpireTime = u32CurrentTime + it->second.u32CycleTime;
                if (u32EarliestTimeout > it->second.u32OsalExpireTime) {
                    u32EarliestTimeout = it->second.u32OsalExpireTime;
                }
            } else {
                it->second.u32OsalExpireTime = 0;
            }

        } else {
            ETG_TRACE_USR1( ( "spm_tclTimerHandler::vCheckTimerTable(): timer %30s active --> check expire time: %dms", it->first.c_str(), it->second.u32OsalExpireTime) );
            if (u32EarliestTimeout > it->second.u32OsalExpireTime) {
                u32EarliestTimeout = it->second.u32OsalExpireTime;
            }
        }
    }

    //and now start timer again
    if ((u32EarliestTimeout != 0xffffffff) && (u32EarliestTimeout > OSAL_ClockGetElapsedTime())){
        //retrigger timer
        ETG_TRACE_USR1( ( "spm_tclTimerHandler::vCheckTimerTable(): retrigger timer with: %dms", u32EarliestTimeout - OSAL_ClockGetElapsedTime()) );
        if ( OSAL_s32TimerSetTime( _hTimerHdl, u32EarliestTimeout - OSAL_ClockGetElapsedTime(), 0 ) != OSAL_OK ) {
        }
    }
}

